var documenterSearchIndex = {"docs":
[{"title":"External Profiler Support","location":"devdocs/external_profilers.html#External-Profiler-Support","category":"section","text":"Julia provides explicit support for some external tracing profilers, enabling you to obtain a high-level overview of the runtime's execution behavior.\n\nThe currently supported profilers are:\n\nTracy\nIntel VTune (ITTAPI)","page":"External Profiler Support"},{"title":"Adding New Zones","location":"devdocs/external_profilers.html#Adding-New-Zones","category":"section","text":"To add new zones, use the JL_TIMING macro. You can find numerous examples throughout the codebase by searching for JL_TIMING. To add a new type of zone you add it to JL_TIMING_OWNERS (and possibly JL_TIMING_EVENTS).","page":"External Profiler Support"},{"title":"Dynamically Enabling and Disabling Zones","location":"devdocs/external_profilers.html#Dynamically-Enabling-and-Disabling-Zones","category":"section","text":"The JULIA_TIMING_SUBSYSTEMS environment variable allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE zones.","page":"External Profiler Support"},{"title":"Tracy Profiler","location":"devdocs/external_profilers.html#Tracy-Profiler","category":"section","text":"Tracy  is a flexible profiler that can be optionally integrated with Julia.\n\nA typical Tracy session might look like this:\n\n(Image: Typical Tracy usage)","page":"External Profiler Support"},{"title":"Building Julia with Tracy","location":"devdocs/external_profilers.html#Building-Julia-with-Tracy","category":"section","text":"To enable Tracy integration, build Julia with the extra option WITH_TRACY=1 in the Make.user file.","page":"External Profiler Support"},{"title":"Installing the Tracy Profile Viewer","location":"devdocs/external_profilers.html#Installing-the-Tracy-Profile-Viewer","category":"section","text":"The easiest way to obtain the profile viewer is by adding the TracyProfiler_jll package and launching the profiler with:\n\nrun(TracyProfiler_jll.tracy())\n\nnote: Note\nOn macOS, you may want to set the TRACY_DPI_SCALE environment variable to 1.0 if the UI elements in the profiler appear excessively large.\n\nTo run a \"headless\" instance that saves the trace to disk, use\n\nrun(`$(TracyProfiler_jll.capture()) -o mytracefile.tracy`)\n\ninstead.\n\nFor information on using the Tracy UI, refer to the Tracy manual.","page":"External Profiler Support"},{"title":"Profiling Julia with Tracy","location":"devdocs/external_profilers.html#Profiling-Julia-with-Tracy","category":"section","text":"A typical workflow for profiling Julia with Tracy involves starting Julia using:\n\nJULIA_WAIT_FOR_TRACY=1 ./julia -e '...'\n\nThe environment variable ensures that Julia waits until it has successfully connected to the Tracy profiler before continuing execution. Afterward, use the Tracy profiler UI, click Connect, and Julia execution should resume and profiling should start.","page":"External Profiler Support"},{"title":"Profiling package precompilation with Tracy","location":"devdocs/external_profilers.html#Profiling-package-precompilation-with-Tracy","category":"section","text":"To profile a package precompilation process it is easiest to explicitly call into Base.compilecache with the package you want to precompile:\n\npkg = Base.identify_package(\"SparseArrays\")\nwithenv(\"JULIA_WAIT_FOR_TRACY\" => 1, \"TRACY_PORT\" => 9001) do\n    Base.compilecache(pkg)\nend\n\nHere, we use a custom port for tracy which makes it easier to find the correct client in the Tracy UI to connect to.","page":"External Profiler Support"},{"title":"Adding metadata to zones","location":"devdocs/external_profilers.html#Adding-metadata-to-zones","category":"section","text":"The various jl_timing_show_* and jl_timing_printf functions can be used to attach a string (or strings) to a zone. For example, the trace zone for inference shows the method instance that is being inferred.\n\nThe TracyCZoneColor function can be used to set the color of a certain zone. Search through the codebase to see how it is used.","page":"External Profiler Support"},{"title":"Viewing Tracy files in your browser","location":"devdocs/external_profilers.html#Viewing-Tracy-files-in-your-browser","category":"section","text":"Visit https://topolarity.github.io/trace-viewer/ for an (experimental) web viewer for Tracy traces.\n\nYou can open a local .tracy file or provide a URL from the web (e.g. a file in a Github repo). If you load a trace file from the web, you can also share the page URL directly with others, enabling them to view the same trace.","page":"External Profiler Support"},{"title":"Enabling stack trace samples","location":"devdocs/external_profilers.html#Enabling-stack-trace-samples","category":"section","text":"To enable call stack sampling in Tracy, build Julia with these options in your Make.user file:\n\nWITH_TRACY := 1\nWITH_TRACY_CALLSTACKS := 1\nUSE_BINARYBUILDER_LIBTRACYCLIENT := 0\n\nYou may also need to run make -C deps clean-libtracyclient to force a re-build of Tracy.\n\nThis feature has a significant impact on trace size and profiling overhead, so it is recommended to leave call stack sampling off when possible, especially if you intend to share your trace files online.\n\nNote that the Julia JIT runtime does not yet have integration for Tracy's symbolification, so Julia functions will typically be unknown in these stack traces.","page":"External Profiler Support"},{"title":"Intel VTune (ITTAPI) Profiler","location":"devdocs/external_profilers.html#Intel-VTune-(ITTAPI)-Profiler","category":"section","text":"This section is yet to be written.","page":"External Profiler Support"},{"title":"Embedding Julia","location":"manual/embedding.html#Embedding-Julia","category":"section","text":"As we have seen in Calling C and Fortran Code, Julia has a simple and efficient way to call functions written in C. But there are situations where the opposite is needed: calling Julia functions from C code. This can be used to integrate Julia code into a larger C/C++ project, without the need to rewrite everything in C/C++. Julia has a C API to make this possible. As almost all programming languages have some way to call C functions, the Julia C API can also be used to build further language bridges (e.g. calling Julia from Python, Rust or C#). Even though Rust and C++ can use the C embedding API directly, both have packages helping with it, for C++ Jluna is useful.","page":"Embedding Julia"},{"title":"High-Level Embedding","location":"manual/embedding.html#High-Level-Embedding","category":"section","text":"Note: This section covers embedding Julia code in C on Unix-like operating systems. For doing this on Windows, please see the section following this, High-Level Embedding on Windows with Visual Studio.\n\nWe start with a simple C program that initializes Julia and calls some Julia code:\n\n#include <julia.h>\nJULIA_DEFINE_FAST_TLS // only define this once, in an executable (not in a shared library) if you want fast code.\n\nint main(int argc, char *argv[])\n{\n    /* required: setup the Julia context */\n    jl_init();\n\n    /* run Julia commands */\n    jl_eval_string(\"print(sqrt(2.0))\");\n\n    /* strongly recommended: notify Julia that the\n         program is about to terminate. this allows\n         Julia time to cleanup pending write requests\n         and run all finalizers\n    */\n    jl_atexit_hook(0);\n    return 0;\n}\n\nIn order to build this program you must add the path to the Julia header to the include path and link against libjulia. For instance, when Julia is installed to $JULIA_DIR, one can compile the above test program test.c with gcc using:\n\ngcc -o test -fPIC -I$JULIA_DIR/include/julia -L$JULIA_DIR/lib -Wl,-rpath,$JULIA_DIR/lib test.c -ljulia\n\nAlternatively, look at the embedding.c program in the Julia source tree in the test/embedding/ folder. The file cli/loader_exe.c program is another simple example of how to set jl_options options while linking against libjulia.\n\nThe first thing that must be done before calling any other Julia C function is to initialize Julia. This is done by calling jl_init, which tries to automatically determine Julia's install location. If you need to specify a custom location, or specify which system image to load, use jl_init_with_image_file or jl_init_with_image_handle instead.\n\nThe second statement in the test program evaluates a Julia statement using a call to jl_eval_string.\n\nBefore the program terminates, it is strongly recommended that jl_atexit_hook is called. The above example program calls this just before returning from main.\n\nnote: Note\nCurrently, dynamically linking with the libjulia shared library requires passing the RTLD_GLOBAL option. In Python, this looks like:>>> julia=CDLL('./libjulia.dylib',RTLD_GLOBAL)\n>>> julia.jl_init.argtypes = []\n>>> julia.jl_init()\n250593296\n\nnote: Note\nIf the julia program needs to access symbols from the main executable, it may be necessary to add the -Wl,--export-dynamic linker flag at compile time on Linux in addition to the ones generated by julia-config.jl described below. This is not necessary when compiling a shared library.","page":"Embedding Julia"},{"title":"Using julia-config to automatically determine build parameters","location":"manual/embedding.html#Using-julia-config-to-automatically-determine-build-parameters","category":"section","text":"The script julia-config.jl was created to aid in determining what build parameters are required by a program that uses embedded Julia. This script uses the build parameters and system configuration of the particular Julia distribution it is invoked by to export the necessary compiler flags for an embedding program to interact with that distribution. This script is located in the Julia shared data directory.","page":"Embedding Julia"},{"title":"Example","location":"manual/embedding.html#Example","category":"section","text":"#include <julia.h>\n\nint main(int argc, char *argv[])\n{\n    jl_init();\n    (void)jl_eval_string(\"println(sqrt(2.0))\");\n    jl_atexit_hook(0);\n    return 0;\n}","page":"Embedding Julia"},{"title":"On the command line","location":"manual/embedding.html#On-the-command-line","category":"section","text":"A simple use of this script is from the command line. Assuming that julia-config.jl is located in /usr/local/julia/share/julia, it can be invoked on the command line directly and takes any combination of three flags:\n\n/usr/local/julia/share/julia/julia-config.jl\nUsage: julia-config [--cflags|--ldflags|--ldlibs]\n\nIf the above example source is saved in the file embed_example.c, then the following command will compile it into an executable program on Linux and Windows (MSYS2 environment). On macOS, substitute clang for gcc.:\n\n/usr/local/julia/share/julia/julia-config.jl --cflags --ldflags --ldlibs | xargs gcc embed_example.c","page":"Embedding Julia"},{"title":"Use in Makefiles","location":"manual/embedding.html#Use-in-Makefiles","category":"section","text":"In general, embedding projects will be more complicated than the above example, and so the following allows general makefile support as well – assuming GNU make because of the use of the shell macro expansions. Furthermore, although julia-config.jl is usually in the /usr/local directory, if it isn't, then Julia itself can be used to find julia-config.jl, and the makefile can take advantage of this. The above example is extended to use a makefile:\n\nJL_SHARE = $(shell julia -e 'print(joinpath(Sys.BINDIR, Base.DATAROOTDIR, \"julia\"))')\nCFLAGS   += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nCXXFLAGS += $(shell $(JL_SHARE)/julia-config.jl --cflags)\nLDFLAGS  += $(shell $(JL_SHARE)/julia-config.jl --ldflags)\nLDLIBS   += $(shell $(JL_SHARE)/julia-config.jl --ldlibs)\n\nall: embed_example\n\nNow the build command is simply make.","page":"Embedding Julia"},{"title":"High-Level Embedding on Windows with Visual Studio","location":"manual/embedding.html#High-Level-Embedding-on-Windows-with-Visual-Studio","category":"section","text":"If the JULIA_DIR environment variable hasn't been setup, add it using the System panel before starting Visual Studio. The bin folder under JULIA_DIR should be on the system PATH.\n\nWe start by opening Visual Studio and creating a new Console Application project. Open the 'stdafx.h' header file, and add the following lines at the end:\n\n#include <julia.h>\n\nThen, replace the main() function in the project with this code:\n\nint main(int argc, char *argv[])\n{\n    /* required: setup the Julia context */\n    jl_init();\n\n    /* run Julia commands */\n    jl_eval_string(\"print(sqrt(2.0))\");\n\n    /* strongly recommended: notify Julia that the\n         program is about to terminate. this allows\n         Julia time to cleanup pending write requests\n         and run all finalizers\n    */\n    jl_atexit_hook(0);\n    return 0;\n}\n\nThe next step is to set up the project to find the Julia include files and the libraries. It's important to know whether the Julia installation is 32- or 64-bit. Remove any platform configuration that doesn't correspond to the Julia installation before proceeding.\n\nUsing the project Properties dialog, go to C/C++ | General and add $(JULIA_DIR)\\include\\julia\\ to the Additional Include Directories property. Then, go to the Linker | General section and add $(JULIA_DIR)\\lib to the Additional Library Directories property. Finally, under Linker | Input, add libjulia.dll.a;libopenlibm.dll.a; to the list of libraries.\n\nAt this point, the project should build and run.","page":"Embedding Julia"},{"title":"Converting Types","location":"manual/embedding.html#Converting-Types","category":"section","text":"Real applications will not only need to execute expressions, but also return their values to the host program. jl_eval_string returns a jl_value_t*, which is a pointer to a heap-allocated Julia object. Storing simple data types like Float64 in this way is called boxing, and extracting the stored primitive data is called unboxing. Our improved sample program that calculates the square root of 2 in Julia and reads back the result in C has a body that now contains this code:\n\njl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\n\nif (jl_typeis(ret, jl_float64_type)) {\n    double ret_unboxed = jl_unbox_float64(ret);\n    printf(\"sqrt(2.0) in C: %e \\n\", ret_unboxed);\n}\nelse {\n    printf(\"ERROR: unexpected return type from sqrt(::Float64)\\n\");\n}\n\nIn order to check whether ret is of a specific Julia type, we can use the jl_isa, jl_typeis, or jl_is_... functions. By typing typeof(sqrt(2.0)) into the Julia shell we can see that the return type is Float64 (double in C). To convert the boxed Julia value into a C double the jl_unbox_float64 function is used in the above code snippet.\n\nCorresponding jl_box_... functions are used to convert the other way:\n\njl_value_t *a = jl_box_float64(3.0);\njl_value_t *b = jl_box_float32(3.0f);\njl_value_t *c = jl_box_int32(3);\n\nAs we will see next, boxing is required to call Julia functions with specific arguments.","page":"Embedding Julia"},{"title":"Calling Julia Functions","location":"manual/embedding.html#Calling-Julia-Functions","category":"section","text":"While jl_eval_string allows C to obtain the result of a Julia expression, it does not allow passing arguments computed in C to Julia. For this you will need to invoke Julia functions directly, using jl_call:\n\njl_function_t *func = jl_get_function(jl_base_module, \"sqrt\");\njl_value_t *argument = jl_box_float64(2.0);\njl_value_t *ret = jl_call1(func, argument);\n\nIn the first step, a handle to the Julia function sqrt is retrieved by calling jl_get_function. The first argument passed to jl_get_function is a pointer to the Base module in which sqrt is defined. Then, the double value is boxed using jl_box_float64. Finally, in the last step, the function is called using jl_call1. jl_call0, jl_call2, and jl_call3 functions also exist, to conveniently handle different numbers of arguments. To pass more arguments, use jl_call:\n\njl_value_t *jl_call(jl_function_t *f, jl_value_t **args, int32_t nargs)\n\nIts second argument args is an array of jl_value_t* arguments and nargs is the number of arguments.\n\nThere is also an alternative, possibly simpler, way of calling Julia functions and that is via @cfunction. Using @cfunction allows you to do the type conversions on the Julia side, which is typically easier than doing it on the C side. The sqrt example above would with @cfunction be written as:\n\ndouble (*sqrt_jl)(double) = jl_unbox_voidpointer(jl_eval_string(\"@cfunction(sqrt, Float64, (Float64,))\"));\ndouble ret = sqrt_jl(2.0);\n\nwhere we first define a C callable function in Julia, extract the function pointer from it, and finally call it. In addition to simplifying type conversions by doing them in the higher-level language, calling Julia functions via @cfunction pointers eliminates the dynamic-dispatch overhead required by jl_call (for which all of the arguments are \"boxed\"), and should have performance equivalent to native C function pointers.","page":"Embedding Julia"},{"title":"Memory Management","location":"manual/embedding.html#Memory-Management","category":"section","text":"As we have seen, Julia objects are represented in C as pointers of type jl_value_t*. This raises the question of who is responsible for freeing these objects.\n\nTypically, Julia objects are freed by the garbage collector (GC), but the GC does not automatically know that we are holding a reference to a Julia value from C. This means the GC can free objects out from under you, rendering pointers invalid.\n\nThe GC will only run when new Julia objects are being allocated. Calls like jl_box_float64 perform allocation, but allocation might also happen at any point in running Julia code.\n\nWhen writing code that embeds Julia, it is generally safe to use jl_value_t* values in between jl_... calls (as GC will only get triggered by those calls). But in order to make sure that values can survive jl_... calls, we have to tell Julia that we still hold a reference to Julia root values, a process called \"GC rooting\". Rooting a value will ensure that the garbage collector does not accidentally identify this value as unused and free the memory backing that value. This can be done using the JL_GC_PUSH macros:\n\njl_value_t *ret = jl_eval_string(\"sqrt(2.0)\");\nJL_GC_PUSH1(&ret);\n// Do something with ret\nJL_GC_POP();\n\nThe JL_GC_POP call releases the references established by the previous JL_GC_PUSH. Note that JL_GC_PUSH stores references on the C stack, so it must be exactly paired with a JL_GC_POP before the scope is exited. That is, before the function returns, or control flow otherwise leaves the block in which the JL_GC_PUSH was invoked.\n\nSeveral Julia values can be pushed at once using the JL_GC_PUSH2 to JL_GC_PUSH6 macros:\n\nJL_GC_PUSH2(&ret1, &ret2);\n// ...\nJL_GC_PUSH6(&ret1, &ret2, &ret3, &ret4, &ret5, &ret6);\n\nTo push an array of Julia values one can use the JL_GC_PUSHARGS macro, which can be used as follows:\n\njl_value_t **args;\nJL_GC_PUSHARGS(args, 2); // args can now hold 2 `jl_value_t*` objects\nargs[0] = some_value;\nargs[1] = some_other_value;\n// Do something with args (e.g. call jl_... functions)\nJL_GC_POP();\n\nEach scope must have only one call to JL_GC_PUSH*, and should be paired with only a single JL_GC_POP call. If all necessary variables you want to root cannot be pushed by a one single call to JL_GC_PUSH*, or if there are more than 6 variables to be pushed and using an array of arguments is not an option, then one can use inner blocks:\n\njl_value_t *ret1 = jl_eval_string(\"sqrt(2.0)\");\nJL_GC_PUSH1(&ret1);\njl_value_t *ret2 = 0;\n{\n    jl_function_t *func = jl_get_function(jl_base_module, \"exp\");\n    ret2 = jl_call1(func, ret1);\n    JL_GC_PUSH1(&ret2);\n    // Do something with ret2.\n    JL_GC_POP();    // This pops ret2.\n}\nJL_GC_POP();    // This pops ret1.\n\nNote that it is not necessary to have valid jl_value_t* values before calling JL_GC_PUSH*. It is fine to have a number of them initialized to NULL, pass those to JL_GC_PUSH* and then create the actual Julia values. For example:\n\njl_value_t *ret1 = NULL, *ret2 = NULL;\nJL_GC_PUSH2(&ret1, &ret2);\nret1 = jl_eval_string(\"sqrt(2.0)\");\nret2 = jl_eval_string(\"sqrt(3.0)\");\n// Use ret1 and ret2\nJL_GC_POP();\n\nIf it is required to hold the pointer to a variable between functions (or block scopes), then it is not possible to use JL_GC_PUSH*. In this case, it is necessary to create and keep a reference to the variable in the Julia global scope. One simple way to accomplish this is to use a global IdDict that will hold the references, avoiding deallocation by the GC. However, this method will only work properly with mutable types.\n\n// This functions shall be executed only once, during the initialization.\njl_value_t* refs = jl_eval_string(\"refs = IdDict()\");\njl_function_t* setindex = jl_get_function(jl_base_module, \"setindex!\");\n\n...\n\n// `var` is the variable we want to protect between function calls.\njl_value_t* var = 0;\n\n...\n\n// `var` is a `Vector{Float64}`, which is mutable.\nvar = jl_eval_string(\"[sqrt(2.0); sqrt(4.0); sqrt(6.0)]\");\n\n// To protect `var`, add its reference to `refs`.\njl_call3(setindex, refs, var, var);\n\nIf the variable is immutable, then it needs to be wrapped in an equivalent mutable container or, preferably, in a RefValue{Any} before it is pushed to IdDict. In this approach, the container has to be created or filled in via C code using, for example, the function jl_new_struct. If the container is created by jl_call*, then you will need to reload the pointer to be used in C code.\n\n// This functions shall be executed only once, during the initialization.\njl_value_t* refs = jl_eval_string(\"refs = IdDict()\");\njl_function_t* setindex = jl_get_function(jl_base_module, \"setindex!\");\njl_datatype_t* reft = (jl_datatype_t*)jl_eval_string(\"Base.RefValue{Any}\");\n\n...\n\n// `var` is the variable we want to protect between function calls.\njl_value_t* var = 0;\n\n...\n\n// `var` is a `Float64`, which is immutable.\nvar = jl_eval_string(\"sqrt(2.0)\");\n\n// Protect `var` until we add its reference to `refs`.\nJL_GC_PUSH1(&var);\n\n// Wrap `var` in `RefValue{Any}` and push to `refs` to protect it.\njl_value_t* rvar = jl_new_struct(reft, var);\nJL_GC_POP();\n\njl_call3(setindex, refs, rvar, rvar);\n\nThe GC can be allowed to deallocate a variable by removing the reference to it from refs using the function delete!, provided that no other reference to the variable is kept anywhere:\n\njl_function_t* delete = jl_get_function(jl_base_module, \"delete!\");\njl_call2(delete, refs, rvar);\n\nAs an alternative for very simple cases, it is possible to just create a global container of type Vector{Any} and fetch the elements from that when necessary, or even to create one global variable per pointer using\n\njl_module_t *mod = jl_main_module;\njl_sym_t *var = jl_symbol(\"var\");\njl_binding_t *bp = jl_get_binding_wr(mod, var, 1);\njl_checked_assignment(bp, mod, var, val);","page":"Embedding Julia"},{"title":"Updating fields of GC-managed objects","location":"manual/embedding.html#Updating-fields-of-GC-managed-objects","category":"section","text":"The garbage collector also operates under the assumption that it is aware of every older-generation object pointing to a younger-generation one. Any time a pointer is updated breaking that assumption, it must be signaled to the collector with the jl_gc_wb (write barrier) function like so:\n\njl_value_t *parent = some_old_value, *child = some_young_value;\n((some_specific_type*)parent)->field = child;\njl_gc_wb(parent, child);\n\nIt is in general impossible to predict which values will be old at runtime, so the write barrier must be inserted after all explicit stores. One notable exception is if the parent object has just been allocated and no garbage collection has run since then. Note that most jl_... functions can sometimes invoke garbage collection.\n\nThe write barrier is also necessary for arrays of pointers when updating their data directly. Calling jl_array_ptr_set is usually much preferred. But direct updates can be done. For example:\n\njl_array_t *some_array = ...; // e.g. a Vector{Any}\nvoid **data = jl_array_data(some_array, void*);\njl_value_t *some_value = ...;\ndata[0] = some_value;\njl_gc_wb(jl_array_owner(some_array), some_value);","page":"Embedding Julia"},{"title":"Controlling the Garbage Collector","location":"manual/embedding.html#Controlling-the-Garbage-Collector","category":"section","text":"There are some functions to control the GC. In normal use cases, these should not be necessary.\n\nFunction Description\njl_gc_collect() Force a GC run\njl_gc_enable(0) Disable the GC, return previous state as int\njl_gc_enable(1) Enable the GC,  return previous state as int\njl_gc_is_enabled() Return current state as int","page":"Embedding Julia"},{"title":"Working with Arrays","location":"manual/embedding.html#Working-with-Arrays","category":"section","text":"Julia and C can share array data without copying. The next example will show how this works.\n\nJulia arrays are represented in C by the datatype jl_array_t*. Basically, jl_array_t is a struct that contains:\n\nInformation about the datatype\nA pointer to the data block\nInformation about the sizes of the array\n\nTo keep things simple, we start with a 1D array. Creating an array containing Float64 elements of length 10 can be done like this:\n\njl_value_t* array_type = jl_apply_array_type((jl_value_t*)jl_float64_type, 1);\njl_array_t* x          = jl_alloc_array_1d(array_type, 10);\n\nAlternatively, if you have already allocated the array you can generate a thin wrapper around its data:\n\ndouble *existingArray = (double*)malloc(sizeof(double)*10);\njl_array_t *x = jl_ptr_to_array_1d(array_type, existingArray, 10, 0);\n\nThe last argument is a boolean indicating whether Julia should take ownership of the data. If this argument is non-zero, the GC will call free on the data pointer when the array is no longer referenced.\n\nIn order to access the data of x, we can use jl_array_data:\n\ndouble *xData = jl_array_data(x, double);\n\nNow we can fill the array:\n\nfor (size_t i = 0; i < jl_array_nrows(x); i++)\n    xData[i] = i;\n\nNow let us call a Julia function that performs an in-place operation on x:\n\njl_function_t *func = jl_get_function(jl_base_module, \"reverse!\");\njl_call1(func, (jl_value_t*)x);\n\nBy printing the array, one can verify that the elements of x are now reversed.","page":"Embedding Julia"},{"title":"Accessing Returned Arrays","location":"manual/embedding.html#Accessing-Returned-Arrays","category":"section","text":"If a Julia function returns an array, the return value of jl_eval_string and jl_call can be cast to a jl_array_t*:\n\njl_function_t *func  = jl_get_function(jl_base_module, \"reverse\");\njl_array_t *y = (jl_array_t*)jl_call1(func, (jl_value_t*)x);\n\nNow the content of y can be accessed as before using jl_array_data. As always, be sure to keep a reference to the array while it is in use.","page":"Embedding Julia"},{"title":"Multidimensional Arrays","location":"manual/embedding.html#Multidimensional-Arrays","category":"section","text":"Julia's multidimensional arrays are stored in memory in column-major order. Here is some code that creates a 2D array and accesses its properties:\n\n// Create 2D array of float64 type\njl_value_t *array_type = jl_apply_array_type((jl_value_t*)jl_float64_type, 2);\nint dims[] = {10,5};\njl_array_t *x  = jl_alloc_array_nd(array_type, dims, 2);\n\n// Get array pointer\ndouble *p = jl_array_data(x, double);\n// Get number of dimensions\nint ndims = jl_array_ndims(x);\n// Get the size of the i-th dim\nsize_t size0 = jl_array_dim(x,0);\nsize_t size1 = jl_array_dim(x,1);\n\n// Fill array with data\nfor(size_t i=0; i<size1; i++)\n    for(size_t j=0; j<size0; j++)\n        p[j + size0*i] = i + j;\n\nNotice that while Julia arrays use 1-based indexing, the C API uses 0-based indexing (for example in calling jl_array_dim) in order to read as idiomatic C code.","page":"Embedding Julia"},{"title":"Exceptions","location":"manual/embedding.html#Exceptions","category":"section","text":"Julia code can throw exceptions. For example, consider:\n\njl_eval_string(\"this_function_does_not_exist()\");\n\nThis call will appear to do nothing. However, it is possible to check whether an exception was thrown:\n\nif (jl_exception_occurred())\n    printf(\"%s \\n\", jl_typeof_str(jl_exception_occurred()));\n\nIf you are using the Julia C API from a language that supports exceptions (e.g. Python, C#, C++), it makes sense to wrap each call into libjulia with a function that checks whether an exception was thrown, and then rethrows the exception in the host language.","page":"Embedding Julia"},{"title":"Throwing Julia Exceptions","location":"manual/embedding.html#Throwing-Julia-Exceptions","category":"section","text":"When writing Julia callable functions, it might be necessary to validate arguments and throw exceptions to indicate errors. A typical type check looks like:\n\nif (!jl_typeis(val, jl_float64_type)) {\n    jl_type_error(function_name, (jl_value_t*)jl_float64_type, val);\n}\n\nGeneral exceptions can be raised using the functions:\n\nvoid jl_error(const char *str);\nvoid jl_errorf(const char *fmt, ...);\n\njl_error takes a C string, and jl_errorf is called like printf:\n\njl_errorf(\"argument x = %d is too large\", x);\n\nwhere in this example x is assumed to be an integer.","page":"Embedding Julia"},{"title":"Thread-safety","location":"manual/embedding.html#Thread-safety","category":"section","text":"In general, the Julia C API is not fully thread-safe. When embedding Julia in a multi-threaded application care needs to be taken not to violate the following restrictions:\n\njl_init() may only be called once in the application life-time. The same applies to jl_atexit_hook(), and it may only be called after jl_init().\njl_...() API functions may only be called from the thread in which jl_init() was called, or from threads started by the Julia runtime. Calling Julia API functions from user-started threads is not supported, and may lead to undefined behaviour and crashes.\n\nThe second condition above implies that you can not safely call jl_...() functions from threads that were not started by Julia (the thread calling jl_init() being the exception). For example, the following is not supported and will most likely segfault:\n\nvoid *func(void*)\n{\n    // Wrong, jl_eval_string() called from thread that was not started by Julia\n    jl_eval_string(\"println(Threads.threadid())\");\n    return NULL;\n}\n\nint main()\n{\n    pthread_t t;\n\n    jl_init();\n\n    // Start a new thread\n    pthread_create(&t, NULL, func, NULL);\n    pthread_join(t, NULL);\n\n    jl_atexit_hook(0);\n}\n\nInstead, performing all Julia calls from the same user-created thread will work:\n\nvoid *func(void*)\n{\n    // Okay, all jl_...() calls from the same thread,\n    // even though it is not the main application thread\n    jl_init();\n    jl_eval_string(\"println(Threads.threadid())\");\n    jl_atexit_hook(0);\n    return NULL;\n}\n\nint main()\n{\n    pthread_t t;\n    // Create a new thread, which runs func()\n    pthread_create(&t, NULL, func, NULL);\n    pthread_join(t, NULL);\n}\n\nAn example of calling the Julia C API from a thread started by Julia itself:\n\n#include <julia/julia.h>\nJULIA_DEFINE_FAST_TLS\n\ndouble c_func(int i)\n{\n    printf(\"[C %08x] i = %d\\n\", pthread_self(), i);\n\n    // Call the Julia sqrt() function to compute the square root of i, and return it\n    jl_function_t *sqrt = jl_get_function(jl_base_module, \"sqrt\");\n    jl_value_t* arg = jl_box_int32(i);\n    double ret = jl_unbox_float64(jl_call1(sqrt, arg));\n\n    return ret;\n}\n\nint main()\n{\n    jl_init();\n\n    // Define a Julia function func() that calls our c_func() defined in C above\n    jl_eval_string(\"func(i) = ccall(:c_func, Float64, (Int32,), i)\");\n\n    // Call func() multiple times, using multiple threads to do so\n    jl_eval_string(\"println(Threads.threadpoolsize())\");\n    jl_eval_string(\"use(i) = println(\\\"[J $(Threads.threadid())] i = $(i) -> $(func(i))\\\")\");\n    jl_eval_string(\"Threads.@threads for i in 1:5 use(i) end\");\n\n    jl_atexit_hook(0);\n}\n\nIf we run this code with 2 Julia threads we get the following output (note: the output will vary per run and system):\n\n$ JULIA_NUM_THREADS=2 ./thread_example\n2\n[C 3bfd9c00] i = 1\n[C 23938640] i = 4\n[J 1] i = 1 -> 1.0\n[C 3bfd9c00] i = 2\n[J 1] i = 2 -> 1.4142135623730951\n[C 3bfd9c00] i = 3\n[J 2] i = 4 -> 2.0\n[C 23938640] i = 5\n[J 1] i = 3 -> 1.7320508075688772\n[J 2] i = 5 -> 2.23606797749979\n\nAs can be seen, Julia thread 1 corresponds to pthread ID 3bfd9c00, and Julia thread 2 corresponds to ID 23938640, showing that indeed multiple threads are used at the C level, and that we can safely call Julia C API routines from those threads.","page":"Embedding Julia"},{"title":"Mathematical Operations and Elementary Functions","location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","category":"section","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions.","page":"Mathematical Operations and Elementary Functions"},{"title":"Arithmetic Operators","location":"manual/mathematical-operations.html#Arithmetic-Operators","category":"section","text":"The following arithmetic operators are supported on all primitive numeric types:\n\nExpression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x, y)\n\nA numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x + y), is treated as a multiplication, except with higher precedence than other binary operations. See Numeric Literal Coefficients for details.\n\nJulia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system.\n\nThe ÷ sign can be conveniently typed by writing \\div<tab> to the REPL or Julia IDE. See the manual section on Unicode input for more information.\n\nHere are some simple examples using arithmetic operators:\n\njulia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5\n\n(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)\n\nWhen used in multiplication, false acts as a strong zero:\n\njulia> NaN * false\n0.0\n\njulia> false * Inf\n0.0\n\nThis is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation.","page":"Mathematical Operations and Elementary Functions"},{"title":"Boolean Operators","location":"manual/mathematical-operations.html#Boolean-Operators","category":"section","text":"The following Boolean operators are supported on Bool types:\n\nExpression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or\n\nNegation changes true to false and vice versa. The short-circuiting operations are explained on the linked page.\n\nNote that Bool is an integer type and all the usual promotion rules and numeric operators are also defined on it.","page":"Mathematical Operations and Elementary Functions"},{"title":"Bitwise Operators","location":"manual/mathematical-operations.html#Bitwise-Operators","category":"section","text":"The following bitwise operators are supported on all primitive integer types:\n\nExpression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx ⊼ y bitwise nand (not and)\nx ⊽ y bitwise nor (not or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left\n\nHere are some examples with bitwise operators:\n\njulia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> nand(123, 123)\n-124\n\njulia> 123 ⊼ 123\n-124\n\njulia> nor(123, 124)\n-128\n\njulia> 123 ⊽ 124\n-128\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84","page":"Mathematical Operations and Elementary Functions"},{"title":"Updating operators","location":"manual/mathematical-operations.html#Updating-operators","category":"section","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:\n\njulia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4\n\nThe updating versions of all the binary arithmetic and bitwise operators are:\n\n+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=\n\nnote: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64","page":"Mathematical Operations and Elementary Functions"},{"title":"Vectorized \"dot\" operators","location":"manual/mathematical-operations.html#man-dot-operators","category":"section","text":"For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1, 2, 3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" a (non-square) array, but [1, 2, 3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3]. Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise.\n\njulia> [1, 2, 3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27\n\nMore specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element a of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).\n\nFurthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation).\n\nNote the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A, B) = kron(A, B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A, B] .⊗ [C, D] will compute [A⊗C, B⊗D] with no additional coding.\n\nCombining dot operators with numeric literals can be ambiguous. For example, it is not clear whether 1.+x means 1. + x or 1 .+ x. Therefore this syntax is disallowed, and spaces must be used around the operator in such cases.","page":"Mathematical Operations and Elementary Functions"},{"title":"Numeric Comparisons","location":"manual/mathematical-operations.html#Numeric-Comparisons","category":"section","text":"Standard comparison operations are defined for all the primitive numeric types:\n\nOperator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to\n\nHere are some simple examples:\n\njulia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse\n\nIntegers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:\n\nFinite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.\n\nThe last point is potentially surprising and thus worth noting:\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nand can cause headaches when working with arrays:\n\njulia> [1 NaN] == [1 NaN]\nfalse\n\nJulia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:\n\nFunction Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number\n\nisequal considers NaNs equal to each other:\n\njulia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue\n\nisequal can also be used to distinguish signed zeros:\n\njulia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse\n\nMixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly.\n\nFor other types, isequal defaults to calling ==, so if you want to define equality for your own types then you only need to add a == method. If you define your own equality function, you should probably define a corresponding hash method to ensure that isequal(x,y) implies hash(x) == hash(y).","page":"Mathematical Operations and Elementary Functions"},{"title":"Chaining comparisons","location":"manual/mathematical-operations.html#Chaining-comparisons","category":"section","text":"Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:\n\njulia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue\n\nChaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1.\n\nNote the evaluation behavior of chained comparisons:\n\njulia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse\n\nThe middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation).","page":"Mathematical Operations and Elementary Functions"},{"title":"Elementary Functions","location":"manual/mathematical-operations.html#Elementary-Functions","category":"section","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense.\n\nMoreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A.","page":"Mathematical Operations and Elementary Functions"},{"title":"Operator Precedence and Associativity","location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","category":"section","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:\n\nCategory Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - ! ~ ¬ √ ∛ ∜ ⋆ ± ∓ <: >: Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right\n\n[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a)).\n\n[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation.\n\nFor a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scm. Note that some of the operators there are not defined in the Base module but may be given definitions by standard libraries, packages or user code.\n\nYou can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:\n\njulia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)\n\nA symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:\n\njulia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)\n\nNote that symbols such as :sin return precedence 0. This value represents invalid operators and not operators of lowest precedence. Similarly, such operators are assigned associativity :none.\n\nNumeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent.\n\njulia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64\n\nJuxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y).","page":"Mathematical Operations and Elementary Functions"},{"title":"Numerical Conversions","location":"manual/mathematical-operations.html#Numerical-Conversions","category":"section","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions.\n\nThe notation T(x) or convert(T, x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x)).\n\nThe following examples show the different forms.\n\njulia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\nSee Conversion and Promotion for how to define your own conversions and promotions.","page":"Mathematical Operations and Elementary Functions"},{"title":"Rounding functions","location":"manual/mathematical-operations.html#Rounding-functions","category":"section","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T","page":"Mathematical Operations and Elementary Functions"},{"title":"Division functions","location":"manual/mathematical-operations.html#Division-functions","category":"section","text":"Function Description\ndiv(x, y), x÷y truncated division; quotient rounded towards zero\nfld(x, y) floored division; quotient rounded towards -Inf\ncld(x, y) ceiling division; quotient rounded towards +Inf\nrem(x, y), x%y remainder; satisfies x == div(x, y)*y + rem(x, y); sign matches x\nmod(x, y) modulus; satisfies x == fld(x, y)*y + mod(x, y); sign matches y\nmod1(x, y) mod with offset 1; returns r∈(0, y] for y>0 or r∈[y, 0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x, y) returns (div(x, y),rem(x, y))\nfldmod(x, y) returns (fld(x, y), mod(x, y))\ngcd(x, y...) greatest positive common divisor of x, y,...\nlcm(x, y...) least positive common multiple of x, y,...","page":"Mathematical Operations and Elementary Functions"},{"title":"Sign and absolute value functions","location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","category":"section","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x, y) a value with the magnitude of x and the sign of y\nflipsign(x, y) a value with the magnitude of x and the sign of x*y","page":"Mathematical Operations and Elementary Functions"},{"title":"Powers, logs and roots","location":"manual/mathematical-operations.html#Powers,-logs-and-roots","category":"section","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nfourthroot(x), ∜x fourth root of x\nhypot(x, y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x) - 1 for x near zero\nldexp(x, n) x * 2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b, x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1 + x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x\n\nFor an overview of why functions like hypot, expm1, and log1p are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot.","page":"Mathematical Operations and Elementary Functions"},{"title":"Trigonometric and hyperbolic functions","location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","category":"section","text":"All the standard trigonometric and hyperbolic functions are also defined:\n\nsin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc\n\nThese are all single-argument functions, with atan also accepting two arguments corresponding to a traditional atan2 function.\n\nAdditionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi * x) and cos(pi * x) respectively.\n\nIn order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:\n\nsind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd","page":"Mathematical Operations and Elementary Functions"},{"title":"Special functions","location":"manual/mathematical-operations.html#Special-functions","category":"section","text":"Many other special mathematical functions are provided by the package SpecialFunctions.jl.","page":"Mathematical Operations and Elementary Functions"},{"title":"Parallel Computing","location":"manual/parallel-computing.html#Parallel-Computing","category":"section","text":"Julia supports these four categories of concurrent and parallel programming:\n\nAsynchronous \"tasks\", or coroutines:\nJulia Tasks allow suspending and resuming computations  for I/O, event handling, producer-consumer processes, and similar patterns.  Tasks can synchronize through operations like wait and fetch, and  communicate via Channels. While strictly not parallel computing by themselves,  Julia lets you schedule Tasks on several threads.\nMulti-threading:\nJulia's multi-threading provides the ability to schedule Tasks  simultaneously on more than one thread or CPU core, sharing memory. This is usually the easiest way  to get parallelism on one's PC or on a single large multi-core server. Julia's multi-threading  is composable. When one multi-threaded function calls another multi-threaded function, Julia  will schedule all the threads globally on available resources, without oversubscribing.\nDistributed computing:\nDistributed computing runs multiple Julia processes with separate memory spaces. These can be on the same  computer or multiple computers. The Distributed standard library provides the capability for remote execution  of a Julia function. With this basic building block, it is possible to build many different kinds of  distributed computing abstractions. Packages like DistributedArrays.jl  are an example of such an abstraction. On the other hand, packages like MPI.jl and  Elemental.jl provide access to the existing MPI ecosystem of libraries.\nGPU computing:\nThe Julia GPU compiler provides the ability to run Julia code natively on GPUs. There  is a rich ecosystem of Julia packages that target GPUs. The JuliaGPU.org  website provides a list of capabilities, supported GPUs, related packages and documentation.","page":"Parallel Computing"},{"title":"Memory Management and Garbage Collection","location":"manual/memory-management.html#man-memory-management","category":"section","text":"Julia uses automatic memory management through its built-in garbage collector (GC). This section provides an overview of how Julia manages memory and how you can configure and optimize memory usage for your applications.","page":"Memory Management and Garbage Collection"},{"title":"Garbage Collection Overview","location":"manual/memory-management.html#man-gc-overview","category":"section","text":"Julia features a garbage collector with the following characteristics:\n\nNon-moving: Objects are not relocated in memory during garbage collection\nGenerational: Younger objects are collected more frequently than older ones\nParallel and partially concurrent: The GC can use multiple threads and run concurrently with your program\nMostly precise: The GC accurately identifies object references for pure Julia code, and it provides conservative scanning APIs for users calling Julia from C\n\nThe garbage collector automatically reclaims memory used by objects that are no longer reachable from your program, freeing you from manual memory management in most cases.","page":"Memory Management and Garbage Collection"},{"title":"Memory Architecture","location":"manual/memory-management.html#man-memory-architecture","category":"section","text":"Julia uses a two-tier allocation strategy:\n\nSmall objects (currently ≤ 2032 bytes but may change): Allocated using a fast per-thread pool allocator\nLarge objects : Allocated directly through the system's malloc\n\nThis hybrid approach optimizes for both allocation speed and memory efficiency, with the pool allocator providing fast allocation for the many small objects typical in Julia programs.","page":"Memory Management and Garbage Collection"},{"title":"System Memory Requirements","location":"manual/memory-management.html#man-system-memory","category":"section","text":"","page":"Memory Management and Garbage Collection"},{"title":"Swap Space","location":"manual/memory-management.html#Swap-Space","category":"section","text":"Julia's garbage collector is designed with the expectation that your system has adequate swap space configured. The GC uses heuristics that assume it can allocate memory beyond physical RAM when needed, relying on the operating system's virtual memory management.\n\nIf your system has limited or no swap space, you may experience out-of-memory errors during garbage collection. In such cases, you can use the --heap-size-hint option to limit Julia's memory usage.","page":"Memory Management and Garbage Collection"},{"title":"Memory Hints","location":"manual/memory-management.html#Memory-Hints","category":"section","text":"You can provide a hint to Julia about the maximum amount of memory to use:\n\njulia --heap-size-hint=4G  # To set the hint to ~4GB\njulia --heap-size-hint=50% # or to 50% of physical memory\n\nThe --heap-size-hint option tells the garbage collector to trigger collection more aggressively when approaching the specified limit. This is particularly useful in:\n\nContainers with memory limits\nSystems without swap space\nShared systems where you want to limit Julia's memory footprint\n\nYou can also set this via the JULIA_HEAP_SIZE_HINT environment variable:\n\nexport JULIA_HEAP_SIZE_HINT=2G\njulia","page":"Memory Management and Garbage Collection"},{"title":"Multithreaded Garbage Collection","location":"manual/memory-management.html#man-gc-multithreading","category":"section","text":"Julia's garbage collector can leverage multiple threads to improve performance on multi-core systems.","page":"Memory Management and Garbage Collection"},{"title":"GC Thread Configuration","location":"manual/memory-management.html#GC-Thread-Configuration","category":"section","text":"By default, Julia uses multiple threads for garbage collection:\n\nMark threads: Used during the mark phase to trace object references (default: 1, which is shared with the compute thread if there is only one, otherwise half the number of compute threads)\nSweep threads: Used for concurrent sweeping of freed memory (default: 0, disabled)\n\nYou can configure GC threading using:\n\njulia --gcthreads=4,1  # 4 mark threads, 1 sweep thread\njulia --gcthreads=8    # 8 mark threads, 0 sweep threads\n\nOr via environment variable:\n\nexport JULIA_NUM_GC_THREADS=4,1\njulia","page":"Memory Management and Garbage Collection"},{"title":"Recommendations","location":"manual/memory-management.html#Recommendations","category":"section","text":"For compute-intensive workloads:\n\nUse multiple mark threads (the default configuration is usually appropriate)\nConsider enabling concurrent sweeping with 1 sweep thread for allocation-heavy workloads\n\nFor memory-intensive workloads:\n\nEnable concurrent sweeping to reduce GC pauses\nMonitor GC time using @time and adjust thread counts accordingly","page":"Memory Management and Garbage Collection"},{"title":"Monitoring and Debugging","location":"manual/memory-management.html#man-gc-monitoring","category":"section","text":"","page":"Memory Management and Garbage Collection"},{"title":"Basic Memory Monitoring","location":"manual/memory-management.html#Basic-Memory-Monitoring","category":"section","text":"Use the @time macro to see memory allocation and GC overhead:\n\njulia> @time some_computation()\n  2.123456 seconds (1.50 M allocations: 58.725 MiB, 17.17% gc time)","page":"Memory Management and Garbage Collection"},{"title":"GC Logging","location":"manual/memory-management.html#GC-Logging","category":"section","text":"Enable detailed GC logging to understand collection patterns:\n\njulia> GC.enable_logging(true)\njulia> # Run your code\njulia> GC.enable_logging(false)\n\nThis logs each garbage collection event with timing and memory statistics.","page":"Memory Management and Garbage Collection"},{"title":"Manual GC Control","location":"manual/memory-management.html#Manual-GC-Control","category":"section","text":"While generally not recommended, you can manually trigger garbage collection:\n\nGC.gc()          # Force a garbage collection\nGC.enable(false) # Disable automatic GC (use with caution!)\nGC.enable(true)  # Re-enable automatic GC\n\nWarning: Disabling GC can lead to memory exhaustion. Only use this for specific performance measurements or debugging.","page":"Memory Management and Garbage Collection"},{"title":"Performance Considerations","location":"manual/memory-management.html#man-gc-performance","category":"section","text":"","page":"Memory Management and Garbage Collection"},{"title":"Reducing Allocations","location":"manual/memory-management.html#Reducing-Allocations","category":"section","text":"The best way to minimize GC impact is to reduce unnecessary allocations:\n\nUse in-place operations when possible (e.g., x .+= y instead of x = x + y)\nPre-allocate arrays and reuse them\nAvoid creating temporary objects in tight loops\nConsider using StaticArrays.jl for small, fixed-size arrays","page":"Memory Management and Garbage Collection"},{"title":"Memory-Efficient Patterns","location":"manual/memory-management.html#Memory-Efficient-Patterns","category":"section","text":"Avoid global variables that change type\nUse const for global constants","page":"Memory Management and Garbage Collection"},{"title":"Profiling Memory Usage","location":"manual/memory-management.html#Profiling-Memory-Usage","category":"section","text":"For detailed guidance on profiling memory allocations and identifying performance bottlenecks, see the Profiling section.","page":"Memory Management and Garbage Collection"},{"title":"Advanced Configuration","location":"manual/memory-management.html#man-gc-advanced","category":"section","text":"","page":"Memory Management and Garbage Collection"},{"title":"Integration with System Memory Management","location":"manual/memory-management.html#Integration-with-System-Memory-Management","category":"section","text":"Julia works best when:\n\nThe system has adequate swap space (recommended: 2x physical RAM)\nVirtual memory is properly configured\nOther processes leave sufficient memory available\nContainer memory limits are set appropriately with --heap-size-hint","page":"Memory Management and Garbage Collection"},{"title":"Troubleshooting Memory Issues","location":"manual/memory-management.html#man-gc-troubleshooting","category":"section","text":"","page":"Memory Management and Garbage Collection"},{"title":"High GC Overhead","location":"manual/memory-management.html#High-GC-Overhead","category":"section","text":"If garbage collection is taking too much time:\n\nReduce allocation rate: Focus on algorithmic improvements\nAdjust GC threads: Experiment with different --gcthreads settings\nUse concurrent sweeping: Enable background sweeping with --gcthreads=N,1\nProfile memory patterns: Identify allocation hotspots and optimize them","page":"Memory Management and Garbage Collection"},{"title":"Memory Leaks","location":"manual/memory-management.html#Memory-Leaks","category":"section","text":"While Julia's GC prevents most memory leaks, issues can still occur:\n\nGlobal references: Avoid holding references to large objects in global variables\nClosures: Be careful with closures that capture large amounts of data\nC interop: Ensure proper cleanup when interfacing with C libraries\n\nFor more detailed information about Julia's garbage collector internals, see the Garbage Collection section in the Developer Documentation.","page":"Memory Management and Garbage Collection"},{"title":"Multi-Threading","location":"manual/multi-threading.html#man-multithreading","category":"section","text":"Visit this blog post for a presentation of Julia multi-threading features.","page":"Multi-Threading"},{"title":"Starting Julia with multiple threads","location":"manual/multi-threading.html#Starting-Julia-with-multiple-threads","category":"section","text":"By default, Julia starts up with 2 threads of execution; 1 worker thread and 1 interactive thread. This can be verified by using the command Threads.nthreads():\n\njulia> Threads.nthreads(:default)\n1\njulia> Threads.nthreads(:interactive)\n1\n\nThe number of execution threads is controlled either by using the -t/--threads command line argument or by using the JULIA_NUM_THREADS environment variable. When both are specified, then -t/--threads takes precedence.\n\nThe number of threads can either be specified as an integer (--threads=4) or as auto (--threads=auto), where auto tries to infer a useful default number of threads to use (see Command-line Options for more details).\n\nSee threadpools for how to control how many :default and :interactive threads are in each threadpool.\n\ncompat: Julia 1.5\nThe -t/--threads command line argument requires at least Julia 1.5. In older versions you must use the environment variable instead.\n\ncompat: Julia 1.7\nUsing auto as value of the environment variable JULIA_NUM_THREADS requires at least Julia 1.7. In older versions, this value is ignored.\n\ncompat: Julia 1.12\nStarting by default with 1 interactive thread, as well as the 1 worker thread, was made as such in Julia 1.12 If the number of threads is set to 1 by either doing -t1 or JULIA_NUM_THREADS=1 an interactive thread will not be spawned.\n\nLets start Julia with 4 threads:\n\n$ julia --threads 4\n\nLet's verify there are 4 threads at our disposal.\n\njulia> Threads.nthreads()\n4\n\nBut we are currently on the master thread. To check, we use the function Threads.threadid\n\njulia> Threads.threadid()\n1\n\nnote: Note\nIf you prefer to use the environment variable you can set it as follows in Bash (Linux/macOS):export JULIA_NUM_THREADS=4C shell on Linux/macOS, CMD on Windows:set JULIA_NUM_THREADS=4Powershell on Windows:$env:JULIA_NUM_THREADS=4Note that this must be done before starting Julia.\n\nnote: Note\nThe number of threads specified with -t/--threads is propagated to worker processes that are spawned using the -p/--procs or --machine-file command line options. For example, julia -p2 -t2 spawns 1 main process with 2 worker processes, and all three processes have 2 threads enabled. For more fine grained control over worker threads use addprocs and pass -t/--threads as exeflags.","page":"Multi-Threading"},{"title":"Multiple GC Threads","location":"manual/multi-threading.html#Multiple-GC-Threads","category":"section","text":"The Garbage Collector (GC) can use multiple threads. The amount used by default matches the compute worker threads or can configured by either the --gcthreads command line argument or by using the JULIA_NUM_GC_THREADS environment variable.\n\ncompat: Julia 1.10\nThe --gcthreads command line argument requires at least Julia 1.10.\n\nFor more details about garbage collection configuration and performance tuning, see Memory Management and Garbage Collection.","page":"Multi-Threading"},{"title":"Threadpools","location":"manual/multi-threading.html#man-threadpools","category":"section","text":"When a program's threads are busy with many tasks to run, tasks may experience delays which may negatively affect the responsiveness and interactivity of the program. To address this, you can specify that a task is interactive when you Threads.@spawn it:\n\nusing Base.Threads\n@spawn :interactive f()\n\nInteractive tasks should avoid performing high latency operations, and if they are long duration tasks, should yield frequently.\n\nBy default Julia starts with one interactive thread reserved to run interactive tasks, but that number can be controlled with:\n\n$ julia --threads 3,1\njulia> Threads.nthreads(:interactive)\n1\n\n$ julia --threads 3,0\njulia> Threads.nthreads(:interactive)\n0\n\nThe environment variable JULIA_NUM_THREADS can also be used similarly:\n\nexport JULIA_NUM_THREADS=3,1\n\nThis starts Julia with 3 threads in the :default threadpool and 1 thread in the :interactive threadpool:\n\njulia> using Base.Threads\n\njulia> nthreadpools()\n2\n\njulia> threadpool() # the main thread is in the interactive thread pool\n:interactive\n\njulia> nthreads(:default)\n3\n\njulia> nthreads(:interactive)\n1\n\njulia> nthreads()\n3\n\nnote: Note\nExplicitly asking for 1 thread by doing -t1 or JULIA_NUM_THREADS=1 does not add an interactive thread.\n\nnote: Note\nThe zero-argument version of nthreads returns the number of threads in the default pool.\n\nnote: Note\nDepending on whether Julia has been started with interactive threads, the main thread is either in the default or interactive thread pool.\n\nEither or both numbers can be replaced with the word auto, which causes Julia to choose a reasonable default.","page":"Multi-Threading"},{"title":"The @threads Macro","location":"manual/multi-threading.html#The-@threads-Macro","category":"section","text":"Let's work a simple example using our native threads. Let us create an array of zeros:\n\njulia> a = zeros(10)\n10-element Vector{Float64}:\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n\nLet us operate on this array simultaneously using 4 threads. We'll have each thread write its thread ID into each location.\n\nJulia supports parallel loops using the Threads.@threads macro. This macro is affixed in front of a for loop to indicate to Julia that the loop is a multi-threaded region:\n\njulia> Threads.@threads for i = 1:10\n           a[i] = Threads.threadid()\n       end\n\nThe iteration space is split among the threads, after which each thread writes its thread ID to its assigned locations:\n\njulia> a\n10-element Vector{Float64}:\n 1.0\n 1.0\n 1.0\n 2.0\n 2.0\n 2.0\n 3.0\n 3.0\n 4.0\n 4.0\n\nNote that Threads.@threads does not have an optional reduction parameter like @distributed.","page":"Multi-Threading"},{"title":"Using @threads without data-races","location":"manual/multi-threading.html#Using-@threads-without-data-races","category":"section","text":"The concept of a data-race is elaborated on in \"Communication and data races between threads\". For now, just known that a data race can result in incorrect results and dangerous errors.\n\nLets say we want to make the function sum_single below multithreaded.\n\njulia> function sum_single(a)\n           s = 0\n           for i in a\n               s += i\n           end\n           s\n       end\nsum_single (generic function with 1 method)\n\njulia> sum_single(1:1_000_000)\n500000500000\n\nSimply adding @threads exposes a data race with multiple threads reading and writing s at the same time.\n\njulia> function sum_multi_bad(a)\n           s = 0\n           Threads.@threads for i in a\n               s += i\n           end\n           s\n       end\nsum_multi_bad (generic function with 1 method)\n\njulia> sum_multi_bad(1:1_000_000)\n70140554652\n\nNote that the result is not 500000500000 as it should be, and will most likely change each evaluation.\n\nTo fix this, buffers that are specific to the task may be used to segment the sum into chunks that are race-free. Here sum_single is reused, with its own internal buffer s. The input vector a is split into at most nthreads() chunks for parallel work. We then use Threads.@spawn to create tasks that individually sum each chunk. Finally, we sum the results from each task using sum_single again:\n\njulia> function sum_multi_good(a)\n           chunks = Iterators.partition(a, cld(length(a), Threads.nthreads()))\n           tasks = map(chunks) do chunk\n               Threads.@spawn sum_single(chunk)\n           end\n           chunk_sums = fetch.(tasks)\n           return sum_single(chunk_sums)\n       end\nsum_multi_good (generic function with 1 method)\n\njulia> sum_multi_good(1:1_000_000)\n500000500000\n\nnote: Note\nBuffers should not be managed based on threadid() i.e. buffers = zeros(Threads.nthreads()) because concurrent tasks can yield, meaning multiple concurrent tasks may use the same buffer on a given thread, introducing risk of data races. Further, when more than one thread is available tasks may change thread at yield points, which is known as task migration.\n\nAnother option is the use of atomic operations on variables shared across tasks/threads, which may be more performant depending on the characteristics of the operations.","page":"Multi-Threading"},{"title":"Communication and data-races between threads","location":"manual/multi-threading.html#man-communication-and-data-races","category":"section","text":"Although Julia's threads can communicate through shared memory, it is notoriously difficult to write correct and data-race free multi-threaded code. Julia's Channels are thread-safe and may be used to communicate safely. There are also sections below that explain how to use locks and atomics to avoid data-races.\n\nIn certain cases, Julia is able to detect a detect safety violations, in particular in regards to deadlocks or other known-unsafe operations such as yielding to the currently running task. In these cases, a ConcurrencyViolationError is thrown.","page":"Multi-Threading"},{"title":"Data-race freedom","location":"manual/multi-threading.html#Data-race-freedom","category":"section","text":"You are entirely responsible for ensuring that your program is data-race free, and nothing promised here can be assumed if you do not observe that requirement. The observed results may be highly unintuitive.\n\nIf data-races are introduced, Julia is not memory safe. Be very careful about reading any data if another thread might write to it, as it could result in segmentation faults or worse. Below are a couple of unsafe ways to access global variables from different threads:\n\nThread 1:\nglobal b = false\nglobal a = rand()\nglobal b = true\n\nThread 2:\nwhile !b; end\nbad_read1(a) # it is NOT safe to access `a` here!\n\nThread 3:\nwhile !@isdefined(a); end\nbad_read2(a) # it is NOT safe to access `a` here","page":"Multi-Threading"},{"title":"Using locks to avoid data-races","location":"manual/multi-threading.html#man-using-locks","category":"section","text":"An important tool to avoid data-races, and thereby write thread-safe code, is the concept of a \"lock\". A lock can be locked and unlocked. If a thread has locked a lock, and not unlocked it, it is said to \"hold\" the lock. If there is only one lock, and we write code the requires holding the lock to access some data, we can ensure that multiple threads will never access the same data simultaneously. Note that the link between a lock and a variable is made by the programmer, and not the program.\n\nFor example, we can create a lock my_lock, and lock it while we mutate a variable my_variable. This is done most simply with the @lock macro:\n\njulia> my_lock = ReentrantLock();\n\njulia> my_variable = [1, 2, 3];\n\njulia> @lock my_lock my_variable[1] = 100\n100\n\nBy using a similar pattern with the same lock and variable, but on another thread, the operations are free from data-races.\n\nWe could have performed the operation above with the functional version of lock, in the following two ways:\n\njulia> lock(my_lock) do\n           my_variable[1] = 100\n       end\n100\n\njulia> begin\n           lock(my_lock)\n           try\n               my_variable[1] = 100\n           finally\n               unlock(my_lock)\n           end\n       end\n100\n\nAll three options are equivalent. Note how the final version requires an explicit try-block to ensure that the lock is always unlocked, whereas the first two version do this internally. One should always use the lock pattern above when changing data (such as assigning to a global or closure variable) accessed by other threads. Failing to do this could have unforeseen and serious consequences.","page":"Multi-Threading"},{"title":"Atomic Operations","location":"manual/multi-threading.html#man-atomic-operations","category":"section","text":"Julia supports accessing and modifying values atomically, that is, in a thread-safe way to avoid race conditions. A value (which must be of a primitive type) can be wrapped as Threads.Atomic to indicate it must be accessed in this way. Here we can see an example:\n\njulia> i = Threads.Atomic{Int}(0);\n\njulia> ids = zeros(4);\n\njulia> old_is = zeros(4);\n\njulia> Threads.@threads for id in 1:4\n           old_is[id] = Threads.atomic_add!(i, id)\n           ids[id] = id\n       end\n\njulia> old_is\n4-element Vector{Float64}:\n 0.0\n 1.0\n 7.0\n 3.0\n\njulia> i[]\n 10\n\njulia> ids\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0\n\nHad we tried to do the addition without the atomic tag, we might have gotten the wrong answer due to a race condition. An example of what would happen if we didn't avoid the race:\n\njulia> using Base.Threads\n\njulia> Threads.nthreads()\n4\n\njulia> acc = Ref(0)\nBase.RefValue{Int64}(0)\n\njulia> @threads for i in 1:1000\n          acc[] += 1\n       end\n\njulia> acc[]\n926\n\njulia> acc = Atomic{Int64}(0)\nAtomic{Int64}(0)\n\njulia> @threads for i in 1:1000\n          atomic_add!(acc, 1)\n       end\n\njulia> acc[]\n1000","page":"Multi-Threading"},{"title":"Per-field atomics","location":"manual/multi-threading.html#man-atomics","category":"section","text":"We can also use atomics on a more granular level using the @atomic, @atomicswap, @atomicreplace macros, and @atomiconce macros.\n\nSpecific details of the memory model and other details of the design are written in the Julia Atomics Manifesto, which will later be published formally.\n\nAny field in a struct declaration can be decorated with @atomic, and then any write must be marked with @atomic also, and must use one of the defined atomic orderings (:monotonic, :acquire, :release, :acquire_release, or :sequentially_consistent). Any read of an atomic field can also be annotated with an atomic ordering constraint, or will be done with monotonic (relaxed) ordering if unspecified.\n\ncompat: Julia 1.7\nPer-field atomics requires at least Julia 1.7.","page":"Multi-Threading"},{"title":"Side effects and mutable function arguments","location":"manual/multi-threading.html#Side-effects-and-mutable-function-arguments","category":"section","text":"When using multi-threading we have to be careful when using functions that are not pure as we might get a wrong answer. For instance functions that have a name ending with ! by convention modify their arguments and thus are not pure.","page":"Multi-Threading"},{"title":"@threadcall","location":"manual/multi-threading.html#@threadcall","category":"section","text":"External libraries, such as those called via ccall, pose a problem for Julia's task-based I/O mechanism. If a C library performs a blocking operation, that prevents the Julia scheduler from executing any other tasks until the call returns. (Exceptions are calls into custom C code that call back into Julia, which may then yield, or C code that calls jl_yield(), the C equivalent of yield.)\n\nThe @threadcall macro provides a way to avoid stalling execution in such a scenario. It schedules a C function for execution in a separate thread. A threadpool with a default size of 4 is used for this. The size of the threadpool is controlled via environment variable UV_THREADPOOL_SIZE. While waiting for a free thread, and during function execution once a thread is available, the requesting task (on the main Julia event loop) yields to other tasks. Note that @threadcall does not return until the execution is complete. From a user point of view, it is therefore a blocking call like other Julia APIs.\n\nIt is very important that the called function does not call back into Julia, as it will segfault.\n\n@threadcall may be removed/changed in future versions of Julia.","page":"Multi-Threading"},{"title":"Caveats","location":"manual/multi-threading.html#Caveats","category":"section","text":"At this time, most operations in the Julia runtime and standard libraries can be used in a thread-safe manner, if the user code is data-race free. However, in some areas work on stabilizing thread support is ongoing. Multi-threaded programming has many inherent difficulties, and if a program using threads exhibits unusual or undesirable behavior (e.g. crashes or mysterious results), thread interactions should typically be suspected first.\n\nThere are a few specific limitations and warnings to be aware of when using threads in Julia:\n\nBase collection types require manual locking if used simultaneously by multiple threads where at least one thread modifies the collection (common examples include push! on arrays, or inserting items into a Dict).\nThe schedule used by @spawn is nondeterministic and should not be relied on.\nCompute-bound, non-memory-allocating tasks can prevent garbage collection from running in other threads that are allocating memory. In these cases it may be necessary to insert a manual call to GC.safepoint() to allow GC to run. This limitation will be removed in the future.\nAvoid running top-level operations, e.g. include, or eval of type, method, and module definitions in parallel.\nBe aware that finalizers registered by a library may break if threads are enabled. This may require some transitional work across the ecosystem before threading can be widely adopted with confidence. See the section on the safe use of finalizers for further details.","page":"Multi-Threading"},{"title":"Task Migration","location":"manual/multi-threading.html#man-task-migration","category":"section","text":"After a task starts running on a certain thread it may move to a different thread if the task yields.\n\nSuch tasks may have been started with @spawn or @threads, although the :static schedule option for @threads does freeze the threadid.\n\nThis means that in most cases threadid() should not be treated as constant within a task, and therefore should not be used to index into a vector of buffers or stateful objects.\n\ncompat: Julia 1.7\nTask migration was introduced in Julia 1.7. Before this tasks always remained on the same thread that they were started on.","page":"Multi-Threading"},{"title":"Safe use of Finalizers","location":"manual/multi-threading.html#man-finalizers","category":"section","text":"Because finalizers can interrupt any code, they must be very careful in how they interact with any global state. Unfortunately, the main reason that finalizers are used is to update global state (a pure function is generally rather pointless as a finalizer). This leads us to a bit of a conundrum. There are a few approaches to dealing with this problem:\n\nWhen single-threaded, code could call the internal jl_gc_enable_finalizers C function to prevent finalizers from being scheduled inside a critical region. Internally, this is used inside some functions (such as our C locks) to prevent recursion when doing certain operations (incremental package loading, codegen, etc.). The combination of a lock and this flag can be used to make finalizers safe.\nA second strategy, employed by Base in a couple places, is to explicitly delay a finalizer until it may be able to acquire its lock non-recursively. The following example demonstrates how this strategy could be applied to Distributed.finalize_ref:\nfunction finalize_ref(r::AbstractRemoteRef)\n    if r.where > 0 # Check if the finalizer is already run\n        if islocked(client_refs) || !trylock(client_refs)\n            # delay finalizer for later if we aren't free to acquire the lock\n            finalizer(finalize_ref, r)\n            return nothing\n        end\n        try # `lock` should always be followed by `try`\n            if r.where > 0 # Must check again here\n                # Do actual cleanup here\n                r.where = 0\n            end\n        finally\n            unlock(client_refs)\n        end\n    end\n    nothing\nend\nA related third strategy is to use a yield-free queue. We don't currently have a lock-free queue implemented in Base, but Base.IntrusiveLinkedListSynchronized{T} is suitable. This can frequently be a good strategy to use for code with event loops. For example, this strategy is employed by Gtk.jl to manage lifetime ref-counting. In this approach, we don't do any explicit work inside the finalizer, and instead add it to a queue to run at a safer time. In fact, Julia's task scheduler already uses this, so defining the finalizer as x -> @spawn do_cleanup(x) is one example of this approach. Note however that this doesn't control which thread do_cleanup runs on, so do_cleanup would still need to acquire a lock. That doesn't need to be true if you implement your own queue, as you can explicitly only drain that queue from your thread.","page":"Multi-Threading"},{"title":"Static analyzer annotations for GC correctness in C code","location":"devdocs/gc-sa.html#Static-analyzer-annotations-for-GC-correctness-in-C-code","category":"section","text":"","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Running the analysis","location":"devdocs/gc-sa.html#Running-the-analysis","category":"section","text":"The analyzer plugin that drives the analysis ships with julia. Its source code can be found in src/clangsa. Running it requires the clang dependency to be build. Set the BUILD_LLVM_CLANG variable in your Make.user in order to build an appropriate version of clang. You may also want to use the prebuilt binaries using the USE_BINARYBUILDER_LLVM options.\n\nAlternatively (or if these do not suffice), try\n\nmake -C src install-analysis-deps\n\nfrom Julia's toplevel directory.\n\nAfterwards, running the analysis over the source tree is as simple as running make -C src analyzegc.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"General Overview","location":"devdocs/gc-sa.html#General-Overview","category":"section","text":"Since Julia's GC is precise, it needs to maintain correct rooting information for any value that may be referenced at any time GC may occur. These places are known as safepoints and in the function local context, we extend this designation to any function call that may recursively end up at a safepoint.\n\nIn generated code, this is taken care of automatically by the GC root placement pass (see the chapter on GC rooting in the LLVM codegen devdocs). However, in C code, we need to inform the runtime of any GC roots manually. This is done using the following macros:\n\n// The value assigned to any slot passed as an argument to these\n// is rooted for the duration of this GC frame.\nJL_GC_PUSH{1,...,6}(args...)\n// The values assigned into the size `n` array `rts` are rooted\n// for the duration of this GC frame.\nJL_GC_PUSHARGS(rts, n)\n// Pop a GC frame\nJL_GC_POP\n\nIf these macros are not used where they need to be, or they are used incorrectly, the result is silent memory corruption. As such it is very important that they are placed correctly in all applicable code.\n\nAs such, we employ static analysis (and in particular the clang static analyzer) to help ensure that these macros are used correctly. The remainder of this document gives an overview of this static analysis and describes the support needed in the julia code base to make things work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"GC Invariants","location":"devdocs/gc-sa.html#GC-Invariants","category":"section","text":"There is two simple invariants correctness:\n\nAll GC_PUSH calls need to be followed by an appropriate GC_POP (in practice we enforce this at the function level)\nIf a value was previously not rooted at any safepoint, it may no longer be referenced afterwards\n\nOf course the devil is in the details here. In particular to satisfy the second of the above conditions, we need to know:\n\nWhich calls are safepoints and which are not\nWhich values are rooted at any given safepoint and which are not\nWhen is a value referenced\n\nFor the second point in particular, we need to know which memory locations will be considered rooting at runtime (i.e. values assigned to such locations are rooted). This includes locations explicitly designated as such by passing them to one of the GC_PUSH macros, globally rooted locations and values, as well as any location recursively reachable from one of those locations.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Static Analysis Algorithm","location":"devdocs/gc-sa.html#Static-Analysis-Algorithm","category":"section","text":"The idea itself is very simple, although the implementation is quite a bit more complicated (mainly due to a large number of special cases and intricacies of C and C++). In essence, we keep track of all locations that are rooting, all values that are rootable and any expression (assignments, allocations, etc) affect the rootedness of any rootable values. Then, at any safepoint, we perform a \"symbolic GC\" and poison any values that are not rooted at said location. If these values are later referenced, we emit an error.\n\nThe clang static analyzer works by constructing a graph of states and exploring this graph for sources of errors. Several nodes in this graph are generated by the analyzer itself (e.g. for control flow), but the definitions above augment this graph with our own state.\n\nThe static analyzer is interprocedural and can analyze control flow across function boundaries. However, the static analyzer is not fully recursive and makes heuristic decisions about which calls to explore (additionally some calls are cross-translation unit and invisible to the analyzer). In our case, our definition of correctness requires total information. As such, we need to annotate the prototypes of all function calls with whatever information the analysis required, even if that information would otherwise be available by interprocedural static analysis.\n\nLuckily however, we can still use this interprocedural analysis to ensure that the annotations we place on a given function are indeed correct given the implementation of said function.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"The analyzer annotations","location":"devdocs/gc-sa.html#The-analyzer-annotations","category":"section","text":"These annotations are found in src/support/analyzer_annotations.h. The are only active when the analyzer is being used and expand either to nothing (for prototype annotations) or to no-ops (for function like annotations).","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_NOTSAFEPOINT","location":"devdocs/gc-sa.html#JL_NOTSAFEPOINT","category":"section","text":"This is perhaps the most common annotation, and should be placed on any function that is known not to possibly lead to reaching a GC safepoint. In general, it is only safe for such a function to perform arithmetic, memory accesses and calls to functions either annotated JL_NOTSAFEPOINT or otherwise known not to be safepoints (e.g. function in the C standard library, which are hardcoded as such in the analyzer)\n\nIt is valid to keep values unrooted across calls to any function annotated with this attribute:\n\nUsage Example:\n\nvoid jl_get_one() JL_NOTSAFEPOINT {\n  return 1;\n}\n\njl_value_t *example() {\n  jl_value_t *val = jl_alloc_whatever();\n  // This is valid, even though `val` is unrooted, because\n  // jl_get_one is not a safepoint\n  jl_get_one();\n  return val;\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","location":"devdocs/gc-sa.html#JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY","category":"section","text":"When JL_MAYBE_UNROOTED is annotated as an argument on a function, indicates that said argument may be passed, even if it is not rooted. In the ordinary course of events, the julia ABI guarantees that callers root values before passing them to callees. However, some functions do not follow this ABI and allow values to be passed to them even though they are not rooted. Note however, that this does not automatically imply that said argument will be preserved. The ROOTS_TEMPORARILY annotation provides the stronger guarantee that, not only may the value be unrooted when passed, it will also be preserved across any internal safepoints by the callee.\n\nNote that JL_NOTSAFEPOINT essentially implies JL_MAYBE_UNROOTED/JL_ROOTS_TEMPORARILY, because the rootedness of an argument is irrelevant if the function contains no safepoints.\n\nOne additional point to note is that these annotations apply on both the caller and the callee side. On the caller side, they lift rootedness restrictions that are normally required for julia ABI functions. On the callee side, they have the reverse effect of preventing these arguments from being considered implicitly rooted.\n\nIf either of these annotations is applied to the function as a whole, it applies to all arguments of the function. This should generally only be necessary for varargs functions.\n\nUsage example:\n\nJL_DLLEXPORT void JL_NORETURN jl_throw(jl_value_t *e JL_MAYBE_UNROOTED);\njl_value_t *jl_alloc_error();\n\nvoid example() {\n  // The return value of the allocation is unrooted. This would normally\n  // be an error, but is allowed because of the above annotation.\n  jl_throw(jl_alloc_error());\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_PROPAGATES_ROOT","location":"devdocs/gc-sa.html#JL_PROPAGATES_ROOT","category":"section","text":"This annotation is commonly found on accessor functions that return one rootable object stored within another. When annotated on a function argument, it tells the analyzer that the root for that argument also applies to the value returned by the function.\n\nUsage Example:\n\njl_value_t *jl_svecref(jl_svec_t *t JL_PROPAGATES_ROOT, size_t i) JL_NOTSAFEPOINT;\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_svecref(svec, 1)\n  // This is valid, because, as annotated by the PROPAGATES_ROOT annotation,\n  // jl_svecref propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","location":"devdocs/gc-sa.html#JL_ROOTING_ARGUMENT/JL_ROOTED_ARGUMENT","category":"section","text":"This is essentially the assignment counterpart to JL_PROPAGATES_ROOT. When assigning a value to a field of another value that is already rooted, the assigned value will inherit the root of the value it is assigned into.\n\nUsage Example:\n\nvoid jl_svecset(void *t JL_ROOTING_ARGUMENT, size_t i, void *x JL_ROOTED_ARGUMENT) JL_NOTSAFEPOINT\n\n\nsize_t example(jl_svec_t *svec) {\n  jl_value_t *val = jl_box_long(10000);\n  jl_svecset(svec, val);\n  // This is valid, because the annotations imply that the\n  // jl_svecset propagates the rooted-ness from `svec` to `val`\n  jl_gc_safepoint();\n  return jl_unbox_long(val);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_DISABLED","location":"devdocs/gc-sa.html#JL_GC_DISABLED","category":"section","text":"This annotation implies that this function is only called with the GC runtime-disabled. Functions of this kind are most often encountered during startup and in the GC code itself. Note that this annotation is checked against the runtime enable/disable calls, so clang will know if you lie. This is not a good way to disable processing of a given function if the GC is not actually disabled (use ifdef __clang_analyzer__ for that if you must).\n\nUsage example:\n\nvoid jl_do_magic() JL_GC_DISABLED {\n  // Wildly allocate here with no regard for roots\n}\n\nvoid example() {\n  int en = jl_gc_enable(0);\n  jl_do_magic();\n  jl_gc_enable(en);\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_REQUIRE_ROOTED_SLOT","location":"devdocs/gc-sa.html#JL_REQUIRE_ROOTED_SLOT","category":"section","text":"This annotation requires the caller to pass in a slot that is rooted (i.e. values assigned to this slot will be rooted).\n\nUsage example:\n\nvoid jl_do_processing(jl_value_t **slot JL_REQUIRE_ROOTED_SLOT) {\n  *slot = jl_box_long(1);\n  // Ok, only, because the slot was annotated as rooting\n  jl_gc_safepoint();\n}\n\nvoid example() {\n  jl_value_t *slot = NULL;\n  JL_GC_PUSH1(&slot);\n  jl_do_processing(&slot);\n  JL_GC_POP();\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GLOBALLY_ROOTED","location":"devdocs/gc-sa.html#JL_GLOBALLY_ROOTED","category":"section","text":"This annotation implies that a given value is always globally rooted. It can be applied to global variable declarations, in which case it will apply to the value of those variables (or values if the declaration if for an array), or to functions, in which case it will apply to the return value of such functions (e.g. for functions that always return some private, globally rooted value).\n\nUsage example:\n\nextern JL_DLLEXPORT jl_datatype_t *jl_any_type JL_GLOBALLY_ROOTED;\njl_ast_context_t *jl_ast_ctx(fl_context_t *fl) JL_GLOBALLY_ROOTED;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_ALWAYS_LEAFTYPE","location":"devdocs/gc-sa.html#JL_ALWAYS_LEAFTYPE","category":"section","text":"This annotations is essentially equivalent to JL_GLOBALLY_ROOTED, except that is should only be used if those values are globally rooted by virtue of being a leaftype. The rooting of leaftypes is a bit complicated. They are generally rooted through cache field of the corresponding TypeName, which itself is rooted by the containing module (so they're rooted as long as the containing module is ok) and we can generally assume that leaftypes are rooted where they are used, but we may refine this property in the future, so the separate annotation helps split out the reason for being globally rooted.\n\nThe analyzer also automatically detects checks for leaftype-ness and will not complain about missing GC roots on these paths.\n\nJL_DLLEXPORT jl_value_t *jl_apply_array_type(jl_value_t *type, size_t dim) JL_ALWAYS_LEAFTYPE;","page":"Static analyzer annotations for GC correctness in C code"},{"title":"JL_GC_PROMISE_ROOTED","location":"devdocs/gc-sa.html#JL_GC_PROMISE_ROOTED","category":"section","text":"This is a function-like annotation. Any value passed to this annotation will be considered rooted for the scope of the current function. It is designed as an escape hatch for analyzer inadequacy or complicated situations. However, it should be used sparingly, in favor of improving the analyzer itself.\n\nvoid example() {\n  jl_value_t *val = jl_alloc_something();\n  if (some_condition) {\n    // We happen to know for complicated external reasons\n    // that val is rooted under these conditions\n    JL_GC_PROMISE_ROOTED(val);\n  }\n}","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Completeness of analysis","location":"devdocs/gc-sa.html#Completeness-of-analysis","category":"section","text":"The analyzer only looks at local information. In particular, e.g. in the PROPAGATES_ROOT case above, it assumes that such memory is only modified in ways it can see, not in any called functions (unless it happens to decide to consider them in its analysis) and not in any concurrently running threads. As such, it may miss a few problematic cases, though in practice such concurrent modification is fairly rare. Improving the analyzer to handle more such cases may be an interesting topic for future work.","page":"Static analyzer annotations for GC correctness in C code"},{"title":"Iteration utilities","location":"base/iterators.html#Iteration-utilities","category":"section","text":"","page":"Iteration utilities"},{"title":"Base.Iterators.Stateful","location":"base/iterators.html#Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.zip","location":"base/iterators.html#Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.enumerate","location":"base/iterators.html#Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a different element. This will happen if iter has indices that do not start at 1, and may happen for strings, dictionaries, etc. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.rest","location":"base/iterators.html#Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state.\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> collect(Iterators.rest([1,2,3,4], 2))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.countfrom","location":"base/iterators.html#Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.take","location":"base/iterators.html#Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.takewhile","location":"base/iterators.html#Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.drop","location":"base/iterators.html#Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.dropwhile","location":"base/iterators.html#Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.cycle","location":"base/iterators.html#Base.Iterators.cycle","category":"function","text":"cycle(iter[, n::Int])\n\nAn iterator that cycles through iter forever. If n is specified, then it cycles through iter that many times. When iter is empty, so are cycle(iter) and cycle(iter, n).\n\nIterators.cycle(iter, n) is the lazy equivalent of Base.repeat(vector, n), while Iterators.repeated(iter, n) is the lazy Base.fill(item, n).\n\ncompat: Julia 1.11\nThe method cycle(iter, n) was added in Julia 1.11.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\njulia> foreach(print, Iterators.cycle(['j', 'u', 'l', 'i', 'a'], 3))\njuliajuliajulia\n\njulia> repeat([1,2,3], 4) == collect(Iterators.cycle([1,2,3], 4))\ntrue\n\njulia> fill([1,2,3], 4) == collect(Iterators.repeated([1,2,3], 4))\ntrue\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.repeated","location":"base/iterators.html#Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nSee also fill, and compare Iterators.cycle.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\njulia> ans == fill([1 2], 4)\ntrue\n\njulia> Iterators.cycle([1 2], 4) |> collect |> println\n[1, 2, 1, 2, 1, 2, 1, 2]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.product","location":"base/iterators.html#Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.flatten","location":"base/iterators.html#Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.flatmap","location":"base/iterators.html#Base.Iterators.flatmap","category":"function","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.partition","location":"base/iterators.html#Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.map","location":"base/iterators.html#Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.filter","location":"base/iterators.html#Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nwarning: Warning\nSubsequent lazy transformations on the iterator returned from filter, such as those performed by Iterators.reverse or cycle, will also delay calls to flt until collecting or iterating over the returned iterable object. If the filter predicate is nondeterministic or its return values depend on the order of iteration over the elements of itr, composition with lazy transformations may result in surprising behavior. If this is undesirable, either ensure that flt is a pure function or collect intermediate filter iterators before further transformations.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.accumulate","location":"base/iterators.html#Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.reverse","location":"base/iterators.html#Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order. This iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns an Iterators.Reverse object wrapping itr, which is iterable if the corresponding iterate methods are defined, but some itr types may implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr; more generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.only","location":"base/iterators.html#Base.Iterators.only","category":"function","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the collection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Base.Iterators.peel","location":"base/iterators.html#Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n","page":"Iteration utilities"},{"title":"Garbage Collection in Julia","location":"devdocs/gc.html#Garbage-Collection-in-Julia","category":"section","text":"","page":"Garbage Collection in Julia"},{"title":"Introduction","location":"devdocs/gc.html#Introduction","category":"section","text":"Julia has a non-moving, partially concurrent, parallel, generational and mostly precise mark-sweep collector (an interface for conservative stack scanning is provided as an option for users who wish to call Julia from C).","page":"Garbage Collection in Julia"},{"title":"Allocation","location":"devdocs/gc.html#Allocation","category":"section","text":"Julia uses two types of allocators, the size of the allocation request determining which one is used. Objects up to 2k bytes are allocated on a per-thread free-list pool allocator, while objects larger than 2k bytes are allocated through libc malloc.\n\nJulia’s pool allocator partitions objects on different size classes, so that a memory page managed by the pool allocator (which spans 4 operating system pages on 64bit platforms) only contains objects of the same size class. Each memory page from the pool allocator is paired with some page metadata stored on per-thread lock-free lists. The page metadata contains information such as whether the page has live objects at all, number of free slots, and offsets to the first and last objects in the free-list contained in that page. These metadata are used to optimize the collection phase: a page which has no live objects at all may be returned to the operating system without any need of scanning it, for example.\n\nWhile a page that has no objects may be returned to the operating system, its associated metadata is permanently allocated and may outlive the given page. As mentioned above, metadata for allocated pages are stored on per-thread lock-free lists. Metadata for free pages, however, may be stored into three separate lock-free lists depending on whether the page has been mapped but never accessed (page_pool_clean), or whether the page has been lazily sweeped and it's waiting to be madvised by a background GC thread (page_pool_lazily_freed), or whether the page has been madvised (page_pool_freed).\n\nJulia's pool allocator follows a \"tiered\" allocation discipline. When requesting a memory page for the pool allocator, Julia will:\n\nTry to claim a page from page_pool_lazily_freed, which contains pages which were empty on the last stop-the-world phase, but not yet madvised by a concurrent sweeper GC thread.\nIf it failed claiming a page from page_pool_lazily_freed, it will try to claim a page from the page_pool_clean, which contains pages which were mmaped on a previous page allocation request but never accessed.\nIf it failed claiming a page from pool_page_clean and from page_pool_lazily_freed, it will try to claim a page from page_pool_freed, which contains pages which have already been madvised by a concurrent sweeper GC thread and whose underlying virtual address can be recycled.\nIf it failed in all of the attempts mentioned above, it will mmap a batch of pages, claim one page for itself, and insert the remaining pages into page_pool_clean.\n\n(Image: Diagram of tiered pool allocation)","page":"Garbage Collection in Julia"},{"title":"Marking and Generational Collection","location":"devdocs/gc.html#Marking-and-Generational-Collection","category":"section","text":"Julia’s mark phase is implemented through a parallel iterative depth-first-search over the object graph. Julia’s collector is non-moving, so object age information can’t be determined through the memory region in which the object resides alone, but has to be somehow encoded in the object header or on a side table. The lowest two bits of an object’s header are used to store, respectively, a mark bit that is set when an object is scanned during the mark phase and an age bit for the generational collection.\n\nGenerational collection is implemented through sticky bits: objects are only pushed to the mark-stack, and therefore traced, if their mark-bits are not set. When objects reach the oldest generation, their mark-bits are not reset during the so-called \"quick-sweep\", which leads to these objects not being traced in a subsequent mark phase. A \"full-sweep\", however, causes the mark-bits of all objects to be reset, leading to all objects being traced in a subsequent mark phase. Objects are promoted to the next generation during every sweep phase they survive. On the mutator side, field writes are intercepted through a write barrier that pushes an object’s address into a per-thread remembered set if the object is in the last generation, and if the object at the field being written is not. Objects in this remembered set are then traced during the mark phase.","page":"Garbage Collection in Julia"},{"title":"Sweeping","location":"devdocs/gc.html#Sweeping","category":"section","text":"Sweeping of object pools for Julia may fall into two categories: if a given page managed by the pool allocator contains at least one live object, then a free-list must be threaded through its dead objects; if a given page contains no live objects at all, then its underlying physical memory may be returned to the operating system through, for instance, the use of madvise system calls on Linux.\n\nThe first category of sweeping is parallelized through work-stealing. For the second category of sweeping, if concurrent page sweeping is enabled through the flag --gcthreads=X,1 we perform the madvise system calls in a background sweeper thread, concurrently with the mutator threads. During the stop-the-world phase of the collector, pool allocated pages which contain no live objects are initially pushed into the pool_page_lazily_freed. The background sweeping thread is then woken up and is responsible for removing pages from pool_page_lazily_freed, calling madvise on them, and inserting them into pool_page_freed. As described above, pool_page_lazily_freed is also shared with mutator threads. This implies that on allocation-heavy multithreaded workloads, mutator threads would often avoid a page fault on allocation (coming from accessing a fresh mmaped page or accessing a madvised page) by directly allocating from a page in pool_page_lazily_freed, while the background sweeper thread needs to madvise a reduce number of pages given some of them were already claimed by the mutators.","page":"Garbage Collection in Julia"},{"title":"Heuristics","location":"devdocs/gc.html#Heuristics","category":"section","text":"GC heuristics tune the GC by changing the size of the allocation interval between garbage collections.\n\nThe GC heuristics measure how big the heap size is after a collection and set the next collection according to the algorithm described by https://dl.acm.org/doi/10.1145/3563323, in summary, it argues that the heap target should have a square root relationship with the live heap, and that it should also be scaled by how fast the GC is freeing objects and how fast the mutators are allocating. The heuristics measure the heap size by counting the number of pages that are in use and the objects that use malloc. Previously we measured the heap size by counting the alive objects, but that doesn't take into account fragmentation which could lead to bad decisions, that also meant that we used thread local information (allocations) to make decisions about a process wide (when to GC), measuring pages means the decision is global.\n\nThe GC will do full collections when the heap size reaches 80% of the maximum allowed size.","page":"Garbage Collection in Julia"},{"title":"RISC-V (Linux)","location":"devdocs/build/riscv.html#RISC-V-(Linux)","category":"section","text":"Julia has experimental support for 64-bit RISC-V (RV64) processors running Linux. This file provides general guidelines for compilation, in addition to instructions for specific devices.\n\nA list of known issues for RISC-V is available. If you encounter difficulties, please create an issue including the output from cat /proc/cpuinfo.","page":"RISC-V (Linux)"},{"title":"Compiling Julia","location":"devdocs/build/riscv.html#Compiling-Julia","category":"section","text":"For now, Julia will need to be compiled entirely from source, i.e., including all of its dependencies. This can be accomplished with the following Make.user:\n\nUSE_BINARYBUILDER := 0\n\nAdditionally, it is required to indicate what architecture, and optionally which CPU to build for. This can be done by setting the MARCH and MCPU variables in Make.user\n\nThe MARCH variable needs to be set to a RISC-V ISA string, which can be found by looking at the documentation of your device, or by inspecting /proc/cpuinfo. Only use flags that your compiler supports, e.g., run gcc -march=help to see a list of supported flags. A common value is rv64gc, which is a good starting point.\n\nThe MCPU variable is optional, and can be used to further optimize the generated code for a specific CPU. If you are unsure, it is recommended to leave it unset. You can find a list of supported values by running gcc --target-help.\n\nFor example, if you are using a StarFive VisionFive2, which contains a JH7110 processor based on the SiFive U74, you can set these flags as follows:\n\nMARCH := rv64gc_zba_zbb\nMCPU := sifive-u74\n\nIf you prefer a portable build, you could use:\n\nMARCH := rv64gc\n\n# also set JULIA_CPU_TARGET to the expanded form of rv64gc\n# (it normally copies the value of MCPU, which we don't set)\nJULIA_CPU_TARGET := generic-rv64,i,m,a,f,d,zicsr,zifencei,c","page":"RISC-V (Linux)"},{"title":"Cross-compilation","location":"devdocs/build/riscv.html#Cross-compilation","category":"section","text":"A native build on a RISC-V device may take a very long time, so it's also possible to cross-compile Julia on a faster machine.\n\nFirst, get a hold of a RISC-V cross-compilation toolchain that provides support for C, C++ and Fortran. This can be done by checking-out the riscv-gnu-toolchain repository and building it as follows:\n\nsudo mkdir /opt/riscv && sudo chown $USER /opt/riscv\n./configure --prefix=/opt/riscv --with-languages=c,c++,fortran\nmake linux -j$(nproc)\n\nThen, install the QEMU user-mode emulator for RISC-V, along with binfmt support to enable execution of RISC-V binaries on the host machine. The exact steps depend on your distribution, e.g., on Arch Linux it involves installing the qemu-user-static and qemu-user-static-binfmt packages. Note that to actually execute RISC-V binaries, QEMU will need to be able to find the RISC-V system root, which can be achieved by setting the QEMU_LD_PREFIX environment variable to the path of the root filesystem.\n\nFinally, compile Julia with the following Make.user variables (in addition to the ones from the previous section):\n\nXC_HOST=riscv64-unknown-linux-gnu\nOS=Linux\nexport QEMU_LD_PREFIX=/opt/riscv/sysroot\n\nNote that you will have to execute make with PATH set to include the cross-compilation toolchain, e.g., by running:\n\nPATH=/opt/riscv/bin:$PATH make -j$(nproc)\n\nBecause of the RISC-V sysroot we use being very barren, you may need to add additional libraries that the Julia build system currently expects to be available system-wide. For example, the build currently relies on a system-provided libz, so you may need to copy this library from the Julia build into the system root:\n\nmake -C deps install-zlib\ncp -v usr/lib/libz.*   /opt/riscv/sysroot/usr/lib\ncp -v usr/include/z*.h /opt/riscv/sysroot/usr/include","page":"RISC-V (Linux)"},{"title":"C Standard Library","location":"base/libc.html#C-Standard-Library","category":"section","text":"","page":"C Standard Library"},{"title":"Base.Libc.malloc","location":"base/libc.html#Base.Libc.malloc","category":"function","text":"malloc(size::Integer) -> Ptr{Cvoid}\n\nCall malloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.calloc","location":"base/libc.html#Base.Libc.calloc","category":"function","text":"calloc(num::Integer, size::Integer) -> Ptr{Cvoid}\n\nCall calloc from the C standard library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.realloc","location":"base/libc.html#Base.Libc.realloc","category":"function","text":"realloc(addr::Ptr, size::Integer) -> Ptr{Cvoid}\n\nCall realloc from the C standard library.\n\nSee warning in the documentation for free regarding only using this on memory originally obtained from malloc.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.memcpy","location":"base/libc.html#Base.memcpy","category":"function","text":"memcpy(dst::Ptr, src::Ptr, n::Integer) -> Ptr{Cvoid}\n\nCall memcpy from the C standard library.\n\ncompat: Julia 1.10\nSupport for memcpy requires at least Julia 1.10.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.memmove","location":"base/libc.html#Base.memmove","category":"function","text":"memmove(dst::Ptr, src::Ptr, n::Integer) -> Ptr{Cvoid}\n\nCall memmove from the C standard library.\n\ncompat: Julia 1.10\nSupport for memmove requires at least Julia 1.10.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.memset","location":"base/libc.html#Base.memset","category":"function","text":"memset(dst::Ptr, val, n::Integer) -> Ptr{Cvoid}\n\nCall memset from the C standard library.\n\ncompat: Julia 1.10\nSupport for memset requires at least Julia 1.10.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.memcmp","location":"base/libc.html#Base.memcmp","category":"function","text":"memcmp(a::Ptr, b::Ptr, n::Integer) -> Int\n\nCall memcmp from the C standard library.\n\ncompat: Julia 1.10\nSupport for memcmp requires at least Julia 1.9.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.free","location":"base/libc.html#Base.Libc.free","category":"function","text":"free(addr::Ptr)\n\nCall free from the C standard library. Only use this on memory obtained from malloc, not on pointers retrieved from other C libraries. Ptr objects obtained from C libraries should be freed by the free functions defined in that library, to avoid assertion failures if multiple libc libraries exist on the system.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.errno","location":"base/libc.html#Base.Libc.errno","category":"function","text":"errno([code])\n\nGet the value of the C library's errno. If an argument is specified, it is used to set the value of errno.\n\nThe value of errno is only valid immediately after a ccall to a C library routine that sets it. Specifically, you cannot call errno at the next prompt in a REPL, because lots of code is executed between prompts.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strerror","location":"base/libc.html#Base.Libc.strerror","category":"function","text":"strerror(n=errno())\n\nConvert a system call error code to a descriptive string\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.GetLastError","location":"base/libc.html#Base.Libc.GetLastError","category":"function","text":"GetLastError()\n\nCall the Win32 GetLastError function [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.FormatMessage","location":"base/libc.html#Base.Libc.FormatMessage","category":"function","text":"FormatMessage(n=GetLastError())\n\nConvert a Win32 system call error code to a descriptive string [only available on Windows].\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.time","location":"base/libc.html#Base.Libc.time-Tuple{Base.Libc.TmStruct}","category":"method","text":"time(t::TmStruct) -> Float64\n\nConverts a TmStruct struct to a number of seconds since the epoch.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strftime","location":"base/libc.html#Base.Libc.strftime","category":"function","text":"strftime([format], time)\n\nConvert time, given as a number of seconds since the epoch or a TmStruct, to a formatted string using the given format. Supported formats are the same as those in the standard C library.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.strptime","location":"base/libc.html#Base.Libc.strptime","category":"function","text":"strptime([format], timestr)\n\nParse a formatted time string into a TmStruct giving the seconds, minute, hour, date, etc. Supported formats are the same as those in the standard C library. On some platforms, timezones will not be parsed correctly. If the result of this function will be passed to time to convert it to seconds since the epoch, the isdst field should be filled in manually. Setting it to -1 will tell the C library to use the current system settings to determine the timezone.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.TmStruct","location":"base/libc.html#Base.Libc.TmStruct","category":"type","text":"TmStruct([seconds])\n\nConvert a number of seconds since the epoch to broken-down format, with fields sec, min, hour, mday, month, year, wday, yday, and isdst.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.FILE","location":"base/libc.html#Base.Libc.FILE","category":"type","text":"FILE(::Ptr)\nFILE(::IO)\n\nA libc FILE*, representing an opened file.\n\nIt can be passed as a Ptr{FILE} argument to ccall and also supports seek, position and close.\n\nA FILE can be constructed from an ordinary IO object, provided it is an open file. It must be closed afterward.\n\nExamples\n\njulia> using Base.Libc\n\njulia> mktemp() do _, io\n           # write to the temporary file using `puts(char*, FILE*)` from libc\n           file = FILE(io)\n           ccall(:fputs, Cint, (Cstring, Ptr{FILE}), \"hello world\", file)\n           close(file)\n           # read the file again\n           seek(io, 0)\n           read(io, String)\n       end\n\"hello world\"\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.dup","location":"base/libc.html#Base.Libc.dup","category":"function","text":"dup(src::RawFD[, target::RawFD])::RawFD\n\nDuplicate the file descriptor src so that the duplicate refers to the same OS resource (e.g. a file or socket). A target file descriptor may be optionally be passed to use for the new duplicate.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.flush_cstdio","location":"base/libc.html#Base.Libc.flush_cstdio","category":"function","text":"flush_cstdio()\n\nFlushes the C stdout and stderr streams (which may have been written to by external C code).\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.systemsleep","location":"base/libc.html#Base.Libc.systemsleep","category":"function","text":"systemsleep(s::Real)\n\nSuspends execution for s seconds. This function does not yield to Julia's scheduler and therefore blocks the Julia thread that it is running on for the duration of the sleep time.\n\nSee also sleep.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Base.Libc.mkfifo","location":"base/libc.html#Base.Libc.mkfifo","category":"function","text":"mkfifo(path::AbstractString, [mode::Integer]) -> path\n\nMake a FIFO special file (a named pipe) at path.  Return path as-is on success.\n\nmkfifo is supported only in Unix platforms.\n\ncompat: Julia 1.11\nmkfifo requires at least Julia 1.11.\n\n\n\n\n\n","page":"C Standard Library"},{"title":"Pkg","location":"stdlib/Pkg.html#Pkg","category":"section","text":"Pkg is Julia's builtin package manager, and handles operations such as installing, updating and removing packages.\n\nnote: Note\nWhat follows is a very brief introduction to Pkg. For more information on Project.toml files, Manifest.toml files, package version compatibility ([compat]), environments, registries, etc., it is highly recommended to read the full manual, which is available here: https://pkgdocs.julialang.org.\n\nimport Markdown\nfile = joinpath(Sys.STDLIB, \"Pkg\", \"docs\", \"src\", \"getting-started.md\")\nstr = read(file, String)\nstr = replace(str, r\"^#.*$\"m => \"\")\nstr = replace(str, \"[API Reference](@ref)\" => \"[API Reference](https://pkgdocs.julialang.org/v1/api/)\")\nstr = replace(str, \"(@ref Working-with-Environments)\" => \"(https://pkgdocs.julialang.org/v1/environments/)\")\nstr = replace(str, \"(@ref Managing-Packages)\" => \"(https://pkgdocs.julialang.org/v1/managing-packages/)\")\nMarkdown.parse(str)","page":"Pkg"},{"title":"Random Numbers","location":"stdlib/Random.html#Random-Numbers","category":"section","text":"Random number generation in Julia uses the Xoshiro256++ algorithm by default, with per-Task state. Other RNG types can be plugged in by inheriting the AbstractRNG type; they can then be used to obtain multiple streams of random numbers.\n\nThe PRNGs (pseudorandom number generators) exported by the Random package are:\n\nTaskLocalRNG: a token that represents use of the currently active Task-local stream, deterministically seeded from the parent task, or by RandomDevice (with system randomness) at program start\nXoshiro: generates a high-quality stream of random numbers with a small state vector and high performance using the Xoshiro256++ algorithm\nRandomDevice: for OS-provided entropy. This may be used for cryptographically secure random numbers (CS(P)RNG).\nMersenneTwister: an alternate high-quality PRNG which was the default in older versions of Julia, and is also quite fast, but requires much more space to store the state vector and generate a random sequence.\n\nMost functions related to random generation accept an optional AbstractRNG object as first argument. Some also accept dimension specifications dims... (which can also be given as a tuple) to generate arrays of random values. In a multi-threaded program, you should generally use different RNG objects from different threads or tasks in order to be thread-safe. However, the default RNG is thread-safe as of Julia 1.3 (using a per-thread RNG up to version 1.6, and per-task thereafter).\n\nThe provided RNGs can generate uniform random numbers of the following types: Float16, Float32, Float64, BigFloat, Bool, Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, BigInt (or complex numbers of those types). Random floating point numbers are generated uniformly in 0 1). As BigInt represents unbounded integers, the interval must be specified (e.g. rand(big.(1:6))).\n\nAdditionally, normal and exponential distributions are implemented for some AbstractFloat and Complex types, see randn and randexp for details.\n\nTo generate random numbers from other distributions, see the Distributions.jl package.\n\nwarning: Warning\nBecause the precise way in which random numbers are generated is considered an implementation detail, bug fixes and speed improvements may change the stream of numbers that are generated after a version change. Relying on a specific seed or generated stream of numbers during unit testing is thus discouraged - consider testing properties of the methods in question instead.","page":"Random Numbers"},{"title":"Random numbers module","location":"stdlib/Random.html#Random-numbers-module","category":"section","text":"","page":"Random Numbers"},{"title":"Random generation functions","location":"stdlib/Random.html#Random-generation-functions","category":"section","text":"","page":"Random Numbers"},{"title":"Subsequences, permutations and shuffling","location":"stdlib/Random.html#Subsequences,-permutations-and-shuffling","category":"section","text":"","page":"Random Numbers"},{"title":"Generators (creation and seeding)","location":"stdlib/Random.html#Generators-(creation-and-seeding)","category":"section","text":"","page":"Random Numbers"},{"title":"Hooking into the Random API","location":"stdlib/Random.html#rand-api-hook","category":"section","text":"There are two mostly orthogonal ways to extend Random functionalities:\n\ngenerating random values of custom types\ncreating new generators\n\nThe API for 1) is quite functional, but is relatively recent so it may still have to evolve in subsequent releases of the Random module. For example, it's typically sufficient to implement one rand method in order to have all other usual methods work automatically.\n\nThe API for 2) is still rudimentary, and may require more work than strictly necessary from the implementer, in order to support usual types of generated values.","page":"Random Numbers"},{"title":"Generating random values of custom types","location":"stdlib/Random.html#Generating-random-values-of-custom-types","category":"section","text":"Generating random values for some distributions may involve various trade-offs. Pre-computed values, such as an alias table for discrete distributions, or “squeezing” functions for univariate distributions, can speed up sampling considerably. How much information should be pre-computed can depend on the number of values we plan to draw from a distribution. Also, some random number generators can have certain properties that various algorithms may want to exploit.\n\nThe Random module defines a customizable framework for obtaining random values that can address these issues. Each invocation of rand generates a sampler which can be customized with the above trade-offs in mind, by adding methods to Sampler, which in turn can dispatch on the random number generator, the object that characterizes the distribution, and a suggestion for the number of repetitions. Currently, for the latter, Val{1} (for a single sample) and Val{Inf} (for an arbitrary number) are used, with Random.Repetition an alias for both.\n\nThe object returned by Sampler is then used to generate the random values. When implementing the random generation interface for a value X that can be sampled from, the implementer should define the method\n\nrand(rng, sampler)\n\nfor the particular sampler returned by Sampler(rng, X, repetition).\n\nSamplers can be arbitrary values that implement rand(rng, sampler), but for most applications the following predefined samplers may be sufficient:\n\nSamplerType{T}() can be used for implementing samplers that draw from type T (e.g. rand(Int)). This is the default returned by Sampler for types.\nSamplerTrivial(self) is a simple wrapper for self, which can be accessed with []. This is the recommended sampler when no pre-computed information is needed (e.g. rand(1:3)), and is the default returned by Sampler for values.\nSamplerSimple(self, data) also contains the additional data field, which can be used to store arbitrary pre-computed values, which should be computed in a custom method of Sampler.\n\nWe provide examples for each of these. We assume here that the choice of algorithm is independent of the RNG, so we use AbstractRNG in our signatures.\n\nDecoupling pre-computation from actually generating the values is part of the API, and is also available to the user. As an example, assume that rand(rng, 1:20) has to be called repeatedly in a loop: the way to take advantage of this decoupling is as follows:\n\nrng = Xoshiro()\nsp = Random.Sampler(rng, 1:20) # or Random.Sampler(Xoshiro, 1:20)\nfor x in X\n    n = rand(rng, sp) # similar to n = rand(rng, 1:20)\n    # use n\nend\n\nThis is the mechanism that is also used in the standard library, e.g. by the default implementation of random array generation (like in rand(1:20, 10)).","page":"Random Numbers"},{"title":"Generating values from a type","location":"stdlib/Random.html#Generating-values-from-a-type","category":"section","text":"Given a type T, it's currently assumed that if rand(T) is defined, an object of type T will be produced. SamplerType is the default sampler for types. In order to define random generation of values of type T, the rand(rng::AbstractRNG, ::Random.SamplerType{T}) method should be defined, and should return values what rand(rng, T) is expected to return.\n\nLet's take the following example: we implement a Die type, with a variable number n of sides, numbered from 1 to n. We want rand(Die) to produce a Die with a random number of up to 20 sides (and at least 4):\n\nstruct Die\n    nsides::Int # number of sides\nend\n\nRandom.rand(rng::AbstractRNG, ::Random.SamplerType{Die}) = Die(rand(rng, 4:20))\n\n# output\n\n\nScalar and array methods for Die now work as expected:\n\njulia> rand(Die)\nDie(5)\n\njulia> rand(Xoshiro(0), Die)\nDie(10)\n\njulia> rand(Die, 3)\n3-element Vector{Die}:\n Die(9)\n Die(15)\n Die(14)\n\njulia> a = Vector{Die}(undef, 3); rand!(a)\n3-element Vector{Die}:\n Die(19)\n Die(7)\n Die(17)","page":"Random Numbers"},{"title":"A simple sampler without pre-computed data","location":"stdlib/Random.html#A-simple-sampler-without-pre-computed-data","category":"section","text":"Here we define a sampler for a collection. If no pre-computed data is required, it can be implemented with a SamplerTrivial sampler, which is in fact the default fallback for values.\n\nIn order to define random generation out of objects of type S, the following method should be defined: rand(rng::AbstractRNG, sp::Random.SamplerTrivial{S}). Here, sp simply wraps an object of type S, which can be accessed via sp[]. Continuing the Die example, we want now to define rand(d::Die) to produce an Int corresponding to one of d's sides:\n\njulia> Random.rand(rng::AbstractRNG, d::Random.SamplerTrivial{Die}) = rand(rng, 1:d[].nsides);\n\njulia> rand(Die(4))\n1\n\njulia> rand(Die(4), 3)\n3-element Vector{Any}:\n 2\n 3\n 3\n\nGiven a collection type S, it's currently assumed that if rand(::S) is defined, an object of type eltype(S) will be produced. In the last example, a Vector{Any} is produced; the reason is that eltype(Die) == Any. The remedy is to define Base.eltype(::Type{Die}) = Int.","page":"Random Numbers"},{"title":"Generating values for an AbstractFloat type","location":"stdlib/Random.html#Generating-values-for-an-AbstractFloat-type","category":"section","text":"AbstractFloat types are special-cased, because by default random values are not produced in the whole type domain, but rather in [0,1). The following method should be implemented for T <: AbstractFloat: Random.rand(::AbstractRNG, ::Random.SamplerTrivial{Random.CloseOpen01{T}})","page":"Random Numbers"},{"title":"An optimized sampler with pre-computed data","location":"stdlib/Random.html#An-optimized-sampler-with-pre-computed-data","category":"section","text":"Consider a discrete distribution, where numbers 1:n are drawn with given probabilities that sum to one. When many values are needed from this distribution, the fastest method is using an alias table. We don't provide the algorithm for building such a table here, but suppose it is available in make_alias_table(probabilities) instead, and draw_number(rng, alias_table) can be used to draw a random number from it.\n\nSuppose that the distribution is described by\n\nstruct DiscreteDistribution{V <: AbstractVector}\n    probabilities::V\nend\n\nand that we always want to build an alias table, regardless of the number of values needed (we learn how to customize this below). The methods\n\nRandom.eltype(::Type{<:DiscreteDistribution}) = Int\n\nfunction Random.Sampler(::Type{<:AbstractRNG}, distribution::DiscreteDistribution, ::Repetition)\n    SamplerSimple(distribution, make_alias_table(distribution.probabilities))\nend\n\nshould be defined to return a sampler with pre-computed data, then\n\nfunction rand(rng::AbstractRNG, sp::SamplerSimple{<:DiscreteDistribution})\n    draw_number(rng, sp.data)\nend\n\nwill be used to draw the values.","page":"Random Numbers"},{"title":"Custom sampler types","location":"stdlib/Random.html#Custom-sampler-types","category":"section","text":"The SamplerSimple type is sufficient for most use cases with precomputed data. However, in order to demonstrate how to use custom sampler types, here we implement something similar to SamplerSimple.\n\nGoing back to our Die example: rand(::Die) uses random generation from a range, so there is an opportunity for this optimization. We call our custom sampler SamplerDie.\n\nimport Random: Sampler, rand\n\nstruct SamplerDie <: Sampler{Int} # generates values of type Int\n    die::Die\n    sp::Sampler{Int} # this is an abstract type, so this could be improved\nend\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerDie(die, Sampler(RNG, 1:die.nsides, r))\n# the `r` parameter will be explained later on\n\nrand(rng::AbstractRNG, sp::SamplerDie) = rand(rng, sp.sp)\n\nIt's now possible to get a sampler with sp = Sampler(rng, die), and use sp instead of die in any rand call involving rng. In the simplistic example above, die doesn't need to be stored in SamplerDie but this is often the case in practice.\n\nOf course, this pattern is so frequent that the helper type used above, namely Random.SamplerSimple, is available, saving us the definition of SamplerDie: we could have implemented our decoupling with:\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerSimple(die, Sampler(RNG, 1:die.nsides, r))\n\nrand(rng::AbstractRNG, sp::SamplerSimple{Die}) = rand(rng, sp.data)\n\nHere, sp.data refers to the second parameter in the call to the SamplerSimple constructor (in this case equal to Sampler(rng, 1:die.nsides, r)), while the Die object can be accessed via sp[].\n\nLike SamplerDie, any custom sampler must be a subtype of Sampler{T} where T is the type of the generated values. Note that SamplerSimple(x, data) isa Sampler{eltype(x)}, so this constrains what the first argument to SamplerSimple can be (it's recommended to use SamplerSimple like in the Die example, where x is simply forwarded while defining a Sampler method). Similarly, SamplerTrivial(x) isa Sampler{eltype(x)}.\n\nAnother helper type is currently available for other cases, Random.SamplerTag, but is considered as internal API, and can break at any time without proper deprecations.","page":"Random Numbers"},{"title":"Using distinct algorithms for scalar or array generation","location":"stdlib/Random.html#Using-distinct-algorithms-for-scalar-or-array-generation","category":"section","text":"In some cases, whether one wants to generate only a handful of values or a large number of values will have an impact on the choice of algorithm. This is handled with the third parameter of the Sampler constructor. Let's assume we defined two helper types for Die, say SamplerDie1 which should be used to generate only few random values, and SamplerDieMany for many values. We can use those types as follows:\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{1}) = SamplerDie1(...)\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{Inf}) = SamplerDieMany(...)\n\nOf course, rand must also be defined on those types (i.e. rand(::AbstractRNG, ::SamplerDie1) and rand(::AbstractRNG, ::SamplerDieMany)). Note that, as usual, SamplerTrivial and SamplerSimple can be used if custom types are not necessary.\n\nNote: Sampler(rng, x) is simply a shorthand for Sampler(rng, x, Val(Inf)), and Random.Repetition is an alias for Union{Val{1}, Val{Inf}}.","page":"Random Numbers"},{"title":"Creating new generators","location":"stdlib/Random.html#Creating-new-generators","category":"section","text":"The API is not clearly defined yet, but as a rule of thumb:\n\nany rand method producing \"basic\" types (isbitstype integer and floating types in Base) should be defined for this specific RNG, if they are needed;\nother documented rand methods accepting an AbstractRNG should work out of the box, (provided the methods from 1) what are relied on are implemented), but can of course be specialized for this RNG if there is room for optimization;\ncopy for pseudo-RNGs should return an independent copy that generates the exact same random sequence as the original from that point when called in the same way. When this is not feasible (e.g. hardware-based RNGs), copy must not be implemented.\n\nConcerning 1), a rand method may happen to work automatically, but it's not officially supported and may break without warnings in a subsequent release.\n\nTo define a new rand method for an hypothetical MyRNG generator, and a value specification s (e.g. s == Int, or s == 1:10) of type S==typeof(s) or S==Type{s} if s is a type, the same two methods as we saw before must be defined:\n\nSampler(::Type{MyRNG}, ::S, ::Repetition), which returns an object of type say SamplerS\nrand(rng::MyRNG, sp::SamplerS)\n\nIt can happen that Sampler(rng::AbstractRNG, ::S, ::Repetition) is already defined in the Random module. It would then be possible to skip step 1) in practice (if one wants to specialize generation for this particular RNG type), but the corresponding SamplerS type is considered as internal detail, and may be changed without warning.","page":"Random Numbers"},{"title":"Specializing array generation","location":"stdlib/Random.html#Specializing-array-generation","category":"section","text":"In some cases, for a given RNG type, generating an array of random values can be more efficient with a specialized method than by merely using the decoupling technique explained before. This is for example the case for MersenneTwister, which natively writes random values in an array.\n\nTo implement this specialization for MyRNG and for a specification s, producing elements of type S, the following method can be defined: rand!(rng::MyRNG, a::AbstractArray{S}, ::SamplerS), where SamplerS is the type of the sampler returned by Sampler(MyRNG, s, Val(Inf)). Instead of AbstractArray, it's possible to implement the functionality only for a subtype, e.g. Array{S}. The non-mutating array method of rand will automatically call this specialization internally.","page":"Random Numbers"},{"title":"Reproducibility","location":"stdlib/Random.html#Reproducibility","category":"section","text":"By using an RNG parameter initialized with a given seed, you can reproduce the same pseudorandom number sequence when running your program multiple times. However, a minor release of Julia (e.g. 1.3 to 1.4) may change the sequence of pseudorandom numbers generated from a specific seed. (Even if the sequence produced by a low-level function like rand does not change, the output of higher-level functions like randsubseq may change due to algorithm updates.) Rationale: guaranteeing that pseudorandom streams never change prohibits many algorithmic improvements.\n\nIf you need to guarantee exact reproducibility of random data, it is advisable to simply save the data (e.g. as a supplementary attachment in a scientific publication). (You can also, of course, specify a particular Julia version and package manifest, especially if you require bit reproducibility.)\n\nSoftware tests that rely on specific \"random\" data should also generally either save the data, embed it into the test code, or use third-party packages like StableRNGs.jl. On the other hand, tests that should pass for most random data (e.g. testing A \\ (A*x) ≈ x for a random matrix A = randn(n,n)) can use an RNG with a fixed seed to ensure that simply running the test many times does not encounter a failure due to very improbable data (e.g. an extremely ill-conditioned matrix).\n\nThe statistical distribution from which random samples are drawn is guaranteed to be the same across any minor Julia releases.","page":"Random Numbers"},{"title":"Random.Random","location":"stdlib/Random.html#Random.Random","category":"module","text":"Random\n\nSupport for generating random numbers. Provides rand, randn, AbstractRNG, Xoshiro, MersenneTwister, and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Base.rand","location":"stdlib/Random.html#Base.rand","category":"function","text":"rand([rng=default_rng()], [S], [dims...])\n\nPick a random element or array of random elements from the set of values specified by S; S can be\n\nan indexable collection (for example 1:9 or ('x', \"y\", :z))\nan AbstractDict or AbstractSet object\na string (considered as a collection of characters), or\na type from the list below, corresponding to the specified set of values\nconcrete integer types sample from typemin(S):typemax(S) (excepting BigInt which is not supported)\nconcrete floating point types sample from [0, 1)\nconcrete complex types Complex{T} if T is a sampleable type take their real and imaginary components independently from the set of values corresponding to T, but are not supported if T is not sampleable.\nall <:AbstractChar types sample from the set of valid Unicode scalars\na user-defined type and set of values; for implementation guidance please see Hooking into the Random API\na tuple type of known size and where each parameter of S is itself a sampleable type; return a value of type S. Note that tuple types such as Tuple{Vararg{T}} (unknown size) and Tuple{1:2} (parameterized with a value) are not supported\na Pair type, e.g. Pair{X, Y} such that rand is defined for X and Y, in which case random pairs are produced.\n\nS defaults to Float64. When only one argument is passed besides the optional rng and is a Tuple, it is interpreted as a collection of values (S) and not as dims.\n\nSee also randn for normally distributed numbers, and rand! and randn! for the in-place equivalents.\n\ncompat: Julia 1.1\nSupport for S as a tuple requires at least Julia 1.1.\n\ncompat: Julia 1.11\nSupport for S as a Tuple type requires at least Julia 1.11.\n\nExamples\n\njulia> rand(Int, 2)\n2-element Vector{Int64}:\n 1339893410598768192\n 1575814717733606317\n\njulia> using Random\n\njulia> rand(Xoshiro(0), Dict(1=>2, 3=>4))\n3 => 4\n\njulia> rand((2, 3))\n3\n\njulia> rand(Float64, (2, 3))\n2×3 Matrix{Float64}:\n 0.999717  0.0143835  0.540787\n 0.696556  0.783855   0.938235\n\nnote: Note\nThe complexity of rand(rng, s::Union{AbstractDict,AbstractSet}) is linear in the length of s, unless an optimized method with constant complexity is available, which is the case for Dict, Set and dense BitSets. For more than a few calls, use rand(rng, collect(s)) instead, or either rand(rng, Dict(s)) or rand(rng, Set(s)) as appropriate.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.rand!","location":"stdlib/Random.html#Random.rand!","category":"function","text":"rand!([rng=default_rng()], A, [S=eltype(A)])\n\nPopulate the array A with random values. If S is specified (S can be a type or a collection, cf. rand for details), the values are picked randomly from S. This is equivalent to copyto!(A, rand(rng, S, size(A))) but without allocating a new array.\n\nExamples\n\njulia> rand!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n 0.521213795535383\n 0.5868067574533484\n 0.8908786980927811\n 0.19090669902576285\n 0.5256623915420473\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.bitrand","location":"stdlib/Random.html#Random.bitrand","category":"function","text":"bitrand([rng=default_rng()], [dims...])\n\nGenerate a BitArray of random boolean values.\n\nExamples\n\njulia> bitrand(Xoshiro(123), 10)\n10-element BitVector:\n 0\n 1\n 0\n 1\n 0\n 1\n 0\n 0\n 1\n 1\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Base.randn","location":"stdlib/Random.html#Base.randn","category":"function","text":"randn([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a normally-distributed random number of type T with mean 0 and standard deviation 1. Given the optional dims argument(s), generate an array of size dims of such numbers. Julia's standard library supports randn for any floating-point type that implements rand, e.g. the Base types Float16, Float32, Float64 (the default), and BigFloat, along with their Complex counterparts.\n\n(When T is complex, the values are drawn from the circularly symmetric complex normal distribution of variance 1, corresponding to real and imaginary parts having independent normal distribution with mean zero and variance 1/2).\n\nSee also randn! to act in-place.\n\nExamples\n\nGenerating a single random number (with the default Float64 type):\n\njulia> randn()\n-0.942481877315864\n\nGenerating a matrix of normal random numbers (with the default Float64 type):\n\njulia> randn(2,3)\n2×3 Matrix{Float64}:\n  1.18786   -0.678616   1.49463\n -0.342792  -0.134299  -1.45005\n\nSetting up of the random number generator rng with a user-defined seed (for reproducible numbers) and using it to generate a random Float32 number or a matrix of ComplexF32 random numbers:\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> randn(rng, Float32)\n-0.6457307f0\n\njulia> randn(rng, ComplexF32, (2, 3))\n2×3 Matrix{ComplexF32}:\n  -1.03467-1.14806im  0.693657+0.056538im   0.291442+0.419454im\n -0.153912+0.34807im    1.0954-0.948661im  -0.543347-0.0538589im\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randn!","location":"stdlib/Random.html#Random.randn!","category":"function","text":"randn!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with normally-distributed (mean 0, standard deviation 1) random numbers. Also see the rand function.\n\nExamples\n\njulia> randn!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n -0.6457306721039767\n -1.4632513788889214\n -1.6236037455860806\n -0.21766510678354617\n  0.4922456865251828\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp","location":"stdlib/Random.html#Random.randexp","category":"function","text":"randexp([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a random number of type T according to the exponential distribution with scale 1. Optionally generate an array of such random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default).\n\nExamples\n\njulia> rng = Xoshiro(123);\n\njulia> randexp(rng, Float32)\n1.1757717f0\n\njulia> randexp(rng, 3, 3)\n3×3 Matrix{Float64}:\n 1.37766  0.456653  0.236418\n 3.40007  0.229917  0.0684921\n 0.48096  0.577481  0.71835\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randexp!","location":"stdlib/Random.html#Random.randexp!","category":"function","text":"randexp!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with random numbers following the exponential distribution (with scale 1).\n\nExamples\n\njulia> randexp!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n 1.1757716836348473\n 1.758884569451514\n 1.0083623637301151\n 0.3510644315565272\n 0.6348266443720407\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randstring","location":"stdlib/Random.html#Random.randstring","category":"function","text":"randstring([rng=default_rng()], [chars], [len=8])\n\nCreate a random string of length len, consisting of characters from chars, which defaults to the set of upper- and lower-case letters and the digits 0-9. The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> Random.seed!(3); randstring()\n\"Lxz5hUwn\"\n\njulia> randstring(Xoshiro(3), 'a':'z', 6)\n\"iyzcsm\"\n\njulia> randstring(\"ACGT\")\n\"TGCTCCTC\"\n\nnote: Note\nchars can be any collection of characters, of type Char or UInt8 (more efficient), provided rand can randomly pick characters from it.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randsubseq","location":"stdlib/Random.html#Random.randsubseq","category":"function","text":"randsubseq([rng=default_rng(),] A, p) -> Vector\n\nReturn a vector consisting of a random subsequence of the given array A, where each element of A is included (in order) with independent probability p. (Complexity is linear in p*length(A), so this function is efficient even if p is small and A is large.) Technically, this process is known as \"Bernoulli sampling\" of A.\n\nExamples\n\njulia> randsubseq(Xoshiro(123), 1:8, 0.3)\n2-element Vector{Int64}:\n 4\n 7\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randsubseq!","location":"stdlib/Random.html#Random.randsubseq!","category":"function","text":"randsubseq!([rng=default_rng(),] S, A, p)\n\nLike randsubseq, but the results are stored in S (which is resized as needed).\n\nExamples\n\njulia> S = Int64[];\n\njulia> randsubseq!(Xoshiro(123), S, 1:8, 0.3)\n2-element Vector{Int64}:\n 4\n 7\n\njulia> S\n2-element Vector{Int64}:\n 4\n 7\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm","location":"stdlib/Random.html#Random.randperm","category":"function","text":"randperm([rng=default_rng(),] n::Integer)\n\nConstruct a random permutation of length n. The optional rng argument specifies a random number generator (see Random Numbers). The element type of the result is the same as the type of n.\n\nTo randomly permute an arbitrary vector, see shuffle or shuffle!.\n\ncompat: Julia 1.1\nIn Julia 1.1 randperm returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randperm(Xoshiro(123), 4)\n4-element Vector{Int64}:\n 1\n 4\n 2\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randperm!","location":"stdlib/Random.html#Random.randperm!","category":"function","text":"randperm!([rng=default_rng(),] A::Array{<:Integer})\n\nConstruct in A a random permutation of length length(A). The optional rng argument specifies a random number generator (see Random Numbers). To randomly permute an arbitrary vector, see shuffle or shuffle!.\n\nExamples\n\njulia> randperm!(Xoshiro(123), Vector{Int}(undef, 4))\n4-element Vector{Int64}:\n 1\n 4\n 2\n 3\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle","location":"stdlib/Random.html#Random.randcycle","category":"function","text":"randcycle([rng=default_rng(),] n::Integer)\n\nConstruct a random cyclic permutation of length n. The optional rng argument specifies a random number generator, see Random Numbers. The element type of the result is the same as the type of n.\n\nHere, a \"cyclic permutation\" means that all of the elements lie within a single cycle.  If n > 0, there are (n-1) possible cyclic permutations, which are sampled uniformly.  If n == 0, randcycle returns an empty vector.\n\nrandcycle! is an in-place variant of this function.\n\ncompat: Julia 1.1\nIn Julia 1.1 and above, randcycle returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randcycle(Xoshiro(123), 6)\n6-element Vector{Int64}:\n 5\n 4\n 2\n 6\n 3\n 1\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.randcycle!","location":"stdlib/Random.html#Random.randcycle!","category":"function","text":"randcycle!([rng=default_rng(),] A::Array{<:Integer})\n\nConstruct in A a random cyclic permutation of length n = length(A). The optional rng argument specifies a random number generator, see Random Numbers.\n\nHere, a \"cyclic permutation\" means that all of the elements lie within a single cycle. If A is nonempty (n > 0), there are (n-1) possible cyclic permutations, which are sampled uniformly.  If A is empty, randcycle! leaves it unchanged.\n\nrandcycle is a variant of this function that allocates a new vector.\n\nExamples\n\njulia> randcycle!(Xoshiro(123), Vector{Int}(undef, 6))\n6-element Vector{Int64}:\n 5\n 4\n 2\n 6\n 3\n 1\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle","location":"stdlib/Random.html#Random.shuffle","category":"function","text":"shuffle([rng=default_rng(),] v::AbstractArray)\n\nReturn a randomly permuted copy of v. The optional rng argument specifies a random number generator (see Random Numbers). To permute v in-place, see shuffle!. To obtain randomly permuted indices, see randperm.\n\nExamples\n\njulia> shuffle(Xoshiro(123), Vector(1:10))\n10-element Vector{Int64}:\n  5\n  4\n  2\n  3\n  6\n 10\n  8\n  1\n  9\n  7\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.shuffle!","location":"stdlib/Random.html#Random.shuffle!","category":"function","text":"shuffle!([rng=default_rng(),] v::AbstractArray)\n\nIn-place version of shuffle: randomly permute v in-place, optionally supplying the random-number generator rng.\n\nExamples\n\njulia> shuffle!(Xoshiro(123), Vector(1:10))\n10-element Vector{Int64}:\n  5\n  4\n  2\n  3\n  6\n 10\n  8\n  1\n  9\n  7\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.default_rng","location":"stdlib/Random.html#Random.default_rng","category":"function","text":"Random.default_rng() -> rng\n\nReturn the default global random number generator (RNG), which is used by rand-related functions when no explicit RNG is provided.\n\nWhen the Random module is loaded, the default RNG is randomly seeded, via Random.seed!(): this means that each time a new julia session is started, the first call to rand() produces a different result, unless seed!(seed) is called first.\n\nIt is thread-safe: distinct threads can safely call rand-related functions on default_rng() concurrently, e.g. rand(default_rng()).\n\nnote: Note\nThe type of the default RNG is an implementation detail. Across different versions of Julia, you should not expect the default RNG to always have the same type, nor that it will produce the same stream of random numbers for a given seed.\n\ncompat: Julia 1.3\nThis function was introduced in Julia 1.3.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.seed!","location":"stdlib/Random.html#Random.seed!","category":"function","text":"seed!([rng=default_rng()], seed) -> rng\nseed!([rng=default_rng()]) -> rng\n\nReseed the random number generator: rng will give a reproducible sequence of numbers if and only if a seed is provided. Some RNGs don't accept a seed, like RandomDevice. After the call to seed!, rng is equivalent to a newly created object initialized with the same seed. The types of accepted seeds depend on the type of rng, but in general, integer seeds should work.\n\nIf rng is not specified, it defaults to seeding the state of the shared task-local generator.\n\nExamples\n\njulia> Random.seed!(1234);\n\njulia> x1 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> Random.seed!(1234);\n\njulia> x2 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\njulia> rng = Xoshiro(1234); rand(rng, 2) == x1\ntrue\n\njulia> Xoshiro(1) == Random.seed!(rng, 1)\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible either\nfalse\n\njulia> rand(Xoshiro(), Bool) # not reproducible either\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.AbstractRNG","location":"stdlib/Random.html#Random.AbstractRNG","category":"type","text":"AbstractRNG\n\nSupertype for random number generators such as MersenneTwister and RandomDevice.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.TaskLocalRNG","location":"stdlib/Random.html#Random.TaskLocalRNG","category":"type","text":"TaskLocalRNG\n\nThe TaskLocalRNG has state that is local to its task, not its thread. It is seeded upon task creation, from the state of its parent task, but without advancing the state of the parent's RNG.\n\nAs an upside, the TaskLocalRNG is pretty fast, and permits reproducible multithreaded simulations (barring race conditions), independent of scheduler decisions. As long as the number of threads is not used to make decisions on task creation, simulation results are also independent of the number of available threads / CPUs. The random stream should not depend on hardware specifics, up to endianness and possibly word size.\n\nUsing or seeding the RNG of any other task than the one returned by current_task() is undefined behavior: it will work most of the time, and may sometimes fail silently.\n\nWhen seeding TaskLocalRNG() with seed!, the passed seed, if any, may be any integer.\n\ncompat: Julia 1.11\nSeeding TaskLocalRNG() with a negative integer seed requires at least Julia 1.11.\n\ncompat: Julia 1.10\nTask creation no longer advances the parent task's RNG state as of Julia 1.10.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.Xoshiro","location":"stdlib/Random.html#Random.Xoshiro","category":"type","text":"Xoshiro(seed::Union{Integer, AbstractString})\nXoshiro()\n\nXoshiro256++ is a fast pseudorandom number generator described by David Blackman and Sebastiano Vigna in \"Scrambled Linear Pseudorandom Number Generators\", ACM Trans. Math. Softw., 2021. Reference implementation is available at https://prng.di.unimi.it\n\nApart from the high speed, Xoshiro has a small memory footprint, making it suitable for applications where many different random states need to be held for long time.\n\nJulia's Xoshiro implementation has a bulk-generation mode; this seeds new virtual PRNGs from the parent, and uses SIMD to generate in parallel (i.e. the bulk stream consists of multiple interleaved xoshiro instances). The virtual PRNGs are discarded once the bulk request has been serviced (and should cause no heap allocations).\n\nIf no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing Xoshiro object.\n\ncompat: Julia 1.11\nPassing a negative integer seed requires at least Julia 1.11.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(1234);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> rng = Xoshiro(1234);\n\njulia> x2 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.MersenneTwister","location":"stdlib/Random.html#Random.MersenneTwister","category":"type","text":"MersenneTwister(seed)\nMersenneTwister()\n\nCreate a MersenneTwister RNG object. Different RNG objects can have their own seeds, which may be useful for generating different streams of random numbers. The seed may be an integer, a string, or a vector of UInt32 integers. If no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing MersenneTwister object.\n\ncompat: Julia 1.11\nPassing a negative integer seed requires at least Julia 1.11.\n\nExamples\n\njulia> rng = MersenneTwister(123);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.37453777969575874\n 0.8735343642013971\n\njulia> x2 = rand(MersenneTwister(123), 2)\n2-element Vector{Float64}:\n 0.37453777969575874\n 0.8735343642013971\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.RandomDevice","location":"stdlib/Random.html#Random.RandomDevice","category":"type","text":"RandomDevice()\n\nCreate a RandomDevice RNG object. Two such objects will always generate different streams of random numbers. The entropy is obtained from the operating system.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.Sampler","location":"stdlib/Random.html#Random.Sampler","category":"type","text":"Sampler(rng, x, repetition = Val(Inf))\n\nReturn a sampler object that can be used to generate random values from rng for x.\n\nWhen sp = Sampler(rng, x, repetition), rand(rng, sp) will be used to draw random values, and should be defined accordingly.\n\nrepetition can be Val(1) or Val(Inf), and should be used as a suggestion for deciding the amount of precomputation, if applicable.\n\nRandom.SamplerType and Random.SamplerTrivial are default fallbacks for types and values, respectively. Random.SamplerSimple can be used to store pre-computed values without defining extra types for only this purpose.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerType","location":"stdlib/Random.html#Random.SamplerType","category":"type","text":"SamplerType{T}()\n\nA sampler for types, containing no other information. The default fallback for Sampler when called with types.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerTrivial","location":"stdlib/Random.html#Random.SamplerTrivial","category":"type","text":"SamplerTrivial(x)\n\nCreate a sampler that just wraps the given value x. This is the default fall-back for values. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values without precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Random.SamplerSimple","location":"stdlib/Random.html#Random.SamplerSimple","category":"type","text":"SamplerSimple(x, data)\n\nCreate a sampler that wraps the given value x and the data. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values with precomputed data.\n\n\n\n\n\n","page":"Random Numbers"},{"title":"Package Images","location":"devdocs/pkgimg.html#pkgimages","category":"section","text":"Julia package images provide object (native code) caches for Julia packages. They are similar to Julia's system image and support many of the same features. In fact the underlying serialization format is the same, and the system image is the base image that the package images are build against.","page":"Package Images"},{"title":"High-level overview","location":"devdocs/pkgimg.html#High-level-overview","category":"section","text":"Package images are shared libraries that contain both code and data. Like .ji cache files, they are generated per package. The data section contains both global data (global variables in the package) as well as the necessary metadata about what methods and types are defined by the package. The code section contains native objects that cache the final output of Julia's LLVM-based compiler.\n\nThe command line option --pkgimages=no can be used to turn off object caching for this session. Note that this means that cache files have to likely be regenerated. See JULIA_MAX_NUM_PRECOMPILE_FILES for the upper limit of variants Julia caches per default.\n\nnote: Note\nWhile the package images present themselves as native shared libraries, they are only an approximation thereof. You will not be able to link against them from a native program and they must be loaded from Julia.","page":"Package Images"},{"title":"Linking","location":"devdocs/pkgimg.html#Linking","category":"section","text":"Since the package images contain native code, we must run a linker over them before we can use them. You can set the environment variable JULIA_VERBOSE_LINKING to true to make the package image linking process verbose.\n\nFurthermore, we cannot assume that the user has a working system linker installed. Therefore, Julia ships with LLD, the LLVM linker, to provide a working out of the box experience. In base/linking.jl, we implement a limited interface to be able to link package images on all supported platforms.","page":"Package Images"},{"title":"Quirks","location":"devdocs/pkgimg.html#Quirks","category":"section","text":"Despite LLD being a multi-platform linker, it does not provide a consistent interface across platforms. Furthermore, it is meant to be used from clang or another compiler driver, we therefore reimplement some of the logic from llvm-project/clang/lib/Driver/ToolChains. Thankfully one can use lld -flavor to set lld to the right platform","page":"Package Images"},{"title":"Windows","location":"devdocs/pkgimg.html#Windows","category":"section","text":"To avoid having to deal with link.exe we use -flavor gnu, effectively turning lld into a cross-linker from a mingw32 environment. Windows DLLs are required to contain a _DllMainCRTStartup function and to minimize our dependence on mingw32 libraries, we inject a stub definition ourselves.","page":"Package Images"},{"title":"MacOS","location":"devdocs/pkgimg.html#MacOS","category":"section","text":"Dynamic libraries on macOS need to link against -lSystem. On recent macOS versions, -lSystem is only available for linking when Xcode is available. To that effect we link with -undefined dynamic_lookup.","page":"Package Images"},{"title":"Package images optimized for multiple microarchitectures","location":"devdocs/pkgimg.html#pkgimgs-multi-versioning","category":"section","text":"Similar to multi-versioning for system images, package images support multi-versioning. If you are in a heterogeneous environment, with a unified cache, you can set the environment variable JULIA_CPU_TARGET=generic to multi-version the object caches.","page":"Package Images"},{"title":"Flags that impact package image creation and selection","location":"devdocs/pkgimg.html#Flags-that-impact-package-image-creation-and-selection","category":"section","text":"These are the Julia command line flags that impact cache selection. Package images that were created with different flags will be rejected.\n\n-g, --debug-info: Exact match required since it changes code generation.\n--check-bounds: Exact match required since it changes code generation.\n--inline: Exact match required since it changes code generation.\n--pkgimages: To allow running without object caching enabled.\n-O, --optimize: Reject package images generated for a lower optimization level, but allow for higher optimization levels to be loaded.","page":"Package Images"},{"title":"Mathematics","location":"base/math.html#Mathematics","category":"section","text":"","page":"Mathematics"},{"title":"Mathematical Operators","location":"base/math.html#math-ops","category":"section","text":"","page":"Mathematics"},{"title":"Mathematical Functions","location":"base/math.html#Mathematical-Functions","category":"section","text":"","page":"Mathematics"},{"title":"Customizable binary operators","location":"base/math.html#Customizable-binary-operators","category":"section","text":"Some unicode characters can be used to define new binary operators that support infix notation. For example ⊗(x,y) = kron(x,y) defines the ⊗ (otimes) function to be the Kronecker product, and one can call it as binary operator using infix syntax: C = A ⊗ B as well as with the usual prefix syntax C = ⊗(A,B).\n\nOther characters that support such extensions include \\odot ⊙ and \\oplus ⊕\n\nThe complete list is in the parser code: https://github.com/JuliaLang/julia/blob/master/src/julia-parser.scm\n\nThose that are parsed like * (in terms of precedence) include * / ÷ % & ⋅ ∘ × |\\\\| ∩ ∧ ⊗ ⊘ ⊙ ⊚ ⊛ ⊠ ⊡ ⊓ ∗ ∙ ∤ ⅋ ≀ ⊼ ⋄ ⋆ ⋇ ⋉ ⋊ ⋋ ⋌ ⋏ ⋒ ⟑ ⦸ ⦼ ⦾ ⦿ ⧶ ⧷ ⨇ ⨰ ⨱ ⨲ ⨳ ⨴ ⨵ ⨶ ⨷ ⨸ ⨻ ⨼ ⨽ ⩀ ⩃ ⩄ ⩋ ⩍ ⩎ ⩑ ⩓ ⩕ ⩘ ⩚ ⩜ ⩞ ⩟ ⩠ ⫛ ⊍ ▷ ⨝ ⟕ ⟖ ⟗ and those that are parsed like + include + - |\\|| ⊕ ⊖ ⊞ ⊟ |++| ∪ ∨ ⊔ ± ∓ ∔ ∸ ≏ ⊎ ⊻ ⊽ ⋎ ⋓ ⟇ ⧺ ⧻ ⨈ ⨢ ⨣ ⨤ ⨥ ⨦ ⨧ ⨨ ⨩ ⨪ ⨫ ⨬ ⨭ ⨮ ⨹ ⨺ ⩁ ⩂ ⩅ ⩊ ⩌ ⩏ ⩐ ⩒ ⩔ ⩖ ⩗ ⩛ ⩝ ⩡ ⩢ ⩣ There are many others that are related to arrows, comparisons, and powers.","page":"Mathematics"},{"title":"Base.:-","location":"base/math.html#Base.:--Tuple{Any}","category":"method","text":"-(x)\n\nUnary minus operator.\n\nSee also: abs, flipsign.\n\nExamples\n\njulia> -1\n-1\n\njulia> -(2)\n-2\n\njulia> -[1 2; 3 4]\n2×2 Matrix{Int64}:\n -1  -2\n -3  -4\n\njulia> -(true)  # promotes to Int\n-1\n\njulia> -(0x003)\n0xfffd\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:+","location":"base/math.html#Base.:+","category":"function","text":"+(x, y...)\n\nAddition operator.\n\nInfix x+y+z+... calls this function with all arguments, i.e. +(x, y, z, ...), which by default then calls (x+y) + z + ... starting from the left.\n\nNote that overflow is possible for most integer types, including the default Int, when adding large numbers.\n\nExamples\n\njulia> 1 + 20 + 4\n25\n\njulia> +(1, 20, 4)\n25\n\njulia> [1,2] + [3,4]\n2-element Vector{Int64}:\n 4\n 6\n\njulia> typemax(Int) + 1 < 0\ntrue\n\n\n\n\n\ndt::Date + t::Time -> DateTime\n\nThe addition of a Date with a Time produces a DateTime. The hour, minute, second, and millisecond parts of the Time are used along with the year, month, and day of the Date to create the new DateTime. Non-zero microseconds or nanoseconds in the Time type will result in an InexactError being thrown.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:-","location":"base/math.html#Base.:--Tuple{Any, Any}","category":"method","text":"-(x, y)\n\nSubtraction operator.\n\nExamples\n\njulia> 2 - 3\n-1\n\njulia> -(2, 4.5)\n-2.5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:*","location":"base/math.html#Base.:*-Tuple{Any, Vararg{Any}}","category":"method","text":"*(x, y...)\n\nMultiplication operator.\n\nInfix x*y*z*... calls this function with all arguments, i.e. *(x, y, z, ...), which by default then calls (x*y) * z * ... starting from the left.\n\nJuxtaposition such as 2pi also calls *(2, pi). Note that this operation has higher precedence than a literal *. Note also that juxtaposition \"0x...\" (integer zero times a variable whose name starts with x) is forbidden as it clashes with unsigned integer literals: 0x01 isa UInt8.\n\nNote that overflow is possible for most integer types, including the default Int, when multiplying large numbers.\n\nExamples\n\njulia> 2 * 7 * 8\n112\n\njulia> *(2, 7, 8)\n112\n\njulia> [2 0; 0 3] * [1, 10]  # matrix * vector\n2-element Vector{Int64}:\n  2\n 30\n\njulia> 1/2pi, 1/2*pi  # juxtaposition has higher precedence\n(0.15915494309189535, 1.5707963267948966)\n\njulia> x = [1, 2]; x'x  # adjoint vector * vector\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:/","location":"base/math.html#Base.:/","category":"function","text":"/(x, y)\n\nRight division operator: multiplication of x by the inverse of y on the right.\n\nGives floating-point results for integer arguments. See ÷ for integer division, or // for Rational results.\n\nExamples\n\njulia> 1/2\n0.5\n\njulia> 4/2\n2.0\n\njulia> 4.5/2\n2.25\n\n\n\n\n\nA / B\n\nMatrix right-division: A / B is equivalent to (B' \\ A')' where \\ is the left-division operator. For square matrices, the result X is such that A == X*B.\n\nSee also: rdiv!.\n\nExamples\n\njulia> A = Float64[1 4 5; 3 9 2]; B = Float64[1 4 2; 3 4 2; 8 7 1];\n\njulia> X = A / B\n2×3 Matrix{Float64}:\n -0.65   3.75  -1.2\n  3.25  -2.75   1.0\n\njulia> isapprox(A, X*B)\ntrue\n\njulia> isapprox(X, A*pinv(B))\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:\\","location":"base/math.html#Base.:\\-Tuple{Any, Any}","category":"method","text":"\\(x, y)\n\nLeft division operator: multiplication of y by the inverse of x on the left. Gives floating-point results for integer arguments.\n\nExamples\n\njulia> 3 \\ 6\n2.0\n\njulia> inv(3) * 6\n2.0\n\njulia> A = [4 3; 2 1]; x = [5, 6];\n\njulia> A \\ x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\njulia> inv(A) * x\n2-element Vector{Float64}:\n  6.5\n -7.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:^","location":"base/math.html#Base.:^-Tuple{Number, Number}","category":"method","text":"^(x, y)\n\nExponentiation operator.\n\nIf x and y are integers, the result may overflow. To enter numbers in scientific notation, use Float64 literals such as 1.2e3 rather than 1.2 * 10^3.\n\nIf y is an Int literal (e.g. 2 in x^2 or -3 in x^-3), the Julia code x^y is transformed by the compiler to Base.literal_pow(^, x, Val(y)), to enable compile-time specialization on the value of the exponent. (As a default fallback we have Base.literal_pow(^, x, Val(y)) = ^(x,y), where usually ^ == Base.^ unless ^ has been defined in the calling namespace.) If y is a negative integer literal, then Base.literal_pow transforms the operation to inv(x)^-y by default, where -y is positive.\n\nSee also exp2, <<.\n\nExamples\n\njulia> 3^5\n243\n\njulia> 3^-1  # uses Base.literal_pow\n0.3333333333333333\n\njulia> p = -1;\n\njulia> 3^p\nERROR: DomainError with -1:\nCannot raise an integer x to a negative power -1.\n[...]\n\njulia> 3.0^p\n0.3333333333333333\n\njulia> 10^19 > 0  # integer overflow\nfalse\n\njulia> big(10)^19 == 1e19\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fma","location":"base/math.html#Base.fma","category":"function","text":"fma(x, y, z)\n\nComputes x*y+z without rounding the intermediate result x*y. On some systems this is significantly more expensive than x*y+z. fma is used to improve accuracy in certain algorithms. See muladd.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.muladd","location":"base/math.html#Base.muladd","category":"function","text":"muladd(x, y, z)\n\nCombined multiply-add: computes x*y+z, but allowing the add and multiply to be merged with each other or with surrounding operations for performance. For example, this may be implemented as an fma if the hardware supports it efficiently. The result can be different on different machines and can also be different on the same machine due to constant propagation or other optimizations. See fma.\n\nExamples\n\njulia> muladd(3, 2, 1)\n7\n\njulia> 3 * 2 + 1\n7\n\n\n\n\n\nmuladd(A, y, z)\n\nCombined multiply-add, A*y .+ z, for matrix-matrix or matrix-vector multiplication. The result is always the same size as A*y, but z may be smaller, or a scalar.\n\ncompat: Julia 1.6\nThese methods require Julia 1.6 or later.\n\nExamples\n\njulia> A=[1.0 2.0; 3.0 4.0]; B=[1.0 1.0; 1.0 1.0]; z=[0, 100];\n\njulia> muladd(A, B, z)\n2×2 Matrix{Float64}:\n   3.0    3.0\n 107.0  107.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.inv","location":"base/math.html#Base.inv-Tuple{Number}","category":"method","text":"inv(x)\n\nReturn the multiplicative inverse of x, such that x*inv(x) or inv(x)*x yields one(x) (the multiplicative identity) up to roundoff errors.\n\nIf x is a number, this is essentially the same as one(x)/x, but for some types inv(x) may be slightly more efficient.\n\nExamples\n\njulia> inv(2)\n0.5\n\njulia> inv(1 + 2im)\n0.2 - 0.4im\n\njulia> inv(1 + 2im) * (1 + 2im)\n1.0 + 0.0im\n\njulia> inv(2//3)\n3//2\n\ncompat: Julia 1.2\ninv(::Missing) requires at least Julia 1.2.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.div","location":"base/math.html#Base.div","category":"function","text":"div(x, y)\n÷(x, y)\n\nThe quotient from Euclidean (integer) division. Generally equivalent to a mathematical operation x/y without a fractional part.\n\nSee also: cld, fld, rem, divrem.\n\nExamples\n\njulia> 9 ÷ 4\n2\n\njulia> -5 ÷ 3\n-1\n\njulia> 5.0 ÷ 2\n2.0\n\njulia> div.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  0  0  1  1  1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.div","location":"base/math.html#Base.div-Tuple{Any, Any, RoundingMode}","category":"method","text":"div(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient from Euclidean (integer) division. Computes x / y, rounded to an integer according to the rounding mode r. In other words, the quantity\n\nround(x / y, r)\n\nwithout any intermediate rounding.\n\ncompat: Julia 1.4\nThe three-argument method taking a RoundingMode requires Julia 1.4 or later.\n\nSee also fld and cld, which are special cases of this function.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> div(4, 3, RoundToZero) # Matches div(4, 3)\n1\njulia> div(4, 3, RoundDown) # Matches fld(4, 3)\n1\njulia> div(4, 3, RoundUp) # Matches cld(4, 3)\n2\njulia> div(5, 2, RoundNearest)\n2\njulia> div(5, 2, RoundNearestTiesAway)\n3\njulia> div(-5, 2, RoundNearest)\n-2\njulia> div(-5, 2, RoundNearestTiesAway)\n-3\njulia> div(-5, 2, RoundNearestTiesUp)\n-2\njulia> div(4, 3, RoundFromZero)\n2\njulia> div(-4, 3, RoundFromZero)\n-2\n\nBecause div(x, y) implements strictly correct truncated rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> div(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but div(6.0, 0.1, RoundToZero) always truncates the true value, so the result is 59.0.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fld","location":"base/math.html#Base.fld","category":"function","text":"fld(x, y)\n\nLargest integer less than or equal to x / y. Equivalent to div(x, y, RoundDown).\n\nSee also div, cld, fld1.\n\nExamples\n\njulia> fld(7.3, 5.5)\n1.0\n\njulia> fld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -2  -1  -1  -1  0  0  0  1  1  1\n\nBecause fld(x, y) implements strictly correct floored rounding based on the true value of floating-point numbers, unintuitive situations can arise. For example:\n\njulia> fld(6.0, 0.1)\n59.0\njulia> 6.0 / 0.1\n60.0\njulia> 6.0 / big(0.1)\n59.99999999999999666933092612453056361837965690217069245739573412231113406246995\n\nWhat is happening here is that the true value of the floating-point number written as 0.1 is slightly larger than the numerical value 1/10 while 6.0 represents the number 6 precisely. Therefore the true value of 6.0 / 0.1 is slightly less than 60. When doing division, this is rounded to precisely 60.0, but fld(6.0, 0.1) always takes the floor of the true value, so the result is 59.0.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cld","location":"base/math.html#Base.cld","category":"function","text":"cld(x, y)\n\nSmallest integer larger than or equal to x / y. Equivalent to div(x, y, RoundUp).\n\nSee also div, fld.\n\nExamples\n\njulia> cld(5.5, 2.2)\n3.0\n\njulia> cld.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -1  -1  -1  0  0  0  1  1  1  2  2\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.mod","location":"base/math.html#Base.mod","category":"function","text":"rem(x::Integer, T::Type{<:Integer}) -> T\nmod(x::Integer, T::Type{<:Integer}) -> T\n%(x::Integer, T::Type{<:Integer}) -> T\n\nFind y::T such that x ≡ y (mod n), where n is the number of integers representable in T, and y is an integer in [typemin(T),typemax(T)]. If T can represent any integer (e.g. T == BigInt), then this operation corresponds to a conversion to T.\n\nExamples\n\njulia> x = 129 % Int8\n-127\n\njulia> typeof(x)\nInt8\n\njulia> x = 129 % BigInt\n129\n\njulia> typeof(x)\nBigInt\n\n\n\n\n\nmod(x, y)\nrem(x, y, RoundDown)\n\nThe reduction of x modulo y, or equivalently, the remainder of x after floored division by y, i.e. x - y*fld(x,y) if computed without intermediate rounding.\n\nThe result will have the same sign as y, and magnitude less than abs(y) (with some exceptions, see note below).\n\nnote: Note\nWhen used with floating point values, the exact result may not be representable by the type, and so rounding error may occur. In particular, if the exact result is very close to y, then it may be rounded to y.\n\nSee also: rem, div, fld, mod1, invmod.\n\njulia> mod(8, 3)\n2\n\njulia> mod(9, 3)\n0\n\njulia> mod(8.9, 3)\n2.9000000000000004\n\njulia> mod(eps(), 3)\n2.220446049250313e-16\n\njulia> mod(-eps(), 3)\n3.0\n\njulia> mod.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  0  1  2  0  1  2  0  1  2\n\n\n\n\n\nmod(x::Integer, r::AbstractUnitRange)\n\nFind y in the range r such that x ≡ y (mod n), where n = length(r), i.e. y = mod(x - first(r), n) + first(r).\n\nSee also mod1.\n\nExamples\n\njulia> mod(0, Base.OneTo(3))  # mod1(0, 3)\n3\n\njulia> mod(3, 0:2)  # mod(3, 3)\n0\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.rem","location":"base/math.html#Base.rem","category":"function","text":"rem(x, y)\n%(x, y)\n\nRemainder from Euclidean division, returning a value of the same sign as x, and smaller in magnitude than y. This value is always exact.\n\nSee also: div, mod, mod1, divrem.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> x % y\n3\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\njulia> rem.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n -2  -1  0  -2  -1  0  1  2  0  1  2\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.rem","location":"base/math.html#Base.rem-Tuple{Any, Any, RoundingMode}","category":"method","text":"rem(x, y, r::RoundingMode=RoundToZero)\n\nCompute the remainder of x after integer division by y, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - y * round(x / y, r)\n\nwithout any intermediate rounding.\n\nif r == RoundNearest, then the result is exact, and in the interval -y  2 y  2. See also RoundNearest.\nif r == RoundToZero (default), then the result is exact, and in the interval 0 y) if x is positive, or (-y 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 y) if y is positive, or (y 0 otherwise. The result may not be exact if x and y have different signs, and abs(x) < abs(y). See also RoundDown.\nif r == RoundUp, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundUp.\nif r == RoundFromZero, then the result is in the interval (-y 0 if y is positive, or 0 -y) otherwise. The result may not be exact if x and y have the same sign, and abs(x) < abs(y). See also RoundFromZero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9.\n\nExamples:\n\njulia> x = 9; y = 4;\n\njulia> x % y  # same as rem(x, y)\n1\n\njulia> x ÷ y  # same as div(x, y)\n2\n\njulia> x == div(x, y) * y + rem(x, y)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.rem2pi","location":"base/math.html#Base.Math.rem2pi","category":"function","text":"rem2pi(x, r::RoundingMode)\n\nCompute the remainder of x after integer division by 2π, with the quotient rounded according to the rounding mode r. In other words, the quantity\n\nx - 2π*round(x/(2π),r)\n\nwithout any intermediate rounding. This internally uses a high precision approximation of 2π, and so will give a more accurate result than rem(x,2π,r)\n\nif r == RoundNearest, then the result is in the interval -π π. This will generally be the most accurate result. See also RoundNearest.\nif r == RoundToZero, then the result is in the interval 0 2π if x is positive,. or -2π 0 otherwise. See also RoundToZero.\nif r == RoundDown, then the result is in the interval 0 2π. See also RoundDown.\nif r == RoundUp, then the result is in the interval -2π 0. See also RoundUp.\n\nExamples\n\njulia> rem2pi(7pi/4, RoundNearest)\n-0.7853981633974485\n\njulia> rem2pi(7pi/4, RoundDown)\n5.497787143782138\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.mod2pi","location":"base/math.html#Base.Math.mod2pi","category":"function","text":"mod2pi(x)\n\nModulus after division by 2π, returning in the range 02π).\n\nThis function computes a floating point representation of the modulus after division by numerically exact 2π, and is therefore not exactly the same as mod(x,2π), which would compute the modulus of x relative to division by the floating-point number 2π.\n\nnote: Note\nDepending on the format of the input value, the closest representable value to 2π may be less than 2π. For example, the expression mod2pi(2π) will not return 0, because the intermediate value of 2*π is a Float64 and 2*Float64(π) < 2*big(π). See rem2pi for more refined control of this behavior.\n\nExamples\n\njulia> mod2pi(9*pi/4)\n0.7853981633974481\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.divrem","location":"base/math.html#Base.divrem","category":"function","text":"divrem(x, y, r::RoundingMode=RoundToZero)\n\nThe quotient and remainder from Euclidean division. Equivalent to (div(x, y, r), rem(x, y, r)). Equivalently, with the default value of r, this call is equivalent to (x ÷ y, x % y).\n\nSee also: fldmod, cld.\n\nExamples\n\njulia> divrem(3, 7)\n(0, 3)\n\njulia> divrem(7, 3)\n(2, 1)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fldmod","location":"base/math.html#Base.fldmod","category":"function","text":"fldmod(x, y)\n\nThe floored quotient and modulus after division. A convenience wrapper for divrem(x, y, RoundDown). Equivalent to (fld(x, y), mod(x, y)).\n\nSee also: fld, cld, fldmod1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fld1","location":"base/math.html#Base.fld1","category":"function","text":"fld1(x, y)\n\nFlooring division, returning a value consistent with mod1(x,y)\n\nSee also mod1, fldmod1.\n\nExamples\n\njulia> x = 15; y = 4;\n\njulia> fld1(x, y)\n4\n\njulia> x == fld(x, y) * y + mod(x, y)\ntrue\n\njulia> x == (fld1(x, y) - 1) * y + mod1(x, y)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.mod1","location":"base/math.html#Base.mod1","category":"function","text":"mod1(x, y)\n\nModulus after flooring division, returning a value r such that mod(r, y) == mod(x, y) in the range (0 y for positive y and in the range y0) for negative y.\n\nWith integer arguments and positive y, this is equal to mod(x, 1:y), and hence natural for 1-based indexing. By comparison, mod(x, y) == mod(x, 0:y-1) is natural for computations with offsets or strides.\n\nSee also mod, fld1, fldmod1.\n\nExamples\n\njulia> mod1(4, 2)\n2\n\njulia> mod1.(-5:5, 3)'\n1×11 adjoint(::Vector{Int64}) with eltype Int64:\n 1  2  3  1  2  3  1  2  3  1  2\n\njulia> mod1.([-0.1, 0, 0.1, 1, 2, 2.9, 3, 3.1]', 3)\n1×8 Matrix{Float64}:\n 2.9  3.0  0.1  1.0  2.0  2.9  3.0  0.1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.fldmod1","location":"base/math.html#Base.fldmod1","category":"function","text":"fldmod1(x, y)\n\nReturn (fld1(x,y), mod1(x,y)).\n\nSee also fld1, mod1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.://","location":"base/math.html#Base.://","category":"function","text":"//(num, den)\n\nDivide two integers or rational numbers, giving a Rational result. More generally, // can be used for exact rational division of other numeric types with integer or rational components, such as complex numbers with integer components.\n\nNote that floating-point (AbstractFloat) arguments are not permitted by // (even if the values are rational). The arguments must be subtypes of Integer, Rational, or composites thereof.\n\nExamples\n\njulia> 3 // 5\n3//5\n\njulia> (3 // 5) // (2 // 1)\n3//10\n\njulia> (1+2im) // (3+4im)\n11//25 + 2//25*im\n\njulia> 1.0 // 2\nERROR: MethodError: no method matching //(::Float64, ::Int64)\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.rationalize","location":"base/math.html#Base.rationalize","category":"function","text":"rationalize([T<:Integer=Int,] x; tol::Real=eps(x))\n\nApproximate floating point number x as a Rational number with components of the given integer type. The result will differ from x by no more than tol.\n\nExamples\n\njulia> rationalize(5.6)\n28//5\n\njulia> a = rationalize(BigInt, 10.3)\n103//10\n\njulia> typeof(numerator(a))\nBigInt\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.numerator","location":"base/math.html#Base.numerator","category":"function","text":"numerator(x)\n\nNumerator of the rational representation of x.\n\nExamples\n\njulia> numerator(2//3)\n2\n\njulia> numerator(4)\n4\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.denominator","location":"base/math.html#Base.denominator","category":"function","text":"denominator(x)\n\nDenominator of the rational representation of x.\n\nExamples\n\njulia> denominator(2//3)\n3\n\njulia> denominator(4)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<<","location":"base/math.html#Base.:<<","category":"function","text":"<<(B::BitVector, n) -> BitVector\n\nLeft bit shift operator, B << n. For n >= 0, the result is B with elements shifted n positions backwards, filling with false values. If n < 0, elements are shifted forwards. Equivalent to B >> -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B << 1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\njulia> B << -1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\n\n\n\n\n<<(x, n)\n\nLeft bit shift operator, x << n. For n >= 0, the result is x shifted left by n bits, filling with 0s. This is equivalent to x * 2^n. For n < 0, this is equivalent to x >> -n.\n\nExamples\n\njulia> Int8(3) << 2\n12\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> bitstring(Int8(12))\n\"00001100\"\n\nSee also >>, >>>, exp2, ldexp.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>>","location":"base/math.html#Base.:>>","category":"function","text":">>(B::BitVector, n) -> BitVector\n\nRight bit shift operator, B >> n. For n >= 0, the result is B with elements shifted n positions forward, filling with false values. If n < 0, elements are shifted backwards. Equivalent to B << -n.\n\nExamples\n\njulia> B = BitVector([true, false, true, false, false])\n5-element BitVector:\n 1\n 0\n 1\n 0\n 0\n\njulia> B >> 1\n5-element BitVector:\n 0\n 1\n 0\n 1\n 0\n\njulia> B >> -1\n5-element BitVector:\n 0\n 1\n 0\n 0\n 0\n\n\n\n\n\n>>(x, n)\n\nRight bit shift operator, x >> n. For n >= 0, the result is x shifted right by n bits, filling with 0s if x >= 0, 1s if x < 0, preserving the sign of x. This is equivalent to fld(x, 2^n). For n < 0, this is equivalent to x << -n.\n\nExamples\n\njulia> Int8(13) >> 2\n3\n\njulia> bitstring(Int8(13))\n\"00001101\"\n\njulia> bitstring(Int8(3))\n\"00000011\"\n\njulia> Int8(-14) >> 2\n-4\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(-4))\n\"11111100\"\n\nSee also >>>, <<.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>>>","location":"base/math.html#Base.:>>>","category":"function","text":">>>(B::BitVector, n) -> BitVector\n\nUnsigned right bitshift operator, B >>> n. Equivalent to B >> n. See >> for details and examples.\n\n\n\n\n\n>>>(x, n)\n\nUnsigned right bit shift operator, x >>> n. For n >= 0, the result is x shifted right by n bits, filling with 0s. For n < 0, this is equivalent to x << -n.\n\nFor Unsigned integer types, this is equivalent to >>. For Signed integer types, this is equivalent to signed(unsigned(x) >> n).\n\nExamples\n\njulia> Int8(-14) >>> 2\n60\n\njulia> bitstring(Int8(-14))\n\"11110010\"\n\njulia> bitstring(Int8(60))\n\"00111100\"\n\nBigInts are treated as if having infinite size, so no filling is required and this is equivalent to >>.\n\nSee also >>, <<.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.bitrotate","location":"base/math.html#Base.bitrotate","category":"function","text":"bitrotate(x::Base.BitInteger, k::Integer)\n\nbitrotate(x, k) implements bitwise rotation. It returns the value of x with its bits rotated left k times. A negative value of k will rotate to the right instead.\n\ncompat: Julia 1.5\nThis function requires Julia 1.5 or later.\n\nSee also: <<, circshift, BitArray.\n\njulia> bitrotate(UInt8(114), 2)\n0xc9\n\njulia> bitstring(bitrotate(0b01110010, 2))\n\"11001001\"\n\njulia> bitstring(bitrotate(0b01110010, -2))\n\"10011100\"\n\njulia> bitstring(bitrotate(0b01110010, 8))\n\"01110010\"\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.::","location":"base/math.html#Base.::","category":"function","text":":expr\n\nQuote an expression expr, returning the abstract syntax tree (AST) of expr. The AST may be of type Expr, Symbol, or a literal value. The syntax :identifier evaluates to a Symbol.\n\nSee also: Expr, Symbol, Meta.parse\n\nExamples\n\njulia> expr = :(a = b + 2*x)\n:(a = b + 2x)\n\njulia> sym = :some_identifier\n:some_identifier\n\njulia> value = :0xff\n0xff\n\njulia> typeof((expr, sym, value))\nTuple{Expr, Symbol, UInt8}\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.range","location":"base/math.html#Base.range","category":"function","text":"range(start, stop, length)\nrange(start, stop; length, step)\nrange(start; length, stop, step)\nrange(;start, length, stop, step)\n\nConstruct a specialized array with evenly spaced elements and optimized storage (an AbstractRange) from the arguments. Mathematically a range is uniquely determined by any three of start, step, stop and length. Valid invocations of range are:\n\nCall range with any three of start, step, stop, length.\nCall range with two of start, stop, length. In this case step will be assumed to be positive one. If both arguments are Integers, a UnitRange will be returned.\nCall range with one of stop or length. start and step will be assumed to be positive one.\n\nTo construct a descending range, specify a negative step size, e.g. range(5, 1; step = -1) => [5,4,3,2,1]. Otherwise, a stop value less than the start value, with the default step of +1, constructs an empty range. Empty ranges are normalized such that the stop is one less than the start, e.g. range(5, 1) == 5:4.\n\nSee Extended Help for additional details on the returned type. See also logrange for logarithmically spaced points.\n\nExamples\n\njulia> range(1, length=100)\n1:100\n\njulia> range(1, stop=100)\n1:100\n\njulia> range(1, step=5, length=100)\n1:5:496\n\njulia> range(1, step=5, stop=100)\n1:5:96\n\njulia> range(1, 10, length=101)\n1.0:0.09:10.0\n\njulia> range(1, 100, step=5)\n1:5:96\n\njulia> range(stop=10, length=5)\n6:10\n\njulia> range(stop=10, step=1, length=5)\n6:1:10\n\njulia> range(start=1, step=1, stop=10)\n1:1:10\n\njulia> range(; length = 10)\nBase.OneTo(10)\n\njulia> range(; stop = 6)\nBase.OneTo(6)\n\njulia> range(; stop = 6.5)\n1.0:1.0:6.0\n\nIf length is not specified and stop - start is not an integer multiple of step, a range that ends before stop will be produced.\n\njulia> range(1, 3.5, step=2)\n1.0:2.0:3.0\n\nSpecial care is taken to ensure intermediate values are computed rationally. To avoid this induced overhead, see the LinRange constructor.\n\ncompat: Julia 1.1\nstop as a positional argument requires at least Julia 1.1.\n\ncompat: Julia 1.7\nThe versions without keyword arguments and start as a keyword argument require at least Julia 1.7.\n\ncompat: Julia 1.8\nThe versions with stop as a sole keyword argument, or length as a sole keyword argument require at least Julia 1.8.\n\nExtended Help\n\nrange will produce a Base.OneTo when the arguments are Integers and\n\nOnly length is provided\nOnly stop is provided\n\nrange will produce a UnitRange when the arguments are Integers and\n\nOnly start  and stop are provided\nOnly length and stop are provided\n\nA UnitRange is not produced if step is provided even if specified as one.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.OneTo","location":"base/math.html#Base.OneTo","category":"type","text":"Base.OneTo(n)\n\nDefine an AbstractUnitRange that behaves like 1:n, with the added distinction that the lower limit is guaranteed (by the type system) to be 1.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.StepRangeLen","location":"base/math.html#Base.StepRangeLen","category":"type","text":"StepRangeLen(         ref::R, step::S, len, [offset=1]) where {  R,S}\nStepRangeLen{T,R,S}(  ref::R, step::S, len, [offset=1]) where {T,R,S}\nStepRangeLen{T,R,S,L}(ref::R, step::S, len, [offset=1]) where {T,R,S,L}\n\nA range r where r[i] produces values of type T (in the first form, T is deduced automatically), parameterized by a reference value, a step, and the length. By default ref is the starting value r[1], but alternatively you can supply it as the value of r[offset] for some other index 1 <= offset <= len. The syntax a:b or a:b:c, where any of a, b, or c are floating-point numbers, creates a StepRangeLen.\n\ncompat: Julia 1.7\nThe 4th type parameter L requires at least Julia 1.7.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.logrange","location":"base/math.html#Base.logrange","category":"function","text":"logrange(start, stop, length)\nlogrange(start, stop; length)\n\nConstruct a specialized array whose elements are spaced logarithmically between the given endpoints. That is, the ratio of successive elements is a constant, calculated from the length.\n\nThis is similar to geomspace in Python. Unlike PowerRange in Mathematica, you specify the number of elements not the ratio. Unlike logspace in Python and Matlab, the start and stop arguments are always the first and last elements of the result, not powers applied to some base.\n\nExamples\n\njulia> logrange(10, 4000, length=3)\n3-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 10.0, 200.0, 4000.0\n\njulia> ans[2] ≈ sqrt(10 * 4000)  # middle element is the geometric mean\ntrue\n\njulia> range(10, 40, length=3)[2] ≈ (10 + 40)/2  # arithmetic mean\ntrue\n\njulia> logrange(1f0, 32f0, 11)\n11-element Base.LogRange{Float32, Float64}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0, 5.65685, 8.0, 11.3137, 16.0, 22.6274, 32.0\n\njulia> logrange(1, 1000, length=4) ≈ 10 .^ (0:3)\ntrue\n\nSee the LogRange type for further details.\n\nSee also range for linearly spaced points.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.LogRange","location":"base/math.html#Base.LogRange","category":"type","text":"LogRange{T}(start, stop, len) <: AbstractVector{T}\n\nA range whose elements are spaced logarithmically between start and stop, with spacing controlled by len. Returned by logrange.\n\nLike LinRange, the first and last elements will be exactly those provided, but intermediate values may have small floating-point errors. These are calculated using the logs of the endpoints, which are stored on construction, often in higher precision than T.\n\nExamples\n\njulia> logrange(1, 4, length=5)\n5-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 1.41421, 2.0, 2.82843, 4.0\n\njulia> Base.LogRange{Float16}(1, 4, 5)\n5-element Base.LogRange{Float16, Float64}:\n 1.0, 1.414, 2.0, 2.828, 4.0\n\njulia> logrange(1e-310, 1e-300, 11)[1:2:end]\n6-element Vector{Float64}:\n 1.0e-310\n 9.999999999999974e-309\n 9.999999999999981e-307\n 9.999999999999988e-305\n 9.999999999999994e-303\n 1.0e-300\n\njulia> prevfloat(1e-308, 5) == ans[2]\ntrue\n\nNote that integer eltype T is not allowed. Use for instance round.(Int, xs), or explicit powers of some integer base:\n\njulia> xs = logrange(1, 512, 4)\n4-element Base.LogRange{Float64, Base.TwicePrecision{Float64}}:\n 1.0, 8.0, 64.0, 512.0\n\njulia> 2 .^ (0:3:9) |> println\n[1, 8, 64, 512]\n\ncompat: Julia 1.11\nThis type requires at least Julia 1.11.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:==","location":"base/math.html#Base.:==","category":"function","text":"==(x, y)\n\nGeneric equality operator. Falls back to ===. Should be implemented for all types with a notion of equality, based on the abstract value that an instance represents. For example, all numeric types are compared by numeric value, ignoring type. Strings are compared as sequences of characters, ignoring encoding. Collections of the same type generally compare their key sets, and if those are ==, then compare the values for each of those keys, returning true if all such pairs are ==. Other properties are typically not taken into account (such as the exact type).\n\nThis operator follows IEEE semantics for floating-point numbers: 0.0 == -0.0 and NaN != NaN.\n\nThe result is of type Bool, except when one of the operands is missing, in which case missing is returned (three-valued logic). Collections generally implement three-valued logic akin to all, returning missing if any operands contain missing values and all other pairs are equal. Use isequal or === to always get a Bool result.\n\nImplementation\n\nNew numeric types should implement this function for two arguments of the new type, and handle comparison to other types via promotion rules where possible.\n\nEquality and hashing are intimately related; two values that are considered isequal must have the same hash and by default isequal falls back to ==. If a type customizes the behavior of == and/or isequal, then hash must be similarly implemented to ensure isequal and hash agree. Sets, Dicts, and many other internal implementations assume that this invariant holds.\n\nIf some type defines ==, isequal, and isless then it should also implement < to ensure consistency of comparisons.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:!=","location":"base/math.html#Base.:!=","category":"function","text":"!=(x)\n\nCreate a function that compares its argument to x using !=, i.e. a function equivalent to y -> y != x. The returned function is of type Base.Fix2{typeof(!=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n!=(x, y)\n≠(x,y)\n\nNot-equals comparison operator. Always gives the opposite answer as ==.\n\nImplementation\n\nNew types should generally not implement this, and rely on the fallback definition !=(x,y) = !(x==y) instead.\n\nExamples\n\njulia> 3 != 2\ntrue\n\njulia> \"foo\" ≠ \"foo\"\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Core.:!==","location":"base/math.html#Core.:!==","category":"function","text":"!==(x, y)\n≢(x,y)\n\nAlways gives the opposite answer as ===.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a ≢ b\ntrue\n\njulia> a ≢ a\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<","location":"base/math.html#Base.:<","category":"function","text":"<(x)\n\nCreate a function that compares its argument to x using <, i.e. a function equivalent to y -> y < x. The returned function is of type Base.Fix2{typeof(<)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<(x, y)\n\nLess-than comparison operator. Falls back to isless. Because of the behavior of floating-point NaN values, this operator implements a partial order.\n\nImplementation\n\nNew types with a canonical partial order should implement this function for two arguments of the new type. Types with a canonical total order should implement isless instead.\n\nSee also isunordered.\n\nExamples\n\njulia> 'a' < 'b'\ntrue\n\njulia> \"abc\" < \"abd\"\ntrue\n\njulia> 5 < 3\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:<=","location":"base/math.html#Base.:<=","category":"function","text":"<=(x)\n\nCreate a function that compares its argument to x using <=, i.e. a function equivalent to y -> y <= x. The returned function is of type Base.Fix2{typeof(<=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n<=(x, y)\n≤(x,y)\n\nLess-than-or-equals comparison operator. Falls back to (x < y) | (x == y).\n\nExamples\n\njulia> 'a' <= 'b'\ntrue\n\njulia> 7 ≤ 7 ≤ 9\ntrue\n\njulia> \"abc\" ≤ \"abc\"\ntrue\n\njulia> 5 <= 3\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>","location":"base/math.html#Base.:>","category":"function","text":">(x)\n\nCreate a function that compares its argument to x using >, i.e. a function equivalent to y -> y > x. The returned function is of type Base.Fix2{typeof(>)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>(x, y)\n\nGreater-than comparison operator. Falls back to y < x.\n\nImplementation\n\nGenerally, new types should implement < instead of this function, and rely on the fallback definition >(x, y) = y < x.\n\nExamples\n\njulia> 'a' > 'b'\nfalse\n\njulia> 7 > 3 > 1\ntrue\n\njulia> \"abc\" > \"abd\"\nfalse\n\njulia> 5 > 3\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:>=","location":"base/math.html#Base.:>=","category":"function","text":">=(x)\n\nCreate a function that compares its argument to x using >=, i.e. a function equivalent to y -> y >= x. The returned function is of type Base.Fix2{typeof(>=)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n>=(x, y)\n≥(x,y)\n\nGreater-than-or-equals comparison operator. Falls back to y <= x.\n\nExamples\n\njulia> 'a' >= 'b'\nfalse\n\njulia> 7 ≥ 7 ≥ 3\ntrue\n\njulia> \"abc\" ≥ \"abc\"\ntrue\n\njulia> 5 >= 3\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cmp","location":"base/math.html#Base.cmp","category":"function","text":"cmp(a::AbstractString, b::AbstractString) -> Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\ncmp(<, x, y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. The first argument specifies a less-than comparison function to use.\n\n\n\n\n\ncmp(x,y)\n\nReturn -1, 0, or 1 depending on whether x is less than, equal to, or greater than y, respectively. Uses the total order implemented by isless.\n\nExamples\n\njulia> cmp(1, 2)\n-1\n\njulia> cmp(2, 1)\n1\n\njulia> cmp(2+im, 3-im)\nERROR: MethodError: no method matching isless(::Complex{Int64}, ::Complex{Int64})\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:~","location":"base/math.html#Base.:~","category":"function","text":"~(x)\n\nBitwise not.\n\nSee also: !, &, |.\n\nExamples\n\njulia> ~4\n-5\n\njulia> ~10\n-11\n\njulia> ~true\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:&","location":"base/math.html#Base.:&","category":"function","text":"x & y\n\nBitwise and. Implements three-valued logic, returning missing if one operand is missing and the other is true. Add parentheses for function application form: (&)(x, y).\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> 4 & 10\n0\n\njulia> 4 & 12\n4\n\njulia> true & missing\nmissing\n\njulia> false & missing\nfalse\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:|","location":"base/math.html#Base.:|","category":"function","text":"x | y\n\nBitwise or. Implements three-valued logic, returning missing if one operand is missing and the other is false.\n\nSee also: &, xor, ||.\n\nExamples\n\njulia> 4 | 10\n14\n\njulia> 4 | 1\n5\n\njulia> true | missing\ntrue\n\njulia> false | missing\nmissing\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.xor","location":"base/math.html#Base.xor","category":"function","text":"xor(x, y)\n⊻(x, y)\n\nBitwise exclusive or of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊻ b is a synonym for xor(a,b), and ⊻ can be typed by tab-completing \\xor or \\veebar in the Julia REPL.\n\nExamples\n\njulia> xor(true, false)\ntrue\n\njulia> xor(true, true)\nfalse\n\njulia> xor(true, missing)\nmissing\n\njulia> false ⊻ false\nfalse\n\njulia> [true; true; false] .⊻ [true; false; false]\n3-element BitVector:\n 0\n 1\n 0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nand","location":"base/math.html#Base.nand","category":"function","text":"nand(x, y)\n⊼(x, y)\n\nBitwise nand (not and) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing.\n\nThe infix operation a ⊼ b is a synonym for nand(a,b), and ⊼ can be typed by tab-completing \\nand or \\barwedge in the Julia REPL.\n\nExamples\n\njulia> nand(true, false)\ntrue\n\njulia> nand(true, true)\nfalse\n\njulia> nand(true, missing)\nmissing\n\njulia> false ⊼ false\ntrue\n\njulia> [true; true; false] .⊼ [true; false; false]\n3-element BitVector:\n 0\n 1\n 1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nor","location":"base/math.html#Base.nor","category":"function","text":"nor(x, y)\n⊽(x, y)\n\nBitwise nor (not or) of x and y. Implements three-valued logic, returning missing if one of the arguments is missing and the other is not true.\n\nThe infix operation a ⊽ b is a synonym for nor(a,b), and ⊽ can be typed by tab-completing \\nor or \\barvee in the Julia REPL.\n\nExamples\n\njulia> nor(true, false)\nfalse\n\njulia> nor(true, true)\nfalse\n\njulia> nor(true, missing)\nfalse\n\njulia> false ⊽ false\ntrue\n\njulia> false ⊽ missing\nmissing\n\njulia> [true; true; false] .⊽ [true; false; false]\n3-element BitVector:\n 0\n 0\n 1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.:!","location":"base/math.html#Base.:!","category":"function","text":"!f::Function\n\nPredicate function negation: when the argument of ! is a function, it returns a composed function which computes the boolean negation of f.\n\nSee also ∘.\n\nExamples\n\njulia> str = \"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\"∀ ε > 0, ∃ δ > 0: |x-y| < δ ⇒ |f(x)-f(y)| < ε\"\n\njulia> filter(isletter, str)\n\"εδxyδfxfyε\"\n\njulia> filter(!isletter, str)\n\"∀  > 0, ∃  > 0: |-| <  ⇒ |()-()| < \"\n\ncompat: Julia 1.9\nStarting with Julia 1.9, !f returns a ComposedFunction instead of an anonymous function.\n\n\n\n\n\n!(x)\n\nBoolean not. Implements three-valued logic, returning missing if x is missing.\n\nSee also ~ for bitwise not.\n\nExamples\n\njulia> !true\nfalse\n\njulia> !false\ntrue\n\njulia> !missing\nmissing\n\njulia> .![true false true]\n1×3 BitMatrix:\n 0  1  0\n\n\n\n\n\n","page":"Mathematics"},{"title":"&&","location":"base/math.html#&&","category":"keyword","text":"x && y\n\nShort-circuiting boolean AND.\n\nThis is equivalent to x ? y : false: it returns false if x is false and the result of evaluating y if x is true. Note that if y is an expression, it is only evaluated when x is true, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) && (statement) can be used as shorthand for if condition; statement; end for an arbitrary statement.\n\nSee also &, the ternary operator ? :, and the manual section on control flow.\n\nExamples\n\njulia> x = 3;\n\njulia> x > 1 && x < 10 && x isa Int\ntrue\n\njulia> x < 0 && error(\"expected positive x\")\nfalse\n\njulia> x > 0 && \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n","page":"Mathematics"},{"title":"||","location":"base/math.html#||","category":"keyword","text":"x || y\n\nShort-circuiting boolean OR.\n\nThis is equivalent to x ? true : y: it returns true if x is true and the result of evaluating y if x is false. Note that if y is an expression, it is only evaluated when x is false, which is called \"short-circuiting\" behavior.\n\nAlso, y does not need to have a boolean value.  This means that (condition) || (statement) can be used as shorthand for if !(condition); statement; end for an arbitrary statement.\n\nSee also: |, xor, &&.\n\nExamples\n\njulia> pi < 3 || ℯ < 3\ntrue\n\njulia> false || true || println(\"neither is true!\")\ntrue\n\njulia> pi < 3 || \"not a boolean\"\n\"not a boolean\"\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.isapprox","location":"base/math.html#Base.isapprox","category":"function","text":"isapprox(x; kwargs...) / ≈(x; kwargs...)\n\nCreate a function that compares its argument to x using ≈, i.e. a function equivalent to y -> y ≈ x.\n\nThe keyword arguments supported here are the same as those in the 2-argument isapprox.\n\ncompat: Julia 1.5\nThis method requires Julia 1.5 or later.\n\n\n\n\n\nisapprox(x, y; atol::Real=0, rtol::Real=atol>0 ? 0 : √eps, nans::Bool=false[, norm::Function])\n\nInexact equality comparison. Two numbers compare equal if their relative distance or their absolute distance is within tolerance bounds: isapprox returns true if norm(x-y) <= max(atol, rtol*max(norm(x), norm(y))). The default atol (absolute tolerance) is zero and the default rtol (relative tolerance) depends on the types of x and y. The keyword argument nans determines whether or not NaN values are considered equal (defaults to false).\n\nFor real or complex floating-point values, if an atol > 0 is not specified, rtol defaults to the square root of eps of the type of x or y, whichever is bigger (least precise). This corresponds to requiring equality of about half of the significant digits. Otherwise, e.g. for integer arguments or if an atol > 0 is supplied, rtol defaults to zero.\n\nThe norm keyword defaults to abs for numeric (x,y) and to LinearAlgebra.norm for arrays (where an alternative norm choice is sometimes useful). When x and y are arrays, if norm(x-y) is not finite (i.e. ±Inf or NaN), the comparison falls back to checking whether all elements of x and y are approximately equal component-wise.\n\nThe binary operator ≈ is equivalent to isapprox with the default arguments, and x ≉ y is equivalent to !isapprox(x,y).\n\nNote that x ≈ 0 (i.e., comparing to zero with the default tolerances) is equivalent to x == 0 since the default atol is 0.  In such cases, you should either supply an appropriate atol (or use norm(x) ≤ atol) or rearrange your code (e.g. use x ≈ y rather than x - y ≈ 0).   It is not possible to pick a nonzero atol automatically because it depends on the overall scaling (the \"units\") of your problem: for example, in x - y ≈ 0, atol=1e-9 is an absurdly small tolerance if x is the radius of the Earth in meters, but an absurdly large tolerance if x is the radius of a Hydrogen atom in meters.\n\ncompat: Julia 1.6\nPassing the norm keyword argument when comparing numeric (non-array) arguments requires Julia 1.6 or later.\n\nExamples\n\njulia> isapprox(0.1, 0.15; atol=0.05)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.34)\ntrue\n\njulia> isapprox(0.1, 0.15; rtol=0.33)\nfalse\n\njulia> 0.1 + 1e-10 ≈ 0.1\ntrue\n\njulia> 1e-10 ≈ 0\nfalse\n\njulia> isapprox(1e-10, 0, atol=1e-8)\ntrue\n\njulia> isapprox([10.0^9, 1.0], [10.0^9, 2.0]) # using `norm`\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sin","location":"base/math.html#Base.sin-Tuple{Number}","category":"method","text":"sin(x::T) where {T <: Number} -> float(T)\n\nCompute sine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, sinpi, sincos, cis, asin.\n\nExamples\n\njulia> round.(sin.(range(0, 2pi, length=9)'), digits=3)\n1×9 Matrix{Float64}:\n 0.0  0.707  1.0  0.707  0.0  -0.707  -1.0  -0.707  -0.0\n\njulia> sind(45)\n0.7071067811865476\n\njulia> sinpi(1/4)\n0.7071067811865475\n\njulia> round.(sincos(pi/6), digits=3)\n(0.5, 0.866)\n\njulia> round(cis(pi/6), digits=3)\n0.866 + 0.5im\n\njulia> round(exp(im*pi/6), digits=3)\n0.866 + 0.5im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cos","location":"base/math.html#Base.cos-Tuple{Number}","category":"method","text":"cos(x::T) where {T <: Number} -> float(T)\n\nCompute cosine of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also cosd, cospi, sincos, cis.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sincos","location":"base/math.html#Base.Math.sincos-Tuple{Float64}","category":"method","text":"sincos(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in radians, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) if isnan(x).\n\nSee also cis, sincospi, sincosd.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.tan","location":"base/math.html#Base.tan-Tuple{Number}","category":"method","text":"tan(x::T) where {T <: Number} -> float(T)\n\nCompute tangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also tanh.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sind","location":"base/math.html#Base.Math.sind","category":"function","text":"sind(x::T) where T -> float(T)\n\nCompute sine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cosd","location":"base/math.html#Base.Math.cosd","category":"function","text":"cosd(x::T) where T -> float(T)\n\nCompute cosine of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.tand","location":"base/math.html#Base.Math.tand","category":"function","text":"tand(x::T) where T -> float(T)\n\nCompute tangent of x, where x is in degrees. If x is a matrix, x needs to be a square matrix.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sincosd","location":"base/math.html#Base.Math.sincosd","category":"function","text":"sincosd(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute the sine and cosine of x, where x is in degrees, returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sinpi","location":"base/math.html#Base.Math.sinpi","category":"function","text":"sinpi(x::T) where T -> float(T)\n\nCompute sin(pi x) more accurately than sin(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also sind, cospi, sincospi.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cospi","location":"base/math.html#Base.Math.cospi","category":"function","text":"cospi(x::T) where T -> float(T)\n\nCompute cos(pi x) more accurately than cos(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\nSee also: cispi, sincosd, cospi.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.tanpi","location":"base/math.html#Base.Math.tanpi","category":"function","text":"tanpi(x::T) where T -> float(T)\n\nCompute tan(pi x) more accurately than tan(pi*x), especially for large x.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also tand, sinpi, cospi, sincospi.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sincospi","location":"base/math.html#Base.Math.sincospi","category":"function","text":"sincospi(x::T) where T -> Tuple{float(T),float(T)}\n\nSimultaneously compute sinpi(x) and cospi(x) (the sine and cosine of π*x, where x is in radians), returning a tuple (sine, cosine).\n\nThrow a DomainError if isinf(x), return a (T(NaN), T(NaN)) tuple if isnan(x).\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\nSee also: cispi, sincosd, sinpi.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sinh","location":"base/math.html#Base.sinh-Tuple{Number}","category":"method","text":"sinh(x)\n\nCompute hyperbolic sine of x.\n\nSee also sin.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cosh","location":"base/math.html#Base.cosh-Tuple{Number}","category":"method","text":"cosh(x)\n\nCompute hyperbolic cosine of x.\n\nSee also cos.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.tanh","location":"base/math.html#Base.tanh-Tuple{Number}","category":"method","text":"tanh(x)\n\nCompute hyperbolic tangent of x.\n\nSee also tan, atanh.\n\nExamples\n\njulia> tanh.(-3:3f0)  # Here 3f0 isa Float32\n7-element Vector{Float32}:\n -0.9950548\n -0.9640276\n -0.7615942\n  0.0\n  0.7615942\n  0.9640276\n  0.9950548\n\njulia> tan.(im .* (1:3))\n3-element Vector{ComplexF64}:\n 0.0 + 0.7615941559557649im\n 0.0 + 0.9640275800758169im\n 0.0 + 0.9950547536867306im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.asin","location":"base/math.html#Base.asin-Tuple{Number}","category":"method","text":"asin(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse sine of x, where the output is in radians.\n\nReturn a T(NaN) if isnan(x).\n\nSee also asind for output in degrees.\n\nExamples\n\njulia> asin.((0, 1/2, 1))\n(0.0, 0.5235987755982989, 1.5707963267948966)\n\njulia> asind.((0, 1/2, 1))\n(0.0, 30.000000000000004, 90.0)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.acos","location":"base/math.html#Base.acos-Tuple{Number}","category":"method","text":"acos(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosine of x, where the output is in radians\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.atan","location":"base/math.html#Base.atan-Tuple{Number}","category":"method","text":"atan(y)\natan(y, x)\n\nCompute the inverse tangent of y or y/x, respectively.\n\nFor one real argument, this is the angle in radians between the positive x-axis and the point (1, y), returning a value in the interval -pi2 pi2.\n\nFor two arguments, this is the angle in radians between the positive x-axis and the point (x, y), returning a value in the interval -pi pi. This corresponds to a standard atan2 function. Note that by convention atan(0.0,x) is defined as pi and atan(-0.0,x) is defined as -pi when x < 0.\n\nSee also atand for degrees.\n\nExamples\n\njulia> rad2deg(atan(-1/√3))\n-30.000000000000004\n\njulia> rad2deg(atan(-1, √3))\n-30.000000000000004\n\njulia> rad2deg(atan(1, -√3))\n150.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asind","location":"base/math.html#Base.Math.asind","category":"function","text":"asind(x)\n\nCompute the inverse sine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acosd","location":"base/math.html#Base.Math.acosd","category":"function","text":"acosd(x)\n\nCompute the inverse cosine of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.atand","location":"base/math.html#Base.Math.atand","category":"function","text":"atand(y::T) where T -> float(T)\natand(y::T, x::S) where {T,S} -> promote_type(T,S)\natand(y::AbstractMatrix{T}) where T -> AbstractMatrix{Complex{float(T)}}\n\nCompute the inverse tangent of y or y/x, respectively, where the output is in degrees.\n\nReturn a NaN if isnan(y) or isnan(x). The returned NaN is either a T in the single argument version, or a promote_type(T,S) in the two argument version.\n\ncompat: Julia 1.7\nThe one-argument method supports square matrix arguments as of Julia 1.7.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sec","location":"base/math.html#Base.Math.sec-Tuple{Number}","category":"method","text":"sec(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.csc","location":"base/math.html#Base.Math.csc-Tuple{Number}","category":"method","text":"csc(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cot","location":"base/math.html#Base.Math.cot-Tuple{Number}","category":"method","text":"cot(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in radians.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.secd","location":"base/math.html#Base.Math.secd","category":"function","text":"secd(x::T) where {T <: Number} -> float(T)\n\nCompute the secant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cscd","location":"base/math.html#Base.Math.cscd","category":"function","text":"cscd(x::T) where {T <: Number} -> float(T)\n\nCompute the cosecant of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cotd","location":"base/math.html#Base.Math.cotd","category":"function","text":"cotd(x::T) where {T <: Number} -> float(T)\n\nCompute the cotangent of x, where x is in degrees.\n\nThrow a DomainError if isinf(x), return a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asec","location":"base/math.html#Base.Math.asec-Tuple{Number}","category":"method","text":"asec(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse secant of x, where the output is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acsc","location":"base/math.html#Base.Math.acsc-Tuple{Number}","category":"method","text":"acsc(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cosecant of x, where the output is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acot","location":"base/math.html#Base.Math.acot-Tuple{Number}","category":"method","text":"acot(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse cotangent of x, where the output is in radians.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asecd","location":"base/math.html#Base.Math.asecd","category":"function","text":"asecd(x)\n\nCompute the inverse secant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acscd","location":"base/math.html#Base.Math.acscd","category":"function","text":"acscd(x)\n\nCompute the inverse cosecant of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acotd","location":"base/math.html#Base.Math.acotd","category":"function","text":"acotd(x)\n\nCompute the inverse cotangent of x, where the output is in degrees. If x is a matrix, x needs to be a square matrix.\n\ncompat: Julia 1.7\nMatrix arguments require Julia 1.7 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sech","location":"base/math.html#Base.Math.sech-Tuple{Number}","category":"method","text":"sech(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic secant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.csch","location":"base/math.html#Base.Math.csch-Tuple{Number}","category":"method","text":"csch(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cosecant of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.coth","location":"base/math.html#Base.Math.coth-Tuple{Number}","category":"method","text":"coth(x::T) where {T <: Number} -> float(T)\n\nCompute the hyperbolic cotangent of x.\n\nReturn a T(NaN) if isnan(x).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.asinh","location":"base/math.html#Base.asinh-Tuple{Number}","category":"method","text":"asinh(x)\n\nCompute the inverse hyperbolic sine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.acosh","location":"base/math.html#Base.acosh-Tuple{Number}","category":"method","text":"acosh(x)\n\nCompute the inverse hyperbolic cosine of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.atanh","location":"base/math.html#Base.atanh-Tuple{Number}","category":"method","text":"atanh(x)\n\nCompute the inverse hyperbolic tangent of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.asech","location":"base/math.html#Base.Math.asech-Tuple{Number}","category":"method","text":"asech(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic secant of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acsch","location":"base/math.html#Base.Math.acsch-Tuple{Number}","category":"method","text":"acsch(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cosecant of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.acoth","location":"base/math.html#Base.Math.acoth-Tuple{Number}","category":"method","text":"acoth(x::T) where {T <: Number} -> float(T)\n\nCompute the inverse hyperbolic cotangent of x.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.sinc","location":"base/math.html#Base.Math.sinc","category":"function","text":"sinc(x::T) where {T <: Number} -> float(T)\n\nCompute normalized sinc function operatornamesinc(x) = sin(pi x)  (pi x) if x neq 0, and 1 if x = 0.\n\nReturn a T(NaN) if isnan(x).\n\nSee also cosc, its derivative.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cosc","location":"base/math.html#Base.Math.cosc","category":"function","text":"cosc(x::T) where {T <: Number} -> float(T)\n\nCompute cos(pi x)  x - sin(pi x)  (pi x^2) if x neq 0, and 0 if x = 0. This is the derivative of sinc(x).\n\nReturn a T(NaN) if isnan(x).\n\nSee also sinc.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.deg2rad","location":"base/math.html#Base.Math.deg2rad","category":"function","text":"deg2rad(x)\n\nConvert x from degrees to radians.\n\nSee also rad2deg, sind, pi.\n\nExamples\n\njulia> deg2rad(90)\n1.5707963267948966\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.rad2deg","location":"base/math.html#Base.Math.rad2deg","category":"function","text":"rad2deg(x)\n\nConvert x from radians to degrees.\n\nSee also deg2rad.\n\nExamples\n\njulia> rad2deg(pi)\n180.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.hypot","location":"base/math.html#Base.Math.hypot","category":"function","text":"hypot(x, y)\n\nCompute the hypotenuse sqrtx^2+y^2 avoiding overflow and underflow.\n\nThis code is an implementation of the algorithm described in: An Improved Algorithm for hypot(a,b) by Carlos F. Borges The article is available online at arXiv at the link   https://arxiv.org/abs/1904.09481\n\nhypot(x...)\n\nCompute the hypotenuse sqrtsum x_i^2 avoiding overflow and underflow.\n\nSee also norm in the LinearAlgebra standard library.\n\nExamples\n\njulia> a = Int64(10)^10;\n\njulia> hypot(a, a)\n1.4142135623730951e10\n\njulia> √(a^2 + a^2) # a^2 overflows\nERROR: DomainError with -2.914184810805068e18:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> hypot(3, 4im)\n5.0\n\njulia> hypot(-5.7)\n5.7\n\njulia> hypot(3, 4im, 12.0)\n13.0\n\njulia> using LinearAlgebra\n\njulia> norm([a, a, a, a]) == hypot(a, a, a, a)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log","location":"base/math.html#Base.log-Tuple{Number}","category":"method","text":"log(x)\n\nCompute the natural logarithm of x.\n\nThrow a DomainError for negative Real arguments. Use Complex arguments to obtain Complex results.\n\nnote: Branch cut\nlog has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also ℯ, log1p, log2, log10.\n\nExamples\n\njulia> log(2)\n0.6931471805599453\n\njulia> log(-3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(-3 + 0im)\n1.0986122886681098 + 3.141592653589793im\n\njulia> log(-3 - 0.0im)\n1.0986122886681098 - 3.141592653589793im\n\njulia> log.(exp.(-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log","location":"base/math.html#Base.log-Tuple{Number, Number}","category":"method","text":"log(b,x)\n\nCompute the base b logarithm of x. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log(4,8)\n1.5\n\njulia> log(4,2)\n0.5\n\njulia> log(-2, 3)\nERROR: DomainError with -2.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\njulia> log(2, -3)\nERROR: DomainError with -3.0:\nlog was called with a negative real argument but will only return a complex result if called with a complex argument. Try log(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\nnote: Note\nIf b is a power of 2 or 10, log2 or log10 should be used, as these will typically be faster and more accurate. For example,julia> log(100,1000000)\n2.9999999999999996\n\njulia> log10(1000000)/2\n3.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log2","location":"base/math.html#Base.log2","category":"function","text":"log2(x)\n\nCompute the logarithm of x to base 2. Throw a DomainError for negative Real arguments.\n\nSee also: exp2, ldexp, ispow2.\n\nExamples\n\njulia> log2(4)\n2.0\n\njulia> log2(10)\n3.321928094887362\n\njulia> log2(-2)\nERROR: DomainError with -2.0:\nlog2 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log2(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\njulia> log2.(2.0 .^ (-1:1))\n3-element Vector{Float64}:\n -1.0\n  0.0\n  1.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log10","location":"base/math.html#Base.log10","category":"function","text":"log10(x)\n\nCompute the logarithm of x to base 10. Throw a DomainError for negative Real arguments.\n\nExamples\n\njulia> log10(100)\n2.0\n\njulia> log10(2)\n0.3010299956639812\n\njulia> log10(-2)\nERROR: DomainError with -2.0:\nlog10 was called with a negative real argument but will only return a complex result if called with a complex argument. Try log10(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(f::Symbol, x::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.log1p","location":"base/math.html#Base.log1p","category":"function","text":"log1p(x)\n\nAccurate natural logarithm of 1+x. Throw a DomainError for Real arguments less than -1.\n\nExamples\n\njulia> log1p(-0.5)\n-0.6931471805599453\n\njulia> log1p(0)\n0.0\n\njulia> log1p(-2)\nERROR: DomainError with -2.0:\nlog1p was called with a real argument < -1 but will only return a complex result if called with a complex argument. Try log1p(Complex(x)).\nStacktrace:\n [1] throw_complex_domainerror(::Symbol, ::Float64) at ./math.jl:31\n[...]\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.frexp","location":"base/math.html#Base.Math.frexp","category":"function","text":"frexp(val)\n\nReturn (x,exp) such that x has a magnitude in the interval 12 1) or 0, and val is equal to x times 2^exp.\n\nSee also significand, exponent, ldexp.\n\nExamples\n\njulia> frexp(6.0)\n(0.75, 3)\n\njulia> significand(6.0), exponent(6.0)  # interval [1, 2) instead\n(1.5, 2)\n\njulia> frexp(0.0), frexp(NaN), frexp(-Inf)  # exponent would give an error\n((0.0, 0), (NaN, 0), (-Inf, 0))\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp","location":"base/math.html#Base.exp-Tuple{Float64}","category":"method","text":"exp(x)\n\nCompute the natural base exponential of x, in other words ℯ^x.\n\nSee also exp2, exp10 and cis.\n\nExamples\n\njulia> exp(1.0)\n2.718281828459045\n\njulia> exp(im * pi) ≈ cis(pi)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp2","location":"base/math.html#Base.exp2","category":"function","text":"exp2(x)\n\nCompute the base 2 exponential of x, in other words 2^x.\n\nSee also ldexp, <<.\n\nExamples\n\njulia> exp2(5)\n32.0\n\njulia> 2^5\n32\n\njulia> exp2(63) > typemax(Int)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.exp10","location":"base/math.html#Base.exp10","category":"function","text":"exp10(x)\n\nCompute the base 10 exponential of x, in other words 10^x.\n\nExamples\n\njulia> exp10(2)\n100.0\n\njulia> 10^2\n100\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.ldexp","location":"base/math.html#Base.Math.ldexp","category":"function","text":"ldexp(x, n)\n\nCompute x times 2^n.\n\nSee also frexp, exponent.\n\nExamples\n\njulia> ldexp(5.0, 2)\n20.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.modf","location":"base/math.html#Base.Math.modf","category":"function","text":"modf(x)\n\nReturn a tuple (fpart, ipart) of the fractional and integral parts of a number. Both parts have the same sign as the argument.\n\nExamples\n\njulia> modf(3.5)\n(0.5, 3.0)\n\njulia> modf(-3.5)\n(-0.5, -3.0)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.expm1","location":"base/math.html#Base.expm1","category":"function","text":"expm1(x)\n\nAccurately compute e^x-1. It avoids the loss of precision involved in the direct evaluation of exp(x)-1 for small values of x.\n\nExamples\n\njulia> expm1(1e-16)\n1.0e-16\n\njulia> exp(1e-16) - 1\n0.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.round","location":"base/math.html#Base.round","category":"function","text":"round([T,] x, [r::RoundingMode])\nround(x, [r::RoundingMode]; digits::Integer=0, base = 10)\nround(x, [r::RoundingMode]; sigdigits::Integer, base = 10)\n\nRounds the number x.\n\nWithout keyword arguments, x is rounded to an integer value, returning a value of type T, or of the same type of x if no T is provided. An InexactError will be thrown if the value is not representable by T, similar to convert.\n\nIf the digits keyword argument is provided, it rounds to the specified number of digits after the decimal place (or before if negative), in base base.\n\nIf the sigdigits keyword argument is provided, it rounds to the specified number of significant digits, in base base.\n\nThe RoundingMode r controls the direction of the rounding; the default is RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer. Note that round may give incorrect results if the global rounding mode is changed (see rounding).\n\nWhen rounding to a floating point type, will round to integers representable by that type (and Inf) rather than true integers. Inf is treated as one ulp greater than the floatmax(T) for purposes of determining \"nearest\", similar to convert.\n\nExamples\n\njulia> round(1.7)\n2.0\n\njulia> round(Int, 1.7)\n2\n\njulia> round(1.5)\n2.0\n\njulia> round(2.5)\n2.0\n\njulia> round(pi; digits=2)\n3.14\n\njulia> round(pi; digits=3, base=2)\n3.125\n\njulia> round(123.456; sigdigits=2)\n120.0\n\njulia> round(357.913; sigdigits=4, base=2)\n352.0\n\njulia> round(Float16, typemax(UInt128))\nInf16\n\njulia> floor(Float16, typemax(UInt128))\nFloat16(6.55e4)\n\nnote: Note\nRounding to specified digits in bases other than 2 can be inexact when operating on binary floating point numbers. For example, the Float64 value represented by 1.15 is actually less than 1.15, yet will be rounded to 1.2. For example:julia> x = 1.15\n1.15\n\njulia> big(1.15)\n1.149999999999999911182158029987476766109466552734375\n\njulia> x < 115//100\ntrue\n\njulia> round(x, digits=1)\n1.2\n\nExtensions\n\nTo extend round to new numeric types, it is typically sufficient to define Base.round(x::NewType, r::RoundingMode).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundingMode","location":"base/math.html#Base.Rounding.RoundingMode","category":"type","text":"RoundingMode\n\nA type used for controlling the rounding mode of floating point operations (via rounding/setrounding functions), or as optional arguments for rounding to the nearest integer (via the round function).\n\nCurrently supported rounding modes are:\n\nRoundNearest (default)\nRoundNearestTiesAway\nRoundNearestTiesUp\nRoundToZero\nRoundFromZero\nRoundUp\nRoundDown\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearest","location":"base/math.html#Base.Rounding.RoundNearest","category":"constant","text":"RoundNearest\n\nThe default rounding mode. Rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearestTiesAway","location":"base/math.html#Base.Rounding.RoundNearestTiesAway","category":"constant","text":"RoundNearestTiesAway\n\nRounds to nearest integer, with ties rounded away from zero (C/C++ round behaviour).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundNearestTiesUp","location":"base/math.html#Base.Rounding.RoundNearestTiesUp","category":"constant","text":"RoundNearestTiesUp\n\nRounds to nearest integer, with ties rounded toward positive infinity (Java/JavaScript round behaviour).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundToZero","location":"base/math.html#Base.Rounding.RoundToZero","category":"constant","text":"RoundToZero\n\nround using this rounding mode is an alias for trunc.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundFromZero","location":"base/math.html#Base.Rounding.RoundFromZero","category":"constant","text":"RoundFromZero\n\nRounds away from zero.\n\ncompat: Julia 1.9\nRoundFromZero requires at least Julia 1.9. Prior versions support RoundFromZero for BigFloats only.\n\nExamples\n\njulia> BigFloat(\"1.0000000000000001\", 5, RoundFromZero)\n1.06\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundUp","location":"base/math.html#Base.Rounding.RoundUp","category":"constant","text":"RoundUp\n\nround using this rounding mode is an alias for ceil.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Rounding.RoundDown","location":"base/math.html#Base.Rounding.RoundDown","category":"constant","text":"RoundDown\n\nround using this rounding mode is an alias for floor.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.round","location":"base/math.html#Base.round-Tuple{Complex{<:AbstractFloat}, RoundingMode, RoundingMode}","category":"method","text":"round(z::Complex[, RoundingModeReal, [RoundingModeImaginary]])\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; digits=0, base=10)\nround(z::Complex[, RoundingModeReal, [RoundingModeImaginary]]; sigdigits, base=10)\n\nReturn the nearest integral value of the same type as the complex-valued z to z, breaking ties using the specified RoundingModes. The first RoundingMode is used for rounding the real components while the second is used for rounding the imaginary components.\n\nRoundingModeReal and RoundingModeImaginary default to RoundNearest, which rounds to the nearest integer, with ties (fractional values of 0.5) being rounded to the nearest even integer.\n\nExamples\n\njulia> round(3.14 + 4.5im)\n3.0 + 4.0im\n\njulia> round(3.14 + 4.5im, RoundUp, RoundNearestTiesUp)\n4.0 + 5.0im\n\njulia> round(3.14159 + 4.512im; digits = 1)\n3.1 + 4.5im\n\njulia> round(3.14159 + 4.512im; sigdigits = 3)\n3.14 + 4.51im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ceil","location":"base/math.html#Base.ceil","category":"function","text":"ceil([T,] x)\nceil(x; digits::Integer= [, base = 10])\nceil(x; sigdigits::Integer= [, base = 10])\n\nceil(x) returns the nearest integral value of the same type as x that is greater than or equal to x.\n\nceil(T, x) converts the result to type T, throwing an InexactError if the ceiled value is not representable as a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support ceil for a new type, define Base.round(x::NewType, ::RoundingMode{:Up}).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.floor","location":"base/math.html#Base.floor","category":"function","text":"floor([T,] x)\nfloor(x; digits::Integer= [, base = 10])\nfloor(x; sigdigits::Integer= [, base = 10])\n\nfloor(x) returns the nearest integral value of the same type as x that is less than or equal to x.\n\nfloor(T, x) converts the result to type T, throwing an InexactError if the floored value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support floor for a new type, define Base.round(x::NewType, ::RoundingMode{:Down}).\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.trunc","location":"base/math.html#Base.trunc","category":"function","text":"trunc([T,] x)\ntrunc(x; digits::Integer= [, base = 10])\ntrunc(x; sigdigits::Integer= [, base = 10])\n\ntrunc(x) returns the nearest integral value of the same type as x whose absolute value is less than or equal to the absolute value of x.\n\ntrunc(T, x) converts the result to type T, throwing an InexactError if the truncated value is not representable a T.\n\nKeywords digits, sigdigits and base work as for round.\n\nTo support trunc for a new type, define Base.round(x::NewType, ::RoundingMode{:ToZero}).\n\nSee also: %, floor, unsigned, unsafe_trunc.\n\nExamples\n\njulia> trunc(2.22)\n2.0\n\njulia> trunc(-2.22, digits=1)\n-2.2\n\njulia> trunc(Int, -2.22)\n-2\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.unsafe_trunc","location":"base/math.html#Base.unsafe_trunc","category":"function","text":"unsafe_trunc(T, x)\n\nReturn the nearest integral value of type T whose absolute value is less than or equal to the absolute value of x. If the value is not representable by T, an arbitrary value will be returned. See also trunc.\n\nExamples\n\njulia> unsafe_trunc(Int, -2.2)\n-2\n\njulia> unsafe_trunc(Int, NaN)\n-9223372036854775808\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.min","location":"base/math.html#Base.min","category":"function","text":"min(x, y, ...)\n\nReturn the minimum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the minimum function to take the minimum element from a collection.\n\nExamples\n\njulia> min(2, 5, 1)\n1\n\njulia> min(4, missing, 6)\nmissing\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.max","location":"base/math.html#Base.max","category":"function","text":"max(x, y, ...)\n\nReturn the maximum of the arguments, with respect to isless. If any of the arguments is missing, return missing. See also the maximum function to take the maximum element from a collection.\n\nExamples\n\njulia> max(2, 5, 1)\n5\n\njulia> max(5, missing, 6)\nmissing\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.minmax","location":"base/math.html#Base.minmax","category":"function","text":"minmax(x, y)\n\nReturn (min(x,y), max(x,y)).\n\nSee also extrema that returns (minimum(x), maximum(x)).\n\nExamples\n\njulia> minmax('c','b')\n('b', 'c')\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.clamp","location":"base/math.html#Base.clamp","category":"function","text":"clamp(x::Integer, r::AbstractUnitRange)\n\nClamp x to lie within range r.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\n\n\n\n\nclamp(x, T)::T\n\nClamp x between typemin(T) and typemax(T) and convert the result to type T.\n\nSee also trunc.\n\nExamples\n\njulia> clamp(200, Int8)\n127\n\njulia> clamp(-200, Int8)\n-128\n\njulia> trunc(Int, 4pi^2)\n39\n\n\n\n\n\nclamp(x, lo, hi)\n\nReturn x if lo <= x <= hi. If x > hi, return hi. If x < lo, return lo. Arguments are promoted to a common type.\n\nSee also clamp!, min, max.\n\ncompat: Julia 1.3\nmissing as the first argument requires at least Julia 1.3.\n\nExamples\n\njulia> clamp.([pi, 1.0, big(10)], 2.0, 9.0)\n3-element Vector{BigFloat}:\n 3.141592653589793238462643383279502884197169399375105820974944592307816406286198\n 2.0\n 9.0\n\njulia> clamp.([11, 8, 5], 10, 6)  # an example where lo > hi\n3-element Vector{Int64}:\n  6\n  6\n 10\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.clamp!","location":"base/math.html#Base.clamp!","category":"function","text":"clamp!(array::AbstractArray, lo, hi)\n\nRestrict values in array to the specified range, in-place. See also clamp.\n\ncompat: Julia 1.3\nmissing entries in array require at least Julia 1.3.\n\nExamples\n\njulia> row = collect(-4:4)';\n\njulia> clamp!(row, 0, Inf)\n1×9 adjoint(::Vector{Int64}) with eltype Int64:\n 0  0  0  0  0  1  2  3  4\n\njulia> clamp.((-4:4)', 0, Inf)\n1×9 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0  1.0  2.0  3.0  4.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.abs","location":"base/math.html#Base.abs","category":"function","text":"abs(x)\n\nThe absolute value of x.\n\nWhen abs is applied to signed integers, overflow may occur, resulting in the return of a negative value. This overflow occurs only when abs is applied to the minimum representable value of a signed integer. That is, when x == typemin(typeof(x)), abs(x) == x < 0, not -x as might be expected.\n\nSee also: abs2, unsigned, sign.\n\nExamples\n\njulia> abs(-3)\n3\n\njulia> abs(1 + im)\n1.4142135623730951\n\njulia> abs.(Int8[-128 -127 -126 0 126 127])  # overflow at typemin(Int8)\n1×6 Matrix{Int8}:\n -128  127  126  0  126  127\n\njulia> maximum(abs, [1, -2, 3, -4])\n4\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked","location":"base/math.html#Base.Checked","category":"module","text":"Checked\n\nThe Checked module provides arithmetic functions for the built-in signed and unsigned Integer types which throw an error when an overflow occurs. They are named like checked_sub, checked_div, etc. In addition, add_with_overflow, sub_with_overflow, mul_with_overflow return both the unchecked results and a boolean value denoting the presence of an overflow.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_abs","location":"base/math.html#Base.Checked.checked_abs","category":"function","text":"Base.checked_abs(x)\n\nCalculates abs(x), checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent abs(typemin(Int)), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_neg","location":"base/math.html#Base.Checked.checked_neg","category":"function","text":"Base.checked_neg(x)\n\nCalculates -x, checking for overflow errors where applicable. For example, standard two's complement signed integers (e.g. Int) cannot represent -typemin(Int), thus leading to an overflow.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_add","location":"base/math.html#Base.Checked.checked_add","category":"function","text":"Base.checked_add(x, y)\n\nCalculates x+y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_sub","location":"base/math.html#Base.Checked.checked_sub","category":"function","text":"Base.checked_sub(x, y)\n\nCalculates x-y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_mul","location":"base/math.html#Base.Checked.checked_mul","category":"function","text":"Base.checked_mul(x, y)\n\nCalculates x*y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_div","location":"base/math.html#Base.Checked.checked_div","category":"function","text":"Base.checked_div(x, y)\n\nCalculates div(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_rem","location":"base/math.html#Base.Checked.checked_rem","category":"function","text":"Base.checked_rem(x, y)\n\nCalculates x%y, checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_fld","location":"base/math.html#Base.Checked.checked_fld","category":"function","text":"Base.checked_fld(x, y)\n\nCalculates fld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_mod","location":"base/math.html#Base.Checked.checked_mod","category":"function","text":"Base.checked_mod(x, y)\n\nCalculates mod(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_cld","location":"base/math.html#Base.Checked.checked_cld","category":"function","text":"Base.checked_cld(x, y)\n\nCalculates cld(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.checked_pow","location":"base/math.html#Base.Checked.checked_pow","category":"function","text":"Base.checked_pow(x, y)\n\nCalculates ^(x,y), checking for overflow errors where applicable.\n\nThe overflow protection may impose a perceptible performance penalty.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.add_with_overflow","location":"base/math.html#Base.Checked.add_with_overflow","category":"function","text":"Base.add_with_overflow(x, y) -> (r, f)\n\nCalculates r = x+y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.sub_with_overflow","location":"base/math.html#Base.Checked.sub_with_overflow","category":"function","text":"Base.sub_with_overflow(x, y) -> (r, f)\n\nCalculates r = x-y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Checked.mul_with_overflow","location":"base/math.html#Base.Checked.mul_with_overflow","category":"function","text":"Base.mul_with_overflow(x, y) -> (r, f)\n\nCalculates r = x*y, with the flag f indicating whether overflow has occurred.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.abs2","location":"base/math.html#Base.abs2","category":"function","text":"abs2(x)\n\nSquared absolute value of x.\n\nThis can be faster than abs(x)^2, especially for complex numbers where abs(x) requires a square root via hypot.\n\nSee also abs, conj, real.\n\nExamples\n\njulia> abs2(-3)\n9\n\njulia> abs2(3.0 + 4.0im)\n25.0\n\njulia> sum(abs2, [1+2im, 3+4im])  # LinearAlgebra.norm(x)^2\n30\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.copysign","location":"base/math.html#Base.copysign","category":"function","text":"copysign(x, y) -> z\n\nReturn z which has the magnitude of x and the same sign as y.\n\nExamples\n\njulia> copysign(1, -2)\n-1\n\njulia> copysign(-1, 2)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sign","location":"base/math.html#Base.sign","category":"function","text":"sign(x)\n\nReturn zero if x==0 and xx otherwise (i.e., ±1 for real x).\n\nSee also signbit, zero, copysign, flipsign.\n\nExamples\n\njulia> sign(-4.0)\n-1.0\n\njulia> sign(99)\n1\n\njulia> sign(-0.0)\n-0.0\n\njulia> sign(0 + im)\n0.0 + 1.0im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.signbit","location":"base/math.html#Base.signbit","category":"function","text":"signbit(x)\n\nReturn true if the value of the sign of x is negative, otherwise false.\n\nSee also sign and copysign.\n\nExamples\n\njulia> signbit(-4)\ntrue\n\njulia> signbit(5)\nfalse\n\njulia> signbit(5.5)\nfalse\n\njulia> signbit(-4.1)\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.flipsign","location":"base/math.html#Base.flipsign","category":"function","text":"flipsign(x, y)\n\nReturn x with its sign flipped if y is negative. For example abs(x) = flipsign(x,x).\n\nExamples\n\njulia> flipsign(5, 3)\n5\n\njulia> flipsign(5, -3)\n-5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.sqrt","location":"base/math.html#Base.sqrt-Tuple{Number}","category":"method","text":"sqrt(x)\n\nReturn sqrtx.\n\nThrow a DomainError for negative Real arguments. Use Complex negative arguments instead to obtain a Complex result.\n\nThe prefix operator √ is equivalent to sqrt.\n\nnote: Branch cut\nsqrt has a branch cut along the negative real axis; -0.0im is taken to be below the axis.\n\nSee also: hypot.\n\nExamples\n\njulia> sqrt(big(81))\n9.0\n\njulia> sqrt(big(-81))\nERROR: DomainError with -81.0:\nNaN result for non-NaN input.\nStacktrace:\n [1] sqrt(::BigFloat) at ./mpfr.jl:501\n[...]\n\njulia> sqrt(big(complex(-81)))\n0.0 + 9.0im\n\njulia> sqrt(-81 - 0.0im)  # -0.0im is below the branch cut\n0.0 - 9.0im\n\njulia> .√(1:4)\n4-element Vector{Float64}:\n 1.0\n 1.4142135623730951\n 1.7320508075688772\n 2.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.isqrt","location":"base/math.html#Base.isqrt","category":"function","text":"isqrt(n::Integer)\n\nInteger square root: the largest integer m such that m*m <= n.\n\njulia> isqrt(5)\n2\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.cbrt","location":"base/math.html#Base.Math.cbrt-Tuple{AbstractFloat}","category":"method","text":"cbrt(x::Real)\n\nReturn the cube root of x, i.e. x^13. Negative values are accepted (returning the negative real root when x  0).\n\nThe prefix operator ∛ is equivalent to cbrt.\n\nExamples\n\njulia> cbrt(big(27))\n3.0\n\njulia> cbrt(big(-27))\n-3.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.fourthroot","location":"base/math.html#Base.Math.fourthroot-Tuple{Number}","category":"method","text":"fourthroot(x)\n\nReturn the fourth root of x by applying sqrt twice successively.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.real","location":"base/math.html#Base.real","category":"function","text":"real(A::AbstractArray)\n\nReturn an array containing the real part of each entry in array A.\n\nEquivalent to real.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> real([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 1\n 0\n 3\n\njulia> real(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n2\n\n\n\n\n\nreal(T::Type)\n\nReturn the type that represents the real part of a value of type T. e.g: for T == Complex{R}, returns R. Equivalent to typeof(real(zero(T))).\n\nExamples\n\njulia> real(Complex{Int})\nInt64\n\njulia> real(Float64)\nFloat64\n\n\n\n\n\nreal(z)\n\nReturn the real part of the complex number z.\n\nSee also: imag, reim, complex, isreal, Real.\n\nExamples\n\njulia> real(1 + 3im)\n1\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.imag","location":"base/math.html#Base.imag","category":"function","text":"imag(A::AbstractArray)\n\nReturn an array containing the imaginary part of each entry in array A.\n\nEquivalent to imag.(A), except that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> imag([1, 2im, 3 + 4im])\n3-element Vector{Int64}:\n 0\n 2\n 4\n\njulia> imag(fill(2 - im))\n0-dimensional Array{Int64, 0}:\n-1\n\n\n\n\n\nimag(z)\n\nReturn the imaginary part of the complex number z.\n\nSee also: conj, reim, adjoint, angle.\n\nExamples\n\njulia> imag(1 + 3im)\n3\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.reim","location":"base/math.html#Base.reim","category":"function","text":"reim(A::AbstractArray)\n\nReturn a tuple of two arrays containing respectively the real and the imaginary part of each entry in A.\n\nEquivalent to (real.(A), imag.(A)), except that when eltype(A) <: Real A is returned without copying to represent the real part, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> reim([1, 2im, 3 + 4im])\n([1, 0, 3], [0, 2, 4])\n\njulia> reim(fill(2 - im))\n(fill(2), fill(-1))\n\n\n\n\n\nreim(z)\n\nReturn a tuple of the real and imaginary parts of the complex number z.\n\nExamples\n\njulia> reim(1 + 3im)\n(1, 3)\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.conj","location":"base/math.html#Base.conj","category":"function","text":"conj(A::AbstractArray)\n\nReturn an array containing the complex conjugate of each entry in array A.\n\nEquivalent to conj.(A), except that when eltype(A) <: Real A is returned without copying, and that when A has zero dimensions, a 0-dimensional array is returned (rather than a scalar).\n\nExamples\n\njulia> conj([1, 2im, 3 + 4im])\n3-element Vector{Complex{Int64}}:\n 1 + 0im\n 0 - 2im\n 3 - 4im\n\njulia> conj(fill(2 - im))\n0-dimensional Array{Complex{Int64}, 0}:\n2 + 1im\n\n\n\n\n\nconj(z)\n\nCompute the complex conjugate of a complex number z.\n\nSee also: angle, adjoint.\n\nExamples\n\njulia> conj(1 + 3im)\n1 - 3im\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.angle","location":"base/math.html#Base.angle","category":"function","text":"angle(z)\n\nCompute the phase angle in radians of a complex number z.\n\nReturns a number -pi ≤ angle(z) ≤ pi, and is thus discontinuous along the negative real axis.\n\nSee also: atan, cis, rad2deg.\n\nExamples\n\njulia> rad2deg(angle(1 + im))\n45.0\n\njulia> rad2deg(angle(1 - im))\n-45.0\n\njulia> rad2deg(angle(-1 + 1e-20im))\n180.0\n\njulia> rad2deg(angle(-1 - 1e-20im))\n-180.0\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cis","location":"base/math.html#Base.cis","category":"function","text":"cis(x)\n\nMore efficient method for exp(im*x) by using Euler's formula: cos(x) + i sin(x) = exp(i x).\n\nSee also cispi, sincos, exp, angle.\n\nExamples\n\njulia> cis(π) ≈ -1\ntrue\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.cispi","location":"base/math.html#Base.cispi","category":"function","text":"cispi(x)\n\nMore accurate method for cis(pi*x) (especially for large x).\n\nSee also cis, sincospi, exp, angle.\n\nExamples\n\njulia> cispi(10000)\n1.0 + 0.0im\n\njulia> cispi(0.25 + 1im)\n0.030556854645954562 + 0.03055685464595456im\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.binomial","location":"base/math.html#Base.binomial","category":"function","text":"binomial(x::Number, k::Integer)\n\nThe generalized binomial coefficient, defined for k ≥ 0 by the polynomial\n\nfrac1k prod_j=0^k-1 (x - j)\n\nWhen k < 0 it returns zero.\n\nFor the case of integer x, this is equivalent to the ordinary integer binomial coefficient\n\nbinomnk = fracnk (n-k)\n\nFurther generalizations to non-integer k are mathematically possible, but involve the Gamma function and/or the beta function, which are not provided by the Julia standard library but are available in external packages such as SpecialFunctions.jl.\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\nbinomial(n::Integer, k::Integer)\n\nThe binomial coefficient binomnk, being the coefficient of the kth term in the polynomial expansion of (1+x)^n.\n\nIf n is non-negative, then it is the number of ways to choose k out of n items:\n\nbinomnk = fracnk (n-k)\n\nwhere n is the factorial function.\n\nIf n is negative, then it is defined in terms of the identity\n\nbinomnk = (-1)^k binomk-n-1k\n\nSee also factorial.\n\nExamples\n\njulia> binomial(5, 3)\n10\n\njulia> factorial(5) ÷ (factorial(5-3) * factorial(3))\n10\n\njulia> binomial(-5, 3)\n-35\n\nExternal links\n\nBinomial coefficient on Wikipedia.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.factorial","location":"base/math.html#Base.factorial","category":"function","text":"factorial(n::Integer)\n\nFactorial of n. If n is an Integer, the factorial is computed as an integer (promoted to at least 64 bits). Note that this may overflow if n is not small, but you can use factorial(big(n)) to compute the result exactly in arbitrary precision.\n\nSee also binomial.\n\nExamples\n\njulia> factorial(6)\n720\n\njulia> factorial(21)\nERROR: OverflowError: 21 is too large to look up in the table; consider using `factorial(big(21))` instead\nStacktrace:\n[...]\n\njulia> factorial(big(21))\n51090942171709440000\n\nExternal links\n\nFactorial on Wikipedia.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.gcd","location":"base/math.html#Base.gcd","category":"function","text":"gcd(x, y...)\n\nGreatest common (positive) divisor (or zero if all arguments are zero). The arguments may be integer and rational numbers.\n\na is a divisor of b if there exists an integer m such that ma=b.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> gcd(6, 9)\n3\n\njulia> gcd(6, -9)\n3\n\njulia> gcd(6, 0)\n6\n\njulia> gcd(0, 0)\n0\n\njulia> gcd(1//3, 2//3)\n1//3\n\njulia> gcd(1//3, -2//3)\n1//3\n\njulia> gcd(1//3, 2)\n1//3\n\njulia> gcd(0, 0, 10, 15)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.lcm","location":"base/math.html#Base.lcm","category":"function","text":"lcm(x, y...)\n\nLeast common (positive) multiple (or zero if any argument is zero). The arguments may be integer and rational numbers.\n\na is a multiple of b if there exists an integer m such that a=mb.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\nExamples\n\njulia> lcm(2, 3)\n6\n\njulia> lcm(-2, 3)\n6\n\njulia> lcm(0, 3)\n0\n\njulia> lcm(0, 0)\n0\n\njulia> lcm(1//3, 2//3)\n2//3\n\njulia> lcm(1//3, -2//3)\n2//3\n\njulia> lcm(1//3, 2)\n2//1\n\njulia> lcm(1, 3, 5, 7)\n105\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.gcdx","location":"base/math.html#Base.gcdx","category":"function","text":"gcdx(a, b...)\n\nComputes the greatest common (positive) divisor of a and b and their Bézout coefficients, i.e. the integer coefficients u and v that satisfy u*a + v*b = d = gcd(a b). gcdx(a b) returns (d u v).\n\nFor more arguments than two, i.e., gcdx(a, b, c, ...) the Bézout coefficients are computed recursively, returning a solution (d, u, v, w, ...) to u*a + v*b + w*c +  = d = gcd(a b c ).\n\nThe arguments may be integer and rational numbers.\n\ncompat: Julia 1.4\nRational arguments require Julia 1.4 or later.\n\ncompat: Julia 1.12\nMore or fewer arguments than two require Julia 1.12 or later.\n\nExamples\n\njulia> gcdx(12, 42)\n(6, -3, 1)\n\njulia> gcdx(240, 46)\n(2, -9, 47)\n\njulia> gcdx(15, 12, 20)\n(1, 7, -7, -1)\n\nnote: Note\nBézout coefficients are not uniquely defined. gcdx returns the minimal Bézout coefficients that are computed by the extended Euclidean algorithm. (Ref: D. Knuth, TAoCP, 2/e, p. 325, Algorithm X.) For signed integers, these coefficients u and v are minimal in the sense that u  bd and v  ad. Furthermore, the signs of u and v are chosen so that d is positive. For unsigned integers, the coefficients u and v might be near their typemax, and the identity then holds only via the unsigned integers' modulo arithmetic.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ispow2","location":"base/math.html#Base.ispow2","category":"function","text":"ispow2(n::Number) -> Bool\n\nTest whether n is an integer power of two.\n\nSee also count_ones, prevpow, nextpow.\n\nExamples\n\njulia> ispow2(4)\ntrue\n\njulia> ispow2(5)\nfalse\n\njulia> ispow2(4.5)\nfalse\n\njulia> ispow2(0.25)\ntrue\n\njulia> ispow2(1//8)\ntrue\n\ncompat: Julia 1.6\nSupport for non-Integer arguments was added in Julia 1.6.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nextpow","location":"base/math.html#Base.nextpow","category":"function","text":"nextpow(a, x)\n\nThe smallest a^n not less than x, where n is a non-negative integer. a must be greater than 1, and x must be greater than 0.\n\nSee also prevpow.\n\nExamples\n\njulia> nextpow(2, 7)\n8\n\njulia> nextpow(2, 9)\n16\n\njulia> nextpow(5, 20)\n25\n\njulia> nextpow(4, 16)\n16\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.prevpow","location":"base/math.html#Base.prevpow","category":"function","text":"prevpow(a, x)\n\nThe largest a^n not greater than x, where n is a non-negative integer. a must be greater than 1, and x must not be less than 1.\n\nSee also nextpow, isqrt.\n\nExamples\n\njulia> prevpow(2, 7)\n4\n\njulia> prevpow(2, 9)\n8\n\njulia> prevpow(5, 20)\n5\n\njulia> prevpow(4, 16)\n16\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.nextprod","location":"base/math.html#Base.nextprod","category":"function","text":"nextprod(factors::Union{Tuple,AbstractVector}, n)\n\nNext integer greater than or equal to n that can be written as prod k_i^p_i for integers p_1, p_2, etcetera, for factors k_i in factors.\n\nExamples\n\njulia> nextprod((2, 3), 105)\n108\n\njulia> 2^2 * 3^3\n108\n\ncompat: Julia 1.6\nThe method that accepts a tuple requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.invmod","location":"base/math.html#Base.invmod","category":"function","text":"invmod(n::Integer, T) where {T <: Base.BitInteger}\ninvmod(n::T) where {T <: Base.BitInteger}\n\nCompute the modular inverse of n in the integer ring of type T, i.e. modulo 2^N where N = 8*sizeof(T) (e.g. N = 32 for Int32). In other words, these methods satisfy the following identities:\n\nn * invmod(n) == 1\n(n * invmod(n, T)) % T == 1\n(n % T) * invmod(n, T) == 1\n\nNote that * here is modular multiplication in the integer ring, T.  This will throw an error if n is even, because then it is not relatively prime with 2^N and thus has no such inverse.\n\nSpecifying the modulus implied by an integer type as an explicit value is often inconvenient since the modulus is by definition too big to be represented by the type.\n\nThe modular inverse is computed much more efficiently than the general case using the algorithm described in https://arxiv.org/pdf/2204.04342.pdf.\n\ncompat: Julia 1.11\nThe invmod(n) and invmod(n, T) methods require Julia 1.11 or later.\n\n\n\n\n\ninvmod(n::Integer, m::Integer)\n\nTake the inverse of n modulo m: y such that n y = 1 pmod m, and div(ym) = 0. This will throw an error if m = 0, or if gcd(nm) neq 1.\n\nExamples\n\njulia> invmod(2, 5)\n3\n\njulia> invmod(2, 3)\n2\n\njulia> invmod(5, 6)\n5\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.powermod","location":"base/math.html#Base.powermod","category":"function","text":"powermod(x::Integer, p::Integer, m)\n\nCompute x^p pmod m.\n\nExamples\n\njulia> powermod(2, 6, 5)\n4\n\njulia> mod(2^6, 5)\n4\n\njulia> powermod(5, 2, 20)\n5\n\njulia> powermod(5, 2, 19)\n6\n\njulia> powermod(5, 3, 19)\n11\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.ndigits","location":"base/math.html#Base.ndigits","category":"function","text":"ndigits(n::Integer; base::Integer=10, pad::Integer=1)\n\nCompute the number of digits in integer n written in base base (base must not be in [-1, 0, 1]), optionally padded with zeros to a specified size (the result will never be less than pad).\n\nSee also digits, count_ones.\n\nExamples\n\njulia> ndigits(0)\n1\n\njulia> ndigits(12345)\n5\n\njulia> ndigits(1022, base=16)\n3\n\njulia> string(1022, base=16)\n\"3fe\"\n\njulia> ndigits(123, pad=5)\n5\n\njulia> ndigits(-123)\n3\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.add_sum","location":"base/math.html#Base.add_sum","category":"function","text":"Base.add_sum(x, y)\n\nThe reduction operator used in sum. The main difference from + is that small integers are promoted to Int/UInt.\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.widemul","location":"base/math.html#Base.widemul","category":"function","text":"widemul(x, y)\n\nMultiply x and y, giving the result as a larger type.\n\nSee also promote, Base.add_sum.\n\nExamples\n\njulia> widemul(Float32(3.0), 4.0) isa BigFloat\ntrue\n\njulia> typemax(Int8) * typemax(Int8)\n1\n\njulia> widemul(typemax(Int8), typemax(Int8))  # == 127^2\n16129\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.evalpoly","location":"base/math.html#Base.Math.evalpoly","category":"function","text":"evalpoly(x, p)\n\nEvaluate the polynomial sum_k x^k-1 pk for the coefficients p[1], p[2], ...; that is, the coefficients are given in ascending order by power of x. Loops are unrolled at compile time if the number of coefficients is statically known, i.e. when p is a Tuple. This function generates efficient code using Horner's method if x is real, or using a Goertzel-like [DK62] algorithm if x is complex.\n\n[DK62]: Donald Knuth, Art of Computer Programming, Volume 2: Seminumerical Algorithms, Sec. 4.6.4.\n\ncompat: Julia 1.4\nThis function requires Julia 1.4 or later.\n\nExamples\n\njulia> evalpoly(2, (1, 2, 3))\n17\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.Math.@evalpoly","location":"base/math.html#Base.Math.@evalpoly","category":"macro","text":"@evalpoly(z, c...)\n\nEvaluate the polynomial sum_k z^k-1 ck for the coefficients c[1], c[2], ...; that is, the coefficients are given in ascending order by power of z.  This macro expands to efficient inline code that uses either Horner's method or, for complex z, a more efficient Goertzel-like algorithm.\n\nSee also evalpoly.\n\nExamples\n\njulia> @evalpoly(3, 1, 0, 1)\n10\n\njulia> @evalpoly(2, 1, 0, 1)\n5\n\njulia> @evalpoly(2, 1, 1, 1)\n7\n\n\n\n\n\n","page":"Mathematics"},{"title":"Base.FastMath.@fastmath","location":"base/math.html#Base.FastMath.@fastmath","category":"macro","text":"@fastmath expr\n\nExecute a transformed version of the expression, which calls functions that may violate strict IEEE semantics. This allows the fastest possible operation, but results are undefined – be careful when doing this, as it may change numerical results.\n\nThis sets the LLVM Fast-Math flags, and corresponds to the -ffast-math option in clang. See the notes on performance annotations for more details.\n\nExamples\n\njulia> @fastmath 1+2\n3\n\njulia> @fastmath(sin(3))\n0.1411200080598672\n\n\n\n\n\n","page":"Mathematics"},{"title":"Windows","location":"devdocs/build/windows.html#Windows","category":"section","text":"This file describes how to install, or build, and use Julia on Windows.\n\nFor more general information about Julia, please see the main README or the documentation.","page":"Windows"},{"title":"General Information for Windows","location":"devdocs/build/windows.html#General-Information-for-Windows","category":"section","text":"We highly recommend running Julia using a modern terminal application, in particular Windows Terminal, which can be installed from the Microsoft Store.","page":"Windows"},{"title":"Line endings","location":"devdocs/build/windows.html#Line-endings","category":"section","text":"Julia uses binary-mode files exclusively. Unlike many other Windows programs, if you write \\n to a file, you get a \\n in the file, not some other bit pattern. This matches the behavior exhibited by other operating systems. If you have installed Git for Windows, it is suggested, but not required, that you configure your system Git to use the same convention:\n\ngit config --global core.eol lf\ngit config --global core.autocrlf input\n\nor edit %USERPROFILE%\\.gitconfig and add/edit the lines:\n\n[core]\n    eol = lf\n    autocrlf = input","page":"Windows"},{"title":"Binary distribution","location":"devdocs/build/windows.html#Binary-distribution","category":"section","text":"For the binary distribution installation notes on Windows please see the instructions at https://julialang.org/downloads/platform/#windows. Note, however, that on all platforms using juliaup is recommended over manually installing binaries.","page":"Windows"},{"title":"Source distribution","location":"devdocs/build/windows.html#Source-distribution","category":"section","text":"","page":"Windows"},{"title":"Cygwin-to-MinGW cross-compiling","location":"devdocs/build/windows.html#Cygwin-to-MinGW-cross-compiling","category":"section","text":"The recommended way of compiling Julia from source on Windows is by cross compiling from Cygwin, using versions of the MinGW-w64 compilers available through Cygwin's package manager.\n\nDownload and run Cygwin setup for 32 bit or 64 bit. Note, that you can compile either 32 or 64 bit Julia from either 32 or 64 bit Cygwin. 64 bit Cygwin has a slightly smaller but often more up-to-date selection of packages.\nAdvanced: you may skip steps 2-4 by running:\nsetup-x86_64.exe -s <url> -q -P cmake,gcc-g++,git,make,patch,curl,m4,python3,p7zip,mingw64-i686-gcc-g++,mingw64-i686-gcc-fortran,mingw64-x86_64-gcc-g++,mingw64-x86_64-gcc-fortran\nreplacing <url> with a site from https://cygwin.com/mirrors.html or run setup manually first and select a mirror.\nSelect installation location and a mirror to download from.\nAt the Select Packages step, select the following:\nFrom the Devel category: cmake, gcc-g++, git, make, patch\nFrom the Net category: curl\nFrom Interpreters (or Python) category: m4, python3\nFrom the Archive category: p7zip\nFor 32 bit Julia, and also from the Devel category: mingw64-i686-gcc-g++ and mingw64-i686-gcc-fortran\nFor 64 bit Julia, and also from the Devel category: mingw64-x86_64-gcc-g++ and mingw64-x86_64-gcc-fortran\nAllow Cygwin installation to finish, then start from the installed shortcut 'Cygwin Terminal', or 'Cygwin64 Terminal', respectively.\nBuild Julia and its dependencies from source:\nGet the Julia sources\ngit clone https://github.com/JuliaLang/julia.git\ncd julia\nTip: If you get an error: cannot fork() for fetch-pack: Resource temporarily unavailable from git, add alias git=\"env PATH=/usr/bin git\" to ~/.bashrc and restart Cygwin.\nSet the XC_HOST variable in Make.user to indicate MinGW-w64 cross compilation\necho 'XC_HOST = i686-w64-mingw32' > Make.user     # for 32 bit Julia\n# or\necho 'XC_HOST = x86_64-w64-mingw32' > Make.user   # for 64 bit Julia\nStart the build\nmake -j 4       # Adjust the number of threads (4) to match your build environment.\nmake -j 4 debug # This builds julia-debug.exe\nRun Julia using the Julia executables directly\nusr/bin/julia.exe\nusr/bin/julia-debug.exe\n\nnote: Pro tip: build both!\nmake O=julia-win32 configure\nmake O=julia-win64 configure\necho 'XC_HOST = i686-w64-mingw32' > julia-win32/Make.user\necho 'XC_HOST = x86_64-w64-mingw32' > julia-win64/Make.user\necho 'ifeq ($(BUILDROOT),$(JULIAHOME))\n        $(error \"in-tree build disabled\")\n      endif' >> Make.user\nmake -C julia-win32  # build for Windows x86 in julia-win32 folder\nmake -C julia-win64  # build for Windows x86-64 in julia-win64 folder","page":"Windows"},{"title":"Compiling with MinGW/MSYS2","location":"devdocs/build/windows.html#Compiling-with-MinGW/MSYS2","category":"section","text":"MSYS2 is a software distribution and build environment for Windows.\n\nNote: MSYS2 requires 64 bit Windows 7 or newer.\n\nInstall and configure MSYS2.\nDownload and run the latest installer for the  64-bit distribution.  The installer will have a name like msys2-x86_64-yyyymmdd.exe.\nOpen the MSYS2 shell. Update the package database and base packages:\npacman -Syu\nExit and restart MSYS2. Update the rest of the base packages:\npacman -Syu\nThen install tools required to build julia:\npacman -S cmake diffutils git m4 make patch tar p7zip curl python\nFor 64 bit Julia, install the x86_64 version:\npacman -S mingw-w64-x86_64-gcc\nFor 32 bit Julia, install the i686 version:\npacman -S mingw-w64-i686-gcc\nConfiguration of MSYS2 is complete. Now exit the MSYS2 shell.\nBuild Julia and its dependencies with pre-build dependencies.\nOpen a new MINGW64/MINGW32 shell.  Currently we can't use both mingw32 and mingw64,  so if you want to build the x86_64 and i686 versions,  you'll need to build them in each environment separately.\nClone the Julia sources:\ngit clone https://github.com/JuliaLang/julia.git  cd julia\nStart the build\nmake -j$(nproc)\n\nnote: Pro tip: build in dir\nmake O=julia-mingw-w64 configure\necho 'ifeq ($(BUILDROOT),$(JULIAHOME))\n        $(error \"in-tree build disabled\")\n      endif' >> Make.user\nmake -C julia-mingw-w64","page":"Windows"},{"title":"Cross-compiling from Unix (Linux/Mac/WSL)","location":"devdocs/build/windows.html#Cross-compiling-from-Unix-(Linux/Mac/WSL)","category":"section","text":"You can also use MinGW-w64 cross compilers to build a Windows version of Julia from Linux, Mac, or the Windows Subsystem for Linux (WSL).\n\nFirst, you will need to ensure your system has the required dependencies. We need wine (>=1.7.5), a system compiler, and some downloaders. Note: a Cygwin install might interfere with this method if using WSL.\n\nOn Ubuntu (on other Linux systems the dependency names are likely to be similar):\n\napt-get install wine-stable gcc wget p7zip-full winbind mingw-w64 gfortran-mingw-w64\ndpkg --add-architecture i386 && apt-get update && apt-get install wine32 # add sudo to each if needed\n# switch all of the following to their \"-posix\" variants (interactively):\nfor pkg in i686-w64-mingw32-g++ i686-w64-mingw32-gcc i686-w64-mingw32-gfortran x86_64-w64-mingw32-g++ x86_64-w64-mingw32-gcc x86_64-w64-mingw32-gfortran; do\n    sudo update-alternatives --config $pkg\ndone\n\nOn Mac: Install XCode, XCode command line tools, X11 (now XQuartz), and MacPorts or Homebrew. Then run port install wine wget mingw-w64, or brew install wine wget mingw-w64, as appropriate.\n\nThen run the build:\n\ngit clone https://github.com/JuliaLang/julia.git julia-win32\ncd julia-win32\necho override XC_HOST = i686-w64-mingw32 >> Make.user\nmake\nmake win-extras (Necessary before running make binary-dist)\nmake binary-dist then make exe to create the Windows installer.\nmove the julia-*.exe installer to the target machine\n\nIf you are building for 64-bit Windows, the steps are essentially the same. Just replace i686 in XC_HOST with x86_64. (Note: on Mac, wine only runs in 32-bit mode).","page":"Windows"},{"title":"Debugging a cross-compiled build under wine","location":"devdocs/build/windows.html#Debugging-a-cross-compiled-build-under-wine","category":"section","text":"The most effective way to debug a cross-compiled version of Julia on the cross-compilation host is to install a Windows version of GDB and run it under wine as usual. The pre-built packages available as part of the MSYS2 project are known to work. Apart from the GDB package you may also need the python and termcap packages. Finally, GDB's prompt may not work when launched from the command line. This can be worked around by prepending wineconsole to the regular GDB invocation.","page":"Windows"},{"title":"After compiling","location":"devdocs/build/windows.html#After-compiling","category":"section","text":"Compiling using one of the options above creates a basic Julia build, but not some extra components that are included if you run the full Julia binary installer. If you need these components, the easiest way to get them is to build the installer yourself using make win-extras followed by make binary-dist and make exe. Then run the resulting installer.","page":"Windows"},{"title":"Windows Build Debugging","location":"devdocs/build/windows.html#Windows-Build-Debugging","category":"section","text":"","page":"Windows"},{"title":"GDB hangs with Cygwin mintty","location":"devdocs/build/windows.html#GDB-hangs-with-Cygwin-mintty","category":"section","text":"Run GDB under the Windows console (cmd) instead. GDB may not function properly under mintty with non- Cygwin applications. You can use cmd /c start to start the Windows console from mintty if necessary.","page":"Windows"},{"title":"GDB not attaching to the right process","location":"devdocs/build/windows.html#GDB-not-attaching-to-the-right-process","category":"section","text":"Use the PID from the Windows task manager or WINPID from the ps command instead of the PID from unix-style command line tools (e.g. pgrep). You may need to add the PID column if it is not shown by default in the Windows task manager.","page":"Windows"},{"title":"GDB not showing the right backtrace","location":"devdocs/build/windows.html#GDB-not-showing-the-right-backtrace","category":"section","text":"When attaching to the julia process, GDB may not be attaching to the right thread. Use info threads command to show all the threads and thread <threadno> to switch threads.\nBe sure to use a 32 bit version of GDB to debug a 32 bit build of Julia, or a 64 bit version of GDB to debug a 64 bit build of Julia.","page":"Windows"},{"title":"Build process is slow/eats memory/hangs my computer","location":"devdocs/build/windows.html#Build-process-is-slow/eats-memory/hangs-my-computer","category":"section","text":"Disable the Windows Superfetch and Program Compatibility Assistant services, as they are known to have spurious interactions with MinGW/Cygwin.\nAs mentioned in the link above: excessive memory use by svchost specifically may be investigated in the Task Manager by clicking on the high-memory svchost.exe process and selecting Go to Services. Disable child services one-by-one until a culprit is found.\nBeware of BLODA. The vmmap tool is indispensable for identifying such software conflicts. Use vmmap to inspect the list of loaded DLLs for bash, mintty, or another persistent process used to drive the build. Essentially any DLL outside of the Windows System directory is potential BLODA.","page":"Windows"},{"title":"The World Age mechanism","location":"manual/worldage.html#The-World-Age-mechanism","category":"section","text":"note: Note\nWorld age is an advanced concept. For the vast majority of Julia users, the world age mechanism operates invisibly in the background. This documentation is intended for the few users who may encounter world-age related issues or error messages.\n\ncompat: Julia 1.12\nPrior to Julia 1.12, the world age mechanism did not apply to changes to the global binding table. The documentation in this chapter is specific to Julia 1.12+.\n\nwarning: Warning\nThis manual chapter uses internal functions to introspect world age and runtime data structures as an explanatory aid. In general, unless otherwise noted the world age mechanism is not a stable interface and should be interacted with in packages through stable APIs (e.g. invokelatest) only. In particular, do not assume that world ages are always integers or that they have a linear order.","page":"The World Age mechanism"},{"title":"World age in general","location":"manual/worldage.html#World-age-in-general","category":"section","text":"The \"world age counter\" is a monotonically increasing counter that is incremented for every change to the global method table or the global binding table (e.g. through method definition, type definition, import/using declaration, creation of (typed) globals or definition of constants).\n\nThe current value of the global world age counter can be retrieved using the (internal) function Base.get_world_counter.\n\njulia> Base.get_world_counter()\n0x0000000000009632\n\njulia> const x = 1\n\njulia> Base.get_world_counter()\n0x0000000000009633\n\nIn addition, each Task stores a local world age that determines which modifications to the global binding and method tables are currently visible to the running task. The world age of the running task will never exceed the global world age counter, but may run arbitrarily behind it. In general the term \"current world age\" refers to the local world age of the currently running task. The current world age may be retrieved using the (internal) function Base.tls_world_age\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           Core.eval(@__MODULE__, :(f() = 1))\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           f()\n       end\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38452, 38452)\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38453, 38452)\nERROR: MethodError: no method matching f()\nThe applicable method may be too new: running in current world age 38452, while global world is 38453.\n\nClosest candidates are:\n  f() (method too new to be called from this world context.)\n   @ Main REPL[2]:3\n\nStacktrace:\n [1] top-level scope\n   @ REPL[2]:5\n\njulia> (f(), Int(Base.tls_world_age()))\n(1, 38453)\n\nHere the definition of the method f raised the global world counter, but the current world age did not change. As a result, the definition of f was not visible in the currently executing task and a MethodError resulted.\n\nnote: Note\nThe method error printing provided additional information that f() is available in a newer world age. This information is added by the error display, not the task that threw the MethodError. The thrown MethodError is identical whether or not a matching definition of f() exists in a newer world age.\n\nHowever, note that the definition of f() was subsequently available at the next REPL prompt, because the current task's world age had been raised. In general, certain syntactic constructs (in particular most definitions) will raise the current task's world age to the latest global world age, thus making all changes (both from the current task and any concurrently executing other tasks) visible. The following statements raise the current world age:\n\nAn explicit invocation of Core.@latestworld\nThe start of every top-level statement\nThe start of every REPL prompt\nAny type or struct definition\nAny method definition\nAny constant declaration\nAny global variable declaration (but not a global variable assignment)\nAny using, import, export or public statement\nCertain other macros like @eval (depends on the macro implementation)\n\nNote, however, that the current task's world age may only ever be permanently incremented at top level. As a general rule, using any of the above statements in non-top-level scope is a syntax error:\n\njulia> f() = Core.@latestworld\nERROR: syntax: World age increment not at top level\nStacktrace:\n [1] top-level scope\n   @ REPL[5]:1\n\nWhen it isn't (for example for @eval), the world age side effect is ignored.\n\nAs a result of these rules, Julia may assume that the world age does not change within the execution of an ordinary function.\n\nfunction my_function()\n    before = Base.tls_world_age()\n    # Any arbitrary code\n    after = Base.tls_world_age()\n    @assert before === after # always true\nend\n\nThis is the key invariant that allows Julia to optimize based on the current state of its global data structures, while still having the well-defined ability to change these data structures.","page":"The World Age mechanism"},{"title":"Temporarily raising the world age using invokelatest","location":"manual/worldage.html#Temporarily-raising-the-world-age-using-invokelatest","category":"section","text":"As described above, it is not possible to permanently raise the world age for the remainder of a Task's execution unless the task is executing top-level statements. However, it is possible to temporarily change the world age in a scoped manner using invokelatest:\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           Core.eval(@__MODULE__, :(f() = 1))\n           invokelatest(f)\n       end\n1\n\ninvokelatest will temporarily raise the current task's world age to the latest global world age (at entry to invokelatest) and execute the provided function. Note that the world age will return to its prior value upon exit from invokelatest.","page":"The World Age mechanism"},{"title":"World age and const struct redefinitions","location":"manual/worldage.html#World-age-and-const-struct-redefinitions","category":"section","text":"The semantics described above for method redefinition also apply to redefinition of constants:\n\njulia> const x = 1\n1\n\njulia> get_const() = x\nget_const (generic function with 1 method)\n\njulia> begin\n           @show get_const()\n           Core.eval(@__MODULE__, :(const x = 2))\n           @show get_const()\n           Core.@latestworld\n           @show get_const()\n       end\nget_const() = 1\nget_const() = 1\nget_const() = 2\n2\n\nHowever, for the avoidance of doubt, they do not apply to ordinary assignment to global variables, which becomes visible immediately:\n\njulia> global y = 1\n1\n\njulia> get_global() = y\nget_global (generic function with 1 method)\n\njulia> begin\n           @show get_global()\n           Core.eval(@__MODULE__, :(y = 2))\n           @show get_global()\n       end\nget_global() = 1\nget_global() = 2\n2\n\nOne particular special case of constant reassignment is the redefinition of struct types:\n\njulia> struct MyStruct\n           x::Int\n       end\n\njulia> const one_field = MyStruct(1)\nMyStruct(1)\n\njulia> struct MyStruct\n           x::Int\n           y::Float64\n       end\n\njulia> const two_field = MyStruct(1, 2.0)\nMyStruct(1, 2.0)\n\njulia> one_field\n@world(MyStruct, 38452:38455)(1)\n\njulia> two_field\nMyStruct(1, 2.0)\n\nInternally the two definitions of MyStruct are entirely separate types. However, after the new MyStruct type is defined, there is no longer any default binding for the original definition of MyStruct. To nevertheless facilitate access to these types, the special @world macro may be used to access the meaning of a name in a previous world. However, this facility is intended for introspection only and in particular note that world age numbers are not stable across precompilation and should in general be treated opaquely.","page":"The World Age mechanism"},{"title":"Binding partition introspection","location":"manual/worldage.html#Binding-partition-introspection","category":"section","text":"In certain cases, it can be helpful to introspect the system's understanding of what a binding means in any particular world age. The default display printing of Core.Binding provides a helpful summary (e.g. on the MyStruct example from above):\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :MyStruct))\nBinding Main.MyStruct\n   38456:∞ - constant binding to MyStruct\n   38452:38455 - constant binding to @world(MyStruct, 38452:38455)\n   38451:38451 - backdated constant binding to @world(MyStruct, 38452:38455)\n   0:38450 - backdated constant binding to @world(MyStruct, 38452:38455)","page":"The World Age mechanism"},{"title":"World age and using/import","location":"manual/worldage.html#World-age-and-using/import","category":"section","text":"Bindings provided via using and import also operate via the world age mechanism. Binding resolution is a stateless function of the import and using definitions visible in the current world age. For example:\n\njulia> module M1; const x = 1; export x; end\n\njulia> module M2; const x = 2; export x; end\n\njulia> using .M1\n\njulia> x\n1\n\njulia> using .M2\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\nHint: It looks like two or more modules export different bindings with this name, resulting in ambiguity. Try explicitly importing it from a particular module, or qualifying the name with the module it should come from.\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :x))\nBinding Main.x\n   38458:∞ - ambiguous binding - guard entry\n   38457:38457 - implicit `using` resolved to constant 1","page":"The World Age mechanism"},{"title":"World age capture","location":"manual/worldage.html#World-age-capture","category":"section","text":"Certain language features capture the current task's world age. Perhaps the most common of these is creation of new tasks. Newly created tasks will inherit the creating task's local world age at creation time and will retain said world age (unless explicitly raised) even if the originating tasks raises its world age:\n\njulia> const x = 1\n\njulia> t = @task (wait(); println(\"Running now\"); x);\n\njulia> const x = 2\n\njulia> schedule(t);\nRunning now\n\njulia> x\n2\n\njulia> fetch(t)\n1\n\nIn addition to tasks, opaque closures also capture their world age at creation. See Base.Experimental.@opaque.","page":"The World Age mechanism"},{"title":"Base.@world","location":"manual/worldage.html#Base.@world","category":"macro","text":"@world(sym, world)\n\nResolve the binding sym in world world. See invoke_in_world for running arbitrary code in fixed worlds. world may be UnitRange, in which case the macro will error unless the binding is valid and has the same value across the entire world range.\n\nAs a special case, the world ∞ always refers to the latest world, even if that world is newer than the world currently running.\n\nThe @world macro is primarily used in the printing of bindings that are no longer available in the current world.\n\nExample\n\njulia> struct Foo; a::Int; end\nFoo\n\njulia> fold = Foo(1)\n\njulia> Int(Base.get_world_counter())\n26866\n\njulia> struct Foo; a::Int; b::Int end\nFoo\n\njulia> fold\n@world(Foo, 26866)(1)\n\ncompat: Julia 1.12\nThis functionality requires at least Julia 1.12.\n\n\n\n\n\n","page":"The World Age mechanism"},{"title":"Base.get_world_counter","location":"manual/worldage.html#Base.get_world_counter","category":"function","text":"get_world_counter()\n\nReturns the current maximum world-age counter. This counter is global and monotonically increasing.\n\n\n\n\n\n","page":"The World Age mechanism"},{"title":"Base.tls_world_age","location":"manual/worldage.html#Base.tls_world_age","category":"function","text":"tls_world_age()\n\nReturns the world the current_task() is executing within.\n\n\n\n\n\n","page":"The World Age mechanism"},{"title":"Base.invoke_in_world","location":"manual/worldage.html#Base.invoke_in_world","category":"function","text":"invoke_in_world(world, f, args...; kwargs...)\n\nCall f(args...; kwargs...) in a fixed world age, world.\n\nThis is useful for infrastructure running in the user's Julia session which is not part of the user's program. For example, things related to the REPL, editor support libraries, etc. In these cases it can be useful to prevent unwanted method invalidation and recompilation latency, and to prevent the user from breaking supporting infrastructure by mistake.\n\nThe global world age can be queried using Base.get_world_counter() and stored for later use within the lifetime of the current Julia session, or when serializing and reloading the system image.\n\nTechnically, invoke_in_world will prevent any function called by f from being extended by the user during their Julia session. That is, generic function method tables seen by f (and any functions it calls) will be frozen as they existed at the given world age. In a sense, this is like the opposite of invokelatest.\n\nnote: Note\nIt is not valid to store world ages obtained in precompilation for later use. This is because precompilation generates a \"parallel universe\" where the world age refers to system state unrelated to the main Julia session.\n\n\n\n\n\n","page":"The World Age mechanism"},{"title":"Base.Experimental.@opaque","location":"manual/worldage.html#Base.Experimental.@opaque","category":"macro","text":"@opaque ([type, ]args...) -> body\n\nMarks a given closure as \"opaque\". Opaque closures capture the world age of their creation (as opposed to their invocation). This allows for more aggressive optimization of the capture list, but trades off against the ability to inline opaque closures at the call site, if their creation is not statically visible.\n\nAn argument tuple type (type) may optionally be specified, to specify allowed argument types in a more flexible way. In particular, the argument type may be fixed length even if the function is variadic.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n","page":"The World Age mechanism"},{"title":"isbits Union Optimizations","location":"devdocs/isbitsunionarrays.html#isbits-Union-Optimizations","category":"section","text":"In Julia, the Array type holds both \"bits\" values as well as heap-allocated \"boxed\" values. The distinction is whether the value itself is stored inline (in the direct allocated memory of the array), or if the memory of the array is simply a collection of pointers to objects allocated elsewhere. In terms of performance, accessing values inline is clearly an advantage over having to follow a pointer to the actual value. The definition of \"isbits\" generally means any Julia type with a fixed, determinate size, meaning no \"pointer\" fields, see ?isbitstype.\n\nJulia also supports Union types, quite literally the union of a set of types. Custom Union type definitions can be extremely handy for applications wishing to \"cut across\" the nominal type system (i.e. explicit subtype relationships) and define methods or functionality on these, otherwise unrelated, set of types. A compiler challenge, however, is in determining how to treat these Union types. The naive approach (and indeed, what Julia itself did pre-0.7), is to simply make a \"box\" and then a pointer in the box to the actual value, similar to the previously mentioned \"boxed\" values. This is unfortunate, however, because of the number of small, primitive \"bits\" types (think UInt8, Int32, Float64, etc.) that would easily fit themselves inline in this \"box\" without needing any indirection for value access. There are two main ways Julia can take advantage of this optimization as of 0.7: isbits Union fields in types, and isbits Union Arrays.","page":"isbits Union Optimizations"},{"title":"isbits Union Structs","location":"devdocs/isbitsunionarrays.html#isbits-Union-Structs","category":"section","text":"Julia now includes an optimization wherein \"isbits Union\" fields in types (mutable struct, struct, etc.) will be stored inline. This is accomplished by determining the \"inline size\" of the Union type (e.g. Union{UInt8, Int16} will have a size of two bytes, which represents the size needed of the largest Union type Int16), and in addition, allocating an extra \"type tag byte\" (UInt8), whose value signals the type of the actual value stored inline of the \"Union bytes\". The type tag byte value is the index of the actual value's type in the Union type's order of types. For example, a type tag value of 0x02 for a field with type Union{Nothing, UInt8, Int16} would indicate that an Int16 value is stored in the 16 bits of the field in the structure's memory; a 0x01 value would indicate that a UInt8 value was stored in the first 8 bits of the 16 bits of the field's memory. Lastly, a value of 0x00 signals that the nothing value will be returned for this field, even though, as a singleton type with a single type instance, it technically has a size of 0. The type tag byte for a type's Union field is stored directly after the field's computed Union memory.","page":"isbits Union Optimizations"},{"title":"isbits Union Memory","location":"devdocs/isbitsunionarrays.html#isbits-Union-Memory","category":"section","text":"Julia can now also store \"isbits Union\" values inline in a Memory, as opposed to requiring an indirection box. The optimization is accomplished by storing an extra \"type tag memory\" of bytes, one byte per element, alongside the bytes of the actual data. This type tag memory serves the same function as the type field case: its value signals the type of the actual stored Union value. The \"type tag memory\" directly follows the regular data space. So the formula to access an isbits Union Array's type tag bytes is a->data + a->length * a->elsize.","page":"isbits Union Optimizations"},{"title":"SubArrays","location":"devdocs/subarrays.html#SubArrays","category":"section","text":"Julia's SubArray type is a container encoding a \"view\" of a parent AbstractArray. This page documents some of the design principles and implementation of SubArrays.\n\nOne of the major design goals is to ensure high performance for views of both IndexLinear and IndexCartesian arrays. Furthermore, views of IndexLinear arrays should themselves be IndexLinear to the extent that it is possible.","page":"SubArrays"},{"title":"Index replacement","location":"devdocs/subarrays.html#Index-replacement","category":"section","text":"Consider making 2d slices of a 3d array:\n\njulia> A = rand(2,3,4);\n\njulia> S1 = view(A, :, 1, 2:3)\n2×2 view(::Array{Float64, 3}, :, 1, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.967143  0.103929\n\njulia> S2 = view(A, 1, :, 2:3)\n3×2 view(::Array{Float64, 3}, 1, :, 2:3) with eltype Float64:\n 0.839622  0.711389\n 0.789764  0.806704\n 0.566704  0.962715\n\nview drops \"singleton\" dimensions (ones that are specified by an Int), so both S1 and S2 are two-dimensional SubArrays. Consequently, the natural way to index these is with S1[i,j]. To extract the value from the parent array A, the natural approach is to replace S1[i,j] with A[i,1,(2:3)[j]] and S2[i,j] with A[1,i,(2:3)[j]].\n\nThe key feature of the design of SubArrays is that this index replacement can be performed without any runtime overhead.","page":"SubArrays"},{"title":"SubArray design","location":"devdocs/subarrays.html#SubArray-design","category":"section","text":"","page":"SubArrays"},{"title":"Type parameters and fields","location":"devdocs/subarrays.html#Type-parameters-and-fields","category":"section","text":"The strategy adopted is first and foremost expressed in the definition of the type:\n\nstruct SubArray{T,N,P,I,L} <: AbstractArray{T,N}\n    parent::P\n    indices::I\n    offset1::Int       # for linear indexing and pointer, only valid when L==true\n    stride1::Int       # used only for linear indexing\n    ...\nend\n\nSubArray has 5 type parameters. The first two are the standard element type and dimensionality.  The next is the type of the parent AbstractArray. The most heavily-used is the fourth parameter, a Tuple of the types of the indices for each dimension. The final one, L, is only provided as a convenience for dispatch; it's a boolean that represents whether the index types support fast linear indexing. More on that later.\n\nIf in our example above A is a Array{Float64, 3}, our S1 case above would be a SubArray{Float64,2,Array{Float64,3},Tuple{Base.Slice{Base.OneTo{Int64}},Int64,UnitRange{Int64}},false}. Note in particular the tuple parameter, which stores the types of the indices used to create S1. Likewise,\n\njulia> S1.indices\n(Base.Slice(Base.OneTo(2)), 1, 2:3)\n\nStoring these values allows index replacement, and having the types encoded as parameters allows one to dispatch to efficient algorithms.","page":"SubArrays"},{"title":"Index translation","location":"devdocs/subarrays.html#Index-translation","category":"section","text":"Performing index translation requires that you do different things for different concrete SubArray types. For example, for S1, one needs to apply the i,j indices to the first and third dimensions of the parent array, whereas for S2 one needs to apply them to the second and third. The simplest approach to indexing would be to do the type-analysis at runtime:\n\nparentindices = Vector{Any}()\nfor thisindex in S.indices\n    ...\n    if isa(thisindex, Int)\n        # Don't consume one of the input indices\n        push!(parentindices, thisindex)\n    elseif isa(thisindex, AbstractVector)\n        # Consume an input index\n        push!(parentindices, thisindex[inputindex[j]])\n        j += 1\n    elseif isa(thisindex, AbstractMatrix)\n        # Consume two input indices\n        push!(parentindices, thisindex[inputindex[j], inputindex[j+1]])\n        j += 2\n    elseif ...\nend\nS.parent[parentindices...]\n\nUnfortunately, this would be disastrous in terms of performance: each element access would allocate memory, and involves the running of a lot of poorly-typed code.\n\nThe better approach is to dispatch to specific methods to handle each type of stored index. That's what reindex does: it dispatches on the type of the first stored index and consumes the appropriate number of input indices, and then it recurses on the remaining indices. In the case of S1, this expands to\n\nBase.reindex(S1, S1.indices, (i, j)) == (i, S1.indices[2], S1.indices[3][j])\n\nfor any pair of indices (i,j) (except CartesianIndexs and arrays thereof, see below).\n\nThis is the core of a SubArray; indexing methods depend upon reindex to do this index translation. Sometimes, though, we can avoid the indirection and make it even faster.","page":"SubArrays"},{"title":"Linear indexing","location":"devdocs/subarrays.html#Linear-indexing","category":"section","text":"Linear indexing can be implemented efficiently when the entire array has a single stride that separates successive elements, starting from some offset. This means that we can pre-compute these values and represent linear indexing simply as an addition and multiplication, avoiding the indirection of reindex and (more importantly) the slow computation of the cartesian coordinates entirely.\n\nFor SubArray types, the availability of efficient linear indexing is based purely on the types of the indices, and does not depend on values like the size of the parent array. You can ask whether a given set of indices supports fast linear indexing with the internal Base.viewindexing function:\n\njulia> Base.viewindexing(S1.indices)\nIndexCartesian()\n\njulia> Base.viewindexing(S2.indices)\nIndexLinear()\n\nThis is computed during construction of the SubArray and stored in the L type parameter as a boolean that encodes fast linear indexing support. While not strictly necessary, it means that we can define dispatch directly on SubArray{T,N,A,I,true} without any intermediaries.\n\nSince this computation doesn't depend on runtime values, it can miss some cases in which the stride happens to be uniform:\n\njulia> A = reshape(1:4*2, 4, 2)\n4×2 reshape(::UnitRange{Int64}, 4, 2) with eltype Int64:\n 1  5\n 2  6\n 3  7\n 4  8\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 2\n 2\n\nA view constructed as view(A, 2:2:4, :) happens to have uniform stride, and therefore linear indexing indeed could be performed efficiently. However, success in this case depends on the size of the array: if the first dimension instead were odd,\n\njulia> A = reshape(1:5*2, 5, 2)\n5×2 reshape(::UnitRange{Int64}, 5, 2) with eltype Int64:\n 1   6\n 2   7\n 3   8\n 4   9\n 5  10\n\njulia> diff(A[2:2:4,:][:])\n3-element Vector{Int64}:\n 2\n 3\n 2\n\nthen A[2:2:4,:] does not have uniform stride, so we cannot guarantee efficient linear indexing.  Since we have to base this decision based purely on types encoded in the parameters of the SubArray, S = view(A, 2:2:4, :) cannot implement efficient linear indexing.","page":"SubArrays"},{"title":"A few details","location":"devdocs/subarrays.html#A-few-details","category":"section","text":"Note that the Base.reindex function is agnostic to the types of the input indices; it simply determines how and where the stored indices should be reindexed. It not only supports integer indices, but it supports non-scalar indexing, too. This means that views of views don't need two levels of indirection; they can simply re-compute the indices into the original parent array!\nHopefully by now it's fairly clear that supporting slices means that the dimensionality, given by the parameter N, is not necessarily equal to the dimensionality of the parent array or the length of the indices tuple. Neither do user-supplied indices necessarily line up with entries in the indices tuple (e.g., the second user-supplied index might correspond to the third dimension of the parent array, and the third element in the indices tuple).\nWhat might be less obvious is that the dimensionality of the stored parent array must be equal to the number of effective indices in the indices tuple. Some examples:\nA = reshape(1:35, 5, 7) # A 2d parent Array\nS = view(A, 2:7)         # A 1d view created by linear indexing\nS = view(A, :, :, 1:1)   # Appending extra indices is supported\nNaively, you'd think you could just set S.parent = A and S.indices = (:,:,1:1), but supporting this dramatically complicates the reindexing process, especially for views of views. Not only do you need to dispatch on the types of the stored indices, but you need to examine whether a given index is the final one and \"merge\" any remaining stored indices together. This is not an easy task, and even worse: it's slow since it implicitly depends upon linear indexing.\nFortunately, this is precisely the computation that ReshapedArray performs, and it does so linearly if possible. Consequently, view ensures that the parent array is the appropriate dimensionality for the given indices by reshaping it if needed. The inner SubArray constructor ensures that this invariant is satisfied.\nCartesianIndex and arrays thereof throw a nasty wrench into the reindex scheme. Recall that reindex simply dispatches on the type of the stored indices in order to determine how many passed indices should be used and where they should go. But with CartesianIndex, there's no longer a one-to-one correspondence between the number of passed arguments and the number of dimensions that they index into. If we return to the above example of Base.reindex(S1, S1.indices, (i, j)), you can see that the expansion is incorrect for i, j = CartesianIndex(), CartesianIndex(2,1). It should skip the CartesianIndex() entirely and return:\n(CartesianIndex(2,1)[1], S1.indices[2], S1.indices[3][CartesianIndex(2,1)[2]])\nInstead, though, we get:\n(CartesianIndex(), S1.indices[2], S1.indices[3][CartesianIndex(2,1)])\nDoing this correctly would require combined dispatch on both the stored and passed indices across all combinations of dimensionalities in an intractable manner. As such, reindex must never be called with CartesianIndex indices. Fortunately, the scalar case is easily handled by first flattening the CartesianIndex arguments to plain integers. Arrays of CartesianIndex, however, cannot be split apart into orthogonal pieces so easily. Before attempting to use reindex, view must ensure that there are no arrays of CartesianIndex in the argument list. If there are, it can simply \"punt\" by avoiding the reindex calculation entirely, constructing a nested SubArray with two levels of indirection instead.","page":"SubArrays"},{"title":"Essentials","location":"base/base.html#Essentials","category":"section","text":"","page":"Essentials"},{"title":"Introduction","location":"base/base.html#Introduction","category":"section","text":"Julia Base contains a range of functions and macros appropriate for performing scientific and numerical computing, but is also as broad as those of many general purpose programming languages. Additional functionality is available from a growing collection of available packages. Functions are grouped by topic below.\n\nSome general notes:\n\nTo use module functions, use import Module to import the module, and Module.fn(x) to use the functions.\nAlternatively, using Module will import all exported Module functions into the current namespace.\nBy convention, function names ending with an exclamation point (!) modify their arguments. Some functions have both modifying (e.g., sort!) and non-modifying (sort) versions.\n\nThe behaviors of Base and standard libraries are stable as defined in SemVer only if they are documented; i.e., included in the Julia documentation and not marked as unstable. See API FAQ for more information.","page":"Essentials"},{"title":"Getting Around","location":"base/base.html#Getting-Around","category":"section","text":"","page":"Essentials"},{"title":"Keywords","location":"base/base.html#Keywords","category":"section","text":"This is the list of reserved keywords in Julia: baremodule, begin, break, catch, const, continue, do, else, elseif, end, export, false, finally, for, function, global, if, import, let, local, macro, module, quote, return, struct, true, try, using, while. Those keywords are not allowed to be used as variable names.\n\nThe following two-word sequences are reserved: abstract type, mutable struct, primitive type. However, you can create variables with names: abstract, mutable, primitive and type.\n\nFinally: where is parsed as an infix operator for writing parametric method and type definitions; in and isa are parsed as infix operators; public is parsed as a keyword when beginning a toplevel statement; outer is parsed as a keyword when used to modify the scope of a variable in an iteration specification of a for loop; and as is used as a keyword to rename an identifier brought into scope by import or using. Creation of variables named where, in, isa, outer and as is allowed, though.","page":"Essentials"},{"title":"Standard Modules","location":"base/base.html#standard-modules","category":"section","text":"","page":"Essentials"},{"title":"Base Submodules","location":"base/base.html#base-submodules","category":"section","text":"","page":"Essentials"},{"title":"All Objects","location":"base/base.html#All-Objects","category":"section","text":"","page":"Essentials"},{"title":"Properties of Types","location":"base/base.html#Properties-of-Types","category":"section","text":"","page":"Essentials"},{"title":"Type relations","location":"base/base.html#Type-relations","category":"section","text":"","page":"Essentials"},{"title":"Declared structure","location":"base/base.html#Declared-structure","category":"section","text":"","page":"Essentials"},{"title":"Memory layout","location":"base/base.html#Memory-layout","category":"section","text":"","page":"Essentials"},{"title":"Special values","location":"base/base.html#Special-values","category":"section","text":"","page":"Essentials"},{"title":"Special Types","location":"base/base.html#Special-Types","category":"section","text":"","page":"Essentials"},{"title":"Generic Functions","location":"base/base.html#Generic-Functions","category":"section","text":"","page":"Essentials"},{"title":"Syntax","location":"base/base.html#Syntax","category":"section","text":"","page":"Essentials"},{"title":"Managing deprecations","location":"base/base.html#Managing-deprecations","category":"section","text":"","page":"Essentials"},{"title":"Missing Values","location":"base/base.html#Missing-Values","category":"section","text":"","page":"Essentials"},{"title":"System","location":"base/base.html#System","category":"section","text":"","page":"Essentials"},{"title":"Versioning","location":"base/base.html#Versioning","category":"section","text":"","page":"Essentials"},{"title":"Errors","location":"base/base.html#Errors","category":"section","text":"","page":"Essentials"},{"title":"Events","location":"base/base.html#Events","category":"section","text":"","page":"Essentials"},{"title":"Reflection","location":"base/base.html#Reflection","category":"section","text":"","page":"Essentials"},{"title":"Documentation","location":"base/base.html#Documentation","category":"section","text":"(See also the documentation chapter.)","page":"Essentials"},{"title":"Code loading","location":"base/base.html#Code-loading","category":"section","text":"","page":"Essentials"},{"title":"Internals","location":"base/base.html#Internals","category":"section","text":"","page":"Essentials"},{"title":"Meta","location":"base/base.html#Meta","category":"section","text":"","page":"Essentials"},{"title":"Base.exit","location":"base/base.html#Base.exit","category":"function","text":"exit(code=0)\n\nStop the program with an exit code. The default exit code is zero, indicating that the program completed successfully. In an interactive session, exit() can be called with the keyboard shortcut ^D.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.atexit","location":"base/base.html#Base.atexit","category":"function","text":"atexit(f)\n\nRegister a zero- or one-argument function f() to be called at process exit. atexit() hooks are called in last in first out (LIFO) order and run before object finalizers.\n\nIf f has a method defined for one integer argument, it will be called as f(n::Int32), where n is the current exit code, otherwise it will be called as f().\n\ncompat: Julia 1.9\nThe one-argument form requires Julia 1.9\n\nExit hooks are allowed to call exit(n), in which case Julia will exit with exit code n (instead of the original exit code). If more than one exit hook calls exit(n), then Julia will exit with the exit code corresponding to the last called exit hook that calls exit(n). (Because exit hooks are called in LIFO order, \"last called\" is equivalent to \"first registered\".)\n\nNote: Once all exit hooks have been called, no more exit hooks can be registered, and any call to atexit(f) after all hooks have completed will throw an exception. This situation may occur if you are registering exit hooks from background Tasks that may still be executing concurrently during shutdown.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isinteractive","location":"base/base.html#Base.isinteractive","category":"function","text":"isinteractive() -> Bool\n\nDetermine whether Julia is running an interactive session.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.summarysize","location":"base/base.html#Base.summarysize","category":"function","text":"Base.summarysize(obj; exclude=Union{...}, chargeall=Union{...}) -> Int\n\nCompute the amount of memory, in bytes, used by all unique objects reachable from the argument.\n\nKeyword Arguments\n\nexclude: specifies the types of objects to exclude from the traversal.\nchargeall: specifies the types of objects to always charge the size of all of their fields, even if those fields would normally be excluded.\n\nSee also sizeof.\n\nExamples\n\njulia> Base.summarysize(1.0)\n8\n\njulia> Base.summarysize(Ref(rand(100)))\n848\n\njulia> sizeof(Ref(rand(100)))\n8\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.__precompile__","location":"base/base.html#Base.__precompile__","category":"function","text":"__precompile__(isprecompilable::Bool)\n\nSpecify whether the file calling this function is precompilable, defaulting to true. If a module or file is not safely precompilable, it should call __precompile__(false) in order to throw an error if Julia attempts to precompile it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include","location":"base/base.html#Base.include","category":"function","text":"Base.include([mapexpr::Function,] m::Module, path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of module m. Every module (except those defined with baremodule) has its own definition of include omitting the m argument, which evaluates the file in that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n","page":"Essentials"},{"title":"include","location":"base/base.html#include","category":"function","text":"include([mapexpr::Function,] path::AbstractString)\n\nEvaluate the contents of the input source file in the global scope of the containing module. Every Module (except those defined with baremodule) has a private 1-argument definition of include, which evaluates the file in that module, for use inside that module. Returns the result of the last evaluated expression of the input file. During including, a task-local include path is set to the directory containing the file. Nested calls to include will search relative to that path. This function is typically used to load source interactively, or to combine files in packages that are broken into multiple source files. The argument path is normalized using normpath which will resolve relative path tokens such as .. and convert / to the appropriate path separator.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in path, the include function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\nUse Base.include to evaluate a file into another module.\n\nnote: Note\nJulia's syntax lowering recognizes an explicit call to a literal include at top-level and inserts an implicit @Core.latestworld to make any include'd definitions visible to subsequent code. Note however that this recognition is syntactic. I.e. assigning const myinclude = include may require and explicit @Core.latestworld call after myinclude.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include_string","location":"base/base.html#Base.include_string","category":"function","text":"include_string([mapexpr::Function,] m::Module, code::AbstractString, filename::AbstractString=\"string\")\n\nLike include, except reads code from the given string rather than from a file.\n\nThe optional first argument mapexpr can be used to transform the included code before it is evaluated: for each parsed expression expr in code, the include_string function actually evaluates mapexpr(expr).  If it is omitted, mapexpr defaults to identity.\n\ncompat: Julia 1.5\nJulia 1.5 is required for passing the mapexpr argument.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.include_dependency","location":"base/base.html#Base.include_dependency","category":"function","text":"include_dependency(path::AbstractString; track_content::Bool=true)\n\nIn a module, declare that the file, directory, or symbolic link specified by path (relative or absolute) is a dependency for precompilation; that is, if track_content=true the module will need to be recompiled if the content of path changes (if path is a directory the content equals join(readdir(path))). If track_content=false recompilation is triggered when the modification time mtime of path changes.\n\nThis is only needed if your module depends on a path that is not used via include. It has no effect outside of compilation.\n\ncompat: Julia 1.11\nKeyword argument track_content requires at least Julia 1.11. An error is now thrown if path is not readable.\n\n\n\n\n\n","page":"Essentials"},{"title":"__init__","location":"base/base.html#__init__","category":"keyword","text":"__init__\n\nThe __init__() function in a module executes immediately after the module is loaded at runtime for the first time. It is called once, after all other statements in the module have been executed. Because it is called after fully importing the module, __init__ functions of submodules will be executed first. Two typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries. See the manual section about modules for more details.\n\nSee also: OncePerProcess.\n\nExamples\n\nconst foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.OncePerProcess","location":"base/base.html#Base.OncePerProcess","category":"type","text":"OncePerProcess{T}(init::Function)() -> T\n\nCalling a OncePerProcess object returns a value of type T by running the function initializer exactly once per process. All concurrent and future calls in the same process will return exactly the same value. This is useful in code that will be precompiled, as it allows setting up caches or other state which won't get serialized.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const global_state = Base.OncePerProcess{Vector{UInt32}}() do\n           println(\"Making lazy global value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (procstate = global_state()) |> typeof\nMaking lazy global value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> procstate === global_state()\ntrue\n\njulia> procstate === fetch(@async global_state())\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.OncePerTask","location":"base/base.html#Base.OncePerTask","category":"type","text":"OncePerTask{T}(init::Function)() -> T\n\nCalling a OncePerTask object returns a value of type T by running the function initializer exactly once per Task. All future calls in the same Task will return exactly the same value.\n\nSee also: task_local_storage.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const task_state = Base.OncePerTask{Vector{UInt32}}() do\n           println(\"Making lazy task value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (taskvec = task_state()) |> typeof\nMaking lazy task value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> taskvec === task_state()\ntrue\n\njulia> taskvec === fetch(@async task_state())\nMaking lazy task value...done.\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.OncePerThread","location":"base/base.html#Base.OncePerThread","category":"type","text":"OncePerThread{T}(init::Function)() -> T\n\nCalling a OncePerThread object returns a value of type T by running the function initializer exactly once per thread. All future calls in the same thread, and concurrent or future calls with the same thread id, will return exactly the same value. The object can also be indexed by the threadid for any existing thread, to get (or initialize on this thread) the value stored for that thread. Incorrect usage can lead to data-races or memory corruption so use only if that behavior is correct within your library's threading-safety design.\n\nwarning: Warning\nIt is not necessarily true that a Task only runs on one thread, therefore the value returned here may alias other values or change in the middle of your program. This function may get deprecated in the future. If initializer yields, the thread running the current task after the call might not be the same as the one at the start of the call.\n\nSee also: OncePerTask.\n\ncompat: Julia 1.12\nThis type requires Julia 1.12 or later.\n\nExample\n\njulia> const thread_state = Base.OncePerThread{Vector{UInt32}}() do\n           println(\"Making lazy thread value...done.\")\n           return [Libc.rand()]\n       end;\n\njulia> (threadvec = thread_state()) |> typeof\nMaking lazy thread value...done.\nVector{UInt32} (alias for Array{UInt32, 1})\n\njulia> threadvec === fetch(@async thread_state())\ntrue\n\njulia> threadvec === thread_state[Threads.threadid()]\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.which","location":"base/base.html#Base.which-Tuple{Any, Any}","category":"method","text":"which(f, types)\n\nReturns the method of f (a Method object) that would be called for arguments of the given types.\n\nIf types is an abstract type, then the method that would be called by invoke is returned.\n\nSee also: parentmodule, @which, and @edit.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.methods","location":"base/base.html#Base.methods","category":"function","text":"methods(f, [types], [module])\n\nReturn the method table for f.\n\nIf types is specified, return an array of methods whose types match. If module is specified, return an array of methods defined in that module. A list of modules can also be specified as an array or set.\n\ncompat: Julia 1.4\nAt least Julia 1.4 is required for specifying a module.\n\nSee also: which, @which and methodswith.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@show","location":"base/base.html#Base.@show","category":"macro","text":"@show exs...\n\nPrints one or more expressions, and their results, to stdout, and returns the last result.\n\nSee also: show, @info, println.\n\nExamples\n\njulia> x = @show 1+2\n1 + 2 = 3\n3\n\njulia> @show x^2 x/2;\nx ^ 2 = 9\nx / 2 = 1.5\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MainInclude.ans","location":"base/base.html#Base.MainInclude.ans","category":"constant","text":"ans\n\nA variable referring to the last computed value, automatically imported to the interactive prompt.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MainInclude.err","location":"base/base.html#Base.MainInclude.err","category":"constant","text":"err\n\nA variable referring to the last thrown errors, automatically imported to the interactive prompt. The thrown errors are collected in a stack of exceptions.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.active_project","location":"base/base.html#Base.active_project","category":"function","text":"active_project()\n\nReturn the path of the active Project.toml file. See also Base.set_active_project.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.set_active_project","location":"base/base.html#Base.set_active_project","category":"function","text":"set_active_project(projfile::Union{AbstractString,Nothing})\n\nSet the active Project.toml file to projfile. See also Base.active_project.\n\ncompat: Julia 1.8\nThis function requires at least Julia 1.8.\n\n\n\n\n\n","page":"Essentials"},{"title":"module","location":"base/base.html#module","category":"keyword","text":"module\n\nmodule declares a Module, which is a separate global variable workspace. Within a module, you can control which names from other modules are visible (via importing), and specify which of your names are intended to be public (via export and public). Modules allow you to create top-level definitions without worrying about name conflicts when your code is used together with somebody else’s. See the manual section about modules for more details.\n\nExamples\n\nmodule Foo\nimport Base.show\nexport MyType, foo\n\nstruct MyType\n    x\nend\n\nbar(x) = 2x\nfoo(a::MyType) = bar(a.x) + 1\nshow(io::IO, a::MyType) = print(io, \"MyType $(a.x)\")\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"export","location":"base/base.html#export","category":"keyword","text":"export\n\nexport is used within modules to tell Julia which names should be made available to the user. For example: export foo makes the name foo available when using the module. See the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"public","location":"base/base.html#public","category":"keyword","text":"public\n\npublic is used within modules to tell Julia which names are part of the public API of the module. For example: public foo indicates that the name foo is public, without making it available when using the module.\n\nAs export already indicates that a name is public, it is unnecessary and an error to declare a name both as public and as exported. See the manual section about modules for details.\n\ncompat: Julia 1.11\nThe public keyword was added in Julia 1.11. Prior to this the notion of publicness was less explicit.\n\n\n\n\n\n","page":"Essentials"},{"title":"import","location":"base/base.html#import","category":"keyword","text":"import\n\nimport Foo will load the module or package Foo. Names from the imported Foo module can be accessed with dot syntax (e.g. Foo.foo to access the name foo). See the manual section about modules for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"using","location":"base/base.html#using","category":"keyword","text":"using\n\nusing Foo will load the module or package Foo and make its exported names available for direct use. Names can also be used via dot syntax (e.g. Foo.foo to access the name foo), whether they are exported or not. See the manual section about modules for details.\n\nnote: Note\nWhen two or more packages/modules export a name and that name does not refer to the same thing in each of the packages, and the packages are loaded via using without an explicit list of names, it is an error to reference that name without qualification. It is thus recommended that code intended to be forward-compatible with future versions of its dependencies and of Julia, e.g., code in released packages, list the names it uses from each loaded package, e.g., using Foo: Foo, f rather than using Foo.\n\n\n\n\n\n","page":"Essentials"},{"title":"as","location":"base/base.html#as","category":"keyword","text":"as\n\nas is used as a keyword to rename an identifier brought into scope by import or using, for the purpose of working around name conflicts as well as for shortening names.  (Outside of import or using statements, as is not a keyword and can be used as an ordinary identifier.)\n\nimport LinearAlgebra as LA brings the imported LinearAlgebra standard library into scope as LA.\n\nimport LinearAlgebra: eigen as eig, cholesky as chol brings the eigen and cholesky methods from LinearAlgebra into scope as eig and chol respectively.\n\nas works with using only when individual identifiers are brought into scope. For example, using LinearAlgebra: eigen as eig or using LinearAlgebra: eigen as eig, cholesky as chol works, but using LinearAlgebra as LA is invalid syntax, since it is nonsensical to rename all exported names from LinearAlgebra to LA.\n\n\n\n\n\n","page":"Essentials"},{"title":"baremodule","location":"base/base.html#baremodule","category":"keyword","text":"baremodule\n\nbaremodule declares a module that does not contain using Base or local definitions of eval and include. It does still import Core. In other words,\n\nmodule Mod\n\n...\n\nend\n\nis equivalent to\n\nbaremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"function","location":"base/base.html#function","category":"keyword","text":"function\n\nFunctions are defined with the function keyword:\n\nfunction add(a, b)\n    return a + b\nend\n\nOr the short form notation:\n\nadd(a, b) = a + b\n\nThe use of the return keyword is exactly the same as in other languages, but is often optional. A function without an explicit return statement will return the last expression in the function body.\n\n\n\n\n\n","page":"Essentials"},{"title":"macro","location":"base/base.html#macro","category":"keyword","text":"macro\n\nmacro defines a method for inserting generated code into a program. A macro maps a sequence of argument expressions to a returned expression, and the resulting expression is substituted directly into the program at the point where the macro is invoked. Macros are a way to run generated code without calling eval, since the generated code instead simply becomes part of the surrounding program. Macro arguments may include expressions, literal values, and symbols. Macros can be defined for variable number of arguments (varargs), but do not accept keyword arguments. Every macro also implicitly gets passed the arguments __source__, which contains the line number and file name the macro is called from, and __module__, which is the module the macro is expanded in.\n\nSee the manual section on Metaprogramming for more information about how to write a macro.\n\nExamples\n\njulia> macro sayhello(name)\n           return :( println(\"Hello, \", $name, \"!\") )\n       end\n@sayhello (macro with 1 method)\n\njulia> @sayhello \"Charlie\"\nHello, Charlie!\n\njulia> macro saylots(x...)\n           return :( println(\"Say: \", $(x...)) )\n       end\n@saylots (macro with 1 method)\n\njulia> @saylots \"hey \" \"there \" \"friend\"\nSay: hey there friend\n\n\n\n\n\n","page":"Essentials"},{"title":"return","location":"base/base.html#return","category":"keyword","text":"return\n\nreturn x causes the enclosing function to exit early, passing the given value x back to its caller. return by itself with no value is equivalent to return nothing (see nothing).\n\nfunction compare(a, b)\n    a == b && return \"equal to\"\n    a < b ? \"less than\" : \"greater than\"\nend\n\nIn general you can place a return statement anywhere within a function body, including within deeply nested loops or conditionals, but be careful with do blocks. For example:\n\nfunction test1(xs)\n    for x in xs\n        iseven(x) && return 2x\n    end\nend\n\nfunction test2(xs)\n    map(xs) do x\n        iseven(x) && return 2x\n        x\n    end\nend\n\nIn the first example, the return breaks out of test1 as soon as it hits an even number, so test1([5,6,7]) returns 12.\n\nYou might expect the second example to behave the same way, but in fact the return there only breaks out of the inner function (inside the do block) and gives a value back to map. test2([5,6,7]) then returns [5,12,7].\n\nWhen used in a top-level expression (i.e. outside any function), return causes the entire current top-level expression to terminate early.\n\n\n\n\n\n","page":"Essentials"},{"title":"do","location":"base/base.html#do","category":"keyword","text":"do\n\nCreate an anonymous function and pass it as the first argument to a function call. For example:\n\nmap(1:10) do x\n    2x\nend\n\nis equivalent to map(x->2x, 1:10).\n\nUse multiple arguments like so:\n\nmap(1:10, 11:20) do x, y\n    x + y\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"begin","location":"base/base.html#begin","category":"keyword","text":"begin\n\nbegin...end denotes a block of code.\n\nbegin\n    println(\"Hello, \")\n    println(\"World!\")\nend\n\nUsually begin will not be necessary, since keywords such as function and let implicitly begin blocks of code. See also ;.\n\nbegin may also be used when indexing to represent the first index of a collection or the first index of a dimension of an array. For example, a[begin] is the first element of an array a.\n\ncompat: Julia 1.4\nUse of begin as an index requires Julia 1.4 or later.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> A[begin, :]\n2-element Matrix{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Essentials"},{"title":"end","location":"base/base.html#end","category":"keyword","text":"end\n\nend marks the conclusion of a block of expressions, for example module, struct, mutable struct, begin, let, for etc.\n\nend may also be used when indexing to represent the last index of a collection or the last index of a dimension of an array.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> A[end, :]\n2-element Vector{Int64}:\n 3\n 4\n\n\n\n\n\n","page":"Essentials"},{"title":"let","location":"base/base.html#let","category":"keyword","text":"let\n\nlet blocks create a new hard scope and optionally introduce new local bindings.\n\nJust like the other scope constructs, let blocks define the block of code where newly introduced local variables are accessible. Additionally, the syntax has a special meaning for comma-separated assignments and variable names that may optionally appear on the same line as the let:\n\nlet var1 = value1, var2, var3 = value3\n    code\nend\n\nThe variables introduced on this line are local to the let block and the assignments are evaluated in order, with each right-hand side evaluated in the scope without considering the name on the left-hand side. Therefore it makes sense to write something like let x = x, since the two x variables are distinct with the left-hand side locally shadowing the x from the outer scope. This can even be a useful idiom as new local variables are freshly created each time local scopes are entered, but this is only observable in the case of variables that outlive their scope via closures.  A let variable without an assignment, such as var2 in the example above, declares a new local variable that is not yet bound to a value.\n\nBy contrast, begin blocks also group multiple expressions together but do not introduce scope or have the special assignment syntax.\n\nExamples\n\nIn the function below, there is a single x that is iteratively updated three times by the map. The closures returned all reference that one x at its final value:\n\njulia> function test_outer_x()\n           x = 0\n           map(1:3) do _\n               x += 1\n               return ()->x\n           end\n       end\ntest_outer_x (generic function with 1 method)\n\njulia> [f() for f in test_outer_x()]\n3-element Vector{Int64}:\n 3\n 3\n 3\n\nIf, however, we add a let block that introduces a new local variable we will end up with three distinct variables being captured (one at each iteration) even though we chose to use (shadow) the same name.\n\njulia> function test_let_x()\n           x = 0\n           map(1:3) do _\n               x += 1\n               let x = x\n                   return ()->x\n               end\n           end\n       end\ntest_let_x (generic function with 1 method)\n\njulia> [f() for f in test_let_x()]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\nAll scope constructs that introduce new local variables behave this way when repeatedly run; the distinctive feature of let is its ability to succinctly declare new locals that may shadow outer variables of the same name. For example, directly using the argument of the do function similarly captures three distinct variables:\n\njulia> function test_do_x()\n           map(1:3) do x\n               return ()->x\n           end\n       end\ntest_do_x (generic function with 1 method)\n\njulia> [f() for f in test_do_x()]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Essentials"},{"title":"if","location":"base/base.html#if","category":"keyword","text":"if/elseif/else\n\nif/elseif/else performs conditional evaluation, which allows portions of code to be evaluated or not evaluated depending on the value of a boolean expression. Here is the anatomy of the if/elseif/else conditional syntax:\n\nif x < y\n    println(\"x is less than y\")\nelseif x > y\n    println(\"x is greater than y\")\nelse\n    println(\"x is equal to y\")\nend\n\nIf the condition expression x < y is true, then the corresponding block is evaluated; otherwise the condition expression x > y is evaluated, and if it is true, the corresponding block is evaluated; if neither expression is true, the else block is evaluated. The elseif and else blocks are optional, and as many elseif blocks as desired can be used.\n\nIn contrast to some other languages conditions must be of type Bool. It does not suffice for conditions to be convertible to Bool.\n\njulia> if 1 end\nERROR: TypeError: non-boolean (Int64) used in boolean context\n\n\n\n\n\n","page":"Essentials"},{"title":"for","location":"base/base.html#for","category":"keyword","text":"for\n\nfor loops repeatedly evaluate a block of statements while iterating over a sequence of values.\n\nThe iteration variable is always a new variable, even if a variable of the same name exists in the enclosing scope. Use outer to reuse an existing local variable for iteration.\n\nExamples\n\njulia> for i in [1, 4, 0]\n           println(i)\n       end\n1\n4\n0\n\n\n\n\n\n","page":"Essentials"},{"title":"while","location":"base/base.html#while","category":"keyword","text":"while\n\nwhile loops repeatedly evaluate a conditional expression, and continue evaluating the body of the while loop as long as the expression remains true. If the condition expression is false when the while loop is first reached, the body is never evaluated.\n\nExamples\n\njulia> i = 1\n1\n\njulia> while i < 5\n           println(i)\n           global i += 1\n       end\n1\n2\n3\n4\n\n\n\n\n\n","page":"Essentials"},{"title":"break","location":"base/base.html#break","category":"keyword","text":"break\n\nBreak out of a loop immediately.\n\nExamples\n\njulia> i = 0\n0\n\njulia> while true\n           global i += 1\n           i > 5 && break\n           println(i)\n       end\n1\n2\n3\n4\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"continue","location":"base/base.html#continue","category":"keyword","text":"continue\n\nSkip the rest of the current loop iteration.\n\nExamples\n\njulia> for i = 1:6\n           iseven(i) && continue\n           println(i)\n       end\n1\n3\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"try","location":"base/base.html#try","category":"keyword","text":"try/catch\n\nA try/catch statement allows intercepting errors (exceptions) thrown by throw so that program execution can continue. For example, the following code attempts to write a file, but warns the user and proceeds instead of terminating execution if the file cannot be written:\n\ntry\n    open(\"/danger\", \"w\") do f\n        println(f, \"Hello\")\n    end\ncatch\n    @warn \"Could not write file.\"\nend\n\nor, when the file cannot be read into a variable:\n\nlines = try\n    open(\"/danger\", \"r\") do f\n        readlines(f)\n    end\ncatch\n    @warn \"File not found.\"\nend\n\nThe syntax catch e (where e is any variable) assigns the thrown exception object to the given variable within the catch block.\n\nThe power of the try/catch construct lies in the ability to unwind a deeply nested computation immediately to a much higher level in the stack of calling functions.\n\nA try/catch block can also have an else clause that executes only if no exception occurred:\n\ntry\n    a_dangerous_operation()\ncatch\n    @warn \"The operation failed.\"\nelse\n    @info \"The operation succeeded.\"\nend\n\nA try or try/catch block can also have a finally clause that executes at the end, regardless of whether an exception occurred.  For example, this can be used to guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    operate_on_file(f)\ncatch\n    @warn \"An error occurred!\"\nfinally\n    close(f)\nend\n\n(finally can also be used without a catch block.)\n\ncompat: Julia 1.8\nElse clauses require at least Julia 1.8.\n\n\n\n\n\n","page":"Essentials"},{"title":"finally","location":"base/base.html#finally","category":"keyword","text":"finally\n\nRun some code when a given try block of code exits, regardless of how it exits. For example, here is how we can guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    operate_on_file(f)\nfinally\n    close(f)\nend\n\nWhen control leaves the try block (for example, due to a return, or just finishing normally), close(f) will be executed. If the try block exits due to an exception, the exception will continue propagating. A catch block may be combined with try and finally as well. In this case the finally block will run after catch has handled the error.\n\n\n\n\n\n","page":"Essentials"},{"title":"quote","location":"base/base.html#quote","category":"keyword","text":"quote\n\nquote creates multiple expression objects in a block without using the explicit Expr constructor. For example:\n\nex = quote\n    x = 1\n    y = 2\n    x + y\nend\n\nUnlike the other means of quoting, :( ... ), this form introduces QuoteNode elements to the expression tree, which must be considered when directly manipulating the tree. For other purposes, :( ... ) and quote .. end blocks are treated identically.\n\n\n\n\n\n","page":"Essentials"},{"title":"local","location":"base/base.html#local","category":"keyword","text":"local\n\nlocal introduces a new local variable. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> function foo(n)\n           x = 0\n           for i = 1:n\n               local x # introduce a loop-local x\n               x = i\n           end\n           x\n       end\nfoo (generic function with 1 method)\n\njulia> foo(10)\n0\n\n\n\n\n\n","page":"Essentials"},{"title":"global","location":"base/base.html#global","category":"keyword","text":"global\n\nglobal x makes x in the current scope and its inner scopes refer to the global variable of that name. See the manual section on variable scoping for more information.\n\nExamples\n\njulia> z = 3\n3\n\njulia> function foo()\n           global z = 6 # use the z variable defined outside foo\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n6\n\njulia> z\n6\n\n\n\n\n\n","page":"Essentials"},{"title":"outer","location":"base/base.html#outer","category":"keyword","text":"for outer\n\nReuse an existing local variable for iteration in a for loop.\n\nSee the manual section on variable scoping for more information.\n\nSee also for.\n\nExamples\n\njulia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0\n\njulia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3\n\njulia> i = 0 # global variable\n       for outer i = 1:3\n       end\nERROR: syntax: no outer local variable declaration exists for \"for outer\"\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"const","location":"base/base.html#const","category":"keyword","text":"const\n\nconst is used to declare global variables whose values will not change. In almost all code (and particularly performance sensitive code) global variables should be declared constant in this way.\n\nconst x = 5\n\nMultiple variables can be declared within a single const:\n\nconst y, z = 7, 11\n\nNote that const only applies to one = operation, therefore const x = y = 1 declares x to be constant but not y. On the other hand, const x = const y = 1 declares both x and y constant.\n\nNote that \"constant-ness\" does not extend into mutable containers; only the association between a variable and its value is constant. If x is an array or dictionary (for example) you can still modify, add, or remove elements.\n\nIn some cases changing the value of a const variable gives a warning instead of an error. However, this can produce unpredictable behavior or corrupt the state of your program, and so should be avoided. This feature is intended only for convenience during interactive use.\n\n\n\n\n\n","page":"Essentials"},{"title":"struct","location":"base/base.html#struct","category":"keyword","text":"struct\n\nThe most commonly used kind of type in Julia is a struct, specified as a name and a set of fields.\n\nstruct Point\n    x\n    y\nend\n\nFields can have type restrictions, which may be parameterized:\n\nstruct Point{X}\n    x::X\n    y::Float64\nend\n\nA struct can also declare an abstract super type via <: syntax:\n\nstruct Point <: AbstractPoint\n    x\n    y\nend\n\nstructs are immutable by default; an instance of one of these types cannot be modified after construction. Use mutable struct instead to declare a type whose instances can be modified.\n\nSee the manual section on Composite Types for more details, such as how to define constructors.\n\n\n\n\n\n","page":"Essentials"},{"title":"mutable struct","location":"base/base.html#mutable struct","category":"keyword","text":"mutable struct\n\nmutable struct is similar to struct, but additionally allows the fields of the type to be set after construction.\n\nIndividual fields of a mutable struct can be marked as const to make them immutable:\n\nmutable struct Baz\n    a::Int\n    const b::Float64\nend\n\ncompat: Julia 1.8\nThe const keyword for fields of mutable structs requires at least Julia 1.8.\n\nSee the manual section on Composite Types for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@kwdef","location":"base/base.html#Base.@kwdef","category":"macro","text":"@kwdef typedef\n\nThis is a helper macro that automatically defines a keyword-based constructor for the type declared in the expression typedef, which must be a struct or mutable struct expression. The default argument is supplied by declaring fields of the form field::T = default or field = default. If no default is provided then the keyword argument becomes a required keyword argument in the resulting type constructor.\n\nInner constructors can still be defined, but at least one should accept arguments in the same form as the default inner constructor (i.e. one positional argument per field) in order to function correctly with the keyword outer constructor.\n\ncompat: Julia 1.1\nBase.@kwdef for parametric structs, and structs with supertypes requires at least Julia 1.1.\n\ncompat: Julia 1.9\nThis macro is exported as of Julia 1.9.\n\nExamples\n\njulia> @kwdef struct Foo\n           a::Int = 1         # specified default\n           b::String          # required keyword\n       end\nFoo\n\njulia> Foo(b=\"hi\")\nFoo(1, \"hi\")\n\njulia> Foo()\nERROR: UndefKeywordError: keyword argument `b` not assigned\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"abstract type","location":"base/base.html#abstract type","category":"keyword","text":"abstract type\n\nabstract type declares a type that cannot be instantiated, and serves only as a node in the type graph, thereby describing sets of related concrete types: those concrete types which are their descendants. Abstract types form the conceptual hierarchy which makes Julia’s type system more than just a collection of object implementations. For example:\n\nabstract type Number end\nabstract type Real <: Number end\n\nNumber has no supertype, whereas Real is an abstract subtype of Number.\n\n\n\n\n\n","page":"Essentials"},{"title":"primitive type","location":"base/base.html#primitive type","category":"keyword","text":"primitive type\n\nprimitive type declares a concrete type whose data consists only of a series of bits. Classic examples of primitive types are integers and floating-point values. Some example built-in primitive type declarations:\n\nprimitive type Char 32 end\nprimitive type Bool <: Integer 8 end\n\nThe number after the name indicates how many bits of storage the type requires. Currently, only sizes that are multiples of 8 bits are supported. The Bool declaration shows how a primitive type can be optionally declared to be a subtype of some supertype.\n\n\n\n\n\n","page":"Essentials"},{"title":"where","location":"base/base.html#where","category":"keyword","text":"where\n\nThe where keyword creates a UnionAll type, which may be thought of as an iterated union of other types, over all values of some variable. For example Vector{T} where T<:Real includes all Vectors where the element type is some kind of Real number.\n\nThe variable bound defaults to Any if it is omitted:\n\nVector{T} where T    # short for `where T<:Any`\n\nVariables can also have lower bounds:\n\nVector{T} where T>:Int\nVector{T} where Int<:T<:Real\n\nThere is also a concise syntax for nested where expressions. For example, this:\n\nPair{T, S} where S<:Array{T} where T<:Number\n\ncan be shortened to:\n\nPair{T, S} where {T<:Number, S<:Array{T}}\n\nThis form is often found on method signatures.\n\nNote that in this form, the variables are listed outermost-first. This matches the order in which variables are substituted when a type is \"applied\" to parameter values using the syntax T{p1, p2, ...}.\n\n\n\n\n\n","page":"Essentials"},{"title":"...","location":"base/base.html#...","category":"keyword","text":"...\n\nThe \"splat\" operator, ..., represents a sequence of arguments. ... can be used in function definitions, to indicate that the function accepts an arbitrary number of arguments. ... can also be used to apply a function to a sequence of arguments.\n\nExamples\n\njulia> add(xs...) = reduce(+, xs)\nadd (generic function with 1 method)\n\njulia> add(1, 2, 3, 4, 5)\n15\n\njulia> add([1, 2, 3]...)\n6\n\njulia> add(7, 1:100..., 1000:1100...)\n111107\n\n\n\n\n\n","page":"Essentials"},{"title":";","location":"base/base.html#;","category":"keyword","text":";\n\nSemicolons are used as statement separators and mark the beginning of keyword arguments in function declarations or calls.\n\n\n\n\n\n","page":"Essentials"},{"title":"=","location":"base/base.html#=","category":"keyword","text":"=\n\n= is the assignment operator.\n\nFor variable a and expression b, a = b makes a refer to the value of b.\nFor functions f(x), f(x) = x defines a new function constant f, or adds a new method to f if f is already defined; this usage is equivalent to function f(x); x; end.\na[i] = v calls setindex!(a,v,i).\na.b = c calls setproperty!(a,:b,c).\nInside a function call, f(a=b) passes b as the value of keyword argument a.\nInside parentheses with commas, (a=1,) constructs a NamedTuple.\n\nExamples\n\nAssigning a to b does not create a copy of b; instead use copy or deepcopy.\n\njulia> b = [1]; a = b; b[1] = 2; a\n1-element Vector{Int64}:\n 2\n\njulia> b = [1]; a = copy(b); b[1] = 2; a\n1-element Vector{Int64}:\n 1\n\n\nCollections passed to functions are also not copied. Functions can modify (mutate) the contents of the objects their arguments refer to. (The names of functions which do this are conventionally suffixed with '!'.)\n\njulia> function f!(x); x[:] .+= 1; end\nf! (generic function with 1 method)\n\njulia> a = [1]; f!(a); a\n1-element Vector{Int64}:\n 2\n\n\nAssignment can operate on multiple variables in parallel, taking values from an iterable:\n\njulia> a, b = 4, 5\n(4, 5)\n\njulia> a, b = 1:3\n1:3\n\njulia> a, b\n(1, 2)\n\n\nAssignment can operate on multiple variables in series, and will return the value of the right-hand-most expression:\n\njulia> a = [1]; b = [2]; c = [3]; a = b = c\n1-element Vector{Int64}:\n 3\n\njulia> b[1] = 2; a, b, c\n([2], [2], [2])\n\n\nAssignment at out-of-bounds indices does not grow a collection. If the collection is a Vector it can instead be grown with push! or append!.\n\njulia> a = [1, 1]; a[3] = 2\nERROR: BoundsError: attempt to access 2-element Vector{Int64} at index [3]\n[...]\n\njulia> push!(a, 2, 3)\n4-element Vector{Int64}:\n 1\n 1\n 2\n 3\n\n\nAssigning [] does not eliminate elements from a collection; instead use filter!.\n\njulia> a = collect(1:3); a[a .<= 1] = []\nERROR: DimensionMismatch: tried to assign 0 elements to 1 destinations\n[...]\n\njulia> filter!(x -> x > 1, a) # in-place & thus more efficient than a = a[a .> 1]\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\n\n","page":"Essentials"},{"title":"?:","location":"base/base.html#?:","category":"keyword","text":"a ? b : c\n\nShort form for conditionals; read \"if a, evaluate b otherwise evaluate c\". Also known as the ternary operator.\n\nThis syntax is equivalent to if a; b else c end, but is often used to emphasize the value b-or-c which is being used as part of a larger expression, rather than the side effects that evaluating b or c may have.\n\nSee the manual section on control flow for more details.\n\nExamples\n\njulia> x = 1; y = 2;\n\njulia> x > y ? println(\"x is larger\") : println(\"x is not larger\")\nx is not larger\n\njulia> x > y ? \"x is larger\" : x == y ? \"x and y are equal\" : \"y is larger\"\n\"y is larger\"\n\n\n\n\n\n","page":"Essentials"},{"title":".=","location":"base/base.html#.=","category":"keyword","text":".=\n\nPerform broadcasted assignment. The right-side argument is expanded as in broadcast and then assigned into the left-side argument in-place. Fuses with other dotted operators in the same expression; i.e. the whole assignment expression is converted into a single loop.\n\nA .= B is similar to broadcast!(identity, A, B).\n\nExamples\n\njulia> A = zeros(4, 4); B = [1, 2, 3, 4];\n\njulia> A .= B\n4×4 Matrix{Float64}:\n 1.0  1.0  1.0  1.0\n 2.0  2.0  2.0  2.0\n 3.0  3.0  3.0  3.0\n 4.0  4.0  4.0  4.0\n\njulia> A\n4×4 Matrix{Float64}:\n 1.0  1.0  1.0  1.0\n 2.0  2.0  2.0  2.0\n 3.0  3.0  3.0  3.0\n 4.0  4.0  4.0  4.0\n\n\n\n\n\n","page":"Essentials"},{"title":".","location":"base/base.html#.","category":"keyword","text":".\n\nThe dot operator is used to access fields or properties of objects and access variables defined inside modules.\n\nIn general, a.b calls getproperty(a, :b) (see getproperty).\n\nExamples\n\njulia> z = 1 + 2im; z.im\n2\n\njulia> Iterators.product\nproduct (generic function with 1 method)\n\n\n\n\n\n","page":"Essentials"},{"title":"->","location":"base/base.html#->","category":"keyword","text":"x -> y\n\nCreate an anonymous function mapping argument(s) x to the function body y.\n\njulia> f = x -> x^2 + 2x - 1\n#1 (generic function with 1 method)\n\njulia> f(2)\n7\n\nAnonymous functions can also be defined for multiple arguments.\n\njulia> g = (x,y) -> x^2 + y^2\n#2 (generic function with 1 method)\n\njulia> g(2,3)\n13\n\nSee the manual section on anonymous functions for more details.\n\n\n\n\n\n","page":"Essentials"},{"title":"::","location":"base/base.html#::","category":"keyword","text":"::\n\nThe :: operator either asserts that a value has the given type, or declares that a local variable or function return always has the given type.\n\nGiven expression::T, expression is first evaluated. If the result is of type T, the value is simply returned. Otherwise, a TypeError is thrown.\n\nIn local scope, the syntax local x::T or x::T = expression declares that local variable x always has type T. When a value is assigned to the variable, it will be converted to type T by calling convert.\n\nIn a method declaration, the syntax function f(x)::T causes any value returned by the method to be converted to type T.\n\nSee the manual section on Type Declarations.\n\nExamples\n\njulia> (1+2)::AbstractFloat\nERROR: TypeError: typeassert: expected AbstractFloat, got a value of type Int64\n\njulia> (1+2)::Int\n3\n\njulia> let\n           local x::Int\n           x = 2.0\n           x\n       end\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"[]","location":"base/base.html#[]","category":"keyword","text":"[]\n\nSquare brackets are used for indexing (getindex), indexed assignment (setindex!), array literals (Base.vect), array concatenation (vcat, hcat, hvcat, hvncat), and array comprehensions (collect).\n\n\n\n\n\n","page":"Essentials"},{"title":"Main","location":"base/base.html#Main","category":"module","text":"Main\n\nMain is the top-level module, and Julia starts with Main set as the current module.  Variables defined at the prompt go in Main, and varinfo lists variables in Main.\n\njulia> @__MODULE__\nMain\n\n\n\n\n\n","page":"Essentials"},{"title":"Core","location":"base/base.html#Core","category":"module","text":"Core\n\nCore is the module that contains all identifiers considered \"built in\" to the language, i.e. part of the core language and not libraries. Every module implicitly specifies using Core, since you can't do anything without those definitions.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base","location":"base/base.html#Base","category":"module","text":"Base\n\nThe base library of Julia. Base is a module that contains basic functionality (the contents of base/). All modules implicitly contain using Base, since this is needed in the vast majority of cases.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Broadcast","location":"base/base.html#Base.Broadcast","category":"module","text":"Base.Broadcast\n\nModule containing the broadcasting implementation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs","location":"base/base.html#Base.Docs","category":"module","text":"Docs\n\nThe Docs module provides the @doc macro which can be used to set and retrieve documentation metadata for Julia objects.\n\nPlease see the manual section on documentation for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Iterators","location":"base/base.html#Base.Iterators","category":"module","text":"Methods for working with Iterators.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc","location":"base/base.html#Base.Libc","category":"module","text":"Interface to libc, the C standard library.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta","location":"base/base.html#Base.Meta","category":"module","text":"Convenience functions for metaprogramming.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.StackTraces","location":"base/base.html#Base.StackTraces","category":"module","text":"Tools for collecting and manipulating stack traces. Mainly used for building errors.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys","location":"base/base.html#Base.Sys","category":"module","text":"Provide methods for retrieving information about hardware and the operating system.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Threads","location":"base/base.html#Base.Threads","category":"module","text":"Multithreading support.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC","location":"base/base.html#Base.GC","category":"module","text":"Base.GC\n\nModule with garbage collection utilities.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.:===","location":"base/base.html#Core.:===","category":"function","text":"===(x,y) -> Bool\n≡(x,y) -> Bool\n\nDetermine whether x and y are identical, in the sense that no program could distinguish them. First the types of x and y are compared. If those are identical, mutable objects are compared by address in memory and immutable objects (such as numbers) are compared by contents at the bit level. This function is sometimes called \"egal\". It always returns a Bool value.\n\nExamples\n\njulia> a = [1, 2]; b = [1, 2];\n\njulia> a == b\ntrue\n\njulia> a === b\nfalse\n\njulia> a === a\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.isa","location":"base/base.html#Core.isa","category":"function","text":"isa(x, type) -> Bool\n\nDetermine whether x is of the given type. Can also be used as an infix operator, e.g. x isa type.\n\nExamples\n\njulia> isa(1, Int)\ntrue\n\njulia> isa(1, Matrix)\nfalse\n\njulia> isa(1, Char)\nfalse\n\njulia> isa(1, Number)\ntrue\n\njulia> 1 isa Number\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isequal","location":"base/base.html#Base.isequal","category":"function","text":"isequal(x)\n\nCreate a function that compares its argument to x using isequal, i.e. a function equivalent to y -> isequal(y, x).\n\nThe returned function is of type Base.Fix2{typeof(isequal)}, which can be used to implement specialized methods.\n\n\n\n\n\nisequal(x, y) -> Bool\n\nSimilar to ==, except for the treatment of floating point numbers and of missing values. isequal treats all floating-point NaN values as equal to each other, treats -0.0 as unequal to 0.0, and missing as equal to missing. Always returns a Bool value.\n\nisequal is an equivalence relation - it is reflexive (=== implies isequal), symmetric (isequal(a, b) implies isequal(b, a)) and transitive (isequal(a, b) and isequal(b, c) implies isequal(a, c)).\n\nImplementation\n\nThe default implementation of isequal calls ==, so a type that does not involve floating-point values generally only needs to define ==.\n\nisequal is the comparison function used by hash tables (Dict). isequal(x,y) must imply that hash(x) == hash(y).\n\nThis typically means that types for which a custom == or isequal method exists must implement a corresponding hash method (and vice versa). Collections typically implement isequal by calling isequal recursively on all contents.\n\nFurthermore, isequal is linked with isless, and they work together to define a fixed total ordering, where exactly one of isequal(x, y), isless(x, y), or isless(y, x) must be true (and the other two false).\n\nScalar types generally do not need to implement isequal separate from ==, unless they represent floating-point numbers amenable to a more efficient implementation than that provided as a generic fallback (based on isnan, signbit, and ==).\n\nExamples\n\njulia> isequal([1., NaN], [1., NaN])\ntrue\n\njulia> [1., NaN] == [1., NaN]\nfalse\n\njulia> 0.0 == -0.0\ntrue\n\njulia> isequal(0.0, -0.0)\nfalse\n\njulia> missing == missing\nmissing\n\njulia> isequal(missing, missing)\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isless","location":"base/base.html#Base.isless","category":"function","text":"isless(x, y)\n\nTest whether x is less than y, according to a fixed total order (defined together with isequal). isless is not defined for pairs (x, y) of all types. However, if it is defined, it is expected to satisfy the following:\n\nIf isless(x, y) is defined, then so is isless(y, x) and isequal(x, y), and exactly one of those three yields true.\nThe relation defined by isless is transitive, i.e., isless(x, y) && isless(y, z) implies isless(x, z).\n\nValues that are normally unordered, such as NaN, are ordered after regular values. missing values are ordered last.\n\nThis is the default comparison used by sort!.\n\nImplementation\n\nNon-numeric types with a total order should implement this function. Numeric types only need to implement it if they have special values such as NaN. Types with a partial order should implement <. See the documentation on Alternate Orderings for how to define alternate ordering methods that can be used in sorting and related functions.\n\nExamples\n\njulia> isless(1, 3)\ntrue\n\njulia> isless(\"Red\", \"Blue\")\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isunordered","location":"base/base.html#Base.isunordered","category":"function","text":"isunordered(x)\n\nReturn true if x is a value that is not orderable according to <, such as NaN or missing.\n\nThe values that evaluate to true with this predicate may be orderable with respect to other orderings such as isless.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ifelse","location":"base/base.html#Base.ifelse","category":"function","text":"ifelse(condition::Bool, x, y)\n\nReturn x if condition is true, otherwise return y. This differs from ? or if in that it is an ordinary function, so all the arguments are evaluated first. In some cases, using ifelse instead of an if statement can eliminate the branch in generated code and provide higher performance in tight loops.\n\nExamples\n\njulia> ifelse(1 > 2, 1, 2)\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.typeassert","location":"base/base.html#Core.typeassert","category":"function","text":"typeassert(x, type)\n\nThrow a TypeError unless x isa type. The syntax x::type calls this function.\n\nExamples\n\njulia> typeassert(2.5, Int)\nERROR: TypeError: in typeassert, expected Int64, got a value of type Float64\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.typeof","location":"base/base.html#Core.typeof","category":"function","text":"typeof(x)\n\nGet the concrete type of x.\n\nSee also eltype.\n\nExamples\n\njulia> a = 1//2;\n\njulia> typeof(a)\nRational{Int64}\n\njulia> M = [1 2; 3.5 4];\n\njulia> typeof(M)\nMatrix{Float64} (alias for Array{Float64, 2})\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.tuple","location":"base/base.html#Core.tuple","category":"function","text":"tuple(xs...)\n\nConstruct a tuple of the given objects.\n\nSee also Tuple, ntuple, NamedTuple.\n\nExamples\n\njulia> tuple(1, 'b', pi)\n(1, 'b', π)\n\njulia> ans === (1, 'b', π)\ntrue\n\njulia> Tuple(Real[1, 2, pi])  # takes a collection\n(1, 2, π)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ntuple","location":"base/base.html#Base.ntuple","category":"function","text":"ntuple(f, ::Val{N})\n\nCreate a tuple of length N, computing each element as f(i), where i is the index of the element. By taking a Val(N) argument, it is possible that this version of ntuple may generate more efficient code than the version taking the length as an integer. But ntuple(f, N) is preferable to ntuple(f, Val(N)) in cases where N cannot be determined at compile time.\n\nExamples\n\njulia> ntuple(i -> 2*i, Val(4))\n(2, 4, 6, 8)\n\n\n\n\n\nntuple(f, n::Integer)\n\nCreate a tuple of length n, computing each element as f(i), where i is the index of the element.\n\nExamples\n\njulia> ntuple(i -> 2*i, 4)\n(2, 4, 6, 8)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.objectid","location":"base/base.html#Base.objectid","category":"function","text":"objectid(x) -> UInt\n\nGet a hash value for x based on object identity.\n\nIf x === y then objectid(x) == objectid(y), and usually when x !== y, objectid(x) != objectid(y).\n\nSee also hash, IdDict.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hash","location":"base/base.html#Base.hash","category":"function","text":"hash(x[, h::UInt]) -> UInt\n\nCompute an integer hash code such that isequal(x,y) implies hash(x)==hash(y). The optional second argument h is another hash code to be mixed with the result.\n\nNew types should implement the 2-argument form, typically by calling the 2-argument hash method recursively in order to mix hashes of the contents with each other (and with h). Typically, any type that implements hash should also implement its own == (hence isequal) to guarantee the property mentioned above.\n\nThe hash value may change when a new Julia process is started.\n\njulia> a = hash(10)\n0x95ea2955abd45275\n\njulia> hash(10, a) # only use the output of another hash function as the second argument\n0xd42bad54a8575b16\n\nSee also: objectid, Dict, Set.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.finalizer","location":"base/base.html#Base.finalizer","category":"function","text":"finalizer(f, x)\n\nRegister a function f(x) to be called when there are no program-accessible references to x, and return x. The type of x must be a mutable struct, otherwise the function will throw.\n\nf must not cause a task switch, which excludes most I/O operations such as println. Using the @async macro (to defer context switching to outside of the finalizer) or ccall to directly invoke IO functions in C may be helpful for debugging purposes.\n\nNote that there is no guaranteed world age for the execution of f. It may be called in the world age in which the finalizer was registered or any later world age.\n\nExamples\n\nfinalizer(my_mutable_struct) do x\n    @async println(\"Finalizing $x.\")\nend\n\nfinalizer(my_mutable_struct) do x\n    ccall(:jl_safe_printf, Cvoid, (Cstring, Cstring), \"Finalizing %s.\", repr(x))\nend\n\nA finalizer may be registered at object construction. In the following example note that we implicitly rely on the finalizer returning the newly created mutable struct x.\n\nmutable struct MyMutableStruct\n    bar\n    function MyMutableStruct(bar)\n        x = new(bar)\n        f(t) = @async println(\"Finalizing $t.\")\n        finalizer(f, x)\n    end\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.finalize","location":"base/base.html#Base.finalize","category":"function","text":"finalize(x)\n\nImmediately run finalizers registered for object x.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.copy","location":"base/base.html#Base.copy","category":"function","text":"copy(x)\n\nCreate a shallow copy of x: the outer structure is copied, but not all internal values. For example, copying an array produces a new array with identically-same elements as the original.\n\nSee also copy!, copyto!, deepcopy.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.deepcopy","location":"base/base.html#Base.deepcopy","category":"function","text":"deepcopy(x)\n\nCreate a deep copy of x: everything is copied recursively, resulting in a fully independent object. For example, deep-copying an array creates deep copies of all the objects it contains and produces a new array with the consistent relationship structure (e.g., if the first two elements are the same object in the original array, the first two elements of the new array will also be the same deepcopyed object). Calling deepcopy on an object should generally have the same effect as serializing and then deserializing it.\n\nWhile it isn't normally necessary, user-defined types can override the default deepcopy behavior by defining a specialized version of the function deepcopy_internal(x::T, dict::IdDict) (which shouldn't otherwise be used), where T is the type to be specialized for, and dict keeps track of objects copied so far within the recursion. Within the definition, deepcopy_internal should be used in place of deepcopy, and the dict variable should be updated as appropriate before returning.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.getproperty","location":"base/base.html#Base.getproperty","category":"function","text":"getproperty(value, name::Symbol)\ngetproperty(value, name::Symbol, order::Symbol)\n\nThe syntax a.b calls getproperty(a, :b). The syntax @atomic order a.b calls getproperty(a, :b, :order) and the syntax @atomic a.b calls getproperty(a, :b, :sequentially_consistent).\n\nExamples\n\njulia> struct MyType{T <: Number}\n           x::T\n       end\n\njulia> function Base.getproperty(obj::MyType, sym::Symbol)\n           if sym === :special\n               return obj.x + 1\n           else # fallback to getfield\n               return getfield(obj, sym)\n           end\n       end\n\njulia> obj = MyType(1);\n\njulia> obj.special\n2\n\njulia> obj.x\n1\n\nOne should overload getproperty only when necessary, as it can be confusing if the behavior of the syntax obj.f is unusual. Also note that using methods is often preferable. See also this style guide documentation for more information: Prefer exported methods over direct field access.\n\nSee also getfield, propertynames and setproperty!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setproperty!","location":"base/base.html#Base.setproperty!","category":"function","text":"setproperty!(value, name::Symbol, x)\nsetproperty!(value, name::Symbol, x, order::Symbol)\n\nThe syntax a.b = c calls setproperty!(a, :b, c). The syntax @atomic order a.b = c calls setproperty!(a, :b, c, :order) and the syntax @atomic a.b = c calls setproperty!(a, :b, c, :sequentially_consistent).\n\ncompat: Julia 1.8\nsetproperty! on modules requires at least Julia 1.8.\n\nSee also setfield!, propertynames and getproperty.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.replaceproperty!","location":"base/base.html#Base.replaceproperty!","category":"function","text":"replaceproperty!(x, f::Symbol, expected, desired, success_order::Symbol=:not_atomic, fail_order::Symbol=success_order)\n\nPerform a compare-and-swap operation on x.f from expected to desired, per egal. The syntax @atomicreplace x.f expected => desired can be used instead of the function call form.\n\nSee also replacefield! setproperty!, setpropertyonce!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.swapproperty!","location":"base/base.html#Base.swapproperty!","category":"function","text":"swapproperty!(x, f::Symbol, v, order::Symbol=:not_atomic)\n\nThe syntax @atomic a.b, _ = c, a.b returns (c, swapproperty!(a, :b, c, :sequentially_consistent)), where there must be one getproperty expression common to both sides.\n\nSee also swapfield! and setproperty!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.modifyproperty!","location":"base/base.html#Base.modifyproperty!","category":"function","text":"modifyproperty!(x, f::Symbol, op, v, order::Symbol=:not_atomic)\n\nThe syntax @atomic op(x.f, v) (and its equivalent @atomic x.f op v) returns modifyproperty!(x, :f, op, v, :sequentially_consistent), where the first argument must be a getproperty expression and is modified atomically.\n\nInvocation of op(getproperty(x, f), v) must return a value that can be stored in the field f of the object x by default.  In particular, unlike the default behavior of setproperty!, the convert function is not called automatically.\n\nSee also modifyfield! and setproperty!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setpropertyonce!","location":"base/base.html#Base.setpropertyonce!","category":"function","text":"setpropertyonce!(x, f::Symbol, value, success_order::Symbol=:not_atomic, fail_order::Symbol=success_order)\n\nPerform a compare-and-swap operation on x.f to set it to value if previously unset. The syntax @atomiconce x.f = value can be used instead of the function call form.\n\nSee also setfieldonce!, setproperty!, replaceproperty!.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.propertynames","location":"base/base.html#Base.propertynames","category":"function","text":"propertynames(x, private=false)\n\nGet a tuple or a vector of the properties (x.property) of an object x. This is typically the same as fieldnames(typeof(x)), but types that overload getproperty should generally overload propertynames as well to get the properties of an instance of the type.\n\npropertynames(x) may return only \"public\" property names that are part of the documented interface of x.   If you want it to also return \"private\" property names intended for internal use, pass true for the optional second argument. REPL tab completion on x. shows only the private=false properties.\n\nSee also: hasproperty, hasfield.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasproperty","location":"base/base.html#Base.hasproperty","category":"function","text":"hasproperty(x, s::Symbol)\n\nReturn a boolean indicating whether the object x has s as one of its own properties.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\nSee also: propertynames, hasfield.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.getfield","location":"base/base.html#Core.getfield","category":"function","text":"getfield(value, name::Symbol, [order::Symbol])\ngetfield(value, i::Int, [order::Symbol])\n\nExtract a field from a composite value by name or position. Optionally, an ordering can be defined for the operation. If the field was declared @atomic, the specification is strongly recommended to be compatible with the stores to that location. Otherwise, if not declared as @atomic, this parameter must be :not_atomic if specified. See also getproperty and fieldnames.\n\nExamples\n\njulia> a = 1//2\n1//2\n\njulia> getfield(a, :num)\n1\n\njulia> a.num\n1\n\njulia> getfield(a, 1)\n1\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.setfield!","location":"base/base.html#Core.setfield!","category":"function","text":"setfield!(value, name::Symbol, x, [order::Symbol])\nsetfield!(value, i::Int, x, [order::Symbol])\n\nAssign x to a named field in value of composite type. The value must be mutable and x must be a subtype of fieldtype(typeof(value), name). Additionally, an ordering can be specified for this operation. If the field was declared @atomic, this specification is mandatory. Otherwise, if not declared as @atomic, it must be :not_atomic if specified. See also setproperty!.\n\nExamples\n\njulia> mutable struct MyMutableStruct\n           field::Int\n       end\n\njulia> a = MyMutableStruct(1);\n\njulia> setfield!(a, :field, 2);\n\njulia> getfield(a, :field)\n2\n\njulia> a = 1//2\n1//2\n\njulia> setfield!(a, :num, 3);\nERROR: setfield!: immutable struct of type Rational cannot be changed\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.modifyfield!","location":"base/base.html#Core.modifyfield!","category":"function","text":"modifyfield!(value, name::Symbol, op, x, [order::Symbol]) -> Pair\nmodifyfield!(value, i::Int, op, x, [order::Symbol]) -> Pair\n\nAtomically perform the operations to get and set a field after applying the function op.\n\ny = getfield(value, name)\nz = op(y, x)\nsetfield!(value, name, z)\nreturn y => z\n\nIf supported by the hardware (for example, atomic increment), this may be optimized to the appropriate hardware instruction, otherwise it'll use a loop.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.replacefield!","location":"base/base.html#Core.replacefield!","category":"function","text":"replacefield!(value, name::Symbol, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\nreplacefield!(value, i::Int, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\n\nAtomically perform the operations to get and conditionally set a field to a given value.\n\ny = getfield(value, name, fail_order)\nok = y === expected\nif ok\n    setfield!(value, name, desired, success_order)\nend\nreturn (; old = y, success = ok)\n\nIf supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise it'll use a loop.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.swapfield!","location":"base/base.html#Core.swapfield!","category":"function","text":"swapfield!(value, name::Symbol, x, [order::Symbol])\nswapfield!(value, i::Int, x, [order::Symbol])\n\nAtomically perform the operations to simultaneously get and set a field:\n\ny = getfield(value, name)\nsetfield!(value, name, x)\nreturn y\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.setfieldonce!","location":"base/base.html#Core.setfieldonce!","category":"function","text":"setfieldonce!(value, name::Union{Int,Symbol}, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> success::Bool\n\nAtomically perform the operations to set a field to a given value, only if it was previously not set.\n\nok = !isdefined(value, name, fail_order)\nif ok\n    setfield!(value, name, desired, success_order)\nend\nreturn ok\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.isdefined","location":"base/base.html#Core.isdefined","category":"function","text":"isdefined(m::Module, s::Symbol, [order::Symbol])\nisdefined(object, s::Symbol, [order::Symbol])\nisdefined(object, index::Int, [order::Symbol])\n\nTests whether a global variable or object field is defined. The arguments can be a module and a symbol or a composite object and field name (as a symbol) or index. Optionally, an ordering can be defined for the operation. If the field was declared @atomic, the specification is strongly recommended to be compatible with the stores to that location. Otherwise, if not declared as @atomic, this parameter must be :not_atomic if specified.\n\nTo test whether an array element is defined, use isassigned instead.\n\nThe global variable variant is supported for compatibility with older julia releases. For new code, prefer isdefinedglobal.\n\nSee also @isdefined.\n\nExamples\n\njulia> isdefined(Base, :sum)\ntrue\n\njulia> isdefined(Base, :NonExistentMethod)\nfalse\n\njulia> a = 1//2;\n\njulia> isdefined(a, 2)\ntrue\n\njulia> isdefined(a, 3)\nfalse\n\njulia> isdefined(a, :num)\ntrue\n\njulia> isdefined(a, :numerator)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.isdefinedglobal","location":"base/base.html#Core.isdefinedglobal","category":"function","text":"isdefinedglobal(m::Module, s::Symbol, [allow_import::Bool=true, [order::Symbol=:unordered]])\n\nTests whether a global variable s is defined in module m (in the current world age). A variable is considered defined if and only if a value may be read from this global variable and an access will not throw. This includes both constants and global variables that have a value set.\n\nIf allow_import is false, the global variable must be defined inside m and may not be imported from another module.\n\ncompat: Julia 1.12\nThis function requires Julia 1.12 or later.\n\nSee also @isdefined.\n\nExamples\n\njulia> isdefinedglobal(Base, :sum)\ntrue\n\njulia> isdefinedglobal(Base, :NonExistentMethod)\nfalse\n\njulia> isdefinedglobal(Base, :sum, false)\ntrue\n\njulia> isdefinedglobal(Main, :sum, false)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@isdefined","location":"base/base.html#Base.@isdefined","category":"macro","text":"@isdefined s -> Bool\n\nTests whether variable s is defined in the current scope.\n\nSee also isdefined for field properties and isassigned for array indexes or haskey for other mappings.\n\nExamples\n\njulia> @isdefined newvar\nfalse\n\njulia> newvar = 1\n1\n\njulia> @isdefined newvar\ntrue\n\njulia> function f()\n           println(@isdefined x)\n           x = 3\n           println(@isdefined x)\n       end\nf (generic function with 1 method)\n\njulia> f()\nfalse\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.convert","location":"base/base.html#Base.convert","category":"function","text":"convert(T, x)\n\nConvert x to a value of type T.\n\nIf T is an Integer type, an InexactError will be raised if x is not representable by T, for example if x is not integer-valued, or is outside the range supported by T.\n\nExamples\n\njulia> convert(Int, 3.0)\n3\n\njulia> convert(Int, 3.5)\nERROR: InexactError: Int64(3.5)\nStacktrace:\n[...]\n\nIf T is a AbstractFloat type, then it will return the closest value to x representable by T. Inf is treated as one ulp greater than floatmax(T) for purposes of determining nearest.\n\njulia> x = 1/3\n0.3333333333333333\n\njulia> convert(Float32, x)\n0.33333334f0\n\njulia> convert(BigFloat, x)\n0.333333333333333314829616256247390992939472198486328125\n\nIf T is a collection type and x a collection, the result of convert(T, x) may alias all or part of x.\n\njulia> x = Int[1, 2, 3];\n\njulia> y = convert(Vector{Int}, x);\n\njulia> y === x\ntrue\n\nSee also: round, trunc, oftype, reinterpret.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote","location":"base/base.html#Base.promote","category":"function","text":"promote(xs...)\n\nConvert all arguments to a common type, and return them all (as a tuple). If no arguments can be converted, an error is raised.\n\nSee also: promote_type, promote_rule.\n\nExamples\n\njulia> promote(Int8(1), Float16(4.5), Float32(4.1))\n(1.0f0, 4.5f0, 4.1f0)\n\njulia> promote_type(Int8, Float16, Float32)\nFloat32\n\njulia> reduce(Base.promote_typejoin, (Int8, Float16, Float32))\nReal\n\njulia> promote(1, \"x\")\nERROR: promotion of types Int64 and String failed to change any arguments\n[...]\n\njulia> promote_type(Int, String)\nAny\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.oftype","location":"base/base.html#Base.oftype","category":"function","text":"oftype(x, y)\n\nConvert y to the type of x i.e. convert(typeof(x), y).\n\nExamples\n\njulia> x = 4;\n\njulia> y = 3.;\n\njulia> oftype(x, y)\n3\n\njulia> oftype(y, x)\n4.0\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.widen","location":"base/base.html#Base.widen","category":"function","text":"widen(x)\n\nIf x is a type, return a \"larger\" type, defined so that arithmetic operations + and - are guaranteed not to overflow nor lose precision for any combination of values that type x can hold.\n\nFor fixed-size integer types less than 128 bits, widen will return a type with twice the number of bits.\n\nIf x is a value, it is converted to widen(typeof(x)).\n\nExamples\n\njulia> widen(Int32)\nInt64\n\njulia> widen(1.5f0)\n1.5\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.identity","location":"base/base.html#Base.identity","category":"function","text":"identity(x)\n\nThe identity function. Returns its argument.\n\nSee also: one, oneunit, and LinearAlgebra's I.\n\nExamples\n\njulia> identity(\"Well, what did you expect?\")\n\"Well, what did you expect?\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.WeakRef","location":"base/base.html#Core.WeakRef","category":"type","text":"WeakRef(x)\n\nw = WeakRef(x) constructs a weak reference to the Julia value x: although w contains a reference to x, it does not prevent x from being garbage collected. w.value is either x (if x has not been garbage-collected yet) or nothing (if x has been garbage-collected).\n\njulia> x = \"a string\"\n\"a string\"\n\njulia> w = WeakRef(x)\nWeakRef(\"a string\")\n\njulia> GC.gc()\n\njulia> w           # a reference is maintained via `x`\nWeakRef(\"a string\")\n\njulia> x = nothing # clear reference\n\njulia> GC.gc()\n\njulia> w\nWeakRef(nothing)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.supertype","location":"base/base.html#Base.supertype","category":"function","text":"supertype(T::Union{DataType, UnionAll})\n\nReturn the direct supertype of type T. T can be a DataType or a UnionAll type. Does not support type Unions. Also see info on Types.\n\nExamples\n\njulia> supertype(Int32)\nSigned\n\njulia> supertype(Vector)\nDenseVector (alias for DenseArray{T, 1} where T)\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Type","location":"base/base.html#Core.Type","category":"type","text":"Core.Type{T}\n\nCore.Type is an abstract type which has all type objects as its instances. The only instance of the singleton type Core.Type{T} is the object T.\n\nExamples\n\njulia> isa(Type{Float64}, Type)\ntrue\n\njulia> isa(Float64, Type)\ntrue\n\njulia> isa(Real, Type{Float64})\nfalse\n\njulia> isa(Real, Type{Real})\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DataType","location":"base/base.html#Core.DataType","category":"type","text":"DataType <: Type{T}\n\nDataType represents explicitly declared types that have names, explicitly declared supertypes, and, optionally, parameters.  Every concrete value in the system is an instance of some DataType.\n\nExamples\n\njulia> typeof(Real)\nDataType\n\njulia> typeof(Int)\nDataType\n\njulia> struct Point\n           x::Int\n           y\n       end\n\njulia> typeof(Point)\nDataType\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.:<:","location":"base/base.html#Core.:<:","category":"function","text":"<:(T1, T2)::Bool\n\nSubtyping relation, defined between two types. In Julia, a type S is said to be a subtype of a type T if and only if we have S <: T.\n\nFor any type L and any type R, L <: R implies that any value v of type L is also of type R. I.e., (L <: R) && (v isa L) implies v isa R.\n\nThe subtyping relation is a partial order. I.e., <: is:\n\nreflexive: for any type T, T <: T holds\nantisymmetric: for any type A and any type B, (A <: B) && (B <: A) implies A == B\ntransitive: for any type A, any type B and any type C; (A <: B) && (B <: C) implies A <: C\n\nSee also info on Types, Union{}, Any, isa.\n\nExamples\n\njulia> Float64 <: AbstractFloat\ntrue\n\njulia> Vector{Int} <: AbstractArray\ntrue\n\njulia> Matrix{Float64} <: Matrix{AbstractFloat}  # `Matrix` is invariant\nfalse\n\njulia> Tuple{Float64} <: Tuple{AbstractFloat}    # `Tuple` is covariant\ntrue\n\njulia> Union{} <: Int  # The bottom type, `Union{}`, subtypes each type.\ntrue\n\njulia> Union{} <: Float32 <: AbstractFloat <: Real <: Number <: Any  # Operator chaining\ntrue\n\nThe <: keyword also has several syntactic uses which represent the same subtyping relation, but which do not execute the operator or return a Bool:\n\nTo specify the lower bound and the upper bound on a parameter of a UnionAll type in a where statement.\nTo specify the lower bound and the upper bound on a (static) parameter of a method, see Parametric Methods.\nTo define a subtyping relation while declaring a new type, see struct and abstract type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.:>:","location":"base/base.html#Core.:>:","category":"function","text":">:(T1, T2)\n\nSupertype operator, equivalent to T2 <: T1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typejoin","location":"base/base.html#Base.typejoin","category":"function","text":"typejoin(T, S, ...)\n\nReturn the closest common ancestor of types T and S, i.e. the narrowest type from which they both inherit. Recurses on additional varargs.\n\nExamples\n\njulia> typejoin(Int, Float64)\nReal\n\njulia> typejoin(Int, Float64, ComplexF32)\nNumber\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typeintersect","location":"base/base.html#Base.typeintersect","category":"function","text":"typeintersect(T::Type, S::Type)\n\nCompute a type that contains the intersection of T and S. Usually this will be the smallest such type or one close to it.\n\nA special case where exact behavior is guaranteed: when T <: S, typeintersect(S, T) == T == typeintersect(T, S).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote_type","location":"base/base.html#Base.promote_type","category":"function","text":"promote_type(type1, type2, ...)\n\nPromotion refers to converting values of mixed types to a single common type. promote_type represents the default promotion behavior in Julia when operators (usually mathematical) are given arguments of differing types. promote_type generally tries to return a type which can at least approximate most values of either input type without excessively widening.  Some loss is tolerated; for example, promote_type(Int64, Float64) returns Float64 even though strictly, not all Int64 values can be represented exactly as Float64 values.\n\nSee also: promote, promote_typejoin, promote_rule.\n\nExamples\n\njulia> promote_type(Int64, Float64)\nFloat64\n\njulia> promote_type(Int32, Int64)\nInt64\n\njulia> promote_type(Float32, BigInt)\nBigFloat\n\njulia> promote_type(Int16, Float16)\nFloat16\n\njulia> promote_type(Int64, Float16)\nFloat16\n\njulia> promote_type(Int8, UInt16)\nUInt16\n\nwarning: Don't overload this directly\nTo overload promotion for your own types you should overload promote_rule. promote_type calls promote_rule internally to determine the type. Overloading promote_type directly can cause ambiguity errors.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote_rule","location":"base/base.html#Base.promote_rule","category":"function","text":"promote_rule(type1, type2)\n\nSpecifies what type should be used by promote when given values of types type1 and type2. This function should not be called directly, but should have definitions added to it for new types as appropriate.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.promote_typejoin","location":"base/base.html#Base.promote_typejoin","category":"function","text":"promote_typejoin(T, S)\n\nCompute a type that contains both T and S, which could be either a parent of both types, or a Union if appropriate. Falls back to typejoin.\n\nSee instead promote, promote_type.\n\nExamples\n\njulia> Base.promote_typejoin(Int, Float64)\nReal\n\njulia> Base.promote_type(Int, Float64)\nFloat64\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isdispatchtuple","location":"base/base.html#Base.isdispatchtuple","category":"function","text":"isdispatchtuple(T)\n\nDetermine whether type T is a tuple of concrete types, meaning it could appear as a type signature in dispatch and has no subtypes (or supertypes) which could appear in a call. If T is not a type, then return false.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ismutable","location":"base/base.html#Base.ismutable","category":"function","text":"ismutable(v) -> Bool\n\nReturn true if and only if value v is mutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a DataType, it will tell you that a value of the type is mutable.\n\nnote: Note\nFor technical reasons, ismutable returns true for values of certain special types (for example String and Symbol) even though they cannot be mutated in a permissible way.\n\nSee also isbits, isstructtype.\n\nExamples\n\njulia> ismutable(1)\nfalse\n\njulia> ismutable([1,2])\ntrue\n\ncompat: Julia 1.5\nThis function requires at least Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isimmutable","location":"base/base.html#Base.isimmutable","category":"function","text":"isimmutable(v) -> Bool\n\nwarning: Warning\nConsider using !ismutable(v) instead, as isimmutable(v) will be replaced by !ismutable(v) in a future release. (Since Julia 1.5)\n\nReturn true iff value v is immutable.  See Mutable Composite Types for a discussion of immutability. Note that this function works on values, so if you give it a type, it will tell you that a value of DataType is mutable.\n\nExamples\n\njulia> isimmutable(1)\ntrue\n\njulia> isimmutable([1,2])\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ismutabletype","location":"base/base.html#Base.ismutabletype","category":"function","text":"ismutabletype(T) -> Bool\n\nDetermine whether type T was declared as a mutable type (i.e. using mutable struct keyword). If T is not a type, then return false.\n\ncompat: Julia 1.7\nThis function requires at least Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isabstracttype","location":"base/base.html#Base.isabstracttype","category":"function","text":"isabstracttype(T)\n\nDetermine whether type T was declared as an abstract type (i.e. using the abstract type syntax). Note that this is not the negation of isconcretetype(T). If T is not a type, then return false.\n\nExamples\n\njulia> isabstracttype(AbstractArray)\ntrue\n\njulia> isabstracttype(Vector)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isprimitivetype","location":"base/base.html#Base.isprimitivetype","category":"function","text":"isprimitivetype(T) -> Bool\n\nDetermine whether type T was declared as a primitive type (i.e. using the primitive type syntax). If T is not a type, then return false.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.issingletontype","location":"base/base.html#Base.issingletontype","category":"function","text":"Base.issingletontype(T)\n\nDetermine whether type T has exactly one possible instance; for example, a struct type with no fields except other singleton values. If T is not a concrete type, then return false.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isstructtype","location":"base/base.html#Base.isstructtype","category":"function","text":"isstructtype(T) -> Bool\n\nDetermine whether type T was declared as a struct type (i.e. using the struct or mutable struct keyword). If T is not a type, then return false.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{DataType}","category":"method","text":"nameof(t::DataType) -> Symbol\n\nGet the name of a (potentially UnionAll-wrapped) DataType (without its parent module) as a symbol.\n\nExamples\n\njulia> module Foo\n           struct S{T}\n           end\n       end\nFoo\n\njulia> nameof(Foo.S{T} where T)\n:S\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldnames","location":"base/base.html#Base.fieldnames","category":"function","text":"fieldnames(x::DataType)\n\nGet a tuple with the names of the fields of a DataType.\n\nEach name is a Symbol, except when x <: Tuple, in which case each name (actually the index of the field) is an Int.\n\nSee also propertynames, hasfield.\n\nExamples\n\njulia> fieldnames(Rational)\n(:num, :den)\n\njulia> fieldnames(typeof(1+im))\n(:re, :im)\n\njulia> fieldnames(Tuple{String,Int})\n(1, 2)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldname","location":"base/base.html#Base.fieldname","category":"function","text":"fieldname(x::DataType, i::Integer)\n\nGet the name of field i of a DataType.\n\nThe return type is Symbol, except when x <: Tuple, in which case the index of the field is returned, of type Int.\n\nExamples\n\njulia> fieldname(Rational, 1)\n:num\n\njulia> fieldname(Rational, 2)\n:den\n\njulia> fieldname(Tuple{String,Int}, 2)\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.fieldtype","location":"base/base.html#Core.fieldtype","category":"function","text":"fieldtype(T, name::Symbol | index::Int)\n\nDetermine the declared type of a field (specified by name or index) in a composite DataType T.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtype(Foo, :x)\nInt64\n\njulia> fieldtype(Foo, 2)\nString\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldtypes","location":"base/base.html#Base.fieldtypes","category":"function","text":"fieldtypes(T::Type)\n\nThe declared types of all fields in a composite DataType T as a tuple.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nExamples\n\njulia> struct Foo\n           x::Int64\n           y::String\n       end\n\njulia> fieldtypes(Foo)\n(Int64, String)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldcount","location":"base/base.html#Base.fieldcount","category":"function","text":"fieldcount(t::Type)\n\nGet the number of fields that an instance of the given type would have. An error is thrown if the type is too abstract to determine this.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasfield","location":"base/base.html#Base.hasfield","category":"function","text":"hasfield(T::Type, name::Symbol)\n\nReturn a boolean indicating whether T has name as one of its own fields.\n\nSee also fieldnames, fieldcount, hasproperty.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\nExamples\n\njulia> struct Foo\n            bar::Int\n       end\n\njulia> hasfield(Foo, :bar)\ntrue\n\njulia> hasfield(Foo, :x)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.nfields","location":"base/base.html#Core.nfields","category":"function","text":"nfields(x) -> Int\n\nGet the number of fields in the given object.\n\nExamples\n\njulia> a = 1//2;\n\njulia> nfields(a)\n2\n\njulia> b = 1\n1\n\njulia> nfields(b)\n0\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> nfields(ex)\n1\n\nIn these examples, a is a Rational, which has two fields. b is an Int, which is a primitive bitstype with no fields at all. ex is an ErrorException, which has one field.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isconst","location":"base/base.html#Base.isconst","category":"function","text":"isconst(t::DataType, s::Union{Int,Symbol}) -> Bool\n\nDetermine whether a field s is const in a given type t in the sense that a read from said field is consistent for egal objects. Note in particular that out-of-bounds fields are considered const under this definition (because they always throw).\n\n\n\n\n\nisconst(m::Module, s::Symbol) -> Bool\nisconst(g::GlobalRef)\n\nDetermine whether a global is const in a given module m, either because it was declared constant or because it was imported from a constant binding. Note that constant-ness is specific to a particular world age, so the result of this function may not be assumed to hold after a world age update.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isfieldatomic","location":"base/base.html#Base.isfieldatomic","category":"function","text":"isfieldatomic(t::DataType, s::Union{Int,Symbol}) -> Bool\n\nDetermine whether a field s is declared @atomic in a given type t.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.sizeof","location":"base/base.html#Base.sizeof-Tuple{Type}","category":"method","text":"sizeof(T::DataType)\nsizeof(obj)\n\nSize, in bytes, of the canonical binary representation of the given DataType T, if any. Or the size, in bytes, of object obj if it is not a DataType.\n\nSee also Base.summarysize.\n\nExamples\n\njulia> sizeof(Float32)\n4\n\njulia> sizeof(ComplexF64)\n16\n\njulia> sizeof(1.0)\n8\n\njulia> sizeof(collect(1.0:10.0))\n80\n\njulia> struct StructWithPadding\n           x::Int64\n           flag::Bool\n       end\n\njulia> sizeof(StructWithPadding) # not the sum of `sizeof` of fields due to padding\n16\n\njulia> sizeof(Int64) + sizeof(Bool) # different from above\n9\n\nIf DataType T does not have a specific size, an error is thrown.\n\njulia> sizeof(AbstractArray)\nERROR: Abstract type AbstractArray does not have a definite size.\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isconcretetype","location":"base/base.html#Base.isconcretetype","category":"function","text":"isconcretetype(T)\n\nDetermine whether type T is a concrete type, meaning it could have direct instances (values x such that typeof(x) === T). Note that this is not the negation of isabstracttype(T). If T is not a type, then return false.\n\nSee also: isbits, isabstracttype, issingletontype.\n\nExamples\n\njulia> isconcretetype(Complex)\nfalse\n\njulia> isconcretetype(Complex{Float32})\ntrue\n\njulia> isconcretetype(Vector{Complex})\ntrue\n\njulia> isconcretetype(Vector{Complex{Float32}})\ntrue\n\njulia> isconcretetype(Union{})\nfalse\n\njulia> isconcretetype(Union{Int,String})\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbits","location":"base/base.html#Base.isbits","category":"function","text":"isbits(x)\n\nReturn true if x is an instance of an isbitstype type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbitstype","location":"base/base.html#Base.isbitstype","category":"function","text":"isbitstype(T)\n\nReturn true if type T is a \"plain data\" type, meaning it is immutable and contains no references to other values, only primitive types and other isbitstype types. Typical examples are numeric types such as UInt8, Float64, and Complex{Float64}. This category of types is significant since they are valid as type parameters, may not track isdefined / isassigned status, and have a defined layout that is compatible with C. If T is not a type, then return false.\n\nSee also isbits, isprimitivetype, ismutable.\n\nExamples\n\njulia> isbitstype(Complex{Float64})\ntrue\n\njulia> isbitstype(Complex)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fieldoffset","location":"base/base.html#Base.fieldoffset","category":"function","text":"fieldoffset(type, i)\n\nThe byte offset of field i of a type relative to the data start. For example, we could use it in the following manner to summarize information about a struct:\n\njulia> structinfo(T) = [(fieldoffset(T,i), fieldname(T,i), fieldtype(T,i)) for i = 1:fieldcount(T)];\n\njulia> structinfo(Base.Filesystem.StatStruct)\n14-element Vector{Tuple{UInt64, Symbol, Type}}:\n (0x0000000000000000, :desc, Union{RawFD, String})\n (0x0000000000000008, :device, UInt64)\n (0x0000000000000010, :inode, UInt64)\n (0x0000000000000018, :mode, UInt64)\n (0x0000000000000020, :nlink, Int64)\n (0x0000000000000028, :uid, UInt64)\n (0x0000000000000030, :gid, UInt64)\n (0x0000000000000038, :rdev, UInt64)\n (0x0000000000000040, :size, Int64)\n (0x0000000000000048, :blksize, Int64)\n (0x0000000000000050, :blocks, Int64)\n (0x0000000000000058, :mtime, Float64)\n (0x0000000000000060, :ctime, Float64)\n (0x0000000000000068, :ioerrno, Int32)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_alignment","location":"base/base.html#Base.datatype_alignment","category":"function","text":"Base.datatype_alignment(dt::DataType) -> Int\n\nMemory allocation minimum alignment for instances of this type. Can be called on any isconcretetype, although for Memory it will give the alignment of the elements, not the whole object.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_haspadding","location":"base/base.html#Base.datatype_haspadding","category":"function","text":"Base.datatype_haspadding(dt::DataType) -> Bool\n\nReturn whether the fields of instances of this type are packed in memory, with no intervening padding bits (defined as bits whose value does not impact the semantic value of the instance itself). Can be called on any isconcretetype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.datatype_pointerfree","location":"base/base.html#Base.datatype_pointerfree","category":"function","text":"Base.datatype_pointerfree(dt::DataType) -> Bool\n\nReturn whether instances of this type can contain references to gc-managed memory. Can be called on any isconcretetype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typemin","location":"base/base.html#Base.typemin","category":"function","text":"typemin(T)\n\nThe lowest value representable by the given (real) numeric DataType T.\n\nSee also: floatmin, typemax, eps.\n\nExamples\n\njulia> typemin(Int8)\n-128\n\njulia> typemin(UInt32)\n0x00000000\n\njulia> typemin(Float16)\n-Inf16\n\njulia> typemin(Float32)\n-Inf32\n\njulia> nextfloat(-Inf32)  # smallest finite Float32 floating point number\n-3.4028235f38\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.typemax","location":"base/base.html#Base.typemax","category":"function","text":"typemax(T)\n\nThe highest value representable by the given (real) numeric DataType.\n\nSee also: floatmax, typemin, eps.\n\nExamples\n\njulia> typemax(Int8)\n127\n\njulia> typemax(UInt32)\n0xffffffff\n\njulia> typemax(Float64)\nInf\n\njulia> typemax(Float32)\nInf32\n\njulia> floatmax(Float32)  # largest finite Float32 floating point number\n3.4028235f38\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.floatmin","location":"base/base.html#Base.floatmin","category":"function","text":"floatmin(T = Float64)\n\nReturn the smallest positive normal number representable by the floating-point type T.\n\nExamples\n\njulia> floatmin(Float16)\nFloat16(6.104e-5)\n\njulia> floatmin(Float32)\n1.1754944f-38\n\njulia> floatmin()\n2.2250738585072014e-308\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.floatmax","location":"base/base.html#Base.floatmax","category":"function","text":"floatmax(T = Float64)\n\nReturn the largest finite number representable by the floating-point type T.\n\nSee also: typemax, floatmin, eps.\n\nExamples\n\njulia> floatmax(Float16)\nFloat16(6.55e4)\n\njulia> floatmax(Float32)\n3.4028235f38\n\njulia> floatmax()\n1.7976931348623157e308\n\njulia> typemax(Float64)\nInf\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.maxintfloat","location":"base/base.html#Base.maxintfloat","category":"function","text":"maxintfloat(T, S)\n\nThe largest consecutive integer representable in the given floating-point type T that also does not exceed the maximum integer representable by the integer type S.  Equivalently, it is the minimum of maxintfloat(T) and typemax(S).\n\n\n\n\n\nmaxintfloat(T=Float64)\n\nThe largest consecutive integer-valued floating-point number that is exactly represented in the given floating-point type T (which defaults to Float64).\n\nThat is, maxintfloat returns the smallest positive integer-valued floating-point number n such that n+1 is not exactly representable in the type T.\n\nWhen an Integer-type value is needed, use Integer(maxintfloat(T)).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.eps","location":"base/base.html#Base.eps-Tuple{Type{<:AbstractFloat}}","category":"method","text":"eps(::Type{T}) where T<:AbstractFloat\neps()\n\nReturn the machine epsilon of the floating point type T (T = Float64 by default). This is defined as the gap between 1 and the next largest value representable by typeof(one(T)), and is equivalent to eps(one(T)).  (Since eps(T) is a bound on the relative error of T, it is a \"dimensionless\" quantity like one.)\n\nExamples\n\njulia> eps()\n2.220446049250313e-16\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> 1.0 + eps()\n1.0000000000000002\n\njulia> 1.0 + eps()/2\n1.0\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.eps","location":"base/base.html#Base.eps-Tuple{AbstractFloat}","category":"method","text":"eps(x::AbstractFloat)\n\nReturn the unit in last place (ulp) of x. This is the distance between consecutive representable floating point values at x. In most cases, if the distance on either side of x is different, then the larger of the two is taken, that is\n\neps(x) == max(x-prevfloat(x), nextfloat(x)-x)\n\nThe exceptions to this rule are the smallest and largest finite values (e.g. nextfloat(-Inf) and prevfloat(Inf) for Float64), which round to the smaller of the values.\n\nThe rationale for this behavior is that eps bounds the floating point rounding error. Under the default RoundNearest rounding mode, if y is a real number and x is the nearest floating point number to y, then\n\ny-x leq operatornameeps(x)2\n\nSee also: nextfloat, issubnormal, floatmax.\n\nExamples\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(prevfloat(2.0))\n2.220446049250313e-16\n\njulia> eps(2.0)\n4.440892098500626e-16\n\njulia> x = prevfloat(Inf)      # largest finite Float64\n1.7976931348623157e308\n\njulia> x + eps(x)/2            # rounds up\nInf\n\njulia> x + prevfloat(eps(x)/2) # rounds down\n1.7976931348623157e308\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.instances","location":"base/base.html#Base.instances","category":"function","text":"instances(T::Type)\n\nReturn a collection of all instances of the given type, if applicable. Mostly used for enumerated types (see @enum).\n\nExamples\n\njulia> @enum Color red blue green\n\njulia> instances(Color)\n(red, blue, green)\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Any","location":"base/base.html#Core.Any","category":"type","text":"Any::DataType\n\nAny is the union of all types. It has the defining property isa(x, Any) == true for any x. Any therefore describes the entire universe of possible values. For example Integer is a subset of Any that includes Int, Int8, and other integer types.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Union","location":"base/base.html#Core.Union","category":"type","text":"Union{Types...}\n\nA Union type is an abstract type which includes all instances of any of its argument types. This means that T <: Union{T,S} and S <: Union{T,S}.\n\nLike other abstract types, it cannot be instantiated, even if all of its arguments are non abstract.\n\nExamples\n\njulia> IntOrString = Union{Int,AbstractString}\nUnion{Int64, AbstractString}\n\njulia> 1 isa IntOrString # instance of Int is included in the union\ntrue\n\njulia> \"Hello!\" isa IntOrString # String is also included\ntrue\n\njulia> 1.0 isa IntOrString # Float64 is not included because it is neither Int nor AbstractString\nfalse\n\nExtended Help\n\nUnlike most other parametric types, unions are covariant in their parameters. For example, Union{Real, String} is a subtype of Union{Number, AbstractString}.\n\nThe empty union Union{} is the bottom type of Julia.\n\n\n\n\n\n","page":"Essentials"},{"title":"Union{}","location":"base/base.html#Union{}","category":"keyword","text":"Union{}\n\nUnion{}, the empty Union of types, is the bottom type of the type system. That is, for each T::Type, Union{} <: T. Also see the subtyping operator's documentation: <:.\n\nAs such, Union{} is also an empty/uninhabited type, meaning that it has no values. That is, for each x, isa(x, Union{}) == false.\n\nBase.Bottom is defined as its alias and the type of Union{} is Core.TypeofBottom.\n\nExamples\n\njulia> isa(nothing, Union{})\nfalse\n\njulia> Union{} <: Int\ntrue\n\njulia> typeof(Union{}) === Core.TypeofBottom\ntrue\n\njulia> isa(Union{}, Union)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UnionAll","location":"base/base.html#Core.UnionAll","category":"type","text":"UnionAll\n\nA union of types over all values of a type parameter. UnionAll is used to describe parametric types where the values of some parameters are not known. See the manual section on UnionAll Types.\n\nExamples\n\njulia> typeof(Vector)\nUnionAll\n\njulia> typeof(Vector{Int})\nDataType\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Tuple","location":"base/base.html#Core.Tuple","category":"type","text":"Tuple{Types...}\n\nA tuple is a fixed-length container that can hold any values of different types, but cannot be modified (it is immutable). The values can be accessed via indexing. Tuple literals are written with commas and parentheses:\n\njulia> (1, 1+1)\n(1, 2)\n\njulia> (1,)\n(1,)\n\njulia> x = (0.0, \"hello\", 6*7)\n(0.0, \"hello\", 42)\n\njulia> x[2]\n\"hello\"\n\njulia> typeof(x)\nTuple{Float64, String, Int64}\n\nA length-1 tuple must be written with a comma, (1,), since (1) would just be a parenthesized value. () represents the empty (length-0) tuple.\n\nA tuple can be constructed from an iterator by using a Tuple type as constructor:\n\njulia> Tuple([\"a\", 1])\n(\"a\", 1)\n\njulia> Tuple{String, Float64}([\"a\", 1])\n(\"a\", 1.0)\n\nTuple types are covariant in their parameters: Tuple{Int} is a subtype of Tuple{Any}. Therefore Tuple{Any} is considered an abstract type, and tuple types are only concrete if their parameters are. Tuples do not have field names; fields are only accessed by index. Tuple types may have any number of parameters.\n\nSee the manual section on Tuple Types.\n\nSee also Vararg, NTuple, ntuple, tuple, NamedTuple.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.NTuple","location":"base/base.html#Core.NTuple","category":"type","text":"NTuple{N, T}\n\nA compact way of representing the type for a tuple of length N where all elements are of type T.\n\nExamples\n\njulia> isa((1, 2, 3, 4, 5, 6), NTuple{6, Int})\ntrue\n\nSee also ntuple.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.NamedTuple","location":"base/base.html#Core.NamedTuple","category":"type","text":"NamedTuple\n\nNamedTuples are, as their name suggests, named Tuples. That is, they're a tuple-like collection of values, where each entry has a unique name, represented as a Symbol. Like Tuples, NamedTuples are immutable; neither the names nor the values can be modified in place after construction.\n\nA named tuple can be created as a tuple literal with keys, e.g. (a=1, b=2), or as a tuple literal with semicolon after the opening parenthesis, e.g. (; a=1, b=2) (this form also accepts programmatically generated names as described below), or using a NamedTuple type as constructor, e.g. NamedTuple{(:a, :b)}((1,2)).\n\nAccessing the value associated with a name in a named tuple can be done using field access syntax, e.g. x.a, or using getindex, e.g. x[:a] or x[(:a, :b)]. A tuple of the names can be obtained using keys, and a tuple of the values can be obtained using values.\n\nnote: Note\nIteration over NamedTuples produces the values without the names. (See example below.) To iterate over the name-value pairs, use the pairs function.\n\nThe @NamedTuple macro can be used for conveniently declaring NamedTuple types.\n\nExamples\n\njulia> x = (a=1, b=2)\n(a = 1, b = 2)\n\njulia> x.a\n1\n\njulia> x[:a]\n1\n\njulia> x[(:a,)]\n(a = 1,)\n\njulia> keys(x)\n(:a, :b)\n\njulia> values(x)\n(1, 2)\n\njulia> collect(x)\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(pairs(x))\n2-element Vector{Pair{Symbol, Int64}}:\n :a => 1\n :b => 2\n\nIn a similar fashion as to how one can define keyword arguments programmatically, a named tuple can be created by giving pairs name::Symbol => value after a semicolon inside a tuple literal. This and the name=value syntax can be mixed:\n\njulia> (; :a => 1, :b => 2, c=3)\n(a = 1, b = 2, c = 3)\n\nThe name-value pairs can also be provided by splatting a named tuple or any iterator that yields two-value collections holding each a symbol as first value:\n\njulia> keys = (:a, :b, :c); values = (1, 2, 3);\n\njulia> NamedTuple{keys}(values)\n(a = 1, b = 2, c = 3)\n\njulia> (; (keys .=> values)...)\n(a = 1, b = 2, c = 3)\n\njulia> nt1 = (a=1, b=2);\n\njulia> nt2 = (c=3, d=4);\n\njulia> (; nt1..., nt2..., b=20) # the final b overwrites the value from nt1\n(a = 1, b = 20, c = 3, d = 4)\n\njulia> (; zip(keys, values)...) # zip yields tuples such as (:a, 1)\n(a = 1, b = 2, c = 3)\n\nAs in keyword arguments, identifiers and dot expressions imply names:\n\njulia> x = 0\n0\n\njulia> t = (; x)\n(x = 0,)\n\njulia> (; t.x)\n(x = 0,)\n\ncompat: Julia 1.5\nImplicit names from identifiers and dot expressions are available as of Julia 1.5.\n\ncompat: Julia 1.7\nUse of getindex methods with multiple Symbols is available as of Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@NamedTuple","location":"base/base.html#Base.@NamedTuple","category":"macro","text":"@NamedTuple{key1::Type1, key2::Type2, ...}\n@NamedTuple begin key1::Type1; key2::Type2; ...; end\n\nThis macro gives a more convenient syntax for declaring NamedTuple types. It returns a NamedTuple type with the given keys and types, equivalent to NamedTuple{(:key1, :key2, ...), Tuple{Type1,Type2,...}}. If the ::Type declaration is omitted, it is taken to be Any.   The begin ... end form allows the declarations to be split across multiple lines (similar to a struct declaration), but is otherwise equivalent. The NamedTuple macro is used when printing NamedTuple types to e.g. the REPL.\n\nFor example, the tuple (a=3.1, b=\"hello\") has a type NamedTuple{(:a, :b), Tuple{Float64, String}}, which can also be declared via @NamedTuple as:\n\njulia> @NamedTuple{a::Float64, b::String}\n@NamedTuple{a::Float64, b::String}\n\njulia> @NamedTuple begin\n           a::Float64\n           b::String\n       end\n@NamedTuple{a::Float64, b::String}\n\ncompat: Julia 1.5\nThis macro is available as of Julia 1.5.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@Kwargs","location":"base/base.html#Base.@Kwargs","category":"macro","text":"@Kwargs{key1::Type1, key2::Type2, ...}\n\nThis macro gives a convenient way to construct the type representation of keyword arguments from the same syntax as @NamedTuple. For example, when we have a function call like func([positional arguments]; kw1=1.0, kw2=\"2\"), we can use this macro to construct the internal type representation of the keyword arguments as @Kwargs{kw1::Float64, kw2::String}. The macro syntax is specifically designed to simplify the signature type of a keyword method when it is printed in the stack trace view.\n\njulia> @Kwargs{init::Int} # the internal representation of keyword arguments\nBase.Pairs{Symbol, Int64, Nothing, @NamedTuple{init::Int64}}\n\njulia> sum(\"julia\"; init=1)\nERROR: MethodError: no method matching +(::Char, ::Char)\nThe function `+` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  +(::Any, ::Any, ::Any, ::Any...)\n   @ Base operators.jl:585\n  +(::Integer, ::AbstractChar)\n   @ Base char.jl:247\n  +(::T, ::Integer) where T<:AbstractChar\n   @ Base char.jl:237\n\nStacktrace:\n  [1] add_sum(x::Char, y::Char)\n    @ Base ./reduce.jl:24\n  [2] BottomRF\n    @ Base ./reduce.jl:86 [inlined]\n  [3] _foldl_impl(op::Base.BottomRF{typeof(Base.add_sum)}, init::Int64, itr::String)\n    @ Base ./reduce.jl:62\n  [4] foldl_impl(op::Base.BottomRF{typeof(Base.add_sum)}, nt::Int64, itr::String)\n    @ Base ./reduce.jl:48 [inlined]\n  [5] mapfoldl_impl(f::typeof(identity), op::typeof(Base.add_sum), nt::Int64, itr::String)\n    @ Base ./reduce.jl:44 [inlined]\n  [6] mapfoldl(f::typeof(identity), op::typeof(Base.add_sum), itr::String; init::Int64)\n    @ Base ./reduce.jl:175 [inlined]\n  [7] mapreduce(f::typeof(identity), op::typeof(Base.add_sum), itr::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:307 [inlined]\n  [8] sum(f::typeof(identity), a::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:535 [inlined]\n  [9] sum(a::String; kw::@Kwargs{init::Int64})\n    @ Base ./reduce.jl:564 [inlined]\n [10] top-level scope\n    @ REPL[12]:1\n\ncompat: Julia 1.10\nThis macro is available as of Julia 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Val","location":"base/base.html#Base.Val","category":"type","text":"Val(c)\n\nReturn Val{c}(), which contains no run-time data. Types like this can be used to pass the information between functions through the value c, which must be an isbits value or a Symbol. The intent of this construct is to be able to dispatch on constants directly (at compile time) without having to test the value of the constant at run time.\n\nExamples\n\njulia> f(::Val{true}) = \"Good\"\nf (generic function with 1 method)\n\njulia> f(::Val{false}) = \"Bad\"\nf (generic function with 2 methods)\n\njulia> f(Val(true))\n\"Good\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Vararg","location":"base/base.html#Core.Vararg","category":"constant","text":"Vararg{T,N}\n\nThe last parameter of a tuple type Tuple can be the special value Vararg, which denotes any number of trailing elements. Vararg{T,N} corresponds to exactly N elements of type T. Finally Vararg{T} corresponds to zero or more elements of type T. Vararg tuple types are used to represent the arguments accepted by varargs methods (see the section on Varargs Functions in the manual.)\n\nSee also NTuple.\n\nExamples\n\njulia> mytupletype = Tuple{AbstractString, Vararg{Int}}\nTuple{AbstractString, Vararg{Int64}}\n\njulia> isa((\"1\",), mytupletype)\ntrue\n\njulia> isa((\"1\",1), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2), mytupletype)\ntrue\n\njulia> isa((\"1\",1,2,3.0), mytupletype)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Nothing","location":"base/base.html#Core.Nothing","category":"type","text":"Nothing\n\nA type with no fields that is the type of nothing.\n\nSee also: isnothing, Some, Missing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isnothing","location":"base/base.html#Base.isnothing","category":"function","text":"isnothing(x)\n\nReturn true if x === nothing, and return false if not.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nSee also something, Base.notnothing, ismissing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.notnothing","location":"base/base.html#Base.notnothing","category":"function","text":"notnothing(x)\n\nThrow an error if x === nothing, and return x if not.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Some","location":"base/base.html#Base.Some","category":"type","text":"Some{T}\n\nA wrapper type used in Union{Some{T}, Nothing} to distinguish between the absence of a value (nothing) and the presence of a nothing value (i.e. Some(nothing)).\n\nUse something to access the value wrapped by a Some object.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.something","location":"base/base.html#Base.something","category":"function","text":"something(x...)\n\nReturn the first value in the arguments which is not equal to nothing, if any. Otherwise throw an error. Arguments of type Some are unwrapped.\n\nSee also coalesce, skipmissing, @something.\n\nExamples\n\njulia> something(nothing, 1)\n1\n\njulia> something(Some(1), nothing)\n1\n\njulia> something(Some(nothing), 2) === nothing\ntrue\n\njulia> something(missing, nothing)\nmissing\n\njulia> something(nothing, nothing)\nERROR: ArgumentError: No value arguments present\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@something","location":"base/base.html#Base.@something","category":"macro","text":"@something(x...)\n\nShort-circuiting version of something.\n\nExamples\n\njulia> f(x) = (println(\"f($x)\"); nothing);\n\njulia> a = 1;\n\njulia> a = @something a f(2) f(3) error(\"Unable to find default for `a`\")\n1\n\njulia> b = nothing;\n\njulia> b = @something b f(2) f(3) error(\"Unable to find default for `b`\")\nf(2)\nf(3)\nERROR: Unable to find default for `b`\n[...]\n\njulia> b = @something b f(2) f(3) Some(nothing)\nf(2)\nf(3)\n\njulia> b === nothing\ntrue\n\ncompat: Julia 1.7\nThis macro is available as of Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Enums.Enum","location":"base/base.html#Base.Enums.Enum","category":"type","text":"Enum{T<:Integer}\n\nThe abstract supertype of all enumerated types defined with @enum.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Enums.@enum","location":"base/base.html#Base.Enums.@enum","category":"macro","text":"@enum EnumName[::BaseType] value1[=x] value2[=y]\n\nCreate an Enum{BaseType} subtype with name EnumName and enum member values of value1 and value2 with optional assigned values of x and y, respectively. EnumName can be used just like other types and enum member values as regular values, such as\n\nExamples\n\njulia> @enum Fruit apple=1 orange=2 kiwi=3\n\njulia> f(x::Fruit) = \"I'm a Fruit with value: $(Int(x))\"\nf (generic function with 1 method)\n\njulia> f(apple)\n\"I'm a Fruit with value: 1\"\n\njulia> Fruit(1)\napple::Fruit = 1\n\nValues can also be specified inside a begin block, e.g.\n\n@enum EnumName begin\n    value1\n    value2\nend\n\nBaseType, which defaults to Int32, must be a primitive subtype of Integer. Member values can be converted between the enum type and BaseType. read and write perform these conversions automatically. In case the enum is created with a non-default BaseType, Integer(value1) will return the integer value1 with the type BaseType.\n\nTo list all the instances of an enum use instances, e.g.\n\njulia> instances(Fruit)\n(apple, orange, kiwi)\n\nIt is possible to construct a symbol from an enum instance:\n\njulia> Symbol(apple)\n:apple\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Expr","location":"base/base.html#Core.Expr","category":"type","text":"Expr(head::Symbol, args...)\n\nA type representing compound expressions in parsed julia code (ASTs). Each expression consists of a head Symbol identifying which kind of expression it is (e.g. a call, for loop, conditional statement, etc.), and subexpressions (e.g. the arguments of a call). The subexpressions are stored in a Vector{Any} field called args.\n\nSee the manual chapter on Metaprogramming and the developer documentation Julia ASTs.\n\nExamples\n\njulia> Expr(:call, :+, 1, 2)\n:(1 + 2)\n\njulia> dump(:(a ? b : c))\nExpr\n  head: Symbol if\n  args: Array{Any}((3,))\n    1: Symbol a\n    2: Symbol b\n    3: Symbol c\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Symbol","location":"base/base.html#Core.Symbol","category":"type","text":"Symbol\n\nThe type of object used to represent identifiers in parsed julia code (ASTs). Also often used as a name or label to identify an entity (e.g. as a dictionary key). Symbols can be entered using the : quote operator:\n\njulia> :name\n:name\n\njulia> typeof(:name)\nSymbol\n\njulia> x = 42\n42\n\njulia> eval(:x)\n42\n\nSymbols can also be constructed from strings or other values by calling the constructor Symbol(x...).\n\nSymbols are immutable and their implementation re-uses the same object for all Symbols with the same name.\n\nUnlike strings, Symbols are \"atomic\" or \"scalar\" entities that do not support iteration over characters.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Symbol","location":"base/base.html#Core.Symbol-Tuple","category":"method","text":"Symbol(x...) -> Symbol\n\nCreate a Symbol by concatenating the string representations of the arguments together.\n\nExamples\n\njulia> Symbol(\"my\", \"name\")\n:myname\n\njulia> Symbol(\"day\", 4)\n:day4\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Module","location":"base/base.html#Core.Module","category":"type","text":"Module\n\nA Module is a separate global variable workspace. See module and the manual section about modules for details.\n\nModule(name::Symbol=:anonymous, std_imports=true, default_names=true)\n\nReturn a module with the specified name. A baremodule corresponds to Module(:ModuleName, false)\n\nAn empty module containing no names at all can be created with Module(:ModuleName, false, false). This module will not import Base or Core and does not contain a reference to itself.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.Function","location":"base/base.html#Core.Function","category":"type","text":"Function\n\nAbstract type of all functions.\n\nExamples\n\njulia> isa(+, Function)\ntrue\n\njulia> typeof(sin)\ntypeof(sin) (singleton type of function sin, subtype of Function)\n\njulia> ans <: Function\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.hasmethod","location":"base/base.html#Base.hasmethod","category":"function","text":"hasmethod(f, t::Type{<:Tuple}[, kwnames]; world=get_world_counter()) -> Bool\n\nDetermine whether the given generic function has a method matching the given Tuple of argument types with the upper bound of world age given by world.\n\nIf a tuple of keyword argument names kwnames is provided, this also checks whether the method of f matching t has the given keyword argument names. If the matching method accepts a variable number of keyword arguments, e.g. with kwargs..., any names given in kwnames are considered valid. Otherwise the provided names must be a subset of the method's keyword arguments.\n\nSee also applicable.\n\ncompat: Julia 1.2\nProviding keyword argument names requires Julia 1.2 or later.\n\nExamples\n\njulia> hasmethod(length, Tuple{Array})\ntrue\n\njulia> f(; oranges=0) = oranges;\n\njulia> hasmethod(f, Tuple{}, (:oranges,))\ntrue\n\njulia> hasmethod(f, Tuple{}, (:apples, :bananas))\nfalse\n\njulia> g(; xs...) = 4;\n\njulia> hasmethod(g, Tuple{}, (:a, :b, :c, :d))  # g accepts arbitrary kwargs\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.applicable","location":"base/base.html#Core.applicable","category":"function","text":"applicable(f, args...) -> Bool\n\nDetermine whether the given generic function has a method applicable to the given arguments.\n\nSee also hasmethod.\n\nExamples\n\njulia> function f(x, y)\n           x + y\n       end;\n\njulia> applicable(f, 1)\nfalse\n\njulia> applicable(f, 1, 2)\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isambiguous","location":"base/base.html#Base.isambiguous","category":"function","text":"Base.isambiguous(m1, m2; ambiguous_bottom=false) -> Bool\n\nDetermine whether two methods m1 and m2 may be ambiguous for some call signature. This test is performed in the context of other methods of the same function; in isolation, m1 and m2 might be ambiguous, but if a third method resolving the ambiguity has been defined, this returns false. Alternatively, in isolation m1 and m2 might be ordered, but if a third method cannot be sorted with them, they may cause an ambiguity together.\n\nFor parametric types, the ambiguous_bottom keyword argument controls whether Union{} counts as an ambiguous intersection of type parameters – when true, it is considered ambiguous, when false it is not.\n\nExamples\n\njulia> foo(x::Complex{<:Integer}) = 1\nfoo (generic function with 1 method)\n\njulia> foo(x::Complex{<:Rational}) = 2\nfoo (generic function with 2 methods)\n\njulia> m1, m2 = collect(methods(foo));\n\njulia> typeintersect(m1.sig, m2.sig)\nTuple{typeof(foo), Complex{Union{}}}\n\njulia> Base.isambiguous(m1, m2, ambiguous_bottom=true)\ntrue\n\njulia> Base.isambiguous(m1, m2, ambiguous_bottom=false)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.invoke","location":"base/base.html#Core.invoke","category":"function","text":"invoke(f, argtypes::Type, args...; kwargs...)\ninvoke(f, argtypes::Method, args...; kwargs...)\ninvoke(f, argtypes::CodeInstance, args...; kwargs...)\n\nInvoke a method for the given generic function f matching the specified types argtypes on the specified arguments args and passing the keyword arguments kwargs. The arguments args must conform with the specified types in argtypes, i.e. conversion is not automatically performed. This method allows invoking a method other than the most specific matching method, which is useful when the behavior of a more general definition is explicitly needed (often as part of the implementation of a more specific method of the same function). However, because this means the runtime must do more work, invoke is generally also slower–sometimes significantly so–than doing normal dispatch with a regular call.\n\nBe careful when using invoke for functions that you don't write. What definition is used for given argtypes is an implementation detail unless the function is explicitly states that calling with certain argtypes is a part of public API.  For example, the change between f1 and f2 in the example below is usually considered compatible because the change is invisible by the caller with a normal (non-invoke) call.  However, the change is visible if you use invoke.\n\nPassing a Method instead of a signature\n\nThe argtypes argument may be a Method, in which case the ordinary method table lookup is bypassed entirely and the given method is invoked directly. Needing this feature is uncommon. Note in particular that the specified Method may be entirely unreachable from ordinary dispatch (or ordinary invoke), e.g. because it was replaced or fully covered by more specific methods. If the method is part of the ordinary method table, this call behaves similar to invoke(f, method.sig, args...).\n\ncompat: Julia 1.12\nPassing a Method requires Julia 1.12.\n\nPassing a CodeInstance instead of a signature\n\nThe argtypes argument may be a CodeInstance, bypassing both method lookup and specialization. The semantics of this invocation are similar to a function pointer call of the CodeInstance's invoke pointer. It is an error to invoke a CodeInstance with arguments that do not match its parent MethodInstance or from a world age not included in the min_world/max_world range. It is undefined behavior to invoke a CodeInstance whose behavior does not match the constraints specified in its fields. For some code instances with owner !== nothing (i.e. those generated by external compilers), it may be an error to invoke them after passing through precompilation. This is an advanced interface intended for use with external compiler plugins.\n\ncompat: Julia 1.12\nPassing a CodeInstance requires Julia 1.12.\n\nExamples\n\njulia> f(x::Real) = x^2;\n\njulia> f(x::Integer) = 1 + invoke(f, Tuple{Real}, x);\n\njulia> f(2)\n5\n\njulia> f1(::Integer) = Integer\n       f1(::Real) = Real;\n\njulia> f2(x::Real) = _f2(x)\n       _f2(::Integer) = Integer\n       _f2(_) = Real;\n\njulia> f1(1)\nInteger\n\njulia> f2(1)\nInteger\n\njulia> invoke(f1, Tuple{Real}, 1)\nReal\n\njulia> invoke(f2, Tuple{Real}, 1)\nInteger\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@invoke","location":"base/base.html#Base.@invoke","category":"macro","text":"@invoke f(arg::T, ...; kwargs...)\n\nProvides a convenient way to call invoke by expanding @invoke f(arg1::T1, arg2::T2; kwargs...) to invoke(f, Tuple{T1,T2}, arg1, arg2; kwargs...). When an argument's type annotation is omitted, it's replaced with Core.Typeof that argument. To invoke a method where an argument is untyped or explicitly typed as Any, annotate the argument with ::Any.\n\nIt also supports the following syntax:\n\n@invoke (x::X).f expands to invoke(getproperty, Tuple{X,Symbol}, x, :f)\n@invoke (x::X).f = v::V expands to invoke(setproperty!, Tuple{X,Symbol,V}, x, :f, v)\n@invoke (xs::Xs)[i::I] expands to invoke(getindex, Tuple{Xs,I}, xs, i)\n@invoke (xs::Xs)[i::I] = v::V expands to invoke(setindex!, Tuple{Xs,V,I}, xs, v, i)\n\nExamples\n\njulia> @macroexpand @invoke f(x::T, y)\n:(Core.invoke(f, Tuple{T, Core.Typeof(y)}, x, y))\n\njulia> @invoke 420::Integer % Unsigned\n0x00000000000001a4\n\njulia> @macroexpand @invoke (x::X).f\n:(Core.invoke(Base.getproperty, Tuple{X, Core.Typeof(:f)}, x, :f))\n\njulia> @macroexpand @invoke (x::X).f = v::V\n:(Core.invoke(Base.setproperty!, Tuple{X, Core.Typeof(:f), V}, x, :f, v))\n\njulia> @macroexpand @invoke (xs::Xs)[i::I]\n:(Core.invoke(Base.getindex, Tuple{Xs, I}, xs, i))\n\njulia> @macroexpand @invoke (xs::Xs)[i::I] = v::V\n:(Core.invoke(Base.setindex!, Tuple{Xs, V, I}, xs, v, i))\n\ncompat: Julia 1.7\nThis macro requires Julia 1.7 or later.\n\ncompat: Julia 1.9\nThis macro is exported as of Julia 1.9.\n\ncompat: Julia 1.10\nThe additional syntax is supported as of Julia 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.invokelatest","location":"base/base.html#Base.invokelatest","category":"function","text":"invokelatest(f, args...; kwargs...)\n\nCalls f(args...; kwargs...), but guarantees that the most recent method of f will be executed.   This is useful in specialized circumstances, e.g. long-running event loops or callback functions that may call obsolete versions of a function f. (The drawback is that invokelatest is somewhat slower than calling f directly, and the type of the result cannot be inferred by the compiler.)\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this function was not exported, and was called as Base.invokelatest.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@invokelatest","location":"base/base.html#Base.@invokelatest","category":"macro","text":"@invokelatest f(args...; kwargs...)\n\nProvides a convenient way to call invokelatest. @invokelatest f(args...; kwargs...) will simply be expanded into Base.invokelatest(f, args...; kwargs...).\n\nIt also supports the following syntax:\n\n@invokelatest x.f expands to Base.invokelatest(getproperty, x, :f)\n@invokelatest x.f = v expands to Base.invokelatest(setproperty!, x, :f, v)\n@invokelatest xs[i] expands to Base.invokelatest(getindex, xs, i)\n@invokelatest xs[i] = v expands to Base.invokelatest(setindex!, xs, v, i)\n\nnote: Note\nIf f is a global, it will be resolved consistently in the (latest) world as the call target. However, all other arguments (as well as f itself if it is not a literal global) will be evaluated in the current world age.\n\ncompat: Julia 1.7\nThis macro requires Julia 1.7 or later.\n\ncompat: Julia 1.9\nPrior to Julia 1.9, this macro was not exported, and was called as Base.@invokelatest.\n\ncompat: Julia 1.10\nThe additional x.f and xs[i] syntax requires Julia 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"new","location":"base/base.html#new","category":"keyword","text":"new, or new{A,B,...}\n\nSpecial function available to inner constructors which creates a new object of the type. The form new{A,B,...} explicitly specifies values of parameters for parametric types. See the manual section on Inner Constructor Methods for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.:|>","location":"base/base.html#Base.:|>","category":"function","text":"|>(x, f)\n\nInfix operator which applies function f to the argument x. This allows f(g(x)) to be written x |> g |> f. When used with anonymous functions, parentheses are typically required around the definition to get the intended chain.\n\nExamples\n\njulia> 4 |> inv\n0.25\n\njulia> [2, 3, 5] |> sum |> inv\n0.1\n\njulia> [0 1; 2 3] .|> (x -> x^2) |> sum\n14\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.:∘","location":"base/base.html#Base.:∘","category":"function","text":"f ∘ g\n\nCompose functions: i.e. (f ∘ g)(args...; kwargs...) means f(g(args...; kwargs...)). The ∘ symbol can be entered in the Julia REPL (and most editors, appropriately configured) by typing \\circ<tab>.\n\nFunction composition also works in prefix form: ∘(f, g) is the same as f ∘ g. The prefix form supports composition of multiple functions: ∘(f, g, h) = f ∘ g ∘ h and splatting ∘(fs...) for composing an iterable collection of functions. The last argument to ∘ execute first.\n\ncompat: Julia 1.4\nMultiple function composition requires at least Julia 1.4.\n\ncompat: Julia 1.5\nComposition of one function ∘(f) requires at least Julia 1.5.\n\ncompat: Julia 1.7\nUsing keyword arguments requires at least Julia 1.7.\n\nExamples\n\njulia> map(uppercase∘first, [\"apple\", \"banana\", \"carrot\"])\n3-element Vector{Char}:\n 'A': ASCII/Unicode U+0041 (category Lu: Letter, uppercase)\n 'B': ASCII/Unicode U+0042 (category Lu: Letter, uppercase)\n 'C': ASCII/Unicode U+0043 (category Lu: Letter, uppercase)\n\njulia> (==(6)∘length).([\"apple\", \"banana\", \"carrot\"])\n3-element BitVector:\n 0\n 1\n 1\n\njulia> fs = [\n           x -> 2x\n           x -> x-1\n           x -> x/2\n           x -> x+1\n       ];\n\njulia> ∘(fs...)(3)\n2.0\n\nSee also ComposedFunction, !f::Function.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ComposedFunction","location":"base/base.html#Base.ComposedFunction","category":"type","text":"ComposedFunction{Outer,Inner} <: Function\n\nRepresents the composition of two callable objects outer::Outer and inner::Inner. That is\n\nComposedFunction(outer, inner)(args...; kw...) === outer(inner(args...; kw...))\n\nThe preferred way to construct an instance of ComposedFunction is to use the composition operator ∘:\n\njulia> sin ∘ cos === ComposedFunction(sin, cos)\ntrue\n\njulia> typeof(sin∘cos)\nComposedFunction{typeof(sin), typeof(cos)}\n\nThe composed pieces are stored in the fields of ComposedFunction and can be retrieved as follows:\n\njulia> composition = sin ∘ cos\nsin ∘ cos\n\njulia> composition.outer === sin\ntrue\n\njulia> composition.inner === cos\ntrue\n\ncompat: Julia 1.6\nComposedFunction requires at least Julia 1.6. In earlier versions ∘ returns an anonymous function instead.\n\nSee also ∘.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.splat","location":"base/base.html#Base.splat","category":"function","text":"splat(f)\n\nEquivalent to\n\n    my_splat(f) = args->f(args...)\n\ni.e. given a function returns a new function that takes one argument and splats it into the original function. This is useful as an adaptor to pass a multi-argument function in a context that expects a single argument, but passes a tuple as that single argument.\n\nExamples\n\njulia> map(splat(+), zip(1:3,4:6))\n3-element Vector{Int64}:\n 5\n 7\n 9\n\njulia> my_add = splat(+)\nsplat(+)\n\njulia> my_add((1,2,3))\n6\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Fix","location":"base/base.html#Base.Fix","category":"type","text":"Fix{N}(f, x)\n\nA type representing a partially-applied version of a function f, with the argument x fixed at position N::Int. In other words, Fix{3}(f, x) behaves similarly to (y1, y2, y3...; kws...) -> f(y1, y2, x, y3...; kws...).\n\ncompat: Julia 1.12\nThis general functionality requires at least Julia 1.12, while Fix1 and Fix2 are available earlier.\n\nnote: Note\nWhen nesting multiple Fix, note that the N in Fix{N} is relative to the current available arguments, rather than an absolute ordering on the target function. For example, Fix{1}(Fix{2}(f, 4), 4) fixes the first and second arg, while Fix{2}(Fix{1}(f, 4), 4) fixes the first and third arg.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Fix1","location":"base/base.html#Base.Fix1","category":"type","text":"Alias for Fix{1}. See Fix.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Fix2","location":"base/base.html#Base.Fix2","category":"type","text":"Alias for Fix{2}. See Fix.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.eval","location":"base/base.html#Core.eval","category":"function","text":"Core.eval(m::Module, expr)\n\nEvaluate an expression in the given module and return the result.\n\n\n\n\n\n","page":"Essentials"},{"title":"eval","location":"base/base.html#eval","category":"function","text":"eval(expr)\n\nEvaluate an expression in the global scope of the containing module. Every Module (except those defined with baremodule) has a private 1-argument definition of eval, which evaluates expressions in that module, for use inside that module.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@eval","location":"base/base.html#Base.@eval","category":"macro","text":"@eval [mod,] ex\n\nEvaluate an expression with values interpolated into it using eval. If two arguments are provided, the first is the module to evaluate in.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.evalfile","location":"base/base.html#Base.evalfile","category":"function","text":"evalfile(path::AbstractString, args::Vector{String}=String[])\n\nLoad the file into an anonymous module using include, evaluate all expressions, and return the value of the last expression. The optional args argument can be used to set the input arguments of the script (i.e. the global ARGS variable). Note that definitions (e.g. methods, globals) are evaluated in the anonymous module and do not affect the current module.\n\nExamples\n\njulia> write(\"testfile.jl\", \"\"\"\n           @show ARGS\n           1 + 1\n       \"\"\");\n\njulia> x = evalfile(\"testfile.jl\", [\"ARG1\", \"ARG2\"]);\nARGS = [\"ARG1\", \"ARG2\"]\n\njulia> x\n2\n\njulia> rm(\"testfile.jl\")\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.esc","location":"base/base.html#Base.esc","category":"function","text":"esc(e)\n\nOnly valid in the context of an Expr returned from a macro. Prevents the macro hygiene pass from turning embedded variables into gensym variables. See the Macros section of the Metaprogramming chapter of the manual for more details and examples.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@inbounds","location":"base/base.html#Base.@inbounds","category":"macro","text":"@inbounds(blk)\n\nEliminates array bounds checking within expressions.\n\nIn the example below the in-range check for referencing element i of array A is skipped to improve performance.\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nwarning: Warning\nUsing @inbounds may return incorrect results/crashes/corruption for out-of-bounds indices. The user is responsible for checking it manually. Only use @inbounds when you are certain that all accesses are in bounds (as undefined behavior, e.g. crashes, might occur if this assertion is violated). For example, using 1:length(A) instead of eachindex(A) in a function like the one above is not safely inbounds because the first index of A may not be 1 for all user defined types that subtype AbstractArray.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@boundscheck","location":"base/base.html#Base.@boundscheck","category":"macro","text":"@boundscheck(blk)\n\nAnnotates the expression blk as a bounds checking block, allowing it to be elided by @inbounds.\n\nnote: Note\nThe function in which @boundscheck is written must be inlined into its caller in order for @inbounds to have effect.\n\nExamples\n\njulia> @inline function g(A, i)\n           @boundscheck checkbounds(A, i)\n           return \"accessing ($A)[$i]\"\n       end;\n\njulia> f1() = return g(1:2, -1);\n\njulia> f2() = @inbounds return g(1:2, -1);\n\njulia> f1()\nERROR: BoundsError: attempt to access 2-element UnitRange{Int64} at index [-1]\nStacktrace:\n [1] throw_boundserror(::UnitRange{Int64}, ::Tuple{Int64}) at ./abstractarray.jl:455\n [2] checkbounds at ./abstractarray.jl:420 [inlined]\n [3] g at ./none:2 [inlined]\n [4] f1() at ./none:1\n [5] top-level scope\n\njulia> f2()\n\"accessing (1:2)[-1]\"\n\nwarning: Warning\nThe @boundscheck annotation allows you, as a library writer, to opt-in to allowing other code to remove your bounds checks with @inbounds. As noted there, the caller must verify—using information they can access—that their accesses are valid before using @inbounds. For indexing into your AbstractArray subclasses, for example, this involves checking the indices against its axes. Therefore, @boundscheck annotations should only be added to a getindex or setindex! implementation after you are certain its behavior is correct.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@propagate_inbounds","location":"base/base.html#Base.@propagate_inbounds","category":"macro","text":"@propagate_inbounds\n\nTells the compiler to inline a function while retaining the caller's inbounds context.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@inline","location":"base/base.html#Base.@inline","category":"macro","text":"@inline\n\nGive a hint to the compiler that this function is worth inlining.\n\nSmall functions typically do not need the @inline annotation, as the compiler does it automatically. By using @inline on bigger functions, an extra nudge can be given to the compiler to inline it.\n\n@inline can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\n@inline function longdef(x)\n    ...\nend\n\n# annotate short-form definition\n@inline shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    @inline\n    ...\nend\n\ncompat: Julia 1.8\nThe usage within a function body requires at least Julia 1.8.\n\n\n\n@inline block\n\nGive a hint to the compiler that calls within block are worth inlining.\n\n# The compiler will try to inline `f`\n@inline f(...)\n\n# The compiler will try to inline `f`, `g` and `+`\n@inline f(...) + g(...)\n\nnote: Note\nA callsite annotation always has the precedence over the annotation applied to the definition of the called function:@noinline function explicit_noinline(args...)\n    # body\nend\n\nlet\n    @inline explicit_noinline(args...) # will be inlined\nend\n\nnote: Note\nWhen there are nested callsite annotations, the innermost annotation has the precedence:@noinline let a0, b0 = ...\n    a = @inline f(a0)  # the compiler will try to inline this call\n    b = f(b0)          # the compiler will NOT try to inline this call\n    return a, b\nend\n\nwarning: Warning\nAlthough a callsite annotation will try to force inlining in regardless of the cost model, there are still chances it can't succeed in it. Especially, recursive calls can not be inlined even if they are annotated as @inlined.\n\ncompat: Julia 1.8\nThe callsite annotation requires at least Julia 1.8.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@noinline","location":"base/base.html#Base.@noinline","category":"macro","text":"@noinline\n\nGive a hint to the compiler that it should not inline a function.\n\nSmall functions are typically inlined automatically. By using @noinline on small functions, auto-inlining can be prevented.\n\n@noinline can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\n@noinline function longdef(x)\n    ...\nend\n\n# annotate short-form definition\n@noinline shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    @noinline\n    ...\nend\n\ncompat: Julia 1.8\nThe usage within a function body requires at least Julia 1.8.\n\n\n\n@noinline block\n\nGive a hint to the compiler that it should not inline the calls within block.\n\n# The compiler will try to not inline `f`\n@noinline f(...)\n\n# The compiler will try to not inline `f`, `g` and `+`\n@noinline f(...) + g(...)\n\nnote: Note\nA callsite annotation always has the precedence over the annotation applied to the definition of the called function:@inline function explicit_inline(args...)\n    # body\nend\n\nlet\n    @noinline explicit_inline(args...) # will not be inlined\nend\n\nnote: Note\nWhen there are nested callsite annotations, the innermost annotation has the precedence:@inline let a0, b0 = ...\n    a = @noinline f(a0)  # the compiler will NOT try to inline this call\n    b = f(b0)            # the compiler will try to inline this call\n    return a, b\nend\n\ncompat: Julia 1.8\nThe callsite annotation requires at least Julia 1.8.\n\n\n\nnote: Note\nIf the function is trivial (for example returning a constant) it might get inlined anyway.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@nospecialize","location":"base/base.html#Base.@nospecialize","category":"macro","text":"@nospecialize\n\nApplied to a function argument name, hints to the compiler that the method implementation should not be specialized for different types of that argument, but instead use the declared type for that argument. It can be applied to an argument within a formal argument list, or in the function body. When applied to an argument, the macro must wrap the entire argument expression, e.g., @nospecialize(x::Real) or @nospecialize(i::Integer...) rather than wrapping just the argument name. When used in a function body, the macro must occur in statement position and before any code.\n\nWhen used without arguments, it applies to all arguments of the parent scope. In local scope, this means all arguments of the containing function. In global (top-level) scope, this means all methods subsequently defined in the current module.\n\nSpecialization can reset back to the default by using @specialize.\n\nfunction example_function(@nospecialize x)\n    ...\nend\n\nfunction example_function(x, @nospecialize(y = 1))\n    ...\nend\n\nfunction example_function(x, y, z)\n    @nospecialize x y\n    ...\nend\n\n@nospecialize\nf(y) = [x for x in y]\n@specialize\n\nnote: Note\n@nospecialize affects code generation but not inference: it limits the diversity of the resulting native code, but it does not impose any limitations (beyond the standard ones) on type-inference. Use Base.@nospecializeinfer together with @nospecialize to additionally suppress inference.\n\nExamples\n\njulia> f(A::AbstractArray) = g(A)\nf (generic function with 1 method)\n\njulia> @noinline g(@nospecialize(A::AbstractArray)) = A[1]\ng (generic function with 1 method)\n\njulia> @code_typed f([1.0])\nCodeInfo(\n1 ─ %1 = invoke Main.g(_2::AbstractArray)::Float64\n└──      return %1\n) => Float64\n\nHere, the @nospecialize annotation results in the equivalent of\n\nf(A::AbstractArray) = invoke(g, Tuple{AbstractArray}, A)\n\nensuring that only one version of native code will be generated for g, one that is generic for any AbstractArray. However, the specific return type is still inferred for both g and f, and this is still used in optimizing the callers of f and g.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@specialize","location":"base/base.html#Base.@specialize","category":"macro","text":"@specialize\n\nReset the specialization hint for an argument back to the default. For details, see @nospecialize.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@nospecializeinfer","location":"base/base.html#Base.@nospecializeinfer","category":"macro","text":"Base.@nospecializeinfer function f(args...)\n    @nospecialize ...\n    ...\nend\nBase.@nospecializeinfer f(@nospecialize args...) = ...\n\nTells the compiler to infer f using the declared types of @nospecialized arguments. This can be used to limit the number of compiler-generated specializations during inference.\n\nExamples\n\njulia> f(A::AbstractArray) = g(A)\nf (generic function with 1 method)\n\njulia> @noinline Base.@nospecializeinfer g(@nospecialize(A::AbstractArray)) = A[1]\ng (generic function with 1 method)\n\njulia> @code_typed f([1.0])\nCodeInfo(\n1 ─ %1 = invoke Main.g(_2::AbstractArray)::Any\n└──      return %1\n) => Any\n\nIn this example, f will be inferred for each specific type of A, but g will only be inferred once with the declared argument type A::AbstractArray, meaning that the compiler will not likely see the excessive inference time on it while it can not infer the concrete return type of it. Without the @nospecializeinfer, f([1.0]) would infer the return type of g as Float64, indicating that inference ran for g(::Vector{Float64}) despite the prohibition on specialized code generation.\n\ncompat: Julia 1.10\nUsing Base.@nospecializeinfer requires Julia version 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@constprop","location":"base/base.html#Base.@constprop","category":"macro","text":"Base.@constprop setting [ex]\n\nControl the mode of interprocedural constant propagation for the annotated function.\n\nTwo settings are supported:\n\nBase.@constprop :aggressive [ex]: apply constant propagation aggressively. For a method where the return type depends on the value of the arguments, this can yield improved inference results at the cost of additional compile time.\nBase.@constprop :none [ex]: disable constant propagation. This can reduce compile times for functions that Julia might otherwise deem worthy of constant-propagation. Common cases are for functions with Bool- or Symbol-valued arguments or keyword arguments.\n\nBase.@constprop can be applied immediately before a function definition or within a function body.\n\n# annotate long-form definition\nBase.@constprop :aggressive function longdef(x)\n    ...\nend\n\n# annotate short-form definition\nBase.@constprop :aggressive shortdef(x) = ...\n\n# annotate anonymous function that a `do` block creates\nf() do\n    Base.@constprop :aggressive\n    ...\nend\n\ncompat: Julia 1.10\nThe usage within a function body requires at least Julia 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.gensym","location":"base/base.html#Base.gensym","category":"function","text":"gensym([tag])\n\nGenerates a symbol which will not conflict with other variable names (in the same module).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@gensym","location":"base/base.html#Base.@gensym","category":"macro","text":"@gensym\n\nGenerates a gensym symbol for a variable. For example, @gensym x y is transformed into x = gensym(\"x\"); y = gensym(\"y\").\n\n\n\n\n\n","page":"Essentials"},{"title":"var\"name\"","location":"base/base.html#var\"name\"","category":"keyword","text":"var\n\nThe syntax var\"#example#\" refers to a variable named Symbol(\"#example#\"), even though #example# is not a valid Julia identifier name.\n\nThis can be useful for interoperability with programming languages which have different rules for the construction of valid identifiers. For example, to refer to the R variable draw.segments, you can use var\"draw.segments\" in your Julia code.\n\nIt is also used to show julia source code which has gone through macro hygiene or otherwise contains variable names which can't be parsed normally.\n\nNote that this syntax requires parser support so it is expanded directly by the parser rather than being implemented as a normal string macro @var_str.\n\ncompat: Julia 1.3\nThis syntax requires at least Julia 1.3.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@goto","location":"base/base.html#Base.@goto","category":"macro","text":"@goto name\n\n@goto name unconditionally jumps to the statement at the location @label name.\n\n@label and @goto cannot create jumps to different top-level statements. Attempts cause an error. To still use @goto, enclose the @label and @goto in a block.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@label","location":"base/base.html#Base.@label","category":"macro","text":"@label name\n\nLabels a statement with the symbolic label name. The label marks the end-point of an unconditional jump with @goto name.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.SimdLoop.@simd","location":"base/base.html#Base.SimdLoop.@simd","category":"macro","text":"@simd\n\nAnnotate a for loop to allow the compiler to take extra liberties to allow loop re-ordering\n\nwarning: Warning\nThis feature is experimental and could change or disappear in future versions of Julia. Incorrect use of the @simd macro may cause unexpected results.\n\nThe object iterated over in a @simd for loop should be a one-dimensional range. By using @simd, you are asserting several properties of the loop:\n\nIt is safe to execute iterations in arbitrary or overlapping order, with special consideration for reduction variables.\nFloating-point operations on reduction variables can be reordered or contracted, possibly causing different results than without @simd.\n\nIn many cases, Julia is able to automatically vectorize inner for loops without the use of @simd. Using @simd gives the compiler a little extra leeway to make it possible in more situations. In either case, your inner loop should have the following properties to allow vectorization:\n\nThe loop must be an innermost loop\nThe loop body must be straight-line code. Therefore, @inbounds is   currently needed for all array accesses. The compiler can sometimes turn   short &&, ||, and ?: expressions into straight-line code if it is safe   to evaluate all operands unconditionally. Consider using the ifelse   function instead of ?: in the loop if it is safe to do so.\nAccesses must have a stride pattern and cannot be \"gathers\" (random-index   reads) or \"scatters\" (random-index writes).\nThe stride should be unit stride.\n\nnote: Note\nThe @simd does not assert by default that the loop is completely free of loop-carried memory dependencies, which is an assumption that can easily be violated in generic code. If you are writing non-generic code, you can use @simd ivdep for ... end to also assert that:\n\nThere exists no loop-carried memory dependencies\nNo iteration ever waits on a previous iteration to make forward progress.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@polly","location":"base/base.html#Base.@polly","category":"macro","text":"@polly\n\nTells the compiler to apply the polyhedral optimizer Polly to a function.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@generated","location":"base/base.html#Base.@generated","category":"macro","text":"@generated f\n\n@generated is used to annotate a function which will be generated. In the body of the generated function, only types of arguments can be read (not the values). The function returns a quoted expression evaluated when the function is called. The @generated macro should not be used on functions mutating the global scope or depending on mutable elements.\n\nSee Metaprogramming for further details.\n\nExamples\n\njulia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@assume_effects","location":"base/base.html#Base.@assume_effects","category":"macro","text":"Base.@assume_effects setting... [ex]\n\nOverride the compiler's effect modeling. This macro can be used in several contexts:\n\nImmediately before a method definition, to override the entire effect modeling of the applied method.\nWithin a function body without any arguments, to override the entire effect modeling of the enclosing method.\nApplied to a code block, to override the local effect modeling of the applied code block.\n\nExamples\n\njulia> Base.@assume_effects :terminates_locally function fact(x)\n           # usage 1:\n           # this :terminates_locally allows `fact` to be constant-folded\n           res = 1\n           0 ≤ x < 20 || error(\"bad fact\")\n           while x > 1\n               res *= x\n               x -= 1\n           end\n           return res\n       end\nfact (generic function with 1 method)\n\njulia> code_typed() do\n           fact(12)\n       end |> only\nCodeInfo(\n1 ─     return 479001600\n) => Int64\n\njulia> code_typed() do\n           map((2,3,4)) do x\n               # usage 2:\n               # this :terminates_locally allows this anonymous function to be constant-folded\n               Base.@assume_effects :terminates_locally\n               res = 1\n               0 ≤ x < 20 || error(\"bad fact\")\n               while x > 1\n                   res *= x\n                   x -= 1\n               end\n               return res\n           end\n       end |> only\nCodeInfo(\n1 ─     return (2, 6, 24)\n) => Tuple{Int64, Int64, Int64}\n\njulia> code_typed() do\n           map((2,3,4)) do x\n               res = 1\n               0 ≤ x < 20 || error(\"bad fact\")\n               # usage 3:\n               # with this :terminates_locally annotation the compiler skips tainting\n               # `:terminates` effect within this `while` block, allowing the parent\n               # anonymous function to be constant-folded\n               Base.@assume_effects :terminates_locally while x > 1\n                   res *= x\n                   x -= 1\n               end\n               return res\n           end\n       end |> only\nCodeInfo(\n1 ─     return (2, 6, 24)\n) => Tuple{Int64, Int64, Int64}\n\ncompat: Julia 1.8\nUsing Base.@assume_effects requires Julia version 1.8.\n\ncompat: Julia 1.10\nThe usage within a function body requires at least Julia 1.10.\n\ncompat: Julia 1.11\nThe code block annotation requires at least Julia 1.11.\n\nwarning: Warning\nImproper use of this macro causes undefined behavior (including crashes, incorrect answers, or other hard to track bugs). Use with care and only as a last resort if absolutely required. Even in such a case, you SHOULD take all possible steps to minimize the strength of the effect assertion (e.g., do not use :total if :nothrow would have been sufficient).\n\nIn general, each setting value makes an assertion about the behavior of the function, without requiring the compiler to prove that this behavior is indeed true. These assertions are made for all world ages. It is thus advisable to limit the use of generic functions that may later be extended to invalidate the assumption (which would cause undefined behavior).\n\nThe following settings are supported.\n\n:consistent\n:effect_free\n:nothrow\n:terminates_globally\n:terminates_locally\n:notaskstate\n:inaccessiblememonly\n:noub\n:noub_if_noinbounds\n:nortcall\n:foldable\n:removable\n:total\n\nExtended help\n\n\n\n:consistent\n\nThe :consistent setting asserts that for egal (===) inputs:\n\nThe manner of termination (return value, exception, non-termination) will always be the same.\nIf the method returns, the results will always be egal.\n\nnote: Note\nThis in particular implies that the method must not return a freshly allocated mutable object. Multiple allocations of mutable objects (even with identical contents) are not egal.\n\nnote: Note\nThe :consistent-cy assertion is made with respect to a particular world range R. More formally, write fᵢ for the evaluation of f in world-age i, then this setting requires: i  R j  R x y x  y  fᵢ(x)  fⱼ(y)For @assume_effects, the range R is m.primary_world:m.deleted_world of the annotated or containing method.For ordinary code instances, R is ci.min_world:ci.max_world.A further implication is that :consistent functions may not make their return value dependent on the state of the heap or any other global state that is not constant over the given world age range.\n\nnote: Note\nThe :consistent-cy includes all legal rewrites performed by the optimizer. For example, floating-point fastmath operations are not considered :consistent, because the optimizer may rewrite them causing the output to not be :consistent, even for the same world age (e.g. because one ran in the interpreter, while the other was optimized).\n\nnote: Note\nIf :consistent functions terminate by throwing an exception, that exception itself is not required to meet the egality requirement specified above.\n\n\n\n:effect_free\n\nThe :effect_free setting asserts that the method is free of externally semantically visible side effects. The following is an incomplete list of externally semantically visible side effects:\n\nChanging the value of a global variable.\nMutating the heap (e.g. an array or mutable value), except as noted below\nChanging the method table (e.g. through calls to eval)\nFile/Network/etc. I/O\nTask switching\n\nHowever, the following are explicitly not semantically visible, even if they may be observable:\n\nMemory allocations (both mutable and immutable)\nElapsed time\nGarbage collection\nHeap mutations of objects whose lifetime does not exceed the method (i.e. were allocated in the method and do not escape).\nThe returned value (which is externally visible, but not a side effect)\n\nThe rule of thumb here is that an externally visible side effect is anything that would affect the execution of the remainder of the program if the function were not executed.\n\nnote: Note\nThe :effect_free assertion is made both for the method itself and any code that is executed by the method. Keep in mind that the assertion must be valid for all world ages and limit use of this assertion accordingly.\n\n\n\n:nothrow\n\nThe :nothrow settings asserts that this method does not throw an exception (i.e. will either always return a value or never return).\n\nnote: Note\nIt is permissible for :nothrow annotated methods to make use of exception handling internally as long as the exception is not rethrown out of the method itself.\n\nnote: Note\nIf the execution of a method may raise MethodErrors and similar exceptions, then the method is not considered as :nothrow. However, note that environment-dependent errors like StackOverflowError or InterruptException are not modeled by this effect and thus a method that may result in StackOverflowError does not necessarily need to be !:nothrow (although it should usually be !:terminates too).\n\n\n\n:terminates_globally\n\nThe :terminates_globally settings asserts that this method will eventually terminate (either normally or abnormally), i.e. does not loop indefinitely.\n\nnote: Note\nThis :terminates_globally assertion covers any other methods called by the annotated method.\n\nnote: Note\nThe compiler will consider this a strong indication that the method will terminate relatively quickly and may (if otherwise legal) call this method at compile time. I.e. it is a bad idea to annotate this setting on a method that technically, but not practically, terminates.\n\n\n\n:terminates_locally\n\nThe :terminates_locally setting is like :terminates_globally, except that it only applies to syntactic control flow within the annotated method. It is thus a much weaker (and thus safer) assertion that allows for the possibility of non-termination if the method calls some other method that does not terminate.\n\nnote: Note\n:terminates_globally implies :terminates_locally.\n\n\n\n:notaskstate\n\nThe :notaskstate setting asserts that the method does not use or modify the local task state (task local storage, RNG state, etc.) and may thus be safely moved between tasks without observable results.\n\nnote: Note\nThe implementation of exception handling makes use of state stored in the task object. However, this state is currently not considered to be within the scope of :notaskstate and is tracked separately using the :nothrow effect.\n\nnote: Note\nThe :notaskstate assertion concerns the state of the currently running task. If a reference to a Task object is obtained by some other means that does not consider which task is currently running, the :notaskstate effect need not be tainted. This is true, even if said task object happens to be === to the currently running task.\n\nnote: Note\nAccess to task state usually also results in the tainting of other effects, such as :effect_free (if task state is modified) or :consistent (if task state is used in the computation of the result). In particular, code that is not :notaskstate, but is :effect_free and :consistent may still be dead-code-eliminated and thus promoted to :total.\n\n\n\n:inaccessiblememonly\n\nThe :inaccessiblememonly setting asserts that the method does not access or modify externally accessible mutable memory. This means the method can access or modify mutable memory for newly allocated objects that is not accessible by other methods or top-level execution before return from the method, but it can not access or modify any mutable global state or mutable memory pointed to by its arguments.\n\nnote: Note\nBelow is an incomplete list of examples that invalidate this assumption:a global reference or getglobal call to access a mutable global variable\na global assignment or setglobal! call to perform assignment to a non-constant global variable\nsetfield! call that changes a field of a global mutable variable\n\nnote: Note\nThis :inaccessiblememonly assertion covers any other methods called by the annotated method.\n\n\n\n:noub\n\nThe :noub setting asserts that the method will not execute any undefined behavior (for any input). Note that undefined behavior may technically cause the method to violate any other effect assertions (such as :consistent or :effect_free) as well, but we do not model this, and they assume the absence of undefined behavior.\n\n\n\n:nortcall\n\nThe :nortcall setting asserts that the method does not call Core.Compiler.return_type, and that any other methods this method might call also do not call Core.Compiler.return_type.\n\nnote: Note\nTo be precise, this assertion can be used when a call to Core.Compiler.return_type is not made at runtime; that is, when the result of Core.Compiler.return_type is known exactly at compile time and the call is eliminated by the optimizer. However, since whether the result of Core.Compiler.return_type is folded at compile time depends heavily on the compiler's implementation, it is generally risky to assert this if the method in question uses Core.Compiler.return_type in any form.\n\n\n\n:foldable\n\nThis setting is a convenient shortcut for the set of effects that the compiler requires to be guaranteed to constant fold a call at compile time. It is currently equivalent to the following settings:\n\n:consistent\n:effect_free\n:terminates_globally\n:noub\n:nortcall\n\nnote: Note\nThis list in particular does not include :nothrow. The compiler will still attempt constant propagation and note any thrown error at compile time. Note however, that by the :consistent-cy requirements, any such annotated call must consistently throw given the same argument values.\n\nnote: Note\nAn explicit @inbounds annotation inside the function will also disable constant folding and not be overridden by :foldable.\n\n\n\n:removable\n\nThis setting is a convenient shortcut for the set of effects that the compiler requires to be guaranteed to delete a call whose result is unused at compile time. It is currently equivalent to the following settings:\n\n:effect_free\n:nothrow\n:terminates_globally\n\n\n\n:total\n\nThis setting is the maximum possible set of effects. It currently implies the following other settings:\n\n:consistent\n:effect_free\n:nothrow\n:terminates_globally\n:notaskstate\n:inaccessiblememonly\n:noub\n:nortcall\n\nwarning: Warning\n:total is a very strong assertion and will likely gain additional semantics in future versions of Julia (e.g. if additional effects are added and included in the definition of :total). As a result, it should be used with care. Whenever possible, prefer to use the minimum possible set of specific effect assertions required for a particular application. In cases where a large number of effect overrides apply to a set of functions, a custom macro is recommended over the use of :total.\n\n\n\nNegated effects\n\nEffect names may be prefixed by ! to indicate that the effect should be removed from an earlier meta effect. For example, :total !:nothrow indicates that while the call is generally total, it may however throw.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@deprecate","location":"base/base.html#Base.@deprecate","category":"macro","text":"@deprecate old new [export_old=true]\n\nDeprecate method old and specify the replacement call new, defining a new method old with the specified signature in the process.\n\nTo prevent old from being exported, set export_old to false.\n\nSee also Base.depwarn().\n\ncompat: Julia 1.5\nAs of Julia 1.5, functions defined by @deprecate do not print warning when julia is run without the --depwarn=yes flag set, as the default value of --depwarn option is no.  The warnings are printed from tests run by Pkg.test().\n\nExamples\n\njulia> @deprecate old_export(x) new(x)\nold_export (generic function with 1 method)\n\njulia> @deprecate old_public(x) new(x) false\nold_public (generic function with 1 method)\n\nCalls to @deprecate without explicit type-annotations will define deprecated methods accepting any number of positional and keyword arguments of type Any.\n\ncompat: Julia 1.9\nKeyword arguments are forwarded when there is no explicit type annotation as of Julia 1.9. For older versions, you can manually forward positional and keyword arguments by doing @deprecate old(args...; kwargs...) new(args...; kwargs...).\n\nTo restrict deprecation to a specific signature, annotate the arguments of old. For example,\n\njulia> new(x::Int) = x;\n\njulia> new(x::Float64) = 2x;\n\njulia> @deprecate old(x::Int) new(x);\n\njulia> methods(old)\n# 1 method for generic function \"old\" from Main:\n [1] old(x::Int64)\n     @ deprecated.jl:94\n\nwill define and deprecate a method old(x::Int) that mirrors new(x::Int) but will not define nor deprecate the method old(x::Float64).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.depwarn","location":"base/base.html#Base.depwarn","category":"function","text":"Base.depwarn(msg::String, funcsym::Symbol; force=false)\n\nPrint msg as a deprecation warning. The symbol funcsym should be the name of the calling function, which is used to ensure that the deprecation warning is only printed the first time for each call place. Set force=true to force the warning to always be shown, even if Julia was started with --depwarn=no (the default).\n\nSee also @deprecate.\n\nExamples\n\nfunction deprecated_func()\n    Base.depwarn(\"Don't use `deprecated_func()`!\", :deprecated_func)\n\n    1 + 1\nend\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Missing","location":"base/base.html#Base.Missing","category":"type","text":"Missing\n\nA type with no fields whose singleton instance missing is used to represent missing values.\n\nSee also: skipmissing, nonmissingtype, Nothing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.missing","location":"base/base.html#Base.missing","category":"constant","text":"missing\n\nThe singleton instance of type Missing representing a missing value.\n\nSee also: NaN, skipmissing, nonmissingtype.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.coalesce","location":"base/base.html#Base.coalesce","category":"function","text":"coalesce(x...)\n\nReturn the first value in the arguments which is not equal to missing, if any. Otherwise return missing.\n\nSee also skipmissing, something, @coalesce.\n\nExamples\n\njulia> coalesce(missing, 1)\n1\n\njulia> coalesce(1, missing)\n1\n\njulia> coalesce(nothing, 1)  # returns `nothing`\n\njulia> coalesce(missing, missing)\nmissing\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@coalesce","location":"base/base.html#Base.@coalesce","category":"macro","text":"@coalesce(x...)\n\nShort-circuiting version of coalesce.\n\nExamples\n\njulia> f(x) = (println(\"f($x)\"); missing);\n\njulia> a = 1;\n\njulia> a = @coalesce a f(2) f(3) error(\"`a` is still missing\")\n1\n\njulia> b = missing;\n\njulia> b = @coalesce b f(2) f(3) error(\"`b` is still missing\")\nf(2)\nf(3)\nERROR: `b` is still missing\n[...]\n\ncompat: Julia 1.7\nThis macro is available as of Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ismissing","location":"base/base.html#Base.ismissing","category":"function","text":"ismissing(x)\n\nIndicate whether x is missing.\n\nSee also: skipmissing, isnothing, isnan.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.skipmissing","location":"base/base.html#Base.skipmissing","category":"function","text":"skipmissing(itr)\n\nReturn an iterator over the elements in itr skipping missing values. The returned object can be indexed using indices of itr if the latter is indexable. Indices corresponding to missing values are not valid: they are skipped by keys and eachindex, and a MissingException is thrown when trying to use them.\n\nUse collect to obtain an Array containing the non-missing values in itr. Note that even if itr is a multidimensional array, the result will always be a Vector since it is not possible to remove missings while preserving dimensions of the input.\n\nSee also coalesce, ismissing, something.\n\nExamples\n\njulia> x = skipmissing([1, missing, 2])\nskipmissing(Union{Missing, Int64}[1, missing, 2])\n\njulia> sum(x)\n3\n\njulia> x[1]\n1\n\njulia> x[2]\nERROR: MissingException: the value at index (2,) is missing\n[...]\n\njulia> argmax(x)\n3\n\njulia> collect(keys(x))\n2-element Vector{Int64}:\n 1\n 3\n\njulia> collect(skipmissing([1, missing, 2]))\n2-element Vector{Int64}:\n 1\n 2\n\njulia> collect(skipmissing([1 missing; 2 missing]))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nonmissingtype","location":"base/base.html#Base.nonmissingtype","category":"function","text":"nonmissingtype(T::Type)\n\nIf T is a union of types containing Missing, return a new type with Missing removed.\n\nExamples\n\njulia> nonmissingtype(Union{Int64,Missing})\nInt64\n\njulia> nonmissingtype(Any)\nAny\n\ncompat: Julia 1.3\nThis function is exported as of Julia 1.3.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.run","location":"base/base.html#Base.run","category":"function","text":"run(command, args...; wait::Bool = true)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual). Throws an error if anything goes wrong, including the process exiting with a non-zero status (when wait is true).\n\nThe args... allow you to pass through file descriptors to the command, and are ordered like regular unix file descriptors (eg stdin, stdout, stderr, FD(3), FD(4)...).\n\nIf wait is false, the process runs asynchronously. You can later wait for it and check its exit status by calling success on the returned process object.\n\nWhen wait is false, the process' I/O streams are directed to devnull. When wait is true, I/O streams are shared with the parent process. Use pipeline to control I/O redirection.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.devnull","location":"base/base.html#Base.devnull","category":"constant","text":"devnull\n\nUsed in a stream redirect to discard all data written to it. Essentially equivalent to /dev/null on Unix or NUL on Windows. Usage:\n\nrun(pipeline(`cat test.txt`, devnull))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.success","location":"base/base.html#Base.success","category":"function","text":"success(command)\n\nRun a command object, constructed with backticks (see the Running External Programs section in the manual), and tell whether it was successful (exited with a code of 0). An exception is raised if the process cannot be started.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.process_running","location":"base/base.html#Base.process_running","category":"function","text":"process_running(p::Process)\n\nDetermine whether a process is currently running.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.process_exited","location":"base/base.html#Base.process_exited","category":"function","text":"process_exited(p::Process)\n\nDetermine whether a process has exited.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.kill","location":"base/base.html#Base.kill-Tuple{Base.Process, Integer}","category":"method","text":"kill(p::Process, signum=Base.SIGTERM)\n\nSend a signal to a process. The default is to terminate the process. Returns successfully if the process has already exited, but throws an error if killing the process failed for other reasons (e.g. insufficient permissions).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.set_process_title","location":"base/base.html#Base.Sys.set_process_title","category":"function","text":"Sys.set_process_title(title::AbstractString)\n\nSet the process title. No-op on some operating systems.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.get_process_title","location":"base/base.html#Base.Sys.get_process_title","category":"function","text":"Sys.get_process_title()\n\nGet the process title. On some systems, will always return an empty string.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ignorestatus","location":"base/base.html#Base.ignorestatus","category":"function","text":"ignorestatus(command)\n\nMark a command object so that running it will not throw an error if the result code is non-zero.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.detach","location":"base/base.html#Base.detach","category":"function","text":"detach(command)\n\nMark a command object so that it will be run in a new process group, allowing it to outlive the julia process, and not have Ctrl-C interrupts passed to it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Cmd","location":"base/base.html#Base.Cmd","category":"type","text":"Cmd(cmd::Cmd; ignorestatus, detach, windows_verbatim, windows_hide, env, dir)\nCmd(exec::Vector{String})\n\nConstruct a new Cmd object, representing an external program and arguments, from cmd, while changing the settings of the optional keyword arguments:\n\nignorestatus::Bool: If true (defaults to false), then the Cmd will not throw an error if the return code is nonzero.\ndetach::Bool: If true (defaults to false), then the Cmd will be run in a new process group, allowing it to outlive the julia process and not have Ctrl-C passed to it.\nwindows_verbatim::Bool: If true (defaults to false), then on Windows the Cmd will send a command-line string to the process with no quoting or escaping of arguments, even arguments containing spaces. (On Windows, arguments are sent to a program as a single \"command-line\" string, and programs are responsible for parsing it into arguments. By default, empty arguments and arguments with spaces or tabs are quoted with double quotes \" in the command line, and \\ or \" are preceded by backslashes. windows_verbatim=true is useful for launching programs that parse their command line in nonstandard ways.) Has no effect on non-Windows systems.\nwindows_hide::Bool: If true (defaults to false), then on Windows no new console window is displayed when the Cmd is executed. This has no effect if a console is already open or on non-Windows systems.\nenv: Set environment variables to use when running the Cmd. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", an array or tuple of \"var\"=>val pairs. In order to modify (rather than replace) the existing environment, initialize env with copy(ENV) and then set env[\"var\"]=val as desired.  To add to an environment block within a Cmd object without replacing all elements, use addenv() which will return a Cmd object with the updated environment.\ndir::AbstractString: Specify a working directory for the command (instead of the current directory).\n\nFor any keywords that are not specified, the current settings from cmd are used.\n\nNote that the Cmd(exec) constructor does not create a copy of exec. Any subsequent changes to exec will be reflected in the Cmd object.\n\nThe most common way to construct a Cmd object is with command literals (backticks), e.g.\n\n`ls -l`\n\nThis can then be passed to the Cmd constructor to modify its settings, e.g.\n\nCmd(`echo \"Hello world\"`, ignorestatus=true, detach=false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setenv","location":"base/base.html#Base.setenv","category":"function","text":"setenv(command::Cmd, env; dir)\n\nSet environment variables to use when running the given command. env is either a dictionary mapping strings to strings, an array of strings of the form \"var=val\", or zero or more \"var\"=>val pair arguments. In order to modify (rather than replace) the existing environment, create env through copy(ENV) and then setting env[\"var\"]=val as desired, or use addenv.\n\nThe dir keyword argument can be used to specify a working directory for the command. dir defaults to the currently set dir for command (which is the current working directory if not specified already).\n\nSee also Cmd, addenv, ENV, pwd.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.addenv","location":"base/base.html#Base.addenv","category":"function","text":"addenv(command::Cmd, env...; inherit::Bool = true)\n\nMerge new environment mappings into the given Cmd object, returning a new Cmd object. Duplicate keys are replaced.  If command does not contain any environment values set already, it inherits the current environment at time of addenv() call if inherit is true. Keys with value nothing are deleted from the env.\n\nSee also Cmd, setenv, ENV.\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.withenv","location":"base/base.html#Base.withenv","category":"function","text":"withenv(f, kv::Pair...)\n\nExecute f in an environment that is temporarily modified (not replaced as in setenv) by zero or more \"var\"=>val arguments kv. withenv is generally used via the withenv(kv...) do ... end syntax. A value of nothing can be used to temporarily unset an environment variable (if it is set). When withenv returns, the original environment has been restored.\n\nwarning: Warning\nChanging the environment is not thread-safe. For running external commands with a different environment from the parent process, prefer using addenv over withenv.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.shell_escape","location":"base/base.html#Base.shell_escape","category":"function","text":"shell_escape(args::Union{Cmd,AbstractString...}; special::AbstractString=\"\")\n\nThe unexported shell_escape function is the inverse of the unexported Base.shell_split() function: it takes a string or command object and escapes any special characters in such a way that calling Base.shell_split() on it would give back the array of words in the original command. The special keyword argument controls what characters in addition to whitespace, backslashes, quotes and dollar signs are considered to be special (default: none).\n\nExamples\n\njulia> Base.shell_escape(\"cat\", \"/foo/bar baz\", \"&&\", \"echo\", \"done\")\n\"cat '/foo/bar baz' && echo done\"\n\njulia> Base.shell_escape(\"echo\", \"this\", \"&&\", \"that\")\n\"echo this && that\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.shell_split","location":"base/base.html#Base.shell_split","category":"function","text":"shell_split(command::AbstractString)\n\nSplit a shell command string into its individual components.\n\nExamples\n\njulia> Base.shell_split(\"git commit -m 'Initial commit'\")\n4-element Vector{String}:\n \"git\"\n \"commit\"\n \"-m\"\n \"Initial commit\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.shell_escape_posixly","location":"base/base.html#Base.shell_escape_posixly","category":"function","text":"shell_escape_posixly(args::Union{Cmd,AbstractString...})\n\nThe unexported shell_escape_posixly function takes a string or command object and escapes any special characters in such a way that it is safe to pass it as an argument to a posix shell.\n\nSee also: Base.shell_escape()\n\nExamples\n\njulia> Base.shell_escape_posixly(\"cat\", \"/foo/bar baz\", \"&&\", \"echo\", \"done\")\n\"cat '/foo/bar baz' '&&' echo done\"\n\njulia> Base.shell_escape_posixly(\"echo\", \"this\", \"&&\", \"that\")\n\"echo this '&&' that\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.shell_escape_csh","location":"base/base.html#Base.shell_escape_csh","category":"function","text":"shell_escape_csh(args::Union{Cmd,AbstractString...})\nshell_escape_csh(io::IO, args::Union{Cmd,AbstractString...})\n\nThis function quotes any metacharacters in the string arguments such that the string returned can be inserted into a command-line for interpretation by the Unix C shell (csh, tcsh), where each string argument will form one word.\n\nIn contrast to a POSIX shell, csh does not support the use of the backslash as a general escape character in double-quoted strings. Therefore, this function wraps strings that might contain metacharacters in single quotes, except for parts that contain single quotes, which it wraps in double quotes instead. It switches between these types of quotes as needed. Linefeed characters are escaped with a backslash.\n\nThis function should also work for a POSIX shell, except if the input string contains a linefeed (\"\\n\") character.\n\nSee also: Base.shell_escape_posixly()\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.shell_escape_wincmd","location":"base/base.html#Base.shell_escape_wincmd","category":"function","text":"shell_escape_wincmd(s::AbstractString)\nshell_escape_wincmd(io::IO, s::AbstractString)\n\nThe unexported shell_escape_wincmd function escapes Windows cmd.exe shell meta characters. It escapes ()!^<>&| by placing a ^ in front. An @ is only escaped at the start of the string. Pairs of \" characters and the strings they enclose are passed through unescaped. Any remaining \" is escaped with ^ to ensure that the number of unescaped \" characters in the result remains even.\n\nSince cmd.exe substitutes variable references (like %USER%) before processing the escape characters ^ and \", this function makes no attempt to escape the percent sign (%), the presence of % in the input may cause severe breakage, depending on where the result is used.\n\nInput strings with ASCII control characters that cannot be escaped (NUL, CR, LF) will cause an ArgumentError exception.\n\nThe result is safe to pass as an argument to a command call being processed by CMD.exe /S /C \" ... \" (with surrounding double-quote pair) and will be received verbatim by the target application if the input does not contain % (else this function will fail with an ArgumentError). The presence of % in the input string may result in command injection vulnerabilities and may invalidate any claim of suitability of the output of this function for use as an argument to cmd (due to the ordering described above), so use caution when assembling a string from various sources.\n\nThis function may be useful in concert with the windows_verbatim flag to Cmd when constructing process pipelines.\n\nwincmd(c::String) =\n   run(Cmd(Cmd([\"cmd.exe\", \"/s /c \\\" $c \\\"\"]);\n           windows_verbatim=true))\nwincmd_echo(s::String) =\n   wincmd(\"echo \" * Base.shell_escape_wincmd(s))\nwincmd_echo(\"hello $(ENV[\"USERNAME\"]) & the \\\"whole\\\" world! (=^I^=)\")\n\nBut take note that if the input string s contains a %, the argument list and echo'ed text may get corrupted, resulting in arbitrary command execution. The argument can alternatively be passed as an environment variable, which avoids the problem with % and the need for the windows_verbatim flag:\n\ncmdargs = Base.shell_escape_wincmd(\"Passing args with %cmdargs% works 100%!\")\nrun(setenv(`cmd /C echo %cmdargs%`, \"cmdargs\" => cmdargs))\n\nwarning: Warning\nThe argument parsing done by CMD when calling batch files (either inside .bat files or as arguments to them) is not fully compatible with the output of this function. In particular, the processing of % is different.\n\nimportant: Important\nDue to a peculiar behavior of the CMD parser/interpreter, each command after a literal | character (indicating a command pipeline) must have shell_escape_wincmd applied twice since it will be parsed twice by CMD. This implies ENV variables would also be expanded twice! For example:to_print = \"All for 1 & 1 for all!\"\nto_print_esc = Base.shell_escape_wincmd(Base.shell_escape_wincmd(to_print))\nrun(Cmd(Cmd([\"cmd\", \"/S /C \\\" break | echo $(to_print_esc) \\\"\"]), windows_verbatim=true))\n\nWith an I/O stream parameter io, the result will be written there, rather than returned as a string.\n\nSee also Base.escape_microsoft_c_args(), Base.shell_escape_posixly().\n\nExamples\n\njulia> Base.shell_escape_wincmd(\"a^\\\"^o\\\"^u\\\"\")\n\"a^^\\\"^o\\\"^^u^\\\"\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.escape_microsoft_c_args","location":"base/base.html#Base.escape_microsoft_c_args","category":"function","text":"escape_microsoft_c_args(args::Union{Cmd,AbstractString...})\nescape_microsoft_c_args(io::IO, args::Union{Cmd,AbstractString...})\n\nConvert a collection of string arguments into a string that can be passed to many Windows command-line applications.\n\nMicrosoft Windows passes the entire command line as a single string to the application (unlike POSIX systems, where the shell splits the command line into a list of arguments). Many Windows API applications (including julia.exe), use the conventions of the Microsoft C/C++ runtime to split that command line into a list of strings.\n\nThis function implements an inverse for a parser compatible with these rules. It joins command-line arguments to be passed to a Windows C/C++/Julia application into a command line, escaping or quoting the meta characters space, TAB, double quote and backslash where needed.\n\nSee also Base.shell_escape_wincmd(), Base.escape_raw_string().\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.setcpuaffinity","location":"base/base.html#Base.setcpuaffinity","category":"function","text":"setcpuaffinity(original_command::Cmd, cpus) -> command::Cmd\n\nSet the CPU affinity of the command by a list of CPU IDs (1-based) cpus.  Passing cpus = nothing means to unset the CPU affinity if the original_command has any.\n\nThis function is supported only in Linux and Windows.  It is not supported in macOS because libuv does not support affinity setting.\n\ncompat: Julia 1.8\nThis function requires at least Julia 1.8.\n\nExamples\n\nIn Linux, the taskset command line program can be used to see how setcpuaffinity works.\n\njulia> run(setcpuaffinity(`sh -c 'taskset -p $$'`, [1, 2, 5]));\npid 2273's current affinity mask: 13\n\nNote that the mask value 13 reflects that the first, second, and the fifth bits (counting from the least significant position) are turned on:\n\njulia> 0b010011\n0x13\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pipeline","location":"base/base.html#Base.pipeline-Tuple{Any, Any, Any, Vararg{Any}}","category":"method","text":"pipeline(from, to, ...)\n\nCreate a pipeline from a data source to a destination. The source and destination can be commands, I/O streams, strings, or results of other pipeline calls. At least one argument must be a command. Strings refer to filenames. When called with more than two arguments, they are chained together from left to right. For example, pipeline(a,b,c) is equivalent to pipeline(pipeline(a,b),c). This provides a more concise way to specify multi-stage pipelines.\n\nExamples:\n\nrun(pipeline(`ls`, `grep xyz`))\nrun(pipeline(`ls`, \"out.txt\"))\nrun(pipeline(\"out.txt\", `grep xyz`))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pipeline","location":"base/base.html#Base.pipeline-Tuple{Base.AbstractCmd}","category":"method","text":"pipeline(command; stdin, stdout, stderr, append=false)\n\nRedirect I/O to or from the given command. Keyword arguments specify which of the command's streams should be redirected. append controls whether file output appends to the file. This is a more general version of the 2-argument pipeline function. pipeline(from, to) is equivalent to pipeline(from, stdout=to) when from is a command, and to pipeline(to, stdin=from) when from is another kind of data source.\n\nExamples:\n\nrun(pipeline(`dothings`, stdout=\"out.txt\", stderr=\"errs.txt\"))\nrun(pipeline(`update`, stdout=\"log.txt\", append=true))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.gethostname","location":"base/base.html#Base.Libc.gethostname","category":"function","text":"gethostname() -> String\n\nGet the local machine's host name.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.getpid","location":"base/base.html#Base.Libc.getpid","category":"function","text":"getpid() -> Int32\n\nGet Julia's process ID.\n\n\n\n\n\ngetpid(process) -> Int32\n\nGet the child process ID, if it still exists.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Libc.time","location":"base/base.html#Base.Libc.time-Tuple{}","category":"method","text":"time() -> Float64\n\nGet the system time in seconds since the epoch, with fairly high (typically, microsecond) resolution.\n\nSee also time_ns.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.time_ns","location":"base/base.html#Base.time_ns","category":"function","text":"time_ns() -> UInt64\n\nGet the time in nanoseconds relative to some machine-specific arbitrary time in the past. The primary use is for measuring elapsed times during program execution. The return value is guaranteed to be monotonic (mod 2⁶⁴) while the system is running, and is unaffected by clock drift or changes to local calendar time, but it may change arbitrarily across system reboots or suspensions.\n\n(Although the returned time is always in nanoseconds, the timing resolution is platform-dependent.)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@time","location":"base/base.html#Base.@time","category":"macro","text":"@time expr\n@time \"description\" expr\n\nA macro to execute an expression, printing the time it took to execute, the number of allocations, and the total number of bytes its execution caused to be allocated, before returning the value of the expression. Any time spent garbage collecting (gc), compiling new code, or recompiling invalidated code is shown as a percentage. Any lock conflicts where a ReentrantLock had to wait are shown as a count.\n\nOptionally provide a description string to print before the time report.\n\nIn some cases the system will look inside the @time expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @time @eval ....\n\nSee also @showtime, @timev, @timed, @elapsed, @allocated, and @allocations.\n\nnote: Note\nFor more serious benchmarking, consider the @btime macro from the BenchmarkTools.jl package which among other things evaluates the function multiple times in order to reduce noise.\n\ncompat: Julia 1.8\nThe option to add a description was introduced in Julia 1.8.Recompilation time being shown separately from compilation time was introduced in Julia 1.8\n\ncompat: Julia 1.11\nThe reporting of any lock conflicts was added in Julia 1.11.\n\njulia> x = rand(10,10);\n\njulia> @time x * x;\n  0.606588 seconds (2.19 M allocations: 116.555 MiB, 3.75% gc time, 99.94% compilation time)\n\njulia> @time x * x;\n  0.000009 seconds (1 allocation: 896 bytes)\n\njulia> @time begin\n           sleep(0.3)\n           1+1\n       end\n  0.301395 seconds (8 allocations: 336 bytes)\n2\n\njulia> @time \"A one second sleep\" sleep(1)\nA one second sleep: 1.005750 seconds (5 allocations: 144 bytes)\n\njulia> for loop in 1:3\n            @time loop sleep(1)\n        end\n1: 1.006760 seconds (5 allocations: 144 bytes)\n2: 1.001263 seconds (5 allocations: 144 bytes)\n3: 1.003676 seconds (5 allocations: 144 bytes)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@showtime","location":"base/base.html#Base.@showtime","category":"macro","text":"@showtime expr\n\nLike @time but also prints the expression being evaluated for reference.\n\ncompat: Julia 1.8\nThis macro was added in Julia 1.8.\n\nSee also @time.\n\njulia> @showtime sleep(1)\nsleep(1): 1.002164 seconds (4 allocations: 128 bytes)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@timev","location":"base/base.html#Base.@timev","category":"macro","text":"@timev expr\n@timev \"description\" expr\n\nThis is a verbose version of the @time macro. It first prints the same information as @time, then any non-zero memory allocation counters, and then returns the value of the expression.\n\nOptionally provide a description string to print before the time report.\n\ncompat: Julia 1.8\nThe option to add a description was introduced in Julia 1.8.\n\nSee also @time, @timed, @elapsed, @allocated, and @allocations.\n\njulia> x = rand(10,10);\n\njulia> @timev x * x;\n  0.546770 seconds (2.20 M allocations: 116.632 MiB, 4.23% gc time, 99.94% compilation time)\nelapsed time (ns): 546769547\ngc time (ns):      23115606\nbytes allocated:   122297811\npool allocs:       2197930\nnon-pool GC allocs:1327\nmalloc() calls:    36\nrealloc() calls:   5\nGC pauses:         3\n\njulia> @timev x * x;\n  0.000010 seconds (1 allocation: 896 bytes)\nelapsed time (ns): 9848\nbytes allocated:   896\npool allocs:       1\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@timed","location":"base/base.html#Base.@timed","category":"macro","text":"@timed\n\nA macro to execute an expression, and return the value of the expression, elapsed time in seconds, total bytes allocated, garbage collection time, an object with various memory allocation counters, compilation time in seconds, and recompilation time in seconds. Any lock conflicts where a ReentrantLock had to wait are shown as a count.\n\nIn some cases the system will look inside the @timed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @timed @eval ....\n\nSee also @time, @timev, @elapsed, @allocated, @allocations, and @lock_conflicts.\n\njulia> stats = @timed rand(10^6);\n\njulia> stats.time\n0.006634834\n\njulia> stats.bytes\n8000256\n\njulia> stats.gctime\n0.0055765\n\njulia> propertynames(stats.gcstats)\n(:allocd, :malloc, :realloc, :poolalloc, :bigalloc, :freecall, :total_time, :pause, :full_sweep)\n\njulia> stats.gcstats.total_time\n5576500\n\njulia> stats.compile_time\n0.0\n\njulia> stats.recompile_time\n0.0\n\n\ncompat: Julia 1.5\nThe return type of this macro was changed from Tuple to NamedTuple in Julia 1.5.\n\ncompat: Julia 1.11\nThe lock_conflicts, compile_time, and recompile_time fields were added in Julia 1.11.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@elapsed","location":"base/base.html#Base.@elapsed","category":"macro","text":"@elapsed\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the number of seconds it took to execute as a floating-point number.\n\nIn some cases the system will look inside the @elapsed expression and compile some of the called code before execution of the top-level expression begins. When that happens, some compilation time will not be counted. To include this time you can run @elapsed @eval ....\n\nSee also @time, @timev, @timed, @allocated, and @allocations.\n\njulia> @elapsed sleep(0.3)\n0.301391426\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@allocated","location":"base/base.html#Base.@allocated","category":"macro","text":"@allocated\n\nA macro to evaluate an expression, discarding the resulting value, instead returning the total number of bytes allocated during evaluation of the expression.\n\nIf the expression is a function call, an effort is made to measure only allocations from the argument expressions and during the function, excluding any overhead from calling it and not performing constant propagation with the provided argument values. If you want to include those effects, i.e. measuring the call site as well, use the syntax @allocated (()->f(1))().\n\nIt is recommended to measure function calls with only simple argument expressions, e.g. x = []; @allocated f(x) instead of @allocated f([]) to clarify that only f is being measured.\n\nFor more complex expressions, the code is simply run in place and therefore may see allocations due to the surrounding context. For example it is possible for @allocated f(1) and @allocated x = f(1) to give different results.\n\nSee also @allocations, @time, @timev, @timed, and @elapsed.\n\njulia> @allocated rand(10^6)\n8000080\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@allocations","location":"base/base.html#Base.@allocations","category":"macro","text":"@allocations\n\nA macro to evaluate an expression, discard the resulting value, and instead return the total number of allocations during evaluation of the expression.\n\nSee also @allocated, @time, @timev, @timed, and @elapsed.\n\njulia> @allocations rand(10^6)\n2\n\ncompat: Julia 1.9\nThis macro was added in Julia 1.9.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@lock_conflicts","location":"base/base.html#Base.@lock_conflicts","category":"macro","text":"@lock_conflicts\n\nA macro to evaluate an expression, discard the resulting value, and instead return the total number of lock conflicts during evaluation, where a lock attempt on a ReentrantLock resulted in a wait because the lock was already held.\n\nSee also @time, @timev and @timed.\n\njulia> @lock_conflicts begin\n    l = ReentrantLock()\n    Threads.@threads for i in 1:Threads.nthreads()\n        lock(l) do\n        sleep(1)\n        end\n    end\nend\n5\n\ncompat: Julia 1.11\nThis macro was added in Julia 1.11.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.EnvDict","location":"base/base.html#Base.EnvDict","category":"type","text":"EnvDict() -> EnvDict\n\nA singleton of this type provides a hash table interface to environment variables.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ENV","location":"base/base.html#Base.ENV","category":"constant","text":"ENV\n\nReference to the singleton EnvDict, providing a dictionary interface to system environment variables.\n\n(On Windows, system environment variables are case-insensitive, and ENV correspondingly converts all keys to uppercase for display, iteration, and copying. Portable code should not rely on the ability to distinguish variables by case, and should beware that setting an ostensibly lowercase variable may result in an uppercase ENV key.)\n\nwarning: Warning\nMutating the environment is not thread-safe.\n\nExamples\n\njulia> ENV\nBase.EnvDict with \"50\" entries:\n  \"SECURITYSESSIONID\"            => \"123\"\n  \"USER\"                         => \"username\"\n  \"MallocNanoZone\"               => \"0\"\n  ⋮                              => ⋮\n\njulia> ENV[\"JULIA_EDITOR\"] = \"vim\"\n\"vim\"\n\njulia> ENV[\"JULIA_EDITOR\"]\n\"vim\"\n\nSee also: withenv, addenv.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.STDLIB","location":"base/base.html#Base.Sys.STDLIB","category":"constant","text":"Sys.STDLIB::String\n\nA string containing the full path to the directory containing the stdlib packages.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isunix","location":"base/base.html#Base.Sys.isunix","category":"function","text":"Sys.isunix([os])\n\nPredicate for testing if the OS provides a Unix-like interface. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isapple","location":"base/base.html#Base.Sys.isapple","category":"function","text":"Sys.isapple([os])\n\nPredicate for testing if the OS is a derivative of Apple Macintosh OS X or Darwin. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.islinux","location":"base/base.html#Base.Sys.islinux","category":"function","text":"Sys.islinux([os])\n\nPredicate for testing if the OS is a derivative of Linux. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isbsd","location":"base/base.html#Base.Sys.isbsd","category":"function","text":"Sys.isbsd([os])\n\nPredicate for testing if the OS is a derivative of BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nThe Darwin kernel descends from BSD, which means that Sys.isbsd() is true on macOS systems. To exclude macOS from a predicate, use Sys.isbsd() && !Sys.isapple().\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isfreebsd","location":"base/base.html#Base.Sys.isfreebsd","category":"function","text":"Sys.isfreebsd([os])\n\nPredicate for testing if the OS is a derivative of FreeBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on FreeBSD but also on other BSD-based systems. Sys.isfreebsd() refers only to FreeBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isopenbsd","location":"base/base.html#Base.Sys.isopenbsd","category":"function","text":"Sys.isopenbsd([os])\n\nPredicate for testing if the OS is a derivative of OpenBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on OpenBSD but also on other BSD-based systems. Sys.isopenbsd() refers only to OpenBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isnetbsd","location":"base/base.html#Base.Sys.isnetbsd","category":"function","text":"Sys.isnetbsd([os])\n\nPredicate for testing if the OS is a derivative of NetBSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on NetBSD but also on other BSD-based systems. Sys.isnetbsd() refers only to NetBSD.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isdragonfly","location":"base/base.html#Base.Sys.isdragonfly","category":"function","text":"Sys.isdragonfly([os])\n\nPredicate for testing if the OS is a derivative of DragonFly BSD. See documentation in Handling Operating System Variation.\n\nnote: Note\nNot to be confused with Sys.isbsd(), which is true on DragonFly but also on other BSD-based systems. Sys.isdragonfly() refers only to DragonFly.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.iswindows","location":"base/base.html#Base.Sys.iswindows","category":"function","text":"Sys.iswindows([os])\n\nPredicate for testing if the OS is a derivative of Microsoft Windows NT. See documentation in Handling Operating System Variation.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.windows_version","location":"base/base.html#Base.Sys.windows_version","category":"function","text":"Sys.windows_version()\n\nReturn the version number for the Windows NT Kernel as a VersionNumber, i.e. v\"major.minor.build\", or v\"0.0.0\" if this is not running on Windows.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.free_memory","location":"base/base.html#Base.Sys.free_memory","category":"function","text":"Sys.free_memory()\n\nGet the total free memory in RAM in bytes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.total_memory","location":"base/base.html#Base.Sys.total_memory","category":"function","text":"Sys.total_memory()\n\nGet the total memory in RAM (including that which is currently used) in bytes. This amount may be constrained, e.g., by Linux control groups. For the unconstrained amount, see Sys.total_physical_memory().\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.free_physical_memory","location":"base/base.html#Base.Sys.free_physical_memory","category":"function","text":"Sys.free_physical_memory()\n\nGet the free memory of the system in bytes. The entire amount may not be available to the current process; use Sys.free_memory() for the actually available amount.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.total_physical_memory","location":"base/base.html#Base.Sys.total_physical_memory","category":"function","text":"Sys.total_physical_memory()\n\nGet the total memory in RAM (including that which is currently used) in bytes. The entire amount may not be available to the current process; see Sys.total_memory().\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.uptime","location":"base/base.html#Base.Sys.uptime","category":"function","text":"Sys.uptime()\n\nGets the current system uptime in seconds.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isjsvm","location":"base/base.html#Base.Sys.isjsvm","category":"function","text":"Sys.isjsvm([os])\n\nPredicate for testing if Julia is running in a JavaScript VM (JSVM), including e.g. a WebAssembly JavaScript embedding in a web browser.\n\ncompat: Julia 1.2\nThis function requires at least Julia 1.2.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.loadavg","location":"base/base.html#Base.Sys.loadavg","category":"function","text":"Sys.loadavg()\n\nGet the load average. See: https://en.wikipedia.org/wiki/Load_(computing).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isexecutable","location":"base/base.html#Base.Sys.isexecutable","category":"function","text":"isexecutable(path::String)\n\nReturn true if the given path has executable permissions.\n\nnote: Note\nThis permission may change before the user executes path, so it is recommended to execute the file and handle the error if that fails, rather than calling isexecutable first.\n\nnote: Note\nPrior to Julia 1.6, this did not correctly interrogate filesystem ACLs on Windows, therefore it would return true for any file.  From Julia 1.6 on, it correctly determines whether the file is marked as executable or not.\n\nSee also ispath, isreadable, iswritable.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.isreadable","location":"base/base.html#Base.Sys.isreadable","category":"function","text":"isreadable(path::String)\n\nReturn true if the access permissions for the given path permitted reading by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling isreadable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, iswritable.\n\n\n\n\n\nisreadable(io) -> Bool\n\nReturn false if the specified IO object is not readable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           isreadable(io)\n       end\nfalse\n\njulia> open(\"myfile.txt\", \"r\") do io\n           isreadable(io)\n       end\ntrue\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.iswritable","location":"base/base.html#Base.Sys.iswritable","category":"function","text":"iswritable(path::String)\n\nReturn true if the access permissions for the given path permitted writing by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling iswritable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, isreadable.\n\n\n\n\n\niswritable(io) -> Bool\n\nReturn false if the specified IO object is not writable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           iswritable(io)\n       end\ntrue\n\njulia> open(\"myfile.txt\", \"r\") do io\n           iswritable(io)\n       end\nfalse\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Sys.username","location":"base/base.html#Base.Sys.username","category":"function","text":"Sys.username() -> String\n\nReturn the username for the current user. If the username cannot be determined or is empty, this function throws an error.\n\nTo retrieve a username that is overridable via an environment variable, e.g., USER, consider using\n\nuser = get(Sys.username, ENV, \"USER\")\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also homedir.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@static","location":"base/base.html#Base.@static","category":"macro","text":"@static\n\nPartially evaluate an expression at macro expansion time.\n\nThis is useful in cases where a construct would be invalid in some cases, such as a ccall to an os-dependent function, or macros defined in packages that are not imported.\n\n@static requires a conditional. The conditional can be in an if statement, a ternary operator, or &&||. The conditional is evaluated by recursively expanding macros, lowering and executing the resulting expressions. Then, the matching branch (if any) is returned. All the other branches of the conditional are deleted before they are macro-expanded (and lowered or executed).\n\nExample\n\nSuppose we want to parse an expression expr that is valid only on macOS. We could solve this problem using @static with @static if Sys.isapple() expr end. In case we had expr_apple for macOS and expr_others for the other operating systems, the solution with @static would be @static Sys.isapple() ? expr_apple : expr_others.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.VersionNumber","location":"base/base.html#Base.VersionNumber","category":"type","text":"VersionNumber\n\nVersion number type which follows the specifications of semantic versioning (semver), composed of major, minor and patch numeric values, followed by pre-release and build alphanumeric annotations.\n\nVersionNumber objects can be compared with all of the standard comparison operators (==, <, <=, etc.), with the result following semver v2.0.0-rc.2 rules.\n\nVersionNumber has the following public fields:\n\nv.major::Integer\nv.minor::Integer\nv.patch::Integer\nv.prerelease::Tuple{Vararg{Union{Integer, AbstractString}}}\nv.build::Tuple{Vararg{Union{Integer, AbstractString}}}\n\nSee also @v_str to efficiently construct VersionNumber objects from semver-format literal strings, VERSION for the VersionNumber of Julia itself, and Version Number Literals in the manual.\n\nExamples\n\njulia> a = VersionNumber(1, 2, 3)\nv\"1.2.3\"\n\njulia> a >= v\"1.2\"\ntrue\n\njulia> b = VersionNumber(\"2.0.1-rc1\")\nv\"2.0.1-rc1\"\n\njulia> b >= v\"2.0.1\"\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@v_str","location":"base/base.html#Base.@v_str","category":"macro","text":"@v_str\n\nString macro used to parse a string to a VersionNumber.\n\nExamples\n\njulia> v\"1.2.3\"\nv\"1.2.3\"\n\njulia> v\"2.0.1-rc1\"\nv\"2.0.1-rc1\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.error","location":"base/base.html#Base.error","category":"function","text":"error(msg...)\n\nRaise an ErrorException with a message constructed by string(msg...).\n\n\n\n\n\nerror(message::AbstractString)\n\nRaise an ErrorException with the given message.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.throw","location":"base/base.html#Core.throw","category":"function","text":"throw(e)\n\nThrow an object as an exception.\n\nSee also: rethrow, error.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.rethrow","location":"base/base.html#Base.rethrow","category":"function","text":"rethrow()\n\nRethrow the current exception from within a catch block. The rethrown exception will continue propagation as if it had not been caught.\n\nnote: Note\nThe alternative form rethrow(e) allows you to associate an alternative exception object e with the current backtrace. However this misrepresents the program state at the time of the error so you're encouraged to instead throw a new exception using throw(e). In Julia 1.1 and above, using throw(e) will preserve the root cause exception on the stack, as described in current_exceptions.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.backtrace","location":"base/base.html#Base.backtrace","category":"function","text":"backtrace()\n\nGet a backtrace object for the current program point.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.catch_backtrace","location":"base/base.html#Base.catch_backtrace","category":"function","text":"catch_backtrace()\n\nGet the backtrace of the current exception, for use within catch blocks.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.current_exceptions","location":"base/base.html#Base.current_exceptions","category":"function","text":"current_exceptions(task::Task=current_task(); [backtrace::Bool=true])\n\nGet the stack of exceptions currently being handled. For nested catch blocks there may be more than one current exception in which case the most recently thrown exception is last in the stack. The stack is returned as an ExceptionStack which is an AbstractVector of named tuples (exception,backtrace). If backtrace is false, the backtrace in each pair will be set to nothing.\n\nExplicitly passing task will return the current exception stack on an arbitrary task. This is useful for inspecting tasks which have failed due to uncaught exceptions.\n\ncompat: Julia 1.7\nThis function went by the experimental name catch_stack() in Julia 1.1–1.6, and had a plain Vector-of-tuples as a return type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@assert","location":"base/base.html#Base.@assert","category":"macro","text":"@assert cond [text]\n\nThrow an AssertionError if cond is false. This is the preferred syntax for writing assertions, which are conditions that are assumed to be true, but that the user might decide to check anyways, as an aid to debugging if they fail. The optional message text is displayed upon assertion failure.\n\nwarning: Warning\nAn assert might be disabled at some optimization levels. Assert should therefore only be used as a debugging tool and not used for authentication verification (e.g., verifying passwords or checking array bounds). The code must not rely on the side effects of running cond for the correct behavior of a function.\n\nExamples\n\njulia> @assert iseven(3) \"3 is an odd number!\"\nERROR: AssertionError: 3 is an odd number!\n\njulia> @assert isodd(3) \"What even are numbers?\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Experimental.register_error_hint","location":"base/base.html#Base.Experimental.register_error_hint","category":"function","text":"Experimental.register_error_hint(handler, exceptiontype)\n\nRegister a \"hinting\" function handler(io, exception) that can suggest potential ways for users to circumvent errors.  handler should examine exception to see whether the conditions appropriate for a hint are met, and if so generate output to io. Packages should call register_error_hint from within their __init__ function.\n\nFor specific exception types, handler is required to accept additional arguments:\n\nMethodError: provide handler(io, exc::MethodError, argtypes, kwargs), which splits the combined arguments into positional and keyword arguments.\n\nWhen issuing a hint, the output should typically start with \\n.\n\nIf you define custom exception types, your showerror method can support hints by calling Experimental.show_error_hints.\n\nExamples\n\njulia> module Hinter\n\n       only_int(x::Int)      = 1\n       any_number(x::Number) = 2\n\n       function __init__()\n           Base.Experimental.register_error_hint(MethodError) do io, exc, argtypes, kwargs\n               if exc.f == only_int\n                    # Color is not necessary, this is just to show it's possible.\n                    print(io, \"\\nDid you mean to call \")\n                    printstyled(io, \"`any_number`?\", color=:cyan)\n               end\n           end\n       end\n\n       end\n\nThen if you call Hinter.only_int on something that isn't an Int (thereby triggering a MethodError), it issues the hint:\n\njulia> Hinter.only_int(1.0)\nERROR: MethodError: no method matching only_int(::Float64)\nThe function `only_int` exists, but no method is defined for this combination of argument types.\nDid you mean to call `any_number`?\nClosest candidates are:\n    ...\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice. To insulate yourself against changes, consider putting any registrations inside an if isdefined(Base.Experimental, :register_error_hint) ... end block.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Experimental.show_error_hints","location":"base/base.html#Base.Experimental.show_error_hints","category":"function","text":"Experimental.show_error_hints(io, ex, args...)\n\nInvoke all handlers from Experimental.register_error_hint for the particular exception type typeof(ex). args must contain any other arguments expected by the handler for that type.\n\ncompat: Julia 1.5\nCustom error hints are available as of Julia 1.5.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ArgumentError","location":"base/base.html#Core.ArgumentError","category":"type","text":"ArgumentError(msg)\n\nThe arguments passed to a function are invalid. msg is a descriptive error message.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.AssertionError","location":"base/base.html#Core.AssertionError","category":"type","text":"AssertionError([msg])\n\nThe asserted condition did not evaluate to true. Optional argument msg is a descriptive error string.\n\nExamples\n\njulia> @assert false \"this is not true\"\nERROR: AssertionError: this is not true\n\nAssertionError is usually thrown from @assert.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.BoundsError","location":"base/base.html#Core.BoundsError","category":"type","text":"BoundsError([a],[i])\n\nAn indexing operation into an array, a, tried to access an out-of-bounds element at index i.\n\nExamples\n\njulia> A = fill(1.0, 7);\n\njulia> A[8]\nERROR: BoundsError: attempt to access 7-element Vector{Float64} at index [8]\n\n\njulia> B = fill(1.0, (2,3));\n\njulia> B[2, 4]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [2, 4]\n\n\njulia> B[9]\nERROR: BoundsError: attempt to access 2×3 Matrix{Float64} at index [9]\n\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.CompositeException","location":"base/base.html#Base.CompositeException","category":"type","text":"CompositeException\n\nWrap a Vector of exceptions thrown by a Task (e.g. generated from a remote worker over a channel or an asynchronously executing local I/O write or a remote worker under pmap) with information about the series of exceptions. For example, if a group of workers are executing several tasks, and multiple workers fail, the resulting CompositeException will contain a \"bundle\" of information from each worker indicating where and why the exception(s) occurred.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.DimensionMismatch","location":"base/base.html#Base.DimensionMismatch","category":"type","text":"DimensionMismatch([msg])\n\nThe objects called do not have matching dimensionality. Optional argument msg is a descriptive error string.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DivideError","location":"base/base.html#Core.DivideError","category":"type","text":"DivideError()\n\nInteger division was attempted with a denominator value of 0.\n\nExamples\n\njulia> 2/0\nInf\n\njulia> div(2, 0)\nERROR: DivideError: integer division error\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.DomainError","location":"base/base.html#Core.DomainError","category":"type","text":"DomainError(val)\nDomainError(val, msg)\n\nThe argument val to a function or constructor is outside the valid domain.\n\nExamples\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.EOFError","location":"base/base.html#Base.EOFError","category":"type","text":"EOFError()\n\nNo more data was available to read from a file or stream.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ErrorException","location":"base/base.html#Core.ErrorException","category":"type","text":"ErrorException(msg)\n\nGeneric error type. The error message, in the .msg field, may provide more specific details.\n\nExamples\n\njulia> ex = ErrorException(\"I've done a bad thing\");\n\njulia> ex.msg\n\"I've done a bad thing\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.FieldError","location":"base/base.html#Core.FieldError","category":"type","text":"FieldError(type::DataType, field::Symbol)\n\nAn operation tried to access invalid field on an object of type.\n\ncompat: Julia 1.12\nPrior to Julia 1.12, invalid field access threw an ErrorException\n\nSee getfield\n\nExamples\n\njulia> struct AB\n          a::Float32\n          b::Float64\n       end\n\njulia> ab = AB(1, 3)\nAB(1.0f0, 3.0)\n\njulia> ab.c # field `c` doesn't exist\nERROR: FieldError: type AB has no field `c`, available fields: `a`, `b`\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InexactError","location":"base/base.html#Core.InexactError","category":"type","text":"InexactError(name::Symbol, T, val)\n\nCannot exactly convert val to type T in a method of function name.\n\nExamples\n\njulia> convert(Float64, 1+2im)\nERROR: InexactError: Float64(1 + 2im)\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InterruptException","location":"base/base.html#Core.InterruptException","category":"type","text":"InterruptException()\n\nThe process was stopped by a terminal interrupt (CTRL+C).\n\nNote that, in Julia script started without -i (interactive) option, InterruptException is not thrown by default.  Calling Base.exit_on_sigint(false) in the script can recover the behavior of the REPL.  Alternatively, a Julia script can be started with\n\njulia -e \"include(popfirst!(ARGS))\" script.jl\n\nto let InterruptException be thrown by CTRL+C during the execution.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.KeyError","location":"base/base.html#Base.KeyError","category":"type","text":"KeyError(key)\n\nAn indexing operation into an AbstractDict (Dict) or Set like object tried to access or delete a non-existent element.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.LoadError","location":"base/base.html#Core.LoadError","category":"type","text":"LoadError(file::AbstractString, line::Int, error)\n\nAn error occurred while includeing, requireing, or using a file. The error specifics should be available in the .error field.\n\ncompat: Julia 1.7\nLoadErrors are no longer emitted by @macroexpand, @macroexpand1, and macroexpand as of Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.MethodError","location":"base/base.html#Core.MethodError","category":"type","text":"MethodError(f, args)\n\nA method with the required type signature does not exist in the given generic function. Alternatively, there is no unique most-specific method.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.MissingException","location":"base/base.html#Base.MissingException","category":"type","text":"MissingException(msg)\n\nException thrown when a missing value is encountered in a situation where it is not supported. The error message, in the msg field may provide more specific details.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.OutOfMemoryError","location":"base/base.html#Core.OutOfMemoryError","category":"type","text":"OutOfMemoryError()\n\nAn operation allocated too much memory for either the system or the garbage collector to handle properly.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.ReadOnlyMemoryError","location":"base/base.html#Core.ReadOnlyMemoryError","category":"type","text":"ReadOnlyMemoryError()\n\nAn operation tried to write to memory that is read-only.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.OverflowError","location":"base/base.html#Core.OverflowError","category":"type","text":"OverflowError(msg)\n\nThe result of an expression is too large for the specified type and will cause a wraparound.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ProcessFailedException","location":"base/base.html#Base.ProcessFailedException","category":"type","text":"ProcessFailedException\n\nIndicates problematic exit status of a process. When running commands or pipelines, this is thrown to indicate a nonzero exit code was returned (i.e. that the invoked process failed).\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.TaskFailedException","location":"base/base.html#Base.TaskFailedException","category":"type","text":"TaskFailedException\n\nThis exception is thrown by a wait(t) call when task t fails. TaskFailedException wraps the failed task t.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.StackOverflowError","location":"base/base.html#Core.StackOverflowError","category":"type","text":"StackOverflowError()\n\nThe function call grew beyond the size of the call stack. This usually happens when a call recurses infinitely.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.SystemError","location":"base/base.html#Base.SystemError","category":"type","text":"SystemError(prefix::AbstractString, [errno::Int32])\n\nA system call failed with an error code (in the errno global variable).\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.TypeError","location":"base/base.html#Core.TypeError","category":"type","text":"TypeError(func::Symbol, context::AbstractString, expected::Type, got)\n\nA type assertion failure, or calling an intrinsic function with an incorrect argument type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefKeywordError","location":"base/base.html#Core.UndefKeywordError","category":"type","text":"UndefKeywordError(var::Symbol)\n\nThe required keyword argument var was not assigned in a function call.\n\nExamples\n\njulia> function my_func(;my_arg)\n           return my_arg + 1\n       end\nmy_func (generic function with 1 method)\n\njulia> my_func()\nERROR: UndefKeywordError: keyword argument `my_arg` not assigned\nStacktrace:\n [1] my_func() at ./REPL[1]:2\n [2] top-level scope at REPL[2]:1\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefRefError","location":"base/base.html#Core.UndefRefError","category":"type","text":"UndefRefError()\n\nThe item or field is not defined for the given object.\n\nExamples\n\njulia> struct MyType\n           a::Vector{Int}\n           MyType() = new()\n       end\n\njulia> A = MyType()\nMyType(#undef)\n\njulia> A.a\nERROR: UndefRefError: access to undefined reference\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.UndefVarError","location":"base/base.html#Core.UndefVarError","category":"type","text":"UndefVarError(var::Symbol, [scope])\n\nA symbol in the current scope is not defined.\n\nExamples\n\njulia> a\nERROR: UndefVarError: `a` not defined in `Main`\n\njulia> a = 1;\n\njulia> a\n1\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.StringIndexError","location":"base/base.html#Base.StringIndexError","category":"type","text":"StringIndexError(str, i)\n\nAn error occurred when trying to access str at index i that is not valid.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.InitError","location":"base/base.html#Core.InitError","category":"type","text":"InitError(mod::Symbol, error)\n\nAn error occurred when running a module's __init__ function. The actual error thrown is available in the .error field.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.retry","location":"base/base.html#Base.retry","category":"function","text":"retry(f;  delays=ExponentialBackOff(), check=nothing) -> Function\n\nReturn an anonymous function that calls function f.  If an exception arises, f is repeatedly called again, each time check returns true, after waiting the number of seconds specified in delays.  check should input delays's current state and the Exception.\n\ncompat: Julia 1.2\nBefore Julia 1.2 this signature was restricted to f::Function.\n\nExamples\n\nretry(f, delays=fill(5.0, 3))\nretry(f, delays=rand(5:10, 2))\nretry(f, delays=Base.ExponentialBackOff(n=3, first_delay=5, max_delay=1000))\nretry(http_get, check=(s,e)->e.status == \"503\")(url)\nretry(read, check=(s,e)->isa(e, IOError))(io, 128; all=false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ExponentialBackOff","location":"base/base.html#Base.ExponentialBackOff","category":"type","text":"ExponentialBackOff(; n=1, first_delay=0.05, max_delay=10.0, factor=5.0, jitter=0.1)\n\nA Float64 iterator of length n whose elements exponentially increase at a rate in the interval factor * (1 ± jitter).  The first element is first_delay and all elements are clamped to max_delay.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Timer","location":"base/base.html#Base.Timer-Tuple{Function, Real}","category":"method","text":"Timer(callback::Function, delay; interval = 0, spawn::Union{Nothing,Bool}=nothing)\n\nCreate a timer that runs the function callback at each timer expiration.\n\nWaiting tasks are woken and the function callback is called after an initial delay of delay seconds, and then repeating with the given interval in seconds. If interval is equal to 0, the callback is only run once. The function callback is called with a single argument, the timer itself. Stop a timer by calling close. The callback may still be run one final time, if the timer has already expired.\n\nIf spawn is true, the created task will be spawned, meaning that it will be allowed to move thread, which avoids the side-effect of forcing the parent task to get stuck to the thread it is on. If spawn is nothing (default), the task will be spawned if the parent task isn't sticky.\n\ncompat: Julia 1.12\nThe spawn argument was introduced in Julia 1.12.\n\nExamples\n\nHere the first number is printed after a delay of two seconds, then the following numbers are printed quickly.\n\njulia> begin\n           i = 0\n           cb(timer) = (global i += 1; println(i))\n           t = Timer(cb, 2, interval=0.2)\n           wait(t)\n           sleep(0.5)\n           close(t)\n       end\n1\n2\n3\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Timer","location":"base/base.html#Base.Timer","category":"type","text":"Timer(delay; interval = 0)\n\nCreate a timer that wakes up tasks waiting for it (by calling wait on the timer object).\n\nWaiting tasks are woken after an initial delay of at least delay seconds, and then repeating after at least interval seconds again elapse. If interval is equal to 0, the timer is only triggered once. When the timer is closed (by close) waiting tasks are woken with an error. Use isopen to check whether a timer is still active. Use t.timeout and t.interval to read the setup conditions of a Timer t.\n\njulia> t = Timer(1.0; interval=0.5)\nTimer (open, timeout: 1.0 s, interval: 0.5 s) @0x000000010f4e6e90\n\njulia> isopen(t)\ntrue\n\njulia> t.timeout\n1.0\n\njulia> close(t)\n\njulia> isopen(t)\nfalse\n\nnote: Note\ninterval is subject to accumulating time skew. If you need precise events at a particular absolute time, create a new timer at each expiration with the difference to the next time computed.\n\nnote: Note\nA Timer requires yield points to update its state. For instance, isopen(t::Timer) cannot be used to timeout a non-yielding while loop.\n\n!!! compat \"Julia 1.12     The timeout and interval readable properties were added in Julia 1.12.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.AsyncCondition","location":"base/base.html#Base.AsyncCondition","category":"type","text":"AsyncCondition()\n\nCreate a async condition that wakes up tasks waiting for it (by calling wait on the object) when notified from C by a call to uv_async_send. Waiting tasks are woken with an error when the object is closed (by close). Use isopen to check whether it is still active.\n\nThis provides an implicit acquire & release memory ordering between the sending and waiting threads.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.AsyncCondition","location":"base/base.html#Base.AsyncCondition-Tuple{Function}","category":"method","text":"AsyncCondition(callback::Function)\n\nCreate a async condition that calls the given callback function. The callback is passed one argument, the async condition object itself.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{Module}","category":"method","text":"nameof(m::Module) -> Symbol\n\nGet the name of a Module as a Symbol.\n\nExamples\n\njulia> nameof(Base.Broadcast)\n:Broadcast\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.parentmodule","location":"base/base.html#Base.parentmodule","category":"function","text":"parentmodule(m::Method) -> Module\n\nReturn the module in which the given method m is defined.\n\ncompat: Julia 1.9\nPassing a Method as an argument requires Julia 1.9 or later.\n\n\n\n\n\nparentmodule(f::Function, types) -> Module\n\nDetermine the module containing the first method of a generic function f matching the specified types.\n\n\n\n\n\nparentmodule(f::Function) -> Module\n\nDetermine the module containing the (first) definition of a generic function.\n\n\n\n\n\nparentmodule(t::DataType) -> Module\n\nDetermine the module containing the definition of a (potentially UnionAll-wrapped) DataType.\n\nExamples\n\njulia> module Foo\n           struct Int end\n       end\nFoo\n\njulia> parentmodule(Int)\nCore\n\njulia> parentmodule(Foo.Int)\nFoo\n\n\n\n\n\nparentmodule(m::Module) -> Module\n\nGet a module's enclosing Module. Main is its own parent.\n\nSee also: names, nameof, fullname, @__MODULE__.\n\nExamples\n\njulia> parentmodule(Main)\nMain\n\njulia> parentmodule(Base.Broadcast)\nBase\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pathof","location":"base/base.html#Base.pathof-Tuple{Module}","category":"method","text":"pathof(m::Module)\n\nReturn the path of the m.jl file that was used to import module m, or nothing if m was not imported from a package.\n\nUse dirname to get the directory part and basename to get the file name part of the path.\n\nSee also pkgdir.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pkgdir","location":"base/base.html#Base.pkgdir-Tuple{Module}","category":"method","text":"pkgdir(m::Module[, paths::String...])\n\nReturn the root directory of the package that declared module m, or nothing if m was not declared in a package. Optionally further path component strings can be provided to construct a path within the package root.\n\nTo get the root directory of the package that implements the current module the form pkgdir(@__MODULE__) can be used.\n\nIf an extension module is given, the root of the parent package is returned.\n\njulia> pkgdir(Foo)\n\"/path/to/Foo.jl\"\n\njulia> pkgdir(Foo, \"src\", \"file.jl\")\n\"/path/to/Foo.jl/src/file.jl\"\n\nSee also pathof.\n\ncompat: Julia 1.7\nThe optional argument paths requires at least Julia 1.7.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.pkgversion","location":"base/base.html#Base.pkgversion-Tuple{Module}","category":"method","text":"pkgversion(m::Module)\n\nReturn the version of the package that imported module m, or nothing if m was not imported from a package, or imported from a package without a version field set.\n\nThe version is read from the package's Project.toml during package load.\n\nTo get the version of the package that imported the current module the form pkgversion(@__MODULE__) can be used.\n\ncompat: Julia 1.9\nThis function was introduced in Julia 1.9.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.moduleroot","location":"base/base.html#Base.moduleroot","category":"function","text":"moduleroot(m::Module) -> Module\n\nFind the root module of a given module. This is the first module in the chain of parent modules of m which is either a registered root module or which is its own parent module.\n\n\n\n\n\n","page":"Essentials"},{"title":"__module__","location":"base/base.html#__module__","category":"keyword","text":"__module__\n\nThe argument __module__ is only visible inside the macro, and it provides information (in the form of a Module object) about the expansion context of the macro invocation. See the manual section on Macro invocation for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"__source__","location":"base/base.html#__source__","category":"keyword","text":"__source__\n\nThe argument __source__ is only visible inside the macro, and it provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. See the manual section on Macro invocation for more information.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__MODULE__","location":"base/base.html#Base.@__MODULE__","category":"macro","text":"@__MODULE__ -> Module\n\nGet the Module of the toplevel eval, which is the Module code is currently being read from.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__FILE__","location":"base/base.html#Base.@__FILE__","category":"macro","text":"@__FILE__ -> String\n\nExpand to a string with the path to the file containing the macrocall, or an empty string if evaluated by julia -e <expr>. Return nothing if the macro was missing parser source information. Alternatively see PROGRAM_FILE.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__DIR__","location":"base/base.html#Base.@__DIR__","category":"macro","text":"@__DIR__ -> String\n\nMacro to obtain the absolute path of the current directory as a string.\n\nIf in a script, returns the directory of the script containing the @__DIR__ macrocall. If run from a REPL or if evaluated by julia -e <expr>, returns the current working directory.\n\nExamples\n\nThe example illustrates the difference in the behaviors of @__DIR__ and pwd(), by creating a simple script in a different directory than the current working one and executing both commands:\n\njulia> cd(\"/home/JuliaUser\") # working directory\n\njulia> # create script at /home/JuliaUser/Projects\n       open(\"/home/JuliaUser/Projects/test.jl\",\"w\") do io\n           print(io, \"\"\"\n               println(\"@__DIR__ = \", @__DIR__)\n               println(\"pwd() = \", pwd())\n           \"\"\")\n       end\n\njulia> # outputs script directory and current working directory\n       include(\"/home/JuliaUser/Projects/test.jl\")\n@__DIR__ = /home/JuliaUser/Projects\npwd() = /home/JuliaUser\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@__LINE__","location":"base/base.html#Base.@__LINE__","category":"macro","text":"@__LINE__ -> Int\n\nExpand to the line number of the location of the macrocall. Return 0 if the line number could not be determined.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.fullname","location":"base/base.html#Base.fullname","category":"function","text":"fullname(m::Module)\n\nGet the fully-qualified name of a module as a tuple of symbols. For example,\n\nExamples\n\njulia> fullname(Base.Iterators)\n(:Base, :Iterators)\n\njulia> fullname(Main)\n(:Main,)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.names","location":"base/base.html#Base.names","category":"function","text":"names(x::Module; all::Bool=false, imported::Bool=false, usings::Bool=false) -> Vector{Symbol}\n\nGet a vector of the public names of a Module, excluding deprecated names. If all is true, then the list also includes non-public names defined in the module, deprecated names, and compiler-generated names. If imported is true, then names explicitly imported from other modules are also included. If usings is true, then names explicitly or implicitly imported via using are also included. Names are returned in sorted order.\n\nAs a special case, all names defined in Main are considered \"public\", since it is not idiomatic to explicitly mark names from Main as public.\n\nnote: Note\nsym ∈ names(SomeModule) does not imply isdefined(SomeModule, sym). names may return symbols marked with public or export, even if they are not defined in the module.\n\nwarning: Warning\nnames may return duplicate names. The duplication happens, e.g. if an imported name conflicts with an already existing identifier.\n\ncompat: Julia 1.12\nThe usings argument requires Julia 1.12 or later.\n\nSee also: Base.isexported, Base.ispublic, Base.@locals, @__MODULE__.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isexported","location":"base/base.html#Base.isexported","category":"function","text":"isexported(m::Module, s::Symbol) -> Bool\n\nReturns whether a symbol is exported from a module.\n\nSee also: ispublic, names\n\njulia> module Mod\n           export foo\n           public bar\n       end\nMod\n\njulia> Base.isexported(Mod, :foo)\ntrue\n\njulia> Base.isexported(Mod, :bar)\nfalse\n\njulia> Base.isexported(Mod, :baz)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.ispublic","location":"base/base.html#Base.ispublic","category":"function","text":"ispublic(m::Module, s::Symbol) -> Bool\n\nReturns whether a symbol is marked as public in a module.\n\nExported symbols are considered public.\n\ncompat: Julia 1.11\nThis function and the notion of publicity were added in Julia 1.11.\n\nSee also: isexported, names\n\njulia> module Mod\n           export foo\n           public bar\n       end\nMod\n\njulia> Base.ispublic(Mod, :foo)\ntrue\n\njulia> Base.ispublic(Mod, :bar)\ntrue\n\njulia> Base.ispublic(Mod, :baz)\nfalse\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.nameof","location":"base/base.html#Base.nameof-Tuple{Function}","category":"method","text":"nameof(f::Function) -> Symbol\n\nGet the name of a generic Function as a symbol. For anonymous functions, this is a compiler-generated name. For explicitly-declared subtypes of Function, it is the name of the function's type.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.functionloc","location":"base/base.html#Base.functionloc-Tuple{Any, Any}","category":"method","text":"functionloc(f::Function, types)\n\nReturn a tuple (filename,line) giving the location of a generic Function definition.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.functionloc","location":"base/base.html#Base.functionloc-Tuple{Method}","category":"method","text":"functionloc(m::Method)\n\nReturn a tuple (filename,line) giving the location of a Method definition.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@locals","location":"base/base.html#Base.@locals","category":"macro","text":"@locals()\n\nConstruct a dictionary of the names (as symbols) and values of all local variables defined as of the call site.\n\ncompat: Julia 1.1\nThis macro requires at least Julia 1.1.\n\nExamples\n\njulia> let x = 1, y = 2\n           Base.@locals\n       end\nDict{Symbol, Any} with 2 entries:\n  :y => 2\n  :x => 1\n\njulia> function f(x)\n           local y\n           show(Base.@locals); println()\n           for i = 1:1\n               show(Base.@locals); println()\n           end\n           y = 2\n           show(Base.@locals); println()\n           nothing\n       end;\n\njulia> f(42)\nDict{Symbol, Any}(:x => 42)\nDict{Symbol, Any}(:i => 1, :x => 42)\nDict{Symbol, Any}(:y => 2, :x => 42)\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.getglobal","location":"base/base.html#Core.getglobal","category":"function","text":"getglobal(module::Module, name::Symbol, [order::Symbol=:monotonic])\n\nRetrieve the value of the binding name from the module module. Optionally, an atomic ordering can be defined for the operation, otherwise it defaults to monotonic.\n\nWhile accessing module bindings using getfield is still supported to maintain compatibility, using getglobal should always be preferred since getglobal allows for control over atomic ordering (getfield is always monotonic) and better signifies the code's intent both to the user as well as the compiler.\n\nMost users should not have to call this function directly – The getproperty function or corresponding syntax (i.e. module.name) should be preferred in all but few very specific use cases.\n\ncompat: Julia 1.9\nThis function requires Julia 1.9 or later.\n\nSee also getproperty and setglobal!.\n\nExamples\n\njulia> a = 1\n1\n\njulia> module M\n       a = 2\n       end;\n\njulia> getglobal(@__MODULE__, :a)\n1\n\njulia> getglobal(M, :a)\n2\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.setglobal!","location":"base/base.html#Core.setglobal!","category":"function","text":"setglobal!(module::Module, name::Symbol, x, [order::Symbol=:monotonic])\n\nSet or change the value of the binding name in the module module to x. No type conversion is performed, so if a type has already been declared for the binding, x must be of appropriate type or an error is thrown.\n\nAdditionally, an atomic ordering can be specified for this operation, otherwise it defaults to monotonic.\n\nUsers will typically access this functionality through the setproperty! function or corresponding syntax (i.e. module.name = x) instead, so this is intended only for very specific use cases.\n\ncompat: Julia 1.9\nThis function requires Julia 1.9 or later.\n\nSee also setproperty! and getglobal\n\nExamples\n\njulia> module M; global a; end;\n\njulia> M.a  # same as `getglobal(M, :a)`\nERROR: UndefVarError: `a` not defined in `M`\nSuggestion: add an appropriate import or assignment. This global was declared but not assigned.\nStacktrace:\n [1] getproperty(x::Module, f::Symbol)\n   @ Base ./Base_compiler.jl:40\n [2] top-level scope\n   @ none:1\n\njulia> setglobal!(M, :a, 1)\n1\n\njulia> M.a\n1\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.modifyglobal!","location":"base/base.html#Core.modifyglobal!","category":"function","text":"modifyglobal!(module::Module, name::Symbol, op, x, [order::Symbol=:monotonic]) -> Pair\n\nAtomically perform the operations to get and set a global after applying the function op.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also modifyproperty! and setglobal!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.swapglobal!","location":"base/base.html#Core.swapglobal!","category":"function","text":"swapglobal!(module::Module, name::Symbol, x, [order::Symbol=:monotonic])\n\nAtomically perform the operations to simultaneously get and set a global.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also swapproperty! and setglobal!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.setglobalonce!","location":"base/base.html#Core.setglobalonce!","category":"function","text":"setglobalonce!(module::Module, name::Symbol, value,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> success::Bool\n\nAtomically perform the operations to set a global to a given value, only if it was previously not set.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also setpropertyonce! and setglobal!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.replaceglobal!","location":"base/base.html#Core.replaceglobal!","category":"function","text":"replaceglobal!(module::Module, name::Symbol, expected, desired,\n              [success_order::Symbol, [fail_order::Symbol=success_order]) -> (; old, success::Bool)\n\nAtomically perform the operations to get and conditionally set a global to a given value.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also replaceproperty! and setglobal!.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.@doc","location":"base/base.html#Core.@doc","category":"macro","text":"Documentation\n\nFunctions, methods and types can be documented by placing a string before the definition:\n\n\"\"\"\n    foo(x)\n\nReturn a fooified version of `x`.\n\"\"\"\nfoo(x) = ...\n\nThe @doc macro can be used directly to both set and retrieve documentation / metadata. The macro has special parsing so that the documented object may occur on the next line:\n\n@doc \"blah\"\nfunction foo() ...\n\nBy default, documentation is written as Markdown, but any object can be used as the first argument.\n\nDocumenting objects separately from their definitions\n\nYou can document an object before or after its definition with\n\n@doc \"foo\" function_to_doc\n@doc \"bar\" TypeToDoc\n\nFor macros, the syntax is @doc \"macro doc\" :(Module.@macro) or @doc \"macro doc\" :(string_macro\"\") for string macros. Without the quote :() the expansion of the macro will be documented.\n\nRetrieving Documentation\n\nYou can retrieve docs for functions, macros and other objects as follows:\n\n@doc foo\n@doc @time\n@doc md\"\"\n\nFunctions & Methods\n\nPlacing documentation before a method definition (e.g. function foo() ... or foo() = ...) will cause that specific method to be documented, as opposed to the whole function. Method docs are concatenated together in the order they were defined to provide docs for the function.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs.HTML","location":"base/base.html#Base.Docs.HTML","category":"type","text":"HTML(s): Create an object that renders s as html.\n\nHTML(\"<div>foo</div>\")\n\nYou can also use a stream for large amounts of data:\n\nHTML() do io\n  println(io, \"<div>foo</div>\")\nend\n\nwarning: Warning\nHTML is currently exported to maintain backwards compatibility, but this export is deprecated. It is recommended to use this type as Docs.HTML or to explicitly import it from Docs.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs.Text","location":"base/base.html#Base.Docs.Text","category":"type","text":"Text(s): Create an object that renders s as plain text.\n\nText(\"foo\")\n\nYou can also use a stream for large amounts of data:\n\nText() do io\n  println(io, \"foo\")\nend\n\nwarning: Warning\nText is currently exported to maintain backwards compatibility, but this export is deprecated. It is recommended to use this type as Docs.Text or to explicitly import it from Docs.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs.hasdoc","location":"base/base.html#Base.Docs.hasdoc","category":"function","text":"Docs.hasdoc(mod::Module, sym::Symbol)::Bool\n\nReturn true if sym in mod has a docstring and false otherwise.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Docs.undocumented_names","location":"base/base.html#Base.Docs.undocumented_names","category":"function","text":"undocumented_names(mod::Module; private=false)\n\nReturn a sorted vector of undocumented symbols in module (that is, lacking docstrings). private=false (the default) returns only identifiers declared with public and/or export, whereas private=true returns all symbols in the module (excluding compiler-generated hidden symbols starting with #).\n\nSee also: names, Docs.hasdoc, Base.ispublic.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.identify_package","location":"base/base.html#Base.identify_package","category":"function","text":"Base.identify_package(name::String)::Union{PkgId, Nothing}\nBase.identify_package(where::Union{Module,PkgId}, name::String)::Union{PkgId, Nothing}\n\nIdentify the package by its name from the current environment stack, returning its PkgId, or nothing if it cannot be found.\n\nIf only the name argument is provided, it searches each environment in the stack and its named direct dependencies.\n\nThe where argument provides the context from where to search for the package: in this case it first checks if the name matches the context itself, otherwise it searches all recursive dependencies (from the resolved manifest of each environment) until it locates the context where, and from there identifies the dependency with the corresponding name.\n\njulia> Base.identify_package(\"Pkg\") # Pkg is a dependency of the default environment\nPkg [44cfe95a-1eb2-52ea-b672-e2afdf69b78f]\n\njulia> using LinearAlgebra\n\njulia> Base.identify_package(LinearAlgebra, \"Pkg\") # Pkg is not a dependency of LinearAlgebra\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.locate_package","location":"base/base.html#Base.locate_package","category":"function","text":"Base.locate_package(pkg::PkgId)::Union{String, Nothing}\n\nThe path to the entry-point file for the package corresponding to the identifier pkg, or nothing if not found. See also identify_package.\n\njulia> pkg = Base.identify_package(\"Pkg\")\nPkg [44cfe95a-1eb2-52ea-b672-e2afdf69b78f]\n\njulia> Base.locate_package(pkg)\n\"/path/to/julia/stdlib/v1.12/Pkg/src/Pkg.jl\"\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.require","location":"base/base.html#Base.require","category":"function","text":"require(into::Module, module::Symbol)\n\nThis function is part of the implementation of using / import, if a module is not already defined in Main. It can also be called directly to force reloading a module, regardless of whether it has been loaded before (for example, when interactively developing libraries).\n\nLoads a source file, in the context of the Main module, on every active node, searching standard locations for files. require is considered a top-level operation, so it sets the current include path but does not use it to search for files (see help for include). This function is typically used to load library code, and is implicitly called by using to load packages.\n\nWhen searching for files, require first looks for package code in the global array LOAD_PATH. require is case-sensitive on all platforms, including those with case-insensitive filesystems like macOS and Windows.\n\nFor more details regarding code loading, see the manual sections on modules and parallel computing.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.compilecache","location":"base/base.html#Base.compilecache","category":"function","text":"Base.compilecache(module::PkgId)\n\nCreates a precompiled cache file for a module and all of its dependencies. This can be used to reduce package load times. Cache files are stored in DEPOT_PATH[1]/compiled. See Module initialization and precompilation for important notes.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isprecompiled","location":"base/base.html#Base.isprecompiled","category":"function","text":"Base.isprecompiled(pkg::PkgId; ignore_loaded::Bool=false)\n\nReturns whether a given PkgId within the active project is precompiled.\n\nBy default this check observes the same approach that code loading takes with respect to when different versions of dependencies are currently loaded to that which is expected. To ignore loaded modules and answer as if in a fresh julia session specify ignore_loaded=true.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.get_extension","location":"base/base.html#Base.get_extension","category":"function","text":"get_extension(parent::Module, extension::Symbol)\n\nReturn the module for extension of parent or return nothing if the extension is not loaded.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.gc","location":"base/base.html#Base.GC.gc","category":"function","text":"GC.gc([full=true])\n\nPerform garbage collection. The argument full determines the kind of collection: a full collection (default) traverses all live objects (i.e. full mark) and should reclaim memory from all unreachable objects. An incremental collection only reclaims memory from young objects which are not reachable.\n\nThe GC may decide to perform a full collection even if an incremental collection was requested.\n\nwarning: Warning\nExcessive use will likely lead to poor performance.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.enable","location":"base/base.html#Base.GC.enable","category":"function","text":"GC.enable(on::Bool)\n\nControl whether garbage collection is enabled using a boolean argument (true for enabled, false for disabled). Return previous GC state.\n\nwarning: Warning\nDisabling garbage collection should be used only with caution, as it can cause memory use to grow without bound.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.@preserve","location":"base/base.html#Base.GC.@preserve","category":"macro","text":"GC.@preserve x1 x2 ... xn expr\n\nMark the objects x1, x2, ... as being in use during the evaluation of the expression expr. This is only required in unsafe code where expr implicitly uses memory or other resources owned by one of the xs.\n\nImplicit use of x covers any indirect use of resources logically owned by x which the compiler cannot see. Some examples:\n\nAccessing memory of an object directly via a Ptr\nPassing a pointer to x to ccall\nUsing resources of x which would be cleaned up in the finalizer.\n\n@preserve should generally not have any performance impact in typical use cases where it briefly extends object lifetime. In implementation, @preserve has effects such as protecting dynamically allocated objects from garbage collection.\n\nExamples\n\nWhen loading from a pointer with unsafe_load, the underlying object is implicitly used, for example x is implicitly used by unsafe_load(p) in the following:\n\njulia> let\n           x = Ref{Int}(101)\n           p = Base.unsafe_convert(Ptr{Int}, x)\n           GC.@preserve x unsafe_load(p)\n       end\n101\n\nWhen passing pointers to ccall, the pointed-to object is implicitly used and should be preserved. (Note however that you should normally just pass x directly to ccall which counts as an explicit use.)\n\njulia> let\n           x = \"Hello\"\n           p = pointer(x)\n           Int(GC.@preserve x @ccall strlen(p::Cstring)::Csize_t)\n           # Preferred alternative\n           Int(@ccall strlen(x::Cstring)::Csize_t)\n       end\n5\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.safepoint","location":"base/base.html#Base.GC.safepoint","category":"function","text":"GC.safepoint()\n\nInserts a point in the program where garbage collection may run.\n\nSafepoints are fast and do not themselves trigger garbage collection. However, if another thread has requested the GC to run, reaching a safepoint will cause the current thread to block and wait for the GC.\n\nThis can be useful in rare cases in multi-threaded programs where some tasks are allocating memory (and hence may need to run GC) but other tasks are doing only simple operations (no allocation, task switches, or I/O), which do not yield control to Julia's runtime, and therefore blocks the GC from running. Calling this function periodically in the non-allocating tasks allows garbage collection to run.\n\nNote that even though safepoints are fast (typically around 2 clock cycles), they can still degrade performance if called in a tight loop.\n\ncompat: Julia 1.4\nThis function is available as of Julia 1.4.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.enable_logging","location":"base/base.html#Base.GC.enable_logging","category":"function","text":"GC.enable_logging(on::Bool)\n\nWhen turned on, print statistics about each GC to stderr.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.GC.logging_enabled","location":"base/base.html#Base.GC.logging_enabled","category":"function","text":"GC.logging_enabled()\n\nReturn whether GC logging has been enabled via GC.enable_logging.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.lower","location":"base/base.html#Base.Meta.lower","category":"function","text":"lower(m, x)\n\nTakes the expression x and returns an equivalent expression in lowered form for executing in module m. See also code_lowered.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.@lower","location":"base/base.html#Base.Meta.@lower","category":"macro","text":"@lower [m] x\n\nReturn lowered form of the expression x in module m. By default m is the module in which the macro is called. See also lower.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.parse","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString, Int32}","category":"method","text":"parse(str, start; greedy=true, raise=true, depwarn=true, filename=\"none\")\n\nParse the expression string and return an expression (which could later be passed to eval for execution). start is the code unit index into str of the first character to start parsing at (as with all string indexing, these are not character indices). If greedy is true (default), parse will try to consume as much input as it can; otherwise, it will stop as soon as it has parsed a valid expression. Incomplete but otherwise syntactically valid expressions will return Expr(:incomplete, \"(error message)\"). If raise is true (default), syntax errors other than incomplete expressions will raise an error. If raise is false, parse will return an expression that will raise an error upon evaluation. If depwarn is false, deprecation warnings will be suppressed. The filename argument is used to display diagnostics when an error is raised.\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 1) # start of string\n(:((α, β) = (3, 5)), 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 1, greedy=false)\n(:((α, β)), 9)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 16) # end of string\n(nothing, 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 11) # index of 3\n(:((3, 5)), 16)\n\njulia> Meta.parse(\"(α, β) = 3, 5\", 11, greedy=false)\n(3, 13)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.parse","location":"base/base.html#Base.Meta.parse-Tuple{AbstractString}","category":"method","text":"parse(str; raise=true, depwarn=true, filename=\"none\")\n\nParse the expression string greedily, returning a single expression. An error is thrown if there are additional characters after the first expression. If raise is true (default), syntax errors will raise an error; otherwise, parse will return an expression that will raise an error upon evaluation.  If depwarn is false, deprecation warnings will be suppressed. The filename argument is used to display diagnostics when an error is raised.\n\njulia> Meta.parse(\"x = 3\")\n:(x = 3)\n\njulia> Meta.parse(\"1.0.2\")\nERROR: ParseError:\n# Error @ none:1:1\n1.0.2\n└──┘ ── invalid numeric constant\n[...]\n\njulia> Meta.parse(\"1.0.2\"; raise = false)\n:($(Expr(:error, \"invalid numeric constant \"1.0.\"\")))\n\njulia> Meta.parse(\"x = \")\n:($(Expr(:incomplete, \"incomplete: premature end of input\")))\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.ParseError","location":"base/base.html#Base.Meta.ParseError","category":"type","text":"ParseError(msg)\n\nThe expression passed to the parse function could not be interpreted as a valid Julia expression.\n\n\n\n\n\n","page":"Essentials"},{"title":"Core.QuoteNode","location":"base/base.html#Core.QuoteNode","category":"type","text":"QuoteNode\n\nA quoted piece of code, that does not support interpolation. See the manual section about QuoteNodes for details.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.macroexpand","location":"base/base.html#Base.macroexpand","category":"function","text":"macroexpand(m::Module, x; recursive=true)\n\nTake the expression x and return an equivalent expression with all macros removed (expanded) for executing in module m. The recursive keyword controls whether deeper levels of nested macros are also expanded. This is demonstrated in the example below:\n\njulia> module M\n           macro m1()\n               42\n           end\n           macro m2()\n               :(@m1())\n           end\n       end\nM\n\njulia> macroexpand(M, :(@m2()), recursive=true)\n42\n\njulia> macroexpand(M, :(@m2()), recursive=false)\n:(#= REPL[16]:6 =# M.@m1)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@macroexpand","location":"base/base.html#Base.@macroexpand","category":"macro","text":"@macroexpand [mod,] ex\n\nReturn equivalent expression with all macros removed (expanded). If two arguments are provided, the first is the module to evaluate in.\n\nThere are differences between @macroexpand and macroexpand.\n\nWhile macroexpand takes a keyword argument recursive, @macroexpand is always recursive. For a non recursive macro version, see @macroexpand1.\nWhile macroexpand has an explicit module argument, @macroexpand always expands with respect to the module in which it is called.\n\nThis is best seen in the following example:\n\njulia> module M\n           macro m()\n               1\n           end\n           function f()\n               (@macroexpand(@m),\n                macroexpand(M, :(@m)),\n                macroexpand(Main, :(@m))\n               )\n           end\n       end\nM\n\njulia> macro m()\n           2\n       end\n@m (macro with 1 method)\n\njulia> M.f()\n(1, 1, 2)\n\nWith @macroexpand the expression expands where @macroexpand appears in the code (module M in the example). With macroexpand the expression expands in the module given as the first argument.\n\ncompat: Julia 1.11\nThe two-argument form requires at least Julia 1.11.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.@macroexpand1","location":"base/base.html#Base.@macroexpand1","category":"macro","text":"@macroexpand1 [mod,] ex\n\nNon recursive version of @macroexpand.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.code_lowered","location":"base/base.html#Base.code_lowered","category":"function","text":"code_lowered(f, types; generated=true, debuginfo=:default)\n\nReturn an array of the lowered forms (IR) for the methods matching the given generic function and type signature.\n\nIf generated is false, the returned CodeInfo instances will correspond to fallback implementations. An error is thrown if no fallback implementation exists. If generated is true, these CodeInfo instances will correspond to the method bodies yielded by expanding the generators.\n\nThe keyword debuginfo controls the amount of code metadata present in the output.\n\nNote that an error will be thrown if types are not concrete types when generated is true and any of the corresponding methods are an @generated method.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.code_typed","location":"base/base.html#Base.code_typed","category":"function","text":"code_typed(f, types; kw...)\n\nReturns an array of type-inferred lowered form (IR) for the methods matching the given generic function and type signature.\n\nKeyword Arguments\n\noptimize::Bool = true: optional, controls whether additional optimizations, such as inlining, are also applied.\ndebuginfo::Symbol = :default: optional, controls the amount of code metadata present in the output, possible options are :source or :none.\n\nInternal Keyword Arguments\n\nThis section should be considered internal, and is only for who understands Julia compiler internals.\n\nworld::UInt = Base.get_world_counter(): optional, controls the world age to use when looking up methods, use current world age if not specified.\ninterp::Core.Compiler.AbstractInterpreter = Core.Compiler.NativeInterpreter(world): optional, controls the abstract interpreter to use, use the native interpreter if not specified.\n\nExamples\n\nOne can put the argument types in a tuple to get the corresponding code_typed.\n\njulia> code_typed(+, (Float64, Float64))\n1-element Vector{Any}:\n CodeInfo(\n1 ─ %1 = Base.add_float(x, y)::Float64\n└──      return %1\n) => Float64\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.precompile","location":"base/base.html#Base.precompile","category":"function","text":"precompile(f, argtypes::Tuple{Vararg{Any}}, m::Method)\n\nPrecompile a specific method for the given argument types. This may be used to precompile a different method than the one that would ordinarily be chosen by dispatch, thus mimicking invoke.\n\n\n\n\n\nprecompile(f, argtypes::Tuple{Vararg{Any}})\n\nCompile the given function f for the argument tuple (of types) argtypes, but do not execute it.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.jit_total_bytes","location":"base/base.html#Base.jit_total_bytes","category":"function","text":"Base.jit_total_bytes()\n\nReturn the total amount (in bytes) allocated by the just-in-time compiler for e.g. native code and data.\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.quot","location":"base/base.html#Base.Meta.quot","category":"function","text":"Meta.quot(ex)::Expr\n\nQuote expression ex to produce an expression with head quote. This can for instance be used to represent objects of type Expr in the AST. See also the manual section about QuoteNode.\n\nExamples\n\njulia> eval(Meta.quot(:x))\n:x\n\njulia> dump(Meta.quot(:x))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Symbol x\n\njulia> eval(Meta.quot(:(1+2)))\n:(1 + 2)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isexpr","location":"base/base.html#Base.isexpr","category":"function","text":"Meta.isexpr(ex, head[, n])::Bool\n\nReturn true if ex is an Expr with the given type head and optionally that the argument list is of length n. head may be a Symbol or collection of Symbols. For example, to check that a macro was passed a function call expression, you might use isexpr(ex, :call).\n\nExamples\n\njulia> ex = :(f(x))\n:(f(x))\n\njulia> Meta.isexpr(ex, :block)\nfalse\n\njulia> Meta.isexpr(ex, :call)\ntrue\n\njulia> Meta.isexpr(ex, [:block, :call]) # multiple possible heads\ntrue\n\njulia> Meta.isexpr(ex, :call, 1)\nfalse\n\njulia> Meta.isexpr(ex, :call, 2)\ntrue\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isidentifier","location":"base/base.html#Base.isidentifier","category":"function","text":" isidentifier(s) -> Bool\n\nReturn whether the symbol or string s contains characters that are parsed as a valid ordinary identifier (not a binary/unary operator) in Julia code; see also Base.isoperator.\n\nInternally Julia allows any sequence of characters in a Symbol (except \\0s), and macros automatically use variable names containing # in order to avoid naming collision with the surrounding code. In order for the parser to recognize a variable, it uses a limited set of characters (greatly extended by Unicode). isidentifier() makes it possible to query the parser directly whether a symbol contains valid characters.\n\nExamples\n\njulia> Meta.isidentifier(:x), Meta.isidentifier(\"1x\")\n(true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isoperator","location":"base/base.html#Base.isoperator","category":"function","text":"isoperator(s::Symbol)\n\nReturn true if the symbol can be used as an operator, false otherwise.\n\nExamples\n\njulia> Meta.isoperator(:+), Meta.isoperator(:f)\n(true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isunaryoperator","location":"base/base.html#Base.isunaryoperator","category":"function","text":"isunaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a unary (prefix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isunaryoperator(:-), Meta.isunaryoperator(:√), Meta.isunaryoperator(:f)\n(true, true, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.isbinaryoperator","location":"base/base.html#Base.isbinaryoperator","category":"function","text":"isbinaryoperator(s::Symbol)\n\nReturn true if the symbol can be used as a binary (infix) operator, false otherwise.\n\nExamples\n\njulia> Meta.isbinaryoperator(:-), Meta.isbinaryoperator(:√), Meta.isbinaryoperator(:f)\n(true, false, false)\n\n\n\n\n\n","page":"Essentials"},{"title":"Base.Meta.show_sexpr","location":"base/base.html#Base.Meta.show_sexpr","category":"function","text":"Meta.show_sexpr([io::IO,], ex)\n\nShow expression ex as a lisp style S-expression.\n\nExamples\n\njulia> Meta.show_sexpr(:(f(x, g(y,z))))\n(:call, :f, :x, (:call, :g, :y, :z))\n\n\n\n\n\n","page":"Essentials"},{"title":"Bounds checking","location":"devdocs/boundscheck.html#Bounds-checking","category":"section","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection.","page":"Bounds checking"},{"title":"Eliding bounds checks","location":"devdocs/boundscheck.html#Eliding-bounds-checks","category":"section","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWith a custom array-like type MyArray having:\n\n@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A, i); A.data[to_index(i)])\n\nThen when getindex is inlined into sum, the call to checkbounds(A, i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack.","page":"Bounds checking"},{"title":"Caution!","location":"devdocs/boundscheck.html#Caution!","category":"section","text":"It is easy to accidentally expose unsafe operations with @inbounds. You might be tempted to write the above example as\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWhich quietly assumes 1-based indexing and therefore exposes unsafe memory access when used with OffsetArrays:\n\njulia> using OffsetArrays\n\njulia> sum(OffsetArray([1, 2, 3], -10))\n9164911648 # inconsistent results or segfault\n\nWhile the original source of the error here is 1:length(A), the use of @inbounds increases the consequences from a bounds error to a less easily caught and debugged unsafe memory access. It is often difficult or impossible to prove that a method which uses @inbounds is safe, so one must weigh the benefits of performance improvements against the risk of segfaults and silent misbehavior, especially in public facing APIs.","page":"Bounds checking"},{"title":"Propagating inbounds","location":"devdocs/boundscheck.html#Propagating-inbounds","category":"section","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).\n\nTo override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining.","page":"Bounds checking"},{"title":"The bounds checking call hierarchy","location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","category":"section","text":"The overall hierarchy is:\n\ncheckbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension\n\nHere A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.\n\ncheckbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,\n\ncheckbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)\n\nso checkindex checks a single dimension. All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .\n\nIf you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.\n\nIf you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array. If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.\n\nNote this hierarchy has been designed to reduce the likelihood of method ambiguities. We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument).","page":"Bounds checking"},{"title":"Emit bounds checks","location":"devdocs/boundscheck.html#Emit-bounds-checks","category":"section","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations.","page":"Bounds checking"},{"title":"Multi-Threading","location":"base/multi-threading.html#lib-multithreading","category":"section","text":"See also Multi-Threading.","page":"Multi-Threading"},{"title":"Atomic operations","location":"base/multi-threading.html#Atomic-operations","category":"section","text":"There are also optional memory ordering parameters for the unsafe set of functions, that select the C/C++-compatible versions of these atomic operations, if that parameter is specified to unsafe_load, unsafe_store!, unsafe_swap!, unsafe_replace!, and unsafe_modify!.\n\nwarning: Warning\nThe following APIs are deprecated, though support for them is likely to remain for several releases.","page":"Multi-Threading"},{"title":"ccall using a libuv threadpool (Experimental)","location":"base/multi-threading.html#ccall-using-a-libuv-threadpool-(Experimental)","category":"section","text":"","page":"Multi-Threading"},{"title":"Low-level synchronization primitives","location":"base/multi-threading.html#Low-level-synchronization-primitives","category":"section","text":"These building blocks are used to create the regular synchronization objects.","page":"Multi-Threading"},{"title":"Task metrics (Experimental)","location":"base/multi-threading.html#Task-metrics-(Experimental)","category":"section","text":"","page":"Multi-Threading"},{"title":"Base.Threads.@threads","location":"base/multi-threading.html#Base.Threads.@threads","category":"macro","text":"Threads.@threads [schedule] for ... end\n\nA macro to execute a for loop in parallel. The iteration space is distributed to coarse-grained tasks. This policy can be specified by the schedule argument. The execution of the loop waits for the evaluation of all iterations.\n\nTasks spawned by @threads are scheduled on the :default threadpool. This means that @threads will not use threads from the :interactive threadpool, even if called from the main thread or from a task in the interactive pool. The :default threadpool is intended for compute-intensive parallel workloads.\n\nSee also: @spawn and pmap in Distributed. For more information on threadpools, see the chapter on threadpools.\n\nExtended help\n\nSemantics\n\nUnless stronger guarantees are specified by the scheduling option, the loop executed by @threads macro have the following semantics.\n\nThe @threads macro executes the loop body in an unspecified order and potentially concurrently. It does not specify the exact assignments of the tasks and the worker threads. The assignments can be different for each execution. The loop body code (including any code transitively called from it) must not make any assumptions about the distribution of iterations to tasks or the worker thread in which they are executed. The loop body for each iteration must be able to make forward progress independent of other iterations and be free from data races. As such, invalid synchronizations across iterations may deadlock while unsynchronized memory accesses may result in undefined behavior.\n\nFor example, the above conditions imply that:\n\nA lock taken in an iteration must be released within the same iteration.\nCommunicating between iterations using blocking primitives like Channels is incorrect.\nWrite only to locations not shared across iterations (unless a lock or atomic operation is used).\nUnless the :static schedule is used, the value of threadid() may change even within a single iteration. See Task Migration.\n\nSchedulers\n\nWithout the scheduler argument, the exact scheduling is unspecified and varies across Julia releases. Currently, :dynamic is used when the scheduler is not specified.\n\ncompat: Julia 1.5\nThe schedule argument is available as of Julia 1.5.\n\n:dynamic (default)\n\n:dynamic scheduler executes iterations dynamically to available worker threads. Current implementation assumes that the workload for each iteration is uniform. However, this assumption may be removed in the future.\n\nThis scheduling option is merely a hint to the underlying execution mechanism. However, a few properties can be expected. The number of Tasks used by :dynamic scheduler is bounded by a small constant multiple of the number of available worker threads (Threads.threadpoolsize()). Each task processes contiguous regions of the iteration space. Thus, @threads :dynamic for x in xs; f(x); end is typically more efficient than @sync for x in xs; @spawn f(x); end if length(xs) is significantly larger than the number of the worker threads and the run-time of f(x) is relatively smaller than the cost of spawning and synchronizing a task (typically less than 10 microseconds).\n\ncompat: Julia 1.8\nThe :dynamic option for the schedule argument is available and the default as of Julia 1.8.\n\n:greedy\n\n:greedy scheduler spawns up to Threads.threadpoolsize() tasks, each greedily working on the given iterated values as they are produced. As soon as one task finishes its work, it takes the next value from the iterator. Work done by any individual task is not necessarily on contiguous values from the iterator. The given iterator may produce values forever, only the iterator interface is required (no indexing).\n\nThis scheduling option is generally a good choice if the workload of individual iterations is not uniform/has a large spread.\n\ncompat: Julia 1.11\nThe :greedy option for the schedule argument is available as of Julia 1.11.\n\n:static\n\n:static scheduler creates one task per thread and divides the iterations equally among them, assigning each task specifically to each thread. In particular, the value of threadid() is guaranteed to be constant within one iteration. Specifying :static is an error if used from inside another @threads loop or from a thread other than 1.\n\nnote: Note\n:static scheduling exists for supporting transition of code written before Julia 1.3. In newly written library functions, :static scheduling is discouraged because the functions using this option cannot be called from arbitrary worker threads.\n\nExamples\n\nTo illustrate of the different scheduling strategies, consider the following function busywait containing a non-yielding timed loop that runs for a given number of seconds.\n\njulia> function busywait(seconds)\n            tstart = time_ns()\n            while (time_ns() - tstart) / 1e9 < seconds\n            end\n        end\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :static for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n6.003001 seconds (16.33 k allocations: 899.255 KiB, 0.25% compilation time)\n\njulia> @time begin\n            Threads.@spawn busywait(5)\n            Threads.@threads :dynamic for i in 1:Threads.threadpoolsize()\n                busywait(1)\n            end\n        end\n2.012056 seconds (16.05 k allocations: 883.919 KiB, 0.66% compilation time)\n\nThe :dynamic example takes 2 seconds since one of the non-occupied threads is able to run two of the 1-second iterations to complete the for loop.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.foreach","location":"base/multi-threading.html#Base.Threads.foreach","category":"function","text":"Threads.foreach(f, channel::Channel;\n                schedule::Threads.AbstractSchedule=Threads.FairSchedule(),\n                ntasks=Threads.threadpoolsize())\n\nSimilar to foreach(f, channel), but iteration over channel and calls to f are split across ntasks tasks spawned by Threads.@spawn. This function will wait for all internally spawned tasks to complete before returning.\n\nIf schedule isa FairSchedule, Threads.foreach will attempt to spawn tasks in a manner that enables Julia's scheduler to more freely load-balance work items across threads. This approach generally has higher per-item overhead, but may perform better than StaticSchedule in concurrence with other multithreaded workloads.\n\nIf schedule isa StaticSchedule, Threads.foreach will spawn tasks in a manner that incurs lower per-item overhead than FairSchedule, but is less amenable to load-balancing. This approach thus may be more suitable for fine-grained, uniform workloads, but may perform worse than FairSchedule in concurrence with other multithreaded workloads.\n\nExamples\n\njulia> n = 20\n\njulia> c = Channel{Int}(ch -> foreach(i -> put!(ch, i), 1:n), 1)\n\njulia> d = Channel{Int}(n) do ch\n           f = i -> put!(ch, i^2)\n           Threads.foreach(f, c)\n       end\n\njulia> collect(d)\ncollect(d) = [1, 4, 9, 16, 25, 36, 49, 64, 81, 100, 121, 144, 169, 196, 225, 256, 289, 324, 361, 400]\n\ncompat: Julia 1.6\nThis function requires Julia 1.6 or later.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.@spawn","location":"base/multi-threading.html#Base.Threads.@spawn","category":"macro","text":"Threads.@spawn [:default|:interactive|:samepool] expr\n\nCreate a Task and schedule it to run on any available thread in the specified threadpool: :default, :interactive, or :samepool to use the same as the caller. :default is used if unspecified. The task is allocated to a thread once one becomes available. To wait for the task to finish, call wait on the result of this macro, or call fetch to wait and then obtain its return value.\n\nValues can be interpolated into @spawn via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nnote: Note\nThe thread that the task runs on may change if the task yields, therefore threadid() should not be treated as constant for a task. See Task Migration, and the broader multi-threading manual for further important caveats. See also the chapter on threadpools.\n\ncompat: Julia 1.3\nThis macro is available as of Julia 1.3.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\ncompat: Julia 1.9\nA threadpool may be specified as of Julia 1.9.\n\ncompat: Julia 1.12\nThe same threadpool may be specified as of Julia 1.12.\n\nExamples\n\njulia> t() = println(\"Hello from \", Threads.threadid());\n\njulia> tasks = fetch.([Threads.@spawn t() for i in 1:4]);\nHello from 1\nHello from 1\nHello from 3\nHello from 4\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadid","location":"base/multi-threading.html#Base.Threads.threadid","category":"function","text":"Threads.threadid([t::Task]) -> Int\n\nGet the ID number of the current thread of execution, or the thread of task t. The master thread has ID 1.\n\nExamples\n\njulia> Threads.threadid()\n1\n\njulia> Threads.@threads for i in 1:4\n          println(Threads.threadid())\n       end\n4\n2\n5\n4\n\njulia> Threads.threadid(Threads.@spawn \"foo\")\n2\n\nnote: Note\nThe thread that a task runs on may change if the task yields, which is known as Task Migration. For this reason in most cases it is not safe to use threadid([task]) to index into, say, a vector of buffers or stateful objects.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.maxthreadid","location":"base/multi-threading.html#Base.Threads.maxthreadid","category":"function","text":"Threads.maxthreadid() -> Int\n\nGet a lower bound on the number of threads (across all thread pools) available to the Julia process, with atomic-acquire semantics. The result will always be greater than or equal to threadid() as well as threadid(task) for any task you were able to observe before calling maxthreadid.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.nthreads","location":"base/multi-threading.html#Base.Threads.nthreads","category":"function","text":"Threads.nthreads(:default | :interactive) -> Int\n\nGet the current number of threads within the specified thread pool. The threads in :interactive have id numbers 1:nthreads(:interactive), and the threads in :default have id numbers in nthreads(:interactive) .+ (1:nthreads(:default)).\n\nSee also BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library and Threads.maxthreadid().\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadpool","location":"base/multi-threading.html#Base.Threads.threadpool","category":"function","text":"Threads.threadpool(tid = threadid()) -> Symbol\n\nReturns the specified thread's threadpool; either :default, :interactive, or :foreign.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.nthreadpools","location":"base/multi-threading.html#Base.Threads.nthreadpools","category":"function","text":"Threads.nthreadpools() -> Int\n\nReturns the number of threadpools currently configured.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.threadpoolsize","location":"base/multi-threading.html#Base.Threads.threadpoolsize","category":"function","text":"Threads.threadpoolsize(pool::Symbol = :default) -> Int\n\nGet the number of threads available to the default thread pool (or to the specified thread pool).\n\nSee also: BLAS.get_num_threads and BLAS.set_num_threads in the LinearAlgebra standard library, and nprocs() in the Distributed standard library.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.ngcthreads","location":"base/multi-threading.html#Base.Threads.ngcthreads","category":"function","text":"Threads.ngcthreads() -> Int\n\nReturns the number of GC threads currently configured. This includes both mark threads and concurrent sweep threads.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"atomic","location":"base/multi-threading.html#atomic","category":"keyword","text":"Unsafe pointer operations are compatible with loading and storing pointers declared with _Atomic and std::atomic type in C11 and C++23 respectively. An error may be thrown if there is not support for atomically loading the Julia type T.\n\nSee also: unsafe_load, unsafe_modify!, unsafe_replace!, unsafe_store!, unsafe_swap!\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomic","location":"base/multi-threading.html#Base.@atomic","category":"macro","text":"@atomic var\n@atomic order ex\n\nMark var or ex as being performed atomically, if ex is a supported expression. If no order is specified it defaults to :sequentially_consistent.\n\n@atomic a.b.x = new\n@atomic a.b.x += addend\n@atomic :release a.b.x = new\n@atomic :acquire_release a.b.x += addend\n@atomic m[idx] = new\n@atomic m[idx] += addend\n@atomic :release m[idx] = new\n@atomic :acquire_release m[idx] += addend\n\nPerform the store operation expressed on the right atomically and return the new value.\n\nWith assignment (=), this operation translates to a setproperty!(a.b, :x, new) or, in case of reference, to a setindex_atomic!(m, order, new, idx) call, with order defaulting to :sequentially_consistent.\n\nWith any modifying operator this operation translates to a modifyproperty!(a.b, :x, op, addend)[2] or, in case of reference, to a modifyindex_atomic!(m, order, op, addend, idx...)[2] call, with order defaulting to :sequentially_consistent.\n\n@atomic a.b.x max arg2\n@atomic a.b.x + arg2\n@atomic max(a.b.x, arg2)\n@atomic :acquire_release max(a.b.x, arg2)\n@atomic :acquire_release a.b.x + arg2\n@atomic :acquire_release a.b.x max arg2\n@atomic m[idx] max arg2\n@atomic m[idx] + arg2\n@atomic max(m[idx], arg2)\n@atomic :acquire_release max(m[idx], arg2)\n@atomic :acquire_release m[idx] + arg2\n@atomic :acquire_release m[idx] max arg2\n\nPerform the binary operation expressed on the right atomically. Store the result into the field or the reference in the first argument, and return the values (old, new).\n\nThis operation translates to a modifyproperty!(a.b, :x, func, arg2) or, in case of reference to a modifyindex_atomic!(m, order, func, arg2, idx) call, with order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomic :sequentially_consistent a.x = 2 # set field x of a, with sequential consistency\n2\n\njulia> @atomic a.x += 1 # increment field x of a, with sequential consistency\n3\n\njulia> @atomic a.x + 1 # increment field x of a, with sequential consistency\n3 => 4\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> @atomic max(a.x, 10) # change field x of a to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic a.x max 5 # again change field x of a to the max value, with sequential consistency\n10 => 10\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 2 # set mem[1] to value 2 with sequential consistency\n2\n\njulia> @atomic :monotonic mem[1] # fetch the first value of mem, with monotonic consistency\n2\n\njulia> @atomic mem[1] += 1 # increment the first value of mem, with sequential consistency\n3\n\njulia> @atomic mem[1] + 1 # increment the first value of mem, with sequential consistency\n3 => 4\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\njulia> @atomic max(mem[1], 10) # change the first value of mem to the max value, with sequential consistency\n4 => 10\n\njulia> @atomic mem[1] max 5 # again change the first value of mem to the max value, with sequential consistency\n10 => 10\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomicswap","location":"base/multi-threading.html#Base.@atomicswap","category":"macro","text":"@atomicswap a.b.x = new\n@atomicswap :sequentially_consistent a.b.x = new\n@atomicswap m[idx] = new\n@atomicswap :sequentially_consistent m[idx] = new\n\nStores new into a.b.x (m[idx] in case of reference) and returns the old value of a.b.x (the old value stored at m[idx], respectively).\n\nThis operation translates to a swapproperty!(a.b, :x, new) or, in case of reference, swapindex_atomic!(mem, order, new, idx) call, with order defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicswap a.x = 2+2 # replace field x of a with 4, with sequential consistency\n1\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n4\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicswap mem[1] = 4 # replace the first value of `mem` with 4, with sequential consistency\n1\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n4\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomicreplace","location":"base/multi-threading.html#Base.@atomicreplace","category":"macro","text":"@atomicreplace a.b.x expected => desired\n@atomicreplace :sequentially_consistent a.b.x expected => desired\n@atomicreplace :sequentially_consistent :monotonic a.b.x expected => desired\n@atomicreplace m[idx] expected => desired\n@atomicreplace :sequentially_consistent m[idx] expected => desired\n@atomicreplace :sequentially_consistent :monotonic m[idx] expected => desired\n\nPerform the conditional replacement expressed by the pair atomically, returning the values (old, success::Bool). Where success indicates whether the replacement was completed.\n\nThis operation translates to a replaceproperty!(a.b, :x, expected, desired) or, in case of reference, to a replaceindex_atomic!(mem, success_order, fail_order, expected, desired, idx) call, with both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct Atomic{T}; @atomic x::T; end\n\njulia> a = Atomic(1)\nAtomic{Int64}(1)\n\njulia> @atomicreplace a.x 1 => 2 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n2\n\njulia> @atomicreplace a.x 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace a.x xchg\n(old = 2, success = true)\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n0\n\njulia> mem = AtomicMemory{Int}(undef, 2);\n\njulia> @atomic mem[1] = 1;\n\njulia> @atomicreplace mem[1] 1 => 2 # replace the first value of mem with 2 if it was 1, with sequential consistency\n(old = 1, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n2\n\njulia> @atomicreplace mem[1] 1 => 3 # replace field x of a with 2 if it was 1, with sequential consistency\n(old = 2, success = false)\n\njulia> xchg = 2 => 0; # replace field x of a with 0 if it was 2, with sequential consistency\n\njulia> @atomicreplace mem[1] xchg\n(old = 2, success = true)\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n0\n\ncompat: Julia 1.7\nAtomic fields functionality requires at least Julia 1.7.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@atomiconce","location":"base/multi-threading.html#Base.@atomiconce","category":"macro","text":"@atomiconce a.b.x = value\n@atomiconce :sequentially_consistent a.b.x = value\n@atomiconce :sequentially_consistent :monotonic a.b.x = value\n@atomiconce m[idx] = value\n@atomiconce :sequentially_consistent m[idx] = value\n@atomiconce :sequentially_consistent :monotonic m[idx] = value\n\nPerform the conditional assignment of value atomically if it was previously unset. Returned value success::Bool indicates whether the assignment was completed.\n\nThis operation translates to a setpropertyonce!(a.b, :x, value) or, in case of reference, to a setindexonce_atomic!(m, success_order, fail_order, value, idx) call, with both orders defaulting to :sequentially_consistent.\n\nSee Per-field atomics section in the manual for more details.\n\nExamples\n\njulia> mutable struct AtomicOnce\n           @atomic x\n           AtomicOnce() = new()\n       end\n\njulia> a = AtomicOnce()\nAtomicOnce(#undef)\n\njulia> @atomiconce a.x = 1 # set field x of a to 1, if unset, with sequential consistency\ntrue\n\njulia> @atomic a.x # fetch field x of a, with sequential consistency\n1\n\njulia> @atomiconce :monotonic a.x = 2 # set field x of a to 1, if unset, with monotonic consistence\nfalse\n\njulia> mem = AtomicMemory{Vector{Int}}(undef, 1);\n\njulia> isassigned(mem, 1)\nfalse\n\njulia> @atomiconce mem[1] = [1] # set the first value of mem to [1], if unset, with sequential consistency\ntrue\n\njulia> isassigned(mem, 1)\ntrue\n\njulia> @atomic mem[1] # fetch the first value of mem, with sequential consistency\n1-element Vector{Int64}:\n 1\n\njulia> @atomiconce :monotonic mem[1] = [2] # set the first value of mem to [2], if unset, with monotonic\nfalse\n\njulia> @atomic mem[1]\n1-element Vector{Int64}:\n 1\n\ncompat: Julia 1.11\nAtomic fields functionality requires at least Julia 1.11.\n\ncompat: Julia 1.12\nAtomic reference functionality requires at least Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Core.AtomicMemory","location":"base/multi-threading.html#Core.AtomicMemory","category":"type","text":"AtomicMemory{T} == GenericMemory{:atomic, T, Core.CPU}\n\nFixed-size DenseVector{T}. Fetching of any of its individual elements is performed atomically (with :monotonic ordering by default).\n\nwarning: Warning\nThe access to AtomicMemory must be done by either using the @atomic macro or the lower level interface functions: Base.getindex_atomic, Base.setindex_atomic!, Base.setindexonce_atomic!, Base.swapindex_atomic!, Base.modifyindex_atomic!, and Base.replaceindex_atomic!.\n\nFor details, see Atomic Operations as well as macros @atomic, @atomiconce, @atomicswap, and @atomicreplace.\n\ncompat: Julia 1.11\nThis type requires Julia 1.11 or later.\n\ncompat: Julia 1.12\nLower level interface functions or @atomic macro requires Julia 1.12 or later.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.Atomic","location":"base/multi-threading.html#Base.Threads.Atomic","category":"type","text":"Threads.Atomic{T}\n\nHolds a reference to an object of type T, ensuring that it is only accessed atomically, i.e. in a thread-safe manner.\n\nOnly certain \"simple\" types can be used atomically, namely the primitive boolean, integer, and float-point types. These are Bool, Int8...Int128, UInt8...UInt128, and Float16...Float64.\n\nNew atomic objects can be created from a non-atomic values; if none is specified, the atomic object is initialized with zero.\n\nAtomic objects can be accessed using the [] notation:\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> x[] = 1\n1\n\njulia> x[]\n1\n\nAtomic operations use an atomic_ prefix, such as atomic_add!, atomic_xchg!, etc.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_cas!","location":"base/multi-threading.html#Base.Threads.atomic_cas!","category":"function","text":"Threads.atomic_cas!(x::Atomic{T}, cmp::T, newval::T) where T\n\nAtomically compare-and-set x\n\nAtomically compares the value in x with cmp. If equal, write newval to x. Otherwise, leaves x unmodified. Returns the old value in x. By comparing the returned value to cmp (via ===) one knows whether x was modified and now holds the new value newval.\n\nFor further details, see LLVM's cmpxchg instruction.\n\nThis function can be used to implement transactional semantics. Before the transaction, one records the value in x. After the transaction, the new value is stored only if x has not been modified in the mean time.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 4, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_cas!(x, 3, 2);\n\njulia> x\nBase.Threads.Atomic{Int64}(2)\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xchg!","location":"base/multi-threading.html#Base.Threads.atomic_xchg!","category":"function","text":"Threads.atomic_xchg!(x::Atomic{T}, newval::T) where T\n\nAtomically exchange the value in x\n\nAtomically exchanges the value in x with newval. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xchg instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_xchg!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_add!","location":"base/multi-threading.html#Base.Threads.atomic_add!","category":"function","text":"Threads.atomic_add!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically add val to x\n\nPerforms x[] += val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw add instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_add!(x, 2)\n3\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_sub!","location":"base/multi-threading.html#Base.Threads.atomic_sub!","category":"function","text":"Threads.atomic_sub!(x::Atomic{T}, val::T) where T <: ArithmeticTypes\n\nAtomically subtract val from x\n\nPerforms x[] -= val atomically. Returns the old value. Not defined for Atomic{Bool}.\n\nFor further details, see LLVM's atomicrmw sub instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_sub!(x, 2)\n3\n\njulia> x[]\n1\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_and!","location":"base/multi-threading.html#Base.Threads.atomic_and!","category":"function","text":"Threads.atomic_and!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-and x with val\n\nPerforms x[] &= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw and instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_and!(x, 2)\n3\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_nand!","location":"base/multi-threading.html#Base.Threads.atomic_nand!","category":"function","text":"Threads.atomic_nand!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-nand (not-and) x with val\n\nPerforms x[] = ~(x[] & val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw nand instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(3)\nBase.Threads.Atomic{Int64}(3)\n\njulia> Threads.atomic_nand!(x, 2)\n3\n\njulia> x[]\n-3\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_or!","location":"base/multi-threading.html#Base.Threads.atomic_or!","category":"function","text":"Threads.atomic_or!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-or x with val\n\nPerforms x[] |= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw or instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_or!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_xor!","location":"base/multi-threading.html#Base.Threads.atomic_xor!","category":"function","text":"Threads.atomic_xor!(x::Atomic{T}, val::T) where T\n\nAtomically bitwise-xor (exclusive-or) x with val\n\nPerforms x[] $= val atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw xor instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_xor!(x, 7)\n5\n\njulia> x[]\n2\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_max!","location":"base/multi-threading.html#Base.Threads.atomic_max!","category":"function","text":"Threads.atomic_max!(x::Atomic{T}, val::T) where T\n\nAtomically store the maximum of x and val in x\n\nPerforms x[] = max(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw max instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(5)\nBase.Threads.Atomic{Int64}(5)\n\njulia> Threads.atomic_max!(x, 7)\n5\n\njulia> x[]\n7\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_min!","location":"base/multi-threading.html#Base.Threads.atomic_min!","category":"function","text":"Threads.atomic_min!(x::Atomic{T}, val::T) where T\n\nAtomically store the minimum of x and val in x\n\nPerforms x[] = min(x[], val) atomically. Returns the old value.\n\nFor further details, see LLVM's atomicrmw min instruction.\n\nExamples\n\njulia> x = Threads.Atomic{Int}(7)\nBase.Threads.Atomic{Int64}(7)\n\njulia> Threads.atomic_min!(x, 5)\n7\n\njulia> x[]\n5\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.atomic_fence","location":"base/multi-threading.html#Base.Threads.atomic_fence","category":"function","text":"Threads.atomic_fence()\n\nInsert a sequential-consistency memory fence\n\nInserts a memory fence with sequentially-consistent ordering semantics. There are algorithms where this is needed, i.e. where an acquire/release ordering is insufficient.\n\nThis is likely a very expensive operation. Given that all other atomic operations in Julia already have acquire/release semantics, explicit fences should not be necessary in most cases.\n\nFor further details, see LLVM's fence instruction.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.@threadcall","location":"base/multi-threading.html#Base.@threadcall","category":"macro","text":"@threadcall((cfunc, clib), rettype, (argtypes...), argvals...)\n\nThe @threadcall macro is called in the same way as ccall but does the work in a different thread. This is useful when you want to call a blocking C function without causing the current julia thread to become blocked. Concurrency is limited by size of the libuv thread pool, which defaults to 4 threads but can be increased by setting the UV_THREADPOOL_SIZE environment variable and restarting the julia process.\n\nNote that the called function should never call back into Julia.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Threads.SpinLock","location":"base/multi-threading.html#Base.Threads.SpinLock","category":"type","text":"SpinLock()\n\nCreate a non-reentrant, test-and-test-and-set spin lock. Recursive use will result in a deadlock. This kind of lock should only be used around code that takes little time to execute and does not block (e.g. perform I/O). In general, ReentrantLock should be used instead.\n\nEach lock must be matched with an unlock. If !islocked(lck::SpinLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\nTest-and-test-and-set spin locks are quickest up to about 30ish contending threads. If you have more contention than that, different synchronization approaches should be considered.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Experimental.task_metrics","location":"base/multi-threading.html#Base.Experimental.task_metrics","category":"function","text":"Base.Experimental.task_metrics(::Bool)\n\nEnable or disable the collection of per-task metrics. A Task created when Base.Experimental.task_metrics(true) is in effect will have Base.Experimental.task_running_time_ns and Base.Experimental.task_wall_time_ns timing information available.\n\nnote: Note\nTask metrics can be enabled at start-up via the --task-metrics=yes command line option.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Experimental.task_running_time_ns","location":"base/multi-threading.html#Base.Experimental.task_running_time_ns","category":"function","text":"Base.Experimental.task_running_time_ns(t::Task) -> Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t has spent running. This metric is only updated when t yields or completes unless t is the current task, in which it will be updated continuously. See also Base.Experimental.task_wall_time_ns.\n\nReturns nothing if task timings are not enabled. See Base.Experimental.task_metrics.\n\nnote: This metric is from the Julia scheduler\nA task may be running on an OS thread that is descheduled by the OS scheduler, this time still counts towards the metric.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Base.Experimental.task_wall_time_ns","location":"base/multi-threading.html#Base.Experimental.task_wall_time_ns","category":"function","text":"Base.Experimental.task_wall_time_ns(t::Task) -> Union{UInt64, Nothing}\n\nReturn the total nanoseconds that the task t was runnable. This is the time since the task first entered the run queue until the time at which it completed, or until the current time if the task has not yet completed. See also Base.Experimental.task_running_time_ns.\n\nReturns nothing if task timings are not enabled. See Base.Experimental.task_metrics.\n\ncompat: Julia 1.12\nThis method was added in Julia 1.12.\n\n\n\n\n\n","page":"Multi-Threading"},{"title":"Documentation","location":"manual/documentation.html#man-documentation","category":"section","text":"","page":"Documentation"},{"title":"Accessing Documentation","location":"manual/documentation.html#Accessing-Documentation","category":"section","text":"Documentation can be accessed at the REPL or in IJulia by typing ? followed by the name of a function or macro, and pressing Enter. For example,\n\n?cos\n?@time\n?r\"\"\n\nwill show documentation for the relevant function, macro or string macro respectively. Most Julia environments provide a way to access documentation directly:\n\nVS Code shows documentation when you hover over a function name. You can also use the Julia panel in the sidebar to search for documentation.\nIn Pluto, open the \"Live Docs\" panel on the bottom right.\nIn Juno using Ctrl-J, Ctrl-D will show the documentation for the object under the cursor.\n\nDocs.hasdoc(module, name)::Bool tells whether a name has a docstring. Docs.undocumented_names(module; all) returns the undocumented names in a module.","page":"Documentation"},{"title":"Writing Documentation","location":"manual/documentation.html#Writing-Documentation","category":"section","text":"Julia enables package developers and users to document functions, types and other objects easily via a built-in documentation system.\n\nThe basic syntax is simple: any string appearing just before an object (function, macro, type or instance) will be interpreted as documenting it (these are called docstrings). Note that no blank lines or comments may intervene between a docstring and the documented object. Here is a basic example:\n\n\"Tell whether there are too foo items in the array.\"\nfoo(xs::Array) = ...\n\nDocumentation is interpreted as Markdown, so you can use indentation and code fences to delimit code examples from text. Technically, any object can be associated with any other as metadata; Markdown happens to be the default, but one can construct other string macros and pass them to the @doc macro just as well.\n\nnote: Note\nMarkdown support is implemented in the Markdown standard library and for a full list of supported syntax see the documentation.\n\nHere is a more complex example, still using Markdown:\n\n\"\"\"\n    bar(x[, y])\n\nCompute the Bar index between `x` and `y`.\n\nIf `y` is unspecified, compute the Bar index between all pairs of columns of `x`.\n\n# Examples\n```julia-repl\njulia> bar([1, 2], [1, 2])\n1\n```\n\"\"\"\nfunction bar(x, y) ...\n\nAs in the example above, we recommend following some simple conventions when writing documentation:\n\nAlways show the signature of a function at the top of the documentation, with a four-space indent so that it is printed as Julia code.\nThis can be identical to the signature present in the Julia code (like mean(x::AbstractArray)), or a simplified form. Optional arguments should be represented with their default values (i.e. f(x, y=1)) when possible, following the actual Julia syntax. Optional arguments which do not have a default value should be put in brackets (i.e. f(x[, y]) and f(x[, y[, z]])). An alternative solution is to use several lines: one without optional arguments, the other(s) with them. This solution can also be used to document several related methods of a given function. When a function accepts many keyword arguments, only include a <keyword arguments> placeholder in the signature (i.e. f(x; <keyword arguments>)), and give the complete list under an # Arguments section (see point 4 below).\nInclude a single one-line sentence describing what the function does or what the object represents after the simplified signature block. If needed, provide more details in a second paragraph, after a blank line.\nThe one-line sentence should use the imperative form (\"Do this\", \"Return that\") instead of the third person (do not write \"Returns the length...\") when documenting functions. It should end with a period. If the meaning of a function cannot be summarized easily, splitting it into separate composable parts could be beneficial (this should not be taken as an absolute requirement for every single case though).\nDo not repeat yourself.\nSince the function name is given by the signature, there is no need to start the documentation with \"The function bar...\": go straight to the point. Similarly, if the signature specifies the types of the arguments, mentioning them in the description is redundant.\nOnly provide an argument list when really necessary.\nFor simple functions, it is often clearer to mention the role of the arguments directly in the description of the function's purpose. An argument list would only repeat information already provided elsewhere. However, providing an argument list can be a good idea for complex functions with many arguments (in particular keyword arguments). In that case, insert it after the general description of the function, under an # Arguments header, with one - bullet for each argument. The list should mention the types and default values (if any) of the arguments:\n\"\"\"\n...\n# Arguments\n- `n::Integer`: the number of elements to compute.\n- `dim::Integer=1`: the dimensions along which to perform the computation.\n...\n\"\"\"\nProvide hints to related functions.\nSometimes there are functions of related functionality. To increase discoverability please provide a short list of these in a See also paragraph.\nSee also [`bar!`](@ref), [`baz`](@ref), [`baaz`](@ref).\nInclude any code examples in an # Examples section.\nExamples should, whenever possible, be written as doctests. A doctest is a fenced code block (see Code blocks) starting with ```jldoctest and contains any number of julia> prompts together with inputs and expected outputs that mimic the Julia REPL.\nnote: Note\nDoctests are enabled by Documenter.jl. For more detailed documentation see Documenter's manual.\nFor example in the following docstring a variable a is defined and the expected result, as printed in a Julia REPL, appears afterwards:\n\"\"\"\nSome nice documentation here.\n\n# Examples\n```jldoctest\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n```\n\"\"\"\nwarning: Warning\nCalling rand and other RNG-related functions should be avoided in doctests since they will not produce consistent outputs during different Julia sessions. If you would like to show some random number generation related functionality, one option is to explicitly construct and seed your own RNG object (see Random) and pass it to the functions you are doctesting.Operating system word size (Int32 or Int64) as well as path separator differences (/ or \\) will also affect the reproducibility of some doctests.Note that whitespace in your doctest is significant! The doctest will fail if you misalign the output of pretty-printing an array, for example.\nYou can then run make -C doc doctest=true to run all the doctests in the Julia Manual and API documentation, which will ensure that your example works.\nTo indicate that the output result is truncated, you may write [...] at the line where checking should stop. This is useful to hide a stacktrace (which contains non-permanent references to lines of julia code) when the doctest shows that an exception is thrown, for example:\n```jldoctest\njulia> div(1, 0)\nERROR: DivideError: integer division error\n[...]\n```\nExamples that are untestable should be written within fenced code blocks starting with ```julia so that they are highlighted correctly in the generated documentation.\ntip: Tip\nWherever possible examples should be self-contained and runnable so that readers are able to try them out without having to include any dependencies.\nUse backticks to identify code and equations.\nJulia identifiers and code excerpts should always appear between backticks ` to enable highlighting. Equations in the LaTeX syntax can be inserted between double backticks ``. Use Unicode characters rather than their LaTeX escape sequence, i.e. ``α = 1`` rather than ``\\\\alpha = 1``.\nPlace the starting and ending \"\"\" characters on lines by themselves.\nThat is, write:\n\"\"\"\n...\n\n...\n\"\"\"\nf(x, y) = ...\nrather than:\n\"\"\"...\n\n...\"\"\"\nf(x, y) = ...\nThis makes it clearer where docstrings start and end.\nRespect the line length limit used in the surrounding code.\nDocstrings are edited using the same tools as code. Therefore, the same conventions should apply. It is recommended that lines are at most 92 characters wide.\nProvide information allowing custom types to implement the function in an # Implementation section. These implementation details are intended for developers rather than users, explaining e.g. which functions should be overridden and which functions automatically use appropriate fallbacks. Such details are best kept separate from the main description of the function's behavior.\nFor long docstrings, consider splitting the documentation with an # Extended help header. The typical help-mode will show only the material above the header; you can access the full help by adding a '?' at the beginning of the expression (i.e., \"??foo\" rather than \"?foo\").","page":"Documentation"},{"title":"Functions & Methods","location":"manual/documentation.html#Functions-and-Methods","category":"section","text":"Functions in Julia may have multiple implementations, known as methods. While it's good practice for generic functions to have a single purpose, Julia allows methods to be documented individually if necessary. In general, only the most generic method should be documented, or even the function itself (i.e. the object created without any methods by function bar end). Specific methods should only be documented if their behaviour differs from the more generic ones. In any case, they should not repeat the information provided elsewhere. For example:\n\n\"\"\"\n    *(x, y, z...)\n\nMultiplication operator. `x * y * z *...` calls this function with multiple\narguments, i.e. `*(x, y, z...)`.\n\"\"\"\nfunction *(x, y, z...)\n    # ... [implementation sold separately] ...\nend\n\n\"\"\"\n    *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\nWhen applied to strings, concatenates them.\n\"\"\"\nfunction *(x::AbstractString, y::AbstractString, z::AbstractString...)\n    # ... [insert secret sauce here] ...\nend\n\nhelp?> *\nsearch: * .*\n\n  *(x, y, z...)\n\n  Multiplication operator. x * y * z *... calls this function with multiple\n  arguments, i.e. *(x,y,z...).\n\n  *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\n  When applied to strings, concatenates them.\n\nWhen retrieving documentation for a generic function, the metadata for each method is concatenated with the catdoc function, which can of course be overridden for custom types.","page":"Documentation"},{"title":"Advanced Usage","location":"manual/documentation.html#Advanced-Usage","category":"section","text":"The @doc macro associates its first argument with its second in a per-module dictionary called META.\n\nTo make it easier to write documentation, the parser treats the macro name @doc specially: if a call to @doc has one argument, but another expression appears after a single line break, then that additional expression is added as an argument to the macro. Therefore the following syntax is parsed as a 2-argument call to @doc:\n\n@doc raw\"\"\"\n...\n\"\"\"\nf(x) = x\n\nThis makes it possible to use expressions other than normal string literals (such as the raw\"\" string macro) as a docstring.\n\nWhen used for retrieving documentation, the @doc macro (or equally, the doc function) will search all META dictionaries for metadata relevant to the given object and return it. The returned object (some Markdown content, for example) will by default display itself intelligently. This design also makes it easy to use the doc system in a programmatic way; for example, to re-use documentation between different versions of a function:\n\n@doc \"...\" foo!\n@doc (@doc foo!) foo\n\nOr for use with Julia's metaprogramming functionality:\n\nfor (f, op) in ((:add, :+), (:subtract, :-), (:multiply, :*), (:divide, :/))\n    @eval begin\n        $f(a, b) = $op(a, b)\n    end\nend\n@doc \"`add(a, b)` adds `a` and `b` together\" add\n@doc \"`subtract(a, b)` subtracts `b` from `a`\" subtract\n\nDocumentation in non-toplevel blocks, such as begin, if, for, let, and inner constructors, should be added to the documentation system via @doc as well. For example:\n\nif condition()\n    @doc \"...\"\n    f(x) = x\nend\n\nwill add documentation to f(x) when condition() is true. Note that even if f(x) goes out of scope at the end of a block, its documentation will remain.\n\nIt is possible to make use of metaprogramming to assist in the creation of documentation. When using string-interpolation within the docstring you will need to use an extra $ as shown with $($name):\n\nfor func in (:day, :dayofmonth)\n    name = string(func)\n    @eval begin\n        @doc \"\"\"\n            $($name)(dt::TimeType) -> Int64\n\n        The day of month of a `Date` or `DateTime` as an `Int64`.\n        \"\"\" $func(dt::Dates.TimeType)\n    end\nend","page":"Documentation"},{"title":"Dynamic documentation","location":"manual/documentation.html#Dynamic-documentation","category":"section","text":"Sometimes the appropriate documentation for an instance of a type depends on the field values of that instance, rather than just on the type itself. In these cases, you can add a method to Docs.getdoc for your custom type that returns the documentation on a per-instance basis. For instance,\n\nstruct MyType\n    value::Int\nend\n\nDocs.getdoc(t::MyType) = \"Documentation for MyType with value $(t.value)\"\n\nx = MyType(1)\ny = MyType(2)\n\n?x will display \"Documentation for MyType with value 1\" while ?y will display \"Documentation for MyType with value 2\".","page":"Documentation"},{"title":"Syntax Guide","location":"manual/documentation.html#Syntax-Guide","category":"section","text":"This guide provides a comprehensive overview of how to attach documentation to all Julia syntax constructs for which providing documentation is possible.\n\nIn the following examples \"...\" is used to illustrate an arbitrary docstring.","page":"Documentation"},{"title":"$ and \\ characters","location":"manual/documentation.html#and-\\-characters","category":"section","text":"The $ and \\ characters are still parsed as string interpolation or start of an escape sequence in docstrings too. The raw\"\" string macro together with the @doc macro can be used to avoid having to escape them. This is handy when the docstrings include LaTeX or Julia source code examples containing interpolation:\n\n@doc raw\"\"\"\n```math\n\\LaTeX\n```\n\"\"\"\nfunction f end","page":"Documentation"},{"title":"Functions and Methods","location":"manual/documentation.html#Functions-and-Methods-2","category":"section","text":"\"...\"\nfunction f end\n\n\"...\"\nf\n\nAdds docstring \"...\" to the function f. The first version is the preferred syntax, however both are equivalent.\n\n\"...\"\nf(x) = x\n\n\"...\"\nfunction f(x)\n    return x\nend\n\n\"...\"\nf(x)\n\nAdds docstring \"...\" to the method f(::Any).\n\n\"...\"\nf(x, y = 1) = x + y\n\nAdds docstring \"...\" to two Methods, namely f(::Any) and f(::Any, ::Any).","page":"Documentation"},{"title":"Macros","location":"manual/documentation.html#Macros","category":"section","text":"\"...\"\nmacro m(x) end\n\nAdds docstring \"...\" to the @m(::Any) macro definition.\n\n\"...\"\n:(@m1)\n\n\"...\"\nmacro m2 end\n\nAdds docstring \"...\" to the macros named @m1 and @m2.","page":"Documentation"},{"title":"Types","location":"manual/documentation.html#Types","category":"section","text":"\"...\"\nabstract type T1 end\n\n\"...\"\nmutable struct T2\n    ...\nend\n\n\"...\"\nstruct T3\n    ...\nend\n\nAdds the docstring \"...\" to types T1, T2, and T3.\n\n\"...\"\nT1\n\n\"...\"\nT2\n\n\"...\"\nT3\n\nAdds the docstring \"...\" to types T1, T2, and T3. The previous version is the preferred syntax, however both are equivalent.\n\n\"...\"\nstruct T\n    \"x\"\n    x\n    \"y\"\n    y\n\n    @doc \"Inner constructor\"\n    function T()\n        new(...)\n    end\nend\n\nAdds docstring \"...\" to type T, \"x\" to field T.x, \"y\" to field T.y, and \"Inner constructor\" to the inner constructor T(). Also applicable to mutable struct types.","page":"Documentation"},{"title":"Modules","location":"manual/documentation.html#Modules","category":"section","text":"\"...\"\nmodule M end\n\nmodule M\n\n\"...\"\nM\n\nend\n\nAdds docstring \"...\" to the Module M. Adding the docstring above the Module is the preferred syntax, however both are equivalent.\n\nThe module docstring is evaluated inside the scope of the module, allowing access to all the symbols defined in and imported into the module:\n\n\"The magic number is $(MAGIC).\"\nmodule DocStringEval\nconst MAGIC = 42\nend\n\nDocumenting a baremodule by placing a docstring above the expression automatically imports @doc into the module. These imports must be done manually when the module expression is not documented:\n\n\"...\"\nbaremodule M\n# ...\nend\n\nbaremodule M\n\nimport Base: @doc\n\n\"...\"\nf(x) = x\n\nend","page":"Documentation"},{"title":"Global Variables","location":"manual/documentation.html#Global-Variables","category":"section","text":"\"...\"\nconst a = 1\n\n\"...\"\nb = 2\n\n\"...\"\nglobal c = 3\n\nAdds docstring \"...\" to the Bindings a, b, and c.\n\nBindings are used to store a reference to a particular Symbol in a Module without storing the referenced value itself.\n\nnote: Note\nWhen a const definition is only used to define an alias of another definition, such as is the case with the function div and its alias ÷ in Base, do not document the alias and instead document the actual function.If the alias is documented and not the real definition then the docsystem (? mode) will not return the docstring attached to the alias when the real definition is searched for.For example you should write\"...\"\nf(x) = x + 1\nconst alias = frather thanf(x) = x + 1\n\"...\"\nconst alias = f\n\n\"...\"\nsym\n\nAdds docstring \"...\" to the value associated with sym. However, it is preferred that sym is documented where it is defined.","page":"Documentation"},{"title":"Multiple Objects","location":"manual/documentation.html#Multiple-Objects","category":"section","text":"\"...\"\na, b\n\nAdds docstring \"...\" to a and b each of which should be a documentable expression. This syntax is equivalent to\n\n\"...\"\na\n\n\"...\"\nb\n\nAny number of expressions many be documented together in this way. This syntax can be useful when two functions are related, such as non-mutating and mutating versions f and f!.","page":"Documentation"},{"title":"Macro-generated code","location":"manual/documentation.html#Macro-generated-code","category":"section","text":"\"...\"\n@m expression\n\nAdds docstring \"...\" to the expression generated by expanding @m expression. This allows for expressions decorated with @inline, @noinline, @generated, or any other macro to be documented in the same way as undecorated expressions.\n\nMacro authors should take note that only macros that generate a single expression will automatically support docstrings. If a macro returns a block containing multiple subexpressions then the subexpression that should be documented must be marked using the @__doc__ macro.\n\nThe @enum macro makes use of @__doc__ to allow for documenting Enums. Examining its definition should serve as an example of how to use @__doc__ correctly.","page":"Documentation"},{"title":"Core.@__doc__","location":"manual/documentation.html#Core.@__doc__","category":"macro","text":"@__doc__(ex)\n\nLow-level macro used to mark expressions returned by a macro that should be documented. If more than one expression is marked then the same docstring is applied to each expression.\n\nmacro example(f)\n    quote\n        $(f)() = 0\n        @__doc__ $(f)(x) = 1\n        $(f)(x, y) = 2\n    end |> esc\nend\n\n@__doc__ has no effect when a macro that uses it is not documented.\n\ncompat: Julia 1.12\nThis section documents a very subtle corner case that is only relevant to macros which themselves both define other macros and then attempt to use them within the same expansion. Such macros were impossible to write prior to Julia 1.12 and are still quite rare. If you are not writing such a macro, you may ignore this note.In versions prior to Julia 1.12, macroexpansion would recursively expand through Expr(:toplevel) blocks. This behavior was changed in Julia 1.12 to allow macros to recursively define other macros and use them in the same returned expression. However, to preserve backwards compatibility with existing uses of @__doc__, the doc system will still expand through Expr(:toplevel) blocks when looking for @__doc__ markers. As a result, macro-defining-macros will have an observable behavior difference when annotated with a docstring:julia> macro macroception()\n    Expr(:toplevel, :(macro foo() 1 end), :(@foo))\nend\n\njulia> @macroception\n1\n\njulia> \"Docstring\" @macroception\nERROR: LoadError: UndefVarError: `@foo` not defined in `Main`The supported workaround is to manually expand the @__doc__ macro in the defining macro, which the docsystem will recognize and suppress the recursive expansion:julia> macro macroception()\n    Expr(:toplevel,\n        macroexpand(__module__, :(@__doc__ macro foo() 1 end); recursive=false),\n        :(@foo))\nend\n\njulia> @macroception\n1\n\njulia> \"Docstring\" @macroception\n1\n\n\n\n\n\n","page":"Documentation"},{"title":"Tasks","location":"base/parallel.html#Tasks","category":"section","text":"","page":"Tasks"},{"title":"Scheduling","location":"base/parallel.html#Scheduling","category":"section","text":"","page":"Tasks"},{"title":"Synchronization","location":"base/parallel.html#lib-task-sync","category":"section","text":"","page":"Tasks"},{"title":"Channels","location":"base/parallel.html#Channels","category":"section","text":"","page":"Tasks"},{"title":"Low-level synchronization using schedule and wait","location":"base/parallel.html#low-level-schedule-wait","category":"section","text":"The easiest correct use of schedule is on a Task that is not started (scheduled) yet. However, it is possible to use schedule and wait as a very low-level building block for constructing synchronization interfaces. A crucial pre-condition of calling schedule(task) is that the caller must \"own\" the task; i.e., it must know that the call to wait in the given task is happening at the locations known to the code calling schedule(task). One strategy for ensuring such pre-condition is to use atomics, as demonstrated in the following example:\n\n@enum OWEState begin\n    OWE_EMPTY\n    OWE_WAITING\n    OWE_NOTIFYING\nend\n\nmutable struct OneWayEvent\n    @atomic state::OWEState\n    task::Task\n    OneWayEvent() = new(OWE_EMPTY)\nend\n\nfunction Base.notify(ev::OneWayEvent)\n    state = @atomic ev.state\n    while state !== OWE_NOTIFYING\n        # Spin until we successfully update the state to OWE_NOTIFYING:\n        state, ok = @atomicreplace(ev.state, state => OWE_NOTIFYING)\n        if ok\n            if state == OWE_WAITING\n                # OWE_WAITING -> OWE_NOTIFYING transition means that the waiter task is\n                # already waiting or about to call `wait`. The notifier task must wake up\n                # the waiter task.\n                schedule(ev.task)\n            else\n                @assert state == OWE_EMPTY\n                # Since we are assuming that there is only one notifier task (for\n                # simplicity), we know that the other possible case here is OWE_EMPTY.\n                # We do not need to do anything because we know that the waiter task has\n                # not called `wait(ev::OneWayEvent)` yet.\n            end\n            break\n        end\n    end\n    return\nend\n\nfunction Base.wait(ev::OneWayEvent)\n    ev.task = current_task()\n    state, ok = @atomicreplace(ev.state, OWE_EMPTY => OWE_WAITING)\n    if ok\n        # OWE_EMPTY -> OWE_WAITING transition means that the notifier task is guaranteed to\n        # invoke OWE_WAITING -> OWE_NOTIFYING transition. The waiter task must call\n        # `wait()` immediately. In particular, it MUST NOT invoke any function that may\n        # yield to the scheduler at this point in code.\n        wait()\n    else\n        @assert state == OWE_NOTIFYING\n        # Otherwise, the `state` must have already been moved to OWE_NOTIFYING by the\n        # notifier task.\n    end\n    return\nend\n\nev = OneWayEvent()\n@sync begin\n    Threads.@spawn begin\n        wait(ev)\n        println(\"done\")\n    end\n    println(\"notifying...\")\n    notify(ev)\nend\n\n# output\nnotifying...\ndone\n\nOneWayEvent lets one task to wait for another task's notify. It is a limited communication interface since wait can only be used once from a single task (note the non-atomic assignment of ev.task)\n\nIn this example, notify(ev::OneWayEvent) is allowed to call schedule(ev.task) if and only if it modifies the state from OWE_WAITING to OWE_NOTIFYING. This lets us know that the task executing wait(ev::OneWayEvent) is now in the ok branch and that there cannot be other tasks that tries to schedule(ev.task) since their @atomicreplace(ev.state, state => OWE_NOTIFYING) will fail.","page":"Tasks"},{"title":"Core.Task","location":"base/parallel.html#Core.Task","category":"type","text":"Task(func[, reserved_stack::Int])\n\nCreate a Task (i.e. coroutine) to execute the given function func (which must be callable with no arguments). The task exits when this function returns. The task will run in the \"world age\" from the parent at construction when scheduled.\n\nThe optional reserved_stack argument specifies the size of the stack available for this task, in bytes. The default, 0, uses the system-dependent stack size default.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@task","location":"base/parallel.html#Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only creates a task, and does not run it.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@async","location":"base/parallel.html#Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nwarning: Warning\nIt is strongly encouraged to favor Threads.@spawn over @async always even when no parallelism is required especially in publicly distributed libraries.  This is because a use of @async disables the migration of the parent task across worker threads in the current implementation of Julia.  Thus, seemingly innocent use of @async in a library function can have a large impact on the performance of very different parts of user applications.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap","location":"base/parallel.html#Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple equal length collections). For multiple collection arguments, f is applied elementwise.\n\nThe output is guaranteed to be the same order as the elements of the collection(s) c.\n\nntasks specifies the number of tasks to run concurrently. Depending on the length of the collections, if ntasks is unspecified, up to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the number of tasks to run in parallel is checked before processing every element and a new task started if the value of ntasks_func is greater than the current number of tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must then be a function that must accept a Vector of argument tuples and must return a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning the objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Vector{UInt64}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Vector{UInt64}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array of argument tuples and return an array of results. map is used in the modified mapping function to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Vector{String}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.asyncmap!","location":"base/parallel.html#Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than returning a collection.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.current_task","location":"base/parallel.html#Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskdone","location":"base/parallel.html#Base.istaskdone","category":"function","text":"istaskdone(t::Task) -> Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskstarted","location":"base/parallel.html#Base.istaskstarted","category":"function","text":"istaskstarted(t::Task) -> Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.istaskfailed","location":"base/parallel.html#Base.istaskfailed","category":"function","text":"istaskfailed(t::Task) -> Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.task_local_storage","location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to key; the previous value of key, or lack thereof, is restored afterwards. Useful for emulating dynamic scoping.\n\n\n\n\n\n","page":"Tasks"},{"title":"Core.ConcurrencyViolationError","location":"base/parallel.html#Core.ConcurrencyViolationError","category":"type","text":"ConcurrencyViolationError(msg) <: Exception\n\nAn error thrown when a detectable violation of concurrent semantics has occurred.\n\nA non-exhaustive list of examples of when this is used include:\n\nThrowing when a deadlock has been detected (e.g. wait(current_task()))\nKnown-unsafe behavior is attempted (e.g. yield(current_task))\nA known non-threadsafe datastructure is attempted to be modified from multiple concurrent tasks\nA lock is being unlocked that wasn't locked by this task\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.yield","location":"base/parallel.html#Base.yield","category":"function","text":"yield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which immediately yields to t before calling the scheduler.\n\nThrows a ConcurrencyViolationError if t is the currently running task.\n\n\n\n\n\nyield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this function is still runnable, and will be restarted immediately if there are no other runnable tasks.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.yieldto","location":"base/parallel.html#Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is called with no arguments. On subsequent switches, arg is returned from the task's last call to yieldto. This is a low-level call that only switches tasks, not considering states or scheduling in any way. Its use is discouraged.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.sleep","location":"base/parallel.html#Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1 millisecond or input of 0.001.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.schedule","location":"base/parallel.html#Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system is otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of yieldto) when it runs again. If error is true, the value is raised as an exception in the woken task.\n\nwarning: Warning\nIt is incorrect to use schedule on an arbitrary Task that has already been started. See the API reference for more information.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.errormonitor","location":"base/parallel.html#Base.errormonitor","category":"function","text":"errormonitor(t::Task)\n\nPrint an error log to stderr if task t fails.\n\nExamples\n\njulia> wait(errormonitor(Threads.@spawn error(\"task failed\")); throw = false)\nUnhandled Task ERROR: task failed\nStacktrace:\n[...]\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@sync","location":"base/parallel.html#Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn, Distributed.@spawnat and Distributed.@distributed are complete. All exceptions thrown by enclosed async operations are collected and thrown as a CompositeException.\n\nExamples\n\njulia> Threads.nthreads()\n4\n\njulia> @sync begin\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 1\")\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 2\")\n       end;\nThread-id 3, task 1\nThread-id 1, task 2\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.wait","location":"base/parallel.html#Base.wait","category":"function","text":"wait([x])\n\nBlock the current task until some event occurs.\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition and return the val parameter passed to notify. See the Condition-specific docstring of wait for the exact behavior.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process can be used to determine success or failure.\nTask: Wait for a Task to finish. See the Task-specific docstring of wait for the exact behavior.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be restarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before proceeding.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.waitany","location":"base/parallel.html#Base.waitany","category":"function","text":"waitany(tasks; throw=true) -> (done_tasks, remaining_tasks)\n\nWait until at least one of the given tasks have been completed.\n\nIf throw is true, throw CompositeException when one of the completed tasks completes with an exception.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\nwarning: Warning\nThis may scale poorly compared to writing code that uses multiple individual tasks that each runs serially, since this needs to scan the list of tasks each time and synchronize with each one every time this is called. Or consider using waitall(tasks; failfast=true) instead.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.waitall","location":"base/parallel.html#Base.waitall","category":"function","text":"waitall(tasks; failfast=true, throw=true) -> (done_tasks, remaining_tasks)\n\nWait until all the given tasks have been completed.\n\nIf failfast is true, the function will return when at least one of the given tasks is finished by exception. If throw is true, throw CompositeException when one of the completed tasks has failed.\n\nfailfast and throw keyword arguments work independently; when only throw=true is specified, this function waits for all the tasks to complete.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Task}","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Any}","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.timedwait","location":"base/parallel.html#Base.timedwait","category":"function","text":"timedwait(testcb, timeout::Real; pollint::Real=0.1)\n\nWait until testcb() returns true or timeout seconds have passed, whichever is earlier. The test function is polled every pollint seconds. The minimum value for pollint is 0.001 seconds, that is, 1 millisecond.\n\nReturn :ok or :timed_out.\n\nExamples\n\njulia> cb() = (sleep(5); return);\n\njulia> t = @async cb();\n\njulia> timedwait(()->istaskdone(t), 1)\n:timed_out\n\njulia> timedwait(()->istaskdone(t), 6.5)\n:ok\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Condition","location":"base/parallel.html#Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a Condition are suspended and queued. Tasks are woken up when notify is later called on the Condition. Waiting on a condition can return a value or raise an error if the optional arguments of notify are used. Edge triggering means that only tasks waiting at the time notify is called can be woken up. For level-triggered notifications, you must keep extra state to keep track of whether a notification has happened. The Channel and Threads.Event types do this, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Threads.Condition","location":"base/parallel.html#Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call lock on it. When wait is called, the lock is atomically released during blocking, and will be reacquired before wait returns. Therefore idiomatic use of a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Event","location":"base/parallel.html#Base.Event","category":"type","text":"Event([autoreset=false])\n\nCreate a level-triggered event source. Tasks that call wait on an Event are suspended and queued until notify is called on the Event. After notify is called, the Event remains in a signaled state and tasks will no longer block when waiting for it, until reset is called.\n\nIf autoreset is true, at most one task will be released from wait for each call to notify.\n\nThis provides an acquire & release memory ordering on notify/wait.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\ncompat: Julia 1.8\nThe autoreset functionality and memory ordering guarantee requires at least Julia 1.8.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.notify","location":"base/parallel.html#Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default), all waiting tasks are woken, otherwise only one is. If error is true, the passed value is raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.reset","location":"base/parallel.html#Base.reset-Tuple{Base.Event}","category":"method","text":"reset(::Event)\n\nReset an Event back into an un-set state. Then any future calls to wait will block until notify is called again.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Semaphore","location":"base/parallel.html#Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size acquires to be in use at any time. Each acquire must be matched with a release.\n\nThis provides a acquire & release memory ordering on acquire/release calls.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.acquire","location":"base/parallel.html#Base.acquire","category":"function","text":"acquire(f, s::Semaphore)\n\nExecute f after acquiring from Semaphore s, and release on completion or error.\n\nFor example, a do-block form that ensures only 2 calls of foo will be active at the same time:\n\ns = Base.Semaphore(2)\n@sync for _ in 1:100\n    Threads.@spawn begin\n        Base.acquire(s) do\n            foo()\n        end\n    end\nend\n\ncompat: Julia 1.8\nThis method requires at least Julia 1.8.\n\n\n\n\n\nacquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available, blocking until one can be acquired.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.release","location":"base/parallel.html#Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool, possibly allowing another task to acquire it and resume execution.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.AbstractLock","location":"base/parallel.html#Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that implement the synchronization primitives: lock, trylock, unlock, and islocked.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.lock","location":"base/parallel.html#Base.lock","category":"function","text":"lock(f::Function, l::Lockable)\n\nAcquire the lock associated with l, execute f with the lock held, and release the lock when f returns. f will receive one positional argument: the value wrapped by l. If the lock is already locked by a different task/thread, wait for it to become available. When this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f returns. If the lock is already locked by a different task/thread, wait for it to become available.\n\nWhen this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\nSee also: @lock.\n\ncompat: Julia 1.7\nUsing a Channel as the second argument requires Julia 1.7 or later.\n\n\n\n\n\nlock(lock)\n\nAcquire the lock when it becomes available. If the lock is already locked by a different task/thread, wait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.unlock","location":"base/parallel.html#Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an internal counter and return immediately.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.trylock","location":"base/parallel.html#Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available, and return true if successful. If the lock is already locked by a different task/thread, return false.\n\nEach successful trylock must be matched by an unlock.\n\nFunction trylock combined with islocked can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.islocked","location":"base/parallel.html#Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread. This function alone should not be used for synchronization. However, islocked combined with trylock can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\nExtended help\n\nFor example, an exponential backoff can be implemented as follows if the lock implementation satisfied the properties documented below.\n\nnspins = 0\nwhile true\n    while islocked(lock)\n        GC.safepoint()\n        nspins += 1\n        nspins > LIMIT && error(\"timeout\")\n    end\n    trylock(lock) && break\n    backoff()\nend\n\nImplementation\n\nA lock implementation is advised to define islocked with the following properties and note it in its docstring.\n\nislocked(lock) is data-race-free.\nIf islocked(lock) returns false, an immediate invocation of trylock(lock) must succeed (returns true) if there is no interference from other tasks.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.ReentrantLock","location":"base/parallel.html#Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreates a re-entrant lock for synchronizing Tasks. The same task can acquire the lock as many times as required (this is what the \"Reentrant\" part of the name means). Each lock must be matched with an unlock.\n\nCalling lock will also inhibit running of finalizers on that thread until the corresponding unlock. Use of the standard lock pattern illustrated below should naturally be supported, but beware of inverting the try/lock order or missing the try block entirely (e.g. attempting to return with the lock still held):\n\nThis provides a acquire/release memory ordering on lock/unlock calls.\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\nIf !islocked(lck::ReentrantLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.@lock","location":"base/parallel.html#Base.@lock","category":"macro","text":"@lock l expr\n\nMacro version of lock(f, l::AbstractLock) but with expr instead of f function. Expands to:\n\nlock(l)\ntry\n    expr\nfinally\n    unlock(l)\nend\n\nThis is similar to using lock with a do block, but avoids creating a closure and thus can improve the performance.\n\ncompat: Compat\n@lock was added in Julia 1.3, and exported in Julia 1.7.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Lockable","location":"base/parallel.html#Base.Lockable","category":"type","text":"Lockable(value, lock = ReentrantLock())\n\nCreates a Lockable object that wraps value and associates it with the provided lock. This object supports @lock, lock, trylock, unlock. To access the value, index the lockable object while holding the lock.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\nExample\n\njulia> locked_list = Base.Lockable(Int[]);\n\njulia> @lock(locked_list, push!(locked_list[], 1)) # must hold the lock to access the value\n1-element Vector{Int64}:\n 1\n\njulia> lock(summary, locked_list)\n\"1-element Vector{Int64}\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.AbstractChannel","location":"base/parallel.html#Base.AbstractChannel","category":"type","text":"AbstractChannel{T}\n\nRepresentation of a channel passing objects of type T.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstructs a Channel with an internal buffer that can hold a maximum of size objects of type T. put! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called. And vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.Channel","location":"base/parallel.html#Base.Channel-Tuple{Function}","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false, threadpool=nothing)\n\nCreate a new task from func, bind it to a new channel of type T and size size, and schedule the task, all in a single call. The channel is automatically closed when the task terminates.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via the keyword argument taskref.\n\nIf spawn=true, the Task created for func may be scheduled on another thread in parallel, equivalent to creating a task via Threads.@spawn.\n\nIf spawn=true and the threadpool argument is not set, it defaults to :default.\n\nIf the threadpool argument is set (to :default or :interactive), this implies that spawn=true and the new Task is spawned to the specified threadpool.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3. In earlier versions of Julia, Channel used keyword arguments to set size and T, but those constructors are deprecated.\n\ncompat: Julia 1.9\nThe threadpool= argument was added in Julia 1.9.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end;\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.put!","location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different task.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.take!","location":"base/parallel.html#Base.take!-Tuple{Channel}","category":"method","text":"take!(c::Channel)\n\nRemoves and returns a value from a Channel in order. Blocks until data is available. For unbuffered channels, blocks until a put! is performed by a different task.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(1);\n\njulia> put!(c, 1);\n\njulia> take!(c)\n1\n\nUnbuffered channel:\n\njulia> c = Channel(0);\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);\n\njulia> take!(c)\n1\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.isfull","location":"base/parallel.html#Base.isfull-Tuple{Channel}","category":"method","text":"isfull(c::Channel)\n\nDetermines if a Channel is full, in the sense that calling put!(c, some_value) would have blocked. Returns immediately, does not block.\n\nNote that it may frequently be the case that put! will not block after this returns true. Users must take precautions not to accidentally create live-lock bugs in their code by calling this method, as these are generally harder to debug than deadlocks. It is also possible that put! will block after this call returns false, if there are multiple producer tasks calling put! in parallel.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(1); # capacity = 1\n\njulia> isfull(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isfull(c)\ntrue\n\nUnbuffered channel:\n\njulia> c = Channel(); # capacity = 0\n\njulia> isfull(c) # unbuffered channel is always full\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.isready","location":"base/parallel.html#Base.isready-Tuple{Channel}","category":"method","text":"isready(c::Channel)\n\nDetermines whether a Channel has a value stored in it. Returns immediately, does not block.\n\nFor unbuffered channels, return true if there are tasks waiting on a put!.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isready(c)\ntrue\n\nUnbuffered channel:\n\njulia> c = Channel();\n\njulia> isready(c)  # no tasks waiting to put!\nfalse\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);  # schedule a put! task\n\njulia> isready(c)\ntrue\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.isopen","location":"base/parallel.html#Base.isopen-Tuple{Channel}","category":"method","text":"isopen(c::Channel)\n\nDetermines whether a Channel is open for new put! operations. Notice that a Channelcan be closed and still have buffered elements which can be consumed with [take!`](@ref).\n\nExamples\n\nBuffered channel with task:\n\njulia> c = Channel(ch -> put!(ch, 1), 1);\n\njulia> isopen(c) # The channel is closed to new `put!`s\nfalse\n\njulia> isready(c) # The channel is closed but still contains elements\ntrue\n\njulia> take!(c)\n1\n\njulia> isready(c)\nfalse\n\nUnbuffered channel:\n\njulia> c = Channel{Int}();\n\njulia> isopen(c)\ntrue\n\njulia> close(c)\n\njulia> isopen(c)\nfalse\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.fetch","location":"base/parallel.html#Base.fetch-Tuple{Channel}","category":"method","text":"fetch(c::Channel)\n\nWaits for and returns (without removing) the first available item from the Channel. Note: fetch is unsupported on an unbuffered (0-size) Channel.\n\nExamples\n\nBuffered channel:\n\njulia> c = Channel(3) do ch\n           foreach(i -> put!(ch, i), 1:3)\n       end;\n\njulia> fetch(c)\n1\n\njulia> collect(c)  # item is not removed\n3-element Vector{Any}:\n 1\n 2\n 3\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.close","location":"base/parallel.html#Base.close-Tuple{Channel}","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n","page":"Tasks"},{"title":"Base.bind","location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task. Channel chnl is automatically closed when the task terminates. Any uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination. Terminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will close the channel. When multiple channels are bound to the same task, termination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n","page":"Tasks"},{"title":"Frequently Asked Questions","location":"manual/faq.html#Frequently-Asked-Questions","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"General","location":"manual/faq.html#General","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Is Julia named after someone or something?","location":"manual/faq.html#Is-Julia-named-after-someone-or-something?","category":"section","text":"No.","page":"Frequently Asked Questions"},{"title":"Why don't you compile Matlab/Python/R/… code to Julia?","location":"manual/faq.html#Why-don't-you-compile-Matlab/Python/R/…-code-to-Julia?","category":"section","text":"Since many people are familiar with the syntax of other dynamic languages, and lots of code has already been written in those languages, it is natural to wonder why we didn't just plug a Matlab or Python front-end into a Julia back-end (or “transpile” code to Julia) in order to get all the performance benefits of Julia without requiring programmers to learn a new language. Simple, right?\n\nThe basic issue is that there is nothing special about Julia's compiler: we use a commonplace compiler (LLVM) with no “secret sauce” that other language developers don't know about. Indeed, Julia's compiler is in many ways much simpler than those of other dynamic languages (e.g. PyPy or LuaJIT). Julia's performance advantage derives almost entirely from its front-end: its language semantics allow a well-written Julia program to give more opportunities to the compiler to generate efficient code and memory layouts. If you tried to compile Matlab or Python code to Julia, our compiler would be limited by the semantics of Matlab or Python to producing code no better than that of existing compilers for those languages (and probably worse). The key role of semantics is also why several existing Python compilers (like Numba and Pythran) only attempt to optimize a small subset of the language (e.g. operations on Numpy arrays and scalars), and for this subset they are already doing at least as well as we could for the same semantics. The people working on those projects are incredibly smart and have accomplished amazing things, but retrofitting a compiler onto a language that was designed to be interpreted is a very difficult problem.\n\nJulia's advantage is that good performance is not limited to a small subset of “built-in” types and operations, and one can write high-level type-generic code that works on arbitrary user-defined types while remaining fast and memory-efficient. Types in languages like Python simply don't provide enough information to the compiler for similar capabilities, so as soon as you used those languages as a Julia front-end you would be stuck.\n\nFor similar reasons, automated translation to Julia would also typically generate unreadable, slow, non-idiomatic code that would not be a good starting point for a native Julia port from another language.\n\nOn the other hand, language interoperability is extremely useful: we want to exploit existing high-quality code in other languages from Julia (and vice versa)!  The best way to enable this is not a transpiler, but rather via easy inter-language calling facilities. We have worked hard on this, from the built-in ccall intrinsic (to call C and Fortran libraries) to JuliaInterop packages that connect Julia to Python, Matlab, C++, and more.","page":"Frequently Asked Questions"},{"title":"Public API","location":"manual/faq.html#man-api","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How does Julia define its public API?","location":"manual/faq.html#How-does-Julia-define-its-public-API?","category":"section","text":"Julia's public API is the behavior described in documentation of public bindings from Base and the standard libraries. Functions, types, and constants are not part of the public API if they are not public, even if they have docstrings or are described in the documentation. Further, only the documented behavior of public bindings is part of the public API. Undocumented behavior of public bindings is internal.\n\nPublic bindings are those marked with either public foo or export foo.\n\nIn other words:\n\nDocumented behavior of public bindings is part of the public API.\nUndocumented behavior of public bindings is not part of the public API.\nDocumented behavior of private bindings is not part of the public API.\nUndocumented behavior of private bindings is not part of the public API.\n\nYou can get a complete list of the public bindings from a module with names(MyModule).\n\nPackage authors are encouraged to define their public API similarly.\n\nAnything in Julia's Public API is covered by SemVer and therefore will not be removed or receive meaningful breaking changes before Julia 2.0.","page":"Frequently Asked Questions"},{"title":"There is a useful undocumented function/type/constant. Can I use it?","location":"manual/faq.html#There-is-a-useful-undocumented-function/type/constant.-Can-I-use-it?","category":"section","text":"Updating Julia may break your code if you use non-public API. If the code is self-contained, it may be a good idea to copy it into your project. If you want to rely on a complex non-public API, especially when using it from a stable package, it is a good idea to open an issue or pull request to start a discussion for turning it into a public API. However, we do not discourage the attempt to create packages that expose stable public interfaces while relying on non-public implementation details of Julia and buffering the differences across different Julia versions.","page":"Frequently Asked Questions"},{"title":"The documentation is not accurate enough. Can I rely on the existing behavior?","location":"manual/faq.html#The-documentation-is-not-accurate-enough.-Can-I-rely-on-the-existing-behavior?","category":"section","text":"Please open an issue or pull request to start a discussion for turning the existing behavior into a public API.","page":"Frequently Asked Questions"},{"title":"Sessions and the REPL","location":"manual/faq.html#Sessions-and-the-REPL","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I delete an object in memory?","location":"manual/faq.html#How-do-I-delete-an-object-in-memory?","category":"section","text":"Julia does not have an analog of MATLAB's clear function; once a name is defined in a Julia session (technically, in module Main), it is always present.\n\nIf memory usage is your concern, you can always replace objects with ones that consume less memory.  For example, if A is a gigabyte-sized array that you no longer need, you can free the memory with A = nothing. The memory will be released the next time the garbage collector runs; you can force this to happen with GC.gc(). Moreover, an attempt to use A will likely result in an error, because most methods are not defined on type Nothing.","page":"Frequently Asked Questions"},{"title":"Scripting","location":"manual/faq.html#man-scripting","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I check if the current file is being run as the main script?","location":"manual/faq.html#How-do-I-check-if-the-current-file-is-being-run-as-the-main-script?","category":"section","text":"When a file is run as the main script using julia file.jl one might want to activate extra functionality like command line argument handling. A way to determine that a file is run in this fashion is to check if abspath(PROGRAM_FILE) == @__FILE__ is true.\n\nHowever, it is recommended to not write files that double as a script and as an importable library. If one needs functionality both available as a library and a script, it is better to write is as a library, then import the functionality into a distinct script.","page":"Frequently Asked Questions"},{"title":"How do I catch CTRL-C in a script?","location":"manual/faq.html#catch-ctrl-c","category":"section","text":"Running a Julia script using julia file.jl does not throw InterruptException when you try to terminate it with CTRL-C (SIGINT). To run a certain code before terminating a Julia script, which may or may not be caused by CTRL-C, use atexit. Alternatively, you can use julia -e 'include(popfirst!(ARGS))' file.jl to execute a script while being able to catch InterruptException in the try block. Note that with this strategy PROGRAM_FILE will not be set.","page":"Frequently Asked Questions"},{"title":"How do I pass options to julia using #!/usr/bin/env?","location":"manual/faq.html#How-do-I-pass-options-to-julia-using-#!/usr/bin/env?","category":"section","text":"Passing options to julia in a so-called shebang line, as in #!/usr/bin/env julia --startup-file=no, will not work on many platforms (BSD, macOS, Linux) where the kernel, unlike the shell, does not split arguments at space characters. The option env -S, which splits a single argument string into multiple arguments at spaces, similar to a shell, offers a simple workaround:\n\n#!/usr/bin/env -S julia --color=yes --startup-file=no\n@show ARGS  # put any Julia code here\n\nnote: Note\nOption env -S appeared in FreeBSD 6.0 (2005), macOS Sierra (2016) and GNU/Linux coreutils 8.30 (2018).","page":"Frequently Asked Questions"},{"title":"Why doesn't run support * or pipes for scripting external programs?","location":"manual/faq.html#Why-doesn't-run-support-*-or-pipes-for-scripting-external-programs?","category":"section","text":"Julia's run function launches external programs directly, without invoking an operating-system shell (unlike the system(\"...\") function in other languages like Python, R, or C). That means that run does not perform wildcard expansion of * (\"globbing\"), nor does it interpret shell pipelines like | or >.\n\nYou can still do globbing and pipelines using Julia features, however. For example, the built-in pipeline function allows you to chain external programs and files, similar to shell pipes, and the Glob.jl package implements POSIX-compatible globbing.\n\nYou can, of course, run programs through the shell by explicitly passing a shell and a command string to run, e.g. run(`sh -c \"ls > files.txt\"`) to use the Unix Bourne shell, but you should generally prefer pure-Julia scripting like run(pipeline(`ls`, \"files.txt\")). The reason why we avoid the shell by default is that shelling out sucks: launching processes via the shell is slow, fragile to quoting of special characters,  has poor error handling, and is problematic for portability.  (The Python developers came to a similar conclusion.)","page":"Frequently Asked Questions"},{"title":"Variables and Assignments","location":"manual/faq.html#Variables-and-Assignments","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why am I getting UndefVarError from a simple loop?","location":"manual/faq.html#Why-am-I-getting-UndefVarError-from-a-simple-loop?","category":"section","text":"You might have something like:\n\nx = 0\nwhile x < 10\n    x += 1\nend\n\nand notice that it works fine in an interactive environment (like the Julia REPL), but gives UndefVarError: `x` not defined when you try to run it in script or other file. What is going on is that Julia generally requires you to be explicit about assigning to global variables in a local scope.\n\nHere, x is a global variable, while defines a local scope, and x += 1 is an assignment to a global in that local scope.\n\nAs mentioned above, Julia (version 1.5 or later) allows you to omit the global keyword for code in the REPL (and many other interactive environments), to simplify exploration (e.g. copy-pasting code from a function to run interactively). However, once you move to code in files, Julia requires a more disciplined approach to global variables. You have least three options:\n\nPut the code into a function (so that x is a local variable in a function). In general, it is good software engineering to use functions rather than global scripts (search online for \"why global variables bad\" to see many explanations). In Julia, global variables are also slow.\nWrap the code in a let block.  (This makes x a local variable within the let ... end statement, again eliminating the need for global).\nExplicitly mark x as global inside the local scope before assigning to it, e.g. write global x += 1.\n\nMore explanation can be found in the manual section on soft scope.","page":"Frequently Asked Questions"},{"title":"Functions","location":"manual/faq.html#Functions","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"I passed an argument x to a function, modified it inside that function, but on the outside, the variable x is still unchanged. Why?","location":"manual/faq.html#I-passed-an-argument-x-to-a-function,-modified-it-inside-that-function,-but-on-the-outside,-the-variable-x-is-still-unchanged.-Why?","category":"section","text":"Suppose you call a function like this:\n\njulia> x = 10\n10\n\njulia> function change_value!(y)\n           y = 17\n       end\nchange_value! (generic function with 1 method)\n\njulia> change_value!(x)\n17\n\njulia> x # x is unchanged!\n10\n\nIn Julia, the binding of a variable x cannot be changed by passing x as an argument to a function. When calling change_value!(x) in the above example, y is a newly created variable, bound initially to the value of x, i.e. 10; then y is rebound to the constant 17, while the variable x of the outer scope is left untouched.\n\nHowever, if x is bound to an object of type Array (or any other mutable type). From within the function, you cannot \"unbind\" x from this Array, but you can change its content. For example:\n\njulia> x = [1,2,3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> function change_array!(A)\n           A[1] = 5\n       end\nchange_array! (generic function with 1 method)\n\njulia> change_array!(x)\n5\n\njulia> x\n3-element Vector{Int64}:\n 5\n 2\n 3\n\nHere we created a function change_array!, that assigns 5 to the first element of the passed array (bound to x at the call site, and bound to A within the function). Notice that, after the function call, x is still bound to the same array, but the content of that array changed: the variables A and x were distinct bindings referring to the same mutable Array object.","page":"Frequently Asked Questions"},{"title":"Can I use using or import inside a function?","location":"manual/faq.html#Can-I-use-using-or-import-inside-a-function?","category":"section","text":"No, you are not allowed to have a using or import statement inside a function. If you want to import a module but only use its bindings inside a specific function or set of functions, you have two options:\n\nUse import:\nimport Foo\nfunction bar(...)\n    # ... refer to Foo bindings via Foo.baz ...\nend\nThis loads the module Foo and defines a variable Foo that refers to the module, but does not import any of the other bindings from the module into the current namespace. You refer to the Foo bindings by their qualified names Foo.bar etc.\nWrap your function in a module:\nmodule Bar\nexport bar\nusing Foo\nfunction bar(...)\n    # ... refer to Foo.baz as simply baz ....\nend\nend\nusing Bar\nThis imports all the bindings from Foo, but only inside the module Bar.","page":"Frequently Asked Questions"},{"title":"What does the ... operator do?","location":"manual/faq.html#What-does-the-...-operator-do?","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"The two uses of the ... operator: slurping and splatting","location":"manual/faq.html#The-two-uses-of-the-...-operator:-slurping-and-splatting","category":"section","text":"Many newcomers to Julia find the use of ... operator confusing. Part of what makes the ... operator confusing is that it means two different things depending on context.","page":"Frequently Asked Questions"},{"title":"... combines many arguments into one argument in function definitions","location":"manual/faq.html#...-combines-many-arguments-into-one-argument-in-function-definitions","category":"section","text":"In the context of function definitions, the ... operator is used to combine many different arguments into a single argument. This use of ... for combining many different arguments into a single argument is called slurping:\n\njulia> function printargs(args...)\n           println(typeof(args))\n           for (i, arg) in enumerate(args)\n               println(\"Arg #$i = $arg\")\n           end\n       end\nprintargs (generic function with 1 method)\n\njulia> printargs(1, 2, 3)\nTuple{Int64, Int64, Int64}\nArg #1 = 1\nArg #2 = 2\nArg #3 = 3\n\nIf Julia were a language that made more liberal use of ASCII characters, the slurping operator might have been written as <-... instead of ....","page":"Frequently Asked Questions"},{"title":"... splits one argument into many different arguments in function calls","location":"manual/faq.html#...-splits-one-argument-into-many-different-arguments-in-function-calls","category":"section","text":"In contrast to the use of the ... operator to denote slurping many different arguments into one argument when defining a function, the ... operator is also used to cause a single function argument to be split apart into many different arguments when used in the context of a function call. This use of ... is called splatting:\n\njulia> function threeargs(a, b, c)\n           println(\"a = $a::$(typeof(a))\")\n           println(\"b = $b::$(typeof(b))\")\n           println(\"c = $c::$(typeof(c))\")\n       end\nthreeargs (generic function with 1 method)\n\njulia> x = [1, 2, 3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> threeargs(x...)\na = 1::Int64\nb = 2::Int64\nc = 3::Int64\n\nIf Julia were a language that made more liberal use of ASCII characters, the splatting operator might have been written as ...-> instead of ....","page":"Frequently Asked Questions"},{"title":"What is the return value of an assignment?","location":"manual/faq.html#What-is-the-return-value-of-an-assignment?","category":"section","text":"The operator = always returns the right-hand side, therefore:\n\njulia> function threeint()\n           x::Int = 3.0\n           x # returns variable x\n       end\nthreeint (generic function with 1 method)\n\njulia> function threefloat()\n           x::Int = 3.0 # returns 3.0\n       end\nthreefloat (generic function with 1 method)\n\njulia> threeint()\n3\n\njulia> threefloat()\n3.0\n\nand similarly:\n\njulia> function twothreetup()\n           x, y = [2, 3] # assigns 2 to x and 3 to y\n           x, y # returns a tuple\n       end\ntwothreetup (generic function with 1 method)\n\njulia> function twothreearr()\n           x, y = [2, 3] # returns an array\n       end\ntwothreearr (generic function with 1 method)\n\njulia> twothreetup()\n(2, 3)\n\njulia> twothreearr()\n2-element Vector{Int64}:\n 2\n 3","page":"Frequently Asked Questions"},{"title":"Types, type declarations, and constructors","location":"manual/faq.html#Types,-type-declarations,-and-constructors","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What does \"type-stable\" mean?","location":"manual/faq.html#man-type-stability","category":"section","text":"It means that the type of the output is predictable from the types of the inputs. In particular, it means that the type of the output cannot vary depending on the values of the inputs. The following code is not type-stable:\n\njulia> function unstable(flag::Bool)\n           if flag\n               return 1\n           else\n               return 1.0\n           end\n       end\nunstable (generic function with 1 method)\n\nIt returns either an Int or a Float64 depending on the value of its argument. Since Julia can't predict the return type of this function at compile-time, any computation that uses it must be able to cope with values of both types, which makes it hard to produce fast machine code.","page":"Frequently Asked Questions"},{"title":"Why does Julia give a DomainError for certain seemingly-sensible operations?","location":"manual/faq.html#faq-domain-errors","category":"section","text":"Certain operations make mathematical sense but result in errors:\n\njulia> sqrt(-2.0)\nERROR: DomainError with -2.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\nThis behavior is an inconvenient consequence of the requirement for type-stability. In the case of sqrt, most users want sqrt(2.0) to give a real number, and would be unhappy if it produced the complex number 1.4142135623730951 + 0.0im. One could write the sqrt function to switch to a complex-valued output only when passed a negative number (which is what sqrt does in some other languages), but then the result would not be type-stable and the sqrt function would have poor performance.\n\nIn these and other cases, you can get the result you want by choosing an input type that conveys your willingness to accept an output type in which the result can be represented:\n\njulia> sqrt(-2.0+0im)\n0.0 + 1.4142135623730951im","page":"Frequently Asked Questions"},{"title":"How can I constrain or compute type parameters?","location":"manual/faq.html#How-can-I-constrain-or-compute-type-parameters?","category":"section","text":"The parameters of a parametric type can hold either types or bits values, and the type itself chooses how it makes use of these parameters. For example, Array{Float64, 2} is parameterized by the type Float64 to express its element type and the integer value 2 to express its number of dimensions. When defining your own parametric type, you can use subtype constraints to declare that a certain parameter must be a subtype (<:) of some abstract type or a previous type parameter. There is not, however, a dedicated syntax to declare that a parameter must be a value of a given type — that is, you cannot directly declare that a dimensionality-like parameter isa Int within the struct definition, for example. Similarly, you cannot do computations (including simple things like addition or subtraction) on type parameters. Instead, these sorts of constraints and relationships may be expressed through additional type parameters that are computed and enforced within the type's constructors.\n\nAs an example, consider\n\nstruct ConstrainedType{T,N,N+1} # NOTE: INVALID SYNTAX\n    A::Array{T,N}\n    B::Array{T,N+1}\nend\n\nwhere the user would like to enforce that the third type parameter is always the second plus one. This can be implemented with an explicit type parameter that is checked by an inner constructor method (where it can be combined with other checks):\n\nstruct ConstrainedType{T,N,M}\n    A::Array{T,N}\n    B::Array{T,M}\n    function ConstrainedType(A::Array{T,N}, B::Array{T,M}) where {T,N,M}\n        N + 1 == M || throw(ArgumentError(\"second argument should have one more axis\" ))\n        new{T,N,M}(A, B)\n    end\nend\n\nThis check is usually costless, as the compiler can elide the check for valid concrete types. If the second argument is also computed, it may be advantageous to provide an outer constructor method that performs this calculation:\n\nConstrainedType(A) = ConstrainedType(A, compute_B(A))","page":"Frequently Asked Questions"},{"title":"Why does Julia use native machine integer arithmetic?","location":"manual/faq.html#faq-integer-arithmetic","category":"section","text":"Julia uses machine arithmetic for integer computations. This means that the range of Int values is bounded and wraps around at either end so that adding, subtracting and multiplying integers can overflow or underflow, leading to some results that can be unsettling at first:\n\njulia> x = typemax(Int)\n9223372036854775807\n\njulia> y = x+1\n-9223372036854775808\n\njulia> z = -y\n-9223372036854775808\n\njulia> 2*z\n0\n\nClearly, this is far from the way mathematical integers behave, and you might think it less than ideal for a high-level programming language to expose this to the user. For numerical work where efficiency and transparency are at a premium, however, the alternatives are worse.\n\nOne alternative to consider would be to check each integer operation for overflow and promote results to bigger integer types such as Int128 or BigInt in the case of overflow. Unfortunately, this introduces major overhead on every integer operation (think incrementing a loop counter) – it requires emitting code to perform run-time overflow checks after arithmetic instructions and branches to handle potential overflows. Worse still, this would cause every computation involving integers to be type-unstable. As we mentioned above, type-stability is crucial for effective generation of efficient code. If you can't count on the results of integer operations being integers, it's impossible to generate fast, simple code the way C and Fortran compilers do.\n\nA variation on this approach, which avoids the appearance of type instability is to merge the Int and BigInt types into a single hybrid integer type, that internally changes representation when a result no longer fits into the size of a machine integer. While this superficially avoids type-instability at the level of Julia code, it just sweeps the problem under the rug by foisting all of the same difficulties onto the C code implementing this hybrid integer type. This approach can be made to work and can even be made quite fast in many cases, but has several drawbacks. One problem is that the in-memory representation of integers and arrays of integers no longer match the natural representation used by C, Fortran and other languages with native machine integers. Thus, to interoperate with those languages, we would ultimately need to introduce native integer types anyway. Any unbounded representation of integers cannot have a fixed number of bits, and thus cannot be stored inline in an array with fixed-size slots – large integer values will always require separate heap-allocated storage. And of course, no matter how clever a hybrid integer implementation one uses, there are always performance traps – situations where performance degrades unexpectedly. Complex representation, lack of interoperability with C and Fortran, the inability to represent integer arrays without additional heap storage, and unpredictable performance characteristics make even the cleverest hybrid integer implementations a poor choice for high-performance numerical work.\n\nAn alternative to using hybrid integers or promoting to BigInts is to use saturating integer arithmetic, where adding to the largest integer value leaves it unchanged and likewise for subtracting from the smallest integer value. This is precisely what Matlab™ does:\n\n>> int64(9223372036854775807)\n\nans =\n\n  9223372036854775807\n\n>> int64(9223372036854775807) + 1\n\nans =\n\n  9223372036854775807\n\n>> int64(-9223372036854775808)\n\nans =\n\n -9223372036854775808\n\n>> int64(-9223372036854775808) - 1\n\nans =\n\n -9223372036854775808\n\nAt first blush, this seems reasonable enough since 9223372036854775807 is much closer to 9223372036854775808 than -9223372036854775808 is and integers are still represented with a fixed size in a natural way that is compatible with C and Fortran. Saturated integer arithmetic, however, is deeply problematic. The first and most obvious issue is that this is not the way machine integer arithmetic works, so implementing saturated operations requires emitting instructions after each machine integer operation to check for underflow or overflow and replace the result with typemin(Int) or typemax(Int) as appropriate. This alone expands each integer operation from a single, fast instruction into half a dozen instructions, probably including branches. Ouch. But it gets worse – saturating integer arithmetic isn't associative. Consider this Matlab computation:\n\n>> n = int64(2)^62\n4611686018427387904\n\n>> n + (n - 1)\n9223372036854775807\n\n>> (n + n) - 1\n9223372036854775806\n\nThis makes it hard to write many basic integer algorithms since a lot of common techniques depend on the fact that machine addition with overflow is associative. Consider finding the midpoint between integer values lo and hi in Julia using the expression (lo + hi) >>> 1:\n\njulia> n = 2^62\n4611686018427387904\n\njulia> (n + 2n) >>> 1\n6917529027641081856\n\nSee? No problem. That's the correct midpoint between 2^62 and 2^63, despite the fact that n + 2n is -4611686018427387904. Now try it in Matlab:\n\n>> (n + 2*n)/2\n\nans =\n\n  4611686018427387904\n\nOops. Adding a >>> operator to Matlab wouldn't help, because saturation that occurs when adding n and 2n has already destroyed the information necessary to compute the correct midpoint.\n\nNot only is lack of associativity unfortunate for programmers who cannot rely it for techniques like this, but it also defeats almost anything compilers might want to do to optimize integer arithmetic. For example, since Julia integers use normal machine integer arithmetic, LLVM is free to aggressively optimize simple little functions like f(k) = 5k-1. The machine code for this function is just this:\n\njulia> code_native(f, Tuple{Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 1\n  leaq  -1(%rdi,%rdi,4), %rax\n  popq  %rbp\n  retq\n  nopl  (%rax,%rax)\n\nThe actual body of the function is a single leaq instruction, which computes the integer multiply and add at once. This is even more beneficial when f gets inlined into another function:\n\njulia> function g(k, n)\n           for i = 1:n\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 1 methods)\n\njulia> code_native(g, Tuple{Int,Int})\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 2\n  testq %rsi, %rsi\n  jle L26\n  nopl  (%rax)\nSource line: 3\nL16:\n  leaq  -1(%rdi,%rdi,4), %rdi\nSource line: 2\n  decq  %rsi\n  jne L16\nSource line: 5\nL26:\n  movq  %rdi, %rax\n  popq  %rbp\n  retq\n  nop\n\nSince the call to f gets inlined, the loop body ends up being just a single leaq instruction. Next, consider what happens if we make the number of loop iterations fixed:\n\njulia> function g(k)\n           for i = 1:10\n               k = f(k)\n           end\n           return k\n       end\ng (generic function with 2 methods)\n\njulia> code_native(g,(Int,))\n  .text\nFilename: none\n  pushq %rbp\n  movq  %rsp, %rbp\nSource line: 3\n  imulq $9765625, %rdi, %rax    # imm = 0x9502F9\n  addq  $-2441406, %rax         # imm = 0xFFDABF42\nSource line: 5\n  popq  %rbp\n  retq\n  nopw  %cs:(%rax,%rax)\n\nBecause the compiler knows that integer addition and multiplication are associative and that multiplication distributes over addition – neither of which is true of saturating arithmetic – it can optimize the entire loop down to just a multiply and an add. Saturated arithmetic completely defeats this kind of optimization since associativity and distributivity can fail at each loop iteration, causing different outcomes depending on which iteration the failure occurs in. The compiler can unroll the loop, but it cannot algebraically reduce multiple operations into fewer equivalent operations.\n\nThe most reasonable alternative to having integer arithmetic silently overflow is to do checked arithmetic everywhere, raising errors when adds, subtracts, and multiplies overflow, producing values that are not value-correct. In this blog post, Dan Luu analyzes this and finds that rather than the trivial cost that this approach should in theory have, it ends up having a substantial cost due to compilers (LLVM and GCC) not gracefully optimizing around the added overflow checks. If this improves in the future, we could consider defaulting to checked integer arithmetic in Julia, but for now, we have to live with the possibility of overflow.\n\nIn the meantime, overflow-safe integer operations can be achieved through the use of external libraries such as SaferIntegers.jl. Note that, as stated previously, the use of these libraries significantly increases the execution time of code using the checked integer types. However, for limited usage, this is far less of an issue than if it were used for all integer operations. You can follow the status of the discussion here.","page":"Frequently Asked Questions"},{"title":"What are the possible causes of an UndefVarError during remote execution?","location":"manual/faq.html#What-are-the-possible-causes-of-an-UndefVarError-during-remote-execution?","category":"section","text":"As the error states, an immediate cause of an UndefVarError on a remote node is that a binding by that name does not exist. Let us explore some of the possible causes.\n\njulia> module Foo\n           foo() = remotecall_fetch(x->x, 2, \"Hello\")\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: `Foo` not defined in `Main`\nStacktrace:\n[...]\n\nThe closure x->x carries a reference to Foo, and since Foo is unavailable on node 2, an UndefVarError is thrown.\n\nGlobals under modules other than Main are not serialized by value to the remote node. Only a reference is sent. Functions which create global bindings (except under Main) may cause an UndefVarError to be thrown later.\n\njulia> @everywhere module Foo\n           function foo()\n               global gvar = \"Hello\"\n               remotecall_fetch(()->gvar, 2)\n           end\n       end\n\njulia> Foo.foo()\nERROR: On worker 2:\nUndefVarError: `gvar` not defined in `Main.Foo`\nStacktrace:\n[...]\n\nIn the above example, @everywhere module Foo defined Foo on all nodes. However the call to Foo.foo() created a new global binding gvar on the local node, but this was not found on node 2 resulting in an UndefVarError error.\n\nNote that this does not apply to globals created under module Main. Globals under module Main are serialized and new bindings created under Main on the remote node.\n\njulia> gvar_self = \"Node1\"\n\"Node1\"\n\njulia> remotecall_fetch(()->gvar_self, 2)\n\"Node1\"\n\njulia> remotecall_fetch(varinfo, 2)\nname          size summary\n––––––––– –––––––– –––––––\nBase               Module\nCore               Module\nMain               Module\ngvar_self 13 bytes String\n\nThis does not apply to function or struct declarations. However, anonymous functions bound to global variables are serialized as can be seen below.\n\njulia> bar() = 1\nbar (generic function with 1 method)\n\njulia> remotecall_fetch(bar, 2)\nERROR: On worker 2:\nUndefVarError: `#bar` not defined in `Main`\n[...]\n\njulia> anon_bar  = ()->1\n(::#21) (generic function with 1 method)\n\njulia> remotecall_fetch(anon_bar, 2)\n1","page":"Frequently Asked Questions"},{"title":"Troubleshooting \"method not matched\": parametric type invariance and MethodErrors","location":"manual/faq.html#Troubleshooting-\"method-not-matched\":-parametric-type-invariance-and-MethodErrors","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why doesn't it work to declare foo(bar::Vector{Real}) = 42 and then call foo([1])?","location":"manual/faq.html#Why-doesn't-it-work-to-declare-foo(bar::Vector{Real})-42-and-then-call-foo([1])?","category":"section","text":"As you'll see if you try this, the result is a MethodError:\n\njulia> foo(x::Vector{Real}) = 42\nfoo (generic function with 1 method)\n\njulia> foo([1])\nERROR: MethodError: no method matching foo(::Vector{Int64})\nThe function `foo` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  foo(!Matched::Vector{Real})\n   @ Main none:1\n\nStacktrace:\n[...]\n\nThis is because Vector{Real} is not a supertype of Vector{Int}! You can solve this problem with something like foo(bar::Vector{T}) where {T<:Real} (or the short form foo(bar::Vector{<:Real}) if the static parameter T is not needed in the body of the function). The T is a wild card: you first specify that it must be a subtype of Real, then specify the function takes a Vector of with elements of that type.\n\nThis same issue goes for any composite type Comp, not just Vector. If Comp has a parameter declared of type Y, then another type Comp2 with a parameter of type X<:Y is not a subtype of Comp. This is type-invariance (by contrast, Tuple is type-covariant in its parameters). See Parametric Composite Types for more explanation of these.","page":"Frequently Asked Questions"},{"title":"Why does Julia use * for string concatenation? Why not + or something else?","location":"manual/faq.html#Why-does-Julia-use-*-for-string-concatenation?-Why-not-or-something-else?","category":"section","text":"The main argument against + is that string concatenation is not commutative, while + is generally used as a commutative operator. While the Julia community recognizes that other languages use different operators and * may be unfamiliar for some users, it communicates certain algebraic properties.\n\nNote that you can also use string(...) to concatenate strings (and other values converted to strings); similarly, repeat can be used instead of ^ to repeat strings. The interpolation syntax is also useful for constructing strings.","page":"Frequently Asked Questions"},{"title":"Packages and Modules","location":"manual/faq.html#Packages-and-Modules","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What is the difference between \"using\" and \"import\"?","location":"manual/faq.html#What-is-the-difference-between-\"using\"-and-\"import\"?","category":"section","text":"There are several differences between using and import (see the Modules section), but there is an important difference that may not seem intuitive at first glance, and on the surface (i.e. syntax-wise) it may seem very minor. When loading modules with using, you need to say function Foo.bar(... to extend module Foo's function bar with a new method, but with import Foo.bar, you only need to say function bar(... and it automatically extends module Foo's function bar.\n\nThe reason this is important enough to have been given separate syntax is that you don't want to accidentally extend a function that you didn't know existed, because that could easily cause a bug. This is most likely to happen with a method that takes a common type like a string or integer, because both you and the other module could define a method to handle such a common type. If you use import, then you'll replace the other module's implementation of bar(s::AbstractString) with your new implementation, which could easily do something completely different (and break all/many future usages of the other functions in module Foo that depend on calling bar).","page":"Frequently Asked Questions"},{"title":"Nothingness and missing values","location":"manual/faq.html#Nothingness-and-missing-values","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How does \"null\", \"nothingness\" or \"missingness\" work in Julia?","location":"manual/faq.html#faq-nothing","category":"section","text":"Unlike many languages (for example, C and Java), Julia objects cannot be \"null\" by default. When a reference (variable, object field, or array element) is uninitialized, accessing it will immediately throw an error. This situation can be detected using the isdefined or isassigned functions.\n\nSome functions are used only for their side effects, and do not need to return a value. In these cases, the convention is to return the value nothing, which is just a singleton object of type Nothing. This is an ordinary type with no fields; there is nothing special about it except for this convention, and that the REPL does not print anything for it. Some language constructs that would not otherwise have a value also yield nothing, for example if false; end.\n\nFor situations where a value x of type T exists only sometimes, the Union{T, Nothing} type can be used for function arguments, object fields and array element types as the equivalent of Nullable, Option or Maybe in other languages. If the value itself can be nothing (notably, when T is Any), the Union{Some{T}, Nothing} type is more appropriate since x == nothing then indicates the absence of a value, and x == Some(nothing) indicates the presence of a value equal to nothing. The something function allows unwrapping Some objects and using a default value instead of nothing arguments. Note that the compiler is able to generate efficient code when working with Union{T, Nothing} arguments or fields.\n\nTo represent missing data in the statistical sense (NA in R or NULL in SQL), use the missing object. See the Missing Values section for more details.\n\nIn some languages, the empty tuple (()) is considered the canonical form of nothingness. However, in julia it is best thought of as just a regular tuple that happens to contain zero values.\n\nThe empty (or \"bottom\") type, written as Union{} (an empty union type), is a type with no values and no subtypes (except itself). You will generally not need to use this type.","page":"Frequently Asked Questions"},{"title":"Memory","location":"manual/faq.html#Memory","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why does x += y allocate memory when x and y are arrays?","location":"manual/faq.html#Why-does-x-y-allocate-memory-when-x-and-y-are-arrays?","category":"section","text":"In Julia, x += y gets replaced during lowering by x = x + y. For arrays, this has the consequence that, rather than storing the result in the same location in memory as x, it allocates a new array to store the result. If you prefer to mutate x, use x .+= y to update each element individually.\n\nWhile this behavior might surprise some, the choice is deliberate. The main reason is the presence of immutable objects within Julia, which cannot change their value once created. Indeed, a number is an immutable object; the statements x = 5; x += 1 do not modify the meaning of 5, they modify the value bound to x. For an immutable, the only way to change the value is to reassign it.\n\nTo amplify a bit further, consider the following function:\n\nfunction power_by_squaring(x, n::Int)\n    ispow2(n) || error(\"This implementation only works for powers of 2\")\n    while n >= 2\n        x *= x\n        n >>= 1\n    end\n    x\nend\n\nAfter a call like x = 5; y = power_by_squaring(x, 4), you would get the expected result: x == 5 && y == 625.  However, now suppose that *=, when used with matrices, instead mutated the left hand side.  There would be two problems:\n\nFor general square matrices, A = A*B cannot be implemented without temporary storage: A[1,1] gets computed and stored on the left hand side before you're done using it on the right hand side.\nSuppose you were willing to allocate a temporary for the computation (which would eliminate most of the point of making *= work in-place); if you took advantage of the mutability of x, then this function would behave differently for mutable vs. immutable inputs. In particular, for immutable x, after the call you'd have (in general) y != x, but for mutable x you'd have y == x.\n\nBecause supporting generic programming is deemed more important than potential performance optimizations that can be achieved by other means (e.g., using broadcasting or explicit loops), operators like += and *= work by rebinding new values.","page":"Frequently Asked Questions"},{"title":"Asynchronous IO and concurrent synchronous writes","location":"manual/faq.html#faq-async-io","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Why do concurrent writes to the same stream result in inter-mixed output?","location":"manual/faq.html#Why-do-concurrent-writes-to-the-same-stream-result-in-inter-mixed-output?","category":"section","text":"While the streaming I/O API is synchronous, the underlying implementation is fully asynchronous.\n\nConsider the printed output from the following:\n\njulia> @sync for i in 1:3\n           Threads.@spawn write(stdout, string(i), \" Foo \", \" Bar \")\n       end\n123 Foo  Foo  Foo  Bar  Bar  Bar\n\nThis is happening because, while the write call is synchronous, the writing of each argument yields to other tasks while waiting for that part of the I/O to complete.\n\nprint and println \"lock\" the stream during a call. Consequently changing write to println in the above example results in:\n\njulia> @sync for i in 1:3\n           Threads.@spawn println(stdout, string(i), \" Foo \", \" Bar \")\n       end\n1 Foo  Bar\n2 Foo  Bar\n3 Foo  Bar\n\nYou can lock your writes with a ReentrantLock like this:\n\njulia> l = ReentrantLock();\n\njulia> @sync for i in 1:3\n           Threads.@spawn begin\n               lock(l)\n               try\n                   write(stdout, string(i), \" Foo \", \" Bar \")\n               finally\n                   unlock(l)\n               end\n           end\n       end\n1 Foo  Bar 2 Foo  Bar 3 Foo  Bar","page":"Frequently Asked Questions"},{"title":"Arrays","location":"manual/faq.html#Arrays","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"What are the differences between zero-dimensional arrays and scalars?","location":"manual/faq.html#faq-array-0dim","category":"section","text":"Zero-dimensional arrays are arrays of the form Array{T,0}. They behave similar to scalars, but there are important differences. They deserve a special mention because they are a special case which makes logical sense given the generic definition of arrays, but might be a bit unintuitive at first. The following line defines a zero-dimensional array:\n\njulia> A = zeros()\n0-dimensional Array{Float64,0}:\n0.0\n\nIn this example, A is a mutable container that contains one element, which can be set by A[] = 1.0 and retrieved with A[]. All zero-dimensional arrays have the same size (size(A) == ()), and length (length(A) == 1). In particular, zero-dimensional arrays are not empty. If you find this unintuitive, here are some ideas that might help to understand Julia's definition.\n\nZero-dimensional arrays are the \"point\" to vector's \"line\" and matrix's \"plane\". Just as a line has no area (but still represents a set of things), a point has no length or any dimensions at all (but still represents a thing).\nWe define prod(()) to be 1, and the total number of elements in an array is the product of the size. The size of a zero-dimensional array is (), and therefore its length is 1.\nZero-dimensional arrays don't natively have any dimensions into which you index – they’re just A[]. We can apply the same \"trailing one\" rule for them as for all other array dimensionalities, so you can indeed index them as A[1], A[1,1], etc; see Omitted and extra indices.\n\nIt is also important to understand the differences to ordinary scalars. Scalars are not mutable containers (even though they are iterable and define things like length, getindex, e.g. 1[] == 1). In particular, if x = 0.0 is defined as a scalar, it is an error to attempt to change its value via x[] = 1.0. A scalar x can be converted into a zero-dimensional array containing it via fill(x), and conversely, a zero-dimensional array a can be converted to the contained scalar via a[]. Another difference is that a scalar can participate in linear algebra operations such as 2 * rand(2,2), but the analogous operation with a zero-dimensional array fill(2) * rand(2,2) is an error.","page":"Frequently Asked Questions"},{"title":"Why are my Julia benchmarks for linear algebra operations different from other languages?","location":"manual/faq.html#Why-are-my-Julia-benchmarks-for-linear-algebra-operations-different-from-other-languages?","category":"section","text":"You may find that simple benchmarks of linear algebra building blocks like\n\nusing BenchmarkTools\nA = randn(1000, 1000)\nB = randn(1000, 1000)\n@btime $A \\ $B\n@btime $A * $B\n\ncan be different when compared to other languages like Matlab or R.\n\nSince operations like this are very thin wrappers over the relevant BLAS functions, the reason for the discrepancy is very likely to be\n\nthe BLAS library each language is using,\nthe number of concurrent threads.\n\nJulia compiles and uses its own copy of OpenBLAS, with threads currently capped at 8 (or the number of your cores).\n\nModifying OpenBLAS settings or compiling Julia with a different BLAS library, eg Intel MKL, may provide performance improvements. You can use MKL.jl, a package that makes Julia's linear algebra use Intel MKL BLAS and LAPACK instead of OpenBLAS, or search the discussion forum for suggestions on how to set this up manually. Note that Intel MKL cannot be bundled with Julia, as it is not open source.","page":"Frequently Asked Questions"},{"title":"Computing cluster","location":"manual/faq.html#Computing-cluster","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"How do I manage precompilation caches in distributed file systems?","location":"manual/faq.html#How-do-I-manage-precompilation-caches-in-distributed-file-systems?","category":"section","text":"When using Julia in high-performance computing (HPC) facilities with shared filesystems, it is recommended to use a shared depot (via the JULIA_DEPOT_PATH environment variable). Since Julia v1.10, multiple Julia processes on functionally similar workers and using the same depot will coordinate via pidfile locks to only spend effort precompiling on one process while the others wait. The precompilation process will indicate when the process is precompiling or waiting for another that is precompiling. If non-interactive the messages are via @debug.\n\nHowever, due to caching of binary code, the cache rejection since v1.9 is more strict and users may need to set the JULIA_CPU_TARGET environment variable appropriately to get a single cache that is usable throughout the HPC environment.","page":"Frequently Asked Questions"},{"title":"Julia Releases","location":"manual/faq.html#Julia-Releases","category":"section","text":"","page":"Frequently Asked Questions"},{"title":"Do I want to use the Stable, LTS, or nightly version of Julia?","location":"manual/faq.html#Do-I-want-to-use-the-Stable,-LTS,-or-nightly-version-of-Julia?","category":"section","text":"The Stable version of Julia is the latest released version of Julia, this is the version most people will want to run. It has the latest features, including improved performance. The Stable version of Julia is versioned according to SemVer as v1.x.y. A new minor release of Julia corresponding to a new Stable version is made approximately every 4-5 months after a few weeks of testing as a release candidate. Unlike the LTS version the Stable version will not normally receive bugfixes after another Stable version of Julia has been released. However, upgrading to the next Stable release will always be possible as each release of Julia v1.x will continue to run code written for earlier versions.\n\nYou may prefer the LTS (Long Term Support) version of Julia if you are looking for a very stable code base. The current LTS version of Julia is versioned according to SemVer as v1.6.x; this branch will continue to receive bugfixes until a new LTS branch is chosen, at which point the v1.6.x series will no longer received regular bug fixes and all but the most conservative users will be advised to upgrade to the new LTS version series. As a package developer, you may prefer to develop for the LTS version, to maximize the number of users who can use your package. As per SemVer, code written for v1.0 will continue to work for all future LTS and Stable versions. In general, even if targeting the LTS, one can develop and run code in the latest Stable version, to take advantage of the improved performance; so long as one avoids using new features (such as added library functions or new methods).\n\nYou may prefer the nightly version of Julia if you want to take advantage of the latest updates to the language, and don't mind if the version available today occasionally doesn't actually work. As the name implies, releases to the nightly version are made roughly every night (depending on build infrastructure stability). In general nightly released are fairly safe to use—your code will not catch on fire. However, they may be occasional regressions and or issues that will not be found until more thorough pre-release testing. You may wish to test against the nightly version to ensure that such regressions that affect your use case are caught before a release is made.\n\nFinally, you may also consider building Julia from source for yourself. This option is mainly for those individuals who are comfortable at the command line, or interested in learning. If this describes you, you may also be interested in reading our guidelines for contributing.\n\nThe juliaup install manager has pre-defined channels named release and lts for the latest stable release and the current LTS release, as well as version-specific channels.","page":"Frequently Asked Questions"},{"title":"How can I transfer the list of installed packages after updating my version of Julia?","location":"manual/faq.html#How-can-I-transfer-the-list-of-installed-packages-after-updating-my-version-of-Julia?","category":"section","text":"Each minor version of julia has its own default environment. As a result, upon installing a new minor version of Julia, the packages you added using the previous minor version will not be available by default. The environment for a given julia version is defined by the files Project.toml and Manifest.toml in a folder matching the version number in .julia/environments/, for instance, .julia/environments/v1.3.\n\nIf you install a new minor version of Julia, say 1.4, and want to use in its default environment the same packages as in a previous version (e.g. 1.3), you can copy the contents of the file Project.toml from the 1.3 folder to 1.4. Then, in a session of the new Julia version, enter the \"package management mode\" by typing the key ], and run the command instantiate.\n\nThis operation will resolve a set of feasible packages from the copied file that are compatible with the target Julia version, and will install or update them if suitable. If you want to reproduce not only the set of packages, but also the versions you were using in the previous Julia version, you should also copy the Manifest.toml file before running the Pkg command instantiate. However, note that packages may define compatibility constraints that may be affected by changing the version of Julia, so the exact set of versions you had in 1.3 may not work for 1.4.","page":"Frequently Asked Questions"},{"title":"Metaprogramming","location":"manual/metaprogramming.html#Metaprogramming","category":"section","text":"The strongest legacy of Lisp in the Julia language is its metaprogramming support. Like Lisp, Julia represents its own code as a data structure of the language itself. Since code is represented by objects that can be created and manipulated from within the language, it is possible for a program to transform and generate its own code. This allows sophisticated code generation without extra build steps, and also allows true Lisp-style macros operating at the level of abstract syntax trees. In contrast, preprocessor \"macro\" systems, like that of C and C++, perform textual manipulation and substitution before any actual parsing or interpretation occurs. Because all data types and code in Julia are represented by Julia data structures, powerful reflection capabilities are available to explore the internals of a program and its types just like any other data.\n\nwarning: Warning\nMetaprogramming is a powerful tool, but it introduces complexity that can make code more difficult to understand. For example, it can be surprisingly hard to get scope rules correct. Metaprogramming should typically be used only when other approaches such as higher order functions and closures cannot be applied.eval and defining new macros should be typically used as a last resort. It is almost never a good idea to use Meta.parse or convert an arbitrary string into Julia code. For manipulating Julia code, use the Expr data structure directly to avoid the complexity of how Julia syntax is parsed.The best uses of metaprogramming often implement most of their functionality in runtime helper functions, striving to minimize the amount of code they generate.","page":"Metaprogramming"},{"title":"Program representation","location":"manual/metaprogramming.html#Program-representation","category":"section","text":"Every Julia program starts life as a string:\n\njulia> prog = \"1 + 1\"\n\"1 + 1\"\n\nWhat happens next?\n\nThe next step is to parse each string into an object called an expression, represented by the Julia type Expr:\n\njulia> ex1 = Meta.parse(prog)\n:(1 + 1)\n\njulia> typeof(ex1)\nExpr\n\nExpr objects contain two parts:\n\na Symbol identifying the kind of expression. A symbol is an interned string identifier (more discussion below).\n\njulia> ex1.head\n:call\n\nthe expression arguments, which may be symbols, other expressions, or literal values:\n\njulia> ex1.args\n3-element Vector{Any}:\n  :+\n 1\n 1\n\nExpressions may also be constructed directly in prefix notation:\n\njulia> ex2 = Expr(:call, :+, 1, 1)\n:(1 + 1)\n\nThe two expressions constructed above – by parsing and by direct construction – are equivalent:\n\njulia> ex1 == ex2\ntrue\n\nThe key point here is that Julia code is internally represented as a data structure that is accessible from the language itself.\n\nThe dump function provides indented and annotated display of Expr objects:\n\njulia> dump(ex2)\nExpr\n  head: Symbol call\n  args: Array{Any}((3,))\n    1: Symbol +\n    2: Int64 1\n    3: Int64 1\n\nExpr objects may also be nested:\n\njulia> ex3 = Meta.parse(\"(4 + 4) / 2\")\n:((4 + 4) / 2)\n\nAnother way to view expressions is with Meta.show_sexpr, which displays the S-expression form of a given Expr, which may look very familiar to users of Lisp. Here's an example illustrating the display on a nested Expr:\n\njulia> Meta.show_sexpr(ex3)\n(:call, :/, (:call, :+, 4, 4), 2)","page":"Metaprogramming"},{"title":"Symbols","location":"manual/metaprogramming.html#Symbols","category":"section","text":"The : character has two syntactic purposes in Julia. The first form creates a Symbol, an interned string used as one building-block of expressions, from valid identifiers:\n\njulia> s = :foo\n:foo\n\njulia> typeof(s)\nSymbol\n\nThe Symbol constructor takes any number of arguments and creates a new symbol by concatenating their string representations together:\n\njulia> :foo === Symbol(\"foo\")\ntrue\n\njulia> Symbol(\"1foo\") # `:1foo` would not work, as `1foo` is not a valid identifier\nSymbol(\"1foo\")\n\njulia> Symbol(\"func\",10)\n:func10\n\njulia> Symbol(:var,'_',\"sym\")\n:var_sym\n\nIn the context of an expression, symbols are used to indicate access to variables; when an expression is evaluated, a symbol is replaced with the value bound to that symbol in the appropriate scope.\n\nSometimes extra parentheses around the argument to : are needed to avoid ambiguity in parsing:\n\njulia> :(:)\n:(:)\n\njulia> :(::)\n:(::)","page":"Metaprogramming"},{"title":"Expressions and evaluation","location":"manual/metaprogramming.html#Expressions-and-evaluation","category":"section","text":"","page":"Metaprogramming"},{"title":"Quoting","location":"manual/metaprogramming.html#Quoting","category":"section","text":"The second syntactic purpose of the : character is to create expression objects without using the explicit Expr constructor. This is referred to as quoting. The : character, followed by paired parentheses around a single statement of Julia code, produces an Expr object based on the enclosed code. Here is an example of the short form used to quote an arithmetic expression:\n\njulia> ex = :(a+b*c+1)\n:(a + b * c + 1)\n\njulia> typeof(ex)\nExpr\n\n(to view the structure of this expression, try ex.head and ex.args, or use dump as above or Meta.@dump)\n\nNote that equivalent expressions may be constructed using Meta.parse or the direct Expr form:\n\njulia>      :(a + b*c + 1)       ==\n       Meta.parse(\"a + b*c + 1\") ==\n       Expr(:call, :+, :a, Expr(:call, :*, :b, :c), 1)\ntrue\n\nExpressions provided by the parser generally only have symbols, other expressions, and literal values as their args, whereas expressions constructed by Julia code can have arbitrary run-time values without literal forms as args. In this specific example, + and a are symbols, *(b,c) is a subexpression, and 1 is a literal 64-bit signed integer.\n\nThere is a second syntactic form of quoting for multiple expressions: blocks of code enclosed in quote ... end.\n\njulia> ex = quote\n           x = 1\n           y = 2\n           x + y\n       end\nquote\n    #= none:2 =#\n    x = 1\n    #= none:3 =#\n    y = 2\n    #= none:4 =#\n    x + y\nend\n\njulia> typeof(ex)\nExpr","page":"Metaprogramming"},{"title":"Interpolation","location":"manual/metaprogramming.html#man-expression-interpolation","category":"section","text":"Direct construction of Expr objects with value arguments is powerful, but Expr constructors can be tedious compared to \"normal\" Julia syntax. As an alternative, Julia allows interpolation of literals or expressions into quoted expressions. Interpolation is indicated by a prefix $.\n\nIn this example, the value of variable a is interpolated:\n\njulia> a = 1;\n\njulia> ex = :($a + b)\n:(1 + b)\n\nInterpolating into an unquoted expression is not supported and will cause a compile-time error:\n\njulia> $a + b\nERROR: syntax: \"$\" expression outside quote\n\nIn this example, the tuple (1,2,3) is interpolated as an expression into a conditional test:\n\njulia> ex = :(a in $:((1,2,3)) )\n:(a in (1, 2, 3))\n\nThe use of $ for expression interpolation is intentionally reminiscent of string interpolation and command interpolation. Expression interpolation allows convenient, readable programmatic construction of complex Julia expressions.","page":"Metaprogramming"},{"title":"Splatting interpolation","location":"manual/metaprogramming.html#Splatting-interpolation","category":"section","text":"Notice that the $ interpolation syntax allows inserting only a single expression into an enclosing expression. Occasionally, you have an array of expressions and need them all to become arguments of the surrounding expression. This can be done with the syntax $(xs...). For example, the following code generates a function call where the number of arguments is determined programmatically:\n\njulia> args = [:x, :y, :z];\n\njulia> :(f(1, $(args...)))\n:(f(1, x, y, z))","page":"Metaprogramming"},{"title":"Nested quote","location":"manual/metaprogramming.html#Nested-quote","category":"section","text":"Naturally, it is possible for quote expressions to contain other quote expressions. Understanding how interpolation works in these cases can be a bit tricky. Consider this example:\n\njulia> x = :(1 + 2);\n\njulia> e = quote quote $x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :x))\nend))\nend\n\nNotice that the result contains $x, which means that x has not been evaluated yet. In other words, the $ expression \"belongs to\" the inner quote expression, and so its argument is only evaluated when the inner quote expression is:\n\njulia> eval(e)\nquote\n    #= none:1 =#\n    1 + 2\nend\n\nHowever, the outer quote expression is able to interpolate values inside the $ in the inner quote. This is done with multiple $s:\n\njulia> e = quote quote $$x end end\nquote\n    #= none:1 =#\n    $(Expr(:quote, quote\n    #= none:1 =#\n    $(Expr(:$, :(1 + 2)))\nend))\nend\n\nNotice that (1 + 2) now appears in the result instead of the symbol x. Evaluating this expression yields an interpolated 3:\n\njulia> eval(e)\nquote\n    #= none:1 =#\n    3\nend\n\nThe intuition behind this behavior is that x is evaluated once for each $: one $ works similarly to eval(:x), giving x's value, while two $s do the equivalent of eval(eval(:x)).","page":"Metaprogramming"},{"title":"QuoteNode","location":"manual/metaprogramming.html#man-quote-node","category":"section","text":"The usual representation of a quote form in an AST is an Expr with head :quote:\n\njulia> dump(Meta.parse(\":(1+2)\"))\nExpr\n  head: Symbol quote\n  args: Array{Any}((1,))\n    1: Expr\n      head: Symbol call\n      args: Array{Any}((3,))\n        1: Symbol +\n        2: Int64 1\n        3: Int64 2\n\nAs we have seen, such expressions support interpolation with $. However, in some situations it is necessary to quote code without performing interpolation. This kind of quoting does not yet have syntax, but is represented internally as an object of type QuoteNode:\n\njulia> eval(Meta.quot(Expr(:$, :(1+2))))\n3\n\njulia> eval(QuoteNode(Expr(:$, :(1+2))))\n:($(Expr(:$, :(1 + 2))))\n\nThe parser yields QuoteNodes for simple quoted items like symbols:\n\njulia> dump(Meta.parse(\":x\"))\nQuoteNode\n  value: Symbol x\n\nQuoteNode can also be used for certain advanced metaprogramming tasks.","page":"Metaprogramming"},{"title":"Evaluating expressions","location":"manual/metaprogramming.html#Evaluating-expressions","category":"section","text":"Given an expression object, one can cause Julia to evaluate (execute) it at global scope using eval:\n\njulia> ex1 = :(1 + 2)\n:(1 + 2)\n\njulia> eval(ex1)\n3\n\njulia> ex = :(a + b)\n:(a + b)\n\njulia> eval(ex)\nERROR: UndefVarError: `b` not defined in `Main`\n[...]\n\njulia> a = 1; b = 2;\n\njulia> eval(ex)\n3\n\nEvery module has its own eval function that evaluates expressions in its global scope. Expressions passed to eval are not limited to returning values – they can also have side-effects that alter the state of the enclosing module's environment:\n\njulia> ex = :(x = 1)\n:(x = 1)\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\njulia> eval(ex)\n1\n\njulia> x\n1\n\nHere, the evaluation of an expression object causes a value to be assigned to the global variable x.\n\nSince expressions are just Expr objects which can be constructed programmatically and then evaluated, it is possible to dynamically generate arbitrary code which can then be run using eval. Here is a simple example:\n\njulia> a = 1;\n\njulia> ex = Expr(:call, :+, a, :b)\n:(1 + b)\n\njulia> a = 0; b = 2;\n\njulia> eval(ex)\n3\n\nThe value of a is used to construct the expression ex which applies the + function to the value 1 and the variable b. Note the important distinction between the way a and b are used:\n\nThe value of the variable a at expression construction time is used as an immediate value in the expression. Thus, the value of a when the expression is evaluated no longer matters: the value in the expression is already 1, independent of whatever the value of a might be.\nOn the other hand, the symbol :b is used in the expression construction, so the value of the variable b at that time is irrelevant – :b is just a symbol and the variable b need not even be defined. At expression evaluation time, however, the value of the symbol :b is resolved by looking up the value of the variable b.","page":"Metaprogramming"},{"title":"Functions on Expressions","location":"manual/metaprogramming.html#Functions-on-Expressions","category":"section","text":"As hinted above, one extremely useful feature of Julia is the capability to generate and manipulate Julia code within Julia itself. We have already seen one example of a function returning Expr objects: the Meta.parse function, which takes a string of Julia code and returns the corresponding Expr. A function can also take one or more Expr objects as arguments, and return another Expr. Here is a simple, motivating example:\n\njulia> function math_expr(op, op1, op2)\n           expr = Expr(:call, op, op1, op2)\n           return expr\n       end\nmath_expr (generic function with 1 method)\n\njulia>  ex = math_expr(:+, 1, Expr(:call, :*, 4, 5))\n:(1 + 4 * 5)\n\njulia> eval(ex)\n21\n\nAs another example, here is a function that doubles any numeric argument, but leaves expressions alone:\n\njulia> function make_expr2(op, opr1, opr2)\n           opr1f, opr2f = map(x -> isa(x, Number) ? 2*x : x, (opr1, opr2))\n           retexpr = Expr(:call, op, opr1f, opr2f)\n           return retexpr\n       end\nmake_expr2 (generic function with 1 method)\n\njulia> make_expr2(:+, 1, 2)\n:(2 + 4)\n\njulia> ex = make_expr2(:+, 1, Expr(:call, :*, 5, 8))\n:(2 + 5 * 8)\n\njulia> eval(ex)\n42","page":"Metaprogramming"},{"title":"Macros","location":"manual/metaprogramming.html#man-macros","category":"section","text":"Macros provide a mechanism to include generated code in the final body of a program. A macro maps a tuple of arguments to a returned expression, and the resulting expression is compiled directly rather than requiring a runtime eval call. Macro arguments may include expressions, literal values, and symbols.","page":"Metaprogramming"},{"title":"Basics","location":"manual/metaprogramming.html#Basics","category":"section","text":"Here is an extraordinarily simple macro:\n\njulia> macro sayhello()\n           return :( println(\"Hello, world!\") )\n       end\n@sayhello (macro with 1 method)\n\nMacros have a dedicated character in Julia's syntax: the @ (at-sign), followed by the unique name declared in a macro NAME ... end block. In this example, the compiler will replace all instances of @sayhello with:\n\n:( println(\"Hello, world!\") )\n\nWhen @sayhello is entered in the REPL, the expression executes immediately, thus we only see the evaluation result:\n\njulia> @sayhello()\nHello, world!\n\nNow, consider a slightly more complex macro:\n\njulia> macro sayhello(name)\n           return :( println(\"Hello, \", $name) )\n       end\n@sayhello (macro with 1 method)\n\nThis macro takes one argument: name. When @sayhello is encountered, the quoted expression is expanded to interpolate the value of the argument into the final expression:\n\njulia> @sayhello(\"human\")\nHello, human\n\nWe can view the quoted return expression using the function macroexpand (important note: this is an extremely useful tool for debugging macros):\n\njulia> ex = macroexpand(Main, :(@sayhello(\"human\")) )\n:(Main.println(\"Hello, \", \"human\"))\n\njulia> typeof(ex)\nExpr\n\nWe can see that the \"human\" literal has been interpolated into the expression.\n\nThere also exists a macro @macroexpand that is perhaps a bit more convenient than the macroexpand function:\n\njulia> @macroexpand @sayhello \"human\"\n:(println(\"Hello, \", \"human\"))","page":"Metaprogramming"},{"title":"Hold up: why macros?","location":"manual/metaprogramming.html#Hold-up:-why-macros?","category":"section","text":"We have already seen a function f(::Expr...) -> Expr in a previous section. In fact, macroexpand is also such a function. So, why do macros exist?\n\nMacros are necessary because they execute when code is parsed, therefore, macros allow the programmer to generate and include fragments of customized code before the full program is run. To illustrate the difference, consider the following example:\n\njulia> macro twostep(arg)\n           println(\"I execute at parse time. The argument is: \", arg)\n           return :(println(\"I execute at runtime. The argument is: \", $arg))\n       end\n@twostep (macro with 1 method)\n\njulia> ex = macroexpand(Main, :(@twostep :(1, 2, 3)) );\nI execute at parse time. The argument is: :((1, 2, 3))\n\nThe first call to println is executed when macroexpand is called. The resulting expression contains only the second println:\n\njulia> typeof(ex)\nExpr\n\njulia> ex\n:(println(\"I execute at runtime. The argument is: \", $(Expr(:copyast, :($(QuoteNode(:((1, 2, 3)))))))))\n\njulia> eval(ex)\nI execute at runtime. The argument is: (1, 2, 3)","page":"Metaprogramming"},{"title":"Macro invocation","location":"manual/metaprogramming.html#Macro-invocation","category":"section","text":"Macros are invoked with the following general syntax:\n\n@name expr1 expr2 ...\n@name(expr1, expr2, ...)\n\nNote the distinguishing @ before the macro name and the lack of commas between the argument expressions in the first form, and the lack of whitespace after @name in the second form. The two styles should not be mixed. For example, the following syntax is different from the examples above; it passes the tuple (expr1, expr2, ...) as one argument to the macro:\n\n@name (expr1, expr2, ...)\n\nAn alternative way to invoke a macro over an array literal (or comprehension) is to juxtapose both without using parentheses. In this case, the array will be the only expression fed to the macro. The following syntax is equivalent (and different from @name [a b] * v):\n\n@name[a b] * v\n@name([a b]) * v\n\nIt is important to emphasize that macros receive their arguments as expressions, literals, or symbols. One way to explore macro arguments is to call the show function within the macro body:\n\njulia> macro showarg(x)\n           show(x)\n           # ... remainder of macro, returning an expression\n       end\n@showarg (macro with 1 method)\n\njulia> @showarg(a)\n:a\n\njulia> @showarg(1+1)\n:(1 + 1)\n\njulia> @showarg(println(\"Yo!\"))\n:(println(\"Yo!\"))\n\njulia> @showarg(1)        # Numeric literal\n1\n\njulia> @showarg(\"Yo!\")    # String literal\n\"Yo!\"\n\njulia> @showarg(\"Yo! $(\"hello\")\")    # String with interpolation is an Expr rather than a String\n:(\"Yo! $(\"hello\")\")\n\nIn addition to the given argument list, every macro is passed extra arguments named __source__ and __module__.\n\nThe argument __source__ provides information (in the form of a LineNumberNode object) about the parser location of the @ sign from the macro invocation. This allows macros to include better error diagnostic information, and is commonly used by logging, string-parser macros, and docs, for example, as well as to implement the @__LINE__, @__FILE__, and @__DIR__ macros.\n\nThe location information can be accessed by referencing __source__.line and __source__.file:\n\njulia> macro __LOCATION__(); return QuoteNode(__source__); end\n@__LOCATION__ (macro with 1 method)\n\njulia> dump(\n            @__LOCATION__(\n       ))\nLineNumberNode\n  line: Int64 2\n  file: Symbol none\n\nThe argument __module__ provides information (in the form of a Module object) about the expansion context of the macro invocation. This allows macros to look up contextual information, such as existing bindings, or to insert the value as an extra argument to a runtime function call doing self-reflection in the current module.","page":"Metaprogramming"},{"title":"Building an advanced macro","location":"manual/metaprogramming.html#Building-an-advanced-macro","category":"section","text":"Here is a simplified definition of Julia's @assert macro:\n\njulia> macro assert(ex)\n           return :( $ex ? nothing : throw(AssertionError($(string(ex)))) )\n       end\n@assert (macro with 1 method)\n\nThis macro can be used like this:\n\njulia> @assert 1 == 1.0\n\njulia> @assert 1 == 0\nERROR: AssertionError: 1 == 0\n\nIn place of the written syntax, the macro call is expanded at parse time to its returned result. This is equivalent to writing:\n\n1 == 1.0 ? nothing : throw(AssertionError(\"1 == 1.0\"))\n1 == 0 ? nothing : throw(AssertionError(\"1 == 0\"))\n\nThat is, in the first call, the expression :(1 == 1.0) is spliced into the test condition slot, while the value of string(:(1 == 1.0)) is spliced into the assertion message slot. The entire expression, thus constructed, is placed into the syntax tree where the @assert macro call occurs. Then at execution time, if the test expression evaluates to true, then nothing is returned, whereas if the test is false, an error is raised indicating the asserted expression that was false. Notice that it would not be possible to write this as a function, since only the value of the condition is available and it would be impossible to display the expression that computed it in the error message.\n\nThe actual definition of @assert in Julia Base is more complicated. It allows the user to optionally specify their own error message, instead of just printing the failed expression. Just like in functions with a variable number of arguments (Varargs Functions), this is specified with an ellipses following the last argument:\n\njulia> macro assert(ex, msgs...)\n           msg_body = isempty(msgs) ? ex : msgs[1]\n           msg = string(msg_body)\n           return :($ex ? nothing : throw(AssertionError($msg)))\n       end\n@assert (macro with 1 method)\n\nNow @assert has two modes of operation, depending upon the number of arguments it receives! If there's only one argument, the tuple of expressions captured by msgs will be empty and it will behave the same as the simpler definition above. But now if the user specifies a second argument, it is printed in the message body instead of the failing expression. You can inspect the result of a macro expansion with the aptly named @macroexpand macro:\n\njulia> @macroexpand @assert a == b\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a == b\"))\n    end)\n\njulia> @macroexpand @assert a==b \"a should equal b!\"\n:(if Main.a == Main.b\n        Main.nothing\n    else\n        Main.throw(Main.AssertionError(\"a should equal b!\"))\n    end)\n\nThere is yet another case that the actual @assert macro handles: what if, in addition to printing \"a should equal b,\" we wanted to print their values? One might naively try to use string interpolation in the custom message, e.g., @assert a==b \"a ($a) should equal b ($b)!\", but this won't work as expected with the above macro. Can you see why? Recall from string interpolation that an interpolated string is rewritten to a call to string. Compare:\n\njulia> typeof(:(\"a should equal b\"))\nString\n\njulia> typeof(:(\"a ($a) should equal b ($b)!\"))\nExpr\n\njulia> dump(:(\"a ($a) should equal b ($b)!\"))\nExpr\n  head: Symbol string\n  args: Array{Any}((5,))\n    1: String \"a (\"\n    2: Symbol a\n    3: String \") should equal b (\"\n    4: Symbol b\n    5: String \")!\"\n\nSo now instead of getting a plain string in msg_body, the macro is receiving a full expression that will need to be evaluated in order to display as expected. This can be spliced directly into the returned expression as an argument to the string call; see error.jl for the complete implementation.\n\nThe @assert macro makes great use of splicing into quoted expressions to simplify the manipulation of expressions inside the macro body.","page":"Metaprogramming"},{"title":"Hygiene","location":"manual/metaprogramming.html#Hygiene","category":"section","text":"An issue that arises in more complex macros is that of hygiene. In short, macros must ensure that the variables they introduce in their returned expressions do not accidentally clash with existing variables in the surrounding code they expand into. Conversely, the expressions that are passed into a macro as arguments are often expected to evaluate in the context of the surrounding code, interacting with and modifying the existing variables. Another concern arises from the fact that a macro may be called in a different module from where it was defined. In this case we need to ensure that all global variables are resolved to the correct module. Julia already has a major advantage over languages with textual macro expansion (like C) in that it only needs to consider the returned expression. All the other variables (such as msg in @assert above) follow the normal scoping block behavior.\n\nTo demonstrate these issues, let us consider writing a @time macro that takes an expression as its argument, records the time, evaluates the expression, records the time again, prints the difference between the before and after times, and then has the value of the expression as its final value. The macro might look like this:\n\nmacro time(ex)\n    return quote\n        local t0 = time_ns()\n        local val = $ex\n        local t1 = time_ns()\n        println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n        val\n    end\nend\n\nHere, we want t0, t1, and val to be private temporary variables, and we want time_ns to refer to the time_ns function in Julia Base, not to any time_ns variable the user might have (the same applies to println). Imagine the problems that could occur if the user expression ex also contained assignments to a variable called t0, or defined its own time_ns variable. We might get errors, or mysteriously incorrect behavior.\n\nJulia's macro expander solves these problems in the following way. First, variables within a macro result are classified as either local or global. A variable is considered local if it is assigned to (and not declared global), declared local, or used as a function argument name. Otherwise, it is considered global. Local variables are then renamed to be unique (using the gensym function, which generates new symbols), and global variables are resolved within the macro definition environment. Therefore both of the above concerns are handled; the macro's locals will not conflict with any user variables, and time_ns and println will refer to the Julia Base definitions.\n\nOne problem remains however. Consider the following use of this macro:\n\nmodule MyModule\nimport Base.@time\n\ntime_ns() = ... # compute something\n\n@time time_ns()\nend\n\nHere the user expression ex is a call to time_ns, but not the same time_ns function that the macro uses. It clearly refers to MyModule.time_ns. Therefore we must arrange for the code in ex to be resolved in the macro call environment. This is done by \"escaping\" the expression with esc:\n\nmacro time(ex)\n    ...\n    local val = $(esc(ex))\n    ...\nend\n\nAn expression wrapped in this manner is left alone by the macro expander and simply pasted into the output verbatim. Therefore it will be resolved in the macro call environment.\n\nThis escaping mechanism can be used to \"violate\" hygiene when necessary, in order to introduce or manipulate user variables. For example, the following macro sets x to zero in the call environment:\n\njulia> macro zerox()\n           return esc(:(x = 0))\n       end\n@zerox (macro with 1 method)\n\njulia> function foo()\n           x = 1\n           @zerox\n           return x # is zero\n       end\nfoo (generic function with 1 method)\n\njulia> foo()\n0\n\nThis kind of manipulation of variables should be used judiciously, but is occasionally quite handy.\n\nGetting the hygiene rules correct can be a formidable challenge. Before using a macro, you might want to consider whether a function closure would be sufficient. Another useful strategy is to defer as much work as possible to runtime. For example, many macros simply wrap their arguments in a QuoteNode or other similar Expr. Some examples of this include @task body which simply returns schedule(Task(() -> $body)), and @eval expr, which simply returns eval(QuoteNode(expr)).\n\nTo demonstrate, we might rewrite the @time example above as:\n\nmacro time(expr)\n    return :(timeit(() -> $(esc(expr))))\nend\nfunction timeit(f)\n    t0 = time_ns()\n    val = f()\n    t1 = time_ns()\n    println(\"elapsed time: \", (t1-t0)/1e9, \" seconds\")\n    return val\nend\n\nHowever, we don't do this for a good reason: wrapping the expr in a new scope block (the anonymous function) also slightly changes the meaning of the expression (the scope of any variables in it), while we want @time to be usable with minimum impact on the wrapped code.","page":"Metaprogramming"},{"title":"Macros and dispatch","location":"manual/metaprogramming.html#Macros-and-dispatch","category":"section","text":"Macros, just like Julia functions, are generic. This means they can also have multiple method definitions, thanks to multiple dispatch:\n\njulia> macro m end\n@m (macro with 0 methods)\n\njulia> macro m(args...)\n           println(\"$(length(args)) arguments\")\n       end\n@m (macro with 1 method)\n\njulia> macro m(x,y)\n           println(\"Two arguments\")\n       end\n@m (macro with 2 methods)\n\njulia> @m \"asd\"\n1 arguments\n\njulia> @m 1 2\nTwo arguments\n\nHowever one should keep in mind, that macro dispatch is based on the types of AST that are handed to the macro, not the types that the AST evaluates to at runtime:\n\njulia> macro m(::Int)\n           println(\"An Integer\")\n       end\n@m (macro with 3 methods)\n\njulia> @m 2\nAn Integer\n\njulia> x = 2\n2\n\njulia> @m x\n1 arguments","page":"Metaprogramming"},{"title":"Code Generation","location":"manual/metaprogramming.html#Code-Generation","category":"section","text":"When a significant amount of repetitive boilerplate code is required, it is common to generate it programmatically to avoid redundancy. In most languages, this requires an extra build step, and a separate program to generate the repetitive code. In Julia, expression interpolation and eval allow such code generation to take place in the normal course of program execution. For example, consider the following custom type\n\nstruct MyNumber\n    x::Float64\nend\n# output\n\n\nfor which we want to add a number of methods to. We can do this programmatically in the following loop:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    eval(quote\n        Base.$op(a::MyNumber) = MyNumber($op(a.x))\n    end)\nend\n# output\n\n\nand we can now use those functions with our custom type:\n\njulia> x = MyNumber(π)\nMyNumber(3.141592653589793)\n\njulia> sin(x)\nMyNumber(1.2246467991473532e-16)\n\njulia> cos(x)\nMyNumber(-1.0)\n\nIn this manner, Julia acts as its own preprocessor, and allows code generation from inside the language. The above code could be written slightly more tersely using the : prefix quoting form:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    eval(:(Base.$op(a::MyNumber) = MyNumber($op(a.x))))\nend\n\nThis sort of in-language code generation, however, using the eval(quote(...)) pattern, is common enough that Julia comes with a macro to abbreviate this pattern:\n\nfor op = (:sin, :cos, :tan, :log, :exp)\n    @eval Base.$op(a::MyNumber) = MyNumber($op(a.x))\nend\n\nThe @eval macro rewrites this call to be precisely equivalent to the above longer versions. For longer blocks of generated code, the expression argument given to @eval can be a block:\n\n@eval begin\n    # multiple lines\nend","page":"Metaprogramming"},{"title":"Non-Standard String Literals","location":"manual/metaprogramming.html#meta-non-standard-string-literals","category":"section","text":"Recall from Strings that string literals prefixed by an identifier are called non-standard string literals, and can have different semantics than un-prefixed string literals. For example:\n\nr\"^\\s*(?:#|$)\" produces a regular expression object rather than a string\nb\"DATA\\xff\\u2200\" is a byte array literal for [68,65,84,65,255,226,136,128].\n\nPerhaps surprisingly, these behaviors are not hard-coded into the Julia parser or compiler. Instead, they are custom behaviors provided by a general mechanism that anyone can use: prefixed string literals are parsed as calls to specially-named macros. For example, the regular expression macro is just the following:\n\nmacro r_str(p)\n    Regex(p)\nend\n\nThat's all. This macro says that the literal contents of the string literal r\"^\\s*(?:#|$)\" should be passed to the @r_str macro and the result of that expansion should be placed in the syntax tree where the string literal occurs. In other words, the expression r\"^\\s*(?:#|$)\" is equivalent to placing the following object directly into the syntax tree:\n\nRegex(\"^\\\\s*(?:#|\\$)\")\n\nNot only is the string literal form shorter and far more convenient, but it is also more efficient: since the regular expression is compiled and the Regex object is actually created when the code is compiled, the compilation occurs only once, rather than every time the code is executed. Consider if the regular expression occurs in a loop:\n\nfor line = lines\n    m = match(r\"^\\s*(?:#|$)\", line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend\n\nSince the regular expression r\"^\\s*(?:#|$)\" is compiled and inserted into the syntax tree when this code is parsed, the expression is only compiled once instead of each time the loop is executed. In order to accomplish this without macros, one would have to write this loop like this:\n\nre = Regex(\"^\\\\s*(?:#|\\$)\")\nfor line = lines\n    m = match(re, line)\n    if m === nothing\n        # non-comment\n    else\n        # comment\n    end\nend\n\nMoreover, if the compiler could not determine that the regex object was constant over all loops, certain optimizations might not be possible, making this version still less efficient than the more convenient literal form above. Of course, there are still situations where the non-literal form is more convenient: if one needs to interpolate a variable into the regular expression, one must take this more verbose approach; in cases where the regular expression pattern itself is dynamic, potentially changing upon each loop iteration, a new regular expression object must be constructed on each iteration. In the vast majority of use cases, however, regular expressions are not constructed based on run-time data. In this majority of cases, the ability to write regular expressions as compile-time values is invaluable.\n\nThe mechanism for user-defined string literals is deeply, profoundly powerful. Not only are Julia's non-standard literals implemented using it, but the command literal syntax (`echo \"Hello, $person\"`) is also implemented using the following innocuous-looking macro:\n\nmacro cmd(str)\n    :(cmd_gen($(shell_parse(str)[1])))\nend\n\nOf course, a large amount of complexity is hidden in the functions used in this macro definition, but they are just functions, written entirely in Julia. You can read their source and see precisely what they do – and all they do is construct expression objects to be inserted into your program's syntax tree.\n\nLike string literals, command literals can also be prefixed by an identifier to form what are called non-standard command literals. These command literals are parsed as calls to specially-named macros. For example, the syntax custom`literal` is parsed as @custom_cmd \"literal\". Julia itself does not contain any non-standard command literals, but packages can make use of this syntax. Aside from the different syntax and the _cmd suffix instead of the _str suffix, non-standard command literals behave exactly like non-standard string literals.\n\nIn the event that two modules provide non-standard string or command literals with the same name, it is possible to qualify the string or command literal with a module name. For instance, if both Foo and Bar provide non-standard string literal @x_str, then one can write Foo.x\"literal\" or Bar.x\"literal\" to disambiguate between the two.\n\nAnother way to define a macro would be like this:\n\nmacro foo_str(str, flag)\n    # do stuff\nend\n\nThis macro can then be called with the following syntax:\n\nfoo\"str\"flag\n\nThe type of flag in the above mentioned syntax would be a String with contents of whatever trails after the string literal.","page":"Metaprogramming"},{"title":"Generated functions","location":"manual/metaprogramming.html#Generated-functions","category":"section","text":"A very special macro is @generated, which allows you to define so-called generated functions. These have the capability to generate specialized code depending on the types of their arguments with more flexibility and/or less code than what can be achieved with multiple dispatch. While macros work with expressions at parse time and cannot access the types of their inputs, a generated function gets expanded at a time when the types of the arguments are known, but the function is not yet compiled.\n\nInstead of performing some calculation or action, a generated function declaration returns a quoted expression which then forms the body for the method corresponding to the types of the arguments. When a generated function is called, the expression it returns is compiled and then run. To make this efficient, the result is usually cached. And to make this inferable, only a limited subset of the language is usable. Thus, generated functions provide a flexible way to move work from run time to compile time, at the expense of greater restrictions on allowed constructs.\n\nWhen defining generated functions, there are five main differences to ordinary functions:\n\nYou annotate the function declaration with the @generated macro. This adds some information to the AST that lets the compiler know that this is a generated function.\nIn the body of the generated function you only have access to the types of the arguments – not their values.\nInstead of calculating something or performing some action, you return a quoted expression which, when evaluated, does what you want.\nGenerated functions are only permitted to call functions that were defined before the definition of the generated function. (Failure to follow this may result in getting MethodErrors referring to functions from a future world-age.)\nGenerated functions must not mutate or observe any non-constant global state (including, for example, IO, locks, non-local dictionaries, or using hasmethod). This means they can only read global constants, and cannot have any side effects. In other words, they must be completely pure. Due to an implementation limitation, this also means that they currently cannot define a closure or generator.\n\nIt's easiest to illustrate this with an example. We can declare a generated function foo as\n\njulia> @generated function foo(x)\n           Core.println(x)\n           return :(x * x)\n       end\nfoo (generic function with 1 method)\n\nNote that the body returns a quoted expression, namely :(x * x), rather than just the value of x * x.\n\nFrom the caller's perspective, this is identical to a regular function; in fact, you don't have to know whether you're calling a regular or generated function. Let's see how foo behaves:\n\njulia> x = foo(2); # note: output is from println() statement in the body\nInt64\n\njulia> x           # now we print x\n4\n\njulia> y = foo(\"bar\");\nString\n\njulia> y\n\"barbar\"\n\nSo, we see that in the body of the generated function, x is the type of the passed argument, and the value returned by the generated function, is the result of evaluating the quoted expression we returned from the definition, now with the value of x.\n\nWhat happens if we evaluate foo again with a type that we have already used?\n\njulia> foo(4)\n16\n\nNote that there is no printout of Int64. We can see that the body of the generated function was only executed once here, for the specific set of argument types, and the result was cached. After that, for this example, the expression returned from the generated function on the first invocation was re-used as the method body. However, the actual caching behavior is an implementation-defined performance optimization, so it is invalid to depend too closely on this behavior.\n\nThe number of times a generated function is generated might be only once, but it might also be more often, or appear to not happen at all. As a consequence, you should never write a generated function with side effects - when, and how often, the side effects occur is undefined. (This is true for macros too - and just like for macros, the use of eval in a generated function is a sign that you're doing something the wrong way.) However, unlike macros, the runtime system cannot correctly handle a call to eval, so it is disallowed.\n\nIt is also important to see how @generated functions interact with method redefinition. Following the principle that a correct @generated function must not observe any mutable state or cause any mutation of global state, we see the following behavior. Observe that the generated function cannot call any method that was not defined prior to the definition of the generated function itself.\n\nInitially f(x) has one definition\n\njulia> f(x) = \"original definition\";\n\nDefine other operations that use f(x):\n\njulia> g(x) = f(x);\n\njulia> @generated gen1(x) = f(x);\n\njulia> @generated gen2(x) = :(f(x));\n\nWe now add some new definitions for f(x):\n\njulia> f(x::Int) = \"definition for Int\";\n\njulia> f(x::Type{Int}) = \"definition for Type{Int}\";\n\nand compare how these results differ:\n\njulia> f(1)\n\"definition for Int\"\n\njulia> g(1)\n\"definition for Int\"\n\njulia> gen1(1)\n\"original definition\"\n\njulia> gen2(1)\n\"definition for Int\"\n\nEach method of a generated function has its own view of defined functions:\n\njulia> @generated gen1(x::Real) = f(x);\n\njulia> gen1(1)\n\"definition for Type{Int}\"\n\nThe example generated function foo above did not do anything a normal function foo(x) = x * x could not do (except printing the type on the first invocation, and incurring higher overhead). However, the power of a generated function lies in its ability to compute different quoted expressions depending on the types passed to it:\n\njulia> @generated function bar(x)\n           if x <: Integer\n               return :(x ^ 2)\n           else\n               return :(x)\n           end\n       end\nbar (generic function with 1 method)\n\njulia> bar(4)\n16\n\njulia> bar(\"baz\")\n\"baz\"\n\n(although of course this contrived example would be more easily implemented using multiple dispatch...)\n\nAbusing this will corrupt the runtime system and cause undefined behavior:\n\njulia> @generated function baz(x)\n           if rand() < .9\n               return :(x^2)\n           else\n               return :(\"boo!\")\n           end\n       end\nbaz (generic function with 1 method)\n\nSince the body of the generated function is non-deterministic, its behavior, and the behavior of all subsequent code is undefined.\n\nDon't copy these examples!\n\nThese examples are hopefully helpful to illustrate how generated functions work, both in the definition end and at the call site; however, don't copy them, for the following reasons:\n\nthe foo function has side-effects (the call to Core.println), and it is undefined exactly when, how often or how many times these side-effects will occur\nthe bar function solves a problem that is better solved with multiple dispatch - defining bar(x) = x and bar(x::Integer) = x ^ 2 will do the same thing, but it is both simpler and faster.\nthe baz function is pathological\n\nNote that the set of operations that should not be attempted in a generated function is unbounded, and the runtime system can currently only detect a subset of the invalid operations. There are many other operations that will simply corrupt the runtime system without notification, usually in subtle ways not obviously connected to the bad definition. Because the function generator is run during inference, it must respect all of the limitations of that code.\n\nSome operations that should not be attempted include:\n\nCaching of native pointers.\nInteracting with the contents or methods of Core.Compiler in any way.\nObserving any mutable state.\nInference on the generated function may be run at any time, including while your code is attempting to observe or mutate this state.\nTaking any locks: C code you call out to may use locks internally, (for example, it is not problematic to call malloc, even though most implementations require locks internally) but don't attempt to hold or acquire any while executing Julia code.\nCalling any function that is defined after the body of the generated function. This condition is relaxed for incrementally-loaded precompiled modules to allow calling any function in the module.\n\nAlright, now that we have a better understanding of how generated functions work, let's use them to build some more advanced (and valid) functionality...","page":"Metaprogramming"},{"title":"An advanced example","location":"manual/metaprogramming.html#An-advanced-example","category":"section","text":"Julia's base library has an internal sub2ind function to calculate a linear index into an n-dimensional array, based on a set of n multilinear indices - in other words, to calculate the index i that can be used to index into an array A using A[i], instead of A[x,y,z,...]. One possible implementation is the following:\n\njulia> function sub2ind_loop(dims::NTuple{N}, I::Integer...) where N\n           ind = I[N] - 1\n           for i = N-1:-1:1\n               ind = I[i]-1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> sub2ind_loop((3, 5), 1, 2)\n4\n\nThe same thing can be done using recursion:\n\njulia> sub2ind_rec(dims::Tuple{}) = 1;\n\njulia> sub2ind_rec(dims::Tuple{}, i1::Integer, I::Integer...) =\n           i1 == 1 ? sub2ind_rec(dims, I...) : throw(BoundsError());\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer) = i1;\n\njulia> sub2ind_rec(dims::Tuple{Integer, Vararg{Integer}}, i1::Integer, I::Integer...) =\n           i1 + dims[1] * (sub2ind_rec(Base.tail(dims), I...) - 1);\n\njulia> sub2ind_rec((3, 5), 1, 2)\n4\n\nBoth these implementations, although different, do essentially the same thing: a runtime loop over the dimensions of the array, collecting the offset in each dimension into the final index.\n\nHowever, all the information we need for the loop is embedded in the type information of the arguments. This allows the compiler to move the iteration to compile time and eliminate the runtime loops altogether. We can utilize generated functions to achieve a similar effect; in compiler parlance, we use generated functions to manually unroll the loop. The body becomes almost identical, but instead of calculating the linear index, we build up an expression that calculates the index:\n\njulia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nWhat code will this generate?\n\nAn easy way to find out is to extract the body into another (regular) function:\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           length(I) == N || return :(error(\"partial indexing is unsupported\"))\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> @generated function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           return sub2ind_gen_impl(dims, I...)\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nWe can now execute sub2ind_gen_impl and examine the expression it returns:\n\njulia> sub2ind_gen_impl(Tuple{Int,Int}, Int, Int)\n:(((I[1] - 1) + dims[1] * (I[2] - 1)) + 1)\n\nSo, the method body that will be used here doesn't include a loop at all - just indexing into the two tuples, multiplication and addition/subtraction. All the looping is performed compile-time, and we avoid looping during execution entirely. Thus, we only loop once per type, in this case once per N (except in edge cases where the function is generated more than once - see disclaimer above).","page":"Metaprogramming"},{"title":"Optionally-generated functions","location":"manual/metaprogramming.html#Optionally-generated-functions","category":"section","text":"Generated functions can achieve high efficiency at run time, but come with a compile time cost: a new function body must be generated for every combination of concrete argument types. Typically, Julia is able to compile \"generic\" versions of functions that will work for any arguments, but with generated functions this is impossible. This means that programs making heavy use of generated functions might be impossible to statically compile.\n\nTo solve this problem, the language provides syntax for writing normal, non-generated alternative implementations of generated functions. Applied to the sub2ind example above, it would look like this:\n\njulia> function sub2ind_gen_impl(dims::Type{T}, I...) where T <: NTuple{N,Any} where N\n           ex = :(I[$N] - 1)\n           for i = (N - 1):-1:1\n               ex = :(I[$i] - 1 + dims[$i] * $ex)\n           end\n           return :($ex + 1)\n       end;\n\njulia> function sub2ind_gen_fallback(dims::NTuple{N}, I) where N\n           ind = I[N] - 1\n           for i = (N - 1):-1:1\n               ind = I[i] - 1 + dims[i]*ind\n           end\n           return ind + 1\n       end;\n\njulia> function sub2ind_gen(dims::NTuple{N}, I::Integer...) where N\n           length(I) == N || error(\"partial indexing is unsupported\")\n           if @generated\n               return sub2ind_gen_impl(dims, I...)\n           else\n               return sub2ind_gen_fallback(dims, I)\n           end\n       end;\n\njulia> sub2ind_gen((3, 5), 1, 2)\n4\n\nInternally, this code creates two implementations of the function: a generated one where the first block in if @generated is used, and a normal one where the else block is used. Inside the then part of the if @generated block, code has the same semantics as other generated functions: argument names refer to types, and the code should return an expression. Multiple if @generated blocks may occur, in which case the generated implementation uses all of the then blocks and the alternate implementation uses all of the else blocks.\n\nNotice that we added an error check to the top of the function. This code will be common to both versions, and is run-time code in both versions (it will be quoted and returned as an expression from the generated version). That means that the values and types of local variables are not available at code generation time –- the code-generation code can only see the types of arguments.\n\nIn this style of definition, the code generation feature is essentially an optional optimization. The compiler will use it if convenient, but otherwise may choose to use the normal implementation instead. This style is preferred, since it allows the compiler to make more decisions and compile programs in more ways, and since normal code is more readable than code-generating code. However, which implementation is used depends on compiler implementation details, so it is essential for the two implementations to behave identically.","page":"Metaprogramming"},{"title":"Multi-processing and Distributed Computing","location":"manual/distributed-computing.html#Multi-processing-and-Distributed-Computing","category":"section","text":"An implementation of distributed memory parallel computing is provided by module Distributed as part of the standard library shipped with Julia.\n\nMost modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.\n\nJulia's implementation of message passing is different from other environments such as MPI[1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.\n\nDistributed programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.\n\nRemote references come in two flavors: Future and RemoteChannel.\n\nA remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait on the returned Future, and you can obtain the full value of the result using fetch.\n\nOn the other hand, RemoteChannel s are rewritable. For example, multiple processes can coordinate their processing by referencing the same remote Channel.\n\nEach process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1. As a result, adding 2 or more processes is required to gain benefits from parallel processing methods like pmap. Adding a single process is beneficial if you just wish to do other things in the main process while a long computation is running on the worker.\n\nLet's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU threads (logical cores) on the machine. Note that the -p argument implicitly loads module Distributed.\n\n$ julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.18526  1.50912\n 1.16296  1.60607\n\nThe first argument to remotecall is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall is considered a low-level interface providing finer control. The second argument to remotecall is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.\n\nAs you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.\n\nOccasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.\n\njulia> remotecall_fetch(r-> fetch(r)[1, 1], 2, r)\n0.18526337335308085\n\nThis fetches the array on worker 2 and returns the first value. Note, that fetch doesn't move any data in this case, since it's executed on the worker that owns the array. One can also write:\n\njulia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.10824216411304866\n\nRemember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.\n\nTo make things easier, the symbol :any can be passed to @spawnat, which picks where to do the operation for you:\n\njulia> r = @spawnat :any rand(2,2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat :any 1 .+ fetch(r)\nFuture(3, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.38854  1.9098\n 1.20939  1.57158\n\nNote that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch might be required to move r to the process doing the addition. In this case, @spawnat is smart enough to perform the computation on the process that owns r, so the fetch will be a no-op (no work is done).\n\n(It is worth noting that @spawnat is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)\n\nAn important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted.\n\nThreads.@spawn is similar to @spawnat, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indices. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. As for v0.7 and beyond, the feeder tasks are able to share state via nextidx because they all run on the same process. Even if Tasks are scheduled cooperatively, locking may still be required in some contexts, as in asynchronous I/O. This means context switches only occur at well-defined points: in this case, when remotecall_fetch is called. This is the current state of implementation and it may change for future Julia versions, as it is intended to make it possible to run up to N Tasks on M Process, aka M:N Threading. Then a lock acquiring\\releasing model for nextidx will be needed, as it is not safe to let multiple processes read-write a resource at the same time.","page":"Multi-processing and Distributed Computing"},{"title":"Code Availability and Loading Packages","location":"manual/distributed-computing.html#code-availability","category":"section","text":"Your code must be available on any process that runs it. For example, type the following into the Julia prompt:\n\njulia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Matrix{Float64}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawnat :any rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))))\nStacktrace:\n[...]\n\nProcess 1 knew about the function rand2, but process 2 did not.\n\nMost commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:\n\nmodule DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nend\n\nIn order to refer to MyType across all processes, DummyModule.jl needs to be loaded on every process. Calling include(\"DummyModule.jl\") loads it only on a single process. To load it on every process, use the @everywhere macro (starting Julia with julia -p 2):\n\njulia> @everywhere include(\"DummyModule.jl\")\nloaded\n      From worker 3:    loaded\n      From worker 2:    loaded\n\nAs usual, this does not bring DummyModule into scope on any of the process, which requires using or import. Moreover, when DummyModule is brought into scope on one process, it is not on any other:\n\njulia> using .DummyModule\n\njulia> MyType(7)\nMyType(7)\n\njulia> fetch(@spawnat 2 MyType(7))\nERROR: On worker 2:\nUndefVarError: `MyType` not defined in `Main`\n⋮\n\njulia> fetch(@spawnat 2 DummyModule.MyType(7))\nMyType(7)\n\nHowever, it's still possible, for instance, to send a MyType to a process which has loaded DummyModule even if it's not in scope:\n\njulia> put!(RemoteChannel(2), MyType(7))\nRemoteChannel{Channel{Any}}(2, 1, 13)\n\nA file can also be preloaded on multiple processes at startup with the -L flag, and a driver script can be used to drive the computation:\n\njulia -p <n> -L file1.jl -L file2.jl driver.jl\n\nThe Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.\n\nFinally, if DummyModule.jl is not a standalone file but a package, then using DummyModule will load DummyModule.jl on all processes, but only bring it into scope on the process where using was called.","page":"Multi-processing and Distributed Computing"},{"title":"Starting and managing worker processes","location":"manual/distributed-computing.html#Starting-and-managing-worker-processes","category":"section","text":"The base Julia installation has in-built support for two types of clusters:\n\nA local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machine-file option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers should use to connect to this worker.\n\nnote: Note\nWhile Julia generally strives for backward compatibility, distribution of code to worker processes relies on Serialization.serialize. As pointed out in the corresponding documentation, this can not be guaranteed to work across different Julia versions, so it is advised that all workers on all machines use the same version.\n\nFunctions addprocs, rmprocs, workers, and others are available as a programmatic means of adding, removing and querying the processes in a cluster.\n\njulia> using Distributed\n\njulia> addprocs(2)\n2-element Vector{Int64}:\n 2\n 3\n\nModule Distributed must be explicitly loaded on the master process before invoking addprocs. It is automatically made available on the worker processes.\n\nnote: Note\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes. You may use addprocs(exeflags=\"--project\") to initialize a worker with a particular environment, and then @everywhere using <modulename> or @everywhere include(\"file.jl\").\n\nOther types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section.","page":"Multi-processing and Distributed Computing"},{"title":"Data Movement","location":"manual/distributed-computing.html#Data-Movement","category":"section","text":"Sending messages and moving data constitute most of the overhead in a distributed program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various distributed programming constructs.\n\nfetch can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawnat (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:\n\nMethod 1:\n\njulia> A = rand(1000,1000);\n\njulia> Bref = @spawnat :any A^2;\n\n[...]\n\njulia> fetch(Bref);\n\nMethod 2:\n\njulia> Bref = @spawnat :any rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);\n\nThe difference seems trivial, but in fact is quite significant due to the behavior of @spawnat. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.\n\nIn this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawnat and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawnat statement just for this step.","page":"Multi-processing and Distributed Computing"},{"title":"Global variables","location":"manual/distributed-computing.html#Global-variables","category":"section","text":"Expressions executed remotely via @spawnat, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:\n\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2)\n\nIn this case sum MUST be defined in the remote process. Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->sum(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:\n\nNew global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.\n\nAs you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.\n\nThus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.\n\nFor example:\n\njulia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @fetchfrom 2 InteractiveUtils.varinfo()\nname           size summary\n––––––––– ––––––––– ––––––––––––––––––––––\nA         800 bytes 10×10 Array{Float64,2}\nBase                Module\nCore                Module\nMain                Module\n\nAs can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2.","page":"Multi-processing and Distributed Computing"},{"title":"Parallel Map and Loops","location":"manual/distributed-computing.html#Parallel-Map-and-Loops","category":"section","text":"Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawnat to flip coins on two processes. First, write the following function in count_heads.jl:\n\nfunction count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nend\n\nThe function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:\n\njulia> @everywhere include_string(Main, $(read(\"count_heads.jl\", String)), \"count_heads.jl\")\n\njulia> a = @spawnat :any count_heads(100000000)\nFuture(2, 1, 6, nothing)\n\njulia> b = @spawnat :any count_heads(100000000)\nFuture(3, 1, 7, nothing)\n\njulia> fetch(a)+fetch(b)\n100001564\n\nThis example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.\n\nNotice that our use of this pattern with count_heads can be generalized. We used two explicit @spawnat statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, running in distributed memory, which can be written in Julia using @distributed like this:\n\nnheads = @distributed (+) for i = 1:200000000\n    Int(rand(Bool))\nend\n\nThis construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.\n\nNote that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.\n\nFor example, the following code will not work as intended:\n\na = zeros(100000)\n@distributed for i = 1:100000\n    a[i] = i\nend\n\nThis code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:\n\nusing SharedArrays\n\na = SharedArray{Float64}(10)\n@distributed for i = 1:10\n    a[i] = i\nend\n\nUsing \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:\n\na = randn(1000)\n@distributed (+) for i = 1:100000\n    f(a[rand(1:end)])\nend\n\nHere each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.\n\nAs you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @distributed for.\n\nIn some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap function. For example, we could compute the singular values of several large random matrices in parallel as follows:\n\njulia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svdvals, M);\n\nJulia's pmap is designed for the case where each function call does a large amount of work. In contrast, @distributed for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap and @distributed for for the parallel computation. In case of @distributed for, the final reduction is done on the calling process.","page":"Multi-processing and Distributed Computing"},{"title":"Remote References and AbstractChannels","location":"manual/distributed-computing.html#Remote-References-and-AbstractChannels","category":"section","text":"Remote references always refer to an implementation of an AbstractChannel.\n\nA concrete implementation of an AbstractChannel (like Channel), is required to implement put!, take!, fetch, isready and wait. The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.\n\nRemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.\n\nThe constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f is a function executed on pid and it must return an AbstractChannel.\n\nFor example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.\n\nMethods put!, take!, fetch, isready and wait on a RemoteChannel are proxied onto the backing store on the remote process.\n\nRemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is the following DictChannel which uses a dictionary as its remote store:\n\njulia> struct DictChannel{T} <: AbstractChannel{T}\n           d::Dict\n           cond_take::Threads.Condition    # waiting for data to become available\n           DictChannel{T}() where {T} = new(Dict(), Threads.Condition())\n           DictChannel() = DictChannel{Any}()\n       end\n\njulia> begin\n       function Base.put!(D::DictChannel, k, v)\n           @lock D.cond_take begin\n               D.d[k] = v\n               notify(D.cond_take)\n           end\n           return D\n       end\n       function Base.take!(D::DictChannel, k)\n           @lock D.cond_take begin\n               v = fetch(D, k)\n               delete!(D.d, k)\n               return v\n           end\n       end\n       Base.isready(D::DictChannel) = @lock D.cond_take !isempty(D.d)\n       Base.isready(D::DictChannel, k) = @lock D.cond_take haskey(D.d, k)\n       function Base.fetch(D::DictChannel, k)\n           @lock D.cond_take begin\n               wait(D, k)\n               return D.d[k]\n           end\n       end\n       function Base.wait(D::DictChannel, k)\n           @lock D.cond_take begin\n               while !isready(D, k)\n                   wait(D.cond_take)\n               end\n           end\n       end\n       end;\n\njulia> d = DictChannel();\n\njulia> isready(d)\nfalse\n\njulia> put!(d, :k, :v);\n\njulia> isready(d, :k)\ntrue\n\njulia> fetch(d, :k)\n:v\n\njulia> wait(d, :k)\n\njulia> take!(d, :k)\n:v\n\njulia> isready(d, :k)\nfalse","page":"Multi-processing and Distributed Computing"},{"title":"Channels and RemoteChannels","location":"manual/distributed-computing.html#Channels-and-RemoteChannels","category":"section","text":"A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.\n\nThe channels example from above can be modified for interprocess communication, as shown below.\n\nWe start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.\n\njulia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(Threads.@spawn make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, where = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds on worker $where\")\n           global n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741","page":"Multi-processing and Distributed Computing"},{"title":"Remote References and Distributed Garbage Collection","location":"manual/distributed-computing.html#Remote-References-and-Distributed-Garbage-Collection","category":"section","text":"Objects referred to by remote references can be freed only when all held references in the cluster are deleted.\n\nThe node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified. This is implemented in an internal cluster aware serializer. Remote references are only valid in the context of a running cluster. Serializing and deserializing references to and from regular IO objects is not supported.\n\nThe notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.\n\nSince Futures are write-once and cached locally, the act of fetching a Future also updates reference tracking information on the node owning the value.\n\nThe node which owns the value frees it once all references to it are cleared.\n\nWith Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.\n\nIt is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.\n\nIn case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize results in an immediate message sent to the remote node to go ahead and remove its reference to the value.\n\nOnce finalized, a reference becomes invalid and cannot be used in any further calls.","page":"Multi-processing and Distributed Computing"},{"title":"Local invocations","location":"manual/distributed-computing.html#Local-invocations","category":"section","text":"Data is necessarily copied over to the remote node for execution. This is the case for both remotecalls and when data is stored to a RemoteChannel / Future on a different node. As expected, this results in a copy of the serialized objects on the remote node. However, when the destination node is the local node, i.e. the calling process id is the same as the remote node id, it is executed as a local call. It is usually (not always) executed in a different task - but there is no serialization/deserialization of data. Consequently, the call refers to the same object instances as passed - no copies are created. This behavior is highlighted below:\n\njulia> using Distributed\n\njulia> rc = RemoteChannel(()->Channel(3));   # RemoteChannel created on local node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i                          # Reusing `v`\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[3], [3], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 1\n\njulia> addprocs(1);\n\njulia> rc = RemoteChannel(()->Channel(3), workers()[1]);   # RemoteChannel created on remote node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[1], [2], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 3\n\nAs can be seen, put! on a locally owned RemoteChannel with the same object v modified between calls results in the same single object instance stored. As opposed to copies of v being created when the node owning rc is a different node.\n\nIt is to be noted that this is generally not an issue. It is something to be factored in only if the object is both being stored locally and modified post the call. In such cases it may be appropriate to store a deepcopy of the object.\n\nThis is also true for remotecalls on the local node as seen in the following example:\n\njulia> using Distributed; addprocs(1);\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), myid(), v);     # Executed on local node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[1], v2=[1], true\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), workers()[1], v); # Executed on remote node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[0], v2=[1], false\n\nAs can be seen once again, a remote call onto the local node behaves just like a direct invocation. The call modifies local objects passed as arguments. In the remote invocation, it operates on a copy of the arguments.\n\nTo repeat, in general this is not an issue. If the local node is also being used as a compute node, and the arguments used post the call, this behavior needs to be factored in and if required deep copies of arguments must be passed to the call invoked on the local node. Calls on remote nodes will always operate on copies of arguments.","page":"Multi-processing and Distributed Computing"},{"title":"Shared Arrays","location":"manual/distributed-computing.html#man-shared-arrays","category":"section","text":"Shared Arrays use system shared memory to map the same array across many processes. A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine. Shared Array support is available via the module SharedArrays, which must be explicitly loaded on all participating workers.\n\nA complementary data structure is provided by the external package DistributedArrays.jl in the form of a DArray. While there are some similarities to a SharedArray, the behavior of a DArray is quite different. In a SharedArray, each \"participating\" process has access to the entire array; in contrast, in a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk.\n\nSharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata. For other AbstractArray types, sdata just returns the object itself, so it's safe to use sdata on any Array-type object.\n\nThe constructor for a shared array is of the form:\n\nSharedArray{T,N}(dims::NTuple; init=false, pids=Int[])\n\nwhich creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host). Note that only elements that are isbits are supported in a SharedArray.\n\nIf an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.\n\nHere's a brief example:\n\njulia> using Distributed\n\njulia> addprocs(3)\n3-element Vector{Int64}:\n 2\n 3\n 4\n\njulia> @everywhere using SharedArrays\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[localindices(S)] = repeat([myid()], length(localindices(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4\n\nSharedArrays.localindices provides disjoint one-dimensional ranges of indices, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = repeat([myid()], length( indexpids(S):length(procs(S)):length(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4\n\nSince all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:\n\n@sync begin\n    for p in procs(S)\n        Threads.@spawn begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nend\n\nwould result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.\n\nAs a more extended and complex example, consider running the following \"kernel\" in parallel:\n\nq[i,j,t+1] = q[i,j,t] + u[i,j,t]\n\nIn this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indices assigned to this worker:\n\njulia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in range(0, stop=size(q,2), length=nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       end\n\nNext, define the kernel:\n\njulia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       end\n\nWe also define a convenience wrapper for a SharedArray implementation\n\njulia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)\n\nNow let's compare three different versions, one that runs in a single process:\n\njulia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);\n\none that uses @distributed:\n\njulia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @distributed for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;\n\nand one that delegates in chunks:\n\njulia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   Threads.@spawn remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;\n\nIf we create SharedArrays and time these functions, we get the following results (with julia -p 4):\n\njulia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));\n\nRun the functions once to JIT-compile and @time them on the second run:\n\njulia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)\n\nThe biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece.","page":"Multi-processing and Distributed Computing"},{"title":"Shared Arrays and Distributed Garbage Collection","location":"manual/distributed-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection","category":"section","text":"Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner.","page":"Multi-processing and Distributed Computing"},{"title":"ClusterManagers","location":"manual/distributed-computing.html#ClusterManagers","category":"section","text":"The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible for\n\nlaunching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transport\n\nA Julia cluster has the following characteristics:\n\nThe initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.\n\nConnections between workers (using the in-built TCP/IP transport) is established in the following manner:\n\naddprocs is called on the master process with a ClusterManager object.\naddprocs calls the appropriate launch method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to stdout.\nThe cluster manager captures the stdout of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.\n\nWhile the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.\n\nJulia provides two in-built cluster managers:\n\nLocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnames\n\nLocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.\n\nThus, a minimal cluster manager would need to:\n\nbe a subtype of the abstract ClusterManager\nimplement launch, a method responsible for launching new workers\nimplement manage, which is called at various events during a worker's lifetime (for example, sending an interrupt signal)\n\naddprocs(manager::FooManager) requires FooManager to implement:\n\nfunction launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nAs an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:\n\nstruct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nThe launch method takes the following arguments:\n\nmanager::ClusterManager: the cluster manager that addprocs is called with\nparams::Dict: all the keyword arguments passed to addprocs\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launched\n\nThe launch method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch function MUST exit as soon as all the requested workers have been launched.\n\nNewly launched workers are connected to each other and the master process in an all-to-all manner. Specifying the command line argument --worker[=<cookie>] results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets.\n\nAll workers in a cluster share the same cookie as the master. When the cookie is unspecified, i.e, with the --worker option, the worker tries to read it from its standard input.  LocalManager and SSHManager both pass the cookie to newly launched workers via their  standard inputs.\n\nBy default a worker will listen on a free port at the address returned by a call to getipaddr(). A specific address to listen on may be specified by optional argument --bind-to bind_addr[:port]. This is useful for multi-homed hosts.\n\nAs an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.\n\nFor every worker launched, the launch method must add a WorkerConfig object (with appropriate fields initialized) to launched\n\nmutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Union{IO, Nothing}\n    host::Union{AbstractString, Nothing}\n    port::Union{Integer, Nothing}\n\n    # Used when launching additional workers at a host\n    count::Union{Int, Symbol, Nothing}\n    exename::Union{AbstractString, Cmd, Nothing}\n    exeflags::Union{Cmd, Nothing}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Any\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Union{Bool, Nothing}\n    bind_addr::Union{AbstractString, Nothing}\n    sshflags::Union{Cmd, Nothing}\n    max_parallel::Union{Integer, Nothing}\n\n    # Used by Local/SSH managers\n    connect_at::Any\n\n    [...]\nend\n\nMost of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:\n\nIf io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of CPU threads (logical cores) on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.\n\nmanage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:\n\nwith :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes.","page":"Multi-processing and Distributed Computing"},{"title":"Cluster Managers with Custom Transports","location":"manual/distributed-computing.html#Cluster-Managers-with-Custom-Transports","category":"section","text":"Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:\n\nEach Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.\n\nReplacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:\n\nconnect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)\n\nThe default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).\n\nconnect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.\n\nA BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.\n\nThe folder clustermanager/0mq in the Examples repository contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.\n\nWhen using custom transports:\n\nJulia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) must be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch is called. The value of this field is passed in all connect callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.\n\nkill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.\n\nThe Examples folder clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup.","page":"Multi-processing and Distributed Computing"},{"title":"Network Requirements for LocalManager and SSHManager","location":"manual/distributed-computing.html#Network-Requirements-for-LocalManager-and-SSHManager","category":"section","text":"Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:\n\nThe master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on an ephemeral port number assigned by the OS.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-i <keyfile>`.\nIn an all-to-all topology (the default), all workers connect to each other via plain TCP sockets. The security policy on the cluster nodes must thus ensure free connectivity between workers for the ephemeral port range (varies by OS).\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager.\nIf you specify multiplex=true as an option to addprocs, SSH multiplexing is used to create a tunnel between the master and workers. If you have configured SSH multiplexing on your own and the connection has already been established, SSH multiplexing is used regardless of multiplex option. If multiplexing is enabled, forwarding is set by using the existing connection (-O forward option in ssh). This is beneficial if your servers require password authentication; you can avoid authentication in Julia by logging in to the server ahead of addprocs. The control socket will be located at ~/.ssh/julia-%r@%h:%p during the session unless the existing multiplexing connection is used. Note that bandwidth may be limited if you create multiple processes on a node and enable multiplexing, because in that case processes share a single multiplexing TCP connection.","page":"Multi-processing and Distributed Computing"},{"title":"Cluster Cookie","location":"manual/distributed-computing.html#man-cluster-cookie","category":"section","text":"All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:\n\ncluster_cookie() returns the cookie, while cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie may be passed to the workers at startup via argument --worker=<cookie>. If argument --worker is specified without the cookie, the worker tries to read the cookie from its standard input (stdin). The stdin is closed immediately after the cookie is retrieved.\nClusterManagers can retrieve the cookie on the master by calling cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.\n\nNote that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument.","page":"Multi-processing and Distributed Computing"},{"title":"Specifying Network Topology (Experimental)","location":"manual/distributed-computing.html#Specifying-Network-Topology-(Experimental)","category":"section","text":"The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other:\n\n:all_to_all, the default: all workers are connected to each other.\n:master_worker: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.\n\nKeyword argument lazy=true|false only affects topology option :all_to_all. If true, the cluster starts off with the master connected to all workers. Specific worker-worker connections are established at the first remote invocation between two workers. This helps in reducing initial resources allocated for intra-cluster communication. Connections are setup depending on the runtime requirements of a parallel program. Default value for lazy is true.\n\nCurrently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases.","page":"Multi-processing and Distributed Computing"},{"title":"Noteworthy external packages","location":"manual/distributed-computing.html#Noteworthy-external-packages","category":"section","text":"Outside of Julia parallelism there are plenty of external packages that should be mentioned. For example, MPI.jl is a Julia wrapper for the MPI protocol, Dagger.jl provides functionality similar to Python's Dask, and DistributedArrays.jl provides array operations distributed across workers, as outlined above.\n\nA mention must be made of Julia's GPU programming ecosystem, which includes:\n\nCUDA.jl wraps the various CUDA libraries and supports compiling Julia kernels for Nvidia GPUs.\noneAPI.jl wraps the oneAPI unified programming model, and supports executing Julia kernels on supported accelerators. Currently only Linux is supported.\nAMDGPU.jl wraps the AMD ROCm libraries and supports compiling Julia kernels for AMD GPUs. Currently only Linux is supported.\nHigh-level libraries like KernelAbstractions.jl, Tullio.jl and ArrayFire.jl.\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes by first casting it through distribute() and CuArray().\n\nRemember when importing DistributedArrays.jl to import it across all processes using @everywhere\n\n$ ./julia -p 4\n\njulia> addprocs()\n\njulia> @everywhere using DistributedArrays\n\njulia> using CUDA\n\njulia> B = ones(10_000) ./ 2;\n\njulia> A = ones(10_000) .* π;\n\njulia> C = 2 .* A ./ B;\n\njulia> all(C .≈ 4*π)\ntrue\n\njulia> typeof(C)\nVector{Float64} (alias for Array{Float64, 1})\n\njulia> dB = distribute(B);\n\njulia> dA = distribute(A);\n\njulia> dC = 2 .* dA ./ dB;\n\njulia> all(dC .≈ 4*π)\ntrue\n\njulia> typeof(dC)\nDistributedArrays.DArray{Float64,1,Vector{Float64}}\n\njulia> cuB = CuArray(B);\n\njulia> cuA = CuArray(A);\n\njulia> cuC = 2 .* cuA ./ cuB;\n\njulia> all(cuC .≈ 4*π);\ntrue\n\njulia> typeof(cuC)\nCuArray{Float64,1}\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes and call a generic function on it.\n\nfunction power_method(M, v)\n    for i in 1:100\n        v = M*v\n        v /= norm(v)\n    end\n\n    return v, norm(M*v) / norm(v)  # or  (M*v) ./ v\nend\n\npower_method repeatedly creates a new vector and normalizes it. We have not specified any type signature in function declaration, let's see if it works with the aforementioned datatypes:\n\njulia> M = [2. 1; 1 1];\n\njulia> v = rand(2)\n2-element Vector{Float64}:\n0.40395\n0.445877\n\njulia> power_method(M,v)\n([0.850651, 0.525731], 2.618033988749895)\n\njulia> cuM = CuArray(M);\n\njulia> cuv = CuArray(v);\n\njulia> curesult = power_method(cuM, cuv);\n\njulia> typeof(curesult)\nCuArray{Float64,1}\n\njulia> dM = distribute(M);\n\njulia> dv = distribute(v);\n\njulia> dC = power_method(dM, dv);\n\njulia> typeof(dC)\nTuple{DistributedArrays.DArray{Float64,1,Vector{Float64}},Float64}\n\nTo end this short exposure to external packages, we can consider MPI.jl, a Julia wrapper of the MPI protocol. As it would take too long to consider every inner function, it would be better to simply appreciate the approach used to implement the protocol.\n\nConsider this toy script which simply calls each subprocess, instantiate its rank and when the master process is reached, performs the ranks' sum\n\nimport MPI\n\nMPI.Init()\n\ncomm = MPI.COMM_WORLD\nMPI.Barrier(comm)\n\nroot = 0\nr = MPI.Comm_rank(comm)\n\nsr = MPI.Reduce(r, MPI.SUM, root, comm)\n\nif(MPI.Comm_rank(comm) == root)\n   @printf(\"sum of ranks: %s\\n\", sr)\nend\n\nMPI.Finalize()\n\nmpirun -np 4 ./julia example.jl\n\n[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding rma to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see https://mpi-forum.org/docs.","page":"Multi-processing and Distributed Computing"},{"title":"Memory-mapped I/O","location":"stdlib/Mmap.html#Memory-mapped-I/O","category":"section","text":"Low level module for mmap (memory mapping of files).","page":"Memory-mapped I/O"},{"title":"Mmap.Anonymous","location":"stdlib/Mmap.html#Mmap.Anonymous","category":"type","text":"Mmap.Anonymous(name::AbstractString=\"\", readonly::Bool=false, create::Bool=true)\n\nCreate an IO-like object for creating zeroed-out mmapped-memory that is not tied to a file for use in mmap. Used by SharedArray for creating shared memory arrays.\n\nExamples\n\njulia> using Mmap\n\njulia> anon = Mmap.Anonymous();\n\njulia> isreadable(anon)\ntrue\n\njulia> iswritable(anon)\ntrue\n\njulia> isopen(anon)\ntrue\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Mmap.mmap","location":"stdlib/Mmap.html#Mmap.mmap","category":"function","text":"mmap(io::Union{IOStream,AbstractString,Mmap.AnonymousMmap}[, type::Type{Array{T,N}}, dims, offset]; grow::Bool=true, shared::Bool=true)\nmmap(type::Type{Array{T,N}}, dims)\n\nCreate an Array whose values are linked to a file, using memory-mapping. This provides a convenient way of working with data too large to fit in the computer's memory.\n\nThe type is an Array{T,N} with a bits-type element of T and dimension N that determines how the bytes of the array are interpreted. Note that the file must be stored in binary format, and no format conversions are possible (this is a limitation of operating systems, not Julia).\n\ndims is a tuple or single Integer specifying the size or length of the array.\n\nThe file is passed via the stream argument, either as an open IOStream or filename string. When you initialize the stream, use \"r\" for a \"read-only\" array, and \"w+\" to create a new array used to write values to disk.\n\nIf no type argument is specified, the default is Vector{UInt8}.\n\nOptionally, you can specify an offset (in bytes) if, for example, you want to skip over a header in the file. The default value for the offset is the current stream position for an IOStream.\n\nThe grow keyword argument specifies whether the disk file should be grown to accommodate the requested size of array (if the total file size is < requested array size). Write privileges are required to grow the file.\n\nThe shared keyword argument specifies whether the resulting Array and changes made to it will be visible to other processes mapping the same file.\n\nFor example, the following code\n\n# Create a file for mmapping\n# (you could alternatively use mmap to do this step, too)\nusing Mmap\nA = rand(1:20, 5, 30)\ns = open(\"/tmp/mmap.bin\", \"w+\")\n# We'll write the dimensions of the array as the first two Ints in the file\nwrite(s, size(A,1))\nwrite(s, size(A,2))\n# Now write the data\nwrite(s, A)\nclose(s)\n\n# Test by reading it back in\ns = open(\"/tmp/mmap.bin\")   # default is read-only\nm = read(s, Int)\nn = read(s, Int)\nA2 = mmap(s, Matrix{Int}, (m,n))\n\ncreates a m-by-n Matrix{Int}, linked to the file associated with stream s.\n\nA more portable file would need to encode the word size – 32 bit or 64 bit – and endianness information in the header. In practice, consider encoding binary data using standard formats like HDF5 (which can be used with memory-mapping).\n\n\n\n\n\nmmap(io, BitArray, [dims, offset])\n\nCreate a BitArray whose values are linked to a file, using memory-mapping; it has the same purpose, works in the same way, and has the same arguments, as mmap, but the byte representation is different.\n\nExamples\n\njulia> using Mmap\n\njulia> io = open(\"mmap.bin\", \"w+\");\n\njulia> B = mmap(io, BitArray, (25,30000));\n\njulia> B[3, 4000] = true;\n\njulia> Mmap.sync!(B);\n\njulia> close(io);\n\njulia> io = open(\"mmap.bin\", \"r+\");\n\njulia> C = mmap(io, BitArray, (25,30000));\n\njulia> C[3, 4000]\ntrue\n\njulia> C[2, 4000]\nfalse\n\njulia> close(io)\n\njulia> rm(\"mmap.bin\")\n\nThis creates a 25-by-30000 BitArray, linked to the file associated with stream io.\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Mmap.sync!","location":"stdlib/Mmap.html#Mmap.sync!","category":"function","text":"Mmap.sync!(array)\n\nForces synchronization between the in-memory version of a memory-mapped Array or BitArray and the on-disk version.\n\n\n\n\n\n","page":"Memory-mapped I/O"},{"title":"Network Options","location":"stdlib/NetworkOptions.html#Network-Options","category":"section","text":"","page":"Network Options"},{"title":"NetworkOptions.ca_roots","location":"stdlib/NetworkOptions.html#NetworkOptions.ca_roots","category":"function","text":"ca_roots() :: Union{Nothing, String}\n\nThe ca_roots() function tells the caller where, if anywhere, to find a file or directory of PEM-encoded certificate authority roots. By default, on systems like Windows and macOS where the built-in TLS engines know how to verify hosts using the system's built-in certificate verification mechanism, this function will return nothing. On classic UNIX systems (excluding macOS), root certificates are typically stored in a file in /etc: the common places for the current UNIX system will be searched and if one of these paths exists, it will be returned; if none of these typical root certificate paths exist, then the path to the set of root certificates that are bundled with Julia is returned.\n\nThe default value returned by ca_roots() may be overridden by setting the JULIA_SSL_CA_ROOTS_PATH, SSL_CERT_DIR, or SSL_CERT_FILE environment variables, in which case this function will always return the value of the first of these variables that is set (whether the path exists or not). If JULIA_SSL_CA_ROOTS_PATH is set to the empty string, then the other variables are ignored (as if unset); if the other variables are set to the empty string, they behave is if they are not set.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ca_roots_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ca_roots_path","category":"function","text":"ca_roots_path() :: String\n\nThe ca_roots_path() function is similar to the ca_roots() function except that it always returns a path to a file or directory of PEM-encoded certificate authority roots. When called on a system like Windows or macOS, where system root certificates are not stored in the file system, it will currently return the path to the set of root certificates that are bundled with Julia. (In the future, this function may instead extract the root certificates from the system and save them to a file whose path would be returned.)\n\nIf it is possible to configure a library that uses TLS to use the system certificates that is generally preferable: i.e. it is better to use ca_roots() which returns nothing to indicate that the system certs should be used. The ca_roots_path() function should only be used when configuring libraries which require a path to a file or directory for root certificates.\n\nThe default value returned by ca_roots_path() may be overridden by setting the JULIA_SSL_CA_ROOTS_PATH, SSL_CERT_DIR, or SSL_CERT_FILE environment variables, in which case this function will always return the value of the first of these variables that is set (whether the path exists or not). If JULIA_SSL_CA_ROOTS_PATH is set to the empty string, then the other variables are ignored (as if unset); if the other variables are set to the empty string, they behave is if they are not set.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_dir","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_dir","category":"function","text":"ssh_dir() :: String\n\nThe ssh_dir() function returns the location of the directory where the ssh program keeps/looks for configuration files. By default this is ~/.ssh but this can be overridden by setting the environment variable SSH_DIR.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_key_pass","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_pass","category":"function","text":"ssh_key_pass() :: String\n\nThe ssh_key_pass() function returns the value of the environment variable SSH_KEY_PASS if it is set or nothing if it is not set. In the future, this may be able to find a password by other means, such as secure system storage, so packages that need a password to decrypt an SSH private key should use this API instead of directly checking the environment variable so that they gain such capabilities automatically when they are added.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_key_name","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_name","category":"function","text":"ssh_key_name() :: String\n\nThe ssh_key_name() function returns the base name of key files that SSH should use for when establishing a connection. There is usually no reason that this function should be called directly and libraries should generally use the ssh_key_path and ssh_pub_key_path functions to get full paths. If the environment variable SSH_KEY_NAME is set then this function returns that; otherwise it returns id_rsa by default.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_key_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_key_path","category":"function","text":"ssh_key_path() :: String\n\nThe ssh_key_path() function returns the path of the SSH private key file that should be used for SSH connections. If the SSH_KEY_PATH environment variable is set then it will return that value. Otherwise it defaults to returning\n\njoinpath(ssh_dir(), ssh_key_name())\n\nThis default value in turn depends on the SSH_DIR and SSH_KEY_NAME environment variables.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_pub_key_path","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_pub_key_path","category":"function","text":"ssh_pub_key_path() :: String\n\nThe ssh_pub_key_path() function returns the path of the SSH public key file that should be used for SSH connections. If the SSH_PUB_KEY_PATH environment variable is set then it will return that value. If that isn't set but SSH_KEY_PATH is set, it will return that path with the .pub suffix appended. If neither is set, it defaults to returning\n\njoinpath(ssh_dir(), ssh_key_name() * \".pub\")\n\nThis default value in turn depends on the SSH_DIR and SSH_KEY_NAME environment variables.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_known_hosts_files","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_known_hosts_files","category":"function","text":"ssh_known_hosts_files() :: Vector{String}\n\nThe ssh_known_hosts_files() function returns a vector of paths of SSH known hosts files that should be used when establishing the identities of remote servers for SSH connections. By default this function returns\n\n[joinpath(ssh_dir(), \"known_hosts\"), bundled_known_hosts]\n\nwhere bundled_known_hosts is the path of a copy of a known hosts file that is bundled with this package (containing known hosts keys for github.com and gitlab.com). If the environment variable SSH_KNOWN_HOSTS_FILES is set, however, then its value is split into paths on the : character (or on ; on Windows) and this vector of paths is returned instead. If any component of this vector is empty, it is expanded to the default known hosts paths.\n\nPackages that use ssh_known_hosts_files() should ideally look for matching entries by comparing the host name and key types, considering the first entry in any of the files which matches to be the definitive identity of the host. If the caller cannot compare the key type (e.g. because it has been hashes) then it must approximate the above algorithm by looking for all matching entries for a host in each file: if a file has any entries for a host then one of them must match; the caller should only continue to search further known hosts files if there are no entries for the host in question in an earlier file.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.ssh_known_hosts_file","location":"stdlib/NetworkOptions.html#NetworkOptions.ssh_known_hosts_file","category":"function","text":"ssh_known_hosts_file() :: String\n\nThe ssh_known_hosts_file() function returns a single path of an SSH known hosts file that should be used when establishing the identities of remote servers for SSH connections. It returns the first path returned by ssh_known_hosts_files that actually exists. Callers who can look in more than one known hosts file should use ssh_known_hosts_files instead and look for host matches in all the files returned as described in that function's docs.\n\n\n\n\n\n","page":"Network Options"},{"title":"NetworkOptions.verify_host","location":"stdlib/NetworkOptions.html#NetworkOptions.verify_host","category":"function","text":"verify_host(url::AbstractString, [transport::AbstractString]) :: Bool\n\nThe verify_host function tells the caller whether the identity of a host should be verified when communicating over secure transports like TLS or SSH. The url argument may be:\n\na proper URL staring with proto://\nan ssh-style bare host name or host name prefixed with user@\nan scp-style host as above, followed by : and a path location\n\nIn each case the host name part is parsed out and the decision about whether to verify or not is made based solely on the host name, not anything else about the input URL. In particular, the protocol of the URL does not matter (more below).\n\nThe transport argument indicates the kind of transport that the query is about. The currently known values are SSL/ssl (alias TLS/tls) and SSH/ssh. If the transport is omitted, the query will return true only if the host name should not be verified regardless of transport.\n\nThe host name is matched against the host patterns in the relevant environment variables depending on whether transport is supplied and what its value is:\n\nJULIA_NO_VERIFY_HOSTS — hosts that should not be verified for any transport\nJULIA_SSL_NO_VERIFY_HOSTS — hosts that should not be verified for SSL/TLS\nJULIA_SSH_NO_VERIFY_HOSTS — hosts that should not be verified for SSH\nJULIA_ALWAYS_VERIFY_HOSTS — hosts that should always be verified\n\nThe values of each of these variables is a comma-separated list of host name patterns with the following syntax — each pattern is split on . into parts and each part must one of:\n\nA literal domain name component consisting of one or more ASCII letter, digit, hyphen or underscore (technically not part of a legal host name, but sometimes used). A literal domain name component matches only itself.\nA **, which matches zero or more domain name components.\nA *, which match any one domain name component.\n\nWhen matching a host name against a pattern list in one of these variables, the host name is split on . into components and that sequence of words is matched against the pattern: a literal pattern matches exactly one host name component with that value; a * pattern matches exactly one host name component with any value; a ** pattern matches any number of host name components. For example:\n\n** matches any host name\n**.org matches any host name in the .org top-level domain\nexample.com matches only the exact host name example.com\n*.example.com matches api.example.com but not example.com or v1.api.example.com\n**.example.com matches any domain under example.com, including example.com itself, api.example.com and v1.api.example.com\n\n\n\n\n\n","page":"Network Options"},{"title":"Code Loading","location":"manual/code-loading.html#code-loading","category":"section","text":"note: Note\nThis chapter covers the technical details of package loading. To install packages, use Pkg, Julia's built-in package manager, to add packages to your active environment. To use packages already in your active environment, write import X or using X, as described in the Modules documentation.","page":"Code Loading"},{"title":"Definitions","location":"manual/code-loading.html#Definitions","category":"section","text":"Julia has two mechanisms for loading code:\n\nCode inclusion: e.g. include(\"source.jl\"). Inclusion allows you to split a single program across multiple source files. The expression include(\"source.jl\") causes the contents of the file source.jl to be evaluated in the global scope of the module where the include call occurs. If include(\"source.jl\") is called multiple times, source.jl is evaluated multiple times. The included path, source.jl, is interpreted relative to the file where the include call occurs. This makes it simple to relocate a subtree of source files. In the REPL, included paths are interpreted relative to the current working directory, pwd().\nPackage loading: e.g. import X or using X. The import mechanism allows you to load a package—i.e. an independent, reusable collection of Julia code, wrapped in a module—and makes the resulting module available by the name X inside of the importing module. If the same X package is imported multiple times in the same Julia session, it is only loaded the first time—on subsequent imports, the importing module gets a reference to the same module. Note though, that import X can load different packages in different contexts: X can refer to one package named X in the main project but potentially to different packages also named X in each dependency. More on this below.\n\nCode inclusion is quite straightforward and simple: it evaluates the given source file in the context of the caller. Package loading is built on top of code inclusion and serves a different purpose. The rest of this chapter focuses on the behavior and mechanics of package loading.\n\nA package is a source tree with a standard layout providing functionality that can be reused by other Julia projects. A package is loaded by import X or  using X statements. These statements also make the module named X—which results from loading the package code—available within the module where the import statement occurs. The meaning of X in import X is context-dependent: which X package is loaded depends on what code the statement occurs in. Thus, handling of import X happens in two stages: first, it determines what package is defined to be X in this context; second, it determines where that particular X package is found.\n\nThese questions are answered by searching through the project environments listed in LOAD_PATH for project files (Project.toml or JuliaProject.toml), manifest files (Manifest.toml or JuliaManifest.toml, or the same names suffixed by -v{major}.{minor}.toml for specific versions), or folders of source files.","page":"Code Loading"},{"title":"Federation of packages","location":"manual/code-loading.html#Federation-of-packages","category":"section","text":"Most of the time, a package is uniquely identifiable simply from its name. However, sometimes a project might encounter a situation where it needs to use two different packages that share the same name. While you might be able fix this by renaming one of the packages, being forced to do so can be highly disruptive in a large, shared code base. Instead, Julia's code loading mechanism allows the same package name to refer to different packages in different components of an application.\n\nJulia supports federated package management, which means that multiple independent parties can maintain both public and private packages and registries of packages, and that projects can depend on a mix of public and private packages from different registries. Packages from various registries are installed and managed using a common set of tools and workflows. The Pkg package manager that ships with Julia lets you install and manage your projects' dependencies. It assists in creating and manipulating project files (which describe what other projects that your project depends on), and manifest files (which snapshot exact versions of your project's complete dependency graph).\n\nOne consequence of federation is that there cannot be a central authority for package naming. Different entities may use the same name to refer to unrelated packages. This possibility is unavoidable since these entities do not coordinate and may not even know about each other. Because of the lack of a central naming authority, a single project may end up depending on different packages that have the same name. Julia's package loading mechanism does not require package names to be globally unique, even within the dependency graph of a single project. Instead, packages are identified by universally unique identifiers (UUIDs), which get assigned when each package is created. Usually you won't have to work directly with these somewhat cumbersome 128-bit identifiers since Pkg will take care of generating and tracking them for you. However, these UUIDs provide the definitive answer to the question of \"what package does X refer to?\"\n\nSince the decentralized naming problem is somewhat abstract, it may help to walk through a concrete scenario to understand the issue. Suppose you're developing an application called App, which uses two packages: Pub and  Priv. Priv is a private package that you created, whereas Pub is a public package that you use but don't control. When you created Priv, there was no public package by the name Priv. Subsequently, however, an unrelated package also named Priv has been published and become popular. In fact, the Pub package has started to use it. Therefore, when you next upgrade Pub to get the latest bug fixes and features, App will end up depending on two different packages named Priv—through no action of yours other than upgrading. App has a direct dependency on your private Priv package, and an indirect dependency, through Pub, on the new public Priv package. Since these two Priv packages are different but are both required for App to continue working correctly, the expression import Priv must refer to different Priv packages depending on whether it occurs in App's code or in Pub's code. To handle this, Julia's package loading mechanism distinguishes the two Priv packages by their UUID and picks the correct one based on its context (the module that called import). How this distinction works is determined by environments, as explained in the following sections.","page":"Code Loading"},{"title":"Environments","location":"manual/code-loading.html#Environments","category":"section","text":"An environment determines what import X and using X mean in various code contexts and what files these statements cause to be loaded. Julia understands two kinds of environments:\n\nA project environment is a directory with a project file and an optional manifest file, and forms an explicit environment. The project file determines what the names and identities of the direct dependencies of a project are. The manifest file, if present, gives a complete dependency graph, including all direct and indirect dependencies, exact versions of each dependency, and sufficient information to locate and load the correct version.\nA package directory is a directory containing the source trees of a set of packages as subdirectories, and forms an implicit environment. If X is a subdirectory of a package directory and X/src/X.jl exists, then the package X is available in the package directory environment and X/src/X.jl is the source file by which it is loaded.\n\nThese can be intermixed to create a stacked environment: an ordered set of project environments and package directories, overlaid to make a single composite environment. The precedence and visibility rules then combine to determine which packages are available and where they get loaded from. Julia's load path forms a stacked environment, for example.\n\nThese environment each serve a different purpose:\n\nProject environments provide reproducibility. By checking a project environment into version control—e.g. a git repository—along with the rest of the project's source code, you can reproduce the exact state of the project and all of its dependencies. The manifest file, in particular, captures the exact version of every dependency, identified by a cryptographic hash of its source tree, which makes it possible for Pkg to retrieve the correct versions and be sure that you are running the exact code that was recorded for all dependencies.\nPackage directories provide convenience when a full carefully-tracked project environment is unnecessary. They are useful when you want to put a set of packages somewhere and be able to directly use them, without needing to create a project environment for them.\nStacked environments allow for adding tools to the primary environment. You can push an environment of development tools onto the end of the stack to make them available from the REPL and scripts, but not from inside packages.\n\nnote: Note\nWhen loading a package from another environment in the stack other than the active environment the package is loaded in the context of the active environment. This means that the package will be loaded as if it were imported in the active environment, which may affect how its dependencies versions are resolved. When such a package is precompiling it will be marked as a (serial) precompile job, which means that its dependencies will be precompiled in series within the same job, which will likely be slower.\n\nAt a high-level, each environment conceptually defines three maps: roots, graph and paths. When resolving the meaning of import X, the roots and graph maps are used to determine the identity of X, while the paths map is used to locate the source code of X. The specific roles of the three maps are:\n\nroots: name::Symbol ⟶ uuid::UUID\nAn environment's roots map assigns package names to UUIDs for all the top-level dependencies that the environment makes available to the main project (i.e. the ones that can be loaded in Main). When Julia encounters import X in the main project, it looks up the identity of X as roots[:X].\ngraph: context::UUID ⟶ name::Symbol ⟶ uuid::UUID\nAn environment's graph is a multilevel map which assigns, for each context UUID, a map from names to UUIDs, similar to the roots map but specific to that context. When Julia sees import X in the code of the package whose UUID is context, it looks up the identity of X as graph[context][:X]. In particular, this means that import X can refer to different packages depending on context.\npaths: uuid::UUID × name::Symbol ⟶ path::String\nThe paths map assigns to each package UUID-name pair, the location of that package's entry-point source file. After the identity of X in import X has been resolved to a UUID via roots or graph (depending on whether it is loaded from the main project or a dependency), Julia determines what file to load to acquire X by looking up paths[uuid,:X] in the environment. Including this file should define a module named X. Once this package is loaded, any subsequent import resolving to the same uuid will create a new binding to the already-loaded package module.\n\nEach kind of environment defines these three maps differently, as detailed in the following sections.\n\nnote: Note\nFor ease of understanding, the examples throughout this chapter show full data structures for roots, graph and paths. However, Julia's package loading code does not explicitly create these. Instead, it lazily computes only as much of each structure as it needs to load a given package.","page":"Code Loading"},{"title":"Project environments","location":"manual/code-loading.html#Project-environments","category":"section","text":"A project environment is determined by a directory containing a project file called Project.toml, and optionally a manifest file called Manifest.toml. These files may also be called JuliaProject.toml and JuliaManifest.toml, in which case Project.toml and Manifest.toml are ignored. This allows for coexistence with other tools that might consider files called Project.toml and Manifest.toml significant. For pure Julia projects, however, the names Project.toml and Manifest.toml are preferred. However, from Julia v1.10.8 onwards, (Julia)Manifest-v{major}.{minor}.toml is recognized as a format to make a given julia version use a specific manifest file i.e. in the same folder, a Manifest-v1.11.toml would be used by v1.11 and Manifest.toml by any other julia version.\n\nThe roots, graph and paths maps of a project environment are defined as follows:\n\nThe roots map of the environment is determined by the contents of the project file, specifically, its top-level name and uuid entries and its [deps] section (all optional). Consider the following example project file for the hypothetical application, App, as described earlier:\n\nname = \"App\"\nuuid = \"8f986787-14fe-4607-ba5d-fbff2944afa9\"\n\n[deps]\nPriv = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\nPub  = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"\n\nThis project file implies the following roots map, if it was represented by a Julia dictionary:\n\nroots = Dict(\n    :App  => UUID(\"8f986787-14fe-4607-ba5d-fbff2944afa9\"),\n    :Priv => UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"),\n    :Pub  => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n)\n\nGiven this roots map, in App's code the statement import Priv will cause Julia to look up roots[:Priv], which yields ba13f791-ae1d-465a-978b-69c3ad90f72b, the UUID of the Priv package that is to be loaded in that context. This UUID identifies which Priv package to load and use when the main application evaluates import Priv.\n\nThe dependency graph of a project environment is determined by the contents of the manifest file, if present. If there is no manifest file, graph is empty. A manifest file contains a stanza for each of a project's direct or indirect dependencies. For each dependency, the file lists the package's UUID and a source tree hash or an explicit path to the source code. Consider the following example manifest file for App:\n\n[[Priv]] # the private one\ndeps = [\"Pub\", \"Zebra\"]\nuuid = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\npath = \"deps/Priv\"\n\n[[Priv]] # the public one\nuuid = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\ngit-tree-sha1 = \"1bf63d3be994fe83456a03b874b409cfd59a6373\"\nversion = \"0.1.5\"\n\n[[Pub]]\nuuid = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"\ngit-tree-sha1 = \"9ebd50e2b0dd1e110e842df3b433cb5869b0dd38\"\nversion = \"2.1.4\"\n\n  [Pub.deps]\n  Priv = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\n  Zebra = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\n\n[[Zebra]]\nuuid = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\ngit-tree-sha1 = \"e808e36a5d7173974b90a15a353b564f3494092f\"\nversion = \"3.4.2\"\n\nThis manifest file describes a possible complete dependency graph for the App project:\n\nThere are two different packages named Priv that the application uses. It uses a private package, which is a root dependency, and a public one, which is an indirect dependency through Pub. These are differentiated by their distinct UUIDs, and they have different deps:\nThe private Priv depends on the Pub and Zebra packages.\nThe public Priv has no dependencies.\nThe application also depends on the Pub package, which in turn depends on the public Priv and the same Zebra package that the private Priv package depends on.\n\nThis dependency graph represented as a dictionary, looks like this:\n\ngraph = Dict(\n    # Priv – the private one:\n    UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\") => Dict(\n        :Pub   => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Priv – the public one:\n    UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\") => Dict(),\n    # Pub:\n    UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\") => Dict(\n        :Priv  => UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Zebra:\n    UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\") => Dict(),\n)\n\nGiven this dependency graph, when Julia sees import Priv in the Pub package—which has UUID c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1—it looks up:\n\ngraph[UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\")][:Priv]\n\nand gets 2d15fe94-a1f7-436c-a4d8-07a9a496e01c, which indicates that in the context of the Pub package, import Priv refers to the public Priv package, rather than the private one which the app depends on directly. This is how the name Priv can refer to different packages in the main project than it does in one of its package's dependencies, which allows for duplicate names in the package ecosystem.\n\nWhat happens if import Zebra is evaluated in the main App code base? Since Zebra does not appear in the project file, the import will fail even though Zebra does appear in the manifest file. Moreover, if import Zebra occurs in the public Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—then that would also fail since that Priv package has no declared dependencies in the manifest file and therefore cannot load any packages. The Zebra package can only be loaded by packages for which it appear as an explicit dependency in the manifest file: the  Pub package and one of the Priv packages.\n\nThe paths map of a project environment is extracted from the manifest file. The path of a package uuid named X is determined by these rules (in order):\n\nIf the project file in the directory matches uuid and name X, then either:\nIt has a toplevel entryfile entry, then uuid will be mapped to that path, interpreted relative to the directory containing the project file.\nOtherwise, uuid is mapped to src/X.jl relative to the directory containing the project file.\nIf the above is not the case and the project file has a corresponding manifest file and the manifest contains a stanza matching uuid then:\nIf it has a path entry, use that path (relative to the directory containing the manifest file).\nIf it has a git-tree-sha1 entry, compute a deterministic hash function of uuid and git-tree-sha1—call it slug—and look for a directory named packages/X/$slug in each directory in the Julia DEPOT_PATH global array. Use the first such directory that exists.\nIf this is a directory then uuid is mapped to src/X.jl unless the matching manifest stanza has an entryfile entry in which case this is used. In both cases, these are relative to the directory in 2.1.\n\nIf any of these result in success, the path to the source code entry point will be either that result, the relative path from that result plus src/X.jl; otherwise, there is no path mapping for uuid. When loading X, if no source code path is found, the lookup will fail, and the user may be prompted to install the appropriate package version or to take other corrective action (e.g. declaring X as a dependency).\n\nIn the example manifest file above, to find the path of the first Priv package—the one with UUID ba13f791-ae1d-465a-978b-69c3ad90f72b—Julia looks for its stanza in the manifest file, sees that it has a path entry, looks at deps/Priv relative to the App project directory—let's suppose the App code lives in /home/me/projects/App—sees that /home/me/projects/App/deps/Priv exists and therefore loads Priv from there.\n\nIf, on the other hand, Julia was loading the other Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—it finds its stanza in the manifest, see that it does not have a path entry, but that it does have a git-tree-sha1 entry. It then computes the slug for this UUID/SHA-1 pair, which is HDkrT (the exact details of this computation aren't important, but it is consistent and deterministic). This means that the path to this Priv package will be packages/Priv/HDkrT/src/Priv.jl in one of the package depots. Suppose the contents of DEPOT_PATH is [\"/home/me/.julia\", \"/usr/local/julia\"], then Julia will look at the following paths to see if they exist:\n\n/home/me/.julia/packages/Priv/HDkrT\n/usr/local/julia/packages/Priv/HDkrT\n\nJulia uses the first of these that exists to try to load the public Priv package from the file packages/Priv/HDKrT/src/Priv.jl in the depot where it was found.\n\nHere is a representation of a possible paths map for our example App project environment, as provided in the Manifest given above for the dependency graph, after searching the local file system:\n\npaths = Dict(\n    # Priv – the private one:\n    (UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"), :Priv) =>\n        # relative entry-point inside `App` repo:\n        \"/home/me/projects/App/deps/Priv/src/Priv.jl\",\n    # Priv – the public one:\n    (UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"), :Priv) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Priv/HDkr/src/Priv.jl\",\n    # Pub:\n    (UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"), :Pub) =>\n        # package installed in the user depot:\n        \"/home/me/.julia/packages/Pub/oKpw/src/Pub.jl\",\n    # Zebra:\n    (UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"), :Zebra) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Zebra/me9k/src/Zebra.jl\",\n)\n\nThis example map includes three different kinds of package locations (the first and third are part of the default load path):\n\nThe private Priv package is \"vendored\" inside the App repository.\nThe public Priv and Zebra packages are in the system depot, where packages installed and managed by the system administrator live. These are available to all users on the system.\nThe Pub package is in the user depot, where packages installed by the user live. These are only available to the user who installed them.","page":"Code Loading"},{"title":"Package directories","location":"manual/code-loading.html#Package-directories","category":"section","text":"Package directories provide a simpler kind of environment without the ability to handle name collisions. In a package directory, the set of top-level packages is the set of subdirectories that \"look like\" packages. A package X exists in a package directory if the directory contains one of the following \"entry point\" files:\n\nX.jl\nX/src/X.jl\nX.jl/src/X.jl\n\nWhich dependencies a package in a package directory can import depends on whether the package contains a project file:\n\nIf it has a project file, it can only import those packages which are identified in the [deps] section of the project file.\nIf it does not have a project file, it can import any top-level package—i.e. the same packages that can be loaded in Main or the REPL.\n\nThe roots map is determined by examining the contents of the package directory to generate a list of all packages that exist. Additionally, a UUID will be assigned to each entry as follows: For a given package found inside the folder X...\n\nIf X/Project.toml exists and has a uuid entry, then uuid is that value.\nIf X/Project.toml exists and but does not have a top-level UUID entry, uuid is a dummy UUID generated by hashing the canonical (real) path to X/Project.toml.\nOtherwise (if Project.toml does not exist), then uuid is the all-zero nil UUID.\n\nThe dependency graph of a project directory is determined by the presence and contents of project files in the subdirectory of each package. The rules are:\n\nIf a package subdirectory has no project file, then it is omitted from graph and import statements in its code are treated as top-level, the same as the main project and REPL.\nIf a package subdirectory has a project file, then the graph entry for its UUID is the [deps] map of the project file, which is considered to be empty if the section is absent.\n\nAs an example, suppose a package directory has the following structure and content:\n\nAardvark/\n    src/Aardvark.jl:\n        import Bobcat\n        import Cobra\n\nBobcat/\n    Project.toml:\n        [deps]\n        Cobra = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Bobcat.jl:\n        import Cobra\n        import Dingo\n\nCobra/\n    Project.toml:\n        uuid = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        [deps]\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Cobra.jl:\n        import Dingo\n\nDingo/\n    Project.toml:\n        uuid = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Dingo.jl:\n        # no imports\n\nHere is a corresponding roots structure, represented as a dictionary:\n\nroots = Dict(\n    :Aardvark => UUID(\"00000000-0000-0000-0000-000000000000\"), # no project file, nil UUID\n    :Bobcat   => UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), # dummy UUID based on path\n    :Cobra    => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), # UUID from project file\n    :Dingo    => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), # UUID from project file\n)\n\nHere is the corresponding graph structure, represented as a dictionary:\n\ngraph = Dict(\n    # Bobcat:\n    UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\") => Dict(\n        :Cobra => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"),\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Cobra:\n    UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\") => Dict(\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Dingo:\n    UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\") => Dict(),\n)\n\nA few general rules to note:\n\nA package without a project file can depend on any top-level dependency, and since every package in a package directory is available at the top-level, it can import all packages in the environment.\nA package with a project file cannot depend on one without a project file since packages with project files can only load packages in graph and packages without project files do not appear in graph.\nA package with a project file but no explicit UUID can only be depended on by packages without project files since dummy UUIDs assigned to these packages are strictly internal.\n\nObserve the following specific instances of these rules in our example:\n\nAardvark can import on any of Bobcat, Cobra or Dingo; it does import Bobcat and Cobra.\nBobcat can and does import both Cobra and Dingo, which both have project files with UUIDs and are declared as dependencies in Bobcat's [deps] section.\nBobcat cannot depend on Aardvark since Aardvark does not have a project file.\nCobra can and does import Dingo, which has a project file and UUID, and is declared as a dependency in Cobra's  [deps] section.\nCobra cannot depend on Aardvark or Bobcat since neither have real UUIDs.\nDingo cannot import anything because it has a project file without a [deps] section.\n\nThe paths map in a package directory is simple: it maps subdirectory names to their corresponding entry-point paths. In other words, if the path to our example project directory is /home/me/animals then the paths map could be represented by this dictionary:\n\npaths = Dict(\n    (UUID(\"00000000-0000-0000-0000-000000000000\"), :Aardvark) =>\n        \"/home/me/AnimalPackages/Aardvark/src/Aardvark.jl\",\n    (UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), :Bobcat) =>\n        \"/home/me/AnimalPackages/Bobcat/src/Bobcat.jl\",\n    (UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), :Cobra) =>\n        \"/home/me/AnimalPackages/Cobra/src/Cobra.jl\",\n    (UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), :Dingo) =>\n        \"/home/me/AnimalPackages/Dingo/src/Dingo.jl\",\n)\n\nSince all packages in a package directory environment are, by definition, subdirectories with the expected entry-point files, their paths map entries always have this form.","page":"Code Loading"},{"title":"Environment stacks","location":"manual/code-loading.html#Environment-stacks","category":"section","text":"The third and final kind of environment is one that combines other environments by overlaying several of them, making the packages in each available in a single composite environment. These composite environments are called environment stacks. The Julia LOAD_PATH global defines an environment stack—the environment in which the Julia process operates. If you want your Julia process to have access only to the packages in one project or package directory, make it the only entry in LOAD_PATH. It is often quite useful, however, to have access to some of your favorite tools—standard libraries, profilers, debuggers, personal utilities, etc.—even if they are not dependencies of the project you're working on. By adding an environment containing these tools to the load path, you immediately have access to them in top-level code without needing to add them to your project.\n\nThe mechanism for combining the roots, graph and paths data structures of the components of an environment stack is simple: they are merged as dictionaries, favoring earlier entries over later ones in the case of key collisions. In other words, if we have stack = [env₁, env₂, …] then we have:\n\nroots = reduce(merge, reverse([roots₁, roots₂, …]))\ngraph = reduce(merge, reverse([graph₁, graph₂, …]))\npaths = reduce(merge, reverse([paths₁, paths₂, …]))\n\nThe subscripted rootsᵢ, graphᵢ and pathsᵢ variables correspond to the subscripted environments, envᵢ, contained in stack. The reverse is present because merge favors the last argument rather than first when there are collisions between keys in its argument dictionaries. There are a couple of noteworthy features of this design:\n\nThe primary environment—i.e. the first environment in a stack—is faithfully embedded in a stacked environment. The full dependency graph of the first environment in a stack is guaranteed to be included intact in the stacked environment including the same versions of all dependencies.\nPackages in non-primary environments can end up using incompatible versions of their dependencies even if their own environments are entirely compatible. This can happen when one of their dependencies is shadowed by a version in an earlier environment in the stack (either by graph or path, or both).\n\nSince the primary environment is typically the environment of a project you're working on, while environments later in the stack contain additional tools, this is the right trade-off: it's better to break your development tools but keep the project working. When such incompatibilities occur, you'll typically want to upgrade your dev tools to versions that are compatible with the main project.","page":"Code Loading"},{"title":"Package Extensions","location":"manual/code-loading.html#man-extensions","category":"section","text":"A package \"extension\" is a module that is automatically loaded when a specified set of other packages (its \"triggers\") are loaded in the current Julia session. Extensions are defined under the [extensions] section in the project file. The triggers of an extension are a subset of those packages listed under the [weakdeps] (and possibly, but uncommonly the [deps]) section of the project file. Those packages can have compat entries like other packages.\n\nname = \"MyPackage\"\n\n[compat]\nExtDep = \"1.0\"\nOtherExtDep = \"1.0\"\n\n[weakdeps]\nExtDep = \"c9a23...\" # uuid\nOtherExtDep = \"862e...\" # uuid\n\n[extensions]\nBarExt = [\"ExtDep\", \"OtherExtDep\"]\nFooExt = \"ExtDep\"\n...\n\nThe keys under extensions are the names of the extensions. They are loaded when all the packages on the right hand side (the triggers) of that extension are loaded. If an extension only has one trigger the list of triggers can be written as just a string for brevity. The location for the entry point of the extension is either in ext/FooExt.jl or ext/FooExt/FooExt.jl for extension FooExt. The content of an extension is often structured as:\n\nmodule FooExt\n\n# Load main package and triggers\nusing MyPackage, ExtDep\n\n# Extend functionality in main package with types from the triggers\nMyPackage.func(x::ExtDep.SomeStruct) = ...\n\nend\n\nWhen a package with extensions is added to an environment, the weakdeps and extensions sections are stored in the manifest file in the section for that package. The dependency lookup rules for a package are the same as for its \"parent\" except that the listed triggers are also considered as dependencies.","page":"Code Loading"},{"title":"Workspaces","location":"manual/code-loading.html#workspaces","category":"section","text":"A project file can define a workspace by giving a set of projects that is part of that workspace:\n\n[workspace]\nprojects = [\"test\", \"benchmarks\", \"docs\", \"SomePackage\"]\n\nEach project listed in the projects array is specified by its relative path from the workspace root. This can be a direct child directory (e.g., \"test\") or a nested subdirectory (e.g., \"nested/subdir/MyPackage\"). Each project contains its own Project.toml file, which may include additional dependencies and compatibility constraints. In such cases, the package manager gathers all dependency information from all the projects in the workspace generating a single manifest file that combines the versions of all dependencies.\n\nWhen Julia loads a project, it searches upward through parent directories until it reaches the user's home directory to find a workspace that includes that project. This allows workspace projects to be nested at arbitrary depth within the workspace directory tree.\n\nFurthermore, workspaces can be \"nested\", meaning a project defining a workspace can also be part of another workspace. In this scenario, a single manifest file is still utilized, stored alongside the \"root project\" (the project that doesn't have another workspace including it). An example file structure could look like this:\n\nProject.toml # projects = [\"MyPackage\"]\nManifest.toml\nMyPackage/\n    Project.toml # projects = [\"test\"]\n    test/\n        Project.toml","page":"Code Loading"},{"title":"Package/Environment Preferences","location":"manual/code-loading.html#preferences","category":"section","text":"Preferences are dictionaries of metadata that influence package behavior within an environment. The preferences system supports reading preferences at compile-time, which means that at code-loading time, we must ensure that the precompilation files selected by Julia were built with the same preferences as the current environment before loading them. The public API for modifying Preferences is contained within the Preferences.jl package. Preferences are stored as TOML dictionaries within a (Julia)LocalPreferences.toml file next to the currently-active project. If a preference is \"exported\", it is instead stored within the (Julia)Project.toml instead. The intention is to allow shared projects to contain shared preferences, while allowing for users themselves to override those preferences with their own settings in the LocalPreferences.toml file, which should be .gitignored as the name implies.\n\nPreferences that are accessed during compilation are automatically marked as compile-time preferences, and any change recorded to these preferences will cause the Julia compiler to recompile any cached precompilation file(s) (.ji and corresponding .so, .dll, or .dylib files) for that module. This is done by serializing the hash of all compile-time preferences during compilation, then checking that hash against the current environment when searching for the proper file(s) to load.\n\nPreferences can be set with depot-wide defaults; if package Foo is installed within your global environment and it has preferences set, these preferences will apply as long as your global environment is part of your LOAD_PATH. Preferences in environments higher up in the environment stack get overridden by the more proximal entries in the load path, ending with the currently active project. This allows depot-wide preference defaults to exist, with active projects able to merge or even completely overwrite these inherited preferences. See the docstring for Preferences.set_preferences!() for the full details of how to set preferences to allow or disallow merging.","page":"Code Loading"},{"title":"Conclusion","location":"manual/code-loading.html#Conclusion","category":"section","text":"Federated package management and precise software reproducibility are difficult but worthy goals in a package system. In combination, these goals lead to a more complex package loading mechanism than most dynamic languages have, but it also yields scalability and reproducibility that is more commonly associated with static languages. Typically, Julia users should be able to use the built-in package manager to manage their projects without needing a precise understanding of these interactions. A call to Pkg.add(\"X\") will add to the appropriate project and manifest files, selected via Pkg.activate(\"Y\"), so that a future call to import X will load X without further thought.","page":"Code Loading"},{"title":"Artifacts","location":"stdlib/Artifacts.html#Artifacts","category":"section","text":"Starting with Julia 1.6, the artifacts support has moved from Pkg.jl to Julia itself. Until proper documentation can be added here, you can learn more about artifacts in the Pkg.jl manual at https://julialang.github.io/Pkg.jl/v1/artifacts/.\n\ncompat: Julia 1.6\nJulia's artifacts API requires at least Julia 1.6. In Julia versions 1.3 to 1.5, you can use Pkg.Artifacts instead.","page":"Artifacts"},{"title":"Artifacts.artifact_meta","location":"stdlib/Artifacts.html#Artifacts.artifact_meta","category":"function","text":"artifact_meta(name::String, artifacts_toml::String;\n              platform::AbstractPlatform = HostPlatform(),\n              pkg_uuid::Union{Base.UUID,Nothing}=nothing)\n\nGet metadata about a given artifact (identified by name) stored within the given (Julia)Artifacts.toml file.  If the artifact is platform-specific, use platform to choose the most appropriate mapping.  If none is found, return nothing.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.artifact_hash","location":"stdlib/Artifacts.html#Artifacts.artifact_hash","category":"function","text":"artifact_hash(name::String, artifacts_toml::String;\n              platform::AbstractPlatform = HostPlatform())\n\nThin wrapper around artifact_meta() to return the hash of the specified, platform- collapsed artifact.  Returns nothing if no mapping can be found.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.find_artifacts_toml","location":"stdlib/Artifacts.html#Artifacts.find_artifacts_toml","category":"function","text":"find_artifacts_toml(path::String)\n\nGiven the path to a .jl file, (such as the one returned by __source__.file in a macro context), find the (Julia)Artifacts.toml that is contained within the containing project (if it exists), otherwise return nothing.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.@artifact_str","location":"stdlib/Artifacts.html#Artifacts.@artifact_str","category":"macro","text":"macro artifact_str(name)\n\nReturn the on-disk path to an artifact. Automatically looks the artifact up by name in the project's (Julia)Artifacts.toml file. Throws an error on if the requested artifact is not present. If run in the REPL, searches for the toml file starting in the current directory, see find_artifacts_toml() for more.\n\nIf the artifact is marked \"lazy\" and the package has using LazyArtifacts defined, the artifact will be downloaded on-demand with Pkg the first time this macro tries to compute the path. The files will then be left installed locally for later.\n\nIf name contains a forward or backward slash, all elements after the first slash will be taken to be path names indexing into the artifact, allowing for an easy one-liner to access a single file/directory within an artifact.  Example:\n\nffmpeg_path = @artifact\"FFMPEG/bin/ffmpeg\"\n\ncompat: Julia 1.3\nThis macro requires at least Julia 1.3.\n\ncompat: Julia 1.6\nSlash-indexing requires at least Julia 1.6.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.artifact_exists","location":"stdlib/Artifacts.html#Artifacts.artifact_exists","category":"function","text":"artifact_exists(hash::SHA1; honor_overrides::Bool=true)\n\nReturn whether or not the given artifact (identified by its sha1 git tree hash) exists on-disk.  Note that it is possible that the given artifact exists in multiple locations (e.g. within multiple depots).\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.artifact_path","location":"stdlib/Artifacts.html#Artifacts.artifact_path","category":"function","text":"artifact_path(hash::SHA1; honor_overrides::Bool=true)\n\nGiven an artifact (identified by SHA1 git tree hash), return its installation path.  If the artifact does not exist, returns the location it would be installed to.\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Artifacts.select_downloadable_artifacts","location":"stdlib/Artifacts.html#Artifacts.select_downloadable_artifacts","category":"function","text":"select_downloadable_artifacts(artifacts_toml::String;\n                              platform = HostPlatform,\n                              include_lazy = false,\n                              pkg_uuid = nothing)\n\nReturn a dictionary where every entry is an artifact from the given Artifacts.toml that should be downloaded for the requested platform.  Lazy artifacts are included if include_lazy is set.\n\n\n\n\n\n","page":"Artifacts"},{"title":"Interfaces","location":"manual/interfaces.html#Interfaces","category":"section","text":"A lot of the power and extensibility in Julia comes from a collection of informal interfaces.  By extending a few specific methods to work for a custom type, objects of that type not only receive those functionalities, but they are also able to be used in other methods that are written to generically build upon those behaviors.","page":"Interfaces"},{"title":"Iteration","location":"manual/interfaces.html#man-interface-iteration","category":"section","text":"There are two methods that are always required:\n\nRequired method Brief description\niterate(iter) Returns either a tuple of the first item and initial state or nothing if empty\niterate(iter, state) Returns either a tuple of the next item and next state or nothing if no items remain\n\nThere are several more methods that should be defined in some circumstances. Please note that you should always define at least one of Base.IteratorSize(IterType) and length(iter) because the default definition of Base.IteratorSize(IterType) is Base.HasLength().\n\nMethod When should this method be defined? Default definition Brief description\nBase.IteratorSize(IterType) If default is not appropriate Base.HasLength() One of Base.HasLength(), Base.HasShape{N}(), Base.IsInfinite(), or Base.SizeUnknown() as appropriate\nlength(iter) If Base.IteratorSize() returns Base.HasLength() or Base.HasShape{N}() (undefined) The number of items, if known\nsize(iter, [dim]) If Base.IteratorSize() returns Base.HasShape{N}() (undefined) The number of items in each dimension, if known\nBase.IteratorEltype(IterType) If default is not appropriate Base.HasEltype() Either Base.EltypeUnknown() or Base.HasEltype() as appropriate\neltype(IterType) If default is not appropriate Any The type of the first entry of the tuple returned by iterate()\nBase.isdone(iter, [state]) Must be defined if iterator is stateful missing Fast-path hint for iterator completion. If not defined for a stateful iterator then functions that check for done-ness, like isempty() and zip(), may mutate the iterator and cause buggy behaviour!\n\nSequential iteration is implemented by the iterate function. Instead of mutating objects as they are iterated over, Julia iterators may keep track of the iteration state externally from the object. The return value from iterate is always either a tuple of a value and a state, or nothing if no elements remain. The state object will be passed back to the iterate function on the next iteration and is generally considered an implementation detail private to the iterable object.\n\nAny object that defines this function is iterable and can be used in the many functions that rely upon iteration. It can also be used directly in a for loop since the syntax:\n\nfor item in iter   # or  \"for item = iter\"\n    # body\nend\n\nis translated into:\n\nnext = iterate(iter)\nwhile next !== nothing\n    (item, state) = next\n    # body\n    next = iterate(iter, state)\nend\n\nA simple example is an iterable sequence of square numbers with a defined length:\n\njulia> struct Squares\n           count::Int\n       end\n\njulia> Base.iterate(S::Squares, state=1) = state > S.count ? nothing : (state*state, state+1)\n\nWith only iterate definition, the Squares type is already pretty powerful. We can iterate over all the elements:\n\njulia> for item in Squares(7)\n           println(item)\n       end\n1\n4\n9\n16\n25\n36\n49\n\nWe can use many of the builtin methods that work with iterables, like in or sum:\n\njulia> 25 in Squares(10)\ntrue\n\njulia> sum(Squares(100))\n338350\n\nThere are a few more methods we can extend to give Julia more information about this iterable collection. We know that the elements in a Squares sequence will always be Int. By extending the eltype method, we can give that information to Julia and help it make more specialized code in the more complicated methods. We also know the number of elements in our sequence, so we can extend length, too:\n\njulia> Base.eltype(::Type{Squares}) = Int # Note that this is defined for the type\n\njulia> Base.length(S::Squares) = S.count\n\nNow, when we ask Julia to collect all the elements into an array it can preallocate a Vector{Int} of the right size instead of naively push!ing each element into a Vector{Any}:\n\njulia> collect(Squares(4))\n4-element Vector{Int64}:\n  1\n  4\n  9\n 16\n\nWhile we can rely upon generic implementations, we can also extend specific methods where we know there is a simpler algorithm. For example, there's a formula to compute the sum of squares, so we can override the generic iterative version with a more performant solution:\n\njulia> Base.sum(S::Squares) = (n = S.count; return n*(n+1)*(2n+1)÷6)\n\njulia> sum(Squares(1803))\n1955361914\n\nThis is a very common pattern throughout Julia Base: a small set of required methods define an informal interface that enable many fancier behaviors. In some cases, types will want to additionally specialize those extra behaviors when they know a more efficient algorithm can be used in their specific case.\n\nIt is also often useful to allow iteration over a collection in reverse order by iterating over Iterators.reverse(iterator). To actually support reverse-order iteration, however, an iterator type T needs to implement iterate for Iterators.Reverse{T}. (Given r::Iterators.Reverse{T}, the underling iterator of type T is r.itr.) In our Squares example, we would implement Iterators.Reverse{Squares} methods:\n\njulia> Base.iterate(rS::Iterators.Reverse{Squares}, state=rS.itr.count) = state < 1 ? nothing : (state*state, state-1)\n\njulia> collect(Iterators.reverse(Squares(4)))\n4-element Vector{Int64}:\n 16\n  9\n  4\n  1","page":"Interfaces"},{"title":"Indexing","location":"manual/interfaces.html#Indexing","category":"section","text":"Methods to implement Brief description\ngetindex(X, i) X[i], indexed access, non-scalar i should allocate a copy\nsetindex!(X, v, i) X[i] = v, indexed assignment\nfirstindex(X) The first index, used in X[begin]\nlastindex(X) The last index, used in X[end]\n\nFor the Squares iterable above, we can easily compute the ith element of the sequence by squaring it. We can expose this as an indexing expression S[i]. To opt into this behavior, Squares simply needs to define getindex:\n\njulia> function Base.getindex(S::Squares, i::Int)\n           1 <= i <= S.count || throw(BoundsError(S, i))\n           return i*i\n       end\n\njulia> Squares(100)[23]\n529\n\nAdditionally, to support the syntax S[begin] and S[end], we must define firstindex and lastindex to specify the first and last valid indices, respectively:\n\njulia> Base.firstindex(S::Squares) = 1\n\njulia> Base.lastindex(S::Squares) = length(S)\n\njulia> Squares(23)[end]\n529\n\nFor multi-dimensional begin/end indexing as in a[3, begin, 7], for example, you should define firstindex(a, dim) and lastindex(a, dim) (which default to calling first and last on axes(a, dim), respectively).\n\nNote, though, that the above only defines getindex with one integer index. Indexing with anything other than an Int will throw a MethodError saying that there was no matching method. In order to support indexing with ranges or vectors of Ints, separate methods must be written:\n\njulia> Base.getindex(S::Squares, i::Number) = S[convert(Int, i)]\n\njulia> Base.getindex(S::Squares, I) = [S[i] for i in I]\n\njulia> Squares(10)[[3,4.,5]]\n3-element Vector{Int64}:\n  9\n 16\n 25\n\nWhile this is starting to support more of the indexing operations supported by some of the builtin types, there's still quite a number of behaviors missing. This Squares sequence is starting to look more and more like a vector as we've added behaviors to it. Instead of defining all these behaviors ourselves, we can officially define it as a subtype of an AbstractArray.","page":"Interfaces"},{"title":"Abstract Arrays","location":"manual/interfaces.html#man-interface-array","category":"section","text":"Methods to implement  Brief description\nsize(A)  Returns a tuple containing the dimensions of A\ngetindex(A, i::Int)  (if IndexLinear) Linear scalar indexing\ngetindex(A, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexing\nOptional methods Default definition Brief description\nIndexStyle(::Type) IndexCartesian() Returns either IndexLinear() or IndexCartesian(). See the description below.\nsetindex!(A, v, i::Int)  (if IndexLinear) Scalar indexed assignment\nsetindex!(A, v, I::Vararg{Int, N})  (if IndexCartesian, where N = ndims(A)) N-dimensional scalar indexed assignment\ngetindex(A, I...) defined in terms of scalar getindex Multidimensional and nonscalar indexing\nsetindex!(A, X, I...) defined in terms of scalar setindex! Multidimensional and nonscalar indexed assignment\niterate defined in terms of scalar getindex Iteration\nlength(A) prod(size(A)) Number of elements\nsimilar(A) similar(A, eltype(A), size(A)) Return a mutable array with the same shape and element type\nsimilar(A, ::Type{S}) similar(A, S, size(A)) Return a mutable array with the same shape and the specified element type\nsimilar(A, dims::Dims) similar(A, eltype(A), dims) Return a mutable array with the same element type and size dims\nsimilar(A, ::Type{S}, dims::Dims) Array{S}(undef, dims) Return a mutable array with the specified element type and size\nNon-traditional indices Default definition Brief description\naxes(A) map(OneTo, size(A)) Return a tuple of AbstractUnitRange{<:Integer} of valid indices. The axes should be their own axes, that is axes.(axes(A),1) == axes(A) should be satisfied.\nsimilar(A, ::Type{S}, inds) similar(A, S, Base.to_shape(inds)) Return a mutable array with the specified indices inds (see below)\nsimilar(T::Union{Type,Function}, inds) T(Base.to_shape(inds)) Return an array similar to T with the specified indices inds (see below)\n\nIf a type is defined as a subtype of AbstractArray, it inherits a very large set of rich behaviors including iteration and multidimensional indexing built on top of single-element access. See the arrays manual page and the Julia Base section for more supported methods.\n\nA key part in defining an AbstractArray subtype is IndexStyle. Since indexing is such an important part of an array and often occurs in hot loops, it's important to make both indexing and indexed assignment as efficient as possible. Array data structures are typically defined in one of two ways: either it most efficiently accesses its elements using just one index (linear indexing) or it intrinsically accesses the elements with indices specified for every dimension.  These two modalities are identified by Julia as IndexLinear() and IndexCartesian().  Converting a linear index to multiple indexing subscripts is typically very expensive, so this provides a traits-based mechanism to enable efficient generic code for all array types.\n\nThis distinction determines which scalar indexing methods the type must define. IndexLinear() arrays are simple: just define getindex(A::ArrayType, i::Int). When the array is subsequently indexed with a multidimensional set of indices, the fallback getindex(A::AbstractArray, I...) efficiently converts the indices into one linear index and then calls the above method. IndexCartesian() arrays, on the other hand, require methods to be defined for each supported dimensionality with ndims(A) Int indices. For example, SparseMatrixCSC from the SparseArrays standard library module, only supports two dimensions, so it just defines getindex(A::SparseMatrixCSC, i::Int, j::Int). The same holds for setindex!.\n\nReturning to the sequence of squares from above, we could instead define it as a subtype of an AbstractArray{Int, 1}:\n\njulia> struct SquaresVector <: AbstractArray{Int, 1}\n           count::Int\n       end\n\njulia> Base.size(S::SquaresVector) = (S.count,)\n\njulia> Base.IndexStyle(::Type{<:SquaresVector}) = IndexLinear()\n\njulia> Base.getindex(S::SquaresVector, i::Int) = i*i\n\nNote that it's very important to specify the two parameters of the AbstractArray; the first defines the eltype, and the second defines the ndims. That supertype and those three methods are all it takes for SquaresVector to be an iterable, indexable, and completely functional array:\n\njulia> s = SquaresVector(4)\n4-element SquaresVector:\n  1\n  4\n  9\n 16\n\njulia> s[s .> 8]\n2-element Vector{Int64}:\n  9\n 16\n\njulia> s + s\n4-element Vector{Int64}:\n  2\n  8\n 18\n 32\n\njulia> sin.(s)\n4-element Vector{Float64}:\n  0.8414709848078965\n -0.7568024953079282\n  0.4121184852417566\n -0.2879033166650653\n\nAs a more complicated example, let's define our own toy N-dimensional sparse-like array type built on top of Dict:\n\njulia> struct SparseArray{T,N} <: AbstractArray{T,N}\n           data::Dict{NTuple{N,Int}, T}\n           dims::NTuple{N,Int}\n       end\n\njulia> SparseArray(::Type{T}, dims::Int...) where {T} = SparseArray(T, dims);\n\njulia> SparseArray(::Type{T}, dims::NTuple{N,Int}) where {T,N} = SparseArray{T,N}(Dict{NTuple{N,Int}, T}(), dims);\n\njulia> Base.size(A::SparseArray) = A.dims\n\njulia> Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where {T} = SparseArray(T, dims)\n\njulia> Base.getindex(A::SparseArray{T,N}, I::Vararg{Int,N}) where {T,N} = get(A.data, I, zero(T))\n\njulia> Base.setindex!(A::SparseArray{T,N}, v, I::Vararg{Int,N}) where {T,N} = (A.data[I] = v)\n\nNotice that this is an IndexCartesian array, so we must manually define getindex and setindex! at the dimensionality of the array. Unlike the SquaresVector, we are able to define setindex!, and so we can mutate the array:\n\njulia> A = SparseArray(Float64, 3, 3)\n3×3 SparseArray{Float64, 2}:\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n 0.0  0.0  0.0\n\njulia> fill!(A, 2)\n3×3 SparseArray{Float64, 2}:\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n 2.0  2.0  2.0\n\njulia> A[:] = 1:length(A); A\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0\n\nThe result of indexing an AbstractArray can itself be an array (for instance when indexing by an AbstractRange). The AbstractArray fallback methods use similar to allocate an Array of the appropriate size and element type, which is filled in using the basic indexing method described above. However, when implementing an array wrapper you often want the result to be wrapped as well:\n\njulia> A[1:2,:]\n2×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n\nIn this example it is accomplished by defining Base.similar(A::SparseArray, ::Type{T}, dims::Dims) where T to create the appropriate wrapped array. (Note that while similar supports 1- and 2-argument forms, in most case you only need to specialize the 3-argument form.) For this to work it's important that SparseArray is mutable (supports setindex!). Defining similar, getindex and setindex! for SparseArray also makes it possible to copy the array:\n\njulia> copy(A)\n3×3 SparseArray{Float64, 2}:\n 1.0  4.0  7.0\n 2.0  5.0  8.0\n 3.0  6.0  9.0\n\nIn addition to all the iterable and indexable methods from above, these types can also interact with each other and use most of the methods defined in Julia Base for AbstractArrays:\n\njulia> A[SquaresVector(3)]\n3-element SparseArray{Float64, 1}:\n 1.0\n 4.0\n 9.0\n\njulia> sum(A)\n45.0\n\nIf you are defining an array type that allows non-traditional indexing (indices that start at something other than 1), you should specialize axes. You should also specialize similar so that the dims argument (ordinarily a Dims size-tuple) can accept AbstractUnitRange objects, perhaps range-types Ind of your own design. For more information, see Arrays with custom indices.","page":"Interfaces"},{"title":"Strided Arrays","location":"manual/interfaces.html#man-interface-strided-arrays","category":"section","text":"Methods to implement  Brief description\nstrides(A)  Return the distance in memory (in number of elements) between adjacent elements in each dimension as a tuple. If A is an AbstractArray{T,0}, this should return an empty tuple.\nBase.unsafe_convert(::Type{Ptr{T}}, A)  Return the native address of an array.\nBase.elsize(::Type{<:A})  Return the stride between consecutive elements in the array.\nOptional methods Default definition Brief description\nstride(A, i::Int) strides(A)[i] Return the distance in memory (in number of elements) between adjacent elements in dimension k.\n\nA strided array is a subtype of AbstractArray whose entries are stored in memory with fixed strides. Provided the element type of the array is compatible with BLAS, a strided array can utilize BLAS and LAPACK routines for more efficient linear algebra routines. A typical example of a user-defined strided array is one that wraps a standard Array with additional structure.\n\nWarning: do not implement these methods if the underlying storage is not actually strided, as it may lead to incorrect results or segmentation faults.\n\nHere are some examples to demonstrate which type of arrays are strided and which are not:\n\n1:5   # not strided (there is no storage associated with this array.)\nVector(1:5)  # is strided with strides (1,)\nA = [1 5; 2 6; 3 7; 4 8]  # is strided with strides (1,4)\nV = view(A, 1:2, :)   # is strided with strides (1,4)\nV = view(A, 1:2:3, 1:2)   # is strided with strides (2,4)\nV = view(A, [1,2,4], :)   # is not strided, as the spacing between rows is not fixed.","page":"Interfaces"},{"title":"Customizing broadcasting","location":"manual/interfaces.html#man-interfaces-broadcasting","category":"section","text":"Methods to implement Brief description\nBase.BroadcastStyle(::Type{SrcType}) = SrcStyle() Broadcasting behavior of SrcType\nBase.similar(bc::Broadcasted{DestStyle}, ::Type{ElType}) Allocation of output container\nOptional methods \nBase.BroadcastStyle(::Style1, ::Style2) = Style12() Precedence rules for mixing styles\nBase.axes(x) Declaration of the indices of x, as per axes(x).\nBase.broadcastable(x) Convert x to an object that has axes and supports indexing\nBypassing default machinery \nBase.copy(bc::Broadcasted{DestStyle}) Custom implementation of broadcast\nBase.copyto!(dest, bc::Broadcasted{DestStyle}) Custom implementation of broadcast!, specializing on DestStyle\nBase.copyto!(dest::DestType, bc::Broadcasted{Nothing}) Custom implementation of broadcast!, specializing on DestType\nBase.Broadcast.broadcasted(f, args...) Override the default lazy behavior within a fused expression\nBase.Broadcast.instantiate(bc::Broadcasted{DestStyle}) Override the computation of the lazy broadcast's axes\n\nBroadcasting is triggered by an explicit call to broadcast or broadcast!, or implicitly by \"dot\" operations like A .+ b or f.(x, y). Any object that has axes and supports indexing can participate as an argument in broadcasting, and by default the result is stored in an Array. This basic framework is extensible in three major ways:\n\nEnsuring that all arguments support broadcast\nSelecting an appropriate output array for the given set of arguments\nSelecting an efficient implementation for the given set of arguments\n\nNot all types support axes and indexing, but many are convenient to allow in broadcast. The Base.broadcastable function is called on each argument to broadcast, allowing it to return something different that supports axes and indexing. By default, this is the identity function for all AbstractArrays and Numbers — they already support axes and indexing.\n\nIf a type is intended to act like a \"0-dimensional scalar\" (a single object) rather than as a container for broadcasting, then the following method should be defined:\n\nBase.broadcastable(o::MyType) = Ref(o)\n\nthat returns the argument wrapped in a 0-dimensional Ref container. For example, such a wrapper method is defined for types themselves, functions, special singletons like missing and nothing, and dates.\n\nCustom array-like types can specialize Base.broadcastable to define their shape, but they should follow the convention that collect(Base.broadcastable(x)) == collect(x). A notable exception is AbstractString; strings are special-cased to behave as scalars for the purposes of broadcast even though they are iterable collections of their characters (see Strings for more).\n\nThe next two steps (selecting the output array and implementation) are dependent upon determining a single answer for a given set of arguments. Broadcast must take all the varied types of its arguments and collapse them down to just one output array and one implementation. Broadcast calls this single answer a \"style\". Every broadcastable object each has its own preferred style, and a promotion-like system is used to combine these styles into a single answer — the \"destination style\".","page":"Interfaces"},{"title":"Broadcast Styles","location":"manual/interfaces.html#Broadcast-Styles","category":"section","text":"Base.BroadcastStyle is the abstract type from which all broadcast styles are derived. When used as a function it has two possible forms, unary (single-argument) and binary. The unary variant states that you intend to implement specific broadcasting behavior and/or output type, and do not wish to rely on the default fallback Broadcast.DefaultArrayStyle.\n\nTo override these defaults, you can define a custom BroadcastStyle for your object:\n\nstruct MyStyle <: Broadcast.BroadcastStyle end\nBase.BroadcastStyle(::Type{<:MyType}) = MyStyle()\n\nIn some cases it might be convenient not to have to define MyStyle, in which case you can leverage one of the general broadcast wrappers:\n\nBase.BroadcastStyle(::Type{<:MyType}) = Broadcast.Style{MyType}() can be used for arbitrary types.\nBase.BroadcastStyle(::Type{<:MyType}) = Broadcast.ArrayStyle{MyType}() is preferred if MyType is an AbstractArray.\nFor AbstractArrays that only support a certain dimensionality, create a subtype of Broadcast.AbstractArrayStyle{N} (see below).\n\nWhen your broadcast operation involves several arguments, individual argument styles get combined to determine a single DestStyle that controls the type of the output container. For more details, see below.","page":"Interfaces"},{"title":"Selecting an appropriate output array","location":"manual/interfaces.html#Selecting-an-appropriate-output-array","category":"section","text":"The broadcast style is computed for every broadcasting operation to allow for dispatch and specialization. The actual allocation of the result array is handled by similar, using the Broadcasted object as its first argument.\n\nBase.similar(bc::Broadcasted{DestStyle}, ::Type{ElType})\n\nThe fallback definition is\n\nsimilar(bc::Broadcasted{DefaultArrayStyle{N}}, ::Type{ElType}) where {N,ElType} =\n    similar(Array{ElType}, axes(bc))\n\nHowever, if needed you can specialize on any or all of these arguments. The final argument bc is a lazy representation of a (potentially fused) broadcast operation, a Broadcasted object. For these purposes, the most important fields of the wrapper are f and args, describing the function and argument list, respectively. Note that the argument list can — and often does — include other nested Broadcasted wrappers.\n\nFor a complete example, let's say you have created a type, ArrayAndChar, that stores an array and a single character:\n\nstruct ArrayAndChar{T,N} <: AbstractArray{T,N}\n    data::Array{T,N}\n    char::Char\nend\nBase.size(A::ArrayAndChar) = size(A.data)\nBase.getindex(A::ArrayAndChar{T,N}, inds::Vararg{Int,N}) where {T,N} = A.data[inds...]\nBase.setindex!(A::ArrayAndChar{T,N}, val, inds::Vararg{Int,N}) where {T,N} = A.data[inds...] = val\nBase.showarg(io::IO, A::ArrayAndChar, toplevel) = print(io, typeof(A), \" with char '\", A.char, \"'\")\n\nYou might want broadcasting to preserve the char \"metadata\". First we define\n\nBase.BroadcastStyle(::Type{<:ArrayAndChar}) = Broadcast.ArrayStyle{ArrayAndChar}()\n\nThis means we must also define a corresponding similar method:\n\nfunction Base.similar(bc::Broadcast.Broadcasted{Broadcast.ArrayStyle{ArrayAndChar}}, ::Type{ElType}) where ElType\n    # Scan the inputs for the ArrayAndChar:\n    A = find_aac(bc)\n    # Use the char field of A to create the output\n    ArrayAndChar(similar(Array{ElType}, axes(bc)), A.char)\nend\n\n\"`A = find_aac(As)` returns the first ArrayAndChar among the arguments.\"\nfind_aac(bc::Base.Broadcast.Broadcasted) = find_aac(bc.args)\nfind_aac(args::Tuple) = find_aac(find_aac(args[1]), Base.tail(args))\nfind_aac(x) = x\nfind_aac(::Tuple{}) = nothing\nfind_aac(a::ArrayAndChar, rest) = a\nfind_aac(::Any, rest) = find_aac(rest)\n\nFrom these definitions, one obtains the following behavior:\n\njulia> a = ArrayAndChar([1 2; 3 4], 'x')\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 1  2\n 3  4\n\njulia> a .+ 1\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n 2  3\n 4  5\n\njulia> a .+ [5,10]\n2×2 ArrayAndChar{Int64, 2} with char 'x':\n  6   7\n 13  14","page":"Interfaces"},{"title":"Extending broadcast with custom implementations","location":"manual/interfaces.html#extending-in-place-broadcast","category":"section","text":"In general, a broadcast operation is represented by a lazy Broadcasted container that holds onto the function to be applied alongside its arguments. Those arguments may themselves be more nested Broadcasted containers, forming a large expression tree to be evaluated. A nested tree of Broadcasted containers is directly constructed by the implicit dot syntax; 5 .+ 2.*x is transiently represented by Broadcasted(+, 5, Broadcasted(*, 2, x)), for example. This is invisible to users as it is immediately realized through a call to copy, but it is this container that provides the basis for broadcast's extensibility for authors of custom types. The built-in broadcast machinery will then determine the result type and size based upon the arguments, allocate it, and then finally copy the realization of the Broadcasted object into it with a default copyto!(::AbstractArray, ::Broadcasted) method. The built-in fallback broadcast and broadcast! methods similarly construct a transient Broadcasted representation of the operation so they can follow the same codepath. This allows custom array implementations to provide their own copyto! specialization to customize and optimize broadcasting. This is again determined by the computed broadcast style. This is such an important part of the operation that it is stored as the first type parameter of the Broadcasted type, allowing for dispatch and specialization.\n\nFor some types, the machinery to \"fuse\" operations across nested levels of broadcasting is not available or could be done more efficiently incrementally. In such cases, you may need or want to evaluate x .* (x .+ 1) as if it had been written broadcast(*, x, broadcast(+, x, 1)), where the inner operation is evaluated before tackling the outer operation. This sort of eager operation is directly supported by a bit of indirection; instead of directly constructing Broadcasted objects, Julia lowers the fused expression x .* (x .+ 1) to Broadcast.broadcasted(*, x, Broadcast.broadcasted(+, x, 1)). Now, by default, broadcasted just calls the Broadcasted constructor to create the lazy representation of the fused expression tree, but you can choose to override it for a particular combination of function and arguments.\n\nAs an example, the builtin AbstractRange objects use this machinery to optimize pieces of broadcasted expressions that can be eagerly evaluated purely in terms of the start, step, and length (or stop) instead of computing every single element. Just like all the other machinery, broadcasted also computes and exposes the combined broadcast style of its arguments, so instead of specializing on broadcasted(f, args...), you can specialize on broadcasted(::DestStyle, f, args...) for any combination of style, function, and arguments.\n\nFor example, the following definition supports the negation of ranges:\n\nbroadcasted(::DefaultArrayStyle{1}, ::typeof(-), r::OrdinalRange) = range(-first(r), step=-step(r), length=length(r))","page":"Interfaces"},{"title":"Extending in-place broadcasting","location":"manual/interfaces.html#extending-in-place-broadcast-2","category":"section","text":"In-place broadcasting can be supported by defining the appropriate copyto!(dest, bc::Broadcasted) method. Because you might want to specialize either on dest or the specific subtype of bc, to avoid ambiguities between packages we recommend the following convention.\n\nIf you wish to specialize on a particular style DestStyle, define a method for\n\ncopyto!(dest, bc::Broadcasted{DestStyle})\n\nOptionally, with this form you can also specialize on the type of dest.\n\nIf instead you want to specialize on the destination type DestType without specializing on DestStyle, then you should define a method with the following signature:\n\ncopyto!(dest::DestType, bc::Broadcasted{Nothing})\n\nThis leverages a fallback implementation of copyto! that converts the wrapper into a Broadcasted{Nothing}. Consequently, specializing on DestType has lower precedence than methods that specialize on DestStyle.\n\nSimilarly, you can completely override out-of-place broadcasting with a copy(::Broadcasted) method.","page":"Interfaces"},{"title":"Working with Broadcasted objects","location":"manual/interfaces.html#Working-with-Broadcasted-objects","category":"section","text":"In order to implement such a copy or copyto!, method, of course, you must work with the Broadcasted wrapper to compute each element. There are two main ways of doing so:\n\nBroadcast.flatten recomputes the potentially nested operation into a single function and flat list of arguments. You are responsible for implementing the broadcasting shape rules yourself, but this may be helpful in limited situations.\nIterating over the CartesianIndices of the axes(::Broadcasted) and using indexing with the resulting CartesianIndex object to compute the result.","page":"Interfaces"},{"title":"Writing binary broadcasting rules","location":"manual/interfaces.html#writing-binary-broadcasting-rules","category":"section","text":"The precedence rules are defined by binary BroadcastStyle calls:\n\nBase.BroadcastStyle(::Style1, ::Style2) = Style12()\n\nwhere Style12 is the BroadcastStyle you want to choose for outputs involving arguments of Style1 and Style2. For example,\n\nBase.BroadcastStyle(::Broadcast.Style{Tuple}, ::Broadcast.AbstractArrayStyle{0}) = Broadcast.Style{Tuple}()\n\nindicates that Tuple \"wins\" over zero-dimensional arrays (the output container will be a tuple). It is worth noting that you do not need to (and should not) define both argument orders of this call; defining one is sufficient no matter what order the user supplies the arguments in.\n\nFor AbstractArray types, defining a BroadcastStyle supersedes the fallback choice, Broadcast.DefaultArrayStyle. DefaultArrayStyle and the abstract supertype, AbstractArrayStyle, store the dimensionality as a type parameter to support specialized array types that have fixed dimensionality requirements.\n\nDefaultArrayStyle \"loses\" to any other AbstractArrayStyle that has been defined because of the following methods:\n\nBroadcastStyle(a::AbstractArrayStyle{Any}, ::DefaultArrayStyle) = a\nBroadcastStyle(a::AbstractArrayStyle{N}, ::DefaultArrayStyle{N}) where N = a\nBroadcastStyle(a::AbstractArrayStyle{M}, ::DefaultArrayStyle{N}) where {M,N} =\n    typeof(a)(Val(max(M, N)))\n\nYou do not need to write binary BroadcastStyle rules unless you want to establish precedence for two or more non-DefaultArrayStyle types.\n\nIf your array type does have fixed dimensionality requirements, then you should subtype AbstractArrayStyle. For example, the sparse array code has the following definitions:\n\nstruct SparseVecStyle <: Broadcast.AbstractArrayStyle{1} end\nstruct SparseMatStyle <: Broadcast.AbstractArrayStyle{2} end\nBase.BroadcastStyle(::Type{<:SparseVector}) = SparseVecStyle()\nBase.BroadcastStyle(::Type{<:SparseMatrixCSC}) = SparseMatStyle()\n\nWhenever you subtype AbstractArrayStyle, you also need to define rules for combining dimensionalities, by creating a constructor for your style that takes a Val(N) argument. For example:\n\nSparseVecStyle(::Val{0}) = SparseVecStyle()\nSparseVecStyle(::Val{1}) = SparseVecStyle()\nSparseVecStyle(::Val{2}) = SparseMatStyle()\nSparseVecStyle(::Val{N}) where N = Broadcast.DefaultArrayStyle{N}()\n\nThese rules indicate that the combination of a SparseVecStyle with 0- or 1-dimensional arrays yields another SparseVecStyle, that its combination with a 2-dimensional array yields a SparseMatStyle, and anything of higher dimensionality falls back to the dense arbitrary-dimensional framework. These rules allow broadcasting to keep the sparse representation for operations that result in one or two dimensional outputs, but produce an Array for any other dimensionality.","page":"Interfaces"},{"title":"Instance Properties","location":"manual/interfaces.html#man-instance-properties","category":"section","text":"Methods to implement Default definition Brief description\npropertynames(x::ObjType, private::Bool=false) fieldnames(typeof(x)) Return a tuple of the properties (x.property) of an object x. If private=true, also return property names intended to be kept as private\ngetproperty(x::ObjType, s::Symbol) getfield(x, s) Return property s of x. x.s calls getproperty(x, :s).\nsetproperty!(x::ObjType, s::Symbol, v) setfield!(x, s, v) Set property s of x to v. x.s = v calls setproperty!(x, :s, v). Should return v.\n\nSometimes, it is desirable to change how the end-user interacts with the fields of an object. Instead of granting direct access to type fields, an extra layer of abstraction between the user and the code can be provided by overloading object.field. Properties are what the user sees of the object, fields what the object actually is.\n\nBy default, properties and fields are the same. However, this behavior can be changed. For example, take this representation of a point in a plane in polar coordinates:\n\njulia> mutable struct Point\n           r::Float64\n           ϕ::Float64\n       end\n\njulia> p = Point(7.0, pi/4)\nPoint(7.0, 0.7853981633974483)\n\nAs described in the table above dot access p.r is the same as getproperty(p, :r) which is by default the same as getfield(p, :r):\n\njulia> propertynames(p)\n(:r, :ϕ)\n\njulia> getproperty(p, :r), getproperty(p, :ϕ)\n(7.0, 0.7853981633974483)\n\njulia> p.r, p.ϕ\n(7.0, 0.7853981633974483)\n\njulia> getfield(p, :r), getproperty(p, :ϕ)\n(7.0, 0.7853981633974483)\n\nHowever, we may want users to be unaware that Point stores the coordinates as r and ϕ (fields), and instead interact with x and y (properties). The methods in the first column can be defined to add new functionality:\n\njulia> Base.propertynames(::Point, private::Bool=false) = private ? (:x, :y, :r, :ϕ) : (:x, :y)\n\njulia> function Base.getproperty(p::Point, s::Symbol)\n           if s === :x\n               return getfield(p, :r) * cos(getfield(p, :ϕ))\n           elseif s === :y\n               return getfield(p, :r) * sin(getfield(p, :ϕ))\n           else\n               # This allows accessing fields with p.r and p.ϕ\n               return getfield(p, s)\n           end\n       end\n\njulia> function Base.setproperty!(p::Point, s::Symbol, f)\n           if s === :x\n               y = p.y\n               setfield!(p, :r, sqrt(f^2 + y^2))\n               setfield!(p, :ϕ, atan(y, f))\n               return f\n           elseif s === :y\n               x = p.x\n               setfield!(p, :r, sqrt(x^2 + f^2))\n               setfield!(p, :ϕ, atan(f, x))\n               return f\n           else\n               # This allow modifying fields with p.r and p.ϕ\n               return setfield!(p, s, f)\n           end\n       end\n\nIt is important that getfield and setfield are used inside getproperty and setproperty! instead of the dot syntax, since the dot syntax would make the functions recursive which can lead to type inference issues. We can now try out the new functionality:\n\njulia> propertynames(p)\n(:x, :y)\n\njulia> p.x\n4.949747468305833\n\njulia> p.y = 4.0\n4.0\n\njulia> p.r\n6.363961030678928\n\nFinally, it is worth noting that adding instance properties like this is quite rarely done in Julia and should in general only be done if there is a good reason for doing so.","page":"Interfaces"},{"title":"Rounding","location":"manual/interfaces.html#man-rounding-interface","category":"section","text":"Methods to implement Default definition Brief description\nround(x::ObjType, r::RoundingMode) none Round x and return the result. If possible, round should return an object of the same type as x\nround(T::Type, x::ObjType, r::RoundingMode) convert(T, round(x, r)) Round x, returning the result as a T\n\nTo support rounding on a new type it is typically sufficient to define the single method round(x::ObjType, r::RoundingMode). The passed rounding mode determines in which direction the value should be rounded. The most commonly used rounding modes are RoundNearest, RoundToZero, RoundDown, and RoundUp, as these rounding modes are used in the definitions of the one argument round, method, and trunc, floor, and ceil, respectively.\n\nIn some cases, it is possible to define a three-argument round method that is more accurate or performant than the two-argument method followed by conversion. In this case it is acceptable to define the three argument method in addition to the two argument method. If it is impossible to represent the rounded result as an object of the type T, then the three argument method should throw an InexactError.\n\nFor example, if we have an Interval type which represents a range of possible values similar to https://github.com/JuliaPhysics/Measurements.jl, we may define rounding on that type with the following\n\njulia> struct Interval{T}\n           min::T\n           max::T\n       end\n\njulia> Base.round(x::Interval, r::RoundingMode) = Interval(round(x.min, r), round(x.max, r))\n\njulia> x = Interval(1.7, 2.2)\nInterval{Float64}(1.7, 2.2)\n\njulia> round(x)\nInterval{Float64}(2.0, 2.0)\n\njulia> floor(x)\nInterval{Float64}(1.0, 2.0)\n\njulia> ceil(x)\nInterval{Float64}(2.0, 3.0)\n\njulia> trunc(x)\nInterval{Float64}(1.0, 2.0)","page":"Interfaces"},{"title":"Future","location":"stdlib/Future.html#Future","category":"section","text":"The Future module implements future behavior of already existing functions, which will replace the current version in a future release of Julia.","page":"Future"},{"title":"Future.copy!","location":"stdlib/Future.html#Future.copy!","category":"function","text":"Future.copy!(dst, src) -> dst\n\nCopy src into dst.\n\ncompat: Julia 1.1\nThis function has moved to Base with Julia 1.1, consider using copy!(dst, src) instead. Future.copy! will be deprecated in the future.\n\n\n\n\n\n","page":"Future"},{"title":"Future.randjump","location":"stdlib/Future.html#Future.randjump","category":"function","text":"randjump(r::MersenneTwister, steps::Integer) -> MersenneTwister\n\nCreate an initialized MersenneTwister object, whose state is moved forward (without generating numbers) from r by steps steps. One such step corresponds to the generation of two Float64 numbers. For each different value of steps, a large polynomial has to be generated internally. One is already pre-computed for steps=big(10)^20.\n\n\n\n\n\n","page":"Future"},{"title":"gdb debugging tips","location":"devdocs/debuggingtips.html#gdb-debugging-tips","category":"section","text":"","page":"gdb debugging tips"},{"title":"Displaying Julia variables","location":"devdocs/debuggingtips.html#Displaying-Julia-variables","category":"section","text":"Within gdb, any jl_value_t* object obj can be displayed using\n\n(gdb) call jl_(obj)\n\nThe object will be displayed in the julia session, not in the gdb session. This is a useful way to discover the types and values of objects being manipulated by Julia's C code.\n\nSimilarly, if you're debugging some of Julia's internals (e.g., compiler.jl), you can print obj using\n\nccall(:jl_, Cvoid, (Any,), obj)\n\nThis is a good way to circumvent problems that arise from the order in which julia's output streams are initialized.\n\nJulia's flisp interpreter uses value_t objects; these can be displayed with call fl_print(fl_ctx, ios_stdout, obj).","page":"gdb debugging tips"},{"title":"Useful Julia variables for Inspecting","location":"devdocs/debuggingtips.html#Useful-Julia-variables-for-Inspecting","category":"section","text":"While the addresses of many variables, like singletons, can be useful to print for many failures, there are a number of additional variables (see julia.h for a complete list) that are even more useful.\n\n(when in jl_apply_generic) mfunc and jl_uncompress_ast(mfunc->def, mfunc->code) :: for figuring out a bit about the call-stack\njl_lineno and jl_filename :: for figuring out what line in a test to go start debugging from (or figure out how far into a file has been parsed)\n$1 :: not really a variable, but still a useful shorthand for referring to the result of the last gdb command (such as print)\njl_options :: sometimes useful, since it lists all of the command line options that were successfully parsed\njl_uv_stderr :: because who doesn't like to be able to interact with stdio","page":"gdb debugging tips"},{"title":"Useful Julia functions for Inspecting those variables","location":"devdocs/debuggingtips.html#Useful-Julia-functions-for-Inspecting-those-variables","category":"section","text":"jl_print_task_backtraces(0) :: Similar to gdb's thread apply all bt or lldb's thread backtrace all. Runs all threads while printing backtraces for all existing tasks.\njl_gdblookup($pc) :: For looking up the current function and line.\njl_gdblookupinfo($pc) :: For looking up the current method instance object.\njl_gdbdumpcode(mi) :: For dumping all of code_typed/code_llvm/code_asm when the REPL is not working right.\njlbacktrace() :: For dumping the current Julia backtrace stack to stderr. Only usable after record_backtrace() has been called.\njl_dump_llvm_value(Value*) :: For invoking Value->dump() in gdb, where it doesn't work natively. For example, f->linfo->functionObject, f->linfo->specFunctionObject, and to_function(f->linfo).\njl_dump_llvm_module(Module*) :: For invoking Module->dump() in gdb, where it doesn't work natively.\nType->dump() :: only works in lldb. Note: add something like ;1 to prevent lldb from printing its prompt over the output\njl_eval_string(\"expr\") :: for invoking side-effects to modify the current state or to lookup symbols\njl_typeof(jl_value_t*) :: for extracting the type tag of a Julia value (in gdb, call macro define jl_typeof jl_typeof first, or pick something short like ty for the first arg to define a shorthand)","page":"gdb debugging tips"},{"title":"Inserting breakpoints for inspection from gdb","location":"devdocs/debuggingtips.html#Inserting-breakpoints-for-inspection-from-gdb","category":"section","text":"In your gdb session, set a breakpoint in jl_breakpoint like so:\n\n(gdb) break jl_breakpoint\n\nThen within your Julia code, insert a call to jl_breakpoint by adding\n\nccall(:jl_breakpoint, Cvoid, (Any,), obj)\n\nwhere obj can be any variable or tuple you want to be accessible in the breakpoint.\n\nIt's particularly helpful to back up to the jl_apply frame, from which you can display the arguments to a function using, e.g.,\n\n(gdb) call jl_(args[0])\n\nAnother useful frame is to_function(jl_method_instance_t *li, bool cstyle). The jl_method_instance_t* argument is a struct with a reference to the final AST sent into the compiler. However, the AST at this point will usually be compressed; to view the AST, call jl_uncompress_ast and then pass the result to jl_:\n\n#2  0x00007ffff7928bf7 in to_function (li=0x2812060, cstyle=false) at codegen.cpp:584\n584          abort();\n(gdb) p jl_(jl_uncompress_ast(li, li->ast))","page":"gdb debugging tips"},{"title":"Inserting breakpoints upon certain conditions","location":"devdocs/debuggingtips.html#Inserting-breakpoints-upon-certain-conditions","category":"section","text":"","page":"gdb debugging tips"},{"title":"Loading a particular file","location":"devdocs/debuggingtips.html#Loading-a-particular-file","category":"section","text":"Let's say the file is sysimg.jl:\n\n(gdb) break jl_load if strcmp(fname, \"sysimg.jl\")==0","page":"gdb debugging tips"},{"title":"Calling a particular method","location":"devdocs/debuggingtips.html#Calling-a-particular-method","category":"section","text":"(gdb) break jl_apply_generic if strcmp((char*)(jl_symbol_name)(jl_gf_mtable(F)->name), \"method_to_break\")==0\n\nSince this function is used for every call, you will make everything 1000x slower if you do this.","page":"gdb debugging tips"},{"title":"Dealing with signals","location":"devdocs/debuggingtips.html#Dealing-with-signals","category":"section","text":"Julia requires a few signals to function properly. The profiler uses SIGUSR2 for sampling and the garbage collector uses SIGSEGV for threads synchronization. If you are debugging some code that uses the profiler or multiple threads, you may want to let the debugger ignore these signals since they can be triggered very often during normal operations. The command to do this in GDB is (replace SIGSEGV with SIGUSR2 or other signals you want to ignore):\n\n(gdb) handle SIGSEGV noprint nostop pass\n\nThe corresponding LLDB command is (after the process is started):\n\n(lldb) pro hand -p true -s false -n false SIGSEGV\n\nIf you are debugging a segfault with threaded code, you can set a breakpoint on jl_critical_error (sigdie_handler should also work on Linux and BSD) in order to only catch the actual segfault rather than the GC synchronization points.","page":"gdb debugging tips"},{"title":"Debugging during Julia's build process (bootstrap)","location":"devdocs/debuggingtips.html#Debugging-during-Julia's-build-process-(bootstrap)","category":"section","text":"Errors that occur during make need special handling. Julia is built in two stages, constructing sys0 and sys.ji. To see what commands are running at the time of failure, use make VERBOSE=1.\n\nAt the time of this writing, you can debug build errors during the sys0 phase from the base directory using:\n\njulia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys0 sysimg.jl\n\nYou might need to delete all the files in usr/lib/julia/ to get this to work.\n\nYou can debug the sys.ji phase using:\n\njulia/base$ gdb --args ../usr/bin/julia-debug -C native --build ../usr/lib/julia/sys -J ../usr/lib/julia/sys0.ji sysimg.jl\n\nBy default, any errors will cause Julia to exit, even under gdb. To catch an error \"in the act\", set a breakpoint in jl_error (there are several other useful spots, for specific kinds of failures, including: jl_too_few_args, jl_too_many_args, and jl_throw).\n\nOnce an error is caught, a useful technique is to walk up the stack and examine the function by inspecting the related call to jl_apply. To take a real-world example:\n\nBreakpoint 1, jl_throw (e=0x7ffdf42de400) at task.c:802\n802 {\n(gdb) p jl_(e)\nErrorException(\"auto_unbox: unable to determine argument type\")\n$2 = void\n(gdb) bt 10\n#0  jl_throw (e=0x7ffdf42de400) at task.c:802\n#1  0x00007ffff65412fe in jl_error (str=0x7ffde56be000 <_j_str267> \"auto_unbox:\n   unable to determine argument type\")\n   at builtins.c:39\n#2  0x00007ffde56bd01a in julia_convert_16886 ()\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n...\n\nThe most recent jl_apply is at frame #3, so we can go back there and look at the AST for the function julia_convert_16886. This is the uniqued name for some method of convert. f in this frame is a jl_function_t*, so we can look at the type signature, if any, from the specTypes field:\n\n(gdb) f 3\n#3  0x00007ffff6541154 in jl_apply (f=0x7ffdf367f630, args=0x7fffffffc2b0, nargs=2) at julia.h:1281\n1281            return f->fptr((jl_value_t*)f, args, nargs);\n(gdb) p f->linfo->specTypes\n$4 = (jl_tupletype_t *) 0x7ffdf39b1030\n(gdb) p jl_( f->linfo->specTypes )\nTuple{Type{Float32}, Float64}           # <-- type signature for julia_convert_16886\n\nThen, we can look at the AST for this function:\n\n(gdb) p jl_( jl_uncompress_ast(f->linfo, f->linfo->ast) )\nExpr(:lambda, Array{Any, 1}[:#s29, :x], Array{Any, 1}[Array{Any, 1}[], Array{Any, 1}[Array{Any, 1}[:#s29, :Any, 0], Array{Any, 1}[:x, :Any, 0]], Array{Any, 1}[], 0], Expr(:body,\nExpr(:line, 90, :float.jl)::Any,\nExpr(:return, Expr(:call, :box, :Float32, Expr(:call, :fptrunc, :Float32, :x)::Any)::Any)::Any)::Any)::Any\n\nFinally, and perhaps most usefully, we can force the function to be recompiled in order to step through the codegen process. To do this, clear the cached functionObject from the jl_lamdbda_info_t*:\n\n(gdb) p f->linfo->functionObject\n$8 = (void *) 0x1289d070\n(gdb) set f->linfo->functionObject = NULL\n\nThen, set a breakpoint somewhere useful (e.g. emit_function, emit_expr, emit_call, etc.), and run codegen:\n\n(gdb) p jl_compile(f)\n... # your breakpoint here","page":"gdb debugging tips"},{"title":"Debugging precompilation errors","location":"devdocs/debuggingtips.html#Debugging-precompilation-errors","category":"section","text":"Module precompilation spawns a separate Julia process to precompile each module. Setting a breakpoint or catching failures in a precompile worker requires attaching a debugger to the worker. The easiest approach is to set the debugger watch for new process launches matching a given name. For example:\n\n(gdb) attach -w -n julia-debug\n\nor:\n\n(lldb) process attach -w -n julia-debug\n\nThen run a script/command to start precompilation. As described earlier, use conditional breakpoints in the parent process to catch specific file-loading events and narrow the debugging window. (some operating systems may require alternative approaches, such as following each fork from the parent process)","page":"gdb debugging tips"},{"title":"Mozilla's Record and Replay Framework (rr)","location":"devdocs/debuggingtips.html#Mozilla's-Record-and-Replay-Framework-(rr)","category":"section","text":"Julia now works out of the box with rr, the lightweight recording and deterministic debugging framework from Mozilla. This allows you to replay the trace of an execution deterministically. The replayed execution's address spaces, register contents, syscall data etc are exactly the same in every run.\n\nA recent version of rr (3.1.0 or higher) is required.","page":"gdb debugging tips"},{"title":"Reproducing concurrency bugs with rr","location":"devdocs/debuggingtips.html#Reproducing-concurrency-bugs-with-rr","category":"section","text":"rr simulates a single-threaded machine by default. In order to debug concurrent code you can use rr record --chaos which will cause rr to simulate between one to eight cores, chosen randomly. You might therefore want to set JULIA_NUM_THREADS=8 and rerun your code under rr until you have caught your bug.","page":"gdb debugging tips"},{"title":"ArgTools","location":"stdlib/ArgTools.html#ArgTools","category":"section","text":"","page":"ArgTools"},{"title":"Argument Handling","location":"stdlib/ArgTools.html#Argument-Handling","category":"section","text":"","page":"ArgTools"},{"title":"Function Testing","location":"stdlib/ArgTools.html#Function-Testing","category":"section","text":"","page":"ArgTools"},{"title":"ArgTools.ArgRead","location":"stdlib/ArgTools.html#ArgTools.ArgRead","category":"type","text":"ArgRead = Union{AbstractString, AbstractCmd, IO}\n\nThe ArgRead types is a union of the types that the arg_read function knows how to convert into readable IO handles. See arg_read for details.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.ArgWrite","location":"stdlib/ArgTools.html#ArgTools.ArgWrite","category":"type","text":"ArgWrite = Union{AbstractString, AbstractCmd, IO}\n\nThe ArgWrite types is a union of the types that the arg_write function knows how to convert into writeable IO handles, except for Nothing which arg_write handles by generating a temporary file. See arg_write for details.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_read","location":"stdlib/ArgTools.html#ArgTools.arg_read","category":"function","text":"arg_read(f::Function, arg::ArgRead) -> f(arg_io)\n\nThe arg_read function accepts an argument arg that can be any of these:\n\nAbstractString: a file path to be opened for reading\nAbstractCmd: a command to be run, reading from its standard output\nIO: an open IO handle to be read from\n\nWhether the body returns normally or throws an error, a path which is opened will be closed before returning from arg_read and an IO handle will be flushed but not closed before returning from arg_read.\n\nNote: when opening a file, ArgTools will pass lock = false to the file open(...) call. Therefore, the object returned by this function should not be used from multiple threads. This restriction may be relaxed in the future, which would not break any working code.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_write","location":"stdlib/ArgTools.html#ArgTools.arg_write","category":"function","text":"arg_write(f::Function, arg::ArgWrite) -> arg\narg_write(f::Function, arg::Nothing) -> tempname()\n\nThe arg_write function accepts an argument arg that can be any of these:\n\nAbstractString: a file path to be opened for writing\nAbstractCmd: a command to be run, writing to its standard input\nIO: an open IO handle to be written to\nNothing: a temporary path should be written to\n\nIf the body returns normally, a path that is opened will be closed upon completion; an IO handle argument is left open but flushed before return. If the argument is nothing then a temporary path is opened for writing and closed open completion and the path is returned from arg_write. In all other cases, arg itself is returned. This is a useful pattern since you can consistently return whatever was written, whether an argument was passed or not.\n\nIf there is an error during the evaluation of the body, a path that is opened by arg_write for writing will be deleted, whether it's passed in as a string or a temporary path generated when arg is nothing.\n\nNote: when opening a file, ArgTools will pass lock = false to the file open(...) call. Therefore, the object returned by this function should not be used from multiple threads. This restriction may be relaxed in the future, which would not break any working code.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_isdir","location":"stdlib/ArgTools.html#ArgTools.arg_isdir","category":"function","text":"arg_isdir(f::Function, arg::AbstractString) -> f(arg)\n\nThe arg_isdir function takes arg which must be the path to an existing directory (an error is raised otherwise) and passes that path to f finally returning the result of f(arg). This is definitely the least useful tool offered by ArgTools and mostly exists for symmetry with arg_mkdir and to give consistent error messages.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_mkdir","location":"stdlib/ArgTools.html#ArgTools.arg_mkdir","category":"function","text":"arg_mkdir(f::Function, arg::AbstractString) -> arg\narg_mkdir(f::Function, arg::Nothing) -> mktempdir()\n\nThe arg_mkdir function takes arg which must either be one of:\n\na path to an already existing empty directory,\na non-existent path which can be created as a directory, or\nnothing in which case a temporary directory is created.\n\nIn all cases the path to the directory is returned. If an error occurs during f(arg), the directory is returned to its original state: if it already existed but was empty, it will be emptied; if it did not exist it will be deleted.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_readers","location":"stdlib/ArgTools.html#ArgTools.arg_readers","category":"function","text":"arg_readers(arg :: AbstractString, [ type = ArgRead ]) do arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        arg :: ArgRead\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThe arg_readers function takes a path to be read and a single-argument do block, which is invoked once for each test reader type that arg_read can handle. If the optional type argument is given then the do block is only invoked for readers that produce arguments of that type.\n\nThe arg passed to the do block is not the argument value itself, because some of test argument types need to be initialized and finalized for each test case. Consider an open file handle argument: once you've used it for one test, you can't use it again; you need to close it and open the file again for the next test. This function arg can be converted into an ArgRead instance using @arg_test arg begin ... end.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.arg_writers","location":"stdlib/ArgTools.html#ArgTools.arg_writers","category":"function","text":"arg_writers([ type = ArgWrite ]) do path::String, arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        arg :: ArgWrite\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThe arg_writers function takes a do block, which is invoked once for each test writer type that arg_write can handle with a temporary (non-existent) path and arg which can be converted into various writable argument types which write to path. If the optional type argument is given then the do block is only invoked for writers that produce arguments of that type.\n\nThe arg passed to the do block is not the argument value itself, because some of test argument types need to be initialized and finalized for each test case. Consider an open file handle argument: once you've used it for one test, you can't use it again; you need to close it and open the file again for the next test. This function arg can be converted into an ArgWrite instance using @arg_test arg begin ... end.\n\nThere is also an arg_writers method that takes a path name like arg_readers:\n\narg_writers(path::AbstractString, [ type = ArgWrite ]) do arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        # here `arg :: ArgWrite`\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThis method is useful if you need to specify path instead of using path name generated by tempname(). Since path is passed from outside of arg_writers, the path is not an argument to the do block in this form.\n\n\n\n\n\n","page":"ArgTools"},{"title":"ArgTools.@arg_test","location":"stdlib/ArgTools.html#ArgTools.@arg_test","category":"macro","text":"@arg_test arg1 arg2 ... body\n\nThe @arg_test macro is used to convert arg functions provided by arg_readers and arg_writers into actual argument values. When you write @arg_test arg body it is equivalent to arg(arg -> body).\n\n\n\n\n\n","page":"ArgTools"},{"title":"SIMD Support","location":"base/simd-types.html#SIMD-Support","category":"section","text":"Type VecElement{T} is intended for building libraries of SIMD operations. Practical use of it requires using llvmcall. The type is defined as:\n\nstruct VecElement{T}\n    value::T\nend\n\nIt has a special compilation rule: a homogeneous tuple of VecElement{T} maps to an LLVM vector type when T is a primitive bits type.\n\nAt -O3, the compiler might automatically vectorize operations on such tuples. For example, the following program, when compiled with julia -O3 generates two SIMD addition instructions (addps) on x86 systems:\n\nconst m128 = NTuple{4,VecElement{Float32}}\n\nfunction add(a::m128, b::m128)\n    (VecElement(a[1].value+b[1].value),\n     VecElement(a[2].value+b[2].value),\n     VecElement(a[3].value+b[3].value),\n     VecElement(a[4].value+b[4].value))\nend\n\ntriple(c::m128) = add(add(c,c),c)\n\ncode_native(triple,(m128,))\n\nHowever, since the automatic vectorization cannot be relied upon, future use will mostly be via libraries that use llvmcall.","page":"SIMD Support"},{"title":"CRC32c","location":"stdlib/CRC32c.html#CRC32c","category":"section","text":"Standard library module for computing the CRC-32c checksum.","page":"CRC32c"},{"title":"CRC32c.crc32c","location":"stdlib/CRC32c.html#CRC32c.crc32c","category":"function","text":"crc32c(data, crc::UInt32=0x00000000)\n\nCompute the CRC-32c checksum of the given data, which can be an Array{UInt8}, a contiguous subarray thereof, an AbstractVector{UInt8}, or a String. Optionally, you can pass a starting crc integer to be mixed in with the checksum. The crc parameter can be used to compute a checksum on data divided into chunks: performing crc32c(data2, crc32c(data1)) is equivalent to the checksum of [data1; data2]. (Technically, a little-endian checksum is computed.)\n\nThere is also a method crc32c(io, nb, crc) to checksum nb bytes from a stream io, or crc32c(io, crc) to checksum all the remaining bytes. Hence you can do open(crc32c, filename) to checksum an entire file, or crc32c(seekstart(buf)) to checksum an IOBuffer without calling take!.\n\nFor a String, note that the result is specific to the UTF-8 encoding (a different checksum would be obtained from a different Unicode encoding). To checksum an a::AbstractArray of some other bitstype without padding, you can do crc32c(vec(reinterpret(UInt8,a))), but note that the result may be endian-dependent.\n\n\n\n\n\n","page":"CRC32c"},{"title":"CRC32c.crc32c","location":"stdlib/CRC32c.html#CRC32c.crc32c-Tuple{IO, Integer, UInt32}","category":"method","text":"crc32c(io::IO, [nb::Integer,] crc::UInt32=0x00000000)\n\nRead up to nb bytes from io and return the CRC-32c checksum, optionally mixed with a starting crc integer.  If nb is not supplied, then io will be read until the end of the stream.\n\n\n\n\n\n","page":"CRC32c"},{"title":"Markdown","location":"stdlib/Markdown.html#markdown_stdlib","category":"section","text":"This section describes Julia's markdown syntax, which is enabled by the Markdown standard library. The following Markdown elements are supported:","page":"Markdown"},{"title":"Inline elements","location":"stdlib/Markdown.html#Inline-elements","category":"section","text":"Here \"inline\" refers to elements that can be found within blocks of text, i.e. paragraphs. These include the following elements.","page":"Markdown"},{"title":"Bold","location":"stdlib/Markdown.html#Bold","category":"section","text":"Surround words with two asterisks, **, to display the enclosed text in boldface.\n\nA paragraph containing a **bold** word.","page":"Markdown"},{"title":"Italics","location":"stdlib/Markdown.html#Italics","category":"section","text":"Surround words with one asterisk, *, to display the enclosed text in italics.\n\nA paragraph containing an *italicized* word.","page":"Markdown"},{"title":"Literals","location":"stdlib/Markdown.html#Literals","category":"section","text":"Surround text that should be displayed exactly as written with single backticks, ` .\n\nA paragraph containing a `literal` word.\n\nLiterals should be used when writing text that refers to names of variables, functions, or other parts of a Julia program.\n\ntip: Tip\nTo include a backtick character within literal text use three backticks rather than one to enclose the text.A paragraph containing ``` `backtick` characters ```.By extension any odd number of backticks may be used to enclose a lesser number of backticks.","page":"Markdown"},{"title":"LaTeX","location":"stdlib/Markdown.html#\\LaTeX","category":"section","text":"Surround text that should be displayed as mathematics using LaTeX syntax with double backticks, `` .\n\nA paragraph containing some ``\\LaTeX`` markup.\n\ntip: Tip\nAs with literals in the previous section, if literal backticks need to be written within double backticks use an even number greater than two. Note that if a single literal backtick needs to be included within LaTeX markup then two enclosing backticks is sufficient.\n\nnote: Note\nThe \\ character should be escaped appropriately if the text is embedded in a Julia source code, for example, \"``\\\\LaTeX`` syntax in a docstring.\", since it is interpreted as a string literal. Alternatively, in order to avoid escaping, it is possible to use the raw string macro together with the @doc macro:@doc raw\"``\\LaTeX`` syntax in a docstring.\" functionname","page":"Markdown"},{"title":"Links","location":"stdlib/Markdown.html#Links","category":"section","text":"Links to either external or internal targets can be written using the following syntax, where the text enclosed in square brackets, [ ], is the name of the link and the text enclosed in parentheses, ( ), is the URL.\n\nA paragraph containing a link to [Julia](https://www.julialang.org).\n\nIt's also possible to add cross-references to other documented functions/methods/variables within the Julia documentation itself. For example:\n\n\"\"\"\n    tryparse(type, str; base)\n\nLike [`parse`](@ref), but returns either a value of the requested type,\nor [`nothing`](@ref) if the string does not contain a valid number.\n\"\"\"\n\nThis will create a link in the generated docs to the parse documentation (which has more information about what this function actually does), and to the nothing documentation. It's good to include cross references to mutating/non-mutating versions of a function, or to highlight a difference between two similar-seeming functions.\n\nnote: Note\nThe above cross referencing is not a Markdown feature, and relies on Documenter.jl, which is used to build base Julia's documentation.","page"