var documenterSearchIndex = {"docs":
[{"location":"manual/variables.html#man-variables","page":"Variables","title":"Variables","category":"section","text":"A variable, in Julia, is a name associated (or bound) to a value. It's useful when you want to store a value (that you obtained after some math, for example) for later use. For example:\n\njulia> x = 10   # Assign the value 10 to the variable x\n10\n\njulia> x + 1    # Doing math with x's value\n11\n\njulia> x = 1 + 1   # Reassign x's value\n2\n\njulia> x = \"Hello World!\"   # You can assign values of other types, like strings of text\n\"Hello World!\"\n\nJulia provides an extremely flexible system for naming variables. Variable names are case-sensitive, and have no semantic meaning (that is, the language will not treat variables differently based on their names).\n\njulia> x = 1.0\n1.0\n\njulia> y = -3\n-3\n\njulia> Z = \"My string\"\n\"My string\"\n\njulia> customary_phrase = \"Hello world!\"\n\"Hello world!\"\n\njulia> UniversalDeclarationOfHumanRightsStart = \"人人生而自由，在尊严和权利上一律平等。\"\n\"人人生而自由，在尊严和权利上一律平等。\"\n\nUnicode names (in UTF-8 encoding) are allowed:\n\njulia> δ = 0.00001\n1.0e-5\n\njulia> 안녕하세요 = \"Hello\"\n\"Hello\"\n\nIn the Julia REPL and several other Julia editing environments, you can type many Unicode math symbols by typing the backslashed LaTeX symbol name followed by tab. For example, the variable name δ can be entered by typing \\delta-tab, or even α̂⁽²⁾ by \\alpha-tab-\\hat- tab-\\^(2)-tab. (If you find a symbol somewhere, e.g. in someone else's code, that you don't know how to type, the REPL help will tell you: just type ? and then paste the symbol.)\n\nJulia will even let you shadow existing exported constants and functions with local ones (although this is not recommended to avoid potential confusions):\n\njulia> pi = 3\n3\n\njulia> pi\n3\n\njulia> sqrt = 4\n4\n\njulia> length() = 5\nlength (generic function with 1 method)\n\njulia> Base.length\nlength (generic function with 79 methods)\n\nHowever, if you try to redefine a built-in constant or function that you have explicitly imported, Julia will give you an error:\n\njulia> using Base: pi, sqrt\n\njulia> pi\nπ = 3.1415926535897...\n\njulia> pi = 3\nERROR: cannot assign a value to imported variable Base.pi from module Main\n\njulia> sqrt(100)\n10.0\n\njulia> sqrt = 4\nERROR: cannot assign a value to imported variable Base.sqrt from module Main\n\ncompat: Julia 1.12\n\n\nNote that in versions prior to Julia 1.12, these errors depended on use rather than definition of   the conflicting binding."},{"location":"manual/variables.html#man-allowed-variable-names","page":"Variables","title":"Allowed Variable Names","category":"section","text":"Variable names must begin with a letter (A-Z or a-z), underscore, or a subset of Unicode code points greater than 00A0; in particular, Unicode character categories Lu/Ll/Lt/Lm/Lo/Nl (letters), Sc/So (currency and other symbols), and a few other letter-like characters (e.g. a subset of the Sm math symbols) are allowed. Subsequent characters may also include ! and digits (0-9 and other characters in categories Nd/No), as well as other Unicode code points: diacritics and other modifying marks (categories Mn/Mc/Me/Sk), some punctuation connectors (category Pc), primes, and a few other characters.\n\nOperators like + are also valid identifiers, but are parsed specially. In some contexts, operators can be used just like variables; for example (+) refers to the addition function, and (+) = f will reassign it. Most of the Unicode infix operators (in category Sm), such as ⊕, are parsed as infix operators and are available for user-defined methods (e.g. you can use const ⊗ = kron to define ⊗ as an infix Kronecker product). Operators can also be suffixed with modifying marks, primes, and sub/superscripts, e.g. +̂ₐ″ is parsed as an infix operator with the same precedence as +. A space is required between an operator that ends with a subscript/superscript letter and a subsequent variable name. For example, if +ᵃ is an operator, then +ᵃx must be written as +ᵃ x to distinguish it from + ᵃx where ᵃx is the variable name.\n\nA particular class of variable names is one that contains only underscores. These identifiers are write-only. I.e. they can only be assigned values, which are immediately discarded, and their values cannot be used in any way.\n\njulia> x, ___ = size([2 2; 1 1])\n(2, 2)\n\njulia> y = ___\nERROR: syntax: all-underscore identifiers are write-only and their values cannot be used in expressions\n\njulia> println(___)\nERROR: syntax: all-underscore identifiers are write-only and their values cannot be used in expressions\n\nThe only explicitly disallowed names for variables are the names of the built-in Keywords:\n\njulia> else = false\nERROR: ParseError:\n# Error @ none:1:1\nelse = false\n└──┘ ── invalid identifier\n\njulia> try = \"No\"\nERROR: ParseError:\n# Error @ none:1:1\ntry = \"No\"\n└────────┘ ── try without catch or finally\n\nSome Unicode characters are considered to be equivalent in identifiers. Different ways of entering Unicode combining characters (e.g., accents) are treated as equivalent (specifically, Julia identifiers are NFC. Julia also includes a few non-standard equivalences for characters that are visually similar and are easily entered by some input methods. The Unicode characters ɛ (U+025B: Latin small letter open e) and µ (U+00B5: micro sign) are treated as equivalent to the corresponding Greek letters. The middle dot · (U+00B7) and the Greek interpunct · (U+0387) are both treated as the mathematical dot operator ⋅ (U+22C5). The minus sign − (U+2212) is treated as equivalent to the hyphen-minus sign - (U+002D)."},{"location":"manual/variables.html#man-assignment-expressions","page":"Variables","title":"Assignment expressions and assignment versus mutation","category":"section","text":"An assignment variable = value \"binds\" the name variable to the value computed on the right-hand side, and the whole assignment is treated by Julia as an expression equal to the right-hand-side value. This means that assignments can be chained (the same value assigned to multiple variables with variable1 = variable2 = value) or used in other expressions, and is also why their result is shown in the REPL as the value of the right-hand side.  (In general, the REPL displays the value of whatever expression you evaluate.)  For example, here the value 4 of b = 2+2 is used in another arithmetic operation and assignment:\n\njulia> a = (b = 2+2) + 3\n7\n\njulia> a\n7\n\njulia> b\n4\n\nA common confusion is the distinction between assignment (giving a new \"name\" to a value) and mutation (changing a value). If you run a = 2 followed by a = 3, you have changed the \"name\" a to refer to a new value 3 … you haven't changed the number 2, so 2+2 will still give 4 and not 6!   This distinction becomes more clear when dealing with mutable types like arrays, whose contents can be changed:\n\njulia> a = [1,2,3] # an array of 3 integers\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> b = a   # both b and a are names for the same array!\n3-element Vector{Int64}:\n 1\n 2\n 3\n\nHere, the line b = a does not make a copy of the array a, it simply binds the name b to the same array a: both b and a \"point\" to one array [1,2,3] in memory. In contrast, an assignment a[i] = value changes the contents of the array, and the modified array will be visible through both the names a and b:\n\njulia> a[1] = 42     # change the first element\n42\n\njulia> a = 3.14159   # a is now the name of a different object\n3.14159\n\njulia> b   # b refers to the original array object, which has been mutated\n3-element Vector{Int64}:\n 42\n  2\n  3\n\nThat is, a[i] = value (an alias for setindex!) mutates an existing array object in memory, accessible via either a or b. Subsequently setting a = 3.14159 does not change this array, it simply binds a to a different object; the array is still accessible via b. Another common syntax to mutate an existing object is a.field = value (an alias for setproperty!), which can be used to change a mutable struct. There is also mutation via dot assignment, for example b .= 5:7 (which mutates our array b in-place to contain [5,6,7]), as part of Julia's vectorized \"dot\" syntax.\n\nWhen you call a function in Julia, it behaves as if you assigned the argument values to new variable names corresponding to the function arguments, as discussed in Argument-Passing Behavior.  (By convention, functions that mutate one or more of their arguments have names ending with !.)"},{"location":"manual/variables.html#Stylistic-Conventions","page":"Variables","title":"Stylistic Conventions","category":"section","text":"While Julia imposes few restrictions on valid names, it has become useful to adopt the following conventions:\n\nNames of variables are in lower case.\nWord separation can be indicated by underscores ('_'), but use of underscores is discouraged unless the name would be hard to read otherwise.\nNames of Types and Modules begin with a capital letter and word separation is shown with upper camel case instead of underscores.\nNames of functions and macros are in lower case, without underscores.\nFunctions that write to their arguments have names that end in !. These are sometimes called \"mutating\" or \"in-place\" functions because they are intended to produce changes in their arguments after the function is called, not just return a value.\n\nFor more information about stylistic conventions, see the Style Guide."},{"location":"devdocs/build/freebsd.html#FreeBSD","page":"FreeBSD","title":"FreeBSD","category":"section","text":"Clang is the default compiler on FreeBSD 11.0-RELEASE and above. The remaining build tools are available from the Ports Collection, and can be installed using pkg install git gcc gmake cmake pkgconf. To build Julia, simply run gmake. (Note that gmake must be used rather than make, since make on FreeBSD corresponds to the incompatible BSD Make rather than GNU Make.)\n\nAs mentioned above, it is important to note that the USE_SYSTEM_* flags should be used with caution on FreeBSD. This is because many system libraries, and even libraries from the Ports Collection, link to the system's libgcc_s.so.1, or to another library which links to the system libgcc_s. This library declares its GCC version to be 4.6, which is too old to build Julia, and conflicts with other libraries when linking. Thus it is highly recommended to simply allow Julia to build all of its dependencies. If you do choose to use the USE_SYSTEM_* flags, note that /usr/local is not on the compiler path by default, so you may need to add LDFLAGS=-L/usr/local/lib and CPPFLAGS=-I/usr/local/include to your Make.user, though doing so may interfere with other dependencies.\n\nNote that the x86 architecture does not support threading due to lack of compiler runtime library support, so you may need to set JULIA_THREADS=0 in your Make.user if you're on a 32-bit system."},{"location":"manual/worldage.html#man-world-age","page":"The World Age mechanism","title":"The World Age mechanism","category":"section","text":"note: Note\nWorld age is an advanced concept. For the vast majority of Julia users, the world age mechanism operates invisibly in the background. This documentation is intended for the few users who may encounter world-age related issues or error messages.\n\ncompat: Julia 1.12\nPrior to Julia 1.12, the world age mechanism did not apply to changes to the global binding table. The documentation in this chapter is specific to Julia 1.12+.\n\nwarning: Warning\nThis manual chapter uses internal functions to introspect world age and runtime data structures as an explanatory aid. In general, unless otherwise noted the world age mechanism is not a stable interface and should be interacted with in packages through stable APIs (e.g. invokelatest) only. In particular, do not assume that world ages are always integers or that they have a linear order."},{"location":"manual/worldage.html#World-age-in-general","page":"The World Age mechanism","title":"World age in general","category":"section","text":"The \"world age counter\" is a monotonically increasing counter that is incremented for every change to the global method table or the global binding table (e.g. through method definition, type definition, import/using declaration, creation of (typed) globals or definition of constants).\n\nThe current value of the global world age counter can be retrieved using the (internal) function Base.get_world_counter.\n\njulia> Base.get_world_counter()\n0x0000000000009632\n\njulia> const x = 1\n\njulia> Base.get_world_counter()\n0x0000000000009633\n\nIn addition, each Task stores a local world age that determines which modifications to the global binding and method tables are currently visible to the running task. The world age of the running task will never exceed the global world age counter, but may run arbitrarily behind it. In general the term \"current world age\" refers to the local world age of the currently running task. The current world age may be retrieved using the (internal) function Base.tls_world_age\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           Core.eval(@__MODULE__, :(f() = 1))\n           @show (Int(Base.get_world_counter()), Int(Base.tls_world_age()))\n           f()\n       end\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38452, 38452)\n(Int(Base.get_world_counter()), Int(Base.tls_world_age())) = (38453, 38452)\nERROR: MethodError: no method matching f()\nThe applicable method may be too new: running in current world age 38452, while global world is 38453.\n\nClosest candidates are:\n  f() (method too new to be called from this world context.)\n   @ Main REPL[2]:3\n\nStacktrace:\n [1] top-level scope\n   @ REPL[2]:5\n\njulia> (f(), Int(Base.tls_world_age()))\n(1, 38453)\n\nHere the definition of the method f raised the global world counter, but the current world age did not change. As a result, the definition of f was not visible in the currently executing task and a MethodError resulted.\n\nnote: Note\nThe method error printing provided additional information that f() is available in a newer world age. This information is added by the error display, not the task that threw the MethodError. The thrown MethodError is identical whether or not a matching definition of f() exists in a newer world age.\n\nHowever, note that the definition of f() was subsequently available at the next REPL prompt, because the current task's world age had been raised. In general, certain syntactic constructs (in particular most definitions) will raise the current task's world age to the latest global world age, thus making all changes (both from the current task and any concurrently executing other tasks) visible. The following statements raise the current world age:\n\nAn explicit invocation of Core.@latestworld\nThe start of every top-level statement\nThe start of every REPL prompt\nAny type or struct definition\nAny method definition\nAny constant declaration\nAny global variable declaration (but not a global variable assignment)\nAny using, import, export or public statement\nCertain other macros like @eval (depends on the macro implementation)\n\nNote, however, that the current task's world age may only ever be permanently incremented at top level. As a general rule, using any of the above statements in non-top-level scope is a syntax error:\n\njulia> f() = Core.@latestworld\nERROR: syntax: World age increment not at top level\nStacktrace:\n [1] top-level scope\n   @ REPL[5]:1\n\nWhen it isn't (for example for @eval), the world age side effect is ignored.\n\nAs a result of these rules, Julia may assume that the world age does not change within the execution of an ordinary function.\n\nfunction my_function()\n    before = Base.tls_world_age()\n    # Any arbitrary code\n    after = Base.tls_world_age()\n    @assert before === after # always true\nend\n\nThis is the key invariant that allows Julia to optimize based on the current state of its global data structures, while still having the well-defined ability to change these data structures."},{"location":"manual/worldage.html#Temporarily-raising-the-world-age-using-invokelatest","page":"The World Age mechanism","title":"Temporarily raising the world age using invokelatest","category":"section","text":"As described above, it is not possible to permanently raise the world age for the remainder of a Task's execution unless the task is executing top-level statements. However, it is possible to temporarily change the world age in a scoped manner using invokelatest:\n\njulia> function f end\nf (generic function with 0 methods)\n\njulia> begin\n           Core.eval(@__MODULE__, :(f() = 1))\n           invokelatest(f)\n       end\n1\n\ninvokelatest will temporarily raise the current task's world age to the latest global world age (at entry to invokelatest) and execute the provided function. Note that the world age will return to its prior value upon exit from invokelatest."},{"location":"manual/worldage.html#World-age-and-const-struct-redefinitions","page":"The World Age mechanism","title":"World age and const struct redefinitions","category":"section","text":"The semantics described above for method redefinition also apply to redefinition of constants:\n\njulia> const x = 1\n1\n\njulia> get_const() = x\nget_const (generic function with 1 method)\n\njulia> begin\n           @show get_const()\n           Core.eval(@__MODULE__, :(const x = 2))\n           @show get_const()\n           Core.@latestworld\n           @show get_const()\n       end\nget_const() = 1\nget_const() = 1\nget_const() = 2\n2\n\nHowever, for the avoidance of doubt, they do not apply to ordinary assignment to global variables, which becomes visible immediately:\n\njulia> global y = 1\n1\n\njulia> get_global() = y\nget_global (generic function with 1 method)\n\njulia> begin\n           @show get_global()\n           Core.eval(@__MODULE__, :(y = 2))\n           @show get_global()\n       end\nget_global() = 1\nget_global() = 2\n2\n\nOne particular special case of constant reassignment is the redefinition of struct types:\n\njulia> struct MyStruct\n           x::Int\n       end\n\njulia> const one_field = MyStruct(1)\nMyStruct(1)\n\njulia> struct MyStruct\n           x::Int\n           y::Float64\n       end\n\njulia> const two_field = MyStruct(1, 2.0)\nMyStruct(1, 2.0)\n\njulia> one_field\n@world(MyStruct, 38452:38455)(1)\n\njulia> two_field\nMyStruct(1, 2.0)\n\nInternally the two definitions of MyStruct are entirely separate types. However, after the new MyStruct type is defined, there is no longer any default binding for the original definition of MyStruct. To nevertheless facilitate access to these types, the special @world macro may be used to access the meaning of a name in a previous world. However, this facility is intended for introspection only and in particular note that world age numbers are not stable across precompilation and should in general be treated opaquely."},{"location":"manual/worldage.html#Binding-partition-introspection","page":"The World Age mechanism","title":"Binding partition introspection","category":"section","text":"In certain cases, it can be helpful to introspect the system's understanding of what a binding means in any particular world age. The default display printing of Core.Binding provides a helpful summary (e.g. on the MyStruct example from above):\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :MyStruct))\nBinding Main.MyStruct\n   38456:∞ - constant binding to MyStruct\n   38452:38455 - constant binding to @world(MyStruct, 38452:38455)\n   38451:38451 - backdated constant binding to @world(MyStruct, 38452:38455)\n   0:38450 - backdated constant binding to @world(MyStruct, 38452:38455)"},{"location":"manual/worldage.html#World-age-and-using/import","page":"The World Age mechanism","title":"World age and using/import","category":"section","text":"Bindings provided via using and import also operate via the world age mechanism. Binding resolution is a stateless function of the import and using definitions visible in the current world age. For example:\n\njulia> module M1; const x = 1; export x; end\n\njulia> module M2; const x = 2; export x; end\n\njulia> using .M1\n\njulia> x\n1\n\njulia> using .M2\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\nHint: It looks like two or more modules export different bindings with this name, resulting in ambiguity. Try explicitly importing it from a particular module, or qualifying the name with the module it should come from.\n\njulia> convert(Core.Binding, GlobalRef(@__MODULE__, :x))\nBinding Main.x\n   38458:∞ - ambiguous binding - guard entry\n   38457:38457 - implicit `using` resolved to constant 1"},{"location":"manual/worldage.html#World-age-capture","page":"The World Age mechanism","title":"World age capture","category":"section","text":"Certain language features capture the current task's world age. Perhaps the most common of these is creation of new tasks. Newly created tasks will inherit the creating task's local world age at creation time and will retain said world age (unless explicitly raised) even if the originating tasks raises its world age:\n\njulia> const x = 1\n\njulia> t = @task (wait(); println(\"Running now\"); x);\n\njulia> const x = 2\n\njulia> schedule(t);\nRunning now\n\njulia> x\n2\n\njulia> fetch(t)\n1\n\nIn addition to tasks, opaque closures also capture their world age at creation. See Base.Experimental.@opaque."},{"location":"manual/worldage.html#Base.@world","page":"The World Age mechanism","title":"Base.@world","category":"macro","text":"@world(sym, world)\n\nResolve the binding sym in world world. See invoke_in_world for running arbitrary code in fixed worlds. world may be UnitRange, in which case the macro will error unless the binding is valid and has the same value across the entire world range.\n\nAs a special case, the world ∞ always refers to the latest world, even if that world is newer than the world currently running.\n\nThe @world macro is primarily used in the printing of bindings that are no longer available in the current world.\n\nExample\n\njulia> struct Foo; a::Int; end\nFoo\n\njulia> fold = Foo(1)\n\njulia> Int(Base.get_world_counter())\n26866\n\njulia> struct Foo; a::Int; b::Int end\nFoo\n\njulia> fold\n@world(Foo, 26866)(1)\n\ncompat: Julia 1.12\nThis functionality requires at least Julia 1.12.\n\n\n\n\n\n"},{"location":"manual/worldage.html#Base.get_world_counter","page":"The World Age mechanism","title":"Base.get_world_counter","category":"function","text":"get_world_counter()\n\nReturns the current maximum world-age counter. This counter is monotonically increasing.\n\nwarning: Warning\nThis counter is global and may change at any time between invocations. In general, most reflection functions operate on the current task's world age, rather than the global maximum world age. See tls_world_age as well as the manual chapter of world age.\n\n\n\n\n\n"},{"location":"manual/worldage.html#Base.tls_world_age","page":"The World Age mechanism","title":"Base.tls_world_age","category":"function","text":"tls_world_age()\n\nReturns the world the current_task() is executing within.\n\n\n\n\n\n"},{"location":"manual/worldage.html#Base.invoke_in_world","page":"The World Age mechanism","title":"Base.invoke_in_world","category":"function","text":"invoke_in_world(world, f, args...; kwargs...)\n\nCall f(args...; kwargs...) in a fixed world age, world.\n\nThis is useful for infrastructure running in the user's Julia session which is not part of the user's program. For example, things related to the REPL, editor support libraries, etc. In these cases it can be useful to prevent unwanted method invalidation and recompilation latency, and to prevent the user from breaking supporting infrastructure by mistake.\n\nThe global world age can be queried using Base.get_world_counter() and stored for later use within the lifetime of the current Julia session, or when serializing and reloading the system image.\n\nTechnically, invoke_in_world will prevent any function called by f from being extended by the user during their Julia session. That is, generic function method tables seen by f (and any functions it calls) will be frozen as they existed at the given world age. In a sense, this is like the opposite of invokelatest.\n\nnote: Note\nIt is not valid to store world ages obtained in precompilation for later use. This is because precompilation generates a \"parallel universe\" where the world age refers to system state unrelated to the main Julia session.\n\n\n\n\n\n"},{"location":"manual/worldage.html#Base.Experimental.@opaque","page":"The World Age mechanism","title":"Base.Experimental.@opaque","category":"macro","text":"@opaque ([type, ]args...) -> body\n\nMarks a given closure as \"opaque\". Opaque closures capture the world age of their creation (as opposed to their invocation). This allows for more aggressive optimization of the capture list, but trades off against the ability to inline opaque closures at the call site, if their creation is not statically visible.\n\nAn argument tuple type (type) may optionally be specified, to specify allowed argument types in a more flexible way. In particular, the argument type may be fixed length even if the function is variadic.\n\nwarning: Warning\nThis interface is experimental and subject to change or removal without notice.\n\n\n\n\n\n"},{"location":"index.html","page":"Julia Documentation","title":"Julia Documentation","category":"section","text":"io = IOBuffer()\nrelease = isempty(VERSION.prerelease)\nv = \"$(VERSION.major).$(VERSION.minor)\"\n!release && (v = v*\"-$(first(VERSION.prerelease))\")\nprint(io, \"\"\"\n    # Julia $(v) Documentation\n\n    Welcome to the documentation for Julia $(v).\n\n    \"\"\")\nif !release\n    print(io,\"\"\"\n        !!! warning \"Work in progress!\"\n            This documentation is for an unreleased, in-development, version of Julia.\n        \"\"\")\nend\nimport Markdown\nMarkdown.parse(String(take!(io)))\n\nPlease read the release notes to see what has changed since the last release.\n\nrelease = isempty(VERSION.prerelease)\nfile = release ? \"julia-$(VERSION).pdf\" :\n       \"julia-$(VERSION.major).$(VERSION.minor).$(VERSION.patch)-$(first(VERSION.prerelease)).pdf\"\nurl = \"https://raw.githubusercontent.com/JuliaLang/docs.julialang.org/assets/$(file)\"\nimport Markdown\nMarkdown.parse(\"\"\"\n!!! note\n    The documentation is also available in PDF format: [$file]($url).\n\"\"\")"},{"location":"index.html#man-introduction","page":"Julia Documentation","title":"Introduction","category":"section","text":"Scientific computing has traditionally required the highest performance, yet domain experts have largely moved to slower dynamic languages for daily work. We believe there are many good reasons to prefer dynamic languages for these applications, and we do not expect their use to diminish. Fortunately, modern language design and compiler techniques make it possible to mostly eliminate the performance trade-off and provide a single environment productive enough for prototyping and efficient enough for deploying performance-intensive applications. The Julia programming language fills this role: it is a flexible dynamic language, appropriate for scientific and numerical computing, with performance comparable to traditional statically-typed languages.\n\nBecause Julia's compiler is different from the interpreters used for languages like Python or R, you may find that Julia's performance is unintuitive at first. If you find that something is slow, we highly recommend reading through the Performance Tips section before trying anything else. Once you understand how Julia works, it is easy to write code that is nearly as fast as C."},{"location":"index.html#man-julia-compared-other-languages","page":"Julia Documentation","title":"Julia Compared to Other Languages","category":"section","text":"Julia features optional typing, multiple dispatch, and good performance, achieved using type inference and just-in-time (JIT) compilation (and optional ahead-of-time compilation), implemented using LLVM. It is multi-paradigm, combining features of imperative, functional, and object-oriented programming. Julia provides ease and expressiveness for high-level numerical computing, in the same way as languages such as R, MATLAB, and Python, but also supports general programming. To achieve this, Julia builds upon the lineage of mathematical programming languages, but also borrows much from popular dynamic languages, including Lisp, Perl, Python, Lua, and Ruby.\n\nThe most significant departures of Julia from typical dynamic languages are:\n\nThe core language imposes very little; Julia Base and the standard library are written in Julia itself, including primitive operations like integer arithmetic\nA rich language of types for constructing and describing objects, that can also optionally be used to make type declarations\nThe ability to define function behavior across many combinations of argument types via multiple dispatch\nAutomatic generation of efficient, specialized code for different argument types\nGood performance, approaching that of statically-compiled languages like C\n\nAlthough one sometimes speaks of dynamic languages as being \"typeless\", they are definitely not. Every object, whether primitive or user-defined, has a type. The lack of type declarations in most dynamic languages, however, means that one cannot instruct the compiler about the types of values, and often cannot explicitly talk about types at all. In static languages, on the other hand, while one can – and usually must – annotate types for the compiler, types exist only at compile time and cannot be manipulated or expressed at run time. In Julia, types are themselves run-time objects, and can also be used to convey information to the compiler."},{"location":"index.html#man-what-makes-julia","page":"Julia Documentation","title":"What Makes Julia, Julia?","category":"section","text":"While the casual programmer need not explicitly use types or multiple dispatch, they are the core unifying features of Julia: functions are defined on different combinations of argument types, and applied by dispatching to the most specific matching definition. This model is a good fit for mathematical programming, where it is unnatural for the first argument to \"own\" an operation as in traditional object-oriented dispatch. Operators are just functions with special notation – to extend addition to new user-defined data types, you define new methods for the + function. Existing code then seamlessly applies to the new data types.\n\nPartly because of run-time type inference (augmented by optional type annotations), and partly because of a strong focus on performance from the inception of the project, Julia's computational efficiency exceeds that of other dynamic languages, and even rivals that of statically-compiled languages. For large scale numerical problems, speed always has been, continues to be, and probably always will be crucial: the amount of data being processed has easily kept pace with Moore's Law over the past decades."},{"location":"index.html#man-advantages-of-julia","page":"Julia Documentation","title":"Advantages of Julia","category":"section","text":"Julia aims to create an unprecedented combination of ease-of-use, power, and efficiency in a single language. In addition to the above, some advantages of Julia over comparable systems include:\n\nFree and open source (MIT licensed)\nUser-defined types are as fast and compact as built-ins\nNo need to vectorize code for performance; devectorized code is fast\nDesigned for parallelism and distributed computation\nLightweight \"green\" threading (coroutines)\nUnobtrusive yet powerful type system\nElegant and extensible conversions and promotions for numeric and other types\nEfficient support for Unicode, including but not limited to UTF-8\nCall C functions directly (no wrappers or special APIs needed)\nPowerful shell-like capabilities for managing other processes\nLisp-like macros and other metaprogramming facilities"},{"location":"index.html#man-important-links","page":"Julia Documentation","title":"Important Links","category":"section","text":"A non-exhaustive list of links that will be useful as you learn and use the Julia programming language:\n\nJulia Homepage\nInstall Julia\nDiscussion forum\nJulia YouTube\nFind Julia Packages\nLearning Resources"},{"location":"devdocs/contributing/git-workflow.html#Git-workflow-recommendations","page":"Git workflow recommendations","title":"Git workflow recommendations","category":"section","text":""},{"location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Requests","page":"Git workflow recommendations","title":"Git Recommendations For Pull Requests","category":"section","text":"Avoid working from the master branch of your fork. Create a new branch as it will make it easier to update your pull request if Julia's master changes.\nTry to squash together small commits that make repeated changes to the same section of code, so your pull request is easier to review. A reasonable number of separate well-factored commits is fine, especially for larger changes.\nIf any conflicts arise due to changes in Julia's master, prefer updating your pull request branch with git rebase versus git merge or git pull, since the latter will introduce merge commits that clutter the git history with noise that makes your changes more difficult to review.\nDescriptive commit messages are good.\nUsing git add -p or git add -i can be useful to avoid accidentally committing unrelated changes.\nWhen linking to specific lines of code in discussion of an issue or pull request, hit the y key while viewing code on GitHub to reload the page with a URL that includes the specific version that you're viewing. That way any lines of code that you refer to will still make sense in the future, even if the content of the file changes.\nWhitespace can be automatically removed from existing commits with git rebase.\nTo remove whitespace for the previous commit, run git rebase --whitespace=fix HEAD~1.\nTo remove whitespace relative to the master branch, run git rebase --whitespace=fix master."},{"location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Request-Reviewers","page":"Git workflow recommendations","title":"Git Recommendations For Pull Request Reviewers","category":"section","text":"When merging, we generally like squash+merge. Unless it is the rare case of a PR with carefully staged individual commits that you want in the history separately, in which case merge is acceptable, but usually prefer squash+merge."},{"location":"base/iterators.html#Iteration-utilities","page":"Iteration utilities","title":"Iteration utilities","category":"section","text":""},{"location":"base/iterators.html#Base.Iterators.Stateful","page":"Iteration utilities","title":"Base.Iterators.Stateful","category":"type","text":"Stateful(itr)\n\nThere are several different ways to think about this iterator wrapper:\n\nIt provides a mutable wrapper around an iterator and its iteration state.\nIt turns an iterator-like abstraction into a Channel-like abstraction.\nIt's an iterator that mutates to become its own rest iterator whenever an item is produced.\n\nStateful provides the regular iterator interface. Like other mutable iterators (e.g. Base.Channel), if iteration is stopped early (e.g. by a break in a for loop), iteration can be resumed from the same spot by continuing to iterate over the same iterator object (in contrast, an immutable iterator would restart from the beginning).\n\nExamples\n\njulia> a = Iterators.Stateful(\"abcdef\");\n\njulia> isempty(a)\nfalse\n\njulia> popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(Iterators.take(a, 3))\n3-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n 'd': ASCII/Unicode U+0064 (category Ll: Letter, lowercase)\n\njulia> collect(a)\n2-element Vector{Char}:\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n 'f': ASCII/Unicode U+0066 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a); popfirst!(a)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> Iterators.reset!(a, \"hello\"); popfirst!(a)\n'h': ASCII/Unicode U+0068 (category Ll: Letter, lowercase)\n\njulia> a = Iterators.Stateful([1,1,1,2,3,4]);\n\njulia> for x in a; x == 1 || break; end\n\njulia> peek(a)\n3\n\njulia> sum(a) # Sum the remaining elements\n7\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.zip","page":"Iteration utilities","title":"Base.Iterators.zip","category":"function","text":"zip(iters...)\n\nRun multiple iterators at the same time, until any of them is exhausted. The value type of the zip iterator is a tuple of values of its subiterators.\n\nnote: Note\nzip orders the calls to its subiterators in such a way that stateful iterators will not advance when another iterator finishes in the current iteration.\n\nnote: Note\nzip() with no arguments yields an infinite iterator of empty tuples.\n\nSee also: enumerate, Base.splat.\n\nExamples\n\njulia> a = 1:5\n1:5\n\njulia> b = [\"e\",\"d\",\"b\",\"c\",\"a\"]\n5-element Vector{String}:\n \"e\"\n \"d\"\n \"b\"\n \"c\"\n \"a\"\n\njulia> c = zip(a,b)\nzip(1:5, [\"e\", \"d\", \"b\", \"c\", \"a\"])\n\njulia> length(c)\n5\n\njulia> first(c)\n(1, \"e\")\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.enumerate","page":"Iteration utilities","title":"Base.Iterators.enumerate","category":"function","text":"enumerate(iter)\n\nAn iterator that yields (i, x) where i is a counter starting at 1, and x is the ith value from the given iterator. It's useful when you need not only the values x over which you are iterating, but also the number of iterations so far.\n\nNote that i may not be valid for indexing iter, or may index a different element. This will happen if iter has indices that do not start at 1, and may happen for strings, dictionaries, etc. See the pairs(IndexLinear(), iter) method if you want to ensure that i is an index.\n\nExamples\n\njulia> a = [\"a\", \"b\", \"c\"];\n\njulia> for (index, value) in enumerate(a)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n\njulia> str = \"naïve\";\n\njulia> for (i, val) in enumerate(str)\n           print(\"i = \", i, \", val = \", val, \", \")\n           try @show(str[i]) catch e println(e) end\n       end\ni = 1, val = n, str[i] = 'n'\ni = 2, val = a, str[i] = 'a'\ni = 3, val = ï, str[i] = 'ï'\ni = 4, val = v, StringIndexError(\"naïve\", 4)\ni = 5, val = e, str[i] = 'v'\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.rest","page":"Iteration utilities","title":"Base.Iterators.rest","category":"function","text":"rest(iter, state)\n\nAn iterator that yields the same elements as iter, but starting at the given state, which must be a state obtainable via a sequence of one or more calls to iterate(iter[, state])\n\nSee also: Iterators.drop, Iterators.peel, Base.rest.\n\nExamples\n\njulia> iter = [1,2,3,4];\n\njulia> val, state = iterate(iter)\n(1, 2)\n\njulia> collect(Iterators.rest(iter, state))\n3-element Vector{Int64}:\n 2\n 3\n 4\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.countfrom","page":"Iteration utilities","title":"Base.Iterators.countfrom","category":"function","text":"countfrom(start=1, step=1)\n\nAn iterator that counts forever, starting at start and incrementing by step.\n\nExamples\n\njulia> for v in Iterators.countfrom(5, 2)\n           v > 10 && break\n           println(v)\n       end\n5\n7\n9\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.take","page":"Iteration utilities","title":"Base.Iterators.take","category":"function","text":"take(iter, n)\n\nAn iterator that generates at most the first n elements of iter.\n\nSee also: drop, peel, first, Base.take!.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.take(a,3))\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.takewhile","page":"Iteration utilities","title":"Base.Iterators.takewhile","category":"function","text":"takewhile(pred, iter)\n\nAn iterator that generates element from iter as long as predicate pred is true, afterwards, drops every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.takewhile(<(3),s))\n2-element Vector{Int64}:\n 1\n 2\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.drop","page":"Iteration utilities","title":"Base.Iterators.drop","category":"function","text":"drop(iter, n)\n\nAn iterator that generates all but the first n elements of iter.\n\nExamples\n\njulia> a = 1:2:11\n1:2:11\n\njulia> collect(a)\n6-element Vector{Int64}:\n  1\n  3\n  5\n  7\n  9\n 11\n\njulia> collect(Iterators.drop(a,4))\n2-element Vector{Int64}:\n  9\n 11\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.dropwhile","page":"Iteration utilities","title":"Base.Iterators.dropwhile","category":"function","text":"dropwhile(pred, iter)\n\nAn iterator that drops element from iter as long as predicate pred is true, afterwards, returns every element.\n\ncompat: Julia 1.4\nThis function requires at least Julia 1.4.\n\nExamples\n\njulia> s = collect(1:5)\n5-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n\njulia> collect(Iterators.dropwhile(<(3),s))\n3-element Vector{Int64}:\n 3\n 4\n 5\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.findeach","page":"Iteration utilities","title":"Base.Iterators.findeach","category":"function","text":"findeach(f, it)\nfindeach(it)\n\nAn iterator that generates every key from the key/value pairs of pairs(it), where f(value) returns true.\n\nIf f is not specified, default to identity.\n\nIterators.findeach is the lazy equivalent of findall.\n\ncompat: Julia 1.13\nfindeach requires at least Julia 1.13.\n\nExamples\n\njulia> collect(Iterators.findeach(isodd, Dict(2 => 3, 3 => 2)))\n1-element Vector{Int64}:\n 2\n\njulia> only(Iterators.findeach(==(1), [3,6,2,1]))\n4\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.cycle","page":"Iteration utilities","title":"Base.Iterators.cycle","category":"function","text":"cycle(iter[, n::Int])\n\nAn iterator that cycles through iter forever. If n is specified, then it cycles through iter that many times. When iter is empty, so are cycle(iter) and cycle(iter, n).\n\nIterators.cycle(iter, n) is the lazy equivalent of Base.repeat(vector, n), while Iterators.repeated(iter, n) is the lazy Base.fill(item, n).\n\ncompat: Julia 1.11\nThe method cycle(iter, n) was added in Julia 1.11.\n\nExamples\n\njulia> for (i, v) in enumerate(Iterators.cycle(\"hello\"))\n           print(v)\n           i > 10 && break\n       end\nhellohelloh\n\njulia> foreach(print, Iterators.cycle(['j', 'u', 'l', 'i', 'a'], 3))\njuliajuliajulia\n\njulia> repeat([1,2,3], 4) == collect(Iterators.cycle([1,2,3], 4))\ntrue\n\njulia> fill([1,2,3], 4) == collect(Iterators.repeated([1,2,3], 4))\ntrue\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.repeated","page":"Iteration utilities","title":"Base.Iterators.repeated","category":"function","text":"repeated(x[, n::Int])\n\nAn iterator that generates the value x forever. If n is specified, generates x that many times (equivalent to take(repeated(x), n)).\n\nSee also fill, and compare Iterators.cycle.\n\nExamples\n\njulia> a = Iterators.repeated([1 2], 4);\n\njulia> collect(a)\n4-element Vector{Matrix{Int64}}:\n [1 2]\n [1 2]\n [1 2]\n [1 2]\n\njulia> ans == fill([1 2], 4)\ntrue\n\njulia> Iterators.cycle([1 2], 4) |> collect |> println\n[1, 2, 1, 2, 1, 2, 1, 2]\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.product","page":"Iteration utilities","title":"Base.Iterators.product","category":"function","text":"product(iters...)\n\nReturn an iterator over the product of several iterators. Each generated element is a tuple whose ith element comes from the ith argument iterator. The first iterator changes the fastest.\n\nSee also: zip, Iterators.flatten.\n\nExamples\n\njulia> collect(Iterators.product(1:2, 3:5))\n2×3 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (1, 4)  (1, 5)\n (2, 3)  (2, 4)  (2, 5)\n\njulia> ans == [(x,y) for x in 1:2, y in 3:5]  # collects a generator involving Iterators.product\ntrue\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.flatten","page":"Iteration utilities","title":"Base.Iterators.flatten","category":"function","text":"flatten(iter)\n\nGiven an iterator that yields iterators, return an iterator that yields the elements of those iterators. Put differently, the elements of the argument iterator are concatenated.\n\nExamples\n\njulia> collect(Iterators.flatten((1:2, 8:9)))\n4-element Vector{Int64}:\n 1\n 2\n 8\n 9\n\njulia> [(x,y) for x in 0:1 for y in 'a':'c']  # collects generators involving Iterators.flatten\n6-element Vector{Tuple{Int64, Char}}:\n (0, 'a')\n (0, 'b')\n (0, 'c')\n (1, 'a')\n (1, 'b')\n (1, 'c')\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.flatmap","page":"Iteration utilities","title":"Base.Iterators.flatmap","category":"function","text":"Iterators.flatmap(f, iterators...)\n\nEquivalent to flatten(map(f, iterators...)).\n\nSee also Iterators.flatten, Iterators.map.\n\ncompat: Julia 1.9\nThis function was added in Julia 1.9.\n\nExamples\n\njulia> Iterators.flatmap(n -> -n:2:n, 1:3) |> collect\n9-element Vector{Int64}:\n -1\n  1\n -2\n  0\n  2\n -3\n -1\n  1\n  3\n\njulia> stack(n -> -n:2:n, 1:3)\nERROR: DimensionMismatch: stack expects uniform slices, got axes(x) == (1:3,) while first had (1:2,)\n[...]\n\njulia> Iterators.flatmap(n -> (-n, 10n), 1:2) |> collect\n4-element Vector{Int64}:\n -1\n 10\n -2\n 20\n\njulia> ans == vec(stack(n -> (-n, 10n), 1:2))\ntrue\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.partition","page":"Iteration utilities","title":"Base.Iterators.partition","category":"function","text":"partition(collection, n)\n\nIterate over a collection n elements at a time.\n\nExamples\n\njulia> collect(Iterators.partition([1,2,3,4,5], 2))\n3-element Vector{SubArray{Int64, 1, Vector{Int64}, Tuple{UnitRange{Int64}}, true}}:\n [1, 2]\n [3, 4]\n [5]\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.map","page":"Iteration utilities","title":"Base.Iterators.map","category":"function","text":"Iterators.map(f, iterators...)\n\nCreate a lazy mapping.  This is another syntax for writing (f(args...) for args in zip(iterators...)).\n\ncompat: Julia 1.6\nThis function requires at least Julia 1.6.\n\nExamples\n\njulia> collect(Iterators.map(x -> x^2, 1:3))\n3-element Vector{Int64}:\n 1\n 4\n 9\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.filter","page":"Iteration utilities","title":"Base.Iterators.filter","category":"function","text":"Iterators.filter(flt, itr)\n\nGiven a predicate function flt and an iterable object itr, return an iterable object which upon iteration yields the elements x of itr that satisfy flt(x). The order of the original iterator is preserved.\n\nThis function is lazy; that is, it is guaranteed to return in Θ(1) time and use Θ(1) additional space, and flt will not be called by an invocation of filter. Calls to flt will be made when iterating over the returned iterable object. These calls are not cached and repeated calls will be made when reiterating.\n\nwarning: Warning\nSubsequent lazy transformations on the iterator returned from filter, such as those performed by Iterators.reverse or cycle, will also delay calls to flt until collecting or iterating over the returned iterable object. If the filter predicate is nondeterministic or its return values depend on the order of iteration over the elements of itr, composition with lazy transformations may result in surprising behavior. If this is undesirable, either ensure that flt is a pure function or collect intermediate filter iterators before further transformations.\n\nSee Base.filter for an eager implementation of filtering for arrays.\n\nExamples\n\njulia> f = Iterators.filter(isodd, [1, 2, 3, 4, 5])\nBase.Iterators.Filter{typeof(isodd), Vector{Int64}}(isodd, [1, 2, 3, 4, 5])\n\njulia> foreach(println, f)\n1\n3\n5\n\njulia> [x for x in [1, 2, 3, 4, 5] if isodd(x)]  # collects a generator over Iterators.filter\n3-element Vector{Int64}:\n 1\n 3\n 5\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.accumulate","page":"Iteration utilities","title":"Base.Iterators.accumulate","category":"function","text":"Iterators.accumulate(f, itr; [init])\n\nGiven a 2-argument function f and an iterator itr, return a new iterator that successively applies f to the previous value and the next element of itr.\n\nThis is effectively a lazy version of Base.accumulate.\n\ncompat: Julia 1.5\nKeyword argument init is added in Julia 1.5.\n\nExamples\n\njulia> a = Iterators.accumulate(+, [1,2,3,4]);\n\njulia> foreach(println, a)\n1\n3\n6\n10\n\njulia> b = Iterators.accumulate(/, (2, 5, 2, 5); init = 100);\n\njulia> collect(b)\n4-element Vector{Float64}:\n 50.0\n 10.0\n  5.0\n  1.0\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.reverse","page":"Iteration utilities","title":"Base.Iterators.reverse","category":"function","text":"Iterators.reverse(itr)\n\nGiven an iterator itr, then reverse(itr) is an iterator over the same collection but in the reverse order. This iterator is \"lazy\" in that it does not make a copy of the collection in order to reverse it; see Base.reverse for an eager implementation.\n\n(By default, this returns an Iterators.Reverse object wrapping itr, which is iterable if the corresponding iterate methods are defined, but some itr types may implement more specialized Iterators.reverse behaviors.)\n\nNot all iterator types T support reverse-order iteration.  If T doesn't, then iterating over Iterators.reverse(itr::T) will throw a MethodError because of the missing iterate methods for Iterators.Reverse{T}. (To implement these methods, the original iterator itr::T can be obtained from an r::Iterators.Reverse{T} object by r.itr; more generally, one can use Iterators.reverse(r).)\n\nExamples\n\njulia> foreach(println, Iterators.reverse(1:5))\n5\n4\n3\n2\n1\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.only","page":"Iteration utilities","title":"Base.Iterators.only","category":"function","text":"only(x)\n\nReturn the one and only element of collection x, or throw an ArgumentError if the collection has zero or multiple elements.\n\nSee also first, last.\n\ncompat: Julia 1.4\nThis method requires at least Julia 1.4.\n\nExamples\n\njulia> only([\"a\"])\n\"a\"\n\njulia> only(\"a\")\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> only(())\nERROR: ArgumentError: Tuple contains 0 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\njulia> only(('a', 'b'))\nERROR: ArgumentError: Tuple contains 2 elements, must contain exactly 1 element\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"base/iterators.html#Base.Iterators.peel","page":"Iteration utilities","title":"Base.Iterators.peel","category":"function","text":"peel(iter)\n\nReturns the first element and an iterator over the remaining elements.\n\nIf the iterator is empty return nothing (like iterate).\n\ncompat: Julia 1.7\nPrior versions throw a BoundsError if the iterator is empty.\n\nSee also: Iterators.drop, Iterators.take.\n\nExamples\n\njulia> (a, rest) = Iterators.peel(\"abc\");\n\njulia> a\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> collect(rest)\n2-element Vector{Char}:\n 'b': ASCII/Unicode U+0062 (category Ll: Letter, lowercase)\n 'c': ASCII/Unicode U+0063 (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"location":"devdocs/offset-arrays.html#man-custom-indices","page":"Arrays with custom indices","title":"Arrays with custom indices","category":"section","text":"Conventionally, Julia's arrays are indexed starting at 1, whereas some other languages start numbering at 0, and yet others (e.g., Fortran) allow you to specify arbitrary starting indices. While there is much merit in picking a standard (i.e., 1 for Julia), there are some algorithms which simplify considerably if you can index outside the range 1:size(A,d) (and not just 0:size(A,d)-1, either). To facilitate such computations, Julia supports arrays with arbitrary indices.\n\nThe purpose of this page is to address the question, \"what do I have to do to support such arrays in my own code?\"  First, let's address the simplest case: if you know that your code will never need to handle arrays with unconventional indexing, hopefully the answer is \"nothing.\" Old code, on conventional arrays, should function essentially without alteration as long as it was using the exported interfaces of Julia. If you find it more convenient to just force your users to supply traditional arrays where indexing starts at one, you can add\n\nBase.require_one_based_indexing(arrays...)\n\nwhere arrays... is a list of the array objects that you wish to check for anything that violates 1-based indexing."},{"location":"devdocs/offset-arrays.html#Generalizing-existing-code","page":"Arrays with custom indices","title":"Generalizing existing code","category":"section","text":"As an overview, the steps are:\n\nreplace many uses of size with axes\nreplace 1:length(A) with eachindex(A), or in some cases LinearIndices(A)\nreplace explicit allocations like Array{Int}(undef, size(B)) with similar(Array{Int}, axes(B))\n\nThese are described in more detail below."},{"location":"devdocs/offset-arrays.html#Things-to-watch-out-for","page":"Arrays with custom indices","title":"Things to watch out for","category":"section","text":"Because unconventional indexing breaks many people's assumptions that all arrays start indexing with 1, there is always the chance that using such arrays will trigger errors. The most frustrating bugs would be incorrect results or segfaults (total crashes of Julia). For example, consider the following function:\n\nfunction mycopy!(dest::AbstractVector, src::AbstractVector)\n    length(dest) == length(src) || throw(DimensionMismatch(\"vectors must match\"))\n    # OK, now we're safe to use @inbounds, right? (not anymore!)\n    for i = 1:length(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend\n\nThis code implicitly assumes that vectors are indexed from 1; if dest starts at a different index than src, there is a chance that this code would trigger a segfault. (If you do get segfaults, to help locate the cause try running julia with the option --check-bounds=yes.)"},{"location":"devdocs/offset-arrays.html#Using-axes-for-bounds-checks-and-loop-iteration","page":"Arrays with custom indices","title":"Using axes for bounds checks and loop iteration","category":"section","text":"axes(A) (reminiscent of size(A)) returns a tuple of AbstractUnitRange{<:Integer} objects, specifying the range of valid indices along each dimension of A. When A has unconventional indexing, the ranges may not start at 1. If you just want the range for a particular dimension d, there is axes(A, d).\n\nBase implements a custom range type, OneTo, where OneTo(n) means the same thing as 1:n but in a form that guarantees (via the type system) that the lower index is 1. For any new AbstractArray type, this is the default returned by axes, and it indicates that this array type uses \"conventional\" 1-based indexing.\n\nFor bounds checking, note that there are dedicated functions checkbounds and checkindex which can sometimes simplify such tests."},{"location":"devdocs/offset-arrays.html#Linear-indexing-(LinearIndices)","page":"Arrays with custom indices","title":"Linear indexing (LinearIndices)","category":"section","text":"Some algorithms are most conveniently (or efficiently) written in terms of a single linear index, A[i] even if A is multi-dimensional. Regardless of the array's native indices, linear indices always range from 1:length(A). However, this raises an ambiguity for one-dimensional arrays (a.k.a., AbstractVector): does v[i] mean linear indexing , or Cartesian indexing with the array's native indices?\n\nFor this reason, your best option may be to iterate over the array with eachindex(A), or, if you require the indices to be sequential integers, to get the index range by calling LinearIndices(A). This will return axes(A, 1) if A is an AbstractVector, and the equivalent of 1:length(A) otherwise.\n\nBy this definition, 1-dimensional arrays always use Cartesian indexing with the array's native indices. To help enforce this, it's worth noting that the index conversion functions will throw an error if shape indicates a 1-dimensional array with unconventional indexing (i.e., is a Tuple{UnitRange} rather than a tuple of OneTo). For arrays with conventional indexing, these functions continue to work the same as always.\n\nUsing axes and LinearIndices, here is one way you could rewrite mycopy!:\n\nfunction mycopy!(dest::AbstractVector, src::AbstractVector)\n    axes(dest) == axes(src) || throw(DimensionMismatch(\"vectors must match\"))\n    for i in LinearIndices(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend"},{"location":"devdocs/offset-arrays.html#Allocating-storage-using-generalizations-of-similar","page":"Arrays with custom indices","title":"Allocating storage using generalizations of similar","category":"section","text":"Storage is often allocated with Array{Int}(undef, dims) or similar(A, args...). When the result needs to match the indices of some other array, this may not always suffice. The generic replacement for such patterns is to use similar(storagetype, shape).  storagetype indicates the kind of underlying \"conventional\" behavior you'd like, e.g., Array{Int} or BitArray or even dims->zeros(Float32, dims) (which would allocate an all-zeros array). shape is a tuple of Integer or AbstractUnitRange values, specifying the indices that you want the result to use. Note that a convenient way of producing an all-zeros array that matches the indices of A is simply zeros(A).\n\nLet's walk through a couple of explicit examples. First, if A has conventional indices, then similar(Array{Int}, axes(A)) would end up calling Array{Int}(undef, size(A)), and thus return an array. If A is an AbstractArray type with unconventional indexing, then similar(Array{Int}, axes(A)) should return something that \"behaves like\" an Array{Int} but with a shape (including indices) that matches A.  (The most obvious implementation is to allocate an Array{Int}(undef, size(A)) and then \"wrap\" it in a type that shifts the indices.)\n\nNote also that similar(Array{Int}, (axes(A, 2),)) would allocate an AbstractVector{Int} (i.e., 1-dimensional array) that matches the indices of the columns of A."},{"location":"devdocs/offset-arrays.html#Writing-custom-array-types-with-non-1-indexing","page":"Arrays with custom indices","title":"Writing custom array types with non-1 indexing","category":"section","text":"Most of the methods you'll need to define are standard for any AbstractArray type, see Abstract Arrays. This page focuses on the steps needed to define unconventional indexing."},{"location":"devdocs/offset-arrays.html#Custom-AbstractUnitRange-types","page":"Arrays with custom indices","title":"Custom AbstractUnitRange types","category":"section","text":"If you're writing a non-1 indexed array type, you will want to specialize axes so it returns a UnitRange, or (perhaps better) a custom AbstractUnitRange. The advantage of a custom type is that it \"signals\" the allocation type for functions like similar. If we're writing an array type for which indexing will start at 0, we likely want to begin by creating a new AbstractUnitRange, ZeroRange, where ZeroRange(n) is equivalent to 0:n-1.\n\nIn general, you should probably not export ZeroRange from your package: there may be other packages that implement their own ZeroRange, and having multiple distinct ZeroRange types is (perhaps counterintuitively) an advantage: ModuleA.ZeroRange indicates that similar should create a ModuleA.ZeroArray, whereas ModuleB.ZeroRange indicates a ModuleB.ZeroArray type.  This design allows peaceful coexistence among many different custom array types.\n\nNote that the Julia package CustomUnitRanges.jl can sometimes be used to avoid the need to write your own ZeroRange type."},{"location":"devdocs/offset-arrays.html#Specializing-axes","page":"Arrays with custom indices","title":"Specializing axes","category":"section","text":"Once you have your AbstractUnitRange type, then specialize axes:\n\nBase.axes(A::ZeroArray) = map(n->ZeroRange(n), A.size)\n\nwhere here we imagine that ZeroArray has a field called size (there would be other ways to implement this).\n\nIn some cases, the fallback definition for axes(A, d):\n\naxes(A::AbstractArray{T,N}, d) where {T,N} = d <= N ? axes(A)[d] : OneTo(1)\n\nmay not be what you want: you may need to specialize it to return something other than OneTo(1) when d > ndims(A). Likewise, in Base there is a dedicated function axes1 which is equivalent to axes(A, 1) but which avoids checking (at runtime) whether ndims(A) > 0. (This is purely a performance optimization.)  It is defined as:\n\naxes1(A::AbstractArray{T,0}) where {T} = OneTo(1)\naxes1(A::AbstractArray) = axes(A)[1]\n\nIf the first of these (the zero-dimensional case) is problematic for your custom array type, be sure to specialize it appropriately."},{"location":"devdocs/offset-arrays.html#Specializing-similar","page":"Arrays with custom indices","title":"Specializing similar","category":"section","text":"Given your custom ZeroRange type, then you should also add the following two specializations for similar:\n\nfunction Base.similar(A::AbstractArray, T::Type, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nfunction Base.similar(f::Union{Function,DataType}, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nBoth of these should allocate your custom array type."},{"location":"devdocs/offset-arrays.html#Specializing-reshape","page":"Arrays with custom indices","title":"Specializing reshape","category":"section","text":"Optionally, define a method\n\nBase.reshape(A::AbstractArray, shape::Tuple{ZeroRange,Vararg{ZeroRange}}) = ...\n\nand you can reshape an array so that the result has custom indices."},{"location":"devdocs/offset-arrays.html#For-objects-that-mimic-AbstractArray-but-are-not-subtypes","page":"Arrays with custom indices","title":"For objects that mimic AbstractArray but are not subtypes","category":"section","text":"has_offset_axes depends on having axes defined for the objects you call it on. If there is some reason you don't have an axes method defined for your object, consider defining a method\n\nBase.has_offset_axes(obj::MyNon1IndexedArraylikeObject) = true\n\nThis will allow code that assumes 1-based indexing to detect a problem and throw a helpful error, rather than returning incorrect results or segfaulting julia."},{"location":"devdocs/offset-arrays.html#Catching-errors","page":"Arrays with custom indices","title":"Catching errors","category":"section","text":"If your new array type triggers errors in other code, one helpful debugging step can be to comment out @boundscheck in your getindex and setindex! implementation. This will ensure that every element access checks bounds. Or, restart julia with --check-bounds=yes.\n\nIn some cases it may also be helpful to temporarily disable size and length for your new array type, since code that makes incorrect assumptions frequently uses these functions."},{"location":"manual/memory-management.html#man-memory-management","page":"Memory Management and Garbage Collection","title":"Memory Management and Garbage Collection","category":"section","text":"Julia uses automatic memory management through its built-in garbage collector (GC). This section provides an overview of how Julia manages memory and how you can configure and optimize memory usage for your applications."},{"location":"manual/memory-management.html#man-gc-overview","page":"Memory Management and Garbage Collection","title":"Garbage Collection Overview","category":"section","text":"Julia features a garbage collector with the following characteristics:\n\nNon-moving: Objects are not relocated in memory during garbage collection\nGenerational: Younger objects are collected more frequently than older ones\nParallel and partially concurrent: The GC can use multiple threads and run concurrently with your program\nMostly precise: The GC accurately identifies object references for pure Julia code, and it provides conservative scanning APIs for users calling Julia from C\n\nThe garbage collector automatically reclaims memory used by objects that are no longer reachable from your program, freeing you from manual memory management in most cases."},{"location":"manual/memory-management.html#man-memory-architecture","page":"Memory Management and Garbage Collection","title":"Memory Architecture","category":"section","text":"Julia uses a two-tier allocation strategy:\n\nSmall objects (currently ≤ 2032 bytes but may change): Allocated using a fast per-thread pool allocator\nLarge objects : Allocated directly through the system's malloc\n\nThis hybrid approach optimizes for both allocation speed and memory efficiency, with the pool allocator providing fast allocation for the many small objects typical in Julia programs."},{"location":"manual/memory-management.html#man-system-memory","page":"Memory Management and Garbage Collection","title":"System Memory Requirements","category":"section","text":""},{"location":"manual/memory-management.html#Swap-Space","page":"Memory Management and Garbage Collection","title":"Swap Space","category":"section","text":"Julia's garbage collector is designed with the expectation that your system has adequate swap space configured. The GC uses heuristics that assume it can allocate memory beyond physical RAM when needed, relying on the operating system's virtual memory management.\n\nIf your system has limited or no swap space, you may experience out-of-memory errors during garbage collection. In such cases, you can use the --heap-size-hint option to limit Julia's memory usage."},{"location":"manual/memory-management.html#Memory-Hints","page":"Memory Management and Garbage Collection","title":"Memory Hints","category":"section","text":"You can provide a hint to Julia about the maximum amount of memory to use:\n\njulia --heap-size-hint=4G  # To set the hint to ~4GB\njulia --heap-size-hint=50% # or to 50% of physical memory\n\nThe --heap-size-hint option tells the garbage collector to trigger collection more aggressively when approaching the specified limit. This is particularly useful in:\n\nContainers with memory limits\nSystems without swap space\nShared systems where you want to limit Julia's memory footprint\n\nYou can also set this via the JULIA_HEAP_SIZE_HINT environment variable:\n\nexport JULIA_HEAP_SIZE_HINT=2G\njulia"},{"location":"manual/memory-management.html#man-gc-multithreading","page":"Memory Management and Garbage Collection","title":"Multithreaded Garbage Collection","category":"section","text":"Julia's garbage collector can leverage multiple threads to improve performance on multi-core systems."},{"location":"manual/memory-management.html#GC-Thread-Configuration","page":"Memory Management and Garbage Collection","title":"GC Thread Configuration","category":"section","text":"By default, Julia uses multiple threads for garbage collection:\n\nMark threads: Used during the mark phase to trace object references (default: 1, which is shared with the compute thread if there is only one, otherwise half the number of compute threads)\nSweep threads: Used for concurrent sweeping of freed memory (default: 0, disabled)\n\nYou can configure GC threading using:\n\njulia --gcthreads=4,1  # 4 mark threads, 1 sweep thread\njulia --gcthreads=8    # 8 mark threads, 0 sweep threads\n\nOr via environment variable:\n\nexport JULIA_NUM_GC_THREADS=4,1\njulia"},{"location":"manual/memory-management.html#Recommendations","page":"Memory Management and Garbage Collection","title":"Recommendations","category":"section","text":"For compute-intensive workloads:\n\nUse multiple mark threads (the default configuration is usually appropriate)\nConsider enabling concurrent sweeping with 1 sweep thread for allocation-heavy workloads\n\nFor memory-intensive workloads:\n\nEnable concurrent sweeping to reduce GC pauses\nMonitor GC time using @time and adjust thread counts accordingly"},{"location":"manual/memory-management.html#man-gc-monitoring","page":"Memory Management and Garbage Collection","title":"Monitoring and Debugging","category":"section","text":""},{"location":"manual/memory-management.html#Basic-Memory-Monitoring","page":"Memory Management and Garbage Collection","title":"Basic Memory Monitoring","category":"section","text":"Use the @time macro to see memory allocation and GC overhead:\n\njulia> @time some_computation()\n  2.123456 seconds (1.50 M allocations: 58.725 MiB, 17.17% gc time)"},{"location":"manual/memory-management.html#GC-Logging","page":"Memory Management and Garbage Collection","title":"GC Logging","category":"section","text":"Enable detailed GC logging to understand collection patterns:\n\njulia> GC.enable_logging(true)\njulia> # Run your code\njulia> GC.enable_logging(false)\n\nThis logs each garbage collection event with timing and memory statistics."},{"location":"manual/memory-management.html#Manual-GC-Control","page":"Memory Management and Garbage Collection","title":"Manual GC Control","category":"section","text":"While generally not recommended, you can manually trigger garbage collection:\n\nGC.gc()          # Force a garbage collection\nGC.enable(false) # Disable automatic GC (use with caution!)\nGC.enable(true)  # Re-enable automatic GC\n\nWarning: Disabling GC can lead to memory exhaustion. Only use this for specific performance measurements or debugging."},{"location":"manual/memory-management.html#man-gc-performance","page":"Memory Management and Garbage Collection","title":"Performance Considerations","category":"section","text":""},{"location":"manual/memory-management.html#Reducing-Allocations","page":"Memory Management and Garbage Collection","title":"Reducing Allocations","category":"section","text":"The best way to minimize GC impact is to reduce unnecessary allocations:\n\nUse in-place operations when possible (e.g., x .+= y instead of x = x + y)\nPre-allocate arrays and reuse them\nAvoid creating temporary objects in tight loops\nConsider using StaticArrays.jl for small, fixed-size arrays"},{"location":"manual/memory-management.html#Memory-Efficient-Patterns","page":"Memory Management and Garbage Collection","title":"Memory-Efficient Patterns","category":"section","text":"Avoid global variables that change type\nUse const for global constants"},{"location":"manual/memory-management.html#Profiling-Memory-Usage","page":"Memory Management and Garbage Collection","title":"Profiling Memory Usage","category":"section","text":"For detailed guidance on profiling memory allocations and identifying performance bottlenecks, see the Profiling section."},{"location":"manual/memory-management.html#man-gc-advanced","page":"Memory Management and Garbage Collection","title":"Advanced Configuration","category":"section","text":""},{"location":"manual/memory-management.html#Integration-with-System-Memory-Management","page":"Memory Management and Garbage Collection","title":"Integration with System Memory Management","category":"section","text":"Julia works best when:\n\nThe system has adequate swap space (recommended: 2x physical RAM)\nVirtual memory is properly configured\nOther processes leave sufficient memory available\nContainer memory limits are set appropriately with --heap-size-hint"},{"location":"manual/memory-management.html#man-gc-troubleshooting","page":"Memory Management and Garbage Collection","title":"Troubleshooting Memory Issues","category":"section","text":""},{"location":"manual/memory-management.html#High-GC-Overhead","page":"Memory Management and Garbage Collection","title":"High GC Overhead","category":"section","text":"If garbage collection is taking too much time:\n\nReduce allocation rate: Focus on algorithmic improvements\nAdjust GC threads: Experiment with different --gcthreads settings\nUse concurrent sweeping: Enable background sweeping with --gcthreads=N,1\nProfile memory patterns: Identify allocation hotspots and optimize them"},{"location":"manual/memory-management.html#Memory-Leaks","page":"Memory Management and Garbage Collection","title":"Memory Leaks","category":"section","text":"While Julia's GC prevents most memory leaks, issues can still occur:\n\nGlobal references: Avoid holding references to large objects in global variables\nClosures: Be careful with closures that capture large amounts of data\nC interop: Ensure proper cleanup when interfacing with C libraries\n\nFor more detailed information about Julia's garbage collector internals, see the Garbage Collection section in the Developer Documentation."},{"location":"manual/parallel-computing.html#Parallel-Computing","page":"Parallel Computing","title":"Parallel Computing","category":"section","text":"Julia supports these four categories of concurrent and parallel programming:\n\nAsynchronous \"tasks\", or coroutines:\nJulia Tasks allow suspending and resuming computations  for I/O, event handling, producer-consumer processes, and similar patterns.  Tasks can synchronize through operations like wait and fetch, and  communicate via Channels. While strictly not parallel computing by themselves,  Julia lets you schedule Tasks on several threads.\nMulti-threading:\nJulia's multi-threading provides the ability to schedule Tasks  simultaneously on more than one thread or CPU core, sharing memory. This is usually the easiest way  to get parallelism on one's PC or on a single large multi-core server. Julia's multi-threading  is composable. When one multi-threaded function calls another multi-threaded function, Julia  will schedule all the threads globally on available resources, without oversubscribing.\nDistributed computing:\nDistributed computing runs multiple Julia processes with separate memory spaces. These can be on the same  computer or multiple computers. The Distributed standard library provides the capability for remote execution  of a Julia function. With this basic building block, it is possible to build many different kinds of  distributed computing abstractions. Packages like DistributedArrays.jl  are an example of such an abstraction. On the other hand, packages like MPI.jl and  Elemental.jl provide access to the existing MPI ecosystem of libraries.\nGPU computing:\nThe Julia GPU compiler provides the ability to run Julia code natively on GPUs. There  is a rich ecosystem of Julia packages that target GPUs. The JuliaGPU.org  website provides a list of capabilities, supported GPUs, related packages and documentation."},{"location":"devdocs/ast.html#Julia-ASTs","page":"Julia ASTs","title":"Julia ASTs","category":"section","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser (e.g. the Meta.parse function), and manipulated by macros. It is a structured representation of code as it is written, constructed by julia-parser.scm from a character stream. Next there is a lowered form, or IR (intermediate representation), which is used by type inference and code generation. In the lowered form there are fewer types of nodes, all macros are expanded, and all control flow is converted to explicit branches and sequences of statements. The lowered form is constructed by julia-syntax.scm.\n\nFirst we will focus on the AST, since it is needed to write macros."},{"location":"devdocs/ast.html#Surface-syntax-AST","page":"Julia ASTs","title":"Surface syntax AST","category":"section","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers). There is generally a different expression head for each visually distinct syntactic form. Examples will be given in s-expression syntax. Each parenthesized list corresponds to an Expr, where the first element is the head. For example (call f x) corresponds to Expr(:call, :f, :x) in Julia."},{"location":"devdocs/ast.html#Calls","page":"Julia ASTs","title":"Calls","category":"section","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))\n\ndo syntax:\n\nf(x) do a,b\n    body\nend\n\nparses as (do (call f x) (-> (tuple a b) (block body)))."},{"location":"devdocs/ast.html#Operators","page":"Julia ASTs","title":"Operators","category":"section","text":"Most uses of operators are just function calls, so they are parsed with the head call. However some operators are special forms (not necessarily function calls), and in those cases the operator itself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\". Some operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument call. Finally, chains of comparisons have their own special expression structure.\n\nInput AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))"},{"location":"devdocs/ast.html#Bracketed-forms","page":"Julia ASTs","title":"Bracketed forms","category":"section","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)"},{"location":"devdocs/ast.html#Macros","page":"Julia ASTs","title":"Macros","category":"section","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)"},{"location":"devdocs/ast.html#Strings","page":"Julia ASTs","title":"Strings","category":"section","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")\n\nDoc string syntax:\n\n\"some docs\"\nf(x) = x\n\nparses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x)))."},{"location":"devdocs/ast.html#Imports-and-such","page":"Julia ASTs","title":"Imports and such","category":"section","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)\npublic a, b (public a b)\n\nusing has the same representation as import, but with expression head :using instead of :import.\n\nTo programmatically create a public statement, you can use Expr(:public, :a, :b) or, closer to regular code, Meta.parse(\"public a, b\"). This approach is necessary due to current limitations on public. The public keyword is only recognized at the syntactic top level within a file (parse_stmts) or module. This restriction was implemented to prevent breaking existing code that used public as an identifier when it was introduced in Julia 1.11."},{"location":"devdocs/ast.html#Numbers","page":"Julia ASTs","title":"Numbers","category":"section","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented directly as scheme numbers in the AST.\n\nInput AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")"},{"location":"devdocs/ast.html#Block-forms","page":"Julia ASTs","title":"Block forms","category":"section","text":"A block of statements is parsed as (block stmt1 stmt2 ...).\n\nIf statement:\n\nif a\n    b\nelseif c\n    d\nelse\n    e\nend\n\nparses as:\n\n(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6) e)))\n\nA while loop parses as (while condition body).\n\nA for loop parses as (for (= var iter) body). If there is more than one iteration specification, they are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body).\n\nbreak and continue are parsed as 0-argument expressions (break) and (continue).\n\nlet is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body), like for loops.\n\nA basic function definition is parsed as (function (call f x) body). A more complex example:\n\nfunction f(x::T; k = 1) where T\n    return x+1\nend\n\nparses as:\n\n(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))\n\nType definition:\n\nmutable struct Foo{T<:S}\n    x::T\nend\n\nparses as:\n\n(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))\n\nThe first argument is a boolean telling whether the type is mutable.\n\ntry blocks parse as (try try_block var catch_block finally_block). If no variable is present after catch, var is #f. If there is no finally clause, then the last argument is not present."},{"location":"devdocs/ast.html#Quote-expressions","page":"Julia ASTs","title":"Quote expressions","category":"section","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $. In Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms. Internally, there is also a need for code quoting without interpolation. In Julia's scheme code, non-interpolating quote is represented with the expression head inert.\n\ninert expressions are converted to Julia QuoteNode objects. These objects wrap a single value of any type, and when evaluated simply return that value.\n\nA quote expression whose argument is an atom also gets converted to a QuoteNode."},{"location":"devdocs/ast.html#Line-numbers","page":"Julia ASTs","title":"Line numbers","category":"section","text":"Source location information is represented as (line line_num file_name) where the third component is optional (and omitted when the current line number, but not file name, changes).\n\nThese expressions are represented as LineNumberNodes in Julia."},{"location":"devdocs/ast.html#Macros-2","page":"Julia ASTs","title":"Macros","category":"section","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope. The result of a macro expansion is automatically wrapped in (hygienic-scope block module [lno]), to represent the result of the new scope. The user can insert (escape block) inside to interpolate code from the caller. The lno is the __source__ argument of the macro, if included."},{"location":"devdocs/ast.html#Lowered-form","page":"Julia ASTs","title":"Lowered form","category":"section","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference, optimizations like inlining, and code generation. It is also less obvious to the human, since it results from a significant rearrangement of the input syntax.\n\nIn addition to Symbols and some number types, the following data types exist in lowered form:\n\nExpr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of subexpressions. While almost every part of a surface AST is represented by an Expr, the IR uses only a limited number of Exprs, mostly for calls and some top-level-only forms.\nSlotNumber\nIdentifies arguments and local variables by consecutive numbering. It has an integer-valued id field giving the slot index. The types of these slots can be found in the slottypes field of their CodeInfo object.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the referenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in the code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified by the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function. If the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a QuoteNode whose value field is the symbol a, in order to return the symbol itself instead of evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted by the compiler. The number (id) of an SSAValue is the code array index of the expression whose value it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined."},{"location":"devdocs/ast.html#Expr-types","page":"Julia ASTs","title":"Expr types","category":"section","text":"These symbols appear in the head field of Exprs in lowered form.\n\ncall\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the arguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a SlotNumber or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end. In the 1-argument form, the argument is a symbol. If this symbol already names a function in the current scope, nothing happens. If the symbol is undefined, a new function is created and assigned to the identifier specified by the symbol. If the symbol is defined but names a non-function, an error is raised. The definition of \"names a function\" is that the binding is constant, and refers to an object of singleton type. The rationale for this is that an instance of a singleton type uniquely identifies the type to add the method to. When the type has fields, it wouldn't be clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression first behaves like the 1-argument form above. This argument is ignored from then on. It can be nothing when methods are added strictly by type, (::T)(x) = x, or when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the argument types, and args[2][2] is a SimpleVector of type variables corresponding to the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a method to a function that also has methods defined in different scopes) this is an expression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer, GlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number of fields, or the minimum number of fields called by an inner constructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The arguments are the same as arguments 1, 2, and 4 of struct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4 are the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5 and replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered to this, and the type is always inserted by the compiler. This is very much an internal-only feature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to splat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has already been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception(ct).\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on error. Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a catch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this expression is true or false (true means bounds checks are disabled), it is pushed onto the stack. If the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds, otherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark the inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply copied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the arguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{<:Union{Symbol,Tuple{Symbol,UInt16}, Tuple{Symbol,UInt16,Bool}}: calling convention\nThe calling convention for the call, optionally with effects, and gc_safe (safe to execute concurrently to GC.).\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call. See Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[3] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[4] : constprop\nIndicates whether the opaque closure's identity may be used for constant propagation. The @opaque macro enables this by default, but this will cause additional inference which may be undesirable and prevents the code from running during precompile. If args[4] is a method, the argument is considered skipped.\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7"},{"location":"devdocs/ast.html#ast-lowered-method","page":"Julia ASTs","title":"Method","category":"section","text":"A unique'd container describing the shared metadata for a single method.\n\nname, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness. Uniqueness is required for efficiency, especially for incremental precompile and tracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by compression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, is_for_opaque_closure,\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method."},{"location":"devdocs/ast.html#MethodInstance","page":"Julia ASTs","title":"MethodInstance","category":"section","text":"A unique'd container describing a single callable signature for a Method. See especially Proper maintenance and care of multi-threading locks for important details on how to modify these fields safely.\n\nspecTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a def.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module, if this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes. For the MethodInstance at Method.unspecialized, this is the empty SimpleVector. But for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions. This works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance. Those optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation. Thus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation."},{"location":"devdocs/ast.html#CodeInstance","page":"Julia ASTs","title":"CodeInstance","category":"section","text":"def\nThe MethodInstance that this cache entry is derived from.\nowner\nA token that represents the owner of this CodeInstance. Will use jl_egal to match.\n\nrettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is also the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function, or it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JL_CALLABLE jl_value_t *(*)(jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called. If max_world is the special token value -1, the value is not yet known. It may continue to be used until we encounter a backedge that requires us to reconsider.\nTiming fields\ntime_infer_total: Total cost of computing inferred originally as wall-time from start to finish.\ntime_infer_cache_saved: The cost saved from time_infer_total by having caching. Adding this to time_infer_total should give a stable estimate for comparing the cost of two implementations or one implementation over time. This is generally an over-estimate of the time to infer something, since the cache is frequently effective at handling repeated work.\ntime_infer_self: Self cost of julia inference for inferred (a portion of time_infer_total). This is simply the incremental cost of compiling this one method, if given a fully populated cache of all call targets, even including constant inference results and LimitedAccuracy results, which generally are not in a cache.\ntime_compile: Self cost of llvm JIT compilation (e.g. of computing invoke from inferred). A total cost estimate can be computed by walking all of the edges contents and summing those, while accounting for cycles and duplicates. (This field currently does not include any measured AOT compile times.)"},{"location":"devdocs/ast.html#CodeInfo","page":"Julia ASTs","title":"CodeInfo","category":"section","text":"A (usually temporary) container for holding lowered (and possibly inferred) source code.\n\ncode\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n0x02 - assigned (only false if there are no assignment statements with this var on the left)\n0x08 - used (if there is any read or write of the slot)\n0x10 - statically assigned once\n0x20 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the function (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level 32 bits flags for each expression in the function. See the definition of jl_code_info_t in julia.h for more details.\n\nThese are only populated after inference (or by generated functions in some cases):\n\ndebuginfo\nAn object to retrieve source information for each statements, see How to interpret line numbers in a CodeInfo object.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any. This is mostly present for convenience, as (due to the way OpaqueClosures work) it is not necessarily the rettype used by codegen.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred.\n\nOptional Fields:\n\nslottypes\nAn array of types for the slots.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if necessary during inference.\n\nBoolean properties:\n\npropagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding @boundscheck blocks.\n\nUInt8 settings:\n\nconstprop, inlineable\n0 = use heuristic\n1 = aggressive\n2 = none\npurity Constructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details."},{"location":"devdocs/ast.html#How-to-interpret-line-numbers-in-a-CodeInfo-object","page":"Julia ASTs","title":"How to interpret line numbers in a CodeInfo object","category":"section","text":"There are 2 common forms for this data: one used internally that compresses the data somewhat and one used in the compiler. They contain the same basic info, but the compiler version is all mutable while the version used internally is not.\n\nMany consumers may be able to call Base.IRShow.buildLineInfoNode, Base.IRShow.append_scopes!, or Stacktraces.lookup(::InterpreterIP) to avoid needing to (re-)implement these details specifically.\n\nThe definitions of each of these are:\n\nstruct Core.DebugInfo\n    @noinline\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::SimpleVector{DebugInfo}\n    codelocs::String # compressed data\nend\nmutable struct Core.Compiler.DebugInfoStream\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::Vector{DebugInfo}\n    firstline::Int32 # the starting line for this block (specified by an index of 0)\n    codelocs::Vector{Int32} # for each statement:\n        # index into linetable (if defined), else a line number (in the file represented by def)\n        # then index into edges\n        # then index into edges[linetable]\nend\n\ndef : where this DebugInfo was defined (the Method, MethodInstance, or Symbol of file scope, for example)\nlinetable\nAnother DebugInfo that this was derived from, which contains the actual line numbers, such that this DebugInfo contains only the indexes into it. This avoids making copies, as well as makes it possible to track how each individual statement transformed from source to optimized, not just the separate line numbers. If def is not a Symbol, then that object replaces the current function object for the metadata on what function is conceptually being executed (e.g. think Cassette transforms here). The codelocs values described below also are interpreted as an index into the codelocs in this object, instead of being a line number itself.\nedges : Vector of the unique DebugInfo for every function inlined into this (which recursively have the edges for everything inlined into them).\nfirstline (when uncompressed to DebugInfoStream)\nThe line number associated with the begin statement (or other keyword such as function or quote) that delineates where this code definition \"starts\".\ncodelocs (when uncompressed to DebugInfoStream)\nA vector of indices, with 3 values for each statement in the IR plus one for the starting point of the block, that describe the stacktrace from that point:\nthe integer index into the linetable.codelocs field, giving the original location associated with each statement (including its syntactic edges), or zero indicating no change to the line number from the previously executed statement (which is not necessarily syntactic or lexical prior), or the line number itself if the linetable field is nothing.\nthe integer index into edges, giving the DebugInfo inlined there, or zero if there are no edges.\n(if entry 2 is non-zero) the integer index into edges[].codelocs, to interpret recursively for each function in the inlining stack, or zero indicating to use edges[].firstline as the line number.\nSpecial codes include:\n(zero, zero, *): no change to the line number or edges from the previous statement (you may choose to interpret this either syntactically or lexically). The inlining depth also might have changed, though most callers should ignore that.\n(zero, non-zero, *) : no line number, just edges (usually because of macro-expansion into top-level code)."},{"location":"base/scopedvalues.html#scoped-values","page":"Scoped Values","title":"Scoped Values","category":"section","text":"Scoped values provide an implementation of dynamic scoping in Julia.\n\nnote: Lexical scoping vs dynamic scoping\nLexical scoping is the default behavior in Julia. Under lexical scoping the scope of a variable is determined by the lexical (textual) structure of a program. Under dynamic scoping a variable is bound to the most recent assigned value during the program's execution.\n\nThe state of a scoped value is dependent on the execution path of the program. This means that for a scoped value you may observe multiple different values concurrently.\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.\n\nIn its simplest form you can create a ScopedValue with a default value and then use with or @with to enter a new dynamic scope. The new scope will inherit all values from the parent scope (and recursively from all outer scopes) with the provided scoped value taking priority over previous definitions.\n\nLet's first look at an example of lexical scope. A let statement begins a new lexical scope within which the outer definition of x is shadowed by its inner definition.\n\njulia> x = 1\n1\n\njulia> let x = 5\n           @show x\n       end;\nx = 5\n\njulia> @show x;\nx = 1\n\nIn the following example, since Julia uses lexical scope, the variable x in the body of f refers to the x defined in the global scope, and entering a let scope does not change the value f observes.\n\njulia> x = 1\n1\n\njulia> f() = @show x\nf (generic function with 1 method)\n\njulia> let x = 5\n           f()\n       end;\nx = 1\n\njulia> f();\nx = 1\n\nNow using a ScopedValue we can use dynamic scoping.\n\njulia> using Base.ScopedValues\n\njulia> x = ScopedValue(1)\nScopedValue{Int64}(1)\n\njulia> f() = @show x[]\nf (generic function with 1 method)\n\njulia> with(x=>5) do\n           f()\n       end;\nx[] = 5\n\njulia> f();\nx[] = 1\n\nNote that the observed value of the ScopedValue is dependent on the execution path of the program.\n\nIt often makes sense to use a const variable to point to a scoped value, and you can set the value of multiple ScopedValues with one call to with.\n\nusing Base.ScopedValues\n\nf() = @show a[]\ng() = @show b[]\n\nconst a = ScopedValue(1)\nconst b = ScopedValue(2)\n\nf() # a[] = 1\ng() # b[] = 2\n\n# Enter a new dynamic scope and set value.\nwith(a => 3) do\n    f() # a[] = 3\n    g() # b[] = 2\n    with(a => 4, b => 5) do\n        f() # a[] = 4\n        g() # b[] = 5\n    end\n    f() # a[] = 3\n    g() # b[] = 2\nend\n\nf() # a[] = 1\ng() # b[] = 2\n\nScopedValues provides a macro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. @with var=>val expr is equivalent to with(var=>val) do expr end. However, with requires a zero-argument closure or function, which results in an extra call-frame. As an example, consider the following function f:\n\nusing Base.ScopedValues\nconst a = ScopedValue(1)\nf(x) = a[] + x\n\nIf you wish to run f in a dynamic scope with a set to 2, then you can use with:\n\nwith(() -> f(10), a=>2)\n\nHowever, this requires wrapping f in a zero-argument function. If you wish to avoid the extra call-frame, then you can use the @with macro:\n\n@with a=>2 f(10)\n\nnote: Note\nDynamic scopes are inherited by Tasks, at the moment of task creation. Dynamic scopes are not propagated through Distributed.jl operations.\n\nIn the example below we open a new dynamic scope before launching a task. The parent task and the two child tasks observe independent values of the same scoped value at the same time.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst scoped_val = ScopedValue(1)\n@sync begin\n    with(scoped_val => 2)\n        @spawn @show scoped_val[] # 2\n    end\n    with(scoped_val => 3)\n        @spawn @show scoped_val[] # 3\n    end\n    @show scoped_val[] # 1\nend\n\nScoped values are constant throughout a scope, but you can store mutable state in a scoped value. Just keep in mind that the usual caveats for global variables apply in the context of concurrent programming.\n\nCare is also required when storing references to mutable state in scoped values. You might want to explicitly unshare mutable state when entering a new dynamic scope.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# Example of using a mutable value wrongly\n@sync begin\n    # `Dict` is not thread-safe the usage below is invalid\n    @spawn (sval_dict[][:a] = 3)\n    @spawn (sval_dict[][:b] = 3)\nend\n\n@sync begin\n    # If we instead pass a unique dictionary to each\n    # task we can access the dictionaries race free.\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:a] = 3)\n    end\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:b] = 3)\n    end\nend"},{"location":"base/scopedvalues.html#Example","page":"Scoped Values","title":"Example","category":"section","text":"In the example below we use a scoped value to implement a permission check in a web-application. After determining the permissions of the request, a new dynamic scope is entered and the scoped value LEVEL is set. Other parts of the application can query the scoped value and will receive the appropriate value. Other alternatives like task-local storage and global variables are not well suited for this kind of propagation; our only alternative would have been to thread a value through the entire call-chain.\n\nusing Base.ScopedValues\n\nconst LEVEL = ScopedValue(:GUEST)\n\nfunction serve(request, response)\n    level = isAdmin(request) ? :ADMIN : :GUEST\n    with(LEVEL => level) do\n        Threads.@spawn handle(request, response)\n    end\nend\n\nfunction open(connection::Database)\n    level = LEVEL[]\n    if level !== :ADMIN\n        error(\"Access disallowed\")\n    end\n    # ... open connection\nend\n\nfunction handle(request, response)\n    # ...\n    open(Database(#=...=#))\n    # ...\nend"},{"location":"base/scopedvalues.html#Idioms","page":"Scoped Values","title":"Idioms","category":"section","text":""},{"location":"base/scopedvalues.html#unshare_mutable_state","page":"Scoped Values","title":"Unshare mutable state","category":"section","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# If you want to add new values to the dict, instead of replacing\n# it, unshare the values explicitly. In this example we use `merge`\n# to unshare the state of the dictionary in parent scope.\n@sync begin\n    with(sval_dict => merge(sval_dict[], Dict(:a => 10))) do\n        @spawn @show sval_dict[][:a]\n    end\n    @spawn sval_dict[][:a] = 3 # Not a race since they are unshared.\nend"},{"location":"base/scopedvalues.html#Scoped-values-as-globals","page":"Scoped Values","title":"Scoped values as globals","category":"section","text":"In order to access the value of a scoped value, the scoped value itself has to be in (lexical) scope. This means most often you likely want to use scoped values as constant globals.\n\nusing Base.ScopedValues\nconst sval = ScopedValue(1)\n\nIndeed one can think of scoped values as hidden function arguments.\n\nThis does not preclude their use as non-globals.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nfunction main()\n    role = ScopedValue(:client)\n\n    function launch()\n        #...\n        role[]\n    end\n\n    @with role => :server @spawn launch()\n    launch()\nend\n\nBut it might have been simpler to just directly pass the function argument in these cases."},{"location":"base/scopedvalues.html#Very-many-ScopedValues","page":"Scoped Values","title":"Very many ScopedValues","category":"section","text":"If you find yourself creating many ScopedValue's for one given module, it may be better to use a dedicated struct to hold them.\n\nusing Base.ScopedValues\n\nBase.@kwdef struct Configuration\n    color::Bool = false\n    verbose::Bool = false\nend\n\nconst CONFIG = ScopedValue(Configuration(color=true))\n\n@with CONFIG => Configuration(color=CONFIG[].color, verbose=true) begin\n    @show CONFIG[].color # true\n    @show CONFIG[].verbose # true\nend"},{"location":"base/scopedvalues.html#API-docs","page":"Scoped Values","title":"API docs","category":"section","text":""},{"location":"base/scopedvalues.html#Implementation-notes-and-performance","page":"Scoped Values","title":"Implementation notes and performance","category":"section","text":"Scopes use a persistent dictionary. Lookup and insertion is O(log(32, n)), upon dynamic scope entry a small amount of data is copied and the unchanged data is shared among other scopes.\n\nThe Scope object itself is not user-facing and may be changed in a future version of Julia."},{"location":"base/scopedvalues.html#Design-inspiration","page":"Scoped Values","title":"Design inspiration","category":"section","text":"This design was heavily inspired by JEPS-429, which in turn was inspired by dynamically scoped free variables in many Lisp dialects. In particular Interlisp-D and its deep binding strategy.\n\nA prior design discussed was context variables ala PEPS-567 and implemented in Julia as ContextVariablesX.jl."},{"location":"base/scopedvalues.html#Base.ScopedValues.ScopedValue","page":"Scoped Values","title":"Base.ScopedValues.ScopedValue","category":"type","text":"ScopedValue(x)\n\nCreate a container that propagates values across dynamic scopes. Use with to create and enter a new dynamic scope.\n\nValues can only be set when entering a new dynamic scope, and the value referred to will be constant during the execution of a dynamic scope.\n\nDynamic scopes are propagated across tasks.\n\nExamples\n\njulia> using Base.ScopedValues;\n\njulia> const sval = ScopedValue(1);\n\njulia> sval[]\n1\n\njulia> with(sval => 2) do\n           sval[]\n       end\n2\n\njulia> sval[]\n1\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.with","page":"Scoped Values","title":"Base.ScopedValues.with","category":"function","text":"with(f, (var::ScopedValue{T} => val)...)\n\nExecute f in a new dynamic scope with var set to val. val will be converted to type T.\n\nSee also: ScopedValues.@with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> f(10)\n11\n\njulia> with(a=>2) do\n           f(10)\n       end\n12\n\njulia> f(10)\n11\n\njulia> b = ScopedValue(2);\n\njulia> g(x) = a[] + b[] + x;\n\njulia> with(a=>10, b=>20) do\n           g(30)\n       end\n60\n\njulia> with(() -> a[] * b[], a=>3, b=>4)\n12\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.@with","page":"Scoped Values","title":"Base.ScopedValues.@with","category":"macro","text":"@with (var::ScopedValue{T} => val)... expr\n\nMacro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. val will be converted to type T. @with var=>val expr is equivalent to with(var=>val) do expr end, but @with avoids creating a closure.\n\nSee also: ScopedValues.with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> const a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> @with a=>2 f(10)\n12\n\njulia> @with a=>3 begin\n           x = 100\n           f(x)\n       end\n103\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.isassigned-Tuple{Base.ScopedValues.ScopedValue}","page":"Scoped Values","title":"Base.isassigned","category":"method","text":"isassigned(val::ScopedValue)\n\nTest whether a ScopedValue has an assigned value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1); b = ScopedValue{Int}();\n\njulia> isassigned(a)\ntrue\n\njulia> isassigned(b)\nfalse\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.get","page":"Scoped Values","title":"Base.ScopedValues.get","category":"function","text":"get(val::ScopedValue{T})::Union{Nothing, Some{T}}\nget(val::LazyScopedValue{T})::Union{Nothing, Some{T}}\n\nIf the scoped value isn't set and doesn't have a default value, return nothing. Otherwise returns Some{T} with the current value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.ScopedValue.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(42); b = ScopedValue{Int}();\n\njulia> ScopedValues.get(a)\nSome(42)\n\njulia> isnothing(ScopedValues.get(b))\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#man-logging","page":"Logging","title":"Logging","category":"section","text":"The Logging module provides a way to record the history and progress of a computation as a log of events. Events are created by inserting a logging statement into the source code, for example:\n\n@warn \"Abandon printf debugging, all ye who enter here!\"\n┌ Warning: Abandon printf debugging, all ye who enter here!\n└ @ Main REPL[1]:1\n\nThe system provides several advantages over peppering your source code with calls to println(). First, it allows you to control the visibility and presentation of messages without editing the source code. For example, in contrast to the @warn above\n\n@debug \"The sum of some values $(sum(rand(100)))\"\n\nwill produce no output by default. Furthermore, it's very cheap to leave debug statements like this in the source code because the system avoids evaluating the message if it would later be ignored. In this case sum(rand(100)) and the associated string processing will never be executed unless debug logging is enabled.\n\nSecond, the logging tools allow you to attach arbitrary data to each event as a set of key–value pairs. This allows you to capture local variables and other program state for later analysis. For example, to attach the local array variable A and the sum of a vector v as the key s you can use\n\nA = ones(Int, 4, 4)\nv = ones(100)\n@info \"Some variables\"  A  s=sum(v)\n\n# output\n┌ Info: Some variables\n│   A =\n│    4×4 Matrix{Int64}:\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n│     1  1  1  1\n└   s = 100.0\n\nAll of the logging macros @debug, @info, @warn and @error share common features that are described in detail in the documentation for the more general macro @logmsg."},{"location":"stdlib/Logging.html#Log-event-structure","page":"Logging","title":"Log event structure","category":"section","text":"Each event generates several pieces of data, some provided by the user and some automatically extracted. Let's examine the user-defined data first:\n\nThe log level is a broad category for the message that is used for early filtering. There are several standard levels of type LogLevel; user-defined levels are also possible. Each is distinct in purpose:\nLogging.Debug (log level -1000) is information intended for the developer of the program. These events are disabled by default.\nLogging.Info (log level 0) is for general information to the user. Think of it as an alternative to using println directly.\nLogging.Warn (log level 1000) means something is wrong and action is likely required but that for now the program is still working.\nLogging.Error (log level 2000) means something is wrong and it is unlikely to be recovered, at least by this part of the code. Often this log-level is unneeded as throwing an exception can convey all the required information.\nThe message  is an object describing the event. By convention AbstractStrings passed as messages are assumed to be in markdown format. Other types will be displayed using print(io, obj) or string(obj) for text-based output and possibly show(io,mime,obj) for other multimedia displays used in the installed logger.\nOptional key–value pairs allow arbitrary data to be attached to each event. Some keys have conventional meaning that can affect the way an event is interpreted (see @logmsg).\n\nThe system also generates some standard information for each event:\n\nThe module in which the logging macro was expanded.\nThe file and line where the logging macro occurs in the source code.\nA message id that is a unique, fixed identifier for the source code statement where the logging macro appears. This identifier is designed to be fairly stable even if the source code of the file changes, as long as the logging statement itself remains the same.\nA group for the event, which is set to the base name of the file by default, without extension. This can be used to group messages into categories more finely than the log level (for example, all deprecation warnings have group :depwarn), or into logical groupings across or within modules.\n\nNotice that some useful information such as the event time is not included by default. This is because such information can be expensive to extract and is also dynamically available to the current logger. It's simple to define a custom logger to augment event data with the time, backtrace, values of global variables and other useful information as required."},{"location":"stdlib/Logging.html#Processing-log-events","page":"Logging","title":"Processing log events","category":"section","text":"As you can see in the examples, logging statements make no mention of where log events go or how they are processed. This is a key design feature that makes the system composable and natural for concurrent use. It does this by separating two different concerns:\n\nCreating log events is the concern of the module author who needs to decide where events are triggered and which information to include.\nProcessing of log events — that is, display, filtering, aggregation and recording — is the concern of the application author who needs to bring multiple modules together into a cooperating application."},{"location":"stdlib/Logging.html#Loggers","page":"Logging","title":"Loggers","category":"section","text":"Processing of events is performed by a logger, which is the first piece of user configurable code to see the event. All loggers must be subtypes of AbstractLogger.\n\nWhen an event is triggered, the appropriate logger is found by looking for a task-local logger with the global logger as fallback. The idea here is that the application code knows how log events should be processed and exists somewhere at the top of the call stack. So we should look up through the call stack to discover the logger — that is, the logger should be dynamically scoped. (This is a point of contrast with logging frameworks where the logger is lexically scoped; provided explicitly by the module author or as a simple global variable. In such a system it's awkward to control logging while composing functionality from multiple modules.)\n\nThe global logger may be set with global_logger, and task-local loggers controlled using with_logger. Newly spawned tasks inherit the logger of the parent task.\n\nThere are three logger types provided by the library.  ConsoleLogger is the default logger you see when starting the REPL. It displays events in a readable text format and tries to give simple but user friendly control over formatting and filtering.  NullLogger is a convenient way to drop all messages where necessary; it is the logging equivalent of the devnull stream.  SimpleLogger is a very simplistic text formatting logger, mainly useful for debugging the logging system itself.\n\nCustom loggers should come with overloads for the functions described in the reference section."},{"location":"stdlib/Logging.html#Early-filtering-and-message-handling","page":"Logging","title":"Early filtering and message handling","category":"section","text":"When an event occurs, a few steps of early filtering occur to avoid generating messages that will be discarded:\n\nThe message log level is checked against a global minimum level (set via disable_logging). This is a crude but extremely cheap global setting.\nThe current logger state is looked up and the message level checked against the logger's cached minimum level, as found by calling Logging.min_enabled_level. This behavior can be overridden via environment variables (more on this later).\nThe Logging.shouldlog function is called with the current logger, taking some minimal information (level, module, group, id) which can be computed statically. Most usefully, shouldlog is passed an event id which can be used to discard events early based on a cached predicate.\n\nIf all these checks pass, the message and key–value pairs are evaluated in full and passed to the current logger via the Logging.handle_message function. handle_message() may perform additional filtering as required and display the event to the screen, save it to a file, etc.\n\nExceptions that occur while generating the log event are captured and logged by default. This prevents individual broken events from crashing the application, which is helpful when enabling little-used debug events in a production system. This behavior can be customized per logger type by extending Logging.catch_exceptions."},{"location":"stdlib/Logging.html#Testing-log-events","page":"Logging","title":"Testing log events","category":"section","text":"Log events are a side effect of running normal code, but you might find yourself wanting to test particular informational messages and warnings. The Test module provides a @test_logs macro that can be used to pattern match against the log event stream."},{"location":"stdlib/Logging.html#Environment-variables","page":"Logging","title":"Environment variables","category":"section","text":"Message filtering can be influenced through the JULIA_DEBUG environment variable, and serves as an easy way to enable debug logging for a file or module. Loading julia with JULIA_DEBUG=loading will activate @debug log messages in loading.jl. For example, in Linux shells:\n\n$ JULIA_DEBUG=loading julia -e 'using OhMyREPL'\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n[ Info: Recompiling stale cache file /home/user/.julia/compiled/v0.7/OhMyREPL.ji for module OhMyREPL\n┌ Debug: Rejecting cache file /home/user/.julia/compiled/v0.7/Tokenize.ji due to it containing an incompatible cache header\n└ @ Base loading.jl:1328\n...\n\nOn windows, the same can be achieved in CMD via first running set JULIA_DEBUG=\"loading\" and in Powershell via $env:JULIA_DEBUG=\"loading\".\n\nSimilarly, the environment variable can be used to enable debug logging of modules, such as Pkg, or module roots (see Base.moduleroot). To enable all debug logging, use the special value all.\n\nTo turn debug logging on from the REPL, set ENV[\"JULIA_DEBUG\"] to the name of the module of interest. Functions defined in the REPL belong to module Main; logging for them can be enabled like this:\n\njulia> foo() = @debug \"foo\"\nfoo (generic function with 1 method)\n\njulia> foo()\n\njulia> ENV[\"JULIA_DEBUG\"] = Main\nMain\n\njulia> foo()\n┌ Debug: foo\n└ @ Main REPL[1]:1\n\n\nUse a comma separator to enable debug for multiple modules: JULIA_DEBUG=loading,Main."},{"location":"stdlib/Logging.html#Examples","page":"Logging","title":"Examples","category":"section","text":""},{"location":"stdlib/Logging.html#Example:-Writing-log-events-to-a-file","page":"Logging","title":"Example: Writing log events to a file","category":"section","text":"Sometimes it can be useful to write log events to a file. Here is an example of how to use a task-local and global logger to write information to a text file:\n\n# Load the logging module\njulia> using Logging\n\n# Open a textfile for writing\njulia> io = open(\"log.txt\", \"w+\")\nIOStream(<file log.txt>)\n\n# Create a simple logger\njulia> logger = SimpleLogger(io)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# Log a task-specific message\njulia> with_logger(logger) do\n           @info(\"a context specific log message\")\n       end\n\n# Write all buffered messages to the file\njulia> flush(io)\n\n# Set the global logger to logger\njulia> global_logger(logger)\nSimpleLogger(IOStream(<file log.txt>), Info, Dict{Any,Int64}())\n\n# This message will now also be written to the file\njulia> @info(\"a global log message\")\n\n# Close the file\njulia> close(io)"},{"location":"stdlib/Logging.html#Example:-Enable-debug-level-messages","page":"Logging","title":"Example: Enable debug-level messages","category":"section","text":"Here is an example of creating a ConsoleLogger that lets through any messages with log level higher than, or equal, to Logging.Debug.\n\njulia> using Logging\n\n# Create a ConsoleLogger that prints any log messages with level >= Debug to stderr\njulia> debuglogger = ConsoleLogger(stderr, Logging.Debug)\n\n# Enable debuglogger for a task\njulia> with_logger(debuglogger) do\n           @debug \"a context specific log message\"\n       end\n\n# Set the global logger\njulia> global_logger(debuglogger)"},{"location":"stdlib/Logging.html#Reference","page":"Logging","title":"Reference","category":"section","text":""},{"location":"stdlib/Logging.html#Logging-module","page":"Logging","title":"Logging module","category":"section","text":""},{"location":"stdlib/Logging.html#Creating-events","page":"Logging","title":"Creating events","category":"section","text":""},{"location":"stdlib/Logging.html#AbstractLogger-interface","page":"Logging","title":"Processing events with AbstractLogger","category":"section","text":"Event processing is controlled by overriding functions associated with AbstractLogger:\n\nMethods to implement  Brief description\nLogging.handle_message  Handle a log event\nLogging.shouldlog  Early filtering of events\nLogging.min_enabled_level  Lower bound for log level of accepted events\nOptional methods Default definition Brief description\nLogging.catch_exceptions true Catch exceptions during event evaluation"},{"location":"stdlib/Logging.html#Using-Loggers","page":"Logging","title":"Using Loggers","category":"section","text":"Logger installation and inspection:\n\nLoggers that are supplied with the system:"},{"location":"stdlib/Logging.html#Logging.Logging","page":"Logging","title":"Logging.Logging","category":"module","text":"Utilities for capturing, filtering and presenting streams of log events. Normally you don't need to import Logging to create log events; for this the standard logging macros such as @info are already exported by Base and available by default.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.@logmsg","page":"Logging","title":"Base.CoreLogging.@logmsg","category":"macro","text":"@debug message  [key=value | value ...]\n@info  message  [key=value | value ...]\n@warn  message  [key=value | value ...]\n@error message  [key=value | value ...]\n\n@logmsg level message [key=value | value ...]\n\nCreate a log record with an informational message.  For convenience, four logging macros @debug, @info, @warn and @error are defined which log at the standard severity levels Debug, Info, Warn and Error.  @logmsg allows level to be set programmatically to any LogLevel or custom log level types.\n\nmessage should be an expression which evaluates to a string which is a human readable description of the log event.  By convention, this string will be formatted as markdown when presented.\n\nThe optional list of key=value pairs supports arbitrary user defined metadata which will be passed through to the logging backend as part of the log record.  If only a value expression is supplied, a key representing the expression will be generated using Symbol. For example, x becomes x=x, and foo(10) becomes Symbol(\"foo(10)\")=foo(10).  For splatting a list of key value pairs, use the normal splatting syntax, @info \"blah\" kws....\n\nThere are some keys which allow automatically generated log data to be overridden:\n\n_module=mod can be used to specify a different originating module from the source location of the message.\n_group=symbol can be used to override the message group (this is normally derived from the base name of the source file).\n_id=symbol can be used to override the automatically generated unique message identifier.  This is useful if you need to very closely associate messages generated on different source lines.\n_file=string and _line=integer can be used to override the apparent source location of a log message.\n\nThere's also some key value pairs which have conventional meaning:\n\nmaxlog=integer should be used as a hint to the backend that the message should be displayed no more than maxlog times.\nexception=ex should be used to transport an exception with a log message, often used with @error. An associated backtrace bt may be attached using the tuple exception=(ex,bt).\n\nExamples\n\n@debug \"Verbose debugging information.  Invisible by default\"\n@info  \"An informational message\"\n@warn  \"Something was odd.  You should pay attention\"\n@error \"A non fatal error occurred\"\n\nx = 10\n@info \"Some variables attached to the message\" x a=42.0\n\n@debug begin\n    sA = sum(A)\n    \"sum(A) = $sA is an expensive operation, evaluated only when `shouldlog` returns true\"\nend\n\nfor i=1:10000\n    @info \"With the default backend, you will only see (i = $i) ten times\"  maxlog=10\n    @debug \"Algorithm1\" i progress=i/10000\nend\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.LogLevel","page":"Logging","title":"Base.CoreLogging.LogLevel","category":"type","text":"LogLevel(level)\n\nSeverity/verbosity of a log record.\n\nThe log level provides a key against which potential log records may be filtered, before any other work is done to construct the log record data structure itself.\n\nExamples\n\njulia> Logging.LogLevel(0) == Logging.Info\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.Debug","page":"Logging","title":"Base.CoreLogging.Debug","category":"constant","text":"Debug\n\nAlias for LogLevel(-1000).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.Info","page":"Logging","title":"Base.CoreLogging.Info","category":"constant","text":"Info\n\nAlias for LogLevel(0).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.Warn","page":"Logging","title":"Base.CoreLogging.Warn","category":"constant","text":"Warn\n\nAlias for LogLevel(1000).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.Error","page":"Logging","title":"Base.CoreLogging.Error","category":"constant","text":"Error\n\nAlias for LogLevel(2000).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.BelowMinLevel","page":"Logging","title":"Base.CoreLogging.BelowMinLevel","category":"constant","text":"BelowMinLevel\n\nAlias for LogLevel(-1_000_001).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.AboveMaxLevel","page":"Logging","title":"Base.CoreLogging.AboveMaxLevel","category":"constant","text":"AboveMaxLevel\n\nAlias for LogLevel(1_000_001).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.AbstractLogger","page":"Logging","title":"Base.CoreLogging.AbstractLogger","category":"type","text":"A logger controls how log records are filtered and dispatched.  When a log record is generated, the logger is the first piece of user configurable code which gets to inspect the record and decide what to do with it.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.handle_message","page":"Logging","title":"Base.CoreLogging.handle_message","category":"function","text":"handle_message(logger, level, message, _module, group, id, file, line; key1=val1, ...)\n\nLog a message to logger at level.  The logical location at which the message was generated is given by module _module and group; the source location by file and line. id is an arbitrary unique value (typically a Symbol) to be used as a key to identify the log statement when filtering.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.shouldlog","page":"Logging","title":"Base.CoreLogging.shouldlog","category":"function","text":"shouldlog(logger, level, _module, group, id)\n\nReturn true when logger accepts a message at level, generated for _module, group and with unique log identifier id.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.min_enabled_level","page":"Logging","title":"Base.CoreLogging.min_enabled_level","category":"function","text":"min_enabled_level(logger)\n\nReturn the minimum enabled level for logger for early filtering.  That is, the log level below or equal to which all messages are filtered.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.catch_exceptions","page":"Logging","title":"Base.CoreLogging.catch_exceptions","category":"function","text":"catch_exceptions(logger)\n\nReturn true if the logger should catch exceptions which happen during log record construction.  By default, messages are caught.\n\nBy default all exceptions are caught to prevent log message generation from crashing the program.  This lets users confidently toggle little-used functionality - such as debug logging - in a production system.\n\nIf you want to use logging as an audit trail you should disable this for your logger type.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.disable_logging","page":"Logging","title":"Base.CoreLogging.disable_logging","category":"function","text":"disable_logging(level)\n\nDisable all log messages at log levels equal to or less than level.  This is a global setting, intended to make debug logging extremely cheap when disabled. Note that this cannot be used to enable logging that is currently disabled by other mechanisms.\n\nExamples\n\nLogging.disable_logging(Logging.Info) # Disable debug and info\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.global_logger","page":"Logging","title":"Base.CoreLogging.global_logger","category":"function","text":"global_logger()\n\nReturn the global logger, used to receive messages when no specific logger exists for the current task.\n\nglobal_logger(logger)\n\nSet the global logger to logger, and return the previous global logger.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.with_logger","page":"Logging","title":"Base.CoreLogging.with_logger","category":"function","text":"with_logger(function, logger)\n\nExecute function, directing all log messages to logger.\n\nExamples\n\nfunction test(x)\n    @info \"x = $x\"\nend\n\nwith_logger(logger) do\n    test(1)\n    test([1,2])\nend\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.current_logger","page":"Logging","title":"Base.CoreLogging.current_logger","category":"function","text":"current_logger()\n\nReturn the logger for the current task, or the global logger if none is attached to the task.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.NullLogger","page":"Logging","title":"Base.CoreLogging.NullLogger","category":"type","text":"NullLogger()\n\nLogger which disables all messages and produces no output - the logger equivalent of /dev/null.\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.ConsoleLogger","page":"Logging","title":"Base.CoreLogging.ConsoleLogger","category":"type","text":"ConsoleLogger([stream,] min_level=Info; meta_formatter=default_metafmt,\n              show_limited=true, right_justify=0)\n\nLogger with formatting optimized for readability in a text console, for example interactive work with the Julia REPL.\n\nLog levels less than min_level are filtered out.\n\nThis Logger is thread-safe, with locks for both orchestration of message limits i.e. maxlog, and writes to the stream.\n\nMessage formatting can be controlled by setting keyword arguments:\n\nmeta_formatter is a function which takes the log event metadata (level, _module, group, id, file, line) and returns a color (as would be passed to printstyled), prefix and suffix for the log message.  The default is to prefix with the log level and a suffix containing the module, file and line location.\nshow_limited limits the printing of large data structures to something which can fit on the screen by setting the :limit IOContext key during formatting.\nright_justify is the integer column which log metadata is right justified at. The default is zero (metadata goes on its own line).\n\n\n\n\n\n"},{"location":"stdlib/Logging.html#Base.CoreLogging.SimpleLogger","page":"Logging","title":"Base.CoreLogging.SimpleLogger","category":"type","text":"SimpleLogger([stream,] min_level=Info)\n\nSimplistic logger for logging all messages with level greater than or equal to min_level to stream. If stream is closed then messages with log level greater or equal to Warn will be logged to stderr and below to stdout.\n\nThis Logger is thread-safe, with a lock taken around orchestration of message limits i.e. maxlog, and writes to the stream.\n\n\n\n\n\n"},{"location":"manual/distributed-computing.html#Multi-processing-and-Distributed-Computing","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","category":"section","text":"An implementation of distributed memory parallel computing is provided by module Distributed as part of the standard library shipped with Julia.\n\nMost modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.\n\nJulia's implementation of message passing is different from other environments such as MPI[1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.\n\nDistributed programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.\n\nRemote references come in two flavors: Future and RemoteChannel.\n\nA remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait on the returned Future, and you can obtain the full value of the result using fetch.\n\nOn the other hand, RemoteChannel s are rewritable. For example, multiple processes can coordinate their processing by referencing the same remote Channel.\n\nEach process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1. As a result, adding 2 or more processes is required to gain benefits from parallel processing methods like pmap. Adding a single process is beneficial if you just wish to do other things in the main process while a long computation is running on the worker.\n\nLet's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU threads (logical cores) on the machine. Note that the -p argument implicitly loads module Distributed.\n\n$ julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.18526  1.50912\n 1.16296  1.60607\n\nThe first argument to remotecall is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall is considered a low-level interface providing finer control. The second argument to remotecall is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.\n\nAs you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.\n\nOccasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.\n\njulia> remotecall_fetch(r-> fetch(r)[1, 1], 2, r)\n0.18526337335308085\n\nThis fetches the array on worker 2 and returns the first value. Note, that fetch doesn't move any data in this case, since it's executed on the worker that owns the array. One can also write:\n\njulia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.10824216411304866\n\nRemember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.\n\nTo make things easier, the symbol :any can be passed to @spawnat, which picks where to do the operation for you:\n\njulia> r = @spawnat :any rand(2,2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat :any 1 .+ fetch(r)\nFuture(3, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.38854  1.9098\n 1.20939  1.57158\n\nNote that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch might be required to move r to the process doing the addition. In this case, @spawnat is smart enough to perform the computation on the process that owns r, so the fetch will be a no-op (no work is done).\n\n(It is worth noting that @spawnat is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)\n\nAn important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted.\n\nThreads.@spawn is similar to @spawnat, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indices. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. As for v0.7 and beyond, the feeder tasks are able to share state via nextidx because they all run on the same process. Even if Tasks are scheduled cooperatively, locking may still be required in some contexts, as in asynchronous I/O. This means context switches only occur at well-defined points: in this case, when remotecall_fetch is called. This is the current state of implementation and it may change for future Julia versions, as it is intended to make it possible to run up to N Tasks on M Process, aka M:N Threading. Then a lock acquiring\\releasing model for nextidx will be needed, as it is not safe to let multiple processes read-write a resource at the same time."},{"location":"manual/distributed-computing.html#code-availability","page":"Multi-processing and Distributed Computing","title":"Code Availability and Loading Packages","category":"section","text":"Your code must be available on any process that runs it. For example, type the following into the Julia prompt:\n\njulia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Matrix{Float64}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawnat :any rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))))\nStacktrace:\n[...]\n\nProcess 1 knew about the function rand2, but process 2 did not.\n\nMost commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:\n\nmodule DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nend\n\nIn order to refer to MyType across all processes, DummyModule.jl needs to be loaded on every process. Calling include(\"DummyModule.jl\") loads it only on a single process. To load it on every process, use the @everywhere macro (starting Julia with julia -p 2):\n\njulia> @everywhere include(\"DummyModule.jl\")\nloaded\n      From worker 3:    loaded\n      From worker 2:    loaded\n\nAs usual, this does not bring DummyModule into scope on any of the processes, which requires using or import. Moreover, when DummyModule is brought into scope on one process, it is not on any other:\n\njulia> using .DummyModule\n\njulia> MyType(7)\nMyType(7)\n\njulia> fetch(@spawnat 2 MyType(7))\nERROR: On worker 2:\nUndefVarError: `MyType` not defined in `Main`\n⋮\n\njulia> fetch(@spawnat 2 DummyModule.MyType(7))\nMyType(7)\n\nHowever, it's still possible, for instance, to send a MyType to a process which has loaded DummyModule even if it's not in scope:\n\njulia> put!(RemoteChannel(2), MyType(7))\nRemoteChannel{Channel{Any}}(2, 1, 13)\n\nA file can also be preloaded on multiple processes at startup with the -L flag, and a driver script can be used to drive the computation:\n\njulia -p <n> -L file1.jl -L file2.jl driver.jl\n\nThe Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.\n\nFinally, if DummyModule.jl is not a standalone file but a package, then using DummyModule will load DummyModule.jl on all processes, but only bring it into scope on the process where using was called."},{"location":"manual/distributed-computing.html#Starting-and-managing-worker-processes","page":"Multi-processing and Distributed Computing","title":"Starting and managing worker processes","category":"section","text":"The base Julia installation has in-built support for two types of clusters:\n\nA local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machine-file option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers should use to connect to this worker.\n\nnote: Note\nWhile Julia generally strives for backward compatibility, distribution of code to worker processes relies on Serialization.serialize. As pointed out in the corresponding documentation, this can not be guaranteed to work across different Julia versions, so it is advised that all workers on all machines use the same version.\n\nFunctions addprocs, rmprocs, workers, and others are available as a programmatic means of adding, removing and querying the processes in a cluster.\n\njulia> using Distributed\n\njulia> addprocs(2)\n2-element Vector{Int64}:\n 2\n 3\n\nModule Distributed must be explicitly loaded on the master process before invoking addprocs. It is automatically made available on the worker processes.\n\nnote: Note\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes. You may use addprocs(exeflags=\"--project\") to initialize a worker with a particular environment, and then @everywhere using <modulename> or @everywhere include(\"file.jl\").\n\nOther types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section."},{"location":"manual/distributed-computing.html#Data-Movement","page":"Multi-processing and Distributed Computing","title":"Data Movement","category":"section","text":"Sending messages and moving data constitute most of the overhead in a distributed program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various distributed programming constructs.\n\nfetch can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawnat (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:\n\nMethod 1:\n\njulia> A = rand(1000,1000);\n\njulia> Bref = @spawnat :any A^2;\n\n[...]\n\njulia> fetch(Bref);\n\nMethod 2:\n\njulia> Bref = @spawnat :any rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);\n\nThe difference seems trivial, but in fact is quite significant due to the behavior of @spawnat. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.\n\nIn this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawnat and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawnat statement just for this step."},{"location":"manual/distributed-computing.html#Global-variables","page":"Multi-processing and Distributed Computing","title":"Global variables","category":"section","text":"Expressions executed remotely via @spawnat, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:\n\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2)\n\nIn this case sum MUST be defined in the remote process. Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->sum(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:\n\nNew global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.\n\nAs you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.\n\nThus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.\n\nFor example:\n\njulia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @fetchfrom 2 InteractiveUtils.varinfo()\nname           size summary\n––––––––– ––––––––– ––––––––––––––––––––––\nA         800 bytes 10×10 Array{Float64,2}\nBase                Module\nCore                Module\nMain                Module\n\nAs can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2."},{"location":"manual/distributed-computing.html#Parallel-Map-and-Loops","page":"Multi-processing and Distributed Computing","title":"Parallel Map and Loops","category":"section","text":"Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawnat to flip coins on two processes. First, write the following function in count_heads.jl:\n\nfunction count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nend\n\nThe function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:\n\njulia> @everywhere include_string(Main, $(read(\"count_heads.jl\", String)), \"count_heads.jl\")\n\njulia> a = @spawnat :any count_heads(100000000)\nFuture(2, 1, 6, nothing)\n\njulia> b = @spawnat :any count_heads(100000000)\nFuture(3, 1, 7, nothing)\n\njulia> fetch(a)+fetch(b)\n100001564\n\nThis example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.\n\nNotice that our use of this pattern with count_heads can be generalized. We used two explicit @spawnat statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, running in distributed memory, which can be written in Julia using @distributed like this:\n\nnheads = @distributed (+) for i = 1:200000000\n    Int(rand(Bool))\nend\n\nThis construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.\n\nNote that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.\n\nFor example, the following code will not work as intended:\n\na = zeros(100000)\n@distributed for i = 1:100000\n    a[i] = i\nend\n\nThis code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:\n\nusing SharedArrays\n\na = SharedArray{Float64}(10)\n@distributed for i = 1:10\n    a[i] = i\nend\n\nUsing \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:\n\na = randn(1000)\n@distributed (+) for i = 1:100000\n    f(a[rand(1:end)])\nend\n\nHere each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.\n\nAs you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @distributed for.\n\nIn some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap function. For example, we could compute the singular values of several large random matrices in parallel as follows:\n\njulia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svdvals, M);\n\nJulia's pmap is designed for the case where each function call does a large amount of work. In contrast, @distributed for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap and @distributed for for the parallel computation. In case of @distributed for, the final reduction is done on the calling process."},{"location":"manual/distributed-computing.html#Remote-References-and-AbstractChannels","page":"Multi-processing and Distributed Computing","title":"Remote References and AbstractChannels","category":"section","text":"Remote references always refer to an implementation of an AbstractChannel.\n\nA concrete implementation of an AbstractChannel (like Channel), is required to implement put!, take!, fetch, isready and wait. The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.\n\nRemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.\n\nThe constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f is a function executed on pid and it must return an AbstractChannel.\n\nFor example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.\n\nMethods put!, take!, fetch, isready and wait on a RemoteChannel are proxied onto the backing store on the remote process.\n\nRemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is the following DictChannel which uses a dictionary as its remote store:\n\njulia> struct DictChannel{T} <: AbstractChannel{T}\n           d::Dict\n           cond_take::Threads.Condition    # waiting for data to become available\n           DictChannel{T}() where {T} = new(Dict(), Threads.Condition())\n           DictChannel() = DictChannel{Any}()\n       end\n\njulia> begin\n       function Base.put!(D::DictChannel, k, v)\n           @lock D.cond_take begin\n               D.d[k] = v\n               notify(D.cond_take)\n           end\n           return D\n       end\n       function Base.take!(D::DictChannel, k)\n           @lock D.cond_take begin\n               v = fetch(D, k)\n               delete!(D.d, k)\n               return v\n           end\n       end\n       Base.isready(D::DictChannel) = @lock D.cond_take !isempty(D.d)\n       Base.isready(D::DictChannel, k) = @lock D.cond_take haskey(D.d, k)\n       function Base.fetch(D::DictChannel, k)\n           @lock D.cond_take begin\n               wait(D, k)\n               return D.d[k]\n           end\n       end\n       function Base.wait(D::DictChannel, k)\n           @lock D.cond_take begin\n               while !isready(D, k)\n                   wait(D.cond_take)\n               end\n           end\n       end\n       end;\n\njulia> d = DictChannel();\n\njulia> isready(d)\nfalse\n\njulia> put!(d, :k, :v);\n\njulia> isready(d, :k)\ntrue\n\njulia> fetch(d, :k)\n:v\n\njulia> wait(d, :k)\n\njulia> take!(d, :k)\n:v\n\njulia> isready(d, :k)\nfalse"},{"location":"manual/distributed-computing.html#Channels-and-RemoteChannels","page":"Multi-processing and Distributed Computing","title":"Channels and RemoteChannels","category":"section","text":"A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.\n\nThe channels example from above can be modified for interprocess communication, as shown below.\n\nWe start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.\n\njulia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(Threads.@spawn make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, worker_id = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds on worker $worker_id\")\n           global n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741"},{"location":"manual/distributed-computing.html#Remote-References-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Remote References and Distributed Garbage Collection","category":"section","text":"Objects referred to by remote references can be freed only when all held references in the cluster are deleted.\n\nThe node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified. This is implemented in an internal cluster aware serializer. Remote references are only valid in the context of a running cluster. Serializing and deserializing references to and from regular IO objects is not supported.\n\nThe notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.\n\nSince Futures are write-once and cached locally, the act of fetching a Future also updates reference tracking information on the node owning the value.\n\nThe node which owns the value frees it once all references to it are cleared.\n\nWith Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.\n\nIt is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.\n\nIn case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize results in an immediate message sent to the remote node to go ahead and remove its reference to the value.\n\nOnce finalized, a reference becomes invalid and cannot be used in any further calls."},{"location":"manual/distributed-computing.html#Local-invocations","page":"Multi-processing and Distributed Computing","title":"Local invocations","category":"section","text":"Data is necessarily copied over to the remote node for execution. This is the case for both remotecalls and when data is stored to a RemoteChannel / Future on a different node. As expected, this results in a copy of the serialized objects on the remote node. However, when the destination node is the local node, i.e. the calling process id is the same as the remote node id, it is executed as a local call. It is usually (not always) executed in a different task - but there is no serialization/deserialization of data. Consequently, the call refers to the same object instances as passed - no copies are created. This behavior is highlighted below:\n\njulia> using Distributed\n\njulia> rc = RemoteChannel(()->Channel(3));   # RemoteChannel created on local node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i                          # Reusing `v`\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[3], [3], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 1\n\njulia> addprocs(1);\n\njulia> rc = RemoteChannel(()->Channel(3), workers()[1]);   # RemoteChannel created on remote node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[1], [2], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 3\n\nAs can be seen, put! on a locally owned RemoteChannel with the same object v modified between calls results in the same single object instance stored. As opposed to copies of v being created when the node owning rc is a different node.\n\nIt is to be noted that this is generally not an issue. It is something to be factored in only if the object is both being stored locally and modified post the call. In such cases it may be appropriate to store a deepcopy of the object.\n\nThis is also true for remotecalls on the local node as seen in the following example:\n\njulia> using Distributed; addprocs(1);\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), myid(), v);     # Executed on local node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[1], v2=[1], true\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), workers()[1], v); # Executed on remote node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[0], v2=[1], false\n\nAs can be seen once again, a remote call onto the local node behaves just like a direct invocation. The call modifies local objects passed as arguments. In the remote invocation, it operates on a copy of the arguments.\n\nTo repeat, in general this is not an issue. If the local node is also being used as a compute node, and the arguments used post the call, this behavior needs to be factored in and if required deep copies of arguments must be passed to the call invoked on the local node. Calls on remote nodes will always operate on copies of arguments."},{"location":"manual/distributed-computing.html#man-shared-arrays","page":"Multi-processing and Distributed Computing","title":"Shared Arrays","category":"section","text":"Shared Arrays use system shared memory to map the same array across many processes. A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine. Shared Array support is available via the module SharedArrays, which must be explicitly loaded on all participating workers.\n\nA complementary data structure is provided by the external package DistributedArrays.jl in the form of a DArray. While there are some similarities to a SharedArray, the behavior of a DArray is quite different. In a SharedArray, each \"participating\" process has access to the entire array; in contrast, in a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk.\n\nSharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata. For other AbstractArray types, sdata just returns the object itself, so it's safe to use sdata on any Array-type object.\n\nThe constructor for a shared array is of the form:\n\nSharedArray{T,N}(dims::NTuple; init=false, pids=Int[])\n\nwhich creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host). Note that only elements that are isbits are supported in a SharedArray.\n\nIf an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.\n\nHere's a brief example:\n\njulia> using Distributed\n\njulia> addprocs(3)\n3-element Vector{Int64}:\n 2\n 3\n 4\n\njulia> @everywhere using SharedArrays\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[localindices(S)] = repeat([myid()], length(localindices(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4\n\nSharedArrays.localindices provides disjoint one-dimensional ranges of indices, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = repeat([myid()], length( indexpids(S):length(procs(S)):length(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4\n\nSince all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:\n\n@sync begin\n    for p in procs(S)\n        Threads.@spawn begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nend\n\nwould result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.\n\nAs a more extended and complex example, consider running the following \"kernel\" in parallel:\n\nq[i,j,t+1] = q[i,j,t] + u[i,j,t]\n\nIn this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indices assigned to this worker:\n\njulia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in range(0, stop=size(q,2), length=nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       end\n\nNext, define the kernel:\n\njulia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       end\n\nWe also define a convenience wrapper for a SharedArray implementation\n\njulia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)\n\nNow let's compare three different versions, one that runs in a single process:\n\njulia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);\n\none that uses @distributed:\n\njulia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @distributed for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;\n\nand one that delegates in chunks:\n\njulia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   Threads.@spawn remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;\n\nIf we create SharedArrays and time these functions, we get the following results (with julia -p 4):\n\njulia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));\n\nRun the functions once to JIT-compile and @time them on the second run:\n\njulia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)\n\nThe biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece."},{"location":"manual/distributed-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Shared Arrays and Distributed Garbage Collection","category":"section","text":"Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner."},{"location":"manual/distributed-computing.html#ClusterManagers","page":"Multi-processing and Distributed Computing","title":"ClusterManagers","category":"section","text":"The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible for\n\nlaunching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transport\n\nA Julia cluster has the following characteristics:\n\nThe initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.\n\nConnections between workers (using the in-built TCP/IP transport) is established in the following manner:\n\naddprocs is called on the master process with a ClusterManager object.\naddprocs calls the appropriate launch method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to stdout.\nThe cluster manager captures the stdout of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.\n\nWhile the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.\n\nJulia provides two in-built cluster managers:\n\nLocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnames\n\nLocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.\n\nThus, a minimal cluster manager would need to:\n\nbe a subtype of the abstract ClusterManager\nimplement launch, a method responsible for launching new workers\nimplement manage, which is called at various events during a worker's lifetime (for example, sending an interrupt signal)\n\naddprocs(manager::FooManager) requires FooManager to implement:\n\nfunction launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nAs an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:\n\nstruct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nThe launch method takes the following arguments:\n\nmanager::ClusterManager: the cluster manager that addprocs is called with\nparams::Dict: all the keyword arguments passed to addprocs\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launched\n\nThe launch method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch function MUST exit as soon as all the requested workers have been launched.\n\nNewly launched workers are connected to each other and the master process in an all-to-all manner. Specifying the command line argument --worker[=<cookie>] results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets.\n\nAll workers in a cluster share the same cookie as the master. When the cookie is unspecified, i.e, with the --worker option, the worker tries to read it from its standard input.  LocalManager and SSHManager both pass the cookie to newly launched workers via their  standard inputs.\n\nBy default a worker will listen on a free port at the address returned by a call to getipaddr(). A specific address to listen on may be specified by optional argument --bind-to bind_addr[:port]. This is useful for multi-homed hosts.\n\nAs an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.\n\nFor every worker launched, the launch method must add a WorkerConfig object (with appropriate fields initialized) to launched\n\nmutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Union{IO, Nothing}\n    host::Union{AbstractString, Nothing}\n    port::Union{Integer, Nothing}\n\n    # Used when launching additional workers at a host\n    count::Union{Int, Symbol, Nothing}\n    exename::Union{AbstractString, Cmd, Nothing}\n    exeflags::Union{Cmd, Nothing}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Any\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Union{Bool, Nothing}\n    bind_addr::Union{AbstractString, Nothing}\n    sshflags::Union{Cmd, Nothing}\n    max_parallel::Union{Integer, Nothing}\n\n    # Used by Local/SSH managers\n    connect_at::Any\n\n    [...]\nend\n\nMost of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:\n\nIf io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of CPU threads (logical cores) on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.\n\nmanage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:\n\nwith :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes."},{"location":"manual/distributed-computing.html#Cluster-Managers-with-Custom-Transports","page":"Multi-processing and Distributed Computing","title":"Cluster Managers with Custom Transports","category":"section","text":"Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:\n\nEach Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.\n\nReplacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:\n\nconnect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)\n\nThe default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).\n\nconnect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.\n\nA BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.\n\nThe folder clustermanager/0mq in the Examples repository contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.\n\nWhen using custom transports:\n\nJulia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) must be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch is called. The value of this field is passed in all connect callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.\n\nkill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.\n\nThe Examples folder clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup."},{"location":"manual/distributed-computing.html#Network-Requirements-for-LocalManager-and-SSHManager","page":"Multi-processing and Distributed Computing","title":"Network Requirements for LocalManager and SSHManager","category":"section","text":"Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:\n\nThe master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on an ephemeral port number assigned by the OS.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-i <keyfile>`.\nIn an all-to-all topology (the default), all workers connect to each other via plain TCP sockets. The security policy on the cluster nodes must thus ensure free connectivity between workers for the ephemeral port range (varies by OS).\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager.\nIf you specify multiplex=true as an option to addprocs, SSH multiplexing is used to create a tunnel between the master and workers. If you have configured SSH multiplexing on your own and the connection has already been established, SSH multiplexing is used regardless of multiplex option. If multiplexing is enabled, forwarding is set by using the existing connection (-O forward option in ssh). This is beneficial if your servers require password authentication; you can avoid authentication in Julia by logging in to the server ahead of addprocs. The control socket will be located at ~/.ssh/julia-%r@%h:%p during the session unless the existing multiplexing connection is used. Note that bandwidth may be limited if you create multiple processes on a node and enable multiplexing, because in that case processes share a single multiplexing TCP connection."},{"location":"manual/distributed-computing.html#man-cluster-cookie","page":"Multi-processing and Distributed Computing","title":"Cluster Cookie","category":"section","text":"All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:\n\ncluster_cookie() returns the cookie, while cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie may be passed to the workers at startup via argument --worker=<cookie>. If argument --worker is specified without the cookie, the worker tries to read the cookie from its standard input (stdin). The stdin is closed immediately after the cookie is retrieved.\nClusterManagers can retrieve the cookie on the master by calling cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.\n\nNote that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument."},{"location":"manual/distributed-computing.html#Specifying-Network-Topology-(Experimental)","page":"Multi-processing and Distributed Computing","title":"Specifying Network Topology (Experimental)","category":"section","text":"The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other:\n\n:all_to_all, the default: all workers are connected to each other.\n:master_worker: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.\n\nKeyword argument lazy=true|false only affects topology option :all_to_all. If true, the cluster starts off with the master connected to all workers. Specific worker-worker connections are established at the first remote invocation between two workers. This helps in reducing initial resources allocated for intra-cluster communication. Connections are setup depending on the runtime requirements of a parallel program. Default value for lazy is true.\n\nCurrently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases."},{"location":"manual/distributed-computing.html#Noteworthy-external-packages","page":"Multi-processing and Distributed Computing","title":"Noteworthy external packages","category":"section","text":"Outside of Julia parallelism there are plenty of external packages that should be mentioned. For example, MPI.jl is a Julia wrapper for the MPI protocol, Dagger.jl provides functionality similar to Python's Dask, and DistributedArrays.jl provides array operations distributed across workers, as outlined above.\n\nA mention must be made of Julia's GPU programming ecosystem, which includes:\n\nCUDA.jl wraps the various CUDA libraries and supports compiling Julia kernels for Nvidia GPUs.\noneAPI.jl wraps the oneAPI unified programming model, and supports executing Julia kernels on supported accelerators. Currently only Linux is supported.\nAMDGPU.jl wraps the AMD ROCm libraries and supports compiling Julia kernels for AMD GPUs. Currently only Linux is supported.\nHigh-level libraries like KernelAbstractions.jl, Tullio.jl and ArrayFire.jl.\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes by first casting it through distribute() and CuArray().\n\nRemember when importing DistributedArrays.jl to import it across all processes using @everywhere\n\n$ ./julia -p 4\n\njulia> addprocs()\n\njulia> @everywhere using DistributedArrays\n\njulia> using CUDA\n\njulia> B = ones(10_000) ./ 2;\n\njulia> A = ones(10_000) .* π;\n\njulia> C = 2 .* A ./ B;\n\njulia> all(C .≈ 4*π)\ntrue\n\njulia> typeof(C)\nVector{Float64} (alias for Array{Float64, 1})\n\njulia> dB = distribute(B);\n\njulia> dA = distribute(A);\n\njulia> dC = 2 .* dA ./ dB;\n\njulia> all(dC .≈ 4*π)\ntrue\n\njulia> typeof(dC)\nDistributedArrays.DArray{Float64,1,Vector{Float64}}\n\njulia> cuB = CuArray(B);\n\njulia> cuA = CuArray(A);\n\njulia> cuC = 2 .* cuA ./ cuB;\n\njulia> all(cuC .≈ 4*π);\ntrue\n\njulia> typeof(cuC)\nCuArray{Float64,1}\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes and call a generic function on it.\n\nfunction power_method(M, v)\n    for i in 1:100\n        v = M*v\n        v /= norm(v)\n    end\n\n    return v, norm(M*v) / norm(v)  # or  (M*v) ./ v\nend\n\npower_method repeatedly creates a new vector and normalizes it. We have not specified any type signature in function declaration, let's see if it works with the aforementioned datatypes:\n\njulia> M = [2. 1; 1 1];\n\njulia> v = rand(2)\n2-element Vector{Float64}:\n0.40395\n0.445877\n\njulia> power_method(M,v)\n([0.850651, 0.525731], 2.618033988749895)\n\njulia> cuM = CuArray(M);\n\njulia> cuv = CuArray(v);\n\njulia> curesult = power_method(cuM, cuv);\n\njulia> typeof(curesult)\nCuArray{Float64,1}\n\njulia> dM = distribute(M);\n\njulia> dv = distribute(v);\n\njulia> dC = power_method(dM, dv);\n\njulia> typeof(dC)\nTuple{DistributedArrays.DArray{Float64,1,Vector{Float64}},Float64}\n\nTo end this short exposure to external packages, we can consider MPI.jl, a Julia wrapper of the MPI protocol. As it would take too long to consider every inner function, it would be better to simply appreciate the approach used to implement the protocol.\n\nConsider this toy script which simply calls each subprocess, instantiate its rank and when the master process is reached, performs the ranks' sum\n\nimport MPI\n\nMPI.Init()\n\ncomm = MPI.COMM_WORLD\nMPI.Barrier(comm)\n\nroot = 0\nr = MPI.Comm_rank(comm)\n\nsr = MPI.Reduce(r, MPI.SUM, root, comm)\n\nif(MPI.Comm_rank(comm) == root)\n   @printf(\"sum of ranks: %s\\n\", sr)\nend\n\nMPI.Finalize()\n\nmpirun -np 4 ./julia example.jl\n\n[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding rma to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see https://mpi-forum.org/docs."},{"location":"manual/workflow-tips.html#man-workflow-tips","page":"Workflow Tips","title":"Workflow Tips","category":"section","text":"Here are some tips for working with Julia efficiently."},{"location":"manual/workflow-tips.html#REPL-based-workflow","page":"Workflow Tips","title":"REPL-based workflow","category":"section","text":"As already elaborated in The Julia REPL, Julia's REPL provides rich functionality that facilitates an efficient interactive workflow. Here are some tips that might further enhance your experience at the command line."},{"location":"manual/workflow-tips.html#A-basic-editor/REPL-workflow","page":"Workflow Tips","title":"A basic editor/REPL workflow","category":"section","text":"The most basic Julia workflows involve using a text editor in conjunction with the julia command line.\n\nCreate a file, say Tmp.jl, and include within it\n\nmodule Tmp\n\nsay_hello() = println(\"Hello!\")\n\n# Your other definitions here\n\nend # module\n\nusing .Tmp\n\nThen, in the same directory, start the Julia REPL (using the julia command). Run the new file as follows:\n\njulia> include(\"Tmp.jl\")\n\njulia> Tmp.say_hello()\nHello!\n\nExplore ideas in the REPL. Save good ideas in Tmp.jl. To reload the file after it has been changed, just include it again.\n\nThe key in the above is that your code is encapsulated in a module. That allows you to edit struct definitions and remove methods, without restarting Julia.\n\n(Explanation: structs cannot be edited after definition, nor can methods be deleted. But you can overwrite the definition of a module, which is what we do when we re-include(\"Tmp.jl\")).\n\nIn addition, the encapsulation of code in a module protects it from being influenced by previous state in the REPL, protecting you from hard-to-detect errors."},{"location":"manual/workflow-tips.html#Browser-based-workflow","page":"Workflow Tips","title":"Browser-based workflow","category":"section","text":"There are a few ways to interact with Julia in a browser:\n\nUsing Pluto notebooks through Pluto.jl\nUsing Jupyter notebooks through IJulia.jl"},{"location":"manual/workflow-tips.html#Revise-based-workflows","page":"Workflow Tips","title":"Revise-based workflows","category":"section","text":"Whether you're at the REPL or in IJulia, you can typically improve your development experience with Revise. It is common to configure Revise to start whenever julia is started, as per the instructions in the Revise documentation. Once configured, Revise will track changes to files in any loaded modules, and to any files loaded in to the REPL with includet (but not with plain include); you can then edit the files and the changes take effect without restarting your julia session. A standard workflow is similar to the REPL-based workflow above, with the following modifications:\n\nPut your code in a module somewhere on your load path. There are several options for achieving this, of which two recommended choices are:\nFor long-term projects, use PkgTemplates:\nusing PkgTemplates\nt = Template()\nt(\"MyPkg\")\nThis will create a blank package, \"MyPkg\", in your .julia/dev directory. Note that PkgTemplates allows you to control many different options through its Template constructor.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and MyPkg/test/runtests.jl for the tests.\nFor \"throw-away\" projects, you can avoid any need for cleanup by doing your work in your temporary directory (e.g., /tmp).\nNavigate to your temporary directory and launch Julia, then do the following:\npkg> generate MyPkg            # type ] to enter pkg mode\njulia> push!(LOAD_PATH, pwd())   # hit backspace to exit pkg mode\nIf you restart your Julia session you'll have to re-issue that command modifying LOAD_PATH.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and create any test file of your choosing.\nDevelop your package\nBefore loading any code, make sure you're running Revise: say using Revise or follow its documentation on configuring it to run automatically.\nThen navigate to the directory containing your test file (here assumed to be \"runtests.jl\") and do the following:\njulia> using MyPkg\n\njulia> include(\"runtests.jl\")\nYou can iteratively modify the code in MyPkg in your editor and re-run the tests with include(\"runtests.jl\"). You generally should not need to restart your Julia session to see the changes take effect (subject to a few limitations)."},{"location":"manual/environment-variables.html#Environment-Variables","page":"Environment Variables","title":"Environment Variables","category":"section","text":"Julia can be configured with a number of environment variables, set either in the usual way for each operating system, or in a portable way from within Julia. Supposing that you want to set the environment variable JULIA_EDITOR to vim, you can type ENV[\"JULIA_EDITOR\"] = \"vim\" (for instance, in the REPL) to make this change on a case by case basis, or add the same to the user configuration file ~/.julia/config/startup.jl in the user's home directory to have a permanent effect. The current value of the same environment variable can be determined by evaluating ENV[\"JULIA_EDITOR\"].\n\nThe environment variables that Julia uses generally start with JULIA. If InteractiveUtils.versioninfo is called with the keyword verbose=true, then the output will list any defined environment variables relevant for Julia, including those which include JULIA in their names.\n\nnote: Note\nIt is recommended to avoid changing environment variables during runtime, such as within a ~/.julia/config/startup.jl.One reason is that some julia language variables, such as JULIA_NUM_THREADS and JULIA_PROJECT, need to be set before Julia starts.Similarly, __init__() functions of user modules in the sysimage (via PackageCompiler) are run before startup.jl, so setting environment variables in a startup.jl may be too late for user code.Further, changing environment variables during runtime can introduce data races into otherwise benign code.In Bash, environment variables can either be set manually by running, e.g., export JULIA_NUM_THREADS=4 before starting Julia, or by adding the same command to ~/.bashrc or ~/.bash_profile to set the variable each time Bash is started."},{"location":"manual/environment-variables.html#File-locations","page":"Environment Variables","title":"File locations","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_BINDIR","page":"Environment Variables","title":"JULIA_BINDIR","category":"section","text":"The absolute path of the directory containing the Julia executable, which sets the global variable Sys.BINDIR. If $JULIA_BINDIR is not set, then Julia determines the value Sys.BINDIR at run-time.\n\nThe executable itself is one of\n\n$JULIA_BINDIR/julia\n$JULIA_BINDIR/julia-debug\n\nby default.\n\nThe global variable Base.DATAROOTDIR determines a relative path from Sys.BINDIR to the data directory associated with Julia. Then the path\n\n$JULIA_BINDIR/$DATAROOTDIR/julia/base\n\ndetermines the directory in which Julia initially searches for source files (via Base.find_source_file()).\n\nLikewise, the global variable Base.SYSCONFDIR determines a relative path to the configuration file directory. Then Julia searches for a startup.jl file at\n\n$JULIA_BINDIR/$SYSCONFDIR/julia/startup.jl\n$JULIA_BINDIR/../etc/julia/startup.jl\n\nby default (via Base.load_julia_startup()).\n\nFor example, a Linux installation with a Julia executable located at /bin/julia, a DATAROOTDIR of ../share, and a SYSCONFDIR of ../etc will have JULIA_BINDIR set to /bin, a source-file search path of\n\n/share/julia/base\n\nand a global configuration search path of\n\n/etc/julia/startup.jl"},{"location":"manual/environment-variables.html#JULIA_PROJECT","page":"Environment Variables","title":"JULIA_PROJECT","category":"section","text":"A directory path that indicates which project should be the initial active project. Setting this environment variable has the same effect as specifying the --project start-up option, but --project has higher precedence. If the variable is set to @. (note the trailing dot) then Julia tries to find a project directory that contains Project.toml or JuliaProject.toml file from the current directory and its parents. See also the chapter on Code Loading.\n\nnote: Note\nJULIA_PROJECT must be defined before starting julia; defining it in startup.jl is too late in the startup process."},{"location":"manual/environment-variables.html#JULIA_LOAD_PATH","page":"Environment Variables","title":"JULIA_LOAD_PATH","category":"section","text":"The JULIA_LOAD_PATH environment variable is used to populate the global Julia LOAD_PATH variable, which determines which packages can be loaded via import and using (see Code Loading).\n\nUnlike the shell PATH variable, empty entries in JULIA_LOAD_PATH are expanded to the default value of LOAD_PATH, [\"@\", \"@v#.#\", \"@stdlib\"] when populating LOAD_PATH. This allows easy appending, prepending, etc. of the load path value in shell scripts regardless of whether JULIA_LOAD_PATH is already set or not. For example, to prepend the directory /foo/bar to LOAD_PATH just do\n\nexport JULIA_LOAD_PATH=\"/foo/bar:$JULIA_LOAD_PATH\"\n\nIf the JULIA_LOAD_PATH environment variable is already set, its old value will be prepended with /foo/bar. On the other hand, if JULIA_LOAD_PATH is not set, then it will be set to /foo/bar: which will expand to a LOAD_PATH value of [\"/foo/bar\", \"@\", \"@v#.#\", \"@stdlib\"]. If JULIA_LOAD_PATH is set to the empty string, it expands to an empty LOAD_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty load path via the environment variable. If you want the default load path, either unset the environment variable or if it must have a value, set it to the string :.\n\nnote: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph."},{"location":"manual/environment-variables.html#JULIA_DEPOT_PATH","page":"Environment Variables","title":"JULIA_DEPOT_PATH","category":"section","text":"The JULIA_DEPOT_PATH environment variable is used to populate the global Julia DEPOT_PATH variable, which controls where the package manager, as well as Julia's code loading mechanisms, look for package registries, installed packages, named environments, repo clones, cached compiled package images, configuration files, and the default location of the REPL's history file.\n\nUnlike the shell PATH variable but similar to JULIA_LOAD_PATH, empty entries in JULIA_DEPOT_PATH have special behavior:\n\nAt the end, it is expanded to the default value of DEPOT_PATH, excluding the user depot.\nAt the start, it is expanded to the default value of DEPOT_PATH, including the user depot.\n\nThis allows easy overriding of the user depot, while still retaining access to resources that are bundled with Julia, like cache files, artifacts, etc. For example, to switch the user depot to /foo/bar use a trailing :\n\nexport JULIA_DEPOT_PATH=\"/foo/bar:\"\n\nAll package operations, like cloning registries or installing packages, will now write to /foo/bar, but since the empty entry is expanded to the default system depot, any bundled resources will still be available. If you really only want to use the depot at /foo/bar, and not load any bundled resources, simply set the environment variable to /foo/bar without the trailing colon.\n\nTo append a depot at the end of the full default list, including the default user depot, use a leading :\n\nexport JULIA_DEPOT_PATH=\":/foo/bar\"\n\nThere are two exceptions to the above rule. First, if JULIA_DEPOT_PATH is set to the empty string, it expands to an empty DEPOT_PATH array. In other words, the empty string is interpreted as a zero-element array, not a one-element array of the empty string. This behavior was chosen so that it would be possible to set an empty depot path via the environment variable.\n\nSecond, if no user depot is specified in JULIA_DEPOT_PATH, then the empty entry is expanded to the default depot including the user depot. This makes it possible to use the default depot, as if the environment variable was unset, by setting it to the string :.\n\nnote: Note\nOn Windows, path elements are separated by the ; character, as is the case with most path lists on Windows. Replace : with ; in the above paragraph.\n\nnote: Note\nJULIA_DEPOT_PATH must be defined before starting julia; defining it in startup.jl is too late in the startup process; at that point you can instead directly modify the DEPOT_PATH array, which is populated from the environment variable."},{"location":"manual/environment-variables.html#JULIA_HISTORY","page":"Environment Variables","title":"JULIA_HISTORY","category":"section","text":"The absolute path REPL.find_hist_file() of the REPL's history file. If $JULIA_HISTORY is not set, then REPL.find_hist_file() defaults to\n\n$(DEPOT_PATH[1])/logs/repl_history.jl"},{"location":"manual/environment-variables.html#JULIA_MAX_NUM_PRECOMPILE_FILES","page":"Environment Variables","title":"JULIA_MAX_NUM_PRECOMPILE_FILES","category":"section","text":"Sets the maximum number of different instances of a single package that are to be stored in the precompile cache (default = 10)."},{"location":"manual/environment-variables.html#JULIA_VERBOSE_LINKING","page":"Environment Variables","title":"JULIA_VERBOSE_LINKING","category":"section","text":"If set to true, linker commands will be displayed during precompilation."},{"location":"manual/environment-variables.html#Pkg.jl","page":"Environment Variables","title":"Pkg.jl","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_CI","page":"Environment Variables","title":"JULIA_CI","category":"section","text":"If set to true, this indicates to the package server that any package operations are part of a continuous integration (CI) system for the purposes of gathering package usage statistics."},{"location":"manual/environment-variables.html#JULIA_NUM_PRECOMPILE_TASKS","page":"Environment Variables","title":"JULIA_NUM_PRECOMPILE_TASKS","category":"section","text":"The number of parallel tasks to use when precompiling packages. See Pkg.precompile."},{"location":"manual/environment-variables.html#JULIA_PKG_DEVDIR","page":"Environment Variables","title":"JULIA_PKG_DEVDIR","category":"section","text":"The default directory used by Pkg.develop for downloading packages."},{"location":"manual/environment-variables.html#JULIA_PKG_IGNORE_HASHES","page":"Environment Variables","title":"JULIA_PKG_IGNORE_HASHES","category":"section","text":"If set to 1, this will ignore incorrect hashes in artifacts. This should be used carefully, as it disables verification of downloads, but can resolve issues when moving files across different types of file systems. See Pkg.jl issue #2317 for more details.\n\ncompat: Julia 1.6\nThis is only supported in Julia 1.6 and above."},{"location":"manual/environment-variables.html#JULIA_PKG_OFFLINE","page":"Environment Variables","title":"JULIA_PKG_OFFLINE","category":"section","text":"If set to true, this will enable offline mode: see Pkg.offline.\n\ncompat: Julia 1.5\nPkg's offline mode requires Julia 1.5 or later."},{"location":"manual/environment-variables.html#JULIA_PKG_PRECOMPILE_AUTO","page":"Environment Variables","title":"JULIA_PKG_PRECOMPILE_AUTO","category":"section","text":"If set to 0, this will disable automatic precompilation by package actions which change the manifest. See Pkg.precompile."},{"location":"manual/environment-variables.html#JULIA_PKG_SERVER","page":"Environment Variables","title":"JULIA_PKG_SERVER","category":"section","text":"Specifies the URL of the package registry to use. By default, Pkg uses https://pkg.julialang.org to fetch Julia packages. In addition, you can disable the use of the PkgServer protocol, and instead access the packages directly from their hosts (GitHub, GitLab, etc.) by setting: export JULIA_PKG_SERVER=\"\""},{"location":"manual/environment-variables.html#JULIA_PKG_SERVER_REGISTRY_PREFERENCE","page":"Environment Variables","title":"JULIA_PKG_SERVER_REGISTRY_PREFERENCE","category":"section","text":"Specifies the preferred registry flavor. Currently supported values are conservative (the default), which will only publish resources that have been processed by the storage server (and thereby have a higher probability of being available from the PkgServers), whereas eager will publish registries whose resources have not necessarily been processed by the storage servers. Users behind restrictive firewalls that do not allow downloading from arbitrary servers should not use the eager flavor.\n\ncompat: Julia 1.7\nThis only affects Julia 1.7 and above."},{"location":"manual/environment-variables.html#JULIA_PKG_UNPACK_REGISTRY","page":"Environment Variables","title":"JULIA_PKG_UNPACK_REGISTRY","category":"section","text":"If set to true, this will unpack the registry instead of storing it as a compressed tarball.\n\ncompat: Julia 1.7\nThis only affects Julia 1.7 and above. Earlier versions will always unpack the registry."},{"location":"manual/environment-variables.html#JULIA_PKG_USE_CLI_GIT","page":"Environment Variables","title":"JULIA_PKG_USE_CLI_GIT","category":"section","text":"If set to true, Pkg operations which use the git protocol will use an external git executable instead of the default libgit2 library.\n\ncompat: Julia 1.7\nUse of the git executable is only supported on Julia 1.7 and above."},{"location":"manual/environment-variables.html#JULIA_PKGRESOLVE_ACCURACY","page":"Environment Variables","title":"JULIA_PKGRESOLVE_ACCURACY","category":"section","text":"The accuracy of the package resolver. This should be a positive integer, the default is 1."},{"location":"manual/environment-variables.html#JULIA_PKG_PRESERVE_TIERED_INSTALLED","page":"Environment Variables","title":"JULIA_PKG_PRESERVE_TIERED_INSTALLED","category":"section","text":"Change the default package installation strategy to Pkg.PRESERVE_TIERED_INSTALLED to let the package manager try to install versions of packages while keeping as many versions of packages already installed as possible.\n\ncompat: Julia 1.9\nThis only affects Julia 1.9 and above."},{"location":"manual/environment-variables.html#JULIA_PKG_GC_AUTO","page":"Environment Variables","title":"JULIA_PKG_GC_AUTO","category":"section","text":"If set to false, automatic garbage collection of packages and artifacts will be disabled; see Pkg.gc for more details.\n\ncompat: Julia 1.12\nThis environment variable is only supported on Julia 1.12 and above."},{"location":"manual/environment-variables.html#Network-transport","page":"Environment Variables","title":"Network transport","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_NO_VERIFY_HOSTS","page":"Environment Variables","title":"JULIA_NO_VERIFY_HOSTS","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_SSL_NO_VERIFY_HOSTS","page":"Environment Variables","title":"JULIA_SSL_NO_VERIFY_HOSTS","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_SSH_NO_VERIFY_HOSTS","page":"Environment Variables","title":"JULIA_SSH_NO_VERIFY_HOSTS","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_ALWAYS_VERIFY_HOSTS","page":"Environment Variables","title":"JULIA_ALWAYS_VERIFY_HOSTS","category":"section","text":"Specify hosts whose identity should or should not be verified for specific transport layers. See NetworkOptions.verify_host"},{"location":"manual/environment-variables.html#JULIA_SSL_CA_ROOTS_PATH","page":"Environment Variables","title":"JULIA_SSL_CA_ROOTS_PATH","category":"section","text":"Specify the file or directory containing the certificate authority roots. See NetworkOptions.ca_roots"},{"location":"manual/environment-variables.html#External-applications","page":"Environment Variables","title":"External applications","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_SHELL","page":"Environment Variables","title":"JULIA_SHELL","category":"section","text":"The absolute path of the shell with which Julia should execute external commands (via Base.repl_cmd()). Defaults to the environment variable $SHELL, and falls back to /bin/sh if $SHELL is unset.\n\nnote: Note\nOn Windows, this environment variable is ignored, and external commands are executed directly."},{"location":"manual/environment-variables.html#JULIA_EDITOR","page":"Environment Variables","title":"JULIA_EDITOR","category":"section","text":"The editor returned by InteractiveUtils.editor() and used in, e.g., InteractiveUtils.edit, referring to the command of the preferred editor, for instance vim.\n\n$JULIA_EDITOR takes precedence over $VISUAL, which in turn takes precedence over $EDITOR. If none of these environment variables is set, then the editor is taken to be open on Windows and OS X, or /etc/alternatives/editor if it exists, or emacs otherwise.\n\nTo use Visual Studio Code on Windows, set $JULIA_EDITOR to code.cmd."},{"location":"manual/environment-variables.html#Parallelization","page":"Environment Variables","title":"Parallelization","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_CPU_THREADS","page":"Environment Variables","title":"JULIA_CPU_THREADS","category":"section","text":"Overrides the global variable Base.Sys.CPU_THREADS, the number of logical CPU cores available."},{"location":"manual/environment-variables.html#JULIA_WORKER_TIMEOUT","page":"Environment Variables","title":"JULIA_WORKER_TIMEOUT","category":"section","text":"A Float64 that sets the value of Distributed.worker_timeout() (default: 60.0). This function gives the number of seconds a worker process will wait for a master process to establish a connection before dying."},{"location":"manual/environment-variables.html#JULIA_NUM_THREADS","page":"Environment Variables","title":"JULIA_NUM_THREADS","category":"section","text":"An unsigned 64-bit integer (uint64_t) or string that sets the maximum number of threads available to Julia. If $JULIA_NUM_THREADS is not set or is a non-positive integer, or if the number of CPU threads cannot be determined through system calls, then the number of threads is set to 1.\n\nIf $JULIA_NUM_THREADS is set to auto, then the number of threads will be set to the number of CPU threads. It can also be set to a comma-separated string to specify the size of the :default and :interactive threadpools, respectively:\n\n# 5 threads in the :default pool and 2 in the :interactive pool\nexport JULIA_NUM_THREADS=5,2\n\n# `auto` threads in the :default pool and 1 in the :interactive pool\nexport JULIA_NUM_THREADS=auto,1\n\nnote: Note\nJULIA_NUM_THREADS must be defined before starting Julia; defining it in startup.jl is too late in the startup process.\n\ncompat: Julia 1.5\nIn Julia 1.5 and above the number of threads can also be specified on startup using the -t/--threads command line argument.\n\ncompat: Julia 1.7\nThe auto value for $JULIA_NUM_THREADS requires Julia 1.7 or above.\n\ncompat: Julia 1.9\nThe x,y format for threadpools requires Julia 1.9 or above."},{"location":"manual/environment-variables.html#JULIA_THREAD_SLEEP_THRESHOLD","page":"Environment Variables","title":"JULIA_THREAD_SLEEP_THRESHOLD","category":"section","text":"If set to a string that starts with the case-insensitive substring \"infinite\", then spinning threads never sleep. Otherwise, $JULIA_THREAD_SLEEP_THRESHOLD is interpreted as an unsigned 64-bit integer (uint64_t) and gives, in nanoseconds, the amount of time after which spinning threads should sleep."},{"location":"manual/environment-variables.html#JULIA_NUM_GC_THREADS","page":"Environment Variables","title":"JULIA_NUM_GC_THREADS","category":"section","text":"Sets the number of threads used by Garbage Collection. If unspecified is set to the number of worker threads.\n\ncompat: Julia 1.10\nThe environment variable was added in 1.10"},{"location":"manual/environment-variables.html#JULIA_IMAGE_THREADS","page":"Environment Variables","title":"JULIA_IMAGE_THREADS","category":"section","text":"An unsigned 32-bit integer that sets the number of threads used by image compilation in this Julia process. The value of this variable may be ignored if the module is a small module. If left unspecified, the smaller of the value of JULIA_CPU_THREADS or half the number of logical CPU cores is used in its place."},{"location":"manual/environment-variables.html#JULIA_IMAGE_TIMINGS","page":"Environment Variables","title":"JULIA_IMAGE_TIMINGS","category":"section","text":"A boolean value that determines if detailed timing information is printed during during image compilation. Defaults to 0."},{"location":"manual/environment-variables.html#JULIA_EXCLUSIVE","page":"Environment Variables","title":"JULIA_EXCLUSIVE","category":"section","text":"If set to anything besides 0, then Julia's thread policy is consistent with running on a dedicated machine: each thread in the default threadpool is affinitized.  Interactive threads remain under the control of the operating system scheduler.\n\nOtherwise, Julia lets the operating system handle thread policy."},{"location":"manual/environment-variables.html#Garbage-Collection","page":"Environment Variables","title":"Garbage Collection","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_HEAP_SIZE_HINT","page":"Environment Variables","title":"JULIA_HEAP_SIZE_HINT","category":"section","text":"Environment variable equivalent to the --heap-size-hint=<size>[<unit>] command line option.\n\nForces garbage collection if memory usage is higher than the given value. The value may be specified as a number of bytes, optionally in units of:\n\n- B  (bytes)\n- K  (kibibytes)\n- M  (mebibytes)\n- G  (gibibytes)\n- T  (tebibytes)\n- %  (percentage of physical memory)\n\nFor example, JULIA_HEAP_SIZE_HINT=1G would provide a 1 GB heap size hint to the garbage collector."},{"location":"manual/environment-variables.html#REPL-formatting","page":"Environment Variables","title":"REPL formatting","category":"section","text":"Environment variables that determine how REPL output should be formatted at the terminal. The JULIA_*_COLOR variables should be set to ANSI terminal escape sequences. Julia provides a high-level interface with much of the same functionality; see the section on The Julia REPL."},{"location":"manual/environment-variables.html#JULIA_ERROR_COLOR","page":"Environment Variables","title":"JULIA_ERROR_COLOR","category":"section","text":"The formatting Base.error_color() (default: light red, \"\\033[91m\") that errors should have at the terminal."},{"location":"manual/environment-variables.html#JULIA_WARN_COLOR","page":"Environment Variables","title":"JULIA_WARN_COLOR","category":"section","text":"The formatting Base.warn_color() (default: yellow, \"\\033[93m\") that warnings should have at the terminal."},{"location":"manual/environment-variables.html#JULIA_INFO_COLOR","page":"Environment Variables","title":"JULIA_INFO_COLOR","category":"section","text":"The formatting Base.info_color() (default: cyan, \"\\033[36m\") that info should have at the terminal."},{"location":"manual/environment-variables.html#JULIA_INPUT_COLOR","page":"Environment Variables","title":"JULIA_INPUT_COLOR","category":"section","text":"The formatting Base.input_color() (default: normal, \"\\033[0m\") that input should have at the terminal."},{"location":"manual/environment-variables.html#JULIA_ANSWER_COLOR","page":"Environment Variables","title":"JULIA_ANSWER_COLOR","category":"section","text":"The formatting Base.answer_color() (default: normal, \"\\033[0m\") that output should have at the terminal."},{"location":"manual/environment-variables.html#NO_COLOR","page":"Environment Variables","title":"NO_COLOR","category":"section","text":"When this variable is present and not an empty string (regardless of its value) then colored text will be disabled on the REPL. Can be overridden with the flag --color=yes or with the environment variable FORCE_COLOR. This environment variable is commonly recognized by command-line applications."},{"location":"manual/environment-variables.html#FORCE_COLOR","page":"Environment Variables","title":"FORCE_COLOR","category":"section","text":"When this variable is present and not an empty string (regardless of its value) then colored text will be enabled on the REPL. Can be overridden with the flag --color=no. This environment variable is commonly recognized by command-line applications."},{"location":"manual/environment-variables.html#System-and-Package-Image-Building","page":"Environment Variables","title":"System and Package Image Building","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_CPU_TARGET","page":"Environment Variables","title":"JULIA_CPU_TARGET","category":"section","text":"Modify the target machine architecture for (pre)compiling system and package images. JULIA_CPU_TARGET only affects machine code image generation being output to a disk cache. Unlike the --cpu-target, or -C, command line option, it does not influence just-in-time (JIT) code generation within a Julia session where machine code is only stored in memory.\n\nValid values for JULIA_CPU_TARGET can be obtained by executing julia -C help.\n\nTo get the CPU target string that was used to build the current system image, use Sys.sysimage_target(). This can be useful for reproducing the same system image or understanding what CPU features were enabled during compilation.\n\nSetting JULIA_CPU_TARGET is important for heterogeneous compute systems where processors of distinct types or features may be present. This is commonly encountered in high performance computing (HPC) clusters since the component nodes may be using distinct processors. In this case, you may want to use the sysimage CPU target to maintain the same configuration as the sysimage. See below for more details.\n\nThe CPU target string is a list of strings separated by ; each string starts with a CPU or architecture name and followed by an optional list of features separated by ,. A generic or empty CPU name means the basic required feature set of the target ISA which is at least the architecture the C/C++ runtime is compiled with. Each string is interpreted by LLVM.\n\nnote: Note\nPackage images can only target the same or more specific CPU features than their base system image.\n\nA few special features are supported:\n\nsysimage\nA special keyword that can be used as a CPU target name, which will be replaced   with the CPU target string that was used to build the current system image. This allows   you to specify CPU targets that build upon or extend the current sysimage's target, which   is particularly helpful for creating package images that are as flexible as the sysimage.\nclone_all\nThis forces the target to have all functions in sysimg cloned.   When used in negative form (i.e. -clone_all), this disables full clone that's   enabled by default for certain targets.\nbase([0-9]*)\nThis specifies the (0-based) base target index. The base target is the target   that the current target is based on, i.e. the functions that are not being cloned   will use the version in the base target. This option causes the base target to be   fully cloned (as if clone_all is specified for it) if it is not the default target (0).   The index can only be smaller than the current index.\nopt_size\nOptimize for size with minimum performance impact. Clang/GCC's -Os.\nmin_size\nOptimize only for size. Clang's -Oz."},{"location":"manual/environment-variables.html#Debugging-and-profiling","page":"Environment Variables","title":"Debugging and profiling","category":"section","text":""},{"location":"manual/environment-variables.html#JULIA_DEBUG","page":"Environment Variables","title":"JULIA_DEBUG","category":"section","text":"Enable debug logging for a file or module, see Logging for more information."},{"location":"manual/environment-variables.html#CI-Debug-Environment-Variables","page":"Environment Variables","title":"CI Debug Environment Variables","category":"section","text":"Julia automatically enables verbose debugging options when certain continuous integration (CI) debug environment variables are set. This improves the debugging experience when CI jobs are re-run with debug logging enabled, by automatically:\n\nEnabling --trace-eval (location mode) to show expressions being evaluated\nSetting JULIA_TEST_VERBOSE=true to enable verbose test output\n\nThis allows developers to get detailed debugging information from CI runs without modifying their scripts or workflow files."},{"location":"manual/environment-variables.html#JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","page":"Environment Variables","title":"JULIA_PROFILE_PEEK_HEAP_SNAPSHOT","category":"section","text":"Enable collecting of a heap snapshot during execution via the profiling peek mechanism. See Triggered During Execution."},{"location":"manual/environment-variables.html#JULIA_TIMING_SUBSYSTEMS","page":"Environment Variables","title":"JULIA_TIMING_SUBSYSTEMS","category":"section","text":"Allows you to enable or disable zones for a specific Julia run. For instance, setting the variable to +GC,-INFERENCE will enable the GC zones and disable the INFERENCE zones. See Dynamically Enabling and Disabling Zones."},{"location":"manual/environment-variables.html#JULIA_GC_WAIT_FOR_DEBUGGER","page":"Environment Variables","title":"JULIA_GC_WAIT_FOR_DEBUGGER","category":"section","text":"If set to anything besides 0, then the Julia garbage collector will wait for a debugger to attach instead of aborting whenever there's a critical error.\n\nnote: Note\nThis environment variable only has an effect if Julia was compiled with garbage-collection debugging (that is, if WITH_GC_DEBUG_ENV is set to 1 in the build configuration)."},{"location":"manual/environment-variables.html#ENABLE_JITPROFILING","page":"Environment Variables","title":"ENABLE_JITPROFILING","category":"section","text":"If set to anything besides 0, then the compiler will create and register an event listener for just-in-time (JIT) profiling.\n\nnote: Note\nThis environment variable only has an effect if Julia was compiled with JIT profiling support, using eitherIntel's VTune™ Amplifier (USE_INTEL_JITEVENTS set to 1 in the build configuration), or\nOProfile (USE_OPROFILE_JITEVENTS set to 1 in the build configuration).\nPerf (USE_PERF_JITEVENTS set to 1 in the build configuration). This integration is enabled by default."},{"location":"manual/environment-variables.html#ENABLE_GDBLISTENER","page":"Environment Variables","title":"ENABLE_GDBLISTENER","category":"section","text":"If set to anything besides 0 enables GDB registration of Julia code on release builds. On debug builds of Julia this is always enabled. Recommended to use with -g 2."},{"location":"manual/environment-variables.html#JULIA_LLVM_ARGS","page":"Environment Variables","title":"JULIA_LLVM_ARGS","category":"section","text":"Arguments to be passed to the LLVM backend."},{"location":"manual/environment-variables.html#JULIA_FALLBACK_REPL","page":"Environment Variables","title":"JULIA_FALLBACK_REPL","category":"section","text":"Forces the fallback repl instead of REPL.jl."},{"location":"manual/methods.html#Methods","page":"Methods","title":"Methods","category":"section","text":"Recall from Functions that a function is an object that maps a tuple of arguments to a return value, or throws an exception if no appropriate value can be returned. It is common for the same conceptual function or operation to be implemented quite differently for different types of arguments: adding two integers is very different from adding two floating-point numbers, both of which are distinct from adding an integer to a floating-point number. Despite their implementation differences, these operations all fall under the general concept of \"addition\". Accordingly, in Julia, these behaviors all belong to a single object: the + function.\n\nTo facilitate using many different implementations of the same concept smoothly, functions need not be defined all at once, but can rather be defined piecewise by providing specific behaviors for certain combinations of argument types and counts. A definition of one possible behavior for a function is called a method. Thus far, we have presented only examples of functions defined with a single method, applicable to all types of arguments. However, the signatures of method definitions can be annotated to indicate the types of arguments in addition to their number, and more than a single method definition may be provided. When a function is applied to a particular tuple of arguments, the most specific method applicable to those arguments is applied. Thus, the overall behavior of a function is a patchwork of the behaviors of its various method definitions. If the patchwork is well designed, even though the implementations of the methods may be quite different, the outward behavior of the function will appear seamless and consistent.\n\nThe choice of which method to execute when a function is applied is called dispatch. Julia allows the dispatch process to choose which of a function's methods to call based on the number of arguments given, and on the types of all of the function's arguments. This is different than traditional object-oriented languages, where dispatch occurs based only on the first argument, which often has a special argument syntax, and is sometimes implied rather than explicitly written as an argument. [1] Using all of a function's arguments to choose which method should be invoked, rather than just the first, is known as multiple dispatch. Multiple dispatch is particularly useful for mathematical code, where it makes little sense to artificially deem the operations to \"belong\" to one argument more than any of the others: does the addition operation in x + y belong to x any more than it does to y? The implementation of a mathematical operator generally depends on the types of all of its arguments. Even beyond mathematical operations, however, multiple dispatch ends up being a powerful and convenient paradigm for structuring and organizing programs.\n\n[1]: In C++ or Java, for example, in a method call like obj.meth(arg1,arg2), the object obj \"receives\" the method call and is implicitly passed to the method via the this keyword, rather than as an explicit method argument. When the current this object is the receiver of a method call, it can be omitted altogether, writing just meth(arg1,arg2), with this implied as the receiving object.\n\nnote: Note\nAll the examples in this chapter assume that you are defining methods for a function in the same module. If you want to add methods to a function in another module, you have to import it or use the name qualified with module names. See the section on namespace management."},{"location":"manual/methods.html#Defining-Methods","page":"Methods","title":"Defining Methods","category":"section","text":"Until now, we have, in our examples, defined only functions with a single method having unconstrained argument types. Such functions behave just like they would in traditional dynamically typed languages. Nevertheless, we have used multiple dispatch and methods almost continually without being aware of it: all of Julia's standard functions and operators, like the aforementioned + function, have many methods defining their behavior over various possible combinations of argument type and count.\n\nWhen defining a function, one can optionally constrain the types of parameters it is applicable to, using the :: type-assertion operator, introduced in the section on Composite Types:\n\njulia> f(x::Float64, y::Float64) = 2x + y\nf (generic function with 1 method)\n\nThis function definition applies only to calls where x and y are both values of type Float64:\n\njulia> f(2.0, 3.0)\n7.0\n\nApplying it to any other types of arguments will result in a MethodError:\n\njulia> f(2.0, 3)\nERROR: MethodError: no method matching f(::Float64, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(Float32(2.0), 3.0)\nERROR: MethodError: no method matching f(::Float32, ::Float64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, ::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(2.0, \"3.0\")\nERROR: MethodError: no method matching f(::Float64, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f(\"2.0\", \"3.0\")\nERROR: MethodError: no method matching f(::String, ::String)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nAs you can see, the arguments must be precisely of type Float64. Other numeric types, such as integers or 32-bit floating-point values, are not automatically converted to 64-bit floating-point, nor are strings parsed as numbers. Because Float64 is a concrete type and concrete types cannot be subclassed in Julia, such a definition can only be applied to arguments that are exactly of type Float64. It may often be useful, however, to write more general methods where the declared parameter types are abstract:\n\njulia> f(x::Number, y::Number) = 2x - y\nf (generic function with 2 methods)\n\njulia> f(2.0, 3)\n1.0\n\nThis method definition applies to any pair of arguments that are instances of Number. They need not be of the same type, so long as they are each numeric values. The problem of handling disparate numeric types is delegated to the arithmetic operations in the expression 2x - y.\n\nTo define a function with multiple methods, one simply defines the function multiple times, with different numbers and types of arguments. The first method definition for a function creates the function object, and subsequent method definitions add new methods to the existing function object. The most specific method definition matching the number and types of the arguments will be executed when the function is applied. Thus, the two method definitions above, taken together, define the behavior for f over all pairs of instances of the abstract type Number – but with a different behavior specific to pairs of Float64 values. If one of the arguments is a 64-bit float but the other one is not, then the f(Float64,Float64) method cannot be called and the more general f(Number,Number) method must be used:\n\njulia> f(2.0, 3.0)\n7.0\n\njulia> f(2, 3.0)\n1.0\n\njulia> f(2.0, 3)\n1.0\n\njulia> f(2, 3)\n1\n\nThe 2x + y definition is only used in the first case, while the 2x - y definition is used in the others. No automatic casting or conversion of function arguments is ever performed: all conversion in Julia is non-magical and completely explicit. Conversion and Promotion, however, shows how clever application of sufficiently advanced technology can be indistinguishable from magic. [Clarke61]\n\nFor non-numeric values, and for fewer or more than two arguments, the function f remains undefined, and applying it will still result in a MethodError:\n\njulia> f(\"foo\", 3)\nERROR: MethodError: no method matching f(::String, ::Int64)\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Number, ::Number)\n   @ Main none:1\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> f()\nERROR: MethodError: no method matching f()\nThe function `f` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  f(!Matched::Float64, !Matched::Float64)\n   @ Main none:1\n  f(!Matched::Number, !Matched::Number)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nYou can easily see which methods exist for a function by entering the function object itself in an interactive session:\n\njulia> f\nf (generic function with 2 methods)\n\nThis output tells us that f is a function object with two methods. To find out what the signatures of those methods are, use the methods function:\n\njulia> methods(f)\n# 2 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1\n\nwhich shows that f has two methods, one taking two Float64 arguments and one taking arguments of type Number. It also indicates the file and line number where the methods were defined: because these methods were defined at the REPL, we get the apparent line number none:1.\n\nIn the absence of a type declaration with ::, the type of a method parameter is Any by default, meaning that it is unconstrained since all values in Julia are instances of the abstract type Any. Thus, we can define a catch-all method for f like so:\n\njulia> f(x,y) = println(\"Whoa there, Nelly.\")\nf (generic function with 3 methods)\n\njulia> methods(f)\n# 3 methods for generic function \"f\" from Main:\n [1] f(x::Float64, y::Float64)\n     @ none:1\n [2] f(x::Number, y::Number)\n     @ none:1\n [3] f(x, y)\n     @ none:1\n\njulia> f(\"foo\", 1)\nWhoa there, Nelly.\n\nThis catch-all is less specific than any other possible method definition for a pair of parameter values, so it will only be called on pairs of arguments to which no other method definition applies.\n\nNote that in the signature of the third method, there is no type specified for the arguments x and y. This is a shortened way of expressing f(x::Any, y::Any).\n\nAlthough it seems a simple concept, multiple dispatch on the types of values is perhaps the single most powerful and central feature of the Julia language. Core operations typically have dozens of methods:\n\njulia> methods(+)\n# 180 methods for generic function \"+\":\n[1] +(x::Bool, z::Complex{Bool}) in Base at complex.jl:227\n[2] +(x::Bool, y::Bool) in Base at bool.jl:89\n[3] +(x::Bool) in Base at bool.jl:86\n[4] +(x::Bool, y::T) where T<:AbstractFloat in Base at bool.jl:96\n[5] +(x::Bool, z::Complex) in Base at complex.jl:234\n[6] +(a::Float16, b::Float16) in Base at float.jl:373\n[7] +(x::Float32, y::Float32) in Base at float.jl:375\n[8] +(x::Float64, y::Float64) in Base at float.jl:376\n[9] +(z::Complex{Bool}, x::Bool) in Base at complex.jl:228\n[10] +(z::Complex{Bool}, x::Real) in Base at complex.jl:242\n[11] +(x::Char, y::Integer) in Base at char.jl:40\n[12] +(c::BigInt, x::BigFloat) in Base.MPFR at mpfr.jl:307\n[13] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt, e::BigInt) in Base.GMP at gmp.jl:392\n[14] +(a::BigInt, b::BigInt, c::BigInt, d::BigInt) in Base.GMP at gmp.jl:391\n[15] +(a::BigInt, b::BigInt, c::BigInt) in Base.GMP at gmp.jl:390\n[16] +(x::BigInt, y::BigInt) in Base.GMP at gmp.jl:361\n[17] +(x::BigInt, c::Union{UInt16, UInt32, UInt64, UInt8}) in Base.GMP at gmp.jl:398\n...\n[180] +(a, b, c, xs...) in Base at operators.jl:424\n\nMultiple dispatch together with the flexible parametric type system give Julia its ability to abstractly express high-level algorithms decoupled from implementation details."},{"location":"manual/methods.html#man-method-specializations","page":"Methods","title":"Method specializations","category":"section","text":"When you create multiple methods of the same function, this is sometimes called \"specialization.\" In this case, you're specializing the function by adding additional methods to it: each new method is a new specialization of the function. As shown above, these specializations are returned by methods.\n\nThere's another kind of specialization that occurs without programmer intervention: Julia's compiler can automatically specialize the method for the specific argument types used. Such specializations are not listed by methods, as this doesn't create new Methods, but tools like @code_typed allow you to inspect such specializations.\n\nFor example, if you create a method\n\nmysum(x::Real, y::Real) = x + y\n\nyou've given the function mysum one new method (possibly its only method), and that method takes any pair of Real number inputs. But if you then execute\n\njulia> mysum(1, 2)\n3\n\njulia> mysum(1.0, 2.0)\n3.0\n\nJulia will compile mysum twice, once for x::Int, y::Int and again for x::Float64, y::Float64. The point of compiling twice is performance: the methods that get called for + (which mysum uses) vary depending on the specific types of x and y, and by compiling different specializations Julia can do all the method lookup ahead of time. This allows the program to run much more quickly, since it does not have to bother with method lookup while it is running. Julia's automatic specialization allows you to write generic algorithms and expect that the compiler will generate efficient, specialized code to handle each case you need.\n\nIn cases where the number of potential specializations might be effectively unlimited, Julia may avoid this default specialization. See Be aware of when Julia avoids specializing for more information."},{"location":"manual/methods.html#man-ambiguities","page":"Methods","title":"Method Ambiguities","category":"section","text":"It is possible to define a set of function methods such that there is no unique most specific method applicable to some combinations of arguments:\n\njulia> g(x::Float64, y) = 2x + y\ng (generic function with 1 method)\n\njulia> g(x, y::Float64) = x + 2y\ng (generic function with 2 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\nERROR: MethodError: g(::Float64, ::Float64) is ambiguous.\n\nCandidates:\n  g(x, y::Float64)\n    @ Main none:1\n  g(x::Float64, y)\n    @ Main none:1\n\nPossible fix, define\n  g(::Float64, ::Float64)\n\nStacktrace:\n[...]\n\nHere the call g(2.0, 3.0) could be handled by either the g(::Float64, ::Any) or the g(::Any, ::Float64) method. The order in which the methods are defined does not matter and neither is more specific than the other. In such cases, Julia raises a MethodError rather than arbitrarily picking a method. You can avoid method ambiguities by specifying an appropriate method for the intersection case:\n\njulia> g(x::Float64, y::Float64) = 2x + 2y\ng (generic function with 3 methods)\n\njulia> g(2.0, 3)\n7.0\n\njulia> g(2, 3.0)\n8.0\n\njulia> g(2.0, 3.0)\n10.0\n\nIt is recommended that the disambiguating method be defined first, since otherwise the ambiguity exists, if transiently, until the more specific method is defined.\n\nIn more complex cases, resolving method ambiguities involves a certain element of design; this topic is explored further below."},{"location":"manual/methods.html#Parametric-Methods","page":"Methods","title":"Parametric Methods","category":"section","text":"Method definitions can optionally have type parameters qualifying the signature:\n\njulia> same_type(x::T, y::T) where {T} = true\nsame_type (generic function with 1 method)\n\njulia> same_type(x,y) = false\nsame_type (generic function with 2 methods)\n\nThe first method applies whenever both arguments are of the same concrete type, regardless of what type that is, while the second method acts as a catch-all, covering all other cases. Thus, overall, this defines a boolean function that checks whether its two arguments are of the same type:\n\njulia> same_type(1, 2)\ntrue\n\njulia> same_type(1, 2.0)\nfalse\n\njulia> same_type(1.0, 2.0)\ntrue\n\njulia> same_type(\"foo\", 2.0)\nfalse\n\njulia> same_type(\"foo\", \"bar\")\ntrue\n\njulia> same_type(Int32(1), Int64(2))\nfalse\n\nSuch definitions correspond to methods whose type signatures are UnionAll types (see UnionAll Types).\n\nThis kind of definition of function behavior by dispatch is quite common – idiomatic, even – in Julia. Method type parameters are not restricted to being used as the types of arguments: they can be used anywhere a value would be in the signature of the function or body of the function. Here's an example where the method type parameter T is used as the type parameter to the parametric type Vector{T} in the method signature:\n\njulia> function myappend(v::Vector{T}, x::T) where {T}\n           return [v..., x]\n       end\nmyappend (generic function with 1 method)\n\nThe type parameter T in this example ensures that the added element x is a subtype of the existing eltype of the vector v. The where keyword introduces a list of those constraints after the method signature definition. This works the same for one-line definitions, as seen above, and must appear before the return type declaration, if present, as illustrated below:\n\njulia> (myappend(v::Vector{T}, x::T)::Vector) where {T} = [v..., x]\nmyappend (generic function with 1 method)\n\njulia> myappend([1,2,3],4)\n4-element Vector{Int64}:\n 1\n 2\n 3\n 4\n\njulia> myappend([1,2,3],2.5)\nERROR: MethodError: no method matching myappend(::Vector{Int64}, ::Float64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> myappend([1.0,2.0,3.0],4.0)\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0\n\njulia> myappend([1.0,2.0,3.0],4)\nERROR: MethodError: no method matching myappend(::Vector{Float64}, ::Int64)\nThe function `myappend` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  myappend(::Vector{T}, !Matched::T) where T\n   @ Main none:1\n\nStacktrace:\n[...]\n\nIf the type of the appended element does not match the element type of the vector it is appended to, a MethodError is raised. In the following example, the method's type parameter T is used as the return value:\n\njulia> mytypeof(x::T) where {T} = T\nmytypeof (generic function with 1 method)\n\njulia> mytypeof(1)\nInt64\n\njulia> mytypeof(1.0)\nFloat64\n\nJust as you can put subtype constraints on type parameters in type declarations (see Parametric Types), you can also constrain type parameters of methods:\n\njulia> same_type_numeric(x::T, y::T) where {T<:Number} = true\nsame_type_numeric (generic function with 1 method)\n\njulia> same_type_numeric(x::Number, y::Number) = false\nsame_type_numeric (generic function with 2 methods)\n\njulia> same_type_numeric(1, 2)\ntrue\n\njulia> same_type_numeric(1, 2.0)\nfalse\n\njulia> same_type_numeric(1.0, 2.0)\ntrue\n\njulia> same_type_numeric(\"foo\", 2.0)\nERROR: MethodError: no method matching same_type_numeric(::String, ::Float64)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  same_type_numeric(!Matched::T, ::T) where T<:Number\n   @ Main none:1\n  same_type_numeric(!Matched::Number, ::Number)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> same_type_numeric(\"foo\", \"bar\")\nERROR: MethodError: no method matching same_type_numeric(::String, ::String)\nThe function `same_type_numeric` exists, but no method is defined for this combination of argument types.\n\njulia> same_type_numeric(Int32(1), Int64(2))\nfalse\n\nThe same_type_numeric function behaves much like the same_type function defined above, but is only defined for pairs of numbers.\n\nParametric methods allow the same syntax as where expressions used to write types (see UnionAll Types). If there is only a single parameter, the enclosing curly braces (in where {T}) can be omitted, but are often preferred for clarity. Multiple parameters can be separated with commas, e.g. where {T, S<:Real}, or written using nested where, e.g. where S<:Real where T."},{"location":"manual/methods.html#Redefining-Methods","page":"Methods","title":"Redefining Methods","category":"section","text":"When redefining a method or adding new methods, it is important to realize that these changes don't take effect immediately. This is key to Julia's ability to statically infer and compile code to run fast, without the usual JIT tricks and overhead. Indeed, any new method definition won't be visible to the current runtime environment, including Tasks and Threads (and any previously defined @generated functions). Let's start with an example to see what this means:\n\njulia> function tryeval()\n           @eval newfun() = 1\n           newfun()\n       end\ntryeval (generic function with 1 method)\n\njulia> tryeval()\nERROR: MethodError: no method matching newfun()\nThe applicable method may be too new: running in world age xxxx1, while current world is xxxx2.\nClosest candidates are:\n  newfun() at none:1 (method too new to be called from this world context.)\n in tryeval() at none:1\n ...\n\njulia> newfun()\n1\n\nIn this example, observe that the new definition for newfun has been created, but can't be immediately called. The new global is immediately visible to the tryeval function, so you could write return newfun (without parentheses). But neither you, nor any of your callers, nor the functions they call, or etc. can call this new method definition!\n\nBut there's an exception: future calls to newfun from the REPL work as expected, being able to both see and call the new definition of newfun.\n\nHowever, future calls to tryeval will continue to see the definition of newfun as it was at the previous statement at the REPL, and thus before that call to tryeval.\n\nYou may want to try this for yourself to see how it works.\n\nThe implementation of this behavior is a \"world age counter\", which is further described in the World Age manual chapter."},{"location":"manual/methods.html#Design-Patterns-with-Parametric-Methods","page":"Methods","title":"Design Patterns with Parametric Methods","category":"section","text":"While complex dispatch logic is not required for performance or usability, sometimes it can be the best way to express some algorithm. Here are a few common design patterns that come up sometimes when using dispatch in this way."},{"location":"manual/methods.html#Extracting-the-type-parameter-from-a-super-type","page":"Methods","title":"Extracting the type parameter from a super-type","category":"section","text":"Here is a correct code template for returning the element-type T of any arbitrary subtype of AbstractArray that has well-defined element type:\n\nabstract type AbstractArray{T, N} end\neltype(::Type{<:AbstractArray{T}}) where {T} = T\n\nusing so-called triangular dispatch. Note that UnionAll types, for example eltype(AbstractArray{T} where T <: Integer), do not match the above method. The implementation of eltype in Base adds a fallback method to Any for such cases.\n\nOne common mistake is to try and get the element-type by using introspection:\n\neltype_wrong(::Type{A}) where {A<:AbstractArray} = A.parameters[1]\n\nHowever, it is not hard to construct cases where this will fail:\n\nstruct BitVector <: AbstractArray{Bool, 1}; end\n\nHere we have created a type BitVector which has no parameters, but where the element-type is still fully specified, with T equal to Bool!\n\nAnother mistake is to try to walk up the type hierarchy using supertype:\n\neltype_wrong(::Type{AbstractArray{T}}) where {T} = T\neltype_wrong(::Type{AbstractArray{T, N}}) where {T, N} = T\neltype_wrong(::Type{A}) where {A<:AbstractArray} = eltype_wrong(supertype(A))\n\nWhile this works for declared types, it fails for types without supertypes:\n\njulia> eltype_wrong(Union{Vector{Int}, Matrix{Int}})\nERROR: MethodError: no method matching supertype(::Type{VecOrMat{Int64}})\n\nClosest candidates are:\n  supertype(::UnionAll)\n   @ Base operators.jl:44\n  supertype(::DataType)\n   @ Base operators.jl:43"},{"location":"manual/methods.html#Building-a-similar-type-with-a-different-type-parameter","page":"Methods","title":"Building a similar type with a different type parameter","category":"section","text":"When building generic code, there is often a need for constructing a similar object with some change made to the layout of the type, also necessitating a change of the type parameters. For instance, you might have some sort of abstract array with an arbitrary element type and want to write your computation on it with a specific element type. We must implement a method for each AbstractArray{T} subtype that describes how to compute this type transform. There is no general transform of one subtype into another subtype with a different parameter.\n\nThe subtypes of AbstractArray typically implement two methods to achieve this: A method to convert the input array to a subtype of a specific AbstractArray{T, N} abstract type; and a method to make a new uninitialized array with a specific element type. Sample implementations of these can be found in Julia Base. Here is a basic example usage of them, guaranteeing that input and output are of the same type:\n\ninput = convert(AbstractArray{Eltype}, input)\noutput = similar(input, Eltype)\n\nAs an extension of this, in cases where the algorithm needs a copy of the input array, convert is insufficient as the return value may alias the original input. Combining similar (to make the output array) and copyto! (to fill it with the input data) is a generic way to express the requirement for a mutable copy of the input argument:\n\ncopy_with_eltype(input, Eltype) = copyto!(similar(input, Eltype), input)"},{"location":"manual/methods.html#Iterated-dispatch","page":"Methods","title":"Iterated dispatch","category":"section","text":"In order to dispatch a multi-level parametric argument list, often it is best to separate each level of dispatch into distinct functions. This may sound similar in approach to single-dispatch, but as we shall see below, it is still more flexible.\n\nFor example, trying to dispatch on the element-type of an array will often run into ambiguous situations. Instead, common code will dispatch first on the container type, then recurse down to a more specific method based on eltype. In most cases, the algorithms lend themselves conveniently to this hierarchical approach, while in other cases, this rigor must be resolved manually. This dispatching branching can be observed, for example, in the logic to sum two matrices:\n\n# First dispatch selects the map algorithm for element-wise summation.\n+(a::Matrix, b::Matrix) = map(+, a, b)\n# Then dispatch handles each element and selects the appropriate\n# common element type for the computation.\n+(a, b) = +(promote(a, b)...)\n# Once the elements have the same type, they can be added.\n# For example, via primitive operations exposed by the processor.\n+(a::Float64, b::Float64) = Core.add(a, b)"},{"location":"manual/methods.html#Trait-based-dispatch","page":"Methods","title":"Trait-based dispatch","category":"section","text":"A natural extension to the iterated dispatch above is to add a layer to method selection that allows to dispatch on sets of types which are independent from the sets defined by the type hierarchy. We could construct such a set by writing out a Union of the types in question, but then this set would not be extensible as Union-types cannot be altered after creation. However, such an extensible set can be programmed with a design pattern often referred to as a \"Holy-trait\".\n\nThis pattern is implemented by defining a generic function which computes a different singleton value (or type) for each trait-set to which the function arguments may belong to. If this function is pure there is no impact on performance compared to normal dispatch.\n\nThe example in the previous section glossed over the implementation details of map and promote, which both operate in terms of these traits. When iterating over a matrix, such as in the implementation of map, one important question is what order to use to traverse the data. When AbstractArray subtypes implement the Base.IndexStyle trait, other functions such as map can dispatch on this information to pick the best algorithm (see Abstract Array Interface). This means that each subtype does not need to implement a custom version of map, since the generic definitions + trait classes will enable the system to select the fastest version. Here is a toy implementation of map illustrating the trait-based dispatch:\n\nmap(f, a::AbstractArray, b::AbstractArray) = map(Base.IndexStyle(a, b), f, a, b)\n# generic implementation:\nmap(::Base.IndexCartesian, f, a::AbstractArray, b::AbstractArray) = ...\n# linear-indexing implementation (faster)\nmap(::Base.IndexLinear, f, a::AbstractArray, b::AbstractArray) = ...\n\nThis trait-based approach is also present in the promote mechanism employed by the scalar +. It uses promote_type, which returns the optimal common type to compute the operation given the two types of the operands. This makes it possible to reduce the problem of implementing every function for every pair of possible type arguments, to the much smaller problem of implementing a conversion operation from each type to a common type, plus a table of preferred pair-wise promotion rules."},{"location":"manual/methods.html#Output-type-computation","page":"Methods","title":"Output-type computation","category":"section","text":"The discussion of trait-based promotion provides a transition into our next design pattern: computing the output element type for a matrix operation.\n\nFor implementing primitive operations, such as addition, we use the promote_type function to compute the desired output type. (As before, we saw this at work in the promote call in the call to +).\n\nFor more complex functions on matrices, it may be necessary to compute the expected return type for a more complex sequence of operations. This is often performed by the following steps:\n\nWrite a small function op that expresses the set of operations performed by the kernel of the algorithm.\nCompute the element type R of the result matrix as promote_op(op, argument_types...), where argument_types is computed from eltype applied to each input array.\nBuild the output matrix as similar(R, dims), where dims are the desired dimensions of the output array.\n\nFor a more specific example, a generic square-matrix multiply pseudo-code might look like:\n\nfunction matmul(a::AbstractMatrix, b::AbstractMatrix)\n    op = (ai, bi) -> ai * bi + ai * bi\n\n    ## this is insufficient because it assumes `one(eltype(a))` is constructable:\n    # R = typeof(op(one(eltype(a)), one(eltype(b))))\n\n    ## this fails because it assumes `a[1]` exists and is representative of all elements of the array\n    # R = typeof(op(a[1], b[1]))\n\n    ## this is incorrect because it assumes that `+` calls `promote_type`\n    ## but this is not true for some types, such as Bool:\n    # R = promote_type(ai, bi)\n\n    # this is wrong, since depending on the return value\n    # of type-inference is very brittle (as well as not being optimizable):\n    # R = Base.return_types(op, (eltype(a), eltype(b)))\n\n    ## but, finally, this works:\n    R = promote_op(op, eltype(a), eltype(b))\n    ## although sometimes it may give a larger type than desired\n    ## it will always give a correct type\n\n    output = similar(b, R, (size(a, 1), size(b, 2)))\n    if size(a, 2) > 0\n        for j in 1:size(b, 2)\n            for i in 1:size(a, 1)\n                ## here we don't use `ab = zero(R)`,\n                ## since `R` might be `Any` and `zero(Any)` is not defined\n                ## we also must declare `ab::R` to make the type of `ab` constant in the loop,\n                ## since it is possible that typeof(a * b) != typeof(a * b + a * b) == R\n                ab::R = a[i, 1] * b[1, j]\n                for k in 2:size(a, 2)\n                    ab += a[i, k] * b[k, j]\n                end\n                output[i, j] = ab\n            end\n        end\n    end\n    return output\nend"},{"location":"manual/methods.html#Separate-convert-and-kernel-logic","page":"Methods","title":"Separate convert and kernel logic","category":"section","text":"One way to significantly cut down on compile-times and testing complexity is to isolate the logic for converting to the desired type and the computation. This lets the compiler specialize and inline the conversion logic independent from the rest of the body of the larger kernel.\n\nThis is a common pattern seen when converting from a larger class of types to the one specific argument type that is actually supported by the algorithm:\n\ncomplexfunction(arg::Int) = ...\ncomplexfunction(arg::Any) = complexfunction(convert(Int, arg))\n\nmatmul(a::T, b::T) = ...\nmatmul(a, b) = matmul(promote(a, b)...)"},{"location":"manual/methods.html#Parametrically-constrained-Varargs-methods","page":"Methods","title":"Parametrically-constrained Varargs methods","category":"section","text":"Function parameters can also be used to constrain the number of arguments that may be supplied to a \"varargs\" function (Varargs Functions). The notation Vararg{T,N} is used to indicate such a constraint. For example:\n\njulia> bar(a,b,x::Vararg{Any,2}) = (a,b,x)\nbar (generic function with 1 method)\n\njulia> bar(1,2,3)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, !Matched::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\njulia> bar(1,2,3,4)\n(1, 2, (3, 4))\n\njulia> bar(1,2,3,4,5)\nERROR: MethodError: no method matching bar(::Int64, ::Int64, ::Int64, ::Int64, ::Int64)\nThe function `bar` exists, but no method is defined for this combination of argument types.\n\nClosest candidates are:\n  bar(::Any, ::Any, ::Any, ::Any)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nMore usefully, it is possible to constrain varargs methods by a parameter. For example:\n\nfunction getindex(A::AbstractArray{T,N}, indices::Vararg{Number,N}) where {T,N}\n\nwould be called only when the number of indices matches the dimensionality of the array.\n\nWhen only the type of supplied arguments needs to be constrained Vararg{T} can be equivalently written as T.... For instance f(x::Int...) = x is a shorthand for f(x::Vararg{Int}) = x."},{"location":"manual/methods.html#Note-on-Optional-and-keyword-Arguments","page":"Methods","title":"Note on Optional and keyword Arguments","category":"section","text":"As mentioned briefly in Functions, optional arguments are implemented as syntax for multiple method definitions. For example, this definition:\n\nf(a=1,b=2) = a+2b\n\ntranslates to the following three methods:\n\nf(a,b) = a+2b\nf(a) = f(a,2)\nf() = f(1,2)\n\nThis means that calling f() is equivalent to calling f(1,2). In this case the result is 5, because f(1,2) invokes the first method of f above. However, this need not always be the case. If you define a fourth method that is more specialized for integers:\n\nf(a::Int,b::Int) = a-2b\n\nthen the result of both f() and f(1,2) is -3. In other words, optional arguments are tied to a function, not to any specific method of that function. It depends on the types of the optional arguments which method is invoked. When optional arguments are defined in terms of a global variable, the type of the optional argument may even change at run-time.\n\nKeyword arguments behave quite differently from ordinary positional arguments. In particular, they do not participate in method dispatch. Methods are dispatched based only on positional arguments, with keyword arguments processed after the matching method is identified."},{"location":"manual/methods.html#Function-like-objects","page":"Methods","title":"Function-like objects","category":"section","text":"Methods are associated with types, so it is possible to make any arbitrary Julia object \"callable\" by adding methods to its type.\n\nFor example, you can define a type that stores the coefficients of a polynomial, but behaves like a function evaluating the polynomial:\n\njulia> struct Polynomial{R}\n           coeffs::Vector{R}\n       end\n\njulia> function (p::Polynomial)(x)\n           v = p.coeffs[end]\n           for i = (length(p.coeffs)-1):-1:1\n               v = v*x + p.coeffs[i]\n           end\n           return v\n       end\n\njulia> (p::Polynomial)() = p(5)\n\nNotice that the function is specified by type instead of by name. As with normal functions there is a terse syntax form. In the function body, p will refer to the object that was called. A Polynomial can be used as follows:\n\njulia> poly = Polynomial([1,10,100])\nPolynomial{Int64}([1, 10, 100])\n\njulia> poly(3)\n931\n\njulia> poly()\n2551\n\nThis mechanism is also the key to how type constructors and closures (inner functions that refer to their surrounding environment) work in Julia."},{"location":"manual/methods.html#Empty-generic-functions","page":"Methods","title":"Empty generic functions","category":"section","text":"Occasionally it is useful to introduce a generic function without yet adding methods. This can be used to separate interface definitions from implementations. It might also be done for the purpose of documentation or code readability. The syntax for this is an empty function block without a tuple of arguments:\n\nfunction emptyfunc end"},{"location":"manual/methods.html#man-method-design-ambiguities","page":"Methods","title":"Method design and the avoidance of ambiguities","category":"section","text":"Julia's method polymorphism is one of its most powerful features, yet exploiting this power can pose design challenges. In particular, in more complex method hierarchies it is not uncommon for ambiguities to arise.\n\nAbove, it was pointed out that one can resolve ambiguities like\n\nf(x, y::Int) = 1\nf(x::Int, y) = 2\n\nby defining a method\n\nf(x::Int, y::Int) = 3\n\nThis is often the right strategy; however, there are circumstances where following this advice mindlessly can be counterproductive. In particular, the more methods a generic function has, the more possibilities there are for ambiguities. When your method hierarchies get more complicated than this simple example, it can be worth your while to think carefully about alternative strategies.\n\nBelow we discuss particular challenges and some alternative ways to resolve such issues."},{"location":"manual/methods.html#Tuple-and-NTuple-arguments","page":"Methods","title":"Tuple and NTuple arguments","category":"section","text":"Tuple (and NTuple) arguments present special challenges. For example,\n\nf(x::NTuple{N,Int}) where {N} = 1\nf(x::NTuple{N,Float64}) where {N} = 2\n\nare ambiguous because of the possibility that N == 0: there are no elements to determine whether the Int or Float64 variant should be called. To resolve the ambiguity, one approach is define a method for the empty tuple:\n\nf(x::Tuple{}) = 3\n\nAlternatively, for all methods but one you can insist that there is at least one element in the tuple:\n\nf(x::NTuple{N,Int}) where {N} = 1           # this is the fallback\nf(x::Tuple{Float64, Vararg{Float64}}) = 2   # this requires at least one Float64"},{"location":"manual/methods.html#man-methods-orthogonalize","page":"Methods","title":"Orthogonalize your design","category":"section","text":"When you might be tempted to dispatch on two or more arguments, consider whether a \"wrapper\" function might make for a simpler design. For example, instead of writing multiple variants:\n\nf(x::A, y::A) = ...\nf(x::A, y::B) = ...\nf(x::B, y::A) = ...\nf(x::B, y::B) = ...\n\nyou might consider defining\n\nf(x::A, y::A) = ...\nf(x, y) = f(g(x), g(y))\n\nwhere g converts the argument to type A. This is a very specific example of the more general principle of orthogonal design, in which separate concepts are assigned to separate methods. Here, g will most likely need a fallback definition\n\ng(x::A) = x\n\nA related strategy exploits promote to bring x and y to a common type:\n\nf(x::T, y::T) where {T} = ...\nf(x, y) = f(promote(x, y)...)\n\nOne risk with this design is the possibility that if there is no suitable promotion method converting x and y to the same type, the second method will recurse on itself infinitely and trigger a stack overflow."},{"location":"manual/methods.html#Dispatch-on-one-argument-at-a-time","page":"Methods","title":"Dispatch on one argument at a time","category":"section","text":"If you need to dispatch on multiple arguments, and there are many fallbacks with too many combinations to make it practical to define all possible variants, then consider introducing a \"name cascade\" where (for example) you dispatch on the first argument and then call an internal method:\n\nf(x::A, y) = _fA(x, y)\nf(x::B, y) = _fB(x, y)\n\nThen the internal methods _fA and _fB can dispatch on y without concern about ambiguities with each other with respect to x.\n\nBe aware that this strategy has at least one major disadvantage: in many cases, it is not possible for users to further customize the behavior of f by defining further specializations of your exported function f. Instead, they have to define specializations for your internal methods _fA and _fB, and this blurs the lines between exported and internal methods."},{"location":"manual/methods.html#Abstract-containers-and-element-types","page":"Methods","title":"Abstract containers and element types","category":"section","text":"Where possible, try to avoid defining methods that dispatch on specific element types of abstract containers. For example,\n\n-(A::AbstractArray{T}, b::Date) where {T<:Date}\n\ngenerates ambiguities for anyone who defines a method\n\n-(A::MyArrayType{T}, b::T) where {T}\n\nThe best approach is to avoid defining either of these methods: instead, rely on a generic method -(A::AbstractArray, b) and make sure this method is implemented with generic calls (like similar and -) that do the right thing for each container type and element type separately. This is just a more complex variant of the advice to orthogonalize your methods.\n\nWhen this approach is not possible, it may be worth starting a discussion with other developers about resolving the ambiguity; just because one method was defined first does not necessarily mean that it can't be modified or eliminated. As a last resort, one developer can define the \"band-aid\" method\n\n-(A::MyArrayType{T}, b::Date) where {T<:Date} = ...\n\nthat resolves the ambiguity by brute force."},{"location":"manual/methods.html#Complex-method-\"cascades\"-with-default-arguments","page":"Methods","title":"Complex method \"cascades\" with default arguments","category":"section","text":"If you are defining a method \"cascade\" that supplies defaults, be careful about dropping any arguments that correspond to potential defaults. For example, suppose you're writing a digital filtering algorithm and you have a method that handles the edges of the signal by applying padding:\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel)  # now perform the \"real\" computation\nend\n\nThis will run afoul of a method that supplies default padding:\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate()) # replicate the edge by default\n\nTogether, these two methods generate an infinite recursion with A constantly growing bigger.\n\nThe better design would be to define your call hierarchy like this:\n\nstruct NoPad end  # indicate that no padding is desired, or that it's already applied\n\nmyfilter(A, kernel) = myfilter(A, kernel, Replicate())  # default boundary conditions\n\nfunction myfilter(A, kernel, ::Replicate)\n    Apadded = replicate_edges(A, size(kernel))\n    myfilter(Apadded, kernel, NoPad())  # indicate the new boundary conditions\nend\n\n# other padding methods go here\n\nfunction myfilter(A, kernel, ::NoPad)\n    # Here's the \"real\" implementation of the core computation\nend\n\nNoPad is supplied in the same argument position as any other kind of padding, so it keeps the dispatch hierarchy well organized and with reduced likelihood of ambiguities. Moreover, it extends the \"public\" myfilter interface: a user who wants to control the padding explicitly can call the NoPad variant directly."},{"location":"manual/methods.html#Defining-methods-in-local-scope","page":"Methods","title":"Defining methods in local scope","category":"section","text":"You can define methods within a local scope, for example\n\njulia> function f(x)\n           g(y::Int) = y + x\n           g(y) = y - x\n           g\n       end\nf (generic function with 1 method)\n\njulia> h = f(3);\n\njulia> h(4)\n7\n\njulia> h(4.0)\n1.0\n\nHowever, you should not define local methods conditionally or subject to control flow, as in\n\nfunction f2(inc)\n    if inc\n        g(x) = x + 1\n    else\n        g(x) = x - 1\n    end\nend\n\nfunction f3()\n    function g end\n    return g\n    g() = 0\nend\n\nas it is not clear what function will end up getting defined. In the future, it might be an error to define local methods in this manner.\n\nFor cases like this use anonymous functions instead:\n\nfunction f2(inc)\n    g = if inc\n        x -> x + 1\n    else\n        x -> x - 1\n    end\nend\n\n[Clarke61]: Arthur C. Clarke, Profiles of the Future (1961): Clarke's Third Law."},{"location":"manual/style-guide.html#Style-Guide","page":"Style Guide","title":"Style Guide","category":"section","text":"The following sections explain a few aspects of idiomatic Julia coding style. None of these rules are absolute; they are only suggestions to help familiarize you with the language and to help you choose among alternative designs."},{"location":"manual/style-guide.html#Indentation","page":"Style Guide","title":"Indentation","category":"section","text":"Use 4 spaces per indentation level."},{"location":"manual/style-guide.html#Write-functions,-not-just-scripts","page":"Style Guide","title":"Write functions, not just scripts","category":"section","text":"Writing code as a series of steps at the top level is a quick way to get started solving a problem, but you should try to divide a program into functions as soon as possible. Functions are more reusable and testable, and clarify what steps are being done and what their inputs and outputs are. Furthermore, code inside functions tends to run much faster than top level code, due to how Julia's compiler works.\n\nIt is also worth emphasizing that functions should take arguments, instead of operating directly on global variables (aside from constants like pi)."},{"location":"manual/style-guide.html#Write-docstrings","page":"Style Guide","title":"Write docstrings","category":"section","text":"Comments describing an object should typically be written as docstrings for editor and REPL accessibility. Inline comments (# comment) and multiline comments (#= comment =#) are appropriate for information that is intended only for the reader of the code (as opposed to a user)."},{"location":"manual/style-guide.html#Avoid-writing-overly-specific-types","page":"Style Guide","title":"Avoid writing overly-specific types","category":"section","text":"Code should be as generic as possible. Instead of writing:\n\nComplex{Float64}(x)\n\nit's better to use available generic functions:\n\ncomplex(float(x))\n\nThe second version will convert x to an appropriate type, instead of always the same type.\n\nThis style point is especially relevant to function arguments. For example, don't declare an argument to be of type Int or Int32 if it really could be any integer, expressed with the abstract type Integer. In fact, in many cases you can omit the argument type altogether, unless it is needed to disambiguate from other method definitions, since a MethodError will be thrown anyway if a type is passed that does not support any of the requisite operations. (This is known as duck typing.)\n\nFor example, consider the following definitions of a function addone that returns one plus its argument:\n\naddone(x::Int) = x + 1                 # works only for Int\naddone(x::Integer) = x + oneunit(x)    # any integer type\naddone(x::Number) = x + oneunit(x)     # any numeric type\naddone(x) = x + oneunit(x)             # any type supporting + and oneunit\n\nThe last definition of addone handles any type supporting oneunit (which returns 1 in the same type as x, which avoids unwanted type promotion) and the + function with those arguments. The key thing to realize is that there is no performance penalty to defining only the general addone(x) = x + oneunit(x), because Julia will automatically compile specialized versions as needed. For example, the first time you call addone(12), Julia will automatically compile a specialized addone function for x::Int arguments, with the call to oneunit replaced by its inlined value 1. Therefore, the first three definitions of addone above are completely redundant with the fourth definition."},{"location":"manual/style-guide.html#Handle-excess-argument-diversity-in-the-caller","page":"Style Guide","title":"Handle excess argument diversity in the caller","category":"section","text":"Instead of:\n\nfunction foo(x, y)\n    x = Int(x); y = Int(y)\n    ...\nend\nfoo(x, y)\n\nuse:\n\nfunction foo(x::Int, y::Int)\n    ...\nend\nfoo(Int(x), Int(y))\n\nThis is better style because foo does not really accept numbers of all types; it really needs Int s.\n\nOne issue here is that if a function inherently requires integers, it might be better to force the caller to decide how non-integers should be converted (e.g. floor or ceiling). Another issue is that declaring more specific types leaves more \"space\" for future method definitions."},{"location":"manual/style-guide.html#bang-convention","page":"Style Guide","title":"Append ! to names of functions that modify their arguments","category":"section","text":"Instead of:\n\nfunction double(a::AbstractArray{<:Number})\n    for i in eachindex(a)\n        a[i] *= 2\n    end\n    return a\nend\n\nuse:\n\nfunction double!(a::AbstractArray{<:Number})\n    for i in eachindex(a)\n        a[i] *= 2\n    end\n    return a\nend\n\nJulia Base uses this convention throughout and contains examples of functions with both copying and modifying forms (e.g., sort and sort!), and others which are just modifying (e.g., push!, pop!, splice!). It is typical for such functions to also return the modified array for convenience.\n\nFunctions related to IO or making use of random number generators (RNG) are notable exceptions: Since these functions almost invariably must mutate the IO or RNG, functions ending with ! are used to signify a mutation other than mutating the IO or advancing the RNG state. For example, rand(x) mutates the RNG, whereas rand!(x) mutates both the RNG and x; similarly, read(io) mutates io, whereas read!(io, x) mutates both arguments."},{"location":"manual/style-guide.html#Avoid-strange-type-Unions","page":"Style Guide","title":"Avoid strange type Unions","category":"section","text":"Types such as Union{Function,AbstractString} are often a sign that some design could be cleaner."},{"location":"manual/style-guide.html#Avoid-elaborate-container-types","page":"Style Guide","title":"Avoid elaborate container types","category":"section","text":"It is usually not much help to construct arrays like the following:\n\na = Vector{Union{Int,AbstractString,Tuple,Array}}(undef, n)\n\nIn this case Vector{Any}(undef, n) is better. It is also more helpful to the compiler to annotate specific uses (e.g. a[i]::Int) than to try to pack many alternatives into one type."},{"location":"manual/style-guide.html#Prefer-exported-methods-over-direct-field-access","page":"Style Guide","title":"Prefer exported methods over direct field access","category":"section","text":"Idiomatic Julia code should generally treat a module's exported methods as the interface to its types. An object's fields are generally considered implementation details and user code should only access them directly if this is stated to be the API. This has several benefits:\n\nPackage developers are freer to change the implementation without breaking user code.\nMethods can be passed to higher-order constructs like map (e.g. map(imag, zs)) rather than [z.im for z in zs]).\nMethods can be defined on abstract types.\nMethods can describe a conceptual operation that can be shared across disparate types (e.g. real(z) works on Complex numbers or Quaternions).\n\nJulia's dispatch system encourages this style because play(x::MyType) only defines the play method on that particular type, leaving other types to have their own implementation.\n\nSimilarly, non-exported functions are typically internal and subject to change, unless the documentations states otherwise. Names sometimes are given a _ prefix (or suffix) to further suggest that something is \"internal\" or an implementation-detail, but it is not a rule.\n\nCounter-examples to this rule include NamedTuple, RegexMatch, StatStruct."},{"location":"manual/style-guide.html#Use-naming-conventions-consistent-with-Julia-base/","page":"Style Guide","title":"Use naming conventions consistent with Julia base/","category":"section","text":"modules and type names use capitalization and camel case: module SparseArrays, struct UnitRange.\nconstants use all uppercase and underscores (LOAD_PATH, VERSION).\nwhile anything not marked with public or export is considered internal, a prefix of _ also indicates that an object is not intended for public use.\nfunctions mutating at least one of their arguments end in !.\nconciseness is valued, but avoid abbreviation (indexin rather than indxin) as it becomes difficult to remember whether and how particular words are abbreviated.\n\nIf a function name requires multiple words, consider whether it might represent more than one concept and might be better split into pieces.\n\nFunction names should be written in snake case (minimum, count_zeros, escape_string). Base often breaks this convention by squashing words together (splitpath, readeach) but this style is not recommended for packages."},{"location":"manual/style-guide.html#Write-functions-with-argument-ordering-similar-to-Julia-Base","page":"Style Guide","title":"Write functions with argument ordering similar to Julia Base","category":"section","text":"As a general rule, the Base library uses the following order of arguments to functions, as applicable:\n\nFunction argument. Putting a function argument first permits the use of do blocks for passing multiline anonymous functions.\nI/O stream. Specifying the IO object first permits passing the function to functions such as sprint, e.g. sprint(show, x).\nInput being mutated. For example, in fill!(x, v), x is the object being mutated and it appears before the value to be inserted into x.\nType. Passing a type typically means that the output will have the given type. In parse(Int, \"1\"), the type comes before the string to parse. There are many such examples where the type appears first, but it's useful to note that in read(io, String), the IO argument appears before the type, which is in keeping with the order outlined here.\nInput not being mutated. In fill!(x, v), v is not being mutated and it comes after x.\nKey. For associative collections, this is the key of the key-value pair(s). For other indexed collections, this is the index.\nValue. For associative collections, this is the value of the key-value pair(s). In cases like fill!(x, v), this is v.\nEverything else. Any other arguments.\nVarargs. This refers to arguments that can be listed indefinitely at the end of a function call. For example, in Matrix{T}(undef, dims), the dimensions can be given as a Tuple, e.g. Matrix{T}(undef, (1,2)), or as Varargs, e.g. Matrix{T}(undef, 1, 2).\nKeyword arguments. In Julia keyword arguments have to come last anyway in function definitions; they're listed here for the sake of completeness.\n\nThe vast majority of functions will not take every kind of argument listed above; the numbers merely denote the precedence that should be used for any applicable arguments to a function.\n\nThere are of course a few exceptions. For example, in convert, the type should always come first. In setindex!, the value comes before the indices so that the indices can be provided as varargs.\n\nWhen designing APIs, adhering to this general order as much as possible is likely to give users of your functions a more consistent experience."},{"location":"manual/style-guide.html#Don't-overuse-try-catch","page":"Style Guide","title":"Don't overuse try-catch","category":"section","text":"It is better to avoid errors than to rely on catching them."},{"location":"manual/style-guide.html#Don't-parenthesize-conditions","page":"Style Guide","title":"Don't parenthesize conditions","category":"section","text":"Julia doesn't require parens around conditions in if and while. Write:\n\nif a == b\n\ninstead of:\n\nif (a == b)"},{"location":"manual/style-guide.html#Don't-overuse-...","page":"Style Guide","title":"Don't overuse ...","category":"section","text":"Splicing function arguments can be addictive. Instead of [a..., b...], use simply [a; b], which already concatenates arrays. collect(a) is better than [a...], but since a is already iterable it is often even better to leave it alone, and not convert it to an array."},{"location":"manual/style-guide.html#Ensure-constructors-return-an-instance-of-their-own-type","page":"Style Guide","title":"Ensure constructors return an instance of their own type","category":"section","text":"When a method T(x) is called on a type T, it is generally expected to return a value of type T. Defining a constructor that returns an unexpected type can lead to confusing and unpredictable behavior:\n\njulia> struct Foo{T}\n           x::T\n       end\n\njulia> Base.Float64(foo::Foo) = Foo(Float64(foo.x))  # Do not define methods like this\n\njulia> Float64(Foo(3))  # Should return `Float64`\nFoo{Float64}(3.0)\n\njulia> Foo{Int}(x) = Foo{Float64}(x)  # Do not define methods like this\n\njulia> Foo{Int}(3)  # Should return `Foo{Int}`\nFoo{Float64}(3.0)\n\nTo maintain code clarity and ensure type consistency, always design constructors to return an instance of the type they are supposed to construct."},{"location":"manual/style-guide.html#Don't-use-unnecessary-static-parameters","page":"Style Guide","title":"Don't use unnecessary static parameters","category":"section","text":"A function signature:\n\nfoo(x::T) where {T<:Real} = ...\n\nshould be written as:\n\nfoo(x::Real) = ...\n\ninstead, especially if T is not used in the function body. Even if T is used, it can be replaced with typeof(x) if convenient. There is no performance difference. Note that this is not a general caution against static parameters, just against uses where they are not needed.\n\nNote also that container types, specifically may need type parameters in function calls. See the FAQ Avoid fields with abstract containers for more information."},{"location":"manual/style-guide.html#Avoid-confusion-about-whether-something-is-an-instance-or-a-type","page":"Style Guide","title":"Avoid confusion about whether something is an instance or a type","category":"section","text":"Sets of definitions like the following are confusing:\n\nfoo(::Type{MyType}) = ...\nfoo(::MyType) = foo(MyType)\n\nDecide whether the concept in question will be written as MyType or MyType(), and stick to it.\n\nThe preferred style is to use instances by default, and only add methods involving Type{MyType} later if they become necessary to solve some problems.\n\nIf a type is effectively an enumeration, it should be defined as a single (ideally immutable struct or primitive) type, with the enumeration values being instances of it. Constructors and conversions can check whether values are valid. This design is preferred over making the enumeration an abstract type, with the \"values\" as subtypes."},{"location":"manual/style-guide.html#Don't-overuse-macros","page":"Style Guide","title":"Don't overuse macros","category":"section","text":"Be aware of when a macro could really be a function instead.\n\nCalling eval inside a macro is a particularly dangerous warning sign; it means the macro will only work when called at the top level. If such a macro is written as a function instead, it will naturally have access to the run-time values it needs."},{"location":"manual/style-guide.html#Don't-expose-unsafe-operations-at-the-interface-level","page":"Style Guide","title":"Don't expose unsafe operations at the interface level","category":"section","text":"If you have a type that uses a native pointer:\n\nmutable struct NativeType\n    p::Ptr{UInt8}\n    ...\nend\n\ndon't write definitions like the following:\n\ngetindex(x::NativeType, i) = unsafe_load(x.p, i)\n\nThe problem is that users of this type can write x[i] without realizing that the operation is unsafe, and then be susceptible to memory bugs.\n\nSuch a function should either check the operation to ensure it is safe, or have unsafe somewhere in its name to alert callers."},{"location":"manual/style-guide.html#Don't-overload-methods-of-base-container-types","page":"Style Guide","title":"Don't overload methods of base container types","category":"section","text":"It is possible to write definitions like the following:\n\nshow(io::IO, v::Vector{MyType}) = ...\n\nThis would provide custom showing of vectors with a specific new element type. While tempting, this should be avoided. The trouble is that users will expect a well-known type like Vector() to behave in a certain way, and overly customizing its behavior can make it harder to work with."},{"location":"manual/style-guide.html#avoid-type-piracy","page":"Style Guide","title":"Avoid type piracy","category":"section","text":"\"Type piracy\" refers to the practice of extending or redefining methods in Base or other packages on types that you have not defined. In extreme cases, you can crash Julia (e.g. if your method extension or redefinition causes invalid input to be passed to a ccall). Type piracy can complicate reasoning about code, and may introduce incompatibilities that are hard to predict and diagnose.\n\nAs an example, suppose you wanted to define multiplication on symbols in a module:\n\nmodule A\nimport Base.*\n*(x::Symbol, y::Symbol) = Symbol(x,y)\nend\n\nThe problem is that now any other module that uses Base.* will also see this definition. Since Symbol is defined in Base and is used by other modules, this can change the behavior of unrelated code unexpectedly. There are several alternatives here, including using a different function name, or wrapping the Symbols in another type that you define.\n\nSometimes, coupled packages may engage in type piracy to separate features from definitions, especially when the packages were designed by collaborating authors, and when the definitions are reusable. For example, one package might provide some types useful for working with colors; another package could define methods for those types that enable conversions between color spaces. Another example might be a package that acts as a thin wrapper for some C code, which another package might then pirate to implement a higher-level, Julia-friendly API."},{"location":"manual/style-guide.html#Be-careful-with-type-equality","page":"Style Guide","title":"Be careful with type equality","category":"section","text":"You generally want to use isa and <: for testing types, not ==. Checking types for exact equality typically only makes sense when comparing to a known concrete type (e.g. T == Float64), or if you really, really know what you're doing."},{"location":"manual/style-guide.html#Don't-write-a-trivial-anonymous-function-x-f(x)-for-a-named-function-f","page":"Style Guide","title":"Don't write a trivial anonymous function x->f(x) for a named function f","category":"section","text":"Since higher-order functions are often called with anonymous functions, it is easy to conclude that this is desirable or even necessary. But any function can be passed directly, without being \"wrapped\" in an anonymous function. Instead of writing map(x->f(x), a), write map(f, a)."},{"location":"manual/style-guide.html#Avoid-using-floats-for-numeric-literals-in-generic-code-when-possible","page":"Style Guide","title":"Avoid using floats for numeric literals in generic code when possible","category":"section","text":"If you write generic code which handles numbers, and which can be expected to run with many different numeric type arguments, try using literals of a numeric type that will affect the arguments as little as possible through promotion.\n\nFor example,\n\njulia> f(x) = 2.0 * x\nf (generic function with 1 method)\n\njulia> f(1//2)\n1.0\n\njulia> f(1/2)\n1.0\n\njulia> f(1)\n2.0\n\nwhile\n\njulia> g(x) = 2 * x\ng (generic function with 1 method)\n\njulia> g(1//2)\n1//1\n\njulia> g(1/2)\n1.0\n\njulia> g(1)\n2\n\nAs you can see, the second version, where we used an Int literal, preserved the type of the input argument, while the first didn't. This is because e.g. promote_type(Int, Float64) == Float64, and promotion happens with the multiplication. Similarly, Rational literals are less type disruptive than Float64 literals, but more disruptive than Ints:\n\njulia> h(x) = 2//1 * x\nh (generic function with 1 method)\n\njulia> h(1//2)\n1//1\n\njulia> h(1/2)\n1.0\n\njulia> h(1)\n2//1\n\nThus, use Int literals when possible, with Rational{Int} for literal non-integer numbers, in order to make it easier to use your code."},{"location":"devdocs/pkgimg.html#pkgimages","page":"Package Images","title":"Package Images","category":"section","text":"Julia package images provide object (native code) caches for Julia packages. They are similar to Julia's system image and support many of the same features. In fact the underlying serialization format is the same, and the system image is the base image that the package images are build against."},{"location":"devdocs/pkgimg.html#High-level-overview","page":"Package Images","title":"High-level overview","category":"section","text":"Package images are shared libraries that contain both code and data. Like .ji cache files, they are generated per package. The data section contains both global data (global variables in the package) as well as the necessary metadata about what methods and types are defined by the package. The code section contains native objects that cache the final output of Julia's LLVM-based compiler.\n\nThe command line option --pkgimages=no can be used to turn off object caching for this session. Note that this means that cache files have to likely be regenerated. See JULIA_MAX_NUM_PRECOMPILE_FILES for the upper limit of variants Julia caches per default.\n\nnote: Note\nWhile the package images present themselves as native shared libraries, they are only an approximation thereof. You will not be able to link against them from a native program and they must be loaded from Julia."},{"location":"devdocs/pkgimg.html#Linking","page":"Package Images","title":"Linking","category":"section","text":"Since the package images contain native code, we must run a linker over them before we can use them. You can set the environment variable JULIA_VERBOSE_LINKING to true to make the package image linking process verbose.\n\nFurthermore, we cannot assume that the user has a working system linker installed. Therefore, Julia ships with LLD, the LLVM linker, to provide a working out of the box experience. In base/linking.jl, we implement a limited interface to be able to link package images on all supported platforms."},{"location":"devdocs/pkgimg.html#Quirks","page":"Package Images","title":"Quirks","category":"section","text":"Despite LLD being a multi-platform linker, it does not provide a consistent interface across platforms. Furthermore, it is meant to be used from clang or another compiler driver, we therefore reimplement some of the logic from llvm-project/clang/lib/Driver/ToolChains. Thankfully one can use lld -flavor to set lld to the right platform"},{"location":"devdocs/pkgimg.html#Windows","page":"Package Images","title":"Windows","category":"section","text":"To avoid having to deal with link.exe we use -flavor gnu, effectively turning lld into a cross-linker from a mingw32 environment. Windows DLLs are required to contain a _DllMainCRTStartup function and to minimize our dependence on mingw32 libraries, we inject a stub definition ourselves."},{"location":"devdocs/pkgimg.html#MacOS","page":"Package Images","title":"MacOS","category":"section","text":"Dynamic libraries on macOS need to link against -lSystem. On recent macOS versions, -lSystem is only available for linking when Xcode is available. To that effect we link with -undefined dynamic_lookup."},{"location":"devdocs/pkgimg.html#pkgimgs-multi-versioning","page":"Package Images","title":"Package images optimized for multiple microarchitectures","category":"section","text":"Similar to multi-versioning for system images, package images support multi-versioning. This allows creating package caches that can run efficiently on different CPU architectures within the same environment.\n\nSee the JULIA_CPU_TARGET environment variable for more information on how to set the CPU target for package images."},{"location":"devdocs/pkgimg.html#Flags-that-impact-package-image-creation-and-selection","page":"Package Images","title":"Flags that impact package image creation and selection","category":"section","text":"These are the Julia command line flags that impact cache selection. Package images that were created with different flags will be rejected.\n\n-g, --debug-info: Exact match required since it changes code generation.\n--check-bounds: Exact match required since it changes code generation.\n--inline: Exact match required since it changes code generation.\n--pkgimages: To allow running without object caching enabled.\n-O, --optimize: Reject package images generated for a lower optimization level, but allow for higher optimization levels to be loaded."},{"location":"devdocs/require.html#Module-loading","page":"Module loading","title":"Module loading","category":"section","text":"Base.require is responsible for loading modules and it also manages the precompilation cache. It is the implementation of the import statement."},{"location":"devdocs/require.html#Experimental-features","page":"Module loading","title":"Experimental features","category":"section","text":"The features below are experimental and not part of the stable Julia API. Before building upon them inform yourself about the current thinking and whether they might change soon."},{"location":"devdocs/require.html#Package-loading-callbacks","page":"Module loading","title":"Package loading callbacks","category":"section","text":"It is possible to listen to the packages loaded by Base.require, by registering a callback.\n\nloaded_packages = Base.PkgId[]\ncallback = (pkg::Base.PkgId) -> push!(loaded_packages, pkg)\npush!(Base.package_callbacks, callback)\n\nUsing this would look something like:\n\njulia> using Example\n\njulia> loaded_packages\n1-element Vector{Base.PkgId}:\n Example [7876af07-990d-54b4-ab0e-23690620f79a]"},{"location":"devdocs/build/linux.html#Linux","page":"Linux","title":"Linux","category":"section","text":"GCC version 4.7 or later is required to build Julia.\nTo use external shared libraries not in the system library search path, set USE_SYSTEM_XXX=1 and LDFLAGS=-Wl,-rpath,/path/to/dir/contains/libXXX.so in Make.user.\nInstead of setting LDFLAGS, putting the library directory into the environment variable LD_LIBRARY_PATH (at both compile and run time) also works.\nThe USE_SYSTEM_* flags should be used with caution. These are meant only for troubleshooting, porting, and packaging, where package maintainers work closely with the Julia developers to make sure that Julia is built correctly. Production use cases should use the officially provided binaries. Issues arising from the use of these flags will generally not be accepted.\nSee also the external dependencies."},{"location":"devdocs/build/linux.html#Architecture-Customization","page":"Linux","title":"Architecture Customization","category":"section","text":"Julia can be built for a non-generic architecture by configuring the ARCH Makefile variable in a Make.user file. See the appropriate section of Make.inc for additional customization options, such as MARCH and JULIA_CPU_TARGET.\n\nFor example, to build for Pentium 4, set MARCH=pentium4 and install the necessary system libraries for linking. On Ubuntu, these may include lib32gfortran-6-dev, lib32gcc1, and lib32stdc++6, among others.\n\nYou can also set MARCH=native in Make.user for a maximum-performance build customized for the current machine CPU."},{"location":"devdocs/build/linux.html#Linux-Build-Troubleshooting","page":"Linux","title":"Linux Build Troubleshooting","category":"section","text":"Problem Possible Solution\nOpenBLAS build failure Set one of the following build options in Make.user and build again: <ul><li> OPENBLAS_TARGET_ARCH=BARCELONA (AMD CPUs) or OPENBLAS_TARGET_ARCH=NEHALEM (Intel CPUs)<ul>Set OPENBLAS_DYNAMIC_ARCH = 0 to disable compiling multiple architectures in a single binary.</ul></li><li> OPENBLAS_NO_AVX2 = 1 disables AVX2 instructions, allowing OpenBLAS to compile with OPENBLAS_DYNAMIC_ARCH = 1 using old versions of binutils </li><li> USE_SYSTEM_BLAS=1 uses the system provided libblas <ul><li>Set LIBBLAS=-lopenblas and LIBBLASNAME=libopenblas to force the use of the system provided OpenBLAS when multiple BLAS versions are installed. </li></ul></li></ul><p> If you get an error that looks like ../kernel/x86_64/dgemm_kernel_4x4_haswell.S:1709: Error: no such instruction: `vpermpd $ 0xb1,%ymm0,%ymm0', then you need to set OPENBLAS_DYNAMIC_ARCH = 0 or OPENBLAS_NO_AVX2 = 1, or you need a newer version of binutils (2.18 or newer). (Issue #7653)</p><p> If the linker cannot find gfortran and you get an error like julia /usr/bin/x86_64-linux-gnu-ld: cannot find -lgfortran, check the path with gfortran -print-file-name=libgfortran.so and use the output to export something similar to this: export LDFLAGS=-L/usr/lib/gcc/x86_64-linux-gnu/8/. See Issue #6150.</p>\nIllegal Instruction error Check if your CPU supports AVX while your OS does not (e.g. through virtualization, as described in this issue)."},{"location":"devdocs/contributing/code-changes.html#Code-changes","page":"Code changes","title":"Code changes","category":"section","text":""},{"location":"devdocs/contributing/code-changes.html#Contributing-to-core-functionality-or-base-libraries","page":"Code changes","title":"Contributing to core functionality or base libraries","category":"section","text":"By contributing code to Julia, you are agreeing to release it under the MIT License.\n\nThe Julia community uses GitHub issues to track and discuss problems, feature requests, and pull requests (PR).\n\nIssues and pull requests should have self explanatory titles such that they can be understood from the list of PRs and Issues. i.e. Add {feature} and Fix {bug} are good, Fix #12345. Corrects the bug. is bad.\n\nYou can make pull requests for incomplete features to get code review. The convention is to open these as draft PRs and prefix the pull request title with \"WIP:\" for Work In Progress, or \"RFC:\" for Request for Comments when work is completed and ready for merging. This will prevent accidental merging of work that is in progress.\n\nNote: These instructions are for adding to or improving functionality in the base library. Before getting started, it can be helpful to discuss the proposed changes or additions on the Julia Discourse forum or in a GitHub issue—it's possible your proposed change belongs in a package rather than the core language. Also, keep in mind that changing stuff in the base can potentially break a lot of things. Finally, because of the time required to build Julia, note that it's usually faster to develop your code in stand-alone files, get it working, and then migrate it into the base libraries.\n\nAdd new code to Julia's base libraries as follows (this is the \"basic\" approach; see a more efficient approach in the next section):\n\nEdit the appropriate file in the base/ directory, or add new files if necessary. Create tests for your functionality and add them to files in the test/ directory. If you're editing C or Scheme code, most likely it lives in src/ or one of its subdirectories, although some aspects of Julia's REPL initialization live in cli/.\nAdd any new files to sysimg.jl in order to build them into the Julia system image.\nAdd any necessary export symbols in exports.jl.\nInclude your tests in test/Makefile and test/choosetests.jl.\n\nBuild as usual, and do make clean testall to test your contribution. If your contribution includes changes to Makefiles or external dependencies, make sure you can build Julia from a clean tree using git clean -fdx or equivalent (be careful – this command will delete any files lying around that aren't checked into git)."},{"location":"devdocs/contributing/code-changes.html#Running-specific-tests","page":"Code changes","title":"Running specific tests","category":"section","text":"There are make targets for running specific tests:\n\nmake test-bitarray\n\nYou can also use the runtests.jl script, e.g. to run test/bitarray.jl and test/math.jl:\n\n./usr/bin/julia test/runtests.jl bitarray math"},{"location":"devdocs/contributing/code-changes.html#Modifying-base-more-efficiently-with-Revise.jl","page":"Code changes","title":"Modifying base more efficiently with Revise.jl","category":"section","text":"Revise is a package that tracks changes in source files and automatically updates function definitions in your running Julia session. Using it, you can make extensive changes to Base without needing to rebuild in order to test your changes.\n\nHere is the standard procedure:\n\nIf you are planning changes to any types or macros, make those changes and build julia using make. (This is necessary because Revise cannot handle changes to type definitions or macros.) Unless it's required to get Julia to build, you do not have to add any functionality based on the new types, just the type definitions themselves.\nStart a Julia REPL session. Then issue the following commands:\n\nusing Revise    # if you aren't launching it in your `.julia/config/startup.jl`\nRevise.track(Base)\n\nEdit files in base/, save your edits, and test the functionality.\n\nIf you need to restart your Julia session, just start at step 2 above. Revise.track(Base) will note any changes from when Julia was last built and incorporate them automatically. You only need to rebuild Julia if you made code-changes that Revise cannot handle.\n\nFor convenience, there are also test-revise-* targets for every test-* target that use Revise to load any modifications to Base into the current system image before running the corresponding test. This can be useful as a shortcut on the command line (since tests aren't always designed to be run outside the runtest harness)."},{"location":"devdocs/contributing/code-changes.html#Contributing-to-the-standard-library","page":"Code changes","title":"Contributing to the standard library","category":"section","text":"The standard library (stdlib) packages are baked into the Julia system image. When running the ordinary test workflow on the stdlib packages, the system image version overrides the version you are developing. To test stdlib packages, you can do the following steps:\n\nEdit the UUID field of the Project.toml in the stdlib package\nChange the current directory to the directory of the stdlib you are developing\nStart julia with julia --project=.\nYou can now test the package by running pkg> test in Pkg mode.\n\nBecause you changed the UUID, the package manager treats the stdlib package as different from the one in the system image, and the system image version will not override the package.\n\nBe sure to change the UUID value back before making the pull request."},{"location":"devdocs/contributing/code-changes.html#News-worthy-changes","page":"Code changes","title":"News-worthy changes","category":"section","text":"For new functionality and other substantial changes, add a brief summary to NEWS.md. The news item should cross reference the pull request (PR) parenthetically, in the form ([#pr]). To add the PR reference number, first create the PR, then push an additional commit updating NEWS.md with the PR reference number. We periodically run ./julia doc/NEWS-update.jl from the julia directory to update the cross-reference links, but this should not be done in a typical PR in order to avoid conflicting commits."},{"location":"devdocs/contributing/code-changes.html#Annotations-for-new-features,-deprecations-and-behavior-changes","page":"Code changes","title":"Annotations for new features, deprecations and behavior changes","category":"section","text":"API additions and deprecations, and minor behavior changes are allowed in minor version releases. For documented features that are part of the public API, a compatibility note should be added into the manual or the docstring. It should state the Julia minor version that changed the behavior and have a brief message describing the change.\n\nAt the moment, this should always be done with the following compat admonition (so that it would be possible to programmatically find the annotations in the future):\n\n!!! compat \"Julia 1.X\"       This method was added in Julia 1.X."},{"location":"base/c.html#C-Interface","page":"C Interface","title":"C Interface","category":"section","text":""},{"location":"base/c.html#LLVM-Interface","page":"C Interface","title":"LLVM Interface","category":"section","text":""},{"location":"base/c.html#Base.@ccall","page":"C Interface","title":"Base.@ccall","category":"macro","text":"@ccall library.function_name(argvalue1::argtype1, ...)::returntype\n@ccall function_name(argvalue1::argtype1, ...)::returntype\n@ccall $function_pointer(argvalue1::argtype1, ...)::returntype\n\nCall a function in a C-exported shared library, specified by library.function_name, where library is a string constant or literal. The library may be omitted, in which case the function_name is resolved in the current process. Alternatively, @ccall may also be used to call a function pointer $function_pointer, such as one returned by dlsym.\n\nEach argvalue to @ccall is converted to the corresponding argtype, by automatic insertion of calls to unsafe_convert(argtype, cconvert(argtype, argvalue)). (See also the documentation for unsafe_convert and cconvert for further details.) In most cases, this simply results in a call to convert(argtype, argvalue).\n\nExamples\n\n@ccall strlen(s::Cstring)::Csize_t\n\nThis calls the C standard library function:\n\nsize_t strlen(char *)\n\nwith a Julia variable named s. See also ccall.\n\nVarargs are supported with the following convention:\n\n@ccall printf(\"%s = %d\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint\n\nThe semicolon is used to separate required arguments (of which there must be at least one) from variadic arguments.\n\nExample using an external library:\n\n# C signature of g_uri_escape_string:\n# char *g_uri_escape_string(const char *unescaped, const char *reserved_chars_allowed, gboolean allow_utf8);\n\nconst glib = \"libglib-2.0\"\n@ccall glib.g_uri_escape_string(my_uri::Cstring, \":/\"::Cstring, true::Cint)::Cstring\n\nThe string literal could also be used directly before the function name, if desired \"libglib-2.0\".g_uri_escape_string(...\n\nIt's possible to declare the ccall as gc_safe by using the gc_safe = true option:\n\n@ccall gc_safe=true strlen(s::Cstring)::Csize_t\n\nThis allows the garbage collector to run concurrently with the ccall, which can be useful whenever the ccall may block outside of julia.\n\nwarning: Warning\nThis option should be used with caution, as it can lead to undefined behavior if the ccall calls back into the julia runtime. (@cfunction/@ccallables are safe however)\n\ncompat: Julia 1.12\nThe gc_safe argument requires Julia 1.12 or higher.\n\n\n\n\n\n"},{"location":"base/c.html#ccall","page":"C Interface","title":"ccall","category":"keyword","text":"ccall((function_name, library), returntype, (argtype1, ...), argvalue1, ...)\nccall(function_name, returntype, (argtype1, ...), argvalue1, ...)\nccall(function_pointer, returntype, (argtype1, ...), argvalue1, ...)\n\nCall a function in a C-exported shared library, specified by the tuple (function_name, library), where each component is either a string or symbol. Instead of specifying a library, one can also use a function_name symbol or string, which is resolved in the current process. Alternatively, ccall may also be used to call a function pointer function_pointer, such as one returned by dlsym.\n\nNote that the argument type tuple must be a literal tuple, and not a tuple-valued variable or expression.\n\nEach argvalue to the ccall will be converted to the corresponding argtype, by automatic insertion of calls to unsafe_convert(argtype, cconvert(argtype, argvalue)). (See also the documentation for unsafe_convert and cconvert for further details.) In most cases, this simply results in a call to convert(argtype, argvalue).\n\n\n\n\n\n"},{"location":"base/c.html#Core.Intrinsics.cglobal","page":"C Interface","title":"Core.Intrinsics.cglobal","category":"function","text":"cglobal((symbol, library) [, type=Cvoid])\n\nObtain a pointer to a global variable in a C-exported shared library, specified exactly as in ccall. Returns a Ptr{Type}, defaulting to Ptr{Cvoid} if no Type argument is supplied. The values can be read or written by unsafe_load or unsafe_store!, respectively.\n\n\n\n\n\n"},{"location":"base/c.html#Base.@cfunction","page":"C Interface","title":"Base.@cfunction","category":"macro","text":"@cfunction(callable, ReturnType, (ArgumentTypes...,)) -> Ptr{Cvoid}\n@cfunction($callable, ReturnType, (ArgumentTypes...,)) -> CFunction\n\nGenerate a C-callable function pointer from the Julia function callable for the given type signature. To pass the return value to a ccall, use the argument type Ptr{Cvoid} in the signature.\n\nNote that the argument type tuple must be a literal tuple, and not a tuple-valued variable or expression (although it can include a splat expression). And that these arguments will be evaluated in global scope during compile-time (not deferred until runtime). Adding a '$' in front of the function argument changes this to instead create a runtime closure over the local variable callable (this is not supported on all architectures).\n\nSee manual section on ccall and cfunction usage.\n\nExamples\n\njulia> function foo(x::Int, y::Int)\n           return x + y\n       end\n\njulia> @cfunction(foo, Int, (Int, Int))\nPtr{Cvoid} @0x000000001b82fcd0\n\n\n\n\n\n"},{"location":"base/c.html#Base.CFunction","page":"C Interface","title":"Base.CFunction","category":"type","text":"CFunction struct\n\nGarbage-collection handle for the return value from @cfunction when the first argument is annotated with '$'. Like all cfunction handles, it should be passed to ccall as a Ptr{Cvoid}, and will be converted automatically at the call site to the appropriate type.\n\nSee @cfunction.\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_convert","page":"C Interface","title":"Base.unsafe_convert","category":"function","text":"unsafe_convert(T, x)\n\nConvert x to a C argument of type T where the input x must be the return value of cconvert(T, ...).\n\nIn cases where convert would need to take a Julia object and turn it into a Ptr, this function should be used to define and perform that conversion.\n\nBe careful to ensure that a Julia reference to x exists as long as the result of this function will be used. Accordingly, the argument x to this function should never be an expression, only a variable name or field reference. For example, x=a.b.c is acceptable, but x=[a,b,c] is not.\n\nThe unsafe prefix on this function indicates that using the result of this function after the x argument to this function is no longer accessible to the program may cause undefined behavior, including program corruption or segfaults, at any later time.\n\nSee also cconvert\n\n\n\n\n\n"},{"location":"base/c.html#Base.cconvert","page":"C Interface","title":"Base.cconvert","category":"function","text":"cconvert(T,x)\n\nConvert x to a value to be passed to C code as type T, typically by calling convert(T, x).\n\nIn cases where x cannot be safely converted to T, unlike convert, cconvert may return an object of a type different from T, which however is suitable for unsafe_convert to handle. The result of this function should be kept valid (for the GC) until the result of unsafe_convert is not needed anymore. This can be used to allocate memory that will be accessed by the ccall. If multiple objects need to be allocated, a tuple of the objects can be used as return value.\n\nNeither convert nor cconvert should take a Julia object and turn it into a Ptr.\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_load","page":"C Interface","title":"Base.unsafe_load","category":"function","text":"unsafe_load(p::Ptr{T}, i::Integer=1)\nunsafe_load(p::Ptr{T}, order::Symbol)\nunsafe_load(p::Ptr{T}, i::Integer, order::Symbol)\n\nLoad a value of type T from the address of the ith element (1-indexed) starting at p. This is equivalent to the C expression p[i-1]. Optionally, an atomic memory ordering can be provided.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program or return garbage answers. Unlike C, dereferencing memory region allocated as different type may be valid provided that the types are compatible.\n\ncompat: Julia 1.10\nThe order argument is available as of Julia 1.10.\n\nSee also: atomic\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_store!","page":"C Interface","title":"Base.unsafe_store!","category":"function","text":"unsafe_store!(p::Ptr{T}, x, i::Integer=1)\nunsafe_store!(p::Ptr{T}, x, order::Symbol)\nunsafe_store!(p::Ptr{T}, x, i::Integer, order::Symbol)\n\nStore a value of type T to the address of the ith element (1-indexed) starting at p. This is equivalent to the C expression p[i-1] = x. Optionally, an atomic memory ordering can be provided.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program. Unlike C, storing memory region allocated as different type may be valid provided that the types are compatible.\n\ncompat: Julia 1.10\nThe order argument is available as of Julia 1.10.\n\nSee also: atomic\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_modify!","page":"C Interface","title":"Base.unsafe_modify!","category":"function","text":"unsafe_modify!(p::Ptr{T}, op, x, [order::Symbol])::Pair\n\nThese atomically perform the operations to get and set a memory address after applying the function op. If supported by the hardware (for example, atomic increment), this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p)\nz = op(y, x)\nunsafe_store!(p, z)\nreturn y => z\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: modifyproperty!, atomic\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_replace!","page":"C Interface","title":"Base.unsafe_replace!","category":"function","text":"unsafe_replace!(p::Ptr{T}, expected, desired,\n               [success_order::Symbol[, fail_order::Symbol=success_order]]) -> (; old, success::Bool)\n\nThese atomically perform the operations to get and conditionally set a memory address to a given value. If supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p, fail_order)\nok = y === expected\nif ok\n    unsafe_store!(p, desired, success_order)\nend\nreturn (; old = y, success = ok)\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: replaceproperty!, atomic\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_swap!","page":"C Interface","title":"Base.unsafe_swap!","category":"function","text":"unsafe_swap!(p::Ptr{T}, x, [order::Symbol])\n\nThese atomically perform the operations to simultaneously get and set a memory address. If supported by the hardware, this may be optimized to the appropriate hardware instruction, otherwise its execution will be similar to:\n\ny = unsafe_load(p)\nunsafe_store!(p, x)\nreturn y\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointer p to ensure that it is valid. Like C, the programmer is responsible for ensuring that referenced memory is not freed or garbage collected while invoking this function. Incorrect usage may segfault your program.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\nSee also: swapproperty!, atomic\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_copyto!-Union{Tuple{T}, Tuple{Ptr{T}, Ptr{T}, Any}} where T","page":"C Interface","title":"Base.unsafe_copyto!","category":"method","text":"unsafe_copyto!(dest::Ptr{T}, src::Ptr{T}, N)\n\nCopy N elements from a source pointer to a destination, with no checking. The size of an element is determined by the type of the pointers.\n\nThe unsafe prefix on this function indicates that no validation is performed on the pointers dest and src to ensure that they are valid. Incorrect usage may corrupt or segfault your program, in the same manner as C.\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_copyto!-Tuple{Array, Any, Array, Any, Any}","page":"C Interface","title":"Base.unsafe_copyto!","category":"method","text":"unsafe_copyto!(dest::Array, doffs, src::Array, soffs, n)\n\nCopy n elements from a source array to a destination, starting at the linear index soffs in the source and doffs in the destination (1-indexed).\n\nThe unsafe prefix on this function indicates that no validation is performed to ensure that n is inbounds on either array. Incorrect usage may corrupt or segfault your program, in the same manner as C.\n\n\n\n\n\n"},{"location":"base/c.html#Base.copyto!","page":"C Interface","title":"Base.copyto!","category":"function","text":"copyto!(dest, Rdest::CartesianIndices, src, Rsrc::CartesianIndices) -> dest\n\nCopy the block of src in the range of Rsrc to the block of dest in the range of Rdest. The sizes of the two regions must match.\n\nExamples\n\njulia> A = zeros(5, 5);\n\njulia> B = [1 2; 3 4];\n\njulia> Ainds = CartesianIndices((2:3, 2:3));\n\njulia> Binds = CartesianIndices(B);\n\njulia> copyto!(A, Ainds, B, Binds)\n5×5 Matrix{Float64}:\n 0.0  0.0  0.0  0.0  0.0\n 0.0  1.0  2.0  0.0  0.0\n 0.0  3.0  4.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n 0.0  0.0  0.0  0.0  0.0\n\n\n\n\n\ncopyto!(dest::AbstractArray, src) -> dest\n\nCopy all elements from collection src to array dest, whose length must be greater than or equal to the length n of src. The first n elements of dest are overwritten, the other elements are left untouched.\n\nSee also copy!, copy.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> x = [1., 0., 3., 0., 5.];\n\njulia> y = zeros(7);\n\njulia> copyto!(y, x);\n\njulia> y\n7-element Vector{Float64}:\n 1.0\n 0.0\n 3.0\n 0.0\n 5.0\n 0.0\n 0.0\n\n\n\n\n\ncopyto!(dest, doffs, src, soffs, n)\n\nCopy n elements from collection src starting at the linear index soffs, to array dest starting at the index doffs. Return dest.\n\n\n\n\n\ncopyto!(B::AbstractMatrix, ir_dest::AbstractUnitRange, jr_dest::AbstractUnitRange,\n        tM::AbstractChar,\n        M::AbstractVecOrMat, ir_src::AbstractUnitRange, jr_src::AbstractUnitRange) -> B\n\nEfficiently copy elements of matrix M to B conditioned on the character parameter tM as follows:\n\ntM Destination Source\n'N' B[ir_dest, jr_dest] M[ir_src, jr_src]\n'T' B[ir_dest, jr_dest] transpose(M)[ir_src, jr_src]\n'C' B[ir_dest, jr_dest] adjoint(M)[ir_src, jr_src]\n\nThe elements B[ir_dest, jr_dest] are overwritten. Furthermore, the index range parameters must satisfy length(ir_dest) == length(ir_src) and length(jr_dest) == length(jr_src).\n\nSee also copy_transpose! and copy_adjoint!.\n\n\n\n\n\ncopyto!(dest::AbstractMatrix, src::UniformScaling)\n\nCopies a UniformScaling onto a matrix.\n\ncompat: Julia 1.1\nIn Julia 1.0 this method only supported a square destination matrix. Julia 1.1. added support for a rectangular matrix.\n\n\n\n\n\n"},{"location":"base/c.html#Base.pointer","page":"C Interface","title":"Base.pointer","category":"function","text":"pointer(array [, index])\n\nGet the native address of an array or string, optionally at a given location index.\n\nThis function is \"unsafe\". Be careful to ensure that a Julia reference to array exists as long as this pointer will be used. The GC.@preserve macro should be used to protect the array argument from garbage collection within a given block of code.\n\nCalling Ref(array[, index]) is generally preferable to this function as it guarantees validity.\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_wrap-Union{Tuple{N}, Tuple{T}, Tuple{Union{Type{Array}, Type{Array{T}}, Type{Array{T, N}}}, Ptr{T}, NTuple{N, Int32}}} where {T, N}","page":"C Interface","title":"Base.unsafe_wrap","category":"method","text":"unsafe_wrap(Array, pointer::Ptr{T}, dims; own = false)\n\nWrap a Julia Array object around the data at the address given by pointer, without making a copy.  The pointer element type T determines the array element type. dims is either an integer (for a 1d array) or a tuple of the array dimensions. own optionally specifies whether Julia should take ownership of the memory, calling free on the pointer when the array is no longer referenced.\n\nThis function is labeled \"unsafe\" because it will crash if pointer is not a valid memory address to data of the requested length. Unlike unsafe_load and unsafe_store!, the programmer is responsible also for ensuring that the underlying data is not accessed through two arrays of different element type, similar to the strict aliasing rule in C.\n\n\n\n\n\n"},{"location":"base/c.html#Base.pointer_from_objref","page":"C Interface","title":"Base.pointer_from_objref","category":"function","text":"pointer_from_objref(x)\n\nGet the memory address of a Julia object as a Ptr. The existence of the resulting Ptr will not protect the object from garbage collection, so you must ensure that the object remains referenced for the whole time that the Ptr will be used.\n\nThis function may not be called on immutable objects, since they do not have stable memory addresses.\n\nSee also unsafe_pointer_to_objref.\n\n\n\n\n\n"},{"location":"base/c.html#Base.unsafe_pointer_to_objref","page":"C Interface","title":"Base.unsafe_pointer_to_objref","category":"function","text":"unsafe_pointer_to_objref(p::Ptr)\n\nConvert a Ptr to an object reference. Assumes the pointer refers to a valid heap-allocated Julia object. If this is not the case, undefined behavior results, hence this function is considered \"unsafe\" and should be used with care.\n\nSee also pointer_from_objref.\n\n\n\n\n\n"},{"location":"base/c.html#Base.disable_sigint","page":"C Interface","title":"Base.disable_sigint","category":"function","text":"disable_sigint(f::Function)\n\nDisable Ctrl-C handler during execution of a function on the current task, for calling external code that may call julia code that is not interrupt safe. Intended to be called using do block syntax as follows:\n\ndisable_sigint() do\n    # interrupt-unsafe code\n    ...\nend\n\nThis is not needed on worker threads (Threads.threadid() != 1) since the InterruptException will only be delivered to the master thread. External functions that do not call julia code or julia runtime automatically disable sigint during their execution.\n\n\n\n\n\n"},{"location":"base/c.html#Base.reenable_sigint","page":"C Interface","title":"Base.reenable_sigint","category":"function","text":"reenable_sigint(f::Function)\n\nRe-enable Ctrl-C handler during execution of a function. Temporarily reverses the effect of disable_sigint.\n\n\n\n\n\n"},{"location":"base/c.html#Base.exit_on_sigint","page":"C Interface","title":"Base.exit_on_sigint","category":"function","text":"exit_on_sigint(on::Bool)\n\nSet exit_on_sigint flag of the julia runtime.  If false, Ctrl-C (SIGINT) is capturable as InterruptException in try block. This is the default behavior in REPL, any code run via -e and -E and in Julia script run with -i option.\n\nIf true, InterruptException is not thrown by Ctrl-C.  Running code upon such event requires atexit.  This is the default behavior in Julia script run without -i option.\n\ncompat: Julia 1.5\nFunction exit_on_sigint requires at least Julia 1.5.\n\n\n\n\n\n"},{"location":"base/c.html#Base.systemerror","page":"C Interface","title":"Base.systemerror","category":"function","text":"systemerror(sysfunc[, errno::Cint=Libc.errno()])\nsystemerror(sysfunc, iftrue::Bool)\n\nRaises a SystemError for errno with the descriptive string sysfunc if iftrue is true\n\n\n\n\n\n"},{"location":"base/c.html#Base.windowserror","page":"C Interface","title":"Base.windowserror","category":"function","text":"windowserror(sysfunc[, code::UInt32=Libc.GetLastError()])\nwindowserror(sysfunc, iftrue::Bool)\n\nLike systemerror, but for Windows API functions that use GetLastError to return an error code instead of setting errno.\n\n\n\n\n\n"},{"location":"base/c.html#Core.Ptr","page":"C Interface","title":"Core.Ptr","category":"type","text":"Ptr{T}\n\nA memory address referring to data of type T.  However, there is no guarantee that the memory is actually valid, or that it actually represents data of the specified type.\n\n\n\n\n\n"},{"location":"base/c.html#Core.Ref","page":"C Interface","title":"Core.Ref","category":"type","text":"Ref{T}\n\nAn object that safely references data of type T. This type is guaranteed to point to valid, Julia-allocated memory of the correct type. The underlying data is protected from freeing by the garbage collector as long as the Ref itself is referenced.\n\nIn Julia, Ref objects are dereferenced (loaded or stored) with [].\n\nCreation of a Ref to a value x of type T is usually written Ref(x). Additionally, for creating interior pointers to containers (such as Array or Ptr), it can be written Ref(a, i) for creating a reference to the i-th element of a.\n\nRef{T}() creates a reference to a value of type T without initialization. For a bitstype T, the value will be whatever currently resides in the memory allocated. For a non-bitstype T, the reference will be undefined and attempting to dereference it will result in an error, \"UndefRefError: access to undefined reference\".\n\nTo check if a Ref is an undefined reference, use isassigned(ref::RefValue). For example, isassigned(Ref{T}()) is false if T is not a bitstype. If T is a bitstype, isassigned(Ref{T}()) will always be true.\n\nWhen passed as a ccall argument (either as a Ptr or Ref type), a Ref object will be converted to a native pointer to the data it references. For most T, or when converted to a Ptr{Cvoid}, this is a pointer to the object data. When T is an isbits type, this value may be safely mutated, otherwise mutation is strictly undefined behavior.\n\nAs a special case, setting T = Any will instead cause the creation of a pointer to the reference itself when converted to a Ptr{Any} (a jl_value_t const* const* if T is immutable, else a jl_value_t *const *). When converted to a Ptr{Cvoid}, it will still return a pointer to the data region as for any other T.\n\nA C_NULL instance of Ptr can be passed to a ccall Ref argument to initialize it.\n\nUse in broadcasting\n\nRef is sometimes used in broadcasting in order to treat the referenced values as a scalar.\n\nExamples\n\njulia> r = Ref(5) # Create a Ref with an initial value\nBase.RefValue{Int64}(5)\n\njulia> r[] # Getting a value from a Ref\n5\n\njulia> r[] = 7 # Storing a new value in a Ref\n7\n\njulia> r # The Ref now contains 7\nBase.RefValue{Int64}(7)\n\njulia> isa.(Ref([1,2,3]), [Array, Dict, Int]) # Treat reference values as scalar during broadcasting\n3-element BitVector:\n 1\n 0\n 0\n\njulia> Ref{Function}()  # Undefined reference to a non-bitstype, Function\nBase.RefValue{Function}(#undef)\n\njulia> try\n           Ref{Function}()[] # Dereferencing an undefined reference will result in an error\n       catch e\n           println(e)\n       end\nUndefRefError()\n\njulia> Ref{Int64}()[]; # A reference to a bitstype refers to an undetermined value if not given\n\njulia> isassigned(Ref{Int64}()) # A reference to a bitstype is always assigned\ntrue\n\n\n\n\n\n"},{"location":"base/c.html#Base.isassigned-Tuple{Base.RefValue}","page":"C Interface","title":"Base.isassigned","category":"method","text":"isassigned(ref::RefValue)::Bool\n\nTest whether the given Ref is associated with a value. This is always true for a Ref of a bitstype object. Return false if the reference is undefined.\n\nExamples\n\njulia> ref = Ref{Function}()\nBase.RefValue{Function}(#undef)\n\njulia> isassigned(ref)\nfalse\n\njulia> ref[] = (foobar(x) = x)\nfoobar (generic function with 1 method)\n\njulia> isassigned(ref)\ntrue\n\njulia> isassigned(Ref{Int}())\ntrue\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cchar","page":"C Interface","title":"Base.Cchar","category":"type","text":"Cchar\n\nEquivalent to the native char c-type.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cuchar","page":"C Interface","title":"Base.Cuchar","category":"type","text":"Cuchar\n\nEquivalent to the native unsigned char c-type (UInt8).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cshort","page":"C Interface","title":"Base.Cshort","category":"type","text":"Cshort\n\nEquivalent to the native signed short c-type (Int16).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cstring","page":"C Interface","title":"Base.Cstring","category":"type","text":"Cstring\n\nA C-style string composed of the native character type Cchars. Cstrings are NUL-terminated. For C-style strings composed of the native wide character type, see Cwstring. For more information about string interoperability with C, see the manual.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cushort","page":"C Interface","title":"Base.Cushort","category":"type","text":"Cushort\n\nEquivalent to the native unsigned short c-type (UInt16).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cint","page":"C Interface","title":"Base.Cint","category":"type","text":"Cint\n\nEquivalent to the native signed int c-type (Int32).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cuint","page":"C Interface","title":"Base.Cuint","category":"type","text":"Cuint\n\nEquivalent to the native unsigned int c-type (UInt32).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Clong","page":"C Interface","title":"Base.Clong","category":"type","text":"Clong\n\nEquivalent to the native signed long c-type.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Culong","page":"C Interface","title":"Base.Culong","category":"type","text":"Culong\n\nEquivalent to the native unsigned long c-type.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Clonglong","page":"C Interface","title":"Base.Clonglong","category":"type","text":"Clonglong\n\nEquivalent to the native signed long long c-type (Int64).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Culonglong","page":"C Interface","title":"Base.Culonglong","category":"type","text":"Culonglong\n\nEquivalent to the native unsigned long long c-type (UInt64).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cintmax_t","page":"C Interface","title":"Base.Cintmax_t","category":"type","text":"Cintmax_t\n\nEquivalent to the native intmax_t c-type (Int64).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cuintmax_t","page":"C Interface","title":"Base.Cuintmax_t","category":"type","text":"Cuintmax_t\n\nEquivalent to the native uintmax_t c-type (UInt64).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Csize_t","page":"C Interface","title":"Base.Csize_t","category":"type","text":"Csize_t\n\nEquivalent to the native size_t c-type (UInt).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cssize_t","page":"C Interface","title":"Base.Cssize_t","category":"type","text":"Cssize_t\n\nEquivalent to the native ssize_t c-type.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cptrdiff_t","page":"C Interface","title":"Base.Cptrdiff_t","category":"type","text":"Cptrdiff_t\n\nEquivalent to the native ptrdiff_t c-type (Int).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cwchar_t","page":"C Interface","title":"Base.Cwchar_t","category":"type","text":"Cwchar_t\n\nEquivalent to the native wchar_t c-type (Int32).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cwstring","page":"C Interface","title":"Base.Cwstring","category":"type","text":"Cwstring\n\nA C-style string composed of the native wide character type Cwchar_ts. Cwstrings are NUL-terminated. For C-style strings composed of the native character type, see Cstring. For more information about string interoperability with C, see the manual.\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cfloat","page":"C Interface","title":"Base.Cfloat","category":"type","text":"Cfloat\n\nEquivalent to the native float c-type (Float32).\n\n\n\n\n\n"},{"location":"base/c.html#Base.Cdouble","page":"C Interface","title":"Base.Cdouble","category":"type","text":"Cdouble\n\nEquivalent to the native double c-type (Float64).\n\n\n\n\n\n"},{"location":"base/c.html#Core.Intrinsics.llvmcall","page":"C Interface","title":"Core.Intrinsics.llvmcall","category":"function","text":"llvmcall(fun_ir::String, returntype, Tuple{argtype1, ...}, argvalue1, ...)\nllvmcall((mod_ir::String, entry_fn::String), returntype, Tuple{argtype1, ...}, argvalue1, ...)\nllvmcall((mod_bc::Vector{UInt8}, entry_fn::String), returntype, Tuple{argtype1, ...}, argvalue1, ...)\n\nCall the LLVM code provided in the first argument. There are several ways to specify this first argument:\n\nas a literal string, representing function-level IR (similar to an LLVM define block), with arguments are available as consecutive unnamed SSA variables (%0, %1, etc.);\nas a 2-element tuple, containing a string of module IR and a string representing the name of the entry-point function to call;\nas a 2-element tuple, but with the module provided as an Vector{UInt8} with bitcode.\n\nNote that contrary to ccall, the argument types must be specified as a tuple type, and not a tuple of types. All types, as well as the LLVM code, should be specified as literals, and not as variables or expressions (it may be necessary to use @eval to generate these literals).\n\nSee test/llvmcall.jl for usage examples.\n\n\n\n\n\n"},{"location":"devdocs/gc-mmtk.html#Julia-MMTk","page":"Julia + MMTk","title":"Julia + MMTk","category":"section","text":"There has been quite a lot of effort to refactor the GC code inside Julia to support external GCs. The first step to enable using different GC algorithms for Julia was the design and implementation of a GC interface. To drive that interface, we added support for building Julia with MMTk (Memory Management Toolkit). Using Julia + MMTk enables testing different GC implementations, allowing developers to choose a specific implementation when building Julia from source. The connection between Julia and MMTk is done via a binding, which links the language runtime with MMTk core. The mmtk-julia binding is written in Rust and can be found in this repository.\n\n[!NOTE] Using a different GC requires building Julia from source. It is not possible to switch implementations at runtime. To see what version of the GC is currently being used, run versioninfo() from the Julia REPL and it should show the version under GC: ...."},{"location":"devdocs/gc-mmtk.html#Building-Julia-with-MMTk","page":"Julia + MMTk","title":"Building Julia with MMTk","category":"section","text":"There are 3 different ways of building Julia with MMTk: building from source using a fixed release of the binding, checking out a custom version in the mmtk-julia repository or using a precompiled binary from Julia's BinaryBuilder. The easiest way is to use the BinaryBuilder binary. First, to enable MMTk as a third-party GC, set the variable WITH_THIRD_PARTY_GC to mmtk. Then, for example, to use the Immix as the GC, simply set the variable MMTK_PLAN=Immix and build Julia as usual.\n\nThere are different configurations supported by the following variables, which can be set in a Make.user file or as an environment variable. Note that at this time, setting MMTK_PLAN=StickyImmix (to use a generational version of Immix) or MMTK_MOVING=1 (to enable object movement) will likely cause segmentation faults or other build failures, since we have not added support for these configurations yet. Setting MMTK_BUILD=debug will force a debug build of the binding, which will print some logging information that can be used to find errors that are specific to MMTk.\n\nVariable  \nMMTK_PLAN Immix StickyImmix\nMMTK_MOVING 0 1\nMMTK_BUILD release debug\n\nNote that when setting only MMTK_PLAN, then the default is to do a non-moving, release build."},{"location":"devdocs/gc-mmtk.html#Building-mmtk-julia-from-source","page":"Julia + MMTk","title":"Building mmtk-julia from source","category":"section","text":"It is also possible to build the binding from source. To do so, set the variable USE_BINARYBUILDER_MMTK_JULIA=0 and the latest release version of the binding will be downloaded and built as part of building Julia. Note that this requires an installation of the rust toolchain.\n\nIt is also possible to build a custom version of binding by checking it out from the git repository and setting a variable named MMTK_JULIA_DIR as the path that contains the binding.\n\nFor more information on building Julia with MMTk, please refer to the README file in the binding repo."},{"location":"devdocs/gc-mmtk.html#I've-got-a-build-error-when-building-Julia-with-MMTk,-what-should-I-do?","page":"Julia + MMTk","title":"I've got a build error when building Julia with MMTk, what should I do?","category":"section","text":"If you try to build Julia with MMTk and get an error it is likely due to a change to Julia that has not been yet propagated to the binding or to the code in Julia that is specific to MMTk. Some changes include:\n\n(1) Changing the memory layout of objects in Julia. The binding relies on automatically generated Rust FFI bindings from Julia code. These files are generated using a crate named rust-bindgen. To regenerate those files, check out the latest version of the mmtk-julia binding, set the variable JULIA_PATH to the path of the Julia version you are trying to build and run make regen-bindgen-ffi from the directory containing the binding. This should delete the current version of the FFI bindings and generate a new version based on the Julia code from JULIA_PATH.\n\n(2) Changing the root objects passed to the GC. Julia passes a set of objects to the GC as roots in the function gcmarkroots. At the moment, this set needs to be consistent between both the Stock GC and MMTk (in the function jl_gc_scan_vm_specific_roots).\n\n(3) Changing how objects are scanned. MMTk uses the same strategy to find references in Julia objects as the stock GC (see gcmarkoutrefs). Changing the logic from this function should be reflected in the Rust code in the binding that scan Julia objects.\n\nIf your case is not included in one of the alternatives above, please create an issue in the Julia repository tagging it with the GC: MMTK label."},{"location":"manual/documentation.html#man-documentation","page":"Documentation","title":"Documentation","category":"section","text":""},{"location":"manual/documentation.html#Accessing-Documentation","page":"Documentation","title":"Accessing Documentation","category":"section","text":"Documentation can be accessed at the REPL or in IJulia by typing ? followed by the name of a function or macro, and pressing Enter. For example,\n\n?cos\n?@time\n?r\"\"\n\nwill show documentation for the relevant function, macro or string macro respectively. Most Julia environments provide a way to access documentation directly:\n\nVS Code shows documentation when you hover over a function name. You can also use the Julia panel in the sidebar to search for documentation.\nIn Pluto, open the \"Live Docs\" panel on the bottom right.\nIn Juno using Ctrl-J, Ctrl-D will show the documentation for the object under the cursor.\n\nDocs.hasdoc(module, name)::Bool tells whether a name has a docstring. Docs.undocumented_names(module; all) returns the undocumented names in a module."},{"location":"manual/documentation.html#man-writing-documentation","page":"Documentation","title":"Writing Documentation","category":"section","text":"Julia enables package developers and users to document functions, types and other objects easily via a built-in documentation system.\n\nThe basic syntax is simple: any string appearing just before an object (function, macro, type or instance) will be interpreted as documenting it (these are called docstrings). Note that no blank lines or comments may intervene between a docstring and the documented object. Here is a basic example:\n\n\"Tell whether there are too many foo items in the array.\"\nfoo(xs::Array) = ...\n\nnote: Reminder\nAny empty lines between the docstring and the object being documented detach the former from the latter, making the docstring ineffective.\n\nDocumentation is interpreted as Markdown, so you can use indentation and code fences to delimit code examples from text. Technically, any object can be associated with any other as metadata; Markdown happens to be the default, but one can construct other string macros and pass them to the @doc macro just as well.\n\nnote: Note\nMarkdown support is implemented in the Markdown standard library and for a full list of supported syntax see the documentation.\n\nHere is a more complex example, still using Markdown:\n\n\"\"\"\n    bar(x[, y])\n\nCompute the Bar index between `x` and `y`.\n\nIf `y` is unspecified, compute the Bar index between all pairs of columns of `x`.\n\n# Examples\n```julia-repl\njulia> bar([1, 2], [1, 2])\n1\n```\n\"\"\"\nfunction bar(x, y) ...\n\nAs in the example above, we recommend following some simple conventions when writing documentation:\n\nAlways show the signature of a function at the top of the documentation, with a four-space indent so that it is printed as Julia code.\nThis can be identical to the signature present in the Julia code (like mean(x::AbstractArray)), or a simplified form. Optional arguments should be represented with their default values (i.e. f(x, y=1)) when possible, following the actual Julia syntax. Optional arguments which do not have a default value should be put in brackets (i.e. f(x[, y]) and f(x[, y[, z]])). An alternative solution is to use several lines: one without optional arguments, the other(s) with them. This solution can also be used to document several related methods of a given function. When a function accepts many keyword arguments, only include a <keyword arguments> placeholder in the signature (i.e. f(x; <keyword arguments>)), and give the complete list under an # Arguments section (see point 4 below).\nUse this style to document the return type or give the return value a name:\n# Naming the return value or its type is not necessary (this is the most common case)\n\"\"\"\n   sum(itr; [init])\n\n...\n\"\"\"\n\n# The return type is easily documented and critical to the semantics of this function\n\"\"\"\n   vec(x::AbstractArray)::AbstractVector\n\n...\n\"\"\"\n\n# Naming and/or destructuring the return value clarifies the semantics of this function\n\"\"\"\n   splitdir(path::AbstractString) -> (dir::AbstractString, file::AbstractString)\n...\n\"\"\"\nWhen included, a return type should be written after the signature, separated by ::, while a named return value should be separated by ->, with a space on both sides. Return types and return values should be valid Julia expressions when possible. Macro docstring signatures that annotate return types or return values should use parentheses to clarify where the macro arguments end and return type or return value begins.\nInclude a single one-line sentence describing what the function does or what the object represents after the simplified signature block. If needed, provide more details in a second paragraph, after a blank line.\nThe one-line sentence should use the imperative form (\"Do this\", \"Return that\") instead of the third person (do not write \"Returns the length...\") when documenting functions. It should end with a period. If the meaning of a function cannot be summarized easily, splitting it into separate composable parts could be beneficial (this should not be taken as an absolute requirement for every single case though).\nDo not repeat yourself.\nSince the function name is given by the signature, there is no need to start the documentation with \"The function bar...\": go straight to the point. Similarly, if the signature specifies the types of the arguments, mentioning them in the description is redundant.\nOnly provide an argument list when really necessary.\nFor simple functions, it is often clearer to mention the role of the arguments directly in the description of the function's purpose. An argument list would only repeat information already provided elsewhere. However, providing an argument list can be a good idea for complex functions with many arguments (in particular keyword arguments). In that case, insert it after the general description of the function, under an # Arguments header, with one - bullet for each argument. The list should mention the types and default values (if any) of the arguments:\n\"\"\"\n...\n# Arguments\n- `n::Integer`: the number of elements to compute.\n- `dim::Integer=1`: the dimensions along which to perform the computation.\n...\n\"\"\"\nProvide hints to related functions.\nSometimes there are functions of related functionality. To increase discoverability please provide a short list of these in a See also paragraph.\nSee also [`bar!`](@ref), [`baz`](@ref), [`baaz`](@ref).\nInclude any code examples in an # Examples section.\nExamples should, whenever possible, be written as doctests. A doctest is a fenced code block (see Code blocks) starting with ```jldoctest and contains any number of julia> prompts together with inputs and expected outputs that mimic the Julia REPL.\nnote: Note\nDoctests are enabled by Documenter.jl. For more detailed documentation see Documenter's manual.\nFor example in the following docstring a variable a is defined and the expected result, as printed in a Julia REPL, appears afterwards:\n\"\"\"\nSome nice documentation here.\n\n# Examples\n```jldoctest\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n```\n\"\"\"\nwarning: Warning\nCalling rand and other RNG-related functions should be avoided in doctests since they will not produce consistent outputs during different Julia sessions. If you would like to show some random number generation related functionality, one option is to explicitly construct and seed your own RNG object (see Random) and pass it to the functions you are doctesting.Operating system word size (Int32 or Int64) as well as path separator differences (/ or \\) will also affect the reproducibility of some doctests.Note that whitespace in your doctest is significant! The doctest will fail if you misalign the output of pretty-printing an array, for example.\nYou can then run make -C doc doctest=true to run all the doctests in the Julia Manual and API documentation, which will ensure that your example works.\nTo indicate that the output result is truncated, you may write [...] at the line where checking should stop. This is useful to hide a stacktrace (which contains non-permanent references to lines of julia code) when the doctest shows that an exception is thrown, for example:\n```jldoctest\njulia> div(1, 0)\nERROR: DivideError: integer division error\n[...]\n```\nExamples that are untestable should be written within fenced code blocks starting with ```julia so that they are highlighted correctly in the generated documentation.\ntip: Tip\nWherever possible examples should be self-contained and runnable so that readers are able to try them out without having to include any dependencies.\nUse backticks to identify code and equations.\nJulia identifiers and code excerpts should always appear between backticks ` to enable highlighting. Equations in the LaTeX syntax can be inserted between double backticks ``. Use Unicode characters rather than their LaTeX escape sequence, i.e. ``α = 1`` rather than ``\\\\alpha = 1``.\nPlace the starting and ending \"\"\" characters on lines by themselves.\nThat is, write:\n\"\"\"\n...\n\n...\n\"\"\"\nf(x, y) = ...\nrather than:\n\"\"\"...\n\n...\"\"\"\nf(x, y) = ...\nThis makes it clearer where docstrings start and end.\nRespect the line length limit used in the surrounding code.\nDocstrings are edited using the same tools as code. Therefore, the same conventions should apply. It is recommended that lines are at most 92 characters wide.\nProvide information allowing custom types to implement the function in an # Implementation section. These implementation details are intended for developers rather than users, explaining e.g. which functions should be overridden and which functions automatically use appropriate fallbacks. Such details are best kept separate from the main description of the function's behavior.\nFor long docstrings, consider splitting the documentation with an # Extended help header. The typical help-mode will show only the material above the header; you can access the full help by adding a '?' at the beginning of the expression (i.e., \"??foo\" rather than \"?foo\")."},{"location":"manual/documentation.html#Functions-and-Methods","page":"Documentation","title":"Functions & Methods","category":"section","text":"Functions in Julia may have multiple implementations, known as methods. While it's good practice for generic functions to have a single purpose, Julia allows methods to be documented individually if necessary. In general, only the most generic method should be documented, or even the function itself (i.e. the object created without any methods by function bar end). Specific methods should only be documented if their behaviour differs from the more generic ones. In any case, they should not repeat the information provided elsewhere. For example:\n\n\"\"\"\n    *(x, y, z...)\n\nMultiplication operator. `x * y * z *...` calls this function with multiple\narguments, i.e. `*(x, y, z...)`.\n\"\"\"\nfunction *(x, y, z...)\n    # ... [implementation sold separately] ...\nend\n\n\"\"\"\n    *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\nWhen applied to strings, concatenates them.\n\"\"\"\nfunction *(x::AbstractString, y::AbstractString, z::AbstractString...)\n    # ... [insert secret sauce here] ...\nend\n\nhelp?> *\nsearch: * .*\n\n  *(x, y, z...)\n\n  Multiplication operator. x * y * z *... calls this function with multiple\n  arguments, i.e. *(x,y,z...).\n\n  *(x::AbstractString, y::AbstractString, z::AbstractString...)\n\n  When applied to strings, concatenates them.\n\nWhen retrieving documentation for a generic function, the metadata for each method is concatenated with the catdoc function, which can of course be overridden for custom types."},{"location":"manual/documentation.html#Advanced-Usage","page":"Documentation","title":"Advanced Usage","category":"section","text":"The @doc macro associates its first argument with its second in a per-module dictionary called META.\n\nTo make it easier to write documentation, the parser treats the macro name @doc specially: if a call to @doc has one argument, but another expression appears after a single line break, then that additional expression is added as an argument to the macro. Therefore the following syntax is parsed as a 2-argument call to @doc:\n\n@doc raw\"\"\"\n...\n\"\"\"\nf(x) = x\n\nThis makes it possible to use expressions other than normal string literals (such as the raw\"\" string macro) as a docstring.\n\nWhen used for retrieving documentation, the @doc macro (or equally, the doc function) will search all META dictionaries for metadata relevant to the given object and return it. The returned object (some Markdown content, for example) will by default display itself intelligently. This design also makes it easy to use the doc system in a programmatic way; for example, to re-use documentation between different versions of a function:\n\n@doc \"...\" foo!\n@doc (@doc foo!) foo\n\ncompat: Julia 1.11\nIn Julia 1.11 and newer, retrieving documentation with the @doc macro requires that the REPL stdlib is loaded.\n\nOr for use with Julia's metaprogramming functionality:\n\nfor (f, op) in ((:add, :+), (:subtract, :-), (:multiply, :*), (:divide, :/))\n    @eval begin\n        $f(a, b) = $op(a, b)\n    end\nend\n@doc \"`add(a, b)` adds `a` and `b` together\" add\n@doc \"`subtract(a, b)` subtracts `b` from `a`\" subtract\n\nDocumentation in non-toplevel blocks, such as begin, if, for, let, and inner constructors, should be added to the documentation system via @doc as well. For example:\n\nif condition()\n    @doc \"...\"\n    f(x) = x\nend\n\nwill add documentation to f(x) when condition() is true. Note that even if f(x) goes out of scope at the end of a block, its documentation will remain.\n\nIt is possible to make use of metaprogramming to assist in the creation of documentation. When using string-interpolation within the docstring you will need to use an extra $ as shown with $($name):\n\nfor func in (:day, :dayofmonth)\n    name = string(func)\n    @eval begin\n        @doc \"\"\"\n            $($name)(dt::TimeType) -> Int64\n\n        The day of month of a `Date` or `DateTime` as an `Int64`.\n        \"\"\" $func(dt::Dates.TimeType)\n    end\nend"},{"location":"manual/documentation.html#Dynamic-documentation","page":"Documentation","title":"Dynamic documentation","category":"section","text":"Sometimes the appropriate documentation for an instance of a type depends on the field values of that instance, rather than just on the type itself. In these cases, you can add a method to Docs.getdoc for your custom type that returns the documentation on a per-instance basis. For instance,\n\nstruct MyType\n    value::Int\nend\n\nDocs.getdoc(t::MyType) = \"Documentation for MyType with value $(t.value)\"\n\nx = MyType(1)\ny = MyType(2)\n\n?x will display \"Documentation for MyType with value 1\" while ?y will display \"Documentation for MyType with value 2\"."},{"location":"manual/documentation.html#Syntax-Guide","page":"Documentation","title":"Syntax Guide","category":"section","text":"This guide provides a comprehensive overview of how to attach documentation to all Julia syntax constructs for which providing documentation is possible.\n\nIn the following examples \"...\" is used to illustrate an arbitrary docstring."},{"location":"manual/documentation.html#and-\\-characters","page":"Documentation","title":"$ and \\ characters","category":"section","text":"The $ and \\ characters are still parsed as string interpolation or start of an escape sequence in docstrings too. The raw\"\" string macro together with the @doc macro can be used to avoid having to escape them. This is handy when the docstrings include LaTeX or Julia source code examples containing interpolation:\n\n@doc raw\"\"\"\n```math\n\\LaTeX\n```\n\"\"\"\nfunction f end"},{"location":"manual/documentation.html#Functions-and-Methods-2","page":"Documentation","title":"Functions and Methods","category":"section","text":"\"...\"\nfunction f end\n\n\"...\"\nf\n\nAdds docstring \"...\" to the function f. The first version is the preferred syntax, however both are equivalent.\n\n\"...\"\nf(x) = x\n\n\"...\"\nfunction f(x)\n    return x\nend\n\n\"...\"\nf(x)\n\nAdds docstring \"...\" to the method f(::Any).\n\n\"...\"\nf(x, y = 1) = x + y\n\nAdds docstring \"...\" to two Methods, namely f(::Any) and f(::Any, ::Any)."},{"location":"manual/documentation.html#Macros","page":"Documentation","title":"Macros","category":"section","text":"\"...\"\nmacro m(x) end\n\nAdds docstring \"...\" to the @m(::Any) macro definition.\n\n\"...\"\n:(@m1)\n\n\"...\"\nmacro m2 end\n\nAdds docstring \"...\" to the macros named @m1 and @m2."},{"location":"manual/documentation.html#Types","page":"Documentation","title":"Types","category":"section","text":"\"...\"\nabstract type T1 end\n\n\"...\"\nmutable struct T2\n    ...\nend\n\n\"...\"\nstruct T3\n    ...\nend\n\nAdds the docstring \"...\" to types T1, T2, and T3.\n\n\"...\"\nT1\n\n\"...\"\nT2\n\n\"...\"\nT3\n\nAdds the docstring \"...\" to types T1, T2, and T3. The previous version is the preferred syntax, however both are equivalent.\n\n\"...\"\nstruct T\n    \"x\"\n    x\n    \"y\"\n    y\n\n    @doc \"Inner constructor\"\n    function T()\n        new(...)\n    end\nend\n\nAdds docstring \"...\" to type T, \"x\" to field T.x, \"y\" to field T.y, and \"Inner constructor\" to the inner constructor T(). Also applicable to mutable struct types."},{"location":"manual/documentation.html#Modules","page":"Documentation","title":"Modules","category":"section","text":"\"...\"\nmodule M end\n\nmodule M\n\n\"...\"\nM\n\nend\n\nAdds docstring \"...\" to the Module M. Adding the docstring above the Module is the preferred syntax, however both are equivalent.\n\nThe module docstring is evaluated inside the scope of the module, allowing access to all the symbols defined in and imported into the module:\n\n\"The magic number is $(MAGIC).\"\nmodule DocStringEval\nconst MAGIC = 42\nend\n\nDocumenting a baremodule by placing a docstring above the expression automatically imports @doc into the module. These imports must be done manually when the module expression is not documented:\n\n\"...\"\nbaremodule M\n# ...\nend\n\nbaremodule M\n\nimport Base: @doc\n\n\"...\"\nf(x) = x\n\nend"},{"location":"manual/documentation.html#Global-Variables","page":"Documentation","title":"Global Variables","category":"section","text":"\"...\"\nconst a = 1\n\n\"...\"\nb = 2\n\n\"...\"\nglobal c = 3\n\nAdds docstring \"...\" to the Bindings a, b, and c.\n\nBindings are used to store a reference to a particular Symbol in a Module without storing the referenced value itself.\n\nnote: Note\nWhen a const definition is only used to define an alias of another definition, such as is the case with the function div and its alias ÷ in Base, do not document the alias and instead document the actual function.If the alias is documented and not the real definition then the docsystem (? mode) will not return the docstring attached to the alias when the real definition is searched for.For example you should write\"...\"\nf(x) = x + 1\nconst alias = frather thanf(x) = x + 1\n\"...\"\nconst alias = f\n\n\"...\"\nsym\n\nAdds docstring \"...\" to the value associated with sym. However, it is preferred that sym is documented where it is defined."},{"location":"manual/documentation.html#Multiple-Objects","page":"Documentation","title":"Multiple Objects","category":"section","text":"\"...\"\na, b\n\nAdds docstring \"...\" to a and b each of which should be a documentable expression. This syntax is equivalent to\n\n\"...\"\na\n\n\"...\"\nb\n\nAny number of expressions many be documented together in this way. This syntax can be useful when two functions are related, such as non-mutating and mutating versions f and f!."},{"location":"manual/documentation.html#Macro-generated-code","page":"Documentation","title":"Macro-generated code","category":"section","text":"\"...\"\n@m expression\n\nAdds docstring \"...\" to the expression generated by expanding @m expression. This allows for expressions decorated with @inline, @noinline, @generated, or any other macro to be documented in the same way as undecorated expressions.\n\nMacro authors should take note that only macros that generate a single expression will automatically support docstrings. If a macro returns a block containing multiple subexpressions then the subexpression that should be documented must be marked using the @__doc__ macro.\n\nThe @enum macro makes use of @__doc__ to allow for documenting Enums. Examining its definition should serve as an example of how to use @__doc__ correctly."},{"location":"manual/documentation.html#Core.@__doc__","page":"Documentation","title":"Core.@__doc__","category":"macro","text":"@__doc__(ex)\n\nLow-level macro used to mark expressions returned by a macro that should be documented. If more than one expression is marked then the same docstring is applied to each expression.\n\nmacro example(f)\n    quote\n        $(f)() = 0\n        @__doc__ $(f)(x) = 1\n        $(f)(x, y) = 2\n    end |> esc\nend\n\n@__doc__ has no effect when a macro that uses it is not documented.\n\ncompat: Julia 1.12\nThis section documents a very subtle corner case that is only relevant to macros which themselves both define other macros and then attempt to use them within the same expansion. Such macros were impossible to write prior to Julia 1.12 and are still quite rare. If you are not writing such a macro, you may ignore this note.In versions prior to Julia 1.12, macroexpansion would recursively expand through Expr(:toplevel) blocks. This behavior was changed in Julia 1.12 to allow macros to recursively define other macros and use them in the same returned expression. However, to preserve backwards compatibility with existing uses of @__doc__, the doc system will still expand through Expr(:toplevel) blocks when looking for @__doc__ markers. As a result, macro-defining-macros will have an observable behavior difference when annotated with a docstring:julia> macro macroception()\n    Expr(:toplevel, :(macro foo() 1 end), :(@foo))\nend\n\njulia> @macroception\n1\n\njulia> \"Docstring\" @macroception\nERROR: LoadError: UndefVarError: `@foo` not defined in `Main`The supported workaround is to manually expand the @__doc__ macro in the defining macro, which the docsystem will recognize and suppress the recursive expansion:julia> macro macroception()\n    Expr(:toplevel,\n        macroexpand(__module__, :(@__doc__ macro foo() 1 end); recursive=false),\n        :(@foo))\nend\n\njulia> @macroception\n1\n\njulia> \"Docstring\" @macroception\n1\n\n\n\n\n\n"},{"location":"devdocs/meta.html#Talking-to-the-compiler-(the-:meta-mechanism)","page":"Talking to the compiler (the :meta mechanism)","title":"Talking to the compiler (the :meta mechanism)","category":"section","text":"In some circumstances, one might wish to provide hints or instructions that a given block of code has special properties: you might always want to inline it, or you might want to turn on special compiler optimization passes. Starting with version 0.4, Julia has a convention that these instructions can be placed inside a :meta expression, which is typically (but not necessarily) the first expression in the body of a function.\n\n:meta expressions are created with macros. As an example, consider the implementation of the @inline macro:\n\nmacro inline(ex)\n    esc(isa(ex, Expr) ? pushmeta!(ex, :inline) : ex)\nend\n\nHere, ex is expected to be an expression defining a function. A statement like this:\n\n@inline function myfunction(x)\n    x*(x+3)\nend\n\ngets turned into an expression like this:\n\nquote\n    function myfunction(x)\n        Expr(:meta, :inline)\n        x*(x+3)\n    end\nend\n\nBase.pushmeta!(ex, tag::Union{Symbol,Expr}) appends :tag to the end of the :meta expression, creating a new :meta expression if necessary.\n\nTo use the metadata, you have to parse these :meta expressions. If your implementation can be performed within Julia, Base.popmeta! is very handy: Base.popmeta!(body, :symbol) will scan a function body expression (one without the function signature) for the first :meta expression containing :symbol, extract any arguments, and return a tuple (found::Bool, args::Array{Any}). If the metadata did not have any arguments, or :symbol was not found, the args array will be empty.\n\nNot yet provided is a convenient infrastructure for parsing :meta expressions from C++."},{"location":"base/io-network.html#I/O-and-Network","page":"I/O and Network","title":"I/O and Network","category":"section","text":""},{"location":"base/io-network.html#General-I/O","page":"I/O and Network","title":"General I/O","category":"section","text":""},{"location":"base/io-network.html#Text-I/O","page":"I/O and Network","title":"Text I/O","category":"section","text":""},{"location":"base/io-network.html#Multimedia-I/O","page":"I/O and Network","title":"Multimedia I/O","category":"section","text":"Just as text output is performed by print and user-defined types can indicate their textual representation by overloading show, Julia provides a standardized mechanism for rich multimedia output (such as images, formatted text, or even audio and video), consisting of three parts:\n\nA function display(x) to request the richest available multimedia display of a Julia object x (with a plain-text fallback).\nOverloading show allows one to indicate arbitrary multimedia representations (keyed by standard MIME types) of user-defined types.\nMultimedia-capable display backends may be registered by subclassing a generic AbstractDisplay type and pushing them onto a stack of display backends via pushdisplay.\n\nThe base Julia runtime provides only plain-text display, but richer displays may be enabled by loading external modules or by using graphical Julia environments (such as the IPython-based IJulia notebook).\n\nAs mentioned above, one can also define new display backends. For example, a module that can display PNG images in a window can register this capability with Julia, so that calling display(x) on types with PNG representations will automatically display the image using the module's window.\n\nIn order to define a new display backend, one should first create a subtype D of the abstract class AbstractDisplay. Then, for each MIME type (mime string) that can be displayed on D, one should define a function display(d::D, ::MIME\"mime\", x) = ... that displays x as that MIME type, usually by calling show(io, mime, x) or repr(io, mime, x). A MethodError should be thrown if x cannot be displayed as that MIME type; this is automatic if one calls show or repr. Finally, one should define a function display(d::D, x) that queries showable(mime, x) for the mime types supported by D and displays the \"best\" one; a MethodError should be thrown if no supported MIME types are found for x. Similarly, some subtypes may wish to override redisplay(d::D, ...). (Again, one should import Base.display to add new methods to display.) The return values of these functions are up to the implementation (since in some cases it may be useful to return a display \"handle\" of some type). The display functions for D can then be called directly, but they can also be invoked automatically from display(x) simply by pushing a new display onto the display-backend stack with:"},{"location":"base/io-network.html#Network-I/O","page":"I/O and Network","title":"Network I/O","category":"section","text":""},{"location":"base/io-network.html#Core.IO","page":"I/O and Network","title":"Core.IO","category":"type","text":"IO\n\nAbstract supertype for input/output types.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.stdout","page":"I/O and Network","title":"Base.stdout","category":"constant","text":"stdout::IO\n\nGlobal variable referring to the standard out stream.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.stderr","page":"I/O and Network","title":"Base.stderr","category":"constant","text":"stderr::IO\n\nGlobal variable referring to the standard error stream.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.stdin","page":"I/O and Network","title":"Base.stdin","category":"constant","text":"stdin::IO\n\nGlobal variable referring to the standard input stream.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.read-Tuple{AbstractString}","page":"I/O and Network","title":"Base.read","category":"method","text":"read(filename::AbstractString)\n\nRead the entire contents of a file as a Vector{UInt8}.\n\nread(filename::AbstractString, String)\n\nRead the entire contents of a file as a string.\n\nread(filename::AbstractString, args...)\n\nOpen a file and read its contents. args is passed to read: this is equivalent to open(io->read(io, args...), filename).\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.write-Tuple{AbstractString, Any}","page":"I/O and Network","title":"Base.write","category":"method","text":"write(filename::AbstractString, content)\n\nWrite the canonical binary representation of content to a file, which will be created if it does not exist yet or overwritten if it does exist.\n\nReturn the number of bytes written into the file.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.open","page":"I/O and Network","title":"Base.open","category":"function","text":"open(f::Function, command, args...; kwargs...)\n\nSimilar to open(command, args...; kwargs...), but calls f(stream) on the resulting process stream, then closes the input stream and waits for the process to complete. Return the value returned by f on success. Throw an error if the process failed, or if the process attempts to print anything to stdout.\n\n\n\n\n\nopen(command, stdio=devnull; write::Bool = false, read::Bool = !write)\n\nStart running command asynchronously, and return a process::IO object.  If read is true, then reads from the process come from the process's standard output and stdio optionally specifies the process's standard input stream.  If write is true, then writes go to the process's standard input and stdio optionally specifies the process's standard output stream. The process's standard error stream is connected to the current global stderr.\n\n\n\n\n\nopen(command, mode::AbstractString, stdio=devnull)\n\nRun command asynchronously. Like open(command, stdio; read, write) except specifying the read and write flags via a mode string instead of keyword arguments. Possible mode strings are:\n\nMode Description Keywords\nr read none\nw write write = true\nr+ read, write read = true, write = true\nw+ read, write read = true, write = true\n\n\n\n\n\nopen(fd::OS_HANDLE)::IO\n\nTake a raw file descriptor and wrap it in a Julia-aware IO type, and take ownership of the fd handle. Call open(Libc.dup(fd)) to avoid the ownership capture of the original handle.\n\nwarning: Warning\nDo not call this on a handle that's already owned by some other part of the system.\n\n\n\n\n\nopen(filename::AbstractString, [mode::AbstractString]; lock = true)::IOStream\n\nAlternate syntax for open, where a string-based mode specifier is used instead of the five booleans. The values of mode correspond to those from fopen(3) or Perl open, and are equivalent to setting the following boolean groups:\n\nMode Description Keywords\nr read none\nw write, create, truncate write = true\na write, create, append append = true\nr+ read, write read = true, write = true\nw+ read, write, create, truncate truncate = true, read = true\na+ read, write, create, append append = true, read = true\n\nThe lock keyword argument controls whether operations will be locked for safe multi-threaded access.\n\nExamples\n\njulia> io = open(\"myfile.txt\", \"w\");\n\njulia> write(io, \"Hello world!\");\n\njulia> close(io);\n\njulia> io = open(\"myfile.txt\", \"r\");\n\njulia> read(io, String)\n\"Hello world!\"\n\njulia> write(io, \"This file is read only\")\nERROR: ArgumentError: write failed, IOStream is not writeable\n[...]\n\njulia> close(io)\n\njulia> io = open(\"myfile.txt\", \"a\");\n\njulia> write(io, \"This stream is not read only\")\n28\n\njulia> close(io)\n\njulia> rm(\"myfile.txt\")\n\ncompat: Julia 1.5\nThe lock argument is available as of Julia 1.5.\n\n\n\n\n\nopen(filename::AbstractString; lock = true, keywords...)::IOStream\n\nOpen a file in a mode specified by five boolean keyword arguments:\n\nKeyword Description Default\nread open for reading !write\nwrite open for writing truncate | append\ncreate create if non-existent !read & write | truncate | append\ntruncate truncate to zero size !read & write\nappend seek to end false\n\nThe default when no keywords are passed is to open files for reading only. Returns a stream for accessing the opened file.\n\nThe lock keyword argument controls whether operations will be locked for safe multi-threaded access.\n\ncompat: Julia 1.5\nThe lock argument is available as of Julia 1.5.\n\n\n\n\n\nopen(f::Function, args...; kwargs...)\n\nApply the function f to the result of open(args...; kwargs...) and close the resulting file descriptor upon completion.\n\nExamples\n\njulia> write(\"myfile.txt\", \"Hello world!\");\n\njulia> open(io->read(io, String), \"myfile.txt\")\n\"Hello world!\"\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.IOStream","page":"I/O and Network","title":"Base.IOStream","category":"type","text":"IOStream\n\nA buffered IO stream wrapping an OS file descriptor. Mostly used to represent files returned by open.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.IOBuffer","page":"I/O and Network","title":"Base.IOBuffer","category":"type","text":"IOBuffer(string::String)\n\nCreate a read-only IOBuffer on the data underlying the given string.\n\nExamples\n\njulia> io = IOBuffer(\"Haho\");\n\njulia> takestring!(io)\n\"Haho\"\n\njulia> takestring!(io)\n\"Haho\"\n\n\n\n\n\nIOBuffer([data::AbstractVector{UInt8}]; keywords...)::IOBuffer\n\nCreate an in-memory I/O stream, which may optionally operate on a pre-existing array.\n\nIt may take optional keyword arguments:\n\nread, write, append: restricts operations to the buffer; see open for details.\ntruncate: truncates the buffer size to zero length.\nmaxsize: specifies a size beyond which the buffer may not be grown.\nsizehint: suggests a capacity of the buffer (data must implement sizehint!(data, size)).\n\nWhen data is not given, the buffer will be both readable and writable by default.\n\nwarning: Passing `data` as scratch space to `IOBuffer` with `write=true` may give unexpected behavior\nOnce write is called on an IOBuffer, it is best to consider any previous references to data invalidated; in effect IOBuffer \"owns\" this data until a call to take!. Any indirect mutations to data could lead to undefined behavior by breaking the abstractions expected by IOBuffer. If write=true the IOBuffer may store data at any offset leaving behind arbitrary values at other offsets. If maxsize > length(data), the IOBuffer might re-allocate the data entirely, which may or may not be visible in any outstanding bindings to array.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> write(io, \"JuliaLang is a GitHub organization.\", \" It has many members.\")\n56\n\njulia> takestring!(io)\n\"JuliaLang is a GitHub organization. It has many members.\"\n\njulia> io = IOBuffer(b\"JuliaLang is a GitHub organization.\")\nIOBuffer(data=UInt8[...], readable=true, writable=false, seekable=true, append=false, size=35, maxsize=Inf, ptr=1, mark=-1)\n\njulia> read(io, String)\n\"JuliaLang is a GitHub organization.\"\n\njulia> write(io, \"This isn't writable.\")\nERROR: ArgumentError: ensureroom failed, IOBuffer is not writeable\n\njulia> io = IOBuffer(UInt8[], read=true, write=true, maxsize=34)\nIOBuffer(data=UInt8[...], readable=true, writable=true, seekable=true, append=false, size=0, maxsize=34, ptr=1, mark=-1)\n\njulia> write(io, \"JuliaLang is a GitHub organization.\")\n34\n\njulia> takestring!(io)\n\"JuliaLang is a GitHub organization\"\n\njulia> length(read(IOBuffer(b\"data\", read=true, truncate=false)))\n4\n\njulia> length(read(IOBuffer(b\"data\", read=true, truncate=true)))\n0\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.take!-Tuple{Base.GenericIOBuffer}","page":"I/O and Network","title":"Base.take!","category":"method","text":"take!(b::IOBuffer)\n\nObtain the contents of an IOBuffer as an array. Afterwards, the IOBuffer is reset to its initial state.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> write(io, \"JuliaLang is a GitHub organization.\", \" It has many members.\")\n56\n\njulia> String(take!(io))\n\"JuliaLang is a GitHub organization. It has many members.\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Pipe","page":"I/O and Network","title":"Base.Pipe","category":"type","text":"Pipe()\n\nConstruct an uninitialized Pipe object, especially for IO communication between multiple processes.\n\nThe appropriate end of the pipe will be automatically initialized if the object is used in process spawning. This can be useful to easily obtain references in process pipelines, e.g.:\n\njulia> err = Pipe()\n\n# After this `err` will be initialized and you may read `foo`'s\n# stderr from the `err` pipe, or pass `err` to other pipelines.\njulia> run(pipeline(pipeline(`foo`, stderr=err), `cat`), wait=false)\n\n# Now destroy the write half of the pipe, so that the read half will get EOF\njulia> closewrite(err)\n\njulia> read(err, String)\n\"stderr messages\"\n\nSee also Base.link_pipe!.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.link_pipe!","page":"I/O and Network","title":"Base.link_pipe!","category":"function","text":"link_pipe!(pipe; reader_supports_async=false, writer_supports_async=false)\n\nInitialize pipe and link the in endpoint to the out endpoint. The keyword arguments reader_supports_async/writer_supports_async correspond to OVERLAPPED on Windows and O_NONBLOCK on POSIX systems. They should be true unless they'll be used by an external program (e.g. the output of a command executed with run).\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.fdio","page":"I/O and Network","title":"Base.fdio","category":"function","text":"fdio([name::AbstractString, ]fd::Integer[, own::Bool=false])::IOStream\n\nCreate an IOStream object from an integer file descriptor. If own is true, closing this object will close the underlying descriptor. By default, an IOStream is closed when it is garbage collected. name allows you to associate the descriptor with a named file.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.flush","page":"I/O and Network","title":"Base.flush","category":"function","text":"flush(io::IO)\n\nCommit all currently buffered writes to the given io. This has a default implementation flush(::IO) = nothing, so may be called in generic IO code.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.close","page":"I/O and Network","title":"Base.close","category":"function","text":"close(io::IO)\n\nClose io. Performs a flush first.\n\nClosing an IO signals that its underlying resources (OS handle, network connections, etc) should be destroyed. A closed IO is in an undefined state and should not be written to or read from. When attempting to do so, the IO may throw an exception, continue to behave normally, or read/write zero bytes, depending on the implementation. However, implementations should make sure that reading to or writing from a closed IO does not cause undefined behaviour.\n\nSee also: isopen\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.closewrite","page":"I/O and Network","title":"Base.closewrite","category":"function","text":"closewrite(stream)\n\nShutdown the write half of a full-duplex I/O stream. Performs a flush first. Notify the other end that no more data will be written to the underlying file. This is not supported by all IO types.\n\nIf implemented, closewrite causes subsequent read or eof calls that would block to instead throw EOF or return true, respectively. If the stream is already closed, this is idempotent.\n\nExamples\n\njulia> io = Base.BufferStream(); # this never blocks, so we can read and write on the same Task\n\njulia> write(io, \"request\");\n\njulia> # calling `read(io)` here would block forever\n\njulia> closewrite(io);\n\njulia> read(io, String)\n\"request\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.write","page":"I/O and Network","title":"Base.write","category":"function","text":"write(io::IO, x)\n\nWrite the canonical binary representation of a value to the given I/O stream or file. Return the number of bytes written into the stream. See also print to write a text representation (with an encoding that may depend upon io).\n\nThe endianness of the written value depends on the endianness of the host system. Convert to/from a fixed endianness when writing/reading (e.g. using  htol and ltoh) to get results that are consistent across platforms.\n\nYou can write multiple values with the same write call, i.e. the following are equivalent:\n\nwrite(io, x, y...)\nwrite(io, x) + write(io, y...)\n\nExamples\n\nConsistent serialization:\n\njulia> fname = tempname(); # random temporary filename\n\njulia> open(fname,\"w\") do f\n           # Make sure we write 64bit integer in little-endian byte order\n           write(f,htol(Int64(42)))\n       end\n8\n\njulia> open(fname,\"r\") do f\n           # Convert back to host byte order and host integer type\n           Int(ltoh(read(f,Int64)))\n       end\n42\n\nMerging write calls:\n\njulia> io = IOBuffer();\n\njulia> write(io, \"JuliaLang is a GitHub organization.\", \" It has many members.\")\n56\n\njulia> takestring!(io)\n\"JuliaLang is a GitHub organization. It has many members.\"\n\njulia> write(io, \"Sometimes those members\") + write(io, \" write documentation.\")\n44\n\njulia> takestring!(io)\n\"Sometimes those members write documentation.\"\n\nUser-defined plain-data types without write methods can be written when wrapped in a Ref:\n\njulia> struct MyStruct; x::Float64; end\n\njulia> io = IOBuffer()\nIOBuffer(data=UInt8[...], readable=true, writable=true, seekable=true, append=false, size=0, maxsize=Inf, ptr=1, mark=-1)\n\njulia> write(io, Ref(MyStruct(42.0)))\n8\n\njulia> seekstart(io); read!(io, Ref(MyStruct(NaN)))\nBase.RefValue{MyStruct}(MyStruct(42.0))\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.read","page":"I/O and Network","title":"Base.read","category":"function","text":"read(command::Cmd, String)\n\nRun command and return the resulting output as a String.\n\n\n\n\n\nread(command::Cmd)\n\nRun command and return the resulting output as an array of bytes.\n\n\n\n\n\nread(s::IOStream, nb::Integer; all=true)\n\nRead at most nb bytes from s, returning a Vector{UInt8} of the bytes read.\n\nIf all is true (the default), this function will block repeatedly trying to read all requested bytes, until an error or end-of-file occurs. If all is false, at most one read call is performed, and the amount of data returned is device-dependent. Note that not all stream types support the all option.\n\n\n\n\n\nread(s::IO, nb=typemax(Int))\n\nRead at most nb bytes from s, returning a Vector{UInt8} of the bytes read.\n\n\n\n\n\nread(filename::AbstractString)\n\nRead the entire contents of a file as a Vector{UInt8}.\n\nread(filename::AbstractString, String)\n\nRead the entire contents of a file as a string.\n\nread(filename::AbstractString, args...)\n\nOpen a file and read its contents. args is passed to read: this is equivalent to open(io->read(io, args...), filename).\n\n\n\n\n\nread(io::IO, T)\n\nRead a single value of type T from io, in canonical binary representation.\n\nNote that Julia does not convert the endianness for you. Use ntoh or ltoh for this purpose.\n\nread(io::IO, String)\n\nRead the entirety of io, as a String (see also readchomp).\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization\");\n\njulia> read(io, Char)\n'J': ASCII/Unicode U+004A (category Lu: Letter, uppercase)\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization\");\n\njulia> read(io, String)\n\"JuliaLang is a GitHub organization\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.read!","page":"I/O and Network","title":"Base.read!","category":"function","text":"read!(stream::IO, array::AbstractArray)\nread!(filename::AbstractString, array::AbstractArray)\n\nRead binary data from an I/O stream or file, filling in array.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readbytes!","page":"I/O and Network","title":"Base.readbytes!","category":"function","text":"readbytes!(stream::IOStream, b::AbstractVector{UInt8}, nb=length(b); all::Bool=true)\n\nRead at most nb bytes from stream into b, returning the number of bytes read. The size of b will be increased if needed (i.e. if nb is greater than length(b) and enough bytes could be read), but it will never be decreased.\n\nIf all is true (the default), this function will block repeatedly trying to read all requested bytes, until an error or end-of-file occurs. If all is false, at most one read call is performed, and the amount of data returned is device-dependent. Note that not all stream types support the all option.\n\n\n\n\n\nreadbytes!(stream::IO, b::AbstractVector{UInt8}, nb=length(b))\n\nRead at most nb bytes from stream into b, returning the number of bytes read. The size of b will be increased if needed (i.e. if nb is greater than length(b) and enough bytes could be read), but it will never be decreased.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.unsafe_read","page":"I/O and Network","title":"Base.unsafe_read","category":"function","text":"unsafe_read(io::IO, ref, nbytes::UInt)\n\nCopy nbytes from the IO stream object into ref (converted to a pointer).\n\nIt is recommended that subtypes T<:IO override the following method signature to provide more efficient implementations: unsafe_read(s::T, p::Ptr{UInt8}, n::UInt)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.unsafe_write","page":"I/O and Network","title":"Base.unsafe_write","category":"function","text":"unsafe_write(io::IO, ref, nbytes::UInt)\n\nCopy nbytes from ref (converted to a pointer) into the IO object.\n\nIt is recommended that subtypes T<:IO override the following method signature to provide more efficient implementations: unsafe_write(s::T, p::Ptr{UInt8}, n::UInt)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readeach","page":"I/O and Network","title":"Base.readeach","category":"function","text":"readeach(io::IO, T)\n\nReturn an iterable object yielding read(io, T).\n\nSee also skipchars, eachline, readuntil.\n\ncompat: Julia 1.6\nreadeach requires Julia 1.6 or later.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\\n It has many members.\\n\");\n\njulia> for c in readeach(io, Char)\n           c == '\\n' && break\n           print(c)\n       end\nJuliaLang is a GitHub organization.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.peek","page":"I/O and Network","title":"Base.peek","category":"function","text":"peek(stream[, T=UInt8])\n\nRead and return a value of type T from a stream without advancing the current position in the stream.   See also startswith(stream, char_or_string).\n\nExamples\n\njulia> b = IOBuffer(\"julia\");\n\njulia> peek(b)\n0x6a\n\njulia> position(b)\n0\n\njulia> peek(b, Char)\n'j': ASCII/Unicode U+006A (category Ll: Letter, lowercase)\n\ncompat: Julia 1.5\nThe method which accepts a type requires Julia 1.5 or later.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.position","page":"I/O and Network","title":"Base.position","category":"function","text":"position(s)\n\nGet the current position of a stream.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> seek(io, 5);\n\njulia> position(io)\n5\n\njulia> skip(io, 10);\n\njulia> position(io)\n15\n\njulia> seekend(io);\n\njulia> position(io)\n35\n\n\n\n\n\nposition(l::Lexer)\n\nReturns the current position.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.seek","page":"I/O and Network","title":"Base.seek","category":"function","text":"seek(s, pos)\n\nSeek a stream to the given position.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> seek(io, 5);\n\njulia> read(io, Char)\n'L': ASCII/Unicode U+004C (category Lu: Letter, uppercase)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.seekstart","page":"I/O and Network","title":"Base.seekstart","category":"function","text":"seekstart(s)\n\nSeek a stream to its beginning.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> seek(io, 5);\n\njulia> read(io, Char)\n'L': ASCII/Unicode U+004C (category Lu: Letter, uppercase)\n\njulia> seekstart(io);\n\njulia> read(io, Char)\n'J': ASCII/Unicode U+004A (category Lu: Letter, uppercase)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.seekend","page":"I/O and Network","title":"Base.seekend","category":"function","text":"seekend(s)\n\nSeek a stream to its end.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.skip","page":"I/O and Network","title":"Base.skip","category":"function","text":"skip(s, offset)\n\nSeek a stream relative to the current position.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> seek(io, 5);\n\njulia> skip(io, 10);\n\njulia> read(io, Char)\n'G': ASCII/Unicode U+0047 (category Lu: Letter, uppercase)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.mark","page":"I/O and Network","title":"Base.mark","category":"function","text":"mark(s::IO)\n\nAdd a mark at the current position of stream s. Return the marked position.\n\nSee also unmark, reset, ismarked.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.unmark","page":"I/O and Network","title":"Base.unmark","category":"function","text":"unmark(s::IO)\n\nRemove a mark from stream s. Return true if the stream was marked, false otherwise.\n\nSee also mark, reset, ismarked.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.reset-Tuple{IO}","page":"I/O and Network","title":"Base.reset","category":"method","text":"reset(s::IO)\n\nReset a stream s to a previously marked position, and remove the mark. Return the previously marked position. Throw an error if the stream is not marked.\n\nSee also mark, unmark, ismarked.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.ismarked","page":"I/O and Network","title":"Base.ismarked","category":"function","text":"ismarked(s::IO)\n\nReturn true if stream s is marked.\n\nSee also mark, unmark, reset.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.eof","page":"I/O and Network","title":"Base.eof","category":"function","text":"eof(stream)::Bool\n\nTest whether an I/O stream is at end-of-file. If the stream is not yet exhausted, this function will block to wait for more data if necessary, and then return false. Therefore it is always safe to read one byte after seeing eof return false. eof will return false as long as buffered data is still available, even if the remote end of a connection is closed.\n\nExamples\n\njulia> b = IOBuffer(\"my buffer\");\n\njulia> eof(b)\nfalse\n\njulia> seekend(b);\n\njulia> eof(b)\ntrue\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.isreadonly","page":"I/O and Network","title":"Base.isreadonly","category":"function","text":"isreadonly(io)::Bool\n\nDetermine whether a stream is read-only.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization\");\n\njulia> isreadonly(io)\ntrue\n\njulia> io = IOBuffer();\n\njulia> isreadonly(io)\nfalse\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.iswritable","page":"I/O and Network","title":"Base.iswritable","category":"function","text":"iswritable(io)::Bool\n\nReturn false if the specified IO object is not writable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           iswritable(io)\n       end\ntrue\n\njulia> open(\"myfile.txt\", \"r\") do io\n           iswritable(io)\n       end\nfalse\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\niswritable(path::String)\n\nReturn true if the access permissions for the given path permitted writing by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling iswritable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, isreadable.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.isreadable","page":"I/O and Network","title":"Base.isreadable","category":"function","text":"isreadable(io)::Bool\n\nReturn false if the specified IO object is not readable.\n\nExamples\n\njulia> open(\"myfile.txt\", \"w\") do io\n           print(io, \"Hello world!\");\n           isreadable(io)\n       end\nfalse\n\njulia> open(\"myfile.txt\", \"r\") do io\n           isreadable(io)\n       end\ntrue\n\njulia> rm(\"myfile.txt\")\n\n\n\n\n\nisreadable(path::String)\n\nReturn true if the access permissions for the given path permitted reading by the current user.\n\nnote: Note\nThis permission may change before the user calls open, so it is recommended to just call open alone and handle the error if that fails, rather than calling isreadable first.\n\nnote: Note\nCurrently this function does not correctly interrogate filesystem ACLs on Windows, therefore it can return wrong results.\n\ncompat: Julia 1.11\nThis function requires at least Julia 1.11.\n\nSee also ispath, isexecutable, iswritable.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.isexecutable","page":"I/O and Network","title":"Base.isexecutable","category":"function","text":"isexecutable(path::String)\n\nReturn true if the given path has executable permissions.\n\nnote: Note\nThis permission may change before the user executes path, so it is recommended to execute the file and handle the error if that fails, rather than calling isexecutable first.\n\nnote: Note\nPrior to Julia 1.6, this did not correctly interrogate filesystem ACLs on Windows, therefore it would return true for any file.  From Julia 1.6 on, it correctly determines whether the file is marked as executable or not.\n\nSee also ispath, isreadable, iswritable.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.isopen","page":"I/O and Network","title":"Base.isopen","category":"function","text":"isopen(object)::Bool\n\nDetermine whether an object, such as an IO or timer, is still open and hence active.\n\nSee also: close\n\nExamples\n\njulia> io = open(\"my_file.txt\", \"w+\");\n\njulia> isopen(io)\ntrue\n\njulia> close(io)\n\njulia> isopen(io)\nfalse\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.fd","page":"I/O and Network","title":"Base.fd","category":"function","text":"fd(x)::RawFD\n\nReturn the file descriptor backing the stream, file, or socket.\n\nRawFD objects can be passed directly to other languages via the ccall interface.\n\ncompat: Julia 1.12\nPrior to 1.12, this function returned an Int instead of a RawFD. You may use RawFD(fd(x)) to produce a RawFD in all Julia versions.\n\ncompat: Julia 1.12\nGetting the file descriptor of sockets are supported as of Julia 1.12.\n\nwarning: Warning\nDuplicate the returned file descriptor with Libc.dup() before passing it to another system that will take ownership of it (e.g. a C library). Otherwise both the Julia object x and the other system may try to close the file descriptor, which will cause errors.\n\nwarning: Warning\nThe file descriptors for sockets are asynchronous (i.e. O_NONBLOCK on POSIX and OVERLAPPED on Windows), they may behave differently than regular file descriptors.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stdio","page":"I/O and Network","title":"Base.redirect_stdio","category":"function","text":"redirect_stdio(f; stdin=nothing, stderr=nothing, stdout=nothing)\n\nRedirect a subset of the streams stdin, stderr, stdout, call f() and restore each stream.\n\nPossible values for each stream are:\n\nnothing indicating the stream should not be redirected.\npath::AbstractString redirecting the stream to the file at path.\nio an IOStream, TTY, Pipe, socket, or devnull.\n\nExamples\n\njulia> redirect_stdio(stdout=\"stdout.txt\", stderr=\"stderr.txt\") do\n           print(\"hello stdout\")\n           print(stderr, \"hello stderr\")\n       end\n\njulia> read(\"stdout.txt\", String)\n\"hello stdout\"\n\njulia> read(\"stderr.txt\", String)\n\"hello stderr\"\n\nEdge cases\n\nIt is possible to pass the same argument to stdout and stderr:\n\njulia> redirect_stdio(stdout=\"log.txt\", stderr=\"log.txt\", stdin=devnull) do\n    ...\nend\n\nHowever it is not supported to pass two distinct descriptors of the same file.\n\njulia> io1 = open(\"same/path\", \"w\")\n\njulia> io2 = open(\"same/path\", \"w\")\n\njulia> redirect_stdio(f, stdout=io1, stderr=io2) # not supported\n\nAlso the stdin argument may not be the same descriptor as stdout or stderr.\n\njulia> io = open(...)\n\njulia> redirect_stdio(f, stdout=io, stdin=io) # not supported\n\ncompat: Julia 1.7\nredirect_stdio requires Julia 1.7 or later.\n\n\n\n\n\nredirect_stdio(;stdin=stdin, stderr=stderr, stdout=stdout)\n\nRedirect a subset of the streams stdin, stderr, stdout. Each argument must be an IOStream, TTY, Pipe, socket, or devnull.\n\ncompat: Julia 1.7\nredirect_stdio requires Julia 1.7 or later.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stdout","page":"I/O and Network","title":"Base.redirect_stdout","category":"function","text":"redirect_stdout([stream]) -> stream\n\nCreate a pipe to which all C and Julia level stdout output will be redirected. Return a stream representing the pipe ends. Data written to stdout may now be read from the rd end of the pipe.\n\nnote: Note\nstream must be a compatible objects, such as an IOStream, TTY, Pipe, socket, or devnull.\n\nSee also redirect_stdio.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stdout-Tuple{Function, Any}","page":"I/O and Network","title":"Base.redirect_stdout","category":"method","text":"redirect_stdout(f::Function, stream)\n\nRun the function f while redirecting stdout to stream. Upon completion, stdout is restored to its prior setting.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stderr","page":"I/O and Network","title":"Base.redirect_stderr","category":"function","text":"redirect_stderr([stream]) -> stream\n\nLike redirect_stdout, but for stderr.\n\nnote: Note\nstream must be a compatible objects, such as an IOStream, TTY, Pipe, socket, or devnull.\n\nSee also redirect_stdio.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stderr-Tuple{Function, Any}","page":"I/O and Network","title":"Base.redirect_stderr","category":"method","text":"redirect_stderr(f::Function, stream)\n\nRun the function f while redirecting stderr to stream. Upon completion, stderr is restored to its prior setting.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stdin","page":"I/O and Network","title":"Base.redirect_stdin","category":"function","text":"redirect_stdin([stream]) -> stream\n\nLike redirect_stdout, but for stdin. Note that the direction of the stream is reversed.\n\nnote: Note\nstream must be a compatible objects, such as an IOStream, TTY, Pipe, socket, or devnull.\n\nSee also redirect_stdio.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.redirect_stdin-Tuple{Function, Any}","page":"I/O and Network","title":"Base.redirect_stdin","category":"method","text":"redirect_stdin(f::Function, stream)\n\nRun the function f while redirecting stdin to stream. Upon completion, stdin is restored to its prior setting.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readchomp","page":"I/O and Network","title":"Base.readchomp","category":"function","text":"readchomp(x)\n\nRead the entirety of x as a string and remove a single trailing newline if there is one. Equivalent to chomp(read(x, String)).\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> readchomp(\"my_file.txt\")\n\"JuliaLang is a GitHub organization.\\nIt has many members.\"\n\njulia> rm(\"my_file.txt\");\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.truncate","page":"I/O and Network","title":"Base.truncate","category":"function","text":"truncate(file, n)\n\nResize the file or buffer given by the first argument to exactly n bytes, filling previously unallocated space with '\\0' if the file or buffer is grown.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> write(io, \"JuliaLang is a GitHub organization.\")\n35\n\njulia> truncate(io, 15)\nIOBuffer(data=UInt8[...], readable=true, writable=true, seekable=true, append=false, size=15, maxsize=Inf, ptr=16, mark=-1)\n\njulia> takestring!(io)\n\"JuliaLang is a \"\n\njulia> io = IOBuffer();\n\njulia> write(io, \"JuliaLang is a GitHub organization.\");\n\njulia> truncate(io, 40);\n\njulia> takestring!(io)\n\"JuliaLang is a GitHub organization.\\0\\0\\0\\0\\0\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.skipchars","page":"I/O and Network","title":"Base.skipchars","category":"function","text":"skipchars(predicate, io::IO; linecomment=nothing)\n\nAdvance the stream io such that the next-read character will be the first remaining for which predicate returns false. If the keyword argument linecomment is specified, all characters from that character until the start of the next line are ignored.\n\nExamples\n\njulia> buf = IOBuffer(\"    text\")\nIOBuffer(data=UInt8[...], readable=true, writable=false, seekable=true, append=false, size=8, maxsize=Inf, ptr=1, mark=-1)\n\njulia> skipchars(isspace, buf)\nIOBuffer(data=UInt8[...], readable=true, writable=false, seekable=true, append=false, size=8, maxsize=Inf, ptr=5, mark=-1)\n\njulia> String(readavailable(buf))\n\"text\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.countlines","page":"I/O and Network","title":"Base.countlines","category":"function","text":"countlines(io::IO; eol::AbstractChar = '\\n')\ncountlines(filename::AbstractString; eol::AbstractChar = '\\n')\n\nRead io until the end of the stream/file and count the number of lines. To specify a file pass the filename as the first argument. EOL markers other than '\\n' are supported by passing them as the second argument.  The last non-empty line of io is counted even if it does not end with the EOL, matching the length returned by eachline and readlines.\n\nTo count lines of a String, countlines(IOBuffer(str)) can be used.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\\n\");\n\njulia> countlines(io)\n1\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> countlines(io)\n1\n\njulia> eof(io) # counting lines moves the file pointer\ntrue\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization.\");\n\njulia> countlines(io, eol = '.')\n1\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\n\")\n36\n\njulia> countlines(\"my_file.txt\")\n1\n\njulia> countlines(\"my_file.txt\", eol = 'n')\n4\n\njulia> rm(\"my_file.txt\")\n\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.PipeBuffer","page":"I/O and Network","title":"Base.PipeBuffer","category":"function","text":"PipeBuffer(data::AbstractVector{UInt8}=UInt8[]; maxsize::Integer = typemax(Int))\n\nAn IOBuffer that allows reading and performs writes by appending. Seeking and truncating are not supported. See IOBuffer for the available constructors. If data is given, creates a PipeBuffer to operate on a data vector, optionally specifying a size beyond which the underlying Array may not be grown.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readavailable","page":"I/O and Network","title":"Base.readavailable","category":"function","text":"readavailable(stream)\n\nRead available buffered data from a stream. Actual I/O is performed only if no data has already been buffered. The result is a Vector{UInt8}.\n\nwarning: Warning\nThe amount of data returned is implementation-dependent; for example it can depend on the internal choice of buffer size. Other functions such as read should generally be used instead.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.IOContext","page":"I/O and Network","title":"Base.IOContext","category":"type","text":"IOContext\n\nIOContext provides a mechanism for passing output configuration settings among show methods.\n\nIn short, it is an immutable dictionary that is a subclass of IO. It supports standard dictionary operations such as getindex, and can also be used as an I/O stream.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.IOContext-Tuple{IO, Pair}","page":"I/O and Network","title":"Base.IOContext","category":"method","text":"IOContext(io::IO, KV::Pair...)\n\nCreate an IOContext that wraps a given stream, adding the specified key=>value pairs to the properties of that stream (note that io can itself be an IOContext).\n\nuse (key => value) in io to see if this particular combination is in the properties set\nuse get(io, key, default) to retrieve the most recent value for a particular key\n\nThe following properties are in common use:\n\n:compact: Boolean specifying that values should be printed more compactly, e.g. that numbers should be printed with fewer digits. This is set when printing array elements. :compact output should not contain line breaks.\n:limit: Boolean specifying that containers should be truncated, e.g. showing … in place of most elements.\n:displaysize: A Tuple{Int,Int} giving the size in rows and columns to use for text output. This can be used to override the display size for called functions, but to get the size of the screen use the displaysize function.\n:typeinfo: a Type characterizing the information already printed concerning the type of the object about to be displayed. This is mainly useful when displaying a collection of objects of the same type, so that redundant type information can be avoided (e.g. [Float16(0)] can be shown as \"Float16[0.0]\" instead of \"Float16[Float16(0.0)]\" : while displaying the elements of the array, the :typeinfo property will be set to Float16).\n:color: Boolean specifying whether ANSI color/escape codes are supported/expected. By default, this is determined by whether io is a compatible terminal and by any --color command-line flag when julia was launched.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> printstyled(IOContext(io, :color => true), \"string\", color=:red)\n\njulia> takestring!(io)\n\"\\e[31mstring\\e[39m\"\n\njulia> printstyled(io, \"string\", color=:red)\n\njulia> takestring!(io)\n\"string\"\n\njulia> print(IOContext(stdout, :compact => false), 1.12341234)\n1.12341234\njulia> print(IOContext(stdout, :compact => true), 1.12341234)\n1.12341\n\njulia> function f(io::IO)\n           if get(io, :short, false)\n               print(io, \"short\")\n           else\n               print(io, \"loooooong\")\n           end\n       end\nf (generic function with 1 method)\n\njulia> f(stdout)\nloooooong\njulia> f(IOContext(stdout, :short => true))\nshort\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.IOContext-Tuple{IO, IOContext}","page":"I/O and Network","title":"Base.IOContext","category":"method","text":"IOContext(io::IO, context::IOContext)\n\nCreate an IOContext that wraps an alternate IO but inherits the properties of context.\n\nnote: Note\nUnless explicitly set in the wrapped io the displaysize of io will not be inherited. This is because by default displaysize is not a property of IO objects themselves, but lazily inferred, as the size of the terminal window can change during the lifetime of the IO object.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.show-Tuple{IO, Any}","page":"I/O and Network","title":"Base.show","category":"method","text":"show([io::IO = stdout], x)\n\nWrite a text representation of a value x to the output stream io. New types T should overload show(io::IO, x::T). The representation used by show generally includes Julia-specific formatting and type information, and should be parseable Julia code when possible.\n\nrepr returns the output of show as a string.\n\nFor a more verbose human-readable text output for objects of type T, define show(io::IO, ::MIME\"text/plain\", ::T) in addition. Checking the :compact IOContext key (often checked as get(io, :compact, false)::Bool) of io in such methods is recommended, since some containers show their elements by calling this method with :compact => true.\n\nSee also print, which writes un-decorated representations.\n\nExamples\n\njulia> show(\"Hello World!\")\n\"Hello World!\"\njulia> print(\"Hello World!\")\nHello World!\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.summary","page":"I/O and Network","title":"Base.summary","category":"function","text":"summary(io::IO, x)\nstr = summary(x)\n\nPrint to a stream io, or return a string str, giving a brief description of a value. By default returns string(typeof(x)), e.g. Int64.\n\nFor arrays, returns a string of size and type info, e.g. 10-element Vector{Int64} or 9×4×5 Array{Float64, 3}.\n\nExamples\n\njulia> summary(1)\n\"Int64\"\n\njulia> summary(zeros(2))\n\"2-element Vector{Float64}\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.print","page":"I/O and Network","title":"Base.print","category":"function","text":"print([io::IO], xs...)\n\nWrite to io (or to the default output stream stdout if io is not given) a canonical (un-decorated) text representation. The representation used by print includes minimal formatting and tries to avoid Julia-specific details.\n\nprint falls back to calling the 2-argument show(io, x) for each argument x in xs, so most types should just define show. Define print if your type has a separate \"plain\" representation.  For example, show displays strings with quotes, and print displays strings without quotes.\n\nSee also println, string, printstyled.\n\nExamples\n\njulia> print(\"Hello World!\")\nHello World!\njulia> io = IOBuffer();\n\njulia> print(io, \"Hello\", ' ', :World!)\n\njulia> takestring!(io)\n\"Hello World!\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.println","page":"I/O and Network","title":"Base.println","category":"function","text":"println([io::IO], xs...)\n\nPrint (using print) xs to io followed by a newline. If io is not supplied, prints to the default output stream stdout.\n\nSee also printstyled to add colors etc.\n\nExamples\n\njulia> println(\"Hello, world\")\nHello, world\n\njulia> io = IOBuffer();\n\njulia> println(io, \"Hello\", ',', \" world.\")\n\njulia> takestring!(io)\n\"Hello, world.\\n\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.printstyled","page":"I/O and Network","title":"Base.printstyled","category":"function","text":"printstyled([io], xs...; bold::Bool=false, italic::Bool=false, underline::Bool=false, blink::Bool=false, reverse::Bool=false, hidden::Bool=false, color::Union{Symbol,Int}=:normal)\n\nPrint xs in a color specified as a symbol or integer, optionally in bold.\n\nKeyword color may take any of the values :normal, :italic, :default, :bold, :black, :blink, :blue, :cyan, :green, :hidden, :light_black, :light_blue, :light_cyan, :light_green, :light_magenta, :light_red, :light_white, :light_yellow, :magenta, :nothing, :red, :reverse, :underline, :white, or  :yellow or an integer between 0 and 255 inclusive. Note that not all terminals support 256 colors.\n\nKeywords bold=true, italic=true, underline=true, blink=true are self-explanatory. Keyword reverse=true prints with foreground and background colors exchanged, and hidden=true should be invisible in the terminal but can still be copied. These properties can be used in any combination.\n\nSee also print, println, show.\n\nnote: Note\nNot all terminals support italic output. Some terminals interpret italic as reverse or blink.\n\ncompat: Julia 1.7\nKeywords except color and bold were added in Julia 1.7.\n\ncompat: Julia 1.10\nSupport for italic output was added in Julia 1.10.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.sprint","page":"I/O and Network","title":"Base.sprint","category":"function","text":"sprint(f::Function, args...; context=nothing, sizehint=0)\n\nCall the given function with an I/O stream and the supplied extra arguments. Everything written to this I/O stream is returned as a string.\n\nThe optional keyword argument context can be set to a :key=>value pair, a tuple of :key=>value pairs, or an IO or IOContext object whose attributes are used for the I/O stream passed to f.  The optional sizehint is a suggested size (in bytes) to allocate for the buffer used to write the string.\n\ncompat: Julia 1.7\nPassing a tuple to keyword context requires Julia 1.7 or later.\n\nExamples\n\njulia> sprint(show, 66.66666; context=:compact => true)\n\"66.6667\"\n\njulia> sprint(showerror, BoundsError([1], 100))\n\"BoundsError: attempt to access 1-element Vector{Int64} at index [100]\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.showerror","page":"I/O and Network","title":"Base.showerror","category":"function","text":"showerror(io, e)\n\nShow a descriptive representation of an exception object e. This method is used to display the exception after a call to throw.\n\nExamples\n\njulia> struct MyException <: Exception\n           msg::String\n       end\n\njulia> function Base.showerror(io::IO, err::MyException)\n           print(io, \"MyException: \")\n           print(io, err.msg)\n       end\n\njulia> err = MyException(\"test exception\")\nMyException(\"test exception\")\n\njulia> sprint(showerror, err)\n\"MyException: test exception\"\n\njulia> throw(MyException(\"test exception\"))\nERROR: MyException: test exception\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.dump","page":"I/O and Network","title":"Base.dump","category":"function","text":"dump(x; maxdepth=8)\n\nShow every part of the representation of a value. The depth of the output is truncated at maxdepth.\n\nExamples\n\njulia> struct MyStruct\n           x\n           y\n       end\n\njulia> x = MyStruct(1, (2,3));\n\njulia> dump(x)\nMyStruct\n  x: Int64 1\n  y: Tuple{Int64, Int64}\n    1: Int64 2\n    2: Int64 3\n\njulia> dump(x; maxdepth = 1)\nMyStruct\n  x: Int64 1\n  y: Tuple{Int64, Int64}\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Meta.@dump","page":"I/O and Network","title":"Base.Meta.@dump","category":"macro","text":"@dump expr\n\nShow every part of the representation of the given expression. Equivalent to dump(:(expr)).\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readline","page":"I/O and Network","title":"Base.readline","category":"function","text":"readline(io::IO=stdin; keep::Bool=false)\nreadline(filename::AbstractString; keep::Bool=false)\n\nRead a single line of text from the given I/O stream or file (defaults to stdin). When reading from a file, the text is assumed to be encoded in UTF-8. Lines in the input end with '\\n' or \"\\r\\n\" or the end of an input stream. When keep is false (as it is by default), these trailing newline characters are removed from the line before it is returned. When keep is true, they are returned as part of the line.\n\nReturn a String.   See also copyline to instead write in-place to another stream (which can be a preallocated IOBuffer).\n\nSee also readuntil for reading until more general delimiters.\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> readline(\"my_file.txt\")\n\"JuliaLang is a GitHub organization.\"\n\njulia> readline(\"my_file.txt\", keep=true)\n\"JuliaLang is a GitHub organization.\\n\"\n\njulia> rm(\"my_file.txt\")\n\njulia> print(\"Enter your name: \")\nEnter your name:\n\njulia> your_name = readline()\nLogan\n\"Logan\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readuntil","page":"I/O and Network","title":"Base.readuntil","category":"function","text":"readuntil(stream::IO, delim; keep::Bool = false)\nreaduntil(filename::AbstractString, delim; keep::Bool = false)\n\nRead a string from an I/O stream or a file, up to the given delimiter. The delimiter can be a UInt8, AbstractChar, string, or vector. Keyword argument keep controls whether the delimiter is included in the result. The text is assumed to be encoded in UTF-8.\n\nReturn a String if delim is an AbstractChar or a string or otherwise return a Vector{typeof(delim)}.   See also copyuntil to instead write in-place to another stream (which can be a preallocated IOBuffer).\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> readuntil(\"my_file.txt\", 'L')\n\"Julia\"\n\njulia> readuntil(\"my_file.txt\", '.', keep = true)\n\"JuliaLang is a GitHub organization.\"\n\njulia> rm(\"my_file.txt\")\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.readlines","page":"I/O and Network","title":"Base.readlines","category":"function","text":"readlines(io::IO=stdin; keep::Bool=false)\nreadlines(filename::AbstractString; keep::Bool=false)\n\nRead all lines of an I/O stream or a file as a vector of strings. Behavior is equivalent to saving the result of reading readline repeatedly with the same arguments and saving the resulting lines as a vector of strings.  See also eachline to iterate over the lines without reading them all at once.\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> readlines(\"my_file.txt\")\n2-element Vector{String}:\n \"JuliaLang is a GitHub organization.\"\n \"It has many members.\"\n\njulia> readlines(\"my_file.txt\", keep=true)\n2-element Vector{String}:\n \"JuliaLang is a GitHub organization.\\n\"\n \"It has many members.\\n\"\n\njulia> rm(\"my_file.txt\")\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.eachline","page":"I/O and Network","title":"Base.eachline","category":"function","text":"eachline(io::IO=stdin; keep::Bool=false)\neachline(filename::AbstractString; keep::Bool=false)\n\nCreate an iterable EachLine object that will yield each line from an I/O stream or a file. Iteration calls readline on the stream argument repeatedly with keep passed through, determining whether trailing end-of-line characters are retained. When called with a file name, the file is opened once at the beginning of iteration and closed at the end. If iteration is interrupted, the file will be closed when the EachLine object is garbage collected.\n\nTo iterate over each line of a String, eachline(IOBuffer(str)) can be used.\n\nIterators.reverse can be used on an EachLine object to read the lines in reverse order (for files, buffers, and other I/O streams supporting seek), and first or last can be used to extract the initial or final lines, respectively.\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\n It has many members.\\n\");\n\njulia> for line in eachline(\"my_file.txt\")\n           print(line)\n       end\nJuliaLang is a GitHub organization. It has many members.\n\njulia> rm(\"my_file.txt\");\n\ncompat: Julia 1.8\nJulia 1.8 is required to use Iterators.reverse or last with eachline iterators.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.copyline","page":"I/O and Network","title":"Base.copyline","category":"function","text":"copyline(out::IO, io::IO=stdin; keep::Bool=false)\ncopyline(out::IO, filename::AbstractString; keep::Bool=false)\n\nCopy a single line of text from an I/O stream or a file to the out stream, returning out.\n\nWhen reading from a file, the text is assumed to be encoded in UTF-8. Lines in the input end with '\\n' or \"\\r\\n\" or the end of an input stream. When keep is false (as it is by default), these trailing newline characters are removed from the line before it is returned. When keep is true, they are returned as part of the line.\n\nSimilar to readline, which returns a String; in contrast, copyline writes directly to out, without allocating a string. (This can be used, for example, to read data into a pre-allocated IOBuffer.)\n\nSee also copyuntil for reading until more general delimiters.\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> takestring!(copyline(IOBuffer(), \"my_file.txt\"))\n\"JuliaLang is a GitHub organization.\"\n\njulia> takestring!(copyline(IOBuffer(), \"my_file.txt\", keep=true))\n\"JuliaLang is a GitHub organization.\\n\"\n\njulia> rm(\"my_file.txt\")\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.copyuntil","page":"I/O and Network","title":"Base.copyuntil","category":"function","text":"copyuntil(out::IO, stream::IO, delim; keep::Bool = false)\ncopyuntil(out::IO, filename::AbstractString, delim; keep::Bool = false)\n\nCopy a string from an I/O stream or a file, up to the given delimiter, to the out stream, returning out. The delimiter can be a UInt8, AbstractChar, string, or vector. Keyword argument keep controls whether the delimiter is included in the result. The text is assumed to be encoded in UTF-8.\n\nSimilar to readuntil, which returns a String; in contrast, copyuntil writes directly to out, without allocating a string. (This can be used, for example, to read data into a pre-allocated IOBuffer.)\n\nExamples\n\njulia> write(\"my_file.txt\", \"JuliaLang is a GitHub organization.\\nIt has many members.\\n\");\n\njulia> takestring!(copyuntil(IOBuffer(), \"my_file.txt\", 'L'))\n\"Julia\"\n\njulia> takestring!(copyuntil(IOBuffer(), \"my_file.txt\", '.', keep = true))\n\"JuliaLang is a GitHub organization.\"\n\njulia> rm(\"my_file.txt\")\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.displaysize","page":"I/O and Network","title":"Base.displaysize","category":"function","text":"displaysize([io::IO]) -> (lines, columns)\n\nReturn the nominal size of the screen that may be used for rendering output to this IO object. If no input is provided, the environment variables LINES and COLUMNS are read. If those are not set, a default size of (24, 80) is returned.\n\nExamples\n\njulia> withenv(\"LINES\" => 30, \"COLUMNS\" => 100) do\n           displaysize()\n       end\n(30, 100)\n\nTo get your TTY size,\n\njulia> displaysize(stdout)\n(34, 147)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.AbstractDisplay","page":"I/O and Network","title":"Base.Multimedia.AbstractDisplay","category":"type","text":"AbstractDisplay\n\nAbstract supertype for rich display output devices. TextDisplay is a subtype of this.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.display","page":"I/O and Network","title":"Base.Multimedia.display","category":"function","text":"display(x)\ndisplay(d::AbstractDisplay, x)\ndisplay(mime, x)\ndisplay(d::AbstractDisplay, mime, x)\n\nDisplay x using the topmost applicable display in the display stack, typically using the richest supported multimedia output for x, with plain-text stdout output as a fallback. The display(d, x) variant attempts to display x on the given display d only, throwing a MethodError if d cannot display objects of this type.\n\nIn general, you cannot assume that display output goes to stdout (unlike print(x) or show(x)).  For example, display(x) may open up a separate window with an image. display(x) means \"show x in the best way you can for the current output device(s).\" If you want REPL-like text output that is guaranteed to go to stdout, use show(stdout, \"text/plain\", x) instead.\n\nThere are also two variants with a mime argument (a MIME type string, such as \"image/png\"), which attempt to display x using the requested MIME type only, throwing a MethodError if this type is not supported by either the display(s) or by x. With these variants, one can also supply the \"raw\" data in the requested MIME type by passing x::AbstractString (for MIME types with text-based storage, such as text/html or application/postscript) or x::Vector{UInt8} (for binary MIME types).\n\nTo customize how instances of a type are displayed, overload show rather than display, as explained in the manual section on custom pretty-printing.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.redisplay","page":"I/O and Network","title":"Base.Multimedia.redisplay","category":"function","text":"redisplay(x)\nredisplay(d::AbstractDisplay, x)\nredisplay(mime, x)\nredisplay(d::AbstractDisplay, mime, x)\n\nBy default, the redisplay functions simply call display. However, some display backends may override redisplay to modify an existing display of x (if any). Using redisplay is also a hint to the backend that x may be redisplayed several times, and the backend may choose to defer the display until (for example) the next interactive prompt.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.displayable","page":"I/O and Network","title":"Base.Multimedia.displayable","category":"function","text":"displayable(mime)::Bool\ndisplayable(d::AbstractDisplay, mime)::Bool\n\nReturn a boolean value indicating whether the given mime type (string) is displayable by any of the displays in the current display stack, or specifically by the display d in the second variant.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.show-Tuple{IO, Any, Any}","page":"I/O and Network","title":"Base.show","category":"method","text":"show(io::IO, mime, x)\n\nThe display functions ultimately call show in order to write an object x as a given mime type to a given I/O stream io (usually a memory buffer), if possible. In order to provide a rich multimedia representation of a user-defined type T, it is only necessary to define a new show method for T, via: show(io, ::MIME\"mime\", x::T) = ..., where mime is a MIME-type string and the function body calls write (or similar) to write that representation of x to io. (Note that the MIME\"\" notation only supports literal strings; to construct MIME types in a more flexible manner use MIME{Symbol(\"\")}.)\n\nFor example, if you define a MyImage type and know how to write it to a PNG file, you could define a function show(io, ::MIME\"image/png\", x::MyImage) = ... to allow your images to be displayed on any PNG-capable AbstractDisplay (such as IJulia). As usual, be sure to import Base.show in order to add new methods to the built-in Julia function show.\n\nTechnically, the MIME\"mime\" macro defines a singleton type for the given mime string, which allows us to exploit Julia's dispatch mechanisms in determining how to display objects of any given type.\n\nThe default MIME type is MIME\"text/plain\". There is a fallback definition for text/plain output that calls show with 2 arguments, so it is not always necessary to add a method for that case. If a type benefits from custom human-readable output though, show(::IO, ::MIME\"text/plain\", ::T) should be defined. For example, the Day type uses 1 day as the output for the text/plain MIME type, and Day(1) as the output of 2-argument show.\n\nExamples\n\njulia> struct Day\n           n::Int\n       end\n\njulia> Base.show(io::IO, ::MIME\"text/plain\", d::Day) = print(io, d.n, \" day\")\n\njulia> Day(1)\n1 day\n\nContainer types generally implement 3-argument show by calling show(io, MIME\"text/plain\"(), x) for elements x, with :compact => true set in an IOContext passed as the first argument.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.showable","page":"I/O and Network","title":"Base.Multimedia.showable","category":"function","text":"showable(mime, x)\n\nReturn a boolean value indicating whether or not the object x can be written as the given mime type.\n\n(By default, this is determined automatically by the existence of the corresponding show method for typeof(x).  Some types provide custom showable methods; for example, if the available MIME formats depend on the value of x.)\n\nExamples\n\njulia> showable(MIME(\"text/plain\"), rand(5))\ntrue\n\njulia> showable(\"image/png\", rand(5))\nfalse\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.repr-Tuple{MIME, Any}","page":"I/O and Network","title":"Base.repr","category":"method","text":"repr(mime, x; context=nothing)\n\nReturn an AbstractString or Vector{UInt8} containing the representation of x in the requested mime type, as written by show(io, mime, x) (throwing a MethodError if no appropriate show is available). An AbstractString is returned for MIME types with textual representations (such as \"text/html\" or \"application/postscript\"), whereas binary data is returned as Vector{UInt8}. (The function istextmime(mime) returns whether or not Julia treats a given mime type as text.)\n\nThe optional keyword argument context can be set to :key=>value pair or an IO or IOContext object whose attributes are used for the I/O stream passed to show.\n\nAs a special case, if x is an AbstractString (for textual MIME types) or a Vector{UInt8} (for binary MIME types), the repr function assumes that x is already in the requested mime format and simply returns x. This special case does not apply to the \"text/plain\" MIME type. This is useful so that raw data can be passed to display(m::MIME, x).\n\nIn particular, repr(\"text/plain\", x) is typically a \"pretty-printed\" version of x designed for human consumption.  See also repr(x) to instead return a string corresponding to show(x) that may be closer to how the value of x would be entered in Julia.\n\nExamples\n\njulia> A = [1 2; 3 4];\n\njulia> repr(\"text/plain\", A)\n\"2×2 Matrix{Int64}:\\n 1  2\\n 3  4\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.MIME","page":"I/O and Network","title":"Base.Multimedia.MIME","category":"type","text":"MIME\n\nA type representing a standard internet data format. \"MIME\" stands for \"Multipurpose Internet Mail Extensions\", since the standard was originally used to describe multimedia attachments to email messages.\n\nA MIME object can be passed as the second argument to show to request output in that format.\n\nExamples\n\njulia> show(stdout, MIME(\"text/plain\"), \"hi\")\n\"hi\"\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.@MIME_str","page":"I/O and Network","title":"Base.Multimedia.@MIME_str","category":"macro","text":"@MIME_str\n\nA convenience macro for writing MIME types, typically used when adding methods to show. For example the syntax show(io::IO, ::MIME\"text/html\", x::MyType) = ... could be used to define how to write an HTML representation of MyType.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.pushdisplay","page":"I/O and Network","title":"Base.Multimedia.pushdisplay","category":"function","text":"pushdisplay(d::AbstractDisplay)\n\nPushes a new display d on top of the global display-backend stack. Calling display(x) or display(mime, x) will display x on the topmost compatible backend in the stack (i.e., the topmost backend that does not throw a MethodError).\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.popdisplay","page":"I/O and Network","title":"Base.Multimedia.popdisplay","category":"function","text":"popdisplay()\npopdisplay(d::AbstractDisplay)\n\nPop the topmost backend off of the display-backend stack, or the topmost copy of d in the second variant.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.TextDisplay","page":"I/O and Network","title":"Base.Multimedia.TextDisplay","category":"type","text":"TextDisplay(io::IO)\n\nReturn a TextDisplay <: AbstractDisplay, which displays any object as the text/plain MIME type (by default), writing the text representation to the given I/O stream. (This is how objects are printed in the Julia REPL.)\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.Multimedia.istextmime","page":"I/O and Network","title":"Base.Multimedia.istextmime","category":"function","text":"istextmime(m::MIME)\n\nDetermine whether a MIME type is text data. MIME types are assumed to be binary data except for a set of types known to be text data (possibly Unicode).\n\nExamples\n\njulia> istextmime(MIME(\"text/plain\"))\ntrue\n\njulia> istextmime(MIME(\"image/png\"))\nfalse\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.bytesavailable","page":"I/O and Network","title":"Base.bytesavailable","category":"function","text":"bytesavailable(io)\n\nReturn the number of bytes available for reading before a read from this stream or buffer will block.\n\nExamples\n\njulia> io = IOBuffer(\"JuliaLang is a GitHub organization\");\n\njulia> bytesavailable(io)\n34\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.ntoh","page":"I/O and Network","title":"Base.ntoh","category":"function","text":"ntoh(x)\n\nConvert the endianness of a value from Network byte order (big-endian) to that used by the Host.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.hton","page":"I/O and Network","title":"Base.hton","category":"function","text":"hton(x)\n\nConvert the endianness of a value from that used by the Host to Network byte order (big-endian).\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.ltoh","page":"I/O and Network","title":"Base.ltoh","category":"function","text":"ltoh(x)\n\nConvert the endianness of a value from Little-endian to that used by the Host.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.htol","page":"I/O and Network","title":"Base.htol","category":"function","text":"htol(x)\n\nConvert the endianness of a value from that used by the Host to Little-endian.\n\n\n\n\n\n"},{"location":"base/io-network.html#Base.ENDIAN_BOM","page":"I/O and Network","title":"Base.ENDIAN_BOM","category":"constant","text":"ENDIAN_BOM\n\nThe 32-bit byte-order-mark indicates the native byte order of the host machine. Little-endian machines will contain the value 0x04030201. Big-endian machines will contain the value 0x01020304.\n\n\n\n\n\n"},{"location":"devdocs/eval.html#Eval-of-Julia-code","page":"Eval of Julia code","title":"Eval of Julia code","category":"section","text":"One of the hardest parts about learning how the Julia Language runs code is learning how all of the pieces work together to execute a block of code.\n\nEach chunk of code typically makes a trip through many steps with potentially unfamiliar names, such as (in no particular order): flisp, AST, C++, LLVM, eval, typeinf, macroexpand, sysimg (or system image), bootstrapping, compile, parse, execute, JIT, interpret, box, unbox, intrinsic function, and primitive function, before turning into the desired result (hopefully).\n\nsidebar: Definitions\nREPL\nREPL stands for Read-Eval-Print Loop. It's just what we call the command line environment for short.\nAST\nAbstract Syntax Tree. The AST is the digital representation of the code structure. In this form the code has been tokenized for meaning so that it is more suitable for manipulation and execution.\n\n(Image: Diagram of the compiler flow)"},{"location":"devdocs/eval.html#Julia-Execution","page":"Eval of Julia code","title":"Julia Execution","category":"section","text":"The 10,000 foot view of the whole process is as follows:\n\nThe user starts julia.\nThe C function main() from cli/loader_exe.c gets called. This function processes the command line arguments, filling in the jl_options struct and setting the variable ARGS. It then initializes Julia (by calling julia_init in init.c, which may load a previously compiled sysimg). Finally, it passes off control to Julia by calling Base._start().\nWhen _start() takes over control, the subsequent sequence of commands depends on the command line arguments given. For example, if a filename was supplied, it will proceed to execute that file. Otherwise, it will start an interactive REPL.\nSkipping the details about how the REPL interacts with the user, let's just say the program ends up with a block of code that it wants to run.\nIf the block of code to run is in a file, jl_load(char *filename) gets invoked to load the file and parse it. Each fragment of code is then passed to eval to execute.\nEach fragment of code (or AST), is handed off to eval() to turn into results.\neval() takes each code fragment and tries to run it in jl_toplevel_eval_flex().\njl_toplevel_eval_flex() decides whether the code is a \"toplevel\" action (such as using or module), which would be invalid inside a function. If so, it passes off the code to the toplevel interpreter.\njl_toplevel_eval_flex() then expands the code to eliminate any macros and to \"lower\" the AST to make it simpler to execute.\njl_toplevel_eval_flex() then uses some simple heuristics to decide whether to JIT compile the  AST or to interpret it directly.\nThe bulk of the work to interpret code is handled by eval in interpreter.c.\nIf instead, the code is compiled, the bulk of the work is handled by codegen.cpp. Whenever a  Julia function is called for the first time with a given set of argument types, type inference  will be run on that function. This information is used by the codegen step to generate  faster code.\nEventually, the user quits the REPL, or the end of the program is reached, and the _start()  method returns.\nJust before exiting, main() calls jl_atexit_hook(exit_code).  This calls Base._atexit() (which calls any functions registered to atexit() inside  Julia). Then it calls jl_gc_run_all_finalizers().  Finally, it gracefully cleans up all libuv handles and waits for them to flush and close."},{"location":"devdocs/eval.html#dev-parsing","page":"Eval of Julia code","title":"Parsing","category":"section","text":"By default, Julia uses JuliaSyntax.jl to produce the AST. Historically, it used a small lisp program written in femtolisp, the source-code for which is distributed inside Julia in src/flisp. If the JULIA_USE_FLISP_PARSER environment variable is set to 1, the old parser will be used instead."},{"location":"devdocs/eval.html#dev-macro-expansion","page":"Eval of Julia code","title":"Macro Expansion","category":"section","text":"When eval() encounters a macro, it expands that AST node before attempting to evaluate the expression. Macro expansion involves a handoff from eval() (in Julia), to the parser function jl_macroexpand() (written in flisp) to the Julia macro itself (written in - what else - Julia) via fl_invoke_julia_macro(), and back.\n\nTypically, macro expansion is invoked as a first step during a call to Meta.lower()/Core._lower(), although it can also be invoked directly by a call to macroexpand()/jl_macroexpand()."},{"location":"devdocs/eval.html#dev-type-inference","page":"Eval of Julia code","title":"Type Inference","category":"section","text":"Type inference is implemented in Julia by typeinf() in compiler/typeinfer.jl. Type inference is the process of examining a Julia function and determining bounds for the types of each of its variables, as well as bounds on the type of the return value from the function. This enables many future optimizations, such as unboxing of known immutable values, and compile-time hoisting of various run-time operations such as computing field offsets and function pointers. Type inference may also include other steps such as constant propagation and inlining.\n\nsidebar: More Definitions\nJIT\nJust-In-Time Compilation The process of generating native-machine code into memory right when it is needed.\nLLVM\nLow-Level Virtual Machine (a compiler) The Julia JIT compiler is a program/library called libLLVM. Codegen in Julia refers both to the process of taking a Julia AST and turning it into LLVM instructions, and the process of LLVM optimizing that and turning it into native assembly instructions.\nC++\nThe programming language that LLVM is implemented in, which means that codegen is also implemented in this language. The rest of Julia's library is implemented in C, in part because its smaller feature set makes it more usable as a cross-language interface layer.\nbox\nThis term is used to describe the process of taking a value and allocating a wrapper around the data that is tracked by the garbage collector (gc) and is tagged with the object's type.\nunbox\nThe reverse of boxing a value. This operation enables more efficient manipulation of data when the type of that data is fully known at compile-time (through type inference).\ngeneric function\nA Julia function composed of multiple \"methods\" that are selected for dynamic dispatch based on the argument type-signature\nanonymous function or \"method\"\nA Julia function without a name and without type-dispatch capabilities\nprimitive function\nA function implemented in C but exposed in Julia as a named function \"method\" (albeit without generic function dispatch capabilities, similar to a anonymous function)\nintrinsic function\nA low-level operation exposed as a function in Julia. These pseudo-functions implement operations on raw bits such as add and sign extend that cannot be expressed directly in any other way. Since they operate on bits directly, they must be compiled into a function and surrounded by a call to Core.Intrinsics.box(T, ...) to reassign type information to the value."},{"location":"devdocs/eval.html#dev-codegen","page":"Eval of Julia code","title":"JIT Code Generation","category":"section","text":"Codegen is the process of turning a Julia AST into native machine code.\n\nThe JIT environment is initialized by an early call to jl_init_codegen in codegen.cpp.\n\nOn demand, a Julia method is converted into a native function by the function emit_function(jl_method_instance_t*). (note, when using the MCJIT (in LLVM v3.4+), each function must be JIT into a new module.) This function recursively calls emit_expr() until the entire function has been emitted.\n\nMuch of the remaining bulk of this file is devoted to various manual optimizations of specific code patterns. For example, emit_known_call() knows how to inline many of the primitive functions (defined in builtins.c) for various combinations of argument types.\n\nOther parts of codegen are handled by various helper files:\n\ndebuginfo.cpp\nHandles backtraces for JIT functions\nccall.cpp\nHandles the ccall and llvmcall FFI, along with various abi_*.cpp files\nintrinsics.cpp\nHandles the emission of various low-level intrinsic functions\n\nsidebar: Bootstrapping\nThe process of creating a new system image is called \"bootstrapping\".The etymology of this word comes from the phrase \"pulling oneself up by the bootstraps\", and refers to the idea of starting from a very limited set of available functions and definitions and ending with the creation of a full-featured environment."},{"location":"devdocs/eval.html#dev-sysimg","page":"Eval of Julia code","title":"System Image","category":"section","text":"The system image is a precompiled archive of a set of Julia files. The sys.ji file distributed with Julia is one such system image, generated by executing the file sysimg.jl, and serializing the resulting environment (including Types, Functions, Modules, and all other defined values) into a file. Therefore, it contains a frozen version of the Main, Core, and Base modules (and whatever else was in the environment at the end of bootstrapping). This serializer/deserializer is implemented by jl_save_system_image/jl_restore_system_image in staticdata.c.\n\nIf there is no sysimg file (jl_options.image_file == NULL), this also implies that --build was given on the command line, so the final result should be a new sysimg file. During Julia initialization, minimal Core and Main modules are created. Then a file named boot.jl is evaluated from the current directory. Julia then evaluates any file given as a command line argument until it reaches the end. Finally, it saves the resulting environment to a \"sysimg\" file for use as a starting point for a future Julia run."},{"location":"devdocs/functions.html#Julia-Functions","page":"Julia Functions","title":"Julia Functions","category":"section","text":"This document will explain how functions, method definitions, and method tables work."},{"location":"devdocs/functions.html#Method-Tables","page":"Julia Functions","title":"Method Tables","category":"section","text":"Every function in Julia is a generic function. A generic function is conceptually a single function, but consists of many definitions, or methods. The methods of a generic function are stored in a method table. There is one global method table (type MethodTable) named Core.methodtable. Any default operation on methods (such as calls) uses that table."},{"location":"devdocs/functions.html#Function-calls","page":"Julia Functions","title":"Function calls","category":"section","text":"Given the call f(x, y), the following steps are performed: First, a tuple type is formed, Tuple{typeof(f), typeof(x), typeof(y)}. Note that the type of the function itself is the first element. This is because the function itself participates symmetrically in method lookup with the other arguments. This tuple type is looked up in the global method table. However, the system can then cache the results, so these steps can be skipped later for similar lookups.\n\nThis dispatch process is performed by jl_apply_generic, which takes two arguments: a pointer to an array of the values f, x, and y, and the number of values (in this case 3).\n\nThroughout the system, there are two kinds of APIs that handle functions and argument lists: those that accept the function and arguments separately, and those that accept a single argument structure. In the first kind of API, the \"arguments\" part does not contain information about the function, since that is passed separately. In the second kind of API, the function is the first element of the argument structure.\n\nFor example, the following function for performing a call accepts just an args pointer, so the first element of the args array will be the function to call:\n\njl_value_t *jl_apply(jl_value_t **args, uint32_t nargs)\n\nThis entry point for the same functionality accepts the function separately, so the args array does not contain the function:\n\njl_value_t *jl_call(jl_value_t *f, jl_value_t **args, int32_t nargs);"},{"location":"devdocs/functions.html#Adding-methods","page":"Julia Functions","title":"Adding methods","category":"section","text":"Given the above dispatch process, conceptually all that is needed to add a new method is (1) a tuple type, and (2) code for the body of the method. jl_method_def implements this operation."},{"location":"devdocs/functions.html#Creating-generic-functions","page":"Julia Functions","title":"Creating generic functions","category":"section","text":"Since every object is callable, nothing special is needed to create a generic function. Therefore jl_new_generic_function simply creates a new singleton (0 size) subtype of Function and returns its instance. A function can have a mnemonic \"display name\" which is used in debug info and when printing objects. For example the name of Base.sin is sin. By convention, the name of the created type is the same as the function name, with a # prepended. So typeof(sin) is Base.#sin."},{"location":"devdocs/functions.html#Closures","page":"Julia Functions","title":"Closures","category":"section","text":"A closure is simply a callable object with field names corresponding to captured variables. For example, the following code:\n\nfunction adder(x)\n    return y->x+y\nend\n\nis lowered to (roughly):\n\nstruct ##1{T}\n    x::T\nend\n\n(_::##1)(y) = _.x + y\n\nfunction adder(x)\n    return ##1(x)\nend"},{"location":"devdocs/functions.html#Constructors","page":"Julia Functions","title":"Constructors","category":"section","text":"A constructor call is just a call to a type, to a method defined on Type{T}."},{"location":"devdocs/functions.html#Builtins","page":"Julia Functions","title":"Builtins","category":"section","text":"The \"builtin\" functions, defined in the Core module, are:\n\nfunction lines(words)\n    io = IOBuffer()\n    n = 0\n    for w in words\n        if n+length(w) > 80\n            print(io, '\\n', w)\n            n = length(w)\n        elseif n == 0\n            print(io, w);\n            n += length(w)\n        else\n            print(io, ' ', w);\n            n += length(w)+1\n        end\n    end\n    takestring!(io)\nend\nimport Markdown\n[string(n) for n in names(Core;all=true)\n    if getfield(Core,n) isa Core.Builtin && nameof(getfield(Core,n)) === n] |>\n    lines |>\n    s ->  \"```\\n$s\\n```\" |>\n    Markdown.parse\n\nThese are mostly singleton objects all of whose types are subtypes of Builtin, which is a subtype of Function. Their purpose is to expose entry points in the run time that use the \"jlcall\" calling convention:\n\njl_value_t *(jl_value_t*, jl_value_t**, uint32_t)"},{"location":"devdocs/functions.html#Keyword-arguments","page":"Julia Functions","title":"Keyword arguments","category":"section","text":"Keyword arguments work by adding methods to the kwcall function. This function is usually the \"keyword argument sorter\" or \"keyword sorter\", which then calls the inner body of the function (defined anonymously). Every definition in the kwsorter function has the same arguments as some definition in the normal method table, except with a single NamedTuple argument prepended, which gives the names and values of passed keyword arguments. The kwsorter's job is to move keyword arguments into their canonical positions based on name, plus evaluate and substitute any needed default value expressions. The result is a normal positional argument list, which is then passed to yet another compiler-generated function.\n\nThe easiest way to understand the process is to look at how a keyword argument method definition is lowered. The code:\n\nfunction circle(center, radius; color = black, fill::Bool = true, options...)\n    # draw\nend\n\nactually produces three method definitions. The first is a function that accepts all arguments (including keyword arguments) as positional arguments, and includes the code for the method body. It has an auto-generated name:\n\nfunction #circle#1(color, fill::Bool, options, circle, center, radius)\n    # draw\nend\n\nThe second method is an ordinary definition for the original circle function, which handles the case where no keyword arguments are passed:\n\nfunction circle(center, radius)\n    #circle#1(black, true, pairs(NamedTuple()), circle, center, radius)\nend\n\nThis simply dispatches to the first method, passing along default values. pairs is applied to the named tuple of rest arguments to provide key-value pair iteration. Note that if the method doesn't accept rest keyword arguments then this argument is absent.\n\nFinally there is the kwsorter definition:\n\nfunction (::Core.kwcall)(kws, circle, center, radius)\n    if haskey(kws, :color)\n        color = kws.color\n    else\n        color = black\n    end\n    # etc.\n\n    # put remaining kwargs in `options`\n    options = structdiff(kws, NamedTuple{(:color, :fill)})\n\n    # if the method doesn't accept rest keywords, throw an error\n    # unless `options` is empty\n\n    #circle#1(color, fill, pairs(options), circle, center, radius)\nend"},{"location":"devdocs/functions.html#compiler-efficiency-issues","page":"Julia Functions","title":"Compiler efficiency issues","category":"section","text":"Generating a new type for every function has potentially serious consequences for compiler resource use when combined with Julia's \"specialize on all arguments by default\" design. Indeed, the initial implementation of this design suffered from much longer build and test times, higher memory use, and a system image nearly 2x larger than the baseline. In a naive implementation, the problem is bad enough to make the system nearly unusable. Several significant optimizations were needed to make the design practical.\n\nThe first issue is excessive specialization of functions for different values of function-valued arguments. Many functions simply \"pass through\" an argument to somewhere else, e.g. to another function or to a storage location. Such functions do not need to be specialized for every closure that might be passed in. Fortunately this case is easy to distinguish by simply considering whether a function calls one of its arguments (i.e. the argument appears in \"head position\" somewhere). Performance-critical higher-order functions like map certainly call their argument function and so will still be specialized as expected. This optimization is implemented by recording which arguments are called during the analyze-variables pass in the front end. When cache_method sees an argument in the Function type hierarchy passed to a slot declared as Any or Function, it behaves as if the @nospecialize annotation were applied. This heuristic seems to be extremely effective in practice.\n\nThe next issue concerns the structure of method tables. Empirical studies show that the vast majority of dynamically-dispatched calls involve one or two arguments. In turn, many of these cases can be resolved by considering only the first argument. (Aside: proponents of single dispatch would not be surprised by this at all. However, this argument means \"multiple dispatch is easy to optimize in practice\", and that we should therefore use it, not \"we should use single dispatch\"!). So the method table and cache splits up on the structure based on a left-to-right decision tree so allow efficient nearest-neighbor searches.\n\nThe front end generates type declarations for all closures. Initially, this was implemented by generating normal type declarations. However, this produced an extremely large number of constructors, all of which were trivial (simply passing all arguments through to new). Since methods are partially ordered, inserting all of these methods is O(n²), plus there are just too many of them to keep around. This was optimized by generating struct_type expressions directly (bypassing default constructor generation), and using new directly to create closure instances. Not the prettiest thing ever, but you do what you gotta do.\n\nThe next problem was the @test macro, which generated a 0-argument closure for each test case. This is not really necessary, since each test case is simply run once in place. Therefore, @test was modified to expand to a try-catch block that records the test result (true, false, or exception raised) and calls the test suite handler on it."},{"location":"stdlib/Test.html#Unit-Testing","page":"Unit Testing","title":"Unit Testing","category":"section","text":""},{"location":"stdlib/Test.html#Testing-Base-Julia","page":"Unit Testing","title":"Testing Base Julia","category":"section","text":"Julia is under rapid development and has an extensive test suite to verify functionality across multiple platforms. If you build Julia from source, you can run this test suite with make test. In a binary install, you can run the test suite using Base.runtests()."},{"location":"stdlib/Test.html#Basic-Unit-Tests","page":"Unit Testing","title":"Basic Unit Tests","category":"section","text":"The Test module provides simple unit testing functionality. Unit testing is a way to see if your code is correct by checking that the results are what you expect. It can be helpful to ensure your code still works after you make changes, and can be used when developing as a way of specifying the behaviors your code should have when complete. You may also want to look at the documentation for adding tests to your Julia Package.\n\nSimple unit testing can be performed with the @test and @test_throws macros:\n\nFor example, suppose we want to check our new function foo(x) works as expected:\n\njulia> using Test\n\njulia> foo(x) = length(x)^2\nfoo (generic function with 1 method)\n\nIf the condition is true, a Pass is returned:\n\njulia> @test foo(\"bar\") == 9\nTest Passed\n\njulia> @test foo(\"fizz\") >= 10\nTest Passed\n\nIf the condition is false, then a Fail is returned and an exception is thrown:\n\njulia> @test foo(\"f\") == 20\nTest Failed at none:1\n  Expression: foo(\"f\") == 20\n   Evaluated: 1 == 20\nERROR: There was an error during testing\n\nIf the condition could not be evaluated because an exception was thrown, which occurs in this case because length is not defined for symbols, an Error object is returned and an exception is thrown:\n\njulia> @test foo(:cat) == 1\nError During Test\n  Test threw an exception of type MethodError\n  Expression: foo(:cat) == 1\n  MethodError: no method matching length(::Symbol)\n  The function `length` exists, but no method is defined for this combination of argument types.\n\n  Closest candidates are:\n    length(::SimpleVector) at essentials.jl:256\n    length(::Base.MethodList) at reflection.jl:521\n    length(::MethodTable) at reflection.jl:597\n    ...\n  Stacktrace:\n  [...]\nERROR: There was an error during testing\n\nIf we expect that evaluating an expression should throw an exception, then we can use @test_throws to check that this occurs:\n\njulia> @test_throws MethodError foo(:cat)\nTest Passed\n      Thrown: MethodError"},{"location":"stdlib/Test.html#Working-with-Test-Sets","page":"Unit Testing","title":"Working with Test Sets","category":"section","text":"Typically a large number of tests are used to make sure functions work correctly over a range of inputs. In the event a test fails, the default behavior is to throw an exception immediately. However, it is normally preferable to run the rest of the tests first to get a better picture of how many errors there are in the code being tested.\n\nnote: Note\nThe @testset will create a local scope of its own when running the tests in it.\n\nThe @testset macro can be used to group tests into sets. All the tests in a test set will be run, and at the end of the test set a summary will be printed. If any of the tests failed, or could not be evaluated due to an error, the test set will then throw a TestSetException.\n\nWe can put our tests for the foo(x) function in a test set:\n\njulia> @testset \"Foo Tests\" begin\n           @test foo(\"a\")   == 1\n           @test foo(\"ab\")  == 4\n           @test foo(\"abc\") == 9\n       end;\nTest Summary: | Pass  Total  Time\nFoo Tests     |    3      3  0.0s\n\nTest sets can also be nested:\n\njulia> @testset \"Foo Tests\" begin\n           @testset \"Animals\" begin\n               @test foo(\"cat\") == 9\n               @test foo(\"dog\") == foo(\"cat\")\n           end\n           @testset \"Arrays $i\" for i in 1:3\n               @test foo(zeros(i)) == i^2\n               @test foo(fill(1.0, i)) == i^2\n           end\n       end;\nTest Summary: | Pass  Total  Time\nFoo Tests     |    8      8  0.0s\n\nAs well as call functions:\n\njulia> f(x) = @test isone(x)\nf (generic function with 1 method)\n\njulia> @testset f(1);\nTest Summary: | Pass  Total  Time\nf             |    1      1  0.0s\n\nThis can be used to allow for factorization of test sets, making it easier to run individual test sets by running the associated functions instead. Note that in the case of functions, the test set will be given the name of the called function. In the event that a nested test set has no failures, as happened here, it will be hidden in the summary, unless the verbose=true option is passed:\n\njulia> @testset verbose = true \"Foo Tests\" begin\n           @testset \"Animals\" begin\n               @test foo(\"cat\") == 9\n               @test foo(\"dog\") == foo(\"cat\")\n           end\n           @testset \"Arrays $i\" for i in 1:3\n               @test foo(zeros(i)) == i^2\n               @test foo(fill(1.0, i)) == i^2\n           end\n       end;\nTest Summary: | Pass  Total  Time\nFoo Tests     |    8      8  0.0s\n  Animals     |    2      2  0.0s\n  Arrays 1    |    2      2  0.0s\n  Arrays 2    |    2      2  0.0s\n  Arrays 3    |    2      2  0.0s"},{"location":"stdlib/Test.html#Environment-Variable-Support","page":"Unit Testing","title":"Environment Variable Support","category":"section","text":"The Test module supports the JULIA_TEST_VERBOSE environment variable for controlling verbose behavior globally:\n\nWhen JULIA_TEST_VERBOSE=true, testsets will automatically use verbose=true by default, and additionally print \"Starting testset\" and \"Finished testset\" messages with timing information as testsets are entered and exited.\nWhen JULIA_TEST_VERBOSE=false or unset, testsets use verbose=false by default and no entry/exit messages are printed.\n\nThis environment variable provides a convenient way to enable comprehensive verbose output for debugging test suites without modifying the test code itself.\n\n$ JULIA_TEST_VERBOSE=true julia -e '\nusing Test\n@testset \"Example\" begin\n    @test 1 + 1 == 2\n    @testset \"Nested\" begin\n        @test 2 * 2 == 4\n    end\nend'\n\nStarting testset: Example\n  Starting testset: Nested\n  Finished testset: Nested (0.0s)\nFinished testset: Example (0.0s)\nTest Summary: | Pass  Total  Time\nExample       |    2      2  0.0s\n  Nested      |    1      1  0.0s\n\nIf we do have a test failure, only the details for the failed test sets will be shown:\n\njulia> @testset \"Foo Tests\" begin\n           @testset \"Animals\" begin\n               @testset \"Felines\" begin\n                   @test foo(\"cat\") == 9\n               end\n               @testset \"Canines\" begin\n                   @test foo(\"dog\") == 9\n               end\n           end\n           @testset \"Arrays\" begin\n               @test foo(zeros(2)) == 4\n               @test foo(fill(1.0, 4)) == 15\n           end\n       end\n\nArrays: Test Failed\n  Expression: foo(fill(1.0, 4)) == 15\n   Evaluated: 16 == 15\n[...]\nTest Summary: | Pass  Fail  Total  Time\nFoo Tests     |    3     1      4  0.0s\n  Animals     |    2            2  0.0s\n  Arrays      |    1     1      2  0.0s\nERROR: Some tests did not pass: 3 passed, 1 failed, 0 errored, 0 broken."},{"location":"stdlib/Test.html#Testing-Log-Statements","page":"Unit Testing","title":"Testing Log Statements","category":"section","text":"One can use the @test_logs macro to test log statements, or use a TestLogger."},{"location":"stdlib/Test.html#Other-Test-Macros","page":"Unit Testing","title":"Other Test Macros","category":"section","text":"As calculations on floating-point values can be imprecise, you can perform approximate equality checks using either @test a ≈ b (where ≈, typed via tab completion of \\approx, is the isapprox function) or use isapprox directly.\n\njulia> @test 1 ≈ 0.999999999\nTest Passed\n\njulia> @test 1 ≈ 0.999999\nTest Failed at none:1\n  Expression: 1 ≈ 0.999999\nERROR: There was an error during testing\n\nYou can specify relative and absolute tolerances by setting the rtol and atol keyword arguments of isapprox, respectively, after the ≈ comparison:\n\njulia> @test 1 ≈ 0.999999  rtol=1e-5\nTest Passed\n\nNote that this is not a specific feature of the ≈ but rather a general feature of the @test macro: @test a <op> b key=val is transformed by the macro into @test op(a, b, key=val). It is, however, particularly useful for ≈ tests."},{"location":"stdlib/Test.html#Broken-Tests","page":"Unit Testing","title":"Broken Tests","category":"section","text":"If a test fails consistently it can be changed to use the @test_broken macro. This will denote the test as Broken if the test continues to fail and alerts the user via an Error if the test succeeds.\n\n@test_skip is also available to skip a test without evaluation, but counting the skipped test in the test set reporting. The test will not run but gives a Broken Result."},{"location":"stdlib/Test.html#Test-result-types","page":"Unit Testing","title":"Test result types","category":"section","text":""},{"location":"stdlib/Test.html#Creating-Custom-AbstractTestSet-Types","page":"Unit Testing","title":"Creating Custom AbstractTestSet Types","category":"section","text":"Packages can create their own AbstractTestSet subtypes by implementing the record and finish methods. The subtype should have a one-argument constructor taking a description string, with any options passed in as keyword arguments.\n\nTest takes responsibility for maintaining a stack of nested testsets as they are executed, but any result accumulation is the responsibility of the AbstractTestSet subtype. You can access this stack with the get_testset and get_testset_depth methods. Note that these functions are not exported.\n\nTest also makes sure that nested @testset invocations use the same AbstractTestSet subtype as their parent unless it is set explicitly. It does not propagate any properties of the testset. Option inheritance behavior can be implemented by packages using the stack infrastructure that Test provides.\n\nDefining a basic AbstractTestSet subtype might look like:\n\nimport Test: Test, record, finish\nusing Test: AbstractTestSet, Result, Pass, Fail, Error\nusing Test: get_testset_depth, get_testset\nstruct CustomTestSet <: Test.AbstractTestSet\n    description::AbstractString\n    foo::Int\n    results::Vector\n    # constructor takes a description string and options keyword arguments\n    CustomTestSet(desc; foo=1) = new(desc, foo, [])\nend\n\nrecord(ts::CustomTestSet, child::AbstractTestSet) = push!(ts.results, child)\nrecord(ts::CustomTestSet, res::Result) = push!(ts.results, res)\nfunction finish(ts::CustomTestSet)\n    # just record if we're not the top-level parent\n    if get_testset_depth() > 0\n        record(get_testset(), ts)\n        return ts\n    end\n\n    # so the results are printed if we are at the top level\n    Test.print_test_results(ts)\n    return ts\nend\n\nAnd using that testset looks like:\n\n@testset CustomTestSet foo=4 \"custom testset inner 2\" begin\n    # this testset should inherit the type, but not the argument.\n    @testset \"custom testset inner\" begin\n        @test true\n    end\nend\n\nIn order to use a custom testset and have the recorded results printed as part of any outer default testset, also define Test.get_test_counts. This might look like so:\n\nusing Test: AbstractTestSet, Pass, Fail, Error, Broken, get_test_counts, TestCounts, format_duration\n\nfunction Test.get_test_counts(ts::CustomTestSet)\n    passes, fails, errors, broken = 0, 0, 0, 0\n    # cumulative results\n    c_passes, c_fails, c_errors, c_broken = 0, 0, 0, 0\n\n    for t in ts.results\n        # count up results\n        isa(t, Pass)   && (passes += 1)\n        isa(t, Fail)   && (fails  += 1)\n        isa(t, Error)  && (errors += 1)\n        isa(t, Broken) && (broken += 1)\n        # handle children\n        if isa(t, AbstractTestSet)\n            tc = get_test_counts(t)::TestCounts\n            c_passes += tc.passes + tc.cumulative_passes\n            c_fails  += tc.fails + tc.cumulative_fails\n            c_errors += tc.errors + tc.cumulative_errors\n            c_broken += tc.broken + tc.cumulative_broken\n        end\n    end\n    # get a duration, if we have one\n    duration = format_duration(ts)\n    return TestCounts(true, passes, fails, errors, broken, c_passes, c_fails, c_errors, c_broken, duration)\nend"},{"location":"stdlib/Test.html#Test-utilities","page":"Unit Testing","title":"Test utilities","category":"section","text":""},{"location":"stdlib/Test.html#Workflow-for-Testing-Packages","page":"Unit Testing","title":"Workflow for Testing Packages","category":"section","text":"Using the tools available to us in the previous sections, here is a potential workflow of creating a package and adding tests to it."},{"location":"stdlib/Test.html#Generating-an-Example-Package","page":"Unit Testing","title":"Generating an Example Package","category":"section","text":"For this workflow, we will create a package called Example:\n\npkg> generate Example\nshell> cd Example\nshell> mkdir test\npkg> activate ."},{"location":"stdlib/Test.html#Creating-Sample-Functions","page":"Unit Testing","title":"Creating Sample Functions","category":"section","text":"The number one requirement for testing a package is to have functionality to test. For that, we will add some simple functions to Example that we can test. Add the following to src/Example.jl:\n\nmodule Example\n\nexport greet, simple_add, type_multiply\n\nfunction greet()\n    \"Hello world!\"\nend\n\nfunction simple_add(a, b)\n    a + b\nend\n\nfunction type_multiply(a::Float64, b::Float64)\n    a * b\nend\n\nend"},{"location":"stdlib/Test.html#Creating-a-Test-Environment","page":"Unit Testing","title":"Creating a Test Environment","category":"section","text":"From within the root of the Example package, navigate to the test directory, activate a new environment there, and add the Test package to the environment:\n\nshell> cd test\npkg> activate .\n(test) pkg> add Test"},{"location":"stdlib/Test.html#Testing-Our-Package","page":"Unit Testing","title":"Testing Our Package","category":"section","text":"Now, we are ready to add tests to Example. It is standard practice to create a file within the test directory called runtests.jl which contains the test sets we want to run. Go ahead and create that file within the test directory and add the following code to it:\n\nusing Example\nusing Test\n\n@testset \"Example tests\" begin\n\n    @testset \"Math tests\" begin\n        include(\"math_tests.jl\")\n    end\n\n    @testset \"Greeting tests\" begin\n        include(\"greeting_tests.jl\")\n    end\nend\n\nWe will need to create those two included files, math_tests.jl and greeting_tests.jl, and add some tests to them.\n\nNote: Notice how we did not have to specify add Example into the test environment's Project.toml. This is a benefit of Julia's testing system that you could read about more here."},{"location":"stdlib/Test.html#Writing-Tests-for-math_tests.jl","page":"Unit Testing","title":"Writing Tests for math_tests.jl","category":"section","text":"Using our knowledge of Test.jl, here are some example tests we could add to math_tests.jl:\n\n@testset \"Testset 1\" begin\n    @test 2 == simple_add(1, 1)\n    @test 3.5 == simple_add(1, 2.5)\n    @test_throws MethodError simple_add(1, \"A\")\n    @test_throws MethodError simple_add(1, 2, 3)\nend\n\n@testset \"Testset 2\" begin\n    @test 1.0 == type_multiply(1.0, 1.0)\n    @test isa(type_multiply(2.0, 2.0), Float64)\n    @test_throws MethodError type_multiply(1, 2.5)\nend"},{"location":"stdlib/Test.html#Writing-Tests-for-greeting_tests.jl","page":"Unit Testing","title":"Writing Tests for greeting_tests.jl","category":"section","text":"Using our knowledge of Test.jl, here are some example tests we could add to greeting_tests.jl:\n\n@testset \"Testset 3\" begin\n    @test \"Hello world!\" == greet()\n    @test_throws MethodError greet(\"Antonia\")\nend"},{"location":"stdlib/Test.html#Testing-Our-Package-2","page":"Unit Testing","title":"Testing Our Package","category":"section","text":"Now that we have added our tests and our runtests.jl script in test, we can test our Example package by going back to the root of the Example package environment and reactivating the Example environment:\n\nshell> cd ..\npkg> activate .\n\nFrom there, we can finally run our test suite as follows:\n\n(Example) pkg> test\n     Testing Example\n      Status `/tmp/jl_Yngpvy/Project.toml`\n  [fa318bd2] Example v0.1.0 `/home/src/Projects/tmp/errata/Example`\n  [8dfed614] Test `@stdlib/Test`\n      Status `/tmp/jl_Yngpvy/Manifest.toml`\n  [fa318bd2] Example v0.1.0 `/home/src/Projects/tmp/errata/Example`\n  [2a0f44e3] Base64 `@stdlib/Base64`\n  [b77e0a4c] InteractiveUtils `@stdlib/InteractiveUtils`\n  [56ddb016] Logging `@stdlib/Logging`\n  [d6f4376e] Markdown `@stdlib/Markdown`\n  [9a3f8284] Random `@stdlib/Random`\n  [ea8e919c] SHA `@stdlib/SHA`\n  [9e88b42a] Serialization `@stdlib/Serialization`\n  [8dfed614] Test `@stdlib/Test`\n     Testing Running tests...\nTest Summary: | Pass  Total\nExample tests |    9      9\n     Testing Example tests passed\n\nAnd if all went correctly, you should see a similar output as above. Using Test.jl, more complicated tests can be added for packages but this should ideally point developers in the direction of how to get started with testing their own created packages."},{"location":"stdlib/Test.html#Code-Coverage","page":"Unit Testing","title":"Code Coverage","category":"section","text":"Code coverage tracking during tests can be enabled using the pkg> test --coverage flag (or at a lower level using the --code-coverage julia arg). This is on by default in the julia-runtest GitHub action.\n\nTo evaluate coverage either manually inspect the .cov files that are generated beside the source files locally, or in CI use the julia-processcoverage GitHub action.\n\ncompat: Julia 1.11\nSince Julia 1.11, coverage is not collected during the package precompilation phase."},{"location":"stdlib/Test.html#Base.runtests","page":"Unit Testing","title":"Base.runtests","category":"function","text":"Base.runtests(tests=[\"all\"]; ncores=ceil(Int, Sys.EFFECTIVE_CPU_THREADS / 2),\n              exit_on_error=false, revise=false, propagate_project=true, [seed], [julia_args::Cmd])\n\nRun the Julia unit tests listed in tests, which can be either a string or an array of strings, using ncores processors. If exit_on_error is false, when one test fails, all remaining tests in other files will still be run; they are otherwise discarded, when exit_on_error == true. If revise is true, the Revise package is used to load any modifications to Base or to the standard libraries before running the tests. If propagate_project is true the current project is propagated to the test environment. If a seed is provided via the keyword argument, it is used to seed the global RNG in the context where the tests are run; otherwise the seed is chosen randomly. The argument julia_args can be used to pass custom julia command line flags to the test process.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test","page":"Unit Testing","title":"Test.@test","category":"macro","text":"@test ex\n@test f(args...) key=val ...\n@test ex broken=true\n@test ex skip=true\n\nTest that the expression ex evaluates to true. If executed inside a @testset, return a Pass Result if it does, a Fail Result if it is false, and an Error Result if it could not be evaluated. If executed outside a @testset, throw an exception instead of returning Fail or Error.\n\nExamples\n\njulia> @test true\nTest Passed\n\njulia> @test [1, 2] + [2, 1] == [3, 3]\nTest Passed\n\nThe @test f(args...) key=val... form is equivalent to writing @test f(args..., key=val...) which can be useful when the expression is a call using infix syntax such as approximate comparisons:\n\njulia> @test π ≈ 3.14 atol=0.01\nTest Passed\n\nThis is equivalent to the uglier test @test ≈(π, 3.14, atol=0.01). It is an error to supply more than one expression unless the first is a call expression and the rest are assignments (k=v).\n\nYou can use any key for the key=val arguments, except for broken and skip, which have special meanings in the context of @test:\n\nbroken=cond indicates a test that should pass but currently consistently fails when cond==true.  Tests that the expression ex evaluates to false or causes an exception.  Returns a Broken Result if it does, or an Error Result if the expression evaluates to true.  Regular @test ex is evaluated when cond==false.\nskip=cond marks a test that should not be executed but should be included in test summary reporting as Broken, when cond==true.  This can be useful for tests that intermittently fail, or tests of not-yet-implemented functionality. Regular @test ex is evaluated when cond==false.\n\nExamples\n\njulia> @test 2 + 2 ≈ 6 atol=1 broken=true\nTest Broken\n  Expression: ≈(2 + 2, 6, atol = 1)\n\njulia> @test 2 + 2 ≈ 5 atol=1 broken=false\nTest Passed\n\njulia> @test 2 + 2 == 5 skip=true\nTest Broken\n  Skipped: 2 + 2 == 5\n\njulia> @test 2 + 2 == 4 skip=false\nTest Passed\n\ncompat: Julia 1.7\nThe broken and skip keyword arguments require at least Julia 1.7.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_throws","page":"Unit Testing","title":"Test.@test_throws","category":"macro","text":"@test_throws exception expr\n@test_throws extype pattern expr\n\nTests that the expression expr throws exception. The exception may specify either a type, a string, regular expression, or list of strings occurring in the displayed error message, a matching function, or a value (which will be tested for equality by comparing fields).\n\nIn the two-argument form, @test_throws exception expr, the exception can be a type or a pattern.\n\nIn the three-argument form, @test_throws extype pattern expr, both the exception type and a message pattern are tested. The extype must be a type, and pattern may be a string, regular expression, or list of strings occurring in the displayed error message, a matching function, or a value.\n\nNote that @test_throws does not support a trailing keyword form.\n\ncompat: Julia 1.8\nThe ability to specify anything other than a type or a value as exception requires Julia v1.8 or later.\n\ncompat: Julia 1.13\nThe three-argument form @test_throws extype pattern expr requires Julia v1.12 or later.\n\nExamples\n\njulia> @test_throws BoundsError [1, 2, 3][4]\nTest Passed\n      Thrown: BoundsError\n\njulia> @test_throws DimensionMismatch [1, 2, 3] + [1, 2]\nTest Passed\n      Thrown: DimensionMismatch\n\njulia> @test_throws \"Try sqrt(Complex\" sqrt(-1)\nTest Passed\n     Message: \"DomainError with -1.0:\\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\"\n\njulia> @test_throws ErrorException \"error foo\" error(\"error foo 1\")\nTest Passed\n      Thrown: ErrorException\n\nIn the third example, instead of matching a single string it could alternatively have been performed with:\n\n[\"Try\", \"Complex\"] (a list of strings)\nr\"Try sqrt\\([Cc]omplex\" (a regular expression)\nstr -> occursin(\"complex\", str) (a matching function)\n\nIn the final example, both the exception type (ErrorException) and message pattern (\"error foo\") are tested.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@testset","page":"Unit Testing","title":"Test.@testset","category":"macro","text":"@testset [CustomTestSet] [options...] [\"description\"] begin test_ex end\n@testset [CustomTestSet] [options...] [\"description $v\"] for v in itr test_ex end\n@testset [CustomTestSet] [options...] [\"description $v, $w\"] for v in itrv, w in itrw test_ex end\n@testset [CustomTestSet] [options...] [\"description\"] test_func()\n@testset let v = v, w = w; test_ex; end\n\nWith begin/end or function call\n\nWhen @testset is used, with begin/end or a single function call, the macro starts a new test set in which to evaluate the given expression.\n\nIf no custom testset type is given it defaults to creating a DefaultTestSet. DefaultTestSet records all the results and, if there are any Fails or Errors, throws an exception at the end of the top-level (non-nested) test set, along with a summary of the test results.\n\nAny custom testset type (subtype of AbstractTestSet) can be given and it will also be used for any nested @testset invocations. The given options are only applied to the test set where they are given. The default test set type accepts the following options:\n\nverbose::Bool: if true, the result summary of the nested testsets is shown even when they all pass (the default is false).\nshowtiming::Bool: if true, the duration of each displayed testset is shown (the default is true).\nfailfast::Bool: if true, any test failure or error will cause the testset and any child testsets to return immediately (the default is false). This can also be set globally via the env var JULIA_TEST_FAILFAST.\nrng::Random.AbstractRNG: use the given random number generator (RNG) as the global one for the testset.  rng must be copy!-able.  This option can be useful to locally reproduce stochastic test failures which only depend on the state of the global RNG.\n\ncompat: Julia 1.8\n@testset test_func() requires at least Julia 1.8.\n\ncompat: Julia 1.9\nfailfast requires at least Julia 1.9.\n\ncompat: Julia 1.12\nThe rng option requires at least Julia 1.12.\n\nThe description string accepts interpolation from the loop indices. If no description is provided, one is constructed based on the variables. If a function call is provided, its name will be used. Explicit description strings override this behavior.\n\nBy default the @testset macro will return the testset object itself, though this behavior can be customized in other testset types. If a for loop is used then the macro collects and returns a list of the return values of the finish method, which by default will return a list of the testset objects used in each iteration.\n\nBefore the execution of the body of a @testset, there is an implicit call to copy!(Random.default_rng(), rng) where rng is the RNG of the current task, or the value of the RNG passed via the rng option. Moreover, after the execution of the body, the state of the global RNG is restored to what it was before the @testset. This is meant to ease reproducibility in case of failure, and to allow seamless re-arrangements of @testsets regardless of their side-effect on the global RNG state.\n\nnote: RNG of nested testsets\nUnless changed with the rng option, the same RNG is set at the beginning of all nested testsets.  The RNG printed to screen when a testset has failures is the global RNG of the outermost testset even if inner testsets have different RNGs manually set by the user.\n\nExamples\n\njulia> @testset \"trigonometric identities\" begin\n           θ = 2/3*π\n           @test sin(-θ) ≈ -sin(θ)\n           @test cos(-θ) ≈ cos(θ)\n           @test sin(2θ) ≈ 2*sin(θ)*cos(θ)\n           @test cos(2θ) ≈ cos(θ)^2 - sin(θ)^2\n       end;\nTest Summary:            | Pass  Total  Time\ntrigonometric identities |    4      4  0.2s\n\n@testset for\n\nWhen @testset for is used, the macro starts a new test set for each iteration of the provided loop. The semantics of each test set are otherwise identical to that of the begin/end case (as if used for each loop iteration).\n\n@testset let\n\nWhen @testset let is used, the macro starts a transparent test set with the given object added as a context object to any failing or erroring test contained therein. This is useful when performing a set of related tests on one larger object and it is desirable to print this larger object when any of the individual tests fail. Transparent test sets do not introduce additional levels of nesting in the test set hierarchy and are passed through directly to the parent test set (with the context object appended to any failing tests.)\n\ncompat: Julia 1.9\n@testset let requires at least Julia 1.9.\n\ncompat: Julia 1.10\nMultiple let assignments are supported since Julia 1.10.\n\ncompat: Julia 1.13\nContext is shown when a test errors since Julia 1.13.\n\nSpecial implicit world age increment for @testset begin\n\nWorld age inside @testset begin increments implicitly after every statement. This matches the behavior of ordinary toplevel code, but not that of ordinary begin/end blocks, i.e. with respect to world age, @testset begin behaves as if the body of the begin/end block was written at toplevel.\n\nExamples\n\njulia> @testset let logi = log(im)\n           @test imag(logi) == π/2\n           @test !iszero(real(logi))\n       end\nTest Failed at none:3\n  Expression: !(iszero(real(logi)))\n   Evaluated: !(iszero(0.0))\n     Context: logi = 0.0 + 1.5707963267948966im\nERROR: There was an error during testing\n\njulia> @testset let logi = log(im), op = !iszero\n           @test imag(logi) == π/2\n           @test op(real(logi))\n       end\nTest Failed at none:3\n  Expression: op(real(logi))\n   Evaluated: op(0.0)\n     Context: logi = 0.0 + 1.5707963267948966im\n              op = !iszero\nERROR: There was an error during testing\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.TestSetException","page":"Unit Testing","title":"Test.TestSetException","category":"type","text":"TestSetException\n\nThrown when a test set finishes and not all tests passed.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_logs","page":"Unit Testing","title":"Test.@test_logs","category":"macro","text":"@test_logs [log_patterns...] [keywords] expression\n\nCollect a list of log records generated by expression using collect_test_logs, check that they match the sequence log_patterns, and return the value of expression.  The keywords provide some simple filtering of log records: the min_level keyword controls the minimum log level which will be collected for the test, the match_mode keyword defines how matching will be performed (the default :all checks that all logs and patterns match pairwise; use :any to check that the pattern matches at least once somewhere in the sequence.)\n\nThe most useful log pattern is a simple tuple of the form (level,message). A different number of tuple elements may be used to match other log metadata, corresponding to the arguments to passed to AbstractLogger via the handle_message function: (level,message,module,group,id,file,line). Elements which are present will be matched pairwise with the log record fields using == by default, with the special cases that Symbols may be used for the standard log levels, and Regexs in the pattern will match string or Symbol fields using occursin.\n\nExamples\n\nConsider a function which logs a warning, and several debug messages:\n\nfunction foo(n)\n    @info \"Doing foo with n=$n\"\n    for i=1:n\n        @debug \"Iteration $i\"\n    end\n    42\nend\n\nWe can test the info message using\n\n@test_logs (:info,\"Doing foo with n=2\") foo(2)\n\nIf we also wanted to test the debug messages, these need to be enabled with the min_level keyword:\n\nusing Logging\n@test_logs (:info,\"Doing foo with n=2\") (:debug,\"Iteration 1\") (:debug,\"Iteration 2\") min_level=Logging.Debug foo(2)\n\nIf you want to test that some particular messages are generated while ignoring the rest, you can set the keyword match_mode=:any:\n\nusing Logging\n@test_logs (:info,) (:debug,\"Iteration 42\") min_level=Logging.Debug match_mode=:any foo(100)\n\nThe macro may be chained with @test to also test the returned value:\n\n@test (@test_logs (:info,\"Doing foo with n=2\") foo(2)) == 42\n\nIf you want to test for the absence of warnings, you can omit specifying log patterns and set the min_level accordingly:\n\n# test that the expression logs no messages when the logger level is warn:\n@test_logs min_level=Logging.Warn @info(\"Some information\") # passes\n@test_logs min_level=Logging.Warn @warn(\"Some information\") # fails\n\nIf you want to test the absence of warnings (or error messages) in stderr which are not generated by @warn, see @test_nowarn.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.TestLogger","page":"Unit Testing","title":"Test.TestLogger","category":"type","text":"TestLogger(; min_level=Info, catch_exceptions=false)\n\nCreate a TestLogger which captures logged messages in its logs::Vector{LogRecord} field.\n\nSet min_level to control the LogLevel, catch_exceptions for whether or not exceptions thrown as part of log event generation should be caught, and respect_maxlog for whether or not to follow the convention of logging messages with maxlog=n for some integer n at most n times.\n\nSee also: LogRecord.\n\nExamples\n\njulia> using Test, Logging\n\njulia> f() = @info \"Hi\" number=5;\n\njulia> test_logger = TestLogger();\n\njulia> with_logger(test_logger) do\n           f()\n           @info \"Bye!\"\n       end\n\njulia> @test test_logger.logs[1].message == \"Hi\"\nTest Passed\n\njulia> @test test_logger.logs[1].kwargs[:number] == 5\nTest Passed\n\njulia> @test test_logger.logs[2].message == \"Bye!\"\nTest Passed\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.LogRecord","page":"Unit Testing","title":"Test.LogRecord","category":"type","text":"LogRecord\n\nStores the results of a single log event. Fields:\n\nlevel: the LogLevel of the log message\nmessage: the textual content of the log message\n_module: the module of the log event\ngroup: the logging group (by default, the name of the file containing the log event)\nid: the ID of the log event\nfile: the file containing the log event\nline: the line within the file of the log event\nkwargs: any keyword arguments passed to the log event\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@inferred","page":"Unit Testing","title":"Test.@inferred","category":"macro","text":"@inferred [AllowedType] f(x)\n\nTests that the call expression f(x) returns a value of the same type inferred by the compiler. It is useful to check for type stability.\n\nf(x) can be any call expression. Returns the result of f(x) if the types match, and an Error Result if it finds different types.\n\nOptionally, AllowedType relaxes the test, by making it pass when either the type of f(x) matches the inferred type modulo AllowedType, or when the return type is a subtype of AllowedType. This is useful when testing type stability of functions returning a small union such as Union{Nothing, T} or Union{Missing, T}.\n\njulia> f(a) = a > 1 ? 1 : 1.0\nf (generic function with 1 method)\n\njulia> typeof(f(2))\nInt64\n\njulia> @code_warntype f(2)\nMethodInstance for f(::Int64)\n  from f(a) @ Main none:1\nArguments\n  #self#::Core.Const(f)\n  a::Int64\nBody::UNION{FLOAT64, INT64}\n1 ─ %1 = :>::Core.Const(>)\n│   %2 = (%1)(a, 1)::Bool\n└──      goto #3 if not %2\n2 ─      return 1\n3 ─      return 1.0\n\njulia> @inferred f(2)\nERROR: return type Int64 does not match inferred return type Union{Float64, Int64}\n[...]\n\njulia> @inferred max(1, 2)\n2\n\njulia> g(a) = a < 10 ? missing : 1.0\ng (generic function with 1 method)\n\njulia> @inferred g(20)\nERROR: return type Float64 does not match inferred return type Union{Missing, Float64}\n[...]\n\njulia> @inferred Missing g(20)\n1.0\n\njulia> h(a) = a < 10 ? missing : f(a)\nh (generic function with 1 method)\n\njulia> @inferred Missing h(20)\nERROR: return type Int64 does not match inferred return type Union{Missing, Float64, Int64}\n[...]\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_deprecated","page":"Unit Testing","title":"Test.@test_deprecated","category":"macro","text":"@test_deprecated [pattern] expression\n\nWhen --depwarn=yes, test that expression emits a deprecation warning and return the value of expression.  The log message string will be matched against pattern which defaults to r\"deprecated\"i.\n\nWhen --depwarn=no, simply return the result of executing expression.  When --depwarn=error, check that an ErrorException is thrown.\n\nExamples\n\n# Deprecated in julia 0.7\n@test_deprecated num2hex(1)\n\n# The returned value can be tested by chaining with @test:\n@test (@test_deprecated num2hex(1)) == \"0000000000000001\"\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_warn","page":"Unit Testing","title":"Test.@test_warn","category":"macro","text":"@test_warn msg expr\n\nTest whether evaluating expr results in stderr output that contains the msg string or matches the msg regular expression.  If msg is a boolean function, tests whether msg(output) returns true.  If msg is a tuple or array, checks that the error output contains/matches each item in msg. Returns the result of evaluating expr.\n\nSee also @test_nowarn to check for the absence of error output.\n\nNote: Warnings generated by @warn cannot be tested with this macro. Use @test_logs instead.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_nowarn","page":"Unit Testing","title":"Test.@test_nowarn","category":"macro","text":"@test_nowarn expr\n\nTest whether evaluating expr results in empty stderr output (no warnings or other messages).  Returns the result of evaluating expr.\n\nNote: The absence of warnings generated by @warn cannot be tested with this macro. Use @test_logs instead.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_broken","page":"Unit Testing","title":"Test.@test_broken","category":"macro","text":"@test_broken ex\n@test_broken f(args...) key=val ...\n\nIndicates a test that should pass but currently consistently fails. Tests that the expression ex evaluates to false or causes an exception. Returns a Broken Result if it does, or an Error Result if the expression evaluates to true.  This is equivalent to @test ex broken=true.\n\nThe @test_broken f(args...) key=val... form works as for the @test macro.\n\nExamples\n\njulia> @test_broken 1 == 2\nTest Broken\n  Expression: 1 == 2\n\njulia> @test_broken 1 == 2 atol=0.1\nTest Broken\n  Expression: ==(1, 2, atol = 0.1)\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.@test_skip","page":"Unit Testing","title":"Test.@test_skip","category":"macro","text":"@test_skip ex\n@test_skip f(args...) key=val ...\n\nMarks a test that should not be executed but should be included in test summary reporting as Broken. This can be useful for tests that intermittently fail, or tests of not-yet-implemented functionality.  This is equivalent to @test ex skip=true.\n\nThe @test_skip f(args...) key=val... form works as for the @test macro.\n\nExamples\n\njulia> @test_skip 1 == 2\nTest Broken\n  Skipped: 1 == 2\n\njulia> @test_skip 1 == 2 atol=0.1\nTest Broken\n  Skipped: ==(1, 2, atol = 0.1)\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.Result","page":"Unit Testing","title":"Test.Result","category":"type","text":"Test.Result\n\nAll tests produce a result object. This object may or may not be stored, depending on whether the test is part of a test set.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.Pass","page":"Unit Testing","title":"Test.Pass","category":"type","text":"Test.Pass <: Test.Result\n\nThe test condition was true, i.e. the expression evaluated to true or the correct exception was thrown.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.Fail","page":"Unit Testing","title":"Test.Fail","category":"type","text":"Test.Fail <: Test.Result\n\nThe test condition was false, i.e. the expression evaluated to false or the correct exception was not thrown.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.Error","page":"Unit Testing","title":"Test.Error","category":"type","text":"Test.Error <: Test.Result\n\nThe test condition couldn't be evaluated due to an exception, or it evaluated to something other than a Bool. In the case of @test_broken it is used to indicate that an unexpected Pass Result occurred.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.Broken","page":"Unit Testing","title":"Test.Broken","category":"type","text":"Test.Broken <: Test.Result\n\nThe test condition is the expected (failed) result of a broken test, or was explicitly skipped with @test_skip.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.record","page":"Unit Testing","title":"Test.record","category":"function","text":"record(ts::AbstractTestSet, res::Result)\n\nRecord a result to a testset. This function is called by the @testset infrastructure each time a contained @test macro completes, and is given the test result (which could be an Error). This will also be called with an Error if an exception is thrown inside the test block but outside of a @test context.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.finish","page":"Unit Testing","title":"Test.finish","category":"function","text":"finish(ts::AbstractTestSet)\n\nDo any final processing necessary for the given testset. This is called by the @testset infrastructure after a test block executes.\n\nCustom AbstractTestSet subtypes should call record on their parent (if there is one) to add themselves to the tree of test results. This might be implemented as:\n\nif get_testset_depth() != 0\n    # Attach this test set to the parent test set\n    parent_ts = get_testset()\n    record(parent_ts, self)\n    return self\nend\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.get_testset","page":"Unit Testing","title":"Test.get_testset","category":"function","text":"get_testset()\n\nRetrieve the active test set from the task's local storage. If no test set is active, use the fallback default test set.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.get_testset_depth","page":"Unit Testing","title":"Test.get_testset_depth","category":"function","text":"get_testset_depth()\n\nReturn the number of active test sets, not including the default test set\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.TestCounts","page":"Unit Testing","title":"Test.TestCounts","category":"type","text":"TestCounts\n\nHolds the state for recursively gathering the results of a test set for display purposes.\n\nFields:\n\ncustomized: Whether the function get_test_counts was customized for the AbstractTestSet               this counts object is for. If a custom method was defined, always pass true               to the constructor.\npasses: The number of passing @test invocations.\nfails: The number of failing @test invocations.\nerrors: The number of erroring @test invocations.\nbroken: The number of broken @test invocations.\npasses: The cumulative number of passing @test invocations.\nfails: The cumulative number of failing @test invocations.\nerrors: The cumulative number of erroring @test invocations.\nbroken: The cumulative number of broken @test invocations.\nduration: The total duration the AbstractTestSet in question ran for, as a formatted String.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.get_test_counts","page":"Unit Testing","title":"Test.get_test_counts","category":"function","text":"\"     gettestcounts(::AbstractTestSet)::TestCounts\n\nRecursive function that counts the number of test results of each type directly in the testset, and totals across the child testsets.\n\nCustom AbstractTestSet should implement this function to get their totals counted & displayed with DefaultTestSet as well.\n\nIf this is not implemented for a custom TestSet, the printing falls back to reporting x for failures and ?s for the duration.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.format_duration","page":"Unit Testing","title":"Test.format_duration","category":"function","text":"format_duration(::AbstractTestSet)\n\nReturn a formatted string for printing the duration the testset ran for.\n\nIf not defined, falls back to \"?s\".\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.print_test_results","page":"Unit Testing","title":"Test.print_test_results","category":"function","text":"print_test_results([io::IO], ts::AbstractTestSet, depth_pad=0)\n\nPrint the results of an AbstractTestSet as a formatted table.\n\ndepth_pad refers to how much padding should be added in front of all output. If io is not provided, defaults to stdout.\n\nCalled inside of Test.finish, if the finished testset is the topmost testset.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.GenericArray","page":"Unit Testing","title":"Test.GenericArray","category":"type","text":"The GenericArray can be used to test generic array APIs that program to the AbstractArray interface, in order to ensure that functions can work with array types besides the standard Array type.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.GenericDict","page":"Unit Testing","title":"Test.GenericDict","category":"type","text":"The GenericDict can be used to test generic dict APIs that program to the AbstractDict interface, in order to ensure that functions can work with associative types besides the standard Dict type.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.GenericOrder","page":"Unit Testing","title":"Test.GenericOrder","category":"type","text":"The GenericOrder can be used to test APIs for their support of generic ordered types.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.GenericSet","page":"Unit Testing","title":"Test.GenericSet","category":"type","text":"The GenericSet can be used to test generic set APIs that program to the AbstractSet interface, in order to ensure that functions can work with set types besides the standard Set and BitSet types.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.GenericString","page":"Unit Testing","title":"Test.GenericString","category":"type","text":"The GenericString can be used to test generic string APIs that program to the AbstractString interface, in order to ensure that functions can work with string types besides the standard String type.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.detect_ambiguities","page":"Unit Testing","title":"Test.detect_ambiguities","category":"function","text":"detect_ambiguities(mod1, mod2...; recursive=false,\n                                  ambiguous_bottom=false,\n                                  allowed_undefineds=nothing)\n\nReturn a vector of (Method,Method) pairs of ambiguous methods defined in the specified modules. Use recursive=true to test in all submodules.\n\nambiguous_bottom controls whether ambiguities triggered only by Union{} type parameters are included; in most cases you probably want to set this to false. See Base.isambiguous.\n\nSee Test.detect_unbound_args for an explanation of allowed_undefineds.\n\ncompat: Julia 1.8\nallowed_undefineds requires at least Julia 1.8.\n\n\n\n\n\n"},{"location":"stdlib/Test.html#Test.detect_unbound_args","page":"Unit Testing","title":"Test.detect_unbound_args","category":"function","text":"detect_unbound_args(mod1, mod2...; recursive=false, allowed_undefineds=nothing)\n\nReturn a vector of Methods which may have unbound type parameters. Use recursive=true to test in all submodules.\n\nBy default, any undefined symbols trigger a warning. This warning can be suppressed by supplying a collection of GlobalRefs for which the warning can be skipped. For example, setting\n\nallowed_undefineds = Set([GlobalRef(Base, :active_repl),\n                          GlobalRef(Base, :active_repl_backend)])\n\nwould suppress warnings about Base.active_repl and Base.active_repl_backend.\n\ncompat: Julia 1.8\nallowed_undefineds requires at least Julia 1.8.\n\n\n\n\n\n"},{"location":"devdocs/compiler.html#High-level-Overview-of-the-Native-Code-Generation-Process","page":"High-level Overview of the Native-Code Generation Process","title":"High-level Overview of the Native-Code Generation Process","category":"section","text":""},{"location":"devdocs/compiler.html#Representation-of-Pointers","page":"High-level Overview of the Native-Code Generation Process","title":"Representation of Pointers","category":"section","text":"When emitting code to an object file, pointers will be emitted as relocations. The deserialization code will ensure any object that pointed to one of these constants gets recreated and contains the right runtime pointer.\n\nOtherwise, they will be emitted as literal constants.\n\nTo emit one of these objects, call literal_pointer_val. It'll handle tracking the Julia value and the LLVM global, ensuring they are valid both for the current runtime and after deserialization.\n\nWhen emitted into the object file, these globals are stored as references in a large gvals table. This allows the deserializer to reference them by index, and implement a custom manual mechanism similar to a Global Offset Table (GOT) to restore them.\n\nFunction pointers are handled similarly. They are stored as values in a large fvals table. Like globals, this allows the deserializer to reference them by index.\n\nNote that extern functions are handled separately, with names, via the usual symbol resolution mechanism in the linker.\n\nNote too that ccall functions are also handled separately, via a manual GOT and Procedure Linkage Table (PLT)."},{"location":"devdocs/compiler.html#Representation-of-Intermediate-Values","page":"High-level Overview of the Native-Code Generation Process","title":"Representation of Intermediate Values","category":"section","text":"Values are passed around in a jl_cgval_t struct. This represents an R-value, and includes enough information to determine how to assign or pass it somewhere.\n\nThey are created via one of the helper constructors, usually: mark_julia_type (for immediate values) and mark_julia_slot (for pointers to values).\n\nThe function convert_julia_type can transform between any two types. It returns an R-value with cgval.typ set to typ. It'll cast the object to the requested representation, making heap boxes, allocating stack copies, and computing tagged unions as needed to change the representation.\n\nBy contrast update_julia_type will change cgval.typ to typ, only if it can be done at zero-cost (i.e. without emitting any code)."},{"location":"devdocs/compiler.html#Union-representation","page":"High-level Overview of the Native-Code Generation Process","title":"Union representation","category":"section","text":"Inferred union types may be stack allocated via a tagged type representation.\n\nThe primitive routines that need to be able to handle tagged unions are:\n\nmark-type\nload-local\nstore-local\nisa\nis\nemit_typeof\nemit_sizeof\nboxed\nunbox\nspecialized cc-ret\n\nEverything else should be possible to handle in inference by using these primitives to implement union-splitting.\n\nThe representation of the tagged-union is as a pair of < void* union, byte selector >. The selector is fixed-size as byte & 0x7f, and will union-tag the first 126 isbits. It records the one-based depth-first count into the type-union of the isbits objects inside. An index of zero indicates that the union* is actually a tagged heap-allocated jl_value_t*, and needs to be treated as normal for a boxed object rather than as a tagged union.\n\nThe high bit of the selector (byte & 0x80) can be tested to determine if the void* is actually a heap-allocated (jl_value_t*) box, thus avoiding the cost of re-allocating a box, while maintaining the ability to efficiently handle union-splitting based on the low bits.\n\nIt is guaranteed that byte & 0x7f is an exact test for the type, if the value can be represented by a tag – it will never be marked byte = 0x80. It is not necessary to also test the type-tag when testing isa.\n\nThe union* memory region may be allocated at any size. The only constraint is that it is big enough to contain the data currently specified by selector. It might not be big enough to contain the union of all types that could be stored there according to the associated Union type field. Use appropriate care when copying."},{"location":"devdocs/compiler.html#Specialized-Calling-Convention-Signature-Representation","page":"High-level Overview of the Native-Code Generation Process","title":"Specialized Calling Convention Signature Representation","category":"section","text":"A jl_returninfo_t object describes the specialized calling convention details of any callable. It can be generated from any (specTypes, rettype) pair, such as a CodeInstance, or other place they are declared. This is the expected calling convention for specptr, but other data may be stored there. Only if the function pointer stored there has the expected specialized calling convention will the corresponding flag be set in specsigflags to indicate it is useable.\n\nIf any of the arguments or return type of a method can be represented unboxed, and none are unable to be represented unboxed (such as an unbounded vararg), it will be given an optimized calling convention signature based on the specTypes and rettype values.\n\nThe general principles are that:\n\nPrimitive types get passed in int/float registers.\nTuples of VecElement types get passed in vector registers.\nStructs get passed on the stack.\nReturn values are handled similarly to arguments, with a size-cutoff at which they will instead be returned via a hidden sret argument.\n\nThe total logic for this is implemented by get_specsig_function and deserves_sret.\n\nAdditionally, if the return type is a union, it may be returned as a pair of values (a pointer and a tag). If the union values can be stack-allocated, then sufficient space to store them will also be passed as a hidden first argument. If the struct to return needs gc roots, space for those will be passed as a hidden second argument. It is up to the callee whether the returned pointer will point to this space, a boxed object, or even other constant memory."},{"location":"stdlib/SharedArrays.html#Shared-Arrays","page":"Shared Arrays","title":"Shared Arrays","category":"section","text":"SharedArray represents an array, which is shared across multiple processes, on a single machine."},{"location":"stdlib/SharedArrays.html#SharedArrays.SharedArray","page":"Shared Arrays","title":"SharedArrays.SharedArray","category":"type","text":"SharedArray{T}(dims::NTuple; init=false, pids=Int[])\nSharedArray{T,N}(...)\n\nConstruct a SharedArray of a bits type T and size dims across the processes specified by pids - all of which have to be on the same host.  If N is specified by calling SharedArray{T,N}(dims), then N must match the length of dims.\n\nIf pids is left unspecified, the shared array will be mapped across all processes on the current host, including the master. But, localindices and indexpids will only refer to worker processes. This facilitates work distribution code to use workers for actual computation with the master process acting as a driver.\n\nIf an init function of the type initfn(S::SharedArray) is specified, it is called on all the participating workers.\n\nThe shared array is valid as long as a reference to the SharedArray object exists on the node which created the mapping.\n\nSharedArray{T}(filename::AbstractString, dims::NTuple, [offset=0]; mode=nothing, init=false, pids=Int[])\nSharedArray{T,N}(...)\n\nConstruct a SharedArray backed by the file filename, with element type T (must be a bits type) and size dims, across the processes specified by pids - all of which have to be on the same host. This file is mmapped into the host memory, with the following consequences:\n\nThe array data must be represented in binary format (e.g., an ASCII format like CSV cannot be supported)\nAny changes you make to the array values (e.g., A[3] = 0) will also change the values on disk\n\nIf pids is left unspecified, the shared array will be mapped across all processes on the current host, including the master. But, localindices and indexpids will only refer to worker processes. This facilitates work distribution code to use workers for actual computation with the master process acting as a driver.\n\nmode must be one of \"r\", \"r+\", \"w+\", or \"a+\", and defaults to \"r+\" if the file specified by filename already exists, or \"w+\" if not. If an init function of the type initfn(S::SharedArray) is specified, it is called on all the participating workers. You cannot specify an init function if the file is not writable.\n\noffset allows you to skip the specified number of bytes at the beginning of the file.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#SharedArrays.SharedVector","page":"Shared Arrays","title":"SharedArrays.SharedVector","category":"type","text":"SharedVector\n\nA one-dimensional SharedArray.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#SharedArrays.SharedMatrix","page":"Shared Arrays","title":"SharedArrays.SharedMatrix","category":"type","text":"SharedMatrix\n\nA two-dimensional SharedArray.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#Distributed.procs-Tuple{SharedArray}","page":"Shared Arrays","title":"Distributed.procs","category":"method","text":"procs(S::SharedArray)\n\nGet the vector of processes mapping the shared array.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#SharedArrays.sdata","page":"Shared Arrays","title":"SharedArrays.sdata","category":"function","text":"sdata(S::SharedArray)\n\nReturn the actual Array object backing S.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#SharedArrays.indexpids","page":"Shared Arrays","title":"SharedArrays.indexpids","category":"function","text":"indexpids(S::SharedArray)\n\nReturn the current worker's index in the list of workers mapping the SharedArray (i.e. in the same list returned by procs(S)), or 0 if the SharedArray is not mapped locally.\n\n\n\n\n\n"},{"location":"stdlib/SharedArrays.html#SharedArrays.localindices","page":"Shared Arrays","title":"SharedArrays.localindices","category":"function","text":"localindices(S::SharedArray)\n\nReturn a range describing the \"default\" indices to be handled by the current process.  This range should be interpreted in the sense of linear indexing, i.e., as a sub-range of 1:length(S).  In multi-process contexts, returns an empty range in the parent process (or any process for which indexpids returns 0).\n\nIt's worth emphasizing that localindices exists purely as a convenience, and you can partition work on the array among workers any way you wish. For a SharedArray, all indices should be equally fast for each worker process.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML","page":"TOML","title":"TOML","category":"section","text":"TOML.jl is a Julia standard library for parsing and writing TOML v1.0 files."},{"location":"stdlib/TOML.html#Parsing-TOML-data","page":"TOML","title":"Parsing TOML data","category":"section","text":"julia> using TOML\n\njulia> data = \"\"\"\n           [database]\n           server = \"192.168.1.1\"\n           ports = [ 8001, 8001, 8002 ]\n       \"\"\";\n\njulia> TOML.parse(data)\nDict{String, Any} with 1 entry:\n  \"database\" => Dict{String, Any}(\"server\"=>\"192.168.1.1\", \"ports\"=>[8001, 8001…\n\nTo parse a file, use TOML.parsefile. If the file has a syntax error, an exception is thrown:\n\njulia> using TOML\n\njulia> TOML.parse(\"\"\"\n           value = 0.0.0\n       \"\"\")\nERROR: TOML Parser error:\nnone:1:16 error: failed to parse value\n      value = 0.0.0\n                 ^\n[...]\n\nThere are other versions of the parse functions (TOML.tryparse and TOML.tryparsefile) that instead of throwing exceptions on parser error returns a TOML.ParserError with information:\n\njulia> using TOML\n\njulia> err = TOML.tryparse(\"\"\"\n           value = 0.0.0\n       \"\"\");\n\njulia> err.type\nErrGenericValueError::ErrorType = 14\n\njulia> err.line\n1\n\njulia> err.column\n16"},{"location":"stdlib/TOML.html#Exporting-data-to-TOML-file","page":"TOML","title":"Exporting data to TOML file","category":"section","text":"The TOML.print function is used to print (or serialize) data into TOML format.\n\njulia> using TOML\n\njulia> data = Dict(\n          \"names\" => [\"Julia\", \"Julio\"],\n          \"age\" => [10, 20],\n       );\n\njulia> TOML.print(data)\nnames = [\"Julia\", \"Julio\"]\nage = [10, 20]\n\njulia> fname = tempname();\n\njulia> open(fname, \"w\") do io\n           TOML.print(io, data)\n       end\n\njulia> TOML.parsefile(fname)\nDict{String, Any} with 2 entries:\n  \"names\" => [\"Julia\", \"Julio\"]\n  \"age\"   => [10, 20]\n\nKeys can be sorted according to some value\n\njulia> using TOML\n\njulia> TOML.print(Dict(\n       \"abc\"  => 1,\n       \"ab\"   => 2,\n       \"abcd\" => 3,\n       ); sorted=true, by=length)\nab = 2\nabc = 1\nabcd = 3\n\nFor custom structs, pass a function that converts the struct to a supported type\n\njulia> using TOML\n\njulia> struct MyStruct\n           a::Int\n           b::String\n       end\n\njulia> TOML.print(Dict(\"foo\" => MyStruct(5, \"bar\"))) do x\n           x isa MyStruct && return [x.a, x.b]\n           error(\"unhandled type $(typeof(x))\")\n       end\nfoo = [5, \"bar\"]"},{"location":"stdlib/TOML.html#References","page":"TOML","title":"References","category":"section","text":""},{"location":"stdlib/TOML.html#TOML.parse","page":"TOML","title":"TOML.parse","category":"function","text":"parse(x::Union{AbstractString, IO})\nparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string  or stream x, and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparse.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.parsefile","page":"TOML","title":"TOML.parsefile","category":"function","text":"parsefile(f::AbstractString)\nparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Throw a ParserError upon failure.\n\nSee also TOML.tryparsefile.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.tryparse","page":"TOML","title":"TOML.tryparse","category":"function","text":"tryparse(x::Union{AbstractString, IO})\ntryparse(p::Parser, x::Union{AbstractString, IO})\n\nParse the string or stream x, and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parse.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.tryparsefile","page":"TOML","title":"TOML.tryparsefile","category":"function","text":"tryparsefile(f::AbstractString)\ntryparsefile(p::Parser, f::AbstractString)\n\nParse file f and return the resulting table (dictionary). Return a ParserError upon failure.\n\nSee also TOML.parsefile.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.print","page":"TOML","title":"TOML.print","category":"function","text":"print([to_toml::Function], io::IO [=stdout], data::AbstractDict; sorted=false, by=identity, inline_tables::IdSet{<:AbstractDict})\n\nWrite data as TOML syntax to the stream io. If the keyword argument sorted is set to true, sort tables according to the function given by the keyword argument by. If the keyword argument inline_tables is given, it should be a set of tables that should be printed \"inline\".\n\ncompat: Julia 1.11\nThe inline_tables keyword argument is supported by Julia 1.11 or later.\n\nThe following data types are supported: AbstractDict, AbstractVector, AbstractString, Integer, AbstractFloat, Bool, Dates.DateTime, Dates.Time, Dates.Date. Note that the integers and floats need to be convertible to Float64 and Int64 respectively. For other data types, pass the function to_toml that takes the data types and returns a value of a supported type.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.Parser","page":"TOML","title":"TOML.Parser","category":"type","text":"Parser()\n\nConstructor for a TOML Parser.  Note that in most cases one does not need to explicitly create a Parser but instead one directly uses TOML.parsefile or TOML.parse.  Using an explicit parser will however reuse some internal data structures which can be beneficial for performance if a larger number of small files are parsed.\n\n\n\n\n\n"},{"location":"stdlib/TOML.html#TOML.ParserError","page":"TOML","title":"TOML.ParserError","category":"type","text":"ParserError\n\nType that is returned from tryparse and tryparsefile when parsing fails. It contains (among others) the following fields:\n\npos, the position in the string when the error happened\ntable, the result that so far was successfully parsed\ntype, an error type, different for different types of errors\n\n\n\n\n\n"},{"location":"base/stacktraces.html#StackTraces","page":"StackTraces","title":"StackTraces","category":"section","text":"The following methods and types in Base.StackTraces are not exported and need to be called e.g. as StackTraces.lookup(ptr)."},{"location":"base/stacktraces.html#Base.StackTraces.StackFrame","page":"StackTraces","title":"Base.StackTraces.StackFrame","category":"type","text":"StackFrame\n\nStack information representing execution context, with the following fields:\n\nfunc::Symbol\nThe name of the function containing the execution context.\nlinfo::Union{Method, Core.MethodInstance, Core.CodeInstance, Core.CodeInfo, Nothing}\nThe Method, MethodInstance, CodeInstance, or CodeInfo containing the execution context (if it could be found),    or nothing (for example, if the inlining was a result of macro expansion).\nfile::Symbol\nThe path to the file containing the execution context.\nline::Int\nThe line number in the file containing the execution context.\nfrom_c::Bool\nTrue if the code is from C.\ninlined::Bool\nTrue if the code is from an inlined frame.\npointer::UInt64\nRepresentation of the pointer to the execution context as returned by backtrace.\n\n\n\n\n\n"},{"location":"base/stacktraces.html#Base.StackTraces.StackTrace","page":"StackTraces","title":"Base.StackTraces.StackTrace","category":"type","text":"StackTrace\n\nAn alias for Vector{StackFrame} provided for convenience; returned by calls to stacktrace.\n\n\n\n\n\n"},{"location":"base/stacktraces.html#Base.StackTraces.stacktrace","page":"StackTraces","title":"Base.StackTraces.stacktrace","category":"function","text":"stacktrace([trace::Vector{Ptr{Cvoid}},] [c_funcs::Bool=false])::StackTrace\n\nReturn a stack trace in the form of a vector of StackFrames. (By default stacktrace doesn't return C functions, but this can be enabled.) When called without specifying a trace, stacktrace first calls backtrace.\n\n\n\n\n\n"},{"location":"base/stacktraces.html#Base.StackTraces.lookup","page":"StackTraces","title":"Base.StackTraces.lookup","category":"function","text":"lookup(pointer::Ptr{Cvoid})::Vector{StackFrame}\n\nGiven a pointer to an execution context (usually generated by a call to backtrace), looks up stack frame context information. Returns an array of frame information for all functions inlined at that point, innermost function first.\n\n\n\n\n\n"},{"location":"base/stacktraces.html#Base.StackTraces.remove_frames!","page":"StackTraces","title":"Base.StackTraces.remove_frames!","category":"function","text":"remove_frames!(stack::StackTrace, m::Module)\n\nReturn the StackTrace with all StackFrames from the provided Module removed.\n\n\n\n\n\nremove_frames!(stack::StackTrace, name::Symbol)\n\nTakes a StackTrace (a vector of StackFrames) and a function name (a Symbol) and removes the StackFrame specified by the function name from the StackTrace (also removing all frames above the specified function). Primarily used to remove StackTraces functions from the StackTrace prior to returning it.\n\n\n\n\n\n"},{"location":"devdocs/boundscheck.html#Bounds-checking","page":"Bounds checking","title":"Bounds checking","category":"section","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection."},{"location":"devdocs/boundscheck.html#Eliding-bounds-checks","page":"Bounds checking","title":"Eliding bounds checks","category":"section","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWith a custom array-like type MyArray having:\n\n@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A, i); A.data[to_index(i)])\n\nThen when getindex is inlined into sum, the call to checkbounds(A, i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack."},{"location":"devdocs/boundscheck.html#Caution!","page":"Bounds checking","title":"Caution!","category":"section","text":"It is easy to accidentally expose unsafe operations with @inbounds. You might be tempted to write the above example as\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWhich quietly assumes 1-based indexing and therefore exposes unsafe memory access when used with OffsetArrays:\n\njulia> using OffsetArrays\n\njulia> sum(OffsetArray([1, 2, 3], -10))\n9164911648 # inconsistent results or segfault\n\nWhile the original source of the error here is 1:length(A), the use of @inbounds increases the consequences from a bounds error to a less easily caught and debugged unsafe memory access. It is often difficult or impossible to prove that a method which uses @inbounds is safe, so one must weigh the benefits of performance improvements against the risk of segfaults and silent misbehavior, especially in public facing APIs."},{"location":"devdocs/boundscheck.html#Propagating-inbounds","page":"Bounds checking","title":"Propagating inbounds","category":"section","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).\n\nTo override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining."},{"location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","page":"Bounds checking","title":"The bounds checking call hierarchy","category":"section","text":"The overall hierarchy is:\n\ncheckbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension\n\nHere A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.\n\ncheckbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,\n\ncheckbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)\n\nso checkindex checks a single dimension. All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .\n\nIf you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.\n\nIf you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array. If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.\n\nNote this hierarchy has been designed to reduce the likelihood of method ambiguities. We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument)."},{"location":"devdocs/boundscheck.html#Emit-bounds-checks","page":"Bounds checking","title":"Emit bounds checks","category":"section","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations."},{"location":"devdocs/builtins.html#lib-builtins","page":"Core.Builtins","title":"Core.Builtins","category":"section","text":"The following builtin functions are considered unstable, but provide the basic definitions for what defines the abilities and behaviors of a Julia program. They are typically accessed through a higher level generic API."},{"location":"devdocs/builtins.html#Raw-access-to-memory","page":"Core.Builtins","title":"Raw access to memory","category":"section","text":""},{"location":"devdocs/builtins.html#Managed-memory","page":"Core.Builtins","title":"Managed memory","category":"section","text":""},{"location":"devdocs/builtins.html#Module-bindings","page":"Core.Builtins","title":"Module bindings","category":"section","text":""},{"location":"devdocs/builtins.html#Other","page":"Core.Builtins","title":"Other","category":"section","text":""},{"location":"devdocs/builtins.html#Core.Intrinsics.pointerref","page":"Core.Builtins","title":"Core.Intrinsics.pointerref","category":"function","text":"Core.Intrinsics.pointerref(p::Ptr{T}, i::Int, align::Int)\n\nLoad a value of type T from the address of the ith element (1-indexed) starting at p. This is equivalent to the C expression p[i-1].\n\nThe alignment must be a power of two, or 0, indicating the default alignment for T. If p[i-1] is out of bounds, invalid, or is not aligned, the behavior is undefined. An alignment of 1 is always safe.\n\nSee also unsafe_load.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.pointerset","page":"Core.Builtins","title":"Core.Intrinsics.pointerset","category":"function","text":"Core.Intrinsics.pointerset(p::Ptr{T}, x::T, i::Int, align::Int)\n\nStore a value of type T to the address of the ith element (1-indexed) starting at p.  This is equivalent to the C expression p[i-1] = x.\n\nThe alignment must be a power of two, or 0, indicating the default alignment for T. If p[i-1] is out of bounds, invalid, or is not aligned, the behavior is undefined. An alignment of 1 is always safe.\n\nSee also unsafe_store!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.atomic_pointerref","page":"Core.Builtins","title":"Core.Intrinsics.atomic_pointerref","category":"function","text":"Core.Intrinsics.atomic_pointerref(pointer::Ptr{T}, order::Symbol) --> T\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\nSee unsafe_load.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.atomic_pointerset","page":"Core.Builtins","title":"Core.Intrinsics.atomic_pointerset","category":"function","text":"Core.Intrinsics.atomic_pointerset(pointer::Ptr{T}, new::T, order::Symbol) --> pointer\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\nSee unsafe_store!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.atomic_pointerswap","page":"Core.Builtins","title":"Core.Intrinsics.atomic_pointerswap","category":"function","text":"Core.Intrinsics.atomic_pointerswap(pointer::Ptr{T}, new::T, order::Symbol) --> old\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\nSee unsafe_swap!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.atomic_pointermodify","page":"Core.Builtins","title":"Core.Intrinsics.atomic_pointermodify","category":"function","text":"Core.Intrinsics.atomic_pointermodify(pointer::Ptr{T}, function::(old::T,arg::S)->T, arg::S, order::Symbol) --> old\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\nSee unsafe_modify!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics.atomic_pointerreplace","page":"Core.Builtins","title":"Core.Intrinsics.atomic_pointerreplace","category":"function","text":"Core.Intrinsics.atomic_pointerreplace(pointer::Ptr{T}, expected::Any, new::T, success_order::Symbol, failure_order::Symbol) --> (old, cmp)\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\nSee unsafe_replace!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memorynew","page":"Core.Builtins","title":"Core.memorynew","category":"function","text":"Core.memorynew(::Type{T} where T <: GenericMemory, n::Int)\n\nConstruct an uninitialized GenericMemory of length n.\n\nSee also Memory, Memory{T}(undef, n).\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefnew","page":"Core.Builtins","title":"Core.memoryrefnew","category":"function","text":"Core.memoryrefnew(::GenericMemory)\nCore.memoryrefnew(::GenericMemoryRef, index::Int, [boundscheck::Bool])\n\nReturn a GenericMemoryRef for a GenericMemory. See memoryref.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefoffset","page":"Core.Builtins","title":"Core.memoryrefoffset","category":"function","text":"Core..memoryrefoffset(::GenericMemoryRef)\n\nReturn the offset index that was used to construct the MemoryRef. See memoryref.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefget","page":"Core.Builtins","title":"Core.memoryrefget","category":"function","text":"Core.memoryrefget(::GenericMemoryRef, ordering::Symbol, boundscheck::Bool)\n\nReturn the value stored at the MemoryRef, throwing a BoundsError if the Memory is empty. See ref[]. The memory ordering specified must be compatible with the isatomic parameter.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefset!","page":"Core.Builtins","title":"Core.memoryrefset!","category":"function","text":"Core.memoryrefset!(::GenericMemoryRef, value, ordering::Symbol, boundscheck::Bool)\n\nStore the value to the MemoryRef, throwing a BoundsError if the Memory is empty. See ref[] = value. The memory ordering specified must be compatible with the isatomic parameter.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryref_isassigned","page":"Core.Builtins","title":"Core.memoryref_isassigned","category":"function","text":"Core.memoryref_isassigned(::GenericMemoryRef, ordering::Symbol, boundscheck::Bool)\n\nReturn whether there is a value stored at the MemoryRef, returning false if the Memory is empty. See isassigned(::Base.RefValue), Core.memoryrefget. The memory ordering specified must be compatible with the isatomic parameter.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefswap!","page":"Core.Builtins","title":"Core.memoryrefswap!","category":"function","text":"Core.memoryrefswap!(::GenericMemoryRef, value, ordering::Symbol, boundscheck::Bool)\n\nAtomically perform the operations to simultaneously get and set a MemoryRef value.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also swapproperty! and Core.memoryrefset!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefmodify!","page":"Core.Builtins","title":"Core.memoryrefmodify!","category":"function","text":"Core.memoryrefmodify!(::GenericMemoryRef, op, value, ordering::Symbol, boundscheck::Bool)::Pair\n\nAtomically perform the operations to get and set a MemoryRef value after applying the function op.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also modifyproperty! and Core.memoryrefset!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefreplace!","page":"Core.Builtins","title":"Core.memoryrefreplace!","category":"function","text":"Core.memoryrefreplace!(::GenericMemoryRef, expected, desired,\n                       success_order::Symbol, fail_order::Symbol=success_order, boundscheck::Bool) -> (; old, success::Bool)\n\nAtomically perform the operations to get and conditionally set a MemoryRef value.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also replaceproperty! and Core.memoryrefset!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.memoryrefsetonce!","page":"Core.Builtins","title":"Core.memoryrefsetonce!","category":"function","text":"Core.memoryrefsetonce!(::GenericMemoryRef, value,\n                       success_order::Symbol, fail_order::Symbol=success_order, boundscheck::Bool) -> success::Bool\n\nAtomically perform the operations to set a MemoryRef to a given value, only if it was previously not set.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nSee also setpropertyonce! and Core.memoryrefset!.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.get_binding_type","page":"Core.Builtins","title":"Core.get_binding_type","category":"function","text":"Core.get_binding_type(module::Module, name::Symbol)\n\nRetrieve the declared type of the binding name from the module module.\n\ncompat: Julia 1.9\nThis function requires Julia 1.9 or later.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.IntrinsicFunction","page":"Core.Builtins","title":"Core.IntrinsicFunction","category":"type","text":"Core.IntrinsicFunction <: Core.Builtin <: Function\n\nThe Core.IntrinsicFunction function define some basic primitives for what defines the abilities and behaviors of a Julia program\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.Intrinsics","page":"Core.Builtins","title":"Core.Intrinsics","category":"module","text":"Core.Intrinsics\n\nThe Core.Intrinsics module holds the Core.IntrinsicFunction objects.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Core.IR","page":"Core.Builtins","title":"Core.IR","category":"module","text":"Core.IR\n\nThe Core.IR module exports the IR object model.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Base.quoted","page":"Core.Builtins","title":"Base.quoted","category":"function","text":"quoted(x)\n\nReturn x made safe for inserting as a constant into IR. Note that this does not make it safe for inserting into an AST, since eval will sometimes copy some types of AST object inside, and even may sometimes evaluate and interpolate any $ inside, depending on the context.\n\n\n\n\n\n"},{"location":"devdocs/builtins.html#Base.isa_ast_node","page":"Core.Builtins","title":"Base.isa_ast_node","category":"function","text":"isa_ast_node(x)\n\nReturn false if x is not interpreted specially by any of inference, lowering, or codegen as either an AST or IR special form.\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian","page":"Base.Cartesian","title":"Base.Cartesian","category":"section","text":"The (non-exported) Cartesian module provides macros that facilitate writing multidimensional algorithms. Most often you can write such algorithms with straightforward techniques; however, there are a few cases where Base.Cartesian is still useful or necessary."},{"location":"devdocs/cartesian.html#Principles-of-usage","page":"Base.Cartesian","title":"Principles of usage","category":"section","text":"A simple example of usage is:\n\n@nloops 3 i A begin\n    s += @nref 3 A i\nend\n\nwhich generates the following code:\n\nfor i_3 = axes(A, 3)\n    for i_2 = axes(A, 2)\n        for i_1 = axes(A, 1)\n            s += A[i_1, i_2, i_3]\n        end\n    end\nend\n\nIn general, Cartesian allows you to write generic code that contains repetitive elements, like the nested loops in this example. Other applications include repeated expressions (e.g., loop unwinding) or creating function calls with variable numbers of arguments without using the \"splat\" construct (i...)."},{"location":"devdocs/cartesian.html#Basic-syntax","page":"Base.Cartesian","title":"Basic syntax","category":"section","text":"The (basic) syntax of @nloops is as follows:\n\nThe first argument must be an integer (not a variable) specifying the number of loops.\nThe second argument is the symbol-prefix used for the iterator variable. Here we used i, and variables i_1, i_2, i_3 were generated.\nThe third argument specifies the range for each iterator variable. If you use a variable (symbol) here, it's taken as axes(A, dim). More flexibly, you can use the anonymous-function expression syntax described below.\nThe last argument is the body of the loop. Here, that's what appears between the begin...end.\n\nThere are some additional features of @nloops described in the reference section.\n\n@nref follows a similar pattern, generating A[i_1,i_2,i_3] from @nref 3 A i. The general practice is to read from left to right, which is why @nloops is @nloops 3 i A expr (as in for i_2 = axes(A, 2), where i_2 is to the left and the range is to the right) whereas @nref is @nref 3 A i (as in A[i_1,i_2,i_3], where the array comes first).\n\nIf you're developing code with Cartesian, you may find that debugging is easier when you examine the generated code, using @macroexpand:\n\njulia> @macroexpand @nref 2 A i\n:(A[i_1, i_2])"},{"location":"devdocs/cartesian.html#Supplying-the-number-of-expressions","page":"Base.Cartesian","title":"Supplying the number of expressions","category":"section","text":"The first argument to both of these macros is the number of expressions, which must be an integer. When you're writing a function that you intend to work in multiple dimensions, this may not be something you want to hard-code. The recommended approach is to use a @generated function. Here's an example:\n\n@generated function mysum(A::Array{T,N}) where {T,N}\n    quote\n        s = zero(T)\n        @nloops $N i A begin\n            s += @nref $N A i\n        end\n        s\n    end\nend\n\nNaturally, you can also prepare expressions or perform calculations before the quote block."},{"location":"devdocs/cartesian.html#Anonymous-function-expressions-as-macro-arguments","page":"Base.Cartesian","title":"Anonymous-function expressions as macro arguments","category":"section","text":"Perhaps the single most powerful feature in Cartesian is the ability to supply anonymous-function expressions that get evaluated at parsing time. Let's consider a simple example:\n\n@nexprs 2 j->(i_j = 1)\n\n@nexprs generates n expressions that follow a pattern. This code would generate the following statements:\n\ni_1 = 1\ni_2 = 1\n\nIn each generated statement, an \"isolated\" j (the variable of the anonymous function) gets replaced by values in the range 1:2. Generally speaking, Cartesian employs a LaTeX-like syntax. This allows you to do math on the index j. Here's an example computing the strides of an array:\n\ns_1 = 1\n@nexprs 3 j->(s_{j+1} = s_j * size(A, j))\n\nwould generate expressions\n\ns_1 = 1\ns_2 = s_1 * size(A, 1)\ns_3 = s_2 * size(A, 2)\ns_4 = s_3 * size(A, 3)\n\nAnonymous-function expressions have many uses in practice."},{"location":"devdocs/cartesian.html#dev-cartesian-reference","page":"Base.Cartesian","title":"Macro reference","category":"section","text":""},{"location":"devdocs/cartesian.html#Base.Cartesian.@nloops","page":"Base.Cartesian","title":"Base.Cartesian.@nloops","category":"macro","text":"@nloops N itersym rangeexpr bodyexpr\n@nloops N itersym rangeexpr preexpr bodyexpr\n@nloops N itersym rangeexpr preexpr postexpr bodyexpr\n\nGenerate N nested loops, using itersym as the prefix for the iteration variables. rangeexpr may be an anonymous-function expression, or a simple symbol var in which case the range is axes(var, d) for dimension d.\n\nOptionally, you can provide \"pre\" and \"post\" expressions. These get executed first and last, respectively, in the body of each loop. For example:\n\n@nloops 2 i A d -> j_d = min(i_d, 5) begin\n    s += @nref 2 A j\nend\n\nwould generate:\n\nfor i_2 = axes(A, 2)\n    j_2 = min(i_2, 5)\n    for i_1 = axes(A, 1)\n        j_1 = min(i_1, 5)\n        s += A[j_1, j_2]\n    end\nend\n\nIf you want just a post-expression, supply nothing for the pre-expression. Using parentheses and semicolons, you can supply multi-statement expressions.\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nref","page":"Base.Cartesian","title":"Base.Cartesian.@nref","category":"macro","text":"@nref N A indexexpr\n\nGenerate expressions like A[i_1, i_2, ...]. indexexpr can either be an iteration-symbol prefix, or an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nref 3 A i\n:(A[i_1, i_2, i_3])\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nextract","page":"Base.Cartesian","title":"Base.Cartesian.@nextract","category":"macro","text":"@nextract N esym isym\n\nGenerate N variables esym_1, esym_2, ..., esym_N to extract values from isym. isym can be either a Symbol or anonymous-function expression.\n\n@nextract 2 x y would generate\n\nx_1 = y[1]\nx_2 = y[2]\n\nwhile @nextract 3 x d->y[2d-1] yields\n\nx_1 = y[1]\nx_2 = y[3]\nx_3 = y[5]\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nexprs","page":"Base.Cartesian","title":"Base.Cartesian.@nexprs","category":"macro","text":"@nexprs N expr\n\nGenerate N expressions. expr should be an anonymous-function expression.\n\nExamples\n\njulia> @macroexpand Base.Cartesian.@nexprs 4 i -> y[i] = A[i+j]\nquote\n    y[1] = A[1 + j]\n    y[2] = A[2 + j]\n    y[3] = A[3 + j]\n    y[4] = A[4 + j]\nend\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ncall","page":"Base.Cartesian","title":"Base.Cartesian.@ncall","category":"macro","text":"@ncall N f sym...\n\nGenerate a function call expression. sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nFor example, @ncall 3 func a generates\n\nfunc(a_1, a_2, a_3)\n\nwhile @ncall 2 func a b i->c[i] yields\n\nfunc(a, b, c[1], c[2])\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ncallkw","page":"Base.Cartesian","title":"Base.Cartesian.@ncallkw","category":"macro","text":"@ncallkw N f kw sym...\n\nGenerate a function call expression with keyword arguments kw.... As in the case of @ncall, sym represents any number of function arguments, the last of which may be an anonymous-function expression and is expanded into N arguments.\n\nExamples\n\njulia> using Base.Cartesian\n\njulia> f(x...; a, b = 1, c = 2, d = 3) = +(x..., a, b, c, d);\n\njulia> x_1, x_2 = (-1, -2); b = 0; kw = (c = 0, d = 0);\n\njulia> @ncallkw 2 f (; a = 0, b, kw...) x\n-3\n\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@ntuple","page":"Base.Cartesian","title":"Base.Cartesian.@ntuple","category":"macro","text":"@ntuple N expr\n\nGenerates an N-tuple. @ntuple 2 i would generate (i_1, i_2), and @ntuple 2 k->k+1 would generate (2,3).\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nall","page":"Base.Cartesian","title":"Base.Cartesian.@nall","category":"macro","text":"@nall N expr\n\nCheck whether all of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nall 3 d->(i_d > 1) would generate the expression (i_1 > 1 && i_2 > 1 && i_3 > 1). This can be convenient for bounds-checking.\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nany","page":"Base.Cartesian","title":"Base.Cartesian.@nany","category":"macro","text":"@nany N expr\n\nCheck whether any of the expressions generated by the anonymous-function expression expr evaluate to true.\n\n@nany 3 d->(i_d > 1) would generate the expression (i_1 > 1 || i_2 > 1 || i_3 > 1).\n\n\n\n\n\n"},{"location":"devdocs/cartesian.html#Base.Cartesian.@nif","page":"Base.Cartesian","title":"Base.Cartesian.@nif","category":"macro","text":"@nif N conditionexpr expr\n@nif N conditionexpr expr elseexpr\n\nGenerates a sequence of if ... elseif ... else ... end statements. For example:\n\n@nif 3 d->(i_d >= size(A,d)) d->(error(\"Dimension \", d, \" too big\")) d->println(\"All OK\")\n\nwould generate:\n\nif i_1 > size(A, 1)\n    error(\"Dimension \", 1, \" too big\")\nelseif i_2 > size(A, 2)\n    error(\"Dimension \", 2, \" too big\")\nelse\n    println(\"All OK\")\nend\n\n\n\n\n\n"},{"location":"devdocs/object.html#Memory-layout-of-Julia-Objects","page":"Memory layout of Julia Objects","title":"Memory layout of Julia Objects","category":"section","text":""},{"location":"devdocs/object.html#Object-layout-(jl_value_t)","page":"Memory layout of Julia Objects","title":"Object layout (jl_value_t)","category":"section","text":"The jl_value_t struct is the name for a block of memory owned by the Julia Garbage Collector, representing the data associated with a Julia object in memory. Absent any type information, it is simply an opaque pointer:\n\ntypedef struct jl_value_t* jl_pvalue_t;\n\nEach jl_value_t struct is contained in a jl_typetag_t struct that contains metadata information about the Julia object, such as its type and garbage collector (gc) reachability:\n\ntypedef struct {\n    opaque metadata;\n    jl_value_t value;\n} jl_typetag_t;\n\nThe type of any Julia object is an instance of a leaf jl_datatype_t object. The jl_typeof() function can be used to query for it:\n\njl_value_t *jl_typeof(jl_value_t *v);\n\nThe layout of the object depends on its type. Reflection methods can be used to inspect that layout. A field can be accessed by calling one of the get-field methods:\n\njl_value_t *jl_get_nth_field_checked(jl_value_t *v, size_t i);\njl_value_t *jl_get_field(jl_value_t *o, char *fld);\n\nIf the field types are known, a priori, to be all pointers, the values can also be extracted directly as an array access:\n\njl_value_t *v = value->fieldptr[n];\n\nAs an example, a \"boxed\" uint16_t is stored as follows:\n\nstruct {\n    opaque metadata;\n    struct {\n        uint16_t data;        // -- 2 bytes\n    } jl_value_t;\n};\n\nThis object is created by jl_box_uint16(). Note that the jl_value_t pointer references the data portion, not the metadata at the top of the struct.\n\nA value may be stored \"unboxed\" in many circumstances (just the data, without the metadata, and possibly not even stored but just kept in registers), so it is unsafe to assume that the address of a box is a unique identifier. The \"egal\" test (corresponding to the === function in Julia), should instead be used to compare two unknown objects for equivalence:\n\nint jl_egal(jl_value_t *a, jl_value_t *b);\n\nThis optimization should be relatively transparent to the API, since the object will be \"boxed\" on-demand, whenever a jl_value_t pointer is needed.\n\nNote that modification of a jl_value_t pointer in memory is permitted only if the object is mutable. Otherwise, modification of the value may corrupt the program and the result will be undefined. The mutability property of a value can be queried for with:\n\nint jl_is_mutable(jl_value_t *v);\n\nIf the object being stored is a jl_value_t, the Julia garbage collector must be notified also:\n\nvoid jl_gc_wb(jl_value_t *parent, jl_value_t *ptr);\n\nHowever, the Embedding Julia section of the manual is also required reading at this point, for covering other details of boxing and unboxing various types, and understanding the gc interactions.\n\nMirror structs for some of the built-in types are defined in julia.h. The corresponding global jl_datatype_t objects are created by jl_init_types in jltypes.c."},{"location":"devdocs/object.html#Garbage-collector-mark-bits","page":"Memory layout of Julia Objects","title":"Garbage collector mark bits","category":"section","text":"The garbage collector uses several bits from the metadata portion of the jl_typetag_t to track each object in the system. Further details about this algorithm can be found in the comments of the garbage collector implementation in gc-stock.c."},{"location":"devdocs/object.html#Object-allocation","page":"Memory layout of Julia Objects","title":"Object allocation","category":"section","text":"Most new objects are allocated by jl_new_structv():\n\njl_value_t *jl_new_struct(jl_datatype_t *type, ...);\njl_value_t *jl_new_structv(jl_datatype_t *type, jl_value_t **args, uint32_t na);\n\nAlthough, isbits objects can be also constructed directly from memory:\n\njl_value_t *jl_new_bits(jl_value_t *bt, void *data)\n\nAnd some objects have special constructors that must be used instead of the above functions:\n\nTypes:\n\njl_datatype_t *jl_apply_type(jl_datatype_t *tc, jl_tuple_t *params);\njl_datatype_t *jl_apply_array_type(jl_datatype_t *type, size_t dim);\n\nWhile these are the most commonly used options, there are more low-level constructors too, which you can find declared in julia.h. These are used in jl_init_types() to create the initial types needed to bootstrap the creation of the Julia system image.\n\nTuples:\n\njl_tuple_t *jl_tuple(size_t n, ...);\njl_tuple_t *jl_tuplev(size_t n, jl_value_t **v);\njl_tuple_t *jl_alloc_tuple(size_t n);\n\nThe representation of tuples is highly unique in the Julia object representation ecosystem. In some cases, a Base.tuple() object may be an array of pointers to the objects contained by the tuple equivalent to:\n\ntypedef struct {\n    size_t length;\n    jl_value_t *data[length];\n} jl_tuple_t;\n\nHowever, in other cases, the tuple may be converted to an anonymous isbits type and stored unboxed, or it may not stored at all (if it is not being used in a generic context as a jl_value_t*).\n\nSymbols:\n\njl_sym_t *jl_symbol(const char *str);\n\nFunctions and MethodInstance:\n\njl_value_t *jl_new_generic_function(jl_sym_t *name);\njl_method_instance_t *jl_new_method_instance(jl_value_t *ast, jl_tuple_t *sparams);\n\nArrays:\n\njl_array_t *jl_new_array(jl_value_t *atype, jl_tuple_t *dims);\njl_array_t *jl_alloc_array_1d(jl_value_t *atype, size_t nr);\njl_array_t *jl_alloc_array_nd(jl_value_t *atype, size_t *dims, size_t ndims);\n\nNote that many of these have alternative allocation functions for various special-purposes. The list here reflects the more common usages, but a more complete list can be found by reading the julia.h header file.\n\nInternal to Julia, storage is typically allocated by newstruct() (or newobj() for the special types):\n\njl_value_t *newstruct(jl_value_t *type);\njl_value_t *newobj(jl_value_t *type, size_t nfields);\n\nAnd at the lowest level, memory is getting allocated by a call to the garbage collector (in gc-stock.c), then tagged with its type:\n\njl_value_t *jl_gc_allocobj(size_t nbytes);\nvoid jl_set_typeof(jl_value_t *v, jl_datatype_t *type);\n\nnote: Out of date Warning\nThe documentation and usage for the function jl_gc_allocobj may be out of date\n\nNote that all objects are allocated in multiples of 4 bytes and aligned to the platform pointer size. Memory is allocated from a pool for smaller objects, or directly with malloc() for large objects.\n\nsidebar: Singleton Types\nSingleton types have only one instance and no data fields. Singleton instances have a size of 0 bytes, and consist only of their metadata. e.g. nothing::Nothing.See Singleton Types and Nothingness and missing values"},{"location":"manual/mathematical-operations.html#Mathematical-Operations-and-Elementary-Functions","page":"Mathematical Operations and Elementary Functions","title":"Mathematical Operations and Elementary Functions","category":"section","text":"Julia provides a complete collection of basic arithmetic and bitwise operators across all of its numeric primitive types, as well as providing portable, efficient implementations of a comprehensive collection of standard mathematical functions."},{"location":"manual/mathematical-operations.html#Arithmetic-Operators","page":"Mathematical Operations and Elementary Functions","title":"Arithmetic Operators","category":"section","text":"The following arithmetic operators are supported on all primitive numeric types:\n\nExpression Name Description\n+x unary plus the identity operation\n-x unary minus maps values to their additive inverses\nx + y binary plus performs addition\nx - y binary minus performs subtraction\nx * y times performs multiplication\nx / y divide performs division\nx ÷ y integer divide x / y, truncated to an integer\nx \\ y inverse divide equivalent to y / x\nx ^ y power raises x to the yth power\nx % y remainder equivalent to rem(x, y)\n\nA numeric literal placed directly before an identifier or parentheses, e.g. 2x or 2(x + y), is treated as a multiplication, except with higher precedence than other binary operations. See Numeric Literal Coefficients for details.\n\nJulia's promotion system makes arithmetic operations on mixtures of argument types \"just work\" naturally and automatically. See Conversion and Promotion for details of the promotion system.\n\nThe ÷ sign can be conveniently typed by writing \\div<tab> to the REPL or Julia IDE. See the manual section on Unicode input for more information.\n\nHere are some simple examples using arithmetic operators:\n\njulia> 1 + 2 + 3\n6\n\njulia> 1 - 2\n-1\n\njulia> 3*2/12\n0.5\n\n(By convention, we tend to space operators more tightly if they get applied before other nearby operators. For instance, we would generally write -x + 2 to reflect that first x gets negated, and then 2 is added to that result.)"},{"location":"manual/mathematical-operations.html#Boolean-Operators","page":"Mathematical Operations and Elementary Functions","title":"Boolean Operators","category":"section","text":"The following Boolean operators are supported on Bool types:\n\nExpression Name\n!x negation\nx && y short-circuiting and\nx || y short-circuiting or\n\nNegation changes true to false and vice versa. The short-circuiting operations are explained on the linked page."},{"location":"manual/mathematical-operations.html#Arithmetic-operations-with-Bool-values","page":"Mathematical Operations and Elementary Functions","title":"Arithmetic operations with Bool values","category":"section","text":"Note that Bool is an integer type, such that false is numerically equal to 0 and true is numerically equal to 1. All the usual promotion rules and numeric operators are also defined on it, with a special behavior of arithmetic (non-Boolean) operations when all the arguments are Bool: in those cases, the arguments are promoted to Int instead of keeping their type. Compare e.g. the following equivalent operations with Bool and with a different numeric type (UInt8):\n\njulia> true - true\n0\n\njulia> 0x01 - 0x01\n0x00\n\nAlso, when used in multiplication, false acts as a strong zero:\n\njulia> NaN * false\n0.0\n\njulia> false * Inf\n0.0\n\nThis is useful for preventing the propagation of NaN values in quantities that are known to be zero. See Knuth (1992) for motivation."},{"location":"manual/mathematical-operations.html#Bitwise-Operators","page":"Mathematical Operations and Elementary Functions","title":"Bitwise Operators","category":"section","text":"The following bitwise operators are supported on all primitive integer types:\n\nExpression Name\n~x bitwise not\nx & y bitwise and\nx | y bitwise or\nx ⊻ y bitwise xor (exclusive or)\nx ⊼ y bitwise nand (not and)\nx ⊽ y bitwise nor (not or)\nx >>> y logical shift right\nx >> y arithmetic shift right\nx << y logical/arithmetic shift left\n\nHere are some examples with bitwise operators:\n\njulia> ~123\n-124\n\njulia> 123 & 234\n106\n\njulia> 123 | 234\n251\n\njulia> 123 ⊻ 234\n145\n\njulia> xor(123, 234)\n145\n\njulia> nand(123, 123)\n-124\n\njulia> 123 ⊼ 123\n-124\n\njulia> nor(123, 124)\n-128\n\njulia> 123 ⊽ 124\n-128\n\njulia> ~UInt32(123)\n0xffffff84\n\njulia> ~UInt8(123)\n0x84"},{"location":"manual/mathematical-operations.html#Updating-operators","page":"Mathematical Operations and Elementary Functions","title":"Updating operators","category":"section","text":"Every binary arithmetic and bitwise operator also has an updating version that assigns the result of the operation back into its left operand. The updating version of the binary operator is formed by placing a = immediately after the operator. For example, writing x += 3 is equivalent to writing x = x + 3:\n\njulia> x = 1\n1\n\njulia> x += 3\n4\n\njulia> x\n4\n\nThe updating versions of all the binary arithmetic and bitwise operators are:\n\n+=  -=  *=  /=  \\=  ÷=  %=  ^=  &=  |=  ⊻=  >>>=  >>=  <<=\n\nnote: Note\nAn updating operator rebinds the variable on the left-hand side. As a result, the type of the variable may change.julia> x = 0x01; typeof(x)\nUInt8\n\njulia> x *= 2 # Same as x = x * 2\n2\n\njulia> typeof(x)\nInt64"},{"location":"manual/mathematical-operations.html#man-dot-operators","page":"Mathematical Operations and Elementary Functions","title":"Vectorized \"dot\" operators","category":"section","text":"For every binary operation like ^, there is a corresponding \"dot\" operation .^ that is automatically defined to perform ^ element-by-element on arrays. For example, [1, 2, 3] ^ 3 is not defined, since there is no standard mathematical meaning to \"cubing\" a (non-square) array, but [1, 2, 3] .^ 3 is defined as computing the elementwise (or \"vectorized\") result [1^3, 2^3, 3^3]. Similarly for unary operators like ! or √, there is a corresponding .√ that applies the operator elementwise.\n\njulia> [1, 2, 3] .^ 3\n3-element Vector{Int64}:\n  1\n  8\n 27\n\nMore specifically, a .^ b is parsed as the \"dot\" call (^).(a,b), which performs a broadcast operation: it can combine arrays and scalars, arrays of the same size (performing the operation elementwise), and even arrays of different shapes (e.g. combining row and column vectors to produce a matrix). Moreover, like all vectorized \"dot calls,\" these \"dot operators\" are fusing. For example, if you compute 2 .* A.^2 .+ sin.(A) (or equivalently @. 2A^2 + sin(A), using the @. macro) for an array A, it performs a single loop over A, computing 2a^2 + sin(a) for each element a of A. In particular, nested dot calls like f.(g.(x)) are fused, and \"adjacent\" binary operators like x .+ 3 .* x.^2 are equivalent to nested dot calls (+).(x, (*).(3, (^).(x, 2))).\n\nFurthermore, \"dotted\" updating operators like a .+= b (or @. a += b) are parsed as a .= a .+ b, where .= is a fused in-place assignment operation (see the dot syntax documentation).\n\nNote the dot syntax is also applicable to user-defined operators. For example, if you define ⊗(A, B) = kron(A, B) to give a convenient infix syntax A ⊗ B for Kronecker products (kron), then [A, B] .⊗ [C, D] will compute [A⊗C, B⊗D] with no additional coding.\n\nCombining dot operators with numeric literals can be ambiguous. For example, it is not clear whether 1.+x means 1. + x or 1 .+ x. Therefore this syntax is disallowed, and spaces must be used around the operator in such cases."},{"location":"manual/mathematical-operations.html#Numeric-Comparisons","page":"Mathematical Operations and Elementary Functions","title":"Numeric Comparisons","category":"section","text":"Standard comparison operations are defined for all the primitive numeric types:\n\nOperator Name\n== equality\n!=, ≠ inequality\n< less than\n<=, ≤ less than or equal to\n> greater than\n>=, ≥ greater than or equal to\n\nHere are some simple examples:\n\njulia> 1 == 1\ntrue\n\njulia> 1 == 2\nfalse\n\njulia> 1 != 2\ntrue\n\njulia> 1 == 1.0\ntrue\n\njulia> 1 < 2\ntrue\n\njulia> 1.0 > 3\nfalse\n\njulia> 1 >= 1.0\ntrue\n\njulia> -1 <= 1\ntrue\n\njulia> -1 <= -1\ntrue\n\njulia> -1 <= -2\nfalse\n\njulia> 3 < -0.5\nfalse\n\nIntegers are compared in the standard manner – by comparison of bits. Floating-point numbers are compared according to the IEEE 754 standard:\n\nFinite numbers are ordered in the usual manner.\nPositive zero is equal but not greater than negative zero.\nInf is equal to itself and greater than everything else except NaN.\n-Inf is equal to itself and less than everything else except NaN.\nNaN is not equal to, not less than, and not greater than anything, including itself.\n\nThe last point is potentially surprising and thus worth noting:\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nand can cause headaches when working with arrays:\n\njulia> [1 NaN] == [1 NaN]\nfalse\n\nJulia provides additional functions to test numbers for special values, which can be useful in situations like hash key comparisons:\n\nFunction Tests if\nisequal(x, y) x and y are identical\nisfinite(x) x is a finite number\nisinf(x) x is infinite\nisnan(x) x is not a number\n\nisequal considers NaNs equal to each other:\n\njulia> isequal(NaN, NaN)\ntrue\n\njulia> isequal([1 NaN], [1 NaN])\ntrue\n\njulia> isequal(NaN, NaN32)\ntrue\n\nisequal can also be used to distinguish signed zeros:\n\njulia> -0.0 == 0.0\ntrue\n\njulia> isequal(-0.0, 0.0)\nfalse\n\nMixed-type comparisons between signed integers, unsigned integers, and floats can be tricky. A great deal of care has been taken to ensure that Julia does them correctly.\n\nFor other types, isequal defaults to calling ==, so if you want to define equality for your own types then you only need to add a == method. If you define your own equality function, you should probably define a corresponding hash method to ensure that isequal(x,y) implies hash(x) == hash(y)."},{"location":"manual/mathematical-operations.html#Chaining-comparisons","page":"Mathematical Operations and Elementary Functions","title":"Chaining comparisons","category":"section","text":"Unlike most languages, with the notable exception of Python, comparisons can be arbitrarily chained:\n\njulia> 1 < 2 <= 2 < 3 == 3 > 2 >= 1 == 1 < 3 != 5\ntrue\n\nChaining comparisons is often quite convenient in numerical code. Chained comparisons use the && operator for scalar comparisons, and the & operator for elementwise comparisons, which allows them to work on arrays. For example, 0 .< A .< 1 gives a boolean array whose entries are true where the corresponding elements of A are between 0 and 1.\n\nNote the evaluation behavior of chained comparisons:\n\njulia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> v(1) < v(2) <= v(3)\n2\n1\n3\ntrue\n\njulia> v(1) > v(2) <= v(3)\n2\n1\nfalse\n\nThe middle expression is only evaluated once, rather than twice as it would be if the expression were written as v(1) < v(2) && v(2) <= v(3). However, the order of evaluations in a chained comparison is undefined. It is strongly recommended not to use expressions with side effects (such as printing) in chained comparisons. If side effects are required, the short-circuit && operator should be used explicitly (see Short-Circuit Evaluation)."},{"location":"manual/mathematical-operations.html#Elementary-Functions","page":"Mathematical Operations and Elementary Functions","title":"Elementary Functions","category":"section","text":"Julia provides a comprehensive collection of mathematical functions and operators. These mathematical operations are defined over as broad a class of numerical values as permit sensible definitions, including integers, floating-point numbers, rationals, and complex numbers, wherever such definitions make sense.\n\nMoreover, these functions (like any Julia function) can be applied in \"vectorized\" fashion to arrays and other collections with the dot syntax f.(A), e.g. sin.(A) will compute the sine of each element of an array A."},{"location":"manual/mathematical-operations.html#Operator-Precedence-and-Associativity","page":"Mathematical Operations and Elementary Functions","title":"Operator Precedence and Associativity","category":"section","text":"Julia applies the following order and associativity of operations, from highest precedence to lowest:\n\nCategory Operators Associativity\nSyntax . followed by :: Left\nExponentiation ^ Right\nUnary + - ! ~ ¬ √ ∛ ∜ ⋆ ± ∓ <: >: Right[1]\nBitshifts << >> >>> Left\nFractions // Left\nMultiplication * / % & \\ ÷ Left[2]\nAddition + - | ⊻ Left[2]\nSyntax : .. Left\nSyntax |> Left\nSyntax <| Right\nComparisons > < >= <= == === != !== <: Non-associative\nControl flow && followed by || followed by ? Right\nPair => Right\nAssignments = += -= *= /= //= \\= ^= ÷= %= |= &= ⊻= <<= >>= >>>= Right\n\n[1]: The unary operators + and - require explicit parentheses around their argument to disambiguate them from the operator ++, etc. Other compositions of unary operators are parsed with right-associativity, e. g., √√-a as √(√(-a)).\n\n[2]: The operators +, ++ and * are non-associative. a + b + c is parsed as +(a, b, c) not +(+(a, b), c). However, the fallback methods for +(a, b, c, d...) and *(a, b, c, d...) both default to left-associative evaluation.\n\nFor a complete list of every Julia operator's precedence, see the top of this file: src/julia-parser.scm. Note that some of the operators there are not defined in the Base module but may be given definitions by standard libraries, packages or user code.\n\nYou can also find the numerical precedence for any given operator via the built-in function Base.operator_precedence, where higher numbers take precedence:\n\njulia> Base.operator_precedence(:+), Base.operator_precedence(:*), Base.operator_precedence(:.)\n(11, 12, 17)\n\njulia> Base.operator_precedence(:sin), Base.operator_precedence(:+=), Base.operator_precedence(:(=))  # (Note the necessary parens on `:(=)`)\n(0, 1, 1)\n\nA symbol representing the operator associativity can also be found by calling the built-in function Base.operator_associativity:\n\njulia> Base.operator_associativity(:-), Base.operator_associativity(:+), Base.operator_associativity(:^)\n(:left, :none, :right)\n\njulia> Base.operator_associativity(:⊗), Base.operator_associativity(:sin), Base.operator_associativity(:→)\n(:left, :none, :right)\n\nNote that symbols such as :sin return precedence 0. This value represents invalid operators and not operators of lowest precedence. Similarly, such operators are assigned associativity :none.\n\nNumeric literal coefficients, e.g. 2x, are treated as multiplications with higher precedence than any other binary operation, with the exception of ^ where they have higher precedence only as the exponent.\n\njulia> x = 3; 2x^2\n18\n\njulia> x = 3; 2^2x\n64\n\nJuxtaposition parses like a unary operator, which has the same natural asymmetry around exponents: -x^y and 2x^y parse as -(x^y) and 2(x^y) whereas x^-y and x^2y parse as x^(-y) and x^(2y)."},{"location":"manual/mathematical-operations.html#Numerical-Conversions","page":"Mathematical Operations and Elementary Functions","title":"Numerical Conversions","category":"section","text":"Julia supports three forms of numerical conversion, which differ in their handling of inexact conversions.\n\nThe notation T(x) or convert(T, x) converts x to a value of type T.\nIf T is a floating-point type, the result is the nearest representable value, which could be positive or negative infinity.\nIf T is an integer type, an InexactError is raised if x is not representable by T.\nx % T converts an integer x to a value of integer type T congruent to x modulo 2^n, where n is the number of bits in T. In other words, the binary representation is truncated to fit.\nThe Rounding functions take a type T as an optional argument. For example, round(Int,x) is a shorthand for Int(round(x)).\n\nThe following examples show the different forms.\n\njulia> Int8(127)\n127\n\njulia> Int8(128)\nERROR: InexactError: trunc(Int8, 128)\nStacktrace:\n[...]\n\njulia> Int8(127.0)\n127\n\njulia> Int8(3.14)\nERROR: InexactError: Int8(3.14)\nStacktrace:\n[...]\n\njulia> Int8(128.0)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\njulia> 127 % Int8\n127\n\njulia> 128 % Int8\n-128\n\njulia> round(Int8,127.4)\n127\n\njulia> round(Int8,127.6)\nERROR: InexactError: Int8(128.0)\nStacktrace:\n[...]\n\nSee Conversion and Promotion for how to define your own conversions and promotions."},{"location":"manual/mathematical-operations.html#Rounding-functions","page":"Mathematical Operations and Elementary Functions","title":"Rounding functions","category":"section","text":"Function Description Return type\nround(x) round x to the nearest integer typeof(x)\nround(T, x) round x to the nearest integer T\nfloor(x) round x towards -Inf typeof(x)\nfloor(T, x) round x towards -Inf T\nceil(x) round x towards +Inf typeof(x)\nceil(T, x) round x towards +Inf T\ntrunc(x) round x towards zero typeof(x)\ntrunc(T, x) round x towards zero T"},{"location":"manual/mathematical-operations.html#Division-functions","page":"Mathematical Operations and Elementary Functions","title":"Division functions","category":"section","text":"Function Description\ndiv(x, y), x÷y truncated division; quotient rounded towards zero\nfld(x, y) floored division; quotient rounded towards -Inf\ncld(x, y) ceiling division; quotient rounded towards +Inf\nrem(x, y), x%y remainder; satisfies x == div(x, y)*y + rem(x, y); sign matches x\nmod(x, y) modulus; satisfies x == fld(x, y)*y + mod(x, y); sign matches y\nmod1(x, y) mod with offset 1; returns r∈(0, y] for y>0 or r∈[y, 0) for y<0, where mod(r, y) == mod(x, y)\nmod2pi(x) modulus with respect to 2pi;  0 <= mod2pi(x) < 2pi\ndivrem(x, y) returns (div(x, y),rem(x, y))\nfldmod(x, y) returns (fld(x, y), mod(x, y))\ngcd(x, y...) greatest positive common divisor of x, y,...\nlcm(x, y...) least positive common multiple of x, y,..."},{"location":"manual/mathematical-operations.html#Sign-and-absolute-value-functions","page":"Mathematical Operations and Elementary Functions","title":"Sign and absolute value functions","category":"section","text":"Function Description\nabs(x) a positive value with the magnitude of x\nabs2(x) the squared magnitude of x\nsign(x) indicates the sign of x, returning -1, 0, or +1\nsignbit(x) indicates whether the sign bit is on (true) or off (false)\ncopysign(x, y) a value with the magnitude of x and the sign of y\nflipsign(x, y) a value with the magnitude of x and the sign of x*y"},{"location":"manual/mathematical-operations.html#Powers,-logs-and-roots","page":"Mathematical Operations and Elementary Functions","title":"Powers, logs and roots","category":"section","text":"Function Description\nsqrt(x), √x square root of x\ncbrt(x), ∛x cube root of x\nfourthroot(x), ∜x fourth root of x\nhypot(x, y) hypotenuse of right-angled triangle with other sides of length x and y\nexp(x) natural exponential function at x\nexpm1(x) accurate exp(x) - 1 for x near zero\nldexp(x, n) x * 2^n computed efficiently for integer values of n\nlog(x) natural logarithm of x\nlog(b, x) base b logarithm of x\nlog2(x) base 2 logarithm of x\nlog10(x) base 10 logarithm of x\nlog1p(x) accurate log(1 + x) for x near zero\nexponent(x) binary exponent of x\nsignificand(x) binary significand (a.k.a. mantissa) of a floating-point number x\n\nFor an overview of why functions like hypot, expm1, and log1p are necessary and useful, see John D. Cook's excellent pair of blog posts on the subject: expm1, log1p, erfc, and hypot."},{"location":"manual/mathematical-operations.html#Trigonometric-and-hyperbolic-functions","page":"Mathematical Operations and Elementary Functions","title":"Trigonometric and hyperbolic functions","category":"section","text":"All the standard trigonometric and hyperbolic functions are also defined:\n\nsin    cos    tan    cot    sec    csc\nsinh   cosh   tanh   coth   sech   csch\nasin   acos   atan   acot   asec   acsc\nasinh  acosh  atanh  acoth  asech  acsch\nsinc   cosc\n\nThese are all single-argument functions, with atan also accepting two arguments corresponding to a traditional atan2 function.\n\nAdditionally, sinpi(x) and cospi(x) are provided for more accurate computations of sin(pi * x) and cos(pi * x) respectively.\n\nIn order to compute trigonometric functions with degrees instead of radians, suffix the function with d. For example, sind(x) computes the sine of x where x is specified in degrees. The complete list of trigonometric functions with degree variants is:\n\nsind   cosd   tand   cotd   secd   cscd\nasind  acosd  atand  acotd  asecd  acscd"},{"location":"manual/mathematical-operations.html#Special-functions","page":"Mathematical Operations and Elementary Functions","title":"Special functions","category":"section","text":"Many other special mathematical functions are provided by the package SpecialFunctions.jl."},{"location":"manual/constructors.html#man-constructors","page":"Constructors","title":"Constructors","category":"section","text":"Constructors [1] are functions that create new objects – specifically, instances of Composite Types. In Julia, type objects also serve as constructor functions: they create new instances of themselves when applied to an argument tuple as a function. This much was already mentioned briefly when composite types were introduced. For example:\n\njulia> struct Foo\n           bar\n           baz\n       end\n\njulia> foo = Foo(1, 2)\nFoo(1, 2)\n\njulia> foo.bar\n1\n\njulia> foo.baz\n2\n\nFor many types, forming new objects by binding their field values together is all that is ever needed to create instances. However, in some cases more functionality is required when creating composite objects. Sometimes invariants must be enforced, either by checking arguments or by transforming them. Recursive data structures, especially those that may be self-referential, often cannot be constructed cleanly without first being created in an incomplete state and then altered programmatically to be made whole, as a separate step from object creation. Sometimes, it's just convenient to be able to construct objects with fewer or different types of parameters than they have fields. Julia's system for object construction addresses all of these cases and more.\n\n[1]: Nomenclature: while the term \"constructor\" generally refers to the entire function which constructs objects of a type, it is common to abuse terminology slightly and refer to specific constructor methods as \"constructors\". In such situations, it is generally clear from the context that the term is used to mean \"constructor method\" rather than \"constructor function\", especially as it is often used in the sense of singling out a particular method of the constructor from all of the others."},{"location":"manual/constructors.html#man-outer-constructor-methods","page":"Constructors","title":"Outer Constructor Methods","category":"section","text":"A constructor is just like any other function in Julia in that its overall behavior is defined by the combined behavior of its methods. Accordingly, you can add functionality to a constructor by simply defining new methods. For example, let's say you want to add a constructor method for Foo objects that takes only one argument and uses the given value for both the bar and baz fields. This is simple:\n\njulia> Foo(x) = Foo(x,x)\nFoo\n\njulia> Foo(1)\nFoo(1, 1)\n\nYou could also add a zero-argument Foo constructor method that supplies default values for both of the bar and baz fields:\n\njulia> Foo() = Foo(0)\nFoo\n\njulia> Foo()\nFoo(0, 0)\n\nHere the zero-argument constructor method calls the single-argument constructor method, which in turn calls the automatically provided two-argument constructor method. For reasons that will become clear very shortly, additional constructor methods declared as normal methods like this are called outer constructor methods. Outer constructor methods can only ever create a new instance by calling another constructor method, such as the automatically provided default ones."},{"location":"manual/constructors.html#man-inner-constructor-methods","page":"Constructors","title":"Inner Constructor Methods","category":"section","text":"While outer constructor methods succeed in addressing the problem of providing additional convenience methods for constructing objects, they fail to address the other two use cases mentioned in the introduction of this chapter: enforcing invariants, and allowing construction of self-referential objects. For these problems, one needs inner constructor methods. An inner constructor method is like an outer constructor method, except for two differences:\n\nIt is declared inside the block of a type declaration, rather than outside of it like normal methods.\nIt has access to a special locally existent function called new that creates objects of the block's type.\n\nFor example, suppose one wants to declare a type that holds a pair of real numbers, subject to the constraint that the first number is not greater than the second one. One could declare it like this:\n\njulia> struct OrderedPair\n           x::Real\n           y::Real\n           OrderedPair(x,y) = x > y ? error(\"out of order\") : new(x,y)\n       end\n\nNow OrderedPair objects can only be constructed such that x <= y:\n\njulia> OrderedPair(1, 2)\nOrderedPair(1, 2)\n\njulia> OrderedPair(2,1)\nERROR: out of order\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] OrderedPair(::Int64, ::Int64) at ./none:4\n [3] top-level scope\n\nIf the type were declared mutable, you could reach in and directly change the field values to violate this invariant. Of course, messing around with an object's internals uninvited is bad practice. You (or someone else) can also provide additional outer constructor methods at any later point, but once a type is declared, there is no way to add more inner constructor methods. Since outer constructor methods can only create objects by calling other constructor methods, ultimately, some inner constructor must be called to create an object. This guarantees that all objects of the declared type must come into existence by a call to one of the inner constructor methods provided with the type, thereby giving some degree of enforcement of a type's invariants.\n\nIf any inner constructor method is defined, no default constructor method is provided: it is presumed that you have supplied yourself with all the inner constructors you need. The default constructor is equivalent to writing your own inner constructor method that takes all of the object's fields as parameters (constrained to be of the correct type, if the corresponding field has a type), and passes them to new, returning the resulting object:\n\njulia> struct Foo\n           bar\n           baz\n           Foo(bar,baz) = new(bar,baz)\n       end\n\n\nThis declaration has the same effect as the earlier definition of the Foo type without an explicit inner constructor method. The following two types are equivalent – one with a default constructor, the other with an explicit constructor:\n\njulia> struct T1\n           x::Int64\n       end\n\njulia> struct T2\n           x::Int64\n           T2(x) = new(x)\n       end\n\njulia> T1(1)\nT1(1)\n\njulia> T2(1)\nT2(1)\n\njulia> T1(1.0)\nT1(1)\n\njulia> T2(1.0)\nT2(1)\n\nIt is good practice to provide as few inner constructor methods as possible: only those taking all arguments explicitly and enforcing essential error checking and transformation. Additional convenience constructor methods, supplying default values or auxiliary transformations, should be provided as outer constructors that call the inner constructors to do the heavy lifting. This separation is typically quite natural."},{"location":"manual/constructors.html#Incomplete-Initialization","page":"Constructors","title":"Incomplete Initialization","category":"section","text":"The final problem which has still not been addressed is construction of self-referential objects, or more generally, recursive data structures. Since the fundamental difficulty may not be immediately obvious, let us briefly explain it. Consider the following recursive type declaration:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n       end\n\n\nThis type may appear innocuous enough, until one considers how to construct an instance of it. If a is an instance of SelfReferential, then a second instance can be created by the call:\n\njulia> b = SelfReferential(a)\n\nBut how does one construct the first instance when no instance exists to provide as a valid value for its obj field? The only solution is to allow creating an incompletely initialized instance of SelfReferential with an unassigned obj field, and using that incomplete instance as a valid value for the obj field of another instance, such as, for example, itself.\n\nTo allow for the creation of incompletely initialized objects, Julia allows the new function to be called with fewer than the number of fields that the type has, returning an object with the unspecified fields uninitialized. The inner constructor method can then use the incomplete object, finishing its initialization before returning it. Here, for example, is another attempt at defining the SelfReferential type, this time using a zero-argument inner constructor returning instances having obj fields pointing to themselves:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n           SelfReferential() = (x = new(); x.obj = x)\n       end\n\n\nWe can verify that this constructor works and constructs objects that are, in fact, self-referential:\n\njulia> x = SelfReferential();\n\njulia> x === x\ntrue\n\njulia> x === x.obj\ntrue\n\njulia> x === x.obj.obj\ntrue\n\nAlthough it is generally a good idea to return a fully initialized object from an inner constructor, it is possible to return incompletely initialized objects:\n\njulia> mutable struct Incomplete\n           data\n           Incomplete() = new()\n       end\n\njulia> z = Incomplete();\n\nWhile you are allowed to create objects with uninitialized fields, any access to an uninitialized reference is an immediate error:\n\njulia> z.data\nERROR: UndefRefError: access to undefined reference\n\nThis avoids the need to continually check for null values. However, not all object fields are references. Julia considers some types to be \"plain data\", meaning all of their data is self-contained and does not reference other objects. The plain data types consist of primitive types (e.g. Int) and immutable structs of other plain data types (see also: isbits, isbitstype). The initial contents of a plain data type is undefined:\n\njulia> struct HasPlain\n           n::Int\n           HasPlain() = new()\n       end\n\njulia> HasPlain()\nHasPlain(438103441441)\n\nArrays of plain data types exhibit the same behavior.\n\nYou can pass incomplete objects to other functions from inner constructors to delegate their completion:\n\njulia> mutable struct Lazy\n           data\n           Lazy(v) = complete_me(new(), v)\n       end\n\nAs with incomplete objects returned from constructors, if complete_me or any of its callees try to access the data field of the Lazy object before it has been initialized, an error will be thrown immediately."},{"location":"manual/constructors.html#Parametric-Constructors","page":"Constructors","title":"Parametric Constructors","category":"section","text":"Parametric types add a few wrinkles to the constructor story. Recall from Parametric Types that, by default, instances of parametric composite types can be constructed either with explicitly given type parameters or with type parameters implied by the types of the arguments given to the constructor. Here are some examples:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n       end\n\njulia> Point(1,2) ## implicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point(1.0,2.5) ## implicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point(1,2.5) ## implicit T ##\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, ::T) where T<:Real at none:2\n\njulia> Point{Int64}(1, 2) ## explicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point{Int64}(1.0,2.5) ## explicit T ##\nERROR: InexactError: Int64(2.5)\nStacktrace:\n[...]\n\njulia> Point{Float64}(1.0, 2.5) ## explicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point{Float64}(1,2) ## explicit T ##\nPoint{Float64}(1.0, 2.0)\n\nAs you can see, for constructor calls with explicit type parameters, the arguments are converted to the implied field types: Point{Int64}(1,2) works, but Point{Int64}(1.0,2.5) raises an InexactError when converting 2.5 to Int64. When the type is implied by the arguments to the constructor call, as in Point(1,2), then the types of the arguments must agree – otherwise the T cannot be determined – but any pair of real arguments with matching type may be given to the generic Point constructor.\n\nWhat's really going on here is that Point, Point{Float64} and Point{Int64} are all different constructor functions. In fact, Point{T} is a distinct constructor function for each type T. Without any explicitly provided inner constructors, the declaration of the composite type Point{T<:Real} automatically provides an inner constructor, Point{T}, for each possible type T<:Real, that behaves just like non-parametric default inner constructors do. It also provides a single general outer Point constructor that takes pairs of real arguments, which must be of the same type. This automatic provision of constructors is equivalent to the following explicit declaration:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n           Point{T}(x,y) where {T<:Real} = new(x,y)\n       end\n\njulia> Point(x::T, y::T) where {T<:Real} = Point{T}(x,y);\n\nNotice that each definition looks like the form of constructor call that it handles. The call Point{Int64}(1,2) will invoke the definition Point{T}(x,y) inside the struct block. The outer constructor declaration, on the other hand, defines a method for the general Point constructor which only applies to pairs of values of the same real type. This declaration makes constructor calls without explicit type parameters, like Point(1,2) and Point(1.0,2.5), work. Since the method declaration restricts the arguments to being of the same type, calls like Point(1,2.5), with arguments of different types, result in \"no method\" errors.\n\nSuppose we wanted to make the constructor call Point(1,2.5) work by \"promoting\" the integer value 1 to the floating-point value 1.0. The simplest way to achieve this is to define the following additional outer constructor method:\n\njulia> Point(x::Int64, y::Float64) = Point(convert(Float64,x),y);\n\nThis method uses the convert function to explicitly convert x to Float64 and then delegates construction to the general constructor for the case where both arguments are Float64. With this method definition what was previously a MethodError now successfully creates a point of type Point{Float64}:\n\njulia> p = Point(1,2.5)\nPoint{Float64}(1.0, 2.5)\n\njulia> typeof(p)\nPoint{Float64}\n\nHowever, other similar calls still don't work:\n\njulia> Point(1.5,2)\nERROR: MethodError: no method matching Point(::Float64, ::Int64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T<:Real\n   @ Main none:1\n  Point(!Matched::Int64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nFor a more general way to make all such calls work sensibly, see Conversion and Promotion. At the risk of spoiling the suspense, we can reveal here that all it takes is the following outer method definition to make all calls to the general Point constructor work as one would expect:\n\njulia> Point(x::Real, y::Real) = Point(promote(x,y)...);\n\nThe promote function converts all its arguments to a common type – in this case Float64. With this method definition, the Point constructor promotes its arguments the same way that numeric operators like + do, and works for all kinds of real numbers:\n\njulia> Point(1.5,2)\nPoint{Float64}(1.5, 2.0)\n\njulia> Point(1,1//2)\nPoint{Rational{Int64}}(1//1, 1//2)\n\njulia> Point(1.0,1//2)\nPoint{Float64}(1.0, 0.5)\n\nThus, while the implicit type parameter constructors provided by default in Julia are fairly strict, it is possible to make them behave in a more relaxed but sensible manner quite easily. Moreover, since constructors can leverage all of the power of the type system, methods, and multiple dispatch, defining sophisticated behavior is typically quite simple."},{"location":"manual/constructors.html#Case-Study:-Rational","page":"Constructors","title":"Case Study: Rational","category":"section","text":"Perhaps the best way to tie all these pieces together is to present a real world example of a parametric composite type and its constructor methods. To that end, we implement our own rational number type OurRational, similar to Julia's built-in Rational type, defined in rational.jl:\n\njulia> struct OurRational{T<:Integer} <: Real\n           num::T\n           den::T\n           function OurRational{T}(num::T, den::T) where T<:Integer\n               if num == 0 && den == 0\n                    error(\"invalid rational: 0//0\")\n               end\n               num = flipsign(num, den)\n               den = flipsign(den, den)\n               g = gcd(num, den)\n               num = div(num, g)\n               den = div(den, g)\n               new(num, den)\n           end\n       end\n\njulia> OurRational(n::T, d::T) where {T<:Integer} = OurRational{T}(n,d)\nOurRational\n\njulia> OurRational(n::Integer, d::Integer) = OurRational(promote(n,d)...)\nOurRational\n\njulia> OurRational(n::Integer) = OurRational(n,one(n))\nOurRational\n\njulia> ⊘(n::Integer, d::Integer) = OurRational(n,d)\n⊘ (generic function with 1 method)\n\njulia> ⊘(x::OurRational, y::Integer) = x.num ⊘ (x.den*y)\n⊘ (generic function with 2 methods)\n\njulia> ⊘(x::Integer, y::OurRational) = (x*y.den) ⊘ y.num\n⊘ (generic function with 3 methods)\n\njulia> ⊘(x::Complex, y::Real) = complex(real(x) ⊘ y, imag(x) ⊘ y)\n⊘ (generic function with 4 methods)\n\njulia> ⊘(x::Real, y::Complex) = (x*y') ⊘ real(y*y')\n⊘ (generic function with 5 methods)\n\njulia> function ⊘(x::Complex, y::Complex)\n           xy = x*y'\n           yy = real(y*y')\n           complex(real(xy) ⊘ yy, imag(xy) ⊘ yy)\n       end\n⊘ (generic function with 6 methods)\n\nThe first line – struct OurRational{T<:Integer} <: Real – declares that OurRational takes one type parameter of an integer type, and is itself a real type. The field declarations num::T and den::T indicate that the data held in a OurRational{T} object are a pair of integers of type T, one representing the rational value's numerator and the other representing its denominator.\n\nNow things get interesting. OurRational has a single inner constructor method which checks that num and den aren't both zero and ensures that every rational is constructed in \"lowest terms\" with a non-negative denominator. This is accomplished by first flipping the signs of numerator and denominator if the denominator is negative. Then, both are divided by their greatest common divisor (gcd always returns a non-negative number, regardless of the sign of its arguments). Because this is the only inner constructor for OurRational, we can be certain that OurRational objects are always constructed in this normalized form.\n\nOurRational also provides several outer constructor methods for convenience. The first is the \"standard\" general constructor that infers the type parameter T from the type of the numerator and denominator when they have the same type. The second applies when the given numerator and denominator values have different types: it promotes them to a common type and then delegates construction to the outer constructor for arguments of matching type. The third outer constructor turns integer values into rationals by supplying a value of 1 as the denominator.\n\nFollowing the outer constructor definitions, we defined a number of methods for the ⊘ operator, which provides a syntax for writing rationals (e.g. 1 ⊘ 2). Julia's Rational type uses the // operator for this purpose. Before these definitions, ⊘ is a completely undefined operator with only syntax and no meaning. Afterwards, it behaves just as described in Rational Numbers – its entire behavior is defined in these few lines. Note that the infix use of ⊘ works because Julia has a set of symbols that are recognized to be infix operators. The first and most basic definition just makes a ⊘ b construct a OurRational by applying the OurRational constructor to a and b when they are integers. When one of the operands of ⊘ is already a rational number, we construct a new rational for the resulting ratio slightly differently; this behavior is actually identical to division of a rational with an integer. Finally, applying ⊘ to complex integral values creates an instance of Complex{<:OurRational} – a complex number whose real and imaginary parts are rationals:\n\njulia> z = (1 + 2im) ⊘ (1 - 2im);\n\njulia> typeof(z)\nComplex{OurRational{Int64}}\n\njulia> typeof(z) <: Complex{<:OurRational}\ntrue\n\nThus, although the ⊘ operator usually returns an instance of OurRational, if either of its arguments are complex integers, it will return an instance of Complex{<:OurRational} instead. The interested reader should consider perusing the rest of rational.jl: it is short, self-contained, and implements an entire basic Julia type."},{"location":"manual/constructors.html#Outer-only-constructors","page":"Constructors","title":"Outer-only constructors","category":"section","text":"As we have seen, a typical parametric type has inner constructors that are called when type parameters are known; e.g. they apply to Point{Int} but not to Point. Optionally, outer constructors that determine type parameters automatically can be added, for example constructing a Point{Int} from the call Point(1,2). Outer constructors call inner constructors to actually make instances. However, in some cases one would rather not provide inner constructors, so that specific type parameters cannot be requested manually.\n\nFor example, say we define a type that stores a vector along with an accurate representation of its sum:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nSummedArray{Int32, Int32}(Int32[1, 2, 3], 6)\n\nThe problem is that we want S to be a larger type than T, so that we can sum many elements with less information loss. For example, when T is Int32, we would like S to be Int64. Therefore we want to avoid an interface that allows the user to construct instances of the type SummedArray{Int32,Int32}. One way to do this is to provide a constructor only for SummedArray, but inside the struct definition block to suppress generation of default constructors:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n           function SummedArray(a::Vector{T}) where T\n               S = widen(T)\n               new{T,S}(a, sum(S, a))\n           end\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nERROR: MethodError: no method matching SummedArray(::Vector{Int32}, ::Int32)\nThe type `SummedArray` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  SummedArray(::Vector{T}) where T\n   @ Main none:4\n\nStacktrace:\n[...]\n\nThis constructor will be invoked by the syntax SummedArray(a). The syntax new{T,S} allows specifying parameters for the type to be constructed, i.e. this call will return a SummedArray{T,S}. new{T,S} can be used in any constructor definition, but for convenience the parameters to new{} are automatically derived from the type being constructed when possible."},{"location":"manual/constructors.html#Constructors-are-just-callable-objects","page":"Constructors","title":"Constructors are just callable objects","category":"section","text":"An object of any type may be made callable by defining a method. This includes types, i.e., objects of type Type; and constructors may, in fact, be viewed as just callable type objects. For example, there are many methods defined on Bool and various supertypes of it:\n\nmethods(Bool)\n\nThe usual constructor syntax is exactly equivalent to the function-like object syntax, so trying to define a method with each syntax will cause the first method to be overwritten by the next one:\n\njulia> struct S\n           f::Int\n       end\n\njulia> S() = S(7)\nS\n\njulia> (::Type{S})() = S(8)  # overwrites the previous constructor method\n\njulia> S()\nS(8)"},{"location":"manual/modules.html#modules","page":"Modules","title":"Modules","category":"section","text":"Modules in Julia help organize code into coherent units. They are delimited syntactically inside module NameOfModule ... end, and have the following features:\n\nModules are separate namespaces, each introducing a new global scope. This is useful, because it allows the same name to be used for different functions or global variables without conflict, as long as they are in separate modules.\nModules have facilities for detailed namespace management: each defines a set of names it exports and marks as public, and can import names from other modules with using and import (we explain these below).\nModules can be precompiled for faster loading, and may contain code for runtime initialization.\n\nTypically, in larger Julia packages you will see module code organized into files, eg\n\nmodule SomeModule\n\n# export, public, using, import statements are usually here; we discuss these below\n\ninclude(\"file1.jl\")\ninclude(\"file2.jl\")\n\nend\n\nFiles and file names are mostly unrelated to modules; modules are associated only with module expressions. One can have multiple files per module, and multiple modules per file. include behaves as if the contents of the source file were evaluated in the global scope of the including module. In this chapter, we use short and simplified examples, so we won't use include.\n\nThe recommended style is not to indent the body of the module, since that would typically lead to whole files being indented. Also, it is common to use UpperCamelCase for module names (just like types), and use the plural form if applicable, especially if the module contains a similarly named identifier, to avoid name clashes. For example,\n\nmodule FastThings\n\nstruct FastThing\n    ...\nend\n\nend"},{"location":"manual/modules.html#namespace-management","page":"Modules","title":"Namespace management","category":"section","text":"Namespace management refers to the facilities the language offers for making names in a module available in other modules. We discuss the related concepts and functionality below in detail."},{"location":"manual/modules.html#Qualified-names","page":"Modules","title":"Qualified names","category":"section","text":"Names for functions, variables and types in the global scope like sin, ARGS, and UnitRange always belong to a module, called the parent module, which can be found interactively with parentmodule, for example\n\njulia> parentmodule(UnitRange)\nBase\n\nOne can also refer to these names outside their parent module by prefixing them with their module, eg Base.UnitRange. This is called a qualified name. The parent module may be accessible using a chain of submodules like Base.Math.sin, where Base.Math is called the module path. Due to syntactic ambiguities, qualifying a name that contains only symbols, such as an operator, requires inserting a colon, e.g. Base.:+. A small number of operators additionally require parentheses, e.g. Base.:(==).\n\nIf a name is qualified, then it is always accessible, and in case of a function, it can also have methods added to it by using the qualified name as the function name.\n\nWithin a module, a variable name can be “reserved” without assigning to it by declaring it as global x. This prevents name conflicts for globals initialized after load time. The syntax M.x = y does not work to assign a global in another module; global assignment is always module-local."},{"location":"manual/modules.html#Export-lists","page":"Modules","title":"Export lists","category":"section","text":"Names (referring to functions, types, global variables, and constants) can be added to the export list of a module with export: these are the symbols that are imported when using the module. Typically, they are at or near the top of the module definition so that readers of the source code can find them easily, as in\n\njulia> module NiceStuff\n       export nice, DOG\n       struct Dog end      # singleton type, not exported\n       const DOG = Dog()   # named instance, exported\n       nice(x) = \"nice $x\" # function, exported\n       end;\n\n\nbut this is just a style suggestion — a module can have multiple export statements in arbitrary locations.\n\nIt is common to export names which form part of the API (application programming interface). In the above code, the export list suggests that users should use nice and DOG. However, since qualified names always make identifiers accessible, this is just an option for organizing APIs: unlike other languages, Julia has no facilities for truly hiding module internals.\n\nAlso, some modules don't export names at all. This is usually done if they use common words, such as derivative, in their API, which could easily clash with the export lists of other modules. We will see how to manage name clashes below.\n\nTo mark a name as public without exporting it into the namespace of folks who call using NiceStuff, one can use public instead of export. This marks the public name(s) as part of the public API, but does not have any namespace implications. The public keyword is only available in Julia 1.11 and above. To maintain compatibility with Julia 1.10 and below, use the @compat macro from the Compat package, or the version-aware construct\n\nVERSION >= v\"1.11.0-DEV.469\" && eval(Meta.parse(\"public a, b, c\"))\n\nexport is a keyword wherever it occurs whereas the public keyword is currently limited to the syntactic top level within a file or module. This limitation exists for compatibility reasons, as public was introduced as a new keyword in Julia 1.11 while export has existed since Julia 1.0. However, this restriction on public may be lifted in future releases, so do not use public as an identifier."},{"location":"manual/modules.html#Standalone-using-and-import","page":"Modules","title":"Standalone using and import","category":"section","text":"For interactive use, the most common way of loading a module is using ModuleName. This loads the code associated with ModuleName, and brings\n\nthe module name\nand the elements of the export list into the surrounding global namespace.\n\nTechnically, the statement using ModuleName means that a module called ModuleName will be available for resolving names as needed. When a global variable is encountered that has no definition in the current module, the system will search for it among variables exported by ModuleName and use it if it is found there. This means that all uses of that global within the current module will resolve to the definition of that variable in ModuleName.\n\nTo load a module from a package, the statement using ModuleName can be used. To load a module from a locally defined module, a dot needs to be added before the module name like using .ModuleName.\n\nTo continue with our example,\n\njulia> using .NiceStuff\n\nwould load the above code, making NiceStuff (the module name), DOG and nice available. Dog is not on the export list, but it can be accessed if the name is qualified with the module path (which here is just the module name) as NiceStuff.Dog.\n\nImportantly, using ModuleName is the only form for which export lists matter at all.\n\nIn contrast,\n\njulia> import .NiceStuff\n\nbrings only the module name into scope. Users would need to use NiceStuff.DOG, NiceStuff.Dog, and NiceStuff.nice to access its contents. As we will see in the next section import .NiceStuff is equivalent to using .NiceStuff: NiceStuff. Usually, import ModuleName or using ModuleName: ModuleName is used in contexts when the user wants to keep the namespace clean.\n\nYou can combine multiple using and import statements of the same kind in a comma-separated expression, e.g.\n\njulia> using LinearAlgebra, Random"},{"location":"manual/modules.html#using-and-import-with-specific-identifiers,-and-adding-methods","page":"Modules","title":"using and import with specific identifiers, and adding methods","category":"section","text":"When using ModuleName: or import ModuleName: is followed by a comma-separated list of names, the module is loaded, but only those specific names are brought into the namespace by the statement. For example,\n\njulia> using .NiceStuff: nice, DOG\n\nwill import the names nice and DOG.\n\nImportantly, the module name NiceStuff will not be in the namespace. If you want to make it accessible, you have to list it explicitly, as\n\njulia> using .NiceStuff: nice, DOG, NiceStuff\n\nWhen two or more packages/modules export a name and that name does not refer to the same thing in each of the packages, and the packages are loaded via using without an explicit list of names, it is an error to reference that name without qualification. It is thus recommended that code intended to be forward-compatible with future versions of its dependencies and of Julia, e.g., code in released packages, list the names it uses from each loaded package, e.g., using Foo: Foo, f rather than using Foo.\n\nJulia has two forms for seemingly the same thing because only import ModuleName: f allows adding methods to f without a module path. That is to say, the following example will give an error:\n\njulia> using .NiceStuff: nice\n\njulia> struct Cat end\n\njulia> nice(::Cat) = \"nice 😸\"\nERROR: invalid method definition in Main: function NiceStuff.nice must be explicitly imported to be extended\nStacktrace:\n [1] top-level scope\n   @ none:1\n\nThis error prevents accidentally adding methods to functions in other modules that you only intended to use.\n\nThere are two ways to deal with this. You can always qualify function names with a module path:\n\njulia> using .NiceStuff\n\njulia> struct Cat end\n\njulia> NiceStuff.nice(::Cat) = \"nice 😸\"\n\nAlternatively, you can import the specific function name:\n\njulia> import .NiceStuff: nice\n\njulia> struct Mouse end\n\njulia> nice(::Mouse) = \"nice 🐭\"\nnice (generic function with 3 methods)\n\nWhich one you choose is a matter of style. The first form makes it clear that you are adding a method to a function in another module (remember, that the imports and the method definition may be in separate files), while the second one is shorter, which is especially convenient if you are defining multiple methods.\n\nOnce a variable is made visible via using or import, a module may not create its own variable with the same name. Imported variables are read-only; assigning to a global variable always affects a variable owned by the current module, or else raises an error."},{"location":"manual/modules.html#Renaming-with-as","page":"Modules","title":"Renaming with as","category":"section","text":"An identifier brought into scope by import or using can be renamed with the keyword as. This is useful for working around name conflicts as well as for shortening names. For example, Base exports the function name read, but the CSV.jl package also provides CSV.read. If we are going to invoke CSV reading many times, it would be convenient to drop the CSV. qualifier. But then it is ambiguous whether we are referring to Base.read or CSV.read:\n\njulia> read;\n\njulia> import CSV: read\nWARNING: ignoring conflicting import of CSV.read into Main\n\nRenaming provides a solution:\n\njulia> import CSV: read as rd\n\nImported packages themselves can also be renamed:\n\nimport BenchmarkTools as BT\n\nas works with using only when a single identifier is brought into scope. For example using CSV: read as rd works, but using CSV as C does not, since it operates on all of the exported names in CSV."},{"location":"manual/modules.html#Mixing-multiple-using-and-import-statements","page":"Modules","title":"Mixing multiple using and import statements","category":"section","text":"When multiple using or import statements of any of the forms above are used, their effect is combined in the order they appear. For example,\n\njulia> using .NiceStuff         # exported names and the module name\n\njulia> import .NiceStuff: nice  # allows adding methods to unqualified functions\n\n\nwould bring all the exported names of NiceStuff and the module name itself into scope, and also allow adding methods to nice without prefixing it with a module name."},{"location":"manual/modules.html#Handling-name-conflicts","page":"Modules","title":"Handling name conflicts","category":"section","text":"Consider the situation where two (or more) packages export the same name, as in\n\njulia> module A\n       export f\n       f() = 1\n       end\nA\njulia> module B\n       export f\n       f() = 2\n       end\nB\n\nThe statement using .A, .B works, but when you try to call f, you get an error with a hint\n\njulia> using .A, .B\n\njulia> f\nERROR: UndefVarError: `f` not defined in `Main`\nHint: It looks like two or more modules export different bindings with this name, resulting in ambiguity. Try explicitly importing it from a particular module, or qualifying the name with the module it should come from.\n\nHere, Julia cannot decide which f you are referring to, so you have to make a choice. The following solutions are commonly used:\n\nSimply proceed with qualified names like A.f and B.f. This makes the context clear to the reader of your code, especially if f just happens to coincide but has different meaning in various packages. For example, degree has various uses in mathematics, the natural sciences, and in everyday life, and these meanings should be kept separate.\nUse the as keyword above to rename one or both identifiers, eg\njulia> using .A: f as f\n\njulia> using .B: f as g\n\nwould make B.f available as g. Here, we are assuming that you did not use using A before, which would have brought f into the namespace.\nWhen the names in question do share a meaning, it is common for one module to import it from another, or have a lightweight “base” package with the sole function of defining an interface like this, which can be used by other packages. It is conventional to have such package names end in ...Base (which has nothing to do with Julia's Base module)."},{"location":"manual/modules.html#Precedence-order-of-definitions","page":"Modules","title":"Precedence order of definitions","category":"section","text":"There are in general four kinds of binding definitions:\n\nThose provided via implicit import through using M\nThose provided via explicit import (e.g. using M: x, import M: x)\nThose declared implicitly as global (via global x without type specification)\nThose declared explicitly using definition syntax (const, global x::T, struct, etc.)\n\nSyntactically, we divide these into three precedence levels (from weakest to strongest)\n\nImplicit imports\nImplicit declarations\nExplicit declarations and imports\n\nIn general, we permit replacement of weaker bindings by stronger ones:\n\njulia> module M1; const x = 1; export x; end\nMain.M1\n\njulia> using .M1\n\njulia> x # Implicit import from M1\n1\n\njulia> begin; f() = (global x; x = 1) end\n\njulia> x # Implicit declaration\nERROR: UndefVarError: `x` not defined in `Main`\nSuggestion: add an appropriate import or assignment. This global was declared but not assigned.\n\njulia> const x = 2 # Explicit declaration\n2\n\nHowever, within the explicit precedence level, replacement is syntactically disallowed:\n\njulia> module M1; const x = 1; export x; end\nMain.M1\n\njulia> import .M1: x\n\njulia> const x = 2\nERROR: cannot declare Main.x constant; it was already declared as an import\nStacktrace:\n [1] top-level scope\n   @ REPL[3]:1\n\nor ignored:\n\njulia> const y = 2\n2\n\njulia> import .M1: x as y\nWARNING: import of M1.x into Main conflicts with an existing identifier; ignored.\n\nThe resolution of an implicit binding depends on the set of all using'd modules visible in the current world age. See the manual chapter on world age for more details."},{"location":"manual/modules.html#Default-top-level-definitions-and-bare-modules","page":"Modules","title":"Default top-level definitions and bare modules","category":"section","text":"Modules automatically contain using Core, using Base, and definitions of the eval and include functions, which evaluate expressions/files within the global scope of that module.\n\nIf these default definitions are not wanted, modules can be defined using the keyword baremodule instead (note: Core is still imported). In terms of baremodule, a standard module looks like this:\n\nbaremodule Mod\n\nusing Base\n\neval(x) = Core.eval(Mod, x)\ninclude(p) = Base.include(Mod, p)\n\n...\n\nend\n\nIf even Core is not wanted, a module that imports nothing and defines no names at all can be defined with Module(:YourNameHere, false, false) and code can be evaluated into it with @eval or Core.eval:\n\njulia> arithmetic = Module(:arithmetic, false, false)\nMain.arithmetic\n\njulia> @eval arithmetic add(x, y) = $(+)(x, y)\nadd (generic function with 1 method)\n\njulia> arithmetic.add(12, 13)\n25"},{"location":"manual/modules.html#Standard-modules","page":"Modules","title":"Standard modules","category":"section","text":"There are three important standard modules:\n\nCore contains all functionality \"built into\" the language.\nBase contains basic functionality that is useful in almost all cases.\nMain is the top-level module and the current module, when Julia is started.\n\nnote: Standard library modules\nBy default Julia ships with some standard library modules. These behave like regular Julia packages except that you don't need to install them explicitly. For example, if you wanted to perform some unit testing, you could load the Test standard library as follows:using Test"},{"location":"manual/modules.html#Submodules-and-relative-paths","page":"Modules","title":"Submodules and relative paths","category":"section","text":"Modules can contain submodules, nesting the same syntax module ... end. They can be used to introduce separate namespaces, which can be helpful for organizing complex codebases. Note that each module introduces its own scope, so submodules do not automatically “inherit” names from their parent.\n\nIt is recommended that submodules refer to other modules within the enclosing parent module (including the latter) using relative module qualifiers in using and import statements. A relative module qualifier starts with a period (.), which corresponds to the current module, and each successive . leads to the parent of the current module. This should be followed by modules if necessary, and eventually the actual name to access, all separated by .s. As a special case, however, referring to the module root can be written without ., avoiding the need to count the depth to reach that module.\n\nConsider the following example, where the submodule SubA defines a function, which is then extended in its “sibling” module:\n\njulia> module ParentModule\n       module SubA\n       export add_D  # exported interface\n       const D = 3\n       add_D(x) = x + D\n       end\n       using .SubA  # brings `add_D` into the namespace\n       export add_D # export it from ParentModule too\n       module SubB\n       import ..SubA: add_D # relative path for a “sibling” module\n       # import ParentModule.SubA: add_D # when in a package, such as when this is loaded by using or import, this would be equivalent to the previous import, but not at the REPL\n       struct Infinity end\n       add_D(x::Infinity) = x\n       end\n       end;\n\n\nYou may see code in packages, which, in a similar situation, uses import without the .:\n\njulia> import ParentModule.SubA: add_D\nERROR: ArgumentError: Package ParentModule not found in current path.\n\nHowever, since this operates through code loading, it only works if ParentModule is in a package in a file. If ParentModule was defined at the REPL, it is necessary to use use relative paths:\n\njulia> import .ParentModule.SubA: add_D\n\n\nNote that the order of definitions also matters if you are evaluating values. Consider\n\nmodule TestPackage\n\nexport x, y\n\nx = 0\n\nmodule Sub\nusing ..TestPackage\nz = y # ERROR: UndefVarError: `y` not defined in `Main`\nend\n\ny = 1\n\nend\n\nwhere Sub is trying to use TestPackage.y before it was defined, so it does not have a value.\n\nFor similar reasons, you cannot use a cyclic ordering:\n\nmodule A\n\nmodule B\nusing ..C # ERROR: UndefVarError: `C` not defined in `Main.A`\nend\n\nmodule C\nusing ..B\nend\n\nend"},{"location":"manual/modules.html#Module-initialization-and-precompilation","page":"Modules","title":"Module initialization and precompilation","category":"section","text":"Large modules can take several seconds to load because executing all of the statements in a module often involves compiling a large amount of code. Julia creates precompiled caches of the module to reduce this time.\n\nPrecompiled module files (sometimes called \"cache files\") are created and used automatically when import or using loads a module. If the cache file(s) do not yet exist, the module will be compiled and saved for future reuse. You can also manually call Base.compilecache(Base.identify_package(\"modulename\")) to create these files without loading the module. The resulting cache files will be stored in the compiled subfolder of DEPOT_PATH[1]. If nothing about your system changes, such cache files will be used when you load the module with import or using.\n\nPrecompilation cache files store definitions of modules, types, methods, and constants. They may also store method specializations and the code generated for them, but this typically requires that the developer add explicit precompile directives or execute workloads that force compilation during the package build.\n\nHowever, if you update the module's dependencies or change its source code, the module is automatically recompiled upon using or import. Dependencies are modules it imports, the Julia build, files it includes, or explicit dependencies declared by include_dependency(path) in the module file(s).\n\nFor file dependencies loaded by include, a change is determined by examining whether the file size (fsize) or content (condensed into a hash) is unchanged. For file dependencies loaded by include_dependency a change is determined by examining whether the modification time (mtime) is unchanged, or equal to the modification time truncated to the nearest second (to accommodate systems that can't copy mtime with sub-second accuracy). It also takes into account whether the path to the file chosen by the search logic in require matches the path that had created the precompile file. It also takes into account the set of dependencies already loaded into the current process and won't recompile those modules, even if their files change or disappear, in order to avoid creating incompatibilities between the running system and the precompile cache. Finally, it takes account of changes in any compile-time preferences.\n\nIf you know that a module is not safe to precompile (for example, for one of the reasons described below), you should put __precompile__(false) in the module file (typically placed at the top). This will cause Base.compilecache to throw an error, and will cause using / import to load it directly into the current process and skip the precompile and caching. This also thereby prevents the module from being imported by any other precompiled module.\n\nYou may need to be aware of certain behaviors inherent in the creation of incremental shared libraries which may require care when writing your module. For example, external state is not preserved. To accommodate this, explicitly separate any initialization steps that must occur at runtime from steps that can occur at compile time. For this purpose, Julia allows you to define an __init__() function in your module that executes any initialization steps that must occur at runtime. This function will not be called during compilation (--output-*). Effectively, you can assume it will be run exactly once in the lifetime of the code. You may, of course, call it manually if necessary, but the default is to assume this function deals with computing state for the local machine, which does not need to be – or even should not be – captured in the compiled image. It will be called after the module is loaded into a process, including if it is being loaded into an incremental compile (--output-incremental=yes), but not if it is being loaded into a full-compilation process.\n\nIn particular, if you define a function __init__() in a module, then Julia will call __init__() immediately after the module is loaded (e.g., by import, using, or require) at runtime for the first time (i.e., __init__ is only called once, and only after all statements in the module have been executed). Because it is called after the module is fully imported, any submodules or other imported modules have their __init__ functions called before the __init__ of the enclosing module. This is also synchronized across threads, so that code can safely rely upon this ordering of effects, such that all __init__ will have run, in dependency ordering, before the using result is completed. They may run concurrently with other __init__ methods which are not dependencies however, so be careful when accessing any shared state outside the current module to use locks when needed.\n\nTwo typical uses of __init__ are calling runtime initialization functions of external C libraries and initializing global constants that involve pointers returned by external libraries. For example, suppose that we are calling a C library libfoo that requires us to call a foo_init() initialization function at runtime. Suppose that we also want to define a global constant foo_data_ptr that holds the return value of a void *foo_data() function defined by libfoo – this constant must be initialized at runtime (not at compile time) because the pointer address will change from run to run. You could accomplish this by defining the following __init__ function in your module:\n\nconst foo_data_ptr = Ref{Ptr{Cvoid}}(0)\nfunction __init__()\n    ccall((:foo_init, :libfoo), Cvoid, ())\n    foo_data_ptr[] = ccall((:foo_data, :libfoo), Ptr{Cvoid}, ())\n    nothing\nend\n\nNotice that it is perfectly possible to define a global inside a function like __init__; this is one of the advantages of using a dynamic language. But by making it a constant at global scope, we can ensure that the type is known to the compiler and allow it to generate better optimized code. Obviously, any other globals in your module that depends on foo_data_ptr would also have to be initialized in __init__.\n\nConstants involving most Julia objects that are not produced by ccall do not need to be placed in __init__: their definitions can be precompiled and loaded from the cached module image. This includes complicated heap-allocated objects like arrays. However, any routine that returns a raw pointer value must be called at runtime for precompilation to work (Ptr objects will turn into null pointers unless they are hidden inside an isbits object). This includes the return values of the Julia functions @cfunction and pointer.\n\nWhen using precompilation, it is important to keep a clear sense of the distinction between the compilation phase and the execution phase. In this mode, it will often be much more clearly apparent that Julia is a compiler which allows execution of arbitrary Julia code, not a standalone interpreter that also generates compiled code.\n\nOther known potential failure scenarios include:\n\nGlobal counters (for example, for attempting to uniquely identify objects). Consider the following code snippet:\nmutable struct UniquedById\n    myid::Int\n    let counter = 0\n        UniquedById() = new(counter += 1)\n    end\nend\nwhile the intent of this code was to give every instance a unique id, the counter value is recorded at the end of compilation. All subsequent usages of this incrementally compiled module will start from that same counter value.\nNote that objectid (which works by hashing the memory pointer) has similar issues (see notes on Dict usage below).\nOne alternative is to use a macro to capture @__MODULE__ and store it alone with the current counter value, however, it may be better to redesign the code to not depend on this global state.\nAssociative collections (such as Dict and Set) need to be re-hashed in __init__. (In the future, a mechanism may be provided to register an initializer function.)\nDepending on compile-time side-effects persisting through load-time. Example include: modifying arrays or other variables in other Julia modules; maintaining handles to open files or devices; storing pointers to other system resources (including memory);\nCreating accidental \"copies\" of global state from another module, by referencing it directly instead of via its lookup path. For example, (in global scope):\n#mystdout = Base.stdout #= will not work correctly, since this will copy Base.stdout into this module =#\n# instead use accessor functions:\ngetstdout() = Base.stdout #= best option =#\n# or move the assignment into the runtime:\n__init__() = global mystdout = Base.stdout #= also works =#\n\nSeveral additional restrictions are placed on the operations that can be done while precompiling code to help the user avoid other wrong-behavior situations:\n\nCalling eval to cause a side-effect in another module. This will also cause a warning to be emitted when the incremental precompile flag is set.\nglobal const statements from local scope after __init__() has been started (see issue #12010 for plans to add an error for this)\nReplacing a module is a runtime error while doing an incremental precompile.\n\nA few other points to be aware of:\n\nNo code reload / cache invalidation is performed after changes are made to the source files themselves, (including by Pkg.update), and no cleanup is done after Pkg.rm\nThe memory sharing behavior of a reshaped array is disregarded by precompilation (each view gets its own copy)\nExpecting the filesystem to be unchanged between compile-time and runtime e.g. @__FILE__/source_path() to find resources at runtime, or the BinDeps @checked_lib macro. Sometimes this is unavoidable. However, when possible, it can be good practice to copy resources into the module at compile-time so they won't need to be found at runtime.\nWeakRef objects and finalizers are not currently handled properly by the serializer (this will be fixed in an upcoming release).\nIt is usually best to avoid capturing references to instances of internal metadata objects such as Method, MethodInstance, MethodTable, TypeMapLevel, TypeMapEntry and fields of those objects, as this can confuse the serializer and may not lead to the outcome you desire. It is not necessarily an error to do this, but you simply need to be prepared that the system will try to copy some of these and to create a single unique instance of others.\n\nIt is sometimes helpful during module development to turn off incremental precompilation. The command line flag --compiled-modules={yes|no|existing} enables you to toggle module precompilation on and off. When Julia is started with --compiled-modules=no the serialized modules in the compile cache are ignored when loading modules and module dependencies. In some cases, you may want to load existing precompiled modules, but not create new ones. This can be done by starting Julia with --compiled-modules=existing. More fine-grained control is available with --pkgimages={yes|no|existing}, which only affects native-code storage during precompilation. Base.compilecache can still be called manually. The state of this command line flag is passed to Pkg.build to disable automatic precompilation triggering when installing, updating, and explicitly building packages.\n\nYou can also debug some precompilation failures with environment variables. Setting JULIA_VERBOSE_LINKING=true may help resolve failures in linking shared libraries of compiled native code. See the Developer Documentation part of the Julia manual, where you will find further details in the section documenting Julia's internals under \"Package Images\"."},{"location":"manual/code-loading.html#code-loading","page":"Code Loading","title":"Code Loading","category":"section","text":"note: Note\nThis chapter covers the technical details of package loading. To install packages, use Pkg, Julia's built-in package manager, to add packages to your active environment. To use packages already in your active environment, write import X or using X, as described in the Modules documentation."},{"location":"manual/code-loading.html#Definitions","page":"Code Loading","title":"Definitions","category":"section","text":"Julia has two mechanisms for loading code:\n\nCode inclusion: e.g. include(\"source.jl\"). Inclusion allows you to split a single program across multiple source files. The expression include(\"source.jl\") causes the contents of the file source.jl to be evaluated in the global scope of the module where the include call occurs. If include(\"source.jl\") is called multiple times, source.jl is evaluated multiple times. The included path, source.jl, is interpreted relative to the file where the include call occurs. This makes it simple to relocate a subtree of source files. In the REPL, included paths are interpreted relative to the current working directory, pwd().\nPackage loading: e.g. import X or using X. The import mechanism allows you to load a package—i.e. an independent, reusable collection of Julia code, wrapped in a module—and makes the resulting module available by the name X inside of the importing module. If the same X package is imported multiple times in the same Julia session, it is only loaded the first time—on subsequent imports, the importing module gets a reference to the same module. Note though, that import X can load different packages in different contexts: X can refer to one package named X in the main project but potentially to different packages also named X in each dependency. More on this below.\n\nCode inclusion is quite straightforward and simple: it evaluates the given source file in the context of the caller. Package loading is built on top of code inclusion and serves a different purpose. The rest of this chapter focuses on the behavior and mechanics of package loading.\n\nA package is a source tree with a standard layout providing functionality that can be reused by other Julia projects. A package is loaded by import X or  using X statements. These statements also make the module named X—which results from loading the package code—available within the module where the import statement occurs. The meaning of X in import X is context-dependent: which X package is loaded depends on what code the statement occurs in. Thus, handling of import X happens in two stages: first, it determines what package is defined to be X in this context; second, it determines where that particular X package is found.\n\nThese questions are answered by searching through the project environments listed in LOAD_PATH for project files (Project.toml or JuliaProject.toml), manifest files (Manifest.toml or JuliaManifest.toml, or the same names suffixed by -v{major}.{minor}.toml for specific versions), or folders of source files."},{"location":"manual/code-loading.html#Federation-of-packages","page":"Code Loading","title":"Federation of packages","category":"section","text":"Most of the time, a package is uniquely identifiable simply from its name. However, sometimes a project might encounter a situation where it needs to use two different packages that share the same name. While you might be able fix this by renaming one of the packages, being forced to do so can be highly disruptive in a large, shared code base. Instead, Julia's code loading mechanism allows the same package name to refer to different packages in different components of an application.\n\nJulia supports federated package management, which means that multiple independent parties can maintain both public and private packages and registries of packages, and that projects can depend on a mix of public and private packages from different registries. Packages from various registries are installed and managed using a common set of tools and workflows. The Pkg package manager that ships with Julia lets you install and manage your projects' dependencies. It assists in creating and manipulating project files (which describe what other projects that your project depends on), and manifest files (which snapshot exact versions of your project's complete dependency graph).\n\nOne consequence of federation is that there cannot be a central authority for package naming. Different entities may use the same name to refer to unrelated packages. This possibility is unavoidable since these entities do not coordinate and may not even know about each other. Because of the lack of a central naming authority, a single project may end up depending on different packages that have the same name. Julia's package loading mechanism does not require package names to be globally unique, even within the dependency graph of a single project. Instead, packages are identified by universally unique identifiers (UUIDs), which get assigned when each package is created. Usually you won't have to work directly with these somewhat cumbersome 128-bit identifiers since Pkg will take care of generating and tracking them for you. However, these UUIDs provide the definitive answer to the question of \"what package does X refer to?\"\n\nSince the decentralized naming problem is somewhat abstract, it may help to walk through a concrete scenario to understand the issue. Suppose you're developing an application called App, which uses two packages: Pub and  Priv. Priv is a private package that you created, whereas Pub is a public package that you use but don't control. When you created Priv, there was no public package by the name Priv. Subsequently, however, an unrelated package also named Priv has been published and become popular. In fact, the Pub package has started to use it. Therefore, when you next upgrade Pub to get the latest bug fixes and features, App will end up depending on two different packages named Priv—through no action of yours other than upgrading. App has a direct dependency on your private Priv package, and an indirect dependency, through Pub, on the new public Priv package. Since these two Priv packages are different but are both required for App to continue working correctly, the expression import Priv must refer to different Priv packages depending on whether it occurs in App's code or in Pub's code. To handle this, Julia's package loading mechanism distinguishes the two Priv packages by their UUID and picks the correct one based on its context (the module that called import). How this distinction works is determined by environments, as explained in the following sections."},{"location":"manual/code-loading.html#Environments","page":"Code Loading","title":"Environments","category":"section","text":"An environment determines what import X and using X mean in various code contexts and what files these statements cause to be loaded. Julia understands two kinds of environments:\n\nA project environment is a directory with a project file and an optional manifest file, and forms an explicit environment. The project file determines what the names and identities of the direct dependencies of a project are. The manifest file, if present, gives a complete dependency graph, including all direct and indirect dependencies, exact versions of each dependency, and sufficient information to locate and load the correct version.\nA package directory is a directory containing the source trees of a set of packages as subdirectories, and forms an implicit environment. If X is a subdirectory of a package directory and X/src/X.jl exists, then the package X is available in the package directory environment and X/src/X.jl is the source file by which it is loaded.\n\nThese can be intermixed to create a stacked environment: an ordered set of project environments and package directories, overlaid to make a single composite environment. The precedence and visibility rules then combine to determine which packages are available and where they get loaded from. Julia's load path forms a stacked environment, for example.\n\nThese environments each serve a different purpose:\n\nProject environments provide reproducibility. By checking a project environment into version control—e.g. a git repository—along with the rest of the project's source code, you can reproduce the exact state of the project and all of its dependencies. The manifest file, in particular, captures the exact version of every dependency, identified by a cryptographic hash of its source tree, which makes it possible for Pkg to retrieve the correct versions and be sure that you are running the exact code that was recorded for all dependencies.\nPackage directories provide convenience when a full carefully-tracked project environment is unnecessary. They are useful when you want to put a set of packages somewhere and be able to directly use them, without needing to create a project environment for them.\nStacked environments allow for adding tools to the primary environment. You can push an environment of development tools onto the end of the stack to make them available from the REPL and scripts, but not from inside packages.\n\nnote: Note\nWhen loading a package from another environment in the stack other than the active environment the package is loaded in the context of the active environment. This means that the package will be loaded as if it were imported in the active environment, which may affect how its dependencies versions are resolved. When such a package is precompiling it will be marked as a (serial) precompile job, which means that its dependencies will be precompiled in series within the same job, which will likely be slower.\n\nAt a high-level, each environment conceptually defines three maps: roots, graph and paths. When resolving the meaning of import X, the roots and graph maps are used to determine the identity of X, while the paths map is used to locate the source code of X. The specific roles of the three maps are:\n\nroots: name::Symbol ⟶ uuid::UUID\nAn environment's roots map assigns package names to UUIDs for all the top-level dependencies that the environment makes available to the main project (i.e. the ones that can be loaded in Main). When Julia encounters import X in the main project, it looks up the identity of X as roots[:X].\ngraph: context::UUID ⟶ name::Symbol ⟶ uuid::UUID\nAn environment's graph is a multilevel map which assigns, for each context UUID, a map from names to UUIDs, similar to the roots map but specific to that context. When Julia sees import X in the code of the package whose UUID is context, it looks up the identity of X as graph[context][:X]. In particular, this means that import X can refer to different packages depending on context.\npaths: uuid::UUID × name::Symbol ⟶ path::String\nThe paths map assigns to each package UUID-name pair, the location of that package's entry-point source file. After the identity of X in import X has been resolved to a UUID via roots or graph (depending on whether it is loaded from the main project or a dependency), Julia determines what file to load to acquire X by looking up paths[uuid,:X] in the environment. Including this file should define a module named X. Once this package is loaded, any subsequent import resolving to the same uuid will create a new binding to the already-loaded package module.\n\nEach kind of environment defines these three maps differently, as detailed in the following sections.\n\nnote: Note\nFor ease of understanding, the examples throughout this chapter show full data structures for roots, graph and paths. However, Julia's package loading code does not explicitly create these. Instead, it lazily computes only as much of each structure as it needs to load a given package."},{"location":"manual/code-loading.html#project-environments","page":"Code Loading","title":"Project environments","category":"section","text":"A project environment is determined by a directory containing a project file called Project.toml, and optionally a manifest file called Manifest.toml. These files may also be called JuliaProject.toml and JuliaManifest.toml, in which case Project.toml and Manifest.toml are ignored. This allows for coexistence with other tools that might consider files called Project.toml and Manifest.toml significant. For pure Julia projects, however, the names Project.toml and Manifest.toml are preferred. However, from Julia v1.10.8 onwards, (Julia)Manifest-v{major}.{minor}.toml is recognized as a format to make a given julia version use a specific manifest file i.e. in the same folder, a Manifest-v1.11.toml would be used by v1.11 and Manifest.toml by any other julia version.\n\nThe roots, graph and paths maps of a project environment are defined as follows:\n\nThe roots map of the environment is determined by the contents of the project file, specifically, its top-level name and uuid entries and its [deps] section (all optional). Consider the following example project file for the hypothetical application, App, as described earlier:\n\nname = \"App\"\nuuid = \"8f986787-14fe-4607-ba5d-fbff2944afa9\"\n\n[deps]\nPriv = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\nPub  = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"\n\nThis project file implies the following roots map, if it was represented by a Julia dictionary:\n\nroots = Dict(\n    :App  => UUID(\"8f986787-14fe-4607-ba5d-fbff2944afa9\"),\n    :Priv => UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"),\n    :Pub  => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n)\n\nGiven this roots map, in App's code the statement import Priv will cause Julia to look up roots[:Priv], which yields ba13f791-ae1d-465a-978b-69c3ad90f72b, the UUID of the Priv package that is to be loaded in that context. This UUID identifies which Priv package to load and use when the main application evaluates import Priv.\n\nThe dependency graph of a project environment is determined by the contents of the manifest file, if present. If there is no manifest file, graph is empty. A manifest file contains a stanza for each of a project's direct or indirect dependencies. For each dependency, the file lists the package's UUID and a source tree hash or an explicit path to the source code. Consider the following example manifest file for App:\n\n[[Priv]] # the private one\ndeps = [\"Pub\", \"Zebra\"]\nuuid = \"ba13f791-ae1d-465a-978b-69c3ad90f72b\"\npath = \"deps/Priv\"\n\n[[Priv]] # the public one\nuuid = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\ngit-tree-sha1 = \"1bf63d3be994fe83456a03b874b409cfd59a6373\"\nversion = \"0.1.5\"\n\n[[Pub]]\nuuid = \"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"\ngit-tree-sha1 = \"9ebd50e2b0dd1e110e842df3b433cb5869b0dd38\"\nversion = \"2.1.4\"\n\n  [Pub.deps]\n  Priv = \"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"\n  Zebra = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\n\n[[Zebra]]\nuuid = \"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"\ngit-tree-sha1 = \"e808e36a5d7173974b90a15a353b564f3494092f\"\nversion = \"3.4.2\"\n\nThis manifest file describes a possible complete dependency graph for the App project:\n\nThere are two different packages named Priv that the application uses. It uses a private package, which is a root dependency, and a public one, which is an indirect dependency through Pub. These are differentiated by their distinct UUIDs, and they have different deps:\nThe private Priv depends on the Pub and Zebra packages.\nThe public Priv has no dependencies.\nThe application also depends on the Pub package, which in turn depends on the public Priv and the same Zebra package that the private Priv package depends on.\n\nThis dependency graph represented as a dictionary, looks like this:\n\ngraph = Dict(\n    # Priv – the private one:\n    UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\") => Dict(\n        :Pub   => UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Priv – the public one:\n    UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\") => Dict(),\n    # Pub:\n    UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\") => Dict(\n        :Priv  => UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"),\n        :Zebra => UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"),\n    ),\n    # Zebra:\n    UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\") => Dict(),\n)\n\nGiven this dependency graph, when Julia sees import Priv in the Pub package—which has UUID c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1—it looks up:\n\ngraph[UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\")][:Priv]\n\nand gets 2d15fe94-a1f7-436c-a4d8-07a9a496e01c, which indicates that in the context of the Pub package, import Priv refers to the public Priv package, rather than the private one which the app depends on directly. This is how the name Priv can refer to different packages in the main project than it does in one of its package's dependencies, which allows for duplicate names in the package ecosystem.\n\nWhat happens if import Zebra is evaluated in the main App code base? Since Zebra does not appear in the project file, the import will fail even though Zebra does appear in the manifest file. Moreover, if import Zebra occurs in the public Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—then that would also fail since that Priv package has no declared dependencies in the manifest file and therefore cannot load any packages. The Zebra package can only be loaded by packages for which it appears as an explicit dependency in the manifest file: the Pub package and one of the Priv packages.\n\nThe paths map of a project environment is extracted from the manifest file. The path of a package uuid named X is determined by these rules (in order):\n\nIf the project file in the directory matches uuid and name X, then either:\nIt has a toplevel entryfile entry, then uuid will be mapped to that path, interpreted relative to the directory containing the project file.\nOtherwise, uuid is mapped to src/X.jl relative to the directory containing the project file.\nIf the above is not the case and the project file has a corresponding manifest file and the manifest contains a stanza matching uuid then:\nIf it has a path entry, use that path (relative to the directory containing the manifest file).\nIf it has a git-tree-sha1 entry, compute a deterministic hash function of uuid and git-tree-sha1—call it slug—and look for a directory named packages/X/$slug in each directory in the Julia DEPOT_PATH global array. Use the first such directory that exists.\nIf this is a directory then uuid is mapped to src/X.jl unless the matching manifest stanza has an entryfile entry in which case this is used. In both cases, these are relative to the directory in 2.1.\n\nIf any of these result in success, the path to the source code entry point will be either that result, the relative path from that result plus src/X.jl; otherwise, there is no path mapping for uuid. When loading X, if no source code path is found, the lookup will fail, and the user may be prompted to install the appropriate package version or to take other corrective action (e.g. declaring X as a dependency).\n\nIn the example manifest file above, to find the path of the first Priv package—the one with UUID ba13f791-ae1d-465a-978b-69c3ad90f72b—Julia looks for its stanza in the manifest file, sees that it has a path entry, looks at deps/Priv relative to the App project directory—let's suppose the App code lives in /home/me/projects/App—sees that /home/me/projects/App/deps/Priv exists and therefore loads Priv from there.\n\nIf, on the other hand, Julia was loading the other Priv package—the one with UUID 2d15fe94-a1f7-436c-a4d8-07a9a496e01c—it finds its stanza in the manifest, see that it does not have a path entry, but that it does have a git-tree-sha1 entry. It then computes the slug for this UUID/SHA-1 pair, which is HDkrT (the exact details of this computation aren't important, but it is consistent and deterministic). This means that the path to this Priv package will be packages/Priv/HDkrT/src/Priv.jl in one of the package depots. Suppose the contents of DEPOT_PATH is [\"/home/me/.julia\", \"/usr/local/julia\"], then Julia will look at the following paths to see if they exist:\n\n/home/me/.julia/packages/Priv/HDkrT\n/usr/local/julia/packages/Priv/HDkrT\n\nJulia uses the first of these that exists to try to load the public Priv package from the file packages/Priv/HDKrT/src/Priv.jl in the depot where it was found.\n\nHere is a representation of a possible paths map for our example App project environment, as provided in the Manifest given above for the dependency graph, after searching the local file system:\n\npaths = Dict(\n    # Priv – the private one:\n    (UUID(\"ba13f791-ae1d-465a-978b-69c3ad90f72b\"), :Priv) =>\n        # relative entry-point inside `App` repo:\n        \"/home/me/projects/App/deps/Priv/src/Priv.jl\",\n    # Priv – the public one:\n    (UUID(\"2d15fe94-a1f7-436c-a4d8-07a9a496e01c\"), :Priv) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Priv/HDkrT/src/Priv.jl\",\n    # Pub:\n    (UUID(\"c07ecb7d-0dc9-4db7-8803-fadaaeaf08e1\"), :Pub) =>\n        # package installed in the user depot:\n        \"/home/me/.julia/packages/Pub/oKpw/src/Pub.jl\",\n    # Zebra:\n    (UUID(\"f7a24cb4-21fc-4002-ac70-f0e3a0dd3f62\"), :Zebra) =>\n        # package installed in the system depot:\n        \"/usr/local/julia/packages/Zebra/me9k/src/Zebra.jl\",\n)\n\nThis example map includes three different kinds of package locations (the first and third are part of the default load path):\n\nThe private Priv package is \"vendored\" inside the App repository.\nThe public Priv and Zebra packages are in the system depot, where packages installed and managed by the system administrator live. These are available to all users on the system.\nThe Pub package is in the user depot, where packages installed by the user live. These are only available to the user who installed them."},{"location":"manual/code-loading.html#Package-directories","page":"Code Loading","title":"Package directories","category":"section","text":"Package directories provide a simpler kind of environment without the ability to handle name collisions. In a package directory, the set of top-level packages is the set of subdirectories that \"look like\" packages. A package X exists in a package directory if the directory contains one of the following \"entry point\" files:\n\nX.jl\nX/src/X.jl\nX.jl/src/X.jl\n\nWhich dependencies a package in a package directory can import depends on whether the package contains a project file:\n\nIf it has a project file, it can only import those packages which are identified in the [deps] section of the project file.\nIf it does not have a project file, it can import any top-level package—i.e. the same packages that can be loaded in Main or the REPL.\n\nThe roots map is determined by examining the contents of the package directory to generate a list of all packages that exist. Additionally, a UUID will be assigned to each entry as follows: For a given package found inside the folder X...\n\nIf X/Project.toml exists and has a uuid entry, then uuid is that value.\nIf X/Project.toml exists and but does not have a top-level UUID entry, uuid is a dummy UUID generated by hashing the canonical (real) path to X/Project.toml.\nOtherwise (if Project.toml does not exist), then uuid is the all-zero nil UUID.\n\nThe dependency graph of a project directory is determined by the presence and contents of project files in the subdirectory of each package. The rules are:\n\nIf a package subdirectory has no project file, then it is omitted from graph and import statements in its code are treated as top-level, the same as the main project and REPL.\nIf a package subdirectory has a project file, then the graph entry for its UUID is the [deps] map of the project file, which is considered to be empty if the section is absent.\n\nAs an example, suppose a package directory has the following structure and content:\n\nAardvark/\n    src/Aardvark.jl:\n        import Bobcat\n        import Cobra\n\nBobcat/\n    Project.toml:\n        [deps]\n        Cobra = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Bobcat.jl:\n        import Cobra\n        import Dingo\n\nCobra/\n    Project.toml:\n        uuid = \"4725e24d-f727-424b-bca0-c4307a3456fa\"\n        [deps]\n        Dingo = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Cobra.jl:\n        import Dingo\n\nDingo/\n    Project.toml:\n        uuid = \"7a7925be-828c-4418-bbeb-bac8dfc843bc\"\n\n    src/Dingo.jl:\n        # no imports\n\nHere is a corresponding roots structure, represented as a dictionary:\n\nroots = Dict(\n    :Aardvark => UUID(\"00000000-0000-0000-0000-000000000000\"), # no project file, nil UUID\n    :Bobcat   => UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), # dummy UUID based on path\n    :Cobra    => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), # UUID from project file\n    :Dingo    => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), # UUID from project file\n)\n\nHere is the corresponding graph structure, represented as a dictionary:\n\ngraph = Dict(\n    # Bobcat:\n    UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\") => Dict(\n        :Cobra => UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"),\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Cobra:\n    UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\") => Dict(\n        :Dingo => UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"),\n    ),\n    # Dingo:\n    UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\") => Dict(),\n)\n\nA few general rules to note:\n\nA package without a project file can depend on any top-level dependency, and since every package in a package directory is available at the top-level, it can import all packages in the environment.\nA package with a project file cannot depend on one without a project file since packages with project files can only load packages in graph and packages without project files do not appear in graph.\nA package with a project file but no explicit UUID can only be depended on by packages without project files since dummy UUIDs assigned to these packages are strictly internal.\n\nObserve the following specific instances of these rules in our example:\n\nAardvark can import on any of Bobcat, Cobra or Dingo; it does import Bobcat and Cobra.\nBobcat can and does import both Cobra and Dingo, which both have project files with UUIDs and are declared as dependencies in Bobcat's [deps] section.\nBobcat cannot depend on Aardvark since Aardvark does not have a project file.\nCobra can and does import Dingo, which has a project file and UUID, and is declared as a dependency in Cobra's  [deps] section.\nCobra cannot depend on Aardvark or Bobcat since neither have real UUIDs.\nDingo cannot import anything because it has a project file without a [deps] section.\n\nThe paths map in a package directory is simple: it maps subdirectory names to their corresponding entry-point paths. In other words, if the path to our example project directory is /home/me/animals then the paths map could be represented by this dictionary:\n\npaths = Dict(\n    (UUID(\"00000000-0000-0000-0000-000000000000\"), :Aardvark) =>\n        \"/home/me/AnimalPackages/Aardvark/src/Aardvark.jl\",\n    (UUID(\"85ad11c7-31f6-5d08-84db-0a4914d4cadf\"), :Bobcat) =>\n        \"/home/me/AnimalPackages/Bobcat/src/Bobcat.jl\",\n    (UUID(\"4725e24d-f727-424b-bca0-c4307a3456fa\"), :Cobra) =>\n        \"/home/me/AnimalPackages/Cobra/src/Cobra.jl\",\n    (UUID(\"7a7925be-828c-4418-bbeb-bac8dfc843bc\"), :Dingo) =>\n        \"/home/me/AnimalPackages/Dingo/src/Dingo.jl\",\n)\n\nSince all packages in a package directory environment are, by definition, subdirectories with the expected entry-point files, their paths map entries always have this form."},{"location":"manual/code-loading.html#Environment-stacks","page":"Code Loading","title":"Environment stacks","category":"section","text":"The third and final kind of environment is one that combines other environments by overlaying several of them, making the packages in each available in a single composite environment. These composite environments are called environment stacks. The Julia LOAD_PATH global defines an environment stack—the environment in which the Julia process operates. If you want your Julia process to have access only to the packages in one project or package directory, make it the only entry in LOAD_PATH. It is often quite useful, however, to have access to some of your favorite tools—standard libraries, profilers, debuggers, personal utilities, etc.—even if they are not dependencies of the project you're working on. By adding an environment containing these tools to the load path, you immediately have access to them in top-level code without needing to add them to your project.\n\nThe mechanism for combining the roots, graph and paths data structures of the components of an environment stack is simple: they are merged as dictionaries, favoring earlier entries over later ones in the case of key collisions. In other words, if we have stack = [env₁, env₂, …] then we have:\n\nroots = reduce(merge, reverse([roots₁, roots₂, …]))\ngraph = reduce(merge, reverse([graph₁, graph₂, …]))\npaths = reduce(merge, reverse([paths₁, paths₂, …]))\n\nThe subscripted rootsᵢ, graphᵢ and pathsᵢ variables correspond to the subscripted environments, envᵢ, contained in stack. The reverse is present because merge favors the last argument rather than first when there are collisions between keys in its argument dictionaries. There are a couple of noteworthy features of this design:\n\nThe primary environment—i.e. the first environment in a stack—is faithfully embedded in a stacked environment. The full dependency graph of the first environment in a stack is guaranteed to be included intact in the stacked environment including the same versions of all dependencies.\nPackages in non-primary environments can end up using incompatible versions of their dependencies even if their own environments are entirely compatible. This can happen when one of their dependencies is shadowed by a version in an earlier environment in the stack (either by graph or path, or both).\n\nSince the primary environment is typically the environment of a project you're working on, while environments later in the stack contain additional tools, this is the right trade-off: it's better to break your development tools but keep the project working. When such incompatibilities occur, you'll typically want to upgrade your dev tools to versions that are compatible with the main project."},{"location":"manual/code-loading.html#man-extensions","page":"Code Loading","title":"Package Extensions","category":"section","text":"A package \"extension\" is a module that is automatically loaded when a specified set of other packages (its \"triggers\") are loaded in the current Julia session. Extensions are defined under the [extensions] section in the project file. The triggers of an extension are a subset of those packages listed under the [weakdeps] (and possibly, but uncommonly the [deps]) section of the project file. Those packages can have compat entries like other packages.\n\nname = \"MyPackage\"\n\n[compat]\nExtDep = \"1.0\"\nOtherExtDep = \"1.0\"\n\n[weakdeps]\nExtDep = \"c9a23...\" # uuid\nOtherExtDep = \"862e...\" # uuid\n\n[extensions]\nBarExt = [\"ExtDep\", \"OtherExtDep\"]\nFooExt = \"ExtDep\"\n...\n\nThe keys under extensions are the names of the extensions. They are loaded when all the packages on the right hand side (the triggers) of that extension are loaded. If an extension only has one trigger the list of triggers can be written as just a string for brevity. The location for the entry point of the extension is either in ext/FooExt.jl or ext/FooExt/FooExt.jl for extension FooExt. The content of an extension is often structured as:\n\nmodule FooExt\n\n# Load main package and triggers\nusing MyPackage, ExtDep\n\n# Extend functionality in main package with types from the triggers\nMyPackage.func(x::ExtDep.SomeStruct) = ...\n\nend\n\nWhen a package with extensions is added to an environment, the weakdeps and extensions sections are stored in the manifest file in the section for that package. The dependency lookup rules for a package are the same as for its \"parent\" except that the listed triggers are also considered as dependencies."},{"location":"manual/code-loading.html#workspaces","page":"Code Loading","title":"Workspaces","category":"section","text":"A project file can define a workspace by giving a set of projects that is part of that workspace:\n\n[workspace]\nprojects = [\"test\", \"benchmarks\", \"docs\", \"SomePackage\"]\n\nEach project listed in the projects array is specified by its relative path from the workspace root. This can be a direct child directory (e.g., \"test\") or a nested subdirectory (e.g., \"nested/subdir/MyPackage\"). Each project contains its own Project.toml file, which may include additional dependencies and compatibility constraints. In such cases, the package manager gathers all dependency information from all the projects in the workspace generating a single manifest file that combines the versions of all dependencies.\n\nWhen Julia loads a project, it searches upward through parent directories until it reaches the user's home directory to find a workspace that includes that project. This allows workspace projects to be nested at arbitrary depth within the workspace directory tree.\n\nFurthermore, workspaces can be \"nested\", meaning a project defining a workspace can also be part of another workspace. In this scenario, a single manifest file is still utilized, stored alongside the \"root project\" (the project that doesn't have another workspace including it). An example file structure could look like this:\n\nProject.toml # projects = [\"MyPackage\"]\nManifest.toml\nMyPackage/\n    Project.toml # projects = [\"test\"]\n    test/\n        Project.toml"},{"location":"manual/code-loading.html#preferences","page":"Code Loading","title":"Package/Environment Preferences","category":"section","text":"Preferences are dictionaries of metadata that influence package behavior within an environment. The preferences system supports reading preferences at compile-time, which means that at code-loading time, we must ensure that the precompilation files selected by Julia were built with the same preferences as the current environment before loading them. The public API for modifying Preferences is contained within the Preferences.jl package. Preferences are stored as TOML dictionaries within a (Julia)LocalPreferences.toml file next to the currently-active project. If a preference is \"exported\", it is instead stored within the (Julia)Project.toml instead. The intention is to allow shared projects to contain shared preferences, while allowing for users themselves to override those preferences with their own settings in the LocalPreferences.toml file, which should be .gitignored as the name implies.\n\nPreferences that are accessed during compilation are automatically marked as compile-time preferences, and any change recorded to these preferences will cause the Julia compiler to recompile any cached precompilation file(s) (.ji and corresponding .so, .dll, or .dylib files) for that module. This is done by serializing the hash of all compile-time preferences during compilation, then checking that hash against the current environment when searching for the proper file(s) to load.\n\nPreferences can be set with depot-wide defaults; if package Foo is installed within your global environment and it has preferences set, these preferences will apply as long as your global environment is part of your LOAD_PATH. Preferences in environments higher up in the environment stack get overridden by the more proximal entries in the load path, ending with the currently active project. This allows depot-wide preference defaults to exist, with active projects able to merge or even completely overwrite these inherited preferences. See the docstring for Preferences.set_preferences!() for the full details of how to set preferences to allow or disallow merging."},{"location":"manual/code-loading.html#Conclusion","page":"Code Loading","title":"Conclusion","category":"section","text":"Federated package management and precise software reproducibility are difficult but worthy goals in a package system. In combination, these goals lead to a more complex package loading mechanism than most dynamic languages have, but it also yields scalability and reproducibility that is more commonly associated with static languages. Typically, Julia users should be able to use the built-in package manager to manage their projects without needing a precise understanding of these interactions. A call to Pkg.add(\"X\") will add to the appropriate project and manifest files, selected via Pkg.activate(\"Y\"), so that a future call to import X will load X without further thought."},{"location":"devdocs/jit.html#JIT-Design-and-Implementation","page":"JIT Design and Implementation","title":"JIT Design and Implementation","category":"section","text":"This document explains the design and implementation of Julia's JIT, after codegen has finished and unoptimized LLVM IR has been produced. The JIT is responsible for optimizing and compiling this IR to machine code, and for linking it into the current process and making the code available for execution."},{"location":"devdocs/jit.html#Introduction","page":"JIT Design and Implementation","title":"Introduction","category":"section","text":"The JIT is responsible for managing compilation resources, looking up previously compiled code, and compiling new code. It is primarily built on LLVM's On-Request-Compilation (ORCv2) technology, which provides support for a number of useful features such as concurrent compilation, lazy compilation, and the ability to compile code in a separate process. Though LLVM provides a basic JIT compiler in the form of LLJIT, Julia uses many ORCv2 APIs directly to create its own custom JIT compiler."},{"location":"devdocs/jit.html#Overview","page":"JIT Design and Implementation","title":"Overview","category":"section","text":"(Image: Diagram of the compiler flow)\n\nCodegen produces an LLVM module containing IR for one or more Julia functions from the original Julia SSA IR produced by type inference (labeled as translate on the compiler diagram above). It also produces a mapping of code-instance to LLVM function name. However, though some optimizations have been applied by the Julia-based compiler on Julia IR, the LLVM IR produced by codegen still contains many opportunities for optimization. Thus, the first step the JIT takes is to run a target-independent optimization pipeline[tdp] on the LLVM module. Then, the JIT runs a target-dependent optimization pipeline, which includes target-specific optimizations and code generation, and outputs an object file. Finally, the JIT links the resulting object file into the current process and makes the code available for execution. All of this is controlled by code in src/jitlayers.cpp.\n\n[tdp]: This is not a totally-target independent pipeline, as transformations such as vectorization rely upon target information such as vector register width and cost modeling. Additionally, codegen itself makes a few target-dependent assumptions, and the optimization pipeline will take advantage of that knowledge.\n\nCurrently, only one thread at a time is permitted to enter the optimize-compile-link pipeline at a time, due to restrictions imposed by one of our linkers (RuntimeDyld). However, the JIT is designed to support concurrent optimization and compilation, and the linker restriction is expected to be lifted in the future when RuntimeDyld has been fully superseded on all platforms."},{"location":"devdocs/jit.html#Optimization-Pipeline","page":"JIT Design and Implementation","title":"Optimization Pipeline","category":"section","text":"The optimization pipeline is based off LLVM's new pass manager, but the pipeline is customized for Julia's needs. The pipeline is defined in src/pipeline.cpp, and broadly proceeds through a number of stages as detailed below.\n\nEarly Simplification\nThese passes are mainly used to simplify the IR and canonicalize patterns so that later passes can identify those patterns more easily. Additionally, various intrinsic calls such as branch prediction hints and annotations are lowered into other metadata or other IR features. SimplifyCFG (simplify control flow graph), DCE (dead code elimination), and SROA (scalar replacement of aggregates) are some of the key players here.\nEarly Optimization\nThese passes are typically cheap and are primarily focused around reducing the number of instructions in the IR and propagating knowledge to other instructions. For example, EarlyCSE is used to perform common subexpression elimination, and InstCombine and InstSimplify perform a number of small peephole optimizations to make operations less expensive.\nLoop Optimization\nThese passes canonicalize and simplify loops. Loops are often hot code, which makes loop optimization extremely important for performance. Key players here include LoopRotate, LICM, and LoopFullUnroll. Some bounds check elimination also happens here, as a result of the IRCE pass which can prove certain bounds are never exceeded.\nScalar Optimization\nThe scalar optimization pipeline contains a number of more expensive, but more powerful passes such as GVN (global value numbering), SCCP (sparse conditional constant propagation), and another round of bounds check elimination. These passes are expensive, but they can often remove large amounts of code and make vectorization much more successful and effective. Several other simplification and optimization passes intersperse the more expensive ones to reduce the amount of work they have to do.\nVectorization\nAutomatic vectorization is an extremely powerful transformation for CPU-intensive code. Briefly, vectorization allows execution of a single instruction on multiple data (SIMD), e.g. performing 8 addition operations at the same time. However, proving code to be both capable of vectorization and profitable to vectorize is difficult, and this relies heavily on the prior optimization passes to massage the IR into a state where vectorization is worth it.\nIntrinsic Lowering\nJulia inserts a number of custom intrinsics, for reasons such as object allocation, garbage collection, and exception handling. These intrinsics were originally placed to make optimization opportunities more obvious, but they are now lowered into LLVM IR to enable the IR to be emitted as machine code.\nCleanup\nThese passes are last-chance optimizations, and perform small optimizations such as fused multiply-add propagation and division-remainder simplification. Additionally, targets that do not support half-precision floating point numbers will have their half-precision instructions lowered into single-precision instructions here, and passes are added to provide sanitizer support."},{"location":"devdocs/jit.html#Target-Dependent-Optimization-and-Code-Generation","page":"JIT Design and Implementation","title":"Target-Dependent Optimization and Code Generation","category":"section","text":"LLVM provides target-dependent optimization and machine code generation in the same pipeline, located in the TargetMachine for a given platform. These passes include instruction selection, instruction scheduling, register allocation, and machine code emission. The LLVM documentation provides a good overview of the process, and the LLVM source code is the best place to look for details on the pipeline and passes."},{"location":"devdocs/jit.html#Linking","page":"JIT Design and Implementation","title":"Linking","category":"section","text":"Currently, Julia is transitioning between two linkers: the older RuntimeDyld linker, and the newer JITLink linker. JITLink contains a number of features that RuntimeDyld does not have, such as concurrent and reentrant linking, but currently lacks good support for profiling integrations and does not yet support all of the platforms that RuntimeDyld supports. Over time, JITLink is expected to replace RuntimeDyld entirely. Further details on JITLink can be found in the LLVM documentation."},{"location":"devdocs/jit.html#Execution","page":"JIT Design and Implementation","title":"Execution","category":"section","text":"Once the code has been linked into the current process, it is available for execution. This fact is made known to the generating codeinst by updating the invoke, specsigflags, and specptr fields appropriately. Codeinsts support upgrading invoke, specsigflags, and specptr fields, so long as every combination of these fields that exists at any given point in time is valid to be called. This allows the JIT to update these fields without invalidating existing codeinsts, supporting a potential future concurrent JIT. Specifically, the following states may be valid:\n\ninvoke is NULL, specsigflags is 0b00, specptr is NULL\nThis is the initial state of a codeinst, and indicates that the codeinst has not yet been compiled.\ninvoke is non-null, specsigflags is 0b00, specptr is NULL\nThis indicates that the codeinst was not compiled with any specialization, and that the codeinst should be invoked directly. Note that in this instance, invoke does not read either the specsigflags or specptr fields, and therefore they may be modified without invalidating the invoke pointer.\ninvoke is non-null, specsigflags is 0b10, specptr is non-null\nThis indicates that the codeinst was compiled, but a specialized function signature was deemed unnecessary by codegen.\ninvoke is non-null, specsigflags is 0b11, specptr is non-null\nThis indicates that the codeinst was compiled, and a specialized function signature was deemed necessary by codegen. The specptr field contains a pointer to the specialized function signature. The invoke pointer is permitted to read both specsigflags and specptr fields.\n\nIn addition, there are a number of different transitional states that occur during the update process. To account for these potential situations, the following write and read patterns should be used when dealing with these codeinst fields.\n\nWhen writing invoke, specsigflags, and specptr:\nPerform an atomic compare-exchange operation of specptr assuming the old value was NULL. This compare-exchange operation should have at least acquire-release ordering, to provide ordering guarantees of the remaining memory operations in the write.\nIf specptr was non-null, cease the write operation and wait for bit 0b10 of specsigflags to be written, then restart from step 1 if desired.\nWrite the new low bit of specsigflags to its final value. This may be a relaxed write.\nWrite the new invoke pointer to its final value. This must have at least a release memory ordering to synchronize with reads of invoke.\nSet the second bit of specsigflags to 1. This must be at least a release memory ordering to synchronize with reads of specsigflags. This step completes the write operation and announces to all other threads that all fields have been set.\nWhen reading all of invoke, specsigflags, and specptr:\nRead the specptr field with any memory ordering.\nRead the invoke field with at least an acquire memory ordering. This load will be referred to as initial_invoke.\nIf initial_invoke is NULL, the codeinst is not yet executable. invoke is NULL, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is NULL, then the initial_invoke pointer must not be relying on specptr to guarantee correct execution. Therefore, invoke is non-null, specsigflags may be treated as 0b00, specptr may be treated as NULL.\nIf specptr is non-null, then initial_invoke might not be the final invoke field that uses specptr. This can occur if specptr has been written, but invoke has not yet been written. Therefore, spin on the second bit of specsigflags until it is set to 1 with at least acquire memory ordering.\nRe-read the invoke field with any memory ordering. This load will be referred to as final_invoke.\nRead the specsigflags field with any memory ordering.\ninvoke is final_invoke, specsigflags is the value read in step 7, specptr is the value read in step 3.\nWhen updating a specptr to a different but equivalent function pointer:\nPerform a release store of the new function pointer to specptr. Races here must be benign, as the old function pointer is required to still be valid, and any new ones are also required to be valid as well. Once a pointer has been written to specptr, it must always be callable whether or not it is later overwritten.\n\nCorrectly reading these fields is implemented in jl_read_codeinst_invoke.\n\nAlthough these write, read, and update steps are complicated, they ensure that the JIT can update codeinsts without invalidating existing codeinsts, and that the JIT can update codeinsts without invalidating existing invoke pointers. This allows the JIT to potentially reoptimize functions at higher optimization levels in the future, and also will allow the JIT to support concurrent compilation of functions in the future."},{"location":"manual/profile.html#Profiling","page":"Profiling","title":"Profiling","category":"section","text":"The Profile module provides tools to help developers improve the performance of their code. When used, it takes measurements on running code, and produces output that helps you understand how much time is spent on individual line(s). The most common usage is to identify \"bottlenecks\" as targets for optimization.\n\nProfile implements what is known as a \"sampling\" or statistical profiler.  It works by periodically taking a backtrace during the execution of any task. Each backtrace captures the currently-running function and line number, plus the complete chain of function calls that led to this line, and hence is a \"snapshot\" of the current state of execution.\n\nIf much of your run time is spent executing a particular line of code, this line will show up frequently in the set of all backtraces. In other words, the \"cost\" of a given line–or really, the cost of the sequence of function calls up to and including this line–is proportional to how often it appears in the set of all backtraces.\n\nA sampling profiler does not provide complete line-by-line coverage, because the backtraces occur at intervals (by default, 1 ms on Unix systems and 10 ms on Windows, although the actual scheduling is subject to operating system load). Moreover, as discussed further below, because samples are collected at a sparse subset of all execution points, the data collected by a sampling profiler is subject to statistical noise.\n\nDespite these limitations, sampling profilers have substantial strengths:\n\nYou do not have to make any modifications to your code to take timing measurements.\nIt can profile into Julia's core code and even (optionally) into C and Fortran libraries.\nBy running \"infrequently\" there is very little performance overhead; while profiling, your code can run at nearly native speed.\n\nFor these reasons, it's recommended that you try using the built-in sampling profiler before considering any alternatives."},{"location":"manual/profile.html#Basic-usage","page":"Profiling","title":"Basic usage","category":"section","text":"Let's work with a simple test case:\n\njulia> function myfunc()\n           A = rand(200, 200, 400)\n           maximum(A)\n       end\n\nIt's a good idea to first run the code you intend to profile at least once (unless you want to profile Julia's JIT-compiler):\n\njulia> myfunc() # run once to force compilation\n\nNow we're ready to profile this function:\n\njulia> using Profile\n\njulia> @profile myfunc()\n\nTo see the profiling results, there are several graphical browsers. One \"family\" of visualizers is based on FlameGraphs.jl, with each family member providing a different user interface:\n\nVS Code is a full IDE with built-in support for profile visualization\nProfileView.jl is a stand-alone visualizer based on GTK\nProfileVega.jl uses VegaLight and integrates well with Jupyter notebooks\nStatProfilerHTML.jl produces HTML and presents some additional summaries, and also integrates well with Jupyter notebooks\nProfileSVG.jl renders SVG\nPProf.jl serves a local website for inspecting graphs, flamegraphs and more\nProfileCanvas.jl is a HTML canvas based profile viewer UI, used by the Julia VS Code extension, but can also generate interactive HTML files.\n\nAn entirely independent approach to profile visualization is PProf.jl, which uses the external pprof tool.\n\nHere, though, we'll use the text-based display that comes with the standard library:\n\njulia> Profile.print()\n80 ./event.jl:73; (::Base.REPL.##1#2{Base.REPL.REPLBackend})()\n 80 ./REPL.jl:97; macro expansion\n  80 ./REPL.jl:66; eval_user_input(::Any, ::Base.REPL.REPLBackend)\n   80 ./boot.jl:235; eval(::Module, ::Any)\n    80 ./<missing>:?; anonymous\n     80 ./profile.jl:23; macro expansion\n      52 ./REPL[1]:2; myfunc()\n       38 ./random.jl:431; rand!(::MersenneTwister, ::Array{Float64,3}, ::Int64, ::Type{B...\n        38 ./dSFMT.jl:84; dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_state, ::Ptr{F...\n       14 ./random.jl:278; rand\n        14 ./random.jl:277; rand\n         14 ./random.jl:366; rand\n          14 ./random.jl:369; rand\n      28 ./REPL[1]:3; myfunc()\n       28 ./reduce.jl:270; _mapreduce(::Base.#identity, ::Base.#scalarmax, ::IndexLinear,...\n        3  ./reduce.jl:426; mapreduce_impl(::Base.#identity, ::Base.#scalarmax, ::Array{F...\n        25 ./reduce.jl:428; mapreduce_impl(::Base.#identity, ::Base.#scalarmax, ::Array{F...\n\nEach line of this display represents a particular spot (line number) in the code. Indentation is used to indicate the nested sequence of function calls, with more-indented lines being deeper in the sequence of calls. In each line, the first \"field\" is the number of backtraces (samples) taken at this line or in any functions executed by this line. The second field is the file name and line number and the third field is the function name. Note that the specific line numbers may change as Julia's code changes; if you want to follow along, it's best to run this example yourself.\n\nIn this example, we can see that the top level function called is in the file event.jl. This is the function that runs the REPL when you launch Julia. If you examine line 97 of REPL.jl, you'll see this is where the function eval_user_input() is called. This is the function that evaluates what you type at the REPL, and since we're working interactively these functions were invoked when we entered @profile myfunc(). The next line reflects actions taken in the @profile macro.\n\nThe first line shows that 80 backtraces were taken at line 73 of event.jl, but it's not that this line was \"expensive\" on its own: the third line reveals that all 80 of these backtraces were actually triggered inside its call to eval_user_input, and so on. To find out which operations are actually taking the time, we need to look deeper in the call chain.\n\nThe first \"important\" line in this output is this one:\n\n52 ./REPL[1]:2; myfunc()\n\nREPL refers to the fact that we defined myfunc in the REPL, rather than putting it in a file; if we had used a file, this would show the file name. The [1] shows that the function myfunc was the first expression evaluated in this REPL session. Line 2 of myfunc() contains the call to rand, and there were 52 (out of 80) backtraces that occurred at this line. Below that, you can see a call to dsfmt_fill_array_close_open! inside dSFMT.jl.\n\nA little further down, you see:\n\n28 ./REPL[1]:3; myfunc()\n\nLine 3 of myfunc contains the call to maximum, and there were 28 (out of 80) backtraces taken here. Below that, you can see the specific places in base/reduce.jl that carry out the time-consuming operations in the maximum function for this type of input data.\n\nOverall, we can tentatively conclude that generating the random numbers is approximately twice as expensive as finding the maximum element. We could increase our confidence in this result by collecting more samples:\n\njulia> @profile (for i = 1:100; myfunc(); end)\n\njulia> Profile.print()\n[....]\n 3821 ./REPL[1]:2; myfunc()\n  3511 ./random.jl:431; rand!(::MersenneTwister, ::Array{Float64,3}, ::Int64, ::Type...\n   3511 ./dSFMT.jl:84; dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_state, ::Ptr...\n  310  ./random.jl:278; rand\n   [....]\n 2893 ./REPL[1]:3; myfunc()\n  2893 ./reduce.jl:270; _mapreduce(::Base.#identity, ::Base.#scalarmax, ::IndexLinea...\n   [....]\n\nIn general, if you have N samples collected at a line, you can expect an uncertainty on the order of sqrt(N) (barring other sources of noise, like how busy the computer is with other tasks). The major exception to this rule is garbage collection, which runs infrequently but tends to be quite expensive. (Since Julia's garbage collector is written in C, such events can be detected using the C=true output mode described below, or by using ProfileView.jl.)\n\nThis illustrates the default \"tree\" dump; an alternative is the \"flat\" dump, which accumulates counts independent of their nesting:\n\njulia> Profile.print(format=:flat)\n Count File          Line Function\n  6714 ./<missing>     -1 anonymous\n  6714 ./REPL.jl       66 eval_user_input(::Any, ::Base.REPL.REPLBackend)\n  6714 ./REPL.jl       97 macro expansion\n  3821 ./REPL[1]        2 myfunc()\n  2893 ./REPL[1]        3 myfunc()\n  6714 ./REPL[7]        1 macro expansion\n  6714 ./boot.jl      235 eval(::Module, ::Any)\n  3511 ./dSFMT.jl      84 dsfmt_fill_array_close_open!(::Base.dSFMT.DSFMT_s...\n  6714 ./event.jl      73 (::Base.REPL.##1#2{Base.REPL.REPLBackend})()\n  6714 ./profile.jl    23 macro expansion\n  3511 ./random.jl    431 rand!(::MersenneTwister, ::Array{Float64,3}, ::In...\n   310 ./random.jl    277 rand\n   310 ./random.jl    278 rand\n   310 ./random.jl    366 rand\n   310 ./random.jl    369 rand\n  2893 ./reduce.jl    270 _mapreduce(::Base.#identity, ::Base.#scalarmax, :...\n     5 ./reduce.jl    420 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n   253 ./reduce.jl    426 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n  2592 ./reduce.jl    428 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n    43 ./reduce.jl    429 mapreduce_impl(::Base.#identity, ::Base.#scalarma...\n\nIf your code has recursion, one potentially-confusing point is that a line in a \"child\" function can accumulate more counts than there are total backtraces. Consider the following function definitions:\n\ndumbsum(n::Integer) = n == 1 ? 1 : 1 + dumbsum(n-1)\ndumbsum3() = dumbsum(3)\n\nIf you were to profile dumbsum3, and a backtrace was taken while it was executing dumbsum(1), the backtrace would look like this:\n\ndumbsum3\n    dumbsum(3)\n        dumbsum(2)\n            dumbsum(1)\n\nConsequently, this child function gets 3 counts, even though the parent only gets one. The \"tree\" representation makes this much clearer, and for this reason (among others) is probably the most useful way to view the results."},{"location":"manual/profile.html#Accumulation-and-clearing","page":"Profiling","title":"Accumulation and clearing","category":"section","text":"Results from @profile accumulate in a buffer; if you run multiple pieces of code under @profile, then Profile.print() will show you the combined results. This can be very useful, but sometimes you want to start fresh; you can do so with Profile.clear()."},{"location":"manual/profile.html#Options-for-controlling-the-display-of-profile-results","page":"Profiling","title":"Options for controlling the display of profile results","category":"section","text":"Profile.print has more options than we've described so far. Let's see the full declaration:\n\nfunction print(io::IO = stdout, data = fetch(); kwargs...)\n\nLet's first discuss the two positional arguments, and later the keyword arguments:\n\nio – Allows you to save the results to a buffer, e.g. a file, but the default is to print to stdout (the console).\ndata – Contains the data you want to analyze; by default that is obtained from Profile.fetch(), which pulls out the backtraces from a pre-allocated buffer. For example, if you want to profile the profiler, you could say:\ndata = copy(Profile.fetch())\nProfile.clear()\n@profile Profile.print(stdout, data) # Prints the previous results\nProfile.print()                      # Prints results from Profile.print()\n\nThe keyword arguments can be any combination of:\n\nformat – Introduced above, determines whether backtraces are printed  with (default, :tree) or without (:flat) indentation indicating tree  structure.\nC – If true, backtraces from C and Fortran code are shown (normally they are excluded). Try running the introductory example with Profile.print(C = true). This can be extremely helpful in deciding whether it's Julia code or C code that is causing a bottleneck; setting C = true also improves the interpretability of the nesting, at the cost of longer profile dumps.\ncombine – Some lines of code contain multiple operations; for example, s += A[i] contains both an array reference (A[i]) and a sum operation. These correspond to different lines in the generated machine code, and hence there may be two or more different addresses captured during backtraces on this line. combine = true lumps them together, and is probably what you typically want, but you can generate an output separately for each unique instruction pointer with combine = false.\nmaxdepth – Limits frames at a depth higher than maxdepth in the :tree format.\nsortedby – Controls the order in :flat format. :filefuncline (default) sorts by the source line, whereas :count sorts in order of number of collected samples.\nnoisefloor – Limits frames that are below the heuristic noise floor of the sample (only applies to format :tree). A suggested value to try for this is 2.0 (the default is 0). This parameter hides samples for which n <= noisefloor * √N, where n is the number of samples on this line, and N is the number of samples for the callee.\nmincount – Limits frames with less than mincount occurrences.\n\nFile/function names are sometimes truncated (with ...), and indentation is truncated with a +n at the beginning, where n is the number of extra spaces that would have been inserted, had there been room. If you want a complete profile of deeply-nested code, often a good idea is to save to a file using a wide displaysize in an IOContext:\n\nopen(\"/tmp/prof.txt\", \"w\") do s\n    Profile.print(IOContext(s, :displaysize => (24, 500)))\nend"},{"location":"manual/profile.html#Configuration","page":"Profiling","title":"Configuration","category":"section","text":"@profile just accumulates backtraces, and the analysis happens when you call Profile.print(). For a long-running computation, it's entirely possible that the pre-allocated buffer for storing backtraces will be filled. If that happens, the backtraces stop but your computation continues. As a consequence, you may miss some important profiling data (you will get a warning when that happens).\n\nYou can obtain and configure the relevant parameters this way:\n\nProfile.init() # returns the current settings\nProfile.init(n = 10^7, delay = 0.01)\n\nn is the total number of instruction pointers you can store, with a default value of 10^6. If your typical backtrace is 20 instruction pointers, then you can collect 50000 backtraces, which suggests a statistical uncertainty of less than 1%. This may be good enough for most applications.\n\nConsequently, you are more likely to need to modify delay, expressed in seconds, which sets the amount of time that Julia gets between snapshots to perform the requested computations. A very long-running job might not need frequent backtraces. The default setting is delay = 0.001. Of course, you can decrease the delay as well as increase it; however, the overhead of profiling grows once the delay becomes similar to the amount of time needed to take a backtrace (~30 microseconds on the author's laptop)."},{"location":"manual/profile.html#Wall-time-Profiler","page":"Profiling","title":"Wall-time Profiler","category":"section","text":""},{"location":"manual/profile.html#Introduction-and-Problem-Motivation","page":"Profiling","title":"Introduction & Problem Motivation","category":"section","text":"The profiler described in the previous section is a sampling CPU profiler. At a high level, the profiler periodically stops all Julia compute threads to collect their backtraces and estimates the time spent in each function based on the number of backtrace samples that include a frame from that function. However, note that only tasks currently running on system threads just before the profiler stops them will have their backtraces collected.\n\nWhile this profiler is typically well-suited for workloads where the majority of tasks are compute-bound, it is less helpful for systems where most tasks are IO-heavy or for diagnosing contention on synchronization primitives in your code.\n\nLet's consider this simple workload:\n\nusing Base.Threads\nusing Profile\nusing PProf\n\nch = Channel(1)\n\nconst N_SPAWNED_TASKS = (1 << 10)\nconst WAIT_TIME_NS = 10_000_000\n\nfunction spawn_a_bunch_of_tasks_waiting_on_channel()\n    for i in 1:N_SPAWNED_TASKS\n        Threads.@spawn begin\n            take!(ch)\n        end\n    end\nend\n\nfunction busywait()\n    t0 = time_ns()\n    while true\n        if time_ns() - t0 > WAIT_TIME_NS\n            break\n        end\n    end\nend\n\nfunction main()\n    spawn_a_bunch_of_tasks_waiting_on_channel()\n    for i in 1:N_SPAWNED_TASKS\n        put!(ch, i)\n        busywait()\n    end\nend\n\nProfile.@profile main()\n\nOur goal is to detect whether there is contention on the ch channel—i.e., whether the number of waiters is excessive given the rate at which work items are being produced in the channel.\n\nIf we run this, we obtain the following PProf flame graph:\n\n(Image: CPU Profile)\n\nThis profile provides no information to help determine where contention occurs in the system’s synchronization primitives. Waiters on a channel will be blocked and descheduled, meaning no system thread will be running the tasks assigned to those waiters, and as a result, they won't be sampled by the profiler."},{"location":"manual/profile.html#Wall-time-Profiler-2","page":"Profiling","title":"Wall-time Profiler","category":"section","text":"Instead of sampling threads—and thus only sampling tasks that are running—a wall-time task profiler samples tasks independently of their scheduling state. For example, tasks that are sleeping on a synchronization primitive at the time the profiler is running will be sampled with the same probability as tasks that were actively running when the profiler attempted to capture backtraces.\n\nThis approach allows us to construct a profile where backtraces from tasks blocked on the ch channel, as in the example above, are actually represented.\n\nLet's run the same example, but now with a wall-time profiler:\n\nusing Base.Threads\nusing Profile\nusing PProf\n\nch = Channel(1)\n\nconst N_SPAWNED_TASKS = (1 << 10)\nconst WAIT_TIME_NS = 10_000_000\n\nfunction spawn_a_bunch_of_tasks_waiting_on_channel()\n    for i in 1:N_SPAWNED_TASKS\n        Threads.@spawn begin\n            take!(ch)\n        end\n    end\nend\n\nfunction busywait()\n    t0 = time_ns()\n    while true\n        if time_ns() - t0 > WAIT_TIME_NS\n            break\n        end\n    end\nend\n\nfunction main()\n    spawn_a_bunch_of_tasks_waiting_on_channel()\n    for i in 1:N_SPAWNED_TASKS\n        put!(ch, i)\n        busywait()\n    end\nend\n\nProfile.@profile_walltime main()\n\nWe obtain the following flame graph:\n\n(Image: Wall-time Profile Channel)\n\nWe see that a large number of samples come from channel-related take! functions, which allows us to determine that there is indeed an excessive number of waiters in ch."},{"location":"manual/profile.html#A-Compute-Bound-Workload","page":"Profiling","title":"A Compute-Bound Workload","category":"section","text":"Despite the wall-time profiler sampling all live tasks in the system and not just the currently running ones, it can still be helpful for identifying performance hotspots, even if your code is compute-bound. Let’s consider a simple example:\n\nusing Base.Threads\nusing Profile\nusing PProf\n\nch = Channel(1)\n\nconst MAX_ITERS = (1 << 22)\nconst N_TASKS = (1 << 12)\n\nfunction spawn_a_task_waiting_on_channel()\n    Threads.@spawn begin\n        take!(ch)\n    end\nend\n\nfunction sum_of_sqrt()\n    sum_of_sqrt = 0.0\n    for i in 1:MAX_ITERS\n        sum_of_sqrt += sqrt(i)\n    end\n    return sum_of_sqrt\nend\n\nfunction spawn_a_bunch_of_compute_heavy_tasks()\n    Threads.@sync begin\n        for i in 1:N_TASKS\n            Threads.@spawn begin\n                sum_of_sqrt()\n            end\n        end\n    end\nend\n\nfunction main()\n    spawn_a_task_waiting_on_channel()\n    spawn_a_bunch_of_compute_heavy_tasks()\nend\n\nProfile.@profile_walltime main()\n\nAfter collecting a wall-time profile, we get the following flame graph:\n\n(Image: Wall-time Profile Compute-Bound)\n\nNotice how many of the samples contain sum_of_sqrt, which is the expensive compute function in our example."},{"location":"manual/profile.html#Identifying-Task-Sampling-Failures-in-your-Profile","page":"Profiling","title":"Identifying Task Sampling Failures in your Profile","category":"section","text":"In the current implementation, the wall-time profiler attempts to sample from tasks that have been alive since the last garbage collection, along with those created afterward. However, if most tasks are extremely short-lived, you may end up sampling tasks that have already completed, resulting in missed backtrace captures.\n\nIf you encounter samples containing failed_to_sample_task_fun or failed_to_stop_thread_fun, this likely indicates a high volume of short-lived tasks, which prevented their backtraces from being collected.\n\nLet's consider this simple example:\n\nusing Base.Threads\nusing Profile\nusing PProf\n\nconst N_SPAWNED_TASKS = (1 << 16)\nconst WAIT_TIME_NS = 100_000\n\nfunction spawn_a_bunch_of_short_lived_tasks()\n    for i in 1:N_SPAWNED_TASKS\n        Threads.@spawn begin\n            # Do nothing\n        end\n    end\nend\n\nfunction busywait()\n    t0 = time_ns()\n    while true\n        if time_ns() - t0 > WAIT_TIME_NS\n            break\n        end\n    end\nend\n\nfunction main()\n    GC.enable(false)\n    spawn_a_bunch_of_short_lived_tasks()\n    for i in 1:N_SPAWNED_TASKS\n        busywait()\n    end\n    GC.enable(true)\nend\n\nProfile.@profile_walltime main()\n\nNotice that the tasks spawned in spawn_a_bunch_of_short_lived_tasks are extremely short-lived. Since these tasks constitute the majority in the system, we will likely miss capturing a backtrace for most sampled tasks.\n\nAfter collecting a wall-time profile, we obtain the following flame graph:\n\n(Image: Task Sampling Failure)\n\nThe large number of samples from failed_to_stop_thread_fun confirms that we have a significant number of short-lived tasks in the system."},{"location":"manual/profile.html#Memory-allocation-analysis","page":"Profiling","title":"Memory allocation analysis","category":"section","text":"One of the most common techniques to improve performance is to reduce memory allocation. Julia provides several tools to measure this:"},{"location":"manual/profile.html#@time","page":"Profiling","title":"@time","category":"section","text":"The total amount of allocation can be measured with @time, @allocated and @allocations, and specific lines triggering allocation can often be inferred from profiling via the cost of garbage collection that these lines incur. However, sometimes it is more efficient to directly measure the amount of memory allocated by each line of code."},{"location":"manual/profile.html#GC-Logging","page":"Profiling","title":"GC Logging","category":"section","text":"While @time logs high-level stats about memory usage and garbage collection over the course of evaluating an expression, it can be useful to log each garbage collection event, to get an intuitive sense of how often the garbage collector is running, how long it's running each time, and how much garbage it collects each time. This can be enabled with GC.enable_logging(true), which causes Julia to log to stderr every time a garbage collection happens."},{"location":"manual/profile.html#allocation-profiler","page":"Profiling","title":"Allocation Profiler","category":"section","text":"compat: Julia 1.8\nThis functionality requires at least Julia 1.8.\n\nThe allocation profiler records the stack trace, type, and size of each allocation while it is running. It can be invoked with Profile.Allocs.@profile.\n\nThis information about the allocations is returned as an array of Alloc objects, wrapped in an AllocResults object. The best way to visualize these is currently with the PProf.jl and ProfileCanvas.jl packages, which can visualize the call stacks which are making the most allocations.\n\nThe allocation profiler does have significant overhead, so a sample_rate argument can be passed to speed it up by making it skip some allocations. Passing sample_rate=1.0 will make it record everything (which is slow); sample_rate=0.1 will record only 10% of the allocations (faster), etc.\n\ncompat: Julia 1.11\nOlder versions of Julia could not capture types in all cases. In older versions of Julia, if you see an allocation of type Profile.Allocs.UnknownType, it means that the profiler doesn't know what type of object was allocated. This mainly happened when the allocation was coming from generated code produced by the compiler. See issue #43688 for more info.Since Julia 1.11, all allocations should have a type reported.\n\nFor more details on how to use this tool, please see the talk from JuliaCon 2022."},{"location":"manual/profile.html#Allocation-Profiler-Example","page":"Profiling","title":"Allocation Profiler Example","category":"section","text":"In this simple example, we use PProf to visualize the alloc profile. You could use another visualization tool instead. We collect the profile (specifying a sample rate), then we visualize it.\n\nusing Profile, PProf\nProfile.Allocs.clear()\nProfile.Allocs.@profile sample_rate=0.0001 my_function()\nPProf.Allocs.pprof()\n\nHere is a more in-depth example, showing how we can tune the sample rate. A good number of samples to aim for is around 1 - 10 thousand. Too many, and the profile visualizer can get overwhelmed, and profiling will be slow. Too few, and you don't have a representative sample.\n\njulia> import Profile\n\njulia> @time my_function()  # Estimate allocations from a (second-run) of the function\n  0.110018 seconds (1.50 M allocations: 58.725 MiB, 17.17% gc time)\n500000\n\njulia> Profile.Allocs.clear()\n\njulia> Profile.Allocs.@profile sample_rate=0.001 begin   # 1.5 M * 0.001 = ~1.5K allocs.\n           my_function()\n       end\n500000\n\njulia> prof = Profile.Allocs.fetch();  # If you want, you can also manually inspect the results.\n\njulia> length(prof.allocs)  # Confirm we have expected number of allocations.\n1515\n\njulia> using PProf  # Now, visualize with an external tool, like PProf or ProfileCanvas.\n\njulia> PProf.Allocs.pprof(prof; from_c=false)  # You can optionally pass in a previously fetched profile result.\nAnalyzing 1515 allocation samples... 100%|████████████████████████████████| Time: 0:00:00\nMain binary filename not available.\nServing web UI on http://localhost:62261\n\"alloc-profile.pb.gz\"\n\nThen you can view the profile by navigating to http://localhost:62261, and the profile is saved to disk. See PProf package for more options."},{"location":"manual/profile.html#Allocation-Profiling-Tips","page":"Profiling","title":"Allocation Profiling Tips","category":"section","text":"As stated above, aim for around 1-10 thousand samples in your profile.\n\nNote that we are uniformly sampling in the space of all allocations, and are not weighting our samples by the size of the allocation. So a given allocation profile may not give a representative profile of where most bytes are allocated in your program, unless you had set sample_rate=1.\n\nAllocations can come from users directly constructing objects, but can also come from inside the runtime or be inserted into compiled code to handle type instability. Looking at the \"source code\" view can be helpful to isolate them, and then other external tools such as Cthulhu.jl can be useful for identifying the cause of the allocation."},{"location":"manual/profile.html#Allocation-Profile-Visualization-Tools","page":"Profiling","title":"Allocation Profile Visualization Tools","category":"section","text":"There are several profiling visualization tools now that can all display Allocation Profiles. Here is a small list of some of the main ones we know about:\n\nPProf.jl\nProfileCanvas.jl\nVSCode's built-in profile visualizer (@profview_allocs) [docs needed]\nViewing the results directly in the REPL\nYou can inspect the results in the REPL via Profile.Allocs.fetch(), to view the stacktrace and type of each allocation."},{"location":"manual/profile.html#Line-by-Line-Allocation-Tracking","page":"Profiling","title":"Line-by-Line Allocation Tracking","category":"section","text":"An alternative way to measure allocations is to start Julia with the --track-allocation=<setting> command-line option, for which you can choose none (the default, do not measure allocation), user (measure memory allocation everywhere except Julia's core code), or all (measure memory allocation at each line of Julia code). Allocation gets measured for each line of compiled code. When you quit Julia, the cumulative results are written to text files with .mem appended after the file name, residing in the same directory as the source file. Each line lists the total number of bytes allocated. The Coverage package contains some elementary analysis tools, for example to sort the lines in order of number of bytes allocated.\n\nIn interpreting the results, there are a few important details. Under the user setting, the first line of any function directly called from the REPL will exhibit allocation due to events that happen in the REPL code itself. More significantly, JIT-compilation also adds to allocation counts, because much of Julia's compiler is written in Julia (and compilation usually requires memory allocation). The recommended procedure is to force compilation by executing all the commands you want to analyze, then call Profile.clear_malloc_data() to reset all allocation counters.  Finally, execute the desired commands and quit Julia to trigger the generation of the .mem files.\n\nnote: Note\n--track-allocation changes code generation to log the allocations, and so the allocations may be different than what happens without the option. We recommend using the allocation profiler instead."},{"location":"manual/profile.html#External-Profiling","page":"Profiling","title":"External Profiling","category":"section","text":"Currently Julia supports Intel VTune, OProfile and perf as external profiling tools.\n\nDepending on the tool you choose, compile with USE_INTEL_JITEVENTS, USE_OPROFILE_JITEVENTS and USE_PERF_JITEVENTS set to 1 in Make.user. Multiple flags are supported.\n\nBefore running Julia set the environment variable ENABLE_JITPROFILING to 1.\n\nNow you have a multitude of ways to employ those tools! For example with OProfile you can try a simple recording :\n\n>ENABLE_JITPROFILING=1 sudo operf -Vdebug ./julia test/fastmath.jl\n>opreport -l `which ./julia`\n\nOr similarly with perf :\n\n$ ENABLE_JITPROFILING=1 perf record -o /tmp/perf.data --call-graph dwarf -k 1 ./julia /test/fastmath.jl\n$ perf inject --jit --input /tmp/perf.data --output /tmp/perf-jit.data\n$ perf report --call-graph -G -i /tmp/perf-jit.data\n\nThere are many more interesting things that you can measure about your program, to get a comprehensive list please read the Linux perf examples page.\n\nRemember that perf saves for each execution a perf.data file that, even for small programs, can get quite large. Also the perf LLVM module saves temporarily debug objects in ~/.debug/jit, remember to clean that folder frequently."},{"location":"stdlib/Distributed.html#man-distributed","page":"Distributed Computing","title":"Distributed Computing","category":"section","text":""},{"location":"stdlib/Distributed.html#Cluster-Manager-Interface","page":"Distributed Computing","title":"Cluster Manager Interface","category":"section","text":"This interface provides a mechanism to launch and manage Julia workers on different cluster environments. There are two types of managers present in Base: LocalManager, for launching additional workers on the same host, and SSHManager, for launching on remote hosts via ssh. TCP/IP sockets are used to connect and transport messages between processes. It is possible for Cluster Managers to provide a different transport."},{"location":"stdlib/Distributed.html#Distributed","page":"Distributed Computing","title":"Distributed","category":"module","text":"Tools for distributed parallel processing.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.addprocs","page":"Distributed Computing","title":"Distributed.addprocs","category":"function","text":"addprocs(manager::ClusterManager; kwargs...) -> List of process identifiers\n\nLaunches worker processes via the specified cluster manager.\n\nFor example, Beowulf clusters are supported via a custom cluster manager implemented in the package ClusterManagers.jl.\n\nThe number of seconds a newly launched worker waits for connection establishment from the master can be specified via variable JULIA_WORKER_TIMEOUT in the worker process's environment. Relevant only when using TCP/IP as transport.\n\nTo launch workers without blocking the REPL, or the containing function if launching workers programmatically, execute addprocs in its own task.\n\nExamples\n\n# On busy clusters, call `addprocs` asynchronously\nt = @async addprocs(...)\n\n# Utilize workers as and when they come online\nif nprocs() > 1   # Ensure at least one new worker is available\n   ....   # perform distributed execution\nend\n\n# Retrieve newly launched worker IDs, or any error messages\nif istaskdone(t)   # Check if `addprocs` has completed to ensure `fetch` doesn't block\n    if nworkers() == N\n        new_pids = fetch(t)\n    else\n        fetch(t)\n    end\nend\n\n\n\n\n\naddprocs(machines; tunnel=false, sshflags=``, max_parallel=10, kwargs...) -> List of process identifiers\n\nAdd worker processes on remote machines via SSH. Configuration is done with keyword arguments (see below). In particular, the exename keyword can be used to specify the path to the julia binary on the remote machine(s).\n\nmachines is a vector of \"machine specifications\" which are given as strings of the form [user@]host[:port] [bind_addr[:port]]. user defaults to current user and port to the standard SSH port. If [bind_addr[:port]] is specified, other workers will connect to this worker at the specified bind_addr and port.\n\nIt is possible to launch multiple processes on a remote host by using a tuple in the machines vector or the form (machine_spec, count), where count is the number of workers to be launched on the specified host. Passing :auto as the worker count will launch as many workers as the number of CPU threads on the remote host.\n\nExamples:\n\naddprocs([\n    \"remote1\",               # one worker on 'remote1' logging in with the current username\n    \"user@remote2\",          # one worker on 'remote2' logging in with the 'user' username\n    \"user@remote3:2222\",     # specifying SSH port to '2222' for 'remote3'\n    (\"user@remote4\", 4),     # launch 4 workers on 'remote4'\n    (\"user@remote5\", :auto), # launch as many workers as CPU threads on 'remote5'\n])\n\nKeyword arguments:\n\ntunnel: if true then SSH tunneling will be used to connect to the worker from the master process. Default is false.\nmultiplex: if true then SSH multiplexing is used for SSH tunneling. Default is false.\nssh: the name or path of the SSH client executable used to start the workers. Default is \"ssh\".\nsshflags: specifies additional ssh options, e.g. sshflags=`-i /home/foo/bar.pem`\nmax_parallel: specifies the maximum number of workers connected to in parallel at a host. Defaults to 10.\nshell: specifies the type of shell to which ssh connects on the workers.\nshell=:posix: a POSIX-compatible Unix/Linux shell (sh, ksh, bash, dash, zsh, etc.). The default.\nshell=:csh: a Unix C shell (csh, tcsh).\nshell=:wincmd: Microsoft Windows cmd.exe.\ndir: specifies the working directory on the workers. Defaults to the host's current directory (as found by pwd())\nenable_threaded_blas: if true then  BLAS will run on multiple threads in added processes. Default is false.\nexename: name of the julia executable. Defaults to \"$(Sys.BINDIR)/julia\" or \"$(Sys.BINDIR)/julia-debug\" as the case may be. It is recommended that a common Julia version is used on all remote machines because serialization and code distribution might fail otherwise.\nexeflags: additional flags passed to the worker processes. It can either be a Cmd, a String holding one flag, or a collection of strings, with one element per flag. E.g. --threads=auto project=, \"--compile-trace=stderr\" or [\"--threads=auto\", \"--compile=all\"].\ntopology: Specifies how the workers connect to each other. Sending a message between unconnected workers results in an error.\ntopology=:all_to_all: All processes are connected to each other. The default.\ntopology=:master_worker: Only the driver process, i.e. pid 1 connects to the workers. The workers do not connect to each other.\ntopology=:custom: The launch method of the cluster manager specifies the connection topology via fields ident and connect_idents in WorkerConfig. A worker with a cluster manager identity ident will connect to all workers specified in connect_idents.\nlazy: Applicable only with topology=:all_to_all. If true, worker-worker connections are setup lazily, i.e. they are setup at the first instance of a remote call between workers. Default is true.\nenv: provide an array of string pairs such as env=[\"JULIA_DEPOT_PATH\"=>\"/depot\"] to request that environment variables are set on the remote machine. By default only the environment variable JULIA_WORKER_TIMEOUT is passed automatically from the local to the remote environment.\ncmdline_cookie: pass the authentication cookie via the --worker commandline  option. The (more secure) default behaviour of passing the cookie via ssh stdio  may hang with Windows workers that use older (pre-ConPTY) Julia or Windows versions,  in which case cmdline_cookie=true offers a work-around.\n\ncompat: Julia 1.6\nThe keyword arguments ssh, shell, env and cmdline_cookie were added in Julia 1.6.\n\nEnvironment variables:\n\nIf the master process fails to establish a connection with a newly launched worker within 60.0 seconds, the worker treats it as a fatal situation and terminates. This timeout can be controlled via environment variable JULIA_WORKER_TIMEOUT. The value of JULIA_WORKER_TIMEOUT on the master process specifies the number of seconds a newly launched worker waits for connection establishment.\n\n\n\n\n\naddprocs(np::Integer=Sys.CPU_THREADS; restrict=true, kwargs...) -> List of process identifiers\n\nLaunch np workers on the local host using the in-built LocalManager.\n\nLocal workers inherit the current package environment (i.e., active project, LOAD_PATH, and DEPOT_PATH) from the main process.\n\nwarning: Warning\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes.\n\nKeyword arguments:\n\nrestrict::Bool: if true (default) binding is restricted to 127.0.0.1.\ndir, exename, exeflags, env, topology, lazy, enable_threaded_blas: same effect as for SSHManager, see documentation for addprocs(machines::AbstractVector).\n\ncompat: Julia 1.9\nThe inheriting of the package environment and the env keyword argument were added in Julia 1.9.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.nprocs","page":"Distributed Computing","title":"Distributed.nprocs","category":"function","text":"nprocs()\n\nGet the number of available processes.\n\nExamples\n\njulia> nprocs()\n3\n\njulia> workers()\n2-element Array{Int64,1}:\n 2\n 3\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.nworkers","page":"Distributed Computing","title":"Distributed.nworkers","category":"function","text":"nworkers()\n\nGet the number of available worker processes. This is one less than nprocs(). Equal to nprocs() if nprocs() == 1.\n\nExamples\n\n$ julia -p 2\n\njulia> nprocs()\n3\n\njulia> nworkers()\n2\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.procs-Tuple{}","page":"Distributed Computing","title":"Distributed.procs","category":"method","text":"procs()\n\nReturn a list of all process identifiers, including pid 1 (which is not included by workers()).\n\nExamples\n\n$ julia -p 2\n\njulia> procs()\n3-element Array{Int64,1}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.procs-Tuple{Integer}","page":"Distributed Computing","title":"Distributed.procs","category":"method","text":"procs(pid::Integer)\n\nReturn a list of all process identifiers on the same physical node. Specifically all workers bound to the same ip-address as pid are returned.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.workers","page":"Distributed Computing","title":"Distributed.workers","category":"function","text":"workers()\n\nReturn a list of all worker process identifiers.\n\nExamples\n\n$ julia -p 2\n\njulia> workers()\n2-element Array{Int64,1}:\n 2\n 3\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.rmprocs","page":"Distributed Computing","title":"Distributed.rmprocs","category":"function","text":"rmprocs(pids...; waitfor=typemax(Int))\n\nRemove the specified workers. Note that only process 1 can add or remove workers.\n\nArgument waitfor specifies how long to wait for the workers to shut down:\n\nIf unspecified, rmprocs will wait until all requested pids are removed.\nAn ErrorException is raised if all workers cannot be terminated before the requested waitfor seconds.\nWith a waitfor value of 0, the call returns immediately with the workers scheduled for removal in a different task. The scheduled Task object is returned. The user should call wait on the task before invoking any other parallel calls.\n\nExamples\n\n$ julia -p 5\n\njulia> t = rmprocs(2, 3, waitfor=0)\nTask (runnable) @0x0000000107c718d0\n\njulia> wait(t)\n\njulia> workers()\n3-element Array{Int64,1}:\n 4\n 5\n 6\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.interrupt","page":"Distributed Computing","title":"Distributed.interrupt","category":"function","text":"interrupt(pids::Integer...)\n\nInterrupt the current executing task on the specified workers. This is equivalent to pressing Ctrl-C on the local machine. If no arguments are given, all workers are interrupted.\n\n\n\n\n\ninterrupt(pids::AbstractVector=workers())\n\nInterrupt the current executing task on the specified workers. This is equivalent to pressing Ctrl-C on the local machine. If no arguments are given, all workers are interrupted.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.myid","page":"Distributed Computing","title":"Distributed.myid","category":"function","text":"myid()\n\nGet the id of the current process.\n\nExamples\n\njulia> myid()\n1\n\njulia> remotecall_fetch(() -> myid(), 4)\n4\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.pmap","page":"Distributed Computing","title":"Distributed.pmap","category":"function","text":"pmap(f, [::AbstractWorkerPool], c...; distributed=true, batch_size=1, on_error=nothing, retry_delays=[], retry_check=nothing) -> collection\n\nTransform collection c by applying f to each element using available workers and tasks.\n\nFor multiple collection arguments, apply f elementwise.\n\nNote that f must be made available to all worker processes; see Code Availability and Loading Packages for details.\n\nIf a worker pool is not specified all available workers will be used via a CachingPool.\n\nBy default, pmap distributes the computation over all specified workers. To use only the local process and distribute over tasks, specify distributed=false. This is equivalent to using asyncmap. For example, pmap(f, c; distributed=false) is equivalent to asyncmap(f,c; ntasks=()->nworkers())\n\npmap can also use a mix of processes and tasks via the batch_size argument. For batch sizes greater than 1, the collection is processed in multiple batches, each of length batch_size or less. A batch is sent as a single request to a free worker, where a local asyncmap processes elements from the batch using multiple concurrent tasks.\n\nAny error stops pmap from processing the remainder of the collection. To override this behavior you can specify an error handling function via argument on_error which takes in a single argument, i.e., the exception. The function can stop the processing by rethrowing the error, or, to continue, return any value which is then returned inline with the results to the caller.\n\nConsider the following two examples. The first one returns the exception object inline, the second a 0 in place of any exception:\n\njulia> pmap(x->iseven(x) ? error(\"foo\") : x, 1:4; on_error=identity)\n4-element Array{Any,1}:\n 1\n  ErrorException(\"foo\")\n 3\n  ErrorException(\"foo\")\n\njulia> pmap(x->iseven(x) ? error(\"foo\") : x, 1:4; on_error=ex->0)\n4-element Array{Int64,1}:\n 1\n 0\n 3\n 0\n\nErrors can also be handled by retrying failed computations. Keyword arguments retry_delays and retry_check are passed through to retry as keyword arguments delays and check respectively. If batching is specified, and an entire batch fails, all items in the batch are retried.\n\nNote that if both on_error and retry_delays are specified, the on_error hook is called before retrying. If on_error does not throw (or rethrow) an exception, the element will not be retried.\n\nExample: On errors, retry f on an element a maximum of 3 times without any delay between retries.\n\npmap(f, c; retry_delays = zeros(3))\n\nExample: Retry f only if the exception is not of type InexactError, with exponentially increasing delays up to 3 times. Return a NaN in place for all InexactError occurrences.\n\npmap(f, c; on_error = e->(isa(e, InexactError) ? NaN : rethrow()), retry_delays = ExponentialBackOff(n = 3))\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.RemoteException","page":"Distributed Computing","title":"Distributed.RemoteException","category":"type","text":"RemoteException(captured)\n\nExceptions on remote computations are captured and rethrown locally.  A RemoteException wraps the pid of the worker and a captured exception. A CapturedException captures the remote exception and a serializable form of the call stack when the exception was raised.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.ProcessExitedException","page":"Distributed Computing","title":"Distributed.ProcessExitedException","category":"type","text":"ProcessExitedException(worker_id::Int)\n\nAfter a client Julia process has exited, further attempts to reference the dead child will throw this exception.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.Future","page":"Distributed Computing","title":"Distributed.Future","category":"type","text":"Future(w::Int, rrid::RRID, v::Union{Some, Nothing}=nothing)\n\nA Future is a placeholder for a single computation of unknown termination status and time. For multiple potential computations, see RemoteChannel. See remoteref_id for identifying an AbstractRemoteRef.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.RemoteChannel","page":"Distributed Computing","title":"Distributed.RemoteChannel","category":"type","text":"RemoteChannel(pid::Integer=myid())\n\nMake a reference to a Channel{Any}(1) on process pid. The default pid is the current process.\n\nRemoteChannel(f::Function, pid::Integer=myid())\n\nCreate references to remote channels of a specific size and type. f is a function that when executed on pid must return an implementation of an AbstractChannel.\n\nFor example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10 on pid.\n\nThe default pid is the current process.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.fetch-Tuple{Distributed.Future}","page":"Distributed Computing","title":"Base.fetch","category":"method","text":"fetch(x::Future)\n\nWait for and get the value of a Future. The fetched value is cached locally. Further calls to fetch on the same reference return the cached value. If the remote value is an exception, throws a RemoteException which captures the remote exception and backtrace.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.fetch-Tuple{RemoteChannel}","page":"Distributed Computing","title":"Base.fetch","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\nfetch(c::RemoteChannel)\n\nWait for and get a value from a RemoteChannel. Exceptions raised are the same as for a Future. Does not remove the item fetched.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall","category":"method","text":"remotecall(f, id::Integer, args...; kwargs...) -> Future\n\nCall a function f asynchronously on the given arguments on the specified process. Return a Future. Keyword arguments, if any, are passed through to f.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall_wait-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_wait","category":"method","text":"remotecall_wait(f, id::Integer, args...; kwargs...)\n\nPerform a faster wait(remotecall(...)) in one message on the Worker specified by worker id id. Keyword arguments, if any, are passed through to f.\n\nSee also wait and remotecall.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall_fetch-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_fetch","category":"method","text":"remotecall_fetch(f, id::Integer, args...; kwargs...)\n\nPerform fetch(remotecall(...)) in one message. Keyword arguments, if any, are passed through to f. Any remote exceptions are captured in a RemoteException and thrown.\n\nSee also fetch and remotecall.\n\nExamples\n\n$ julia -p 2\n\njulia> remotecall_fetch(sqrt, 2, 4)\n2.0\n\njulia> remotecall_fetch(sqrt, 2, -4)\nERROR: On worker 2:\nDomainError with -4.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\n...\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall_eval","page":"Distributed Computing","title":"Distributed.remotecall_eval","category":"function","text":"remotecall_eval(m::Module, procs, expression)\n\nExecute an expression under module m on the processes specified in procs. Errors on any of the processes are collected into a CompositeException and thrown.\n\nSee also @everywhere.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remote_do-Tuple{Any, Integer, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remote_do","category":"method","text":"remote_do(f, id::Integer, args...; kwargs...) -> nothing\n\nExecutes f on worker id asynchronously. Unlike remotecall, it does not store the result of computation, nor is there a way to wait for its completion.\n\nA successful invocation indicates that the request has been accepted for execution on the remote node.\n\nWhile consecutive remotecalls to the same worker are serialized in the order they are invoked, the order of executions on the remote worker is undetermined. For example, remote_do(f1, 2); remotecall(f2, 2); remote_do(f3, 2) will serialize the call to f1, followed by f2 and f3 in that order. However, it is not guaranteed that f1 is executed before f3 on worker 2.\n\nAny exceptions thrown by f are printed to stderr on the remote worker.\n\nKeyword arguments, if any, are passed through to f.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.put!-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.put!","category":"method","text":"put!(rr::RemoteChannel, args...)\n\nStore a set of values to the RemoteChannel. If the channel is full, blocks until space is available. Return the first argument.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.put!-Tuple{Distributed.Future, Any}","page":"Distributed Computing","title":"Base.put!","category":"method","text":"put!(rr::Future, v)\n\nStore a value to a Future rr. Futures are write-once remote references. A put! on an already set Future throws an Exception. All asynchronous remote calls return Futures and set the value to the return value of the call upon completion.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.take!-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.take!","category":"method","text":"take!(rr::RemoteChannel, args...)\n\nFetch value(s) from a RemoteChannel rr, removing the value(s) in the process.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.isready-Tuple{RemoteChannel, Vararg{Any}}","page":"Distributed Computing","title":"Base.isready","category":"method","text":"isready(rr::RemoteChannel, args...)\n\nDetermine whether a RemoteChannel has a value stored to it. Note that this function can cause race conditions, since by the time you receive its result it may no longer be true. However, it can be safely used on a Future since they are assigned only once.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.isready-Tuple{Distributed.Future}","page":"Distributed Computing","title":"Base.isready","category":"method","text":"isready(rr::Future)\n\nDetermine whether a Future has a value stored to it.\n\nIf the argument Future is owned by a different node, this call will block to wait for the answer. It is recommended to wait for rr in a separate task instead or to use a local Channel as a proxy:\n\np = 1\nf = Future(p)\nerrormonitor(@async put!(f, remotecall_fetch(long_computation, p)))\nisready(f)  # will not block\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.AbstractWorkerPool","page":"Distributed Computing","title":"Distributed.AbstractWorkerPool","category":"type","text":"AbstractWorkerPool\n\nSupertype for worker pools such as WorkerPool and CachingPool. An AbstractWorkerPool should implement:\n\npush! - add a new worker to the overall pool (available + busy)\nput! - put back a worker to the available pool\ntake! - take a worker from the available pool (to be used for remote function execution)\nwait - block until a worker is available\nlength - number of workers available in the overall pool\nisready - return false if a take! on the pool would block, else true\n\nThe default implementations of the above (on a AbstractWorkerPool) require fields\n\nchannel::Channel{Int}\nworkers::Set{Int}\n\nwhere channel contains free worker pids and workers is the set of all workers associated with this pool.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.WorkerPool","page":"Distributed Computing","title":"Distributed.WorkerPool","category":"type","text":"WorkerPool(workers::Union{Vector{Int},AbstractRange{Int}})\n\nCreate a WorkerPool from a vector or range of worker ids.\n\nExamples\n\n$ julia -p 3\n\njulia> WorkerPool([2, 3])\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:2), Set([2, 3]), RemoteChannel{Channel{Any}}(1, 1, 6))\n\njulia> WorkerPool(2:4)\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:2), Set([4, 2, 3]), RemoteChannel{Channel{Any}}(1, 1, 7))\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.CachingPool","page":"Distributed Computing","title":"Distributed.CachingPool","category":"type","text":"CachingPool(workers::Vector{Int})\n\nAn implementation of an AbstractWorkerPool. remote, remotecall_fetch, pmap (and other remote calls which execute functions remotely) benefit from caching the serialized/deserialized functions on the worker nodes, especially closures (which may capture large amounts of data).\n\nThe remote cache is maintained for the lifetime of the returned CachingPool object. To clear the cache earlier, use clear!(pool).\n\nFor global variables, only the bindings are captured in a closure, not the data. let blocks can be used to capture global data.\n\nExamples\n\nconst foo = rand(10^8);\nwp = CachingPool(workers())\nlet foo = foo\n    pmap(i -> sum(foo) + i, wp, 1:100);\nend\n\nThe above would transfer foo only once to each worker.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.default_worker_pool","page":"Distributed Computing","title":"Distributed.default_worker_pool","category":"function","text":"default_worker_pool()\n\nAbstractWorkerPool containing idle workers - used by remote(f) and pmap (by default). Unless one is explicitly set via default_worker_pool!(pool), the default worker pool is initialized to a WorkerPool.\n\nExamples\n\n$ julia -p 3\n\njulia> default_worker_pool()\nWorkerPool(Channel{Int64}(sz_max:9223372036854775807,sz_curr:3), Set([4, 2, 3]), RemoteChannel{Channel{Any}}(1, 1, 4))\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.clear!","page":"Distributed Computing","title":"Distributed.clear!","category":"function","text":"clear!(syms, pids=workers(); mod=Main)\n\nClears global bindings in modules by initializing them to nothing. syms should be of type Symbol or a collection of Symbols . pids and mod identify the processes and the module in which global variables are to be reinitialized. Only those names found to be defined under mod are cleared.\n\nAn exception is raised if a global constant is requested to be cleared.\n\n\n\n\n\nclear!(pool::CachingPool) -> pool\n\nRemoves all cached functions from all participating workers.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remote","page":"Distributed Computing","title":"Distributed.remote","category":"function","text":"remote([p::AbstractWorkerPool], f) -> Function\n\nReturn an anonymous function that executes function f on an available worker (drawn from WorkerPool p if provided) using remotecall_fetch.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall","category":"method","text":"remotecall(f, pool::AbstractWorkerPool, args...; kwargs...) -> Future\n\nWorkerPool variant of remotecall(f, pid, ....). Wait for and take a free worker from pool and perform a remotecall on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> f = remotecall(maximum, wp, A)\nFuture(2, 1, 6, nothing)\n\nIn this example, the task ran on pid 2, called from pid 1.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall_wait-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_wait","category":"method","text":"remotecall_wait(f, pool::AbstractWorkerPool, args...; kwargs...) -> Future\n\nWorkerPool variant of remotecall_wait(f, pid, ....). Wait for and take a free worker from pool and perform a remotecall_wait on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> f = remotecall_wait(maximum, wp, A)\nFuture(3, 1, 9, nothing)\n\njulia> fetch(f)\n0.9995177101692958\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remotecall_fetch-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remotecall_fetch","category":"method","text":"remotecall_fetch(f, pool::AbstractWorkerPool, args...; kwargs...) -> result\n\nWorkerPool variant of remotecall_fetch(f, pid, ....). Waits for and takes a free worker from pool and performs a remotecall_fetch on it.\n\nExamples\n\n$ julia -p 3\n\njulia> wp = WorkerPool([2, 3]);\n\njulia> A = rand(3000);\n\njulia> remotecall_fetch(maximum, wp, A)\n0.9995177101692958\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remote_do-Tuple{Any, AbstractWorkerPool, Vararg{Any}}","page":"Distributed Computing","title":"Distributed.remote_do","category":"method","text":"remote_do(f, pool::AbstractWorkerPool, args...; kwargs...) -> nothing\n\nWorkerPool variant of remote_do(f, pid, ....). Wait for and take a free worker from pool and perform a remote_do on it.\n\nNote that it's not possible to wait for the result of a remote_do() to finish so the worker will immediately be put back in the pool (i.e. potentially causing oversubscription).\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@spawn","page":"Distributed Computing","title":"Distributed.@spawn","category":"macro","text":"@spawn expr\n\nCreate a closure around an expression and run it on an automatically-chosen process, returning a Future to the result. This macro is deprecated; @spawnat :any expr should be used instead.\n\nExamples\n\njulia> addprocs(3);\n\njulia> f = @spawn myid()\nFuture(2, 1, 5, nothing)\n\njulia> fetch(f)\n2\n\njulia> f = @spawn myid()\nFuture(3, 1, 7, nothing)\n\njulia> fetch(f)\n3\n\ncompat: Julia 1.3\nAs of Julia 1.3 this macro is deprecated. Use @spawnat :any instead.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@spawnat","page":"Distributed Computing","title":"Distributed.@spawnat","category":"macro","text":"@spawnat p expr\n\nCreate a closure around an expression and run the closure asynchronously on process p. Return a Future to the result.\n\nIf p is the quoted literal symbol :any, then the system will pick a processor to use automatically. Using :any will not apply any form of load-balancing, consider using a WorkerPool and remotecall(f, ::WorkerPool) if you need load-balancing.\n\nExamples\n\njulia> addprocs(3);\n\njulia> f = @spawnat 2 myid()\nFuture(2, 1, 3, nothing)\n\njulia> fetch(f)\n2\n\njulia> f = @spawnat :any myid()\nFuture(3, 1, 7, nothing)\n\njulia> fetch(f)\n3\n\ncompat: Julia 1.3\nThe :any argument is available as of Julia 1.3.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@fetch","page":"Distributed Computing","title":"Distributed.@fetch","category":"macro","text":"@fetch expr\n\nEquivalent to fetch(@spawnat :any expr). See fetch and @spawnat.\n\nExamples\n\njulia> addprocs(3);\n\njulia> @fetch myid()\n2\n\njulia> @fetch myid()\n3\n\njulia> @fetch myid()\n4\n\njulia> @fetch myid()\n2\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@fetchfrom","page":"Distributed Computing","title":"Distributed.@fetchfrom","category":"macro","text":"@fetchfrom\n\nEquivalent to fetch(@spawnat p expr). See fetch and @spawnat.\n\nExamples\n\njulia> addprocs(3);\n\njulia> @fetchfrom 2 myid()\n2\n\njulia> @fetchfrom 4 myid()\n4\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@distributed","page":"Distributed Computing","title":"Distributed.@distributed","category":"macro","text":"@distributed\n\nA distributed memory, parallel for loop of the form :\n\n@distributed [reducer] for var = range\n    body\nend\n\nThe specified range is partitioned and locally executed across all workers. In case an optional reducer function is specified, @distributed performs local reductions on each worker with a final reduction on the calling process.\n\nNote that without a reducer function, @distributed executes asynchronously, i.e. it spawns independent tasks on all available workers and returns immediately without waiting for completion. To wait for completion, prefix the call with @sync, like :\n\n@sync @distributed for var = range\n    body\nend\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.@everywhere","page":"Distributed Computing","title":"Distributed.@everywhere","category":"macro","text":"@everywhere [procs()] expr\n\nExecute an expression under Main on all procs. Errors on any of the processes are collected into a CompositeException and thrown. For example:\n\n@everywhere bar = 1\n\nwill define Main.bar on all current processes. Any processes added later (say with addprocs()) will not have the expression defined.\n\nUnlike @spawnat, @everywhere does not capture any local variables. Instead, local variables can be broadcast using interpolation:\n\nfoo = 1\n@everywhere bar = $foo\n\nThe optional argument procs allows specifying a subset of all processes to have execute the expression.\n\nSimilar to calling remotecall_eval(Main, procs, expr), but with two extra features:\n\nusing and import statements run on the calling process first, to ensure packages are precompiled.\nThe current source file path used by include is propagated to other processes.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.remoteref_id","page":"Distributed Computing","title":"Distributed.remoteref_id","category":"function","text":"remoteref_id(r::AbstractRemoteRef) -> RRID\n\nFutures and RemoteChannels are identified by fields:\n\nwhere - refers to the node where the underlying object/storage referred to by the reference actually exists.\nwhence - refers to the node the remote reference was created from. Note that this is different from the node where the underlying object referred to actually exists. For example calling RemoteChannel(2) from the master process would result in a where value of 2 and a whence value of 1.\nid is unique across all references created from the worker specified by whence.\n\nTaken together,  whence and id uniquely identify a reference across all workers.\n\nremoteref_id is a low-level API which returns a RRID object that wraps whence and id values of a remote reference.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.channel_from_id","page":"Distributed Computing","title":"Distributed.channel_from_id","category":"function","text":"channel_from_id(id) -> c\n\nA low-level API which returns the backing AbstractChannel for an id returned by remoteref_id. The call is valid only on the node where the backing channel exists.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.worker_id_from_socket","page":"Distributed Computing","title":"Distributed.worker_id_from_socket","category":"function","text":"worker_id_from_socket(s) -> pid\n\nA low-level API which, given a IO connection or a Worker, returns the pid of the worker it is connected to. This is useful when writing custom serialize methods for a type, which optimizes the data written out depending on the receiving process id.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.cluster_cookie-Tuple{}","page":"Distributed Computing","title":"Distributed.cluster_cookie","category":"method","text":"cluster_cookie() -> cookie\n\nReturn the cluster cookie.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.cluster_cookie-Tuple{Any}","page":"Distributed Computing","title":"Distributed.cluster_cookie","category":"method","text":"cluster_cookie(cookie) -> cookie\n\nSet the passed cookie as the cluster cookie, then returns it.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.ClusterManager","page":"Distributed Computing","title":"Distributed.ClusterManager","category":"type","text":"ClusterManager\n\nSupertype for cluster managers, which control workers processes as a cluster. Cluster managers implement how workers can be added, removed and communicated with. SSHManager and LocalManager are subtypes of this.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.WorkerConfig","page":"Distributed Computing","title":"Distributed.WorkerConfig","category":"type","text":"WorkerConfig\n\nType used by ClusterManagers to control workers added to their clusters. Some fields are used by all cluster managers to access a host:\n\nio – the connection used to access the worker (a subtype of IO or Nothing)\nhost – the host address (either a String or Nothing)\nport – the port on the host used to connect to the worker (either an Int or Nothing)\n\nSome are used by the cluster manager to add workers to an already-initialized host:\n\ncount – the number of workers to be launched on the host\nexename – the path to the Julia executable on the host, defaults to \"$(Sys.BINDIR)/julia\" or \"$(Sys.BINDIR)/julia-debug\"\nexeflags – flags to use when launching Julia remotely\n\nThe userdata field is used to store information for each worker by external managers.\n\nSome fields are used by SSHManager and similar managers:\n\ntunnel – true (use tunneling), false (do not use tunneling), or nothing (use default for the manager)\nmultiplex – true (use SSH multiplexing for tunneling) or false\nforward – the forwarding option used for -L option of ssh\nbind_addr – the address on the remote host to bind to\nsshflags – flags to use in establishing the SSH connection\nmax_parallel – the maximum number of workers to connect to in parallel on the host\n\nSome fields are used by both LocalManagers and SSHManagers:\n\nconnect_at – determines whether this is a worker-to-worker or driver-to-worker setup call\nprocess – the process which will be connected (usually the manager will assign this during addprocs)\nospid – the process ID according to the host OS, used to interrupt worker processes\nenviron – private dictionary used to store temporary information by Local/SSH managers\nident – worker as identified by the ClusterManager\nconnect_idents – list of worker ids the worker must connect to if using a custom topology\nenable_threaded_blas – true, false, or nothing, whether to use threaded BLAS or not on the workers\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.launch","page":"Distributed Computing","title":"Distributed.launch","category":"function","text":"launch(manager::ClusterManager, params::Dict, launched::Array, launch_ntfy::Condition)\n\nImplemented by cluster managers. For every Julia worker launched by this function, it should append a WorkerConfig entry to launched and notify launch_ntfy. The function MUST exit once all workers, requested by manager have been launched. params is a dictionary of all keyword arguments addprocs was called with.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.manage","page":"Distributed Computing","title":"Distributed.manage","category":"function","text":"manage(manager::ClusterManager, id::Integer, config::WorkerConfig. op::Symbol)\n\nImplemented by cluster managers. It is called on the master process, during a worker's lifetime, with appropriate op values:\n\nwith :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Base.kill-Tuple{ClusterManager, Int32, WorkerConfig}","page":"Distributed Computing","title":"Base.kill","category":"method","text":"kill(manager::ClusterManager, pid::Int, config::WorkerConfig)\n\nImplemented by cluster managers. It is called on the master process, by rmprocs. It should cause the remote worker specified by pid to exit. kill(manager::ClusterManager.....) executes a remote exit() on pid.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Sockets.connect-Tuple{ClusterManager, Int32, WorkerConfig}","page":"Distributed Computing","title":"Sockets.connect","category":"method","text":"connect(manager::ClusterManager, pid::Int, config::WorkerConfig) -> (instrm::IO, outstrm::IO)\n\nImplemented by cluster managers using custom transports. It should establish a logical connection to worker with id pid, specified by config and return a pair of IO objects. Messages from pid to current process will be read off instrm, while messages to be sent to pid will be written to outstrm. The custom transport implementation must ensure that messages are delivered and received completely and in order. connect(manager::ClusterManager.....) sets up TCP/IP socket connections in-between workers.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.init_worker","page":"Distributed Computing","title":"Distributed.init_worker","category":"function","text":"init_worker(cookie::AbstractString, manager::ClusterManager=DefaultClusterManager())\n\nCalled by cluster managers implementing custom transports. It initializes a newly launched process as a worker. Command line argument --worker[=<cookie>] has the effect of initializing a process as a worker using TCP/IP sockets for transport. cookie is a cluster_cookie.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.start_worker","page":"Distributed Computing","title":"Distributed.start_worker","category":"function","text":"start_worker([out::IO=stdout], cookie::AbstractString=readline(stdin); close_stdin::Bool=true, stderr_to_stdout::Bool=true)\n\nstart_worker is an internal function which is the default entry point for worker processes connecting via TCP/IP. It sets up the process as a Julia cluster worker.\n\nhost:port information is written to stream out (defaults to stdout).\n\nThe function reads the cookie from stdin if required, and  listens on a free port (or if specified, the port in the --bind-to command line option) and schedules tasks to process incoming TCP connections and requests. It also (optionally) closes stdin and redirects stderr to stdout.\n\nIt does not return.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.process_messages","page":"Distributed Computing","title":"Distributed.process_messages","category":"function","text":"process_messages(r_stream::IO, w_stream::IO, incoming::Bool=true)\n\nCalled by cluster managers using custom transports. It should be called when the custom transport implementation receives the first message from a remote worker. The custom transport must manage a logical connection to the remote worker and provide two IO objects, one for incoming messages and the other for messages addressed to the remote worker. If incoming is true, the remote peer initiated the connection. Whichever of the pair initiates the connection sends the cluster cookie and its Julia version number to perform the authentication handshake.\n\nSee also cluster_cookie.\n\n\n\n\n\n"},{"location":"stdlib/Distributed.html#Distributed.default_addprocs_params","page":"Distributed Computing","title":"Distributed.default_addprocs_params","category":"function","text":"default_addprocs_params(mgr::ClusterManager) -> Dict{Symbol, Any}\n\nImplemented by cluster managers. The default keyword parameters passed when calling addprocs(mgr). The minimal set of options is available by calling default_addprocs_params()\n\n\n\n\n\n"},{"location":"stdlib/Serialization.html#Serialization","page":"Serialization","title":"Serialization","category":"section","text":"Provides serialization of Julia objects."},{"location":"stdlib/Serialization.html#Recommended-File-Extension","page":"Serialization","title":"Recommended File Extension","category":"section","text":"While the Serialization module does not mandate a specific file extension, the Julia community commonly uses the .jls extension for serialized Julia files.\n\nExample:\n\nopen(\"model.jls\", \"w\") do io\n    serialize(io, my_model)\nend"},{"location":"stdlib/Serialization.html#Serialization.serialize","page":"Serialization","title":"Serialization.serialize","category":"function","text":"serialize(stream::IO, value)\n\nWrite an arbitrary value to a stream in an opaque format, such that it can be read back by deserialize. The read-back value will be as identical as possible to the original, but note that Ptr values are serialized as all-zero bit patterns (NULL).\n\nAn 8-byte identifying header is written to the stream first. To avoid writing the header, construct a Serializer and use it as the first argument to serialize instead. See also Serialization.writeheader.\n\nThe data format can change in minor (1.x) Julia releases, but files written by prior 1.x versions will remain readable. The main exception to this is when the definition of a type in an external package changes. If that occurs, it may be necessary to specify an explicit compatible version of the affected package in your environment. Renaming functions, even private functions, inside packages can also put existing files out of sync. Anonymous functions require special care: because their names are automatically generated, minor code changes can cause them to be renamed. Serializing anonymous functions should be avoided in files intended for long-term storage.\n\nIn some cases, the word size (32- or 64-bit) of the reading and writing machines must match. In rarer cases the OS or architecture must also match, for example when using packages that contain platform-dependent code.\n\n\n\n\n\nserialize(filename::AbstractString, value)\n\nOpen a file and serialize the given value to it.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"location":"stdlib/Serialization.html#Serialization.deserialize","page":"Serialization","title":"Serialization.deserialize","category":"function","text":"deserialize(stream)\n\nRead a value written by serialize. deserialize assumes the binary data read from stream is correct and has been serialized by a compatible implementation of serialize. deserialize is designed for simplicity and performance, and so does not validate the data read. Malformed data can result in process termination. The caller must ensure the integrity and correctness of data read from stream.\n\n\n\n\n\ndeserialize(filename::AbstractString)\n\nOpen a file and deserialize its contents.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\n\n\n\n\n"},{"location":"stdlib/Serialization.html#Serialization.writeheader","page":"Serialization","title":"Serialization.writeheader","category":"function","text":"Serialization.writeheader(s::AbstractSerializer)\n\nWrite an identifying header to the specified serializer. The header consists of 8 bytes as follows:\n\nOffset Description\n0 tag byte (0x37)\n1-2 signature bytes \"JL\"\n3 protocol version\n4 bits 0-1: endianness: 0 = little, 1 = big\n4 bits 2-3: platform: 0 = 32-bit, 1 = 64-bit\n5-7 reserved\n\n\n\n\n\n"},{"location":"manual/calling-c-and-fortran-code.html#Calling-C-and-Fortran-Code","page":"Calling C and Fortran Code","title":"Calling C and Fortran Code","category":"section","text":"Though most code can be written in Julia, there are many high-quality, mature libraries for numerical computing already written in C and Fortran. To allow easy use of this existing code, Julia makes it simple and efficient to call C and Fortran functions. Julia has a \"no boilerplate\" philosophy: functions can be called directly from Julia without any \"glue\" code, code generation, or compilation – even from the interactive prompt. This is accomplished just by making an appropriate call with the @ccall macro (or the less convenient ccall syntax, see the ccall syntax section).\n\nThe code to be called must be available as a shared library. Most C and Fortran libraries ship compiled as shared libraries already, but if you are compiling the code yourself using GCC (or Clang), you will need to use the -shared and -fPIC options. The machine instructions generated by Julia's JIT are the same as a native C call would be, so the resulting overhead is the same as calling a library function from C code. [1]\n\nBy default, Fortran compilers generate mangled names (for example, converting function names to lowercase or uppercase, often appending an underscore), and so to call a Fortran function you must pass the mangled identifier corresponding to the rule followed by your Fortran compiler. Also, when calling a Fortran function, all inputs must be passed as pointers to allocated values on the heap or stack. This applies not only to arrays and other mutable objects which are normally heap-allocated, but also to scalar values such as integers and floats which are normally stack-allocated and commonly passed in registers when using C or Julia calling conventions.\n\nThe syntax for @ccall to generate a call to the library function is:\n\n@ccall library.function_name(argvalue1::argtype1, ...)::returntype\n@ccall function_name(argvalue1::argtype1, ...)::returntype\n@ccall $function_pointer(argvalue1::argtype1, ...)::returntype\n\nwhere library is a string constant or global variable name (see Non-constant Function -Specifications below). The library can be just a name, or it can specify a full path to the library. The library may be omitted, in which case the function name is resolved in the current executable, the current libc, or libjulia(-internal). This form can be used to call C library functions, functions in the Julia runtime, or functions in an application linked to Julia. Omitting the library cannot be used to call a function in any library (like specifying RTLD_DEFAULT to dlsym) as such behavior is slow, complicated, and not implemented on all platforms. Alternatively, @ccall may also be used to call a function pointer $function_pointer, such as one returned by Libdl.dlsym. The argtypes corresponds to the C-function signature and the argvalues are the actual argument values to be passed to the function.\n\nnote: Note\nSee below for how to map C types to Julia types.\n\nAs a complete but simple example, the following calls the clock function from the standard C library on most Unix-derived systems:\n\njulia> t = @ccall clock()::Int32\n2292761\n\njulia> typeof(t)\nInt32\n\nclock takes no arguments and returns an Int32. To call the getenv function to get a pointer to the value of an environment variable, one makes a call like this:\n\njulia> path = @ccall getenv(\"SHELL\"::Cstring)::Cstring\nCstring(@0x00007fff5fbffc45)\n\njulia> unsafe_string(path)\n\"/bin/bash\"\n\nIn practice, especially when providing reusable functionality, one generally wraps @ccall uses in Julia functions that set up arguments and then check for errors in whatever manner the C or Fortran function specifies. If an error occurs it is thrown as a normal Julia exception. This is especially important since C and Fortran APIs are notoriously inconsistent about how they indicate error conditions. For example, the getenv C library function is wrapped in the following Julia function, which is a simplified version of the actual definition from env.jl:\n\nfunction getenv(var::AbstractString)\n    val = @ccall getenv(var::Cstring)::Cstring\n    if val == C_NULL\n        error(\"getenv: undefined variable: \", var)\n    end\n    return unsafe_string(val)\nend\n\nThe C getenv function indicates an error by returning C_NULL, but other standard C functions indicate errors in different ways, including by returning -1, 0, 1, and other special values. This wrapper throws an exception indicating the problem if the caller tries to get a non-existent environment variable:\n\njulia> getenv(\"SHELL\")\n\"/bin/bash\"\n\njulia> getenv(\"FOOBAR\")\nERROR: getenv: undefined variable: FOOBAR\n\nHere is a slightly more complex example that discovers the local machine's hostname.\n\nfunction gethostname()\n    hostname = Vector{UInt8}(undef, 256) # MAXHOSTNAMELEN\n    err = @ccall gethostname(hostname::Ptr{UInt8}, sizeof(hostname)::Csize_t)::Int32\n    Base.systemerror(\"gethostname\", err != 0)\n    hostname[end] = 0 # ensure null-termination\n    return GC.@preserve hostname unsafe_string(pointer(hostname))\nend\n\nThis example first allocates an array of bytes. It then calls the C library function gethostname to populate the array with the hostname. Finally, it takes a pointer to the hostname buffer, and converts the pointer to a Julia string, assuming that it is a null-terminated C string.\n\nIt is common for C libraries to use this pattern of requiring the caller to allocate memory to be passed to the callee and populated. Allocation of memory from Julia like this is generally accomplished by creating an uninitialized array and passing a pointer to its data to the C function. This is why we don't use the Cstring type here: as the array is uninitialized, it could contain null bytes. Converting to a Cstring as part of the @ccall checks for contained null bytes and could therefore throw a conversion error.\n\nDereferencing pointer(hostname) with unsafe_string is an unsafe operation as it requires access to the memory allocated for hostname that may have been in the meanwhile garbage collected. The macro GC.@preserve prevents this from happening and therefore accessing an invalid memory location.\n\nFinally, here is an example of specifying a library via a path. We create a shared library with the following content\n\n#include <stdio.h>\n\nvoid say_y(int y)\n{\n    printf(\"Hello from C: got y = %d.\\n\", y);\n}\n\nand compile it with gcc -fPIC -shared -o mylib.so mylib.c. It can then be called by specifying the (absolute) path as the library name:\n\njulia> @ccall \"./mylib.so\".say_y(5::Cint)::Cvoid\nHello from C: got y = 5."},{"location":"manual/calling-c-and-fortran-code.html#Creating-C-Compatible-Julia-Function-Pointers","page":"Calling C and Fortran Code","title":"Creating C-Compatible Julia Function Pointers","category":"section","text":"It is possible to pass Julia functions to native C functions that accept function pointer arguments. For example, to match C prototypes of the form:\n\ntypedef returntype (*functiontype)(argumenttype, ...)\n\nThe macro @cfunction generates the C-compatible function pointer for a call to a Julia function. The arguments to @cfunction are:\n\nA Julia function\nThe function's return type\nA tuple of input types, corresponding to the function signature\n\nnote: Note\nAs with @ccall, the return type and the input types must be literal constants.\n\nnote: Note\nCurrently, only the platform-default C calling convention is supported. This means that @cfunction-generated pointers cannot be used in calls where WINAPI expects a stdcall function on 32-bit Windows, but can be used on WIN64 (where stdcall is unified with the C calling convention).\n\nnote: Note\nCallback functions exposed via @cfunction should not throw errors, as that will return control to the Julia runtime unexpectedly and may leave the program in an undefined state.\n\nA classic example is the standard C library qsort function, declared as:\n\nvoid qsort(void *base, size_t nitems, size_t size,\n           int (*compare)(const void*, const void*));\n\nThe base argument is a pointer to an array of length nitems, with elements of size bytes each. compare is a callback function which takes pointers to two elements a and b and returns an integer less/greater than zero if a should appear before/after b (or zero if any order is permitted).\n\nNow, suppose that we have a 1-d array A of values in Julia that we want to sort using the qsort function (rather than Julia's built-in sort function). Before we consider calling qsort and passing arguments, we need to write a comparison function:\n\njulia> function mycompare(a, b)::Cint\n           return (a < b) ? -1 : ((a > b) ? +1 : 0)\n       end;\n\nqsort expects a comparison function that return a C int, so we annotate the return type to be Cint.\n\nIn order to pass this function to C, we obtain its address using the macro @cfunction:\n\njulia> mycompare_c = @cfunction(mycompare, Cint, (Ref{Cdouble}, Ref{Cdouble}));\n\n@cfunction requires three arguments: the Julia function (mycompare), the return type (Cint), and a literal tuple of the input argument types, in this case to sort an array of Cdouble (Float64) elements.\n\nThe final call to qsort looks like this:\n\njulia> A = [1.3, -2.7, 4.4, 3.1];\n\njulia> @ccall qsort(A::Ptr{Cdouble}, length(A)::Csize_t, sizeof(eltype(A))::Csize_t, mycompare_c::Ptr{Cvoid})::Cvoid\n\njulia> A\n4-element Vector{Float64}:\n -2.7\n  1.3\n  3.1\n  4.4\n\nAs the example shows, the original Julia array A has now been sorted: [-2.7, 1.3, 3.1, 4.4]. Note that Julia takes care of converting the array to a Ptr{Cdouble}, computing the size of the element type in bytes, and so on.\n\nFor fun, try inserting a println(\"mycompare($a, $b)\") line into mycompare, which will allow you to see the comparisons that qsort is performing (and to verify that it is really calling the Julia function that you passed to it)."},{"location":"manual/calling-c-and-fortran-code.html#mapping-c-types-to-julia","page":"Calling C and Fortran Code","title":"Mapping C Types to Julia","category":"section","text":"It is critical to exactly match the declared C type with its declaration in Julia. Inconsistencies can cause code that works correctly on one system to fail or produce indeterminate results on a different system.\n\nNote that no C header files are used anywhere in the process of calling C functions: you are responsible for making sure that your Julia types and call signatures accurately reflect those in the C header file.[2]"},{"location":"manual/calling-c-and-fortran-code.html#automatic-type-conversion","page":"Calling C and Fortran Code","title":"Automatic Type Conversion","category":"section","text":"Julia automatically inserts calls to the Base.cconvert function to convert each argument to the specified type. For example, the following call:\n\n@ccall \"libfoo\".foo(x::Int32, y::Float64)::Cvoid\n\nwill behave as if it were written like this:\n\nc_x = Base.cconvert(Int32, x)\nc_y = Base.cconvert(Float64, y)\nGC.@preserve c_x c_y begin\n    @ccall \"libfoo\".foo(\n        Base.unsafe_convert(Int32, c_x)::Int32,\n        Base.unsafe_convert(Float64, c_y)::Float64\n    )::Cvoid\nend\n\nBase.cconvert normally just calls convert, but can be defined to return an arbitrary new object more appropriate for passing to C. This should be used to perform all allocations of memory that will be accessed by the C code. For example, this is used to convert an Array of objects (e.g. strings) to an array of pointers.\n\nBase.unsafe_convert handles conversion to Ptr types. It is considered unsafe because converting an object to a native pointer can hide the object from the garbage collector, causing it to be freed prematurely."},{"location":"manual/calling-c-and-fortran-code.html#Type-Correspondences","page":"Calling C and Fortran Code","title":"Type Correspondences","category":"section","text":"First, let's review some relevant Julia type terminology:\n\nSyntax / Keyword Example Description\nmutable struct BitSet \"Concrete Type\" :: A group of related data that includes a type-tag, is managed by the Julia GC, and is defined by object-identity. The type parameters of a concrete type must be fully defined (no TypeVars are allowed) in order for the instance to be constructed. Also see isconcretetype.\nabstract type Any, AbstractArray{T, N}, Complex{T} \"Super Type\" :: A super-type (not a concrete type) that cannot be instantiated, but can be used to describe a group of types. Also see isabstracttype.\nT{A} Vector{Int} \"Type Parameter\" :: A specialization of a type (typically used for dispatch or storage optimization).\n  \"TypeVar\" :: The T in the type parameter declaration is referred to as a TypeVar (short for type variable).\nprimitive type Int, Float64 \"Primitive Type\" :: A type with no fields, but a size. It is stored and defined by-value.\nstruct Pair{Int, Int} \"Struct\" :: A type with all fields defined to be constant. It is defined by-value, and may be stored with a type-tag.\n ComplexF64 (isbits) \"Is-Bits\"   :: A primitive type, or a struct type where all fields are other isbits types. It is defined by-value, and is stored without a type-tag.\nstruct ...; end nothing \"Singleton\" :: a concrete Type or Struct with no fields.\n(...) or tuple(...) (1, 2, 3) \"Tuple\" :: an immutable data-structure similar to an anonymous struct type, or a constant array. Represented as either an array or a struct."},{"location":"manual/calling-c-and-fortran-code.html#man-bits-types","page":"Calling C and Fortran Code","title":"Bits Types","category":"section","text":"There are several special types to be aware of, as no other type can be defined to behave the same:\n\nFloat32\nExactly corresponds to the float type in C (or REAL*4 in Fortran).\nFloat64\nExactly corresponds to the double type in C (or REAL*8 in Fortran).\nComplexF32\nExactly corresponds to the complex float type in C (or COMPLEX*8 in Fortran).\nComplexF64\nExactly corresponds to the complex double type in C (or COMPLEX*16 in Fortran).\nSigned\nExactly corresponds to the signed type annotation in C (or any INTEGER type in Fortran). Any Julia type that is not a subtype of Signed is assumed to be unsigned.\n\nRef{T}\nBehaves like a Ptr{T} that can manage its memory via the Julia GC.\n\nArray{T,N}\nWhen an array is passed to C as a Ptr{T} argument, it is not reinterpret-cast: Julia requires that the element type of the array matches T, and the address of the first element is passed.\nTherefore, if an Array contains data in the wrong format, it will have to be explicitly converted using a call such as trunc.(Int32, A).\nTo pass an array A as a pointer of a different type without converting the data beforehand (for example, to pass a Float64 array to a function that operates on uninterpreted bytes), you can declare the argument as Ptr{Cvoid}.\nIf an array of eltype Ptr{T} is passed as a Ptr{Ptr{T}} argument, Base.cconvert will attempt to first make a null-terminated copy of the array with each element replaced by its Base.cconvert version. This allows, for example, passing an argv pointer array of type Vector{String} to an argument of type Ptr{Ptr{Cchar}}.\n\nOn all systems we currently support, basic C/C++ value types may be translated to Julia types as follows. Every C type also has a corresponding Julia type with the same name, prefixed by C. This can help when writing portable code (and remembering that an int in C is not the same as an Int in Julia).\n\nSystem Independent Types\n\nC name Fortran name Standard Julia Alias Julia Base Type\nunsigned char CHARACTER Cuchar UInt8\nbool (_Bool in C99+)  Cuchar UInt8\nshort INTEGER*2, LOGICAL*2 Cshort Int16\nunsigned short  Cushort UInt16\nint, BOOL (C, typical) INTEGER*4, LOGICAL*4 Cint Int32\nunsigned int  Cuint UInt32\nlong long INTEGER*8, LOGICAL*8 Clonglong Int64\nunsigned long long  Culonglong UInt64\nintmax_t  Cintmax_t Int64\nuintmax_t  Cuintmax_t UInt64\nfloat REAL*4 Cfloat Float32\ndouble REAL*8 Cdouble Float64\ncomplex float COMPLEX*8 ComplexF32 Complex{Float32}\ncomplex double COMPLEX*16 ComplexF64 Complex{Float64}\nptrdiff_t  Cptrdiff_t Int\nssize_t  Cssize_t Int\nsize_t  Csize_t UInt\nvoid   Cvoid\nvoid and [[noreturn]] or _Noreturn   Union{}\nvoid*   Ptr{Cvoid} (or similarly Ref{Cvoid})\nT* (where T represents an appropriately defined type)   Ref{T} (T may be safely mutated only if T is an isbits type)\nchar* (or char[], e.g. a string) CHARACTER*N  Cstring if null-terminated, or Ptr{UInt8} if not\nchar** (or *char[])   Ptr{Ptr{UInt8}}\njl_value_t* (any Julia Type)   Any\njl_value_t* const* (a reference to a Julia value)   Ref{Any} (const, since mutation would require a write barrier, which is not possible to insert correctly)\nva_arg   Not supported\n... (variadic function specification)   T... (where T is one of the above types, when using the ccall function)\n... (variadic function specification)   ; va_arg1::T, va_arg2::S, etc. (only supported with @ccall macro)\n\nThe Cstring type is essentially a synonym for Ptr{UInt8}, except the conversion to Cstring throws an error if the Julia string contains any embedded null characters (which would cause the string to be silently truncated if the C routine treats null as the terminator). If you are passing a char* to a C routine that does not assume null termination (e.g. because you pass an explicit string length), or if you know for certain that your Julia string does not contain null and want to skip the check, you can use Ptr{UInt8} as the argument type. Cstring can also be used as the ccall return type, but in that case it obviously does not introduce any extra checks and is only meant to improve the readability of the call.\n\nSystem Dependent Types\n\nC name Standard Julia Alias Julia Base Type\nchar Cchar Int8 (x86, x86_64), UInt8 (powerpc, arm)\nlong Clong Int (UNIX), Int32 (Windows)\nunsigned long Culong UInt (UNIX), UInt32 (Windows)\nwchar_t Cwchar_t Int32 (UNIX), UInt16 (Windows)\n\nnote: Note\nWhen calling Fortran, all inputs must be passed by pointers to heap- or stack-allocated values, so all type correspondences above should contain an additional Ptr{..} or Ref{..} wrapper around their type specification.\n\nwarning: Warning\nFor string arguments (char*) the Julia type should be Cstring (if null-terminated data is expected), or either Ptr{Cchar} or Ptr{UInt8} otherwise (these two pointer types have the same effect), as described above, not String. Similarly, for array arguments (T[] or T*), the Julia type should again be Ptr{T}, not Vector{T}.\n\nwarning: Warning\nJulia's Char type is 32 bits, which is not the same as the wide-character type (wchar_t or wint_t) on all platforms.\n\nwarning: Warning\nA return type of Union{} means the function will not return, i.e., C++11 [[noreturn]] or C11 _Noreturn (e.g. jl_throw or longjmp). Do not use this for functions that return no value (void) but do return, for those, use Cvoid instead.\n\nnote: Note\nFor wchar_t* arguments, the Julia type should be Cwstring (if the C routine expects a null-terminated string), or Ptr{Cwchar_t} otherwise. Note also that UTF-8 string data in Julia is internally null-terminated, so it can be passed to C functions expecting null-terminated data without making a copy (but using the Cwstring type will cause an error to be thrown if the string itself contains null characters).\n\nnote: Note\nC functions that take an argument of type char** can be called by using a Ptr{Ptr{UInt8}} type within Julia. For example, C functions of the form:int main(int argc, char **argv);can be called via the following Julia code:argv = [ \"a.out\", \"arg1\", \"arg2\" ]\n@ccall main(length(argv)::Int32, argv::Ptr{Ptr{UInt8}})::Int32\n\nnote: Note\nFor Fortran functions taking variable length strings of type character(len=*) the string lengths are provided as hidden arguments. Type and position of these arguments in the list are compiler specific, where compiler vendors usually default to using Csize_t as type and append the hidden arguments at the end of the argument list. While this behaviour is fixed for some compilers (GNU), others optionally permit placing hidden arguments directly after the character argument (Intel, PGI). For example, Fortran subroutines of the formsubroutine test(str1, str2)\ncharacter(len=*) :: str1,str2can be called via the following Julia code, where the lengths are appendedstr1 = \"foo\"\nstr2 = \"bar\"\nccall(:test, Cvoid, (Ptr{UInt8}, Ptr{UInt8}, Csize_t, Csize_t),\n                    str1, str2, sizeof(str1), sizeof(str2))\n\nwarning: Warning\nFortran compilers may also add other hidden arguments for pointers, assumed-shape (:) and assumed-size (*) arrays. Such behaviour can be avoided by using ISO_C_BINDING and including bind(c) in the definition of the subroutine, which is strongly recommended for interoperable code. In this case, there will be no hidden arguments, at the cost of some language features (e.g. only character(len=1) will be permitted to pass strings).\n\nnote: Note\nA C function declared to return Cvoid will return the value nothing in Julia."},{"location":"manual/calling-c-and-fortran-code.html#Struct-Type-Correspondences","page":"Calling C and Fortran Code","title":"Struct Type Correspondences","category":"section","text":"Composite types such as struct in C or TYPE in Fortran90 (or STRUCTURE / RECORD in some variants of F77), can be mirrored in Julia by creating a struct definition with the same field layout.\n\nWhen used recursively, isbits types are stored inline. All other types are stored as a pointer to the data. When mirroring a struct used by-value inside another struct in C, it is imperative that you do not attempt to manually copy the fields over, as this will not preserve the correct field alignment. Instead, declare an isbits struct type and use that instead. Unnamed structs are not possible in the translation to Julia.\n\nPacked structs and union declarations are not supported by Julia.\n\nYou can get an approximation of a union if you know, a priori, the field that will have the greatest size (potentially including padding). When translating your fields to Julia, declare the Julia field to be only of that type.\n\nArrays of parameters can be expressed with NTuple. For example, the struct in C notation is written as\n\nstruct B {\n    int A[3];\n};\n\nb_a_2 = B.A[2];\n\ncan be written in Julia as\n\nstruct B\n    A::NTuple{3, Cint}\nend\n\nb_a_2 = B.A[3]  # note the difference in indexing (1-based in Julia, 0-based in C)\n\nArrays of unknown size (C99-compliant variable length structs specified by [] or [0]) are not directly supported. Often the best way to deal with these is to deal with the byte offsets directly. For example, if a C library declared a proper string type and returned a pointer to it:\n\nstruct String {\n    int strlen;\n    char data[];\n};\n\nIn Julia, we can access the parts independently to make a copy of that string:\n\nstr = from_c::Ptr{Cvoid}\nlen = unsafe_load(Ptr{Cint}(str))\nunsafe_string(str + Core.sizeof(Cint), len)"},{"location":"manual/calling-c-and-fortran-code.html#Type-Parameters","page":"Calling C and Fortran Code","title":"Type Parameters","category":"section","text":"The type arguments to @ccall and @cfunction are evaluated statically, when the method containing the usage is defined. They therefore must take the form of a literal tuple, not a variable, and cannot reference local variables.\n\nThis may sound like a strange restriction, but remember that since C is not a dynamic language like Julia, its functions can only accept argument types with a statically-known, fixed signature.\n\nHowever, while the type layout must be known statically to compute the intended C ABI, the static parameters of the function are considered to be part of this static environment. The static parameters of the function may be used as type parameters in the call signature, as long as they don't affect the layout of the type. For example, f(x::T) where {T} = @ccall valid(x::Ptr{T})::Ptr{T} is valid, since Ptr is always a word-size primitive type. But, g(x::T) where {T} = @ccall notvalid(x::T)::T is not valid, since the type layout of T is not known statically."},{"location":"manual/calling-c-and-fortran-code.html#SIMD-Values","page":"Calling C and Fortran Code","title":"SIMD Values","category":"section","text":"If a C/C++ routine has an argument or return value that is a native SIMD type, the corresponding Julia type is a homogeneous tuple of VecElement that naturally maps to the SIMD type. Specifically:\n\nThe tuple must be the same size and elements as the SIMD type. For example, a tuple representing an __m128 on x86 must have a size of 16 bytes and Float32 elements.\nThe element type of the tuple must be an instance of VecElement{T} where T is a primitive type with a power-of-two number of bytes (e.g. 1, 2, 4, 8, 16, etc) such as Int8 or Float64.\n\nFor instance, consider this C routine that uses AVX intrinsics:\n\n#include <immintrin.h>\n\n__m256 dist( __m256 a, __m256 b ) {\n    return _mm256_sqrt_ps(_mm256_add_ps(_mm256_mul_ps(a, a),\n                                        _mm256_mul_ps(b, b)));\n}\n\nThe following Julia code calls dist using ccall:\n\nconst m256 = NTuple{8, VecElement{Float32}}\n\na = m256(ntuple(i -> VecElement(sin(Float32(i))), 8))\nb = m256(ntuple(i -> VecElement(cos(Float32(i))), 8))\n\nfunction call_dist(a::m256, b::m256)\n    @ccall \"libdist\".dist(a::m256, b::m256)::m256\nend\n\nprintln(call_dist(a,b))\n\nThe host machine must have the requisite SIMD registers. For example, the code above will not work on hosts without AVX support."},{"location":"manual/calling-c-and-fortran-code.html#Memory-Ownership","page":"Calling C and Fortran Code","title":"Memory Ownership","category":"section","text":"malloc/free\n\nMemory allocation and deallocation of such objects must be handled by calls to the appropriate cleanup routines in the libraries being used, just like in any C program. Do not try to free an object received from a C library with Libc.free in Julia, as this may result in the free function being called via the wrong library and cause the process to abort. The reverse (passing an object allocated in Julia to be freed by an external library) is equally invalid."},{"location":"manual/calling-c-and-fortran-code.html#When-to-use-T,-Ptr{T}-and-Ref{T}","page":"Calling C and Fortran Code","title":"When to use T, Ptr{T} and Ref{T}","category":"section","text":"In Julia code wrapping calls to external C routines, ordinary (non-pointer) data should be declared to be of type T inside the @ccall, as they are passed by value. For C code accepting pointers, Ref{T} should generally be used for the types of input arguments, allowing the use of pointers to memory managed by either Julia or C through the implicit call to Base.cconvert. In contrast, pointers returned by the C function called should be declared to be of the output type Ptr{T}, reflecting that the memory pointed to is managed by C only. Pointers contained in C structs should be represented as fields of type Ptr{T} within the corresponding Julia struct types designed to mimic the internal structure of corresponding C structs.\n\nIn Julia code wrapping calls to external Fortran routines, all input arguments should be declared as of type Ref{T}, as Fortran passes all variables by pointers to memory locations. The return type should either be Cvoid for Fortran subroutines, or a T for Fortran functions returning the type T."},{"location":"manual/calling-c-and-fortran-code.html#Mapping-C-Functions-to-Julia","page":"Calling C and Fortran Code","title":"Mapping C Functions to Julia","category":"section","text":""},{"location":"manual/calling-c-and-fortran-code.html#@ccall-/-@cfunction-argument-translation-guide","page":"Calling C and Fortran Code","title":"@ccall / @cfunction argument translation guide","category":"section","text":"For translating a C argument list to Julia:\n\nT, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nT, where T is an equivalent Julia Bits Type (per the table above)\nif T is an enum, the argument type should be equivalent to Cint or Cuint\nargument value will be copied (passed by value)\nstruct T (including typedef to a struct)\nT, where T is a concrete Julia type\nargument value will be copied (passed by value)\nvector T (or __attribute__ vector_size, or a typedef such as __m128)\nNTuple{N, VecElement{T}}, where T is a primitive Julia type of the correct size and N is the number of elements in the vector (equal to vector_size / sizeof T).\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Cvoid} if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\njl_value_t* const*\nRef{Any}\nargument list must be a valid Julia object (or C_NULL)\ncannot be used for an output parameter, unless the user is able to separately arrange for the object to be GC-preserved\nT*\nRef{T}, where T is the Julia type corresponding to T\nargument value will be copied if it is an inlinealloc type (which includes isbits otherwise, the value must be a valid Julia object\nT (*)(...) (e.g. a pointer to a function)\nPtr{Cvoid} (you may need to use @cfunction explicitly to create this pointer)\n... (e.g. a vararg)\n[for ccall]: T..., where T is the single Julia type of all remaining arguments\n[for @ccall]: ; va_arg1::T, va_arg2::S, etc, where T and S are the Julia type (i.e. separate the regular arguments from varargs with a ;)\ncurrently unsupported by @cfunction\nva_arg\nnot supported by ccall or @cfunction"},{"location":"manual/calling-c-and-fortran-code.html#@ccall-/-@cfunction-return-type-translation-guide","page":"Calling C and Fortran Code","title":"@ccall / @cfunction return type translation guide","category":"section","text":"For translating a C return type to Julia:\n\nvoid\nCvoid (this will return the singleton instance nothing::Cvoid)\nT, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nsame as C argument list\nargument value will be copied (returned by-value)\nstruct T (including typedef to a struct)\nsame as C argument list\nargument value will be copied (returned by-value)\nvector T\nsame as C argument list\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Cvoid} if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\njl_value_t**\nPtr{Any} (Ref{Any} is invalid as a return type)\nT*\nIf the memory is already owned by Julia, or is an isbits type, and is known to be non-null:\nRef{T}, where T is the Julia type corresponding to T\na return type of Ref{Any} is invalid, it should either be Any (corresponding to jl_value_t*) or Ptr{Any} (corresponding to jl_value_t**)\nC MUST NOT modify the memory returned via Ref{T} if T is an isbits type\nIf the memory is owned by C:\nPtr{T}, where T is the Julia type corresponding to T\nT (*)(...) (e.g. a pointer to a function)\nPtr{Cvoid} to call this directly from Julia you will need to pass this as the first argument to @ccall. See Indirect Calls."},{"location":"manual/calling-c-and-fortran-code.html#Passing-Pointers-for-Modifying-Inputs","page":"Calling C and Fortran Code","title":"Passing Pointers for Modifying Inputs","category":"section","text":"Because C doesn't support multiple return values, often C functions will take pointers to data that the function will modify. To accomplish this within a @ccall, you need to first encapsulate the value inside a Ref{T} of the appropriate type. When you pass this Ref object as an argument, Julia will automatically pass a C pointer to the encapsulated data:\n\nwidth = Ref{Cint}(0)\nrange = Ref{Cfloat}(0)\n@ccall foo(width::Ref{Cint}, range::Ref{Cfloat})::Cvoid\n\nUpon return, the contents of width and range can be retrieved (if they were changed by foo) by width[] and range[]; that is, they act like zero-dimensional arrays."},{"location":"manual/calling-c-and-fortran-code.html#C-Wrapper-Examples","page":"Calling C and Fortran Code","title":"C Wrapper Examples","category":"section","text":"Let's start with a simple example of a C wrapper that returns a Ptr type:\n\nmutable struct gsl_permutation\nend\n\n# The corresponding C signature is\n#     gsl_permutation * gsl_permutation_alloc (size_t n);\nfunction permutation_alloc(n::Integer)\n    output_ptr = @ccall \"libgsl\".gsl_permutation_alloc(n::Csize_t)::Ptr{gsl_permutation}\n    if output_ptr == C_NULL # Could not allocate memory\n        throw(OutOfMemoryError())\n    end\n    return output_ptr\nend\n\nThe GNU Scientific Library (here assumed to be accessible through :libgsl) defines an opaque pointer, gsl_permutation *, as the return type of the C function gsl_permutation_alloc. As user code never has to look inside the gsl_permutation struct, the corresponding Julia wrapper simply needs a new type declaration, gsl_permutation, that has no internal fields and whose sole purpose is to be placed in the type parameter of a Ptr type. The return type of the ccall is declared as Ptr{gsl_permutation}, since the memory allocated and pointed to by output_ptr is controlled by C.\n\nThe input n is passed by value, and so the function's input signature is simply declared as ::Csize_t without any Ref or Ptr necessary. (If the wrapper was calling a Fortran function instead, the corresponding function input signature would instead be ::Ref{Csize_t}, since Fortran variables are passed by pointers.) Furthermore, n can be any type that is convertible to a Csize_t integer; the ccall implicitly calls Base.cconvert(Csize_t, n).\n\nHere is a second example wrapping the corresponding destructor:\n\n# The corresponding C signature is\n#     void gsl_permutation_free (gsl_permutation * p);\nfunction permutation_free(p::Ptr{gsl_permutation})\n    @ccall \"libgsl\".gsl_permutation_free(p::Ptr{gsl_permutation})::Cvoid\nend\n\nHere is a third example passing Julia arrays:\n\n# The corresponding C signature is\n#    int gsl_sf_bessel_Jn_array (int nmin, int nmax, double x,\n#                                double result_array[])\nfunction sf_bessel_Jn_array(nmin::Integer, nmax::Integer, x::Real)\n    if nmax < nmin\n        throw(DomainError())\n    end\n    result_array = Vector{Cdouble}(undef, nmax - nmin + 1)\n    errorcode = @ccall \"libgsl\".gsl_sf_bessel_Jn_array(\n                    nmin::Cint, nmax::Cint, x::Cdouble, result_array::Ref{Cdouble})::Cint\n    if errorcode != 0\n        error(\"GSL error code $errorcode\")\n    end\n    return result_array\nend\n\nThe C function wrapped returns an integer error code; the results of the actual evaluation of the Bessel J function populate the Julia array result_array. This variable is declared as a Ref{Cdouble}, since its memory is allocated and managed by Julia. The implicit call to Base.cconvert(Ref{Cdouble}, result_array) unpacks the Julia pointer to a Julia array data structure into a form understandable by C."},{"location":"manual/calling-c-and-fortran-code.html#Fortran-Wrapper-Example","page":"Calling C and Fortran Code","title":"Fortran Wrapper Example","category":"section","text":"The following example utilizes ccall to call a function in a common Fortran library (libBLAS) to compute a dot product. Notice that the argument mapping is a bit different here than above, as we need to map from Julia to Fortran. On every argument type, we specify Ref or Ptr. This mangling convention may be specific to your Fortran compiler and operating system and is likely undocumented. However, wrapping each in a Ref (or Ptr, where equivalent) is a frequent requirement of Fortran compiler implementations:\n\nfunction compute_dot(DX::Vector{Float64}, DY::Vector{Float64})\n    @assert length(DX) == length(DY)\n    n = length(DX)\n    incx = incy = 1\n    product = @ccall \"libLAPACK\".ddot(\n        n::Ref{Int32}, DX::Ptr{Float64}, incx::Ref{Int32}, DY::Ptr{Float64}, incy::Ref{Int32})::Float64\n    return product\nend"},{"location":"manual/calling-c-and-fortran-code.html#Garbage-Collection-Safety","page":"Calling C and Fortran Code","title":"Garbage Collection Safety","category":"section","text":"When passing data to a @ccall, it is best to avoid using the pointer function. Instead define a Base.cconvert method and pass the variables directly to the @ccall. @ccall automatically arranges that all of its arguments will be preserved from garbage collection until the call returns. If a C API will store a reference to memory allocated by Julia, after the @ccall returns, you must ensure that the object remains visible to the garbage collector. The suggested way to do this is to make a global variable of type Vector{Ref} to hold these values until the C library notifies you that it is finished with them.\n\nWhenever you have created a pointer to Julia data, you must ensure the original data exists until you have finished using the pointer. Many methods in Julia such as unsafe_load and String make copies of data instead of taking ownership of the buffer, so that it is safe to free (or alter) the original data without affecting Julia. A notable exception is unsafe_wrap which, for performance reasons, shares (or can be told to take ownership of) the underlying buffer.\n\nThe garbage collector does not guarantee any order of finalization. That is, if a contained a reference to b and both a and b are due for garbage collection, there is no guarantee that b would be finalized after a. If proper finalization of a depends on b being valid, it must be handled in other ways."},{"location":"manual/calling-c-and-fortran-code.html#Non-constant-Function-Specifications","page":"Calling C and Fortran Code","title":"Non-constant Function Specifications","category":"section","text":"In some cases, the exact name or path of the needed library is not known in advance and must be computed at run time. To handle such cases, the library component specification can be a value such as Libdl.LazyLibrary. The runtime will call Libdl.dlopen on that object when first used by a ccall."},{"location":"manual/calling-c-and-fortran-code.html#man-lazylibrary","page":"Calling C and Fortran Code","title":"Using LazyLibrary for Lazy Loading","category":"section","text":"Libdl.LazyLibrary provides a thread-safe mechanism for deferring library loading until first use. This is the recommended approach for library initialization in modern Julia code.\n\nA LazyLibrary represents a library that opens itself (and its dependencies) automatically on first use in a ccall(), @ccall, dlopen(), dlsym(), dlpath(), or cglobal(). The library is loaded exactly once in a thread-safe manner, and subsequent calls reuse the loaded library handle."},{"location":"manual/calling-c-and-fortran-code.html#Basic-Usage","page":"Calling C and Fortran Code","title":"Basic Usage","category":"section","text":"using Libdl\n\n# Define a LazyLibrary as a const for optimal performance\nconst libz = LazyLibrary(\"libz\")\n\n# Use directly in @ccall - library loads automatically on first call\n@ccall libz.deflate(strm::Ptr{Cvoid}, flush::Cint)::Cint\n\n# Also works with ccall\nccall((:inflate, libz), Cint, (Ptr{Cvoid}, Cint), strm, flush)"},{"location":"manual/calling-c-and-fortran-code.html#Platform-Specific-Libraries","page":"Calling C and Fortran Code","title":"Platform-Specific Libraries","category":"section","text":"For code that needs to work across different platforms:\n\nconst mylib = LazyLibrary(\n    if Sys.iswindows()\n        \"mylib.dll\"\n    elseif Sys.isapple()\n        \"libmylib.dylib\"\n    else\n        \"libmylib.so\"\n    end\n)"},{"location":"manual/calling-c-and-fortran-code.html#Libraries-with-Dependencies","page":"Calling C and Fortran Code","title":"Libraries with Dependencies","category":"section","text":"When a library depends on other libraries, specify the dependencies to ensure they load in the correct order:\n\nconst libfoo = LazyLibrary(\"libfoo\")\nconst libbar = LazyLibrary(\"libbar\"; dependencies=[libfoo])\n\n# When libbar is first used, libfoo is loaded first automatically\n@ccall libbar.bar_function(x::Cint)::Cint"},{"location":"manual/calling-c-and-fortran-code.html#Lazy-Path-Construction","page":"Calling C and Fortran Code","title":"Lazy Path Construction","category":"section","text":"For libraries whose paths are determined at runtime, use LazyLibraryPath:\n\n# Path is constructed when library is first accessed\nconst mylib = LazyLibrary(LazyLibraryPath(artifact_dir, \"lib\", \"libmylib.so\"))"},{"location":"manual/calling-c-and-fortran-code.html#Initialization-Callbacks","page":"Calling C and Fortran Code","title":"Initialization Callbacks","category":"section","text":"If a library requires initialization after loading:\n\nconst mylib = LazyLibrary(\"libmylib\";\n    on_load_callback = () -> @ccall mylib.initialize()::Cvoid\n)\n\nwarning: Warning\nThe on_load_callback should be minimal and must not call wait() on any tasks. It is called exactly once by the thread that loads the library."},{"location":"manual/calling-c-and-fortran-code.html#Conversion-from-__init__()-Pattern","page":"Calling C and Fortran Code","title":"Conversion from __init__() Pattern","category":"section","text":"Before LazyLibrary, library paths were often computed in __init__() functions. This pattern can be replaced with LazyLibrary for better performance and thread safety.\n\nOld pattern using __init__():\n\n# Old: Library path computed in __init__()\nlibmylib_path = \"\"\n\nfunction __init__(\n    # Loads library on startup, whether it is used or not\n    global libmylib_path = find_library([\"libmylib\"])\nend\n\nfunction myfunc(x)\n    ccall((:cfunc, libmylib_path), Cint, (Cint,), x)\nend\n\nNew pattern using LazyLibrary:\n\n# New: Library as const, no __init__() needed\nconst libmylib = LazyLibrary(\"libmylib\")\n\nfunction myfunc(x)\n    # Library loads automatically just before calling `cfunc`\n    @ccall libmylib.cfunc(x::Cint)::Cint\nend\n\nFor more details, see the Libdl.LazyLibrary documentation."},{"location":"manual/calling-c-and-fortran-code.html#Overloading-dlopen-for-Custom-Types","page":"Calling C and Fortran Code","title":"Overloading dlopen for Custom Types","category":"section","text":"The runtime will call dlsym(:function, dlopen(library)::Ptr{Cvoid}) when a @ccall is executed. The Libdl.dlopen function can be overloaded for custom types to provide alternate behaviors. However, it is assumed that the library location and handle does not change once it is determined, so the result of the call may be cached and reused. Therefore, the number of times the dlopen expression executes is unspecified, and returning different values for multiple calls will results in unspecified (but valid) behavior."},{"location":"manual/calling-c-and-fortran-code.html#Computed-Function-Names","page":"Calling C and Fortran Code","title":"Computed Function Names","category":"section","text":"If even more flexibility is needed, it is possible to use computed values as function names by staging through eval as follows:\n\n@eval @ccall \"lib\".$(string(\"a\", \"b\"))()::Cint\n\nThis expression constructs a name using string, then substitutes this name into a new @ccall expression, which is then evaluated. Keep in mind that eval only operates at the top level, so within this expression local variables will not be available (unless their values are substituted with $). For this reason, eval is typically only used to form top-level definitions, for example when wrapping libraries that contain many similar functions."},{"location":"manual/calling-c-and-fortran-code.html#Indirect-Calls","page":"Calling C and Fortran Code","title":"Indirect Calls","category":"section","text":"The first argument to @ccall can also be an expression to be evaluated at run time, each time it is used. In this case, the expression must evaluate to a Ptr, which will be used as the address of the native function to call. This behavior occurs when the first @ccall argument is marked with $ and when the first ccall argument is not a simple constant literal or expression in (). The argument can be any expression and can use local variables and arguments and can return a different value every time.\n\nFor example, you might implement a macro similar to cglobal that looks up the function via dlsym, then caches the pointer in a shared reference (which is auto reset to C_NULL during precompile saving). For example:\n\nmacro dlsym(lib, func)\n    z = Ref(C_NULL)\n    quote\n        local zlocal = $z[]\n        if zlocal == C_NULL\n            zlocal = dlsym($(esc(lib))::Ptr{Cvoid}, $(esc(func)))::Ptr{Cvoid}\n            $z[] = zlocal\n        end\n        zlocal\n    end\nend\n\nconst mylibvar = LazyLibrary(\"mylib\")\n@ccall $(@dlsym(dlopen(mylibvar), \"myfunc\"))()::Cvoid"},{"location":"manual/calling-c-and-fortran-code.html#Closure-cfunctions","page":"Calling C and Fortran Code","title":"Closure cfunctions","category":"section","text":"The first argument to @cfunction can be marked with a $, in which case the return value will instead be a struct CFunction which closes over the argument. You must ensure that this return object is kept alive until all uses of it are done. The contents and code at the cfunction pointer will be erased via a finalizer when this reference is dropped and atexit. This is not usually needed, since this functionality is not present in C, but can be useful for dealing with ill-designed APIs which don't provide a separate closure environment parameter.\n\nfunction qsort(a::Vector{T}, cmp) where T\n    isbits(T) || throw(ArgumentError(\"this method can only qsort isbits arrays\"))\n    callback = @cfunction $cmp Cint (Ref{T}, Ref{T})\n    # Here, `callback` isa Base.CFunction, which will be converted to Ptr{Cvoid}\n    # (and protected against finalization) by the ccall\n    @ccall qsort(a::Ptr{T}, length(a)::Csize_t, Base.elsize(a)::Csize_t, callback::Ptr{Cvoid})\n    # We could instead use:\n    #    GC.@preserve callback begin\n    #        use(Base.unsafe_convert(Ptr{Cvoid}, callback))\n    #    end\n    # if we needed to use it outside of a `ccall`\n    return a\nend\n\nnote: Note\nClosure @cfunction relies on LLVM trampolines, which are not available on all platforms (for example ARM and PowerPC)."},{"location":"manual/calling-c-and-fortran-code.html#Closing-a-Library","page":"Calling C and Fortran Code","title":"Closing a Library","category":"section","text":"It is sometimes useful to close (unload) a library so that it can be reloaded. For instance, when developing C code for use with Julia, one may need to compile, call the C code from Julia, then close the library, make an edit, recompile, and load in the new changes. One can either restart Julia or use the Libdl functions to manage the library explicitly, such as:\n\nlib = Libdl.dlopen(\"./my_lib.so\") # Open the library explicitly.\nsym = Libdl.dlsym(lib, :my_fcn)   # Get a symbol for the function to call.\n@ccall $sym(...) # Use the pointer `sym` instead of the library.symbol tuple.\nLibdl.dlclose(lib) # Close the library explicitly.\n\nNote that when using @ccall with the input (e.g., @ccall \"./my_lib.so\".my_fcn(...)::Cvoid), the library is opened implicitly and it may not be explicitly closed."},{"location":"manual/calling-c-and-fortran-code.html#Variadic-function-calls","page":"Calling C and Fortran Code","title":"Variadic function calls","category":"section","text":"To call variadic C functions a semicolon can be used in the argument list to separate required arguments from variadic arguments. An example with the printf function is given below:\n\njulia> @ccall printf(\"%s = %d\\n\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint\nfoo = 3\n8"},{"location":"manual/calling-c-and-fortran-code.html#ccall-interface","page":"Calling C and Fortran Code","title":"ccall interface","category":"section","text":"There is another alternative interface to @ccall. This interface is slightly less convenient but it does allow one to specify a calling convention.\n\nThe arguments to ccall are:\n\nA (:function, \"library\") pair (most common),\nOR\na :function name symbol or \"function\" name string (for symbols in the current process or libc),\nOR\na function pointer (for example, from dlsym).\nThe function's return type\nA tuple of input types, corresponding to the function signature. One common mistake is forgetting that a 1-tuple of argument types must be written with a trailing comma.\nThe actual argument values to be passed to the function, if any; each is a separate parameter.\n\nnote: Note\nThe (:function, \"library\") pair and the input type list must be syntactic tuples (i.e., they can't be variables or values with a type of Tuple.The rettype and argument type values are evaluated at when the containing method is defined, not runtime.\n\nnote: Function Name vs Pointer Syntax\nThe syntax of the first argument to ccall determines whether you're calling by name or by pointer:Name-based calls (tuple literal syntax):\nBoth the function and library names can be a quoted Symbol, a String, a variable name (a GlobalRef), or a dotted expression ending with a variable name.\nSingle name: (:function_name,) or \"function_name\" - uses default library lookup.\nName with library: (:function_name, \"library\") - specifies both function and library.\nSymbol, string, and tuple literal constants (not expressions that evaluate to those constants, but actual literals) are automatically normalized to tuple form.\nPointer-based calls (non-tuple syntax):\nAnything that is not a literal tuple expression specified above is assumed to be an expression that evaluates to a function pointers at runtime.\nFunction pointer variables: fptr where fptr is a runtime pointer value.\nFunction pointer computations: dlsym(:something) where the result is computed at runtime every time (usually along with some caching logic).\nLibrary name expressions:\nWhen given as a variable, the library name can resolve to a Symbol, a String, or any other value. The runtime will call Libdl.dlopen(name) on the value an unspecified number of times, caching the result. The result is not invalidated if the value of the binding changes or if it becomes undefined, as long as there exists any value for that binding in any past or future worlds, that value may be used.\nDot expressions, such as A.B().c, will be executed at method definition time up to the final c. The first part must resolve to a Module, and the second part to a quoted symbol. The value of that global will be resolved at runtime when the ccall is first executed.\n\nA table of translations between the macro and function interfaces is given below.\n\n@ccall ccall\n@ccall clock()::Int32 ccall(:clock, Int32, ())\n@ccall f(a::Cint)::Cint ccall(:f, Cint, (Cint,), a)\n@ccall \"mylib\".f(a::Cint, b::Cdouble)::Cvoid ccall((:f, \"mylib\"), Cvoid, (Cint, Cdouble), a, b)\n@ccall $fptr.f()::Cvoid ccall(fptr, f, Cvoid, ())\n@ccall printf(\"%s = %d\\n\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint <unavailable>\n@ccall printf(\"%s = %s\\n\"::Cstring ; \"2 + 2\"::Cstring, \"5\"::Cstring)::Cint ccall(:printf, Cint, (Cstring, Cstring...), \"%s = %s\\n\", \"2 + 2\", \"5\")\n<unavailable> ccall(:gethostname, stdcall, Int32, (Ptr{UInt8}, UInt32), hn, length(hn))"},{"location":"manual/calling-c-and-fortran-code.html#calling-convention","page":"Calling C and Fortran Code","title":"Calling Convention","category":"section","text":"The second argument to ccall (immediately preceding return type) can optionally be a calling convention specifier (the @ccall macro currently does not support giving a calling convention). Without any specifier, the platform-default C calling convention is used. Other supported conventions are: stdcall, cdecl, fastcall, and thiscall (no-op on 64-bit Windows). For example (from base/libc.jl) we see the same gethostname ccall as above, but with the correct signature for Windows:\n\nhn = Vector{UInt8}(undef, 256)\nerr = ccall(:gethostname, stdcall, Int32, (Ptr{UInt8}, UInt32), hn, length(hn))\n\nFor more information, please see the LLVM Language Reference.\n\nThere is one additional special calling convention llvmcall, which allows inserting calls to LLVM intrinsics directly. This can be especially useful when targeting unusual platforms such as GPGPUs. For example, for CUDA, we need to be able to read the thread index:\n\nccall(\"llvm.nvvm.read.ptx.sreg.tid.x\", llvmcall, Int32, ())\n\nAs with any ccall, it is essential to get the argument signature exactly correct. Also, note that there is no compatibility layer that ensures the intrinsic makes sense and works on the current target, unlike the equivalent Julia functions exposed by Core.Intrinsics."},{"location":"manual/calling-c-and-fortran-code.html#Accessing-Global-Variables","page":"Calling C and Fortran Code","title":"Accessing Global Variables","category":"section","text":"Global variables exported by native libraries can be accessed by name using the cglobal function. The arguments to cglobal are a symbol specification identical to that used by ccall, and a type describing the value stored in the variable:\n\njulia> cglobal((:errno, :libc), Int32)\nPtr{Int32} @0x00007f418d0816b8\n\nThe result is a pointer giving the address of the value. The value can be manipulated through this pointer using unsafe_load and unsafe_store!.\n\nnote: Note\nThis errno symbol may not be found in a library named \"libc\", as this is an implementation detail of your system compiler. Typically standard library symbols should be accessed just by name, allowing the compiler to fill in the correct one. Also, however, the errno symbol shown in this example is special in most compilers, and so the value seen here is probably not what you expect or want. Compiling the equivalent code in C on any multi-threaded-capable system would typically actually call a different function (via macro preprocessor overloading), and may give a different result than the legacy value printed here."},{"location":"manual/calling-c-and-fortran-code.html#Accessing-Data-through-a-Pointer","page":"Calling C and Fortran Code","title":"Accessing Data through a Pointer","category":"section","text":"The following methods are described as \"unsafe\" because a bad pointer or type declaration can cause Julia to terminate abruptly.\n\nGiven a Ptr{T}, the contents of type T can generally be copied from the referenced memory into a Julia object using unsafe_load(ptr, [index]). The index argument is optional (default is 1), and follows the Julia-convention of 1-based indexing. This function is intentionally similar to the behavior of getindex and setindex! (e.g. [] access syntax).\n\nThe return value will be a new object initialized to contain a copy of the contents of the referenced memory. The referenced memory can safely be freed or released.\n\nIf T is Any, then the memory is assumed to contain a reference to a Julia object (a jl_value_t*), the result will be a reference to this object, and the object will not be copied. You must be careful in this case to ensure that the object was always visible to the garbage collector (pointers do not count, but the new reference does) to ensure the memory is not prematurely freed. Note that if the object was not originally allocated by Julia, the new object will never be finalized by Julia's garbage collector. If the Ptr itself is actually a jl_value_t*, it can be converted back to a Julia object reference by unsafe_pointer_to_objref(ptr). (Julia values v can be converted to jl_value_t* pointers, as Ptr{Cvoid}, by calling pointer_from_objref(v).)\n\nThe reverse operation (writing data to a Ptr{T}), can be performed using unsafe_store!(ptr, value, [index]). Currently, this is only supported for primitive types or other pointer-free (isbits) immutable struct types.\n\nAny operation that throws an error is probably currently unimplemented and should be posted as a bug so that it can be resolved.\n\nIf the pointer of interest is a plain-data array (primitive type or immutable struct), the function unsafe_wrap(Array, ptr,dims, own = false) may be more useful. The final parameter should be true if Julia should \"take ownership\" of the underlying buffer and call free(ptr) when the returned Array object is finalized. If the own parameter is omitted or false, the caller must ensure the buffer remains in existence until all access is complete.\n\nArithmetic on the Ptr type in Julia (e.g. using +) does not behave the same as C's pointer arithmetic. Adding an integer to a Ptr in Julia always moves the pointer by some number of bytes, not elements. This way, the address values obtained from pointer arithmetic do not depend on the element types of pointers."},{"location":"manual/calling-c-and-fortran-code.html#Thread-safety","page":"Calling C and Fortran Code","title":"Thread-safety","category":"section","text":"Some C libraries execute their callbacks from a different thread, and since Julia isn't thread-safe you'll need to take some extra precautions. In particular, you'll need to set up a two-layered system: the C callback should only schedule (via Julia's event loop) the execution of your \"real\" callback. To do this, create an AsyncCondition object and wait on it:\n\ncond = Base.AsyncCondition()\nwait(cond)\n\nThe callback you pass to C should only execute a ccall to :uv_async_send, passing cond.handle as the argument, taking care to avoid any allocations or other interactions with the Julia runtime.\n\nNote that events may be coalesced, so multiple calls to uv_async_send may result in a single wakeup notification to the condition."},{"location":"manual/calling-c-and-fortran-code.html#More-About-Callbacks","page":"Calling C and Fortran Code","title":"More About Callbacks","category":"section","text":"For more details on how to pass callbacks to C libraries, see this blog post."},{"location":"manual/calling-c-and-fortran-code.html#C","page":"Calling C and Fortran Code","title":"C++","category":"section","text":"For tools to create C++ bindings, see the CxxWrap package.\n\n[1]: Non-library function calls in both C and Julia can be inlined and thus may have even less overhead than calls to shared library functions. The point above is that the cost of actually doing foreign function call is about the same as doing a call in either native language.\n\n[2]: The Clang package can be used to auto-generate Julia code from a C header file."},{"location":"stdlib/SparseArrays.html#Sparse-Arrays","page":"Sparse Arrays","title":"Sparse Arrays","category":"section","text":"Julia has support for sparse vectors and sparse matrices in the SparseArrays stdlib module. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays.\n\nExternal packages which implement different sparse storage types, multidimensional sparse arrays, and more can be found in Noteworthy External Sparse Packages"},{"location":"stdlib/SparseArrays.html#man-csc","page":"Sparse Arrays","title":"Compressed Sparse Column (CSC) Sparse Matrix Storage","category":"section","text":"In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format. Julia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the type of the stored values, and Ti is the integer type for storing column pointers and row indices. The internal representation of SparseMatrixCSC is as follows:\n\nstruct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column j is in colptr[j]:(colptr[j+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nend\n\nThe compressed sparse column storage makes it easy and quick to access the elements in the column of a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations such as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is because all elements of the sparse matrix that are beyond the point of insertion have to be moved one place over.\n\nAll operations on sparse matrices are carefully implemented to exploit the CSC data structure for performance, and to avoid expensive operations.\n\nIf you have data in CSC format from a different application or library, and wish to import it in Julia, make sure that you use 1-based indexing. The row indices in every column need to be sorted, and if they are not, the matrix will display incorrectly.  If your SparseMatrixCSC object contains unsorted row indices, one quick way to sort them is by doing a double transpose. Since the transpose operation is lazy, make a copy to materialize each transpose.\n\nIn some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These are accepted by functions in Base (but there is no guarantee that they will be preserved in mutating operations). Such explicitly stored zeros are treated as structural nonzeros by many routines. The nnz function returns the number of elements explicitly stored in the sparse data structure, including non-structural zeros. In order to count the exact number of numerical nonzeros, use count(!iszero, x), which inspects every stored element of a sparse matrix. dropzeros, and the in-place dropzeros!, can be used to remove stored zeros from the sparse matrix.\n\njulia> A = sparse([1, 1, 2, 3], [1, 3, 2, 3], [0, 1, 2, 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 0  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n ⋅  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  ⋅"},{"location":"stdlib/SparseArrays.html#Sparse-Vector-Storage","page":"Sparse Arrays","title":"Sparse Vector Storage","category":"section","text":"Sparse vectors are stored in a close analog to compressed sparse column format for sparse matrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv is the type of the stored values and Ti the integer type for the indices. The internal representation is as follows:\n\nstruct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nend\n\nLike SparseMatrixCSC, the SparseVector type can also contain explicitly stored zeros. (See Sparse Matrix Storage.)."},{"location":"stdlib/SparseArrays.html#Sparse-Vector-and-Matrix-Constructors","page":"Sparse Arrays","title":"Sparse Vector and Matrix Constructors","category":"section","text":"The simplest way to create a sparse array is to use a function equivalent to the zeros function that Julia provides for working with dense arrays. To produce a sparse array instead, you can use the same name with an sp prefix:\n\njulia> spzeros(3)\n3-element SparseVector{Float64, Int64} with 0 stored entries\n\nThe sparse function is often a handy way to construct sparse arrays. For example, to construct a sparse matrix we can input a vector I of row indices, a vector J of column indices, and a vector V of stored values (this is also known as the COO (coordinate) format). sparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The equivalent sparse vector constructor is sparsevec, which takes the (row) index vector I and the vector V with the stored values and constructs a sparse vector R such that R[I[k]] = V[k].\n\njulia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n⎡⠀⠈⠀⠀⠀⠀⠀⠀⢀⎤\n⎣⠀⠀⠀⠂⡀⠀⠀⠀⠀⎦\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64, Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3\n\nThe inverse of the sparse and sparsevec functions is findnz, which retrieves the inputs used to create the sparse array (including stored entries equal to zero). findall(!iszero, x) returns the Cartesian indices of non-zero entries in x (not including stored entries equal to zero).\n\njulia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findall(!iszero, S)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 4)\n CartesianIndex(4, 7)\n CartesianIndex(5, 9)\n CartesianIndex(3, 18)\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findall(!iszero, R)\n4-element Vector{Int64}:\n 1\n 3\n 4\n 5\n\nAnother way to create a sparse array is to convert a dense array into a sparse array using the sparse function:\n\njulia> sparse(Matrix(1.0I, 5, 5))\n5×5 SparseMatrixCSC{Float64, Int64} with 5 stored entries:\n 1.0   ⋅    ⋅    ⋅    ⋅\n  ⋅   1.0   ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅   1.0   ⋅\n  ⋅    ⋅    ⋅    ⋅   1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\nYou can go in the other direction using the Array constructor. The issparse function can be used to query if a matrix is sparse.\n\njulia> issparse(spzeros(5))\ntrue"},{"location":"stdlib/SparseArrays.html#Sparse-matrix-operations","page":"Sparse Arrays","title":"Sparse matrix operations","category":"section","text":"Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of, assignment into, and concatenation of sparse matrices work in the same way as dense matrices. Indexing operations, especially assignment, are expensive, when carried out one element at a time. In many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz, manipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct the sparse matrix."},{"location":"stdlib/SparseArrays.html#Correspondence-of-dense-and-sparse-methods","page":"Sparse Arrays","title":"Correspondence of dense and sparse methods","category":"section","text":"The following table gives a correspondence between built-in methods on sparse matrices and their corresponding methods on dense matrix types. In general, methods that generate sparse matrices differ from their dense counterparts in that the resulting matrix follows the same sparsity pattern as a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix element has a probability d of being non-zero.\n\nDetails can be found in the Sparse Vectors and Matrices section of the standard library reference.\n\nSparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nsparse(I,n,n) Matrix(I,n,n) Creates a n-by-n identity matrix.\nsparse(A) Array(S) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(rng,m,n,d) randn(rng,m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements generated with the rng random number generator"},{"location":"stdlib/SparseArrays.html#stdlib-sparse-arrays","page":"Sparse Arrays","title":"SparseArrays API","category":"section","text":""},{"location":"stdlib/SparseArrays.html#Noteworthy-External-Sparse-Packages","page":"Sparse Arrays","title":"Noteworthy External Sparse Packages","category":"section","text":"Several other Julia packages provide sparse matrix implementations that should be mentioned:\n\nSuiteSparseGraphBLAS.jl is a wrapper over the fast, multithreaded SuiteSparse:GraphBLAS C library. On CPU this is typically the fastest option, often significantly outperforming MKLSparse.\nCUDA.jl exposes the CUSPARSE library for GPU sparse matrix operations.\nSparseMatricesCSR.jl provides a Julia native implementation of the Compressed Sparse Rows (CSR) format.\nMKLSparse.jl accelerates SparseArrays sparse-dense matrix operations using Intel's MKL library.\nSparseArrayKit.jl available for multidimensional sparse arrays.\nLuxurySparse.jl provides static sparse array formats, as well as a coordinate format.\nExtendableSparse.jl enables fast insertion into sparse matrices using a lazy approach to new stored indices.\nFinch.jl supports extensive multidimensional sparse array formats and operations through a mini tensor language and compiler, all in native Julia. Support for COO, CSF, CSR, CSC and more, as well as operations like broadcast, reduce, etc. and custom operations.\n\nExternal packages providing sparse direct solvers:\n\nKLU.jl\nPardiso.jl\n\nExternal packages providing solvers for iterative solution of eigensystems and singular value decompositions:\n\nArnoldiMethods.jl\nKrylovKit\nArpack.jl\n\nExternal packages for working with graphs:\n\nGraphs.jl"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseArray","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseArray","category":"type","text":"AbstractSparseArray{Tv,Ti,N}\n\nSupertype for N-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. SparseMatrixCSC, SparseVector and SuiteSparse.CHOLMOD.Sparse are subtypes of this.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseVector","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseVector","category":"type","text":"AbstractSparseVector{Tv,Ti}\n\nSupertype for one-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,1}.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseMatrix","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseMatrix","category":"type","text":"AbstractSparseMatrix{Tv,Ti}\n\nSupertype for two-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,2}.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.SparseVector","page":"Sparse Arrays","title":"SparseArrays.SparseVector","category":"type","text":"SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n\nVector type for storing sparse vectors. Can be created by passing the length of the vector, a sorted vector of non-zero indices, and a vector of non-zero values.\n\nFor instance, the vector [5, 6, 0, 7] can be represented as\n\nSparseVector(4, [1, 2, 4], [5, 6, 7])\n\nThis indicates that the element at index 1 is 5, at index 2 is 6, at index 3 is zero(Int), and at index 4 is 7.\n\nIt may be more convenient to create sparse vectors directly from dense vectors using sparse as\n\nsparse([5, 6, 0, 7])\n\nyields the same sparse vector.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.SparseMatrixCSC","page":"Sparse Arrays","title":"SparseArrays.SparseMatrixCSC","category":"type","text":"SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n\nMatrix type for storing sparse matrices in the Compressed Sparse Column format. The standard way of constructing SparseMatrixCSC is through the sparse function. See also spzeros, spdiagm and sprand.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse","page":"Sparse Arrays","title":"SparseArrays.sparse","category":"function","text":"sparse(A::Union{AbstractVector, AbstractMatrix})\n\nConvert a vector or matrix A into a sparse array. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> A = Matrix(1.0I, 3, 3)\n3×3 Matrix{Float64}:\n 1.0  0.0  0.0\n 0.0  1.0  0.0\n 0.0  0.0  1.0\n\njulia> sparse(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   1.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> [1.0, 0.0, 1.0]\n3-element Vector{Float64}:\n 1.0\n 0.0\n 1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\nsparse(I, J, V,[ m, n, combine])\n\nCreate a sparse matrix S of dimensions m x n such that S[I[k], J[k]] = V[k]. The combine function is used to combine duplicates. If m and n are not specified, they are set to maximum(I) and maximum(J) respectively. If the combine function is not supplied, combine defaults to + unless the elements of V are Booleans in which case combine defaults to |. All elements of I must satisfy 1 <= I[k] <= m, and all elements of J must satisfy 1 <= J[k] <= n. Numerical zeros in (I, J, V) are retained as structural nonzeros; to drop numerical zeros, use dropzeros!.\n\nFor additional documentation and an expert driver, see SparseArrays.sparse!.\n\nExamples\n\njulia> Is = [1; 2; 3];\n\njulia> Js = [1; 2; 3];\n\njulia> Vs = [1; 2; 3];\n\njulia> sparse(Is, Js, Vs)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse!","page":"Sparse Arrays","title":"SparseArrays.sparse!","category":"function","text":"sparse!(I::AbstractVector{Ti}, J::AbstractVector{Ti}, V::AbstractVector{Tv},\n        m::Integer, n::Integer, combine, klasttouch::Vector{Ti},\n        csrrowptr::Vector{Ti}, csrcolval::Vector{Ti}, csrnzval::Vector{Tv},\n        [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}] ) where {Tv,Ti<:Integer}\n\nParent of and expert driver for sparse; see sparse for basic usage. This method allows the user to provide preallocated storage for sparse's intermediate objects and result as described below. This capability enables more efficient successive construction of SparseMatrixCSCs from coordinate representations, and also enables extraction of an unsorted-column representation of the result's transpose at no additional cost.\n\nThis method consists of three major steps: (1) Counting-sort the provided coordinate representation into an unsorted-row CSR form including repeated entries. (2) Sweep through the CSR form, simultaneously calculating the desired CSC form's column-pointer array, detecting repeated entries, and repacking the CSR form with repeated entries combined; this stage yields an unsorted-row CSR form with no repeated entries. (3) Counting-sort the preceding CSR form into a fully-sorted CSC form with no repeated entries.\n\nInput arrays csrrowptr, csrcolval, and csrnzval constitute storage for the intermediate CSR forms and require length(csrrowptr) >= m + 1, length(csrcolval) >= length(I), and length(csrnzval >= length(I)). Input array klasttouch, workspace for the second stage, requires length(klasttouch) >= n. Optional input arrays csccolptr, cscrowval, and cscnzval constitute storage for the returned CSC form S. If necessary, these are resized automatically to satisfy length(csccolptr) = n + 1, length(cscrowval) = nnz(S) and length(cscnzval) = nnz(S); hence, if nnz(S) is unknown at the outset, passing in empty vectors of the appropriate type (Vector{Ti}() and Vector{Tv}() respectively) suffices, or calling the sparse! method neglecting cscrowval and cscnzval.\n\nOn return, csrrowptr, csrcolval, and csrnzval contain an unsorted-column representation of the result's transpose.\n\nYou may reuse the input arrays' storage (I, J, V) for the output arrays (csccolptr, cscrowval, cscnzval). For example, you may call sparse!(I, J, V, csrrowptr, csrcolval, csrnzval, I, J, V). Note that they will be resized to satisfy the conditions above.\n\nFor the sake of efficiency, this method performs no argument checking beyond 1 <= I[k] <= m and 1 <= J[k] <= n. Use with care. Testing with --check-bounds=yes is wise.\n\nThis method runs in O(m, n, length(I)) time. The HALFPERM algorithm described in F. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3), 250-269 (1978) inspired this method's use of a pair of counting sorts.\n\n\n\n\n\nSparseArrays.sparse!(I, J, V, [m, n, combine]) -> SparseMatrixCSC\n\nVariant of sparse! that re-uses the input vectors (I, J, V) for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I, S.rowval === J, and S.nzval === V holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around sparse!(I, J, V, m, n, combine, klasttouch, csrrowptr, csrcolval, csrnzval, csccolptr, cscrowval, cscnzval) where this method allocates klasttouch, csrrowptr, csrcolval, and csrnzval of appropriate size, but reuses I, J, and V for csccolptr, cscrowval, and cscnzval.\n\nArguments m, n, and combine defaults to maximum(I), maximum(J), and +, respectively.\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparsevec","page":"Sparse Arrays","title":"SparseArrays.sparsevec","category":"function","text":"sparsevec(I, V, [m, combine])\n\nCreate a sparse vector S of length m such that S[I[k]] = V[k]. Duplicates are combined using the combine function, which defaults to + if no combine argument is provided, unless the elements of V are Booleans in which case combine defaults to |.\n\nExamples\n\njulia> II = [1, 3, 3, 5]; V = [0.1, 0.2, 0.3, 0.2];\n\njulia> sparsevec(II, V)\n5-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  0.5\n  [5]  =  0.2\n\njulia> sparsevec(II, V, 8, -)\n8-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  -0.1\n  [5]  =  0.2\n\njulia> sparsevec([1, 3, 1, 2, 2], [true, true, false, false, false])\n3-element SparseVector{Bool, Int64} with 3 stored entries:\n  [1]  =  1\n  [2]  =  0\n  [3]  =  1\n\n\n\n\n\nsparsevec(d::Dict, [m])\n\nCreate a sparse vector of length m where the nonzero indices are keys from the dictionary, and the nonzero values are the values from the dictionary.\n\nExamples\n\njulia> sparsevec(Dict(1 => 3, 2 => 2))\n2-element SparseVector{Int64, Int64} with 2 stored entries:\n  [1]  =  3\n  [2]  =  2\n\n\n\n\n\nsparsevec(A)\n\nConvert a vector A into a sparse vector of length m. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> sparsevec([1.0, 2.0, 0.0, 0.0, 3.0, 0.0])\n6-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  2.0\n  [5]  =  3.0\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#Base.similar-Tuple{SparseArrays.AbstractSparseMatrixCSC, Type}","page":"Sparse Arrays","title":"Base.similar","category":"method","text":"similar(A::AbstractSparseMatrixCSC{Tv,Ti}, [::Type{TvNew}, ::Type{TiNew}, m::Integer, n::Integer]) where {Tv,Ti}\n\nCreate an uninitialized mutable array with the given element type, index type, and size, based upon the given source SparseMatrixCSC. The new sparse matrix maintains the structure of the original sparse matrix, except in the case where dimensions of the output matrix are different from the output.\n\nThe output matrix has zeros in the same locations as the input, but uninitialized values for the nonzero locations.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.issparse","page":"Sparse Arrays","title":"SparseArrays.issparse","category":"function","text":"issparse(S)\n\nReturns true if S is sparse, and false otherwise.\n\nExamples\n\njulia> sv = sparsevec([1, 4], [2.3, 2.2], 10)\n10-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  2.3\n  [4]  =  2.2\n\njulia> issparse(sv)\ntrue\n\njulia> issparse(Array(sv))\nfalse\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nnz","page":"Sparse Arrays","title":"SparseArrays.nnz","category":"function","text":"nnz(A)\n\nReturns the number of stored (filled) elements in a sparse array.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nnz(A)\n3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.findnz","page":"Sparse Arrays","title":"SparseArrays.findnz","category":"function","text":"findnz(A::SparseMatrixCSC)\n\nReturn a tuple (I, J, V) where I and J are the row and column indices of the stored (\"structurally non-zero\") values in sparse matrix A, and V is a vector of the values.\n\nExamples\n\njulia> A = sparse([1 2 0; 0 0 3; 0 4 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 1  2  ⋅\n ⋅  ⋅  3\n ⋅  4  ⋅\n\njulia> findnz(A)\n([1, 1, 3, 2], [1, 2, 2, 3], [1, 2, 4, 3])\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spzeros","page":"Sparse Arrays","title":"SparseArrays.spzeros","category":"function","text":"spzeros([type,]m[,n])\n\nCreate a sparse vector of length m or sparse matrix of size m x n. This sparse array will not contain any nonzero values. No storage will be allocated for nonzero values during construction. The type defaults to Float64 if not specified.\n\nExamples\n\njulia> spzeros(3, 3)\n3×3 SparseMatrixCSC{Float64, Int64} with 0 stored entries:\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n\njulia> spzeros(Float32, 4)\n4-element SparseVector{Float32, Int64} with 0 stored entries\n\n\n\n\n\nspzeros([type], I::AbstractVector, J::AbstractVector, [m, n])\n\nCreate a sparse matrix S of dimensions m x n with structural zeros at S[I[k], J[k]].\n\nThis method can be used to construct the sparsity pattern of the matrix, and is more efficient than using e.g. sparse(I, J, zeros(length(I))).\n\nFor additional documentation and an expert driver, see SparseArrays.spzeros!.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spzeros!","page":"Sparse Arrays","title":"SparseArrays.spzeros!","category":"function","text":"spzeros!(::Type{Tv}, I::AbstractVector{Ti}, J::AbstractVector{Ti}, m::Integer, n::Integer,\n         klasttouch::Vector{Ti}, csrrowptr::Vector{Ti}, csrcolval::Vector{Ti},\n         [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}]) where {Tv,Ti<:Integer}\n\nParent of and expert driver for spzeros(I, J) allowing user to provide preallocated storage for intermediate objects. This method is to spzeros what SparseArrays.sparse! is to sparse. See documentation for SparseArrays.sparse! for details and required buffer lengths.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\nSparseArrays.spzeros!(::Type{Tv}, I, J, [m, n]) -> SparseMatrixCSC{Tv}\n\nVariant of spzeros! that re-uses the input vectors I and J for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I and S.rowval === J holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around spzeros!(Tv, I, J, m, n, klasttouch, csrrowptr, csrcolval, csccolptr, cscrowval) where this method allocates klasttouch, csrrowptr, and csrcolval of appropriate size, but reuses I and J for csccolptr and cscrowval.\n\nArguments m and n defaults to maximum(I) and maximum(J).\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spdiagm","page":"Sparse Arrays","title":"SparseArrays.spdiagm","category":"function","text":"spdiagm(kv::Pair{<:Integer,<:AbstractVector}...)\nspdiagm(m::Integer, n::Integer, kv::Pair{<:Integer,<:AbstractVector}...)\n\nConstruct a sparse diagonal matrix from Pairs of vectors and diagonals. Each vector kv.second will be placed on the kv.first diagonal.  By default, the matrix is square and its size is inferred from kv, but a non-square size m×n (padded with zeros as needed) can be specified by passing m,n as the first arguments.\n\nExamples\n\njulia> spdiagm(-1 => [1,2,3,4], 1 => [4,3,2,1])\n5×5 SparseMatrixCSC{Int64, Int64} with 8 stored entries:\n ⋅  4  ⋅  ⋅  ⋅\n 1  ⋅  3  ⋅  ⋅\n ⋅  2  ⋅  2  ⋅\n ⋅  ⋅  3  ⋅  1\n ⋅  ⋅  ⋅  4  ⋅\n\n\n\n\n\nspdiagm(v::AbstractVector)\nspdiagm(m::Integer, n::Integer, v::AbstractVector)\n\nConstruct a sparse matrix with elements of the vector as diagonal elements. By default (no given m and n), the matrix is square and its size is given by length(v), but a non-square size m×n can be specified by passing m and n as the first arguments.\n\ncompat: Julia 1.6\nThese functions require at least Julia 1.6.\n\nExamples\n\njulia> spdiagm([1,2,3])\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\njulia> spdiagm(sparse([1,0,3]))\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n 1  ⋅  ⋅\n ⋅  ⋅  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_hcat","page":"Sparse Arrays","title":"SparseArrays.sparse_hcat","category":"function","text":"sparse_hcat(A...)\n\nConcatenate along dimension 2. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_vcat","page":"Sparse Arrays","title":"SparseArrays.sparse_vcat","category":"function","text":"sparse_vcat(A...)\n\nConcatenate along dimension 1. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_hvcat","page":"Sparse Arrays","title":"SparseArrays.sparse_hvcat","category":"function","text":"sparse_hvcat(rows::Tuple{Vararg{Int}}, values...)\n\nSparse horizontal and vertical concatenation in one call. This function is called for block matrix syntax. The first argument specifies the number of arguments to concatenate in each block row.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.blockdiag","page":"Sparse Arrays","title":"SparseArrays.blockdiag","category":"function","text":"blockdiag(A...)\n\nConcatenate matrices block-diagonally. Currently only implemented for sparse matrices.\n\nExamples\n\njulia> blockdiag(sparse(2I, 3, 3), sparse(4I, 2, 2))\n5×5 SparseMatrixCSC{Int64, Int64} with 5 stored entries:\n 2  ⋅  ⋅  ⋅  ⋅\n ⋅  2  ⋅  ⋅  ⋅\n ⋅  ⋅  2  ⋅  ⋅\n ⋅  ⋅  ⋅  4  ⋅\n ⋅  ⋅  ⋅  ⋅  4\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sprand","page":"Sparse Arrays","title":"SparseArrays.sprand","category":"function","text":"sprand([rng],[T::Type],m,[n],p::AbstractFloat)\nsprand([rng],m,[n],p::AbstractFloat,[rfn=rand])\n\nCreate a random length m sparse vector or m by n sparse matrix, in which the probability of any element being nonzero is independently given by p (and hence the mean density of nonzeros is also exactly p). The optional rng argument specifies a random number generator, see Random Numbers. The optional T argument specifies the element type, which defaults to Float64.\n\nBy default, nonzero values are sampled from a uniform distribution using the rand function, i.e. by rand(T), or rand(rng, T) if rng is supplied; for the default T=Float64, this corresponds to nonzero values sampled uniformly in [0,1).\n\nYou can sample nonzero values from a different distribution by passing a custom rfn function instead of rand.   This should be a function rfn(k) that returns an array of k random numbers sampled from the desired distribution; alternatively, if rng is supplied, it should instead be a function rfn(rng, k).\n\nExamples\n\njulia> sprand(Bool, 2, 2, 0.5)\n2×2 SparseMatrixCSC{Bool, Int64} with 2 stored entries:\n 1  1\n ⋅  ⋅\n\njulia> sprand(Float64, 3, 0.75)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  0.795547\n  [2]  =  0.49425\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sprandn","page":"Sparse Arrays","title":"SparseArrays.sprandn","category":"function","text":"sprandn([rng][,Type],m[,n],p::AbstractFloat)\n\nCreate a random sparse vector of length m or sparse matrix of size m by n with the specified (independent) probability p of any entry being nonzero, where nonzero values are sampled from the normal distribution. The optional rng argument specifies a random number generator, see Random Numbers.\n\ncompat: Julia 1.1\nSpecifying the output element type Type requires at least Julia 1.1.\n\nExamples\n\njulia> sprandn(2, 2, 0.75)\n2×2 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n -1.20577     ⋅\n  0.311817  -0.234641\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nonzeros","page":"Sparse Arrays","title":"SparseArrays.nonzeros","category":"function","text":"nonzeros(A)\n\nReturn a vector of the structural nonzero values in sparse array A. This includes zeros that are explicitly stored in the sparse array. The returned vector points directly to the internal nonzero storage of A, and any modifications to the returned vector will mutate A as well. See rowvals and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nonzeros(A)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.rowvals","page":"Sparse Arrays","title":"SparseArrays.rowvals","category":"function","text":"rowvals(A)\n\nReturn a vector of the row indices of sparse array A. Any modifications to the returned vector will mutate A as well. Providing access to how the row indices are stored internally can be useful in conjunction with iterating over structural nonzero values. See also nonzeros and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> rowvals(A)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nzrange","page":"Sparse Arrays","title":"SparseArrays.nzrange","category":"function","text":"nzrange(A, col::Integer)\n\nReturn the range of indices to the structural nonzero values of column col of sparse array A. In conjunction with nonzeros and rowvals, this allows for convenient iterating over a sparse matrix :\n\nA = sparse(I,J,V)\nrows = rowvals(A)\nvals = nonzeros(A)\nm, n = size(A)\nfor j = 1:n\n   for i in nzrange(A, j)\n      row = rows[i]\n      val = vals[i]\n      # perform sparse wizardry...\n   end\nend\n\nwarning: Warning\nAdding or removing nonzero elements to the matrix may invalidate the nzrange, one should not mutate the matrix while iterating.\n\n\n\n\n\nnzrange(x::SparseVectorUnion, col)\n\nGive the range of indices to the structural nonzero values of a sparse vector. The column index col is ignored (assumed to be 1).\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.droptol!","page":"Sparse Arrays","title":"SparseArrays.droptol!","category":"function","text":"droptol!(A::AbstractSparseMatrixCSC, tol)\n\nRemoves stored values from A whose absolute value is less than or equal to tol.\n\n\n\n\n\ndroptol!(x::AbstractCompressedVector, tol)\n\nRemoves stored values from x whose absolute value is less than or equal to tol.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.dropzeros!","page":"Sparse Arrays","title":"SparseArrays.dropzeros!","category":"function","text":"dropzeros!(x::AbstractCompressedVector)\n\nRemoves stored numerical zeros from x.\n\nFor an out-of-place version, see dropzeros. For algorithmic information, see fkeep!.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.dropzeros","page":"Sparse Arrays","title":"SparseArrays.dropzeros","category":"function","text":"dropzeros(A::AbstractSparseMatrixCSC;)\n\nGenerates a copy of A and removes stored numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparse([1, 2, 3], [1, 2, 3], [1.0, 0.0, 1.0])\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   0.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 2 stored entries:\n 1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0\n\n\n\n\n\ndropzeros(x::AbstractCompressedVector)\n\nGenerates a copy of x and removes numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparsevec([1, 2, 3], [1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  0.0\n  [3]  =  1.0\n\njulia> dropzeros(A)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.permute","page":"Sparse Arrays","title":"SparseArrays.permute","category":"function","text":"permute(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n        q::AbstractVector{<:Integer}) where {Tv,Ti}\n\nBilaterally permute A, returning PAQ (A[p,q]). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nFor expert drivers and additional information, see permute!.\n\nExamples\n\njulia> A = spdiagm(0 => [1, 2, 3, 4], 1 => [5, 6, 7])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n 1  5  ⋅  ⋅\n ⋅  2  6  ⋅\n ⋅  ⋅  3  7\n ⋅  ⋅  ⋅  4\n\njulia> permute(A, [4, 3, 2, 1], [1, 2, 3, 4])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  ⋅  4\n ⋅  ⋅  3  7\n ⋅  2  6  ⋅\n 1  5  ⋅  ⋅\n\njulia> permute(A, [1, 2, 3, 4], [4, 3, 2, 1])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  5  1\n ⋅  6  2  ⋅\n 7  3  ⋅  ⋅\n 4  ⋅  ⋅  ⋅\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#Base.permute!-Union{Tuple{Tq}, Tuple{Tp}, Tuple{Ti}, Tuple{Tv}, Tuple{SparseMatrixCSC{Tv, Ti}, SparseMatrixCSC{Tv, Ti}, AbstractVector{Tp}, AbstractVector{Tq}}} where {Tv, Ti, Tp<:Integer, Tq<:Integer}","page":"Sparse Arrays","title":"Base.permute!","category":"method","text":"permute!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti},\n         p::AbstractVector{<:Integer}, q::AbstractVector{<:Integer},\n         [C::AbstractSparseMatrixCSC{Tv,Ti}]) where {Tv,Ti}\n\nBilaterally permute A, storing result PAQ (A[p,q]) in X. Stores intermediate result (AQ)^T (transpose(A[:,q])) in optional argument C if present. Requires that none of X, A, and, if present, C alias each other; to store result PAQ back into A, use the following method lacking X:\n\npermute!(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n         q::AbstractVector{<:Integer}[, C::AbstractSparseMatrixCSC{Tv,Ti},\n         [workcolptr::Vector{Ti}]]) where {Tv,Ti}\n\nX's dimensions must match those of A (size(X, 1) == size(A, 1) and size(X, 2) == size(A, 2)), and X must have enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nC's dimensions must match those of transpose(A) (size(C, 1) == size(A, 2) and size(C, 2) == size(A, 1)), and C must have enough storage to accommodate all allocated entries in A (length(rowvals(C)) >= nnz(A) and length(nonzeros(C)) >= nnz(A)).\n\nFor additional (algorithmic) information, and for versions of these methods that forgo argument checking, see (unexported) parent methods unchecked_noalias_permute! and unchecked_aliasing_permute!.\n\nSee also permute.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.halfperm!","page":"Sparse Arrays","title":"SparseArrays.halfperm!","category":"function","text":"halfperm!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{TvA,Ti},\n          q::AbstractVector{<:Integer}, f::Function = identity) where {Tv,TvA,Ti}\n\nColumn-permute and transpose A, simultaneously applying f to each entry of A, storing the result (f(A)Q)^T (map(f, transpose(A[:,q]))) in X.\n\nElement type Tv of X must match f(::TvA), where TvA is the element type of A. X's dimensions must match those of transpose(A) (size(X, 1) == size(A, 2) and size(X, 2) == size(A, 1)), and X must have enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column count (length(q) == size(A, 2)).\n\nThis method is the parent of several methods performing transposition and permutation operations on SparseMatrixCSCs. As this method performs no argument checking, prefer the safer child methods ([c]transpose[!], permute[!]) to direct use.\n\nThis method implements the HALFPERM algorithm described in F. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3), 250-269 (1978). The algorithm runs in O(size(A, 1), size(A, 2), nnz(A)) time and requires no space beyond that passed in.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.ftranspose!","page":"Sparse Arrays","title":"SparseArrays.ftranspose!","category":"function","text":"ftranspose!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti}, f::Function) where {Tv,Ti}\n\nTranspose A and store it in X while applying the function f to the non-zero elements. Does not remove the zeros created by f. size(X) must be equal to size(transpose(A)). No additional memory is allocated other than resizing the rowval and nzval of X, if needed.\n\nSee halfperm!\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics","page":"Statistics","title":"Statistics","category":"section","text":"The Statistics standard library module contains basic statistics functionality."},{"location":"stdlib/Statistics.html#Statistics.std","page":"Statistics","title":"Statistics.std","category":"function","text":"std(itr; corrected::Bool=true, mean=nothing[, dims])\n\nCompute the sample standard deviation of collection itr.\n\nThe algorithm returns an estimator of the generative distribution's standard deviation under the assumption that each entry of itr is a sample drawn from the same unknown distribution, with the samples uncorrelated. For arrays, this computation is equivalent to calculating sqrt.(sum(abs2.(itr .- mean(itr))) / (length(itr) - 1)). If corrected is true, then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false with n the number of elements in itr.\n\nIf itr is an AbstractArray, dims can be provided to compute the standard deviation over dimensions.\n\nA pre-computed mean may be provided. When dims is specified, mean must be an array with the same shape as mean(itr, dims=dims) (additional trailing singleton dimensions are allowed).\n\nnote: Note\nIf array contains NaN or missing values, the result is also NaN or missing (missing takes precedence if array contains both). Use the skipmissing function to omit missing entries and compute the standard deviation of non-missing values.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.stdm","page":"Statistics","title":"Statistics.stdm","category":"function","text":"stdm(itr, mean; corrected::Bool=true[, dims])\n\nCompute the sample standard deviation of collection itr, with known mean(s) mean.\n\nThe algorithm returns an estimator of the generative distribution's standard deviation under the assumption that each entry of itr is a sample drawn from the same unknown distribution, with the samples uncorrelated. For arrays, this computation is equivalent to calculating sqrt.(sum(abs2.(itr .- mean(itr))) / (length(itr) - 1)). If corrected is true, then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false with n the number of elements in itr.\n\nIf itr is an AbstractArray, dims can be provided to compute the standard deviation over dimensions. In that case, mean must be an array with the same shape as mean(itr, dims=dims) (additional trailing singleton dimensions are allowed).\n\nnote: Note\nIf array contains NaN or missing values, the result is also NaN or missing (missing takes precedence if array contains both). Use the skipmissing function to omit missing entries and compute the standard deviation of non-missing values.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.var","page":"Statistics","title":"Statistics.var","category":"function","text":"var(itr; corrected::Bool=true, mean=nothing[, dims])\n\nCompute the sample variance of collection itr.\n\nThe algorithm returns an estimator of the generative distribution's variance under the assumption that each entry of itr is a sample drawn from the same unknown distribution, with the samples uncorrelated. For arrays, this computation is equivalent to calculating sum(abs2.(itr .- mean(itr))) / (length(itr) - 1). If corrected is true, then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false where n is the number of elements in itr.\n\nIf itr is an AbstractArray, dims can be provided to compute the variance over dimensions.\n\nA pre-computed mean may be provided. When dims is specified, mean must be an array with the same shape as mean(itr, dims=dims) (additional trailing singleton dimensions are allowed).\n\nnote: Note\nIf array contains NaN or missing values, the result is also NaN or missing (missing takes precedence if array contains both). Use the skipmissing function to omit missing entries and compute the variance of non-missing values.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.varm","page":"Statistics","title":"Statistics.varm","category":"function","text":"varm(itr, mean; dims, corrected::Bool=true)\n\nCompute the sample variance of collection itr, with known mean(s) mean.\n\nThe algorithm returns an estimator of the generative distribution's variance under the assumption that each entry of itr is a sample drawn from the same unknown distribution, with the samples uncorrelated. For arrays, this computation is equivalent to calculating sum(abs2.(itr .- mean(itr))) / (length(itr) - 1). If corrected is true, then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false with n the number of elements in itr.\n\nIf itr is an AbstractArray, dims can be provided to compute the variance over dimensions. In that case, mean must be an array with the same shape as mean(itr, dims=dims) (additional trailing singleton dimensions are allowed).\n\nnote: Note\nIf array contains NaN or missing values, the result is also NaN or missing (missing takes precedence if array contains both). Use the skipmissing function to omit missing entries and compute the variance of non-missing values.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.cor","page":"Statistics","title":"Statistics.cor","category":"function","text":"cor(x::AbstractVector)\n\nReturn the number one.\n\n\n\n\n\ncor(X::AbstractMatrix; dims::Int=1)\n\nCompute the Pearson correlation matrix of the matrix X along the dimension dims.\n\n\n\n\n\ncor(x::AbstractVector, y::AbstractVector)\n\nCompute the Pearson correlation between the vectors x and y.\n\n\n\n\n\ncor(X::AbstractVecOrMat, Y::AbstractVecOrMat; dims=1)\n\nCompute the Pearson correlation between the vectors or matrices X and Y along the dimension dims.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.cov","page":"Statistics","title":"Statistics.cov","category":"function","text":"cov(x::AbstractVector; corrected::Bool=true)\n\nCompute the variance of the vector x. If corrected is true (the default) then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false where n = length(x).\n\n\n\n\n\ncov(X::AbstractMatrix; dims::Int=1, corrected::Bool=true)\n\nCompute the covariance matrix of the matrix X along the dimension dims. If corrected is true (the default) then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false where n = size(X, dims).\n\n\n\n\n\ncov(x::AbstractVector, y::AbstractVector; corrected::Bool=true)\n\nCompute the covariance between the vectors x and y. If corrected is true (the default), computes frac1n-1sum_i=1^n (x_i-bar x) (y_i-bar y)^* where * denotes the complex conjugate and n = length(x) = length(y). If corrected is false, computes frac1nsum_i=1^n (x_i-bar x) (y_i-bar y)^*.\n\n\n\n\n\ncov(X::AbstractVecOrMat, Y::AbstractVecOrMat; dims::Int=1, corrected::Bool=true)\n\nCompute the covariance between the vectors or matrices X and Y along the dimension dims. If corrected is true (the default) then the sum is scaled with n-1, whereas the sum is scaled with n if corrected is false where n = size(X, dims) = size(Y, dims).\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.mean!","page":"Statistics","title":"Statistics.mean!","category":"function","text":"mean!(r, v)\n\nCompute the mean of v over the singleton dimensions of r, and write results to r. Note that the target must not alias with the source.\n\nExamples\n\njulia> using Statistics\n\njulia> v = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> mean!([1., 1.], v)\n2-element Vector{Float64}:\n 1.5\n 3.5\n\njulia> mean!([1. 1.], v)\n1×2 Matrix{Float64}:\n 2.0  3.0\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.mean","page":"Statistics","title":"Statistics.mean","category":"function","text":"mean(itr)\n\nCompute the mean of all elements in a collection.\n\nnote: Note\nIf itr contains NaN or missing values, the result is also NaN or missing (missing takes precedence if array contains both). Use the skipmissing function to omit missing entries and compute the mean of non-missing values.\n\nExamples\n\njulia> using Statistics\n\njulia> mean(1:20)\n10.5\n\njulia> mean([1, missing, 3])\nmissing\n\njulia> mean(skipmissing([1, missing, 3]))\n2.0\n\n\n\n\n\nmean(f, itr)\n\nApply the function f to each element of collection itr and take the mean.\n\njulia> using Statistics\n\njulia> mean(√, [1, 2, 3])\n1.3820881233139908\n\njulia> mean([√1, √2, √3])\n1.3820881233139908\n\n\n\n\n\nmean(f, A::AbstractArray; dims)\n\nApply the function f to each element of array A and take the mean over dimensions dims.\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\njulia> using Statistics\n\njulia> mean(√, [1, 2, 3])\n1.3820881233139908\n\njulia> mean([√1, √2, √3])\n1.3820881233139908\n\njulia> mean(√, [1 2 3; 4 5 6], dims=2)\n2×1 Matrix{Float64}:\n 1.3820881233139908\n 2.2285192400943226\n\n\n\n\n\nmean(A::AbstractArray; dims)\n\nCompute the mean of an array over the given dimensions.\n\ncompat: Julia 1.1\nmean for empty arrays requires at least Julia 1.1.\n\nExamples\n\njulia> using Statistics\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> mean(A, dims=1)\n1×2 Matrix{Float64}:\n 2.0  3.0\n\njulia> mean(A, dims=2)\n2×1 Matrix{Float64}:\n 1.5\n 3.5\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.median!","page":"Statistics","title":"Statistics.median!","category":"function","text":"median!(v)\n\nLike median, but may overwrite the input vector.\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.median","page":"Statistics","title":"Statistics.median","category":"function","text":"median(itr)\n\nCompute the median of all elements in a collection. For an even number of elements no exact median element exists, so the result is equivalent to calculating mean of two median elements.\n\nnote: Note\nIf itr contains NaN or missing values, the result is also NaN or missing (missing takes precedence if itr contains both). Use the skipmissing function to omit missing entries and compute the median of non-missing values.\n\nExamples\n\njulia> using Statistics\n\njulia> median([1, 2, 3])\n2.0\n\njulia> median([1, 2, 3, 4])\n2.5\n\njulia> median([1, 2, missing, 4])\nmissing\n\njulia> median(skipmissing([1, 2, missing, 4]))\n2.0\n\n\n\n\n\nmedian(A::AbstractArray; dims)\n\nCompute the median of an array along the given dimensions.\n\nExamples\n\njulia> using Statistics\n\njulia> median([1 2; 3 4], dims=1)\n1×2 Matrix{Float64}:\n 2.0  3.0\n\n\n\n\n\nmedian(f, v)\n\nApply the function f to each element of collection v and then compute the median.\n\njulia> using Statistics\n\njulia> median(√, [1, 3, 2])\n1.4142135623730951\n\njulia> median([√1, √3, √2])\n1.4142135623730951\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.middle","page":"Statistics","title":"Statistics.middle","category":"function","text":"middle(x)\n\nCompute the middle of a scalar value, which is equivalent to x itself, but of the type of middle(x, x) for consistency.\n\n\n\n\n\nmiddle(x, y)\n\nCompute the middle of two numbers x and y, which is equivalent in both value and type to computing their mean ((x + y) / 2).\n\n\n\n\n\nmiddle(a::AbstractArray)\n\nCompute the middle of an array a, which consists of finding its extrema and then computing their mean.\n\njulia> using Statistics\n\njulia> middle(1:10)\n5.5\n\njulia> a = [1,2,3.6,10.9]\n4-element Vector{Float64}:\n  1.0\n  2.0\n  3.6\n 10.9\n\njulia> middle(a)\n5.95\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.quantile!","page":"Statistics","title":"Statistics.quantile!","category":"function","text":"quantile!([q::AbstractArray, ] v::AbstractVector, p; sorted=false, alpha::Real=1.0, beta::Real=alpha)\n\nCompute the quantile(s) of a vector v at a specified probability or vector or tuple of probabilities p on the interval [0,1]. If p is a vector, an optional output array q may also be specified. (If not provided, a new output array is created.) The keyword argument sorted indicates whether v can be assumed to be sorted; if false (the default), then the elements of v will be partially sorted in-place.\n\nSamples quantile are defined by Q(p) = (1-γ)*x[j] + γ*x[j+1], where x[j] is the j-th order statistic of v, j = floor(n*p + m), m = alpha + p*(1 - alpha - beta) and γ = n*p + m - j.\n\nBy default (alpha = beta = 1), quantiles are computed via linear interpolation between the points ((k-1)/(n-1), x[k]), for k = 1:n where n = length(v). This corresponds to Definition 7 of Hyndman and Fan (1996), and is the same as the R and NumPy default.\n\nThe keyword arguments alpha and beta correspond to the same parameters in Hyndman and Fan, setting them to different values allows to calculate quantiles with any of the methods 4-9 defined in this paper:\n\nDef. 4: alpha=0, beta=1\nDef. 5: alpha=0.5, beta=0.5 (MATLAB default)\nDef. 6: alpha=0, beta=0 (Excel PERCENTILE.EXC, Python default, Stata altdef)\nDef. 7: alpha=1, beta=1 (Julia, R and NumPy default, Excel PERCENTILE and PERCENTILE.INC, Python 'inclusive')\nDef. 8: alpha=1/3, beta=1/3\nDef. 9: alpha=3/8, beta=3/8\n\nnote: Note\nAn ArgumentError is thrown if v contains NaN or missing values.\n\nReferences\n\nHyndman, R.J and Fan, Y. (1996) \"Sample Quantiles in Statistical Packages\", The American Statistician, Vol. 50, No. 4, pp. 361-365\nQuantile on Wikipedia details the different quantile definitions\n\nExamples\n\njulia> using Statistics\n\njulia> x = [3, 2, 1];\n\njulia> quantile!(x, 0.5)\n2.0\n\njulia> x\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> y = zeros(3);\n\njulia> quantile!(y, x, [0.1, 0.5, 0.9]) === y\ntrue\n\njulia> y\n3-element Vector{Float64}:\n 1.2\n 2.0\n 2.8\n\n\n\n\n\n"},{"location":"stdlib/Statistics.html#Statistics.quantile","page":"Statistics","title":"Statistics.quantile","category":"function","text":"quantile(itr, p; sorted=false, alpha::Real=1.0, beta::Real=alpha)\n\nCompute the quantile(s) of a collection itr at a specified probability or vector or tuple of probabilities p on the interval [0,1]. The keyword argument sorted indicates whether itr can be assumed to be sorted.\n\nSamples quantile are defined by Q(p) = (1-γ)*x[j] + γ*x[j+1], where x[j] is the j-th order statistic of itr, j = floor(n*p + m), m = alpha + p*(1 - alpha - beta) and γ = n*p + m - j.\n\nBy default (alpha = beta = 1), quantiles are computed via linear interpolation between the points ((k-1)/(n-1), x[k]), for k = 1:n where n = length(itr). This corresponds to Definition 7 of Hyndman and Fan (1996), and is the same as the R and NumPy default.\n\nThe keyword arguments alpha and beta correspond to the same parameters in Hyndman and Fan, setting them to different values allows to calculate quantiles with any of the methods 4-9 defined in this paper:\n\nDef. 4: alpha=0, beta=1\nDef. 5: alpha=0.5, beta=0.5 (MATLAB default)\nDef. 6: alpha=0, beta=0 (Excel PERCENTILE.EXC, Python default, Stata altdef)\nDef. 7: alpha=1, beta=1 (Julia, R and NumPy default, Excel PERCENTILE and PERCENTILE.INC, Python 'inclusive')\nDef. 8: alpha=1/3, beta=1/3\nDef. 9: alpha=3/8, beta=3/8\n\nnote: Note\nAn ArgumentError is thrown if v contains NaN or missing values. Use the skipmissing function to omit missing entries and compute the quantiles of non-missing values.\n\nReferences\n\nHyndman, R.J and Fan, Y. (1996) \"Sample Quantiles in Statistical Packages\", The American Statistician, Vol. 50, No. 4, pp. 361-365\nQuantile on Wikipedia details the different quantile definitions\n\nExamples\n\njulia> using Statistics\n\njulia> quantile(0:20, 0.5)\n10.0\n\njulia> quantile(0:20, [0.1, 0.5, 0.9])\n3-element Vector{Float64}:\n  2.0\n 10.0\n 18.0\n\njulia> quantile(skipmissing([1, 10, missing]), 0.5)\n5.5\n\n\n\n\n\nquantile(f, v)\n\nApply the function f to each element of collection v and then compute the quantile(s) at a specified probability or vector or tuple of probabilities p on the interval [0,1].\n\njulia> using Statistics\n\njulia> quantile(√, [1, 3, 2], 0.3)\n1.248528137423857\n\njulia> quantile([√1, √3, √2], 0.3)\n1.248528137423857\n\njulia> quantile(√, [1, 3, 2], (0.3, 0.4, 0.5))\n(1.248528137423857, 1.3313708498984762, 1.4142135623730951)\n\njulia> quantile(.√[1, 3, 2], (0.3, 0.4, 0.5))\n(1.248528137423857, 1.3313708498984762, 1.4142135623730951)\n\n\n\n\n\n"},{"location":"devdocs/aot.html#Ahead-of-Time-Compilation","page":"Ahead of Time Compilation","title":"Ahead of Time Compilation","category":"section","text":"This document describes the design and structure of the ahead-of-time (AOT) compilation system in Julia. This system is used when generating system images and package images. Much of the implementation described here is located in aotcompile.cpp, staticdata.c, and processor.cpp"},{"location":"devdocs/aot.html#Introduction","page":"Ahead of Time Compilation","title":"Introduction","category":"section","text":"Though Julia normally compiles code just-in-time (JIT), it is possible to compile code ahead of time and save the resulting code to a file. This can be useful for a number of reasons:\n\nTo reduce the time it takes to start a Julia process.\nTo reduce the time spent in the JIT compiler instead of executing code (time to first execution, TTFX).\nTo reduce the amount of memory used by the JIT compiler."},{"location":"devdocs/aot.html#High-Level-Overview","page":"Ahead of Time Compilation","title":"High-Level Overview","category":"section","text":"The following descriptions are a snapshot of the current implementation details of the end-to-end pipeline that happens internally when the user compiles a new AOT module, such as occurs when they type using Foo. These details are likely to change over time as we implement better ways to handle them, so current implementations may not exactly match the dataflow and functions described below."},{"location":"devdocs/aot.html#Compiling-Code-Images","page":"Ahead of Time Compilation","title":"Compiling Code Images","category":"section","text":"Firstly, the methods that need to be compiled to native code must be identified. This can only be done by actually executing the code to be compiled, as the set of methods that need to be compiled depends on the types of the arguments passed to the methods, and method invocations with certain combinations of types may not be known until runtime. During this process, the exact methods that the compiler sees are tracked for later compilation, producing a compilation trace.\n\nnote: Note\nCurrently when compiling images, Julia runs the trace generation in a different process than the process performing the AOT compilation. This can have impacts when attempting to use a debugger during precompilation. The best way to debug precompilation with a debugger is to use the rr debugger, record the entire process tree, use rr ps to identify the relevant failing process, and then use rr replay -p PID to replay just the failing process.\n\nOnce the methods to be compiled have been identified, they are passed to the jl_create_system_image function. This function sets up a number of data structures that will be used when serializing native code to a file, and then calls jl_create_native with the array of methods. jl_create_native runs codegen on the methods produces one or more LLVM modules. jl_create_system_image then records some useful information about what codegen produced from the module(s).\n\nThe module(s) are then passed to jl_dump_native, along with the information recorded by jl_create_system_image. jl_dump_native contains the code necessary to serialize the module(s) to bitcode, object, or assembly files depending on the command-line options passed to Julia. The serialized code and information are then written to a file as an archive.\n\nThe final step is to run a system linker on the object files in the archive produced by jl_dump_native. Once this step is complete, a shared library containing the compiled code is produced."},{"location":"devdocs/aot.html#Loading-Code-Images","page":"Ahead of Time Compilation","title":"Loading Code Images","category":"section","text":"When loading a code image, the shared library produced by the linker is loaded into memory. The system image data is then loaded from the shared library. This data contains information about the types, methods, and code instances that were compiled into the shared library. This data is used to restore the state of the runtime to what it was when the code image was compiled.\n\nIf the code image was compiled with multiversioning, the loader will pick the appropriate version of each function to use based on the CPU features available on the current machine.\n\nFor system images, since no other code has been loaded, the state of the runtime is now the same as it was when the code image was compiled. For package images, the environment may have changed compared to when the code was compiled, so each method must be checked against the global method table to determine if it is still valid code."},{"location":"devdocs/aot.html#Compiling-Methods","page":"Ahead of Time Compilation","title":"Compiling Methods","category":"section","text":""},{"location":"devdocs/aot.html#Tracing-Compiled-Methods","page":"Ahead of Time Compilation","title":"Tracing Compiled Methods","category":"section","text":"Julia has a command-line flag to record all of the methods that are compiled by the JIT compiler, --trace-compile=filename. When a function is compiled and this flag has a filename, Julia will print out a precompile statement to that file with the method and argument types it was called with. This therefore generates a precompile script that can be used later in the AOT compilation process. The PrecompileTools package has tooling that can make taking advantage of this functionality easier for package developers."},{"location":"devdocs/aot.html#jl_create_system_image","page":"Ahead of Time Compilation","title":"jl_create_system_image","category":"section","text":"jl_create_system_image saves all of the Julia-specific metadata necessary to later restore the state of the runtime. This includes data such as code instances, method instances, method tables, and type information. This function also sets up the data structures necessary to serialize the native code to a file. Finally, it calls jl_create_native to create one or more LLVM modules containing the native code for the methods passed to it. jl_create_native is responsible for running codegen on the methods passed to it."},{"location":"devdocs/aot.html#jl_dump_native","page":"Ahead of Time Compilation","title":"jl_dump_native","category":"section","text":"jl_dump_native is responsible for serializing the LLVM module containing the native code to a file. In addition to the module, the system image data produced by jl_create_system_image is compiled as a global variable. The output of this method is bitcode, object, and/or assembly archives containing the code and system image data.\n\njl_dump_native is typically one of the larger time sinks when emitting native code, with much of the time spent in optimizing LLVM IR and emitting machine code. Therefore, this function is capable of multithreading the optimization and machine code emission steps. This multithreading is parameterized on the size of the module, but can be explicitly overridden by setting the JULIA_IMAGE_THREADS environment variable. The default maximum number of threads is half the number of available threads, but setting it to be lower can reduce peak memory usage during compilation.\n\njl_dump_native can also produce native code optimized for multiple architectures, when integrated with the Julia loader. This is triggered by setting the JULIA_CPU_TARGET environment variable and mediated by the multiversioning pass in the optimization pipeline. To make this work with multithreading, an annotation step is added before the module is split into submodules that are emitted on their own threads, and this annotation step uses information available throughout the entire module to decide what functions are cloned for different architectures. Once the annotation has happened, individual threads can emit code for different architectures in parallel, knowing that a different submodule is guaranteed to produce the necessary functions that will be called by a cloned function.\n\nSome other metadata about how the module was serialized is also stored in the archive, such as the number of threads used to serialize the module and the number of functions that were compiled."},{"location":"devdocs/aot.html#Static-Linking","page":"Ahead of Time Compilation","title":"Static Linking","category":"section","text":"The final step in the AOT compilation process is to run a linker on the object files in the archive produced by jl_dump_native. This produces a shared library containing the compiled code. This shared library can then be loaded by Julia to restore the state of the runtime. When compiling a system image, the native linker used by a C compiler is used to produce the final shared library. For package images, the LLVM linker LLD is used to provide a more consistent linking interface."},{"location":"devdocs/aot.html#Loading-Code-Images-2","page":"Ahead of Time Compilation","title":"Loading Code Images","category":"section","text":""},{"location":"devdocs/aot.html#Loading-the-Shared-Library","page":"Ahead of Time Compilation","title":"Loading the Shared Library","category":"section","text":"The first step in loading a code image is to load the shared library produced by the linker. This is done by calling jl_dlopen on the path to the shared library. This function is responsible for loading the shared library and resolving all of the symbols in the library."},{"location":"devdocs/aot.html#Loading-Native-Code","page":"Ahead of Time Compilation","title":"Loading Native Code","category":"section","text":"The loader first needs to identify whether the native code that was compiled is valid for the architecture that the loader is running on. This is necessary to avoid executing instructions that older CPUs do not recognize. This is done by checking the CPU features available on the current machine against the CPU features that the code was compiled for. When multiversioning is enabled, the loader will pick the appropriate version of each function to use based on the CPU features available on the current machine. If none of the feature sets that were multiversioned, the loader will throw an error.\n\nPart of the multiversioning pass creates a number of global arrays of all of the functions in the module. When this process is multithreaded, an array of arrays is created, which the loader reorganizes into one large array with all of the functions that were compiled for this architecture. A similar process occurs for the global variables in the module."},{"location":"devdocs/aot.html#Setting-Up-Julia-State","page":"Ahead of Time Compilation","title":"Setting Up Julia State","category":"section","text":"The loader then uses the global variables and functions produced from loading native code to set up Julia runtime core data structures in the current process. This setup involves adding types and methods to the Julia runtime, and making the cached native code available for use by other Julia functions and the interpreter. For package images, each method must be validated, in that the global method table's state must match the state that the package image was compiled for. In particular, if a different set of methods exists at the load time compared to compile time of the package image, the method must be invalidated and recompiled on first use. This is necessary to ensure that execution semantics remain the same regardless of if a package was precompiled or if the code was directly executed. System images do not need to perform this validation, since the global method table is empty at load time. Thus, system images have faster load times than package images."},{"location":"manual/control-flow.html#Control-Flow","page":"Control Flow","title":"Control Flow","category":"section","text":"Julia provides a variety of control flow constructs:\n\nCompound Expressions: begin and ;.\nConditional Evaluation: if-elseif-else and ?: (ternary operator).\nShort-Circuit Evaluation: logical operators && (“and”) and || (“or”), and also chained comparisons.\nRepeated Evaluation: Loops: while and for.\nException Handling: try-catch, error and throw.\nTasks (aka Coroutines): yieldto.\n\nThe first five control flow mechanisms are standard to high-level programming languages. Tasks are not so standard: they provide non-local control flow, making it possible to switch between temporarily-suspended computations. This is a powerful construct: both exception handling and cooperative multitasking are implemented in Julia using tasks. Everyday programming requires no direct usage of tasks, but certain problems can be solved much more easily by using tasks."},{"location":"manual/control-flow.html#man-compound-expressions","page":"Control Flow","title":"Compound Expressions","category":"section","text":"Sometimes it is convenient to have a single expression which evaluates several subexpressions in order, returning the value of the last subexpression as its value. There are two Julia constructs that accomplish this: begin blocks and ; chains. The value of both compound expression constructs is that of the last subexpression. Here's an example of a begin block:\n\njulia> z = begin\n           x = 1\n           y = 2\n           x + y\n       end\n3\n\nSince these are fairly small, simple expressions, they could easily be placed onto a single line, which is where the ; chain syntax comes in handy:\n\njulia> z = (x = 1; y = 2; x + y)\n3\n\nThis syntax is particularly useful with the terse single-line function definition form introduced in Functions. Although it is typical, there is no requirement that begin blocks be multiline or that ; chains be single-line:\n\njulia> begin x = 1; y = 2; x + y end\n3\n\njulia> (x = 1;\n        y = 2;\n        x + y)\n3"},{"location":"manual/control-flow.html#man-conditional-evaluation","page":"Control Flow","title":"Conditional Evaluation","category":"section","text":"Conditional evaluation allows portions of code to be evaluated or not evaluated depending on the value of a boolean expression. Here is the anatomy of the if-elseif-else conditional syntax:\n\nif x < y\n    println(\"x is less than y\")\nelseif x > y\n    println(\"x is greater than y\")\nelse\n    println(\"x is equal to y\")\nend\n\nIf the condition expression x < y is true, then the corresponding block is evaluated; otherwise the condition expression x > y is evaluated, and if it is true, the corresponding block is evaluated; if neither expression is true, the else block is evaluated. Here it is in action:\n\njulia> function test(x, y)\n           if x < y\n               println(\"x is less than y\")\n           elseif x > y\n               println(\"x is greater than y\")\n           else\n               println(\"x is equal to y\")\n           end\n       end\ntest (generic function with 1 method)\n\njulia> test(1, 2)\nx is less than y\n\njulia> test(2, 1)\nx is greater than y\n\njulia> test(1, 1)\nx is equal to y\n\nThe elseif and else blocks are optional, and as many elseif blocks as desired can be used. The condition expressions in the if-elseif-else construct are evaluated until the first one evaluates to true, after which the associated block is evaluated, and no further condition expressions or blocks are evaluated.\n\nif blocks are \"leaky\", i.e. they do not introduce a local scope. This means that new variables defined inside the if clauses can be used after the if block, even if they weren't defined before. So, we could have defined the test function above as\n\njulia> function test(x,y)\n           if x < y\n               relation = \"less than\"\n           elseif x == y\n               relation = \"equal to\"\n           else\n               relation = \"greater than\"\n           end\n           println(\"x is \", relation, \" y.\")\n       end\ntest (generic function with 1 method)\n\njulia> test(2, 1)\nx is greater than y.\n\nThe variable relation is declared inside the if block, but used outside. However, when depending on this behavior, make sure all possible code paths define a value for the variable. The following change to the above function results in a runtime error\n\njulia> function test(x,y)\n           if x < y\n               relation = \"less than\"\n           elseif x == y\n               relation = \"equal to\"\n           end\n           println(\"x is \", relation, \" y.\")\n       end\ntest (generic function with 1 method)\n\njulia> test(1,2)\nx is less than y.\n\njulia> test(2,1)\nERROR: UndefVarError: `relation` not defined in local scope\nStacktrace:\n [1] test(::Int64, ::Int64) at ./none:7\n\nif blocks also return a value, which may seem unintuitive to users coming from many other languages. This value is simply the return value of the last executed statement in the branch that was chosen, so\n\njulia> x = 3\n3\n\njulia> if x > 0\n           \"positive!\"\n       else\n           \"negative...\"\n       end\n\"positive!\"\n\nNote that very short conditional statements (one-liners) are frequently expressed using Short-Circuit Evaluation in Julia, as outlined in the next section.\n\nUnlike C, MATLAB, Perl, Python, and Ruby – but like Java, and a few other stricter, typed languages – it is an error if the value of a conditional expression is anything but true or false:\n\njulia> if 1\n           println(\"true\")\n       end\nERROR: TypeError: non-boolean (Int64) used in boolean context\n\nThis error indicates that the conditional was of the wrong type: Int64 rather than the required Bool.\n\nThe so-called \"ternary operator\", ?:, is closely related to the if-elseif-else syntax, but is used where a conditional choice between single expression values is required, as opposed to conditional execution of longer blocks of code. It gets its name from being the only operator in most languages taking three operands:\n\na ? b : c\n\nThe expression a, before the ?, is a condition expression, and the ternary operation evaluates the expression b, before the :, if the condition a is true or the expression c, after the :, if it is false. Note that the spaces around ? and : are mandatory: an expression like a?b:c is not a valid ternary expression (but a newline is acceptable after both the ? and the :).\n\nThe easiest way to understand this behavior is to see an example. In the previous example, the println call is shared by all three branches: the only real choice is which literal string to print. This could be written more concisely using the ternary operator. For the sake of clarity, let's try a two-way version first:\n\njulia> x = 1; y = 2;\n\njulia> println(x < y ? \"less than\" : \"not less than\")\nless than\n\njulia> x = 1; y = 0;\n\njulia> println(x < y ? \"less than\" : \"not less than\")\nnot less than\n\nIf the expression x < y is true, the entire ternary operator expression evaluates to the string \"less than\" and otherwise it evaluates to the string \"not less than\". The original three-way example requires chaining multiple uses of the ternary operator together:\n\njulia> test(x, y) = println(x < y ? \"x is less than y\"    :\n                            x > y ? \"x is greater than y\" : \"x is equal to y\")\ntest (generic function with 1 method)\n\njulia> test(1, 2)\nx is less than y\n\njulia> test(2, 1)\nx is greater than y\n\njulia> test(1, 1)\nx is equal to y\n\nTo facilitate chaining, the operator associates from right to left.\n\nIt is significant that like if-elseif-else, the expressions before and after the : are only evaluated if the condition expression evaluates to true or false, respectively:\n\njulia> v(x) = (println(x); x)\nv (generic function with 1 method)\n\njulia> 1 < 2 ? v(\"yes\") : v(\"no\")\nyes\n\"yes\"\n\njulia> 1 > 2 ? v(\"yes\") : v(\"no\")\nno\n\"no\""},{"location":"manual/control-flow.html#Short-Circuit-Evaluation","page":"Control Flow","title":"Short-Circuit Evaluation","category":"section","text":"The && and || operators in Julia correspond to logical “and” and “or” operations, respectively, and are typically used for this purpose. However, they have an additional property of short-circuit evaluation: they don't necessarily evaluate their second argument, as explained below.  (There are also bitwise & and | operators that can be used as logical “and” and “or” without short-circuit behavior, but beware that & and | have higher precedence than && and || for evaluation order.)\n\nShort-circuit evaluation is quite similar to conditional evaluation. The behavior is found in most imperative programming languages having the && and || boolean operators: in a series of boolean expressions connected by these operators, only the minimum number of expressions are evaluated as are necessary to determine the final boolean value of the entire chain. Some languages (like Python) refer to them as and (&&) and or (||). Explicitly, this means that:\n\nIn the expression a && b, the subexpression b is only evaluated if a evaluates to true.\nIn the expression a || b, the subexpression b is only evaluated if a evaluates to false.\n\nThe reasoning is that a && b must be false if a is false, regardless of the value of b, and likewise, the value of a || b must be true if a is true, regardless of the value of b. Both && and || associate to the right, but && has higher precedence than || does. It's easy to experiment with this behavior:\n\njulia> t(x) = (println(x); true)\nt (generic function with 1 method)\n\njulia> f(x) = (println(x); false)\nf (generic function with 1 method)\n\njulia> t(1) && t(2)\n1\n2\ntrue\n\njulia> t(1) && f(2)\n1\n2\nfalse\n\njulia> f(1) && t(2)\n1\nfalse\n\njulia> f(1) && f(2)\n1\nfalse\n\njulia> t(1) || t(2)\n1\ntrue\n\njulia> t(1) || f(2)\n1\ntrue\n\njulia> f(1) || t(2)\n1\n2\ntrue\n\njulia> f(1) || f(2)\n1\n2\nfalse\n\nYou can easily experiment in the same way with the associativity and precedence of various combinations of && and || operators.\n\nThis behavior is frequently used in Julia to form an alternative to very short if statements. Instead of if <cond> <statement> end, one can write <cond> && <statement> (which could be read as: <cond> and then <statement>). Similarly, instead of if ! <cond> <statement> end, one can write <cond> || <statement> (which could be read as: <cond> or else <statement>).\n\nFor example, a recursive factorial routine could be defined like this:\n\njulia> function fact(n::Int)\n           n >= 0 || error(\"n must be non-negative\")\n           n == 0 && return 1\n           n * fact(n-1)\n       end\nfact (generic function with 1 method)\n\njulia> fact(5)\n120\n\njulia> fact(0)\n1\n\njulia> fact(-1)\nERROR: n must be non-negative\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] fact(::Int64) at ./none:2\n [3] top-level scope\n\nBoolean operations without short-circuit evaluation can be done with the bitwise boolean operators introduced in Mathematical Operations and Elementary Functions: & and |. These are normal functions, which happen to support infix operator syntax, but always evaluate their arguments:\n\njulia> f(1) & t(2)\n1\n2\nfalse\n\njulia> t(1) | t(2)\n1\n2\ntrue\n\nJust like condition expressions used in if, elseif or the ternary operator, the operands of && or || must be boolean values (true or false). Using a non-boolean value anywhere except for the last entry in a conditional chain is an error:\n\njulia> 1 && true\nERROR: TypeError: non-boolean (Int64) used in boolean context\n\nOn the other hand, any type of expression can be used at the end of a conditional chain. It will be evaluated and returned depending on the preceding conditionals:\n\njulia> true && (x = (1, 2, 3))\n(1, 2, 3)\n\njulia> false && (x = (1, 2, 3))\nfalse"},{"location":"manual/control-flow.html#man-loops","page":"Control Flow","title":"Repeated Evaluation: Loops","category":"section","text":"There are two constructs for repeated evaluation of expressions: the while loop and the for loop. Here is an example of a while loop:\n\njulia> i = 1;\n\njulia> while i <= 3\n           println(i)\n           global i += 1\n       end\n1\n2\n3\n\nThe while loop evaluates the condition expression (i <= 3 in this case), and as long it remains true, keeps also evaluating the body of the while loop. If the condition expression is false when the while loop is first reached, the body is never evaluated.\n\nThe for loop makes common repeated evaluation idioms easier to write. Since counting up and down like the above while loop does is so common, it can be expressed more concisely with a for loop:\n\njulia> for i = 1:3\n           println(i)\n       end\n1\n2\n3\n\nHere the 1:3 is a range object, representing the sequence of numbers 1, 2, 3. The for loop iterates through these values, assigning each one in turn to the variable i. In general, the for construct can loop over any \"iterable\" object (or \"container\"), from a  range like 1:3 or 1:3:13 (a StepRange indicating every 3rd integer 1, 4, 7, …, 13) to more generic containers like arrays, including iterators defined by user code or external packages. For containers other than ranges, the alternative (but fully equivalent) keyword in or ∈ is typically used instead of =, since it makes the code read more clearly:\n\njulia> for i in [1,4,0]\n           println(i)\n       end\n1\n4\n0\n\njulia> for s ∈ [\"foo\",\"bar\",\"baz\"]\n           println(s)\n       end\nfoo\nbar\nbaz\n\nVarious types of iterable containers will be introduced and discussed in later sections of the manual (see, e.g., Multi-dimensional Arrays).\n\nOne rather important distinction between the previous while loop form and the for loop form is the scope during which the variable is visible. A for loop always introduces a new iteration variable in its body, regardless of whether a variable of the same name exists in the enclosing scope. This implies that on the one hand i need not be declared before the loop. On the other hand it will not be visible outside the loop, nor will an outside variable of the same name be affected. You'll either need a new interactive session instance or a different variable name to test this:\n\njulia> for j = 1:3\n           println(j)\n       end\n1\n2\n3\n\njulia> j\nERROR: UndefVarError: `j` not defined in `Main`\n\njulia> j = 0;\n\njulia> for j = 1:3\n           println(j)\n       end\n1\n2\n3\n\njulia> j\n0\n\nUse for outer to modify the latter behavior and reuse an existing local variable.\n\nSee Scope of Variables for a detailed explanation of variable scope, outer, and how it works in Julia.\n\nIt is sometimes convenient to terminate the repetition of a while before the test condition is falsified or stop iterating in a for loop before the end of the iterable object is reached. This can be accomplished with the break keyword:\n\njulia> i = 1;\n\njulia> while true\n           println(i)\n           if i >= 3\n               break\n           end\n           global i += 1\n       end\n1\n2\n3\n\njulia> for j = 1:1000\n           println(j)\n           if j >= 3\n               break\n           end\n       end\n1\n2\n3\n\nWithout the break keyword, the above while loop would never terminate on its own, and the for loop would iterate up to 1000. These loops are both exited early by using break.\n\nIn other circumstances, it is handy to be able to stop an iteration and move on to the next one immediately. The continue keyword accomplishes this:\n\njulia> for i = 1:10\n           if i % 3 != 0\n               continue\n           end\n           println(i)\n       end\n3\n6\n9\n\nThis is a somewhat contrived example since we could produce the same behavior more clearly by negating the condition and placing the println call inside the if block. In realistic usage there is more code to be evaluated after the continue, and often there are multiple points from which one calls continue.\n\nMultiple nested for loops can be combined into a single outer loop, forming the cartesian product of its iterables:\n\njulia> for i = 1:2, j = 3:4\n           println((i, j))\n       end\n(1, 3)\n(1, 4)\n(2, 3)\n(2, 4)\n\nWith this syntax, iterables may still refer to outer loop variables; e.g. for i = 1:n, j = 1:i is valid. However a break statement inside such a loop exits the entire nest of loops, not just the inner one. Both variables (i and j) are set to their current iteration values each time the inner loop runs. Therefore, assignments to i will not be visible to subsequent iterations:\n\njulia> for i = 1:2, j = 3:4\n           println((i, j))\n           i = 0\n       end\n(1, 3)\n(1, 4)\n(2, 3)\n(2, 4)\n\nIf this example were rewritten to use a for keyword for each variable, then the output would be different: the second and fourth values would contain 0.\n\nMultiple containers can be iterated over at the same time in a single for loop using zip:\n\njulia> for (j, k) in zip([1 2 3], [4 5 6 7])\n           println((j,k))\n       end\n(1, 4)\n(2, 5)\n(3, 6)\n\nUsing zip will create an iterator that is a tuple containing the subiterators for the containers passed to it. The zip iterator will iterate over all subiterators in order, choosing the ith element of each subiterator in the ith iteration of the for loop. Once any of the subiterators run out, the for loop will stop."},{"location":"manual/control-flow.html#Exception-Handling","page":"Control Flow","title":"Exception Handling","category":"section","text":"When an unexpected condition occurs, a function may be unable to return a reasonable value to its caller. In such cases, it may be best for the exceptional condition to either terminate the program while printing a diagnostic error message, or if the programmer has provided code to handle such exceptional circumstances then allow that code to take the appropriate action."},{"location":"manual/control-flow.html#Built-in-Exceptions","page":"Control Flow","title":"Built-in Exceptions","category":"section","text":"Exceptions are thrown when an unexpected condition has occurred. The built-in Exceptions listed below all interrupt the normal flow of control.\n\nException\nArgumentError\nBoundsError\nCompositeException\nDimensionMismatch\nDivideError\nDomainError\nEOFError\nErrorException\nFieldError\nInexactError\nInitError\nInterruptException\nInvalidStateException\nKeyError\nLoadError\nOutOfMemoryError\nReadOnlyMemoryError\nRemoteException\nMethodError\nOverflowError\nMeta.ParseError\nSystemError\nTypeError\nUndefRefError\nUndefVarError\nStringIndexError\n\nFor example, the sqrt function throws a DomainError if applied to a negative real value:\n\njulia> sqrt(-1)\nERROR: DomainError with -1.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\nYou may define your own exceptions in the following way:\n\njulia> struct MyCustomException <: Exception end"},{"location":"manual/control-flow.html#The-[throw](@ref)-function","page":"Control Flow","title":"The throw function","category":"section","text":"Exceptions can be created explicitly with throw. For example, a function defined only for non-negative numbers could be written to throw a DomainError if the argument is negative:\n\njulia> f(x) = x>=0 ? exp(-x) : throw(DomainError(x, \"argument must be non-negative\"))\nf (generic function with 1 method)\n\njulia> f(1)\n0.36787944117144233\n\njulia> f(-1)\nERROR: DomainError with -1:\nargument must be non-negative\nStacktrace:\n [1] f(::Int64) at ./none:1\n\nNote that DomainError without parentheses is not an exception, but a type of exception. It needs to be called to obtain an Exception object:\n\njulia> typeof(DomainError(nothing)) <: Exception\ntrue\n\njulia> typeof(DomainError) <: Exception\nfalse\n\nAdditionally, some exception types take one or more arguments that are used for error reporting:\n\njulia> throw(UndefVarError(:x))\nERROR: UndefVarError: `x` not defined\n\nThis mechanism can be implemented easily by custom exception types following the way UndefVarError is written:\n\njulia> struct MyUndefVarError <: Exception\n           var::Symbol\n       end\n\njulia> Base.showerror(io::IO, e::MyUndefVarError) = print(io, e.var, \" not defined\")\n\nnote: Note\nWhen writing an error message, it is preferred to make the first word lowercase. For example,size(A) == size(B) || throw(DimensionMismatch(\"size of A not equal to size of B\"))is preferred oversize(A) == size(B) || throw(DimensionMismatch(\"Size of A not equal to size of B\")).However, sometimes it makes sense to keep the uppercase first letter, for instance if an argument to a function is a capital letter:size(A,1) == size(B,2) || throw(DimensionMismatch(\"A has first dimension...\"))."},{"location":"manual/control-flow.html#Errors","page":"Control Flow","title":"Errors","category":"section","text":"The error function is used to produce an ErrorException that interrupts the normal flow of control.\n\nSuppose we want to stop execution immediately if the square root of a negative number is taken. To do this, we can define a fussy version of the sqrt function that raises an error if its argument is negative:\n\njulia> fussy_sqrt(x) = x >= 0 ? sqrt(x) : error(\"negative x not allowed\")\nfussy_sqrt (generic function with 1 method)\n\njulia> fussy_sqrt(2)\n1.4142135623730951\n\njulia> fussy_sqrt(-1)\nERROR: negative x not allowed\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] fussy_sqrt(::Int64) at ./none:1\n [3] top-level scope\n\nIf fussy_sqrt is called with a negative value from another function, instead of trying to continue execution of the calling function, it returns immediately, displaying the error message in the interactive session:\n\njulia> function verbose_fussy_sqrt(x)\n           println(\"before fussy_sqrt\")\n           r = fussy_sqrt(x)\n           println(\"after fussy_sqrt\")\n           return r\n       end\nverbose_fussy_sqrt (generic function with 1 method)\n\njulia> verbose_fussy_sqrt(2)\nbefore fussy_sqrt\nafter fussy_sqrt\n1.4142135623730951\n\njulia> verbose_fussy_sqrt(-1)\nbefore fussy_sqrt\nERROR: negative x not allowed\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] fussy_sqrt at ./none:1 [inlined]\n [3] verbose_fussy_sqrt(::Int64) at ./none:3\n [4] top-level scope"},{"location":"manual/control-flow.html#The-try/catch-statement","page":"Control Flow","title":"The try/catch statement","category":"section","text":"The try/catch statement allows for Exceptions to be tested for, and for the graceful handling of things that may ordinarily break your application. For example, in the below code the function for square root would normally throw an exception. By placing a try/catch block around it we can mitigate that here. You may choose how you wish to handle this exception, whether logging it, return a placeholder value or as in the case below where we just printed out a statement. One thing to think about when deciding how to handle unexpected situations is that using a try/catch block is much slower than using conditional branching to handle those situations. Below there are more examples of handling exceptions with a try/catch block:\n\njulia> try\n           sqrt(\"ten\")\n       catch e\n           println(\"You should have entered a numeric value\")\n       end\nYou should have entered a numeric value\n\ntry/catch statements also allow the Exception to be saved in a variable. The following contrived example calculates the square root of the second element of x if x is indexable, otherwise assumes x is a real number and returns its square root:\n\njulia> sqrt_second(x) = try\n           sqrt(x[2])\n       catch y\n           if isa(y, DomainError)\n               sqrt(complex(x[2], 0))\n           elseif isa(y, BoundsError)\n               sqrt(x)\n           else\n               rethrow() # ensure other exceptions can bubble up the call stack\n           end\n       end\nsqrt_second (generic function with 1 method)\n\njulia> sqrt_second([1 4])\n2.0\n\njulia> sqrt_second([1 -4])\n0.0 + 2.0im\n\njulia> sqrt_second(9)\n3.0\n\njulia> sqrt_second(-9)\nERROR: DomainError with -9.0:\nsqrt was called with a negative real argument but will only return a complex result if called with a complex argument. Try sqrt(Complex(x)).\nStacktrace:\n[...]\n\njulia> sqrt_second([1 nothing])\nERROR: MethodError: no method matching sqrt(::Nothing)\nThe function `sqrt` exists, but no method is defined for this combination of argument types.\n[...]\n\nUse rethrow as above to continue unwinding the stack with the original exception so that higher-level exception handlers can deal with the exception. When filtering by exception type as above, it is often important to include else rethrow() so that other types of exceptions are not hidden from the caller.\n\nNote that the symbol following catch will always be interpreted as a name for the exception, so care is needed when writing try/catch expressions on a single line. The following code will not work to return the value of x in case of an error:\n\ntry bad() catch x end\n\nInstead, use a semicolon or insert a line break after catch:\n\ntry bad() catch; x end\n\ntry bad()\ncatch\n    x\nend\n\nThe power of the try/catch construct lies in the ability to unwind a deeply nested computation immediately to a much higher level in the stack of calling functions. There are situations where no error has occurred, but the ability to unwind the stack and pass a value to a higher level is desirable. Julia provides the backtrace, catch_backtrace and current_exceptions functions for more advanced error handling."},{"location":"manual/control-flow.html#else-Clauses","page":"Control Flow","title":"else Clauses","category":"section","text":"compat: Julia 1.8\nThis functionality requires at least Julia 1.8.\n\nIn some cases, one may not only want to appropriately handle the error case, but also want to run some code only if the try block succeeds. For this, an else clause can be specified after the catch block that is run whenever no error was thrown previously. The advantage over including this code in the try block instead is that any further errors don't get silently caught by the catch clause.\n\nlocal x\ntry\n    x = read(\"file\", String)\ncatch\n    # handle read errors\nelse\n    # do something with x\nend\n\nnote: Note\nThe try, catch, else, and finally clauses each introduce their own scope blocks, so if a variable is only defined in the try block, it can not be accessed by the else or finally clause:julia> try\n           foo = 1\n       catch\n       else\n           foo\n       end\nERROR: UndefVarError: `foo` not defined in `Main`\nSuggestion: check for spelling errors or missing imports.Use the local keyword outside the try block to make the variable accessible from anywhere within the outer scope."},{"location":"manual/control-flow.html#finally-Clauses","page":"Control Flow","title":"finally Clauses","category":"section","text":"In code that performs state changes or uses resources like files, there is typically clean-up work (such as closing files) that needs to be done when the code is finished. Exceptions potentially complicate this task, since they can cause a block of code to exit before reaching its normal end. The finally keyword provides a way to run some code when a given block of code exits, regardless of how it exits.\n\nFor example, here is how we can guarantee that an opened file is closed:\n\nf = open(\"file\")\ntry\n    # operate on file f\nfinally\n    close(f)\nend\n\nWhen control leaves the try block (for example due to a return, or just finishing normally), close(f) will be executed. If the try block exits due to an exception, the exception will continue propagating. A catch block may be combined with try and finally as well. In this case the finally block will run after catch has handled the error.\n\nWhen evaluating a try/catch/else/finally expression, the value of the entire expression is the value of the last block executed, excluding the finally block. For example:\n\njulia> try\n           1\n       finally\n           2\n       end\n1\n\njulia> try\n           error(\"\")\n       catch\n           1\n       else\n           2\n       finally\n           3\n       end\n1\n\njulia> try\n           0\n       catch\n           1\n       else\n           2\n       finally\n           3\n       end\n2"},{"location":"manual/control-flow.html#man-tasks","page":"Control Flow","title":"Tasks (aka Coroutines)","category":"section","text":"Tasks are a control flow feature that allows computations to be suspended and resumed in a flexible manner. We mention them here only for completeness; for a full discussion see Asynchronous Programming."},{"location":"stdlib/FileWatching.html#lib-filewatching","page":"File Events","title":"File Events","category":"section","text":""},{"location":"stdlib/FileWatching.html#Pidfile","page":"File Events","title":"Pidfile","category":"section","text":"A simple utility tool for creating advisory pidfiles (lock files)."},{"location":"stdlib/FileWatching.html#Primary-Functions","page":"File Events","title":"Primary Functions","category":"section","text":""},{"location":"stdlib/FileWatching.html#Helper-Functions","page":"File Events","title":"Helper Functions","category":"section","text":""},{"location":"stdlib/FileWatching.html#FileWatching.poll_fd","page":"File Events","title":"FileWatching.poll_fd","category":"function","text":"poll_fd(fd, timeout_s::Real=-1; readable=false, writable=false)\n\nMonitor a file descriptor fd for changes in the read or write availability, and with a timeout given by timeout_s seconds.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\nThis is a thin wrapper over calling wait on a FDWatcher, which implements the functionality but requires the user to call close manually when finished with it, or risk serious crashes.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.poll_file","page":"File Events","title":"FileWatching.poll_file","category":"function","text":"poll_file(path::AbstractString, interval_s::Real=5.007, timeout_s::Real=-1) -> (previous::StatStruct, current)\n\nMonitor a file for changes by polling every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds.\n\nReturns a pair of status objects (previous, current) when a change is detected. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (indicating the timeout elapsed), or some other Exception subtype (if the stat operation failed: for example, if the path does not exist).\n\nTo determine when a file was modified, compare !(current isa StatStruct && prev == current) to detect notification of changes to the mtime or inode. However, using watch_file for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\nThis is a thin wrapper over calling wait on a PollingFileWatcher, which implements the functionality, but this function has a small race window between consecutive calls to poll_file where the file might change without being detected.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.watch_file","page":"File Events","title":"FileWatching.watch_file","category":"function","text":"watch_file(path::AbstractString, timeout_s::Real=-1)\n\nWatch file or directory path for changes until a change occurs or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThe returned value is an object with boolean fields renamed, changed, and timedout, giving the result of watching the file.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\nThis is a thin wrapper over calling wait on a FileMonitor. This function has a small race window between consecutive calls to watch_file where the file might change without being detected. To avoid this race, use\n\nfm = FileMonitor(path)\nwait(fm)\n\ndirectly, re-using the same fm each time you wait.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.watch_folder","page":"File Events","title":"FileWatching.watch_folder","category":"function","text":"watch_folder(path::AbstractString, timeout_s::Real=-1)\n\nWatch a file or directory path for changes until a change has occurred or timeout_s seconds have elapsed. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis will continuing tracking changes for path in the background until unwatch_folder is called on the same path.\n\nThe returned value is an pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed, changed, and timedout, giving the event.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\nThis function is a thin wrapper over calling wait on a FolderMonitor, with added timeout support.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.unwatch_folder","page":"File Events","title":"FileWatching.unwatch_folder","category":"function","text":"unwatch_folder(path::AbstractString)\n\nStop background tracking of changes for path. It is not recommended to do this while another task is waiting for watch_folder to return on the same path, as the result may be unpredictable.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.FileMonitor","page":"File Events","title":"FileWatching.FileMonitor","category":"type","text":"FileMonitor(path::AbstractString)\n\nWatch file or directory path (which must exist) for changes until a change occurs. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nfm = FileMonitor(path) acts like an auto-reset Event, so wait(fm) blocks until there has been at least one event in the file originally at the given path and then returns an object with boolean fields renamed, changed, timedout summarizing all changes that have occurred since the last call to wait returned.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.FolderMonitor","page":"File Events","title":"FileWatching.FolderMonitor","category":"type","text":"FolderMonitor(folder::AbstractString)\n\nWatch a file or directory path for changes until a change has occurred. This function does not poll the file system and instead uses platform-specific functionality to receive notifications from the operating system (e.g. via inotify on Linux). See the NodeJS documentation linked below for details.\n\nThis acts similar to a Channel, so calling take! (or wait) blocks until some change has occurred. The wait function will return a pair where the first field is the name of the changed file (if available) and the second field is an object with boolean fields renamed and changed, giving the event that occurred on it.\n\nThis behavior of this function varies slightly across platforms. See https://nodejs.org/api/fs.html#fs_caveats for more detailed information.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.PollingFileWatcher","page":"File Events","title":"FileWatching.PollingFileWatcher","category":"type","text":"PollingFileWatcher(path::AbstractString, interval_s::Real=5.007)\n\nMonitor a file for changes by polling stat every interval_s seconds until a change occurs or timeout_s seconds have elapsed. The interval_s should be a long period; the default is 5.007 seconds. Call stat on it to get the most recent, but old, result.\n\nThis acts like an auto-reset Event, so calling wait blocks until the stat result has changed since the previous value captured upon entry to the wait call. The wait function will return a pair of status objects (previous, current) once any stat change is detected since the previous time that wait was called. The previous status is always a StatStruct, but it may have all of the fields zeroed (indicating the file didn't previously exist, or wasn't previously accessible).\n\nThe current status object may be a StatStruct, an EOFError (if the wait is canceled by closing this object), or some other Exception subtype (if the stat operation failed: for example, if the path is removed). Note that stat value may be outdated if the file has changed again multiple times.\n\nUsing FileMonitor for this operation is preferred, since it is more reliable and efficient, although in some situations it may not be available.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.FDWatcher","page":"File Events","title":"FileWatching.FDWatcher","category":"type","text":"FDWatcher(fd::Union{RawFD,WindowsRawSocket}, readable::Bool, writable::Bool)\n\nMonitor a file descriptor fd for changes in the read or write availability.\n\nThe keyword arguments determine which of read and/or write status should be monitored; at least one of them must be set to true.\n\nThe returned value is an object with boolean fields readable, writable, and timedout, giving the result of the polling.\n\nThis acts like a level-set event, so calling wait blocks until one of those conditions is met, but then continues to return without blocking until the condition is cleared (either there is no more to read, or no more space in the write buffer, or both).\n\nwarning: Warning\nYou must call close manually, when finished with this object, before the fd argument is closed. Failure to do so risks serious crashes.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.mkpidlock","page":"File Events","title":"FileWatching.Pidfile.mkpidlock","category":"function","text":"mkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\nmkpidlock(at::String, proc::Process; kwopts...)\n\nCreate a pidfile lock for the path \"at\" for the current process or the process identified by pid or proc. Can take a function to execute once locked, for usage in do blocks, after which the lock will be automatically closed. If the lock fails and wait is false, then an error is thrown.\n\nThe lock will be released by either close, a finalizer, or shortly after proc exits. Make sure the return value is live through the end of the critical section of your program, so the finalizer does not reclaim it early.\n\nOptional keyword arguments:\n\nmode: file access mode (modified by the process umask). Defaults to world-readable.\npoll_interval: Specify the maximum time to between attempts (if watch_file doesn't work)\nstale_age: Delete an existing pidfile (ignoring the lock) if it is older than this many seconds, based on its mtime.   The file won't be deleted until 5x longer than this if the pid in the file appears that it may be valid.   Or 25x longer if refresh is overridden to 0 to disable lock refreshing.   By default this is disabled (stale_age = 0), but a typical recommended value would be about 3-5x an   estimated normal completion time.\nrefresh: Keeps a lock from becoming stale by updating the mtime every interval of time that passes.   By default, this is set to stale_age/2, which is the recommended value.\nwait: If true, block until we get the lock, if false, raise error if lock fails.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.trymkpidlock","page":"File Events","title":"FileWatching.Pidfile.trymkpidlock","category":"function","text":"trymkpidlock([f::Function], at::String, [pid::Cint]; kwopts...)\ntrymkpidlock(at::String, proc::Process; kwopts...)\n\nLike mkpidlock except returns false instead of waiting if the file is already locked.\n\ncompat: Julia 1.10\nThis function requires at least Julia 1.10.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#Base.close-Tuple{FileWatching.Pidfile.LockMonitor}","page":"File Events","title":"Base.close","category":"method","text":"close(lock::LockMonitor)\n\nRelease a pidfile lock.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.open_exclusive","page":"File Events","title":"FileWatching.Pidfile.open_exclusive","category":"function","text":"open_exclusive(path::String; mode, poll_interval, wait, stale_age, refresh) :: File\n\nCreate a new a file for read-write advisory-exclusive access. If wait is false then error out if the lock files exist otherwise block until we get the lock.\n\nFor a description of the keyword arguments, see mkpidlock.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.tryopen_exclusive","page":"File Events","title":"FileWatching.Pidfile.tryopen_exclusive","category":"function","text":"tryopen_exclusive(path::String, mode::Integer = 0o444) :: Union{Void, File}\n\nTry to create a new file for read-write advisory-exclusive access, return nothing if it already exists.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.write_pidfile","page":"File Events","title":"FileWatching.Pidfile.write_pidfile","category":"function","text":"write_pidfile(io, pid)\n\nWrite our pidfile format to an open IO descriptor.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.parse_pidfile","page":"File Events","title":"FileWatching.Pidfile.parse_pidfile","category":"function","text":"parse_pidfile(file::Union{IO, String}) => (pid, hostname, age)\n\nAttempt to parse our pidfile format, replaced an element with (0, \"\", 0.0), respectively, for any read that failed.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.stale_pidfile","page":"File Events","title":"FileWatching.Pidfile.stale_pidfile","category":"function","text":"stale_pidfile(path::String, stale_age::Real, refresh::Real) :: Bool\n\nHelper function for open_exclusive for deciding if a pidfile is stale.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#FileWatching.Pidfile.isvalidpid","page":"File Events","title":"FileWatching.Pidfile.isvalidpid","category":"function","text":"isvalidpid(hostname::String, pid::Cuint) :: Bool\n\nAttempt to conservatively estimate whether pid is a valid process id.\n\n\n\n\n\n"},{"location":"stdlib/FileWatching.html#Base.Filesystem.touch-Tuple{FileWatching.Pidfile.LockMonitor}","page":"File Events","title":"Base.Filesystem.touch","category":"method","text":"Base.touch(::Pidfile.LockMonitor)\n\nUpdate the mtime on the lock, to indicate it is still fresh.\n\nSee also the refresh keyword in the mkpidlock constructor.\n\n\n\n\n\n"},{"location":"devdocs/ssair.html#Julia-SSA-form-IR","page":"Julia SSA-form IR","title":"Julia SSA-form IR","category":"section","text":"Julia uses a static single assignment intermediate representation (SSA IR) to perform optimization. This IR is different from LLVM IR, and unique to Julia. It allows for Julia specific optimizations.\n\nBasic blocks (regions with no control flow) are explicitly annotated.\nif/else and loops are turned into goto statements.\nlines with multiple operations are split into multiple lines by introducing variables.\n\nFor example the following Julia code:\n\nfunction foo(x)\n    y = sin(x)\n    if x > 5.0\n        y = y + cos(x)\n    end\n    return exp(2) + y\nend\n\nwhen called with a Float64 argument is translated into:\n\nusing InteractiveUtils\n@code_typed foo(1.0)\n\nCodeInfo(\n1 ─ %1 = invoke Main.sin(x::Float64)::Float64\n│   %2 = Base.lt_float(x, 5.0)::Bool\n└──      goto #3 if not %2\n2 ─ %4 = invoke Main.cos(x::Float64)::Float64\n└── %5 = Base.add_float(%1, %4)::Float64\n3 ┄ %6 = φ (#2 => %5, #1 => %1)::Float64\n│   %7 = Base.add_float(7.38905609893065, %6)::Float64\n└──      return %7\n) => Float64\n\nIn this example, we can see all of these changes.\n\nThe first basic block is everything in\n\n1 ─ %1 = invoke Main.sin(x::Float64)::Float64\n│   %2 = Base.lt_float(x, 5.0)::Bool\n└──      goto #3 if not %2\n\nThe if statement is translated into goto #3 if not %2 which goes to the 3rd basic block if x>5 isn't met and otherwise goes to the second basic block.\n%2 is an SSA value introduced to represent x > 5."},{"location":"devdocs/ssair.html#Background","page":"Julia SSA-form IR","title":"Background","category":"section","text":"Beginning in Julia 0.7, parts of the compiler use a new SSA-form intermediate representation (IR). Historically, the compiler would directly generate LLVM IR from a lowered form of the Julia AST. This form had most syntactic abstractions removed, but still looked a lot like an abstract syntax tree. Over time, in order to facilitate optimizations, SSA values were introduced to this IR and the IR was linearized (i.e. turned into a form where function arguments could only be SSA values or constants). However, non-SSA values (slots) remained in the IR due to the lack of Phi nodes in the IR (necessary for back-edges and re-merging of conditional control flow). This negated much of the usefulness of SSA form representation when performing middle end optimizations. Some heroic effort was put into making these optimizations work without a complete SSA form representation, but the lack of such a representation ultimately proved prohibitive."},{"location":"devdocs/ssair.html#Categories-of-IR-nodes","page":"Julia SSA-form IR","title":"Categories of IR nodes","category":"section","text":"The SSA IR representation has four categories of IR nodes: Phi, Pi, PhiC, and Upsilon nodes (the latter two are only used for exception handling)."},{"location":"devdocs/ssair.html#Phi-nodes-and-Pi-nodes","page":"Julia SSA-form IR","title":"Phi nodes and Pi nodes","category":"section","text":"Phi nodes are part of generic SSA abstraction (see the link above if you're not familiar with the concept). In the Julia IR, these nodes are represented as:\n\nstruct PhiNode\n    edges::Vector{Int32}\n    values::Vector{Any}\nend\n\nwhere we ensure that both vectors always have the same length. In the canonical representation (the one handled by codegen and the interpreter), the edge values indicate come-from statement numbers (i.e. if edge has an entry of 15, there must be a goto, gotoifnot or implicit fall through from statement 15 that targets this phi node). Values are either SSA values or constants. It is also possible for a value to be unassigned if the variable was not defined on this path. However, undefinedness checks get explicitly inserted and represented as booleans after middle end optimizations, so code generators may assume that any use of a Phi node will have an assigned value in the corresponding slot. It is also legal for the mapping to be incomplete, i.e. for a Phi node to have missing incoming edges. In that case, it must be dynamically guaranteed that the corresponding value will not be used.\n\nNote that SSA uses semantically occur after the terminator of the corresponding predecessor (\"on the edge\"). Consequently, if multiple Phi nodes appear at the start of a basic block, they are run simultaneously. This means that in the following IR snippet, if we came from block 23, %46 will take the value associated to %45 before we entered this block.\n\n%45 = φ (#18 => %23, #23 => %50)\n%46 = φ (#18 => 1.0, #23 => %45)\n\nPiNodes encode statically proven information that may be implicitly assumed in basic blocks dominated by a given pi node. They are conceptually equivalent to the technique introduced in the paper ABCD: Eliminating Array Bounds Checks on Demand or the predicate info nodes in LLVM. To see how they work, consider, e.g.\n\n%x::Union{Int, Float64} # %x is some Union{Int, Float64} typed ssa value\nif isa(x, Int)\n    # use x\nelse\n    # use x\nend\n\nWe can perform predicate insertion and turn this into:\n\n%x::Union{Int, Float64} # %x is some Union{Int, Float64} typed ssa value\nif isa(x, Int)\n    %x_int = PiNode(x, Int)\n    # use %x_int\nelse\n    %x_float = PiNode(x, Float64)\n    # use %x_float\nend\n\nPi nodes are generally ignored in the interpreter, since they don't have any effect on the values, but they may sometimes lead to code generation in the compiler (e.g. to change from an implicitly union split representation to a plain unboxed representation). The main usefulness of PiNodes stems from the fact that path conditions of the values can be accumulated simply by def-use chain walking that is generally done for most optimizations that care about these conditions anyway."},{"location":"devdocs/ssair.html#PhiC-nodes-and-Upsilon-nodes","page":"Julia SSA-form IR","title":"PhiC nodes and Upsilon nodes","category":"section","text":"Exception handling complicates the SSA story moderately, because exception handling introduces additional control flow edges into the IR across which values must be tracked. One approach to do so, which is followed by LLVM, is to make calls which may throw exceptions into basic block terminators and add an explicit control flow edge to the catch handler:\n\ninvoke @function_that_may_throw() to label %regular unwind to %catch\n\nregular:\n# Control flow continues here\n\ncatch:\n# Exceptions go here\n\nHowever, this is problematic in a language like Julia, where at the start of the optimization pipeline, we do not know which calls throw. We would have to conservatively assume that every call (which in Julia is every statement) throws. This would have several negative effects. On the one hand, it would essentially reduce the scope of every basic block to a single call, defeating the purpose of having operations be performed at the basic block level. On the other hand, every catch basic block would have n*m phi node arguments (n, the number of statements in the critical region, m the number of live values through the catch block).\n\nTo work around this, we use a combination of Upsilon and PhiC nodes (the C standing for catch, written φᶜ in the IR pretty printer, because unicode subscript c is not available). There are several ways to think of these nodes, but perhaps the easiest is to think of each PhiC as a load from a unique store-many, read-once slot, with Upsilon being the corresponding store operation. The PhiC has an operand list of all the upsilon nodes that store to its implicit slot. The Upsilon nodes however, do not record which PhiC node they store to. This is done for more natural integration with the rest of the SSA IR. E.g. if there are no more uses of a PhiC node, it is safe to delete it, and the same is true of an Upsilon node. In most IR passes, PhiC nodes can be treated like Phi nodes. One can follow use-def chains through them, and they can be lifted to new PhiC nodes and new Upsilon nodes (in the same places as the original Upsilon nodes). The result of this scheme is that the number of Upsilon nodes (and PhiC arguments) is proportional to the number of assigned values to a particular variable (before SSA conversion), rather than the number of statements in the critical region.\n\nTo see this scheme in action, consider the function\n\n@noinline opaque() = invokelatest(identity, nothing) # Something opaque\nfunction foo()\n    local y\n    x = 1\n    try\n        y = 2\n        opaque()\n        y = 3\n        error()\n    catch\n    end\n    (x, y)\nend\n\nThe corresponding IR (with irrelevant types stripped) is:\n\n1 ─       nothing::Nothing\n2 ─ %2  = $(Expr(:enter, #4))\n3 ─ %3  = ϒ (false)\n│   %4  = ϒ (#undef)\n│   %5  = ϒ (1)\n│   %6  = ϒ (true)\n│   %7  = ϒ (2)\n│         invoke Main.opaque()::Any\n│   %9  = ϒ (true)\n│   %10 = ϒ (3)\n│         invoke Main.error()::Union{}\n└──       $(Expr(:unreachable))::Union{}\n4 ┄ %13 = φᶜ (%3, %6, %9)::Bool\n│   %14 = φᶜ (%4, %7, %10)::Core.Compiler.MaybeUndef(Int64)\n│   %15 = φᶜ (%5)::Core.Const(1)\n└──       $(Expr(:leave, Core.SSAValue(2)))\n5 ─       $(Expr(:pop_exception, :(%2)))::Any\n│         $(Expr(:throw_undef_if_not, :y, :(%13)))::Any\n│   %19 = Core.tuple(%15, %14)\n└──       return %19\n\nNote in particular that every value live into the critical region gets an upsilon node at the top of the critical region. This is because catch blocks are considered to have an invisible control flow edge from outside the function. As a result, no SSA value dominates the catch blocks, and all incoming values have to come through a φᶜ node."},{"location":"devdocs/ssair.html#Main-SSA-data-structure","page":"Julia SSA-form IR","title":"Main SSA data structure","category":"section","text":"The main SSAIR data structure is worthy of discussion. It draws inspiration from LLVM and Webkit's B3 IR. The core of the data structure is a flat vector of statements. Each statement is implicitly assigned an SSA value based on its position in the vector (i.e. the result of the statement at idx 1 can be accessed using SSAValue(1) etc). For each SSA value, we additionally maintain its type. Since, SSA values are definitionally assigned only once, this type is also the result type of the expression at the corresponding index. However, while this representation is rather efficient (since the assignments don't need to be explicitly encoded), it of course carries the drawback that order is semantically significant, so reorderings and insertions change statement numbers. Additionally, we do not keep use lists (i.e. it is impossible to walk from a def to all its uses without explicitly computing this map–def lists however are trivial since you can look up the corresponding statement from the index), so the LLVM-style RAUW (replace-all-uses-with) operation is unavailable.\n\nInstead, we do the following:\n\nWe keep a separate buffer of nodes to insert (including the position to insert them at, the type of the corresponding value and the node itself). These nodes are numbered by their occurrence in the insertion buffer, allowing their values to be immediately used elsewhere in the IR (i.e. if there are 12 statements in the original statement list, the first new statement will be accessible as SSAValue(13)).\nRAUW style operations are performed by setting the corresponding statement index to the replacement value.\nStatements are erased by setting the corresponding statement to nothing (this is essentially just a special-case convention of the above).\nIf there are any uses of the statement being erased, they will be set to nothing.\n\nThere is a compact! function that compacts the above data structure by performing the insertion of nodes in the appropriate place, trivial copy propagation, and renaming of uses to any changed SSA values. However, the clever part of this scheme is that this compaction can be done lazily as part of the subsequent pass. Most optimization passes need to walk over the entire list of statements, performing analysis or modifications along the way. We provide an IncrementalCompact iterator that can be used to iterate over the statement list. It will perform any necessary compaction and return the new index of the node, as well as the node itself. It is legal at this point to walk def-use chains, as well as make any modifications or deletions to the IR (insertions are disallowed however).\n\nThe idea behind this arrangement is that, since the optimization passes need to touch the corresponding memory anyway and incur the corresponding memory access penalty, performing the extra housekeeping should have comparatively little overhead (and save the overhead of maintaining these data structures during IR modification)."},{"location":"manual/stacktraces.html#Stack-Traces","page":"Stack Traces","title":"Stack Traces","category":"section","text":"The StackTraces module provides simple stack traces that are both human readable and easy to use programmatically."},{"location":"manual/stacktraces.html#Viewing-a-stack-trace","page":"Stack Traces","title":"Viewing a stack trace","category":"section","text":"The primary function used to obtain a stack trace is stacktrace:\n\n6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n\nCalling stacktrace() returns a vector of StackTraces.StackFrame s. For ease of use, the alias StackTraces.StackTrace can be used in place of Vector{StackFrame}. (Examples with [...] indicate that output may vary depending on how the code is run.)\n\njulia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]\n\njulia> @noinline child() = stacktrace()\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> grandparent() = parent()\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\n9-element Array{Base.StackTraces.StackFrame,1}:\n child() at REPL[3]:1\n parent() at REPL[4]:1\n grandparent() at REPL[5]:1\n[...]\n\nNote that when calling stacktrace() you'll typically see a frame with eval at boot.jl. When calling stacktrace() from the REPL you'll also have a few extra frames in the stack from REPL.jl, usually looking something like this:\n\njulia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92"},{"location":"manual/stacktraces.html#Extracting-useful-information","page":"Stack Traces","title":"Extracting useful information","category":"section","text":"Each StackTraces.StackFrame contains the function name, file name, line number, lambda info, a flag indicating whether the frame has been inlined, a flag indicating whether it is a C function (by default C functions do not appear in the stack trace), and an integer representation of the pointer returned by backtrace:\n\njulia> frame = stacktrace()[3]\neval(::Module, ::Expr) at REPL.jl:5\n\njulia> frame.func\n:eval\n\njulia> frame.file\nSymbol(\"~/julia/usr/share/julia/stdlib/v0.7/REPL/src/REPL.jl\")\n\njulia> frame.line\n5\n\njulia> frame.linfo\nMethodInstance for eval(::Module, ::Expr)\n\njulia> frame.inlined\nfalse\n\njulia> frame.from_c\nfalse\n\njulia> frame.pointer\n0x00007f92d6293171\n\nThis makes stack trace information available programmatically for logging, error handling, and more."},{"location":"manual/stacktraces.html#Error-handling","page":"Stack Traces","title":"Error handling","category":"section","text":"While having easy access to information about the current state of the callstack can be helpful in many places, the most obvious application is in error handling and debugging.\n\njulia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace()\n       end\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[2]:4\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]\n\nYou may notice that in the example above the first stack frame points at line 4, where stacktrace is called, rather than line 2, where bad_function is called, and bad_function's frame is missing entirely. This is understandable, given that stacktrace is called from the context of the catch. While in this example it's fairly easy to find the actual source of the error, in complex cases tracking down the source of the error becomes nontrivial.\n\nThis can be remedied by passing the result of catch_backtrace to stacktrace. Instead of returning callstack information for the current context, catch_backtrace returns stack information for the context of the most recent exception:\n\njulia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace(catch_backtrace())\n       end\nexample (generic function with 1 method)\n\njulia> example()\n8-element Array{Base.StackTraces.StackFrame,1}:\n bad_function() at REPL[1]:1\n example() at REPL[2]:2\n[...]\n\nNotice that the stack trace now indicates the appropriate line number and the missing frame.\n\njulia> @noinline child() = error(\"Whoops!\")\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> @noinline function grandparent()\n           try\n               parent()\n           catch err\n               println(\"ERROR: \", err.msg)\n               stacktrace(catch_backtrace())\n           end\n       end\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\nERROR: Whoops!\n10-element Array{Base.StackTraces.StackFrame,1}:\n error at error.jl:33 [inlined]\n child() at REPL[1]:1\n parent() at REPL[2]:1\n grandparent() at REPL[3]:3\n[...]"},{"location":"manual/stacktraces.html#Exception-stacks-and-[current_exceptions](@ref)","page":"Stack Traces","title":"Exception stacks and current_exceptions","category":"section","text":"compat: Julia 1.1\nException stacks requires at least Julia 1.1.\n\nWhile handling an exception further exceptions may be thrown. It can be useful to inspect all these exceptions to identify the root cause of a problem. The julia runtime supports this by pushing each exception onto an internal exception stack as it occurs. When the code exits a catch normally, any exceptions which were pushed onto the stack in the associated try are considered to be successfully handled and are removed from the stack.\n\nThe stack of current exceptions can be accessed using the current_exceptions function. For example,\n\njulia> try\n           error(\"(A) The root cause\")\n       catch\n           try\n               error(\"(B) An exception while handling the exception\")\n           catch\n               for (exc, bt) in current_exceptions()\n                   showerror(stdout, exc, bt)\n                   println(stdout)\n               end\n           end\n       end\n(A) The root cause\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:2\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259\n(B) An exception while handling the exception\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:5\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259\n\nIn this example the root cause exception (A) is first on the stack, with a further exception (B) following it. After exiting both catch blocks normally (i.e., without throwing a further exception) all exceptions are removed from the stack and are no longer accessible.\n\nThe exception stack is stored on the Task where the exceptions occurred. When a task fails with uncaught exceptions, current_exceptions(task) may be used to inspect the exception stack for that task."},{"location":"manual/stacktraces.html#Comparison-with-[backtrace](@ref)","page":"Stack Traces","title":"Comparison with backtrace","category":"section","text":"A call to backtrace returns a vector of Union{Ptr{Nothing}, Base.InterpreterIP}, which may then be passed into stacktrace for translation:\n\njulia> trace = backtrace()\n18-element Array{Union{Ptr{Nothing}, Base.InterpreterIP},1}:\n Ptr{Nothing} @0x00007fd8734c6209\n Ptr{Nothing} @0x00007fd87362b342\n Ptr{Nothing} @0x00007fd87362c136\n Ptr{Nothing} @0x00007fd87362c986\n Ptr{Nothing} @0x00007fd87362d089\n Base.InterpreterIP(CodeInfo(:(begin\n      Core.SSAValue(0) = backtrace()\n      trace = Core.SSAValue(0)\n      return Core.SSAValue(0)\n  end)), 0x0000000000000000)\n Ptr{Nothing} @0x00007fd87362e4cf\n[...]\n\njulia> stacktrace(trace)\n6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n\nNotice that the vector returned by backtrace had 18 elements, while the vector returned by stacktrace only has 6. This is because, by default, stacktrace removes any lower-level C functions from the stack. If you want to include stack frames from C calls, you can do it like this:\n\njulia> stacktrace(trace, true)\n21-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n do_call at interpreter.c:324\n eval_value at interpreter.c:416\n eval_body at interpreter.c:559\n jl_interpret_toplevel_thunk_callback at interpreter.c:798\n top-level scope\n jl_interpret_toplevel_thunk at interpreter.c:807\n jl_toplevel_eval_flex at toplevel.c:856\n jl_toplevel_eval_in at builtins.c:624\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n jl_apply_generic at gf.c:2167\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n jl_apply_generic at gf.c:2167\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n jl_fptr_trampoline at gf.c:1838\n jl_apply_generic at gf.c:2167\n jl_apply at julia.h:1540 [inlined]\n start_task at task.c:268\n ip:0xffffffffffffffff\n\nIndividual pointers returned by backtrace can be translated into StackTraces.StackFrame s by passing them into StackTraces.lookup:\n\njulia> pointer = backtrace()[1];\n\njulia> frame = StackTraces.lookup(pointer)\n1-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n\njulia> println(\"The top frame is from $(frame[1].func)!\")\nThe top frame is from jl_apply_generic!"},{"location":"manual/getting-started.html#man-getting-started","page":"Getting Started","title":"Getting Started","category":"section","text":"Julia installation is straightforward, whether using precompiled binaries or compiling from source. Download and install Julia by following the instructions at https://julialang.org/install/.\n\nIf you are coming to Julia from one of the following languages, then you should start by reading the section on noteworthy differences from MATLAB, R, Python, C/C++ or Common Lisp. This will help you avoid some common pitfalls since Julia differs from those languages in many subtle ways.\n\nThe easiest way to learn and experiment with Julia is by starting an interactive session (also known as a read-eval-print loop or \"REPL\") by double-clicking the Julia executable or running julia from the command line:\n\nusing REPL\nio = IOBuffer()\nREPL.banner(io)\nbanner = takestring!(io)\nimport Markdown\nMarkdown.parse(\"```\\n\\$ julia\\n\\n$(banner)\\njulia> 1 + 2\\n3\\n\\njulia> ans\\n3\\n```\")\n\nTo exit the interactive session, type CTRL-D (press the Control/^ key together with the d key), or type exit(). When run in interactive mode, julia displays a banner and prompts the user for input. Once the user has entered a complete expression, such as 1 + 2, and hits enter, the interactive session evaluates the expression and shows its value. If an expression is entered into an interactive session with a trailing semicolon, its value is not shown. The variable ans is bound to the value of the last evaluated expression whether it is shown or not. The ans variable is only bound in interactive sessions, not when Julia code is run in other ways.\n\nTo evaluate expressions written in a source file file.jl, write include(\"file.jl\").\n\nTo run code in a file non-interactively, you can give it as the first argument to the julia command:\n\n$ julia script.jl\n\nYou can pass additional arguments to Julia, and to your program script.jl. A detailed list of all the available options can be found under Command-line Interface."},{"location":"manual/getting-started.html#Resources","page":"Getting Started","title":"Resources","category":"section","text":"A curated list of useful learning resources to help new users get started can be found on the learning page of the main Julia website.\n\nYou can use the REPL as a learning resource by switching into the help mode. Switch to help mode by pressing ? at an empty julia> prompt, before typing anything else. Typing a keyword in help mode will fetch the documentation for it, along with examples. Similarly for most functions or other objects you might encounter!\n\nhelp?> begin\nsearch: begin disable_sigint reenable_sigint\n\n  begin\n\n  begin...end denotes a block of code.\n\nIf you already know Julia a bit, you might want to peek ahead at Performance Tips and Workflow Tips, or check out the comprehensive ModernJuliaWorkflows blog."},{"location":"manual/installation.html#man-installation","page":"Installation","title":"Installation","category":"section","text":"There are many ways to install Julia. The following sections highlight the recommended method for each of the main supported platforms, and then present alternative ways that might be useful in specialized situations.\n\nThe current installation recommendation is a solution based on Juliaup. If you installed Julia previously with a method that is not based on Juliaup and want to switch your system to an installation that is based on Juliaup, we recommend that you uninstall all previous Julia versions, ensure that you remove anything Julia related from your PATH variable and then install Julia with one of the methods described below."},{"location":"manual/installation.html#Windows","page":"Installation","title":"Windows","category":"section","text":"On Windows Julia can be installed directly from the Windows store here. One can also install exactly the same version by executing\n\nwinget install --name Julia --id 9NJNWW8PVKMN -e -s msstore\n\nin any shell."},{"location":"manual/installation.html#Mac-and-Linux","page":"Installation","title":"Mac and Linux","category":"section","text":"Julia can be installed on Linux or Mac by executing\n\ncurl -fsSL https://install.julialang.org | sh\n\nin a shell."},{"location":"manual/installation.html#Command-line-arguments","page":"Installation","title":"Command line arguments","category":"section","text":"One can pass various command line arguments to the Julia installer. The syntax for installer arguments is\n\ncurl -fsSL https://install.julialang.org | sh -s -- <ARGS>\n\nHere <ARGS> should be replaced with one or more of the following arguments:\n\n--yes (or -y): Run the installer in a non-interactive mode. All configuration values use their default or a value supplied as a command line argument.\n--default-channel=<NAME>: Configure the default Juliaup channel. For example --default-channel lts would install the lts channel and configure it as the default.\n--add-to-path=<yes|no>: Configure whether Julia should be added to the PATH environment variable. Valid values are yes (default) and no.\n--background-selfupdate=<SECONDS>: Configure an optional CRON job that auto-updates Juliaup if <SECONDS> has a value larger than 0. The actual value controls how often the CRON job will run to check for a new Juliaup version in seconds. The default value is 0, i.e. no CRON job will be created.\n--startup-selfupdate=<MINUTES>: Configure how often Julia will check for new versions of Juliaup when Julia is started. The default is every 1440 minutes.\n-p=<PATH> (or --path): Configure where the Julia and Juliaup binaries are installed. The default is ~/.juliaup."},{"location":"manual/installation.html#Alternative-installation-methods","page":"Installation","title":"Alternative installation methods","category":"section","text":"Note that we recommend the following methods only if none of the installation methods described above work for your system.\n\nSome of the installation methods described below recommend installing a package called juliaup. Note that this nevertheless installs a fully functional Julia system, not just Juliaup."},{"location":"manual/installation.html#App-Installer-(Windows)","page":"Installation","title":"App Installer (Windows)","category":"section","text":"If the Windows Store is blocked on a system, we have an alternative MSIX App Installer based setup. To use the App Installer version, download this file and open it by double clicking on it. One can also install exactly the same version by executing the PowerShell command\n\nAdd-AppxPackage -AppInstallerFile https://install.julialang.org/Julia.appinstaller"},{"location":"manual/installation.html#MSI-Installer-(Windows)","page":"Installation","title":"MSI Installer (Windows)","category":"section","text":"If neither the Windows Store nor the App Installer version work on your Windows system, you can also use a MSI based installer. Note that this installation methods comes with serious limitations and is generally not recommended unless no other method works. For example, there is no automatic update mechanism for Juliaup with this installation method. The 64 bit version of the MSI installer can be downloaded from here and the 32 bit version from here.\n\nBy default the install will be a per-user install that does not require  elevation. You can also do a system install by running the following command  from a shell:\n\nmsiexec /i <PATH_TO_JULIA_MSI> ALLUSERS=1"},{"location":"manual/installation.html#[Homebrew](https://brew.sh)-(Mac-and-Linux)","page":"Installation","title":"Homebrew (Mac and Linux)","category":"section","text":"On systems with brew, you can install Julia by running\n\nbrew install juliaup\n\nin a shell. Note that you will have to update Juliaup with standard brew commands."},{"location":"manual/installation.html#[Arch-Linux-AUR](https://aur.archlinux.org/packages/juliaup/)-(Linux)","page":"Installation","title":"Arch Linux - AUR (Linux)","category":"section","text":"On Arch Linux, Juliaup is available in the Arch User Repository (AUR)."},{"location":"manual/installation.html#[openSUSE-Tumbleweed](https://get.opensuse.org/tumbleweed/)-(Linux)","page":"Installation","title":"openSUSE Tumbleweed (Linux)","category":"section","text":"On openSUSE Tumbleweed, you can install Julia by running\n\nzypper install juliaup\n\nin a shell with root privileges."},{"location":"manual/installation.html#[cargo](https://crates.io/crates/juliaup/)-(Windows,-Mac-and-Linux)","page":"Installation","title":"cargo (Windows, Mac and Linux)","category":"section","text":"To install Julia via Rust's cargo, run:\n\ncargo install juliaup"},{"location":"stdlib/Future.html#Future","page":"Future","title":"Future","category":"section","text":"The Future module implements future behavior of already existing functions, which will replace the current version in a future release of Julia."},{"location":"stdlib/Future.html#Future.copy!","page":"Future","title":"Future.copy!","category":"function","text":"Future.copy!(dst, src) -> dst\n\nCopy src into dst.\n\ncompat: Julia 1.1\nThis function has moved to Base with Julia 1.1, consider using copy!(dst, src) instead. Future.copy! will be deprecated in the future.\n\n\n\n\n\n"},{"location":"stdlib/Future.html#Future.randjump","page":"Future","title":"Future.randjump","category":"function","text":"randjump(r::MersenneTwister, steps::Integer)::MersenneTwister\n\nCreate an initialized MersenneTwister object, whose state is moved forward (without generating numbers) from r by steps steps. One such step corresponds to the generation of two Float64 numbers. For each different value of steps, a large polynomial has to be generated internally. One is already pre-computed for steps=big(10)^20.\n\n\n\n\n\n"},{"location":"manual/integers-and-floating-point-numbers.html#Integers-and-Floating-Point-Numbers","page":"Integers and Floating-Point Numbers","title":"Integers and Floating-Point Numbers","category":"section","text":"Integers and floating-point values are the basic building blocks of arithmetic and computation. Built-in representations of such values are called numeric primitives, while representations of integers and floating-point numbers as immediate values in code are known as numeric literals. For example, 1 is an integer literal, while 1.0 is a floating-point literal; their binary in-memory representations as objects are numeric primitives.\n\nJulia provides a broad range of primitive numeric types, and a full complement of arithmetic and bitwise operators as well as standard mathematical functions are defined over them. These map directly onto numeric types and operations that are natively supported on modern computers, thus allowing Julia to take full advantage of computational resources. Additionally, Julia provides software support for Arbitrary Precision Arithmetic, which can handle operations on numeric values that cannot be represented effectively in native hardware representations, but at the cost of relatively slower performance.\n\nThe following are Julia's primitive numeric types:\n\nInteger types:\n\nType Signed? Number of bits Smallest value Largest value\nInt8 ✓ 8 -2^7 2^7 - 1\nUInt8  8 0 2^8 - 1\nInt16 ✓ 16 -2^15 2^15 - 1\nUInt16  16 0 2^16 - 1\nInt32 ✓ 32 -2^31 2^31 - 1\nUInt32  32 0 2^32 - 1\nInt64 ✓ 64 -2^63 2^63 - 1\nUInt64  64 0 2^64 - 1\nInt128 ✓ 128 -2^127 2^127 - 1\nUInt128  128 0 2^128 - 1\nBool N/A 8 false (0) true (1)\n\nFloating-point types:\n\nType Precision Number of bits\nFloat16 half 16\nFloat32 single 32\nFloat64 double 64\n\nAdditionally, full support for Complex and Rational Numbers is built on top of these primitive numeric types. All numeric types interoperate naturally without explicit casting, thanks to a flexible, user-extensible type promotion system."},{"location":"manual/integers-and-floating-point-numbers.html#Integers","page":"Integers and Floating-Point Numbers","title":"Integers","category":"section","text":"Literal integers are represented in the standard manner:\n\njulia> 1\n1\n\njulia> 1234\n1234\n\nThe default type for an integer literal depends on whether the target system has a 32-bit architecture or a 64-bit architecture:\n\n# 32-bit system:\njulia> typeof(1)\nInt32\n\n# 64-bit system:\njulia> typeof(1)\nInt64\n\nThe Julia internal variable Sys.WORD_SIZE indicates whether the target system is 32-bit or 64-bit:\n\n# 32-bit system:\njulia> Sys.WORD_SIZE\n32\n\n# 64-bit system:\njulia> Sys.WORD_SIZE\n64\n\nJulia also defines the types Int and UInt, which are aliases for the system's signed and unsigned native integer types respectively:\n\n# 32-bit system:\njulia> Int\nInt32\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> Int\nInt64\njulia> UInt\nUInt64\n\nLarger integer literals that cannot be represented using only 32 bits but can be represented in 64 bits always create 64-bit integers, regardless of the system type:\n\n# 32-bit or 64-bit system:\njulia> typeof(3000000000)\nInt64\n\nUnsigned integers are input and output using the 0x prefix and hexadecimal (base 16) digits 0-9a-f (the capitalized digits A-F also work for input). The size of the unsigned value is determined by the number of hex digits used:\n\njulia> x = 0x1\n0x01\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x123\n0x0123\n\njulia> typeof(x)\nUInt16\n\njulia> x = 0x1234567\n0x01234567\n\njulia> typeof(x)\nUInt32\n\njulia> x = 0x123456789abcdef\n0x0123456789abcdef\n\njulia> typeof(x)\nUInt64\n\njulia> x = 0x11112222333344445555666677778888\n0x11112222333344445555666677778888\n\njulia> typeof(x)\nUInt128\n\nThis behavior is based on the observation that when one uses unsigned hex literals for integer values, one typically is using them to represent a fixed numeric byte sequence, rather than just an integer value.\n\nBinary and octal literals are also supported:\n\njulia> x = 0b10\n0x02\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0o010\n0x08\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x00000000000000001111222233334444\n0x00000000000000001111222233334444\n\njulia> typeof(x)\nUInt128\n\nAs for hexadecimal literals, binary and octal literals produce unsigned integer types. The size of the binary data item is the minimal needed size, if the leading digit of the literal is not 0. In the case of leading zeros, the size is determined by the minimal needed size for a literal, which has the same length but leading digit 1. It means that:\n\n0x1 and 0x12 are UInt8 literals,\n0x123 and 0x1234 are UInt16 literals,\n0x12345 and 0x12345678 are UInt32 literals,\n0x123456789 and 0x1234567890adcdef are UInt64 literals, etc.\n\nEven if there are leading zero digits which don’t contribute to the value, they count for determining storage size of a literal. So 0x01 is a UInt8 while 0x0001 is a UInt16.\n\nThat allows the user to control the size.\n\nUnsigned literals (starting with 0x) that encode integers too large to be represented as UInt128 values will construct BigInt values instead. This is not an unsigned type but it is the only built-in type big enough to represent such large integer values.\n\nBinary, octal, and hexadecimal literals may be signed by a - immediately preceding the unsigned literal. They produce an unsigned integer of the same size as the unsigned literal would do, with the two's complement of the value:\n\njulia> -0x2\n0xfe\n\njulia> -0x0002\n0xfffe\n\nThe minimum and maximum representable values of primitive numeric types such as integers are given by the typemin and typemax functions:\n\njulia> (typemin(Int32), typemax(Int32))\n(-2147483648, 2147483647)\n\njulia> for T in [Int8,Int16,Int32,Int64,Int128,UInt8,UInt16,UInt32,UInt64,UInt128]\n           println(\"$(lpad(T,7)): [$(typemin(T)),$(typemax(T))]\")\n       end\n   Int8: [-128,127]\n  Int16: [-32768,32767]\n  Int32: [-2147483648,2147483647]\n  Int64: [-9223372036854775808,9223372036854775807]\n Int128: [-170141183460469231731687303715884105728,170141183460469231731687303715884105727]\n  UInt8: [0,255]\n UInt16: [0,65535]\n UInt32: [0,4294967295]\n UInt64: [0,18446744073709551615]\nUInt128: [0,340282366920938463463374607431768211455]\n\nThe values returned by typemin and typemax are always of the given argument type. (The above expression uses several features that have yet to be introduced, including for loops, Strings, and Interpolation, but should be easy enough to understand for users with some existing programming experience.)"},{"location":"manual/integers-and-floating-point-numbers.html#Overflow-behavior","page":"Integers and Floating-Point Numbers","title":"Overflow behavior","category":"section","text":"In Julia, exceeding the maximum representable value of a given type results in a wraparound behavior:\n\njulia> x = typemax(Int64)\n9223372036854775807\n\njulia> x + 1\n-9223372036854775808\n\njulia> x + 1 == typemin(Int64)\ntrue\n\nArithmetic operations with Julia's integer types inherently perform modular arithmetic, mirroring the characteristics of integer arithmetic on modern computer hardware. In scenarios where overflow is a possibility, it is crucial to explicitly check for wraparound effects that can result from such overflows. The Base.Checked module provides a suite of arithmetic operations equipped with overflow checks, which trigger errors if an overflow occurs. For use cases where overflow cannot be tolerated under any circumstances, utilizing the BigInt type, as detailed in Arbitrary Precision Arithmetic, is advisable.\n\nAn example of overflow behavior and how to potentially resolve it is as follows:\n\njulia> 10^19\n-8446744073709551616\n\njulia> big(10)^19\n10000000000000000000"},{"location":"manual/integers-and-floating-point-numbers.html#Division-errors","page":"Integers and Floating-Point Numbers","title":"Division errors","category":"section","text":"Integer division (the div function) has two exceptional cases: dividing by zero, and dividing the lowest negative number (typemin) by -1. Both of these cases throw a DivideError. The remainder and modulus functions (rem and mod) throw a DivideError when their second argument is zero."},{"location":"manual/integers-and-floating-point-numbers.html#Floating-Point-Numbers","page":"Integers and Floating-Point Numbers","title":"Floating-Point Numbers","category":"section","text":"Literal floating-point numbers are represented in the standard formats, using E-notation when necessary:\n\njulia> 1.0\n1.0\n\njulia> 1.\n1.0\n\njulia> 0.5\n0.5\n\njulia> .5\n0.5\n\njulia> -1.23\n-1.23\n\njulia> 1e10\n1.0e10\n\njulia> 2.5e-4\n0.00025\n\nThe above results are all Float64 values. Literal Float32 values can be entered by writing an f in place of e:\n\njulia> x = 0.5f0\n0.5f0\n\njulia> typeof(x)\nFloat32\n\njulia> 2.5f-4\n0.00025f0\n\nValues can be converted to Float32 easily:\n\njulia> x = Float32(-1.5)\n-1.5f0\n\njulia> typeof(x)\nFloat32\n\nHexadecimal floating-point literals are also valid, but only as Float64 values, with p preceding the base-2 exponent:\n\njulia> 0x1p0\n1.0\n\njulia> 0x1.8p3\n12.0\n\njulia> x = 0x.4p-1\n0.125\n\njulia> typeof(x)\nFloat64\n\nHalf-precision floating-point numbers are also supported (Float16) on all platforms, with native instructions used on hardware which supports this number format. Otherwise, operations are implemented in software, and use Float32 for intermediate calculations. As an internal implementation detail, this is achieved under the hood by using LLVM's half type, which behaves similarly to what the GCC -fexcess-precision=16 flag does for C/C++ code.\n\njulia> sizeof(Float16(4.))\n2\n\njulia> 2*Float16(4.)\nFloat16(8.0)\n\nThe underscore _ can be used as digit separator:\n\njulia> 10_000, 0.000_000_005, 0xdead_beef, 0b1011_0010\n(10000, 5.0e-9, 0xdeadbeef, 0xb2)"},{"location":"manual/integers-and-floating-point-numbers.html#Floating-point-zero","page":"Integers and Floating-Point Numbers","title":"Floating-point zero","category":"section","text":"Floating-point numbers have two zeros, positive zero and negative zero. They are equal to each other but have different binary representations, as can be seen using the bitstring function:\n\njulia> 0.0 == -0.0\ntrue\n\njulia> bitstring(0.0)\n\"0000000000000000000000000000000000000000000000000000000000000000\"\n\njulia> bitstring(-0.0)\n\"1000000000000000000000000000000000000000000000000000000000000000\""},{"location":"manual/integers-and-floating-point-numbers.html#Special-floating-point-values","page":"Integers and Floating-Point Numbers","title":"Special floating-point values","category":"section","text":"There are three specified standard floating-point values that do not correspond to any point on the real number line:\n\nFloat16 Float32 Float64 Name Description\nInf16 Inf32 Inf positive infinity a value greater than all finite floating-point values\n-Inf16 -Inf32 -Inf negative infinity a value less than all finite floating-point values\nNaN16 NaN32 NaN not a number a value not == to any floating-point value (including itself)\n\nFor further discussion of how these non-finite floating-point values are ordered with respect to each other and other floats, see Numeric Comparisons. By the IEEE 754 standard, these floating-point values are the results of certain arithmetic operations:\n\njulia> 1/Inf\n0.0\n\njulia> 1/0\nInf\n\njulia> -5/0\n-Inf\n\njulia> 0.000001/0\nInf\n\njulia> 0/0\nNaN\n\njulia> 500 + Inf\nInf\n\njulia> 500 - Inf\n-Inf\n\njulia> Inf + Inf\nInf\n\njulia> Inf - Inf\nNaN\n\njulia> Inf * Inf\nInf\n\njulia> Inf / Inf\nNaN\n\njulia> 0 * Inf\nNaN\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nThe typemin and typemax functions also apply to floating-point types:\n\njulia> (typemin(Float16),typemax(Float16))\n(-Inf16, Inf16)\n\njulia> (typemin(Float32),typemax(Float32))\n(-Inf32, Inf32)\n\njulia> (typemin(Float64),typemax(Float64))\n(-Inf, Inf)"},{"location":"manual/integers-and-floating-point-numbers.html#Machine-epsilon","page":"Integers and Floating-Point Numbers","title":"Machine epsilon","category":"section","text":"Most real numbers cannot be represented exactly with floating-point numbers, and so for many purposes it is important to know the distance between two adjacent representable floating-point numbers, which is often known as machine epsilon.\n\nJulia provides eps, which gives the distance between 1.0 and the next larger representable floating-point value:\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> eps(Float64)\n2.220446049250313e-16\n\njulia> eps() # same as eps(Float64)\n2.220446049250313e-16\n\nThese values are 2.0^-23 and 2.0^-52 as Float32 and Float64 values, respectively. The eps function can also take a floating-point value as an argument, and gives the absolute difference between that value and the next representable floating point value. That is, eps(x) yields a value of the same type as x such that x + eps(x) is the next representable floating-point value larger than x:\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(1000.)\n1.1368683772161603e-13\n\njulia> eps(1e-27)\n1.793662034335766e-43\n\njulia> eps(0.0)\n5.0e-324\n\nThe distance between two adjacent representable floating-point numbers is not constant, but is smaller for smaller values and larger for larger values. In other words, the representable floating-point numbers are densest in the real number line near zero, and grow sparser exponentially as one moves farther away from zero. By definition, eps(1.0) is the same as eps(Float64) since 1.0 is a 64-bit floating-point value.\n\nJulia also provides the nextfloat and prevfloat functions which return the next largest or smallest representable floating-point number to the argument respectively:\n\njulia> x = 1.25f0\n1.25f0\n\njulia> nextfloat(x)\n1.2500001f0\n\njulia> prevfloat(x)\n1.2499999f0\n\njulia> bitstring(prevfloat(x))\n\"00111111100111111111111111111111\"\n\njulia> bitstring(x)\n\"00111111101000000000000000000000\"\n\njulia> bitstring(nextfloat(x))\n\"00111111101000000000000000000001\"\n\nThis example highlights the general principle that the adjacent representable floating-point numbers also have adjacent binary integer representations."},{"location":"manual/integers-and-floating-point-numbers.html#Rounding-modes","page":"Integers and Floating-Point Numbers","title":"Rounding modes","category":"section","text":"If a number doesn't have an exact floating-point representation, it must be rounded to an appropriate representable value. However, the manner in which this rounding is done can be changed if required according to the rounding modes presented in the IEEE 754 standard.\n\nThe default mode used is always RoundNearest, which rounds to the nearest representable value, with ties rounded towards the nearest value with an even least significant bit."},{"location":"manual/integers-and-floating-point-numbers.html#Background-and-References","page":"Integers and Floating-Point Numbers","title":"Background and References","category":"section","text":"Floating-point arithmetic entails many subtleties which can be surprising to users who are unfamiliar with the low-level implementation details. However, these subtleties are described in detail in most books on scientific computation, and also in the following references:\n\nThe definitive guide to floating point arithmetic is the IEEE 754-2008 Standard; however, it is not available for free online.\nFor a brief but lucid presentation of how floating-point numbers are represented, see John D. Cook's article on the subject as well as his introduction to some of the issues arising from how this representation differs in behavior from the idealized abstraction of real numbers.\nAlso recommended is Bruce Dawson's series of blog posts on floating-point numbers.\nFor an excellent, in-depth discussion of floating-point numbers and issues of numerical accuracy encountered when computing with them, see David Goldberg's paper What Every Computer Scientist Should Know About Floating-Point Arithmetic.\nFor even more extensive documentation of the history of, rationale for, and issues with floating-point numbers, as well as discussion of many other topics in numerical computing, see the collected writings of William Kahan, commonly known as the \"Father of Floating-Point\". Of particular interest may be An Interview with the Old Man of Floating-Point."},{"location":"manual/integers-and-floating-point-numbers.html#Arbitrary-Precision-Arithmetic","page":"Integers and Floating-Point Numbers","title":"Arbitrary Precision Arithmetic","category":"section","text":"To allow computations with arbitrary-precision integers and floating point numbers, Julia wraps the GNU Multiple Precision Arithmetic Library (GMP) and the GNU MPFR Library, respectively. The BigInt and BigFloat types are available in Julia for arbitrary precision integer and floating point numbers respectively.\n\nConstructors exist to create these types from primitive numerical types, and the string literal @big_str or parse can be used to construct them from AbstractStrings. BigInts can also be input as integer literals when they are too big for other built-in integer types. Note that as there is no unsigned arbitrary-precision integer type in Base (BigInt is sufficient in most cases), hexadecimal, octal and binary literals can be used (in addition to decimal literals).\n\nOnce created, they participate in arithmetic with all other numeric types thanks to Julia's type promotion and conversion mechanism:\n\njulia> BigInt(typemax(Int64)) + 1\n9223372036854775808\n\njulia> big\"123456789012345678901234567890\" + 1\n123456789012345678901234567891\n\njulia> parse(BigInt, \"123456789012345678901234567890\") + 1\n123456789012345678901234567891\n\njulia> string(big\"2\"^200, base=16)\n\"100000000000000000000000000000000000000000000000000\"\n\njulia> 0x100000000000000000000000000000000-1 == typemax(UInt128)\ntrue\n\njulia> 0x000000000000000000000000000000000\n0\n\njulia> typeof(ans)\nBigInt\n\njulia> big\"1.23456789012345678901\"\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> parse(BigFloat, \"1.23456789012345678901\")\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> BigFloat(2.0^66) / 3\n2.459565876494606882133333333333333333333333333333333333333333333333333333333344e19\n\njulia> factorial(BigInt(40))\n815915283247897734345611269596115894272000000000\n\nHowever, type promotion between the primitive types above and BigInt/BigFloat is not automatic and must be explicitly stated.\n\njulia> x = typemin(Int64)\n-9223372036854775808\n\njulia> x = x - 1\n9223372036854775807\n\njulia> typeof(x)\nInt64\n\njulia> y = BigInt(typemin(Int64))\n-9223372036854775808\n\njulia> y = y - 1\n-9223372036854775809\n\njulia> typeof(y)\nBigInt\n\nThe default precision (in number of bits of the significand) and rounding mode of BigFloat operations can be changed globally by calling setprecision and setrounding, and all further calculations will take these changes in account. Alternatively, the precision or the rounding can be changed only within the execution of a particular block of code by using the same functions with a do block:\n\njulia> setrounding(BigFloat, RoundUp) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.100000000000000000000000000000000000000000000000000000000000000000000000000003\n\njulia> setrounding(BigFloat, RoundDown) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.099999999999999999999999999999999999999999999999999999999999999999999999999986\n\njulia> setprecision(40) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.1000000000004\n\nwarning: Warning\nThe relation between setprecision or setrounding and @big_str, the macro used for big string literals (such as big\"0.3\"), might not be intuitive, as a consequence 