var documenterSearchIndex = {"docs":
[{"location":"index.html","page":"Julia Documentation","title":"Julia Documentation","category":"section","text":"io = IOBuffer()\nrelease = isempty(VERSION.prerelease)\nv = \"$(VERSION.major).$(VERSION.minor)\"\n!release && (v = v*\"-$(first(VERSION.prerelease))\")\nprint(io, \"\"\"\n    # Julia $(v) Documentation\n\n    Welcome to the documentation for Julia $(v).\n\n    \"\"\")\nif !release\n    print(io,\"\"\"\n        !!! warning \"Work in progress!\"\n            This documentation is for an unreleased, in-development, version of Julia.\n        \"\"\")\nend\nimport Markdown\nMarkdown.parse(String(take!(io)))\n\nPlease read the release notes to see what has changed since the last release.\n\nrelease = isempty(VERSION.prerelease)\nfile = release ? \"julia-$(VERSION).pdf\" :\n       \"julia-$(VERSION.major).$(VERSION.minor).$(VERSION.patch)-$(first(VERSION.prerelease)).pdf\"\nurl = \"https://raw.githubusercontent.com/JuliaLang/docs.julialang.org/assets/$(file)\"\nimport Markdown\nMarkdown.parse(\"\"\"\n!!! note\n    The documentation is also available in PDF format: [$file]($url).\n\"\"\")"},{"location":"index.html#man-introduction","page":"Julia Documentation","title":"Introduction","category":"section","text":"Scientific computing has traditionally required the highest performance, yet domain experts have largely moved to slower dynamic languages for daily work. We believe there are many good reasons to prefer dynamic languages for these applications, and we do not expect their use to diminish. Fortunately, modern language design and compiler techniques make it possible to mostly eliminate the performance trade-off and provide a single environment productive enough for prototyping and efficient enough for deploying performance-intensive applications. The Julia programming language fills this role: it is a flexible dynamic language, appropriate for scientific and numerical computing, with performance comparable to traditional statically-typed languages.\n\nBecause Julia's compiler is different from the interpreters used for languages like Python or R, you may find that Julia's performance is unintuitive at first. If you find that something is slow, we highly recommend reading through the Performance Tips section before trying anything else. Once you understand how Julia works, it is easy to write code that is nearly as fast as C."},{"location":"index.html#man-julia-compared-other-languages","page":"Julia Documentation","title":"Julia Compared to Other Languages","category":"section","text":"Julia features optional typing, multiple dispatch, and good performance, achieved using type inference and just-in-time (JIT) compilation (and optional ahead-of-time compilation), implemented using LLVM. It is multi-paradigm, combining features of imperative, functional, and object-oriented programming. Julia provides ease and expressiveness for high-level numerical computing, in the same way as languages such as R, MATLAB, and Python, but also supports general programming. To achieve this, Julia builds upon the lineage of mathematical programming languages, but also borrows much from popular dynamic languages, including Lisp, Perl, Python, Lua, and Ruby.\n\nThe most significant departures of Julia from typical dynamic languages are:\n\nThe core language imposes very little; Julia Base and the standard library are written in Julia itself, including primitive operations like integer arithmetic\nA rich language of types for constructing and describing objects, that can also optionally be used to make type declarations\nThe ability to define function behavior across many combinations of argument types via multiple dispatch\nAutomatic generation of efficient, specialized code for different argument types\nGood performance, approaching that of statically-compiled languages like C\n\nAlthough one sometimes speaks of dynamic languages as being \"typeless\", they are definitely not. Every object, whether primitive or user-defined, has a type. The lack of type declarations in most dynamic languages, however, means that one cannot instruct the compiler about the types of values, and often cannot explicitly talk about types at all. In static languages, on the other hand, while one can – and usually must – annotate types for the compiler, types exist only at compile time and cannot be manipulated or expressed at run time. In Julia, types are themselves run-time objects, and can also be used to convey information to the compiler."},{"location":"index.html#man-what-makes-julia","page":"Julia Documentation","title":"What Makes Julia, Julia?","category":"section","text":"While the casual programmer need not explicitly use types or multiple dispatch, they are the core unifying features of Julia: functions are defined on different combinations of argument types, and applied by dispatching to the most specific matching definition. This model is a good fit for mathematical programming, where it is unnatural for the first argument to \"own\" an operation as in traditional object-oriented dispatch. Operators are just functions with special notation – to extend addition to new user-defined data types, you define new methods for the + function. Existing code then seamlessly applies to the new data types.\n\nPartly because of run-time type inference (augmented by optional type annotations), and partly because of a strong focus on performance from the inception of the project, Julia's computational efficiency exceeds that of other dynamic languages, and even rivals that of statically-compiled languages. For large scale numerical problems, speed always has been, continues to be, and probably always will be crucial: the amount of data being processed has easily kept pace with Moore's Law over the past decades."},{"location":"index.html#man-advantages-of-julia","page":"Julia Documentation","title":"Advantages of Julia","category":"section","text":"Julia aims to create an unprecedented combination of ease-of-use, power, and efficiency in a single language. In addition to the above, some advantages of Julia over comparable systems include:\n\nFree and open source (MIT licensed)\nUser-defined types are as fast and compact as built-ins\nNo need to vectorize code for performance; devectorized code is fast\nDesigned for parallelism and distributed computation\nLightweight \"green\" threading (coroutines)\nUnobtrusive yet powerful type system\nElegant and extensible conversions and promotions for numeric and other types\nEfficient support for Unicode, including but not limited to UTF-8\nCall C functions directly (no wrappers or special APIs needed)\nPowerful shell-like capabilities for managing other processes\nLisp-like macros and other metaprogramming facilities"},{"location":"index.html#man-important-links","page":"Julia Documentation","title":"Important Links","category":"section","text":"A non-exhaustive list of links that will be useful as you learn and use the Julia programming language:\n\nJulia Homepage\nInstall Julia\nDiscussion forum\nJulia YouTube\nFind Julia Packages\nLearning Resources"},{"location":"devdocs/offset-arrays.html#man-custom-indices","page":"Arrays with custom indices","title":"Arrays with custom indices","category":"section","text":"Conventionally, Julia's arrays are indexed starting at 1, whereas some other languages start numbering at 0, and yet others (e.g., Fortran) allow you to specify arbitrary starting indices. While there is much merit in picking a standard (i.e., 1 for Julia), there are some algorithms which simplify considerably if you can index outside the range 1:size(A,d) (and not just 0:size(A,d)-1, either). To facilitate such computations, Julia supports arrays with arbitrary indices.\n\nThe purpose of this page is to address the question, \"what do I have to do to support such arrays in my own code?\"  First, let's address the simplest case: if you know that your code will never need to handle arrays with unconventional indexing, hopefully the answer is \"nothing.\" Old code, on conventional arrays, should function essentially without alteration as long as it was using the exported interfaces of Julia. If you find it more convenient to just force your users to supply traditional arrays where indexing starts at one, you can add\n\nBase.require_one_based_indexing(arrays...)\n\nwhere arrays... is a list of the array objects that you wish to check for anything that violates 1-based indexing."},{"location":"devdocs/offset-arrays.html#Generalizing-existing-code","page":"Arrays with custom indices","title":"Generalizing existing code","category":"section","text":"As an overview, the steps are:\n\nreplace many uses of size with axes\nreplace 1:length(A) with eachindex(A), or in some cases LinearIndices(A)\nreplace explicit allocations like Array{Int}(undef, size(B)) with similar(Array{Int}, axes(B))\n\nThese are described in more detail below."},{"location":"devdocs/offset-arrays.html#Things-to-watch-out-for","page":"Arrays with custom indices","title":"Things to watch out for","category":"section","text":"Because unconventional indexing breaks many people's assumptions that all arrays start indexing with 1, there is always the chance that using such arrays will trigger errors. The most frustrating bugs would be incorrect results or segfaults (total crashes of Julia). For example, consider the following function:\n\nfunction mycopy!(dest::AbstractVector, src::AbstractVector)\n    length(dest) == length(src) || throw(DimensionMismatch(\"vectors must match\"))\n    # OK, now we're safe to use @inbounds, right? (not anymore!)\n    for i = 1:length(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend\n\nThis code implicitly assumes that vectors are indexed from 1; if dest starts at a different index than src, there is a chance that this code would trigger a segfault. (If you do get segfaults, to help locate the cause try running julia with the option --check-bounds=yes.)"},{"location":"devdocs/offset-arrays.html#Using-axes-for-bounds-checks-and-loop-iteration","page":"Arrays with custom indices","title":"Using axes for bounds checks and loop iteration","category":"section","text":"axes(A) (reminiscent of size(A)) returns a tuple of AbstractUnitRange{<:Integer} objects, specifying the range of valid indices along each dimension of A. When A has unconventional indexing, the ranges may not start at 1. If you just want the range for a particular dimension d, there is axes(A, d).\n\nBase implements a custom range type, OneTo, where OneTo(n) means the same thing as 1:n but in a form that guarantees (via the type system) that the lower index is 1. For any new AbstractArray type, this is the default returned by axes, and it indicates that this array type uses \"conventional\" 1-based indexing.\n\nFor bounds checking, note that there are dedicated functions checkbounds and checkindex which can sometimes simplify such tests."},{"location":"devdocs/offset-arrays.html#Linear-indexing-(LinearIndices)","page":"Arrays with custom indices","title":"Linear indexing (LinearIndices)","category":"section","text":"Some algorithms are most conveniently (or efficiently) written in terms of a single linear index, A[i] even if A is multi-dimensional. Regardless of the array's native indices, linear indices always range from 1:length(A). However, this raises an ambiguity for one-dimensional arrays (a.k.a., AbstractVector): does v[i] mean linear indexing , or Cartesian indexing with the array's native indices?\n\nFor this reason, your best option may be to iterate over the array with eachindex(A), or, if you require the indices to be sequential integers, to get the index range by calling LinearIndices(A). This will return axes(A, 1) if A is an AbstractVector, and the equivalent of 1:length(A) otherwise.\n\nBy this definition, 1-dimensional arrays always use Cartesian indexing with the array's native indices. To help enforce this, it's worth noting that the index conversion functions will throw an error if shape indicates a 1-dimensional array with unconventional indexing (i.e., is a Tuple{UnitRange} rather than a tuple of OneTo). For arrays with conventional indexing, these functions continue to work the same as always.\n\nUsing axes and LinearIndices, here is one way you could rewrite mycopy!:\n\nfunction mycopy!(dest::AbstractVector, src::AbstractVector)\n    axes(dest) == axes(src) || throw(DimensionMismatch(\"vectors must match\"))\n    for i in LinearIndices(src)\n        @inbounds dest[i] = src[i]\n    end\n    dest\nend"},{"location":"devdocs/offset-arrays.html#Allocating-storage-using-generalizations-of-similar","page":"Arrays with custom indices","title":"Allocating storage using generalizations of similar","category":"section","text":"Storage is often allocated with Array{Int}(undef, dims) or similar(A, args...). When the result needs to match the indices of some other array, this may not always suffice. The generic replacement for such patterns is to use similar(storagetype, shape).  storagetype indicates the kind of underlying \"conventional\" behavior you'd like, e.g., Array{Int} or BitArray or even dims->zeros(Float32, dims) (which would allocate an all-zeros array). shape is a tuple of Integer or AbstractUnitRange values, specifying the indices that you want the result to use. Note that a convenient way of producing an all-zeros array that matches the indices of A is simply zeros(A).\n\nLet's walk through a couple of explicit examples. First, if A has conventional indices, then similar(Array{Int}, axes(A)) would end up calling Array{Int}(undef, size(A)), and thus return an array. If A is an AbstractArray type with unconventional indexing, then similar(Array{Int}, axes(A)) should return something that \"behaves like\" an Array{Int} but with a shape (including indices) that matches A.  (The most obvious implementation is to allocate an Array{Int}(undef, size(A)) and then \"wrap\" it in a type that shifts the indices.)\n\nNote also that similar(Array{Int}, (axes(A, 2),)) would allocate an AbstractVector{Int} (i.e., 1-dimensional array) that matches the indices of the columns of A."},{"location":"devdocs/offset-arrays.html#Writing-custom-array-types-with-non-1-indexing","page":"Arrays with custom indices","title":"Writing custom array types with non-1 indexing","category":"section","text":"Most of the methods you'll need to define are standard for any AbstractArray type, see Abstract Arrays. This page focuses on the steps needed to define unconventional indexing."},{"location":"devdocs/offset-arrays.html#Custom-AbstractUnitRange-types","page":"Arrays with custom indices","title":"Custom AbstractUnitRange types","category":"section","text":"If you're writing a non-1 indexed array type, you will want to specialize axes so it returns a UnitRange, or (perhaps better) a custom AbstractUnitRange. The advantage of a custom type is that it \"signals\" the allocation type for functions like similar. If we're writing an array type for which indexing will start at 0, we likely want to begin by creating a new AbstractUnitRange, ZeroRange, where ZeroRange(n) is equivalent to 0:n-1.\n\nIn general, you should probably not export ZeroRange from your package: there may be other packages that implement their own ZeroRange, and having multiple distinct ZeroRange types is (perhaps counterintuitively) an advantage: ModuleA.ZeroRange indicates that similar should create a ModuleA.ZeroArray, whereas ModuleB.ZeroRange indicates a ModuleB.ZeroArray type.  This design allows peaceful coexistence among many different custom array types.\n\nNote that the Julia package CustomUnitRanges.jl can sometimes be used to avoid the need to write your own ZeroRange type."},{"location":"devdocs/offset-arrays.html#Specializing-axes","page":"Arrays with custom indices","title":"Specializing axes","category":"section","text":"Once you have your AbstractUnitRange type, then specialize axes:\n\nBase.axes(A::ZeroArray) = map(n->ZeroRange(n), A.size)\n\nwhere here we imagine that ZeroArray has a field called size (there would be other ways to implement this).\n\nIn some cases, the fallback definition for axes(A, d):\n\naxes(A::AbstractArray{T,N}, d) where {T,N} = d <= N ? axes(A)[d] : OneTo(1)\n\nmay not be what you want: you may need to specialize it to return something other than OneTo(1) when d > ndims(A). Likewise, in Base there is a dedicated function axes1 which is equivalent to axes(A, 1) but which avoids checking (at runtime) whether ndims(A) > 0. (This is purely a performance optimization.)  It is defined as:\n\naxes1(A::AbstractArray{T,0}) where {T} = OneTo(1)\naxes1(A::AbstractArray) = axes(A)[1]\n\nIf the first of these (the zero-dimensional case) is problematic for your custom array type, be sure to specialize it appropriately."},{"location":"devdocs/offset-arrays.html#Specializing-similar","page":"Arrays with custom indices","title":"Specializing similar","category":"section","text":"Given your custom ZeroRange type, then you should also add the following two specializations for similar:\n\nfunction Base.similar(A::AbstractArray, T::Type, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nfunction Base.similar(f::Union{Function,DataType}, shape::Tuple{ZeroRange,Vararg{ZeroRange}})\n    # body\nend\n\nBoth of these should allocate your custom array type."},{"location":"devdocs/offset-arrays.html#Specializing-reshape","page":"Arrays with custom indices","title":"Specializing reshape","category":"section","text":"Optionally, define a method\n\nBase.reshape(A::AbstractArray, shape::Tuple{ZeroRange,Vararg{ZeroRange}}) = ...\n\nand you can reshape an array so that the result has custom indices."},{"location":"devdocs/offset-arrays.html#For-objects-that-mimic-AbstractArray-but-are-not-subtypes","page":"Arrays with custom indices","title":"For objects that mimic AbstractArray but are not subtypes","category":"section","text":"has_offset_axes depends on having axes defined for the objects you call it on. If there is some reason you don't have an axes method defined for your object, consider defining a method\n\nBase.has_offset_axes(obj::MyNon1IndexedArraylikeObject) = true\n\nThis will allow code that assumes 1-based indexing to detect a problem and throw a helpful error, rather than returning incorrect results or segfaulting julia."},{"location":"devdocs/offset-arrays.html#Catching-errors","page":"Arrays with custom indices","title":"Catching errors","category":"section","text":"If your new array type triggers errors in other code, one helpful debugging step can be to comment out @boundscheck in your getindex and setindex! implementation. This will ensure that every element access checks bounds. Or, restart julia with --check-bounds=yes.\n\nIn some cases it may also be helpful to temporarily disable size and length for your new array type, since code that makes incorrect assumptions frequently uses these functions."},{"location":"base/constants.html#lib-constants","page":"Constants","title":"Constants","category":"section","text":"See also:\n\nstdin\nstdout\nstderr\nENV\nENDIAN_BOM"},{"location":"base/constants.html#Core.nothing","page":"Constants","title":"Core.nothing","category":"constant","text":"nothing\n\nThe singleton instance of type Nothing, used by convention when there is no value to return (as in a C void function) or when a variable or field holds no value.\n\nA return value of nothing is not displayed by the REPL and similar interactive environments.\n\nSee also: isnothing, something, missing.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.PROGRAM_FILE","page":"Constants","title":"Base.PROGRAM_FILE","category":"constant","text":"PROGRAM_FILE\n\nA string containing the script name passed to Julia from the command line. Note that the script name remains unchanged from within included files. Alternatively see @__FILE__.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.ARGS","page":"Constants","title":"Base.ARGS","category":"constant","text":"ARGS\n\nAn array of the command line arguments passed to Julia, as strings.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.C_NULL","page":"Constants","title":"Base.C_NULL","category":"constant","text":"C_NULL\n\nThe C null pointer constant, sometimes used when calling external code.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.VERSION","page":"Constants","title":"Base.VERSION","category":"constant","text":"VERSION\n\nA VersionNumber object describing which version of Julia is in use. See also Version Number Literals.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.DEPOT_PATH","page":"Constants","title":"Base.DEPOT_PATH","category":"constant","text":"DEPOT_PATH\n\nA stack of \"depot\" locations where the package manager, as well as Julia's code loading mechanisms, look for package registries, installed packages, named environments, repo clones, cached compiled package images, and configuration files. By default it includes:\n\n~/.julia where ~ is the user home as appropriate on the system;\nan architecture-specific shared system directory, e.g. /usr/local/share/julia;\nan architecture-independent shared system directory, e.g. /usr/share/julia.\n\nSo DEPOT_PATH might be:\n\n[joinpath(homedir(), \".julia\"), \"/usr/local/share/julia\", \"/usr/share/julia\"]\n\nThe first entry is the \"user depot\" and should be writable by and owned by the current user. The user depot is where: registries are cloned, new package versions are installed, named environments are created and updated, package repos are cloned, newly compiled package image files are saved, log files are written, development packages are checked out by default, and global configuration data is saved. Later entries in the depot path are treated as read-only and are appropriate for registries, packages, etc. installed and managed by system administrators.\n\nDEPOT_PATH is populated based on the JULIA_DEPOT_PATH environment variable if set.\n\nDEPOT_PATH contents\n\nEach entry in DEPOT_PATH is a path to a directory which contains subdirectories used by Julia for various purposes. Here is an overview of some of the subdirectories that may exist in a depot:\n\nartifacts: Contains content that packages use for which Pkg manages the installation of.\nclones: Contains full clones of package repos. Maintained by Pkg.jl and used as a cache.\nconfig: Contains julia-level configuration such as a startup.jl.\ncompiled: Contains precompiled *.ji files for packages. Maintained by Julia.\ndev: Default directory for Pkg.develop. Maintained by Pkg.jl and the user.\nenvironments: Default package environments. For instance the global environment for a specific julia version. Maintained by Pkg.jl.\nlogs: Contains logs of Pkg and REPL operations. Maintained by Pkg.jl and Julia.\npackages: Contains packages, some of which were explicitly installed and some which are implicit dependencies. Maintained by Pkg.jl.\nregistries: Contains package registries. By default only General. Maintained by Pkg.jl.\nscratchspaces: Contains content that a package itself installs via the Scratch.jl package. Pkg.gc() will delete content that is known to be unused.\n\nnote: Note\nPackages that want to store content should use the scratchspaces subdirectory via Scratch.jl instead of creating new subdirectories in the depot root.\n\nSee also JULIA_DEPOT_PATH, and Code Loading.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.LOAD_PATH","page":"Constants","title":"Base.LOAD_PATH","category":"constant","text":"LOAD_PATH\n\nAn array of paths for using and import statements to consider as project environments or package directories when loading code. It is populated based on the JULIA_LOAD_PATH environment variable if set; otherwise it defaults to [\"@\", \"@v#.#\", \"@stdlib\"]. Entries starting with @ have special meanings:\n\n@ refers to the \"current active environment\", the initial value of which is initially determined by the JULIA_PROJECT environment variable or the --project command-line option.\n@stdlib expands to the absolute path of the current Julia installation's standard library directory.\n@name refers to a named environment, which are stored in depots (see JULIA_DEPOT_PATH) under the environments subdirectory. The user's named environments are stored in ~/.julia/environments so @name would refer to the environment in ~/.julia/environments/name if it exists and contains a Project.toml file. If name contains # characters, then they are replaced with the major, minor and patch components of the Julia version number. For example, if you are running Julia 1.2 then @v#.# expands to @v1.2 and will look for an environment by that name, typically at ~/.julia/environments/v1.2.\n\nThe fully expanded value of LOAD_PATH that is searched for projects and packages can be seen by calling the Base.load_path() function.\n\nSee also JULIA_LOAD_PATH, JULIA_PROJECT, JULIA_DEPOT_PATH, and Code Loading.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.BINDIR","page":"Constants","title":"Base.Sys.BINDIR","category":"constant","text":"Sys.BINDIR::String\n\nA string containing the full path to the directory containing the julia executable.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.CPU_THREADS","page":"Constants","title":"Base.Sys.CPU_THREADS","category":"constant","text":"Sys.CPU_THREADS::Int\n\nThe number of logical CPU cores available in the system, i.e. the number of threads that the CPU can run concurrently. Note that this is not necessarily the number of CPU cores, for example, in the presence of hyper-threading.\n\nSee Hwloc.jl or CpuId.jl for extended information, including number of physical cores.\n\nSee also: Sys.EFFECTIVE_CPU_THREADS for a container-aware CPU count that respects cgroup limits.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.EFFECTIVE_CPU_THREADS","page":"Constants","title":"Base.Sys.EFFECTIVE_CPU_THREADS","category":"constant","text":"Sys.EFFECTIVE_CPU_THREADS::Int\n\nThe effective number of logical CPU cores available to the Julia process, taking into account container limits (e.g., Docker --cpus, Kubernetes CPU limits, cgroup quotas). This is the minimum of the hardware CPU thread count and any imposed CPU limits.\n\nIn non-containerized environments, this typically equals Sys.CPU_THREADS. In containerized environments, it respects cgroup CPU limits and provides a more accurate measure of available parallelism.\n\nUse this constant when determining default thread pool sizes or parallelism levels to ensure proper behavior in containerized deployments.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.WORD_SIZE","page":"Constants","title":"Base.Sys.WORD_SIZE","category":"constant","text":"Sys.WORD_SIZE::Int\n\nStandard word size on the current machine, in bits.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.KERNEL","page":"Constants","title":"Base.Sys.KERNEL","category":"constant","text":"Sys.KERNEL::Symbol\n\nA symbol representing the name of the operating system, as returned by uname of the build configuration.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.ARCH","page":"Constants","title":"Base.Sys.ARCH","category":"constant","text":"Sys.ARCH::Symbol\n\nA symbol representing the architecture of the build configuration.\n\n\n\n\n\n"},{"location":"base/constants.html#Base.Sys.MACHINE","page":"Constants","title":"Base.Sys.MACHINE","category":"constant","text":"Sys.MACHINE::String\n\nA string containing the build triple.\n\n\n\n\n\n"},{"location":"devdocs/backtraces.html#Reporting-and-analyzing-crashes-(segfaults)","page":"Reporting and analyzing crashes (segfaults)","title":"Reporting and analyzing crashes (segfaults)","category":"section","text":"So you managed to break Julia. Congratulations!  Collected here are some general procedures you can undergo for common symptoms encountered when something goes awry. Including the information from these debugging steps can greatly help the maintainers when tracking down a segfault or trying to figure out why your script is running slower than expected.\n\nIf you've been directed to this page, find the symptom that best matches what you're experiencing and follow the instructions to generate the debugging information requested. Table of symptoms:\n\nSegfaults during bootstrap (sysimg.jl)\nSegfaults when running a script\nErrors during Julia startup\nOther generic segfaults or unreachables reached"},{"location":"devdocs/backtraces.html#dev-version-info","page":"Reporting and analyzing crashes (segfaults)","title":"Version/Environment info","category":"section","text":"No matter the error, we will always need to know what version of Julia you are running. When Julia first starts up, a header is printed out with a version number and date. Please also include the output of versioninfo() (exported from the InteractiveUtils standard library) in any report you create:\n\nusing InteractiveUtils\nversioninfo()"},{"location":"devdocs/backtraces.html#Segfaults-during-bootstrap-(sysimg.jl)","page":"Reporting and analyzing crashes (segfaults)","title":"Segfaults during bootstrap (sysimg.jl)","category":"section","text":"Segfaults toward the end of the make process of building Julia are a common symptom of something going wrong while Julia is preparsing the corpus of code in the base/ folder. Many factors can contribute toward this process dying unexpectedly, however it is as often as not due to an error in the C-code portion of Julia, and as such must typically be debugged with a debug build inside of gdb. Explicitly:\n\nCreate a debug build of Julia:\n\n$ cd <julia_root>\n$ make debug\n\nNote that this process will likely fail with the same error as a normal make incantation, however this will create a debug executable that will offer gdb the debugging symbols needed to get accurate backtraces. Next, manually run the bootstrap process inside of gdb:\n\n$ cd base/\n$ gdb -x ../contrib/debug_bootstrap.gdb\n\nThis will start gdb, attempt to run the bootstrap process using the debug build of Julia, and print out a backtrace if (when) it segfaults. You may need to hit <enter> a few times to get the full backtrace. Create a gist with the backtrace, the version info, and any other pertinent information you can think of and open a new issue on Github with a link to the gist."},{"location":"devdocs/backtraces.html#Segfaults-when-running-a-script","page":"Reporting and analyzing crashes (segfaults)","title":"Segfaults when running a script","category":"section","text":"The procedure is very similar to Segfaults during bootstrap (sysimg.jl). Create a debug build of Julia, and run your script inside of a debugged Julia process:\n\n$ cd <julia_root>\n$ make debug\n$ gdb --args usr/bin/julia-debug <path_to_your_script>\n\nNote that gdb will sit there, waiting for instructions. Type r to run the process, and bt to generate a backtrace once it segfaults:\n\n(gdb) r\nStarting program: /home/sabae/src/julia/usr/bin/julia-debug ./test.jl\n...\n(gdb) bt\n\nCreate a gist with the backtrace, the version info, and any other pertinent information you can think of and open a new issue on Github with a link to the gist."},{"location":"devdocs/backtraces.html#Errors-during-Julia-startup","page":"Reporting and analyzing crashes (segfaults)","title":"Errors during Julia startup","category":"section","text":"Occasionally errors occur during Julia's startup process (especially when using binary distributions, as opposed to compiling from source) such as the following:\n\n$ julia\nexec: error -5\n\nThese errors typically indicate something is not getting loaded properly very early on in the bootup phase, and our best bet in determining what's going wrong is to use external tools to audit the disk activity of the julia process:\n\nOn Linux, use strace:\n$ strace julia\nOn OSX, use dtruss:\n$ dtruss -f julia\n\nCreate a gist with the strace/ dtruss output, the version info, and any other pertinent information and open a new issue on Github with a link to the gist."},{"location":"devdocs/backtraces.html#Other-generic-segfaults-or-unreachables-reached","page":"Reporting and analyzing crashes (segfaults)","title":"Other generic segfaults or unreachables reached","category":"section","text":"As mentioned elsewhere, julia has good integration with rr for generating traces; this includes, on Linux, the ability to automatically run julia under rr and share the trace after a crash. This can be immensely helpful when debugging such crashes and is strongly encouraged when reporting crash issues to the JuliaLang/julia repo. To run julia under rr automatically, do:\n\njulia --bug-report=rr\n\nTo generate the rr trace locally, but not share, you can do:\n\njulia --bug-report=rr-local\n\nNote that this is only works on Linux. The blog post on Time Travelling Bug Reporting has many more details."},{"location":"devdocs/backtraces.html#Glossary","page":"Reporting and analyzing crashes (segfaults)","title":"Glossary","category":"section","text":"A few terms have been used as shorthand in this guide:\n\n<julia_root> refers to the root directory of the Julia source tree; e.g. it should contain folders such as base, deps, src, test, etc."},{"location":"base/collections.html#Collections-and-Data-Structures","page":"Collections and Data Structures","title":"Collections and Data Structures","category":"section","text":""},{"location":"base/collections.html#lib-collections-iteration","page":"Collections and Data Structures","title":"Iteration","category":"section","text":"Sequential iteration is implemented by the iterate function. The general for loop:\n\nfor i in iter   # or  \"for i = iter\"\n    # body\nend\n\nis translated into:\n\nnext = iterate(iter)\nwhile next !== nothing\n    (i, state) = next\n    # body\n    next = iterate(iter, state)\nend\n\nThe state object may be anything, and should be chosen appropriately for each iterable type. See the manual section on the iteration interface for more details about defining a custom iterable type.\n\nFully implemented by:\n\nAbstractRange\nUnitRange\nTuple\nNumber\nAbstractArray\nBitSet\nIdDict\nDict\nWeakKeyDict\nEachLine\nAbstractString\nSet\nPair\nNamedTuple"},{"location":"base/collections.html#Constructors-and-Types","page":"Collections and Data Structures","title":"Constructors and Types","category":"section","text":""},{"location":"base/collections.html#General-Collections","page":"Collections and Data Structures","title":"General Collections","category":"section","text":"Fully implemented by:\n\nAbstractRange\nUnitRange\nTuple\nNumber\nAbstractArray\nBitSet\nIdDict\nDict\nWeakKeyDict\nAbstractString\nSet\nNamedTuple"},{"location":"base/collections.html#Iterable-Collections","page":"Collections and Data Structures","title":"Iterable Collections","category":"section","text":""},{"location":"base/collections.html#Indexable-Collections","page":"Collections and Data Structures","title":"Indexable Collections","category":"section","text":"Fully implemented by:\n\nArray\nBitArray\nAbstractArray\nSubArray\n\nPartially implemented by:\n\nAbstractRange\nUnitRange\nTuple\nAbstractString\nDict\nIdDict\nWeakKeyDict\nNamedTuple"},{"location":"base/collections.html#Dictionaries","page":"Collections and Data Structures","title":"Dictionaries","category":"section","text":"Dict is the standard dictionary. Its implementation uses hash as the hashing function for the key, and isequal to determine equality. Define these two functions for custom types to override how they are stored in a hash table.\n\nIdDict is a special hash table where the keys are always object identities.\n\nWeakKeyDict is a hash table implementation where the keys are weak references to objects, and thus may be garbage collected even when referenced in a hash table. Like Dict it uses hash for hashing and isequal for equality, unlike Dict it does not convert keys on insertion.\n\nDicts can be created by passing pair objects constructed with => to a Dict constructor: Dict(\"A\"=>1, \"B\"=>2). This call will attempt to infer type information from the keys and values (i.e. this example creates a Dict{String, Int64}). To explicitly specify types use the syntax Dict{KeyType,ValueType}(...). For example, Dict{String,Int32}(\"A\"=>1, \"B\"=>2).\n\nDictionaries may also be created with generators. For example, Dict(i => f(i) for i = 1:10).\n\nGiven a dictionary D, the syntax D[x] returns the value of key x (if it exists) or throws an error, and D[x] = y stores the key-value pair x => y in D (replacing any existing value for the key x). Multiple arguments to D[...] are converted to tuples; for example, the syntax D[x,y]  is equivalent to D[(x,y)], i.e. it refers to the value keyed by the tuple (x,y).\n\nFully implemented by:\n\nDict\nIdDict\nWeakKeyDict\n\nPartially implemented by:\n\nSet\nBitSet\nIdSet\nEnvDict\nArray\nBitArray\nImmutableDict\nPersistentDict\nIterators.Pairs"},{"location":"base/collections.html#Set-Like-Collections","page":"Collections and Data Structures","title":"Set-Like Collections","category":"section","text":"Fully implemented by:\n\nSet\nBitSet\nIdSet\n\nPartially implemented by:\n\nArray"},{"location":"base/collections.html#Deques","page":"Collections and Data Structures","title":"Deques","category":"section","text":"Fully implemented by:\n\nVector (a.k.a. 1-dimensional Array)\nBitVector (a.k.a. 1-dimensional BitArray)"},{"location":"base/collections.html#Utility-Collections","page":"Collections and Data Structures","title":"Utility Collections","category":"section","text":""},{"location":"base/collections.html#Base.iterate","page":"Collections and Data Structures","title":"Base.iterate","category":"function","text":"iterate(iter [, state])::Union{Nothing, Tuple{Any, Any}}\n\nAdvance the iterator to obtain the next element. If no elements remain, nothing should be returned. Otherwise, a 2-tuple of the next element and the new iteration state should be returned.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.IteratorSize","page":"Collections and Data Structures","title":"Base.IteratorSize","category":"type","text":"IteratorSize(itertype::Type)::IteratorSize\n\nGiven the type of an iterator, return one of the following values:\n\nSizeUnknown() if the length (number of elements) cannot be determined in advance.\nHasLength() if there is a fixed, finite length.\nHasShape{N}() if there is a known length plus a notion of multidimensional shape (as for an array).  In this case N should give the number of dimensions, and the axes function is valid  for the iterator.\nIsInfinite() if the iterator yields values forever.\n\nThe default value (for iterators that do not define this function) is HasLength(). This means that most iterators are assumed to implement length.\n\nThis trait is generally used to select between algorithms that pre-allocate space for their result, and algorithms that resize their result incrementally.\n\njulia> Base.IteratorSize(1:5)\nBase.HasShape{1}()\n\njulia> Base.IteratorSize((2,3))\nBase.HasLength()\n\n\n\n\n\n"},{"location":"base/collections.html#Base.IteratorEltype","page":"Collections and Data Structures","title":"Base.IteratorEltype","category":"type","text":"IteratorEltype(itertype::Type)::IteratorEltype\n\nGiven the type of an iterator, return one of the following values:\n\nEltypeUnknown() if the type of elements yielded by the iterator is not known in advance.\nHasEltype() if the element type is known, and eltype would return a meaningful value.\n\nHasEltype() is the default, since iterators are assumed to implement eltype.\n\nThis trait is generally used to select between algorithms that pre-allocate a specific type of result, and algorithms that pick a result type based on the types of yielded values.\n\njulia> Base.IteratorEltype(1:5)\nBase.HasEltype()\n\n\n\n\n\n"},{"location":"base/collections.html#Base.AbstractRange","page":"Collections and Data Structures","title":"Base.AbstractRange","category":"type","text":"AbstractRange{T} <: AbstractVector{T}\n\nSupertype for linear ranges with elements of type T. UnitRange, LinRange and other types are subtypes of this.\n\nAll subtypes must define step. Thus LogRange is not a subtype of AbstractRange.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.OrdinalRange","page":"Collections and Data Structures","title":"Base.OrdinalRange","category":"type","text":"OrdinalRange{T, S} <: AbstractRange{T}\n\nSupertype for ordinal ranges with elements of type T with spacing(s) of type S. The steps should be always-exact multiples of oneunit, and T should be a \"discrete\" type, which cannot have values smaller than oneunit. For example, Integer or Date types would qualify, whereas Float64 would not (since this type can represent values smaller than oneunit(Float64). UnitRange, StepRange, and other types are subtypes of this.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.AbstractUnitRange","page":"Collections and Data Structures","title":"Base.AbstractUnitRange","category":"type","text":"AbstractUnitRange{T} <: OrdinalRange{T, T}\n\nSupertype for ranges with a step size of oneunit(T) with elements of type T. UnitRange and other types are subtypes of this.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.StepRange","page":"Collections and Data Structures","title":"Base.StepRange","category":"type","text":"StepRange{T, S} <: OrdinalRange{T, S}\n\nRanges with elements of type T with spacing of type S. The step between each element is constant, and the range is defined in terms of a start and stop of type T and a step of type S. Neither T nor S should be floating point types. The syntax a:b:c with b != 0 and a, b, and c all integers creates a StepRange.\n\nExamples\n\njulia> collect(StepRange(1, Int8(2), 10))\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\njulia> typeof(StepRange(1, Int8(2), 10))\nStepRange{Int64, Int8}\n\njulia> typeof(1:3:6)\nStepRange{Int64, Int64}\n\n\n\n\n\n"},{"location":"base/collections.html#Base.UnitRange","page":"Collections and Data Structures","title":"Base.UnitRange","category":"type","text":"UnitRange{T<:Real}\n\nA range parameterized by a start and stop of type T, filled with elements spaced by 1 from start until stop is exceeded. The syntax a:b with a and b both Integers creates a UnitRange.\n\nExamples\n\njulia> collect(UnitRange(2.3, 5.2))\n3-element Vector{Float64}:\n 2.3\n 3.3\n 4.3\n\njulia> typeof(1:10)\nUnitRange{Int64}\n\n\n\n\n\n"},{"location":"base/collections.html#Base.LinRange","page":"Collections and Data Structures","title":"Base.LinRange","category":"type","text":"LinRange{T,L}\n\nA range with len linearly spaced elements between its start and stop. The size of the spacing is controlled by len, which must be an Integer.\n\nExamples\n\njulia> LinRange(1.5, 5.5, 9)\n9-element LinRange{Float64, Int64}:\n 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0, 5.5\n\nCompared to using range, directly constructing a LinRange should have less overhead but won't try to correct for floating point errors:\n\njulia> collect(range(-0.1, 0.3, length=5))\n5-element Vector{Float64}:\n -0.1\n  0.0\n  0.1\n  0.2\n  0.3\n\njulia> collect(LinRange(-0.1, 0.3, 5))\n5-element Vector{Float64}:\n -0.1\n -1.3877787807814457e-17\n  0.09999999999999999\n  0.19999999999999998\n  0.3\n\nSee also Base.LogRange for logarithmically spaced points.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.isempty","page":"Collections and Data Structures","title":"Base.isempty","category":"function","text":"isempty(condition)\n\nReturn true if no tasks are waiting on the condition, false otherwise.\n\n\n\n\n\nisempty(collection)::Bool\n\nDetermine whether a collection is empty (has no elements).\n\nwarning: Warning\nisempty(itr) may consume the next element of a stateful iterator itr unless an appropriate Base.isdone(itr) method is defined. Stateful iterators should implement isdone, but you may want to avoid using isempty when writing generic code which should support any iterator type.\n\nExamples\n\njulia> isempty([])\ntrue\n\njulia> isempty([1 2 3])\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.isdone","page":"Collections and Data Structures","title":"Base.isdone","category":"function","text":"isdone(itr, [state])::Union{Bool, Missing}\n\nThis function provides a fast-path hint for iterator completion. This is useful for stateful iterators that want to avoid having elements consumed if they are not going to be exposed to the user (e.g. when checking for done-ness in isempty or zip).\n\nStateful iterators that want to opt into this feature should define an isdone method that returns true/false depending on whether the iterator is done or not. Stateless iterators need not implement this function.\n\nIf the result is missing, then isdone cannot determine whether the iterator state is terminal, and callers must compute iterate(itr, state) === nothing to obtain a definitive answer.\n\nSee also iterate, isempty\n\n\n\n\n\n"},{"location":"base/collections.html#Base.empty!","page":"Collections and Data Structures","title":"Base.empty!","category":"function","text":"empty!(c::Channel)\n\nEmpty a Channel c by calling empty! on the internal buffer. Return the empty channel.\n\n\n\n\n\nempty!(collection) -> collection\n\nRemove all elements from a collection.\n\nExamples\n\njulia> A = Dict(\"a\" => 1, \"b\" => 2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> empty!(A);\n\njulia> A\nDict{String, Int64}()\n\n\n\n\n\n"},{"location":"base/collections.html#Base.length","page":"Collections and Data Structures","title":"Base.length","category":"function","text":"length(collection)::Integer\n\nReturn the number of elements in the collection.\n\nUse lastindex to get the last valid index of an indexable collection.\n\nSee also: size, ndims, eachindex.\n\nExamples\n\njulia> length(1:5)\n5\n\njulia> length([1, 2, 3, 4])\n4\n\njulia> length([1 2; 3 4])\n4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.checked_length","page":"Collections and Data Structures","title":"Base.checked_length","category":"function","text":"Base.checked_length(r)\n\nCalculates length(r), but may check for overflow errors where applicable when the result doesn't fit into Union{Integer(eltype(r)),Int}.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.in","page":"Collections and Data Structures","title":"Base.in","category":"function","text":"in(item, collection)::Bool\n∈(item, collection)::Bool\n\nDetermine whether an item is in the given collection, in the sense that it is == to one of the values generated by iterating over the collection. Can equivalently be used with infix syntax:\n\nitem in collection\nitem ∈ collection\n\nReturn a Bool value, except if item is missing or collection contains missing but not item, in which case missing is returned (three-valued logic, matching the behavior of any and ==). Some collections follow a slightly different definition. For example, Sets check whether the item isequal to one of the elements; Dicts look for key=>value pairs, and the key is compared using isequal.\n\nTo test for the presence of a key in a dictionary, use haskey or k in keys(dict). For the collections mentioned above, the result is always a Bool.\n\nWhen broadcasting with in.(items, collection) or items .∈ collection, both items and collection are broadcasted over, which is often not what is intended. For example, if both arguments are vectors (and the dimensions match), the result is a vector indicating whether each value in collection items is in the value at the corresponding position in collection. To get a vector indicating whether each value in items is in collection, wrap collection in a tuple or a Ref like this: in.(items, Ref(collection)) or items .∈ Ref(collection).\n\nSee also: ∉, insorted, contains, occursin, issubset.\n\nExamples\n\njulia> a = 1:3:20\n1:3:19\n\njulia> 4 in a\ntrue\n\njulia> 5 in a\nfalse\n\njulia> missing in [1, 2]\nmissing\n\njulia> 1 in [2, missing]\nmissing\n\njulia> 1 in [1, missing]\ntrue\n\njulia> missing in Set([1, 2])\nfalse\n\njulia> (1=>missing) in Dict(1=>10, 2=>20)\nmissing\n\njulia> [1, 2] .∈ [2, 3]\n2-element BitVector:\n 0\n 0\n\njulia> [1, 2] .∈ ([2, 3],)\n2-element BitVector:\n 0\n 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.:∉","page":"Collections and Data Structures","title":"Base.:∉","category":"function","text":"∉(item, collection)::Bool\n∌(collection, item)::Bool\n\nNegation of ∈ and ∋, i.e. checks that item is not in collection.\n\nWhen broadcasting with items .∉ collection, both items and collection are broadcasted over, which is often not what is intended. For example, if both arguments are vectors (and the dimensions match), the result is a vector indicating whether each value in collection items is not in the value at the corresponding position in collection. To get a vector indicating whether each value in items is not in collection, wrap collection in a tuple or a Ref like this: items .∉ Ref(collection).\n\nExamples\n\njulia> 1 ∉ 2:4\ntrue\n\njulia> 1 ∉ 1:3\nfalse\n\njulia> [1, 2] .∉ [2, 3]\n2-element BitVector:\n 1\n 1\n\njulia> [1, 2] .∉ ([2, 3],)\n2-element BitVector:\n 1\n 0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.hasfastin","page":"Collections and Data Structures","title":"Base.hasfastin","category":"function","text":"Base.hasfastin(T)\n\nDetermine whether the computation x ∈ collection where collection::T can be considered as a \"fast\" operation (typically constant or logarithmic complexity). The definition hasfastin(x) = hasfastin(typeof(x)) is provided for convenience so that instances can be passed instead of types. However the form that accepts a type argument should be defined for new types.\n\nThe default for hasfastin(T) is true for subtypes of AbstractSet, AbstractDict and AbstractRange and false otherwise.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.eltype","page":"Collections and Data Structures","title":"Base.eltype","category":"function","text":"eltype(type)\n\nDetermine the type of the elements generated by iterating a collection of the given type. For dictionary types, this will be a Pair{KeyType,ValType}. The definition eltype(x) = eltype(typeof(x)) is provided for convenience so that instances can be passed instead of types. However the form that accepts a type argument should be defined for new types.\n\nSee also: keytype, typeof.\n\nExamples\n\njulia> eltype(fill(1f0, (2,2)))\nFloat32\n\njulia> eltype(fill(0x1, (2,2)))\nUInt8\n\n\n\n\n\n"},{"location":"base/collections.html#Base.indexin","page":"Collections and Data Structures","title":"Base.indexin","category":"function","text":"indexin(a, b)\n\nReturn an array containing the first index in b for each value in a that is a member of b. The output array contains nothing wherever a is not a member of b.\n\nSee also: sortperm, findfirst.\n\nExamples\n\njulia> a = ['a', 'b', 'c', 'b', 'd', 'a'];\n\njulia> b = ['a', 'b', 'c'];\n\njulia> indexin(a, b)\n6-element Vector{Union{Nothing, Int64}}:\n 1\n 2\n 3\n 2\n  nothing\n 1\n\njulia> indexin(b, a)\n3-element Vector{Union{Nothing, Int64}}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.unique","page":"Collections and Data Structures","title":"Base.unique","category":"function","text":"unique(A::AbstractArray; dims::Int)\n\nReturn unique regions of A along dimension dims.\n\nExamples\n\njulia> A = map(isodd, reshape(Vector(1:8), (2,2,2)))\n2×2×2 Array{Bool, 3}:\n[:, :, 1] =\n 1  1\n 0  0\n\n[:, :, 2] =\n 1  1\n 0  0\n\njulia> unique(A)\n2-element Vector{Bool}:\n 1\n 0\n\njulia> unique(A, dims=2)\n2×1×2 Array{Bool, 3}:\n[:, :, 1] =\n 1\n 0\n\n[:, :, 2] =\n 1\n 0\n\njulia> unique(A, dims=3)\n2×2×1 Array{Bool, 3}:\n[:, :, 1] =\n 1  1\n 0  0\n\n\n\n\n\nunique(f, itr)\n\nReturn an array containing one value from itr for each unique value produced by f applied to elements of itr.\n\nExamples\n\njulia> unique(x -> x^2, [1, -1, 3, -3, 4])\n3-element Vector{Int64}:\n 1\n 3\n 4\n\nThis functionality can also be used to extract the indices of the first occurrences of unique elements in an array:\n\njulia> a = [3.1, 4.2, 5.3, 3.1, 3.1, 3.1, 4.2, 1.7];\n\njulia> i = unique(i -> a[i], eachindex(a))\n4-element Vector{Int64}:\n 1\n 2\n 3\n 8\n\njulia> a[i]\n4-element Vector{Float64}:\n 3.1\n 4.2\n 5.3\n 1.7\n\njulia> a[i] == unique(a)\ntrue\n\n\n\n\n\nunique(itr)\n\nReturn an array containing only the unique elements of collection itr, as determined by isequal and hash, in the order that the first of each set of equivalent elements originally appears. The element type of the input is preserved.\n\nSee also: unique!, allunique, allequal.\n\nExamples\n\njulia> unique([1, 2, 6, 2])\n3-element Vector{Int64}:\n 1\n 2\n 6\n\njulia> unique(Real[1, 1.0, 2])\n2-element Vector{Real}:\n 1\n 2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.unique!","page":"Collections and Data Structures","title":"Base.unique!","category":"function","text":"unique!(A::AbstractVector)\n\nRemove duplicate items as determined by isequal and hash, then return the modified A. unique! will return the elements of A in the order that they occur. If you do not care about the order of the returned data, then calling (sort!(A); unique!(A)) will be much more efficient as long as the elements of A can be sorted.\n\nExamples\n\njulia> unique!([1, 1, 1])\n1-element Vector{Int64}:\n 1\n\njulia> A = [7, 3, 2, 3, 7, 5];\n\njulia> unique!(A)\n4-element Vector{Int64}:\n 7\n 3\n 2\n 5\n\njulia> B = [7, 6, 42, 6, 7, 42];\n\njulia> sort!(B);  # unique! is able to process sorted data much more efficiently.\n\njulia> unique!(B)\n3-element Vector{Int64}:\n  6\n  7\n 42\n\n\n\n\n\nunique!(f, A::AbstractVector)\n\nSelects one value from A for each unique value produced by f applied to elements of A, then return the modified A.\n\ncompat: Julia 1.1\nThis method is available as of Julia 1.1.\n\nExamples\n\njulia> unique!(x -> x^2, [1, -1, 3, -3, 4])\n3-element Vector{Int64}:\n 1\n 3\n 4\n\njulia> unique!(n -> n%3, [5, 1, 8, 9, 3, 4, 10, 7, 2, 6])\n3-element Vector{Int64}:\n 5\n 1\n 9\n\njulia> unique!(iseven, [2, 3, 5, 7, 9])\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.allunique","page":"Collections and Data Structures","title":"Base.allunique","category":"function","text":"allunique(itr)::Bool\nallunique(f, itr)::Bool\n\nReturn true if all values from itr are distinct when compared with isequal. Or if all of [f(x) for x in itr] are distinct, for the second method.\n\nNote that allunique(f, itr) may call f fewer than length(itr) times. The precise number of calls is regarded as an implementation detail.\n\nallunique may use a specialized implementation when the input is sorted.\n\nSee also: unique, issorted, allequal.\n\ncompat: Julia 1.11\nThe method allunique(f, itr) requires at least Julia 1.11.\n\nExamples\n\njulia> allunique([1, 2, 3])\ntrue\n\njulia> allunique([1, 2, 1, 2])\nfalse\n\njulia> allunique(Real[1, 1.0, 2])\nfalse\n\njulia> allunique([NaN, 2.0, NaN, 4.0])\nfalse\n\njulia> allunique(abs, [1, -1, 2])\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.allequal","page":"Collections and Data Structures","title":"Base.allequal","category":"function","text":"allequal(itr)::Bool\nallequal(f, itr)::Bool\n\nReturn true if all values from itr are equal when compared with isequal. Or if all of [f(x) for x in itr] are equal, for the second method.\n\nNote that allequal(f, itr) may call f fewer than length(itr) times. The precise number of calls is regarded as an implementation detail.\n\nSee also: unique, allunique.\n\ncompat: Julia 1.8\nThe allequal function requires at least Julia 1.8.\n\ncompat: Julia 1.11\nThe method allequal(f, itr) requires at least Julia 1.11.\n\nExamples\n\njulia> allequal([])\ntrue\n\njulia> allequal([1])\ntrue\n\njulia> allequal([1, 1])\ntrue\n\njulia> allequal([1, 2])\nfalse\n\njulia> allequal(Dict(:a => 1, :b => 1))\nfalse\n\njulia> allequal(abs2, [1, -1])\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.reduce-Tuple{Any, Any}","page":"Collections and Data Structures","title":"Base.reduce","category":"method","text":"reduce(op, itr; [init])\n\nReduce the given collection itr with the given binary operator op. If provided, the initial value init must be a neutral element for op that will be returned for empty collections. It is unspecified whether init is used for non-empty collections.\n\nFor empty collections, providing init will be necessary, except for some special cases (e.g. when op is one of +, *, max, min, &, |) when Julia can determine the neutral element of op.\n\nReductions for certain commonly-used operators may have special implementations, and should be used instead: maximum(itr), minimum(itr), sum(itr), prod(itr), any(itr), all(itr). There are efficient methods for concatenating certain arrays of arrays by calling reduce(vcat, arr) or reduce(hcat, arr).\n\nThe associativity of the reduction is implementation dependent. This means that you can't use non-associative operations like - because it is undefined whether reduce(-,[1,2,3]) should be evaluated as (1-2)-3 or 1-(2-3). Use foldl or foldr instead for guaranteed left or right associativity.\n\nSome operations accumulate error. Parallelism will be easier if the reduction can be executed in groups. Future versions of Julia might change the algorithm. Note that the elements are not reordered if you use an ordered collection.\n\nExamples\n\njulia> reduce(*, [2; 3; 4])\n24\n\njulia> reduce(*, Int[]; init=1)\n1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.reduce-Tuple{Any, AbstractArray}","page":"Collections and Data Structures","title":"Base.reduce","category":"method","text":"reduce(f, A::AbstractArray; dims=:, [init])\n\nReduce 2-argument function f along dimensions of A. dims is a vector specifying the dimensions to reduce, and the keyword argument init is the initial value to use in the reductions. For +, *, max and min the init argument is optional.\n\nThe associativity of the reduction is implementation-dependent; if you need a particular associativity, e.g. left-to-right, you should write your own loop or consider using foldl or foldr. See documentation for reduce.\n\nExamples\n\njulia> a = reshape(Vector(1:16), (4,4))\n4×4 Matrix{Int64}:\n 1  5   9  13\n 2  6  10  14\n 3  7  11  15\n 4  8  12  16\n\njulia> reduce(max, a, dims=2)\n4×1 Matrix{Int64}:\n 13\n 14\n 15\n 16\n\njulia> reduce(max, a, dims=1)\n1×4 Matrix{Int64}:\n 4  8  12  16\n\n\n\n\n\n"},{"location":"base/collections.html#Base.foldl-Tuple{Any, Any}","page":"Collections and Data Structures","title":"Base.foldl","category":"method","text":"foldl(op, itr; [init])\n\nLike reduce, but with guaranteed left associativity. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\nSee also mapfoldl, foldr, accumulate.\n\nExamples\n\njulia> foldl(=>, 1:4)\n((1 => 2) => 3) => 4\n\njulia> foldl(=>, 1:4; init=0)\n(((0 => 1) => 2) => 3) => 4\n\njulia> accumulate(=>, (1,2,3,4))\n(1, 1 => 2, (1 => 2) => 3, ((1 => 2) => 3) => 4)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.foldr-Tuple{Any, Any}","page":"Collections and Data Structures","title":"Base.foldr","category":"method","text":"foldr(op, itr; [init])\n\nLike reduce, but with guaranteed right associativity. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\nExamples\n\njulia> foldr(=>, 1:4)\n1 => (2 => (3 => 4))\n\njulia> foldr(=>, 1:4; init=0)\n1 => (2 => (3 => (4 => 0)))\n\n\n\n\n\n"},{"location":"base/collections.html#Base.maximum","page":"Collections and Data Structures","title":"Base.maximum","category":"function","text":"maximum(f, A::AbstractArray; dims)\n\nCompute the maximum value by calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 9  16\n\njulia> maximum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n  4\n 16\n\n\n\n\n\nmaximum(A::AbstractArray; dims)\n\nCompute the maximum value of an array over the given dimensions. See also the max(a,b) function to take the maximum of two or more arguments, which can be applied elementwise to arrays via max.(a,b).\n\nSee also: maximum!, extrema, findmax, argmax.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum(A, dims=1)\n1×2 Matrix{Int64}:\n 3  4\n\njulia> maximum(A, dims=2)\n2×1 Matrix{Int64}:\n 2\n 4\n\n\n\n\n\nmaximum(itr; [init])\n\nReturn the largest element in a collection.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for max (i.e. which is less than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> maximum(-20.5:10)\n9.5\n\njulia> maximum([1,2,3])\n3\n\njulia> maximum(())\nERROR: ArgumentError: reducing over an empty collection is not allowed; consider supplying `init` to the reducer\nStacktrace:\n[...]\n\njulia> maximum((); init=-Inf)\n-Inf\n\n\n\n\n\nmaximum(f, itr; [init])\n\nReturn the largest result of calling function f on each element of itr.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for max (i.e. which is less than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> maximum(length, [\"Julion\", \"Julia\", \"Jule\"])\n6\n\njulia> maximum(length, []; init=-1)\n-1\n\njulia> maximum(sin, Real[]; init=-1.0)  # good, since output of sin is >= -1\n-1.0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.maximum!","page":"Collections and Data Structures","title":"Base.maximum!","category":"function","text":"maximum!(r, A)\n\nCompute the maximum value of A over the singleton dimensions of r, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> maximum!([1; 1], A)\n2-element Vector{Int64}:\n 2\n 4\n\njulia> maximum!([1 1], A)\n1×2 Matrix{Int64}:\n 3  4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.minimum","page":"Collections and Data Structures","title":"Base.minimum","category":"function","text":"minimum(f, A::AbstractArray; dims)\n\nCompute the minimum value by calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 1  4\n\njulia> minimum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n 1\n 9\n\n\n\n\n\nminimum(A::AbstractArray; dims)\n\nCompute the minimum value of an array over the given dimensions. See also the min(a,b) function to take the minimum of two or more arguments, which can be applied elementwise to arrays via min.(a,b).\n\nSee also: minimum!, extrema, findmin, argmin.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum(A, dims=1)\n1×2 Matrix{Int64}:\n 1  2\n\njulia> minimum(A, dims=2)\n2×1 Matrix{Int64}:\n 1\n 3\n\n\n\n\n\nminimum(itr; [init])\n\nReturn the smallest element in a collection.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for min (i.e. which is greater than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> minimum(-20.5:10)\n-20.5\n\njulia> minimum([1,2,3])\n1\n\njulia> minimum([])\nERROR: ArgumentError: reducing over an empty collection is not allowed; consider supplying `init` to the reducer\nStacktrace:\n[...]\n\njulia> minimum([]; init=Inf)\nInf\n\n\n\n\n\nminimum(f, itr; [init])\n\nReturn the smallest result of calling function f on each element of itr.\n\nThe value returned for empty itr can be specified by init. It must be a neutral element for min (i.e. which is greater than or equal to any other element) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> minimum(length, [\"Julion\", \"Julia\", \"Jule\"])\n4\n\njulia> minimum(length, []; init=typemax(Int64))\n9223372036854775807\n\njulia> minimum(sin, Real[]; init=1.0)  # good, since output of sin is <= 1\n1.0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.minimum!","page":"Collections and Data Structures","title":"Base.minimum!","category":"function","text":"minimum!(r, A)\n\nCompute the minimum value of A over the singleton dimensions of r, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> minimum!([1; 1], A)\n2-element Vector{Int64}:\n 1\n 3\n\njulia> minimum!([1 1], A)\n1×2 Matrix{Int64}:\n 1  2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.extrema","page":"Collections and Data Structures","title":"Base.extrema","category":"function","text":"extrema(f, A::AbstractArray; dims) -> Array{Tuple}\n\nCompute the minimum and maximum of f applied to each element in the given dimensions of A.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\n\n\n\n\nextrema(A::AbstractArray; dims) -> Array{Tuple}\n\nCompute the minimum and maximum elements of an array over the given dimensions.\n\nSee also: minimum, maximum, extrema!.\n\nExamples\n\njulia> A = reshape(Vector(1:2:16), (2,2,2))\n2×2×2 Array{Int64, 3}:\n[:, :, 1] =\n 1  5\n 3  7\n\n[:, :, 2] =\n  9  13\n 11  15\n\njulia> extrema(A, dims = (1,2))\n1×1×2 Array{Tuple{Int64, Int64}, 3}:\n[:, :, 1] =\n (1, 7)\n\n[:, :, 2] =\n (9, 15)\n\n\n\n\n\nextrema(f, itr; [init]) -> (mn, mx)\n\nCompute both the minimum mn and maximum mx of f applied to each element in itr and return them as a 2-tuple. Only one pass is made over itr.\n\nThe value returned for empty itr can be specified by init. It must be a 2-tuple whose first and second elements are neutral elements for min and max respectively (i.e. which are greater/less than or equal to any other element). It is used for non-empty collections. Note: it implies that, for empty itr, the returned value (mn, mx) satisfies mn ≥ mx even though for non-empty itr it  satisfies mn ≤ mx.  This is a \"paradoxical\" but yet expected result.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\ncompat: Julia 1.8\nKeyword argument init requires Julia 1.8 or later.\n\nExamples\n\njulia> extrema(sin, 0:π)\n(0.0, 0.9092974268256817)\n\njulia> extrema(sin, Real[]; init = (1.0, -1.0))  # good, since -1 ≤ sin(::Real) ≤ 1\n(1.0, -1.0)\n\n\n\n\n\nextrema(itr; [init]) -> (mn, mx)\n\nCompute both the minimum mn and maximum mx element in a single pass, and return them as a 2-tuple.\n\nThe value returned for empty itr can be specified by init. It must be a 2-tuple whose first and second elements are neutral elements for min and max respectively (i.e. which are greater/less than or equal to any other element). As a consequence, when itr is empty the returned (mn, mx) tuple will satisfy mn ≥ mx. When init is specified it may be used even for non-empty itr.\n\ncompat: Julia 1.8\nKeyword argument init requires Julia 1.8 or later.\n\nExamples\n\njulia> extrema(2:10)\n(2, 10)\n\njulia> extrema([9,pi,4.5])\n(3.141592653589793, 9.0)\n\njulia> extrema([]; init = (Inf, -Inf))\n(Inf, -Inf)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.extrema!","page":"Collections and Data Structures","title":"Base.extrema!","category":"function","text":"extrema!(r, A)\n\nCompute the minimum and maximum value of A over the singleton dimensions of r, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\ncompat: Julia 1.8\nThis method requires Julia 1.8 or later.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> extrema!([(1, 1); (1, 1)], A)\n2-element Vector{Tuple{Int64, Int64}}:\n (1, 2)\n (3, 4)\n\njulia> extrema!([(1, 1);; (1, 1)], A)\n1×2 Matrix{Tuple{Int64, Int64}}:\n (1, 3)  (2, 4)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.argmax","page":"Collections and Data Structures","title":"Base.argmax","category":"function","text":"argmax(A; dims) -> indices\n\nFor an array input, return the indices of the maximum elements over the given dimensions. NaN is treated as greater than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> argmax(A, dims=1)\n1×2 Matrix{CartesianIndex{2}}:\n CartesianIndex(2, 1)  CartesianIndex(2, 2)\n\njulia> argmax(A, dims=2)\n2×1 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 2)\n CartesianIndex(2, 2)\n\n\n\n\n\nargmax(itr)\n\nReturn the index or key of the maximal element in a collection. If there are multiple maximal elements, then the first one will be returned.\n\nThe collection must not be empty.\n\nIndices are of the same type as those returned by keys(itr) and pairs(itr).\n\nValues are compared with isless.\n\nSee also: argmin, findmax.\n\nExamples\n\njulia> argmax([8, 0.1, -9, pi])\n1\n\njulia> argmax([1, 7, 7, 6])\n2\n\njulia> argmax([1, 7, 7, NaN])\n4\n\n\n\n\n\nargmax(f, domain)\n\nReturn a value x from domain for which f(x) is maximised. If there are multiple maximal values for f(x) then the first one will be found.\n\ndomain must be a non-empty iterable.\n\nValues are compared with isless.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nSee also argmin, findmax.\n\nExamples\n\njulia> argmax(abs, -10:5)\n-10\n\njulia> argmax(cos, 0:π/2:2π)\n0.0\n\n\n\n\n\nargmax(r::AbstractRange)\n\nRanges can have multiple maximal elements. In that case argmax will return a maximal index, but not necessarily the first one.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.argmin","page":"Collections and Data Structures","title":"Base.argmin","category":"function","text":"argmin(A; dims) -> indices\n\nFor an array input, return the indices of the minimum elements over the given dimensions. NaN is treated as less than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> argmin(A, dims=1)\n1×2 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 1)  CartesianIndex(1, 2)\n\njulia> argmin(A, dims=2)\n2×1 Matrix{CartesianIndex{2}}:\n CartesianIndex(1, 1)\n CartesianIndex(2, 1)\n\n\n\n\n\nargmin(itr)\n\nReturn the index or key of the minimal element in a collection. If there are multiple minimal elements, then the first one will be returned.\n\nThe collection must not be empty.\n\nIndices are of the same type as those returned by keys(itr) and pairs(itr).\n\nNaN is treated as less than all other values except missing.\n\nSee also: argmax, findmin.\n\nExamples\n\njulia> argmin([8, 0.1, -9, pi])\n3\n\njulia> argmin([7, 1, 1, 6])\n2\n\njulia> argmin([7, 1, 1, NaN])\n4\n\n\n\n\n\nargmin(f, domain)\n\nReturn a value x from domain for which f(x) is minimised. If there are multiple minimal values for f(x) then the first one will be found.\n\ndomain must be a non-empty iterable.\n\nNaN is treated as less than all other values except missing.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nSee also argmax, findmin.\n\nExamples\n\njulia> argmin(sign, -10:5)\n-10\n\njulia> argmin(x -> -x^3 + x^2 - 10, -5:5)\n5\n\njulia> argmin(acos, 0:0.1:1)\n1.0\n\n\n\n\n\nargmin(r::AbstractRange)\n\nRanges can have multiple minimal elements. In that case argmin will return a minimal index, but not necessarily the first one.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.findmax","page":"Collections and Data Structures","title":"Base.findmax","category":"function","text":"findmax(f, A; dims) -> (f(x), index)\n\nFor an array input, returns the value in the codomain and index of the corresponding value which maximize f over the given dimensions.\n\nExamples\n\njulia> A = [-1.0 1; -0.5 2]\n2×2 Matrix{Float64}:\n -1.0  1.0\n -0.5  2.0\n\njulia> findmax(abs2, A, dims=1)\n([1.0 4.0], CartesianIndex{2}[CartesianIndex(1, 1) CartesianIndex(2, 2)])\n\njulia> findmax(abs2, A, dims=2)\n([1.0; 4.0;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 2);;])\n\n\n\n\n\nfindmax(A; dims) -> (maxval, index)\n\nFor an array input, returns the value and index of the maximum over the given dimensions. NaN is treated as greater than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> findmax(A, dims=1)\n([3.0 4.0], CartesianIndex{2}[CartesianIndex(2, 1) CartesianIndex(2, 2)])\n\njulia> findmax(A, dims=2)\n([2.0; 4.0;;], CartesianIndex{2}[CartesianIndex(1, 2); CartesianIndex(2, 2);;])\n\n\n\n\n\nfindmax(itr) -> (x, index)\n\nReturn the maximal element of the collection itr and its index or key. If there are multiple maximal elements, then the first one will be returned. Values are compared with isless.\n\nIndices are of the same type as those returned by keys(itr) and pairs(itr).\n\nSee also: findmin, argmax, maximum.\n\nExamples\n\njulia> findmax([8, 0.1, -9, pi])\n(8.0, 1)\n\njulia> findmax([1, 7, 7, 6])\n(7, 2)\n\njulia> findmax([1, 7, 7, NaN])\n(NaN, 4)\n\n\n\n\n\nfindmax(f, domain) -> (f(x), index)\n\nReturn a pair of a value in the codomain (outputs of f) and the index or key of the corresponding value in the domain (inputs to f) such that f(x) is maximised. If there are multiple maximal points, then the first one will be returned.\n\ndomain must be a non-empty iterable supporting keys. Indices are of the same type as those returned by keys(domain).\n\nValues are compared with isless.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nExamples\n\njulia> findmax(identity, 5:9)\n(9, 5)\n\njulia> findmax(-, 1:10)\n(-1, 1)\n\njulia> findmax(first, [(1, :a), (3, :b), (3, :c)])\n(3, 2)\n\njulia> findmax(cos, 0:π/2:2π)\n(1.0, 1)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.findmin","page":"Collections and Data Structures","title":"Base.findmin","category":"function","text":"findmin(f, A; dims) -> (f(x), index)\n\nFor an array input, returns the value in the codomain and index of the corresponding value which minimize f over the given dimensions.\n\nExamples\n\njulia> A = [-1.0 1; -0.5 2]\n2×2 Matrix{Float64}:\n -1.0  1.0\n -0.5  2.0\n\njulia> findmin(abs2, A, dims=1)\n([0.25 1.0], CartesianIndex{2}[CartesianIndex(2, 1) CartesianIndex(1, 2)])\n\njulia> findmin(abs2, A, dims=2)\n([1.0; 0.25;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 1);;])\n\n\n\n\n\nfindmin(A; dims) -> (minval, index)\n\nFor an array input, returns the value and index of the minimum over the given dimensions. NaN is treated as less than all other values except missing.\n\nExamples\n\njulia> A = [1.0 2; 3 4]\n2×2 Matrix{Float64}:\n 1.0  2.0\n 3.0  4.0\n\njulia> findmin(A, dims=1)\n([1.0 2.0], CartesianIndex{2}[CartesianIndex(1, 1) CartesianIndex(1, 2)])\n\njulia> findmin(A, dims=2)\n([1.0; 3.0;;], CartesianIndex{2}[CartesianIndex(1, 1); CartesianIndex(2, 1);;])\n\n\n\n\n\nfindmin(itr) -> (x, index)\n\nReturn the minimal element of the collection itr and its index or key. If there are multiple minimal elements, then the first one will be returned. NaN is treated as less than all other values except missing.\n\nIndices are of the same type as those returned by keys(itr) and pairs(itr).\n\nSee also: findmax, argmin, minimum.\n\nExamples\n\njulia> findmin([8, 0.1, -9, pi])\n(-9.0, 3)\n\njulia> findmin([1, 7, 7, 6])\n(1, 1)\n\njulia> findmin([1, 7, 7, NaN])\n(NaN, 4)\n\n\n\n\n\nfindmin(f, domain) -> (f(x), index)\n\nReturn a pair of a value in the codomain (outputs of f) and the index or key of the corresponding value in the domain (inputs to f) such that f(x) is minimised. If there are multiple minimal points, then the first one will be returned.\n\ndomain must be a non-empty iterable.\n\nIndices are of the same type as those returned by keys(domain) and pairs(domain).\n\nNaN is treated as less than all other values except missing.\n\ncompat: Julia 1.7\nThis method requires Julia 1.7 or later.\n\nExamples\n\njulia> findmin(identity, 5:9)\n(5, 1)\n\njulia> findmin(-, 1:10)\n(-10, 10)\n\njulia> findmin(first, [(2, :a), (2, :b), (3, :c)])\n(2, 1)\n\njulia> findmin(cos, 0:π/2:2π)\n(-1.0, 3)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.findmax!","page":"Collections and Data Structures","title":"Base.findmax!","category":"function","text":"findmax!(rval, rind, A) -> (maxval, index)\n\nFind the maximum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind. NaN is treated as greater than all other values except missing.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.findmin!","page":"Collections and Data Structures","title":"Base.findmin!","category":"function","text":"findmin!(rval, rind, A) -> (minval, index)\n\nFind the minimum of A and the corresponding linear index along singleton dimensions of rval and rind, and store the results in rval and rind. NaN is treated as less than all other values except missing.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.sum","page":"Collections and Data Structures","title":"Base.sum","category":"function","text":"sum(f, A::AbstractArray; dims)\n\nSum the results of calling function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 10  20\n\njulia> sum(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n  5\n 25\n\n\n\n\n\nsum(A::AbstractArray; dims)\n\nSum elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum(A, dims=1)\n1×2 Matrix{Int64}:\n 4  6\n\njulia> sum(A, dims=2)\n2×1 Matrix{Int64}:\n 3\n 7\n\n\n\n\n\nsum(itr; [init])\n\nReturn the sum of all elements in a collection.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the additive identity (i.e. zero) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nSee also: reduce, mapreduce, count, union.\n\nExamples\n\njulia> sum(1:20)\n210\n\njulia> sum(1:20; init = 0.0)\n210.0\n\n\n\n\n\nsum(f, itr; [init])\n\nSum the results of calling function f on each element of itr.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the additive identity (i.e. zero) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> sum(abs2, [2; 3; 4])\n29\n\nNote the important difference between sum(A) and reduce(+, A) for arrays with small integer eltype:\n\njulia> sum(Int8[100, 28])\n128\n\njulia> reduce(+, Int8[100, 28])\n-128\n\nIn the former case, the integers are widened to system word size and therefore the result is 128. In the latter case, no such widening happens and integer overflow results in -128.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.sum!","page":"Collections and Data Structures","title":"Base.sum!","category":"function","text":"sum!(r, A)\n\nSum elements of A over the singleton dimensions of r, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> sum!([1; 1], A)\n2-element Vector{Int64}:\n 3\n 7\n\njulia> sum!([1 1], A)\n1×2 Matrix{Int64}:\n 4  6\n\n\n\n\n\n"},{"location":"base/collections.html#Base.prod","page":"Collections and Data Structures","title":"Base.prod","category":"function","text":"prod(f, A::AbstractArray; dims)\n\nMultiply the results of calling the function f on each element of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod(abs2, A, dims=1)\n1×2 Matrix{Int64}:\n 9  64\n\njulia> prod(abs2, A, dims=2)\n2×1 Matrix{Int64}:\n   4\n 144\n\n\n\n\n\nprod(A::AbstractArray; dims)\n\nMultiply elements of an array over the given dimensions.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod(A, dims=1)\n1×2 Matrix{Int64}:\n 3  8\n\njulia> prod(A, dims=2)\n2×1 Matrix{Int64}:\n  2\n 12\n\n\n\n\n\nprod(itr; [init])\n\nReturn the product of all elements of a collection.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the multiplicative identity (i.e. one) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nSee also: reduce, cumprod, any.\n\nExamples\n\njulia> prod(1:5)\n120\n\njulia> prod(1:5; init = 1.0)\n120.0\n\n\n\n\n\nprod(f, itr; [init])\n\nReturn the product of f applied to each element of itr.\n\nThe return type is Int for signed integers of less than system word size, and UInt for unsigned integers of less than system word size.  For all other arguments, a common return type is found to which all arguments are promoted.\n\nThe value returned for empty itr can be specified by init. It must be the multiplicative identity (i.e. one) as it is unspecified whether init is used for non-empty collections.\n\ncompat: Julia 1.6\nKeyword argument init requires Julia 1.6 or later.\n\nExamples\n\njulia> prod(abs2, [2; 3; 4])\n576\n\n\n\n\n\n"},{"location":"base/collections.html#Base.prod!","page":"Collections and Data Structures","title":"Base.prod!","category":"function","text":"prod!(r, A)\n\nMultiply elements of A over the singleton dimensions of r, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> prod!([1; 1], A)\n2-element Vector{Int64}:\n  2\n 12\n\njulia> prod!([1 1], A)\n1×2 Matrix{Int64}:\n 3  8\n\n\n\n\n\n"},{"location":"base/collections.html#Base.any-Tuple{Any}","page":"Collections and Data Structures","title":"Base.any","category":"method","text":"any(itr)::Bool\n\nTest whether any elements of a boolean collection are true, returning true as soon as the first true value in itr is encountered (short-circuiting). To short-circuit on false, use all.\n\nIf the input contains missing values, return missing if all non-missing values are false (or equivalently, if the input contains no true value), following three-valued logic.\n\nSee also: all, count, sum, |, ||.\n\nExamples\n\njulia> a = [true,false,false,true]\n4-element Vector{Bool}:\n 1\n 0\n 0\n 1\n\njulia> any(a)\ntrue\n\njulia> any((println(i); v) for (i, v) in enumerate(a))\n1\ntrue\n\njulia> any([missing, true])\ntrue\n\njulia> any([false, missing])\nmissing\n\n\n\n\n\n"},{"location":"base/collections.html#Base.any-Tuple{AbstractArray, Any}","page":"Collections and Data Structures","title":"Base.any","category":"method","text":"any(p, itr)::Bool\n\nDetermine whether predicate p returns true for any elements of itr, returning true as soon as the first item in itr for which p returns true is encountered (short-circuiting). To short-circuit on false, use all.\n\nIf the input contains missing values, return missing if all non-missing values are false (or equivalently, if the input contains no true value), following three-valued logic.\n\nExamples\n\njulia> any(i->(4<=i<=6), [3,5,7])\ntrue\n\njulia> any(i -> (println(i); i > 3), 1:10)\n1\n2\n3\n4\ntrue\n\njulia> any(i -> i > 0, [1, missing])\ntrue\n\njulia> any(i -> i > 0, [-1, missing])\nmissing\n\njulia> any(i -> i > 0, [-1, 0])\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.any!","page":"Collections and Data Structures","title":"Base.any!","category":"function","text":"any!(r, A)\n\nTest whether any values in A along the singleton dimensions of r are true, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Matrix{Bool}:\n 1  0\n 1  0\n\njulia> any!(Bool[1; 1], A)\n2-element Vector{Bool}:\n 1\n 1\n\njulia> any!(Bool[1 1], A)\n1×2 Matrix{Bool}:\n 1  0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.all-Tuple{Any}","page":"Collections and Data Structures","title":"Base.all","category":"method","text":"all(itr)::Bool\n\nTest whether all elements of a boolean collection are true, returning false as soon as the first false value in itr is encountered (short-circuiting). To short-circuit on true, use any.\n\nIf the input contains missing values, return missing if all non-missing values are true (or equivalently, if the input contains no false value), following three-valued logic.\n\nSee also: all!, any, count, &, &&, allunique.\n\nExamples\n\njulia> a = [true,false,false,true]\n4-element Vector{Bool}:\n 1\n 0\n 0\n 1\n\njulia> all(a)\nfalse\n\njulia> all((println(i); v) for (i, v) in enumerate(a))\n1\n2\nfalse\n\njulia> all([missing, false])\nfalse\n\njulia> all([true, missing])\nmissing\n\n\n\n\n\n"},{"location":"base/collections.html#Base.all-Tuple{AbstractArray, Any}","page":"Collections and Data Structures","title":"Base.all","category":"method","text":"all(p, itr)::Bool\n\nDetermine whether predicate p returns true for all elements of itr, returning false as soon as the first item in itr for which p returns false is encountered (short-circuiting). To short-circuit on true, use any.\n\nIf the input contains missing values, return missing if all non-missing values are true (or equivalently, if the input contains no false value), following three-valued logic.\n\nExamples\n\njulia> all(i->(4<=i<=6), [4,5,6])\ntrue\n\njulia> all(i -> (println(i); i < 3), 1:10)\n1\n2\n3\nfalse\n\njulia> all(i -> i > 0, [1, missing])\nmissing\n\njulia> all(i -> i > 0, [-1, missing])\nfalse\n\njulia> all(i -> i > 0, [1, 2])\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.all!","page":"Collections and Data Structures","title":"Base.all!","category":"function","text":"all!(r, A)\n\nTest whether all values in A along the singleton dimensions of r are true, and write results to r.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> A = [true false; true false]\n2×2 Matrix{Bool}:\n 1  0\n 1  0\n\njulia> all!(Bool[1; 1], A)\n2-element Vector{Bool}:\n 0\n 0\n\njulia> all!(Bool[1 1], A)\n1×2 Matrix{Bool}:\n 1  0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.count","page":"Collections and Data Structures","title":"Base.count","category":"function","text":"count([f=identity,] A::AbstractArray; dims=:)\n\nCount the number of elements in A for which f returns true over the given dimensions.\n\ncompat: Julia 1.5\ndims keyword was added in Julia 1.5.\n\ncompat: Julia 1.6\ninit keyword was added in Julia 1.6.\n\nExamples\n\njulia> A = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> count(<=(2), A, dims=1)\n1×2 Matrix{Int64}:\n 1  1\n\njulia> count(<=(2), A, dims=2)\n2×1 Matrix{Int64}:\n 2\n 0\n\n\n\n\n\ncount(\n    pattern::Union{AbstractChar,AbstractString,AbstractPattern},\n    string::AbstractString;\n    overlap::Bool = false,\n)\n\nReturn the number of matches for pattern in string. This is equivalent to calling length(findall(pattern, string)) but more efficient.\n\nIf overlap=true, the matching sequences are allowed to overlap indices in the original string, otherwise they must be from disjoint character ranges.\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\ncompat: Julia 1.7\nUsing a character as the pattern requires at least Julia 1.7.\n\nExamples\n\njulia> count('a', \"JuliaLang\")\n2\n\njulia> count(r\"a(.)a\", \"cabacabac\", overlap=true)\n3\n\njulia> count(r\"a(.)a\", \"cabacabac\")\n2\n\nSee also\n\neachmatch, occursin, findall\n\n\n\n\n\ncount([f=identity,] itr; init=0)::Integer\n\nCount the number of elements in itr for which the function f returns true. If f is omitted, count the number of true elements in itr (which should be a collection of boolean values). init optionally specifies the value to start counting from and therefore also determines the output type.\n\ncompat: Julia 1.6\ninit keyword was added in Julia 1.6.\n\nSee also: any, sum.\n\nExamples\n\njulia> count(i->(4<=i<=6), [2,3,4,5,6])\n3\n\njulia> count([true, false, true, true])\n3\n\njulia> count(>(3), 1:7, init=UInt(0))\n0x0000000000000004\n\n\n\n\n\n"},{"location":"base/collections.html#Base.foreach","page":"Collections and Data Structures","title":"Base.foreach","category":"function","text":"foreach(f, c...) -> nothing\n\nCall function f on each element of iterable c. For multiple iterable arguments, f is called elementwise, and iteration stops when any iterator is finished.\n\nforeach should be used instead of map when the results of f are not needed, for example in foreach(println, array).\n\nExamples\n\njulia> tri = 1:3:7; res = Int[];\n\njulia> foreach(x -> push!(res, x^2), tri)\n\njulia> res\n3-element Vector{Int32}:\n  1\n 16\n 49\n\njulia> foreach((x, y) -> println(x, \" with \", y), tri, 'a':'z')\n1 with a\n4 with b\n7 with c\n\n\n\n\n\n"},{"location":"base/collections.html#Base.map","page":"Collections and Data Structures","title":"Base.map","category":"function","text":"map(f, A::AbstractArray...) -> N-array\n\nWhen acting on multi-dimensional arrays of the same ndims, they must all have the same axes, and the answer will too.\n\nSee also broadcast, which allows mismatched sizes.\n\nExamples\n\njulia> map(//, [1 2; 3 4], [4 3; 2 1])\n2×2 Matrix{Rational{Int32}}:\n 1//4  2//3\n 3//2  4//1\n\njulia> map(+, [1 2; 3 4], zeros(2,1))\nERROR: DimensionMismatch\n\njulia> map(+, [1 2; 3 4], [1,10,100,1000], zeros(3,1))  # iterates until 3rd is exhausted\n3-element Vector{Float64}:\n   2.0\n  13.0\n 102.0\n\n\n\n\n\nmap(f, c...) -> collection\n\nTransform collection c by applying f to each element. For multiple collection arguments, apply f elementwise, and stop when any of them is exhausted.\n\nThe element type of the result is determined in the same manner as in collect.\n\nSee also map!, foreach, mapreduce, mapslices, zip, Iterators.map.\n\nExamples\n\njulia> map(x -> x * 2, [1, 2, 3])\n3-element Vector{Int64}:\n 2\n 4\n 6\n\njulia> map(+, [1, 2, 3], [10, 20, 30, 400, 5000])\n3-element Vector{Int64}:\n 11\n 22\n 33\n\n\n\n\n\n"},{"location":"base/collections.html#Base.map!","page":"Collections and Data Structures","title":"Base.map!","category":"function","text":"map!(f, values(dict::AbstractDict))\n\nModifies dict by transforming each value from val to f(val). Note that the type of dict cannot be changed: if f(val) is not an instance of the value type of dict then it will be converted to the value type if possible and otherwise raise an error.\n\ncompat: Julia 1.2\nmap!(f, values(dict::AbstractDict)) requires Julia 1.2 or later.\n\nExamples\n\njulia> d = Dict(:a => 1, :b => 2)\nDict{Symbol, Int64} with 2 entries:\n  :a => 1\n  :b => 2\n\njulia> map!(v -> v-1, values(d))\nValueIterator for a Dict{Symbol, Int64} with 2 entries. Values:\n  0\n  1\n\n\n\n\n\nmap!(function, array)\n\nLike map, but stores the result in the same array.\n\ncompat: Julia 1.12\nThis method requires Julia 1.12 or later. To support previous versions too, use the equivalent map!(function, array, array).\n\nExamples\n\njulia> a = [1 2 3; 4 5 6];\n\njulia> map!(x -> x^3, a);\n\njulia> a\n2×3 Matrix{Int32}:\n  1    8   27\n 64  125  216\n\n\n\n\n\nmap!(function, destination, collection...)\n\nLike map, but stores the result in destination rather than a new collection. destination must be at least as large as the smallest collection.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nSee also: map, foreach, zip, copyto!.\n\nExamples\n\njulia> a = zeros(3);\n\njulia> map!(x -> x * 2, a, [1, 2, 3]);\n\njulia> a\n3-element Vector{Float64}:\n 2.0\n 4.0\n 6.0\n\njulia> map!(+, zeros(Int, 5), 100:999, 1:3)\n5-element Vector{Int32}:\n 101\n 103\n 105\n   0\n   0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.mapreduce-Tuple{Any, Any, Any}","page":"Collections and Data Structures","title":"Base.mapreduce","category":"method","text":"mapreduce(f, op, itrs...; [init])\n\nApply function f to each element(s) in itrs, and then reduce the result using the binary function op. If provided, init must be a neutral element for op that will be returned for empty collections. It is unspecified whether init is used for non-empty collections. In general, it will be necessary to provide init to work with empty collections.\n\nmapreduce is functionally equivalent to calling reduce(op, map(f, itr); init=init), but will in general execute faster since no intermediate collection needs to be created. See documentation for reduce and map.\n\ncompat: Julia 1.2\nmapreduce with multiple iterators requires Julia 1.2 or later.\n\nExamples\n\njulia> mapreduce(x->x^2, +, [1:3;]) # == 1 + 4 + 9\n14\n\nThe associativity of the reduction is implementation-dependent. Additionally, some implementations may reuse the return value of f for elements that appear multiple times in itr. Use mapfoldl or mapfoldr instead for guaranteed left or right associativity and invocation of f for every value.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.mapfoldl-Tuple{Any, Any, Any}","page":"Collections and Data Structures","title":"Base.mapfoldl","category":"method","text":"mapfoldl(f, op, itr; [init])\n\nLike mapreduce, but with guaranteed left associativity, as in foldl. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.mapfoldr-Tuple{Any, Any, Any}","page":"Collections and Data Structures","title":"Base.mapfoldr","category":"method","text":"mapfoldr(f, op, itr; [init])\n\nLike mapreduce, but with guaranteed right associativity, as in foldr. If provided, the keyword argument init will be used exactly once. In general, it will be necessary to provide init to work with empty collections.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.first","page":"Collections and Data Structures","title":"Base.first","category":"function","text":"first(s::AbstractString, n::Integer)\n\nGet a string consisting of the first n characters of s.\n\nExamples\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 1)\n\"∀\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 3)\n\"∀ϵ≠\"\n\n\n\n\n\nfirst(itr, n::Integer)\n\nGet the first n elements of the iterable collection itr, or fewer elements if itr is not long enough.\n\nSee also: startswith, Iterators.take.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\nExamples\n\njulia> first([\"foo\", \"bar\", \"qux\"], 2)\n2-element Vector{String}:\n \"foo\"\n \"bar\"\n\njulia> first(1:6, 10)\n1:6\n\njulia> first(Bool[], 1)\nBool[]\n\n\n\n\n\nfirst(coll)\n\nGet the first element of an iterable collection. Return the start point of an AbstractRange even if it is empty.\n\nSee also: only, firstindex, last.\n\nExamples\n\njulia> first(2:2:10)\n2\n\njulia> first([1; 2; 3; 4])\n1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.last","page":"Collections and Data Structures","title":"Base.last","category":"function","text":"last(s::AbstractString, n::Integer)\n\nGet a string consisting of the last n characters of s.\n\nExamples\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 1)\n\"0\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 3)\n\"²>0\"\n\n\n\n\n\nlast(itr, n::Integer)\n\nGet the last n elements of the iterable collection itr, or fewer elements if itr is not long enough.\n\ncompat: Julia 1.6\nThis method requires at least Julia 1.6.\n\nExamples\n\njulia> last([\"foo\", \"bar\", \"qux\"], 2)\n2-element Vector{String}:\n \"bar\"\n \"qux\"\n\njulia> last(1:6, 10)\n1:6\n\njulia> last(Float64[], 1)\nFloat64[]\n\n\n\n\n\nlast(coll)\n\nGet the last element of an ordered collection, if it can be computed in O(1) time. This is accomplished by calling lastindex to get the last index. Return the end point of an AbstractRange even if it is empty.\n\nSee also first, endswith.\n\nExamples\n\njulia> last(1:2:10)\n9\n\njulia> last([1; 2; 3; 4])\n4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.front","page":"Collections and Data Structures","title":"Base.front","category":"function","text":"front(x::Tuple)::Tuple\n\nReturn a Tuple consisting of all but the last component of x.\n\nSee also: first, tail.\n\nExamples\n\njulia> Base.front((1,2,3))\n(1, 2)\n\njulia> Base.front(())\nERROR: ArgumentError: Cannot call front on an empty tuple.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.tail","page":"Collections and Data Structures","title":"Base.tail","category":"function","text":"tail(x::Tuple)::Tuple\n\nReturn a Tuple consisting of all but the first component of x.\n\nSee also: front, rest, first, Iterators.peel.\n\nExamples\n\njulia> Base.tail((1,2,3))\n(2, 3)\n\njulia> Base.tail(())\nERROR: ArgumentError: Cannot call tail on an empty tuple.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.step","page":"Collections and Data Structures","title":"Base.step","category":"function","text":"step(r)\n\nGet the step size of an AbstractRange object.\n\nExamples\n\njulia> step(1:10)\n1\n\njulia> step(1:2:10)\n2\n\njulia> step(2.5:0.3:10.9)\n0.3\n\njulia> step(range(2.5, stop=10.9, length=85))\n0.1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.collect-Tuple{Any}","page":"Collections and Data Structures","title":"Base.collect","category":"method","text":"collect(iterator)\n\nReturn an Array of all items in a collection or iterator. For dictionaries, returns a Vector of key=>value Pairs. If the argument is array-like or is an iterator with the HasShape trait, the result will have the same shape and number of dimensions as the argument.\n\nUsed by comprehensions to turn a generator expression into an Array. Thus, on generators, the square-brackets notation may be used instead of calling collect, see second example.\n\nThe element type of the returned array is based on the types of the values collected. However, if the iterator is empty then the element type of the returned (empty) array is determined by type inference.\n\nExamples\n\nCollect items from a UnitRange{Int64} collection:\n\njulia> collect(1:3)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\nCollect items from a generator (same output as [x^2 for x in 1:3]):\n\njulia> collect(x^2 for x in 1:3)\n3-element Vector{Int64}:\n 1\n 4\n 9\n\nCollecting an empty iterator where the result type depends on type inference:\n\njulia> [rand(Bool) ? 1 : missing for _ in []]\nUnion{Missing, Int64}[]\n\nWhen the iterator is non-empty, the result type depends only on values:\n\njulia> [rand(Bool) ? 1 : missing for _ in [\"\"]]\n1-element Vector{Int64}:\n 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.collect-Tuple{Type, Any}","page":"Collections and Data Structures","title":"Base.collect","category":"method","text":"collect(element_type, collection)\n\nReturn an Array with the given element type of all items in a collection or iterable. The result has the same shape and number of dimensions as collection.\n\nExamples\n\njulia> collect(Float64, 1:2:5)\n3-element Vector{Float64}:\n 1.0\n 3.0\n 5.0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.filter","page":"Collections and Data Structures","title":"Base.filter","category":"function","text":"filter(f, itr::SkipMissing{<:AbstractArray})\n\nReturn a vector similar to the array wrapped by the given SkipMissing iterator but with all missing elements and those for which f returns false removed.\n\ncompat: Julia 1.2\nThis method requires Julia 1.2 or later.\n\nExamples\n\njulia> x = [1 2; missing 4]\n2×2 Matrix{Union{Missing, Int64}}:\n 1         2\n  missing  4\n\njulia> filter(isodd, skipmissing(x))\n1-element Vector{Int64}:\n 1\n\n\n\n\n\nfilter(f, d::AbstractDict)\n\nReturn a copy of d, removing elements for which f is false. The function f is passed key=>value pairs.\n\nExamples\n\njulia> d = Dict(1=>\"a\", 2=>\"b\")\nDict{Int64, String} with 2 entries:\n  2 => \"b\"\n  1 => \"a\"\n\njulia> filter(p->isodd(p.first), d)\nDict{Int64, String} with 1 entry:\n  1 => \"a\"\n\n\n\n\n\nfilter(f)\n\nCreate a function that filters its arguments with function f using filter, i.e. a function equivalent to x -> filter(f, x).\n\nThe returned function is of type Base.Fix1{typeof(filter)}, which can be used to implement specialized methods.\n\nExamples\n\njulia> (1, 2, Inf, 4, NaN, 6) |> filter(isfinite)\n(1, 2, 4, 6)\n\njulia> map(filter(iseven), [1:3, 2:4, 3:5])\n3-element Vector{Vector{Int64}}:\n [2]\n [2, 4]\n [4]\n\ncompat: Julia 1.9\nThis method requires at least Julia 1.9.\n\n\n\n\n\nfilter(f, a)\n\nReturn a copy of collection a, removing elements for which f is false. The function f is passed one argument.\n\ncompat: Julia 1.4\nSupport for a as a tuple requires at least Julia 1.4.\n\nSee also: filter!, Iterators.filter.\n\nExamples\n\njulia> a = 1:10\n1:10\n\njulia> filter(isodd, a)\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\n\n\n\n\n"},{"location":"base/collections.html#Base.filter!","page":"Collections and Data Structures","title":"Base.filter!","category":"function","text":"filter!(f, d::AbstractDict)\n\nUpdate d, removing elements for which f is false. The function f is passed key=>value pairs.\n\nExamples\n\njulia> d = Dict(1=>\"a\", 2=>\"b\", 3=>\"c\")\nDict{Int64, String} with 3 entries:\n  2 => \"b\"\n  3 => \"c\"\n  1 => \"a\"\n\njulia> filter!(p->isodd(p.first), d)\nDict{Int64, String} with 2 entries:\n  3 => \"c\"\n  1 => \"a\"\n\n\n\n\n\nfilter!(f, a)\n\nUpdate collection a, removing elements for which f is false. The function f is passed one argument.\n\nExamples\n\njulia> filter!(isodd, Vector(1:10))\n5-element Vector{Int64}:\n 1\n 3\n 5\n 7\n 9\n\n\n\n\n\n"},{"location":"base/collections.html#Base.replace-Tuple{Any, Vararg{Pair}}","page":"Collections and Data Structures","title":"Base.replace","category":"method","text":"replace(A, old_new::Pair...; [count::Integer])\n\nReturn a copy of collection A where, for each pair old=>new in old_new, all occurrences of old are replaced by new. Equality is determined using isequal. If count is specified, then replace at most count occurrences in total.\n\nThe element type of the result is chosen using promotion (see promote_type) based on the element type of A and on the types of the new values in pairs. If count is omitted and the element type of A is a Union, the element type of the result will not include singleton types which are replaced with values of a different type: for example, Union{T,Missing} will become T if missing is replaced.\n\nSee also replace!, splice!, delete!, insert!.\n\ncompat: Julia 1.7\nVersion 1.7 is required to replace elements of a Tuple.\n\nExamples\n\njulia> replace([1, 2, 1, 3], 1=>0, 2=>4, count=2)\n4-element Vector{Int64}:\n 0\n 4\n 1\n 3\n\njulia> replace([1, missing], missing=>0)\n2-element Vector{Int64}:\n 1\n 0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.replace-Tuple{Union{Function, Type}, Any}","page":"Collections and Data Structures","title":"Base.replace","category":"method","text":"replace(new::Union{Function, Type}, A; [count::Integer])\n\nReturn a copy of A where each value x in A is replaced by new(x). If count is specified, then replace at most count values in total (replacements being defined as new(x) !== x).\n\ncompat: Julia 1.7\nVersion 1.7 is required to replace elements of a Tuple.\n\nExamples\n\njulia> replace(x -> isodd(x) ? 2x : x, [1, 2, 3, 4])\n4-element Vector{Int64}:\n 2\n 2\n 6\n 4\n\njulia> replace(Dict(1=>2, 3=>4)) do kv\n           first(kv) < 3 ? first(kv)=>3 : kv\n       end\nDict{Int64, Int64} with 2 entries:\n  3 => 4\n  1 => 3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.replace!","page":"Collections and Data Structures","title":"Base.replace!","category":"function","text":"replace!(new::Union{Function, Type}, A; [count::Integer])\n\nReplace each element x in collection A by new(x). If count is specified, then replace at most count values in total (replacements being defined as new(x) !== x).\n\nExamples\n\njulia> replace!(x -> isodd(x) ? 2x : x, [1, 2, 3, 4])\n4-element Vector{Int64}:\n 2\n 2\n 6\n 4\n\njulia> replace!(Dict(1=>2, 3=>4)) do kv\n           first(kv) < 3 ? first(kv)=>3 : kv\n       end\nDict{Int64, Int64} with 2 entries:\n  3 => 4\n  1 => 3\n\njulia> replace!(x->2x, Set([3, 6]))\nSet{Int64} with 2 elements:\n  6\n  12\n\n\n\n\n\nreplace!(A, old_new::Pair...; [count::Integer])\n\nFor each pair old=>new in old_new, replace all occurrences of old in collection A by new. Equality is determined using isequal. If count is specified, then replace at most count occurrences in total. See also replace.\n\nExamples\n\njulia> replace!([1, 2, 1, 3], 1=>0, 2=>4, count=2)\n4-element Vector{Int64}:\n 0\n 4\n 1\n 3\n\njulia> replace!(Set([1, 2, 3]), 1=>0)\nSet{Int64} with 3 elements:\n  0\n  2\n  3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.rest","page":"Collections and Data Structures","title":"Base.rest","category":"function","text":"Base.rest(collection[, itr_state])\n\nGeneric function for taking the tail of collection, starting from a specific iteration state itr_state. Return a Tuple, if collection itself is a Tuple, a subtype of AbstractVector, if collection is an AbstractArray, a subtype of AbstractString if collection is an AbstractString, and an arbitrary iterator, falling back to Iterators.rest(collection[, itr_state]), otherwise.\n\nCan be overloaded for user-defined collection types to customize the behavior of slurping in assignments in final position, like a, b... = collection.\n\ncompat: Julia 1.6\nBase.rest requires at least Julia 1.6.\n\nSee also: first, Iterators.rest, Base.split_rest.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> first, state = iterate(a)\n(1, 2)\n\njulia> first, Base.rest(a, state)\n(1, [3, 2, 4])\n\n\n\n\n\n"},{"location":"base/collections.html#Base.split_rest","page":"Collections and Data Structures","title":"Base.split_rest","category":"function","text":"Base.split_rest(collection, n::Int[, itr_state]) -> (rest_but_n, last_n)\n\nGeneric function for splitting the tail of collection, starting from a specific iteration state itr_state. Returns a tuple of two new collections. The first one contains all elements of the tail but the n last ones, which make up the second collection.\n\nThe type of the first collection generally follows that of Base.rest, except that the fallback case is not lazy, but is collected eagerly into a vector.\n\nCan be overloaded for user-defined collection types to customize the behavior of slurping in assignments in non-final position, like a, b..., c = collection.\n\ncompat: Julia 1.9\nBase.split_rest requires at least Julia 1.9.\n\nSee also: Base.rest.\n\nExamples\n\njulia> a = [1 2; 3 4]\n2×2 Matrix{Int64}:\n 1  2\n 3  4\n\njulia> first, state = iterate(a)\n(1, 2)\n\njulia> first, Base.split_rest(a, 1, state)\n(1, ([3, 2], [4]))\n\n\n\n\n\n"},{"location":"base/collections.html#Base.getindex","page":"Collections and Data Structures","title":"Base.getindex","category":"function","text":"getindex(collection, key...)\n\nRetrieve the value(s) stored at the given key or index within a collection. The syntax a[i,j,...] is converted by the compiler to getindex(a, i, j, ...).\n\nSee also get, keys, eachindex.\n\nExamples\n\njulia> A = Dict(\"a\" => 1, \"b\" => 2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> getindex(A, \"a\")\n1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.setindex!","page":"Collections and Data Structures","title":"Base.setindex!","category":"function","text":"setindex!(collection, value, key...)\n\nStore the given value at the given key or index within a collection. The syntax a[i,j,...] = x is converted by the compiler to (setindex!(a, x, i, j, ...); x).\n\nExamples\n\njulia> a = Dict(\"a\"=>1)\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\njulia> setindex!(a, 2, \"b\")\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.firstindex","page":"Collections and Data Structures","title":"Base.firstindex","category":"function","text":"firstindex(collection)::Integer\nfirstindex(collection, d)::Integer\n\nReturn the first index of collection. If d is given, return the first index of collection along dimension d.\n\nThe syntaxes A[begin] and A[1, begin] lower to A[firstindex(A)] and A[1, firstindex(A, 2)], respectively.\n\nSee also: first, axes, lastindex, nextind.\n\nExamples\n\njulia> firstindex([1,2,4])\n1\n\njulia> firstindex(rand(3,4,5), 2)\n1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.lastindex","page":"Collections and Data Structures","title":"Base.lastindex","category":"function","text":"lastindex(collection)::Integer\nlastindex(collection, d)::Integer\n\nReturn the last index of collection. If d is given, return the last index of collection along dimension d.\n\nThe syntaxes A[end] and A[end, end] lower to A[lastindex(A)] and A[lastindex(A, 1), lastindex(A, 2)], respectively.\n\nSee also: axes, firstindex, eachindex, prevind.\n\nExamples\n\njulia> lastindex([1,2,4])\n3\n\njulia> lastindex(rand(3,4,5), 2)\n4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.AbstractDict","page":"Collections and Data Structures","title":"Base.AbstractDict","category":"type","text":"AbstractDict{K, V}\n\nSupertype for dictionary-like types with keys of type K and values of type V. Dict, IdDict and other types are subtypes of this. An AbstractDict{K, V} should be an iterator of Pair{K, V}.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.Dict","page":"Collections and Data Structures","title":"Base.Dict","category":"type","text":"Dict([itr])\n\nDict{K,V}() constructs a hash table with keys of type K and values of type V. Keys are compared with isequal and hashed with hash.\n\nGiven a single iterable argument, constructs a Dict whose key-value pairs are taken from 2-tuples (key,value) generated by the argument.\n\nExamples\n\njulia> Dict([(\"A\", 1), (\"B\", 2)])\nDict{String, Int64} with 2 entries:\n  \"B\" => 2\n  \"A\" => 1\n\nAlternatively, a sequence of pair arguments may be passed.\n\njulia> Dict(\"A\"=>1, \"B\"=>2)\nDict{String, Int64} with 2 entries:\n  \"B\" => 2\n  \"A\" => 1\n\nwarning: Warning\nKeys are allowed to be mutable, but if you do mutate stored keys, the hash table may become internally inconsistent, in which case the Dict will not work properly. IdDict can be an alternative if you need to mutate keys.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.IdDict","page":"Collections and Data Structures","title":"Base.IdDict","category":"type","text":"IdDict([itr])\n\nIdDict{K,V}() constructs a hash table using objectid as hash and === as equality with keys of type K and values of type V. See Dict for further help and IdSet for the set version of this.\n\nIn the example below, the Dict keys are all isequal and therefore get hashed the same, so they get overwritten. The IdDict hashes by object-id, and thus preserves the 3 different keys.\n\nExamples\n\njulia> Dict(true => \"yes\", 1 => \"no\", 1.0 => \"maybe\")\nDict{Real, String} with 1 entry:\n  1.0 => \"maybe\"\n\njulia> IdDict(true => \"yes\", 1 => \"no\", 1.0 => \"maybe\")\nIdDict{Any, String} with 3 entries:\n  true => \"yes\"\n  1.0  => \"maybe\"\n  1    => \"no\"\n\n\n\n\n\n"},{"location":"base/collections.html#Base.WeakKeyDict","page":"Collections and Data Structures","title":"Base.WeakKeyDict","category":"type","text":"WeakKeyDict([itr])\n\nWeakKeyDict() constructs a hash table where the keys are weak references to objects which may be garbage collected even when referenced in a hash table.\n\nSee Dict for further help.  Note, unlike Dict, WeakKeyDict does not convert keys on insertion, as this would imply the key object was unreferenced anywhere before insertion.\n\nSee also WeakRef.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.ImmutableDict","page":"Collections and Data Structures","title":"Base.ImmutableDict","category":"type","text":"ImmutableDict\n\nImmutableDict is a dictionary implemented as an immutable linked list, which is optimal for small dictionaries that are constructed over many individual insertions. Note that it is not possible to remove a value, although it can be partially overridden and hidden by inserting a new value with the same key.\n\nImmutableDict(KV::Pair)\n\nCreate a new entry in the ImmutableDict for a key => value pair\n\nuse (key => value) in dict to see if this particular combination is in the properties set\nuse get(dict, key, default) to retrieve the most recent value for a particular key\n\n\n\n\n\n"},{"location":"base/collections.html#Base.PersistentDict","page":"Collections and Data Structures","title":"Base.PersistentDict","category":"type","text":"PersistentDict\n\nPersistentDict is a dictionary implemented as a hash array mapped trie, which is optimal for situations where you need persistence, each operation returns a new dictionary separate from the previous one, but the underlying implementation is space-efficient and may share storage across multiple separate dictionaries.\n\nnote: Note\nIt behaves like an IdDict.\n\nPersistentDict(KV::Pair)\n\nExamples\n\njulia> dict = Base.PersistentDict(:a=>1)\nBase.PersistentDict{Symbol, Int64} with 1 entry:\n  :a => 1\n\njulia> dict2 = Base.delete(dict, :a)\nBase.PersistentDict{Symbol, Int64}()\n\njulia> dict3 = Base.PersistentDict(dict, :a=>2)\nBase.PersistentDict{Symbol, Int64} with 1 entry:\n  :a => 2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.haskey","page":"Collections and Data Structures","title":"Base.haskey","category":"function","text":"haskey(collection, key)::Bool\n\nDetermine whether a collection has a mapping for a given key.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> haskey(D, 'a')\ntrue\n\njulia> haskey(D, 'c')\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.get","page":"Collections and Data Structures","title":"Base.get","category":"function","text":"get(f::Union{Function, Type}, collection, key)\n\nReturn the value stored for the given key, or if no mapping for the key is present, return f().  Use get! to also store the default value in the dictionary.\n\nThis is intended to be called using do block syntax\n\nget(dict, key) do\n    # default value calculated here\n    time()\nend\n\n\n\n\n\nget(collection, key, default)\n\nReturn the value stored for the given key, or the given default value if no mapping for the key is present.\n\ncompat: Julia 1.7\nFor tuples and numbers, this function requires at least Julia 1.7.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2);\n\njulia> get(d, \"a\", 3)\n1\n\njulia> get(d, \"c\", 3)\n3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.get!","page":"Collections and Data Structures","title":"Base.get!","category":"function","text":"get!(f::Union{Function, Type}, collection, key)\n\nReturn the value stored for the given key, or if no mapping for the key is present, store key => f(), and return f().\n\nThis is intended to be called using do block syntax.\n\nExamples\n\njulia> squares = Dict{Int, Int}();\n\njulia> function get_square!(d, i)\n           get!(d, i) do\n               i^2\n           end\n       end\nget_square! (generic function with 1 method)\n\njulia> get_square!(squares, 2)\n4\n\njulia> squares\nDict{Int64, Int64} with 1 entry:\n  2 => 4\n\n\n\n\n\nget!(collection, key, default)\n\nReturn the value stored for the given key, or if no mapping for the key is present, store key => default, and return default.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> get!(d, \"a\", 5)\n1\n\njulia> get!(d, \"d\", 4)\n4\n\njulia> d\nDict{String, Int64} with 4 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n  \"d\" => 4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.getkey","page":"Collections and Data Structures","title":"Base.getkey","category":"function","text":"getkey(collection, key, default)\n\nReturn the key matching argument key if one exists in collection, otherwise return default.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> getkey(D, 'a', 1)\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> getkey(D, 'd', 'a')\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"location":"base/collections.html#Base.delete!","page":"Collections and Data Structures","title":"Base.delete!","category":"function","text":"delete!(collection, key)\n\nDelete the mapping for the given key in a collection, if any, and return the collection.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2)\nDict{String, Int64} with 2 entries:\n  \"b\" => 2\n  \"a\" => 1\n\njulia> delete!(d, \"b\")\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\njulia> delete!(d, \"b\") # d is left unchanged\nDict{String, Int64} with 1 entry:\n  \"a\" => 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.pop!-Tuple{Any, Any, Any}","page":"Collections and Data Structures","title":"Base.pop!","category":"method","text":"pop!(collection, key[, default])\n\nDelete and return the mapping for key if it exists in collection, otherwise return default, or throw an error if default is not specified.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> pop!(d, \"a\")\n1\n\njulia> pop!(d, \"d\")\nERROR: KeyError: key \"d\" not found\nStacktrace:\n[...]\n\njulia> pop!(d, \"e\", 4)\n4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.keys","page":"Collections and Data Structures","title":"Base.keys","category":"function","text":"keys(iterator)\n\nFor an iterator or collection that has keys and values (e.g. arrays and dictionaries), return an iterator over the keys.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.values","page":"Collections and Data Structures","title":"Base.values","category":"function","text":"values(a::AbstractDict)\n\nReturn an iterator over all values in a collection. collect(values(a)) returns an array of values. When the values are stored internally in a hash table, as is the case for Dict, the order in which they are returned may vary. But keys(a), values(a) and pairs(a) all iterate a and return the elements in the same order.\n\nExamples\n\njulia> D = Dict('a'=>2, 'b'=>3)\nDict{Char, Int64} with 2 entries:\n  'a' => 2\n  'b' => 3\n\njulia> collect(values(D))\n2-element Vector{Int64}:\n 2\n 3\n\n\n\n\n\nvalues(iterator)\n\nFor an iterator or collection that has keys and values, return an iterator over the values. This function simply returns its argument by default, since the elements of a general iterator are normally considered its \"values\".\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2);\n\njulia> values(d)\nValueIterator for a Dict{String, Int64} with 2 entries. Values:\n  2\n  1\n\njulia> values([2])\n1-element Vector{Int64}:\n 2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.pairs","page":"Collections and Data Structures","title":"Base.pairs","category":"function","text":"pairs(collection)\n\nReturn an iterator over key => value pairs for any collection that maps a set of keys to a set of values. This includes arrays, where the keys are the array indices. When the entries are stored internally in a hash table, as is the case for Dict, the order in which they are returned may vary. But keys(a), values(a) and pairs(a) all iterate a and return the elements in the same order.\n\nExamples\n\njulia> a = Dict(zip([\"a\", \"b\", \"c\"], [1, 2, 3]))\nDict{String, Int64} with 3 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n\njulia> pairs(a)\nDict{String, Int64} with 3 entries:\n  \"c\" => 3\n  \"b\" => 2\n  \"a\" => 1\n\njulia> foreach(println, pairs([\"a\", \"b\", \"c\"]))\n1 => \"a\"\n2 => \"b\"\n3 => \"c\"\n\njulia> (;a=1, b=2, c=3) |> pairs |> collect\n3-element Vector{Pair{Symbol, Int64}}:\n :a => 1\n :b => 2\n :c => 3\n\njulia> (;a=1, b=2, c=3) |> collect\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\npairs(IndexLinear(), A)\npairs(IndexCartesian(), A)\npairs(IndexStyle(A), A)\n\nAn iterator that accesses each element of the array A, returning i => x, where i is the index for the element and x = A[i]. Identical to pairs(A), except that the style of index can be selected. Also similar to enumerate(A), except i will be a valid index for A, while enumerate always counts from 1 regardless of the indices of A.\n\nSpecifying IndexLinear() ensures that i will be an integer; specifying IndexCartesian() ensures that i will be a Base.CartesianIndex; specifying IndexStyle(A) chooses whichever has been defined as the native indexing style for array A.\n\nMutation of the bounds of the underlying array will invalidate this iterator.\n\nExamples\n\njulia> A = [\"a\" \"d\"; \"b\" \"e\"; \"c\" \"f\"];\n\njulia> for (index, value) in pairs(IndexStyle(A), A)\n           println(\"$index $value\")\n       end\n1 a\n2 b\n3 c\n4 d\n5 e\n6 f\n\njulia> S = view(A, 1:2, :);\n\njulia> for (index, value) in pairs(IndexStyle(S), S)\n           println(\"$index $value\")\n       end\nCartesianIndex(1, 1) a\nCartesianIndex(2, 1) b\nCartesianIndex(1, 2) d\nCartesianIndex(2, 2) e\n\nSee also IndexStyle, axes.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.merge","page":"Collections and Data Structures","title":"Base.merge","category":"function","text":"merge(a::NamedTuple, iterable)\n\nInterpret an iterable of key-value pairs as a named tuple, and perform a merge.\n\njulia> merge((a=1, b=2, c=3), [:b=>4, :d=>5])\n(a = 1, b = 4, c = 3, d = 5)\n\n\n\n\n\nmerge(a::NamedTuple, bs::NamedTuple...)\n\nConstruct a new named tuple by merging two or more existing ones, in a left-associative manner. Merging proceeds left-to-right, between pairs of named tuples, and so the order of fields present in both the leftmost and rightmost named tuples take the same position as they are found in the leftmost named tuple. However, values are taken from matching fields in the rightmost named tuple that contains that field. Fields present in only the rightmost named tuple of a pair are appended at the end. A fallback is implemented for when only a single named tuple is supplied, with signature merge(a::NamedTuple).\n\ncompat: Julia 1.1\nMerging 3 or more NamedTuple requires at least Julia 1.1.\n\nExamples\n\njulia> merge((a=1, b=2, c=3), (b=4, d=5))\n(a = 1, b = 4, c = 3, d = 5)\n\njulia> merge((a=1, b=2), (b=3, c=(d=1,)), (c=(d=2,),))\n(a = 1, b = 3, c = (d = 2,))\n\n\n\n\n\nmerge(d::AbstractDict, others::AbstractDict...)\n\nConstruct a merged collection from the given collections. If necessary, the types of the resulting collection will be promoted to accommodate the types of the merged collections. If the same key is present in another collection, the value for that key will be the value it has in the last collection listed. See also mergewith for custom handling of values with the same key.\n\nExamples\n\njulia> a = Dict(\"foo\" => 0.0, \"bar\" => 42.0)\nDict{String, Float64} with 2 entries:\n  \"bar\" => 42.0\n  \"foo\" => 0.0\n\njulia> b = Dict(\"baz\" => 17, \"bar\" => 4711)\nDict{String, Int64} with 2 entries:\n  \"bar\" => 4711\n  \"baz\" => 17\n\njulia> merge(a, b)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 4711.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\njulia> merge(b, a)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 42.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\n\n\n\n\nmerge(initial::StyledStrings.Face, others::StyledStrings.Face...)\n\nMerge the properties of the initial face and others, with later faces taking priority.\n\nThis is used to combine the styles of multiple faces, and to resolve inheritance.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.mergewith","page":"Collections and Data Structures","title":"Base.mergewith","category":"function","text":"mergewith(combine, d::AbstractDict, others::AbstractDict...)\nmergewith(combine)\nmerge(combine, d::AbstractDict, others::AbstractDict...)\n\nConstruct a merged collection from the given collections. If necessary, the types of the resulting collection will be promoted to accommodate the types of the merged collections. Values with the same key will be combined using the combiner function.  The curried form mergewith(combine) returns the function (args...) -> mergewith(combine, args...).\n\nMethod merge(combine::Union{Function,Type}, args...) as an alias of mergewith(combine, args...) is still available for backward compatibility.\n\ncompat: Julia 1.5\nmergewith requires Julia 1.5 or later.\n\nExamples\n\njulia> a = Dict(\"foo\" => 0.0, \"bar\" => 42.0)\nDict{String, Float64} with 2 entries:\n  \"bar\" => 42.0\n  \"foo\" => 0.0\n\njulia> b = Dict(\"baz\" => 17, \"bar\" => 4711)\nDict{String, Int64} with 2 entries:\n  \"bar\" => 4711\n  \"baz\" => 17\n\njulia> mergewith(+, a, b)\nDict{String, Float64} with 3 entries:\n  \"bar\" => 4753.0\n  \"baz\" => 17.0\n  \"foo\" => 0.0\n\njulia> ans == mergewith(+)(a, b)\ntrue\n\njulia> mergewith(-, Dict(), Dict(:a=>1))  # Combining function only used if key is present in both\nDict{Any, Any} with 1 entry:\n  :a => 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.merge!","page":"Collections and Data Structures","title":"Base.merge!","category":"function","text":"merge!(d::AbstractDict, others::AbstractDict...)\n\nUpdate collection with pairs from the other collections. See also merge.\n\nExamples\n\njulia> d1 = Dict(1 => 2, 3 => 4);\n\njulia> d2 = Dict(1 => 4, 4 => 5);\n\njulia> merge!(d1, d2);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 4\n  1 => 4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.mergewith!","page":"Collections and Data Structures","title":"Base.mergewith!","category":"function","text":"mergewith!(combine, d::AbstractDict, others::AbstractDict...) -> d\nmergewith!(combine)\nmerge!(combine, d::AbstractDict, others::AbstractDict...) -> d\n\nUpdate collection with pairs from the other collections. Values with the same key will be combined using the combiner function.  The curried form mergewith!(combine) returns the function (args...) -> mergewith!(combine, args...).\n\nMethod merge!(combine::Union{Function,Type}, args...) as an alias of mergewith!(combine, args...) is still available for backward compatibility.\n\ncompat: Julia 1.5\nmergewith! requires Julia 1.5 or later.\n\nExamples\n\njulia> d1 = Dict(1 => 2, 3 => 4);\n\njulia> d2 = Dict(1 => 4, 4 => 5);\n\njulia> mergewith!(+, d1, d2);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 4\n  1 => 6\n\njulia> mergewith!(-, d1, d1);\n\njulia> d1\nDict{Int64, Int64} with 3 entries:\n  4 => 0\n  3 => 0\n  1 => 0\n\njulia> foldl(mergewith!(+), [d1, d2]; init=Dict{Int64, Int64}())\nDict{Int64, Int64} with 3 entries:\n  4 => 5\n  3 => 0\n  1 => 4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.sizehint!","page":"Collections and Data Structures","title":"Base.sizehint!","category":"function","text":"sizehint!(s, n; first::Bool=false, shrink::Bool=true) -> s\n\nSuggest that collection s reserve capacity for at least n elements. That is, if you expect that you're going to have to push a lot of values onto s, you can avoid the cost of incremental reallocation by doing it once up front; this can improve performance.\n\nIf first is true, then any additional space is reserved before the start of the collection. This way, subsequent calls to pushfirst! (instead of push!) may become faster. Supplying this keyword may result in an error if the collection is not ordered or if pushfirst! is not supported for this collection.\n\nIf shrink=true (the default), the collection's capacity may be reduced if its current capacity is greater than n.\n\nSee also resize!.\n\nNotes on the performance model\n\nFor types that support sizehint!,\n\npush! and append! methods generally may (but are not required to) preallocate extra storage. For types implemented in Base, they typically do, using a heuristic optimized for a general use case.\nsizehint! may control this preallocation. Again, it typically does this for types in Base.\nempty! is nearly costless (and O(1)) for types that support this kind of preallocation.\n\ncompat: Julia 1.11\nThe shrink and first arguments were added in Julia 1.11.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.keytype","page":"Collections and Data Structures","title":"Base.keytype","category":"function","text":"keytype(type)\n\nGet the key type of a dictionary type. Behaves similarly to eltype.\n\nExamples\n\njulia> keytype(Dict(Int32(1) => \"foo\"))\nInt32\n\n\n\n\n\nkeytype(T::Type{<:AbstractArray})\nkeytype(A::AbstractArray)\n\nReturn the key type of an array. This is equal to the eltype of the result of keys(...), and is provided mainly for compatibility with the dictionary interface.\n\nExamples\n\njulia> keytype([1, 2, 3]) == Int\ntrue\n\njulia> keytype([1 2; 3 4])\nCartesianIndex{2}\n\ncompat: Julia 1.2\nFor arrays, this function requires at least Julia 1.2.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.valtype","page":"Collections and Data Structures","title":"Base.valtype","category":"function","text":"valtype(type)\n\nGet the value type of a dictionary type. Behaves similarly to eltype.\n\nExamples\n\njulia> valtype(Dict(Int32(1) => \"foo\"))\nString\n\n\n\n\n\nvaltype(T::Type{<:AbstractArray})\nvaltype(A::AbstractArray)\n\nReturn the value type of an array. This is identical to eltype and is provided mainly for compatibility with the dictionary interface.\n\nExamples\n\njulia> valtype([\"one\", \"two\", \"three\"])\nString\n\ncompat: Julia 1.2\nFor arrays, this function requires at least Julia 1.2.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.AbstractSet","page":"Collections and Data Structures","title":"Base.AbstractSet","category":"type","text":"AbstractSet{T}\n\nSupertype for set-like types whose elements are of type T. Set, BitSet and other types are subtypes of this.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.Set","page":"Collections and Data Structures","title":"Base.Set","category":"type","text":"Set{T} <: AbstractSet{T}\n\nSets are mutable containers that provide fast membership testing.\n\nSets have efficient implementations of set operations such as in, union and intersect. Elements in a Set are unique, as determined by the elements' definition of isequal. The order of elements in a Set is an implementation detail and cannot be relied on.\n\nSee also: AbstractSet, BitSet, Dict, push!, empty!, union!, in, isequal\n\nExamples\n\njulia> s = Set(\"aaBca\")\nSet{Char} with 3 elements:\n  'a'\n  'c'\n  'B'\n\njulia> push!(s, 'b')\nSet{Char} with 4 elements:\n  'a'\n  'b'\n  'B'\n  'c'\n\njulia> s = Set([NaN, 0.0, 1.0, 2.0]);\n\njulia> -0.0 in s # isequal(0.0, -0.0) is false\nfalse\n\njulia> NaN in s # isequal(NaN, NaN) is true\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.BitSet","page":"Collections and Data Structures","title":"Base.BitSet","category":"type","text":"BitSet([itr])\n\nConstruct a sorted set of Ints generated by the given iterable object, or an empty set. Implemented as a bit string, and therefore designed for dense integer sets. If the set will be sparse (for example, holding a few very large integers), use Set instead.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.IdSet","page":"Collections and Data Structures","title":"Base.IdSet","category":"type","text":"IdSet{T}([itr])\nIdSet()\n\nIdSet{T}() constructs a set (see Set) using === as equality with values of type T.\n\nIn the example below, the values are all isequal so they get overwritten in the ordinary Set. The IdSet compares by === and so preserves the 3 different values.\n\ncompat: Julia 1.11\nExported in Julia 1.11 and later.\n\nExamples\n\njulia> Set(Any[true, 1, 1.0])\nSet{Any} with 1 element:\n  1.0\n\njulia> IdSet{Any}(Any[true, 1, 1.0])\nIdSet{Any} with 3 elements:\n  1.0\n  1\n  true\n\n\n\n\n\n"},{"location":"base/collections.html#Base.union","page":"Collections and Data Structures","title":"Base.union","category":"function","text":"union(s, itrs...)\n∪(s, itrs...)\n\nConstruct an object containing all distinct elements from all of the arguments.\n\nThe first argument controls what kind of container is returned. If this is an array, it maintains the order in which elements first appear.\n\nUnicode ∪ can be typed by writing \\cup then pressing tab in the Julia REPL, and in many editors. This is an infix operator, allowing s ∪ itr.\n\nSee also unique, intersect, isdisjoint, vcat, Iterators.flatten.\n\nExamples\n\njulia> union([1, 2], [3])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> union([4 2 3 4 4], 1:3, 3.0)\n4-element Vector{Float64}:\n 4.0\n 2.0\n 3.0\n 1.0\n\njulia> (0, 0.0) ∪ (-0.0, NaN)\n3-element Vector{Real}:\n   0\n  -0.0\n NaN\n\njulia> union(Set([1, 2]), 2:3)\nSet{Int64} with 3 elements:\n  2\n  3\n  1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.union!","page":"Collections and Data Structures","title":"Base.union!","category":"function","text":"union!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nConstruct the union of passed in sets and overwrite s with the result. Maintain order with arrays.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> a = Set([3, 4, 5]);\n\njulia> union!(a, 1:2:7);\n\njulia> a\nSet{Int64} with 5 elements:\n  5\n  4\n  7\n  3\n  1\n\n\n\n\n\nunion!(s::IntDisjointSet{T}, x::T, y::T)\n\nMerge the subset containing x and that containing y into one and return the root of the new set.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.intersect","page":"Collections and Data Structures","title":"Base.intersect","category":"function","text":"intersect(s, itrs...)\n∩(s, itrs...)\n\nConstruct the set containing those elements which appear in all of the arguments.\n\nThe first argument controls what kind of container is returned. If this is an array, it maintains the order in which elements first appear.\n\nUnicode ∩ can be typed by writing \\cap then pressing tab in the Julia REPL, and in many editors. This is an infix operator, allowing s ∩ itr.\n\nSee also setdiff, isdisjoint, issubset, issetequal.\n\ncompat: Julia 1.8\nAs of Julia 1.8 intersect returns a result with the eltype of the type-promoted eltypes of the two inputs\n\nExamples\n\njulia> intersect([1, 2, 3], [3, 4, 5])\n1-element Vector{Int64}:\n 3\n\njulia> intersect([1, 4, 4, 5, 6], [6, 4, 6, 7, 8])\n2-element Vector{Int64}:\n 4\n 6\n\njulia> intersect(1:16, 7:99)\n7:16\n\njulia> (0, 0.0) ∩ (-0.0, 0)\n1-element Vector{Real}:\n 0\n\njulia> intersect(Set([1, 2]), BitSet([2, 3]), 1.0:10.0)\nSet{Float64} with 1 element:\n  2.0\n\n\n\n\n\n"},{"location":"base/collections.html#Base.setdiff","page":"Collections and Data Structures","title":"Base.setdiff","category":"function","text":"setdiff(s, itrs...)\n\nConstruct the set of elements in s but not in any of the iterables in itrs. Maintain order with arrays. The result will have the same element type as s.\n\nSee also setdiff!, union and intersect.\n\nExamples\n\njulia> setdiff([1,2,3], [3,4,5])\n2-element Vector{Int64}:\n 1\n 2\n\njulia> setdiff([1,2,3], [1.0, 2.0])\n1-element Vector{Int64}:\n 3\n\n\n\n\n\n"},{"location":"base/collections.html#Base.setdiff!","page":"Collections and Data Structures","title":"Base.setdiff!","category":"function","text":"setdiff!(s, itrs...)\n\nRemove from set s (in-place) each element of each iterable from itrs. Maintain order with arrays.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\nExamples\n\njulia> a = Set([1, 3, 4, 5]);\n\njulia> setdiff!(a, 1:2:6);\n\njulia> a\nSet{Int64} with 1 element:\n  4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.symdiff","page":"Collections and Data Structures","title":"Base.symdiff","category":"function","text":"symdiff(s, itrs...)\n\nConstruct the symmetric difference of elements in the passed in sets. When s is not an AbstractSet, the order is maintained.\n\nSee also symdiff!, setdiff, union and intersect.\n\nExamples\n\njulia> symdiff([1,2,3], [3,4,5], [4,5,6])\n3-element Vector{Int64}:\n 1\n 2\n 6\n\njulia> symdiff([1,2,1], [2, 1, 2])\nInt64[]\n\n\n\n\n\n"},{"location":"base/collections.html#Base.symdiff!","page":"Collections and Data Structures","title":"Base.symdiff!","category":"function","text":"symdiff!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nConstruct the symmetric difference of the passed in sets, and overwrite s with the result. When s is an array, the order is maintained. Note that in this case the multiplicity of elements matters.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.intersect!","page":"Collections and Data Structures","title":"Base.intersect!","category":"function","text":"intersect!(s::Union{AbstractSet,AbstractVector}, itrs...)\n\nIntersect all passed in sets and overwrite s with the result. Maintain order with arrays.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.issubset","page":"Collections and Data Structures","title":"Base.issubset","category":"function","text":"issubset(a, b)::Bool\n⊆(a, b)::Bool\n⊇(b, a)::Bool\n\nDetermine whether every element of a is also in b, using in.\n\nSee also ⊊, ⊈, ∩, ∪, contains.\n\nExamples\n\njulia> issubset([1, 2], [1, 2, 3])\ntrue\n\njulia> [1, 2, 3] ⊆ [1, 2]\nfalse\n\njulia> [1, 2, 3] ⊇ [1, 2]\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.in!","page":"Collections and Data Structures","title":"Base.in!","category":"function","text":"in!(x, s::AbstractSet)::Bool\n\nIf x is in s, return true. If not, push x into s and return false. This is equivalent to in(x, s) ? true : (push!(s, x); false), but may have a more efficient implementation.\n\nSee also: in, push!, Set\n\ncompat: Julia 1.11\nThis function requires at least 1.11.\n\nExamples\n\njulia> s = Set{Any}([1, 2, 3]); in!(4, s)\nfalse\n\njulia> length(s)\n4\n\njulia> in!(0x04, s)\ntrue\n\njulia> s\nSet{Any} with 4 elements:\n  4\n  2\n  3\n  1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.:⊈","page":"Collections and Data Structures","title":"Base.:⊈","category":"function","text":"⊈(a, b)::Bool\n⊉(b, a)::Bool\n\nNegation of ⊆ and ⊇, i.e. checks that a is not a subset of b.\n\nSee also issubset (⊆), ⊊.\n\nExamples\n\njulia> (1, 2) ⊈ (2, 3)\ntrue\n\njulia> (1, 2) ⊈ (1, 2, 3)\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.:⊊","page":"Collections and Data Structures","title":"Base.:⊊","category":"function","text":"⊊(a, b)::Bool\n⊋(b, a)::Bool\n\nDetermines if a is a subset of, but not equal to, b.\n\nSee also issubset (⊆), ⊈.\n\nExamples\n\njulia> (1, 2) ⊊ (1, 2, 3)\ntrue\n\njulia> (1, 2) ⊊ (1, 2)\nfalse\n\n\n\n\n\n"},{"location":"base/collections.html#Base.issetequal","page":"Collections and Data Structures","title":"Base.issetequal","category":"function","text":"issetequal(x)\n\nCreate a function that compares its argument to x using issetequal, i.e. a function equivalent to y -> issetequal(y, x). The returned function is of type Base.Fix2{typeof(issetequal)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.11\nThis functionality requires at least Julia 1.11.\n\n\n\n\n\nissetequal(a, b)::Bool\n\nDetermine whether a and b have the same elements. Equivalent to a ⊆ b && b ⊆ a but more efficient when possible.\n\nSee also: isdisjoint, union.\n\nExamples\n\njulia> issetequal([1, 2], [1, 2, 3])\nfalse\n\njulia> issetequal([1, 2], [2, 1])\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.isdisjoint","page":"Collections and Data Structures","title":"Base.isdisjoint","category":"function","text":"isdisjoint(x)\n\nCreate a function that compares its argument to x using isdisjoint, i.e. a function equivalent to y -> isdisjoint(y, x). The returned function is of type Base.Fix2{typeof(isdisjoint)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.11\nThis functionality requires at least Julia 1.11.\n\n\n\n\n\nisdisjoint(a, b)::Bool\n\nDetermine whether the collections a and b are disjoint. Equivalent to isempty(a ∩ b) but more efficient when possible.\n\nSee also: intersect, isempty, issetequal.\n\ncompat: Julia 1.5\nThis function requires at least Julia 1.5.\n\nExamples\n\njulia> isdisjoint([1, 2], [2, 3, 4])\nfalse\n\njulia> isdisjoint([3, 1], [2, 4])\ntrue\n\n\n\n\n\n"},{"location":"base/collections.html#Base.push!","page":"Collections and Data Structures","title":"Base.push!","category":"function","text":"push!(collection, items...) -> collection\n\nInsert one or more items in collection. If collection is an ordered container, the items are inserted at the end (in the given order).\n\nExamples\n\njulia> push!([1, 2, 3], 4, 5, 6)\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\nIf collection is ordered, use append! to add all the elements of another collection to it. The result of the preceding example is equivalent to append!([1, 2, 3], [4, 5, 6]). For AbstractSet objects, union! can be used instead.\n\nSee sizehint! for notes about the performance model.\n\nSee also pushfirst!.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.pop!","page":"Collections and Data Structures","title":"Base.pop!","category":"function","text":"pop!(collection, key[, default])\n\nDelete and return the mapping for key if it exists in collection, otherwise return default, or throw an error if default is not specified.\n\nExamples\n\njulia> d = Dict(\"a\"=>1, \"b\"=>2, \"c\"=>3);\n\njulia> pop!(d, \"a\")\n1\n\njulia> pop!(d, \"d\")\nERROR: KeyError: key \"d\" not found\nStacktrace:\n[...]\n\njulia> pop!(d, \"e\", 4)\n4\n\n\n\n\n\npop!(collection) -> item\n\nRemove an item in collection and return it. If collection is an ordered container, the last item is returned; for unordered containers, an arbitrary element is returned.\n\nSee also: popfirst!, popat!, delete!, deleteat!, splice!, and push!.\n\nExamples\n\njulia> A=[1, 2, 3]\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> pop!(A)\n3\n\njulia> A\n2-element Vector{Int64}:\n 1\n 2\n\njulia> S = Set([1, 2])\nSet{Int64} with 2 elements:\n  2\n  1\n\njulia> pop!(S)\n2\n\njulia> S\nSet{Int64} with 1 element:\n  1\n\njulia> pop!(Dict(1=>2))\n1 => 2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.popat!","page":"Collections and Data Structures","title":"Base.popat!","category":"function","text":"popat!(a::Vector, i::Integer, [default])\n\nRemove the item at the given i and return it. Subsequent items are shifted to fill the resulting gap. When i is not a valid index for a, return default, or throw an error if default is not specified.\n\nSee also: pop!, popfirst!, deleteat!, splice!.\n\ncompat: Julia 1.5\nThis function is available as of Julia 1.5.\n\nExamples\n\njulia> a = [4, 3, 2, 1]; popat!(a, 2)\n3\n\njulia> a\n3-element Vector{Int64}:\n 4\n 2\n 1\n\njulia> popat!(a, 4, missing)\nmissing\n\njulia> popat!(a, 4)\nERROR: BoundsError: attempt to access 3-element Vector{Int64} at index [4]\n[...]\n\n\n\n\n\n"},{"location":"base/collections.html#Base.pushfirst!","page":"Collections and Data Structures","title":"Base.pushfirst!","category":"function","text":"pushfirst!(collection, items...) -> collection\n\nInsert one or more items at the beginning of collection.\n\nThis function is called unshift in many other programming languages.\n\nExamples\n\njulia> pushfirst!([1, 2, 3, 4], 5, 6)\n6-element Vector{Int64}:\n 5\n 6\n 1\n 2\n 3\n 4\n\n\n\n\n\n"},{"location":"base/collections.html#Base.popfirst!","page":"Collections and Data Structures","title":"Base.popfirst!","category":"function","text":"popfirst!(collection) -> item\n\nRemove the first item from collection.\n\nThis function is called shift in many other programming languages.\n\nSee also: pop!, popat!, delete!.\n\nExamples\n\njulia> A = [1, 2, 3, 4, 5, 6]\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\njulia> popfirst!(A)\n1\n\njulia> A\n5-element Vector{Int64}:\n 2\n 3\n 4\n 5\n 6\n\n\n\n\n\n"},{"location":"base/collections.html#Base.insert!","page":"Collections and Data Structures","title":"Base.insert!","category":"function","text":"insert!(a::Vector, index::Integer, item)\n\nInsert an item into a at the given index. index is the index of item in the resulting a.\n\nSee also: push!, replace, popat!, splice!.\n\nExamples\n\njulia> insert!(Any[1:6;], 3, \"here\")\n7-element Vector{Any}:\n 1\n 2\n  \"here\"\n 3\n 4\n 5\n 6\n\n\n\n\n\n"},{"location":"base/collections.html#Base.deleteat!","page":"Collections and Data Structures","title":"Base.deleteat!","category":"function","text":"deleteat!(a::Vector, inds)\n\nRemove the items at the indices given by inds, and return the modified a. Subsequent items are shifted to fill the resulting gap.\n\ninds can be either an iterator or a collection of sorted and unique integer indices, or a boolean vector of the same length as a with true indicating entries to delete.\n\nExamples\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], 1:2:5)\n3-element Vector{Int64}:\n 5\n 3\n 1\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], [true, false, true, false, true, false])\n3-element Vector{Int64}:\n 5\n 3\n 1\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], (2, 2))\nERROR: ArgumentError: indices must be unique and sorted\nStacktrace:\n[...]\n\n\n\n\n\ndeleteat!(a::Vector, i::Integer)\n\nRemove the item at the given i and return the modified a. Subsequent items are shifted to fill the resulting gap.\n\nSee also: keepat!, delete!, popat!, splice!.\n\nExamples\n\njulia> deleteat!([6, 5, 4, 3, 2, 1], 2)\n5-element Vector{Int64}:\n 6\n 4\n 3\n 2\n 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.keepat!","page":"Collections and Data Structures","title":"Base.keepat!","category":"function","text":"keepat!(a::Vector, m::AbstractVector{Bool})\nkeepat!(a::BitVector, m::AbstractVector{Bool})\n\nThe in-place version of logical indexing a = a[m]. That is, keepat!(a, m) on vectors of equal length a and m will remove all elements from a for which m at the corresponding index is false.\n\nExamples\n\njulia> a = [:a, :b, :c];\n\njulia> keepat!(a, [true, false, true])\n2-element Vector{Symbol}:\n :a\n :c\n\njulia> a\n2-element Vector{Symbol}:\n :a\n :c\n\n\n\n\n\nkeepat!(a::Vector, inds)\nkeepat!(a::BitVector, inds)\n\nRemove the items at all the indices which are not given by inds, and return the modified a. Items which are kept are shifted to fill the resulting gaps.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\ninds must be an iterator of sorted and unique integer indices. See also deleteat!.\n\ncompat: Julia 1.7\nThis function is available as of Julia 1.7.\n\nExamples\n\njulia> keepat!([6, 5, 4, 3, 2, 1], 1:2:5)\n3-element Vector{Int64}:\n 6\n 4\n 2\n\n\n\n\n\n"},{"location":"base/collections.html#Base.splice!","page":"Collections and Data Structures","title":"Base.splice!","category":"function","text":"splice!(a::Vector, indices, [replacement]) -> items\n\nRemove items at specified indices, and return a collection containing the removed items. Subsequent items are shifted left to fill the resulting gaps. If specified, replacement values from an ordered collection will be spliced in place of the removed items; in this case, indices must be a AbstractUnitRange.\n\nTo insert replacement before an index n without removing any items, use splice!(collection, n:n-1, replacement).\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\ncompat: Julia 1.5\nPrior to Julia 1.5, indices must always be a UnitRange.\n\ncompat: Julia 1.8\nPrior to Julia 1.8, indices must be a UnitRange if splicing in replacement values.\n\nExamples\n\njulia> A = [-1, -2, -3, 5, 4, 3, -1]; splice!(A, 4:3, 2)\nInt64[]\n\njulia> A\n8-element Vector{Int64}:\n -1\n -2\n -3\n  2\n  5\n  4\n  3\n -1\n\n\n\n\n\nsplice!(a::Vector, index::Integer, [replacement]) -> item\n\nRemove the item at the given index, and return the removed item. Subsequent items are shifted left to fill the resulting gap. If specified, replacement values from an ordered collection will be spliced in place of the removed item.\n\nSee also: replace, delete!, deleteat!, pop!, popat!.\n\nExamples\n\njulia> A = [6, 5, 4, 3, 2, 1]; splice!(A, 5)\n2\n\njulia> A\n5-element Vector{Int64}:\n 6\n 5\n 4\n 3\n 1\n\njulia> splice!(A, 5, -1)\n1\n\njulia> A\n5-element Vector{Int64}:\n  6\n  5\n  4\n  3\n -1\n\njulia> splice!(A, 1, [-1, -2, -3])\n6\n\njulia> A\n7-element Vector{Int64}:\n -1\n -2\n -3\n  5\n  4\n  3\n -1\n\nTo insert replacement before an index n without removing any items, use splice!(collection, n:n-1, replacement).\n\n\n\n\n\n"},{"location":"base/collections.html#Base.resize!","page":"Collections and Data Structures","title":"Base.resize!","category":"function","text":"resize!(a::Vector, n::Integer) -> a\n\nResize a to contain n elements. If n is smaller than the current collection length, the first n elements will be retained. If n is larger, the new elements are not guaranteed to be initialized.\n\nExamples\n\njulia> resize!([6, 5, 4, 3, 2, 1], 3)\n3-element Vector{Int64}:\n 6\n 5\n 4\n\njulia> a = resize!([6, 5, 4, 3, 2, 1], 8);\n\njulia> length(a)\n8\n\njulia> a[1:6]\n6-element Vector{Int64}:\n 6\n 5\n 4\n 3\n 2\n 1\n\n\n\n\n\n"},{"location":"base/collections.html#Base.append!","page":"Collections and Data Structures","title":"Base.append!","category":"function","text":"append!(collection, collections...) -> collection.\n\nFor an ordered container collection, add the elements of each collections to the end of it.\n\ncompat: Julia 1.6\nSpecifying multiple collections to be appended requires at least Julia 1.6.\n\nExamples\n\njulia> append!([1], [2, 3])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> append!([1, 2, 3], [4, 5], [6])\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\nUse push! to add individual items to collection which are not already themselves in another collection. The result of the preceding example is equivalent to push!([1, 2, 3], 4, 5, 6).\n\nSee sizehint! for notes about the performance model.\n\nSee also vcat for vectors, union! for sets, and prepend! and pushfirst! for the opposite order.\n\n\n\n\n\n"},{"location":"base/collections.html#Base.prepend!","page":"Collections and Data Structures","title":"Base.prepend!","category":"function","text":"prepend!(a::Vector, collections...) -> collection\n\nInsert the elements of each collections to the beginning of a.\n\nWhen collections specifies multiple collections, order is maintained: elements of collections[1] will appear leftmost in a, and so on.\n\ncompat: Julia 1.6\nSpecifying multiple collections to be prepended requires at least Julia 1.6.\n\nExamples\n\njulia> prepend!([3], [1, 2])\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> prepend!([6], [1, 2], [3, 4, 5])\n6-element Vector{Int64}:\n 1\n 2\n 3\n 4\n 5\n 6\n\n\n\n\n\n"},{"location":"base/collections.html#Core.Pair","page":"Collections and Data Structures","title":"Core.Pair","category":"type","text":"Pair(x, y)\nx => y\n\nConstruct a Pair object with type Pair{typeof(x), typeof(y)}. The elements are stored in the fields first and second. They can also be accessed via iteration (but a Pair is treated as a single \"scalar\" for broadcasting operations).\n\nSee also Dict.\n\nExamples\n\njulia> p = \"foo\" => 7\n\"foo\" => 7\n\njulia> typeof(p)\nPair{String, Int64}\n\njulia> p.first\n\"foo\"\n\njulia> for x in p\n           println(x)\n       end\nfoo\n7\n\njulia> replace.([\"xops\", \"oxps\"], \"x\" => \"o\")\n2-element Vector{String}:\n \"oops\"\n \"oops\"\n\n\n\n\n\n"},{"location":"base/collections.html#Base.Pairs","page":"Collections and Data Structures","title":"Base.Pairs","category":"type","text":"Base.Pairs(values, keys) <: AbstractDict{eltype(keys), eltype(values)}\n\nTransforms an indexable container into a Dictionary-view of the same data. Modifying the key-space of the underlying data may invalidate this object.\n\n\n\n\n\n"},{"location":"manual/variables-and-scoping.html#scope-of-variables","page":"Scope of Variables","title":"Scope of Variables","category":"section","text":"The scope of a variable is the region of code within which a variable is accessible. Variable scoping helps avoid variable naming conflicts. The concept is intuitive: two functions can both have arguments called x without the two x's referring to the same thing. Similarly, there are many other cases where different blocks of code can use the same name without referring to the same thing. The rules for when the same variable name does or doesn't refer to the same thing are called scope rules; this section spells them out in detail.\n\nCertain constructs in the language introduce scope blocks, which are regions of code that are eligible to be the scope of some set of variables. The scope of a variable cannot be an arbitrary set of source lines; instead, it will always line up with one of these blocks. There are two main types of scopes in Julia, global scope and local scope. The latter can be nested. There is also a distinction in Julia between constructs which introduce a \"hard scope\" and those which only introduce a \"soft scope\", which affects whether shadowing a global variable by the same name is allowed or not.\n\ninfo: Summary\nVariables defined in global scope may be undefined in inner local scopes, depending on where the code is run, in order to balance safety and convenience. The hard and soft local scoping rules define the interplay between global and local variables.However, variables defined only in local scope behave consistently in all contexts. If the variable is already defined, it will be reused. If the variable is not defined, it will be made available to the current and inner scopes (but not outer scopes).\n\ntip: A Common Confusion\nIf you run into an unexpectedly undefined variable,# Print the numbers 1 through 5\ni = 0\nwhile i < 5\n    i += 1     # ERROR: UndefVarError: `i` not defined\n    println(i)\nenda simple fix is to change all global variable definitions into local definitions by wrapping the code in a let block or function.# Print the numbers 1 through 5\nlet i = 0\n    while i < 5\n        i += 1     # Now outer `i` is defined in the inner scope of the while loop\n        println(i)\n    end\nendThis is a common source of confusion when writing procedural scripts, but it becomes a non-issue if code is moved inside functions or executed interactively in the REPL.See also the global and local keywords to explicitly achieve any desired scoping behavior."},{"location":"manual/variables-and-scoping.html#man-scope-table","page":"Scope of Variables","title":"Scope Constructs","category":"section","text":"The constructs introducing scope blocks are:\n\nConstruct Scope Type Introduced Scope Types Able to Contain Construct\nmodule, baremodule global global\nstruct local (hard) global\nmacro local (hard) global\nfor, while, try local (soft) global, local\nfunction, do, let, comprehensions, generators local (hard) global, local\n\nNotably missing from this table are begin blocks and if blocks which do not introduce new scopes. The three types of scopes follow somewhat different rules which will be explained below.\n\nJulia uses lexical scoping, meaning that a function's scope does not inherit from its caller's scope, but from the scope in which the function was defined. For example, in the following code the x inside foo refers to the x in the global scope of its module Bar:\n\njulia> module Bar\n           x = 1\n           foo() = x\n       end;\n\nand not a x in the scope where foo is used:\n\njulia> import .Bar\n\njulia> x = -1;\n\njulia> Bar.foo()\n1\n\nThus lexical scope means that what a variable in a particular piece of code refers to can be deduced from the code in which it appears alone and does not depend on how the program executes. A scope nested inside another scope can \"see\" variables in all the outer scopes in which it is contained. Outer scopes, on the other hand, cannot see variables in inner scopes."},{"location":"manual/variables-and-scoping.html#Global-Scope","page":"Scope of Variables","title":"Global Scope","category":"section","text":"Each module introduces a new global scope, separate from the global scope of all other modules—there is no all-encompassing global scope. Modules can introduce variables of other modules into their scope through the using or import statements or through qualified access using the dot-notation, i.e. each module is a so-called namespace as well as a first-class data structure associating names with values.\n\nIf a top-level expression contains a variable declaration with keyword local, then that variable is not accessible outside that expression. The variable inside the expression does not affect global variables of the same name. An example is to declare local x in a begin or if block at the top-level:\n\njulia> x = 1\n       begin\n           local x = 0\n           @show x\n       end\n       @show x;\nx = 0\nx = 1\n\nNote that the interactive prompt (aka REPL) is in the global scope of the module Main."},{"location":"manual/variables-and-scoping.html#local-scope","page":"Scope of Variables","title":"Local Scope","category":"section","text":"A new local scope is introduced by most code blocks (see above table for a complete list). If such a block is syntactically nested inside of another local scope, the scope it creates is nested inside of all the local scopes that it appears within, which are all ultimately nested inside of the global scope of the module in which the code is evaluated. Variables in outer scopes are visible from any scope they contain — meaning that they can be read and written in inner scopes — unless there is a variable with the same name that \"shadows\" the outer variable of the same name. This is true even if the outer local is declared after (in the sense of textually below) an inner block. When we say that a variable \"exists\" in a given scope, this means that a variable by that name exists in any of the scopes that the current scope is nested inside of, including the current one. If a variable's value is used in a local scope, but nothing with its name exists in this scope, it is assumed to be a global.\n\nSome programming languages require explicitly declaring new variables before using them. Explicit declaration works in Julia too: in any local scope, writing local x declares a new local variable in that scope, regardless of whether there is already a variable named x in an outer scope or not. Declaring each new variable like this is somewhat verbose and tedious, however, so Julia, like many other languages, considers assignment to a variable name that doesn't already exist to implicitly declare that variable. If the current scope is global, the new variable is global; if the current scope is local, the new variable is local to the innermost local scope and will be visible inside of that scope but not outside of it. If you assign to an existing local, it always updates that existing local: you can only shadow a local by explicitly declaring a new local in a nested scope with the local keyword. In particular, this applies to variables assigned in inner functions, which may surprise users coming from Python where assignment in an inner function creates a new local unless the variable is explicitly declared to be non-local.\n\nMostly this is pretty intuitive, but as with many things that behave intuitively, the details are more subtle than one might naïvely imagine.\n\nWhen x = <value> occurs in a local scope, Julia applies the following rules to decide what the expression means based on where the assignment expression occurs and what x already refers to at that location:\n\nExisting local: If x is already a local variable, then the existing local x is assigned;\nHard scope: If x is not already a local variable and assignment occurs inside of any hard scope construct (i.e. within a let block, function, struct or macro body, comprehension, or generator), a new local named x is created in the scope of the assignment;\nSoft scope: If x is not already a local variable and all of the scope constructs containing the assignment are soft scopes (loops, try/catch blocks), the behavior depends on whether the global variable x is defined:\nif global x is undefined, a new local named x is created in the scope of the assignment;\nif global x is defined, the assignment is considered ambiguous:\nin non-interactive contexts (files, eval), an ambiguity warning is printed and a new local is created;\nin interactive contexts (REPL, notebooks), the global variable x is assigned.\n\nYou may note that in non-interactive contexts the hard and soft scope behaviors are identical except that a warning is printed when an implicitly local variable (i.e. not declared with local x) shadows a global. In interactive contexts, the rules follow a more complex heuristic for the sake of convenience. This is covered in depth in examples that follow.\n\nNow that you know the rules, let's look at some examples. Each example is assumed to be evaluated in a fresh REPL session so that the only globals in each snippet are the ones that are assigned in that block of code.\n\nWe'll begin with a nice and clear-cut situation—assignment inside of a hard scope, in this case a function body, when no local variable by that name already exists:\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\nERROR: UndefVarError: `x` not defined in `Main`\n\nInside of the greet function, the assignment x = \"hello\" causes x to be a new local variable in the function's scope. There are two relevant facts: the assignment occurs in local scope and there is no existing local x variable. Since x is local, it doesn't matter if there is a global named x or not. Here for example we define x = 123 before defining and calling greet:\n\njulia> x = 123 # global\n123\n\njulia> function greet()\n           x = \"hello\" # new local\n           println(x)\n       end\ngreet (generic function with 1 method)\n\njulia> greet()\nhello\n\njulia> x # global\n123\n\nSince the x in greet is local, the value (or lack thereof) of the global x is unaffected by calling greet. The hard scope rule doesn't care whether a global named x exists or not: assignment to x in a hard scope is local (unless x is declared global).\n\nThe next clear cut situation we'll consider is when there is already a local variable named x, in which case x = <value> always assigns to this existing local x. This is true whether the assignment occurs in the same local scope, an inner local scope in the same function body, or in the body of a function nested inside of another function, also known as a closure.\n\nWe'll use the sum_to function, which computes the sum of integers from one up to n, as an example:\n\nfunction sum_to(n)\n    s = 0 # new local\n    for i = 1:n\n        s = s + i # assign existing local\n    end\n    return s # same local\nend\n\nAs in the previous example, the first assignment to s at the top of sum_to causes s to be a new local variable in the body of the function. The for loop has its own inner local scope within the function scope. At the point where s = s + i occurs, s is already a local variable, so the assignment updates the existing s instead of creating a new local. We can test this out by calling sum_to in the REPL:\n\njulia> function sum_to(n)\n           s = 0 # new local\n           for i = 1:n\n               s = s + i # assign existing local\n           end\n           return s # same local\n       end\nsum_to (generic function with 1 method)\n\njulia> sum_to(10)\n55\n\njulia> s # global\nERROR: UndefVarError: `s` not defined in `Main`\n\nSince s is local to the function sum_to, calling the function has no effect on the global variable s. We can also see that the update s = s + i in the for loop must have updated the same s created by the initialization s = 0 since we get the correct sum of 55 for the integers 1 through 10.\n\nLet's dig into the fact that the for loop body has its own scope for a second by writing a slightly more verbose variation which we'll call sum_to_def, in which we save the sum s + i in a variable t before updating s:\n\njulia> function sum_to_def(n)\n           s = 0 # new local\n           for i = 1:n\n               t = s + i # new local `t`\n               s = t # assign existing local `s`\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def (generic function with 1 method)\n\njulia> sum_to_def(10)\n(55, false)\n\nThis version returns s as before but it also uses the @isdefined macro to return a boolean indicating whether there is a local variable named t defined in the function's outermost local scope. As you can see, there is no t defined outside of the for loop body. This is because of the hard scope rule again: since the assignment to t occurs inside of a function, which introduces a hard scope, the assignment causes t to become a new local variable in the local scope where it appears, i.e. inside of the loop body. Even if there were a global named t, it would make no difference—the hard scope rule isn't affected by anything in global scope.\n\nNote that the local scope of a for loop body is no different from the local scope of an inner function. This means that we could rewrite this example so that the loop body is implemented as a call to an inner helper function and it behaves the same way:\n\njulia> function sum_to_def_closure(n)\n           function loop_body(i)\n               t = s + i # new local `t`\n               s = t # assign same local `s` as below\n           end\n           s = 0 # new local\n           for i = 1:n\n               loop_body(i)\n           end\n           return s, @isdefined(t)\n       end\nsum_to_def_closure (generic function with 1 method)\n\njulia> sum_to_def_closure(10)\n(55, false)\n\nThis example illustrates a couple of key points:\n\nInner function scopes are just like any other nested local scope. In particular, if a variable is already a local outside of an inner function and you assign to it in the inner function, the outer local variable is updated.\nIt doesn't matter if the definition of an outer local happens below where it is updated, the rule remains the same. The entire enclosing local scope is parsed and its locals determined before inner local meanings are resolved.\n\nThis design means that you can generally move code in or out of an inner function without changing its meaning, which facilitates a number of common idioms in the language using closures (see do blocks).\n\nLet's move onto some more ambiguous cases covered by the soft scope rule. We'll explore this by extracting the bodies of the greet and sum_to_def functions into soft scope contexts. First, let's put the body of greet in a for loop—which is soft, rather than hard—and evaluate it in the REPL:\n\njulia> for i = 1:3\n           x = \"hello\" # new local\n           println(x)\n       end\nhello\nhello\nhello\n\njulia> x\nERROR: UndefVarError: `x` not defined in `Main`\n\nSince the global x is not defined when the for loop is evaluated, the first clause of the soft scope rule applies and x is created as local to the for loop and therefore global x remains undefined after the loop executes. Next, let's consider the body of sum_to_def extracted into global scope, fixing its argument to n = 10\n\ns = 0\nfor i = 1:10\n    t = s + i\n    s = t\nend\ns\n@isdefined(t)\n\nWhat does this code do? Hint: it's a trick question. The answer is \"it depends.\" If this code is entered interactively, it behaves the same way it does in a function body. But if the code appears in a file, it  prints an ambiguity warning and throws an undefined variable error. Let's see it working in the REPL first:\n\njulia> s = 0 # global\n0\n\njulia> for i = 1:10\n           t = s + i # new local `t`\n           s = t # assign global `s`\n       end\n\njulia> s # global\n55\n\njulia> @isdefined(t) # global\nfalse\n\nThe REPL approximates being in the body of a function by deciding whether assignment inside the loop assigns to a global or creates new local based on whether a global variable by that name is defined or not. If a global by the name exists, then the assignment updates it. If no global exists, then the assignment creates a new local variable. In this example we see both cases in action:\n\nThere is no global named t, so t = s + i creates a new t that is local to the for loop;\nThere is a global named s, so s = t assigns to it.\n\nThe second fact is why execution of the loop changes the global value of s and the first fact is why t is still undefined after the loop executes. Now, let's try evaluating this same code as though it were in a file instead:\n\njulia> code = \"\"\"\n       s = 0 # global\n       for i = 1:10\n           t = s + i # new local `t`\n           s = t # new local `s` with warning\n       end\n       s, # global\n       @isdefined(t) # global\n       \"\"\";\n\njulia> include_string(Main, code)\n┌ Warning: Assignment to `s` in soft scope is ambiguous because a global variable by the same name exists: `s` will be treated as a new local. Disambiguate by using `local s` to suppress this warning or `global s` to assign to the existing global variable.\n└ @ string:4\nERROR: LoadError: UndefVarError: `s` not defined in local scope\n\nHere we use include_string, to evaluate code as though it were the contents of a file. We could also save code to a file and then call include on that file—the result would be the same. As you can see, this behaves quite different from evaluating the same code in the REPL. Let's break down what's happening here:\n\nglobal s is defined with the value 0 before the loop is evaluated\nthe assignment s = t occurs in a soft scope—a for loop outside of any function body or other hard scope construct\ntherefore the second clause of the soft scope rule applies, and the assignment is ambiguous so a warning is emitted\nexecution continues, making s local to the for loop body\nsince s is local to the for loop, it is undefined when t = s + i is evaluated, causing an error\nevaluation stops there, but if it got to s and @isdefined(t), it would return 0 and false.\n\nThis demonstrates some important aspects of scope: in a scope, each variable can only have one meaning, and that meaning is determined regardless of the order of expressions. The presence of the expression s = t in the loop causes s to be local to the loop, which means that it is also local when it appears on the right hand side of t = s + i, even though that expression appears first and is evaluated first. One might imagine that the s on the first line of the loop could be global while the s on the second line of the loop is local, but that's not possible since the two lines are in the same scope block and each variable can only mean one thing in a given scope."},{"location":"manual/variables-and-scoping.html#on-soft-scope","page":"Scope of Variables","title":"On Soft Scope","category":"section","text":"We have now covered all the local scope rules, but before wrapping up this section, perhaps a few words should be said about why the ambiguous soft scope case is handled differently in interactive and non-interactive contexts. There are two obvious questions one could ask:\n\nWhy doesn't it just work like the REPL everywhere?\nWhy doesn't it just work like in files everywhere? And maybe skip the warning?\n\nIn Julia ≤ 0.6, all global scopes did work like the current REPL: when x = <value> occurred in a loop (or try/catch, or struct body) but outside of a function body (or let block or comprehension), it was decided based on whether a global named x was defined or not whether x should be local to the loop. This behavior has the advantage of being intuitive and convenient since it approximates the behavior inside of a function body as closely as possible. In particular, it makes it easy to move code back and forth between a function body and the REPL when trying to debug the behavior of a function. However, it has some downsides. First, it's quite a complex behavior: many people over the years were confused about this behavior and complained that it was complicated and hard both to explain and understand. Fair point. Second, and arguably worse, is that it's bad for programming \"at scale.\" When you see a small piece of code in one place like this, it's quite clear what's going on:\n\ns = 0\nfor i = 1:10\n    s += i\nend\n\nObviously the intention is to modify the existing global variable s. What else could it mean? However, not all real world code is so short or so clear. We found that code like the following often occurs in the wild:\n\nx = 123\n\n# much later\n# maybe in a different file\n\nfor i = 1:10\n    x = \"hello\"\n    println(x)\nend\n\n# much later\n# maybe in yet another file\n# or maybe back in the first one where `x = 123`\n\ny = x + 234\n\nIt's far less clear what should happen here. Since x + \"hello\" is a method error, it seems probable that the intention is for x to be local to the for loop. But runtime values and what methods happen to exist cannot be used to determine the scopes of variables. With the Julia ≤ 0.6 behavior, it's especially concerning that someone might have written the for loop first, had it working just fine, but later when someone else adds a new global far away—possibly in a different file—the code suddenly changes meaning and either breaks noisily or, worse still, silently does the wrong thing. This kind of \"spooky action at a distance\" is something that good programming language designs should prevent.\n\nSo in Julia 1.0, we simplified the rules for scope: in any local scope, assignment to a name that wasn't already a local variable created a new local variable. This eliminated the notion of soft scope entirely as well as removing the potential for spooky action. We uncovered and fixed a significant number of bugs due to the removal of soft scope, vindicating the choice to get rid of it. And there was much rejoicing! Well, no, not really. Because some people were angry that they now had to write:\n\ns = 0\nfor i = 1:10\n    global s += i\nend\n\nDo you see that global annotation in there? Hideous. Obviously this situation could not be tolerated. But seriously, there are two main issues with requiring global for this kind of top-level code:\n\nIt's no longer convenient to copy and paste the code from inside a function body into the REPL to debug it—you have to add global annotations and then remove them again to go back;\nBeginners will write this kind of code without the global and have no idea why their code doesn't work—the error that they get is that s is undefined, which does not seem to enlighten anyone who happens to make this mistake.\n\nAs of Julia 1.5, this code works without the global annotation in interactive contexts like the REPL or Jupyter notebooks (just like Julia 0.6) and in files and other non-interactive contexts, it prints this very direct warning:\n\nAssignment to s in soft scope is ambiguous because a global variable by the same name exists: s will be treated as a new local. Disambiguate by using local s to suppress this warning or global s to assign to the existing global variable.\n\nThis addresses both issues while preserving the \"programming at scale\" benefits of the 1.0 behavior: global variables have no spooky effect on the meaning of code that may be far away; in the REPL copy-and-paste debugging works and beginners don't have any issues; any time someone either forgets a global annotation or accidentally shadows an existing global with a local in a soft scope, which would be confusing anyway, they get a nice clear warning.\n\nAn important property of this design is that any code that executes in a file without a warning will behave the same way in a fresh REPL. And on the flip side, if you take a REPL session and save it to file, if it behaves differently than it did in the REPL, then you will get a warning."},{"location":"manual/variables-and-scoping.html#Let-Blocks","page":"Scope of Variables","title":"Let Blocks","category":"section","text":"let statements create a new hard scope block (see above) and introduce new variable bindings each time they run. The variable need not be immediately assigned:\n\njulia> var1 = let x\n           for i in 1:5\n               (i == 4) && (x = i; break)\n           end\n           x\n       end\n4\n\nWhereas assignments might reassign a new value to an existing value location, let always creates a new location. This difference is usually not important, and is only detectable in the case of variables that outlive their scope via closures. The let syntax accepts a comma-separated series of assignments and variable names:\n\njulia> x, y, z = -1, -1, -1;\n\njulia> let x = 1, z\n           println(\"x: $x, y: $y\") # x is local variable, y the global\n           println(\"z: $z\") # errors as z has not been assigned yet but is local\n       end\nx: 1, y: -1\nERROR: UndefVarError: `z` not defined in local scope\n\nThe assignments are evaluated in order, with each right-hand side evaluated in the scope before the new variable on the left-hand side has been introduced. Therefore it makes sense to write something like let x = x since the two x variables are distinct and have separate storage. Here is an example where the behavior of let is needed:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           Fs[i] = ()->i\n           global i += 1\n       end\n\njulia> Fs[1]()\n3\n\njulia> Fs[2]()\n3\n\nHere we create and store two closures that return variable i. However, it is always the same variable i, so the two closures behave identically. We can use let to create a new binding for i:\n\njulia> Fs = Vector{Any}(undef, 2); i = 1;\n\njulia> while i <= 2\n           let i = i\n               Fs[i] = ()->i\n           end\n           global i += 1\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nSince the begin construct does not introduce a new scope, it can be useful to use a zero-argument let to just introduce a new scope block without creating any new bindings immediately:\n\njulia> let\n           local x = 1\n           let\n               local x = 2\n           end\n           x\n       end\n1\n\nSince let introduces a new scope block, the inner local x is a different variable than the outer local x. This particular example is equivalent to:\n\njulia> let x = 1\n           let x = 2\n           end\n           x\n       end\n1"},{"location":"manual/variables-and-scoping.html#Loops-and-Comprehensions","page":"Scope of Variables","title":"Loops and Comprehensions","category":"section","text":"In loops and comprehensions, new variables introduced in their body scopes are freshly allocated for each loop iteration, as if the loop body were surrounded by a let block, as demonstrated by this example:\n\njulia> Fs = Vector{Any}(undef, 2);\n\njulia> for j = 1:2\n           Fs[j] = ()->j\n       end\n\njulia> Fs[1]()\n1\n\njulia> Fs[2]()\n2\n\nA for loop or comprehension iteration variable is always a new variable:\n\njulia> function f()\n           i = 0\n           for i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n0\n\nHowever, it is occasionally useful to reuse an existing local variable as the iteration variable. This can be done conveniently by adding the keyword outer:\n\njulia> function f()\n           i = 0\n           for outer i = 1:3\n               # empty\n           end\n           return i\n       end;\n\njulia> f()\n3"},{"location":"manual/variables-and-scoping.html#Constants","page":"Scope of Variables","title":"Constants","category":"section","text":"A common use of variables is giving names to specific, unchanging values. Such variables are only assigned once. This intent can be conveyed to the compiler using the const keyword:\n\njulia> const e  = 2.71828182845904523536;\n\njulia> const pi = 3.14159265358979323846;\n\nMultiple variables can be declared in a single const statement:\n\njulia> const a, b = 1, 2\n(1, 2)\n\nThe const declaration should only be used in global scope on globals. It is difficult for the compiler to optimize code involving global variables, since their values (or even their types) might change at almost any time. If a global variable will not change, adding a const declaration solves this performance problem.\n\nLocal constants are quite different. The compiler is able to determine automatically when a local variable is constant, so local constant declarations are not necessary, and in fact are currently not supported.\n\nSpecial top-level assignments, such as those performed by the function and struct keywords, are constant by default.\n\nNote that const only affects the variable binding; the variable may be bound to a mutable object (such as an array), and that object may still be modified. Additionally when one tries to assign a value to a variable that is declared constant the following scenarios are possible:\n\nAttempting to replace a constant without the const keyword is disallowed:\njulia> const x = 1.0\n1.0\n\njulia> x = 1\nERROR: invalid assignment to constant x. This redefinition may be permitted using the `const` keyword.\nAll other definitions of constants are permitted, but may cause significant re-compilation:\njulia> const y = 1.0\n1.0\n\njulia> const y = 2.0\n2.0\n\ncompat: Julia 1.12\nPrior to julia 1.12, redefinition of constants was poorly supported. It was restricted to redefinition of constants of the same type and could lead to observably incorrect behavior or crashes. Constant redefinition is highly discouraged in versions of julia prior to 1.12. See the manual for prior julia versions for further information."},{"location":"manual/variables-and-scoping.html#man-typed-globals","page":"Scope of Variables","title":"Typed Globals","category":"section","text":"compat: Julia 1.8\nSupport for typed globals was added in Julia 1.8\n\nSimilar to being declared as constants, global bindings can also be declared to always be of a constant type. This can either be done without assigning an actual value using the syntax global x::T or upon assignment as x::T = 123.\n\njulia> x::Float64 = 2.718\n2.718\n\njulia> f() = x\nf (generic function with 1 method)\n\njulia> Base.return_types(f)\n1-element Vector{Any}:\n Float64\n\nFor any assignment to a global, Julia will first try to convert it to the appropriate type using convert:\n\njulia> global y::Int\n\njulia> y = 1.0\n1.0\n\njulia> y\n1\n\njulia> y = 3.14\nERROR: InexactError: Int64(3.14)\nStacktrace:\n[...]\n\nThe type does not need to be concrete, but annotations with abstract types typically have little performance benefit.\n\nOnce a global has either been assigned to or its type has been set, the binding type is not allowed to change:\n\njulia> x = 1\n1\n\njulia> global x::Int\nERROR: cannot set type for global x. It already has a value or is already set to a different type.\nStacktrace:\n[...]"},{"location":"base/strings.html#lib-strings","page":"Strings","title":"Strings","category":"section","text":""},{"location":"base/strings.html#AnnotatedStrings","page":"Strings","title":"AnnotatedStrings","category":"section","text":"note: Note\nThe API for AnnotatedStrings is considered experimental and is subject to change between Julia versions."},{"location":"base/strings.html#Core.AbstractString","page":"Strings","title":"Core.AbstractString","category":"type","text":"The AbstractString type is the supertype of all string implementations in Julia. Strings are encodings of sequences of Unicode code points as represented by the AbstractChar type. Julia makes a few assumptions about strings:\n\nStrings are encoded in terms of fixed-size \"code units\"\nCode units can be extracted with codeunit(s, i)\nThe first code unit has index 1\nThe last code unit has index ncodeunits(s)\nAny index i such that 1 ≤ i ≤ ncodeunits(s) is in bounds\nString indexing is done in terms of these code units:\nCharacters are extracted by s[i] with a valid string index i\nEach AbstractChar in a string is encoded by one or more code units\nOnly the index of the first code unit of an AbstractChar is a valid index\nThe encoding of an AbstractChar is independent of what precedes or follows it\nString encodings are self-synchronizing – i.e. isvalid(s, i) is O(1)\n\nSome string functions that extract code units, characters or substrings from strings error if you pass them out-of-bounds or invalid string indices. This includes codeunit(s, i) and s[i]. Functions that do string index arithmetic take a more relaxed approach to indexing and give you the closest valid string index when in-bounds, or when out-of-bounds, behave as if there were an infinite number of characters padding each side of the string. Usually these imaginary padding characters have code unit length 1 but string types may choose different \"imaginary\" character sizes as makes sense for their implementations (e.g. substrings may pass index arithmetic through to the underlying string they provide a view into). Relaxed indexing functions include those intended for index arithmetic: thisind, nextind and prevind. This model allows index arithmetic to work with out-of-bounds indices as intermediate values so long as one never uses them to retrieve a character, which often helps avoid needing to code around edge cases.\n\nSee also codeunit, ncodeunits, thisind, nextind, prevind.\n\n\n\n\n\n"},{"location":"base/strings.html#Core.AbstractChar","page":"Strings","title":"Core.AbstractChar","category":"type","text":"The AbstractChar type is the supertype of all character implementations in Julia. A character represents a Unicode code point, and can be converted to an integer via the codepoint function in order to obtain the numerical value of the code point, or constructed from the same integer. These numerical values determine how characters are compared with < and ==, for example.  New T <: AbstractChar types should define a codepoint(::T) method and a T(::UInt32) constructor, at minimum.\n\nA given AbstractChar subtype may be capable of representing only a subset of Unicode, in which case conversion from an unsupported UInt32 value may throw an error. Conversely, the built-in Char type represents a superset of Unicode (in order to losslessly encode invalid byte streams), in which case conversion of a non-Unicode value to UInt32 throws an error. The isvalid function can be used to check which codepoints are representable in a given AbstractChar type.\n\nInternally, an AbstractChar type may use a variety of encodings.  Conversion via codepoint(char) will not reveal this encoding because it always returns the Unicode value of the character. print(io, c) of any c::AbstractChar produces an encoding determined by io (UTF-8 for all built-in IO types), via conversion to Char if necessary.\n\nwrite(io, c), in contrast, may emit an encoding depending on typeof(c), and read(io, typeof(c)) should read the same encoding as write. New AbstractChar types must provide their own implementations of write and read.\n\n\n\n\n\n"},{"location":"base/strings.html#Core.Char","page":"Strings","title":"Core.Char","category":"type","text":"Char(c::Union{Number,AbstractChar})\n\nChar is a 32-bit AbstractChar type that is the default representation of characters in Julia. Char is the type used for character literals like 'x' and it is also the element type of String.\n\nIn order to losslessly represent arbitrary byte streams stored in a String, a Char value may store information that cannot be converted to a Unicode codepoint — converting such a Char to UInt32 will throw an error. The isvalid(c::Char) function can be used to query whether c represents a valid Unicode character.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.codepoint","page":"Strings","title":"Base.codepoint","category":"function","text":"codepoint(c::AbstractChar)::Integer\n\nReturn the Unicode codepoint (an unsigned integer) corresponding to the character c (or throw an exception if c does not represent a valid character). For Char, this is a UInt32 value, but AbstractChar types that represent only a subset of Unicode may return a different-sized integer (e.g. UInt8).\n\n\n\n\n\n"},{"location":"base/strings.html#Base.length-Tuple{AbstractString}","page":"Strings","title":"Base.length","category":"method","text":"length(s::AbstractString)::Int\nlength(s::AbstractString, i::Integer, j::Integer)::Int\n\nReturn the number of characters in string s from indices i through j.\n\nThis is computed as the number of code unit indices from i to j which are valid character indices. With only a single string argument, this computes the number of characters in the entire string. With i and j arguments it computes the number of indices between i and j inclusive that are valid indices in the string s. In addition to in-bounds values, i may take the out-of-bounds value ncodeunits(s) + 1 and j may take the out-of-bounds value 0.\n\nnote: Note\nThe time complexity of this operation is linear in general. That is, it will take the time proportional to the number of bytes or characters in the string because it counts the value on the fly. This is in contrast to the method for arrays, which is a constant-time operation.\n\nSee also isvalid, ncodeunits, lastindex, thisind, nextind, prevind.\n\nExamples\n\njulia> length(\"jμΛIα\")\n5\n\n\n\n\n\n"},{"location":"base/strings.html#Base.sizeof-Tuple{AbstractString}","page":"Strings","title":"Base.sizeof","category":"method","text":"sizeof(str::AbstractString)\n\nSize, in bytes, of the string str. Equal to the number of code units in str multiplied by the size, in bytes, of one code unit in str.\n\nExamples\n\njulia> sizeof(\"\")\n0\n\njulia> sizeof(\"∀\")\n3\n\n\n\n\n\n"},{"location":"base/strings.html#Base.:*-Tuple{Union{AbstractChar, AbstractString}, Vararg{Union{AbstractChar, AbstractString}}}","page":"Strings","title":"Base.:*","category":"method","text":"*(s::Union{AbstractString, AbstractChar}, t::Union{AbstractString, AbstractChar}...)::AbstractString\n\nConcatenate strings and/or characters, producing a String or AnnotatedString (as appropriate). This is equivalent to calling the string or annotatedstring function on the arguments. Concatenation of built-in string types always produces a value of type String but other string types may choose to return a string of a different type as appropriate.\n\nExamples\n\njulia> \"Hello \" * \"world\"\n\"Hello world\"\n\njulia> 'j' * \"ulia\"\n\"julia\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.:^-Tuple{Union{AbstractChar, AbstractString}, Integer}","page":"Strings","title":"Base.:^","category":"method","text":"^(s::Union{AbstractString,AbstractChar}, n::Integer)::AbstractString\n\nRepeat a string or character n times. This can also be written as repeat(s, n).\n\nSee also repeat.\n\nExamples\n\njulia> \"Test \"^3\n\"Test Test Test \"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.string","page":"Strings","title":"Base.string","category":"function","text":"string(xs...)\n\nCreate a string from any values using the print function.\n\nstring should usually not be defined directly. Instead, define a method print(io::IO, x::MyType). If string(x) for a certain type needs to be highly efficient, then it may make sense to add a method to string and define print(io::IO, x::MyType) = print(io, string(x)) to ensure the functions are consistent.\n\nSee also: String, repr, sprint, show.\n\nExamples\n\njulia> string(\"a\", 1, true)\n\"a1true\"\n\n\n\n\n\nstring(n::Integer; base::Integer = 10, pad::Integer = 1)\n\nConvert an integer n to a string in the given base, optionally specifying a number of digits to pad to.\n\nSee also digits, bitstring, count_zeros, and the Printf standard library.\n\nExamples\n\njulia> string(5, base = 13, pad = 4)\n\"0005\"\n\njulia> string(-13, base = 5, pad = 4)\n\"-0023\"\n\njulia> using Printf\n\njulia> @sprintf(\"%04i\", 5)\n\"0005\"\n\njulia> @sprintf(\"%4i\", 5)\n\"   5\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.repeat-Tuple{AbstractString, Integer}","page":"Strings","title":"Base.repeat","category":"method","text":"repeat(s::AbstractString, r::Integer)\n\nRepeat a string r times. This can be written as s^r.\n\nSee also ^.\n\nExamples\n\njulia> repeat(\"ha\", 3)\n\"hahaha\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.repeat-Tuple{AbstractChar, Integer}","page":"Strings","title":"Base.repeat","category":"method","text":"repeat(c::AbstractChar, r::Integer)::String\n\nRepeat a character r times. This can equivalently be accomplished by calling c^r.\n\nExamples\n\njulia> repeat('A', 3)\n\"AAA\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.repr-Tuple{Any}","page":"Strings","title":"Base.repr","category":"method","text":"repr(x; context=nothing)\n\nCreate a string representation of any value using the 2-argument show(io, x) function, which aims to produce a string that is parseable Julia code, where possible. i.e. eval(Meta.parse(repr(x))) == x should hold true. You should not add methods to repr; define a show method instead.\n\nThe optional keyword argument context can be set to a :key=>value pair, a tuple of :key=>value pairs, or an IO or IOContext object whose attributes are used for the I/O stream passed to show.\n\nNote that repr(x) is usually similar to how the value of x would be entered in Julia.  See also repr(MIME(\"text/plain\"), x) to instead return a \"pretty-printed\" version of x designed more for human consumption, equivalent to the REPL display of x, using the 3-argument show(io, mime, x).\n\ncompat: Julia 1.7\nPassing a tuple to keyword context requires Julia 1.7 or later.\n\nExamples\n\njulia> repr(1)\n\"1\"\n\njulia> repr(zeros(3))\n\"[0.0, 0.0, 0.0]\"\n\njulia> repr(big(1/3))\n\"0.333333333333333314829616256247390992939472198486328125\"\n\njulia> repr(big(1/3), context=:compact => true)\n\"0.333333\"\n\n\n\n\n\n\n"},{"location":"base/strings.html#Core.String-Tuple{AbstractString}","page":"Strings","title":"Core.String","category":"method","text":"String(s::AbstractString)\n\nCreate a new String from an existing AbstractString.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.SubString","page":"Strings","title":"Base.SubString","category":"type","text":"SubString(s::AbstractString, i::Integer, j::Integer=lastindex(s))\nSubString(s::AbstractString, r::UnitRange{<:Integer})\n\nLike getindex, but returns a view into the parent string s within range i:j or r respectively instead of making a copy.\n\nThe @views macro converts any string slices s[i:j] into substrings SubString(s, i, j) in a block of code.\n\nExamples\n\njulia> SubString(\"abc\", 1, 2)\n\"ab\"\n\njulia> SubString(\"abc\", 1:2)\n\"ab\"\n\njulia> SubString(\"abc\", 2)\n\"bc\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.LazyString","page":"Strings","title":"Base.LazyString","category":"type","text":"LazyString <: AbstractString\n\nA lazy representation of string interpolation. This is useful when a string needs to be constructed in a context where performing the actual interpolation and string construction is unnecessary or undesirable (e.g. in error paths of functions).\n\nThis type is designed to be cheap to construct at runtime, trying to offload as much work as possible to either the macro or later printing operations.\n\nExamples\n\njulia> n = 5; str = LazyString(\"n is \", n)\n\"n is 5\"\n\nSee also @lazy_str.\n\ncompat: Julia 1.8\nLazyString requires Julia 1.8 or later.\n\nExtended help\n\nSafety properties for concurrent programs\n\nA lazy string itself does not introduce any concurrency problems even if it is printed in multiple Julia tasks.  However, if print methods on a captured value can have a concurrency issue when invoked without synchronizations, printing the lazy string may cause an issue.  Furthermore, the print methods on the captured values may be invoked multiple times, though only exactly one result will be returned.\n\ncompat: Julia 1.9\nLazyString is safe in the above sense in Julia 1.9 and later.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.@lazy_str","page":"Strings","title":"Base.@lazy_str","category":"macro","text":"lazy\"str\"\n\nCreate a LazyString using regular string interpolation syntax. Note that interpolations are evaluated at LazyString construction time, but printing is delayed until the first access to the string.\n\nSee LazyString documentation for the safety properties for concurrent programs.\n\nExamples\n\njulia> n = 5; str = lazy\"n is $n\"\n\"n is 5\"\n\njulia> typeof(str)\nLazyString\n\ncompat: Julia 1.8\nlazy\"str\" requires Julia 1.8 or later.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.transcode","page":"Strings","title":"Base.transcode","category":"function","text":"transcode(T, src)\n\nConvert string data between Unicode encodings. src is either a String or a Vector{UIntXX} of UTF-XX code units, where XX is 8, 16, or 32. T indicates the encoding of the return value: String to return a (UTF-8 encoded) String or UIntXX to return a Vector{UIntXX} of UTF-XX data. (The alias Cwchar_t can also be used as the integer type, for converting wchar_t* strings used by external C libraries.)\n\nThe transcode function succeeds as long as the input data can be reasonably represented in the target encoding; it always succeeds for conversions between UTF-XX encodings, even for invalid Unicode data.\n\nOnly conversion to/from UTF-8 is currently supported.\n\nExamples\n\njulia> str = \"αβγ\"\n\"αβγ\"\n\njulia> transcode(UInt16, str)\n3-element Vector{UInt16}:\n 0x03b1\n 0x03b2\n 0x03b3\n\njulia> transcode(String, transcode(UInt16, str))\n\"αβγ\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.unsafe_string","page":"Strings","title":"Base.unsafe_string","category":"function","text":"unsafe_string(p::Ptr{T}, [length::Integer]) where {T<:Union{UInt16,UInt32,Cwchar_t}}\nunsafe_string(p::Cwstring)\n\nTranscode a string from the address of a C-style (NUL-terminated) string encoded as UTF-16 (T=UInt16), UTF-32 (T=UInt32), or the system-dependent wchar_t (T=Cwchar_t or Cwstring), returning a String (UTF-8 encoding), similar to transcode but reading directly from a pointer.  (The pointer can be safely freed afterwards.) If length is specified (the length of the data in encoding units), the string does not have to be NUL-terminated.\n\nThis function is labeled \"unsafe\" because it will crash if p is not a valid memory address to data of the requested length (or NUL-terminated data).\n\n\n\n\n\nunsafe_string(p::Ptr{UInt8}, [length::Integer])\nunsafe_string(p::Cstring)\n\nCopy a string from the address of a C-style (NUL-terminated) string encoded as UTF-8. (The pointer can be safely freed afterwards.) If length is specified (the length of the data in bytes), the string does not have to be NUL-terminated.\n\nThis function is labeled \"unsafe\" because it will crash if p is not a valid memory address to data of the requested length.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.ncodeunits-Tuple{AbstractString}","page":"Strings","title":"Base.ncodeunits","category":"method","text":"ncodeunits(s::AbstractString)::Int\n\nReturn the number of code units in a string. Indices that are in bounds to access this string must satisfy 1 ≤ i ≤ ncodeunits(s). Not all such indices are valid – they may not be the start of a character, but they will return a code unit value when calling codeunit(s,i).\n\nExamples\n\njulia> ncodeunits(\"The Julia Language\")\n18\n\njulia> ncodeunits(\"∫eˣ\")\n6\n\njulia> ncodeunits('∫'), ncodeunits('e'), ncodeunits('ˣ')\n(3, 1, 2)\n\nSee also codeunit, checkbounds, sizeof, length, lastindex.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.codeunit","page":"Strings","title":"Base.codeunit","category":"function","text":"codeunit(s::AbstractString, i::Integer)::Union{UInt8, UInt16, UInt32}\n\nReturn the code unit value in the string s at index i. Note that\n\ncodeunit(s, i) :: codeunit(s)\n\nI.e. the value returned by codeunit(s, i) is of the type returned by codeunit(s).\n\nExamples\n\njulia> a = codeunit(\"Hello\", 2)\n0x65\n\njulia> typeof(a)\nUInt8\n\nSee also ncodeunits, checkbounds.\n\n\n\n\n\ncodeunit(s::AbstractString)::Type{<:Union{UInt8, UInt16, UInt32}}\n\nReturn the code unit type of the given string object. For ASCII, Latin-1, or UTF-8 encoded strings, this would be UInt8; for UCS-2 and UTF-16 it would be UInt16; for UTF-32 it would be UInt32. The code unit type need not be limited to these three types, but it's hard to think of widely used string encodings that don't use one of these units. codeunit(s) is the same as typeof(codeunit(s,1)) when s is a non-empty string.\n\nSee also ncodeunits.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.codeunits","page":"Strings","title":"Base.codeunits","category":"function","text":"codeunits(s::AbstractString)\n\nObtain a vector-like object containing the code units of a string. Returns a CodeUnits wrapper by default, but codeunits may optionally be defined for new string types if necessary.\n\nExamples\n\njulia> codeunits(\"Juλia\")\n6-element Base.CodeUnits{UInt8, String}:\n 0x4a\n 0x75\n 0xce\n 0xbb\n 0x69\n 0x61\n\n\n\n\n\n"},{"location":"base/strings.html#Base.ascii","page":"Strings","title":"Base.ascii","category":"function","text":"ascii(s::AbstractString)\n\nConvert a string to String type and check that it contains only ASCII data, otherwise throwing an ArgumentError indicating the position of the first non-ASCII byte.\n\nSee also the isascii predicate to filter or replace non-ASCII characters.\n\nExamples\n\njulia> ascii(\"abcdeγfgh\")\nERROR: ArgumentError: invalid ASCII at index 6 in \"abcdeγfgh\"\nStacktrace:\n[...]\n\njulia> ascii(\"abcdefgh\")\n\"abcdefgh\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Regex","page":"Strings","title":"Base.Regex","category":"type","text":"Regex(pattern[, flags]) <: AbstractPattern\n\nA type representing a regular expression. Regex objects can be used to match strings with match.\n\nRegex objects can be created using the @r_str string macro. The Regex(pattern[, flags]) constructor is usually used if the pattern string needs to be interpolated. See the documentation of the string macro for details on flags.\n\nnote: Note\nTo escape interpolated variables use \\Q and \\E (e.g. Regex(\"\\\\Q$x\\\\E\"))\n\n\n\n\n\n"},{"location":"base/strings.html#Base.@r_str","page":"Strings","title":"Base.@r_str","category":"macro","text":"@r_str -> Regex\n\nConstruct a regex, such as r\"^[a-z]*$\", without interpolation and unescaping (except for quotation mark \" which still has to be escaped). The regex also accepts one or more flags, listed after the ending quote, to change its behaviour:\n\ni enables case-insensitive matching\nm treats the ^ and $ tokens as matching the start and end of individual lines, as opposed to the whole string.\ns allows the . modifier to match newlines.\nx enables \"free-spacing mode\": whitespace between regex tokens is ignored except when escaped with \\,  and # in the regex is treated as starting a comment (which is ignored to the line ending).\na enables ASCII mode (disables UTF and UCP modes). By default \\B, \\b, \\D, \\d, \\S, \\s, \\W, \\w, etc. match based on Unicode character properties. With this option, these sequences only match ASCII characters. This includes \\u also, which will emit the specified character value directly as a single byte, and not attempt to encode it into UTF-8. Importantly, this option allows matching against invalid UTF-8 strings, by treating both matcher and target as simple bytes (as if they were ISO/IEC 8859-1 / Latin-1 bytes) instead of as character encodings. In this case, this option is often combined with s. This option can be further refined by starting the pattern with (UCP) or (UTF).\n\nSee Regex if interpolation is needed.\n\nExamples\n\njulia> match(r\"a+.*b+.*?d$\"ism, \"Goodbye,\\nOh, angry,\\nBad world\\n\")\nRegexMatch(\"angry,\\nBad world\")\n\nThis regex has the first three flags enabled.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.SubstitutionString","page":"Strings","title":"Base.SubstitutionString","category":"type","text":"SubstitutionString(substr) <: AbstractString\n\nStores the given string substr as a SubstitutionString, for use in regular expression substitutions. Most commonly constructed using the @s_str macro.\n\nExamples\n\njulia> SubstitutionString(\"Hello \\\\g<name>, it's \\\\1\")\ns\"Hello \\g<name>, it's \\1\"\n\njulia> subst = s\"Hello \\g<name>, it's \\1\"\ns\"Hello \\g<name>, it's \\1\"\n\njulia> typeof(subst)\nSubstitutionString{String}\n\n\n\n\n\n"},{"location":"base/strings.html#Base.@s_str","page":"Strings","title":"Base.@s_str","category":"macro","text":"@s_str -> SubstitutionString\n\nConstruct a substitution string, used for regular expression substitutions.  Within the string, sequences of the form \\N refer to the Nth capture group in the regex, and \\g<groupname> refers to a named capture group with name groupname.\n\nExamples\n\njulia> msg = \"#Hello# from Julia\";\n\njulia> replace(msg, r\"#(.+)# from (?<from>\\w+)\" => s\"FROM: \\g<from>; MESSAGE: \\1\")\n\"FROM: Julia; MESSAGE: Hello\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.@raw_str","page":"Strings","title":"Base.@raw_str","category":"macro","text":"@raw_str -> String\n\nCreate a raw string without interpolation and unescaping. The exception is that quotation marks still must be escaped. Backslashes escape both quotation marks and other backslashes, but only when a sequence of backslashes precedes a quote character. Thus, 2n backslashes followed by a quote encodes n backslashes and the end of the literal while 2n+1 backslashes followed by a quote encodes n backslashes followed by a quote character.\n\nExamples\n\njulia> println(raw\"\\ $x\")\n\\ $x\n\njulia> println(raw\"\\\"\")\n\"\n\njulia> println(raw\"\\\\\\\"\")\n\\\"\n\njulia> println(raw\"\\\\x \\\\\\\"\")\n\\\\x \\\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.@b_str","page":"Strings","title":"Base.@b_str","category":"macro","text":"@b_str\n\nCreate an immutable byte (UInt8) vector using string syntax.\n\nExamples\n\njulia> v = b\"12\\x01\\x02\"\n4-element Base.CodeUnits{UInt8, String}:\n 0x31\n 0x32\n 0x01\n 0x02\n\njulia> v[2]\n0x32\n\n\n\n\n\n"},{"location":"base/strings.html#Base.takestring!","page":"Strings","title":"Base.takestring!","category":"function","text":"takestring!(io::IOBuffer) -> String\n\nReturn the content of io as a String, resetting the buffer to its initial state. This is preferred over calling String(take!(io)) to create a string from an IOBuffer.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> write(io, [0x61, 0x62, 0x63]);\n\njulia> s = takestring!(io)\n\"abc\"\n\njulia> isempty(take!(io)) # io is now empty\ntrue\n\ncompat: Julia 1.13\nThis function requires at least Julia 1.13.\n\n\n\n\n\ntakestring!(x) -> String\n\nCreate a string from the content of x, emptying x.\n\nExamples\n\njulia> v = [0x61, 0x62, 0x63];\n\njulia> s = takestring!(v)\n\"abc\"\n\njulia> isempty(v)\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Docs.@html_str","page":"Strings","title":"Base.Docs.@html_str","category":"macro","text":"@html_str -> Docs.HTML\n\nCreate an HTML object from a literal string.\n\nExamples\n\njulia> html\"Julia\"\nHTML{String}(\"Julia\")\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Docs.@text_str","page":"Strings","title":"Base.Docs.@text_str","category":"macro","text":"@text_str -> Docs.Text\n\nCreate a Text object from a literal string.\n\nExamples\n\njulia> text\"Julia\"\nJulia\n\n\n\n\n\n"},{"location":"base/strings.html#Base.isvalid-Tuple{Any}","page":"Strings","title":"Base.isvalid","category":"method","text":"isvalid(value)::Bool\n\nReturn true if the given value is valid for its type, which currently can be either AbstractChar or String or SubString{String}.\n\nExamples\n\njulia> isvalid(Char(0xd800))\nfalse\n\njulia> isvalid(SubString(String(UInt8[0xfe,0x80,0x80,0x80,0x80,0x80]),1,2))\nfalse\n\njulia> isvalid(Char(0xd799))\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.isvalid-Tuple{Any, Any}","page":"Strings","title":"Base.isvalid","category":"method","text":"isvalid(T, value)::Bool\n\nReturn true if the given value is valid for that type. Types currently can be either AbstractChar or String. Values for AbstractChar can be of type AbstractChar or UInt32. Values for String can be of that type, SubString{String}, Vector{UInt8}, or a contiguous subarray thereof.\n\nExamples\n\njulia> isvalid(Char, 0xd800)\nfalse\n\njulia> isvalid(String, SubString(\"thisisvalid\",1,5))\ntrue\n\njulia> isvalid(Char, 0xd799)\ntrue\n\ncompat: Julia 1.6\nSupport for subarray values was added in Julia 1.6.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.isvalid-Tuple{AbstractString, Integer}","page":"Strings","title":"Base.isvalid","category":"method","text":"isvalid(s::AbstractString, i::Integer)::Bool\n\nPredicate indicating whether the given index is the start of the encoding of a character in s or not. If isvalid(s, i) is true then s[i] will return the character whose encoding starts at that index, if it's false, then s[i] will raise an invalid index error or a bounds error depending on if i is in bounds. In order for isvalid(s, i) to be an O(1) function, the encoding of s must be self-synchronizing. This is a basic assumption of Julia's generic string support.\n\nSee also getindex, iterate, thisind, nextind, prevind, length.\n\nExamples\n\njulia> str = \"αβγdef\";\n\njulia> isvalid(str, 1)\ntrue\n\njulia> str[1]\n'α': Unicode U+03B1 (category Ll: Letter, lowercase)\n\njulia> isvalid(str, 2)\nfalse\n\njulia> str[2]\nERROR: StringIndexError: invalid index [2], valid nearby indices [1]=>'α', [3]=>'β'\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"base/strings.html#Base.match","page":"Strings","title":"Base.match","category":"function","text":"match(r::Regex, s::AbstractString[, idx::Integer[, addopts]])\n\nSearch for the first match of the regular expression r in s and return a RegexMatch object containing the match, or nothing if the match failed. The optional idx argument specifies an index at which to start the search. The matching substring can be retrieved by accessing m.match, the captured sequences can be retrieved by accessing m.captures. The resulting RegexMatch object can be used to construct other collections: e.g. Tuple(m), NamedTuple(m).\n\ncompat: Julia 1.11\nConstructing NamedTuples and Dicts requires Julia 1.11\n\nExamples\n\njulia> rx = r\"a(.)a\"\nr\"a(.)a\"\n\njulia> m = match(rx, \"cabac\")\nRegexMatch(\"aba\", 1=\"b\")\n\njulia> m.captures\n1-element Vector{Union{Nothing, SubString{String}}}:\n \"b\"\n\njulia> m.match\n\"aba\"\n\njulia> match(rx, \"cabac\", 3) === nothing\ntrue\n\nSee also\n\neachmatch, occursin, findfirst\n\n\n\n\n\n"},{"location":"base/strings.html#Base.eachmatch","page":"Strings","title":"Base.eachmatch","category":"function","text":"eachmatch(r::Regex, s::AbstractString; overlap::Bool=false)\n\nSearch for all matches of the regular expression r in s and return an iterator over the matches. If overlap is true, the matching sequences are allowed to overlap indices in the original string, otherwise they must be from distinct character ranges.\n\nExamples\n\njulia> rx = r\"a.a\"\nr\"a.a\"\n\njulia> m = eachmatch(rx, \"a1a2a3a\")\nBase.RegexMatchIterator{String}(r\"a.a\", \"a1a2a3a\", false)\n\njulia> collect(m)\n2-element Vector{RegexMatch}:\n RegexMatch(\"a1a\")\n RegexMatch(\"a3a\")\n\njulia> collect(eachmatch(rx, \"a1a2a3a\", overlap = true))\n3-element Vector{RegexMatch}:\n RegexMatch(\"a1a\")\n RegexMatch(\"a2a\")\n RegexMatch(\"a3a\")\n\nSee also\n\nmatch, findall, count\n\n\n\n\n\n"},{"location":"base/strings.html#Base.RegexMatch","page":"Strings","title":"Base.RegexMatch","category":"type","text":"RegexMatch <: AbstractMatch\n\nA type representing a single match to a Regex found in a string. Typically created from the match function.\n\nThe match field stores the substring of the entire matched string. The captures field stores the substrings for each capture group, indexed by number. To index by capture group name, the entire match object should be indexed instead, as shown in the examples. The location of the start of the match is stored in the offset field. The offsets field stores the locations of the start of each capture group, with 0 denoting a group that was not captured.\n\nThis type can be used as an iterator over the capture groups of the Regex, yielding the substrings captured in each group. Because of this, the captures of a match can be destructured. If a group was not captured, nothing will be yielded instead of a substring.\n\nMethods that accept a RegexMatch object are defined for iterate, length, eltype, keys, haskey, and getindex, where keys are the names or numbers of a capture group. See keys for more information.\n\nTuple(m), NamedTuple(m), and Dict(m) can be used to construct more flexible collection types from RegexMatch objects.\n\ncompat: Julia 1.11\nConstructing NamedTuples and Dicts from RegexMatches requires Julia 1.11\n\nExamples\n\njulia> m = match(r\"(?<hour>\\d+):(?<minute>\\d+)(am|pm)?\", \"11:30 in the morning\")\nRegexMatch(\"11:30\", hour=\"11\", minute=\"30\", 3=nothing)\n\njulia> m.match\n\"11:30\"\n\njulia> m.captures\n3-element Vector{Union{Nothing, SubString{String}}}:\n \"11\"\n \"30\"\n nothing\n\n\njulia> m[\"minute\"]\n\"30\"\n\njulia> hr, min, ampm = m; # destructure capture groups by iteration\n\njulia> hr\n\"11\"\n\njulia> Dict(m)\nDict{Any, Union{Nothing, SubString{String}}} with 3 entries:\n  \"hour\"   => \"11\"\n  3        => nothing\n  \"minute\" => \"30\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.keys-Tuple{RegexMatch}","page":"Strings","title":"Base.keys","category":"method","text":"keys(m::RegexMatch)::Vector\n\nReturn a vector of keys for all capture groups of the underlying regex. A key is included even if the capture group fails to match. That is, idx will be in the return value even if m[idx] == nothing.\n\nUnnamed capture groups will have integer keys corresponding to their index. Named capture groups will have string keys.\n\ncompat: Julia 1.7\nThis method was added in Julia 1.7\n\nExamples\n\njulia> keys(match(r\"(?<hour>\\d+):(?<minute>\\d+)(am|pm)?\", \"11:30\"))\n3-element Vector{Any}:\n  \"hour\"\n  \"minute\"\n 3\n\n\n\n\n\n"},{"location":"base/strings.html#Base.isless-Tuple{AbstractString, AbstractString}","page":"Strings","title":"Base.isless","category":"method","text":"isless(a::AbstractString, b::AbstractString)::Bool\n\nTest whether string a comes before string b in alphabetical order (technically, in lexicographical order by Unicode code points).\n\nExamples\n\njulia> isless(\"a\", \"b\")\ntrue\n\njulia> isless(\"β\", \"α\")\nfalse\n\njulia> isless(\"a\", \"a\")\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.:==-Tuple{AbstractString, AbstractString}","page":"Strings","title":"Base.:==","category":"method","text":"==(a::AbstractString, b::AbstractString)::Bool\n\nTest whether two strings are equal character by character (technically, Unicode code point by code point). Should either string be a AnnotatedString the string properties must match too.\n\nExamples\n\njulia> \"abc\" == \"abc\"\ntrue\n\njulia> \"abc\" == \"αβγ\"\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.cmp-Tuple{AbstractString, AbstractString}","page":"Strings","title":"Base.cmp","category":"method","text":"cmp(a::AbstractString, b::AbstractString)::Int\n\nCompare two strings. Return 0 if both strings have the same length and the character at each index is the same in both strings. Return -1 if a is a prefix of b, or if a comes before b in alphabetical order. Return 1 if b is a prefix of a, or if b comes before a in alphabetical order (technically, lexicographical order by Unicode code points).\n\nExamples\n\njulia> cmp(\"abc\", \"abc\")\n0\n\njulia> cmp(\"ab\", \"abc\")\n-1\n\njulia> cmp(\"abc\", \"ab\")\n1\n\njulia> cmp(\"ab\", \"ac\")\n-1\n\njulia> cmp(\"ac\", \"ab\")\n1\n\njulia> cmp(\"α\", \"a\")\n1\n\njulia> cmp(\"b\", \"β\")\n-1\n\n\n\n\n\n"},{"location":"base/strings.html#Base.lpad","page":"Strings","title":"Base.lpad","category":"function","text":"lpad(s, n::Integer, p::Union{AbstractChar,AbstractString}=' ')::String\n\nStringify s and pad the resulting string on the left with p to make it n characters (in textwidth) long. If s is already n characters long, an equal string is returned. Pad with spaces by default.\n\nExamples\n\njulia> lpad(\"March\", 10)\n\"     March\"\n\ncompat: Julia 1.7\nIn Julia 1.7, this function was changed to use textwidth rather than a raw character (codepoint) count.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.rpad","page":"Strings","title":"Base.rpad","category":"function","text":"rpad(s, n::Integer, p::Union{AbstractChar,AbstractString}=' ')::String\n\nStringify s and pad the resulting string on the right with p to make it n characters (in textwidth) long. If s is already n characters long, an equal string is returned. Pad with spaces by default.\n\nExamples\n\njulia> rpad(\"March\", 20)\n\"March               \"\n\ncompat: Julia 1.7\nIn Julia 1.7, this function was changed to use textwidth rather than a raw character (codepoint) count.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.ltruncate","page":"Strings","title":"Base.ltruncate","category":"function","text":"ltruncate(str::AbstractString, maxwidth::Integer, replacement::Union{AbstractString,AbstractChar} = '…')\n\nTruncate str to at most maxwidth columns (as estimated by textwidth), replacing the first characters with replacement if necessary. The default replacement string is \"…\".\n\nExamples\n\njulia> s = ltruncate(\"🍕🍕 I love 🍕\", 10)\n\"…I love 🍕\"\n\njulia> textwidth(s)\n10\n\njulia> ltruncate(\"foo\", 3)\n\"foo\"\n\ncompat: Julia 1.12\nThis function was added in Julia 1.12.\n\nSee also rtruncate and ctruncate.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.rtruncate","page":"Strings","title":"Base.rtruncate","category":"function","text":"rtruncate(str::AbstractString, maxwidth::Integer, replacement::Union{AbstractString,AbstractChar} = '…')\n\nTruncate str to at most maxwidth columns (as estimated by textwidth), replacing the last characters with replacement if necessary. The default replacement string is \"…\".\n\nExamples\n\njulia> s = rtruncate(\"🍕🍕 I love 🍕\", 10)\n\"🍕🍕 I lo…\"\n\njulia> textwidth(s)\n10\n\njulia> rtruncate(\"foo\", 3)\n\"foo\"\n\ncompat: Julia 1.12\nThis function was added in Julia 1.12.\n\nSee also ltruncate and ctruncate.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.ctruncate","page":"Strings","title":"Base.ctruncate","category":"function","text":"ctruncate(str::AbstractString, maxwidth::Integer, replacement::Union{AbstractString,AbstractChar} = '…'; prefer_left::Bool = true)\n\nTruncate str to at most maxwidth columns (as estimated by textwidth), replacing the middle characters with replacement if necessary. The default replacement string is \"…\". By default, the truncation prefers keeping chars on the left, but this can be changed by setting prefer_left to false.\n\nExamples\n\njulia> s = ctruncate(\"🍕🍕 I love 🍕\", 10)\n\"🍕🍕 …e 🍕\"\n\njulia> textwidth(s)\n10\n\njulia> ctruncate(\"foo\", 3)\n\"foo\"\n\ncompat: Julia 1.12\nThis function was added in Julia 1.12.\n\nSee also ltruncate and rtruncate.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findfirst-Tuple{AbstractString, AbstractString}","page":"Strings","title":"Base.findfirst","category":"method","text":"findfirst(pattern::AbstractString, string::AbstractString)\nfindfirst(pattern::AbstractPattern, string::String)\n\nFind the first occurrence of pattern in string. Equivalent to findnext(pattern, string, firstindex(s)).\n\nExamples\n\njulia> findfirst(\"z\", \"Hello to the world\") # returns nothing, but not printed in the REPL\n\njulia> findfirst(\"Julia\", \"JuliaLang\")\n1:5\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findnext-Tuple{AbstractString, AbstractString, Integer}","page":"Strings","title":"Base.findnext","category":"method","text":"findnext(pattern::AbstractString, string::AbstractString, start::Integer)\nfindnext(pattern::AbstractPattern, string::String, start::Integer)\n\nFind the next occurrence of pattern in string starting at position start. pattern can be either a string, or a regular expression, in which case string must be of type String.\n\nThe return value is a range of indices where the matching sequence is found, such that s[findnext(x, s, i)] == x:\n\nfindnext(\"substring\", string, i) == start:stop such that string[start:stop] == \"substring\" and i <= start, or nothing if unmatched.\n\nExamples\n\njulia> findnext(\"z\", \"Hello to the world\", 1) === nothing\ntrue\n\njulia> findnext(\"o\", \"Hello to the world\", 6)\n8:8\n\njulia> findnext(\"Lang\", \"JuliaLang\", 2)\n6:9\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findnext-Tuple{AbstractChar, AbstractString, Integer}","page":"Strings","title":"Base.findnext","category":"method","text":"findnext(ch::AbstractChar, string::AbstractString, start::Integer)\n\nFind the next occurrence of character ch in string starting at position start.\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\nExamples\n\njulia> findnext('z', \"Hello to the world\", 1) === nothing\ntrue\n\njulia> findnext('o', \"Hello to the world\", 6)\n8\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findlast-Tuple{AbstractString, AbstractString}","page":"Strings","title":"Base.findlast","category":"method","text":"findlast(pattern::AbstractString, string::AbstractString)\n\nFind the last occurrence of pattern in string. Equivalent to findprev(pattern, string, lastindex(string)).\n\nExamples\n\njulia> findlast(\"o\", \"Hello to the world\")\n15:15\n\njulia> findfirst(\"Julia\", \"JuliaLang\")\n1:5\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findlast-Tuple{AbstractChar, AbstractString}","page":"Strings","title":"Base.findlast","category":"method","text":"findlast(ch::AbstractChar, string::AbstractString)\n\nFind the last occurrence of character ch in string.\n\ncompat: Julia 1.3\nThis method requires at least Julia 1.3.\n\nExamples\n\njulia> findlast('p', \"happy\")\n4\n\njulia> findlast('z', \"happy\") === nothing\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.findprev-Tuple{AbstractString, AbstractString, Integer}","page":"Strings","title":"Base.findprev","category":"method","text":"findprev(pattern::AbstractString, string::AbstractString, start::Integer)\n\nFind the previous occurrence of pattern in string starting at position start.\n\nThe return value is a range of indices where the matching sequence is found, such that s[findprev(x, s, i)] == x:\n\nfindprev(\"substring\", string, i) == start:stop such that string[start:stop] == \"substring\" and stop <= i, or nothing if unmatched.\n\nExamples\n\njulia> findprev(\"z\", \"Hello to the world\", 18) === nothing\ntrue\n\njulia> findprev(\"o\", \"Hello to the world\", 18)\n15:15\n\njulia> findprev(\"Julia\", \"JuliaLang\", 6)\n1:5\n\n\n\n\n\n"},{"location":"base/strings.html#Base.occursin","page":"Strings","title":"Base.occursin","category":"function","text":"occursin(haystack)\n\nCreate a function that checks whether its argument occurs in haystack, i.e. a function equivalent to needle -> occursin(needle, haystack).\n\nThe returned function is of type Base.Fix2{typeof(occursin)}.\n\ncompat: Julia 1.6\nThis method requires Julia 1.6 or later.\n\nExamples\n\njulia> search_f = occursin(\"JuliaLang is a programming language\");\n\njulia> search_f(\"JuliaLang\")\ntrue\n\njulia> search_f(\"Python\")\nfalse\n\n\n\n\n\noccursin(needle::Union{AbstractString,AbstractPattern,AbstractChar}, haystack::AbstractString)\n\nDetermine whether the first argument is a substring of the second. If needle is a regular expression, checks whether haystack contains a match.\n\nExamples\n\njulia> occursin(\"Julia\", \"JuliaLang is pretty cool!\")\ntrue\n\njulia> occursin('a', \"JuliaLang is pretty cool!\")\ntrue\n\njulia> occursin(r\"a.a\", \"aba\")\ntrue\n\njulia> occursin(r\"a.a\", \"abba\")\nfalse\n\nSee also contains.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.reverse-Tuple{Union{SubString{String}, String}}","page":"Strings","title":"Base.reverse","category":"method","text":"reverse(s::AbstractString)::AbstractString\n\nReverses a string. Technically, this function reverses the codepoints in a string and its main utility is for reversed-order string processing, especially for reversed regular-expression searches. See also reverseind to convert indices in s to indices in reverse(s) and vice-versa, and graphemes from module Unicode to operate on user-visible \"characters\" (graphemes) rather than codepoints. See also Iterators.reverse for reverse-order iteration without making a copy. Custom string types must implement the reverse function themselves and should typically return a string with the same type and encoding. If they return a string with a different encoding, they must also override reverseind for that string type to satisfy s[reverseind(s,i)] == reverse(s)[i].\n\nExamples\n\njulia> reverse(\"JuliaLang\")\n\"gnaLailuJ\"\n\nnote: Note\nThe examples below may be rendered differently on different systems. The comments indicate how they're supposed to be rendered\n\nCombining characters can lead to surprising results:\n\njulia> reverse(\"ax̂e\") # hat is above x in the input, above e in the output\n\"êxa\"\n\njulia> using Unicode\n\njulia> join(reverse(collect(graphemes(\"ax̂e\")))) # reverses graphemes; hat is above x in both in- and output\n\"ex̂a\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.replace-Tuple{IO, AbstractString, Vararg{Pair}}","page":"Strings","title":"Base.replace","category":"method","text":"replace([io::IO], s::AbstractString, pat=>r, [pat2=>r2, ...]; [count::Integer])\n\nSearch for the given pattern pat in s, and replace each occurrence with r. If count is provided, replace at most count occurrences. pat may be a single character, a vector or a set of characters, a string, or a regular expression. If r is a function, each occurrence is replaced with r(s) where s is the matched substring (when pat is a AbstractPattern or AbstractString) or character (when pat is an AbstractChar or a collection of AbstractChar). If pat is a regular expression and r is a SubstitutionString, then capture group references in r are replaced with the corresponding matched text. To remove instances of pat from string, set r to the empty String (\"\").\n\nThe return value is a new string after the replacements.  If the io::IO argument is supplied, the transformed string is instead written to io (returning io). (For example, this can be used in conjunction with an IOBuffer to re-use a pre-allocated buffer array in-place.)\n\nMultiple patterns can be specified: The input string will be scanned only once from start (left) to end (right), and the first matching replacement will be applied to each substring. Replacements are applied in the order of the arguments provided if they match substrings starting at the same input string position. Thus, only one pattern will be applied to any character, and the patterns will only be applied to the input text, not the replacements.\n\ncompat: Julia 1.7\nSupport for multiple patterns requires version 1.7.\n\ncompat: Julia 1.10\nThe io::IO argument requires version 1.10.\n\nExamples\n\njulia> replace(\"Python is a programming language.\", \"Python\" => \"Julia\")\n\"Julia is a programming language.\"\n\njulia> replace(\"The quick foxes run quickly.\", \"quick\" => \"slow\", count=1)\n\"The slow foxes run quickly.\"\n\njulia> replace(\"The quick foxes run quickly.\", \"quick\" => \"\", count=1)\n\"The  foxes run quickly.\"\n\njulia> replace(\"The quick foxes run quickly.\", r\"fox(es)?\" => s\"bus\\1\")\n\"The quick buses run quickly.\"\n\njulia> replace(\"abcabc\", \"a\" => \"b\", \"b\" => \"c\", r\".+\" => \"a\")\n\"bca\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.eachsplit","page":"Strings","title":"Base.eachsplit","category":"function","text":"eachsplit(str::AbstractString, dlm; limit::Integer=0, keepempty::Bool=true)\neachsplit(str::AbstractString; limit::Integer=0, keepempty::Bool=false)\n\nSplit str on occurrences of the delimiter(s) dlm and return an iterator over the substrings.  dlm can be any of the formats allowed by findnext's first argument (i.e. as a string, regular expression or a function), or as a single character or collection of characters.\n\nIf dlm is omitted, it defaults to isspace.\n\nThe optional keyword arguments are:\n\nlimit: the maximum size of the result. limit=0 implies no maximum (default)\nkeepempty: whether empty fields should be kept in the result. Default is false without a dlm argument, true with a dlm argument.\n\nSee also split.\n\ncompat: Julia 1.8\nThe eachsplit function requires at least Julia 1.8.\n\nExamples\n\njulia> a = \"Ma.rch\"\n\"Ma.rch\"\n\njulia> b = eachsplit(a, \".\")\nBase.SplitIterator{String, String}(\"Ma.rch\", \".\", 0, true)\n\njulia> collect(b)\n2-element Vector{SubString{String}}:\n \"Ma\"\n \"rch\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.eachrsplit","page":"Strings","title":"Base.eachrsplit","category":"function","text":"eachrsplit(str::AbstractString, dlm; limit::Integer=0, keepempty::Bool=true)\neachrsplit(str::AbstractString; limit::Integer=0, keepempty::Bool=false)\n\nReturn an iterator over SubStrings of str, produced when splitting on the delimiter(s) dlm, and yielded in reverse order (from right to left). dlm can be any of the formats allowed by findprev's first argument (i.e. a string, a single character or a function), or a collection of characters.\n\nIf dlm is omitted, it defaults to isspace, and keepempty default to false.\n\nThe optional keyword arguments are:\n\nIf limit > 0, the iterator will split at most limit - 1 times before returning the rest of the string unsplit. limit < 1 implies no cap to splits (default).\nkeepempty: whether empty fields should be returned when iterating Default is false without a dlm argument, true with a dlm argument.\n\nNote that unlike split, rsplit and eachsplit, this function iterates the substrings right to left as they occur in the input.\n\nSee also eachsplit, rsplit.\n\ncompat: Julia 1.11\nThis function requires Julia 1.11 or later.\n\nExamples\n\njulia> a = \"Ma.r.ch\";\n\njulia> collect(eachrsplit(a, \".\")) == [\"ch\", \"r\", \"Ma\"]\ntrue\n\njulia> collect(eachrsplit(a, \".\"; limit=2)) == [\"ch\", \"Ma.r\"]\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.split","page":"Strings","title":"Base.split","category":"function","text":"split(str::AbstractString, dlm; limit::Integer=0, keepempty::Bool=true)\nsplit(str::AbstractString; limit::Integer=0, keepempty::Bool=false)\n\nSplit str into an array of substrings on occurrences of the delimiter(s) dlm.  dlm can be any of the formats allowed by findnext's first argument (i.e. as a string, regular expression or a function), or as a single character or collection of characters.\n\nIf dlm is omitted, it defaults to isspace.\n\nThe optional keyword arguments are:\n\nlimit: the maximum size of the result. limit=0 implies no maximum (default)\nkeepempty: whether empty fields should be kept in the result. Default is false without a dlm argument, true with a dlm argument.\n\nSee also rsplit, eachsplit.\n\nExamples\n\njulia> a = \"Ma.rch\"\n\"Ma.rch\"\n\njulia> split(a, \".\")\n2-element Vector{SubString{String}}:\n \"Ma\"\n \"rch\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.rsplit","page":"Strings","title":"Base.rsplit","category":"function","text":"rsplit(s::AbstractString; limit::Integer=0, keepempty::Bool=false)\nrsplit(s::AbstractString, chars; limit::Integer=0, keepempty::Bool=true)\n\nSimilar to split, but starting from the end of the string.\n\nExamples\n\njulia> a = \"M.a.r.c.h\"\n\"M.a.r.c.h\"\n\njulia> rsplit(a, \".\")\n5-element Vector{SubString{String}}:\n \"M\"\n \"a\"\n \"r\"\n \"c\"\n \"h\"\n\njulia> rsplit(a, \".\"; limit=1)\n1-element Vector{SubString{String}}:\n \"M.a.r.c.h\"\n\njulia> rsplit(a, \".\"; limit=2)\n2-element Vector{SubString{String}}:\n \"M.a.r.c\"\n \"h\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.strip","page":"Strings","title":"Base.strip","category":"function","text":"strip([pred=isspace,] str::AbstractString)::SubString\nstrip(str::AbstractString, chars)::SubString\n\nRemove leading and trailing characters from str, either those specified by chars or those for which the function pred returns true.\n\nThe default behaviour is to remove leading and trailing whitespace and delimiters: see isspace for precise details.\n\nThe optional chars argument specifies which characters to remove: it can be a single character, vector or set of characters.\n\nSee also lstrip and rstrip.\n\ncompat: Julia 1.2\nThe method which accepts a predicate function requires Julia 1.2 or later.\n\nExamples\n\njulia> strip(\"{3, 5}\\n\", ['{', '}', '\\n'])\n\"3, 5\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.lstrip","page":"Strings","title":"Base.lstrip","category":"function","text":"lstrip([pred=isspace,] str::AbstractString)::SubString\nlstrip(str::AbstractString, chars)::SubString\n\nRemove leading characters from str, either those specified by chars or those for which the function pred returns true.\n\nThe default behaviour is to remove leading whitespace and delimiters: see isspace for precise details.\n\nThe optional chars argument specifies which characters to remove: it can be a single character, or a vector or set of characters.\n\nSee also strip and rstrip.\n\nExamples\n\njulia> a = lpad(\"March\", 20)\n\"               March\"\n\njulia> lstrip(a)\n\"March\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.rstrip","page":"Strings","title":"Base.rstrip","category":"function","text":"rstrip([pred=isspace,] str::AbstractString)::SubString\nrstrip(str::AbstractString, chars)::SubString\n\nRemove trailing characters from str, either those specified by chars or those for which the function pred returns true.\n\nThe default behaviour is to remove trailing whitespace and delimiters: see isspace for precise details.\n\nThe optional chars argument specifies which characters to remove: it can be a single character, or a vector or set of characters.\n\nSee also strip and lstrip.\n\nExamples\n\njulia> a = rpad(\"March\", 20)\n\"March               \"\n\njulia> rstrip(a)\n\"March\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.startswith","page":"Strings","title":"Base.startswith","category":"function","text":"startswith(s::AbstractString, prefix::Regex)\n\nReturn true if s starts with the regex pattern, prefix.\n\nnote: Note\nstartswith does not compile the anchoring into the regular expression, but instead passes the anchoring as match_option to PCRE. If compile time is amortized, occursin(r\"^...\", s) is faster than startswith(s, r\"...\").\n\nSee also occursin, endswith, match\n\ncompat: Julia 1.2\nThis method requires at least Julia 1.2.\n\nExamples\n\njulia> startswith(\"JuliaLang\", r\"Julia|Romeo\")\ntrue\n\n\n\n\n\nstartswith(prefix)\n\nCreate a function that checks whether its argument starts with prefix, i.e. a function equivalent to y -> startswith(y, prefix).\n\nThe returned function is of type Base.Fix2{typeof(startswith)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.5\nThe single argument startswith(prefix) requires at least Julia 1.5.\n\nExamples\n\njulia> startswith(\"Julia\")(\"JuliaLang\")\ntrue\n\njulia> startswith(\"Julia\")(\"Ends with Julia\")\nfalse\n\n\n\n\n\nstartswith(io::IO, prefix::Union{AbstractString,Base.Chars})\n\nCheck if an IO object starts with a prefix, which can be either a string, a character, or a tuple/vector/set of characters.  See also peek.\n\n\n\n\n\nstartswith(s::AbstractString, prefix::Union{AbstractString,Base.Chars})\n\nReturn true if s starts with prefix, which can be a string, a character, or a tuple/vector/set of characters. If prefix is a tuple/vector/set of characters, test whether the first character of s belongs to that set.\n\nSee also endswith, contains.\n\nExamples\n\njulia> startswith(\"JuliaLang\", \"Julia\")\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.endswith","page":"Strings","title":"Base.endswith","category":"function","text":"endswith(s::AbstractString, suffix::Regex)\n\nReturn true if s ends with the regex pattern, suffix.\n\nnote: Note\nendswith does not compile the anchoring into the regular expression, but instead passes the anchoring as match_option to PCRE. If compile time is amortized, occursin(r\"...$\", s) is faster than endswith(s, r\"...\").\n\nSee also occursin, startswith, match\n\ncompat: Julia 1.2\nThis method requires at least Julia 1.2.\n\nExamples\n\njulia> endswith(\"JuliaLang\", r\"Lang|Roberts\")\ntrue\n\n\n\n\n\nendswith(suffix)\n\nCreate a function that checks whether its argument ends with suffix, i.e. a function equivalent to y -> endswith(y, suffix).\n\nThe returned function is of type Base.Fix2{typeof(endswith)}, which can be used to implement specialized methods.\n\ncompat: Julia 1.5\nThe single argument endswith(suffix) requires at least Julia 1.5.\n\nExamples\n\njulia> endswith(\"Julia\")(\"Ends with Julia\")\ntrue\n\njulia> endswith(\"Julia\")(\"JuliaLang\")\nfalse\n\n\n\n\n\nendswith(s::AbstractString, suffix::Union{AbstractString,Base.Chars})\n\nReturn true if s ends with suffix, which can be a string, a character, or a tuple/vector/set of characters. If suffix is a tuple/vector/set of characters, test whether the last character of s belongs to that set.\n\nSee also startswith, contains.\n\nExamples\n\njulia> endswith(\"Sunday\", \"day\")\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.contains","page":"Strings","title":"Base.contains","category":"function","text":"contains(needle)\n\nCreate a function that checks whether its argument contains needle, i.e. a function equivalent to haystack -> contains(haystack, needle).\n\nThe returned function is of type Base.Fix2{typeof(contains)}, which can be used to implement specialized methods.\n\n\n\n\n\ncontains(haystack::AbstractString, needle)\n\nReturn true if haystack contains needle. This is the same as occursin(needle, haystack), but is provided for consistency with startswith(haystack, needle) and endswith(haystack, needle).\n\nSee also occursin, in, issubset.\n\nExamples\n\njulia> contains(\"JuliaLang is pretty cool!\", \"Julia\")\ntrue\n\njulia> contains(\"JuliaLang is pretty cool!\", 'a')\ntrue\n\njulia> contains(\"aba\", r\"a.a\")\ntrue\n\njulia> contains(\"abba\", r\"a.a\")\nfalse\n\ncompat: Julia 1.5\nThe contains function requires at least Julia 1.5.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.first-Tuple{AbstractString, Integer}","page":"Strings","title":"Base.first","category":"method","text":"first(s::AbstractString, n::Integer)\n\nGet a string consisting of the first n characters of s.\n\nExamples\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 1)\n\"∀\"\n\njulia> first(\"∀ϵ≠0: ϵ²>0\", 3)\n\"∀ϵ≠\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.last-Tuple{AbstractString, Integer}","page":"Strings","title":"Base.last","category":"method","text":"last(s::AbstractString, n::Integer)\n\nGet a string consisting of the last n characters of s.\n\nExamples\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 0)\n\"\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 1)\n\"0\"\n\njulia> last(\"∀ϵ≠0: ϵ²>0\", 3)\n\"²>0\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.uppercase","page":"Strings","title":"Base.Unicode.uppercase","category":"function","text":"uppercase(s::AbstractString)\n\nReturn s with all characters converted to uppercase.\n\nSee also lowercase, titlecase, uppercasefirst.\n\nExamples\n\njulia> uppercase(\"Julia\")\n\"JULIA\"\n\n\n\n\n\nuppercase(c::AbstractChar)\n\nConvert c to uppercase.\n\nSee also lowercase, titlecase.\n\nExamples\n\njulia> uppercase('a')\n'A': ASCII/Unicode U+0041 (category Lu: Letter, uppercase)\n\njulia> uppercase('ê')\n'Ê': Unicode U+00CA (category Lu: Letter, uppercase)\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.lowercase","page":"Strings","title":"Base.Unicode.lowercase","category":"function","text":"lowercase(s::AbstractString)\n\nReturn s with all characters converted to lowercase.\n\nSee also uppercase, titlecase, lowercasefirst.\n\nExamples\n\njulia> lowercase(\"STRINGS AND THINGS\")\n\"strings and things\"\n\n\n\n\n\nlowercase(c::AbstractChar)\n\nConvert c to lowercase.\n\nSee also uppercase, titlecase.\n\nExamples\n\njulia> lowercase('A')\n'a': ASCII/Unicode U+0061 (category Ll: Letter, lowercase)\n\njulia> lowercase('Ö')\n'ö': Unicode U+00F6 (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.titlecase","page":"Strings","title":"Base.Unicode.titlecase","category":"function","text":"titlecase(s::AbstractString; [wordsep::Function], strict::Bool=true)::String\n\nCapitalize the first character of each word in s; if strict is true, every other character is converted to lowercase, otherwise they are left unchanged. By default, all non-letters beginning a new grapheme are considered as word separators; a predicate can be passed as the wordsep keyword to determine which characters should be considered as word separators. See also uppercasefirst to capitalize only the first character in s.\n\nSee also uppercase, lowercase, uppercasefirst.\n\nExamples\n\njulia> titlecase(\"the JULIA programming language\")\n\"The Julia Programming Language\"\n\njulia> titlecase(\"ISS - international space station\", strict=false)\n\"ISS - International Space Station\"\n\njulia> titlecase(\"a-a b-b\", wordsep = c->c==' ')\n\"A-a B-b\"\n\n\n\n\n\ntitlecase(c::AbstractChar)\n\nConvert c to titlecase. This may differ from uppercase for digraphs, compare the example below.\n\nSee also uppercase, lowercase.\n\nExamples\n\njulia> titlecase('a')\n'A': ASCII/Unicode U+0041 (category Lu: Letter, uppercase)\n\njulia> titlecase('ǆ')\n'ǅ': Unicode U+01C5 (category Lt: Letter, titlecase)\n\njulia> uppercase('ǆ')\n'Ǆ': Unicode U+01C4 (category Lu: Letter, uppercase)\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.uppercasefirst","page":"Strings","title":"Base.Unicode.uppercasefirst","category":"function","text":"uppercasefirst(s::AbstractString)::String\n\nReturn s with the first character converted to uppercase (technically \"title case\" for Unicode). See also titlecase to capitalize the first character of every word in s.\n\nSee also lowercasefirst, uppercase, lowercase, titlecase.\n\nExamples\n\njulia> uppercasefirst(\"python\")\n\"Python\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.lowercasefirst","page":"Strings","title":"Base.Unicode.lowercasefirst","category":"function","text":"lowercasefirst(s::AbstractString)\n\nReturn s with the first character converted to lowercase.\n\nSee also uppercasefirst, uppercase, lowercase, titlecase.\n\nExamples\n\njulia> lowercasefirst(\"Julia\")\n\"julia\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.join","page":"Strings","title":"Base.join","category":"function","text":"join([io::IO,] iterator [, delim [, last]])\n\nJoin any iterator into a single string, inserting the given delimiter (if any) between adjacent items.  If last is given, it will be used instead of delim between the last two items.  Each item of iterator is converted to a string via print(io::IOBuffer, x). If io is given, the result is written to io rather than returned as a String.\n\nExamples\n\njulia> join([\"apples\", \"bananas\", \"pineapples\"], \", \", \" and \")\n\"apples, bananas and pineapples\"\n\njulia> join([1,2,3,4,5])\n\"12345\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.chop","page":"Strings","title":"Base.chop","category":"function","text":"chop(s::AbstractString; head::Integer = 0, tail::Integer = 1)\n\nRemove the first head and the last tail characters from s. The call chop(s) removes the last character from s. If it is requested to remove more characters than length(s) then an empty string is returned.\n\nSee also chomp, startswith, first.\n\nExamples\n\njulia> a = \"March\"\n\"March\"\n\njulia> chop(a)\n\"Marc\"\n\njulia> chop(a, head = 1, tail = 2)\n\"ar\"\n\njulia> chop(a, head = 5, tail = 5)\n\"\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.chopprefix","page":"Strings","title":"Base.chopprefix","category":"function","text":"chopprefix(s::AbstractString, prefix::Union{AbstractString,Regex,AbstractChar})::SubString\n\nRemove the prefix prefix from s. If s does not start with prefix, a string equal to s is returned.\n\nSee also chopsuffix.\n\ncompat: Julia 1.8\nThis function is available as of Julia 1.8.\n\ncompat: Julia 1.13\nThe method which accepts an AbstractChar prefix is available as of Julia 1.13.\n\nExamples\n\njulia> chopprefix(\"Hamburger\", \"Ham\")\n\"burger\"\n\njulia> chopprefix(\"Hamburger\", \"hotdog\")\n\"Hamburger\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.chopsuffix","page":"Strings","title":"Base.chopsuffix","category":"function","text":"chopsuffix(s::AbstractString, suffix::Union{AbstractString,Regex,AbstractChar})::SubString\n\nRemove the suffix suffix from s. If s does not end with suffix, a string equal to s is returned.\n\nSee also chopprefix.\n\ncompat: Julia 1.8\nThis function is available as of Julia 1.8.\n\ncompat: Julia 1.13\nThe method which accepts an AbstractChar suffix is available as of Julia 1.13.\n\nExamples\n\njulia> chopsuffix(\"Hamburger\", \"er\")\n\"Hamburg\"\n\njulia> chopsuffix(\"Hamburger\", \"hotdog\")\n\"Hamburger\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.chomp","page":"Strings","title":"Base.chomp","category":"function","text":"chomp(s::AbstractString)::SubString\n\nRemove a single trailing newline (i.e. \"\\r\\n\" or \"\\n\") from a string.\n\nSee also chop.\n\nExamples\n\njulia> chomp(\"Hello\\n\")\n\"Hello\"\n\njulia> chomp(\"World\\r\\n\")\n\"World\"\n\njulia> chomp(\"Julia\\r\\n\\n\")\n\"Julia\\r\\n\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.thisind","page":"Strings","title":"Base.thisind","category":"function","text":"thisind(s::AbstractString, i::Integer)::Int\n\nIf i is in bounds in s return the index of the start of the character whose encoding code unit i is part of. In other words, if i is the start of a character, return i; if i is not the start of a character, rewind until the start of a character and return that index. If i is equal to 0 or ncodeunits(s)+1 return i. In all other cases throw BoundsError.\n\nExamples\n\njulia> thisind(\"α\", 0)\n0\n\njulia> thisind(\"α\", 1)\n1\n\njulia> thisind(\"α\", 2)\n1\n\njulia> thisind(\"α\", 3)\n3\n\njulia> thisind(\"α\", 4)\nERROR: BoundsError: attempt to access 2-codeunit String at index [4]\n[...]\n\njulia> thisind(\"α\", -1)\nERROR: BoundsError: attempt to access 2-codeunit String at index [-1]\n[...]\n\n\n\n\n\n"},{"location":"base/strings.html#Base.nextind-Tuple{AbstractString, Integer, Integer}","page":"Strings","title":"Base.nextind","category":"method","text":"nextind(str::AbstractString, i::Integer, n::Integer=1)::Int\n\nCase n == 1\nIf i is in bounds in str return the index of the start of the character whose encoding starts after index i. In other words, if i is the start of a character, return the start of the next character; if i is not the start of a character, move forward until the start of a character and return that index. If i is equal to 0 return 1. If i is in bounds but greater or equal to lastindex(str) return ncodeunits(str)+1. Otherwise throw BoundsError.\nCase n > 1\nBehaves like applying n times nextind for n==1. The only difference is that if n is so large that applying nextind would reach ncodeunits(str)+1 then each remaining iteration increases the returned value by 1. This means that in this case nextind can return a value greater than ncodeunits(str)+1.\nCase n == 0\nReturn i only if i is a valid index in str or is equal to 0. Otherwise StringIndexError or BoundsError is thrown.\n\nExamples\n\njulia> nextind(\"α\", 0)\n1\n\njulia> nextind(\"α\", 1)\n3\n\njulia> nextind(\"α\", 3)\nERROR: BoundsError: attempt to access 2-codeunit String at index [3]\n[...]\n\njulia> nextind(\"α\", 0, 2)\n3\n\njulia> nextind(\"α\", 1, 2)\n4\n\n\n\n\n\n"},{"location":"base/strings.html#Base.prevind-Tuple{AbstractString, Integer, Integer}","page":"Strings","title":"Base.prevind","category":"method","text":"prevind(str::AbstractString, i::Integer, n::Integer=1)::Int\n\nCase n == 1\nIf i is in bounds in str return the index of the start of the character whose encoding starts before index i. In other words, if i is the start of a character, return the start of the previous character; if i is not the start of a character, rewind until the start of a character and return that index. If i is equal to 1 return 0. If i is equal to ncodeunits(str)+1 return lastindex(str). Otherwise throw BoundsError.\nCase n > 1\nBehaves like applying n times prevind for n==1. The only difference is that if n is so large that applying prevind would reach 0 then each remaining iteration decreases the returned value by 1. This means that in this case prevind can return a negative value.\nCase n == 0\nReturn i only if i is a valid index in str or is equal to ncodeunits(str)+1. Otherwise StringIndexError or BoundsError is thrown.\n\nExamples\n\njulia> prevind(\"α\", 3)\n1\n\njulia> prevind(\"α\", 1)\n0\n\njulia> prevind(\"α\", 0)\nERROR: BoundsError: attempt to access 2-codeunit String at index [0]\n[...]\n\njulia> prevind(\"α\", 2, 2)\n0\n\njulia> prevind(\"α\", 2, 3)\n-1\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.textwidth","page":"Strings","title":"Base.Unicode.textwidth","category":"function","text":"textwidth(s::AbstractString)\n\nGive the number of columns needed to print a string.\n\nExamples\n\njulia> textwidth(\"March\")\n5\n\n\n\n\n\ntextwidth(c)\n\nGive the number of columns needed to print a character.\n\nExamples\n\njulia> textwidth('α')\n1\n\njulia> textwidth('⛵')\n2\n\n\n\n\n\n"},{"location":"base/strings.html#Base.isascii","page":"Strings","title":"Base.isascii","category":"function","text":"isascii(cu::AbstractVector{CU}) where {CU <: Integer}::Bool\n\nTest whether all values in the vector belong to the ASCII character set (0x00 to 0x7f). This function is intended to be used by other string implementations that need a fast ASCII check.\n\n\n\n\n\nisascii(c::Union{AbstractChar,AbstractString})::Bool\n\nTest whether a character belongs to the ASCII character set, or whether this is true for all elements of a string.\n\nExamples\n\njulia> isascii('a')\ntrue\n\njulia> isascii('α')\nfalse\n\njulia> isascii(\"abc\")\ntrue\n\njulia> isascii(\"αβγ\")\nfalse\n\nFor example, isascii can be used as a predicate function for filter or replace to remove or replace non-ASCII characters, respectively:\n\njulia> filter(isascii, \"abcdeγfgh\") # discard non-ASCII chars\n\"abcdefgh\"\n\njulia> replace(\"abcdeγfgh\", !isascii=>' ') # replace non-ASCII chars with spaces\n\"abcde fgh\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.iscntrl","page":"Strings","title":"Base.Unicode.iscntrl","category":"function","text":"iscntrl(c::AbstractChar)::Bool\n\nTests whether a character is a control character. Control characters are the non-printing characters of the Latin-1 subset of Unicode.\n\nExamples\n\njulia> iscntrl('\\x01')\ntrue\n\njulia> iscntrl('a')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isdigit","page":"Strings","title":"Base.Unicode.isdigit","category":"function","text":"isdigit(c::AbstractChar)::Bool\n\nTests whether a character is an ASCII decimal digit (0-9).\n\nSee also: isletter.\n\nExamples\n\njulia> isdigit('❤')\nfalse\n\njulia> isdigit('9')\ntrue\n\njulia> isdigit('α')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isletter","page":"Strings","title":"Base.Unicode.isletter","category":"function","text":"isletter(c::AbstractChar)::Bool\n\nTest whether a character is a letter. A character is classified as a letter if it belongs to the Unicode general category Letter, i.e. a character whose category code begins with 'L'.\n\nSee also: isdigit.\n\nExamples\n\njulia> isletter('❤')\nfalse\n\njulia> isletter('α')\ntrue\n\njulia> isletter('9')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.islowercase","page":"Strings","title":"Base.Unicode.islowercase","category":"function","text":"islowercase(c::AbstractChar)::Bool\n\nTests whether a character is a lowercase letter (according to the Unicode standard's Lowercase derived property).\n\nSee also isuppercase.\n\nExamples\n\njulia> islowercase('α')\ntrue\n\njulia> islowercase('Γ')\nfalse\n\njulia> islowercase('❤')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isnumeric","page":"Strings","title":"Base.Unicode.isnumeric","category":"function","text":"isnumeric(c::AbstractChar)::Bool\n\nTests whether a character is numeric. A character is classified as numeric if it belongs to the Unicode general category Number, i.e. a character whose category code begins with 'N'.\n\nNote that this broad category includes characters such as ¾ and ௰. Use isdigit to check whether a character is a decimal digit between 0 and 9.\n\nExamples\n\njulia> isnumeric('௰')\ntrue\n\njulia> isnumeric('9')\ntrue\n\njulia> isnumeric('α')\nfalse\n\njulia> isnumeric('❤')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isprint","page":"Strings","title":"Base.Unicode.isprint","category":"function","text":"isprint(c::AbstractChar)::Bool\n\nTests whether a character is printable, including spaces, but not a control character.\n\nExamples\n\njulia> isprint('\\x01')\nfalse\n\njulia> isprint('A')\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.ispunct","page":"Strings","title":"Base.Unicode.ispunct","category":"function","text":"ispunct(c::AbstractChar)::Bool\n\nTests whether a character belongs to the Unicode general category Punctuation, i.e. a character whose category code begins with 'P'.\n\nnote: Note\nThis behavior is different from the ispunct function in C.\n\nExamples\n\njulia> ispunct('α')\nfalse\n\njulia> ispunct('=')\nfalse\n\njulia> ispunct('/')\ntrue\n\njulia> ispunct(';')\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isspace","page":"Strings","title":"Base.Unicode.isspace","category":"function","text":"isspace(c::AbstractChar)::Bool\n\nTests whether a character is any whitespace character. Includes ASCII characters '\\t', '\\n', '\\v', '\\f', '\\r', and ' ', Latin-1 character U+0085, and characters in Unicode category Zs.\n\nExamples\n\njulia> isspace('\\n')\ntrue\n\njulia> isspace('\\r')\ntrue\n\njulia> isspace(' ')\ntrue\n\njulia> isspace('\\x20')\ntrue\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isuppercase","page":"Strings","title":"Base.Unicode.isuppercase","category":"function","text":"isuppercase(c::AbstractChar)::Bool\n\nTests whether a character is an uppercase letter (according to the Unicode standard's Uppercase derived property).\n\nSee also islowercase.\n\nExamples\n\njulia> isuppercase('γ')\nfalse\n\njulia> isuppercase('Γ')\ntrue\n\njulia> isuppercase('❤')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.Unicode.isxdigit","page":"Strings","title":"Base.Unicode.isxdigit","category":"function","text":"isxdigit(c::AbstractChar)::Bool\n\nTest whether a character is a valid hexadecimal digit. Note that this does not include x (as in the standard 0x prefix).\n\nExamples\n\njulia> isxdigit('a')\ntrue\n\njulia> isxdigit('x')\nfalse\n\n\n\n\n\n"},{"location":"base/strings.html#Base.escape_string","page":"Strings","title":"Base.escape_string","category":"function","text":"escape_string(str::AbstractString[, esc]; keep=(), ascii=false, fullhex=false)::AbstractString\nescape_string(io, str::AbstractString[, esc]; keep=())::Nothing\n\nGeneral escaping of traditional C and Unicode escape sequences. The first form returns the escaped string, the second prints the result to io.\n\nBackslashes (\\) are escaped with a double-backslash (\"\\\\\"). Non-printable characters are escaped either with their standard C escape codes, \"\\0\" for NUL (if unambiguous), unicode code point (\"\\u\" prefix) or hex (\"\\x\" prefix).\n\nThe optional esc argument specifies any additional characters that should also be escaped by a prepending backslash (\" is also escaped by default in the first form).\n\nThe argument keep specifies a collection of characters which are to be kept as they are. Notice that esc has precedence here.\n\nThe argument ascii can be set to true to escape all non-ASCII characters, whereas the default ascii=false outputs printable Unicode characters as-is. (keep takes precedence over ascii.)\n\nThe argument fullhex can be set to true to require all \\u escapes to be printed with 4 hex digits, and \\U escapes to be printed with 8 hex digits, whereas by default (fullhex=false) they are printed with fewer digits if possible (omitting leading zeros).\n\nSee also unescape_string for the reverse operation.\n\ncompat: Julia 1.7\nThe keep argument is available as of Julia 1.7.\n\ncompat: Julia 1.12\nThe ascii and fullhex arguments require Julia 1.12.\n\nExamples\n\njulia> escape_string(\"aaa\\nbbb\")\n\"aaa\\\\nbbb\"\n\njulia> escape_string(\"aaa\\nbbb\"; keep = '\\n')\n\"aaa\\nbbb\"\n\njulia> escape_string(\"\\xfe\\xff\") # invalid utf-8\n\"\\\\xfe\\\\xff\"\n\njulia> escape_string(string('\\u2135','\\0')) # unambiguous\n\"ℵ\\\\0\"\n\njulia> escape_string(string('\\u2135','\\0','0')) # \\0 would be ambiguous\n\"ℵ\\\\x000\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.escape_raw_string","page":"Strings","title":"Base.escape_raw_string","category":"function","text":"escape_raw_string(s::AbstractString, delim='\"')::AbstractString\nescape_raw_string(io, s::AbstractString, delim='\"')\n\nEscape a string in the manner used for parsing raw string literals. For each double-quote (\") character in input string s (or delim if specified), this function counts the number n of preceding backslash (\\) characters, and then increases there the number of backslashes from n to 2n+1 (even for n = 0). It also doubles a sequence of backslashes at the end of the string.\n\nThis escaping convention is used in raw strings and other non-standard string literals. (It also happens to be the escaping convention expected by the Microsoft C/C++ compiler runtime when it parses a command-line string into the argv[] array.)\n\nSee also Base.escape_string().\n\n\n\n\n\n"},{"location":"base/strings.html#Base.unescape_string","page":"Strings","title":"Base.unescape_string","category":"function","text":"unescape_string(str::AbstractString, keep = ())::AbstractString\nunescape_string(io, s::AbstractString, keep = ())::Nothing\n\nGeneral unescaping of traditional C and Unicode escape sequences. The first form returns the escaped string, the second prints the result to io. The argument keep specifies a collection of characters which (along with backlashes) are to be kept as they are.\n\nThe following escape sequences are recognised:\n\nEscaped backslash (\\\\)\nEscaped double-quote (\\\")\nStandard C escape sequences (\\a, \\b, \\t, \\n, \\v, \\f, \\r, \\e)\nUnicode BMP code points (\\u with 1-4 trailing hex digits)\nAll Unicode code points (\\U with 1-8 trailing hex digits; max value = 0010ffff)\nHex bytes (\\x with 1-2 trailing hex digits)\nOctal bytes (\\ with 1-3 trailing octal digits)\n\nSee also escape_string.\n\nExamples\n\njulia> unescape_string(\"aaa\\\\nbbb\") # C escape sequence\n\"aaa\\nbbb\"\n\njulia> unescape_string(\"\\\\u03c0\") # unicode\n\"π\"\n\njulia> unescape_string(\"\\\\101\") # octal\n\"A\"\n\njulia> unescape_string(\"aaa \\\\g \\\\n\", ['g']) # using `keep` argument\n\"aaa \\\\g \\n\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.AnnotatedString","page":"Strings","title":"Base.AnnotatedString","category":"type","text":"AnnotatedString{S <: AbstractString} <: AbstractString\n\nA string with metadata, in the form of annotated regions.\n\nMore specifically, this is a simple wrapper around any other AbstractString that allows for regions of the wrapped string to be annotated with labeled values.\n\n                           C\n                    ┌──────┸─────────┐\n  \"this is an example annotated string\"\n  └──┰────────┼─────┘         │\n     A        └─────┰─────────┘\n                    B\n\nThe above diagram represents a AnnotatedString where three ranges have been annotated (labeled A, B, and C). Each annotation holds a label (Symbol) and a value (Any). These three pieces of information are held as a @NamedTuple{region::UnitRange{Int32}, label::Symbol, value}.\n\nLabels do not need to be unique, the same region can hold multiple annotations with the same label.\n\nCode written for AnnotatedStrings in general should conserve the following properties:\n\nWhich characters an annotation is applied to\nThe order in which annotations are applied to each character\n\nAdditional semantics may be introduced by specific uses of AnnotatedStrings.\n\nA corollary of these rules is that adjacent, consecutively placed, annotations with identical labels and values are equivalent to a single annotation spanning the combined range.\n\nSee also AnnotatedChar, annotatedstring, annotations, and annotate!.\n\nConstructors\n\nAnnotatedString(s::S<:AbstractString) -> AnnotatedString{S}\nAnnotatedString(s::S<:AbstractString, annotations::Vector{@NamedTuple{region::UnitRange{Int32}, label::Symbol, value}})\n\nA AnnotatedString can also be created with annotatedstring, which acts much like string but preserves any annotations present in the arguments.\n\nExamples\n\njulia> AnnotatedString(\"this is an example annotated string\",\n                    [(1:18, :A, 1), (12:28, :B, 2), (18:35, :C, 3)])\n\"this is an example annotated string\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.AnnotatedChar","page":"Strings","title":"Base.AnnotatedChar","category":"type","text":"AnnotatedChar{S <: AbstractChar} <: AbstractChar\n\nA Char with annotations.\n\nMore specifically, this is a simple wrapper around any other AbstractChar, which holds a list of arbitrary labelled annotations (@NamedTuple{label::Symbol, value}) with the wrapped character.\n\nSee also: AnnotatedString, annotatedstring, annotations, and annotate!.\n\nConstructors\n\nAnnotatedChar(s::S) -> AnnotatedChar{S}\nAnnotatedChar(s::S, annotations::Vector{@NamedTuple{label::Symbol, value}})\n\nExamples\n\njulia> AnnotatedChar('j', [(:label, 1)])\n'j': ASCII/Unicode U+006A (category Ll: Letter, lowercase)\n\n\n\n\n\n"},{"location":"base/strings.html#Base.annotatedstring","page":"Strings","title":"Base.annotatedstring","category":"function","text":"annotatedstring(values...)\n\nCreate a AnnotatedString from any number of values using their printed representation.\n\nThis acts like string, but takes care to preserve any annotations present (in the form of AnnotatedString or AnnotatedChar values).\n\nSee also AnnotatedString and AnnotatedChar.\n\nExamples\n\njulia> annotatedstring(\"now an AnnotatedString\")\n\"now an AnnotatedString\"\n\njulia> annotatedstring(AnnotatedString(\"annotated\", [(1:9, :label, 1)]), \", and unannotated\")\n\"annotated, and unannotated\"\n\n\n\n\n\n"},{"location":"base/strings.html#Base.annotations","page":"Strings","title":"Base.annotations","category":"function","text":"annotations(chr::AnnotatedChar)::Vector{@NamedTuple{label::Symbol, value}}\n\nGet all annotations of chr, in the form of a vector of annotation pairs.\n\n\n\n\n\nannotations(str::Union{AnnotatedString, SubString{AnnotatedString}},\n            [position::Union{Integer, UnitRange}]) ->\n    Vector{@NamedTuple{region::UnitRange{Int32}, label::Symbol, value}}\n\nGet all annotations that apply to str. Should position be provided, only annotations that overlap with position will be returned.\n\nAnnotations are provided together with the regions they apply to, in the form of a vector of region–annotation tuples.\n\nIn accordance with the semantics documented in AnnotatedString, the order of annotations returned matches the order in which they were applied.\n\nSee also: annotate!.\n\n\n\n\n\n"},{"location":"base/strings.html#Base.annotate!","page":"Strings","title":"Base.annotate!","category":"function","text":"annotate!(char::AnnotatedChar, label::Symbol, value::Any)\n\nAnnotate char with the labeled value (label, value).\n\n\n\n\n\nannotate!(str::AnnotatedString, [range::UnitRange{Int}], label::Symbol, value)\nannotate!(str::SubString{AnnotatedString}, [range::UnitRange{Int}], label::Symbol, value)\n\nAnnotate a range of str (or the entire string) with a labeled value (label, value). To remove existing label annotations, use a value of nothing.\n\nThe order in which annotations are applied to str is semantically meaningful, as described in AnnotatedString.\n\n\n\n\n\n"},{"location":"devdocs/ast.html#Julia-ASTs","page":"Julia ASTs","title":"Julia ASTs","category":"section","text":"Julia has two representations of code. First there is a surface syntax AST returned by the parser (e.g. the Meta.parse function), and manipulated by macros. It is a structured representation of code as it is written, constructed by julia-parser.scm from a character stream. Next there is a lowered form, or IR (intermediate representation), which is used by type inference and code generation. In the lowered form there are fewer types of nodes, all macros are expanded, and all control flow is converted to explicit branches and sequences of statements. The lowered form is constructed by julia-syntax.scm.\n\nFirst we will focus on the AST, since it is needed to write macros."},{"location":"devdocs/ast.html#Surface-syntax-AST","page":"Julia ASTs","title":"Surface syntax AST","category":"section","text":"Front end ASTs consist almost entirely of Exprs and atoms (e.g. symbols, numbers). There is generally a different expression head for each visually distinct syntactic form. Examples will be given in s-expression syntax. Each parenthesized list corresponds to an Expr, where the first element is the head. For example (call f x) corresponds to Expr(:call, :f, :x) in Julia."},{"location":"devdocs/ast.html#Calls","page":"Julia ASTs","title":"Calls","category":"section","text":"Input AST\nf(x) (call f x)\nf(x, y=1, z=2) (call f x (kw y 1) (kw z 2))\nf(x; y=1) (call f (parameters (kw y 1)) x)\nf(x...) (call f (... x))\n\ndo syntax:\n\nf(x) do a,b\n    body\nend\n\nparses as (do (call f x) (-> (tuple a b) (block body)))."},{"location":"devdocs/ast.html#Operators","page":"Julia ASTs","title":"Operators","category":"section","text":"Most uses of operators are just function calls, so they are parsed with the head call. However some operators are special forms (not necessarily function calls), and in those cases the operator itself is the expression head. In julia-parser.scm these are referred to as \"syntactic operators\". Some operators (+ and *) use N-ary parsing; chained calls are parsed as a single N-argument call. Finally, chains of comparisons have their own special expression structure.\n\nInput AST\nx+y (call + x y)\na+b+c+d (call + a b c d)\n2x (call * 2 x)\na&&b (&& a b)\nx += 1 (+= x 1)\na ? 1 : 2 (if a 1 2)\na,b (tuple a b)\na==b (call == a b)\n1<i<=n (comparison 1 < i <= n)\na.b (. a (quote b))\na.(b) (. a (tuple b))"},{"location":"devdocs/ast.html#Bracketed-forms","page":"Julia ASTs","title":"Bracketed forms","category":"section","text":"Input AST\na[i] (ref a i)\nt[i;j] (typed_vcat t i j)\nt[i j] (typed_hcat t i j)\nt[a b; c d] (typed_vcat t (row a b) (row c d))\nt[a b;;; c d] (typed_ncat t 3 (row a b) (row c d))\na{b} (curly a b)\na{b;c} (curly a (parameters c) b)\n[x] (vect x)\n[x,y] (vect x y)\n[x;y] (vcat x y)\n[x y] (hcat x y)\n[x y; z t] (vcat (row x y) (row z t))\n[x;y;; z;t;;;] (ncat 3 (nrow 2 (nrow 1 x y) (nrow 1 z t)))\n[x for y in z, a in b] (comprehension (generator x (= y z) (= a b)))\nT[x for y in z] (typed_comprehension T (generator x (= y z)))\n(a, b, c) (tuple a b c)\n(a; b; c) (block a b c)"},{"location":"devdocs/ast.html#Macros","page":"Julia ASTs","title":"Macros","category":"section","text":"Input AST\n@m x y (macrocall @m (line) x y)\nBase.@m x y (macrocall (. Base (quote @m)) (line) x y)\n@Base.m x y (macrocall (. Base (quote @m)) (line) x y)"},{"location":"devdocs/ast.html#Strings","page":"Julia ASTs","title":"Strings","category":"section","text":"Input AST\n\"a\" \"a\"\nx\"y\" (macrocall @x_str (line) \"y\")\nx\"y\"z (macrocall @x_str (line) \"y\" \"z\")\n\"x = $x\" (string \"x = \" x)\n`a b c` (macrocall @cmd (line) \"a b c\")\n\nDoc string syntax:\n\n\"some docs\"\nf(x) = x\n\nparses as (macrocall (|.| Core '@doc) (line) \"some docs\" (= (call f x) (block x)))."},{"location":"devdocs/ast.html#Imports-and-such","page":"Julia ASTs","title":"Imports and such","category":"section","text":"Input AST\nimport a (import (. a))\nimport a.b.c (import (. a b c))\nimport ...a (import (. . . . a))\nimport a.b, c.d (import (. a b) (. c d))\nimport Base: x (import (: (. Base) (. x)))\nimport Base: x, y (import (: (. Base) (. x) (. y)))\nexport a, b (export a b)\npublic a, b (public a b)\n\nusing has the same representation as import, but with expression head :using instead of :import.\n\nTo programmatically create a public statement, you can use Expr(:public, :a, :b) or, closer to regular code, Meta.parse(\"public a, b\"). This approach is necessary due to current limitations on public. The public keyword is only recognized at the syntactic top level within a file (parse_stmts) or module. This restriction was implemented to prevent breaking existing code that used public as an identifier when it was introduced in Julia 1.11."},{"location":"devdocs/ast.html#Numbers","page":"Julia ASTs","title":"Numbers","category":"section","text":"Julia supports more number types than many scheme implementations, so not all numbers are represented directly as scheme numbers in the AST.\n\nInput AST\n11111111111111111111 (macrocall @int128_str nothing \"11111111111111111111\")\n0xfffffffffffffffff (macrocall @uint128_str nothing \"0xfffffffffffffffff\")\n1111...many digits... (macrocall @big_str nothing \"1111....\")"},{"location":"devdocs/ast.html#Block-forms","page":"Julia ASTs","title":"Block forms","category":"section","text":"A block of statements is parsed as (block stmt1 stmt2 ...).\n\nIf statement:\n\nif a\n    b\nelseif c\n    d\nelse\n    e\nend\n\nparses as:\n\n(if a (block (line 2) b)\n    (elseif (block (line 3) c) (block (line 4) d)\n            (block (line 6) e)))\n\nA while loop parses as (while condition body).\n\nA for loop parses as (for (= var iter) body). If there is more than one iteration specification, they are parsed as a block: (for (block (= v1 iter1) (= v2 iter2)) body).\n\nbreak and continue are parsed as 0-argument expressions (break) and (continue).\n\nlet is parsed as (let (= var val) body) or (let (block (= var1 val1) (= var2 val2) ...) body), like for loops.\n\nA basic function definition is parsed as (function (call f x) body). A more complex example:\n\nfunction f(x::T; k = 1) where T\n    return x+1\nend\n\nparses as:\n\n(function (where (call f (parameters (kw k 1))\n                       (:: x T))\n                 T)\n          (block (line 2) (return (call + x 1))))\n\nType definition:\n\nmutable struct Foo{T<:S}\n    x::T\nend\n\nparses as:\n\n(struct true (curly Foo (<: T S))\n        (block (line 2) (:: x T)))\n\nThe first argument is a boolean telling whether the type is mutable.\n\ntry blocks parse as (try try_block var catch_block finally_block). If no variable is present after catch, var is #f. If there is no finally clause, then the last argument is not present."},{"location":"devdocs/ast.html#Quote-expressions","page":"Julia ASTs","title":"Quote expressions","category":"section","text":"Julia source syntax forms for code quoting (quote and :( )) support interpolation with $. In Lisp terminology, this means they are actually \"backquote\" or \"quasiquote\" forms. Internally, there is also a need for code quoting without interpolation. In Julia's scheme code, non-interpolating quote is represented with the expression head inert.\n\ninert expressions are converted to Julia QuoteNode objects. These objects wrap a single value of any type, and when evaluated simply return that value.\n\nA quote expression whose argument is an atom also gets converted to a QuoteNode."},{"location":"devdocs/ast.html#Line-numbers","page":"Julia ASTs","title":"Line numbers","category":"section","text":"Source location information is represented as (line line_num file_name) where the third component is optional (and omitted when the current line number, but not file name, changes).\n\nThese expressions are represented as LineNumberNodes in Julia."},{"location":"devdocs/ast.html#Macros-2","page":"Julia ASTs","title":"Macros","category":"section","text":"Macro hygiene is represented through the expression head pair escape and hygienic-scope. The result of a macro expansion is automatically wrapped in (hygienic-scope block module [lno]), to represent the result of the new scope. The user can insert (escape block) inside to interpolate code from the caller. The lno is the __source__ argument of the macro, if included."},{"location":"devdocs/ast.html#Lowered-form","page":"Julia ASTs","title":"Lowered form","category":"section","text":"Lowered form (IR) is more important to the compiler, since it is used for type inference, optimizations like inlining, and code generation. It is also less obvious to the human, since it results from a significant rearrangement of the input syntax.\n\nIn addition to Symbols and some number types, the following data types exist in lowered form:\n\nExpr\nHas a node type indicated by the head field, and an args field which is a Vector{Any} of subexpressions. While almost every part of a surface AST is represented by an Expr, the IR uses only a limited number of Exprs, mostly for calls and some top-level-only forms.\nSlotNumber\nIdentifies arguments and local variables by consecutive numbering. It has an integer-valued id field giving the slot index. The types of these slots can be found in the slottypes field of their CodeInfo object.\nArgument\nThe same as SlotNumber, but appears only post-optimization. Indicates that the referenced slot is an argument of the enclosing function.\nCodeInfo\nWraps the IR of a group of statements. Its code field is an array of expressions to execute.\nGotoNode\nUnconditional branch. The argument is the branch target, represented as an index in the code array to jump to.\nGotoIfNot\nConditional branch. If the cond field evaluates to false, goes to the index identified by the dest field.\nReturnNode\nReturns its argument (the val field) as the value of the enclosing function. If the val field is undefined, then this represents an unreachable statement.\nQuoteNode\nWraps an arbitrary value to reference as data. For example, the function f() = :a contains a QuoteNode whose value field is the symbol a, in order to return the symbol itself instead of evaluating it.\nGlobalRef\nRefers to global variable name in module mod.\nSSAValue\nRefers to a consecutively-numbered (starting at 1) static single assignment (SSA) variable inserted by the compiler. The number (id) of an SSAValue is the code array index of the expression whose value it represents.\nNewvarNode\nMarks a point where a variable (slot) is created. This has the effect of resetting a variable to undefined."},{"location":"devdocs/ast.html#Expr-types","page":"Julia ASTs","title":"Expr types","category":"section","text":"These symbols appear in the head field of Exprs in lowered form.\n\ncall\nFunction call (dynamic dispatch). args[1] is the function to call, args[2:end] are the arguments.\ninvoke\nFunction call (static dispatch). args[1] is the MethodInstance to call, args[2:end] are the arguments (including the function that is being called, at args[2]).\nstatic_parameter\nReference a static parameter by index.\n=\nAssignment. In the IR, the first argument is always a SlotNumber or a GlobalRef.\nmethod\nAdds a method to a generic function and assigns the result if necessary.\nHas a 1-argument form and a 3-argument form. The 1-argument form arises from the syntax function foo end. In the 1-argument form, the argument is a symbol. If this symbol already names a function in the current scope, nothing happens. If the symbol is undefined, a new function is created and assigned to the identifier specified by the symbol. If the symbol is defined but names a non-function, an error is raised. The definition of \"names a function\" is that the binding is constant, and refers to an object of singleton type. The rationale for this is that an instance of a singleton type uniquely identifies the type to add the method to. When the type has fields, it wouldn't be clear whether the method was being added to the instance or its type.\nThe 3-argument form has the following arguments:\nargs[1]\nA function name, or nothing if unknown or unneeded. If a symbol, then the expression first behaves like the 1-argument form above. This argument is ignored from then on. It can be nothing when methods are added strictly by type, (::T)(x) = x, or when a method is being added to an existing function, MyModule.f(x) = x.\nargs[2]\nA SimpleVector of argument type data. args[2][1] is a SimpleVector of the argument types, and args[2][2] is a SimpleVector of type variables corresponding to the method's static parameters.\nargs[3]\nA CodeInfo of the method itself. For \"out of scope\" method definitions (adding a method to a function that also has methods defined in different scopes) this is an expression that evaluates to a :lambda expression.\nstruct_type\nA 7-argument expression that defines a new struct:\nargs[1]\nThe name of the struct\nargs[2]\nA call expression that creates a SimpleVector specifying its parameters\nargs[3]\nA call expression that creates a SimpleVector specifying its fieldnames\nargs[4]\nA Symbol, GlobalRef, or Expr specifying the supertype (e.g., :Integer, GlobalRef(Core, :Any), or :(Core.apply_type(AbstractArray, T, N)))\nargs[5]\nA call expression that creates a SimpleVector specifying its fieldtypes\nargs[6]\nA Bool, true if mutable\nargs[7]\nThe number of arguments to initialize. This will be the number of fields, or the minimum number of fields called by an inner constructor's new statement.\nabstract_type\nA 3-argument expression that defines a new abstract type. The arguments are the same as arguments 1, 2, and 4 of struct_type expressions.\nprimitive_type\nA 4-argument expression that defines a new primitive type. Arguments 1, 2, and 4 are the same as struct_type. Argument 3 is the number of bits.\ncompat: Julia 1.5\nstruct_type, abstract_type, and primitive_type were removed in Julia 1.5 and replaced by calls to new builtins.\nglobal\nDeclares a global binding.\nconst\nDeclares a (global) variable as constant.\nnew\nAllocates a new struct-like object. First argument is the type. The new pseudo-function is lowered to this, and the type is always inserted by the compiler. This is very much an internal-only feature, and does no checking. Evaluating arbitrary new expressions can easily segfault.\nsplatnew\nSimilar to new, except field values are passed as a single tuple. Works similarly to splat(new) if new were a first-class function, hence the name.\nisdefined\nExpr(:isdefined, :x) returns a Bool indicating whether x has already been defined in the current scope.\nthe_exception\nYields the caught exception inside a catch block, as returned by jl_current_exception(ct).\nenter\nEnters an exception handler (setjmp). args[1] is the label of the catch block to jump to on error. Yields a token which is consumed by pop_exception.\nleave\nPop exception handlers. args[1] is the number of handlers to pop.\npop_exception\nPop the stack of current exceptions back to the state at the associated enter when leaving a catch block. args[1] contains the token from the associated enter.\ncompat: Julia 1.1\npop_exception is new in Julia 1.1.\ninbounds\nControls turning bounds checks on or off. A stack is maintained; if the first argument of this expression is true or false (true means bounds checks are disabled), it is pushed onto the stack. If the first argument is :pop, the stack is popped.\nboundscheck\nHas the value false if inlined into a section of code marked with @inbounds, otherwise has the value true.\nloopinfo\nMarks the end of the a loop. Contains metadata that is passed to LowerSimdLoop to either mark the inner loop of @simd expression, or to propagate information to LLVM loop passes.\ncopyast\nPart of the implementation of quasi-quote. The argument is a surface syntax AST that is simply copied recursively and returned at run time.\nmeta\nMetadata. args[1] is typically a symbol specifying the kind of metadata, and the rest of the arguments are free-form. The following kinds of metadata are commonly used:\n:inline and :noinline: Inlining hints.\nforeigncall\nStatically-computed container for ccall information. The fields are:\nargs[1] : name\nThe expression that'll be parsed for the foreign function.\nargs[2]::Type : RT\nThe (literal) return type, computed statically when the containing method was defined.\nargs[3]::SimpleVector (of Types) : AT\nThe (literal) vector of argument types, computed statically when the containing method was defined.\nargs[4]::Int : nreq\nThe number of required arguments for a varargs function definition.\nargs[5]::QuoteNode{<:Union{Symbol,Tuple{Symbol,UInt16}, Tuple{Symbol,UInt16,Bool}}: calling convention\nThe calling convention for the call, optionally with effects, and gc_safe (safe to execute concurrently to GC.).\nargs[6:5+length(args[3])] : arguments\nThe values for all the arguments (with types of each given in args[3]).\nargs[6+length(args[3])+1:end] : gc-roots\nThe additional objects that may need to be gc-rooted for the duration of the call. See Working with LLVM for where these are derived from and how they get handled.\nnew_opaque_closure\nConstructs a new opaque closure. The fields are:\nargs[1] : signature\nThe function signature of the opaque closure. Opaque closures don't participate in dispatch, but the input types can be restricted.\nargs[2] : lb\nLower bound on the output type. (Defaults to Union{})\nargs[3] : ub\nUpper bound on the output type. (Defaults to Any)\nargs[4] : constprop\nIndicates whether the opaque closure's identity may be used for constant propagation. The @opaque macro enables this by default, but this will cause additional inference which may be undesirable and prevents the code from running during precompile. If args[4] is a method, the argument is considered skipped.\nargs[5] : method\nThe actual method as an opaque_closure_method expression.\nargs[6:end] : captures\nThe values captured by the opaque closure.\ncompat: Julia 1.7\nOpaque closures were added in Julia 1.7"},{"location":"devdocs/ast.html#ast-lowered-method","page":"Julia ASTs","title":"Method","category":"section","text":"A unique'd container describing the shared metadata for a single method.\n\nname, module, file, line, sig\nMetadata to uniquely identify the method for the computer and the human.\nambig\nCache of other methods that may be ambiguous with this one.\nspecializations\nCache of all MethodInstance ever created for this Method, used to ensure uniqueness. Uniqueness is required for efficiency, especially for incremental precompile and tracking of method invalidation.\nsource\nThe original source code (if available, usually compressed).\ngenerator\nA callable object which can be executed to get specialized source for a specific method signature.\nroots\nPointers to non-AST things that have been interpolated into the AST, required by compression of the AST, type-inference, or the generation of native code.\nnargs, isva, called, is_for_opaque_closure,\nDescriptive bit-fields for the source code of this Method.\nprimary_world\nThe world age that \"owns\" this Method."},{"location":"devdocs/ast.html#MethodInstance","page":"Julia ASTs","title":"MethodInstance","category":"section","text":"A unique'd container describing a single callable signature for a Method. See especially Proper maintenance and care of multi-threading locks for important details on how to modify these fields safely.\n\nspecTypes\nThe primary key for this MethodInstance. Uniqueness is guaranteed through a def.specializations lookup.\ndef\nThe Method that this function describes a specialization of. Or a Module, if this is a top-level Lambda expanded in Module, and which is not part of a Method.\nsparam_vals\nThe values of the static parameters in specTypes. For the MethodInstance at Method.unspecialized, this is the empty SimpleVector. But for a runtime MethodInstance from the MethodTable cache, this will always be defined and indexable.\nbackedges\nWe store the reverse-list of cache dependencies for efficient tracking of incremental reanalysis/recompilation work that may be needed after a new method definitions. This works by keeping a list of the other MethodInstance that have been inferred or optimized to contain a possible call to this MethodInstance. Those optimization results might be stored somewhere in the cache, or it might have been the result of something we didn't want to cache, such as constant propagation. Thus we merge all of those backedges to various cache entries here (there's almost always only the one applicable cache entry with a sentinel value for max_world anyways).\ncache\nCache of CodeInstance objects that share this template instantiation."},{"location":"devdocs/ast.html#CodeInstance","page":"Julia ASTs","title":"CodeInstance","category":"section","text":"def\nThe MethodInstance that this cache entry is derived from.\nowner\nA token that represents the owner of this CodeInstance. Will use jl_egal to match.\n\nrettype/rettype_const\nThe inferred return type for the specFunctionObject field, which (in most cases) is also the computed return type for the function in general.\ninferred\nMay contain a cache of the inferred source for this function, or it could be set to nothing to just indicate rettype is inferred.\nftpr\nThe generic jlcall entry point.\njlcall_api\nThe ABI to use when calling fptr. Some significant ones include:\n0 - Not compiled yet\n1 - JL_CALLABLE jl_value_t *(*)(jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n2 - Constant (value stored in rettype_const)\n3 - With Static-parameters forwarded jl_value_t *(*)(jl_svec_t *sparams, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\n4 - Run in interpreter jl_value_t *(*)(jl_method_instance_t *meth, jl_value_t *f, jl_value_t *args[nargs], uint32_t nargs)\nmin_world / max_world\nThe range of world ages for which this method instance is valid to be called. If max_world is the special token value -1, the value is not yet known. It may continue to be used until we encounter a backedge that requires us to reconsider.\nTiming fields\ntime_infer_total: Total cost of computing inferred originally as wall-time from start to finish.\ntime_infer_cache_saved: The cost saved from time_infer_total by having caching. Adding this to time_infer_total should give a stable estimate for comparing the cost of two implementations or one implementation over time. This is generally an over-estimate of the time to infer something, since the cache is frequently effective at handling repeated work.\ntime_infer_self: Self cost of julia inference for inferred (a portion of time_infer_total). This is simply the incremental cost of compiling this one method, if given a fully populated cache of all call targets, even including constant inference results and LimitedAccuracy results, which generally are not in a cache.\ntime_compile: Self cost of llvm JIT compilation (e.g. of computing invoke from inferred). A total cost estimate can be computed by walking all of the edges contents and summing those, while accounting for cycles and duplicates. (This field currently does not include any measured AOT compile times.)"},{"location":"devdocs/ast.html#CodeInfo","page":"Julia ASTs","title":"CodeInfo","category":"section","text":"A (usually temporary) container for holding lowered (and possibly inferred) source code.\n\ncode\nAn Any array of statements\nslotnames\nAn array of symbols giving names for each slot (argument or local variable).\nslotflags\nA UInt8 array of slot properties, represented as bit flags:\n0x02 - assigned (only false if there are no assignment statements with this var on the left)\n0x08 - used (if there is any read or write of the slot)\n0x10 - statically assigned once\n0x20 - might be used before assigned. This flag is only valid after type inference.\nssavaluetypes\nEither an array or an Int.\nIf an Int, it gives the number of compiler-inserted temporary locations in the function (the length of code array). If an array, specifies a type for each location.\nssaflags\nStatement-level 32 bits flags for each expression in the function. See the definition of jl_code_info_t in julia.h for more details.\n\nThese are only populated after inference (or by generated functions in some cases):\n\ndebuginfo\nAn object to retrieve source information for each statements, see How to interpret line numbers in a CodeInfo object.\nrettype\nThe inferred return type of the lowered form (IR). Default value is Any. This is mostly present for convenience, as (due to the way OpaqueClosures work) it is not necessarily the rettype used by codegen.\nparent\nThe MethodInstance that \"owns\" this object (if applicable).\nedges\nForward edges to method instances that must be invalidated.\nmin_world/max_world\nThe range of world ages for which this code was valid at the time when it had been inferred.\n\nOptional Fields:\n\nslottypes\nAn array of types for the slots.\nmethod_for_inference_limit_heuristics\nThe method_for_inference_heuristics will expand the given method's generator if necessary during inference.\n\nBoolean properties:\n\npropagate_inbounds\nWhether this should propagate @inbounds when inlined for the purpose of eliding @boundscheck blocks.\n\nUInt8 settings:\n\nconstprop, inlineable\n0 = use heuristic\n1 = aggressive\n2 = none\npurity Constructed from 5 bit flags:\n0x01 << 0 = this method is guaranteed to return or terminate consistently (:consistent)\n0x01 << 1 = this method is free from externally semantically visible side effects (:effect_free)\n0x01 << 2 = this method is guaranteed to not throw an exception (:nothrow)\n0x01 << 3 = this method is guaranteed to terminate (:terminates_globally)\n0x01 << 4 = the syntactic control flow within this method is guaranteed to terminate (:terminates_locally)\nSee the documentation of Base.@assume_effects for more details."},{"location":"devdocs/ast.html#How-to-interpret-line-numbers-in-a-CodeInfo-object","page":"Julia ASTs","title":"How to interpret line numbers in a CodeInfo object","category":"section","text":"There are 2 common forms for this data: one used internally that compresses the data somewhat and one used in the compiler. They contain the same basic info, but the compiler version is all mutable while the version used internally is not.\n\nMany consumers may be able to call Base.IRShow.buildLineInfoNode, Base.IRShow.append_scopes!, or Stacktraces.lookup(::InterpreterIP) to avoid needing to (re-)implement these details specifically.\n\nThe definitions of each of these are:\n\nstruct Core.DebugInfo\n    @noinline\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::SimpleVector{DebugInfo}\n    codelocs::String # compressed data\nend\nmutable struct Core.Compiler.DebugInfoStream\n    def::Union{Method,MethodInstance,Symbol}\n    linetable::Union{Nothing,DebugInfo}\n    edges::Vector{DebugInfo}\n    firstline::Int32 # the starting line for this block (specified by an index of 0)\n    codelocs::Vector{Int32} # for each statement:\n        # index into linetable (if defined), else a line number (in the file represented by def)\n        # then index into edges\n        # then index into edges[linetable]\nend\n\ndef : where this DebugInfo was defined (the Method, MethodInstance, or Symbol of file scope, for example)\nlinetable\nAnother DebugInfo that this was derived from, which contains the actual line numbers, such that this DebugInfo contains only the indexes into it. This avoids making copies, as well as makes it possible to track how each individual statement transformed from source to optimized, not just the separate line numbers. If def is not a Symbol, then that object replaces the current function object for the metadata on what function is conceptually being executed (e.g. think Cassette transforms here). The codelocs values described below also are interpreted as an index into the codelocs in this object, instead of being a line number itself.\nedges : Vector of the unique DebugInfo for every function inlined into this (which recursively have the edges for everything inlined into them).\nfirstline (when uncompressed to DebugInfoStream)\nThe line number associated with the begin statement (or other keyword such as function or quote) that delineates where this code definition \"starts\".\ncodelocs (when uncompressed to DebugInfoStream)\nA vector of indices, with 3 values for each statement in the IR plus one for the starting point of the block, that describe the stacktrace from that point:\nthe integer index into the linetable.codelocs field, giving the original location associated with each statement (including its syntactic edges), or zero indicating no change to the line number from the previously executed statement (which is not necessarily syntactic or lexical prior), or the line number itself if the linetable field is nothing.\nthe integer index into edges, giving the DebugInfo inlined there, or zero if there are no edges.\n(if entry 2 is non-zero) the integer index into edges[].codelocs, to interpret recursively for each function in the inlining stack, or zero indicating to use edges[].firstline as the line number.\nSpecial codes include:\n(zero, zero, *): no change to the line number or edges from the previous statement (you may choose to interpret this either syntactically or lexically). The inlining depth also might have changed, though most callers should ignore that.\n(zero, non-zero, *) : no line number, just edges (usually because of macro-expansion into top-level code)."},{"location":"base/reflection.html#Reflection-and-introspection","page":"Reflection and introspection","title":"Reflection and introspection","category":"section","text":"Julia provides a variety of runtime reflection capabilities."},{"location":"base/reflection.html#Module-bindings","page":"Reflection and introspection","title":"Module bindings","category":"section","text":"The public names for a Module are available using names(m::Module), which will return an array of Symbol elements representing the public bindings. names(m::Module, all = true) returns symbols for all bindings in m, regardless of public status."},{"location":"base/reflection.html#DataType-fields","page":"Reflection and introspection","title":"DataType fields","category":"section","text":"The names of DataType fields may be interrogated using fieldnames. For example, given the following type, fieldnames(Point) returns a tuple of Symbols representing the field names:\n\njulia> struct Point\n           x::Int\n           y\n       end\n\njulia> fieldnames(Point)\n(:x, :y)\n\nThe type of each field in a Point object is stored in the types field of the Point variable itself:\n\njulia> Point.types\nsvec(Int64, Any)\n\nWhile x is annotated as an Int, y was unannotated in the type definition, therefore y defaults to the Any type.\n\nTypes are themselves represented as a structure called DataType:\n\njulia> typeof(Point)\nDataType\n\nNote that fieldnames(DataType) gives the names for each field of DataType itself, and one of these fields is the types field observed in the example above."},{"location":"base/reflection.html#Subtypes","page":"Reflection and introspection","title":"Subtypes","category":"section","text":"The direct subtypes of any DataType may be listed using subtypes. For example, the abstract DataType AbstractFloat has four (concrete) subtypes:\n\njulia> InteractiveUtils.subtypes(AbstractFloat)\n5-element Vector{Any}:\n BigFloat\n Core.BFloat16\n Float16\n Float32\n Float64\n\nAny abstract subtype will also be included in this list, but further subtypes thereof will not; recursive application of subtypes may be used to inspect the full type tree.\n\nNote that subtypes is located inside InteractiveUtils but is automatically exported when using the REPL."},{"location":"base/reflection.html#DataType-layout","page":"Reflection and introspection","title":"DataType layout","category":"section","text":"The internal representation of a DataType is critically important when interfacing with C code and several functions are available to inspect these details. isbitstype(T::DataType) returns true if T is stored with C-compatible alignment. fieldoffset(T::DataType, i::Integer) returns the (byte) offset for field i relative to the start of the type."},{"location":"base/reflection.html#Function-methods","page":"Reflection and introspection","title":"Function methods","category":"section","text":"The methods of any generic function may be listed using methods. The method dispatch table may be searched for methods accepting a given type using methodswith."},{"location":"base/reflection.html#Expansion-and-lowering","page":"Reflection and introspection","title":"Expansion and lowering","category":"section","text":"As discussed in the Metaprogramming section, the macroexpand function gives the unquoted and interpolated expression (Expr) form for a given macro. To use macroexpand, quote the expression block itself (otherwise, the macro will be evaluated and the result will be passed instead!). For example:\n\njulia> macroexpand(@__MODULE__, :(@invoke identity(1::Int)))\n:(Core.invoke(identity, Base.Tuple{Int}, 1))\n\nThe functions Base.Meta.show_sexpr and dump are used to display S-expr style views and depth-nested detail views for any expression.\n\nFinally, the Meta.lower function gives the lowered form of any expression and is of particular interest for understanding how language constructs map to primitive operations such as assignments, branches, and calls:\n\njulia> Meta.lower(@__MODULE__, :( [1+2, sin(0.5)] ))\n:($(Expr(:thunk, CodeInfo(\n1 ─ %1 = :+\n│   %2 =   dynamic (%1)(1, 2)\n│   %3 = sin\n│   %4 =   dynamic (%3)(0.5)\n│   %5 =   dynamic Base.vect(%2, %4)\n└──      return %5\n))))"},{"location":"base/reflection.html#Intermediate-and-compiled-representations","page":"Reflection and introspection","title":"Intermediate and compiled representations","category":"section","text":"Inspecting the lowered form for functions requires selection of the specific method to display, because generic functions may have many methods with different type signatures. For this purpose, method-specific code-lowering is available using code_lowered, and the type-inferred form is available using code_typed. code_warntype adds highlighting to the output of code_typed.\n\nCloser to the machine, the LLVM intermediate representation of a function may be printed using by code_llvm, and finally the compiled machine code is available using code_native (this will trigger JIT compilation/code generation for any function which has not previously been called).\n\nFor convenience, there are macro versions of the above functions which take standard function calls and expand argument types automatically:\n\njulia> @code_llvm +(1,1)\n;  @ int.jl:87 within `+`\n; Function Attrs: sspstrong uwtable\ndefine i64 @\"julia_+_476\"(i64 signext %0, i64 signext %1) #0 {\ntop:\n  %2 = add i64 %1, %0\n  ret i64 %2\n}\n\nFor more information see @code_lowered, @code_typed, @code_warntype, @code_llvm, and @code_native."},{"location":"base/reflection.html#Printing-of-debug-information","page":"Reflection and introspection","title":"Printing of debug information","category":"section","text":"The aforementioned functions and macros take the keyword argument debuginfo that controls the level of debug information printed.\n\njulia> InteractiveUtils.@code_typed debuginfo=:source +(1,1)\nCodeInfo(\n    @ int.jl:87 within `+`\n1 ─ %1 = intrinsic Base.add_int(x, y)::Int64\n└──      return %1\n) => Int64\n\nPossible values for debuginfo are: :none, :source, and :default. Per default debug information is not printed, but that can be changed by setting Base.IRShow.default_debuginfo[] = :source."},{"location":"devdocs/llvm.html#Working-with-LLVM","page":"Working with LLVM","title":"Working with LLVM","category":"section","text":"This is not a replacement for the LLVM documentation, but a collection of tips for working on LLVM for Julia."},{"location":"devdocs/llvm.html#Overview-of-Julia-to-LLVM-Interface","page":"Working with LLVM","title":"Overview of Julia to LLVM Interface","category":"section","text":"Julia dynamically links against LLVM by default. Build with USE_LLVM_SHLIB=0 to link statically.\n\nThe code for lowering Julia AST to LLVM IR or interpreting it directly is in directory src/.\n\nFile Description\naotcompile.cpp Compiler C-interface entry and object file emission\nbuiltins.c Builtin functions\nccall.cpp Lowering ccall\ncgutils.cpp Lowering utilities, notably for array and tuple accesses\ncodegen.cpp Top-level of code generation, pass list, lowering builtins\ndebuginfo.cpp Tracks debug information for JIT code\ndisasm.cpp Handles native object file and JIT code disassembly\ngf.c Generic functions\nintrinsics.cpp Lowering intrinsics\njitlayers.cpp JIT-specific code, ORC compilation layers/utilities\nllvm-alloc-helpers.cpp Julia-specific escape analysis\nllvm-alloc-opt.cpp Custom LLVM pass to demote heap allocations to the stack\nllvm-cpufeatures.cpp Custom LLVM pass to lower CPU-based functions (e.g. haveFMA)\nllvm-demote-float16.cpp Custom LLVM pass to lower 16b float ops to 32b float ops\nllvm-final-gc-lowering.cpp Custom LLVM pass to lower GC calls to their final form\nllvm-gc-invariant-verifier.cpp Custom LLVM pass to verify Julia GC invariants\nllvm-julia-licm.cpp Custom LLVM pass to hoist/sink Julia-specific intrinsics\nllvm-late-gc-lowering.cpp Custom LLVM pass to root GC-tracked values\nllvm-multiversioning.cpp Custom LLVM pass to generate sysimg code on multiple architectures\nllvm-propagate-addrspaces.cpp Custom LLVM pass to canonicalize addrspaces\nllvm-ptls.cpp Custom LLVM pass to lower TLS operations\nllvm-remove-addrspaces.cpp Custom LLVM pass to remove Julia addrspaces\nllvm-remove-ni.cpp Custom LLVM pass to remove Julia non-integral addrspaces\nllvm-simdloop.cpp Custom LLVM pass for @simd\npipeline.cpp New pass manager pipeline, pass pipeline parsing\nsys.c I/O and operating system utility functions\n\nSome of the .cpp files form a group that compile to a single object.\n\nThe difference between an intrinsic and a builtin is that a builtin is a first class function that can be used like any other Julia function. An intrinsic can operate only on unboxed data, and therefore its arguments must be statically typed."},{"location":"devdocs/llvm.html#LLVM-Alias-Analysis","page":"Working with LLVM","title":"Alias Analysis","category":"section","text":"Julia currently uses LLVM's Type Based Alias Analysis. To find the comments that document the inclusion relationships, look for static MDNode* in src/codegen.cpp.\n\nThe -O option enables LLVM's Basic Alias Analysis."},{"location":"devdocs/llvm.html#Building-Julia-with-a-different-version-of-LLVM","page":"Working with LLVM","title":"Building Julia with a different version of LLVM","category":"section","text":"The default version of LLVM is specified in deps/llvm.version. You can override it by creating a file called Make.user in the top-level directory and adding a line to it such as:\n\nLLVM_VER = 13.0.0\n\nBesides the LLVM release numerals, you can also use DEPS_GIT = llvm in combination with USE_BINARYBUILDER_LLVM = 0 to build against the latest development version of LLVM.\n\nYou can also specify to build a debug version of LLVM, by setting either LLVM_DEBUG = 1 or LLVM_DEBUG = Release in your Make.user file. The former will be a fully unoptimized build of LLVM and the latter will produce an optimized build of LLVM. Depending on your needs the latter will suffice and it quite a bit faster. If you use LLVM_DEBUG = Release you will also want to set LLVM_ASSERTIONS = 1 to enable diagnostics for different passes. Only LLVM_DEBUG = 1 implies that option by default."},{"location":"devdocs/llvm.html#Passing-options-to-LLVM","page":"Working with LLVM","title":"Passing options to LLVM","category":"section","text":"You can pass options to LLVM via the environment variable JULIA_LLVM_ARGS. Here are example settings using bash syntax:\n\nexport JULIA_LLVM_ARGS=-print-after-all dumps IR after each pass.\nexport JULIA_LLVM_ARGS=-debug-only=loop-vectorize dumps LLVM DEBUG(...) diagnostics for loop vectorizer. If you get warnings about \"Unknown command line argument\", rebuild LLVM with LLVM_ASSERTIONS = 1.\nexport JULIA_LLVM_ARGS=-help shows a list of available options. export JULIA_LLVM_ARGS=-help-hidden shows even more.\nexport JULIA_LLVM_ARGS=\"-fatal-warnings -print-options\" is an example how to use multiple options."},{"location":"devdocs/llvm.html#Useful-JULIA_LLVM_ARGS-parameters","page":"Working with LLVM","title":"Useful JULIA_LLVM_ARGS parameters","category":"section","text":"-print-after=PASS: prints the IR after any execution of PASS, useful for checking changes done by a pass.\n-print-before=PASS: prints the IR before any execution of PASS, useful for checking the input to a pass.\n-print-changed: prints the IR whenever a pass changes the IR, useful for narrowing down which passes are causing problems.\n-print-(before|after)=MARKER-PASS: the Julia pipeline ships with a number of marker passes in the pipeline, which can be used to identify where problems or optimizations are occurring. A marker pass is defined as a pass which appears once in the pipeline and performs no transformations on the IR, and is only useful for targeting print-before/print-after. Currently, the following marker passes exist in the pipeline:\nBeforeOptimization\nBeforeEarlySimplification\nAfterEarlySimplification\nBeforeEarlyOptimization\nAfterEarlyOptimization\nBeforeLoopOptimization\nBeforeLICM\nAfterLICM\nBeforeLoopSimplification\nAfterLoopSimplification\nAfterLoopOptimization\nBeforeScalarOptimization\nAfterScalarOptimization\nBeforeVectorization\nAfterVectorization\nBeforeIntrinsicLowering\nAfterIntrinsicLowering\nBeforeCleanup\nAfterCleanup\nAfterOptimization\n-time-passes: prints the time spent in each pass, useful for identifying which passes are taking a long time.\n-print-module-scope: used in conjunction with -print-(before|after), gets the entire module rather than the IR unit received by the pass\n-debug: prints out a lot of debugging information throughout LLVM\n-debug-only=NAME, prints out debugging statements from files with DEBUG_TYPE defined to NAME, useful for getting additional context about a problem"},{"location":"devdocs/llvm.html#Debugging-LLVM-transformations-in-isolation","page":"Working with LLVM","title":"Debugging LLVM transformations in isolation","category":"section","text":"On occasion, it can be useful to debug LLVM's transformations in isolation from the rest of the Julia system, e.g. because reproducing the issue inside julia would take too long, or because one wants to take advantage of LLVM's tooling (e.g. bugpoint).\n\nTo start with, you can install the developer tools to work with LLVM via:\n\nmake -C deps install-llvm-tools\n\nTo get unoptimized IR for the entire system image, pass the --output-unopt-bc unopt.bc option to the system image build process, which will output the unoptimized IR to an unopt.bc file. This file can then be passed to LLVM tools as usual. libjulia can function as an LLVM pass plugin and can be loaded into LLVM tools, to make julia-specific passes available in this environment. In addition, it exposes the -julia meta-pass, which runs the entire Julia pass-pipeline over the IR. As an example, to generate a system image with the old pass manager, one could do:\n\n\nllc -o sys.o opt.bc\ncc -shared -o sys.so sys.o\n\nTo generate a system image with the new pass manager, one could do:\n\n./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -o opt.bc unopt.bc\n./usr/tools/llc -o sys.o opt.bc\n./usr/tools/cc -shared -o sys.so sys.o\n\nThis system image can then be loaded by julia as usual.\n\nIt is also possible to dump an LLVM IR module for just one Julia function, using:\n\nfun, T = +, Tuple{Int,Int} # Substitute your function of interest here\noptimize = false\nopen(\"plus.ll\", \"w\") do file\n    code_llvm(file, fun, T; raw=true, dump_module=true, optimize)\nend\n\nThese files can be processed the same way as the unoptimized sysimg IR shown above, or if you want to see the LLVM IR yourself and get extra verification run, you can use\n\n./usr/tools/opt -load-pass-plugin=libjulia-codegen.so --passes='julia' -S -verify-each plus.ll\n\n(note on MacOS this would be libjulia-codegen.dylib and on Windows libjulia-codegen.dll)"},{"location":"devdocs/llvm.html#Running-the-LLVM-test-suite","page":"Working with LLVM","title":"Running the LLVM test suite","category":"section","text":"To run the llvm tests locally, you need to first install the tools, build julia, then you can run the tests:\n\nmake -C deps install-llvm-tools\nmake -j julia-src-release\nmake -C test/llvmpasses\n\nIf you want to run the individual test files directly, via the commands at the top of each test file, the first step here will have installed the tools into ./usr/tools/opt. Then you'll want to manually replace %s with the name of the test file."},{"location":"devdocs/llvm.html#Improving-LLVM-optimizations-for-Julia","page":"Working with LLVM","title":"Improving LLVM optimizations for Julia","category":"section","text":"Improving LLVM code generation usually involves either changing Julia lowering to be more friendly to LLVM's passes, or improving a pass.\n\nIf you are planning to improve a pass, be sure to read the LLVM developer policy. The best strategy is to create a code example in a form where you can use LLVM's opt tool to study it and the pass of interest in isolation.\n\nCreate an example Julia code of interest.\nUse JULIA_LLVM_ARGS=-print-after-all to dump the IR.\nPick out the IR at the point just before the pass of interest runs.\nStrip the debug metadata and fix up the TBAA metadata by hand.\n\nThe last step is labor intensive. Suggestions on a better way would be appreciated."},{"location":"devdocs/llvm.html#The-jlcall-calling-convention","page":"Working with LLVM","title":"The jlcall calling convention","category":"section","text":"Julia has a generic calling convention for unoptimized code, which looks somewhat as follows:\n\njl_value_t *any_unoptimized_call(jl_value_t *, jl_value_t **, int);\n\nwhere the first argument is the boxed function object, the second argument is an on-stack array of arguments and the third is the number of arguments. Now, we could perform a straightforward lowering and emit an alloca for the argument array. However, this would betray the SSA nature of the uses at the call site, making optimizations (including GC root placement), significantly harder. Instead, we emit it as follows:\n\ncall %jl_value_t *@julia.call(jl_value_t *(*)(...) @any_unoptimized_call, %jl_value_t *%arg1, %jl_value_t *%arg2)\n\nThis allows us to retain the SSA-ness of the uses throughout the optimizer. GC root placement will later lower this call to the original C ABI."},{"location":"devdocs/llvm.html#GC-root-placement","page":"Working with LLVM","title":"GC root placement","category":"section","text":"GC root placement is done by an LLVM pass late in the pass pipeline. Doing GC root placement this late enables LLVM to make more aggressive optimizations around code that requires GC roots, as well as allowing us to reduce the number of required GC roots and GC root store operations (since LLVM doesn't understand our GC, it wouldn't otherwise know what it is and is not allowed to do with values stored to the GC frame, so it'll conservatively do very little). As an example, consider an error path\n\nif some_condition()\n    #= Use some variables maybe =#\n    error(\"An error occurred\")\nend\n\nDuring constant folding, LLVM may discover that the condition is always false, and can remove the basic block. However, if GC root lowering is done early, the GC root slots used in the deleted block, as well as any values kept alive in those slots only because they were used in the error path, would be kept alive by LLVM. By doing GC root lowering late, we give LLVM the license to do any of its usual optimizations (constant folding, dead code elimination, etc.), without having to worry (too much) about which values may or may not be GC tracked.\n\nHowever, in order to be able to do late GC root placement, we need to be able to identify a) which pointers are GC tracked and b) all uses of such pointers. The goal of the GC placement pass is thus simple:\n\nMinimize the number of needed GC roots/stores to them subject to the constraint that at every safepoint, any live GC-tracked pointer (i.e. for which there is a path after this point that contains a use of this pointer) is in some GC slot."},{"location":"devdocs/llvm.html#Representation","page":"Working with LLVM","title":"Representation","category":"section","text":"The primary difficulty is thus choosing an IR representation that allows us to identify GC-tracked pointers and their uses, even after the program has been run through the optimizer. Our design makes use of three LLVM features to achieve this:\n\nCustom address spaces\nOperand Bundles\nNon-integral pointers\n\nCustom address spaces allow us to tag every point with an integer that needs to be preserved through optimizations. The compiler may not insert casts between address spaces that did not exist in the original program and it must never change the address space of a pointer on a load/store/etc operation. This allows us to annotate which pointers are GC-tracked in an optimizer-resistant way. Note that metadata would not be able to achieve the same purpose. Metadata is supposed to always be discardable without altering the semantics of the program. However, failing to identify a GC-tracked pointer alters the resulting program behavior dramatically - it'll probably crash or return wrong results. We currently use three different address spaces (their numbers are defined in src/codegen_shared.cpp):\n\nGC Tracked Pointers (currently 10): These are pointers to boxed values that may be put into a GC frame. It is loosely equivalent to a jl_value_t* pointer on the C side. N.B. It is illegal to ever have a pointer in this address space that may not be stored to a GC slot.\nDerived Pointers (currently 11): These are pointers that are derived from some GC tracked pointer. Uses of these pointers generate uses of the original pointer. However, they need not themselves be known to the GC. The GC root placement pass MUST always find the GC tracked pointer from which this pointer is derived and use that as the pointer to root.\nCallee Rooted Pointers (currently 12): This is a utility address space to express the notion of a callee rooted value. All values of this address space MUST be storable to a GC root (though it is possible to relax this condition in the future), but unlike the other pointers need not be rooted if passed to a call (they do still need to be rooted if they are live across another safepoint between the definition and the call).\nPointers loaded from tracked object (currently 13): This is used by arrays, which themselves contain a pointer to the managed data. This data area is owned by the array, but is not a GC-tracked object by itself. The compiler guarantees that as long as this pointer is live, the object that this pointer was loaded from will keep being live."},{"location":"devdocs/llvm.html#Invariants","page":"Working with LLVM","title":"Invariants","category":"section","text":"The GC root placement pass makes use of several invariants, which need to be observed by the frontend and are preserved by the optimizer.\n\nFirst, only the following address space casts are allowed:\n\n0->{Tracked,Derived,CalleeRooted}: It is allowable to decay an untracked pointer to any of the others. However, do note that the optimizer has broad license to not root such a value. It is never safe to have a value in address space 0 in any part of the program if it is (or is derived from) a value that requires a GC root.\nTracked->Derived: This is the standard decay route for interior values. The placement pass will look for these to identify the base pointer for any use.\nTracked->CalleeRooted: Addrspace CalleeRooted serves merely as a hint that a GC root is not required. However, do note that the Derived->CalleeRooted decay is prohibited, since pointers should generally be storable to a GC slot, even in this address space.\n\nNow let us consider what constitutes a use:\n\nLoads whose loaded values is in one of the address spaces\nStores of a value in one of the address spaces to a location\nStores to a pointer in one of the address spaces\nCalls for which a value in one of the address spaces is an operand\nCalls in jlcall ABI, for which the argument array contains a value\nReturn instructions.\n\nWe explicitly allow load/stores and simple calls in address spaces Tracked/Derived. Elements of jlcall argument arrays must always be in address space Tracked (it is required by the ABI that they are valid jl_value_t* pointers). The same is true for return instructions (though note that struct return arguments are allowed to have any of the address spaces). The only allowable use of an address space CalleeRooted pointer is to pass it to a call (which must have an appropriately typed operand).\n\nFurther, we disallow getelementptr in addrspace Tracked. This is because unless the operation is a noop, the resulting pointer will not be validly storable to a GC slot and may thus not be in this address space. If such a pointer is required, it should be decayed to addrspace Derived first.\n\nLastly, we disallow inttoptr/ptrtoint instructions in these address spaces. Having these instructions would mean that some i64 values are really GC tracked. This is problematic, because it breaks that stated requirement that we're able to identify GC-relevant pointers. This invariant is accomplished using the LLVM \"non-integral pointers\" feature, which is new in LLVM 5.0. It prohibits the optimizer from making optimizations that would introduce these operations. Note we can still insert static constants at JIT time by using inttoptr in address space 0 and then decaying to the appropriate address space afterwards."},{"location":"devdocs/llvm.html#Supporting-[ccall](@ref)","page":"Working with LLVM","title":"Supporting ccall","category":"section","text":"One important aspect missing from the discussion so far is the handling of ccall. ccall has the peculiar feature that the location and scope of a use do not coincide. As an example consider:\n\nA = randn(1024)\nccall(:foo, Cvoid, (Ptr{Float64},), A)\n\nIn lowering, the compiler will insert a conversion from the array to the pointer which drops the reference to the array value. However, we of course need to make sure that the array does stay alive while we're doing the ccall. To understand how this is done, lets look at a hypothetical approximate possible lowering of the above code:\n\nreturn $(Expr(:foreigncall, Expr(:tuple, :(:foo)), Cvoid, svec(Ptr{Float64}), 0, :(:ccall), Expr(:foreigncall, Expr(:tuple, :(:jl_array_ptr)), Ptr{Float64}, svec(Any), 0, :(:ccall), :(A)), :(A)))\n\nThe last :(A), is an extra argument list inserted during lowering that informs the code generator which Julia level values need to be kept alive for the duration of this ccall. We then take this information and represent it in an \"operand bundle\" at the IR level. An operand bundle is essentially a fake use that is attached to the call site. At the IR level, this looks like so:\n\ncall void inttoptr (i64 ... to void (double*)*)(double* %5) [ \"jl_roots\"(%jl_value_t addrspace(10)* %A) ]\n\nThe GC root placement pass will treat the jl_roots operand bundle as if it were a regular operand. However, as a final step, after the GC roots are inserted, it will drop the operand bundle to avoid confusing instruction selection."},{"location":"devdocs/llvm.html#Supporting-[pointer_from_objref](@ref)","page":"Working with LLVM","title":"Supporting pointer_from_objref","category":"section","text":"pointer_from_objref is special because it requires the user to take explicit control of GC rooting. By our above invariants, this function is illegal, because it performs an address space cast from 10 to 0. However, it can be useful, in certain situations, so we provide a special intrinsic:\n\ndeclared %jl_value_t *julia.pointer_from_objref(%jl_value_t addrspace(10)*)\n\nwhich is lowered to the corresponding address space cast after GC root lowering. Do note however that by using this intrinsic, the caller assumes all responsibility for making sure that the value in question is rooted. Further this intrinsic is not considered a use, so the GC root placement pass will not provide a GC root for the function. As a result, the external rooting must be arranged while the value is still tracked by the system. I.e. it is not valid to attempt to use the result of this operation to establish a global root - the optimizer may have already dropped the value."},{"location":"devdocs/llvm.html#Keeping-values-alive-in-the-absence-of-uses","page":"Working with LLVM","title":"Keeping values alive in the absence of uses","category":"section","text":"In certain cases it is necessary to keep an object alive, even though there is no compiler-visible use of said object. This may be case for low level code that operates on the memory-representation of an object directly or code that needs to interface with C code. In order to allow this, we provide the following intrinsics at the LLVM level:\n\ntoken @llvm.julia.gc_preserve_begin(...)\nvoid @llvm.julia.gc_preserve_end(token)\n\n(The llvm. in the name is required in order to be able to use the token type). The semantics of these intrinsics are as follows: At any safepoint that is dominated by a gc_preserve_begin call, but that is not not dominated by a corresponding gc_preserve_end call (i.e. a call whose argument is the token returned by a gc_preserve_begin call), the values passed as arguments to that gc_preserve_begin will be kept live. Note that the gc_preserve_begin still counts as a regular use of those values, so the standard lifetime semantics will ensure that the values will be kept alive before entering the preserve region."},{"location":"stdlib/Unicode.html#Unicode","page":"Unicode","title":"Unicode","category":"section","text":"The Unicode module provides essential functionality for managing Unicode characters and strings. It includes validation, category determination, normalization, case transformation, and grapheme segmentation, enabling effective Unicode data handling."},{"location":"stdlib/Unicode.html#Unicode","page":"Unicode","title":"Unicode","category":"module","text":"The Unicode module provides essential functionality for managing Unicode characters and strings. It includes validation, category determination, normalization, case transformation, and grapheme segmentation, enabling effective Unicode data handling.\n\n\n\n\n\n"},{"location":"stdlib/Unicode.html#Unicode.julia_chartransform","page":"Unicode","title":"Unicode.julia_chartransform","category":"function","text":"Unicode.julia_chartransform(c::Union{Char,Integer})\n\nMap the Unicode character (Char) or codepoint (Integer) c to the corresponding \"equivalent\" character or codepoint, respectively, according to the custom equivalence used within the Julia parser (in addition to NFC normalization).\n\nFor example, 'µ' (U+00B5 micro) is treated as equivalent to 'μ' (U+03BC mu) by Julia's parser, so julia_chartransform performs this transformation while leaving other characters unchanged:\n\njulia> Unicode.julia_chartransform('µ')\n'μ': Unicode U+03BC (category Ll: Letter, lowercase)\n\njulia> Unicode.julia_chartransform('x')\n'x': ASCII/Unicode U+0078 (category Ll: Letter, lowercase)\n\njulia_chartransform is mainly useful for passing to the Unicode.normalize function in order to mimic the normalization used by the Julia parser:\n\njulia> s = \"µö\"\n\"µö\"\n\njulia> s2 = Unicode.normalize(s, compose=true, stable=true, chartransform=Unicode.julia_chartransform)\n\"μö\"\n\njulia> collect(s2)\n2-element Vector{Char}:\n 'μ': Unicode U+03BC (category Ll: Letter, lowercase)\n 'ö': Unicode U+00F6 (category Ll: Letter, lowercase)\n\njulia> s2 == string(Meta.parse(s))\ntrue\n\ncompat: Julia 1.8\nThis function was introduced in Julia 1.8.\n\n\n\n\n\n"},{"location":"stdlib/Unicode.html#Unicode.isassigned","page":"Unicode","title":"Unicode.isassigned","category":"function","text":"Unicode.isassigned(c)::Bool\n\nReturn true if the given char or integer is an assigned Unicode code point.\n\nExamples\n\njulia> Unicode.isassigned(101)\ntrue\n\njulia> Unicode.isassigned('\\x01')\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Unicode.html#Unicode.isequal_normalized","page":"Unicode","title":"Unicode.isequal_normalized","category":"function","text":"isequal_normalized(s1::AbstractString, s2::AbstractString; casefold=false, stripmark=false, chartransform=identity)\n\nReturn whether s1 and s2 are canonically equivalent Unicode strings.   If casefold=true, ignores case (performs Unicode case-folding); if stripmark=true, strips diacritical marks and other combining characters.\n\nAs with Unicode.normalize, you can also pass an arbitrary function via the chartransform keyword (mapping Integer codepoints to codepoints) to perform custom normalizations, such as Unicode.julia_chartransform.\n\ncompat: Julia 1.8\nThe isequal_normalized function was added in Julia 1.8.\n\nExamples\n\nFor example, the string \"noël\" can be constructed in two canonically equivalent ways in Unicode, depending on whether \"ë\" is formed from a single codepoint U+00EB or from the ASCII character 'e' followed by the U+0308 combining-diaeresis character.\n\njulia> s1 = \"noël\"\n\"noël\"\n\njulia> s2 = \"noël\"\n\"noël\"\n\njulia> s1 == s2\nfalse\n\njulia> isequal_normalized(s1, s2)\ntrue\n\njulia> isequal_normalized(s1, \"noel\", stripmark=true)\ntrue\n\njulia> isequal_normalized(s1, \"NOËL\", casefold=true)\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Unicode.html#Unicode.normalize","page":"Unicode","title":"Unicode.normalize","category":"function","text":"Unicode.normalize(s::AbstractString; keywords...)\nUnicode.normalize(s::AbstractString, normalform::Symbol)\n\nNormalize the string s. By default, canonical composition (compose=true) is performed without ensuring Unicode versioning stability (compat=false), which produces the shortest possible equivalent string but may introduce composition characters not present in earlier Unicode versions.\n\nAlternatively, one of the four \"normal forms\" of the Unicode standard can be specified: normalform can be :NFC, :NFD, :NFKC, or :NFKD.  Normal forms C (canonical composition) and D (canonical decomposition) convert different visually identical representations of the same abstract string into a single canonical form, with form C being more compact.  Normal forms KC and KD additionally canonicalize \"compatibility equivalents\": they convert characters that are abstractly similar but visually distinct into a single canonical choice (e.g. they expand ligatures into the individual characters), with form KC being more compact.\n\nAlternatively, finer control and additional transformations may be obtained by calling Unicode.normalize(s; keywords...), where any number of the following boolean keywords options (which all default to false except for compose) are specified:\n\ncompose=false: do not perform canonical composition\ndecompose=true: do canonical decomposition instead of canonical composition (compose=true is ignored if present)\ncompat=true: compatibility equivalents are canonicalized\ncasefold=true: perform Unicode case folding, e.g. for case-insensitive string comparison\nnewline2lf=true, newline2ls=true, or newline2ps=true: convert various newline sequences (LF, CRLF, CR, NEL) into a linefeed (LF), line-separation (LS), or paragraph-separation (PS) character, respectively\nstripmark=true: strip diacritical marks (e.g. accents)\nstripignore=true: strip Unicode's \"default ignorable\" characters (e.g. the soft hyphen or the left-to-right marker)\nstripcc=true: strip control characters; horizontal tabs and form feeds are converted to spaces; newlines are also converted to spaces unless a newline-conversion flag was specified\nrejectna=true: throw an error if unassigned code points are found\nstable=true: enforce Unicode versioning stability (never introduce characters missing from earlier Unicode versions)\n\nYou can also use the chartransform keyword (which defaults to identity) to pass an arbitrary function mapping Integer codepoints to codepoints, which is called on each character in s as it is processed, in order to perform arbitrary additional normalizations. For example, by passing chartransform=Unicode.julia_chartransform, you can apply a few Julia-specific character normalizations that are performed by Julia when parsing identifiers (in addition to NFC normalization: compose=true, stable=true).\n\nFor example, NFKC corresponds to the options compose=true, compat=true, stable=true.\n\nExamples\n\njulia> \"é\" == Unicode.normalize(\"é\") #LHS: Unicode U+00e9, RHS: U+0065 & U+0301\ntrue\n\njulia> \"μ\" == Unicode.normalize(\"µ\", compat=true) #LHS: Unicode U+03bc, RHS: Unicode U+00b5\ntrue\n\njulia> Unicode.normalize(\"JuLiA\", casefold=true)\n\"julia\"\n\njulia> Unicode.normalize(\"JúLiA\", stripmark=true)\n\"JuLiA\"\n\ncompat: Julia 1.8\nThe chartransform keyword argument requires Julia 1.8.\n\n\n\n\n\n"},{"location":"stdlib/Unicode.html#Unicode.graphemes","page":"Unicode","title":"Unicode.graphemes","category":"function","text":"graphemes(s::AbstractString)::GraphemeIterator\n\nReturn an iterator over substrings of s that correspond to the extended graphemes in the string, as defined by Unicode UAX #29. (Roughly, these are what users would perceive as single characters, even though they may contain more than one codepoint; for example a letter combined with an accent mark is a single grapheme.)\n\n\n\n\n\ngraphemes(s::AbstractString, m:n)::SubString\n\nReturns a SubString of s consisting of the m-th through n-th graphemes of the string s, where the second argument m:n is an integer-valued AbstractUnitRange.\n\nLoosely speaking, this corresponds to the m:n-th user-perceived \"characters\" in the string.  For example:\n\njulia> s = graphemes(\"exposé\", 3:6)\n\"posé\"\n\njulia> collect(s)\n5-element Vector{Char}:\n 'p': ASCII/Unicode U+0070 (category Ll: Letter, lowercase)\n 'o': ASCII/Unicode U+006F (category Ll: Letter, lowercase)\n 's': ASCII/Unicode U+0073 (category Ll: Letter, lowercase)\n 'e': ASCII/Unicode U+0065 (category Ll: Letter, lowercase)\n '́': Unicode U+0301 (category Mn: Mark, nonspacing)\n\nThis consists of the 3rd to 7th codepoints (Chars) in \"exposé\", because the grapheme \"é\" is actually two Unicode codepoints (an 'e' followed by an acute-accent combining character U+0301).\n\nBecause finding grapheme boundaries requires iteration over the string contents, the graphemes(s, m:n) function requires time proportional to the length of the string (number of codepoints) before the end of the substring.\n\ncompat: Julia 1.9\nThe m:n argument of graphemes requires Julia 1.9.\n\n\n\n\n\n"},{"location":"stdlib/Printf.html#man-printf","page":"Printf","title":"Printf","category":"section","text":"The Printf module provides formatted output functions similar to the C standard library's printf. It allows formatted printing to an output stream or to a string."},{"location":"stdlib/Printf.html#Printf.@printf","page":"Printf","title":"Printf.@printf","category":"macro","text":"@printf([io::IO], \"%Fmt\", args...)\n\nPrint args using C printf style format specification string. Optionally, an IO may be passed as the first argument to redirect output.\n\nExamples\n\njulia> @printf \"Hello %s\" \"world\"\nHello world\n\njulia> @printf \"Scientific notation %e\" 1.234\nScientific notation 1.234000e+00\n\njulia> @printf \"Scientific notation three digits %.3e\" 1.23456\nScientific notation three digits 1.235e+00\n\njulia> @printf \"Decimal two digits %.2f\" 1.23456\nDecimal two digits 1.23\n\njulia> @printf \"Padded to length 5 %5i\" 123\nPadded to length 5   123\n\njulia> @printf \"Padded with zeros to length 6 %06i\" 123\nPadded with zeros to length 6 000123\n\njulia> @printf \"Use shorter of decimal or scientific %g %g\" 1.23 12300000.0\nUse shorter of decimal or scientific 1.23 1.23e+07\n\njulia> @printf \"Use dynamic width and precision  %*.*f\" 10 2 0.12345\nUse dynamic width and precision        0.12\n\nFor a systematic specification of the format, see here. See also @sprintf to get the result as a String instead of it being printed.\n\nCaveats\n\nInf and NaN are printed consistently as Inf and NaN for flags %a, %A, %e, %E, %f, %F, %g, and %G. Furthermore, if a floating point number is equally close to the numeric values of two possible output strings, the output string further away from zero is chosen.\n\nExamples\n\njulia> @printf(\"%f %F %f %F\", Inf, Inf, NaN, NaN)\nInf Inf NaN NaN\n\njulia> @printf \"%.0f %.1f %f\" 0.5 0.025 -0.0078125\n0 0.0 -0.007812\n\ncompat: Julia 1.8\nStarting in Julia 1.8, %s (string) and %c (character) widths are computed using textwidth, which e.g. ignores zero-width characters (such as combining characters for diacritical marks) and treats certain \"wide\" characters (e.g. emoji) as width 2.\n\ncompat: Julia 1.10\nDynamic width specifiers like %*s and %0*.*f require Julia 1.10.\n\n\n\n\n\n"},{"location":"stdlib/Printf.html#Printf.@sprintf","page":"Printf","title":"Printf.@sprintf","category":"macro","text":"@sprintf(\"%Fmt\", args...)\n\nReturn @printf formatted output as string.\n\nExamples\n\njulia> @sprintf \"this is a %s %15.1f\" \"test\" 34.567\n\"this is a test            34.6\"\n\n\n\n\n\n"},{"location":"stdlib/Printf.html#Printf.Format","page":"Printf","title":"Printf.Format","category":"type","text":"Printf.Format(format_str)\n\nCreate a C printf-compatible format object that can be used for formatting values.\n\nThe input format_str can include any valid format specifier character and modifiers.\n\nA Format object can be passed to Printf.format(f::Format, args...) to produce a formatted string, or Printf.format(io::IO, f::Format, args...) to print the formatted string directly to io.\n\nFor convenience, the Printf.format\"...\" string macro form can be used for building a Printf.Format object at macro-expansion-time.\n\ncompat: Julia 1.6\nPrintf.Format requires Julia 1.6 or later.\n\n\n\n\n\n"},{"location":"stdlib/Printf.html#Printf.format","page":"Printf","title":"Printf.format","category":"function","text":"Printf.format(f::Printf.Format, args...) => String\nPrintf.format(io::IO, f::Printf.Format, args...)\n\nApply a printf format object f to provided args and return the formatted string (1st method), or print directly to an io object (2nd method). See @printf for more details on C printf support.\n\n\n\n\n\n"},{"location":"devdocs/probes.html#Instrumenting-Julia-with-DTrace,-and-bpftrace","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Instrumenting Julia with DTrace, and bpftrace","category":"section","text":"DTrace and bpftrace are tools that enable lightweight instrumentation of processes. You can turn the instrumentation on and off while the process is running, and with instrumentation off the overhead is minimal.\n\ncompat: Julia 1.8\nSupport for probes was added in Julia 1.8\n\nnote: Note\nThis documentation has been written from a Linux perspective, most of this should hold on Mac OS/Darwin and FreeBSD."},{"location":"devdocs/probes.html#Enabling-support","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Enabling support","category":"section","text":"On Linux install the systemtap package that has a version of dtrace and create a Make.user file containing\n\nWITH_DTRACE=1\n\nto enable USDT probes."},{"location":"devdocs/probes.html#Verifying","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Verifying","category":"section","text":"> readelf -n usr/lib/libjulia-internal.so.1\n\nDisplaying notes found in: .note.gnu.build-id\n  Owner                Data size  Description\n  GNU                  0x00000014 NT_GNU_BUILD_ID (unique build ID bitstring)\n    Build ID: 57161002f35548772a87418d2385c284ceb3ead8\n\nDisplaying notes found in: .note.stapsdt\n  Owner                Data size  Description\n  stapsdt              0x00000029 NT_STAPSDT (SystemTap probe descriptors)\n    Provider: julia\n    Name: gc__begin\n    Location: 0x000000000013213e, Base: 0x00000000002bb4da, Semaphore: 0x0000000000346cac\n    Arguments:\n  stapsdt              0x00000032 NT_STAPSDT (SystemTap probe descriptors)\n    Provider: julia\n    Name: gc__stop_the_world\n    Location: 0x0000000000132144, Base: 0x00000000002bb4da, Semaphore: 0x0000000000346cae\n    Arguments:\n  stapsdt              0x00000027 NT_STAPSDT (SystemTap probe descriptors)\n    Provider: julia\n    Name: gc__end\n    Location: 0x000000000013214a, Base: 0x00000000002bb4da, Semaphore: 0x0000000000346cb0\n    Arguments:\n  stapsdt              0x0000002d NT_STAPSDT (SystemTap probe descriptors)\n    Provider: julia\n    Name: gc__finalizer\n    Location: 0x0000000000132150, Base: 0x00000000002bb4da, Semaphore: 0x0000000000346cb2\n    Arguments:"},{"location":"devdocs/probes.html#Adding-probes-in-libjulia","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Adding probes in libjulia","category":"section","text":"Probes are declared in dtraces format in the file src/uprobes.d. The generated header file is included in src/julia_internal.h and if you add probes you should provide a noop implementation there.\n\nThe header will contain a semaphore *_ENABLED and the actual call to the probe. If the probe arguments are expensive to compute you should first check if the probe is enabled and then compute the arguments and call the probe.\n\n  if (JL_PROBE_{PROBE}_ENABLED())\n    auto expensive_arg = ...;\n    JL_PROBE_{PROBE}(expensive_arg);\n\nIf your probe has no arguments it is preferred to not include the semaphore check. With USDT probes enabled the cost of a semaphore is a memory load, irrespective of the fact that the probe is enabled or not.\n\n#define JL_PROBE_GC_BEGIN_ENABLED() __builtin_expect (julia_gc__begin_semaphore, 0)\n__extension__ extern unsigned short julia_gc__begin_semaphore __attribute__ ((unused)) __attribute__ ((section (\".probes\")));\n\nWhereas the probe itself is a noop sled that will be patched to a trampoline to the probe handler."},{"location":"devdocs/probes.html#Available-probes","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Available probes","category":"section","text":""},{"location":"devdocs/probes.html#GC-probes","page":"Instrumenting Julia with DTrace, and bpftrace","title":"GC probes","category":"section","text":"julia:gc__begin: GC begins running on one thread and triggers stop-the-world.\njulia:gc__stop_the_world: All threads have reached a safepoint and GC runs.\njulia:gc__mark__begin: Beginning the mark phase\njulia:gc__mark_end(scanned_bytes, perm_scanned): Mark phase ended\njulia:gc__sweep_begin(full): Starting sweep\njulia:gc__sweep_end: Sweep phase finished\njulia:gc__end: GC is finished, other threads continue work\njulia:gc__finalizer: Initial GC thread has finished running finalizers"},{"location":"devdocs/probes.html#Task-runtime-probes","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Task runtime probes","category":"section","text":"julia:rt__run__task(task): Switching to task task on current thread.\njulia:rt__pause__task(task): Switching from task task on current thread.\njulia:rt__new__task(parent, child): Task parent created task child on current thread.\njulia:rt__start__task(task): Task task started for the first time with a new stack.\njulia:rt__finish__task(task): Task task finished and will no longer execute.\njulia:rt__start__process__events(task): Task task started processing libuv events.\njulia:rt__finish__process__events(task): Task task finished processing libuv events."},{"location":"devdocs/probes.html#Task-queue-probes","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Task queue probes","category":"section","text":"julia:rt__taskq__insert(ptls, task): Thread ptls attempted to insert task into a PARTR multiq.\njulia:rt__taskq__get(ptls, task): Thread ptls popped task from a PARTR multiq."},{"location":"devdocs/probes.html#Thread-sleep/wake-probes","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Thread sleep/wake probes","category":"section","text":"julia:rt__sleep__check__wake(ptls, old_state): Thread (PTLS ptls) waking up, previously in state old_state.\njulia:rt__sleep__check__wakeup(ptls): Thread (PTLS ptls) woke itself up.\njulia:rt__sleep__check__sleep(ptls): Thread (PTLS ptls) is attempting to sleep.\njulia:rt__sleep__check__taskq__wake(ptls): Thread (PTLS ptls) fails to sleep due to tasks in PARTR multiq.\njulia:rt__sleep__check__task__wake(ptls): Thread (PTLS ptls) fails to sleep due to tasks in Base workqueue.\njulia:rt__sleep__check__uv__wake(ptls): Thread (PTLS ptls) fails to sleep due to libuv wakeup."},{"location":"devdocs/probes.html#Probe-usage-examples","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Probe usage examples","category":"section","text":""},{"location":"devdocs/probes.html#GC-stop-the-world-latency","page":"Instrumenting Julia with DTrace, and bpftrace","title":"GC stop-the-world latency","category":"section","text":"An example bpftrace script is given in contrib/gc_stop_the_world_latency.bt and it creates a histogram of the latency for all threads to reach a safepoint.\n\nRunning this Julia code, with julia -t 2\n\nusing Base.Threads\n\nfib(x) = x <= 1 ? 1 : fib(x-1) + fib(x-2)\n\nbeaver = @spawn begin\n    while true\n        fib(30)\n        # A manual safepoint is necessary since otherwise this loop\n        # may never yield to GC.\n        GC.safepoint()\n    end\nend\n\nallocator = @spawn begin\n    while true\n        zeros(1024)\n    end\nend\n\nwait(allocator)\n\nand in a second terminal\n\n> sudo contrib/bpftrace/gc_stop_the_world_latency.bt\nAttaching 4 probes...\nTracing Julia GC Stop-The-World Latency... Hit Ctrl-C to end.\n^C\n\n\n@usecs[1743412]:\n[4, 8)               971 |@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@|\n[8, 16)              837 |@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@        |\n[16, 32)             129 |@@@@@@                                              |\n[32, 64)              10 |                                                    |\n[64, 128)              1 |                                                    |\n\nWe can see the latency distribution of the stop-the-world phase in the executed Julia process."},{"location":"devdocs/probes.html#Task-spawn-monitor","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Task spawn monitor","category":"section","text":"It's sometimes useful to know when a task is spawning other tasks. This is very easy to see with rt__new__task. The first argument to the probe, parent, is the existing task which is creating a new task. This means that if you know the address of the task you want to monitor, you can easily just look at the tasks that were spawned by that specific task. Let's see how to do this; first let's start a Julia session and get the PID and REPL's task address:\n\n> julia\n               _\n   _       _ _(_)_     |  Documentation: https://docs.julialang.org\n  (_)     | (_) (_)    |\n   _ _   _| |_  __ _   |  Type \"?\" for help, \"]?\" for Pkg help.\n  | | | | | | |/ _` |  |\n  | | |_| | | | (_| |  |  Version 1.6.2 (2021-07-14)\n _/ |\\__'_|_|_|\\__'_|  |  Official https://julialang.org release\n|__/                   |\n\n1> getpid()\n997825\n\n2> current_task()\nTask (runnable) @0x00007f524d088010\n\nNow we can start bpftrace and have it monitor rt__new__task for only this parent:\n\nsudo bpftrace -p 997825 -e 'usdt:usr/lib/libjulia-internal.so:julia:rt__new__task /arg0==0x00007f524d088010/{ printf(\"Task: %x\\n\", arg0); }'\n\n(Note that in the above, arg0 is the first argument, parent).\n\nAnd if we spawn a single task:\n\nThreads.@spawn 1+1\n\nwe see this task being created:\n\nTask: 4d088010\n\nHowever, if we spawn a bunch of tasks from that newly-spawned task:\n\nThreads.@spawn for i in 1:10\n   Threads.@spawn 1+1\nend\n\nwe still only see one task from bpftrace:\n\nTask: 4d088010\n\nand it's still the same task we were monitoring! Of course, we can remove this filter to see all newly-created tasks just as easily:\n\nsudo bpftrace -p 997825 -e 'usdt:usr/lib/libjulia-internal.so:julia:rt__new__task { printf(\"Task: %x\\n\", arg0); }'\n\nTask: 4d088010\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\nTask: 4dc4e290\n\nWe can see our root task, and the newly-spawned task as the parent of the ten even newer tasks."},{"location":"devdocs/probes.html#Thundering-herd-detection","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Thundering herd detection","category":"section","text":"Task runtimes can often suffer from the \"thundering herd\" problem: when some work is added to a quiet task runtime, all threads may be woken up from their slumber, even if there isn't enough work for each thread to process. This can cause extra latency and CPU cycles while all threads awaken (and simultaneously go back to sleep, not finding any work to execute).\n\nWe can see this problem illustrated with bpftrace quite easily. First, in one terminal we start Julia with multiple threads (6 in this example), and get the PID of that process:\n\n> julia -t 6\n               _\n   _       _ _(_)_     |  Documentation: https://docs.julialang.org\n  (_)     | (_) (_)    |\n   _ _   _| |_  __ _   |  Type \"?\" for help, \"]?\" for Pkg help.\n  | | | | | | |/ _` |  |\n  | | |_| | | | (_| |  |  Version 1.6.2 (2021-07-14)\n _/ |\\__'_|_|_|\\__'_|  |  Official https://julialang.org release\n|__/                   |\n\n1> getpid()\n997825\n\nAnd in another terminal we start bpftrace monitoring our process, specifically probing the rt__sleep__check__wake hook:\n\nsudo bpftrace -p 997825 -e 'usdt:usr/lib/libjulia-internal.so:julia:rt__sleep__check__wake { printf(\"Thread wake up! %x\\n\", arg0); }'\n\nNow, we create and execute a single task in Julia:\n\nThreads.@spawn 1+1\n\nAnd in bpftrace we see printed out something like:\n\nThread wake up! 3f926100\nThread wake up! 3ebd5140\nThread wake up! 3f876130\nThread wake up! 3e2711a0\nThread wake up! 3e312190\n\nEven though we only spawned a single task (which only one thread could process at a time), we woke up all of our other threads! In the future, a smarter task runtime might only wake up a single thread (or none at all; the spawning thread could execute this task!), and we should see this behavior go away."},{"location":"devdocs/probes.html#Task-Monitor-with-BPFnative.jl","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Task Monitor with BPFnative.jl","category":"section","text":"BPFnative.jl is able to attach to USDT probe points just like bpftrace. There is a demo available for monitoring the task runtime, GC, and thread sleep/wake transitions here."},{"location":"devdocs/probes.html#Notes-on-using-bpftrace","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Notes on using bpftrace","category":"section","text":"An example probe in the bpftrace format looks like:\n\nusdt:usr/lib/libjulia-internal.so:julia:gc__begin\n{\n  @start[pid] = nsecs;\n}\n\nThe probe declaration takes the kind usdt, then either the path to the library or the PID, the provider name julia and the probe name gc__begin. Note that I am using a relative path to the libjulia-internal.so, but this might need to be an absolute path on a production system."},{"location":"devdocs/probes.html#Useful-references:","page":"Instrumenting Julia with DTrace, and bpftrace","title":"Useful references:","category":"section","text":"Julia Evans blog on Linux tracing systems\nLWN article on USDT and BPF\nGDB support for probes\nBrendan Gregg – Linux Performance"},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Optimizations","page":"isbits Union Optimizations","title":"isbits Union Optimizations","category":"section","text":"In Julia, the Array type holds both \"bits\" values as well as heap-allocated \"boxed\" values. The distinction is whether the value itself is stored inline (in the direct allocated memory of the array), or if the memory of the array is simply a collection of pointers to objects allocated elsewhere. In terms of performance, accessing values inline is clearly an advantage over having to follow a pointer to the actual value. The definition of \"isbits\" generally means any Julia type with a fixed, determinate size, meaning no \"pointer\" fields, see ?isbitstype.\n\nJulia also supports Union types, quite literally the union of a set of types. Custom Union type definitions can be extremely handy for applications wishing to \"cut across\" the nominal type system (i.e. explicit subtype relationships) and define methods or functionality on these, otherwise unrelated, set of types. A compiler challenge, however, is in determining how to treat these Union types. The naive approach (and indeed, what Julia itself did pre-0.7), is to simply make a \"box\" and then a pointer in the box to the actual value, similar to the previously mentioned \"boxed\" values. This is unfortunate, however, because of the number of small, primitive \"bits\" types (think UInt8, Int32, Float64, etc.) that would easily fit themselves inline in this \"box\" without needing any indirection for value access. There are two main ways Julia can take advantage of this optimization as of 0.7: isbits Union fields in types, and isbits Union Arrays."},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Structs","page":"isbits Union Optimizations","title":"isbits Union Structs","category":"section","text":"Julia now includes an optimization wherein \"isbits Union\" fields in types (mutable struct, struct, etc.) will be stored inline. This is accomplished by determining the \"inline size\" of the Union type (e.g. Union{UInt8, Int16} will have a size of two bytes, which represents the size needed of the largest Union type Int16), and in addition, allocating an extra \"type tag byte\" (UInt8), whose value signals the type of the actual value stored inline of the \"Union bytes\". The type tag byte value is the index of the actual value's type in the Union type's order of types. For example, a type tag value of 0x02 for a field with type Union{Nothing, UInt8, Int16} would indicate that an Int16 value is stored in the 16 bits of the field in the structure's memory; a 0x01 value would indicate that a UInt8 value was stored in the first 8 bits of the 16 bits of the field's memory. Lastly, a value of 0x00 signals that the nothing value will be returned for this field, even though, as a singleton type with a single type instance, it technically has a size of 0. The type tag byte for a type's Union field is stored directly after the field's computed Union memory."},{"location":"devdocs/isbitsunionarrays.html#isbits-Union-Memory","page":"isbits Union Optimizations","title":"isbits Union Memory","category":"section","text":"Julia can now also store \"isbits Union\" values inline in a Memory, as opposed to requiring an indirection box. The optimization is accomplished by storing an extra \"type tag memory\" of bytes, one byte per element, alongside the bytes of the actual data. This type tag memory serves the same function as the type field case: its value signals the type of the actual stored Union value. The \"type tag memory\" directly follows the regular data space. So the formula to access an isbits Union Array's type tag bytes is a->data + a->length * a->elsize."},{"location":"stdlib/Future.html#Future","page":"Future","title":"Future","category":"section","text":"The Future module implements future behavior of already existing functions, which will replace the current version in a future release of Julia."},{"location":"stdlib/Future.html#Future.copy!","page":"Future","title":"Future.copy!","category":"function","text":"Future.copy!(dst, src) -> dst\n\nCopy src into dst.\n\ncompat: Julia 1.1\nThis function has moved to Base with Julia 1.1, consider using copy!(dst, src) instead. Future.copy! will be deprecated in the future.\n\n\n\n\n\n"},{"location":"stdlib/Future.html#Future.randjump","page":"Future","title":"Future.randjump","category":"function","text":"randjump(r::MersenneTwister, steps::Integer)::MersenneTwister\n\nCreate an initialized MersenneTwister object, whose state is moved forward (without generating numbers) from r by steps steps. One such step corresponds to the generation of two Float64 numbers. For each different value of steps, a large polynomial has to be generated internally. One is already pre-computed for steps=big(10)^20.\n\n\n\n\n\n"},{"location":"devdocs/contributing/git-workflow.html#Git-workflow-recommendations","page":"Git workflow recommendations","title":"Git workflow recommendations","category":"section","text":""},{"location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Requests","page":"Git workflow recommendations","title":"Git Recommendations For Pull Requests","category":"section","text":"Avoid working from the master branch of your fork. Create a new branch as it will make it easier to update your pull request if Julia's master changes.\nTry to squash together small commits that make repeated changes to the same section of code, so your pull request is easier to review. A reasonable number of separate well-factored commits is fine, especially for larger changes.\nIf any conflicts arise due to changes in Julia's master, prefer updating your pull request branch with git rebase versus git merge or git pull, since the latter will introduce merge commits that clutter the git history with noise that makes your changes more difficult to review.\nDescriptive commit messages are good.\nUsing git add -p or git add -i can be useful to avoid accidentally committing unrelated changes.\nWhen linking to specific lines of code in discussion of an issue or pull request, hit the y key while viewing code on GitHub to reload the page with a URL that includes the specific version that you're viewing. That way any lines of code that you refer to will still make sense in the future, even if the content of the file changes.\nWhitespace can be automatically removed from existing commits with git rebase.\nTo remove whitespace for the previous commit, run git rebase --whitespace=fix HEAD~1.\nTo remove whitespace relative to the master branch, run git rebase --whitespace=fix master."},{"location":"devdocs/contributing/git-workflow.html#Git-Recommendations-For-Pull-Request-Reviewers","page":"Git workflow recommendations","title":"Git Recommendations For Pull Request Reviewers","category":"section","text":"When merging, we generally like squash+merge. Unless it is the rare case of a PR with carefully staged individual commits that you want in the history separately, in which case merge is acceptable, but usually prefer squash+merge."},{"location":"devdocs/gc-mmtk.html#Julia-MMTk","page":"Julia + MMTk","title":"Julia + MMTk","category":"section","text":"There has been quite a lot of effort to refactor the GC code inside Julia to support external GCs. The first step to enable using different GC algorithms for Julia was the design and implementation of a GC interface. To drive that interface, we added support for building Julia with MMTk (Memory Management Toolkit). Using Julia + MMTk enables testing different GC implementations, allowing developers to choose a specific implementation when building Julia from source. The connection between Julia and MMTk is done via a binding, which links the language runtime with MMTk core. The mmtk-julia binding is written in Rust and can be found in this repository.\n\n[!NOTE] Using a different GC requires building Julia from source. It is not possible to switch implementations at runtime. To see what version of the GC is currently being used, run versioninfo() from the Julia REPL and it should show the version under GC: ...."},{"location":"devdocs/gc-mmtk.html#Building-Julia-with-MMTk","page":"Julia + MMTk","title":"Building Julia with MMTk","category":"section","text":"There are 3 different ways of building Julia with MMTk: building from source using a fixed release of the binding, checking out a custom version in the mmtk-julia repository or using a precompiled binary from Julia's BinaryBuilder. The easiest way is to use the BinaryBuilder binary. First, to enable MMTk as a third-party GC, set the variable WITH_THIRD_PARTY_GC to mmtk. Then, for example, to use the Immix as the GC, simply set the variable MMTK_PLAN=Immix and build Julia as usual.\n\nThere are different configurations supported by the following variables, which can be set in a Make.user file or as an environment variable. Note that at this time, setting MMTK_PLAN=StickyImmix (to use a generational version of Immix) or MMTK_MOVING=1 (to enable object movement) will likely cause segmentation faults or other build failures, since we have not added support for these configurations yet. Setting MMTK_BUILD=debug will force a debug build of the binding, which will print some logging information that can be used to find errors that are specific to MMTk.\n\nVariable  \nMMTK_PLAN Immix StickyImmix\nMMTK_MOVING 0 1\nMMTK_BUILD release debug\n\nNote that when setting only MMTK_PLAN, then the default is to do a non-moving, release build."},{"location":"devdocs/gc-mmtk.html#Building-mmtk-julia-from-source","page":"Julia + MMTk","title":"Building mmtk-julia from source","category":"section","text":"It is also possible to build the binding from source. To do so, set the variable USE_BINARYBUILDER_MMTK_JULIA=0 and the latest release version of the binding will be downloaded and built as part of building Julia. Note that this requires an installation of the rust toolchain.\n\nIt is also possible to build a custom version of binding by checking it out from the git repository and setting a variable named MMTK_JULIA_DIR as the path that contains the binding.\n\nFor more information on building Julia with MMTk, please refer to the README file in the binding repo."},{"location":"devdocs/gc-mmtk.html#I've-got-a-build-error-when-building-Julia-with-MMTk,-what-should-I-do?","page":"Julia + MMTk","title":"I've got a build error when building Julia with MMTk, what should I do?","category":"section","text":"If you try to build Julia with MMTk and get an error it is likely due to a change to Julia that has not been yet propagated to the binding or to the code in Julia that is specific to MMTk. Some changes include:\n\n(1) Changing the memory layout of objects in Julia. The binding relies on automatically generated Rust FFI bindings from Julia code. These files are generated using a crate named rust-bindgen. To regenerate those files, check out the latest version of the mmtk-julia binding, set the variable JULIA_PATH to the path of the Julia version you are trying to build and run make regen-bindgen-ffi from the directory containing the binding. This should delete the current version of the FFI bindings and generate a new version based on the Julia code from JULIA_PATH.\n\n(2) Changing the root objects passed to the GC. Julia passes a set of objects to the GC as roots in the function gcmarkroots. At the moment, this set needs to be consistent between both the Stock GC and MMTk (in the function jl_gc_scan_vm_specific_roots).\n\n(3) Changing how objects are scanned. MMTk uses the same strategy to find references in Julia objects as the stock GC (see gcmarkoutrefs). Changing the logic from this function should be reflected in the Rust code in the binding that scan Julia objects.\n\nIf your case is not included in one of the alternatives above, please create an issue in the Julia repository tagging it with the GC: MMTK label."},{"location":"stdlib/UUIDs.html#UUIDs","page":"UUIDs","title":"UUIDs","category":"section","text":""},{"location":"stdlib/UUIDs.html#UUIDs.uuid1","page":"UUIDs","title":"UUIDs.uuid1","category":"function","text":"uuid1([rng::AbstractRNG])::UUID\n\nGenerates a version 1 (time-based) universally unique identifier (UUID), as specified by RFC 4122. Note that the Node ID is randomly generated (does not identify the host) according to section 4.5 of the RFC.\n\nThe default rng used by uuid1 is not Random.default_rng() and every invocation of uuid1() without an argument should be expected to return a unique identifier. Importantly, the outputs of uuid1 do not repeat even when Random.seed!(seed) is called. Currently (as of Julia 1.6), uuid1 uses Random.RandomDevice as the default rng. However, this is an implementation detail that may change in the future.\n\ncompat: Julia 1.6\nThe output of uuid1 does not depend on Random.default_rng() as of Julia 1.6.\n\nExamples\n\njulia> using Random\n\njulia> rng = MersenneTwister(1234);\n\njulia> uuid1(rng)\nUUID(\"cfc395e8-590f-11e8-1f13-43a2532b2fa8\")\n\n\n\n\n\n"},{"location":"stdlib/UUIDs.html#UUIDs.uuid4","page":"UUIDs","title":"UUIDs.uuid4","category":"function","text":"uuid4([rng::AbstractRNG])::UUID\n\nGenerates a version 4 (random or pseudo-random) universally unique identifier (UUID), as specified by RFC 4122.\n\nThe default rng used by uuid4 is not Random.default_rng() and every invocation of uuid4() without an argument should be expected to return a unique identifier. Importantly, the outputs of uuid4 do not repeat even when Random.seed!(seed) is called. Currently (as of Julia 1.6), uuid4 uses Random.RandomDevice as the default rng. However, this is an implementation detail that may change in the future.\n\ncompat: Julia 1.6\nThe output of uuid4 does not depend on Random.default_rng() as of Julia 1.6.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> uuid4(rng)\nUUID(\"856e446e-0c6a-472a-9638-f7b8557cd282\")\n\n\n\n\n\n"},{"location":"stdlib/UUIDs.html#UUIDs.uuid5","page":"UUIDs","title":"UUIDs.uuid5","category":"function","text":"uuid5(ns::UUID, name::String)::UUID\n\nGenerates a version 5 (namespace and domain-based) universally unique identifier (UUID), as specified by RFC 4122.\n\ncompat: Julia 1.1\nThis function requires at least Julia 1.1.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> u4 = uuid4(rng)\nUUID(\"856e446e-0c6a-472a-9638-f7b8557cd282\")\n\njulia> u5 = uuid5(u4, \"julia\")\nUUID(\"2df91e3f-da06-5362-a6fe-03772f2e14c9\")\n\n\n\n\n\n"},{"location":"stdlib/UUIDs.html#UUIDs.uuid_version","page":"UUIDs","title":"UUIDs.uuid_version","category":"function","text":"uuid_version(u::UUID)::Int\n\nInspects the given UUID and returns its version (see RFC 4122).\n\nExamples\n\njulia> uuid_version(uuid4())\n4\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates","page":"Dates","title":"Dates","category":"section","text":"The Dates module provides two types for working with dates: Date and DateTime, representing day and millisecond precision, respectively; both are subtypes of the abstract TimeType. The motivation for distinct types is simple: some operations are much simpler, both in terms of code and mental reasoning, when the complexities of greater precision don't have to be dealt with. For example, since the Date type only resolves to the precision of a single date (i.e. no hours, minutes, or seconds), normal considerations for time zones, daylight savings/summer time, and leap seconds are unnecessary and avoided.\n\nBoth Date and DateTime are basically immutable Int64 wrappers. The single instant field of either type is actually a UTInstant{P} type, which represents a continuously increasing machine timeline based on the UT second [1]. The DateTime type is not aware of time zones (naive, in Python parlance), analogous to a LocalDateTime in Java 8. Additional time zone functionality can be added through the TimeZones.jl package, which compiles the IANA time zone database. Both Date and DateTime are based on the ISO 8601 standard, which follows the proleptic Gregorian calendar. One note is that the ISO 8601 standard is particular about BC/BCE dates. In general, the last day of the BC/BCE era, 1-12-31 BC/BCE, was followed by 1-1-1 AD/CE, thus no year zero exists. The ISO standard, however, states that 1 BC/BCE is year zero, so 0000-12-31 is the day before 0001-01-01, and year -0001 (yes, negative one for the year) is 2 BC/BCE, year -0002 is 3 BC/BCE, etc.\n\n[1]: The notion of the UT second is actually quite fundamental. There are basically two different notions of time generally accepted, one based on the physical rotation of the earth (one full rotation = 1 day), the other based on the SI second (a fixed, constant value). These are radically different! Think about it, a \"UT second\", as defined relative to the rotation of the earth, may have a different absolute length depending on the day! Anyway, the fact that Date and DateTime are based on UT seconds is a simplifying, yet honest assumption so that things like leap seconds and all their complexity can be avoided. This basis of time is formally called UT or UT1. Basing types on the UT second basically means that every minute has 60 seconds and every day has 24 hours and leads to more natural calculations when working with calendar dates."},{"location":"stdlib/Dates.html#Constructors","page":"Dates","title":"Constructors","category":"section","text":"Date and DateTime types can be constructed by integer or Period types, by parsing, or through adjusters (more on those later):\n\njulia> DateTime(2013)\n2013-01-01T00:00:00\n\njulia> DateTime(2013,7)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1)\n2013-07-01T00:00:00\n\njulia> DateTime(2013,7,1,12)\n2013-07-01T12:00:00\n\njulia> DateTime(2013,7,1,12,30)\n2013-07-01T12:30:00\n\njulia> DateTime(2013,7,1,12,30,59)\n2013-07-01T12:30:59\n\njulia> DateTime(2013,7,1,12,30,59,1)\n2013-07-01T12:30:59.001\n\njulia> Date(2013)\n2013-01-01\n\njulia> Date(2013,7)\n2013-07-01\n\njulia> Date(2013,7,1)\n2013-07-01\n\njulia> Date(Dates.Year(2013),Dates.Month(7),Dates.Day(1))\n2013-07-01\n\njulia> Date(Dates.Month(7),Dates.Year(2013))\n2013-07-01\n\nDate or DateTime parsing is accomplished by the use of format strings. Format strings work by the notion of defining delimited or fixed-width \"slots\" that contain a period to parse and passing the text to parse and format string to a Date or DateTime constructor, of the form Date(\"2015-01-01\",dateformat\"y-m-d\") or DateTime(\"20150101\",dateformat\"yyyymmdd\").\n\nDelimited slots are marked by specifying the delimiter the parser should expect between two subsequent periods; so \"y-m-d\" lets the parser know that between the first and second slots in a date string like \"2014-07-16\", it should find the - character. The y, m, and d characters let the parser know which periods to parse in each slot.\n\nAs in the case of constructors above such as Date(2013), delimited DateFormats allow for missing parts of dates and times so long as the preceding parts are given. The other parts are given the usual default values. For example, Date(\"1981-03\", dateformat\"y-m-d\") returns 1981-03-01, whilst Date(\"31/12\", dateformat\"d/m/y\") gives 0001-12-31.  (Note that the default year is 1 AD/CE.) An empty string, however, always throws an ArgumentError.\n\nFixed-width slots are specified by repeating the period character the number of times corresponding to the width with no delimiter between characters. So dateformat\"yyyymmdd\" would correspond to a date string like \"20140716\". The parser distinguishes a fixed-width slot by the absence of a delimiter, noting the transition \"yyyymm\" from one period character to the next.\n\nSupport for text-form month parsing is also supported through the u and U characters, for abbreviated and full-length month names, respectively. By default, only English month names are supported, so u corresponds to \"Jan\", \"Feb\", \"Mar\", etc. And U corresponds to \"January\", \"February\", \"March\", etc. Similar to other name=>value mapping functions dayname and monthname, custom locales can be loaded by passing in the locale=>Dict{String,Int} mapping to the MONTHTOVALUEABBR and MONTHTOVALUE dicts for abbreviated and full-name month names, respectively.\n\nThe above examples used the dateformat\"\" string macro. This macro creates a DateFormat object once when the macro is expanded and uses the same DateFormat object even if a code snippet is run multiple times.\n\njulia> for i = 1:10^5\n           Date(\"2015-01-01\", dateformat\"y-m-d\")\n       end\n\nOr you can create the DateFormat object explicitly:\n\njulia> df = DateFormat(\"y-m-d\");\n\njulia> dt = Date(\"2015-01-01\",df)\n2015-01-01\n\njulia> dt2 = Date(\"2015-01-02\",df)\n2015-01-02\n\nAlternatively, use broadcasting:\n\njulia> years = [\"2015\", \"2016\"];\n\njulia> Date.(years, DateFormat(\"yyyy\"))\n2-element Vector{Date}:\n 2015-01-01\n 2016-01-01\n\nFor convenience, you may pass the format string directly (e.g., Date(\"2015-01-01\",\"y-m-d\")), although this form incurs performance costs if you are parsing the same format repeatedly, as it internally creates a new DateFormat object each time.\n\nAs well as via the constructors, a Date or DateTime can be constructed from strings using the parse and tryparse functions, but with an optional third argument of type DateFormat specifying the format; for example, parse(Date, \"06.23.2013\", dateformat\"m.d.y\"), or tryparse(DateTime, \"1999-12-31T23:59:59\") which uses the default format. The notable difference between the functions is that with tryparse, an error is not thrown if the string is empty or in an invalid format; instead nothing is returned.\n\ncompat: Julia 1.9\nBefore Julia 1.9, empty strings could be passed to constructors and parse without error, returning as appropriate DateTime(1), Date(1) or Time(0). Likewise, tryparse did not return nothing.\n\nA full suite of parsing and formatting tests and examples is available in stdlib/Dates/test/io.jl."},{"location":"stdlib/Dates.html#Durations/Comparisons","page":"Dates","title":"Durations/Comparisons","category":"section","text":"Finding the length of time between two Date or DateTime is straightforward given their underlying representation as UTInstant{Day} and UTInstant{Millisecond}, respectively. The difference between Date is returned in the number of Day, and DateTime in the number of Millisecond. Similarly, comparing TimeType is a simple matter of comparing the underlying machine instants (which in turn compares the internal Int64 values).\n\njulia> dt = Date(2012,2,29)\n2012-02-29\n\njulia> dt2 = Date(2000,2,1)\n2000-02-01\n\njulia> dump(dt)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 734562\n\njulia> dump(dt2)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 730151\n\njulia> dt > dt2\ntrue\n\njulia> dt != dt2\ntrue\n\njulia> dt + dt2\nERROR: MethodError: no method matching +(::Date, ::Date)\n[...]\n\njulia> dt * dt2\nERROR: MethodError: no method matching *(::Date, ::Date)\n[...]\n\njulia> dt / dt2\nERROR: MethodError: no method matching /(::Date, ::Date)\n\njulia> dt - dt2\n4411 days\n\njulia> dt2 - dt\n-4411 days\n\njulia> dt = DateTime(2012,2,29)\n2012-02-29T00:00:00\n\njulia> dt2 = DateTime(2000,2,1)\n2000-02-01T00:00:00\n\njulia> dt - dt2\n381110400000 milliseconds"},{"location":"stdlib/Dates.html#Accessor-Functions","page":"Dates","title":"Accessor Functions","category":"section","text":"Because the Date and DateTime types are stored as single Int64 values, date parts or fields can be retrieved through accessor functions. The lowercase accessors return the field as an integer:\n\njulia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.year(t)\n2014\n\njulia> Dates.month(t)\n1\n\njulia> Dates.week(t)\n5\n\njulia> Dates.day(t)\n31\n\nWhile propercase return the same value in the corresponding Period type:\n\njulia> Dates.Year(t)\n2014 years\n\njulia> Dates.Day(t)\n31 days\n\nCompound methods are provided because it is more efficient to access multiple fields at the same time than individually:\n\njulia> Dates.yearmonth(t)\n(2014, 1)\n\njulia> Dates.monthday(t)\n(1, 31)\n\njulia> Dates.yearmonthday(t)\n(2014, 1, 31)\n\nOne may also access the underlying UTInstant or integer value:\n\njulia> dump(t)\nDate\n  instant: Dates.UTInstant{Day}\n    periods: Day\n      value: Int64 735264\n\njulia> t.instant\nDates.UTInstant{Day}(Day(735264))\n\njulia> Dates.value(t)\n735264"},{"location":"stdlib/Dates.html#Query-Functions","page":"Dates","title":"Query Functions","category":"section","text":"Query functions provide calendrical information about a TimeType. They include information about the day of the week:\n\njulia> t = Date(2014, 1, 31)\n2014-01-31\n\njulia> Dates.dayofweek(t)\n5\n\njulia> Dates.dayname(t)\n\"Friday\"\n\njulia> Dates.dayofweekofmonth(t) # 5th Friday of January\n5\n\nMonth of the year:\n\njulia> Dates.monthname(t)\n\"January\"\n\njulia> Dates.daysinmonth(t)\n31\n\nAs well as information about the TimeType's year and quarter:\n\njulia> Dates.isleapyear(t)\nfalse\n\njulia> Dates.dayofyear(t)\n31\n\njulia> Dates.quarterofyear(t)\n1\n\njulia> Dates.dayofquarter(t)\n31\n\nThe dayname and monthname methods can also take an optional locale keyword that can be used to return the name of the day or month of the year for other languages/locales. There are also versions of these functions returning the abbreviated names, namely dayabbr and monthabbr. First the mapping is loaded into the LOCALES variable:\n\njulia> french_months = [\"janvier\", \"février\", \"mars\", \"avril\", \"mai\", \"juin\",\n                        \"juillet\", \"août\", \"septembre\", \"octobre\", \"novembre\", \"décembre\"];\n\njulia> french_months_abbrev = [\"janv\",\"févr\",\"mars\",\"avril\",\"mai\",\"juin\",\n                              \"juil\",\"août\",\"sept\",\"oct\",\"nov\",\"déc\"];\n\njulia> french_days = [\"lundi\",\"mardi\",\"mercredi\",\"jeudi\",\"vendredi\",\"samedi\",\"dimanche\"];\n\njulia> Dates.LOCALES[\"french\"] = Dates.DateLocale(french_months, french_months_abbrev, french_days, [\"\"]);\n\nThe above mentioned functions can then be used to perform the queries:\n\njulia> Dates.dayname(t;locale=\"french\")\n\"vendredi\"\n\njulia> Dates.monthname(t;locale=\"french\")\n\"janvier\"\n\njulia> Dates.monthabbr(t;locale=\"french\")\n\"janv\"\n\nSince the abbreviated versions of the days are not loaded, trying to use the function dayabbr will throw an error.\n\njulia> Dates.dayabbr(t;locale=\"french\")\nERROR: BoundsError: attempt to access 1-element Vector{String} at index [5]\nStacktrace:\n[...]"},{"location":"stdlib/Dates.html#TimeType-Period-Arithmetic","page":"Dates","title":"TimeType-Period Arithmetic","category":"section","text":"It's good practice when using any language/date framework to be familiar with how date-period arithmetic is handled as there are some tricky issues to deal with (though much less so for day-precision types).\n\nThe Dates module approach tries to follow the simple principle of trying to change as little as possible when doing Period arithmetic. This approach is also often known as calendrical arithmetic or what you would probably guess if someone were to ask you the same calculation in a conversation. Why all the fuss about this? Let's take a classic example: add 1 month to January 31st, 2014. What's the answer? Javascript will say March 3 (assumes 31 days). PHP says March 2 (assumes 30 days). The fact is, there is no right answer. In the Dates module, it gives the result of February 28th. How does it figure that out? Consider the classic 7-7-7 gambling game in casinos.\n\nNow just imagine that instead of 7-7-7, the slots are Year-Month-Day, or in our example, 2014-01-31. When you ask to add 1 month to this date, the month slot is incremented, so now we have 2014-02-31. Then the day number is checked if it is greater than the last valid day of the new month; if it is (as in the case above), the day number is adjusted down to the last valid day (28). What are the ramifications with this approach? Go ahead and add another month to our date, 2014-02-28 + Month(1) == 2014-03-28. What? Were you expecting the last day of March? Nope, sorry, remember the 7-7-7 slots. As few slots as possible are going to change, so we first increment the month slot by 1, 2014-03-28, and boom, we're done because that's a valid date. On the other hand, if we were to add 2 months to our original date, 2014-01-31, then we end up with 2014-03-31, as expected. The other ramification of this approach is a loss in associativity when a specific ordering is forced (i.e. adding things in different orders results in different outcomes). For example:\n\njulia> (Date(2014,1,29)+Dates.Day(1)) + Dates.Month(1)\n2014-02-28\n\njulia> (Date(2014,1,29)+Dates.Month(1)) + Dates.Day(1)\n2014-03-01\n\nWhat's going on there? In the first line, we're adding 1 day to January 29th, which results in 2014-01-30; then we add 1 month, so we get 2014-02-30, which then adjusts down to 2014-02-28. In the second example, we add 1 month first, where we get 2014-02-29, which adjusts down to 2014-02-28, and then add 1 day, which results in 2014-03-01. One design principle that helps in this case is that, in the presence of multiple Periods, the operations will be ordered by the Periods' types, not their value or positional order; this means Year will always be added first, then Month, then Week, etc. Hence the following does result in associativity and Just Works:\n\njulia> Date(2014,1,29) + Dates.Day(1) + Dates.Month(1)\n2014-03-01\n\njulia> Date(2014,1,29) + Dates.Month(1) + Dates.Day(1)\n2014-03-01\n\nTricky? Perhaps. What is an innocent Dates user to do? The bottom line is to be aware that explicitly forcing a certain associativity, when dealing with months, may lead to some unexpected results, but otherwise, everything should work as expected. Thankfully, that's pretty much the extent of the odd cases in date-period arithmetic when dealing with time in UT (avoiding the \"joys\" of dealing with daylight savings, leap seconds, etc.).\n\nAs a bonus, all period arithmetic objects work directly with ranges:\n\njulia> dr = Date(2014,1,29):Day(1):Date(2014,2,3)\nDate(\"2014-01-29\"):Day(1):Date(\"2014-02-03\")\n\njulia> collect(dr)\n6-element Vector{Date}:\n 2014-01-29\n 2014-01-30\n 2014-01-31\n 2014-02-01\n 2014-02-02\n 2014-02-03\n\njulia> dr = Date(2014,1,29):Dates.Month(1):Date(2014,07,29)\nDate(\"2014-01-29\"):Month(1):Date(\"2014-07-29\")\n\njulia> collect(dr)\n7-element Vector{Date}:\n 2014-01-29\n 2014-02-28\n 2014-03-29\n 2014-04-29\n 2014-05-29\n 2014-06-29\n 2014-07-29"},{"location":"stdlib/Dates.html#Adjuster-Functions","page":"Dates","title":"Adjuster Functions","category":"section","text":"As convenient as date-period arithmetic is, often the kinds of calculations needed on dates take on a calendrical or temporal nature rather than a fixed number of periods. Holidays are a perfect example; most follow rules such as \"Memorial Day = Last Monday of May\", or \"Thanksgiving = 4th Thursday of November\". These kinds of temporal expressions deal with rules relative to the calendar, like first or last of the month, next Tuesday, or the first and third Wednesdays, etc.\n\nThe Dates module provides the adjuster API through several convenient methods that aid in simply and succinctly expressing temporal rules. The first group of adjuster methods deal with the first and last of weeks, months, quarters, and years. They each take a single TimeType as input and return or adjust to the first or last of the desired period relative to the input.\n\njulia> Dates.firstdayofweek(Date(2014,7,16)) # Adjusts the input to the Monday of the input's week\n2014-07-14\n\njulia> Dates.lastdayofmonth(Date(2014,7,16)) # Adjusts to the last day of the input's month\n2014-07-31\n\njulia> Dates.lastdayofquarter(Date(2014,7,16)) # Adjusts to the last day of the input's quarter\n2014-09-30\n\nThe next two higher-order methods, tonext, and toprev, generalize working with temporal expressions by taking a DateFunction as first argument, along with a starting TimeType. A DateFunction is just a function, usually anonymous, that takes a single TimeType as input and returns a Bool, true indicating a satisfied adjustment criterion. For example:\n\njulia> istuesday = x->Dates.dayofweek(x) == Dates.Tuesday; # Returns true if the day of the week of x is Tuesday\n\njulia> Dates.tonext(istuesday, Date(2014,7,13)) # 2014-07-13 is a Sunday\n2014-07-15\n\njulia> Dates.tonext(Date(2014,7,13), Dates.Tuesday) # Convenience method provided for day of the week adjustments\n2014-07-15\n\nThis is useful with the do-block syntax for more complex temporal expressions:\n\njulia> Dates.tonext(Date(2014,7,13)) do x\n           # Return true on the 4th Thursday of November (Thanksgiving)\n           Dates.dayofweek(x) == Dates.Thursday &&\n           Dates.dayofweekofmonth(x) == 4 &&\n           Dates.month(x) == Dates.November\n       end\n2014-11-27\n\nThe Base.filter method can be used to obtain all valid dates/moments in a specified range:\n\n# Pittsburgh street cleaning; Every 2nd Tuesday from April to November\n# Date range from January 1st, 2014 to January 1st, 2015\njulia> dr = Dates.Date(2014):Day(1):Dates.Date(2015);\n\njulia> filter(dr) do x\n           Dates.dayofweek(x) == Dates.Tue &&\n           Dates.April <= Dates.month(x) <= Dates.Nov &&\n           Dates.dayofweekofmonth(x) == 2\n       end\n8-element Vector{Date}:\n 2014-04-08\n 2014-05-13\n 2014-06-10\n 2014-07-08\n 2014-08-12\n 2014-09-09\n 2014-10-14\n 2014-11-11\n\nAdditional examples and tests are available in stdlib/Dates/test/adjusters.jl."},{"location":"stdlib/Dates.html#Period-Types","page":"Dates","title":"Period Types","category":"section","text":"Periods are a human view of discrete, sometimes irregular durations of time. Consider 1 month; it could represent, in days, a value of 28, 29, 30, or 31 depending on the year and month context. Or a year could represent 365 or 366 days in the case of a leap year. Period types are simple Int64 wrappers and are constructed by wrapping any Int64 convertible type, i.e. Year(1) or Month(3.0). Arithmetic between Period of the same type behave like integers, and limited Period-Real arithmetic is available. You can extract the underlying integer with Dates.value.\n\njulia> y1 = Dates.Year(1)\n1 year\n\njulia> y2 = Dates.Year(2)\n2 years\n\njulia> y3 = Dates.Year(10)\n10 years\n\njulia> y1 + y2\n3 years\n\njulia> div(y3,y2)\n5\n\njulia> y3 - y2\n8 years\n\njulia> y3 % y2\n0 years\n\njulia> div(y3,3) # mirrors integer division\n3 years\n\njulia> Dates.value(Dates.Millisecond(10))\n10\n\nRepresenting periods or durations that are not integer multiples of the basic types can be achieved with the Dates.CompoundPeriod type. Compound periods may be constructed manually from simple Period types. Additionally, the canonicalize function can be used to break down a period into a Dates.CompoundPeriod. This is particularly useful to convert a duration, e.g., a difference of two DateTime, into a more convenient representation.\n\njulia> cp = Dates.CompoundPeriod(Day(1),Minute(1))\n1 day, 1 minute\n\njulia> t1 = DateTime(2018,8,8,16,58,00)\n2018-08-08T16:58:00\n\njulia> t2 = DateTime(2021,6,23,10,00,00)\n2021-06-23T10:00:00\n\njulia> canonicalize(t2-t1) # creates a CompoundPeriod\n149 weeks, 6 days, 17 hours, 2 minutes"},{"location":"stdlib/Dates.html#Rounding","page":"Dates","title":"Rounding","category":"section","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round:\n\njulia> floor(Date(1985, 8, 16), Dates.Month)\n1985-08-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Dates.Minute(15))\n2013-02-13T00:45:00\n\njulia> round(DateTime(2016, 8, 6, 20, 15), Dates.Day)\n2016-08-07T00:00:00\n\nUnlike the numeric round method, which breaks ties toward the even number by default, the TimeTyperound method uses the RoundNearestTiesUp rounding mode. (It's difficult to guess what breaking ties to nearest \"even\" TimeType would entail.) Further details on the available RoundingMode s can be found in the API reference.\n\nRounding should generally behave as expected, but there are a few cases in which the expected behaviour is not obvious."},{"location":"stdlib/Dates.html#Rounding-Epoch","page":"Dates","title":"Rounding Epoch","category":"section","text":"In many cases, the resolution specified for rounding (e.g., Dates.Second(30)) divides evenly into the next largest period (in this case, Dates.Minute(1)). But rounding behaviour in cases in which this is not true may lead to confusion. What is the expected result of rounding a DateTime to the nearest 10 hours?\n\njulia> round(DateTime(2016, 7, 17, 11, 55), Dates.Hour(10))\n2016-07-17T12:00:00\n\nThat may seem confusing, given that the hour (12) is not divisible by 10. The reason that 2016-07-17T12:00:00 was chosen is that it is 17,676,660 hours after 0000-01-01T00:00:00, and 17,676,660 is divisible by 10.\n\nAs Julia Date and DateTime values are represented according to the ISO 8601 standard, 0000-01-01T00:00:00 was chosen as base (or \"rounding epoch\") from which to begin the count of days (and milliseconds) used in rounding calculations. (Note that this differs slightly from Julia's internal representation of Date s using Rata Die notation; but since the ISO 8601 standard is most visible to the end user, 0000-01-01T00:00:00 was chosen as the rounding epoch instead of the 0000-12-31T00:00:00 used internally to minimize confusion.)\n\nThe only exception to the use of 0000-01-01T00:00:00 as the rounding epoch is when rounding to weeks. Rounding to the nearest week will always return a Monday (the first day of the week as specified by ISO 8601). For this reason, we use 0000-01-03T00:00:00 (the first day of the first week of year 0000, as defined by ISO 8601) as the base when rounding to a number of weeks.\n\nHere is a related case in which the expected behaviour is not necessarily obvious: What happens when we round to the nearest P(2), where P is a Period type? In some cases (specifically, when P <: Dates.TimePeriod) the answer is clear:\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Hour(2))\n2016-07-17T08:00:00\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Minute(2))\n2016-07-17T08:56:00\n\nThis seems obvious, because two of each of these periods still divides evenly into the next larger order period. But in the case of two months (which still divides evenly into one year), the answer may be surprising:\n\njulia> round(DateTime(2016, 7, 17, 8, 55, 30), Dates.Month(2))\n2016-07-01T00:00:00\n\nWhy round to the first day in July, even though it is month 7 (an odd number)? The key is that months are 1-indexed (the first month is assigned 1), unlike hours, minutes, seconds, and milliseconds (the first of which are assigned 0).\n\nThis means that rounding a DateTime to an even multiple of seconds, minutes, hours, or years (because the ISO 8601 specification includes a year zero) will result in a DateTime with an even value in that field, while rounding a DateTime to an even multiple of months will result in the months field having an odd value. Because both months and years may contain an irregular number of days, whether rounding to an even number of days will result in an even value in the days field is uncertain.\n\nSee the API reference for additional information on methods exported from the Dates module."},{"location":"stdlib/Dates.html#stdlib-dates-api","page":"Dates","title":"API reference","category":"section","text":""},{"location":"stdlib/Dates.html#Dates-and-Time-Types","page":"Dates","title":"Dates and Time Types","category":"section","text":""},{"location":"stdlib/Dates.html#Dates-Functions","page":"Dates","title":"Dates Functions","category":"section","text":""},{"location":"stdlib/Dates.html#Accessor-Functions-2","page":"Dates","title":"Accessor Functions","category":"section","text":""},{"location":"stdlib/Dates.html#Query-Functions-2","page":"Dates","title":"Query Functions","category":"section","text":""},{"location":"stdlib/Dates.html#Adjuster-Functions-2","page":"Dates","title":"Adjuster Functions","category":"section","text":""},{"location":"stdlib/Dates.html#Periods","page":"Dates","title":"Periods","category":"section","text":""},{"location":"stdlib/Dates.html#Rounding-Functions","page":"Dates","title":"Rounding Functions","category":"section","text":"Date and DateTime values can be rounded to a specified resolution (e.g., 1 month or 15 minutes) with floor, ceil, or round.\n\nMost Period values can also be rounded to a specified resolution:\n\nThe following functions are not exported:"},{"location":"stdlib/Dates.html#Conversion-Functions","page":"Dates","title":"Conversion Functions","category":"section","text":""},{"location":"stdlib/Dates.html#Constants","page":"Dates","title":"Constants","category":"section","text":"Days of the Week:\n\nVariable Abbr. Value (Int)\nMonday Mon 1\nTuesday Tue 2\nWednesday Wed 3\nThursday Thu 4\nFriday Fri 5\nSaturday Sat 6\nSunday Sun 7\n\nMonths of the Year:\n\nVariable Abbr. Value (Int)\nJanuary Jan 1\nFebruary Feb 2\nMarch Mar 3\nApril Apr 4\nMay May 5\nJune Jun 6\nJuly Jul 7\nAugust Aug 8\nSeptember Sep 9\nOctober Oct 10\nNovember Nov 11\nDecember Dec 12"},{"location":"stdlib/Dates.html#Common-Date-Formatters","page":"Dates","title":"Common Date Formatters","category":"section","text":""},{"location":"stdlib/Dates.html#Dates.Period","page":"Dates","title":"Dates.Period","category":"type","text":"Period\nYear\nQuarter\nMonth\nWeek\nDay\nHour\nMinute\nSecond\nMillisecond\nMicrosecond\nNanosecond\n\nPeriod types represent discrete, human representations of time.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.CompoundPeriod","page":"Dates","title":"Dates.CompoundPeriod","category":"type","text":"CompoundPeriod\n\nA CompoundPeriod is useful for expressing time periods that are not a fixed multiple of smaller periods. For example, \"a year and a day\" is not a fixed number of days, but can be expressed using a CompoundPeriod. In fact, a CompoundPeriod is automatically generated by addition of different period types, e.g. Year(1) + Day(1) produces a CompoundPeriod result.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Instant","page":"Dates","title":"Dates.Instant","category":"type","text":"Instant\n\nInstant types represent integer-based, machine representations of time as continuous timelines starting from an epoch.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.UTInstant","page":"Dates","title":"Dates.UTInstant","category":"type","text":"UTInstant{T}\n\nThe UTInstant represents a machine timeline based on UT time (1 day = one revolution of the earth). The T is a Period parameter that indicates the resolution or precision of the instant.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.TimeType","page":"Dates","title":"Dates.TimeType","category":"type","text":"TimeType\n\nTimeType types wrap Instant machine instances to provide human representations of the machine instant. Time, DateTime and Date are subtypes of TimeType.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime","page":"Dates","title":"Dates.DateTime","category":"type","text":"DateTime\n\nDateTime represents a point in time according to the proleptic Gregorian calendar. The finest resolution of the time is millisecond (i.e., microseconds or nanoseconds cannot be represented by this type). The type supports fixed-point arithmetic, and thus is prone to underflowing (and overflowing). A notable consequence is rounding when adding a Microsecond or a Nanosecond:\n\njulia> dt = DateTime(2023, 8, 19, 17, 45, 32, 900)\n2023-08-19T17:45:32.900\n\njulia> dt + Millisecond(1)\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1000) # 1000us == 1ms\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(999) # 999us rounded to 1000us\n2023-08-19T17:45:32.901\n\njulia> dt + Microsecond(1499) # 1499 rounded to 1000us\n2023-08-19T17:45:32.901\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date","page":"Dates","title":"Dates.Date","category":"type","text":"Date\n\nDate wraps a UTInstant{Day} and interprets it according to the proleptic Gregorian calendar.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time","page":"Dates","title":"Dates.Time","category":"type","text":"Time\n\nTime wraps a Nanosecond and represents a specific moment in a 24-hour day.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.TimeZone","page":"Dates","title":"Dates.TimeZone","category":"type","text":"TimeZone\n\nGeographic zone generally based on longitude determining what the time is at a certain location. Some time zones observe daylight savings (eg EST -> EDT). For implementations and more support, see the TimeZones.jl package\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.UTC","page":"Dates","title":"Dates.UTC","category":"type","text":"UTC\n\nUTC, or Coordinated Universal Time, is the TimeZone from which all others are measured. It is associated with the time at 0° longitude. It is not adjusted for daylight savings.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-NTuple{7, Int64}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(y, [m, d, h, mi, s, ms])::DateTime\n\nConstruct a DateTime type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{Period}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(periods::Period...)::DateTime\n\nConstruct a DateTime type by Period type parts. Arguments may be in any order. DateTime parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{Function, Vararg{Any}}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(f::Function, y[, m, d, h, mi, s]; step=Day(1), limit=10000)::DateTime\n\nCreate a DateTime through the adjuster API. The starting point will be constructed from the provided y, m, d... arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied).\n\nExamples\n\njulia> DateTime(dt -> second(dt) == 40, 2010, 10, 20, 10; step = Second(1))\n2010-10-20T10:00:40\n\njulia> DateTime(dt -> hour(dt) == 20, 2010, 10, 20, 10; step = Hour(1), limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{TimeType}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(dt::Date)\n\nConvert a Date to a DateTime. The hour, minute, second, and millisecond parts of the new DateTime are assumed to be zero.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(dt::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the format string (see DateFormat  for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> DateTime(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01T00:00:00\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [DateTime(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{DateTime}:\n 2020-01-01T00:00:00\n 2020-01-02T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.format-Tuple{TimeType, AbstractString}","page":"Dates","title":"Dates.format","category":"method","text":"format(dt::TimeType, format::AbstractString; locale=\"english\")::AbstractString\n\nConstruct a string by using a TimeType object and applying the provided format. The following character codes can be used to construct the format string:\n\nCode Examples Comment\ny 6 Numeric year with a fixed width\nY 1996 Numeric year with a minimum width\nm 1, 12 Numeric month with a minimum width\nu Jan Month name shortened to 3-chars according to the locale\nU January Full month name according to the locale keyword\nd 1, 31 Day of the month with a minimum width\nH 0, 23 Hour (24-hour clock) with a minimum width\nM 0, 59 Minute with a minimum width\nS 0, 59 Second with a minimum width\ns 000, 500 Millisecond with a minimum width of 3\ne Mon, Tue Abbreviated days of the week\nE Monday Full day of week name\n\nThe number of sequential code characters indicate the width of the code. A format of yyyy-mm specifies that the code y should have a width of four while m a width of two. Codes that yield numeric digits have an associated mode: fixed-width or minimum-width. The fixed-width mode left-pads the value with zeros when it is shorter than the specified width and truncates the value when longer. Minimum-width mode works the same as fixed-width except that it does not truncate values longer than the width.\n\nWhen creating a format you can use any non-code characters as a separator. For example to generate the string \"1996-01-15T00:00:00\" you could use format: \"yyyy-mm-ddTHH:MM:SS\". Note that if you need to use a code character as a literal you can use the escape character backslash. The string \"1996y01m\" can be produced with the format raw\"yyyy\\ymm\\m\".\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateFormat","page":"Dates","title":"Dates.DateFormat","category":"type","text":"DateFormat(format::AbstractString, locale=\"english\")\n\nConstruct a date formatting object that can be used for parsing date strings or formatting a date object as a string. The following character codes can be used to construct the format string:\n\nCode Matches Comment\nY 1996, 96 Returns year of 1996, 0096\ny 1996, 96 Same as Y on parse but discards excess digits on format\nm 1, 01 Matches 1 or 2-digit months\nu Jan Matches abbreviated months according to the locale keyword\nU January Matches full month names according to the locale keyword\nd 1, 01 Matches 1 or 2-digit days\nH 00 Matches hours (24-hour clock)\nI 00 For outputting hours with 12-hour clock\nM 00 Matches minutes\nS 00 Matches seconds\ns .500 Matches milliseconds\ne Mon, Tues Matches abbreviated days of the week\nE Monday Matches full name days of the week\np AM Matches AM/PM (case-insensitive)\nyyyymmdd 19960101 Matches fixed-width year, month, and day\n\nCharacters not listed above are normally treated as delimiters between date and time slots. For example a dt string of \"1996-01-15T00:00:00.0\" would have a format string like \"y-m-dTH:M:S.s\". If you need to use a code character as a delimiter you can escape it using backslash. The date \"1995y01m\" would have the format \"y\\ym\\m\".\n\nNote that 12:00AM corresponds 00:00 (midnight), and 12:00PM corresponds to 12:00 (noon). When parsing a time with a p specifier, any hour (either H or I) is interpreted as as a 12-hour clock, so the I code is mainly useful for output.\n\nCreating a DateFormat object is expensive. Whenever possible, create it once and use it many times or try the dateformat\"\" string macro. Using this macro creates the DateFormat object once at macro expansion time and reuses it later. There are also several pre-defined formatters, listed later.\n\nSee DateTime and format for how to use a DateFormat object to parse and write Date strings respectively.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.@dateformat_str","page":"Dates","title":"Dates.@dateformat_str","category":"macro","text":"dateformat\"Y-m-d H:M:S\"\n\nCreate a DateFormat object. Similar to DateFormat(\"Y-m-d H:M:S\") but creates the DateFormat object once during macro expansion.\n\nSee DateFormat for details about format specifiers.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.DateTime-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.DateTime","category":"method","text":"DateTime(dt::AbstractString, df::DateFormat=ISODateTimeFormat)\n\nConstruct a DateTime by parsing the dt date time string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\\THH:MM:SS.s\" if omitted.\n\nSimilar to DateTime(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date time strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Int64, Int64, Int64}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(y, [m, d])::Date\n\nConstruct a Date type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Period}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(period::Period...)::Date\n\nConstruct a Date type by Period type parts. Arguments may be in any order. Date parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{Function, Any, Any, Any}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(f::Function, y[, m, d]; step=Day(1), limit=10000)::Date\n\nCreate a Date through the adjuster API. The starting point will be constructed from the provided y, m, d arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (given that f::Function is never satisfied).\n\nExamples\n\njulia> Date(date -> week(date) == 20, 2010, 01, 01)\n2010-05-17\n\njulia> Date(date -> year(date) == 2010, 2000, 01, 01)\n2010-01-01\n\njulia> Date(date -> month(date) == 10, 2000, 01, 01; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{TimeType}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(dt::DateTime)\n\nConvert a DateTime to a Date. The hour, minute, second, and millisecond parts of the DateTime are truncated, so only the year, month and day parts are used in construction.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(d::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a Date by parsing the d date string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Date(\"2020-01-01\", \"yyyy-mm-dd\")\n2020-01-01\n\njulia> a = (\"2020-01-01\", \"2020-01-02\");\n\njulia> [Date(d, dateformat\"yyyy-mm-dd\") for d ∈ a] # preferred\n2-element Vector{Date}:\n 2020-01-01\n 2020-01-02\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Date-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.Date","category":"method","text":"Date(d::AbstractString, df::DateFormat=ISODateFormat)\n\nConstruct a Date by parsing the d date string following the pattern given in the DateFormat object, or dateformat\"yyyy-mm-dd\" if omitted.\n\nSimilar to Date(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted date strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-NTuple{5, Int64}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(h, [mi, s, ms, us, ns])::Time\n\nConstruct a Time type by parts. Arguments must be convertible to Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{TimePeriod}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(period::TimePeriod...)::Time\n\nConstruct a Time type by Period type parts. Arguments may be in any order. Time parts not provided will default to the value of Dates.default(period).\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{Function, Vararg{Any}}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(f::Function, h, mi=0; step::Period=Second(1), limit::Int=10000)\nTime(f::Function, h, mi, s; step::Period=Millisecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms; step::Period=Microsecond(1), limit::Int=10000)\nTime(f::Function, h, mi, s, ms, us; step::Period=Nanosecond(1), limit::Int=10000)\n\nCreate a Time through the adjuster API. The starting point will be constructed from the provided h, mi, s, ms, us arguments, and will be adjusted until f::Function returns true. The step size in adjusting can be provided manually through the step keyword. limit provides a limit to the max number of iterations the adjustment API will pursue before throwing an error (in the case that f::Function is never satisfied). Note that the default step will adjust to allow for greater precision for the given arguments; i.e. if hour, minute, and second arguments are provided, the default step will be Millisecond(1) instead of Second(1).\n\nExamples\n\njulia> Time(t -> minute(t) == 30, 20)\n20:30:00\n\njulia> Time(t -> minute(t) == 0, 20)\n20:00:00\n\njulia> Time(t -> hour(t) == 10, 3; limit = 5)\nERROR: ArgumentError: Adjustment limit reached: 5 iterations\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{DateTime}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(dt::DateTime)\n\nConvert a DateTime to a Time. The hour, minute, second, and millisecond parts of the DateTime are used to create the new Time. Microsecond and nanoseconds are zero by default.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, AbstractString}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(t::AbstractString, format::AbstractString; locale=\"english\")\n\nConstruct a Time by parsing the t time string following the pattern given in the format string (see DateFormat for syntax).\n\nnote: Note\nThis method creates a DateFormat object each time it is called. It is recommended that you create a DateFormat object instead and use that as the second argument to avoid performance loss when using the same format repeatedly.\n\nExamples\n\njulia> Time(\"12:34pm\", \"HH:MMp\")\n12:34:00\n\njulia> a = (\"12:34pm\", \"2:34am\");\n\njulia> [Time(d, dateformat\"HH:MMp\") for d ∈ a] # preferred\n2-element Vector{Time}:\n 12:34:00\n 02:34:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Time-Tuple{AbstractString, DateFormat}","page":"Dates","title":"Dates.Time","category":"method","text":"Time(t::AbstractString, df::DateFormat=ISOTimeFormat)\n\nConstruct a Time by parsing the t date time string following the pattern given in the DateFormat object, or dateformat\"HH:MM:SS.s\" if omitted.\n\nSimilar to Time(::AbstractString, ::AbstractString) but more efficient when repeatedly parsing similarly formatted time strings with a pre-created DateFormat object.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.now-Tuple{}","page":"Dates","title":"Dates.now","category":"method","text":"now()::DateTime\n\nReturn a DateTime corresponding to the user's system time including the system timezone locale.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.now-Tuple{Type{UTC}}","page":"Dates","title":"Dates.now","category":"method","text":"now(::Type{UTC})::DateTime\n\nReturn a DateTime corresponding to the user's system time as UTC/GMT. For other time zones, see the TimeZones.jl package.\n\nExamples\n\njulia> now(UTC)\n2023-01-04T10:52:24.864\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.eps-Tuple{Union{Type{Date}, Type{DateTime}, Type{Time}, TimeType}}","page":"Dates","title":"Base.eps","category":"method","text":"eps(::Type{DateTime})::Millisecond\neps(::Type{Date})::Day\neps(::Type{Time})::Nanosecond\neps(::TimeType)::Period\n\nReturn the smallest unit value supported by the TimeType.\n\nExamples\n\njulia> eps(DateTime)\n1 millisecond\n\njulia> eps(Date)\n1 day\n\njulia> eps(Time)\n1 nanosecond\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.year","page":"Dates","title":"Dates.year","category":"function","text":"year(dt::TimeType)::Int64\n\nThe year of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.month","page":"Dates","title":"Dates.month","category":"function","text":"month(dt::TimeType)::Int64\n\nThe month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.week","page":"Dates","title":"Dates.week","category":"function","text":"week(dt::TimeType)::Int64\n\nReturn the ISO week date of a Date or DateTime as an Int64. Note that the first week of a year is the week that contains the first Thursday of the year, which can result in dates prior to January 4th being in the last week of the previous year. For example, week(Date(2005, 1, 1)) is the 53rd week of 2004.\n\nExamples\n\njulia> week(Date(1989, 6, 22))\n25\n\njulia> week(Date(2005, 1, 1))\n53\n\njulia> week(Date(2004, 12, 31))\n53\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.day","page":"Dates","title":"Dates.day","category":"function","text":"day(dt::TimeType)::Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.hour","page":"Dates","title":"Dates.hour","category":"function","text":"hour(dt::DateTime)::Int64\n\nThe hour of day of a DateTime as an Int64.\n\n\n\n\n\nhour(t::Time)::Int64\n\nThe hour of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.minute","page":"Dates","title":"Dates.minute","category":"function","text":"minute(dt::DateTime)::Int64\n\nThe minute of a DateTime as an Int64.\n\n\n\n\n\nminute(t::Time)::Int64\n\nThe minute of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.second","page":"Dates","title":"Dates.second","category":"function","text":"second(dt::DateTime)::Int64\n\nThe second of a DateTime as an Int64.\n\n\n\n\n\nsecond(t::Time)::Int64\n\nThe second of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.millisecond","page":"Dates","title":"Dates.millisecond","category":"function","text":"millisecond(dt::DateTime)::Int64\n\nThe millisecond of a DateTime as an Int64.\n\n\n\n\n\nmillisecond(t::Time)::Int64\n\nThe millisecond of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.microsecond","page":"Dates","title":"Dates.microsecond","category":"function","text":"microsecond(t::Time)::Int64\n\nThe microsecond of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.nanosecond","page":"Dates","title":"Dates.nanosecond","category":"function","text":"nanosecond(t::Time)::Int64\n\nThe nanosecond of a Time as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Year-Tuple{TimeType}","page":"Dates","title":"Dates.Year","category":"method","text":"Year(v)\n\nConstruct a Year object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Month-Tuple{TimeType}","page":"Dates","title":"Dates.Month","category":"method","text":"Month(v)\n\nConstruct a Month object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Week-Tuple{TimeType}","page":"Dates","title":"Dates.Week","category":"method","text":"Week(v)\n\nConstruct a Week object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Day-Tuple{TimeType}","page":"Dates","title":"Dates.Day","category":"method","text":"Day(v)\n\nConstruct a Day object with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Hour-Tuple{DateTime}","page":"Dates","title":"Dates.Hour","category":"method","text":"Hour(dt::DateTime)\n\nThe hour part of a DateTime as a Hour.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Minute-Tuple{DateTime}","page":"Dates","title":"Dates.Minute","category":"method","text":"Minute(dt::DateTime)\n\nThe minute part of a DateTime as a Minute.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Second-Tuple{DateTime}","page":"Dates","title":"Dates.Second","category":"method","text":"Second(dt::DateTime)\n\nThe second part of a DateTime as a Second.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Millisecond-Tuple{DateTime}","page":"Dates","title":"Dates.Millisecond","category":"method","text":"Millisecond(dt::DateTime)\n\nThe millisecond part of a DateTime as a Millisecond.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Microsecond-Tuple{Time}","page":"Dates","title":"Dates.Microsecond","category":"method","text":"Microsecond(dt::Time)\n\nThe microsecond part of a Time as a Microsecond.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Nanosecond-Tuple{Time}","page":"Dates","title":"Dates.Nanosecond","category":"method","text":"Nanosecond(dt::Time)\n\nThe nanosecond part of a Time as a Nanosecond.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.yearmonth","page":"Dates","title":"Dates.yearmonth","category":"function","text":"yearmonth(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the year and month parts of a Date or DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.monthday","page":"Dates","title":"Dates.monthday","category":"function","text":"monthday(dt::TimeType) -> (Int64, Int64)\n\nSimultaneously return the month and day parts of a Date or DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.yearmonthday","page":"Dates","title":"Dates.yearmonthday","category":"function","text":"yearmonthday(dt::TimeType) -> (Int64, Int64, Int64)\n\nSimultaneously return the year, month and day parts of a Date or DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayname","page":"Dates","title":"Dates.dayname","category":"function","text":"dayname(dt::TimeType; locale=\"english\")::String\ndayname(day::Integer; locale=\"english\")::String\n\nReturn the full day name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayname(Date(\"2000-01-01\"))\n\"Saturday\"\n\njulia> dayname(4)\n\"Thursday\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayabbr","page":"Dates","title":"Dates.dayabbr","category":"function","text":"dayabbr(dt::TimeType; locale=\"english\")::String\ndayabbr(day::Integer; locale=\"english\")::String\n\nReturn the abbreviated name corresponding to the day of the week of the Date or DateTime in the given locale. Also accepts Integer.\n\nExamples\n\njulia> dayabbr(Date(\"2000-01-01\"))\n\"Sat\"\n\njulia> dayabbr(3)\n\"Wed\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayofweek","page":"Dates","title":"Dates.dayofweek","category":"function","text":"dayofweek(dt::TimeType)::Int64\n\nReturn the day of the week as an Int64 with 1 = Monday, 2 = Tuesday, etc..\n\nExamples\n\njulia> dayofweek(Date(\"2000-01-01\"))\n6\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayofmonth","page":"Dates","title":"Dates.dayofmonth","category":"function","text":"dayofmonth(dt::TimeType)::Int64\n\nThe day of month of a Date or DateTime as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayofweekofmonth","page":"Dates","title":"Dates.dayofweekofmonth","category":"function","text":"dayofweekofmonth(dt::TimeType)::Int\n\nFor the day of week of dt, return which number it is in dt's month. So if the day of the week of dt is Monday, then 1 = First Monday of the month, 2 = Second Monday of the month, etc. In the range 1:5.\n\nExamples\n\njulia> dayofweekofmonth(Date(\"2000-02-01\"))\n1\n\njulia> dayofweekofmonth(Date(\"2000-02-08\"))\n2\n\njulia> dayofweekofmonth(Date(\"2000-02-15\"))\n3\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.daysofweekinmonth","page":"Dates","title":"Dates.daysofweekinmonth","category":"function","text":"daysofweekinmonth(dt::TimeType)::Int\n\nFor the day of week of dt, return the total number of that day of the week in dt's month. Returns 4 or 5. Useful in temporal expressions for specifying the last day of a week in a month by including dayofweekofmonth(dt) == daysofweekinmonth(dt) in the adjuster function.\n\nExamples\n\njulia> daysofweekinmonth(Date(\"2005-01-01\"))\n5\n\njulia> daysofweekinmonth(Date(\"2005-01-04\"))\n4\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.monthname","page":"Dates","title":"Dates.monthname","category":"function","text":"monthname(dt::TimeType; locale=\"english\")::String\nmonthname(month::Integer, locale=\"english\")::String\n\nReturn the full name of the month of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthname(Date(\"2005-01-04\"))\n\"January\"\n\njulia> monthname(2)\n\"February\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.monthabbr","page":"Dates","title":"Dates.monthabbr","category":"function","text":"monthabbr(dt::TimeType; locale=\"english\")::String\nmonthabbr(month::Integer, locale=\"english\")::String\n\nReturn the abbreviated month name of the Date or DateTime or Integer in the given locale.\n\nExamples\n\njulia> monthabbr(Date(\"2005-01-04\"))\n\"Jan\"\n\njulia> monthabbr(2)\n\"Feb\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.daysinmonth","page":"Dates","title":"Dates.daysinmonth","category":"function","text":"daysinmonth(dt::TimeType)::Int\n\nReturn the number of days in the month of dt. Value will be 28, 29, 30, or 31.\n\nExamples\n\njulia> daysinmonth(Date(\"2000-01\"))\n31\n\njulia> daysinmonth(Date(\"2001-02\"))\n28\n\njulia> daysinmonth(Date(\"2000-02\"))\n29\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.isleapyear","page":"Dates","title":"Dates.isleapyear","category":"function","text":"isleapyear(dt::TimeType)::Bool\n\nReturn true if the year of dt is a leap year.\n\nExamples\n\njulia> isleapyear(Date(\"2004\"))\ntrue\n\njulia> isleapyear(Date(\"2005\"))\nfalse\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayofyear","page":"Dates","title":"Dates.dayofyear","category":"function","text":"dayofyear(dt::TimeType)::Int\n\nReturn the day of the year for dt with January 1st being day 1.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.daysinyear","page":"Dates","title":"Dates.daysinyear","category":"function","text":"daysinyear(dt::TimeType)::Int\n\nReturn 366 if the year of dt is a leap year, otherwise return 365.\n\nExamples\n\njulia> daysinyear(1999)\n365\n\njulia> daysinyear(2000)\n366\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.quarterofyear","page":"Dates","title":"Dates.quarterofyear","category":"function","text":"quarterofyear(dt::TimeType)::Int\n\nReturn the quarter that dt resides in. Range of value is 1:4.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.dayofquarter","page":"Dates","title":"Dates.dayofquarter","category":"function","text":"dayofquarter(dt::TimeType)::Int\n\nReturn the day of the current quarter of dt. Range of value is 1:92.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.trunc-Tuple{TimeType, Type{Period}}","page":"Dates","title":"Base.trunc","category":"method","text":"trunc(dt::TimeType, ::Type{Period})::TimeType\n\nTruncates the value of dt according to the provided Period type.\n\nExamples\n\njulia> trunc(DateTime(\"1996-01-01T12:30:00\"), Day)\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.firstdayofweek","page":"Dates","title":"Dates.firstdayofweek","category":"function","text":"firstdayofweek(dt::TimeType)::TimeType\n\nAdjusts dt to the Monday of its week.\n\nExamples\n\njulia> firstdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.lastdayofweek","page":"Dates","title":"Dates.lastdayofweek","category":"function","text":"lastdayofweek(dt::TimeType)::TimeType\n\nAdjusts dt to the Sunday of its week.\n\nExamples\n\njulia> lastdayofweek(DateTime(\"1996-01-05T12:30:00\"))\n1996-01-07T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.firstdayofmonth","page":"Dates","title":"Dates.firstdayofmonth","category":"function","text":"firstdayofmonth(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its month.\n\nExamples\n\njulia> firstdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-01T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.lastdayofmonth","page":"Dates","title":"Dates.lastdayofmonth","category":"function","text":"lastdayofmonth(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its month.\n\nExamples\n\njulia> lastdayofmonth(DateTime(\"1996-05-20\"))\n1996-05-31T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.firstdayofyear","page":"Dates","title":"Dates.firstdayofyear","category":"function","text":"firstdayofyear(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its year.\n\nExamples\n\njulia> firstdayofyear(DateTime(\"1996-05-20\"))\n1996-01-01T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.lastdayofyear","page":"Dates","title":"Dates.lastdayofyear","category":"function","text":"lastdayofyear(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its year.\n\nExamples\n\njulia> lastdayofyear(DateTime(\"1996-05-20\"))\n1996-12-31T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.firstdayofquarter","page":"Dates","title":"Dates.firstdayofquarter","category":"function","text":"firstdayofquarter(dt::TimeType)::TimeType\n\nAdjusts dt to the first day of its quarter.\n\nExamples\n\njulia> firstdayofquarter(DateTime(\"1996-05-20\"))\n1996-04-01T00:00:00\n\njulia> firstdayofquarter(DateTime(\"1996-08-20\"))\n1996-07-01T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.lastdayofquarter","page":"Dates","title":"Dates.lastdayofquarter","category":"function","text":"lastdayofquarter(dt::TimeType)::TimeType\n\nAdjusts dt to the last day of its quarter.\n\nExamples\n\njulia> lastdayofquarter(DateTime(\"1996-05-20\"))\n1996-06-30T00:00:00\n\njulia> lastdayofquarter(DateTime(\"1996-08-20\"))\n1996-09-30T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.tonext-Tuple{TimeType, Int32}","page":"Dates","title":"Dates.tonext","category":"method","text":"tonext(dt::TimeType, dow::Int; same::Bool=false)::TimeType\n\nAdjusts dt to the next day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the next dow, allowing for no adjustment to occur.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.toprev-Tuple{TimeType, Int32}","page":"Dates","title":"Dates.toprev","category":"method","text":"toprev(dt::TimeType, dow::Int; same::Bool=false)::TimeType\n\nAdjusts dt to the previous day of week corresponding to dow with 1 = Monday, 2 = Tuesday, etc. Setting same=true allows the current dt to be considered as the previous dow, allowing for no adjustment to occur.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.tofirst","page":"Dates","title":"Dates.tofirst","category":"function","text":"tofirst(dt::TimeType, dow::Int; of=Month)::TimeType\n\nAdjusts dt to the first dow of its month. Alternatively, of=Year will adjust to the first dow of the year.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.tolast","page":"Dates","title":"Dates.tolast","category":"function","text":"tolast(dt::TimeType, dow::Int; of=Month)::TimeType\n\nAdjusts dt to the last dow of its month. Alternatively, of=Year will adjust to the last dow of the year.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.tonext-Tuple{Function, TimeType}","page":"Dates","title":"Dates.tonext","category":"method","text":"tonext(func::Function, dt::TimeType; step=Day(1), limit=10000, same=false)::TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.toprev-Tuple{Function, TimeType}","page":"Dates","title":"Dates.toprev","category":"method","text":"toprev(func::Function, dt::TimeType; step=Day(-1), limit=10000, same=false)::TimeType\n\nAdjusts dt by iterating at most limit iterations by step increments until func returns true. func must take a single TimeType argument and return a Bool. same allows dt to be considered in satisfying func.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.Period-Tuple{Any}","page":"Dates","title":"Dates.Period","category":"method","text":"Year(v)\nQuarter(v)\nMonth(v)\nWeek(v)\nDay(v)\nHour(v)\nMinute(v)\nSecond(v)\nMillisecond(v)\nMicrosecond(v)\nNanosecond(v)\n\nConstruct a Period type with the given v value. Input must be losslessly convertible to an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.CompoundPeriod-Tuple{Vector{<:Period}}","page":"Dates","title":"Dates.CompoundPeriod","category":"method","text":"CompoundPeriod(periods)\n\nConstruct a CompoundPeriod from a Vector of Periods. All Periods of the same type will be added together.\n\nExamples\n\njulia> Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13))\n25 hours\n\njulia> Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1))\n-1 hour, 1 minute\n\njulia> Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2))\n1 month, -2 weeks\n\njulia> Dates.CompoundPeriod(Dates.Minute(50000))\n50000 minutes\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.canonicalize","page":"Dates","title":"Dates.canonicalize","category":"function","text":"canonicalize(::CompoundPeriod)::CompoundPeriod\n\nReduces the CompoundPeriod into its canonical form by applying the following rules:\n\nAny Period large enough be partially representable by a coarser Period will be broken into multiple Periods (eg. Hour(30) becomes Day(1) + Hour(6))\nPeriods with opposite signs will be combined when possible (eg. Hour(1) - Day(1) becomes -Hour(23))\n\nExamples\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(12), Dates.Hour(13)))\n1 day, 1 hour\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Hour(-1), Dates.Minute(1)))\n-59 minutes\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Month(1), Dates.Week(-2)))\n1 month, -2 weeks\n\njulia> canonicalize(Dates.CompoundPeriod(Dates.Minute(50000)))\n4 weeks, 6 days, 17 hours, 20 minutes\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.value","page":"Dates","title":"Dates.value","category":"function","text":"Dates.value(x::Period)::Int64\n\nFor a given period, return the value associated with that period.  For example, value(Millisecond(10)) returns 10 as an integer.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.default","page":"Dates","title":"Dates.default","category":"function","text":"default(p::Period)::Period\n\nReturn a sensible \"default\" value for the input Period by returning T(1) for Year, Month, and Day, and T(0) for Hour, Minute, Second, and Millisecond.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.periods","page":"Dates","title":"Dates.periods","category":"function","text":"Dates.periods(::CompoundPeriod)::Vector{Period}\n\nReturn the Vector of Periods that comprise the given CompoundPeriod.\n\ncompat: Julia 1.7\nThis function requires Julia 1.7 or later.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.floor-Tuple{TimeType, Period}","page":"Dates","title":"Base.floor","category":"method","text":"floor(dt::TimeType, p::Period)::TimeType\n\nReturn the nearest Date or DateTime less than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: floor(dt, Dates.Hour) is a shortcut for floor(dt, Dates.Hour(1)).\n\njulia> floor(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> floor(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> floor(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-06T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.ceil-Tuple{TimeType, Period}","page":"Dates","title":"Base.ceil","category":"method","text":"ceil(dt::TimeType, p::Period)::TimeType\n\nReturn the nearest Date or DateTime greater than or equal to dt at resolution p.\n\nFor convenience, p may be a type instead of a value: ceil(dt, Dates.Hour) is a shortcut for ceil(dt, Dates.Hour(1)).\n\njulia> ceil(Date(1985, 8, 16), Month)\n1985-09-01\n\njulia> ceil(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:45:00\n\njulia> ceil(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.round-Tuple{TimeType, Period, RoundingMode{:NearestTiesUp}}","page":"Dates","title":"Base.round","category":"method","text":"round(dt::TimeType, p::Period, [r::RoundingMode]) -> TimeType\n\nReturn the Date or DateTime nearest to dt at resolution p. By default (RoundNearestTiesUp), ties (e.g., rounding 9:30 to the nearest hour) will be rounded up.\n\nFor convenience, p may be a type instead of a value: round(dt, Dates.Hour) is a shortcut for round(dt, Dates.Hour(1)).\n\njulia> round(Date(1985, 8, 16), Month)\n1985-08-01\n\njulia> round(DateTime(2013, 2, 13, 0, 31, 20), Minute(15))\n2013-02-13T00:30:00\n\njulia> round(DateTime(2016, 8, 6, 12, 0, 0), Day)\n2016-08-07T00:00:00\n\nValid rounding modes for round(::TimeType, ::Period, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.floor-Union{Tuple{T}, Tuple{Union{Day, Week, TimePeriod}, T}} where T<:Union{Day, Week, TimePeriod}","page":"Dates","title":"Base.floor","category":"method","text":"floor(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x down to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: floor(x, Dates.Hour) is a shortcut for floor(x, Dates.Hour(1)).\n\njulia> floor(Day(16), Week)\n2 weeks\n\njulia> floor(Minute(44), Minute(15))\n30 minutes\n\njulia> floor(Hour(36), Day)\n1 day\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.ceil-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}}","page":"Dates","title":"Base.ceil","category":"method","text":"ceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x up to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision.\n\nFor convenience, precision may be a type instead of a value: ceil(x, Dates.Hour) is a shortcut for ceil(x, Dates.Hour(1)).\n\njulia> ceil(Day(16), Week)\n3 weeks\n\njulia> ceil(Minute(44), Minute(15))\n45 minutes\n\njulia> ceil(Hour(36), Day)\n2 days\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Base.round-Tuple{Union{Day, Week, TimePeriod}, Union{Day, Week, TimePeriod}, RoundingMode{:NearestTiesUp}}","page":"Dates","title":"Base.round","category":"method","text":"round(x::Period, precision::T, [r::RoundingMode]) where T <: Union{TimePeriod, Week, Day} -> T\n\nRound x to the nearest multiple of precision. If x and precision are different subtypes of Period, the return value will have the same type as precision. By default (RoundNearestTiesUp), ties (e.g., rounding 90 minutes to the nearest hour) will be rounded up.\n\nFor convenience, precision may be a type instead of a value: round(x, Dates.Hour) is a shortcut for round(x, Dates.Hour(1)).\n\njulia> round(Day(16), Week)\n2 weeks\n\njulia> round(Minute(44), Minute(15))\n45 minutes\n\njulia> round(Hour(36), Day)\n2 days\n\nValid rounding modes for round(::Period, ::T, ::RoundingMode) are RoundNearestTiesUp (default), RoundDown (floor), and RoundUp (ceil).\n\nRounding to a precision of Months or Years is not supported, as these Periods are of inconsistent length.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.floorceil","page":"Dates","title":"Dates.floorceil","category":"function","text":"floorceil(dt::TimeType, p::Period) -> (TimeType, TimeType)\n\nSimultaneously return the floor and ceil of a Date or DateTime at resolution p. More efficient than calling both floor and ceil individually.\n\n\n\n\n\nfloorceil(x::Period, precision::T) where T <: Union{TimePeriod, Week, Day} -> (T, T)\n\nSimultaneously return the floor and ceil of Period at resolution p.  More efficient than calling both floor and ceil individually.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.epochdays2date","page":"Dates","title":"Dates.epochdays2date","category":"function","text":"epochdays2date(days)::Date\n\nTake the number of days since the rounding epoch (0000-01-01T00:00:00) and return the corresponding Date.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.epochms2datetime","page":"Dates","title":"Dates.epochms2datetime","category":"function","text":"epochms2datetime(milliseconds)::DateTime\n\nTake the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) and return the corresponding DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.date2epochdays","page":"Dates","title":"Dates.date2epochdays","category":"function","text":"date2epochdays(dt::Date)::Int64\n\nTake the given Date and return the number of days since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.datetime2epochms","page":"Dates","title":"Dates.datetime2epochms","category":"function","text":"datetime2epochms(dt::DateTime)::Int64\n\nTake the given DateTime and return the number of milliseconds since the rounding epoch (0000-01-01T00:00:00) as an Int64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.today","page":"Dates","title":"Dates.today","category":"function","text":"today()::Date\n\nReturn the date portion of now().\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.unix2datetime","page":"Dates","title":"Dates.unix2datetime","category":"function","text":"unix2datetime(x)::DateTime\n\nTake the number of seconds since unix epoch 1970-01-01T00:00:00 and convert to the corresponding DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.datetime2unix","page":"Dates","title":"Dates.datetime2unix","category":"function","text":"datetime2unix(dt::DateTime)::Float64\n\nTake the given DateTime and return the number of seconds since the unix epoch 1970-01-01T00:00:00 as a Float64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.julian2datetime","page":"Dates","title":"Dates.julian2datetime","category":"function","text":"julian2datetime(julian_days)::DateTime\n\nTake the number of Julian calendar days since epoch -4713-11-24T12:00:00 and return the corresponding DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.datetime2julian","page":"Dates","title":"Dates.datetime2julian","category":"function","text":"datetime2julian(dt::DateTime)::Float64\n\nTake the given DateTime and return the number of Julian calendar days since the julian epoch -4713-11-24T12:00:00 as a Float64.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.rata2datetime","page":"Dates","title":"Dates.rata2datetime","category":"function","text":"rata2datetime(days)::DateTime\n\nTake the number of Rata Die days since epoch 0000-12-31T00:00:00 and return the corresponding DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.datetime2rata","page":"Dates","title":"Dates.datetime2rata","category":"function","text":"datetime2rata(dt::TimeType)::Int64\n\nReturn the number of Rata Die days since epoch from the given Date or DateTime.\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.ISODateTimeFormat","page":"Dates","title":"Dates.ISODateTimeFormat","category":"constant","text":"Dates.ISODateTimeFormat\n\nDescribes the ISO8601 formatting for a date and time. This is the default value for Dates.format of a DateTime.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), ISODateTimeFormat)\n\"2018-08-08T12:00:43.001\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.ISODateFormat","page":"Dates","title":"Dates.ISODateFormat","category":"constant","text":"Dates.ISODateFormat\n\nDescribes the ISO8601 formatting for a date. This is the default value for Dates.format of a Date.\n\nExamples\n\njulia> Dates.format(Date(2018, 8, 8), ISODateFormat)\n\"2018-08-08\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.ISOTimeFormat","page":"Dates","title":"Dates.ISOTimeFormat","category":"constant","text":"Dates.ISOTimeFormat\n\nDescribes the ISO8601 formatting for a time. This is the default value for Dates.format of a Time.\n\nExamples\n\njulia> Dates.format(Time(12, 0, 43, 1), ISOTimeFormat)\n\"12:00:43.001\"\n\n\n\n\n\n"},{"location":"stdlib/Dates.html#Dates.RFC1123Format","page":"Dates","title":"Dates.RFC1123Format","category":"constant","text":"Dates.RFC1123Format\n\nDescribes the RFC1123 formatting for a date and time.\n\nExamples\n\njulia> Dates.format(DateTime(2018, 8, 8, 12, 0, 43, 1), RFC1123Format)\n\"Wed, 08 Aug 2018 12:00:43\"\n\n\n\n\n\n"},{"location":"devdocs/require.html#Module-loading","page":"Module loading","title":"Module loading","category":"section","text":"Base.require is responsible for loading modules and it also manages the precompilation cache. It is the implementation of the import statement."},{"location":"devdocs/require.html#Experimental-features","page":"Module loading","title":"Experimental features","category":"section","text":"The features below are experimental and not part of the stable Julia API. Before building upon them inform yourself about the current thinking and whether they might change soon."},{"location":"devdocs/require.html#Package-loading-callbacks","page":"Module loading","title":"Package loading callbacks","category":"section","text":"It is possible to listen to the packages loaded by Base.require, by registering a callback.\n\nloaded_packages = Base.PkgId[]\ncallback = (pkg::Base.PkgId) -> push!(loaded_packages, pkg)\npush!(Base.package_callbacks, callback)\n\nUsing this would look something like:\n\njulia> using Example\n\njulia> loaded_packages\n1-element Vector{Base.PkgId}:\n Example [7876af07-990d-54b4-ab0e-23690620f79a]"},{"location":"manual/memory-management.html#man-memory-management","page":"Memory Management and Garbage Collection","title":"Memory Management and Garbage Collection","category":"section","text":"Julia uses automatic memory management through its built-in garbage collector (GC). This section provides an overview of how Julia manages memory and how you can configure and optimize memory usage for your applications."},{"location":"manual/memory-management.html#man-gc-overview","page":"Memory Management and Garbage Collection","title":"Garbage Collection Overview","category":"section","text":"Julia features a garbage collector with the following characteristics:\n\nNon-moving: Objects are not relocated in memory during garbage collection\nGenerational: Younger objects are collected more frequently than older ones\nParallel and partially concurrent: The GC can use multiple threads and run concurrently with your program\nMostly precise: The GC accurately identifies object references for pure Julia code, and it provides conservative scanning APIs for users calling Julia from C\n\nThe garbage collector automatically reclaims memory used by objects that are no longer reachable from your program, freeing you from manual memory management in most cases."},{"location":"manual/memory-management.html#man-memory-architecture","page":"Memory Management and Garbage Collection","title":"Memory Architecture","category":"section","text":"Julia uses a two-tier allocation strategy:\n\nSmall objects (currently ≤ 2032 bytes but may change): Allocated using a fast per-thread pool allocator\nLarge objects : Allocated directly through the system's malloc\n\nThis hybrid approach optimizes for both allocation speed and memory efficiency, with the pool allocator providing fast allocation for the many small objects typical in Julia programs."},{"location":"manual/memory-management.html#man-system-memory","page":"Memory Management and Garbage Collection","title":"System Memory Requirements","category":"section","text":""},{"location":"manual/memory-management.html#Swap-Space","page":"Memory Management and Garbage Collection","title":"Swap Space","category":"section","text":"Julia's garbage collector is designed with the expectation that your system has adequate swap space configured. The GC uses heuristics that assume it can allocate memory beyond physical RAM when needed, relying on the operating system's virtual memory management.\n\nIf your system has limited or no swap space, you may experience out-of-memory errors during garbage collection. In such cases, you can use the --heap-size-hint option to limit Julia's memory usage."},{"location":"manual/memory-management.html#Memory-Hints","page":"Memory Management and Garbage Collection","title":"Memory Hints","category":"section","text":"You can provide a hint to Julia about the maximum amount of memory to use:\n\njulia --heap-size-hint=4G  # To set the hint to ~4GB\njulia --heap-size-hint=50% # or to 50% of physical memory\n\nThe --heap-size-hint option tells the garbage collector to trigger collection more aggressively when approaching the specified limit. This is particularly useful in:\n\nContainers with memory limits\nSystems without swap space\nShared systems where you want to limit Julia's memory footprint\n\nYou can also set this via the JULIA_HEAP_SIZE_HINT environment variable:\n\nexport JULIA_HEAP_SIZE_HINT=2G\njulia"},{"location":"manual/memory-management.html#man-gc-multithreading","page":"Memory Management and Garbage Collection","title":"Multithreaded Garbage Collection","category":"section","text":"Julia's garbage collector can leverage multiple threads to improve performance on multi-core systems."},{"location":"manual/memory-management.html#GC-Thread-Configuration","page":"Memory Management and Garbage Collection","title":"GC Thread Configuration","category":"section","text":"By default, Julia uses multiple threads for garbage collection:\n\nMark threads: Used during the mark phase to trace object references (default: 1, which is shared with the compute thread if there is only one, otherwise half the number of compute threads)\nSweep threads: Used for concurrent sweeping of freed memory (default: 0, disabled)\n\nYou can configure GC threading using:\n\njulia --gcthreads=4,1  # 4 mark threads, 1 sweep thread\njulia --gcthreads=8    # 8 mark threads, 0 sweep threads\n\nOr via environment variable:\n\nexport JULIA_NUM_GC_THREADS=4,1\njulia"},{"location":"manual/memory-management.html#Recommendations","page":"Memory Management and Garbage Collection","title":"Recommendations","category":"section","text":"For compute-intensive workloads:\n\nUse multiple mark threads (the default configuration is usually appropriate)\nConsider enabling concurrent sweeping with 1 sweep thread for allocation-heavy workloads\n\nFor memory-intensive workloads:\n\nEnable concurrent sweeping to reduce GC pauses\nMonitor GC time using @time and adjust thread counts accordingly"},{"location":"manual/memory-management.html#man-gc-monitoring","page":"Memory Management and Garbage Collection","title":"Monitoring and Debugging","category":"section","text":""},{"location":"manual/memory-management.html#Basic-Memory-Monitoring","page":"Memory Management and Garbage Collection","title":"Basic Memory Monitoring","category":"section","text":"Use the @time macro to see memory allocation and GC overhead:\n\njulia> @time some_computation()\n  2.123456 seconds (1.50 M allocations: 58.725 MiB, 17.17% gc time)"},{"location":"manual/memory-management.html#GC-Logging","page":"Memory Management and Garbage Collection","title":"GC Logging","category":"section","text":"Enable detailed GC logging to understand collection patterns:\n\njulia> GC.enable_logging(true)\njulia> # Run your code\njulia> GC.enable_logging(false)\n\nThis logs each garbage collection event with timing and memory statistics."},{"location":"manual/memory-management.html#Manual-GC-Control","page":"Memory Management and Garbage Collection","title":"Manual GC Control","category":"section","text":"While generally not recommended, you can manually trigger garbage collection:\n\nGC.gc()          # Force a garbage collection\nGC.enable(false) # Disable automatic GC (use with caution!)\nGC.enable(true)  # Re-enable automatic GC\n\nWarning: Disabling GC can lead to memory exhaustion. Only use this for specific performance measurements or debugging."},{"location":"manual/memory-management.html#man-gc-performance","page":"Memory Management and Garbage Collection","title":"Performance Considerations","category":"section","text":""},{"location":"manual/memory-management.html#Reducing-Allocations","page":"Memory Management and Garbage Collection","title":"Reducing Allocations","category":"section","text":"The best way to minimize GC impact is to reduce unnecessary allocations:\n\nUse in-place operations when possible (e.g., x .+= y instead of x = x + y)\nPre-allocate arrays and reuse them\nAvoid creating temporary objects in tight loops\nConsider using StaticArrays.jl for small, fixed-size arrays"},{"location":"manual/memory-management.html#Memory-Efficient-Patterns","page":"Memory Management and Garbage Collection","title":"Memory-Efficient Patterns","category":"section","text":"Avoid global variables that change type\nUse const for global constants"},{"location":"manual/memory-management.html#Profiling-Memory-Usage","page":"Memory Management and Garbage Collection","title":"Profiling Memory Usage","category":"section","text":"For detailed guidance on profiling memory allocations and identifying performance bottlenecks, see the Profiling section."},{"location":"manual/memory-management.html#man-gc-advanced","page":"Memory Management and Garbage Collection","title":"Advanced Configuration","category":"section","text":""},{"location":"manual/memory-management.html#Integration-with-System-Memory-Management","page":"Memory Management and Garbage Collection","title":"Integration with System Memory Management","category":"section","text":"Julia works best when:\n\nThe system has adequate swap space (recommended: 2x physical RAM)\nVirtual memory is properly configured\nOther processes leave sufficient memory available\nContainer memory limits are set appropriately with --heap-size-hint"},{"location":"manual/memory-management.html#man-gc-troubleshooting","page":"Memory Management and Garbage Collection","title":"Troubleshooting Memory Issues","category":"section","text":""},{"location":"manual/memory-management.html#High-GC-Overhead","page":"Memory Management and Garbage Collection","title":"High GC Overhead","category":"section","text":"If garbage collection is taking too much time:\n\nReduce allocation rate: Focus on algorithmic improvements\nAdjust GC threads: Experiment with different --gcthreads settings\nUse concurrent sweeping: Enable background sweeping with --gcthreads=N,1\nProfile memory patterns: Identify allocation hotspots and optimize them"},{"location":"manual/memory-management.html#Memory-Leaks","page":"Memory Management and Garbage Collection","title":"Memory Leaks","category":"section","text":"While Julia's GC prevents most memory leaks, issues can still occur:\n\nGlobal references: Avoid holding references to large objects in global variables\nClosures: Be careful with closures that capture large amounts of data\nC interop: Ensure proper cleanup when interfacing with C libraries\n\nFor more detailed information about Julia's garbage collector internals, see the Garbage Collection section in the Developer Documentation."},{"location":"devdocs/contributing/tests.html#Writing-tests","page":"Writing tests","title":"Writing tests","category":"section","text":"There are never enough tests. Track code coverage at Codecov, and help improve it.\n\nGo visit https://codecov.io/github/JuliaLang/julia.\nBrowse through the source files and find some untested functionality (highlighted in red) that you think you might be able to write a test for.\nWrite a test that exercises this functionality—you can add your test to one of the existing files, or start a new one, whichever seems most appropriate to you. If you're adding a new test file, make sure you include it in the list of tests in test/choosetests.jl. https://docs.julialang.org/en/v1/stdlib/Test/ may be helpful in explaining how the testing infrastructure works.\nRun make test-all to rebuild Julia and run your new test(s). If you had to fix a bug or add functionality in base, this will ensure that your test passes and that you have not introduced extraneous whitespace.\nSubmit the test as a pull request (PR).\n\nCode for the buildbot configuration is maintained at: https://github.com/staticfloat/julia-buildbot\nYou can see the current buildbot setup at: https://build.julialang.org/builders\nIssue 9493 and issue 11885 have more detailed discussion on code coverage.\n\nCode coverage shows functionality that still needs \"proof of concept\" tests. These are important, as are tests for tricky edge cases, such as converting between integer types when the number to convert is near the maximum of the range of one of the integer types. Even if a function already has some coverage on Codecov, it may still benefit from tests for edge cases."},{"location":"manual/constructors.html#man-constructors","page":"Constructors","title":"Constructors","category":"section","text":"Constructors [1] are functions that create new objects – specifically, instances of Composite Types. In Julia, type objects also serve as constructor functions: they create new instances of themselves when applied to an argument tuple as a function. This much was already mentioned briefly when composite types were introduced. For example:\n\njulia> struct Foo\n           bar\n           baz\n       end\n\njulia> foo = Foo(1, 2)\nFoo(1, 2)\n\njulia> foo.bar\n1\n\njulia> foo.baz\n2\n\nFor many types, forming new objects by binding their field values together is all that is ever needed to create instances. However, in some cases more functionality is required when creating composite objects. Sometimes invariants must be enforced, either by checking arguments or by transforming them. Recursive data structures, especially those that may be self-referential, often cannot be constructed cleanly without first being created in an incomplete state and then altered programmatically to be made whole, as a separate step from object creation. Sometimes, it's just convenient to be able to construct objects with fewer or different types of parameters than they have fields. Julia's system for object construction addresses all of these cases and more.\n\n[1]: Nomenclature: while the term \"constructor\" generally refers to the entire function which constructs objects of a type, it is common to abuse terminology slightly and refer to specific constructor methods as \"constructors\". In such situations, it is generally clear from the context that the term is used to mean \"constructor method\" rather than \"constructor function\", especially as it is often used in the sense of singling out a particular method of the constructor from all of the others."},{"location":"manual/constructors.html#man-outer-constructor-methods","page":"Constructors","title":"Outer Constructor Methods","category":"section","text":"A constructor is just like any other function in Julia in that its overall behavior is defined by the combined behavior of its methods. Accordingly, you can add functionality to a constructor by simply defining new methods. For example, let's say you want to add a constructor method for Foo objects that takes only one argument and uses the given value for both the bar and baz fields. This is simple:\n\njulia> Foo(x) = Foo(x,x)\nFoo\n\njulia> Foo(1)\nFoo(1, 1)\n\nYou could also add a zero-argument Foo constructor method that supplies default values for both of the bar and baz fields:\n\njulia> Foo() = Foo(0)\nFoo\n\njulia> Foo()\nFoo(0, 0)\n\nHere the zero-argument constructor method calls the single-argument constructor method, which in turn calls the automatically provided two-argument constructor method. For reasons that will become clear very shortly, additional constructor methods declared as normal methods like this are called outer constructor methods. Outer constructor methods can only ever create a new instance by calling another constructor method, such as the automatically provided default ones."},{"location":"manual/constructors.html#man-inner-constructor-methods","page":"Constructors","title":"Inner Constructor Methods","category":"section","text":"While outer constructor methods succeed in addressing the problem of providing additional convenience methods for constructing objects, they fail to address the other two use cases mentioned in the introduction of this chapter: enforcing invariants, and allowing construction of self-referential objects. For these problems, one needs inner constructor methods. An inner constructor method is like an outer constructor method, except for two differences:\n\nIt is declared inside the block of a type declaration, rather than outside of it like normal methods.\nIt has access to a special locally existent function called new that creates objects of the block's type.\n\nFor example, suppose one wants to declare a type that holds a pair of real numbers, subject to the constraint that the first number is not greater than the second one. One could declare it like this:\n\njulia> struct OrderedPair\n           x::Real\n           y::Real\n           OrderedPair(x,y) = x > y ? error(\"out of order\") : new(x,y)\n       end\n\nNow OrderedPair objects can only be constructed such that x <= y:\n\njulia> OrderedPair(1, 2)\nOrderedPair(1, 2)\n\njulia> OrderedPair(2,1)\nERROR: out of order\nStacktrace:\n [1] error at ./error.jl:33 [inlined]\n [2] OrderedPair(::Int64, ::Int64) at ./none:4\n [3] top-level scope\n\nIf the type were declared mutable, you could reach in and directly change the field values to violate this invariant. Of course, messing around with an object's internals uninvited is bad practice. You (or someone else) can also provide additional outer constructor methods at any later point, but once a type is declared, there is no way to add more inner constructor methods. Since outer constructor methods can only create objects by calling other constructor methods, ultimately, some inner constructor must be called to create an object. This guarantees that all objects of the declared type must come into existence by a call to one of the inner constructor methods provided with the type, thereby giving some degree of enforcement of a type's invariants.\n\nIf any inner constructor method is defined, no default constructor method is provided: it is presumed that you have supplied yourself with all the inner constructors you need. The default constructor is equivalent to writing your own inner constructor method that takes all of the object's fields as parameters (constrained to be of the correct type, if the corresponding field has a type), and passes them to new, returning the resulting object:\n\njulia> struct Foo\n           bar\n           baz\n           Foo(bar,baz) = new(bar,baz)\n       end\n\n\nThis declaration has the same effect as the earlier definition of the Foo type without an explicit inner constructor method. The following two types are equivalent – one with a default constructor, the other with an explicit constructor:\n\njulia> struct T1\n           x::Int64\n       end\n\njulia> struct T2\n           x::Int64\n           T2(x) = new(x)\n       end\n\njulia> T1(1)\nT1(1)\n\njulia> T2(1)\nT2(1)\n\njulia> T1(1.0)\nT1(1)\n\njulia> T2(1.0)\nT2(1)\n\nIt is good practice to provide as few inner constructor methods as possible: only those taking all arguments explicitly and enforcing essential error checking and transformation. Additional convenience constructor methods, supplying default values or auxiliary transformations, should be provided as outer constructors that call the inner constructors to do the heavy lifting. This separation is typically quite natural."},{"location":"manual/constructors.html#Incomplete-Initialization","page":"Constructors","title":"Incomplete Initialization","category":"section","text":"The final problem which has still not been addressed is construction of self-referential objects, or more generally, recursive data structures. Since the fundamental difficulty may not be immediately obvious, let us briefly explain it. Consider the following recursive type declaration:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n       end\n\n\nThis type may appear innocuous enough, until one considers how to construct an instance of it. If a is an instance of SelfReferential, then a second instance can be created by the call:\n\njulia> b = SelfReferential(a)\n\nBut how does one construct the first instance when no instance exists to provide as a valid value for its obj field? The only solution is to allow creating an incompletely initialized instance of SelfReferential with an unassigned obj field, and using that incomplete instance as a valid value for the obj field of another instance, such as, for example, itself.\n\nTo allow for the creation of incompletely initialized objects, Julia allows the new function to be called with fewer than the number of fields that the type has, returning an object with the unspecified fields uninitialized. The inner constructor method can then use the incomplete object, finishing its initialization before returning it. Here, for example, is another attempt at defining the SelfReferential type, this time using a zero-argument inner constructor returning instances having obj fields pointing to themselves:\n\njulia> mutable struct SelfReferential\n           obj::SelfReferential\n           SelfReferential() = (x = new(); x.obj = x)\n       end\n\n\nWe can verify that this constructor works and constructs objects that are, in fact, self-referential:\n\njulia> x = SelfReferential();\n\njulia> x === x\ntrue\n\njulia> x === x.obj\ntrue\n\njulia> x === x.obj.obj\ntrue\n\nAlthough it is generally a good idea to return a fully initialized object from an inner constructor, it is possible to return incompletely initialized objects:\n\njulia> mutable struct Incomplete\n           data\n           Incomplete() = new()\n       end\n\njulia> z = Incomplete();\n\nWhile you are allowed to create objects with uninitialized fields, any access to an uninitialized reference is an immediate error:\n\njulia> z.data\nERROR: UndefRefError: access to undefined reference\n\nThis avoids the need to continually check for null values. However, not all object fields are references. Julia considers some types to be \"plain data\", meaning all of their data is self-contained and does not reference other objects. The plain data types consist of primitive types (e.g. Int) and immutable structs of other plain data types (see also: isbits, isbitstype). The initial contents of a plain data type is undefined:\n\njulia> struct HasPlain\n           n::Int\n           HasPlain() = new()\n       end\n\njulia> HasPlain()\nHasPlain(438103441441)\n\nArrays of plain data types exhibit the same behavior.\n\nYou can pass incomplete objects to other functions from inner constructors to delegate their completion:\n\njulia> mutable struct Lazy\n           data\n           Lazy(v) = complete_me(new(), v)\n       end\n\nAs with incomplete objects returned from constructors, if complete_me or any of its callees try to access the data field of the Lazy object before it has been initialized, an error will be thrown immediately."},{"location":"manual/constructors.html#Parametric-Constructors","page":"Constructors","title":"Parametric Constructors","category":"section","text":"Parametric types add a few wrinkles to the constructor story. Recall from Parametric Types that, by default, instances of parametric composite types can be constructed either with explicitly given type parameters or with type parameters implied by the types of the arguments given to the constructor. Here are some examples:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n       end\n\njulia> Point(1,2) ## implicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point(1.0,2.5) ## implicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point(1,2.5) ## implicit T ##\nERROR: MethodError: no method matching Point(::Int64, ::Float64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, ::T) where T<:Real at none:2\n\njulia> Point{Int64}(1, 2) ## explicit T ##\nPoint{Int64}(1, 2)\n\njulia> Point{Int64}(1.0,2.5) ## explicit T ##\nERROR: InexactError: Int64(2.5)\nStacktrace:\n[...]\n\njulia> Point{Float64}(1.0, 2.5) ## explicit T ##\nPoint{Float64}(1.0, 2.5)\n\njulia> Point{Float64}(1,2) ## explicit T ##\nPoint{Float64}(1.0, 2.0)\n\nAs you can see, for constructor calls with explicit type parameters, the arguments are converted to the implied field types: Point{Int64}(1,2) works, but Point{Int64}(1.0,2.5) raises an InexactError when converting 2.5 to Int64. When the type is implied by the arguments to the constructor call, as in Point(1,2), then the types of the arguments must agree – otherwise the T cannot be determined – but any pair of real arguments with matching type may be given to the generic Point constructor.\n\nWhat's really going on here is that Point, Point{Float64} and Point{Int64} are all different constructor functions. In fact, Point{T} is a distinct constructor function for each type T. Without any explicitly provided inner constructors, the declaration of the composite type Point{T<:Real} automatically provides an inner constructor, Point{T}, for each possible type T<:Real, that behaves just like non-parametric default inner constructors do. It also provides a single general outer Point constructor that takes pairs of real arguments, which must be of the same type. This automatic provision of constructors is equivalent to the following explicit declaration:\n\njulia> struct Point{T<:Real}\n           x::T\n           y::T\n           Point{T}(x,y) where {T<:Real} = new(x,y)\n       end\n\njulia> Point(x::T, y::T) where {T<:Real} = Point{T}(x,y);\n\nNotice that each definition looks like the form of constructor call that it handles. The call Point{Int64}(1,2) will invoke the definition Point{T}(x,y) inside the struct block. The outer constructor declaration, on the other hand, defines a method for the general Point constructor which only applies to pairs of values of the same real type. This declaration makes constructor calls without explicit type parameters, like Point(1,2) and Point(1.0,2.5), work. Since the method declaration restricts the arguments to being of the same type, calls like Point(1,2.5), with arguments of different types, result in \"no method\" errors.\n\nSuppose we wanted to make the constructor call Point(1,2.5) work by \"promoting\" the integer value 1 to the floating-point value 1.0. The simplest way to achieve this is to define the following additional outer constructor method:\n\njulia> Point(x::Int64, y::Float64) = Point(convert(Float64,x),y);\n\nThis method uses the convert function to explicitly convert x to Float64 and then delegates construction to the general constructor for the case where both arguments are Float64. With this method definition what was previously a MethodError now successfully creates a point of type Point{Float64}:\n\njulia> p = Point(1,2.5)\nPoint{Float64}(1.0, 2.5)\n\njulia> typeof(p)\nPoint{Float64}\n\nHowever, other similar calls still don't work:\n\njulia> Point(1.5,2)\nERROR: MethodError: no method matching Point(::Float64, ::Int64)\nThe type `Point` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  Point(::T, !Matched::T) where T<:Real\n   @ Main none:1\n  Point(!Matched::Int64, !Matched::Float64)\n   @ Main none:1\n\nStacktrace:\n[...]\n\nFor a more general way to make all such calls work sensibly, see Conversion and Promotion. At the risk of spoiling the suspense, we can reveal here that all it takes is the following outer method definition to make all calls to the general Point constructor work as one would expect:\n\njulia> Point(x::Real, y::Real) = Point(promote(x,y)...);\n\nThe promote function converts all its arguments to a common type – in this case Float64. With this method definition, the Point constructor promotes its arguments the same way that numeric operators like + do, and works for all kinds of real numbers:\n\njulia> Point(1.5,2)\nPoint{Float64}(1.5, 2.0)\n\njulia> Point(1,1//2)\nPoint{Rational{Int64}}(1//1, 1//2)\n\njulia> Point(1.0,1//2)\nPoint{Float64}(1.0, 0.5)\n\nThus, while the implicit type parameter constructors provided by default in Julia are fairly strict, it is possible to make them behave in a more relaxed but sensible manner quite easily. Moreover, since constructors can leverage all of the power of the type system, methods, and multiple dispatch, defining sophisticated behavior is typically quite simple."},{"location":"manual/constructors.html#Case-Study:-Rational","page":"Constructors","title":"Case Study: Rational","category":"section","text":"Perhaps the best way to tie all these pieces together is to present a real world example of a parametric composite type and its constructor methods. To that end, we implement our own rational number type OurRational, similar to Julia's built-in Rational type, defined in rational.jl:\n\njulia> struct OurRational{T<:Integer} <: Real\n           num::T\n           den::T\n           function OurRational{T}(num::T, den::T) where T<:Integer\n               if num == 0 && den == 0\n                    error(\"invalid rational: 0//0\")\n               end\n               num = flipsign(num, den)\n               den = flipsign(den, den)\n               g = gcd(num, den)\n               num = div(num, g)\n               den = div(den, g)\n               new(num, den)\n           end\n       end\n\njulia> OurRational(n::T, d::T) where {T<:Integer} = OurRational{T}(n,d)\nOurRational\n\njulia> OurRational(n::Integer, d::Integer) = OurRational(promote(n,d)...)\nOurRational\n\njulia> OurRational(n::Integer) = OurRational(n,one(n))\nOurRational\n\njulia> ⊘(n::Integer, d::Integer) = OurRational(n,d)\n⊘ (generic function with 1 method)\n\njulia> ⊘(x::OurRational, y::Integer) = x.num ⊘ (x.den*y)\n⊘ (generic function with 2 methods)\n\njulia> ⊘(x::Integer, y::OurRational) = (x*y.den) ⊘ y.num\n⊘ (generic function with 3 methods)\n\njulia> ⊘(x::Complex, y::Real) = complex(real(x) ⊘ y, imag(x) ⊘ y)\n⊘ (generic function with 4 methods)\n\njulia> ⊘(x::Real, y::Complex) = (x*y') ⊘ real(y*y')\n⊘ (generic function with 5 methods)\n\njulia> function ⊘(x::Complex, y::Complex)\n           xy = x*y'\n           yy = real(y*y')\n           complex(real(xy) ⊘ yy, imag(xy) ⊘ yy)\n       end\n⊘ (generic function with 6 methods)\n\nThe first line – struct OurRational{T<:Integer} <: Real – declares that OurRational takes one type parameter of an integer type, and is itself a real type. The field declarations num::T and den::T indicate that the data held in a OurRational{T} object are a pair of integers of type T, one representing the rational value's numerator and the other representing its denominator.\n\nNow things get interesting. OurRational has a single inner constructor method which checks that num and den aren't both zero and ensures that every rational is constructed in \"lowest terms\" with a non-negative denominator. This is accomplished by first flipping the signs of numerator and denominator if the denominator is negative. Then, both are divided by their greatest common divisor (gcd always returns a non-negative number, regardless of the sign of its arguments). Because this is the only inner constructor for OurRational, we can be certain that OurRational objects are always constructed in this normalized form.\n\nOurRational also provides several outer constructor methods for convenience. The first is the \"standard\" general constructor that infers the type parameter T from the type of the numerator and denominator when they have the same type. The second applies when the given numerator and denominator values have different types: it promotes them to a common type and then delegates construction to the outer constructor for arguments of matching type. The third outer constructor turns integer values into rationals by supplying a value of 1 as the denominator.\n\nFollowing the outer constructor definitions, we defined a number of methods for the ⊘ operator, which provides a syntax for writing rationals (e.g. 1 ⊘ 2). Julia's Rational type uses the // operator for this purpose. Before these definitions, ⊘ is a completely undefined operator with only syntax and no meaning. Afterwards, it behaves just as described in Rational Numbers – its entire behavior is defined in these few lines. Note that the infix use of ⊘ works because Julia has a set of symbols that are recognized to be infix operators. The first and most basic definition just makes a ⊘ b construct a OurRational by applying the OurRational constructor to a and b when they are integers. When one of the operands of ⊘ is already a rational number, we construct a new rational for the resulting ratio slightly differently; this behavior is actually identical to division of a rational with an integer. Finally, applying ⊘ to complex integral values creates an instance of Complex{<:OurRational} – a complex number whose real and imaginary parts are rationals:\n\njulia> z = (1 + 2im) ⊘ (1 - 2im);\n\njulia> typeof(z)\nComplex{OurRational{Int64}}\n\njulia> typeof(z) <: Complex{<:OurRational}\ntrue\n\nThus, although the ⊘ operator usually returns an instance of OurRational, if either of its arguments are complex integers, it will return an instance of Complex{<:OurRational} instead. The interested reader should consider perusing the rest of rational.jl: it is short, self-contained, and implements an entire basic Julia type."},{"location":"manual/constructors.html#Outer-only-constructors","page":"Constructors","title":"Outer-only constructors","category":"section","text":"As we have seen, a typical parametric type has inner constructors that are called when type parameters are known; e.g. they apply to Point{Int} but not to Point. Optionally, outer constructors that determine type parameters automatically can be added, for example constructing a Point{Int} from the call Point(1,2). Outer constructors call inner constructors to actually make instances. However, in some cases one would rather not provide inner constructors, so that specific type parameters cannot be requested manually.\n\nFor example, say we define a type that stores a vector along with an accurate representation of its sum:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nSummedArray{Int32, Int32}(Int32[1, 2, 3], 6)\n\nThe problem is that we want S to be a larger type than T, so that we can sum many elements with less information loss. For example, when T is Int32, we would like S to be Int64. Therefore we want to avoid an interface that allows the user to construct instances of the type SummedArray{Int32,Int32}. One way to do this is to provide a constructor only for SummedArray, but inside the struct definition block to suppress generation of default constructors:\n\njulia> struct SummedArray{T<:Number,S<:Number}\n           data::Vector{T}\n           sum::S\n           function SummedArray(a::Vector{T}) where T\n               S = widen(T)\n               new{T,S}(a, sum(S, a))\n           end\n       end\n\njulia> SummedArray(Int32[1; 2; 3], Int32(6))\nERROR: MethodError: no method matching SummedArray(::Vector{Int32}, ::Int32)\nThe type `SummedArray` exists, but no method is defined for this combination of argument types when trying to construct it.\n\nClosest candidates are:\n  SummedArray(::Vector{T}) where T\n   @ Main none:4\n\nStacktrace:\n[...]\n\nThis constructor will be invoked by the syntax SummedArray(a). The syntax new{T,S} allows specifying parameters for the type to be constructed, i.e. this call will return a SummedArray{T,S}. new{T,S} can be used in any constructor definition, but for convenience the parameters to new{} are automatically derived from the type being constructed when possible."},{"location":"manual/constructors.html#Constructors-are-just-callable-objects","page":"Constructors","title":"Constructors are just callable objects","category":"section","text":"An object of any type may be made callable by defining a method. This includes types, i.e., objects of type Type; and constructors may, in fact, be viewed as just callable type objects. For example, there are many methods defined on Bool and various supertypes of it:\n\nmethods(Bool)\n\nThe usual constructor syntax is exactly equivalent to the function-like object syntax, so trying to define a method with each syntax will cause the first method to be overwritten by the next one:\n\njulia> struct S\n           f::Int\n       end\n\njulia> S() = S(7)\nS\n\njulia> (::Type{S})() = S(8)  # overwrites the previous constructor method\n\njulia> S()\nS(8)"},{"location":"stdlib/CRC32c.html#CRC32c","page":"CRC32c","title":"CRC32c","category":"section","text":"Standard library module for computing the CRC-32c checksum."},{"location":"stdlib/CRC32c.html#CRC32c.crc32c","page":"CRC32c","title":"CRC32c.crc32c","category":"function","text":"crc32c(data, crc::UInt32=0x00000000)\n\nCompute the CRC-32c checksum of the given data, which can be an Array{UInt8}, a contiguous subarray thereof, an AbstractVector{UInt8}, or a String. Optionally, you can pass a starting crc integer to be mixed in with the checksum. The crc parameter can be used to compute a checksum on data divided into chunks: performing crc32c(data2, crc32c(data1)) is equivalent to the checksum of [data1; data2]. (Technically, a little-endian checksum is computed.)\n\nThere is also a method crc32c(io, nb, crc) to checksum nb bytes from a stream io, or crc32c(io, crc) to checksum all the remaining bytes. Hence you can do open(crc32c, filename) to checksum an entire file, or crc32c(seekstart(buf)) to checksum an IOBuffer without calling take!.\n\nFor a String, note that the result is specific to the UTF-8 encoding (a different checksum would be obtained from a different Unicode encoding). To checksum an a::AbstractArray of some other bitstype without padding, you can do crc32c(vec(reinterpret(UInt8,a))), but note that the result may be endian-dependent.\n\n\n\n\n\n"},{"location":"stdlib/CRC32c.html#CRC32c.crc32c-Tuple{IO, Integer, UInt32}","page":"CRC32c","title":"CRC32c.crc32c","category":"method","text":"crc32c(io::IO, [nb::Integer,] crc::UInt32=0x00000000)\n\nRead up to nb bytes from io and return the CRC-32c checksum, optionally mixed with a starting crc integer.  If nb is not supplied, then io will be read until the end of the stream.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random-Numbers","page":"Random Numbers","title":"Random Numbers","category":"section","text":"Random number generation in Julia uses the Xoshiro256++ algorithm by default, with per-Task state. Other RNG types can be plugged in by inheriting the AbstractRNG type; they can then be used to obtain multiple streams of random numbers.\n\nThe PRNGs (pseudorandom number generators) exported by the Random package are:\n\nTaskLocalRNG: a token that represents use of the currently active Task-local stream, deterministically seeded from the parent task, or by RandomDevice (with system randomness) at program start\nXoshiro: generates a high-quality stream of random numbers with a small state vector and high performance using the Xoshiro256++ algorithm\nRandomDevice: for OS-provided entropy. This may be used for cryptographically secure random numbers (CS(P)RNG).\nMersenneTwister: an alternate high-quality PRNG which was the default in older versions of Julia, and is also quite fast, but requires much more space to store the state vector and generate a random sequence.\n\nMost functions related to random generation accept an optional AbstractRNG object as first argument. Some also accept dimension specifications dims... (which can also be given as a tuple) to generate arrays of random values. In a multi-threaded program, you should generally use different RNG objects from different threads or tasks in order to be thread-safe. However, the default RNG is thread-safe as of Julia 1.3 (using a per-thread RNG up to version 1.6, and per-task thereafter).\n\nThe provided RNGs can generate uniform random numbers of the following types: Float16, Float32, Float64, BigFloat, Bool, Int8, UInt8, Int16, UInt16, Int32, UInt32, Int64, UInt64, Int128, UInt128, BigInt (or complex numbers of those types). Random floating point numbers are generated uniformly in 0 1). As BigInt represents unbounded integers, the interval must be specified (e.g. rand(big.(1:6))).\n\nAdditionally, normal and exponential distributions are implemented for some AbstractFloat and Complex types, see randn and randexp for details.\n\nTo generate random numbers from other distributions, see the Distributions.jl package.\n\nwarning: Warning\nBecause the precise way in which random numbers are generated is considered an implementation detail, bug fixes and speed improvements may change the stream of numbers that are generated after a version change. Relying on a specific seed or generated stream of numbers during unit testing is thus discouraged - consider testing properties of the methods in question instead."},{"location":"stdlib/Random.html#Random-numbers-module","page":"Random Numbers","title":"Random numbers module","category":"section","text":""},{"location":"stdlib/Random.html#Random-generation-functions","page":"Random Numbers","title":"Random generation functions","category":"section","text":""},{"location":"stdlib/Random.html#Subsequences,-permutations-and-shuffling","page":"Random Numbers","title":"Subsequences, permutations and shuffling","category":"section","text":""},{"location":"stdlib/Random.html#Generators-(creation-and-seeding)","page":"Random Numbers","title":"Generators (creation and seeding)","category":"section","text":""},{"location":"stdlib/Random.html#rand-api-hook","page":"Random Numbers","title":"Hooking into the Random API","category":"section","text":"There are two mostly orthogonal ways to extend Random functionalities:\n\ngenerating random values of custom types\ncreating new generators\n\nThe API for 1) is quite functional, but is relatively recent so it may still have to evolve in subsequent releases of the Random module. For example, it's typically sufficient to implement one rand method in order to have all other usual methods work automatically.\n\nThe API for 2) is still rudimentary, and may require more work than strictly necessary from the implementer, in order to support usual types of generated values."},{"location":"stdlib/Random.html#Generating-random-values-of-custom-types","page":"Random Numbers","title":"Generating random values of custom types","category":"section","text":"Generating random values for some distributions may involve various trade-offs. Pre-computed values, such as an alias table for discrete distributions, or “squeezing” functions for univariate distributions, can speed up sampling considerably. How much information should be pre-computed can depend on the number of values we plan to draw from a distribution. Also, some random number generators can have certain properties that various algorithms may want to exploit.\n\nThe Random module defines a customizable framework for obtaining random values that can address these issues. Each invocation of rand generates a sampler which can be customized with the above trade-offs in mind, by adding methods to Sampler, which in turn can dispatch on the random number generator, the object that characterizes the distribution, and a suggestion for the number of repetitions. Currently, for the latter, Val{1} (for a single sample) and Val{Inf} (for an arbitrary number) are used, with Random.Repetition an alias for both.\n\nThe object returned by Sampler is then used to generate the random values. When implementing the random generation interface for a value X that can be sampled from, the implementer should define the method\n\nrand(rng, sampler)\n\nfor the particular sampler returned by Sampler(rng, X, repetition).\n\nSamplers can be arbitrary values that implement rand(rng, sampler), but for most applications the following predefined samplers may be sufficient:\n\nSamplerType{T}() can be used for implementing samplers that draw from type T (e.g. rand(Int)). This is the default returned by Sampler for types.\nSamplerTrivial(self) is a simple wrapper for self, which can be accessed with []. This is the recommended sampler when no pre-computed information is needed (e.g. rand(1:3)), and is the default returned by Sampler for values.\nSamplerSimple(self, data) also contains the additional data field, which can be used to store arbitrary pre-computed values, which should be computed in a custom method of Sampler.\n\nWe provide examples for each of these. We assume here that the choice of algorithm is independent of the RNG, so we use AbstractRNG in our signatures.\n\nDecoupling pre-computation from actually generating the values is part of the API, and is also available to the user. As an example, assume that rand(rng, 1:20) has to be called repeatedly in a loop: the way to take advantage of this decoupling is as follows:\n\nrng = Xoshiro()\nsp = Random.Sampler(rng, 1:20) # or Random.Sampler(Xoshiro, 1:20)\nfor x in X\n    n = rand(rng, sp) # similar to n = rand(rng, 1:20)\n    # use n\nend\n\nThis is the mechanism that is also used in the standard library, e.g. by the default implementation of random array generation (like in rand(1:20, 10))."},{"location":"stdlib/Random.html#Generating-values-from-a-type","page":"Random Numbers","title":"Generating values from a type","category":"section","text":"Given a type T, it's currently assumed that if rand(T) is defined, an object of type T will be produced. SamplerType is the default sampler for types. In order to define random generation of values of type T, the rand(rng::AbstractRNG, ::Random.SamplerType{T}) method should be defined, and should return values what rand(rng, T) is expected to return.\n\nLet's take the following example: we implement a Die type, with a variable number n of sides, numbered from 1 to n. We want rand(Die) to produce a Die with a random number of up to 20 sides (and at least 4):\n\nstruct Die\n    nsides::Int # number of sides\nend\n\nRandom.rand(rng::AbstractRNG, ::Random.SamplerType{Die}) = Die(rand(rng, 4:20))\n\n# output\n\n\nScalar and array methods for Die now work as expected:\n\njulia> rand(Die)\nDie(5)\n\njulia> rand(Xoshiro(0), Die)\nDie(10)\n\njulia> rand(Die, 3)\n3-element Vector{Die}:\n Die(9)\n Die(15)\n Die(14)\n\njulia> a = Vector{Die}(undef, 3); rand!(a)\n3-element Vector{Die}:\n Die(19)\n Die(7)\n Die(17)"},{"location":"stdlib/Random.html#A-simple-sampler-without-pre-computed-data","page":"Random Numbers","title":"A simple sampler without pre-computed data","category":"section","text":"Here we define a sampler for a collection. If no pre-computed data is required, it can be implemented with a SamplerTrivial sampler, which is in fact the default fallback for values.\n\nIn order to define random generation out of objects of type S, the following method should be defined: rand(rng::AbstractRNG, sp::Random.SamplerTrivial{S}). Here, sp simply wraps an object of type S, which can be accessed via sp[]. Continuing the Die example, we want now to define rand(d::Die) to produce an Int corresponding to one of d's sides:\n\njulia> Random.rand(rng::AbstractRNG, d::Random.SamplerTrivial{Die}) = rand(rng, 1:d[].nsides);\n\njulia> rand(Die(4))\n1\n\njulia> rand(Die(4), 3)\n3-element Vector{Any}:\n 2\n 3\n 3\n\nGiven a collection type S, it's currently assumed that if rand(::S) is defined, an object of type eltype(S) will be produced. In the last example, a Vector{Any} is produced; the reason is that eltype(Die) == Any. The remedy is to define Base.eltype(::Type{Die}) = Int."},{"location":"stdlib/Random.html#Generating-values-for-an-AbstractFloat-type","page":"Random Numbers","title":"Generating values for an AbstractFloat type","category":"section","text":"AbstractFloat types are special-cased, because by default random values are not produced in the whole type domain, but rather in [0,1). The following method should be implemented for T <: AbstractFloat: Random.rand(::AbstractRNG, ::Random.SamplerTrivial{Random.CloseOpen01{T}})"},{"location":"stdlib/Random.html#An-optimized-sampler-with-pre-computed-data","page":"Random Numbers","title":"An optimized sampler with pre-computed data","category":"section","text":"Consider a discrete distribution, where numbers 1:n are drawn with given probabilities that sum to one. When many values are needed from this distribution, the fastest method is using an alias table. We don't provide the algorithm for building such a table here, but suppose it is available in make_alias_table(probabilities) instead, and draw_number(rng, alias_table) can be used to draw a random number from it.\n\nSuppose that the distribution is described by\n\nstruct DiscreteDistribution{V <: AbstractVector}\n    probabilities::V\nend\n\nand that we always want to build an alias table, regardless of the number of values needed (we learn how to customize this below). The methods\n\nRandom.eltype(::Type{<:DiscreteDistribution}) = Int\n\nfunction Random.Sampler(::Type{<:AbstractRNG}, distribution::DiscreteDistribution, ::Repetition)\n    SamplerSimple(distribution, make_alias_table(distribution.probabilities))\nend\n\nshould be defined to return a sampler with pre-computed data, then\n\nfunction rand(rng::AbstractRNG, sp::SamplerSimple{<:DiscreteDistribution})\n    draw_number(rng, sp.data)\nend\n\nwill be used to draw the values."},{"location":"stdlib/Random.html#Custom-sampler-types","page":"Random Numbers","title":"Custom sampler types","category":"section","text":"The SamplerSimple type is sufficient for most use cases with precomputed data. However, in order to demonstrate how to use custom sampler types, here we implement something similar to SamplerSimple.\n\nGoing back to our Die example: rand(::Die) uses random generation from a range, so there is an opportunity for this optimization. We call our custom sampler SamplerDie.\n\nimport Random: Sampler, rand\n\nstruct SamplerDie <: Sampler{Int} # generates values of type Int\n    die::Die\n    sp::Sampler{Int} # this is an abstract type, so this could be improved\nend\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerDie(die, Sampler(RNG, 1:die.nsides, r))\n# the `r` parameter will be explained later on\n\nrand(rng::AbstractRNG, sp::SamplerDie) = rand(rng, sp.sp)\n\nIt's now possible to get a sampler with sp = Sampler(rng, die), and use sp instead of die in any rand call involving rng. In the simplistic example above, die doesn't need to be stored in SamplerDie but this is often the case in practice.\n\nOf course, this pattern is so frequent that the helper type used above, namely Random.SamplerSimple, is available, saving us the definition of SamplerDie: we could have implemented our decoupling with:\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, r::Random.Repetition) =\n    SamplerSimple(die, Sampler(RNG, 1:die.nsides, r))\n\nrand(rng::AbstractRNG, sp::SamplerSimple{Die}) = rand(rng, sp.data)\n\nHere, sp.data refers to the second parameter in the call to the SamplerSimple constructor (in this case equal to Sampler(rng, 1:die.nsides, r)), while the Die object can be accessed via sp[].\n\nLike SamplerDie, any custom sampler must be a subtype of Sampler{T} where T is the type of the generated values. Note that SamplerSimple(x, data) isa Sampler{eltype(x)}, so this constrains what the first argument to SamplerSimple can be (it's recommended to use SamplerSimple like in the Die example, where x is simply forwarded while defining a Sampler method). Similarly, SamplerTrivial(x) isa Sampler{eltype(x)}.\n\nAnother helper type is currently available for other cases, Random.SamplerTag, but is considered as internal API, and can break at any time without proper deprecations."},{"location":"stdlib/Random.html#Using-distinct-algorithms-for-scalar-or-array-generation","page":"Random Numbers","title":"Using distinct algorithms for scalar or array generation","category":"section","text":"In some cases, whether one wants to generate only a handful of values or a large number of values will have an impact on the choice of algorithm. This is handled with the third parameter of the Sampler constructor. Let's assume we defined two helper types for Die, say SamplerDie1 which should be used to generate only few random values, and SamplerDieMany for many values. We can use those types as follows:\n\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{1}) = SamplerDie1(...)\nSampler(RNG::Type{<:AbstractRNG}, die::Die, ::Val{Inf}) = SamplerDieMany(...)\n\nOf course, rand must also be defined on those types (i.e. rand(::AbstractRNG, ::SamplerDie1) and rand(::AbstractRNG, ::SamplerDieMany)). Note that, as usual, SamplerTrivial and SamplerSimple can be used if custom types are not necessary.\n\nNote: Sampler(rng, x) is simply a shorthand for Sampler(rng, x, Val(Inf)), and Random.Repetition is an alias for Union{Val{1}, Val{Inf}}."},{"location":"stdlib/Random.html#Creating-new-generators","page":"Random Numbers","title":"Creating new generators","category":"section","text":"The API is not clearly defined yet, but as a rule of thumb:\n\nany rand method producing \"basic\" types (isbitstype integer and floating types in Base) should be defined for this specific RNG, if they are needed;\nother documented rand methods accepting an AbstractRNG should work out of the box, (provided the methods from 1) what are relied on are implemented), but can of course be specialized for this RNG if there is room for optimization;\ncopy for pseudo-RNGs should return an independent copy that generates the exact same random sequence as the original from that point when called in the same way. When this is not feasible (e.g. hardware-based RNGs), copy must not be implemented.\n\nConcerning 1), a rand method may happen to work automatically, but it's not officially supported and may break without warnings in a subsequent release.\n\nTo define a new rand method for an hypothetical MyRNG generator, and a value specification s (e.g. s == Int, or s == 1:10) of type S==typeof(s) or S==Type{s} if s is a type, the same two methods as we saw before must be defined:\n\nSampler(::Type{MyRNG}, ::S, ::Repetition), which returns an object of type say SamplerS\nrand(rng::MyRNG, sp::SamplerS)\n\nIt can happen that Sampler(rng::AbstractRNG, ::S, ::Repetition) is already defined in the Random module. It would then be possible to skip step 1) in practice (if one wants to specialize generation for this particular RNG type), but the corresponding SamplerS type is considered as internal detail, and may be changed without warning."},{"location":"stdlib/Random.html#Specializing-array-generation","page":"Random Numbers","title":"Specializing array generation","category":"section","text":"In some cases, for a given RNG type, generating an array of random values can be more efficient with a specialized method than by merely using the decoupling technique explained before. This is for example the case for MersenneTwister, which natively writes random values in an array.\n\nTo implement this specialization for MyRNG and for a specification s, producing elements of type S, the following method can be defined: rand!(rng::MyRNG, a::AbstractArray{S}, ::SamplerS), where SamplerS is the type of the sampler returned by Sampler(MyRNG, s, Val(Inf)). Instead of AbstractArray, it's possible to implement the functionality only for a subtype, e.g. Array{S}. The non-mutating array method of rand will automatically call this specialization internally."},{"location":"stdlib/Random.html#Reproducibility","page":"Random Numbers","title":"Reproducibility","category":"section","text":"By using an RNG parameter initialized with a given seed, you can reproduce the same pseudorandom number sequence when running your program multiple times. However, a minor release of Julia (e.g. 1.3 to 1.4) may change the sequence of pseudorandom numbers generated from a specific seed. (Even if the sequence produced by a low-level function like rand does not change, the output of higher-level functions like randsubseq may change due to algorithm updates.) Rationale: guaranteeing that pseudorandom streams never change prohibits many algorithmic improvements.\n\nIf you need to guarantee exact reproducibility of random data, it is advisable to simply save the data (e.g. as a supplementary attachment in a scientific publication). (You can also, of course, specify a particular Julia version and package manifest, especially if you require bit reproducibility.)\n\nSoftware tests that rely on specific \"random\" data should also generally either save the data, embed it into the test code, or use third-party packages like StableRNGs.jl. On the other hand, tests that should pass for most random data (e.g. testing A \\ (A*x) ≈ x for a random matrix A = randn(n,n)) can use an RNG with a fixed seed to ensure that simply running the test many times does not encounter a failure due to very improbable data (e.g. an extremely ill-conditioned matrix).\n\nThe statistical distribution from which random samples are drawn is guaranteed to be the same across any minor Julia releases."},{"location":"stdlib/Random.html#Random.Random","page":"Random Numbers","title":"Random.Random","category":"module","text":"Random\n\nSupport for generating random numbers. Provides rand, randn, AbstractRNG, Xoshiro, MersenneTwister, and RandomDevice.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Base.rand","page":"Random Numbers","title":"Base.rand","category":"function","text":"rand([rng=default_rng()], [S], [dims...])\n\nPick a random element or array of random elements from the set of values specified by S; S can be\n\nan indexable collection (for example 1:9 or ('x', \"y\", :z))\nan AbstractDict or AbstractSet object\na string (considered as a collection of characters), or\na type from the list below, corresponding to the specified set of values\nconcrete integer types sample from typemin(S):typemax(S) (excepting BigInt which is not supported)\nconcrete floating point types sample from [0, 1)\nconcrete complex types Complex{T} if T is a sampleable type take their real and imaginary components independently from the set of values corresponding to T, but are not supported if T is not sampleable.\nall <:AbstractChar types sample from the set of valid Unicode scalars\na user-defined type and set of values; for implementation guidance please see Hooking into the Random API\na tuple type of known size and where each parameter of S is itself a sampleable type; return a value of type S. Note that tuple types such as Tuple{Vararg{T}} (unknown size) and Tuple{1:2} (parameterized with a value) are not supported\na Pair type, e.g. Pair{X, Y} such that rand is defined for X and Y, in which case random pairs are produced.\n\nS defaults to Float64. When only one argument is passed besides the optional rng and is a Tuple, it is interpreted as a collection of values (S) and not as dims.\n\nSee also randn for normally distributed numbers, and rand! and randn! for the in-place equivalents.\n\ncompat: Julia 1.1\nSupport for S as a tuple requires at least Julia 1.1.\n\ncompat: Julia 1.11\nSupport for S as a Tuple type requires at least Julia 1.11.\n\nExamples\n\njulia> rand(Int, 2)\n2-element Vector{Int64}:\n 1339893410598768192\n 1575814717733606317\n\njulia> using Random\n\njulia> rand(Xoshiro(0), Dict(1=>2, 3=>4))\n3 => 4\n\njulia> rand((2, 3))\n3\n\njulia> rand(Float64, (2, 3))\n2×3 Matrix{Float64}:\n 0.999717  0.0143835  0.540787\n 0.696556  0.783855   0.938235\n\nnote: Note\nThe complexity of rand(rng, s::Union{AbstractDict,AbstractSet}) is linear in the length of s, unless an optimized method with constant complexity is available, which is the case for Dict, Set and dense BitSets. For more than a few calls, use rand(rng, collect(s)) instead, or either rand(rng, Dict(s)) or rand(rng, Set(s)) as appropriate.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.rand!","page":"Random Numbers","title":"Random.rand!","category":"function","text":"rand!([rng=default_rng()], A, [S=eltype(A)])\n\nPopulate the array A with random values. If S is specified (S can be a type or a collection, cf. rand for details), the values are picked randomly from S. This is equivalent to copyto!(A, rand(rng, S, size(A))) but without allocating a new array.\n\nExamples\n\njulia> rand!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n 0.521213795535383\n 0.5868067574533484\n 0.8908786980927811\n 0.19090669902576285\n 0.5256623915420473\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.bitrand","page":"Random Numbers","title":"Random.bitrand","category":"function","text":"bitrand([rng=default_rng()], [dims...])\n\nGenerate a BitArray of random boolean values.\n\nExamples\n\njulia> bitrand(Xoshiro(123), 10)\n10-element BitVector:\n 0\n 1\n 0\n 1\n 0\n 1\n 0\n 0\n 1\n 1\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Base.randn","page":"Random Numbers","title":"Base.randn","category":"function","text":"randn([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a normally-distributed random number of type T with mean 0 and standard deviation 1. Given the optional dims argument(s), generate an array of size dims of such numbers. Julia's standard library supports randn for any floating-point type that implements rand, e.g. the Base types Float16, Float32, Float64 (the default), and BigFloat, along with their Complex counterparts.\n\n(When T is complex, the values are drawn from the circularly symmetric complex normal distribution of variance 1, corresponding to real and imaginary parts having independent normal distribution with mean zero and variance 1/2).\n\nSee also randn! to act in-place.\n\nExamples\n\nGenerating a single random number (with the default Float64 type):\n\njulia> randn()\n-0.942481877315864\n\nGenerating a matrix of normal random numbers (with the default Float64 type):\n\njulia> randn(2,3)\n2×3 Matrix{Float64}:\n  1.18786   -0.678616   1.49463\n -0.342792  -0.134299  -1.45005\n\nSetting up of the random number generator rng with a user-defined seed (for reproducible numbers) and using it to generate a random Float32 number or a matrix of ComplexF32 random numbers:\n\njulia> using Random\n\njulia> rng = Xoshiro(123);\n\njulia> randn(rng, Float32)\n-0.6457307f0\n\njulia> randn(rng, ComplexF32, (2, 3))\n2×3 Matrix{ComplexF32}:\n  -1.03467-1.14806im  0.693657+0.056538im   0.291442+0.419454im\n -0.153912+0.34807im    1.0954-0.948661im  -0.543347-0.0538589im\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randn!","page":"Random Numbers","title":"Random.randn!","category":"function","text":"randn!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with normally-distributed (mean 0, standard deviation 1) random numbers. Also see the rand function.\n\nExamples\n\njulia> randn!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n -0.6457306721039767\n -1.4632513788889214\n -1.6236037455860806\n -0.21766510678354617\n  0.4922456865251828\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randexp","page":"Random Numbers","title":"Random.randexp","category":"function","text":"randexp([rng=default_rng()], [T=Float64], [dims...])\n\nGenerate a random number of type T according to the exponential distribution with scale 1. Optionally generate an array of such random numbers. The Base module currently provides an implementation for the types Float16, Float32, and Float64 (the default).\n\nExamples\n\njulia> rng = Xoshiro(123);\n\njulia> randexp(rng, Float32)\n1.1757717f0\n\njulia> randexp(rng, 3, 3)\n3×3 Matrix{Float64}:\n 1.37766  0.456653  0.236418\n 3.40007  0.229917  0.0684921\n 0.48096  0.577481  0.71835\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randexp!","page":"Random Numbers","title":"Random.randexp!","category":"function","text":"randexp!([rng=default_rng()], A::AbstractArray) -> A\n\nFill the array A with random numbers following the exponential distribution (with scale 1).\n\nExamples\n\njulia> randexp!(Xoshiro(123), zeros(5))\n5-element Vector{Float64}:\n 1.1757716836348473\n 1.758884569451514\n 1.0083623637301151\n 0.3510644315565272\n 0.6348266443720407\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randstring","page":"Random Numbers","title":"Random.randstring","category":"function","text":"randstring([rng=default_rng()], [chars], [len=8])\n\nCreate a random string of length len, consisting of characters from chars, which defaults to the set of upper- and lower-case letters and the digits 0-9. The optional rng argument specifies a random number generator, see Random Numbers.\n\nExamples\n\njulia> Random.seed!(3); randstring()\n\"Lxz5hUwn\"\n\njulia> randstring(Xoshiro(3), 'a':'z', 6)\n\"iyzcsm\"\n\njulia> randstring(\"ACGT\")\n\"TGCTCCTC\"\n\nnote: Note\nchars can be any collection of characters, of type Char or UInt8 (more efficient), provided rand can randomly pick characters from it.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randsubseq","page":"Random Numbers","title":"Random.randsubseq","category":"function","text":"randsubseq([rng=default_rng(),] A, p) -> Vector\n\nReturn a vector consisting of a random subsequence of the given array A, where each element of A is included (in order) with independent probability p. (Complexity is linear in p*length(A), so this function is efficient even if p is small and A is large.) Technically, this process is known as \"Bernoulli sampling\" of A.\n\nExamples\n\njulia> randsubseq(Xoshiro(123), 1:8, 0.3)\n2-element Vector{Int64}:\n 4\n 7\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randsubseq!","page":"Random Numbers","title":"Random.randsubseq!","category":"function","text":"randsubseq!([rng=default_rng(),] S, A, p)\n\nLike randsubseq, but the results are stored in S (which is resized as needed).\n\nExamples\n\njulia> S = Int64[];\n\njulia> randsubseq!(Xoshiro(123), S, 1:8, 0.3)\n2-element Vector{Int64}:\n 4\n 7\n\njulia> S\n2-element Vector{Int64}:\n 4\n 7\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randperm","page":"Random Numbers","title":"Random.randperm","category":"function","text":"randperm([rng=default_rng(),] n::Integer)\n\nConstruct a random permutation of length n. The optional rng argument specifies a random number generator (see Random Numbers). The element type of the result is the same as the type of n.\n\nTo randomly permute an arbitrary vector, see shuffle or shuffle!.\n\ncompat: Julia 1.1\nIn Julia 1.1 randperm returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randperm(Xoshiro(0), 6)\n6-element Vector{Int64}:\n 5\n 1\n 2\n 6\n 3\n 4\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randperm!","page":"Random Numbers","title":"Random.randperm!","category":"function","text":"randperm!([rng=default_rng(),] A::AbstractArray{<:Integer})\n\nConstruct in A a random permutation of length length(A). The optional rng argument specifies a random number generator (see Random Numbers). To randomly permute an arbitrary vector, see shuffle or shuffle!.\n\ncompat: Julia 1.13\nA isa Array was required prior to Julia v1.13.\n\nExamples\n\njulia> randperm!(Xoshiro(0), Vector{Int}(undef, 6))\n6-element Vector{Int64}:\n 5\n 1\n 2\n 6\n 3\n 4\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randcycle","page":"Random Numbers","title":"Random.randcycle","category":"function","text":"randcycle([rng=default_rng(),] n::Integer)\n\nConstruct a random cyclic permutation of length n. The optional rng argument specifies a random number generator, see Random Numbers. The element type of the result is the same as the type of n.\n\nHere, a \"cyclic permutation\" means that all of the elements lie within a single cycle.  If n > 0, there are (n-1) possible cyclic permutations, which are sampled uniformly.  If n == 0, randcycle returns an empty vector.\n\nrandcycle! is an in-place variant of this function.\n\ncompat: Julia 1.1\nIn Julia 1.1 and above, randcycle returns a vector v with eltype(v) == typeof(n) while in Julia 1.0 eltype(v) == Int.\n\nExamples\n\njulia> randcycle(Xoshiro(0), 6)\n6-element Vector{Int64}:\n 5\n 1\n 4\n 6\n 3\n 2\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.randcycle!","page":"Random Numbers","title":"Random.randcycle!","category":"function","text":"randcycle!([rng=default_rng(),] A::AbstractArray{<:Integer})\n\nConstruct in A a random cyclic permutation of length n = length(A). The optional rng argument specifies a random number generator, see Random Numbers.\n\nHere, a \"cyclic permutation\" means that all of the elements lie within a single cycle. If A is nonempty (n > 0), there are (n-1) possible cyclic permutations, which are sampled uniformly.  If A is empty, randcycle! leaves it unchanged.\n\nrandcycle is a variant of this function that allocates a new vector.\n\ncompat: Julia 1.13\nA isa Array was required prior to Julia v1.13.\n\nExamples\n\njulia> randcycle!(Xoshiro(0), Vector{Int}(undef, 6))\n6-element Vector{Int64}:\n 5\n 1\n 4\n 6\n 3\n 2\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.shuffle","page":"Random Numbers","title":"Random.shuffle","category":"function","text":"shuffle([rng=default_rng(),] v::Union{NTuple,AbstractArray})\n\nReturn a randomly permuted copy of v. The optional rng argument specifies a random number generator (see Random Numbers). To permute v in-place, see shuffle!. To obtain randomly permuted indices, see randperm.\n\ncompat: Julia 1.13\nShuffling an NTuple value requires Julia v1.13 or above.\n\nExamples\n\njulia> shuffle(Xoshiro(0), 1:6)\n6-element Vector{Int64}:\n 5\n 1\n 2\n 6\n 3\n 4\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.shuffle!","page":"Random Numbers","title":"Random.shuffle!","category":"function","text":"shuffle!([rng=default_rng(),] v::AbstractArray)\n\nIn-place version of shuffle: randomly permute v in-place, optionally supplying the random-number generator rng.\n\nExamples\n\njulia> shuffle!(Xoshiro(0), Vector(1:6))\n6-element Vector{Int64}:\n 5\n 1\n 2\n 6\n 3\n 4\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.default_rng","page":"Random Numbers","title":"Random.default_rng","category":"function","text":"Random.default_rng() -> rng\n\nReturn the default global random number generator (RNG), which is used by rand-related functions when no explicit RNG is provided.\n\nWhen the Random module is loaded, the default RNG is randomly seeded, via Random.seed!(): this means that each time a new julia session is started, the first call to rand() produces a different result, unless seed!(seed) is called first.\n\nIt is thread-safe: distinct threads can safely call rand-related functions on default_rng() concurrently, e.g. rand(default_rng()).\n\nnote: Note\nThe type of the default RNG is an implementation detail. Across different versions of Julia, you should not expect the default RNG to always have the same type, nor that it will produce the same stream of random numbers for a given seed.\n\ncompat: Julia 1.3\nThis function was introduced in Julia 1.3.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.seed!","page":"Random Numbers","title":"Random.seed!","category":"function","text":"seed!([rng=default_rng()], seed) -> rng\nseed!([rng=default_rng()]) -> rng\n\nReseed the random number generator: rng will give a reproducible sequence of numbers if and only if a seed is provided. Some RNGs don't accept a seed, like RandomDevice. After the call to seed!, rng is equivalent to a newly created object initialized with the same seed.\n\nThe types of accepted seeds depend on the type of rng, but in general, integer seeds should work. Providing nothing as the seed should be equivalent to not providing one.\n\nIf rng is not specified, it defaults to seeding the state of the shared task-local generator.\n\nExamples\n\njulia> Random.seed!(1234);\n\njulia> x1 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> Random.seed!(1234);\n\njulia> x2 = rand(2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\njulia> rng = Xoshiro(1234); rand(rng, 2) == x1\ntrue\n\njulia> Xoshiro(1) == Random.seed!(rng, 1)\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible\ntrue\n\njulia> rand(Random.seed!(rng), Bool) # not reproducible either\nfalse\n\njulia> rand(Xoshiro(), Bool) # not reproducible either\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.AbstractRNG","page":"Random Numbers","title":"Random.AbstractRNG","category":"type","text":"AbstractRNG\n\nSupertype for random number generators such as MersenneTwister and RandomDevice.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.TaskLocalRNG","page":"Random Numbers","title":"Random.TaskLocalRNG","category":"type","text":"TaskLocalRNG\n\nThe TaskLocalRNG has state that is local to its task, not its thread. It is seeded upon task creation, from the state of its parent task, but without advancing the state of the parent's RNG.\n\nAs an upside, the TaskLocalRNG is pretty fast, and permits reproducible multithreaded simulations (barring race conditions), independent of scheduler decisions. As long as the number of threads is not used to make decisions on task creation, simulation results are also independent of the number of available threads / CPUs. The random stream should not depend on hardware specifics, up to endianness and possibly word size.\n\nWhen seeding TaskLocalRNG() with seed!, the passed seed, if any, may be any integer.\n\ncompat: Julia 1.11\nSeeding TaskLocalRNG() with a negative integer seed requires at least Julia 1.11.\n\ncompat: Julia 1.10\nTask creation no longer advances the parent task's RNG state as of Julia 1.10.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.Xoshiro","page":"Random Numbers","title":"Random.Xoshiro","category":"type","text":"Xoshiro(seed::Union{Integer, AbstractString})\nXoshiro()\n\nXoshiro256++ is a fast pseudorandom number generator described by David Blackman and Sebastiano Vigna in \"Scrambled Linear Pseudorandom Number Generators\", ACM Trans. Math. Softw., 2021. Reference implementation is available at https://prng.di.unimi.it\n\nApart from the high speed, Xoshiro has a small memory footprint, making it suitable for applications where many different random states need to be held for long time.\n\nJulia's Xoshiro implementation has a bulk-generation mode; this seeds new virtual PRNGs from the parent, and uses SIMD to generate in parallel (i.e. the bulk stream consists of multiple interleaved xoshiro instances). The virtual PRNGs are discarded once the bulk request has been serviced (and should cause no heap allocations).\n\nIf no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing Xoshiro object.\n\ncompat: Julia 1.11\nPassing a negative integer seed requires at least Julia 1.11.\n\nExamples\n\njulia> using Random\n\njulia> rng = Xoshiro(1234);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> rng = Xoshiro(1234);\n\njulia> x2 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.32597672886359486\n 0.5490511363155669\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.MersenneTwister","page":"Random Numbers","title":"Random.MersenneTwister","category":"type","text":"MersenneTwister(seed)\nMersenneTwister()\n\nCreate a MersenneTwister RNG object. Different RNG objects can have their own seeds, which may be useful for generating different streams of random numbers. The seed may be an integer, a string, or a vector of UInt32 integers. If no seed is provided, a randomly generated one is created (using entropy from the system). See the seed! function for reseeding an already existing MersenneTwister object.\n\ncompat: Julia 1.11\nPassing a negative integer seed requires at least Julia 1.11.\n\nExamples\n\njulia> rng = MersenneTwister(123);\n\njulia> x1 = rand(rng, 2)\n2-element Vector{Float64}:\n 0.37453777969575874\n 0.8735343642013971\n\njulia> x2 = rand(MersenneTwister(123), 2)\n2-element Vector{Float64}:\n 0.37453777969575874\n 0.8735343642013971\n\njulia> x1 == x2\ntrue\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.RandomDevice","page":"Random Numbers","title":"Random.RandomDevice","category":"type","text":"RandomDevice()\n\nCreate a RandomDevice RNG object. Two such objects will always generate different streams of random numbers. The entropy is obtained from the operating system.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.Sampler","page":"Random Numbers","title":"Random.Sampler","category":"type","text":"Sampler(rng, x, repetition = Val(Inf))\n\nReturn a sampler object that can be used to generate random values from rng for x.\n\nWhen sp = Sampler(rng, x, repetition), rand(rng, sp) will be used to draw random values, and should be defined accordingly.\n\nrepetition can be Val(1) or Val(Inf), and should be used as a suggestion for deciding the amount of precomputation, if applicable.\n\nRandom.SamplerType and Random.SamplerTrivial are default fallbacks for types and values, respectively. Random.SamplerSimple can be used to store pre-computed values without defining extra types for only this purpose.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.SamplerType","page":"Random Numbers","title":"Random.SamplerType","category":"type","text":"SamplerType{T}()\n\nA sampler for types, containing no other information. The default fallback for Sampler when called with types.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.SamplerTrivial","page":"Random Numbers","title":"Random.SamplerTrivial","category":"type","text":"SamplerTrivial(x)\n\nCreate a sampler that just wraps the given value x. This is the default fall-back for values. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values without precomputed data.\n\n\n\n\n\n"},{"location":"stdlib/Random.html#Random.SamplerSimple","page":"Random Numbers","title":"Random.SamplerSimple","category":"type","text":"SamplerSimple(x, data)\n\nCreate a sampler that wraps the given value x and the data. The eltype of this sampler is equal to eltype(x).\n\nThe recommended use case is sampling from values with precomputed data.\n\n\n\n\n\n"},{"location":"stdlib/Downloads.html#Downloads","page":"Downloads","title":"Downloads","category":"section","text":""},{"location":"stdlib/Downloads.html#Downloads.download","page":"Downloads","title":"Downloads.download","category":"function","text":"download(url, [ output = tempname() ];\n    [ method = \"GET\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ downloader = <default>, ]\n) -> output\n\n    url        :: AbstractString\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (total::Integer, now::Integer) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    downloader :: Downloader\n\nDownload a file from the given url, saving it to output or if not specified, a temporary path. The output can also be an IO handle, in which case the body of the response is streamed to that handle and the handle is returned. If output is a command, the command is run and output is sent to it on stdin.\n\nIf the downloader keyword argument is provided, it must be a Downloader object. Resources and connections will be shared between downloads performed by the same Downloader and cleaned up automatically when the object is garbage collected or there have been no downloads performed with it for a grace period. See Downloader for more info about configuration and usage.\n\nIf the headers keyword argument is provided, it must be a vector or dictionary whose elements are all pairs of strings. These pairs are passed as headers when downloading URLs with protocols that supports them, such as HTTP/S.\n\nThe timeout keyword argument specifies a timeout for the download to complete in seconds, with a resolution of milliseconds. By default no timeout is set, but this can also be explicitly requested by passing a timeout value of Inf. Separately, if 20 seconds elapse without receiving any data, the download will timeout. See extended help for how to disable this timeout.\n\nIf the progress keyword argument is provided, it must be a callback function which will be called whenever there are updates about the size and status of the ongoing download. The callback must take two integer arguments: total and now which are the total size of the download in bytes, and the number of bytes which have been downloaded so far. Note that total starts out as zero and remains zero until the server gives an indication of the total size of the download (e.g. with a Content-Length header), which may never happen. So a well-behaved progress callback should handle a total size of zero gracefully.\n\nIf the verbose option is set to true, libcurl, which is used to implement the download functionality will print debugging information to stderr. If the debug option is set to a function accepting two String arguments, then the verbose option is ignored and instead the data that would have been printed to stderr is passed to the debug callback with type and message arguments. The type argument indicates what kind of event has occurred, and is one of: TEXT, HEADER IN, HEADER OUT, DATA IN, DATA OUT, SSL DATA IN or SSL DATA OUT. The message argument is the description of the debug event.\n\nExtended Help\n\nFor further customization, use a Downloader and easy_hooks. For example, to disable the 20 second timeout when no data is received, you may use the following:\n\ndownloader = Downloads.Downloader()\ndownloader.easy_hook = (easy, info) -> Downloads.Curl.setopt(easy, Downloads.Curl.CURLOPT_LOW_SPEED_TIME, 0)\n\nDownloads.download(\"https://httpbingo.julialang.org/delay/30\"; downloader)\n\n\n\n\n\n"},{"location":"stdlib/Downloads.html#Downloads.request","page":"Downloads","title":"Downloads.request","category":"function","text":"request(url;\n    [ input = <none>, ]\n    [ output = <none>, ]\n    [ method = input ? \"PUT\" : output ? \"GET\" : \"HEAD\", ]\n    [ headers = <none>, ]\n    [ timeout = <none>, ]\n    [ progress = <none>, ]\n    [ verbose = false, ]\n    [ debug = <none>, ]\n    [ throw = true, ]\n    [ downloader = <default>, ]\n    [ interrupt = <none>, ]\n) -> Union{Response, RequestError}\n\n    url        :: AbstractString\n    input      :: Union{AbstractString, AbstractCmd, IO}\n    output     :: Union{AbstractString, AbstractCmd, IO}\n    method     :: AbstractString\n    headers    :: Union{AbstractVector, AbstractDict}\n    timeout    :: Real\n    progress   :: (dl_total, dl_now, ul_total, ul_now) --> Any\n    verbose    :: Bool\n    debug      :: (type, message) --> Any\n    throw      :: Bool\n    downloader :: Downloader\n    interrupt  :: Base.Event\n\nMake a request to the given url, returning a Response object capturing the status, headers and other information about the response. The body of the response is written to output if specified and discarded otherwise. For HTTP/S requests, if an input stream is given, a PUT request is made; otherwise if an output stream is given, a GET request is made; if neither is given a HEAD request is made. For other protocols, appropriate default methods are used based on what combination of input and output are requested. The following options differ from the download function:\n\ninput allows providing a request body; if provided default to PUT request\nprogress is a callback taking four integers for upload and download progress\nthrow controls whether to throw or return a RequestError on request error\n\nNote that unlike download which throws an error if the requested URL could not be downloaded (indicated by non-2xx status code), request returns a Response object no matter what the status code of the response is. If there is an error with getting a response at all, then a RequestError is thrown or returned.\n\nIf the interrupt keyword argument is provided, it must be a Base.Event object. If the event is triggered while the request is in progress, the request will be cancelled and an error will be thrown. This can be used to interrupt a long running request, for example if the user wants to cancel a download.\n\n\n\n\n\n"},{"location":"stdlib/Downloads.html#Downloads.Response","page":"Downloads","title":"Downloads.Response","category":"type","text":"struct Response\n    proto   :: String\n    url     :: String\n    status  :: Int\n    message :: String\n    headers :: Vector{Pair{String,String}}\nend\n\nResponse is a type capturing the properties of a successful response to a request as an object. It has the following fields:\n\nproto: the protocol that was used to get the response\nurl: the URL that was ultimately requested after following redirects\nstatus: the status code of the response, indicating success, failure, etc.\nmessage: a textual message describing the nature of the response\nheaders: any headers that were returned with the response\n\nThe meaning and availability of some of these responses depends on the protocol used for the request. For many protocols, including HTTP/S and S/FTP, a 2xx status code indicates a successful response. For responses in protocols that do not support headers, the headers vector will be empty. HTTP/2 does not include a status message, only a status code, so the message will be empty.\n\n\n\n\n\n"},{"location":"stdlib/Downloads.html#Downloads.RequestError","page":"Downloads","title":"Downloads.RequestError","category":"type","text":"struct RequestError <: ErrorException\n    url      :: String\n    code     :: Int\n    message  :: String\n    response :: Response\nend\n\nRequestError is a type capturing the properties of a failed response to a request as an exception object:\n\nurl: the original URL that was requested without any redirects\ncode: the libcurl error code; 0 if a protocol-only error occurred\nmessage: the libcurl error message indicating what went wrong\nresponse: response object capturing what response info is available\n\nThe same RequestError type is thrown by download if the request was successful but there was a protocol-level error indicated by a status code that is not in the 2xx range, in which case code will be zero and the message field will be the empty string. The request API only throws a RequestError if the libcurl error code is non-zero, in which case the included response object is likely to have a status of zero and an empty message. There are, however, situations where a curl-level error is thrown due to a protocol error, in which case both the inner and outer code and message may be of interest.\n\n\n\n\n\n"},{"location":"stdlib/Downloads.html#Downloads.Downloader","page":"Downloads","title":"Downloads.Downloader","category":"type","text":"Downloader(; [ grace::Real = 30 ])\n\nDownloader objects are used to perform individual download operations. Connections, name lookups and other resources are shared within a Downloader. These connections and resources are cleaned up after a configurable grace period (default: 30 seconds) since anything was downloaded with it, or when it is garbage collected, whichever comes first. If the grace period is set to zero, all resources will be cleaned up immediately as soon as there are no more ongoing downloads in progress. If the grace period is set to Inf then resources are not cleaned up until Downloader is garbage collected.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#Sparse-Arrays","page":"Sparse Arrays","title":"Sparse Arrays","category":"section","text":"Julia has support for sparse vectors and sparse matrices in the SparseArrays stdlib module. Sparse arrays are arrays that contain enough zeros that storing them in a special data structure leads to savings in space and execution time, compared to dense arrays.\n\nExternal packages which implement different sparse storage types, multidimensional sparse arrays, and more can be found in Noteworthy External Sparse Packages"},{"location":"stdlib/SparseArrays.html#man-csc","page":"Sparse Arrays","title":"Compressed Sparse Column (CSC) Sparse Matrix Storage","category":"section","text":"In Julia, sparse matrices are stored in the Compressed Sparse Column (CSC) format. Julia sparse matrices have the type SparseMatrixCSC{Tv,Ti}, where Tv is the type of the stored values, and Ti is the integer type for storing column pointers and row indices. The internal representation of SparseMatrixCSC is as follows:\n\nstruct SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n    m::Int                  # Number of rows\n    n::Int                  # Number of columns\n    colptr::Vector{Ti}      # Column j is in colptr[j]:(colptr[j+1]-1)\n    rowval::Vector{Ti}      # Row indices of stored values\n    nzval::Vector{Tv}       # Stored values, typically nonzeros\nend\n\nThe compressed sparse column storage makes it easy and quick to access the elements in the column of a sparse matrix, whereas accessing the sparse matrix by rows is considerably slower. Operations such as insertion of previously unstored entries one at a time in the CSC structure tend to be slow. This is because all elements of the sparse matrix that are beyond the point of insertion have to be moved one place over.\n\nAll operations on sparse matrices are carefully implemented to exploit the CSC data structure for performance, and to avoid expensive operations.\n\nIf you have data in CSC format from a different application or library, and wish to import it in Julia, make sure that you use 1-based indexing. The row indices in every column need to be sorted, and if they are not, the matrix will display incorrectly.  If your SparseMatrixCSC object contains unsorted row indices, one quick way to sort them is by doing a double transpose. Since the transpose operation is lazy, make a copy to materialize each transpose.\n\nIn some applications, it is convenient to store explicit zero values in a SparseMatrixCSC. These are accepted by functions in Base (but there is no guarantee that they will be preserved in mutating operations). Such explicitly stored zeros are treated as structural nonzeros by many routines. The nnz function returns the number of elements explicitly stored in the sparse data structure, including non-structural zeros. In order to count the exact number of numerical nonzeros, use count(!iszero, x), which inspects every stored element of a sparse matrix. dropzeros, and the in-place dropzeros!, can be used to remove stored zeros from the sparse matrix.\n\njulia> A = sparse([1, 1, 2, 3], [1, 3, 2, 3], [0, 1, 2, 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 0  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n ⋅  ⋅  1\n ⋅  2  ⋅\n ⋅  ⋅  ⋅"},{"location":"stdlib/SparseArrays.html#Sparse-Vector-Storage","page":"Sparse Arrays","title":"Sparse Vector Storage","category":"section","text":"Sparse vectors are stored in a close analog to compressed sparse column format for sparse matrices. In Julia, sparse vectors have the type SparseVector{Tv,Ti} where Tv is the type of the stored values and Ti the integer type for the indices. The internal representation is as follows:\n\nstruct SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n    n::Int              # Length of the sparse vector\n    nzind::Vector{Ti}   # Indices of stored values\n    nzval::Vector{Tv}   # Stored values, typically nonzeros\nend\n\nLike SparseMatrixCSC, the SparseVector type can also contain explicitly stored zeros. (See Sparse Matrix Storage.)."},{"location":"stdlib/SparseArrays.html#Sparse-Vector-and-Matrix-Constructors","page":"Sparse Arrays","title":"Sparse Vector and Matrix Constructors","category":"section","text":"The simplest way to create a sparse array is to use a function equivalent to the zeros function that Julia provides for working with dense arrays. To produce a sparse array instead, you can use the same name with an sp prefix:\n\njulia> spzeros(3)\n3-element SparseVector{Float64, Int64} with 0 stored entries\n\nThe sparse function is often a handy way to construct sparse arrays. For example, to construct a sparse matrix we can input a vector I of row indices, a vector J of column indices, and a vector V of stored values (this is also known as the COO (coordinate) format). sparse(I,J,V) then constructs a sparse matrix such that S[I[k], J[k]] = V[k]. The equivalent sparse vector constructor is sparsevec, which takes the (row) index vector I and the vector V with the stored values and constructs a sparse vector R such that R[I[k]] = V[k].\n\njulia> I = [1, 4, 3, 5]; J = [4, 7, 18, 9]; V = [1, 2, -5, 3];\n\njulia> S = sparse(I,J,V)\n5×18 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n⎡⠀⠈⠀⠀⠀⠀⠀⠀⢀⎤\n⎣⠀⠀⠀⠂⡀⠀⠀⠀⠀⎦\n\njulia> R = sparsevec(I,V)\n5-element SparseVector{Int64, Int64} with 4 stored entries:\n  [1]  =  1\n  [3]  =  -5\n  [4]  =  2\n  [5]  =  3\n\nThe inverse of the sparse and sparsevec functions is findnz, which retrieves the inputs used to create the sparse array (including stored entries equal to zero). findall(!iszero, x) returns the Cartesian indices of non-zero entries in x (not including stored entries equal to zero).\n\njulia> findnz(S)\n([1, 4, 5, 3], [4, 7, 9, 18], [1, 2, 3, -5])\n\njulia> findall(!iszero, S)\n4-element Vector{CartesianIndex{2}}:\n CartesianIndex(1, 4)\n CartesianIndex(4, 7)\n CartesianIndex(5, 9)\n CartesianIndex(3, 18)\n\njulia> findnz(R)\n([1, 3, 4, 5], [1, -5, 2, 3])\n\njulia> findall(!iszero, R)\n4-element Vector{Int64}:\n 1\n 3\n 4\n 5\n\nAnother way to create a sparse array is to convert a dense array into a sparse array using the sparse function:\n\njulia> sparse(Matrix(1.0I, 5, 5))\n5×5 SparseMatrixCSC{Float64, Int64} with 5 stored entries:\n 1.0   ⋅    ⋅    ⋅    ⋅\n  ⋅   1.0   ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅   1.0   ⋅\n  ⋅    ⋅    ⋅    ⋅   1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\nYou can go in the other direction using the Array constructor. The issparse function can be used to query if a matrix is sparse.\n\njulia> issparse(spzeros(5))\ntrue"},{"location":"stdlib/SparseArrays.html#Sparse-matrix-operations","page":"Sparse Arrays","title":"Sparse matrix operations","category":"section","text":"Arithmetic operations on sparse matrices also work as they do on dense matrices. Indexing of, assignment into, and concatenation of sparse matrices work in the same way as dense matrices. Indexing operations, especially assignment, are expensive, when carried out one element at a time. In many cases it may be better to convert the sparse matrix into (I,J,V) format using findnz, manipulate the values or the structure in the dense vectors (I,J,V), and then reconstruct the sparse matrix."},{"location":"stdlib/SparseArrays.html#Correspondence-of-dense-and-sparse-methods","page":"Sparse Arrays","title":"Correspondence of dense and sparse methods","category":"section","text":"The following table gives a correspondence between built-in methods on sparse matrices and their corresponding methods on dense matrix types. In general, methods that generate sparse matrices differ from their dense counterparts in that the resulting matrix follows the same sparsity pattern as a given sparse matrix S, or that the resulting sparse matrix has density d, i.e. each matrix element has a probability d of being non-zero.\n\nDetails can be found in the Sparse Vectors and Matrices section of the standard library reference.\n\nSparse Dense Description\nspzeros(m,n) zeros(m,n) Creates a m-by-n matrix of zeros. (spzeros(m,n) is empty.)\nsparse(I,n,n) Matrix(I,n,n) Creates a n-by-n identity matrix.\nsparse(A) Array(S) Interconverts between dense and sparse formats.\nsprand(m,n,d) rand(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed uniformly on the half-open interval 0 1).\nsprandn(m,n,d) randn(m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements distributed according to the standard normal (Gaussian) distribution.\nsprandn(rng,m,n,d) randn(rng,m,n) Creates a m-by-n random matrix (of density d) with iid non-zero elements generated with the rng random number generator"},{"location":"stdlib/SparseArrays.html#stdlib-sparse-arrays","page":"Sparse Arrays","title":"SparseArrays API","category":"section","text":""},{"location":"stdlib/SparseArrays.html#Noteworthy-External-Sparse-Packages","page":"Sparse Arrays","title":"Noteworthy External Sparse Packages","category":"section","text":"Several other Julia packages provide sparse matrix implementations that should be mentioned:\n\nSuiteSparseGraphBLAS.jl is a wrapper over the fast, multithreaded SuiteSparse:GraphBLAS C library. On CPU this is typically the fastest option, often significantly outperforming MKLSparse.\nCUDA.jl exposes the CUSPARSE library for GPU sparse matrix operations.\nSparseMatricesCSR.jl provides a Julia native implementation of the Compressed Sparse Rows (CSR) format.\nMKLSparse.jl accelerates SparseArrays sparse-dense matrix operations using Intel's MKL library.\nSparseArrayKit.jl available for multidimensional sparse arrays.\nLuxurySparse.jl provides static sparse array formats, as well as a coordinate format.\nExtendableSparse.jl enables fast insertion into sparse matrices using a lazy approach to new stored indices.\nFinch.jl supports extensive multidimensional sparse array formats and operations through a mini tensor language and compiler, all in native Julia. Support for COO, CSF, CSR, CSC and more, as well as operations like broadcast, reduce, etc. and custom operations.\n\nExternal packages providing sparse direct solvers:\n\nKLU.jl\nPardiso.jl\n\nExternal packages providing solvers for iterative solution of eigensystems and singular value decompositions:\n\nArnoldiMethods.jl\nKrylovKit\nArpack.jl\n\nExternal packages for working with graphs:\n\nGraphs.jl"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseArray","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseArray","category":"type","text":"AbstractSparseArray{Tv,Ti,N}\n\nSupertype for N-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. SparseMatrixCSC, SparseVector and SuiteSparse.CHOLMOD.Sparse are subtypes of this.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseVector","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseVector","category":"type","text":"AbstractSparseVector{Tv,Ti}\n\nSupertype for one-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,1}.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.AbstractSparseMatrix","page":"Sparse Arrays","title":"SparseArrays.AbstractSparseMatrix","category":"type","text":"AbstractSparseMatrix{Tv,Ti}\n\nSupertype for two-dimensional sparse arrays (or array-like types) with elements of type Tv and index type Ti. Alias for AbstractSparseArray{Tv,Ti,2}.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.SparseVector","page":"Sparse Arrays","title":"SparseArrays.SparseVector","category":"type","text":"SparseVector{Tv,Ti<:Integer} <: AbstractSparseVector{Tv,Ti}\n\nVector type for storing sparse vectors. Can be created by passing the length of the vector, a sorted vector of non-zero indices, and a vector of non-zero values.\n\nFor instance, the vector [5, 6, 0, 7] can be represented as\n\nSparseVector(4, [1, 2, 4], [5, 6, 7])\n\nThis indicates that the element at index 1 is 5, at index 2 is 6, at index 3 is zero(Int), and at index 4 is 7.\n\nIt may be more convenient to create sparse vectors directly from dense vectors using sparse as\n\nsparse([5, 6, 0, 7])\n\nyields the same sparse vector.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.SparseMatrixCSC","page":"Sparse Arrays","title":"SparseArrays.SparseMatrixCSC","category":"type","text":"SparseMatrixCSC{Tv,Ti<:Integer} <: AbstractSparseMatrixCSC{Tv,Ti}\n\nMatrix type for storing sparse matrices in the Compressed Sparse Column format. The standard way of constructing SparseMatrixCSC is through the sparse function. See also spzeros, spdiagm and sprand.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse","page":"Sparse Arrays","title":"SparseArrays.sparse","category":"function","text":"sparse(A::Union{AbstractVector, AbstractMatrix})\n\nConvert a vector or matrix A into a sparse array. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> A = Matrix(1.0I, 3, 3)\n3×3 Matrix{Float64}:\n 1.0  0.0  0.0\n 0.0  1.0  0.0\n 0.0  0.0  1.0\n\njulia> sparse(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   1.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> [1.0, 0.0, 1.0]\n3-element Vector{Float64}:\n 1.0\n 0.0\n 1.0\n\njulia> sparse([1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\nsparse(I, J, V,[ m, n, combine])\n\nCreate a sparse matrix S of dimensions m x n such that S[I[k], J[k]] = V[k]. The combine function is used to combine duplicates. If m and n are not specified, they are set to maximum(I) and maximum(J) respectively. If the combine function is not supplied, combine defaults to + unless the elements of V are Booleans in which case combine defaults to |. All elements of I must satisfy 1 <= I[k] <= m, and all elements of J must satisfy 1 <= J[k] <= n. Numerical zeros in (I, J, V) are retained as structural nonzeros; to drop numerical zeros, use dropzeros!.\n\nFor additional documentation and an expert driver, see SparseArrays.sparse!.\n\nExamples\n\njulia> Is = [1; 2; 3];\n\njulia> Js = [1; 2; 3];\n\njulia> Vs = [1; 2; 3];\n\njulia> sparse(Is, Js, Vs)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse!","page":"Sparse Arrays","title":"SparseArrays.sparse!","category":"function","text":"sparse!(I::AbstractVector{Ti}, J::AbstractVector{Ti}, V::AbstractVector{Tv},\n        m::Integer, n::Integer, combine, klasttouch::Vector{Ti},\n        csrrowptr::Vector{Ti}, csrcolval::Vector{Ti}, csrnzval::Vector{Tv},\n        [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}] ) where {Tv,Ti<:Integer}\n\nParent of and expert driver for sparse; see sparse for basic usage. This method allows the user to provide preallocated storage for sparse's intermediate objects and result as described below. This capability enables more efficient successive construction of SparseMatrixCSCs from coordinate representations, and also enables extraction of an unsorted-column representation of the result's transpose at no additional cost.\n\nThis method consists of three major steps: (1) Counting-sort the provided coordinate representation into an unsorted-row CSR form including repeated entries. (2) Sweep through the CSR form, simultaneously calculating the desired CSC form's column-pointer array, detecting repeated entries, and repacking the CSR form with repeated entries combined; this stage yields an unsorted-row CSR form with no repeated entries. (3) Counting-sort the preceding CSR form into a fully-sorted CSC form with no repeated entries.\n\nInput arrays csrrowptr, csrcolval, and csrnzval constitute storage for the intermediate CSR forms and require length(csrrowptr) >= m + 1, length(csrcolval) >= length(I), and length(csrnzval >= length(I)). Input array klasttouch, workspace for the second stage, requires length(klasttouch) >= n. Optional input arrays csccolptr, cscrowval, and cscnzval constitute storage for the returned CSC form S. If necessary, these are resized automatically to satisfy length(csccolptr) = n + 1, length(cscrowval) = nnz(S) and length(cscnzval) = nnz(S); hence, if nnz(S) is unknown at the outset, passing in empty vectors of the appropriate type (Vector{Ti}() and Vector{Tv}() respectively) suffices, or calling the sparse! method neglecting cscrowval and cscnzval.\n\nOn return, csrrowptr, csrcolval, and csrnzval contain an unsorted-column representation of the result's transpose.\n\nYou may reuse the input arrays' storage (I, J, V) for the output arrays (csccolptr, cscrowval, cscnzval). For example, you may call sparse!(I, J, V, csrrowptr, csrcolval, csrnzval, I, J, V). Note that they will be resized to satisfy the conditions above.\n\nFor the sake of efficiency, this method performs no argument checking beyond 1 <= I[k] <= m and 1 <= J[k] <= n. Use with care. Testing with --check-bounds=yes is wise.\n\nThis method runs in O(m, n, length(I)) time. The HALFPERM algorithm described in F. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3), 250-269 (1978) inspired this method's use of a pair of counting sorts.\n\n\n\n\n\nSparseArrays.sparse!(I, J, V, [m, n, combine]) -> SparseMatrixCSC\n\nVariant of sparse! that re-uses the input vectors (I, J, V) for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I, S.rowval === J, and S.nzval === V holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around sparse!(I, J, V, m, n, combine, klasttouch, csrrowptr, csrcolval, csrnzval, csccolptr, cscrowval, cscnzval) where this method allocates klasttouch, csrrowptr, csrcolval, and csrnzval of appropriate size, but reuses I, J, and V for csccolptr, cscrowval, and cscnzval.\n\nArguments m, n, and combine defaults to maximum(I), maximum(J), and +, respectively.\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparsevec","page":"Sparse Arrays","title":"SparseArrays.sparsevec","category":"function","text":"sparsevec(I, V, [m, combine])\n\nCreate a sparse vector S of length m such that S[I[k]] = V[k]. Duplicates are combined using the combine function, which defaults to + if no combine argument is provided, unless the elements of V are Booleans in which case combine defaults to |.\n\nExamples\n\njulia> II = [1, 3, 3, 5]; V = [0.1, 0.2, 0.3, 0.2];\n\njulia> sparsevec(II, V)\n5-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  0.5\n  [5]  =  0.2\n\njulia> sparsevec(II, V, 8, -)\n8-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  0.1\n  [3]  =  -0.1\n  [5]  =  0.2\n\njulia> sparsevec([1, 3, 1, 2, 2], [true, true, false, false, false])\n3-element SparseVector{Bool, Int64} with 3 stored entries:\n  [1]  =  1\n  [2]  =  0\n  [3]  =  1\n\n\n\n\n\nsparsevec(d::Dict, [m])\n\nCreate a sparse vector of length m where the nonzero indices are keys from the dictionary, and the nonzero values are the values from the dictionary.\n\nExamples\n\njulia> sparsevec(Dict(1 => 3, 2 => 2))\n2-element SparseVector{Int64, Int64} with 2 stored entries:\n  [1]  =  3\n  [2]  =  2\n\n\n\n\n\nsparsevec(A)\n\nConvert a vector A into a sparse vector of length m. Numerical zeros in A are turned into structural zeros.\n\nExamples\n\njulia> sparsevec([1.0, 2.0, 0.0, 0.0, 3.0, 0.0])\n6-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  2.0\n  [5]  =  3.0\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#Base.similar-Tuple{SparseArrays.AbstractSparseMatrixCSC, Type}","page":"Sparse Arrays","title":"Base.similar","category":"method","text":"similar(A::AbstractSparseMatrixCSC{Tv,Ti}, [::Type{TvNew}, ::Type{TiNew}, m::Integer, n::Integer]) where {Tv,Ti}\n\nCreate an uninitialized mutable array with the given element type, index type, and size, based upon the given source SparseMatrixCSC. The new sparse matrix maintains the structure of the original sparse matrix, except in the case where dimensions of the output matrix are different from the output.\n\nThe output matrix has zeros in the same locations as the input, but uninitialized values for the nonzero locations.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.issparse","page":"Sparse Arrays","title":"SparseArrays.issparse","category":"function","text":"issparse(S)\n\nReturns true if S is sparse, and false otherwise.\n\nExamples\n\njulia> sv = sparsevec([1, 4], [2.3, 2.2], 10)\n10-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  2.3\n  [4]  =  2.2\n\njulia> issparse(sv)\ntrue\n\njulia> issparse(Array(sv))\nfalse\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nnz","page":"Sparse Arrays","title":"SparseArrays.nnz","category":"function","text":"nnz(A)\n\nReturns the number of stored (filled) elements in a sparse array.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nnz(A)\n3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.findnz","page":"Sparse Arrays","title":"SparseArrays.findnz","category":"function","text":"findnz(A::SparseMatrixCSC)\n\nReturn a tuple (I, J, V) where I and J are the row and column indices of the stored (\"structurally non-zero\") values in sparse matrix A, and V is a vector of the values.\n\nExamples\n\njulia> A = sparse([1 2 0; 0 0 3; 0 4 0])\n3×3 SparseMatrixCSC{Int64, Int64} with 4 stored entries:\n 1  2  ⋅\n ⋅  ⋅  3\n ⋅  4  ⋅\n\njulia> findnz(A)\n([1, 1, 3, 2], [1, 2, 2, 3], [1, 2, 4, 3])\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spzeros","page":"Sparse Arrays","title":"SparseArrays.spzeros","category":"function","text":"spzeros([type,]m[,n])\n\nCreate a sparse vector of length m or sparse matrix of size m x n. This sparse array will not contain any nonzero values. No storage will be allocated for nonzero values during construction. The type defaults to Float64 if not specified.\n\nExamples\n\njulia> spzeros(3, 3)\n3×3 SparseMatrixCSC{Float64, Int64} with 0 stored entries:\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅    ⋅\n\njulia> spzeros(Float32, 4)\n4-element SparseVector{Float32, Int64} with 0 stored entries\n\n\n\n\n\nspzeros([type], I::AbstractVector, J::AbstractVector, [m, n])\n\nCreate a sparse matrix S of dimensions m x n with structural zeros at S[I[k], J[k]].\n\nThis method can be used to construct the sparsity pattern of the matrix, and is more efficient than using e.g. sparse(I, J, zeros(length(I))).\n\nFor additional documentation and an expert driver, see SparseArrays.spzeros!.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spzeros!","page":"Sparse Arrays","title":"SparseArrays.spzeros!","category":"function","text":"spzeros!(::Type{Tv}, I::AbstractVector{Ti}, J::AbstractVector{Ti}, m::Integer, n::Integer,\n         klasttouch::Vector{Ti}, csrrowptr::Vector{Ti}, csrcolval::Vector{Ti},\n         [csccolptr::Vector{Ti}], [cscrowval::Vector{Ti}, cscnzval::Vector{Tv}]) where {Tv,Ti<:Integer}\n\nParent of and expert driver for spzeros(I, J) allowing user to provide preallocated storage for intermediate objects. This method is to spzeros what SparseArrays.sparse! is to sparse. See documentation for SparseArrays.sparse! for details and required buffer lengths.\n\ncompat: Julia 1.10\nThis methods requires Julia version 1.10 or later.\n\n\n\n\n\nSparseArrays.spzeros!(::Type{Tv}, I, J, [m, n]) -> SparseMatrixCSC{Tv}\n\nVariant of spzeros! that re-uses the input vectors I and J for the final matrix storage. After construction the input vectors will alias the matrix buffers; S.colptr === I and S.rowval === J holds, and they will be resize!d as necessary.\n\nNote that some work buffers will still be allocated. Specifically, this method is a convenience wrapper around spzeros!(Tv, I, J, m, n, klasttouch, csrrowptr, csrcolval, csccolptr, cscrowval) where this method allocates klasttouch, csrrowptr, and csrcolval of appropriate size, but reuses I and J for csccolptr and cscrowval.\n\nArguments m and n defaults to maximum(I) and maximum(J).\n\ncompat: Julia 1.10\nThis method requires Julia version 1.10 or later.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.spdiagm","page":"Sparse Arrays","title":"SparseArrays.spdiagm","category":"function","text":"spdiagm(kv::Pair{<:Integer,<:AbstractVector}...)\nspdiagm(m::Integer, n::Integer, kv::Pair{<:Integer,<:AbstractVector}...)\n\nConstruct a sparse diagonal matrix from Pairs of vectors and diagonals. Each vector kv.second will be placed on the kv.first diagonal.  By default, the matrix is square and its size is inferred from kv, but a non-square size m×n (padded with zeros as needed) can be specified by passing m,n as the first arguments.\n\nExamples\n\njulia> spdiagm(-1 => [1,2,3,4], 1 => [4,3,2,1])\n5×5 SparseMatrixCSC{Int64, Int64} with 8 stored entries:\n ⋅  4  ⋅  ⋅  ⋅\n 1  ⋅  3  ⋅  ⋅\n ⋅  2  ⋅  2  ⋅\n ⋅  ⋅  3  ⋅  1\n ⋅  ⋅  ⋅  4  ⋅\n\n\n\n\n\nspdiagm(v::AbstractVector)\nspdiagm(m::Integer, n::Integer, v::AbstractVector)\n\nConstruct a sparse matrix with elements of the vector as diagonal elements. By default (no given m and n), the matrix is square and its size is given by length(v), but a non-square size m×n can be specified by passing m and n as the first arguments.\n\ncompat: Julia 1.6\nThese functions require at least Julia 1.6.\n\nExamples\n\njulia> spdiagm([1,2,3])\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 1  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  3\n\njulia> spdiagm(sparse([1,0,3]))\n3×3 SparseMatrixCSC{Int64, Int64} with 2 stored entries:\n 1  ⋅  ⋅\n ⋅  ⋅  ⋅\n ⋅  ⋅  3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_hcat","page":"Sparse Arrays","title":"SparseArrays.sparse_hcat","category":"function","text":"sparse_hcat(A...)\n\nConcatenate along dimension 2. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_vcat","page":"Sparse Arrays","title":"SparseArrays.sparse_vcat","category":"function","text":"sparse_vcat(A...)\n\nConcatenate along dimension 1. Return a SparseMatrixCSC object.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sparse_hvcat","page":"Sparse Arrays","title":"SparseArrays.sparse_hvcat","category":"function","text":"sparse_hvcat(rows::Tuple{Vararg{Int}}, values...)\n\nSparse horizontal and vertical concatenation in one call. This function is called for block matrix syntax. The first argument specifies the number of arguments to concatenate in each block row.\n\ncompat: Julia 1.8\nThis method was added in Julia 1.8. It mimics previous concatenation behavior, where the concatenation with specialized \"sparse\" matrix types from LinearAlgebra.jl automatically yielded sparse output even in the absence of any SparseArray argument.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.blockdiag","page":"Sparse Arrays","title":"SparseArrays.blockdiag","category":"function","text":"blockdiag(A...)\n\nConcatenate matrices block-diagonally. Currently only implemented for sparse matrices.\n\nExamples\n\njulia> blockdiag(sparse(2I, 3, 3), sparse(4I, 2, 2))\n5×5 SparseMatrixCSC{Int64, Int64} with 5 stored entries:\n 2  ⋅  ⋅  ⋅  ⋅\n ⋅  2  ⋅  ⋅  ⋅\n ⋅  ⋅  2  ⋅  ⋅\n ⋅  ⋅  ⋅  4  ⋅\n ⋅  ⋅  ⋅  ⋅  4\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sprand","page":"Sparse Arrays","title":"SparseArrays.sprand","category":"function","text":"sprand([rng],[T::Type],m,[n],p::AbstractFloat)\nsprand([rng],m,[n],p::AbstractFloat,[rfn=rand])\n\nCreate a random length m sparse vector or m by n sparse matrix, in which the probability of any element being nonzero is independently given by p (and hence the mean density of nonzeros is also exactly p). The optional rng argument specifies a random number generator, see Random Numbers. The optional T argument specifies the element type, which defaults to Float64.\n\nBy default, nonzero values are sampled from a uniform distribution using the rand function, i.e. by rand(T), or rand(rng, T) if rng is supplied; for the default T=Float64, this corresponds to nonzero values sampled uniformly in [0,1).\n\nYou can sample nonzero values from a different distribution by passing a custom rfn function instead of rand.   This should be a function rfn(k) that returns an array of k random numbers sampled from the desired distribution; alternatively, if rng is supplied, it should instead be a function rfn(rng, k).\n\nExamples\n\njulia> sprand(Bool, 2, 2, 0.5)\n2×2 SparseMatrixCSC{Bool, Int64} with 2 stored entries:\n 1  1\n ⋅  ⋅\n\njulia> sprand(Float64, 3, 0.75)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  0.795547\n  [2]  =  0.49425\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.sprandn","page":"Sparse Arrays","title":"SparseArrays.sprandn","category":"function","text":"sprandn([rng][,Type],m[,n],p::AbstractFloat)\n\nCreate a random sparse vector of length m or sparse matrix of size m by n with the specified (independent) probability p of any entry being nonzero, where nonzero values are sampled from the normal distribution. The optional rng argument specifies a random number generator, see Random Numbers.\n\ncompat: Julia 1.1\nSpecifying the output element type Type requires at least Julia 1.1.\n\nExamples\n\njulia> sprandn(2, 2, 0.75)\n2×2 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n -1.20577     ⋅\n  0.311817  -0.234641\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nonzeros","page":"Sparse Arrays","title":"SparseArrays.nonzeros","category":"function","text":"nonzeros(A)\n\nReturn a vector of the structural nonzero values in sparse array A. This includes zeros that are explicitly stored in the sparse array. The returned vector points directly to the internal nonzero storage of A, and any modifications to the returned vector will mutate A as well. See rowvals and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> nonzeros(A)\n3-element Vector{Int64}:\n 2\n 2\n 2\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.rowvals","page":"Sparse Arrays","title":"SparseArrays.rowvals","category":"function","text":"rowvals(A)\n\nReturn a vector of the row indices of sparse array A. Any modifications to the returned vector will mutate A as well. Providing access to how the row indices are stored internally can be useful in conjunction with iterating over structural nonzero values. See also nonzeros and nzrange.\n\nExamples\n\njulia> A = sparse(2I, 3, 3)\n3×3 SparseMatrixCSC{Int64, Int64} with 3 stored entries:\n 2  ⋅  ⋅\n ⋅  2  ⋅\n ⋅  ⋅  2\n\njulia> rowvals(A)\n3-element Vector{Int64}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.nzrange","page":"Sparse Arrays","title":"SparseArrays.nzrange","category":"function","text":"nzrange(A, col::Integer)\n\nReturn the range of indices to the structural nonzero values of column col of sparse array A. In conjunction with nonzeros and rowvals, this allows for convenient iterating over a sparse matrix :\n\nA = sparse(I,J,V)\nrows = rowvals(A)\nvals = nonzeros(A)\nm, n = size(A)\nfor j = 1:n\n   for i in nzrange(A, j)\n      row = rows[i]\n      val = vals[i]\n      # perform sparse wizardry...\n   end\nend\n\nwarning: Warning\nAdding or removing nonzero elements to the matrix may invalidate the nzrange, one should not mutate the matrix while iterating.\n\n\n\n\n\nnzrange(x::SparseVectorUnion, col)\n\nGive the range of indices to the structural nonzero values of a sparse vector. The column index col is ignored (assumed to be 1).\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.droptol!","page":"Sparse Arrays","title":"SparseArrays.droptol!","category":"function","text":"droptol!(A::AbstractSparseMatrixCSC, tol)\n\nRemoves stored values from A whose absolute value is less than or equal to tol.\n\n\n\n\n\ndroptol!(x::AbstractCompressedVector, tol)\n\nRemoves stored values from x whose absolute value is less than or equal to tol.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.dropzeros!","page":"Sparse Arrays","title":"SparseArrays.dropzeros!","category":"function","text":"dropzeros!(x::AbstractCompressedVector)\n\nRemoves stored numerical zeros from x.\n\nFor an out-of-place version, see dropzeros. For algorithmic information, see fkeep!.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.dropzeros","page":"Sparse Arrays","title":"SparseArrays.dropzeros","category":"function","text":"dropzeros(A::AbstractSparseMatrixCSC;)\n\nGenerates a copy of A and removes stored numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparse([1, 2, 3], [1, 2, 3], [1.0, 0.0, 1.0])\n3×3 SparseMatrixCSC{Float64, Int64} with 3 stored entries:\n 1.0   ⋅    ⋅\n  ⋅   0.0   ⋅\n  ⋅    ⋅   1.0\n\njulia> dropzeros(A)\n3×3 SparseMatrixCSC{Float64, Int64} with 2 stored entries:\n 1.0   ⋅    ⋅\n  ⋅    ⋅    ⋅\n  ⋅    ⋅   1.0\n\n\n\n\n\ndropzeros(x::AbstractCompressedVector)\n\nGenerates a copy of x and removes numerical zeros from that copy.\n\nFor an in-place version and algorithmic information, see dropzeros!.\n\nExamples\n\njulia> A = sparsevec([1, 2, 3], [1.0, 0.0, 1.0])\n3-element SparseVector{Float64, Int64} with 3 stored entries:\n  [1]  =  1.0\n  [2]  =  0.0\n  [3]  =  1.0\n\njulia> dropzeros(A)\n3-element SparseVector{Float64, Int64} with 2 stored entries:\n  [1]  =  1.0\n  [3]  =  1.0\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.permute","page":"Sparse Arrays","title":"SparseArrays.permute","category":"function","text":"permute(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n        q::AbstractVector{<:Integer}) where {Tv,Ti}\n\nBilaterally permute A, returning PAQ (A[p,q]). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nFor expert drivers and additional information, see permute!.\n\nExamples\n\njulia> A = spdiagm(0 => [1, 2, 3, 4], 1 => [5, 6, 7])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n 1  5  ⋅  ⋅\n ⋅  2  6  ⋅\n ⋅  ⋅  3  7\n ⋅  ⋅  ⋅  4\n\njulia> permute(A, [4, 3, 2, 1], [1, 2, 3, 4])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  ⋅  4\n ⋅  ⋅  3  7\n ⋅  2  6  ⋅\n 1  5  ⋅  ⋅\n\njulia> permute(A, [1, 2, 3, 4], [4, 3, 2, 1])\n4×4 SparseMatrixCSC{Int64, Int64} with 7 stored entries:\n ⋅  ⋅  5  1\n ⋅  6  2  ⋅\n 7  3  ⋅  ⋅\n 4  ⋅  ⋅  ⋅\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#Base.permute!-Union{Tuple{Tq}, Tuple{Tp}, Tuple{Ti}, Tuple{Tv}, Tuple{SparseMatrixCSC{Tv, Ti}, SparseMatrixCSC{Tv, Ti}, AbstractVector{Tp}, AbstractVector{Tq}}} where {Tv, Ti, Tp<:Integer, Tq<:Integer}","page":"Sparse Arrays","title":"Base.permute!","category":"method","text":"permute!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti},\n         p::AbstractVector{<:Integer}, q::AbstractVector{<:Integer},\n         [C::AbstractSparseMatrixCSC{Tv,Ti}]) where {Tv,Ti}\n\nBilaterally permute A, storing result PAQ (A[p,q]) in X. Stores intermediate result (AQ)^T (transpose(A[:,q])) in optional argument C if present. Requires that none of X, A, and, if present, C alias each other; to store result PAQ back into A, use the following method lacking X:\n\npermute!(A::AbstractSparseMatrixCSC{Tv,Ti}, p::AbstractVector{<:Integer},\n         q::AbstractVector{<:Integer}[, C::AbstractSparseMatrixCSC{Tv,Ti},\n         [workcolptr::Vector{Ti}]]) where {Tv,Ti}\n\nX's dimensions must match those of A (size(X, 1) == size(A, 1) and size(X, 2) == size(A, 2)), and X must have enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column count (length(q) == size(A, 2)). Row-permutation p's length must match A's row count (length(p) == size(A, 1)).\n\nC's dimensions must match those of transpose(A) (size(C, 1) == size(A, 2) and size(C, 2) == size(A, 1)), and C must have enough storage to accommodate all allocated entries in A (length(rowvals(C)) >= nnz(A) and length(nonzeros(C)) >= nnz(A)).\n\nFor additional (algorithmic) information, and for versions of these methods that forgo argument checking, see (unexported) parent methods unchecked_noalias_permute! and unchecked_aliasing_permute!.\n\nSee also permute.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.halfperm!","page":"Sparse Arrays","title":"SparseArrays.halfperm!","category":"function","text":"halfperm!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{TvA,Ti},\n          q::AbstractVector{<:Integer}, f::Function = identity) where {Tv,TvA,Ti}\n\nColumn-permute and transpose A, simultaneously applying f to each entry of A, storing the result (f(A)Q)^T (map(f, transpose(A[:,q]))) in X.\n\nElement type Tv of X must match f(::TvA), where TvA is the element type of A. X's dimensions must match those of transpose(A) (size(X, 1) == size(A, 2) and size(X, 2) == size(A, 1)), and X must have enough storage to accommodate all allocated entries in A (length(rowvals(X)) >= nnz(A) and length(nonzeros(X)) >= nnz(A)). Column-permutation q's length must match A's column count (length(q) == size(A, 2)).\n\nThis method is the parent of several methods performing transposition and permutation operations on SparseMatrixCSCs. As this method performs no argument checking, prefer the safer child methods ([c]transpose[!], permute[!]) to direct use.\n\nThis method implements the HALFPERM algorithm described in F. Gustavson, \"Two fast algorithms for sparse matrices: multiplication and permuted transposition,\" ACM TOMS 4(3), 250-269 (1978). The algorithm runs in O(size(A, 1), size(A, 2), nnz(A)) time and requires no space beyond that passed in.\n\n\n\n\n\n"},{"location":"stdlib/SparseArrays.html#SparseArrays.ftranspose!","page":"Sparse Arrays","title":"SparseArrays.ftranspose!","category":"function","text":"ftranspose!(X::AbstractSparseMatrixCSC{Tv,Ti}, A::AbstractSparseMatrixCSC{Tv,Ti}, f::Function) where {Tv,Ti}\n\nTranspose A and store it in X while applying the function f to the non-zero elements. Does not remove the zeros created by f. size(X) must be equal to size(transpose(A)). No additional memory is allocated other than resizing the rowval and nzval of X, if needed.\n\nSee halfperm!\n\n\n\n\n\n"},{"location":"devdocs/EscapeAnalysis.html#EscapeAnalysis","page":"EscapeAnalysis","title":"EscapeAnalysis","category":"section","text":"Compiler.EscapeAnalysis is a compiler utility module that aims to analyze escape information of Julia's SSA-form IR a.k.a. IRCode.\n\nThis escape analysis aims to:\n\nleverage Julia's high-level semantics, especially reason about escapes and aliasing via inter-procedural calls\nbe versatile enough to be used for various optimizations including alias-aware SROA, early finalize insertion, copy-free ImmutableArray construction, stack allocation of mutable objects, and so on.\nachieve a simple implementation based on a fully backward data-flow analysis implementation as well as a new lattice design that combines orthogonal lattice properties"},{"location":"devdocs/EscapeAnalysis.html#Try-it-out!","page":"EscapeAnalysis","title":"Try it out!","category":"section","text":"You can give a try to the escape analysis by loading the EAUtils.jl utility script that defines the convenience entries code_escapes and @code_escapes for testing and debugging purposes:\n\n# InteractiveUtils.@activate Compiler # to use the stdlib version of the Compiler\n\nlet JULIA_DIR = normpath(Sys.BINDIR, Base.DATAROOTDIR, \"julia\")\n    include(normpath(JULIA_DIR, \"Compiler\", \"test\", \"EAUtils.jl\"))\n    using .EAUtils\nend\n\nmutable struct SafeRef{T}\n    x::T\nend\nBase.getindex(x::SafeRef) = x.x;\nBase.setindex!(x::SafeRef, v) = x.x = v;\nBase.isassigned(x::SafeRef) = true;\nget′(x) = isassigned(x) ? x[] : throw(x);\n\nresult = code_escapes((Base.RefValue{String},String,String,)) do r1, s2, s3\n    r2 = Ref(s2)\n    r3 = SafeRef(s3)\n    try\n        s1 = get′(r1)\n        ret = sizeof(s1)\n    catch err\n        global GV = err # `r1` may escape\n    end\n    s2 = get′(r2)       # `r2` doesn't escape\n    s3 = get′(r3)       # `r3` doesn't escape\n    return s2, s3, s4\nend\n\nThe symbols on the side of each call argument and SSA statements represent the following meaning:\n\n◌ (plain): this value is not analyzed because escape information of it won't be used anyway (when the object is isbitstype for example)\n✓ (green or cyan): this value never escapes (has_no_escape(result.state[x]) holds), colored blue if it has arg escape also (has_arg_escape(result.state[x]) holds)\n↑ (blue or yellow): this value can escape to the caller via return (has_return_escape(result.state[x]) holds), colored yellow if it has unhandled thrown escape also (has_thrown_escape(result.state[x]) holds)\nX (red): this value can escape to somewhere the escape analysis can't reason about like escapes to a global memory (has_all_escape(result.state[x]) holds)\n* (bold): this value's escape state is between the ReturnEscape and AllEscape in the partial order of EscapeInfo, colored yellow if it has unhandled thrown escape also (has_thrown_escape(result.state[x]) holds)\n′: this value has additional object field / array element information in its AliasInfo property\n\nEscape information of each call argument and SSA value can be inspected programmatically as like:\n\nresult.state[Core.Argument(3)] # get EscapeInfo of `s2`\n\nresult.state[Core.SSAValue(3)] # get EscapeInfo of `r3`"},{"location":"devdocs/EscapeAnalysis.html#Analysis-Design","page":"EscapeAnalysis","title":"Analysis Design","category":"section","text":""},{"location":"devdocs/EscapeAnalysis.html#Lattice-Design","page":"EscapeAnalysis","title":"Lattice Design","category":"section","text":"EscapeAnalysis is implemented as a data-flow analysis that works on a lattice of x::EscapeInfo, which is composed of the following properties:\n\nx.Analyzed::Bool: not formally part of the lattice, only indicates x has not been analyzed or not\nx.ReturnEscape::BitSet: records SSA statements where x can escape to the caller via return\nx.ThrownEscape::BitSet: records SSA statements where x can be thrown as exception (used for the exception handling described below)\nx.AliasInfo: maintains all possible values that can be aliased to fields or array elements of x (used for the alias analysis described below)\nx.ArgEscape::Int (not implemented yet): indicates it will escape to the caller through setfield! on argument(s)\n\nThese attributes can be combined to create a partial lattice that has a finite height, given the invariant that an input program has a finite number of statements, which is assured by Julia's semantics. The clever part of this lattice design is that it enables a simpler implementation of lattice operations by allowing them to handle each lattice property separately[LatticeDesign]."},{"location":"devdocs/EscapeAnalysis.html#Backward-Escape-Propagation","page":"EscapeAnalysis","title":"Backward Escape Propagation","category":"section","text":"This escape analysis implementation is based on the data-flow algorithm described in the paper[MM02]. The analysis works on the lattice of EscapeInfo and transitions lattice elements from the bottom to the top until every lattice element gets converged to a fixed point by maintaining a (conceptual) working set that contains program counters corresponding to remaining SSA statements to be analyzed. The analysis manages a single global state that tracks EscapeInfo of each argument and SSA statement, but also note that some flow-sensitivity is encoded as program counters recorded in EscapeInfo's ReturnEscape property, which can be combined with domination analysis later to reason about flow-sensitivity if necessary.\n\nOne distinctive design of this escape analysis is that it is fully backward, i.e. escape information flows from usages to definitions. For example, in the code snippet below, EA first analyzes the statement return %1 and imposes ReturnEscape on %1 (corresponding to obj), and then it analyzes %1 = %new(Base.RefValue{Base.RefValue{String}, _2})) and propagates the ReturnEscape imposed on %1 to the call argument _2 (corresponding to s):\n\ncode_escapes((Base.RefValue{String},)) do s\n    obj = Ref(s)\n    return obj\nend\n\nThe key observation here is that this backward analysis allows escape information to flow naturally along the use-def chain rather than control-flow[BackandForth]. As a result this scheme enables a simple implementation of escape analysis, e.g. PhiNode for example can be handled simply by propagating escape information imposed on a PhiNode to its predecessor values:\n\ncode_escapes((Bool, Base.RefValue{String}, Base.RefValue{String})) do cnd, s, t\n    if cnd\n        obj = Ref(s)\n    else\n        obj = Ref(t)\n    end\n    return obj\nend"},{"location":"devdocs/EscapeAnalysis.html#EA-Alias-Analysis","page":"EscapeAnalysis","title":"Alias Analysis","category":"section","text":"EscapeAnalysis implements a backward field analysis in order to reason about escapes imposed on object fields with certain accuracy, and x::EscapeInfo's x.AliasInfo property exists for this purpose. It records all possible values that can be aliased to fields of x at \"usage\" sites, and then the escape information of that recorded values are propagated to the actual field values later at \"definition\" sites. More specifically, the analysis records a value that may be aliased to a field of object by analyzing getfield call, and then it propagates its escape information to the field when analyzing %new(...) expression or setfield! call[Dynamism].\n\ncode_escapes((String,)) do s\n    obj = SafeRef(\"init\")\n    obj[] = s\n    v = obj[]\n    return v\nend\n\nIn the example above, ReturnEscape imposed on %3 (corresponding to v) is not directly propagated to %1 (corresponding to obj) but rather that ReturnEscape is only propagated to _2 (corresponding to s). Here %3 is recorded in %1's AliasInfo property as it can be aliased to the first field of %1, and then when analyzing Base.setfield!(%1, :x, _2)::String, that escape information is propagated to _2 but not to %1.\n\nSo EscapeAnalysis tracks which IR elements can be aliased across a getfield-%new/setfield! chain in order to analyze escapes of object fields, but actually this alias analysis needs to be generalized to handle other IR elements as well. This is because in Julia IR the same object is sometimes represented by different IR elements and so we should make sure that those different IR elements that actually can represent the same object share the same escape information. IR elements that return the same object as their operand(s), such as PiNode and typeassert, can cause that IR-level aliasing and thus requires escape information imposed on any of such aliased values to be shared between them. More interestingly, it is also needed for correctly reasoning about mutations on PhiNode. Let's consider the following example:\n\ncode_escapes((Bool, String,)) do cond, x\n    if cond\n        ϕ2 = ϕ1 = SafeRef(\"foo\")\n    else\n        ϕ2 = ϕ1 = SafeRef(\"bar\")\n    end\n    ϕ2[] = x\n    y = ϕ1[]\n    return y\nend\n\nϕ1 = %5 and ϕ2 = %6 are aliased and thus ReturnEscape imposed on %8 = Base.getfield(%6, :x)::String (corresponding to y = ϕ1[]) needs to be propagated to Base.setfield!(%5, :x, _3)::String (corresponding to ϕ2[] = x). In order for such escape information to be propagated correctly, the analysis should recognize that the predecessors of ϕ1 and ϕ2 can be aliased as well and equalize their escape information.\n\nOne interesting property of such aliasing information is that it is not known at \"usage\" site but can only be derived at \"definition\" site (as aliasing is conceptually equivalent to assignment), and thus it doesn't naturally fit in a backward analysis. In order to efficiently propagate escape information between related values, EscapeAnalysis.jl uses an approach inspired by the escape analysis algorithm explained in an old JVM paper[JVM05]. That is, in addition to managing escape lattice elements, the analysis also maintains an \"equi\"-alias set, a disjoint set of aliased arguments and SSA statements. The alias set manages values that can be aliased to each other and allows escape information imposed on any of such aliased values to be equalized between them."},{"location":"devdocs/EscapeAnalysis.html#EA-Array-Analysis","page":"EscapeAnalysis","title":"Array Analysis","category":"section","text":"The alias analysis for object fields described above can also be generalized to analyze array operations. EscapeAnalysis implements handlings for various primitive array operations so that it can propagate escapes via arrayref-arrayset use-def chain and does not escape allocated arrays too conservatively:\n\ncode_escapes((String,)) do s\n    ary = Any[]\n    push!(ary, SafeRef(s))\n    return ary[1], length(ary)\nend\n\nIn the above example EscapeAnalysis understands that %20 and %2 (corresponding to the allocated object SafeRef(s)) are aliased via the arrayset-arrayref chain and imposes ReturnEscape on them, but not impose it on the allocated array %1 (corresponding to ary). EscapeAnalysis still imposes ThrownEscape on ary since it also needs to account for potential escapes via BoundsError, but also note that such unhandled ThrownEscape can often be ignored when optimizing the ary allocation.\n\nFurthermore, in cases when array index information as well as array dimensions can be known precisely, EscapeAnalysis is able to even reason about \"per-element\" aliasing via arrayref-arrayset chain, as EscapeAnalysis does \"per-field\" alias analysis for objects:\n\ncode_escapes((String,String)) do s, t\n    ary = Vector{Any}(undef, 2)\n    ary[1] = SafeRef(s)\n    ary[2] = SafeRef(t)\n    return ary[1], length(ary)\nend\n\nNote that ReturnEscape is only imposed on %2 (corresponding to SafeRef(s)) but not on %4 (corresponding to SafeRef(t)). This is because the allocated array's dimension and indices involved with all arrayref/arrayset operations are available as constant information and EscapeAnalysis can understand that %6 is aliased to %2 but never be aliased to %4. In this kind of case, the succeeding optimization passes will be able to replace Base.arrayref(true, %1, 1)::Any with %2 (a.k.a. \"load-forwarding\") and eventually eliminate the allocation of array %1 entirely (a.k.a. \"scalar-replacement\").\n\nWhen compared to object field analysis, where an access to object field can be analyzed trivially using type information derived by inference, array dimension isn't encoded as type information and so we need an additional analysis to derive that information. EscapeAnalysis at this moment first does an additional simple linear scan to analyze dimensions of allocated arrays before firing up the main analysis routine so that the succeeding escape analysis can precisely analyze operations on those arrays.\n\nHowever, such precise \"per-element\" alias analysis is often hard. Essentially, the main difficulty inherit to array is that array dimension and index are often non-constant:\n\nloop often produces loop-variant, non-constant array indices\n(specific to vectors) array resizing changes array dimension and invalidates its constant-ness\n\nLet's discuss those difficulties with concrete examples.\n\nIn the following example, EscapeAnalysis fails the precise alias analysis since the index at the Base.arrayset(false, %4, %8, %6)::Vector{Any} is not (trivially) constant. Especially Any[nothing, nothing] forms a loop and calls that arrayset operation in a loop, where %6 is represented as a ϕ-node value (whose value is control-flow dependent). As a result, ReturnEscape ends up imposed on both %23 (corresponding to SafeRef(s)) and %25 (corresponding to SafeRef(t)), although ideally we want it to be imposed only on %23 but not on %25:\n\ncode_escapes((String,String)) do s, t\n    ary = Any[nothing, nothing]\n    ary[1] = SafeRef(s)\n    ary[2] = SafeRef(t)\n    return ary[1], length(ary)\nend\n\nThe next example illustrates how vector resizing makes precise alias analysis hard. The essential difficulty is that the dimension of allocated array %1 is first initialized as 0, but it changes by the two :jl_array_grow_end calls afterwards. EscapeAnalysis currently simply gives up precise alias analysis whenever it encounters any array resizing operations and so ReturnEscape is imposed on both %2 (corresponding to SafeRef(s)) and %20 (corresponding to SafeRef(t)):\n\ncode_escapes((String,String)) do s, t\n    ary = Any[]\n    push!(ary, SafeRef(s))\n    push!(ary, SafeRef(t))\n    ary[1], length(ary)\nend\n\nIn order to address these difficulties, we need inference to be aware of array dimensions and propagate array dimensions in a flow-sensitive way[ArrayDimension], as well as come up with nice representation of loop-variant values.\n\nEscapeAnalysis at this moment quickly switches to the more imprecise analysis that doesn't track precise index information in cases when array dimensions or indices are trivially non constant. The switch can naturally be implemented as a lattice join operation of EscapeInfo.AliasInfo property in the data-flow analysis framework."},{"location":"devdocs/EscapeAnalysis.html#EA-Exception-Handling","page":"EscapeAnalysis","title":"Exception Handling","category":"section","text":"It would be also worth noting how EscapeAnalysis handles possible escapes via exceptions. Naively it seems enough to propagate escape information imposed on :the_exception object to all values that may be thrown in a corresponding try block. But there are actually several other ways to access to the exception object in Julia, such as Base.current_exceptions and rethrow. For example, escape analysis needs to account for potential escape of r in the example below:\n\nconst GR = Ref{Any}();\n@noinline function rethrow_escape!()\n    try\n        rethrow()\n    catch err\n        GR[] = err\n    end\nend;\nget′(x) = isassigned(x) ? x[] : throw(x);\n\ncode_escapes() do\n    r = Ref{String}()\n    local t\n    try\n        t = get′(r)\n    catch err\n        t = typeof(err)   # `err` (which `r` aliases to) doesn't escape here\n        rethrow_escape!() # but `r` escapes here\n    end\n    return t\nend\n\nIt requires a global analysis in order to correctly reason about all possible escapes via existing exception interfaces. For now we always propagate the topmost escape information to all potentially thrown objects conservatively, since such an additional analysis might not be worthwhile to do given that exception handling and error path usually don't need to be very performance sensitive, and also optimizations of error paths might be very ineffective anyway since they are often even \"unoptimized\" intentionally for latency reasons.\n\nx::EscapeInfo's x.ThrownEscape property records SSA statements where x can be thrown as an exception. Using this information EscapeAnalysis can propagate possible escapes via exceptions limitedly to only those may be thrown in each try region:\n\nresult = code_escapes((String,String)) do s1, s2\n    r1 = Ref(s1)\n    r2 = Ref(s2)\n    local ret\n    try\n        s1 = get′(r1)\n        ret = sizeof(s1)\n    catch err\n        global GV = err # will definitely escape `r1`\n    end\n    s2 = get′(r2)       # still `r2` doesn't escape fully\n    return s2\nend"},{"location":"devdocs/EscapeAnalysis.html#Analysis-Usage","page":"EscapeAnalysis","title":"Analysis Usage","category":"section","text":"analyze_escapes is the entry point to analyze escape information of SSA-IR elements.\n\nMost optimizations like SROA (sroa_pass!) are more effective when applied to an optimized source that the inlining pass (ssa_inlining_pass!) has simplified by resolving inter-procedural calls and expanding callee sources. Accordingly, analyze_escapes is also able to analyze post-inlining IR and collect escape information that is useful for certain memory-related optimizations.\n\nHowever, since certain optimization passes like inlining can change control flows and eliminate dead code, they can break the inter-procedural validity of escape information. In particularity, in order to collect inter-procedurally valid escape information, we need to analyze a pre-inlining IR.\n\nBecause of this reason, analyze_escapes can analyze IRCode at any Julia-level optimization stage, and especially, it is supposed to be used at the following two stages:\n\nIPO EA: analyze pre-inlining IR to generate IPO-valid escape information cache\nLocal EA: analyze post-inlining IR to collect locally-valid escape information\n\nEscape information derived by IPO EA is transformed to the ArgEscapeCache data structure and cached globally. By passing an appropriate get_escape_cache callback to analyze_escapes, the escape analysis can improve analysis accuracy by utilizing cached inter-procedural information of non-inlined callees that has been derived by previous IPO EA. More interestingly, it is also valid to use IPO EA escape information for type inference, e.g., inference accuracy can be improved by forming Const/PartialStruct/MustAlias of mutable object.\n\n\n\n[LatticeDesign]: Our type inference implementation takes the alternative approach, where each lattice property is represented by a special lattice element type object. It turns out that it started to complicate implementations of the lattice operations mainly because it often requires conversion rules between each lattice element type object. And we are working on overhauling our type inference lattice implementation with EscapeInfo-like lattice design.\n\n[MM02]: A Graph-Free approach to Data-Flow Analysis.      Markas Mohnen, 2002, April.      https://api.semanticscholar.org/CorpusID:28519618.\n\n[BackandForth]: Our type inference algorithm in contrast is implemented as a forward analysis, because type information usually flows from \"definition\" to \"usage\" and it is more natural and effective to propagate such information in a forward way.\n\n[Dynamism]: In some cases, however, object fields can't be analyzed precisely. For example, object may escape to somewhere EscapeAnalysis can't account for possible memory effects on it, or fields of the objects simply can't be known because of the lack of type information. In such cases AliasInfo property is raised to the topmost element within its own lattice order, and it causes succeeding field analysis to be conservative and escape information imposed on fields of an unanalyzable object to be propagated to the object itself.\n\n[JVM05]: Escape Analysis in the Context of Dynamic Compilation and Deoptimization.       Thomas Kotzmann and Hanspeter Mössenböck, 2005, June.       https://dl.acm.org/doi/10.1145/1064979.1064996.\n\n[ArrayDimension]: Otherwise we will need yet another forward data-flow analysis on top of the escape analysis."},{"location":"devdocs/EscapeAnalysis.html#Base.Compiler.EscapeAnalysis.analyze_escapes","page":"EscapeAnalysis","title":"Base.Compiler.EscapeAnalysis.analyze_escapes","category":"function","text":"analyze_escapes(ir::IRCode, nargs::Int, get_escape_cache) -> estate::EscapeState\n\nAnalyzes escape information in ir:\n\nnargs: the number of actual arguments of the analyzed call\nget_escape_cache(::MethodInstance) -> Union{Bool,ArgEscapeCache}: retrieves cached argument escape information\n\n\n\n\n\n"},{"location":"devdocs/EscapeAnalysis.html#Base.Compiler.EscapeAnalysis.EscapeState","page":"EscapeAnalysis","title":"Base.Compiler.EscapeAnalysis.EscapeState","category":"type","text":"estate::EscapeState\n\nExtended lattice that maps arguments and SSA values to escape information represented as EscapeInfo. Escape information imposed on SSA IR element x can be retrieved by estate[x].\n\n\n\n\n\n"},{"location":"devdocs/EscapeAnalysis.html#Base.Compiler.EscapeAnalysis.EscapeInfo","page":"EscapeAnalysis","title":"Base.Compiler.EscapeAnalysis.EscapeInfo","category":"type","text":"x::EscapeInfo\n\nA lattice for escape information, which holds the following properties:\n\nx.Analyzed::Bool: not formally part of the lattice, only indicates whether x has been analyzed\nx.ReturnEscape::Bool: indicates x can escape to the caller via return\nx.ThrownEscape::BitSet: records SSA statement numbers where x can be thrown as exception:\nisempty(x.ThrownEscape): x will never be thrown in this call frame (the bottom)\npc ∈ x.ThrownEscape: x may be thrown at the SSA statement at pc\n-1 ∈ x.ThrownEscape: x may be thrown at arbitrary points of this call frame (the top)\nThis information will be used by escape_exception! to propagate potential escapes via exception.\nx.AliasInfo::Union{Bool,IndexableFields,Unindexable}: maintains all possible values that can be aliased to fields or array elements of x:\nx.AliasInfo === false indicates the fields/elements of x aren't analyzed yet\nx.AliasInfo === true indicates the fields/elements of x can't be analyzed, e.g. the type of x is not known or is not concrete and thus its fields/elements can't be known precisely\nx.AliasInfo::IndexableFields records all the possible values that can be aliased to fields of object x with precise index information\nx.AliasInfo::Unindexable records all the possible values that can be aliased to fields/elements of x without precise index information\nx.Liveness::BitSet: records SSA statement numbers where x should be live, e.g. to be used as a call argument, to be returned to a caller, or preserved for :foreigncall:\nisempty(x.Liveness): x is never be used in this call frame (the bottom)\n0 ∈ x.Liveness also has the special meaning that it's a call argument of the currently analyzed call frame (and thus it's visible from the caller immediately).\npc ∈ x.Liveness: x may be used at the SSA statement at pc\n-1 ∈ x.Liveness: x may be used at arbitrary points of this call frame (the top)\n\nThere are utility constructors to create common EscapeInfos, e.g.,\n\nNoEscape(): the bottom(-like) element of this lattice, meaning it won't escape to anywhere\nAllEscape(): the topmost element of this lattice, meaning it will escape to everywhere\n\nanalyze_escapes will transition these elements from the bottom to the top, in the same direction as Julia's native type inference routine. An abstract state will be initialized with the bottom(-like) elements:\n\nthe call arguments are initialized as ArgEscape(), whose Liveness property includes 0 to indicate that it is passed as a call argument and visible from a caller immediately\nthe other states are initialized as NotAnalyzed(), which is a special lattice element that is slightly lower than NoEscape, but at the same time doesn't represent any meaning other than it's not analyzed yet (thus it's not formally part of the lattice)\n\n\n\n\n\n"},{"location":"devdocs/compiler.html#High-level-Overview-of-the-Native-Code-Generation-Process","page":"High-level Overview of the Native-Code Generation Process","title":"High-level Overview of the Native-Code Generation Process","category":"section","text":""},{"location":"devdocs/compiler.html#Representation-of-Pointers","page":"High-level Overview of the Native-Code Generation Process","title":"Representation of Pointers","category":"section","text":"When emitting code to an object file, pointers will be emitted as relocations. The deserialization code will ensure any object that pointed to one of these constants gets recreated and contains the right runtime pointer.\n\nOtherwise, they will be emitted as literal constants.\n\nTo emit one of these objects, call literal_pointer_val. It'll handle tracking the Julia value and the LLVM global, ensuring they are valid both for the current runtime and after deserialization.\n\nWhen emitted into the object file, these globals are stored as references in a large gvals table. This allows the deserializer to reference them by index, and implement a custom manual mechanism similar to a Global Offset Table (GOT) to restore them.\n\nFunction pointers are handled similarly. They are stored as values in a large fvals table. Like globals, this allows the deserializer to reference them by index.\n\nNote that extern functions are handled separately, with names, via the usual symbol resolution mechanism in the linker.\n\nNote too that ccall functions are also handled separately, via a manual GOT and Procedure Linkage Table (PLT)."},{"location":"devdocs/compiler.html#Representation-of-Intermediate-Values","page":"High-level Overview of the Native-Code Generation Process","title":"Representation of Intermediate Values","category":"section","text":"Values are passed around in a jl_cgval_t struct. This represents an R-value, and includes enough information to determine how to assign or pass it somewhere.\n\nThey are created via one of the helper constructors, usually: mark_julia_type (for immediate values) and mark_julia_slot (for pointers to values).\n\nThe function convert_julia_type can transform between any two types. It returns an R-value with cgval.typ set to typ. It'll cast the object to the requested representation, making heap boxes, allocating stack copies, and computing tagged unions as needed to change the representation.\n\nBy contrast update_julia_type will change cgval.typ to typ, only if it can be done at zero-cost (i.e. without emitting any code)."},{"location":"devdocs/compiler.html#Union-representation","page":"High-level Overview of the Native-Code Generation Process","title":"Union representation","category":"section","text":"Inferred union types may be stack allocated via a tagged type representation.\n\nThe primitive routines that need to be able to handle tagged unions are:\n\nmark-type\nload-local\nstore-local\nisa\nis\nemit_typeof\nemit_sizeof\nboxed\nunbox\nspecialized cc-ret\n\nEverything else should be possible to handle in inference by using these primitives to implement union-splitting.\n\nThe representation of the tagged-union is as a pair of < void* union, byte selector >. The selector is fixed-size as byte & 0x7f, and will union-tag the first 126 isbits. It records the one-based depth-first count into the type-union of the isbits objects inside. An index of zero indicates that the union* is actually a tagged heap-allocated jl_value_t*, and needs to be treated as normal for a boxed object rather than as a tagged union.\n\nThe high bit of the selector (byte & 0x80) can be tested to determine if the void* is actually a heap-allocated (jl_value_t*) box, thus avoiding the cost of re-allocating a box, while maintaining the ability to efficiently handle union-splitting based on the low bits.\n\nIt is guaranteed that byte & 0x7f is an exact test for the type, if the value can be represented by a tag – it will never be marked byte = 0x80. It is not necessary to also test the type-tag when testing isa.\n\nThe union* memory region may be allocated at any size. The only constraint is that it is big enough to contain the data currently specified by selector. It might not be big enough to contain the union of all types that could be stored there according to the associated Union type field. Use appropriate care when copying."},{"location":"devdocs/compiler.html#Specialized-Calling-Convention-Signature-Representation","page":"High-level Overview of the Native-Code Generation Process","title":"Specialized Calling Convention Signature Representation","category":"section","text":"A jl_returninfo_t object describes the specialized calling convention details of any callable. It can be generated from any (specTypes, rettype) pair, such as a CodeInstance, or other place they are declared. This is the expected calling convention for specptr, but other data may be stored there. Only if the function pointer stored there has the expected specialized calling convention will the corresponding flag be set in specsigflags to indicate it is useable.\n\nIf any of the arguments or return type of a method can be represented unboxed, and none are unable to be represented unboxed (such as an unbounded vararg), it will be given an optimized calling convention signature based on the specTypes and rettype values.\n\nThe general principles are that:\n\nPrimitive types get passed in int/float registers.\nTuples of VecElement types get passed in vector registers.\nStructs get passed on the stack.\nReturn values are handled similarly to arguments, with a size-cutoff at which they will instead be returned via a hidden sret argument.\n\nThe total logic for this is implemented by get_specsig_function and deserves_sret.\n\nAdditionally, if the return type is a union, it may be returned as a pair of values (a pointer and a tag). If the union values can be stack-allocated, then sufficient space to store them will also be passed as a hidden first argument. If the struct to return needs gc roots, space for those will be passed as a hidden second argument. It is up to the callee whether the returned pointer will point to this space, a boxed object, or even other constant memory."},{"location":"devdocs/llvm-passes.html#Custom-LLVM-Passes","page":"Custom LLVM Passes","title":"Custom LLVM Passes","category":"section","text":"Julia has a number of custom LLVM passes. Broadly, they can be classified into passes that are required to be run to maintain Julia semantics, and passes that take advantage of Julia semantics to optimize LLVM IR."},{"location":"devdocs/llvm-passes.html#Semantic-Passes","page":"Custom LLVM Passes","title":"Semantic Passes","category":"section","text":"These passes are used to transform LLVM IR into code that is legal to be run on a CPU. Their main purpose is to enable simpler IR to be emitted by codegen, which then enables other LLVM passes to optimize common patterns."},{"location":"devdocs/llvm-passes.html#CPUFeatures","page":"Custom LLVM Passes","title":"CPUFeatures","category":"section","text":"Filename: llvm-cpufeatures.cpp\nClass Name: CPUFeaturesPass\nOpt Name: module(CPUFeatures)\n\nThis pass lowers the julia.cpu.have_fma.(f32|f64) intrinsic to either true or false, depending on the target architecture and target features present on the function. This intrinsic is often used to determine if using algorithms dependent on fast fused multiply-add operations is better than using standard algorithms not dependent on such instructions."},{"location":"devdocs/llvm-passes.html#DemoteFloat16","page":"Custom LLVM Passes","title":"DemoteFloat16","category":"section","text":"Filename: llvm-demote-float16.cpp\nClassName: DemoteFloat16Pass\nOpt Name function(DemoteFloat16)\n\nThis pass replaces float16 operations with float32 operations on architectures that do not natively support float16 operations. This is done by inserting fpext and fptrunc instructions around any float16 operation. On architectures that do support native float16 operations, this pass is a no-op."},{"location":"devdocs/llvm-passes.html#LateGCLowering","page":"Custom LLVM Passes","title":"LateGCLowering","category":"section","text":"Filename: llvm-late-gc-lowering.cpp\nClass Name: LateLowerGCPass\nOpt Name: function(LateLowerGCFrame)\n\nThis pass performs most of the GC rooting work required to track pointers between GC safepoints. It also lowers several intrinsics to their corresponding instruction translation, and is permitted to violate the non-integral invariants previously established (pointer_from_objref is lowered to a ptrtoint instruction here). This pass typically occupies the most time out of all the custom Julia passes, due to its dataflow algorithm to minimize the number of objects live at any safepoint."},{"location":"devdocs/llvm-passes.html#FinalGCLowering","page":"Custom LLVM Passes","title":"FinalGCLowering","category":"section","text":"Filename: llvm-final-gc-lowering.cpp\nClass Name: FinalLowerGCPass\nOpt Name: module(FinalLowerGC)\n\nThis pass lowers a few last intrinsics to their final form targeting functions in the libjulia library. Separating this from LateGCLowering enables other backends (GPU compilation) to supply their own custom lowerings for these intrinsics, enabling the Julia pipeline to be used on those backends as well."},{"location":"devdocs/llvm-passes.html#RemoveNI","page":"Custom LLVM Passes","title":"RemoveNI","category":"section","text":"Filename: llvm-remove-ni.cpp\nClass Name: RemoveNIPass\nOpt Name: module(RemoveNI)\n\nThis pass removes the non-integral address spaces from the module's datalayout string. This enables the backend to lower Julia's custom address spaces directly to machine code, without a costly rewrite of every pointer operation to address space 0."},{"location":"devdocs/llvm-passes.html#SIMDLoop","page":"Custom LLVM Passes","title":"SIMDLoop","category":"section","text":"Filename: llvm-simdloop.cpp\nClass Name: LowerSIMDLoopPass\nOpt Name: loop(LowerSIMDLoop)\n\nThis pass acts as the main driver of the @simd annotation. Codegen inserts a !llvm.loopid marker at the back branch of a loop, which this pass uses to identify loops that were originally marked with @simd. Then, this pass looks for a chain of floating point operations that form a reduce and adds the contract and reassoc fast math flags to allow reassociation (and thus vectorization). This pass does not preserve either loop information nor inference correctness, so it may violate Julia semantics in surprising ways. If the loop was annotated with ivdep as well, then the pass marks the loop as having no loop-carried dependencies (the resulting behavior is undefined if the user annotation was incorrect or gets applied to the wrong loop)."},{"location":"devdocs/llvm-passes.html#LowerPTLS","page":"Custom LLVM Passes","title":"LowerPTLS","category":"section","text":"Filename: llvm-ptls.cpp\nClass Name: LowerPTLSPass\nOpt Name: module(LowerPTLSPass)\n\nThis pass lowers thread-local Julia intrinsics to assembly instructions. Julia relies on thread-local storage for garbage collection and multithreading task scheduling. When compiling code for system images and package images, this pass replaces calls to intrinsics with loads from global variables that are initialized at load time.\n\nIf codegen produces a function with a swiftself argument and calling convention, this pass assumes the swiftself argument is the pgcstack and will replace the intrinsics with that argument. Doing so provides speedups on architectures that have slow thread local storage accesses."},{"location":"devdocs/llvm-passes.html#RemoveAddrspaces","page":"Custom LLVM Passes","title":"RemoveAddrspaces","category":"section","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveAddrspacesPass\nOpt Name: module(RemoveAddrspaces)\n\nThis pass renames pointers in one address space to another address space. This is used to remove Julia-specific address spaces from LLVM IR."},{"location":"devdocs/llvm-passes.html#RemoveJuliaAddrspaces","page":"Custom LLVM Passes","title":"RemoveJuliaAddrspaces","category":"section","text":"Filename: llvm-remove-addrspaces.cpp\nClass Name: RemoveJuliaAddrspacesPass\nOpt Name: module(RemoveJuliaAddrspaces)\n\nThis pass removes Julia-specific address spaces from LLVM IR. It is mostly used for displaying LLVM IR in a less cluttered format. Internally, it is implemented off the RemoveAddrspaces pass."},{"location":"devdocs/llvm-passes.html#Multiversioning","page":"Custom LLVM Passes","title":"Multiversioning","category":"section","text":"Filename: llvm-multiversioning.cpp\nClass Name: MultiVersioningPass\nOpt Name: module(JuliaMultiVersioning)\n\nThis pass performs modifications to a module to create functions that are optimized for running on different architectures (see sysimg.md and pkgimg.md for more details). Implementation-wise, it clones functions and applies different target-specific attributes to them to allow the optimizer to use advanced features such as vectorization and instruction scheduling for that platform. It also creates some infrastructure to enable the Julia image loader to select the appropriate version of the function to call based on the architecture the loader is running on. The target-specific attributes are controlled by the julia.mv.specs module flag, which during compilation is derived from the JULIA_CPU_TARGET environment variable. The pass must also be enabled by providing a julia.mv.enable module flag with a value of 1.\n\nwarning: Warning\nUse of llvmcall with multiversioning is dangerous. llvmcall enables access to features not typically exposed by the Julia APIs, and are therefore usually not available on all architectures. If multiversioning is enabled and code generation is requested for a target architecture that does not support the feature required by an llvmcall expression, LLVM will probably error out, likely with an abort and the message LLVM ERROR: Do not know how to split the result of this operator!."},{"location":"devdocs/llvm-passes.html#GCInvariantVerifier","page":"Custom LLVM Passes","title":"GCInvariantVerifier","category":"section","text":"Filename: llvm-gc-invariant-verifier.cpp\nClass Name: GCInvariantVerifierPass\nOpt Name: module(GCInvariantVerifier)\n\nThis pass is used to verify Julia's invariants about LLVM IR. This includes things such as the nonexistence of ptrtoint in Julia's non-integral address spaces [nislides] and the existence of only blessed addrspacecast instructions (Tracked -> Derived, 0 -> Tracked, etc). It performs no transformations on IR.\n\n[nislides]: https://llvm.org/devmtg/2015-02/slides/chisnall-pointers-not-int.pdf"},{"location":"devdocs/llvm-passes.html#Optimization-Passes","page":"Custom LLVM Passes","title":"Optimization Passes","category":"section","text":"These passes are used to perform transformations on LLVM IR that LLVM will not perform itself, e.g. fast math flag propagation, escape analysis, and optimizations on Julia-specific internal functions. They use knowledge about Julia's semantics to perform these optimizations."},{"location":"devdocs/llvm-passes.html#AllocOpt","page":"Custom LLVM Passes","title":"AllocOpt","category":"section","text":"Filename: llvm-alloc-opt.cpp\nClass Name: AllocOptPass\nOpt Name: function(AllocOpt)\n\nJulia does not have the concept of a program stack as a place to allocate mutable objects. However, allocating objects on the stack reduces GC pressure and is critical for GPU compilation. Thus, AllocOpt performs heap to stack conversion of objects that it can prove do not escape the current function. It also performs a number of other optimizations on allocations, such as removing allocations that are never used, optimizing typeof calls to freshly allocated objects, and removing stores to allocations that are immediately overwritten. The escape analysis implementation is located in llvm-alloc-helpers.cpp. Currently, this pass does not use information from EscapeAnalysis.jl, though that may change in the future."},{"location":"devdocs/llvm-passes.html#PropagateJuliaAddrspaces","page":"Custom LLVM Passes","title":"PropagateJuliaAddrspaces","category":"section","text":"Filename: llvm-propagate-addrspaces.cpp\nClass Name: PropagateJuliaAddrspacesPass\nOpt Name: function(PropagateJuliaAddrspaces)\n\nThis pass is used to propagate Julia-specific address spaces through operations on pointers. LLVM is not allowed to introduce or remove addrspacecast instructions by optimizations, so this pass acts to eliminate redundant addrspace casts by replacing operations with their equivalent in a Julia address space. For more information on Julia's address spaces, see (TODO link to llvm.md)."},{"location":"devdocs/llvm-passes.html#JuliaLICM","page":"Custom LLVM Passes","title":"JuliaLICM","category":"section","text":"Filename: llvm-julia-licm.cpp\nClass Name: JuliaLICMPass\nOpt Name: loop(JuliaLICM)\n\nThis pass is used to hoist Julia-specific intrinsics out of loops. Specifically, it performs the following transformations:\n\nHoist gc_preserve_begin and sink gc_preserve_end out of loops when the preserved objects are loop-invariant.\nSince objects preserved within a loop are likely preserved for the duration of the loop, this transformation can reduce the number of gc_preserve_begin/gc_preserve_end pairs in the IR. This makes it easier for the LateLowerGCPass to identify where particular objects are preserved.\nHoist write barriers with invariant objects\nHere we assume that there are only two generations that an object can be a part of. Given that, a write barrier needs to only execute once for any pair of the same object. Thus, we can hoist write barriers out of loops when the object being written to is loop-invariant.\nHoist allocations out of loops when they do not escape the loop\nWe use a very conservative definition of escape here, the same as the one used in AllocOptPass. This transformation can reduce the number of allocations in the IR, even when an allocation escapes the function altogether.\n\nnote: Note\nThis pass is required to preserve LLVM's MemorySSA (Short Video, Longer Video) and ScalarEvolution (Newer Slides Older Slides) analyses."},{"location":"devdocs/contributing/patch-releases.html#Contributing-to-patch-releases","page":"Contributing to patch releases","title":"Contributing to patch releases","category":"section","text":"The process of creating a patch release is roughly as follows:\n\nCreate a new branch (e.g. backports-release-1.10) against the relevant minor release branch (e.g. release-1.10). Usually a corresponding pull request is created as well.\nAdd commits, nominally from master (hence \"backports\"), to that branch. See below for more information on this process.\nRun the BaseBenchmarks.jl benchmark suite and PkgEval.jl package ecosystem exerciser against that branch. Nominally BaseBenchmarks.jl and PkgEval.jl are invoked via Nanosoldier.jl from the pull request associated with the backports branch. Fix any issues.\nOnce all test and benchmark reports look good, merge the backports branch into the corresponding release branch (e.g. merge backports-release-1.10 into release-1.10).\nOpen a pull request that bumps the version of the relevant minor release to the next patch version, e.g. as in this pull request.\nPing @JuliaLang/releases to tag the patch release and update the website.\nOpen a pull request that bumps the version of the relevant minor release to the next prerelease patch version, e.g. as in this pull request.\n\nStep 2 above, i.e. backporting commits to the backports-release-X.Y branch, has largely been automated via Backporter: Backporter searches for merged pull requests with the relevant backport-X.Y tag, and attempts to cherry-pick the commits from those pull requests onto the backports-release-X.Y branch. Some commits apply successfully without intervention, others not so much. The latter commits require \"manual\" backporting, with which help is generally much appreciated. Backporter generates a report identifying those commits it managed to backport automatically and those that require manual backporting; this report is usually copied into the first post of the pull request associated with backports-release-X.Y and maintained as additional commits are automatically and/or manually backported.\n\nWhen contributing a manual backport, if you have the necessary permissions, please push the backport directly to the backports-release-X.Y branch. If you lack the relevant permissions, please open a pull request against the backports-release-X.Y branch with the manual backport. Once the manual backport is live on the backports-release-X.Y branch, please remove the backport-X.Y tag from the originating pull request for the commits."},{"location":"manual/workflow-tips.html#man-workflow-tips","page":"Workflow Tips","title":"Workflow Tips","category":"section","text":"Here are some tips for working with Julia efficiently."},{"location":"manual/workflow-tips.html#REPL-based-workflow","page":"Workflow Tips","title":"REPL-based workflow","category":"section","text":"As already elaborated in The Julia REPL, Julia's REPL provides rich functionality that facilitates an efficient interactive workflow. Here are some tips that might further enhance your experience at the command line."},{"location":"manual/workflow-tips.html#A-basic-editor/REPL-workflow","page":"Workflow Tips","title":"A basic editor/REPL workflow","category":"section","text":"The most basic Julia workflows involve using a text editor in conjunction with the julia command line.\n\nCreate a file, say Tmp.jl, and include within it\n\nmodule Tmp\n\nsay_hello() = println(\"Hello!\")\n\n# Your other definitions here\n\nend # module\n\nusing .Tmp\n\nThen, in the same directory, start the Julia REPL (using the julia command). Run the new file as follows:\n\njulia> include(\"Tmp.jl\")\n\njulia> Tmp.say_hello()\nHello!\n\nExplore ideas in the REPL. Save good ideas in Tmp.jl. To reload the file after it has been changed, just include it again.\n\nThe key in the above is that your code is encapsulated in a module. That allows you to edit struct definitions and remove methods, without restarting Julia.\n\n(Explanation: structs cannot be edited after definition, nor can methods be deleted. But you can overwrite the definition of a module, which is what we do when we re-include(\"Tmp.jl\")).\n\nIn addition, the encapsulation of code in a module protects it from being influenced by previous state in the REPL, protecting you from hard-to-detect errors."},{"location":"manual/workflow-tips.html#Browser-based-workflow","page":"Workflow Tips","title":"Browser-based workflow","category":"section","text":"There are a few ways to interact with Julia in a browser:\n\nUsing Pluto notebooks through Pluto.jl\nUsing Jupyter notebooks through IJulia.jl"},{"location":"manual/workflow-tips.html#Revise-based-workflows","page":"Workflow Tips","title":"Revise-based workflows","category":"section","text":"Whether you're at the REPL or in IJulia, you can typically improve your development experience with Revise. It is common to configure Revise to start whenever julia is started, as per the instructions in the Revise documentation. Once configured, Revise will track changes to files in any loaded modules, and to any files loaded in to the REPL with includet (but not with plain include); you can then edit the files and the changes take effect without restarting your julia session. A standard workflow is similar to the REPL-based workflow above, with the following modifications:\n\nPut your code in a module somewhere on your load path. There are several options for achieving this, of which two recommended choices are:\nFor long-term projects, use PkgTemplates:\nusing PkgTemplates\nt = Template()\nt(\"MyPkg\")\nThis will create a blank package, \"MyPkg\", in your .julia/dev directory. Note that PkgTemplates allows you to control many different options through its Template constructor.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and MyPkg/test/runtests.jl for the tests.\nFor \"throw-away\" projects, you can avoid any need for cleanup by doing your work in your temporary directory (e.g., /tmp).\nNavigate to your temporary directory and launch Julia, then do the following:\npkg> generate MyPkg            # type ] to enter pkg mode\njulia> push!(LOAD_PATH, pwd())   # hit backspace to exit pkg mode\nIf you restart your Julia session you'll have to re-issue that command modifying LOAD_PATH.\nIn step 2 below, edit MyPkg/src/MyPkg.jl to change the source code, and create any test file of your choosing.\nDevelop your package\nBefore loading any code, make sure you're running Revise: say using Revise or follow its documentation on configuring it to run automatically.\nThen navigate to the directory containing your test file (here assumed to be \"runtests.jl\") and do the following:\njulia> using MyPkg\n\njulia> include(\"runtests.jl\")\nYou can iteratively modify the code in MyPkg in your editor and re-run the tests with include(\"runtests.jl\"). You generally should not need to restart your Julia session to see the changes take effect (subject to a few limitations)."},{"location":"devdocs/eval.html#Eval-of-Julia-code","page":"Eval of Julia code","title":"Eval of Julia code","category":"section","text":"One of the hardest parts about learning how the Julia Language runs code is learning how all of the pieces work together to execute a block of code.\n\nEach chunk of code typically makes a trip through many steps with potentially unfamiliar names, such as (in no particular order): flisp, AST, C++, LLVM, eval, typeinf, macroexpand, sysimg (or system image), bootstrapping, compile, parse, execute, JIT, interpret, box, unbox, intrinsic function, and primitive function, before turning into the desired result (hopefully).\n\nsidebar: Definitions\nREPL\nREPL stands for Read-Eval-Print Loop. It's just what we call the command line environment for short.\nAST\nAbstract Syntax Tree. The AST is the digital representation of the code structure. In this form the code has been tokenized for meaning so that it is more suitable for manipulation and execution.\n\n(Image: Diagram of the compiler flow)"},{"location":"devdocs/eval.html#Julia-Execution","page":"Eval of Julia code","title":"Julia Execution","category":"section","text":"The 10,000 foot view of the whole process is as follows:\n\nThe user starts julia.\nThe C function main() from cli/loader_exe.c gets called. This function processes the command line arguments, filling in the jl_options struct and setting the variable ARGS. It then initializes Julia (by calling julia_init in init.c, which may load a previously compiled sysimg). Finally, it passes off control to Julia by calling Base._start().\nWhen _start() takes over control, the subsequent sequence of commands depends on the command line arguments given. For example, if a filename was supplied, it will proceed to execute that file. Otherwise, it will start an interactive REPL.\nSkipping the details about how the REPL interacts with the user, let's just say the program ends up with a block of code that it wants to run.\nIf the block of code to run is in a file, jl_load(char *filename) gets invoked to load the file and parse it. Each fragment of code is then passed to eval to execute.\nEach fragment of code (or AST), is handed off to eval() to turn into results.\neval() takes each code fragment and tries to run it in jl_toplevel_eval_flex().\njl_toplevel_eval_flex() decides whether the code is a \"toplevel\" action (such as using or module), which would be invalid inside a function. If so, it passes off the code to the toplevel interpreter.\njl_toplevel_eval_flex() then expands the code to eliminate any macros and to \"lower\" the AST to make it simpler to execute.\njl_toplevel_eval_flex() then uses some simple heuristics to decide whether to JIT compile the  AST or to interpret it directly.\nThe bulk of the work to interpret code is handled by eval in interpreter.c.\nIf instead, the code is compiled, the bulk of the work is handled by codegen.cpp. Whenever a  Julia function is called for the first time with a given set of argument types, type inference  will be run on that function. This information is used by the codegen step to generate  faster code.\nEventually, the user quits the REPL, or the end of the program is reached, and the _start()  method returns.\nJust before exiting, main() calls jl_atexit_hook(exit_code).  This calls Base._atexit() (which calls any functions registered to atexit() inside  Julia). Then it calls jl_gc_run_all_finalizers().  Finally, it gracefully cleans up all libuv handles and waits for them to flush and close."},{"location":"devdocs/eval.html#dev-parsing","page":"Eval of Julia code","title":"Parsing","category":"section","text":"By default, Julia uses JuliaSyntax.jl to produce the AST. Historically, it used a small lisp program written in femtolisp, the source-code for which is distributed inside Julia in src/flisp. If the JULIA_USE_FLISP_PARSER environment variable is set to 1, the old parser will be used instead."},{"location":"devdocs/eval.html#dev-macro-expansion","page":"Eval of Julia code","title":"Macro Expansion","category":"section","text":"When eval() encounters a macro, it expands that AST node before attempting to evaluate the expression. Macro expansion involves a handoff from eval() (in Julia), to the parser function jl_macroexpand() (written in flisp) to the Julia macro itself (written in - what else - Julia) via fl_invoke_julia_macro(), and back.\n\nTypically, macro expansion is invoked as a first step during a call to Meta.lower()/Core._lower(), although it can also be invoked directly by a call to macroexpand()/jl_macroexpand()."},{"location":"devdocs/eval.html#dev-type-inference","page":"Eval of Julia code","title":"Type Inference","category":"section","text":"Type inference is implemented in Julia by typeinf() in compiler/typeinfer.jl. Type inference is the process of examining a Julia function and determining bounds for the types of each of its variables, as well as bounds on the type of the return value from the function. This enables many future optimizations, such as unboxing of known immutable values, and compile-time hoisting of various run-time operations such as computing field offsets and function pointers. Type inference may also include other steps such as constant propagation and inlining.\n\nsidebar: More Definitions\nJIT\nJust-In-Time Compilation The process of generating native-machine code into memory right when it is needed.\nLLVM\nLow-Level Virtual Machine (a compiler) The Julia JIT compiler is a program/library called libLLVM. Codegen in Julia refers both to the process of taking a Julia AST and turning it into LLVM instructions, and the process of LLVM optimizing that and turning it into native assembly instructions.\nC++\nThe programming language that LLVM is implemented in, which means that codegen is also implemented in this language. The rest of Julia's library is implemented in C, in part because its smaller feature set makes it more usable as a cross-language interface layer.\nbox\nThis term is used to describe the process of taking a value and allocating a wrapper around the data that is tracked by the garbage collector (gc) and is tagged with the object's type.\nunbox\nThe reverse of boxing a value. This operation enables more efficient manipulation of data when the type of that data is fully known at compile-time (through type inference).\ngeneric function\nA Julia function composed of multiple \"methods\" that are selected for dynamic dispatch based on the argument type-signature\nanonymous function or \"method\"\nA Julia function without a name and without type-dispatch capabilities\nprimitive function\nA function implemented in C but exposed in Julia as a named function \"method\" (albeit without generic function dispatch capabilities, similar to a anonymous function)\nintrinsic function\nA low-level operation exposed as a function in Julia. These pseudo-functions implement operations on raw bits such as add and sign extend that cannot be expressed directly in any other way. Since they operate on bits directly, they must be compiled into a function and surrounded by a call to Core.Intrinsics.box(T, ...) to reassign type information to the value."},{"location":"devdocs/eval.html#dev-codegen","page":"Eval of Julia code","title":"JIT Code Generation","category":"section","text":"Codegen is the process of turning a Julia AST into native machine code.\n\nThe JIT environment is initialized by an early call to jl_init_codegen in codegen.cpp.\n\nOn demand, a Julia method is converted into a native function by the function emit_function(jl_method_instance_t*). (note, when using the MCJIT (in LLVM v3.4+), each function must be JIT into a new module.) This function recursively calls emit_expr() until the entire function has been emitted.\n\nMuch of the remaining bulk of this file is devoted to various manual optimizations of specific code patterns. For example, emit_known_call() knows how to inline many of the primitive functions (defined in builtins.c) for various combinations of argument types.\n\nOther parts of codegen are handled by various helper files:\n\ndebuginfo.cpp\nHandles backtraces for JIT functions\nccall.cpp\nHandles the ccall and llvmcall FFI, along with various abi_*.cpp files\nintrinsics.cpp\nHandles the emission of various low-level intrinsic functions\n\nsidebar: Bootstrapping\nThe process of creating a new system image is called \"bootstrapping\".The etymology of this word comes from the phrase \"pulling oneself up by the bootstraps\", and refers to the idea of starting from a very limited set of available functions and definitions and ending with the creation of a full-featured environment."},{"location":"devdocs/eval.html#dev-sysimg","page":"Eval of Julia code","title":"System Image","category":"section","text":"The system image is a precompiled archive of a set of Julia files. The sys.ji file distributed with Julia is one such system image, generated by executing the file sysimg.jl, and serializing the resulting environment (including Types, Functions, Modules, and all other defined values) into a file. Therefore, it contains a frozen version of the Main, Core, and Base modules (and whatever else was in the environment at the end of bootstrapping). This serializer/deserializer is implemented by jl_save_system_image/jl_restore_system_image in staticdata.c.\n\nIf there is no sysimg file (jl_options.image_file == NULL), this also implies that --build was given on the command line, so the final result should be a new sysimg file. During Julia initialization, minimal Core and Main modules are created. Then a file named boot.jl is evaluated from the current directory. Julia then evaluates any file given as a command line argument until it reaches the end. Finally, it saves the resulting environment to a \"sysimg\" file for use as a starting point for a future Julia run."},{"location":"manual/multi-threading.html#man-multithreading","page":"Multi-Threading","title":"Multi-Threading","category":"section","text":"Visit this blog post for a presentation of Julia multi-threading features."},{"location":"manual/multi-threading.html#Starting-Julia-with-multiple-threads","page":"Multi-Threading","title":"Starting Julia with multiple threads","category":"section","text":"By default, Julia starts up with 2 threads of execution; 1 worker thread and 1 interactive thread. This can be verified by using the command Threads.nthreads():\n\njulia> Threads.nthreads(:default)\n1\njulia> Threads.nthreads(:interactive)\n1\n\nThe number of execution threads is controlled either by using the -t/--threads command line argument or by using the JULIA_NUM_THREADS environment variable. When both are specified, then -t/--threads takes precedence.\n\nThe number of threads can either be specified as an integer (--threads=4) or as auto (--threads=auto), where auto tries to infer a useful default number of threads to use (see Command-line Options for more details).\n\nSee threadpools for how to control how many :default and :interactive threads are in each threadpool.\n\ncompat: Julia 1.5\nThe -t/--threads command line argument requires at least Julia 1.5. In older versions you must use the environment variable instead.\n\ncompat: Julia 1.7\nUsing auto as value of the environment variable JULIA_NUM_THREADS requires at least Julia 1.7. In older versions, this value is ignored.\n\ncompat: Julia 1.12\nStarting by default with 1 interactive thread, as well as the 1 worker thread, was made as such in Julia 1.12 If the number of threads is set to 1 by either doing -t1 or JULIA_NUM_THREADS=1 an interactive thread will not be spawned.\n\nLets start Julia with 4 threads:\n\n$ julia --threads 4\n\nLet's verify there are 4 threads at our disposal.\n\njulia> Threads.nthreads()\n4\n\nBut we are currently on the master thread. To check, we use the function Threads.threadid\n\njulia> Threads.threadid()\n1\n\nnote: Note\nIf you prefer to use the environment variable you can set it as follows in Bash (Linux/macOS):export JULIA_NUM_THREADS=4C shell on Linux/macOS, CMD on Windows:set JULIA_NUM_THREADS=4Powershell on Windows:$env:JULIA_NUM_THREADS=4Note that this must be done before starting Julia.\n\nnote: Note\nThe number of threads specified with -t/--threads is propagated to worker processes that are spawned using the -p/--procs or --machine-file command line options. For example, julia -p2 -t2 spawns 1 main process with 2 worker processes, and all three processes have 2 threads enabled. For more fine grained control over worker threads use addprocs and pass -t/--threads as exeflags."},{"location":"manual/multi-threading.html#Multiple-GC-Threads","page":"Multi-Threading","title":"Multiple GC Threads","category":"section","text":"The Garbage Collector (GC) can use multiple threads. The amount used by default matches the compute worker threads or can configured by either the --gcthreads command line argument or by using the JULIA_NUM_GC_THREADS environment variable.\n\ncompat: Julia 1.10\nThe --gcthreads command line argument requires at least Julia 1.10.\n\nFor more details about garbage collection configuration and performance tuning, see Memory Management and Garbage Collection."},{"location":"manual/multi-threading.html#man-threadpools","page":"Multi-Threading","title":"Threadpools","category":"section","text":"When a program's threads are busy with many tasks to run, tasks may experience delays which may negatively affect the responsiveness and interactivity of the program. To address this, you can specify that a task is interactive when you Threads.@spawn it:\n\nusing Base.Threads\n@spawn :interactive f()\n\nInteractive tasks should avoid performing high latency operations, and if they are long duration tasks, should yield frequently.\n\nBy default Julia starts with one interactive thread reserved to run interactive tasks, but that number can be controlled with:\n\n$ julia --threads 3,1\njulia> Threads.nthreads(:interactive)\n1\n\n$ julia --threads 3,0\njulia> Threads.nthreads(:interactive)\n0\n\nThe environment variable JULIA_NUM_THREADS can also be used similarly:\n\nexport JULIA_NUM_THREADS=3,1\n\nThis starts Julia with 3 threads in the :default threadpool and 1 thread in the :interactive threadpool:\n\njulia> using Base.Threads\n\njulia> nthreadpools()\n2\n\njulia> threadpool() # the main thread is in the interactive thread pool\n:interactive\n\njulia> nthreads(:default)\n3\n\njulia> nthreads(:interactive)\n1\n\njulia> nthreads()\n3\n\nnote: Note\nExplicitly asking for 1 thread by doing -t1 or JULIA_NUM_THREADS=1 does not add an interactive thread.\n\nnote: Note\nThe zero-argument version of nthreads returns the number of threads in the default pool.\n\nnote: Note\nDepending on whether Julia has been started with interactive threads, the main thread is either in the default or interactive thread pool.\n\nEither or both numbers can be replaced with the word auto, which causes Julia to choose a reasonable default."},{"location":"manual/multi-threading.html#The-@threads-Macro","page":"Multi-Threading","title":"The @threads Macro","category":"section","text":"Let's work a simple example using our native threads. Let us create an array of zeros:\n\njulia> a = zeros(10)\n10-element Vector{Float64}:\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n 0.0\n\nLet us operate on this array simultaneously using 4 threads. We'll have each thread write its thread ID into each location.\n\nJulia supports parallel loops using the Threads.@threads macro. This macro is affixed in front of a for loop to indicate to Julia that the loop is a multi-threaded region:\n\njulia> Threads.@threads for i = 1:10\n           a[i] = Threads.threadid()\n       end\n\nThe iteration space is split among the threads, after which each thread writes its thread ID to its assigned locations:\n\njulia> a\n10-element Vector{Float64}:\n 1.0\n 1.0\n 1.0\n 2.0\n 2.0\n 2.0\n 3.0\n 3.0\n 4.0\n 4.0\n\nNote that Threads.@threads does not have an optional reduction parameter like @distributed."},{"location":"manual/multi-threading.html#Using-@threads-without-data-races","page":"Multi-Threading","title":"Using @threads without data-races","category":"section","text":"The concept of a data-race is elaborated on in \"Communication and data races between threads\". For now, just know that a data race can result in incorrect results and dangerous errors.\n\nLets say we want to make the function sum_single below multithreaded.\n\njulia> function sum_single(a)\n           s = 0\n           for i in a\n               s += i\n           end\n           s\n       end\nsum_single (generic function with 1 method)\n\njulia> sum_single(1:1_000_000)\n500000500000\n\nSimply adding @threads exposes a data race with multiple threads reading and writing s at the same time.\n\njulia> function sum_multi_bad(a)\n           s = 0\n           Threads.@threads for i in a\n               s += i\n           end\n           s\n       end\nsum_multi_bad (generic function with 1 method)\n\njulia> sum_multi_bad(1:1_000_000)\n70140554652\n\nNote that the result is not 500000500000 as it should be, and will most likely change each evaluation.\n\nTo fix this, buffers that are specific to the task may be used to segment the sum into chunks that are race-free. Here sum_single is reused, with its own internal buffer s. The input vector a is split into at most nthreads() chunks for parallel work. We then use Threads.@spawn to create tasks that individually sum each chunk. Finally, we sum the results from each task using sum_single again:\n\njulia> function sum_multi_good(a)\n           chunks = Iterators.partition(a, cld(length(a), Threads.nthreads()))\n           tasks = map(chunks) do chunk\n               Threads.@spawn sum_single(chunk)\n           end\n           chunk_sums = fetch.(tasks)\n           return sum_single(chunk_sums)\n       end\nsum_multi_good (generic function with 1 method)\n\njulia> sum_multi_good(1:1_000_000)\n500000500000\n\nnote: Note\nBuffers should not be managed based on threadid() i.e. buffers = zeros(Threads.nthreads()) because concurrent tasks can yield, meaning multiple concurrent tasks may use the same buffer on a given thread, introducing risk of data races. Further, when more than one thread is available tasks may change thread at yield points, which is known as task migration.\n\nAnother option is the use of atomic operations on variables shared across tasks/threads, which may be more performant depending on the characteristics of the operations."},{"location":"manual/multi-threading.html#man-communication-and-data-races","page":"Multi-Threading","title":"Communication and data-races between threads","category":"section","text":"Although Julia's threads can communicate through shared memory, it is notoriously difficult to write correct and data-race free multi-threaded code. Julia's Channels are thread-safe and may be used to communicate safely. There are also sections below that explain how to use locks and atomics to avoid data-races.\n\nIn certain cases, Julia is able to detect a detect safety violations, in particular in regards to deadlocks or other known-unsafe operations such as yielding to the currently running task. In these cases, a ConcurrencyViolationError is thrown."},{"location":"manual/multi-threading.html#Data-race-freedom","page":"Multi-Threading","title":"Data-race freedom","category":"section","text":"You are entirely responsible for ensuring that your program is data-race free, and nothing promised here can be assumed if you do not observe that requirement. The observed results may be highly unintuitive.\n\nIf data-races are introduced, Julia is not memory safe. Be very careful about reading any data if another thread might write to it, as it could result in segmentation faults or worse. Below are a couple of unsafe ways to access global variables from different threads:\n\nThread 1:\nglobal b = false\nglobal a = rand()\nglobal b = true\n\nThread 2:\nwhile !b; end\nbad_read1(a) # it is NOT safe to access `a` here!\n\nThread 3:\nwhile !@isdefined(a); end\nbad_read2(a) # it is NOT safe to access `a` here"},{"location":"manual/multi-threading.html#man-using-locks","page":"Multi-Threading","title":"Using locks to avoid data-races","category":"section","text":"An important tool for avoiding data races, and writing thread-safe code in general, is the concept of a \"lock\". A lock can be locked and unlocked. If a thread has locked a lock, and not unlocked it, it is said to \"hold\" the lock. If there is only one lock, and we write code that requires holding the lock to access some data, we can ensure that multiple threads will never access the same data simultaneously.\n\nNote that the link between a lock and a variable is made by the programmer, and not the program. A helper-type Base.Lockable exists that helps you associate a lock and a value. This is often more safe than keeping track yourself, and is detailed under Using Base.Lockable to associate a lock and a value.\n\nFor example, we can create a lock my_lock, and lock it while we mutate a variable my_variable. This is done most simply with the @lock macro:\n\njulia> my_lock = ReentrantLock();\n\njulia> my_variable = [1, 2, 3];\n\njulia> @lock my_lock my_variable[1] = 100\n100\n\nBy using a similar pattern with the same lock and variable, but on another thread, the operations are free from data-races.\n\nWe could have performed the operation above with the functional version of lock, in the following two ways:\n\njulia> lock(my_lock) do\n           my_variable[1] = 100\n       end\n100\n\njulia> begin\n           lock(my_lock)\n           try\n               my_variable[1] = 100\n           finally\n               unlock(my_lock)\n           end\n       end\n100\n\nAll three options are equivalent. Note how the final version requires an explicit try-block to ensure that the lock is always unlocked, whereas the first two version do this internally. One should always use the lock pattern above when changing data (such as assigning to a global or closure variable) accessed by other threads. Failing to do this could have unforeseen and serious consequences."},{"location":"manual/multi-threading.html#man-lockable","page":"Multi-Threading","title":"Using Base.Lockable to associate a lock and a value","category":"section","text":"As mentioned in the previous section, the helper-type Base.Lockable can be used to programmatically ensure the association between a lock and a value. This is generally recommended, as it is both less prone to error and more readable for others compared to having the association only by convention.\n\nAny object can be wrapped in Base.Lockable:\n\njulia> my_array = [];\n\njulia> my_locked_array = Base.Lockable(my_array);\n\nIf the lock is held, the underlying object can be accessed with the empty indexing notation:\n\njulia> begin\n           lock(my_locked_array)\n           try\n               push!(my_locked_array[], 1)\n           finally\n               unlock(my_locked_array)\n           end\n       end\n1-element Vector{Any}:\n 1\n\nIt is usually easier and safer to pass a function as the first argument to lock. The function is applied to the unlocked object, and the locking/unlocking is handled automatically:\n\njulia> lock(x -> push!(x, 2), my_locked_array);\n\njulia> lock(display, my_locked_array)\n2-element Vector{Any}:\n 1\n 2\n\njulia> lock(my_locked_array) do x\n           x[1] = π\n           display(x)\n       end\n2-element Vector{Any}:\n π = 3.1415926535897...\n 2"},{"location":"manual/multi-threading.html#man-atomic-operations","page":"Multi-Threading","title":"Atomic Operations","category":"section","text":"Julia supports accessing and modifying values atomically, that is, in a thread-safe way to avoid race conditions. A value (which must be of a primitive type) can be wrapped as Threads.Atomic to indicate it must be accessed in this way. Here we can see an example:\n\njulia> i = Threads.Atomic{Int}(0);\n\njulia> ids = zeros(4);\n\njulia> old_is = zeros(4);\n\njulia> Threads.@threads for id in 1:4\n           old_is[id] = Threads.atomic_add!(i, id)\n           ids[id] = id\n       end\n\njulia> old_is\n4-element Vector{Float64}:\n 0.0\n 1.0\n 7.0\n 3.0\n\njulia> i[]\n 10\n\njulia> ids\n4-element Vector{Float64}:\n 1.0\n 2.0\n 3.0\n 4.0\n\nHad we tried to do the addition without the atomic tag, we might have gotten the wrong answer due to a race condition. An example of what would happen if we didn't avoid the race:\n\njulia> using Base.Threads\n\njulia> Threads.nthreads()\n4\n\njulia> acc = Ref(0)\nBase.RefValue{Int64}(0)\n\njulia> @threads for i in 1:1000\n          acc[] += 1\n       end\n\njulia> acc[]\n926\n\njulia> acc = Atomic{Int64}(0)\nAtomic{Int64}(0)\n\njulia> @threads for i in 1:1000\n          atomic_add!(acc, 1)\n       end\n\njulia> acc[]\n1000"},{"location":"manual/multi-threading.html#man-atomics","page":"Multi-Threading","title":"Per-field atomics","category":"section","text":"We can also use atomics on a more granular level using the @atomic, @atomicswap, @atomicreplace macros, and @atomiconce macros.\n\nSpecific details of the memory model and other details of the design are written in the Julia Atomics Manifesto, which will later be published formally.\n\nAny field in a struct declaration can be decorated with @atomic, and then any write must be marked with @atomic also, and must use one of the defined atomic orderings (:monotonic, :acquire, :release, :acquire_release, or :sequentially_consistent). Any read of an atomic field can also be annotated with an atomic ordering constraint, or will be done with monotonic (relaxed) ordering if unspecified.\n\ncompat: Julia 1.7\nPer-field atomics requires at least Julia 1.7."},{"location":"manual/multi-threading.html#Side-effects-and-mutable-function-arguments","page":"Multi-Threading","title":"Side effects and mutable function arguments","category":"section","text":"When using multi-threading we have to be careful when using functions that are not pure as we might get a wrong answer. For instance functions that have a name ending with ! by convention modify their arguments and thus are not pure."},{"location":"manual/multi-threading.html#@threadcall","page":"Multi-Threading","title":"@threadcall","category":"section","text":"External libraries, such as those called via ccall, pose a problem for Julia's task-based I/O mechanism. If a C library performs a blocking operation, that prevents the Julia scheduler from executing any other tasks until the call returns. (Exceptions are calls into custom C code that call back into Julia, which may then yield, or C code that calls jl_yield(), the C equivalent of yield.)\n\nThe @threadcall macro provides a way to avoid stalling execution in such a scenario. It schedules a C function for execution in a separate thread. A threadpool with a default size of 4 is used for this. The size of the threadpool is controlled via environment variable UV_THREADPOOL_SIZE. While waiting for a free thread, and during function execution once a thread is available, the requesting task (on the main Julia event loop) yields to other tasks. Note that @threadcall does not return until the execution is complete. From a user point of view, it is therefore a blocking call like other Julia APIs.\n\nIt is very important that the called function does not call back into Julia, as it will segfault.\n\n@threadcall may be removed/changed in future versions of Julia."},{"location":"manual/multi-threading.html#Caveats","page":"Multi-Threading","title":"Caveats","category":"section","text":"At this time, most operations in the Julia runtime and standard libraries can be used in a thread-safe manner, if the user code is data-race free. However, in some areas work on stabilizing thread support is ongoing. Multi-threaded programming has many inherent difficulties, and if a program using threads exhibits unusual or undesirable behavior (e.g. crashes or mysterious results), thread interactions should typically be suspected first.\n\nThere are a few specific limitations and warnings to be aware of when using threads in Julia:\n\nBase collection types require manual locking if used simultaneously by multiple threads where at least one thread modifies the collection (common examples include push! on arrays, or inserting items into a Dict).\nThe schedule used by @spawn is nondeterministic and should not be relied on.\nCompute-bound, non-memory-allocating tasks can prevent garbage collection from running in other threads that are allocating memory. In these cases it may be necessary to insert a manual call to GC.safepoint() to allow GC to run. This limitation will be removed in the future.\nAvoid running top-level operations, e.g. include, or eval of type, method, and module definitions in parallel.\nBe aware that finalizers registered by a library may break if threads are enabled. This may require some transitional work across the ecosystem before threading can be widely adopted with confidence. See the section on the safe use of finalizers for further details."},{"location":"manual/multi-threading.html#man-task-migration","page":"Multi-Threading","title":"Task Migration","category":"section","text":"After a task starts running on a certain thread it may move to a different thread if the task yields.\n\nSuch tasks may have been started with @spawn or @threads, although the :static schedule option for @threads does freeze the threadid.\n\nThis means that in most cases threadid() should not be treated as constant within a task, and therefore should not be used to index into a vector of buffers or stateful objects.\n\ncompat: Julia 1.7\nTask migration was introduced in Julia 1.7. Before this tasks always remained on the same thread that they were started on."},{"location":"manual/multi-threading.html#man-finalizers","page":"Multi-Threading","title":"Safe use of Finalizers","category":"section","text":"Because finalizers can interrupt any code, they must be very careful in how they interact with any global state. Unfortunately, the main reason that finalizers are used is to update global state (a pure function is generally rather pointless as a finalizer). This leads us to a bit of a conundrum. There are a few approaches to dealing with this problem:\n\nWhen single-threaded, code could call the internal jl_gc_enable_finalizers C function to prevent finalizers from being scheduled inside a critical region. Internally, this is used inside some functions (such as our C locks) to prevent recursion when doing certain operations (incremental package loading, codegen, etc.). The combination of a lock and this flag can be used to make finalizers safe.\nA second strategy, employed by Base in a couple places, is to explicitly delay a finalizer until it may be able to acquire its lock non-recursively. The following example demonstrates how this strategy could be applied to Distributed.finalize_ref:\nfunction finalize_ref(r::AbstractRemoteRef)\n    if r.where > 0 # Check if the finalizer is already run\n        if islocked(client_refs) || !trylock(client_refs)\n            # delay finalizer for later if we aren't free to acquire the lock\n            finalizer(finalize_ref, r)\n            return nothing\n        end\n        try # `lock` should always be followed by `try`\n            if r.where > 0 # Must check again here\n                # Do actual cleanup here\n                r.where = 0\n            end\n        finally\n            unlock(client_refs)\n        end\n    end\n    nothing\nend\nA related third strategy is to use a yield-free queue. We don't currently have a lock-free queue implemented in Base, but Base.IntrusiveLinkedListSynchronized{T} is suitable. This can frequently be a good strategy to use for code with event loops. For example, this strategy is employed by Gtk.jl to manage lifetime ref-counting. In this approach, we don't do any explicit work inside the finalizer, and instead add it to a queue to run at a safer time. In fact, Julia's task scheduler already uses this, so defining the finalizer as x -> @spawn do_cleanup(x) is one example of this approach. Note however that this doesn't control which thread do_cleanup runs on, so do_cleanup would still need to acquire a lock. That doesn't need to be true if you implement your own queue, as you can explicitly only drain that queue from your thread."},{"location":"stdlib/Base64.html#Base64","page":"Base64","title":"Base64","category":"section","text":""},{"location":"stdlib/Base64.html#Base64.Base64","page":"Base64","title":"Base64.Base64","category":"module","text":"Base64\n\nFunctionality for base64 encoding and decoding, a method to represent binary data using text, common on the web.\n\n\n\n\n\n"},{"location":"stdlib/Base64.html#Base64.Base64EncodePipe","page":"Base64","title":"Base64.Base64EncodePipe","category":"type","text":"Base64EncodePipe(ostream)\n\nReturn a new write-only I/O stream, which converts any bytes written to it into base64-encoded ASCII bytes written to ostream.  Calling close on the Base64EncodePipe stream is necessary to complete the encoding (but does not close ostream).\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> iob64_encode = Base64EncodePipe(io);\n\njulia> write(iob64_encode, \"Hello!\")\n6\n\njulia> close(iob64_encode);\n\njulia> str = takestring!(io)\n\"SGVsbG8h\"\n\njulia> String(base64decode(str))\n\"Hello!\"\n\n\n\n\n\n"},{"location":"stdlib/Base64.html#Base64.base64encode","page":"Base64","title":"Base64.base64encode","category":"function","text":"base64encode(writefunc, args...; context=nothing)\nbase64encode(args...; context=nothing)\n\nGiven a write-like function writefunc, which takes an I/O stream as its first argument, base64encode(writefunc, args...) calls writefunc to write args... to a base64-encoded string, and returns the string. base64encode(args...) is equivalent to base64encode(write, args...): it converts its arguments into bytes using the standard write functions and returns the base64-encoded string.\n\nThe optional keyword argument context can be set to :key=>value pair or an IO or IOContext object whose attributes are used for the I/O stream passed to writefunc or write.\n\nSee also base64decode.\n\n\n\n\n\n"},{"location":"stdlib/Base64.html#Base64.Base64DecodePipe","page":"Base64","title":"Base64.Base64DecodePipe","category":"type","text":"Base64DecodePipe(istream)\n\nReturn a new read-only I/O stream, which decodes base64-encoded data read from istream.\n\nExamples\n\njulia> io = IOBuffer();\n\njulia> iob64_decode = Base64DecodePipe(io);\n\njulia> write(io, \"SGVsbG8h\")\n8\n\njulia> seekstart(io);\n\njulia> String(read(iob64_decode))\n\"Hello!\"\n\n\n\n\n\n"},{"location":"stdlib/Base64.html#Base64.base64decode","page":"Base64","title":"Base64.base64decode","category":"function","text":"base64decode(string)\n\nDecode the base64-encoded string and returns a Vector{UInt8} of the decoded bytes.\n\nSee also base64encode.\n\nExamples\n\njulia> b = base64decode(\"SGVsbG8h\")\n6-element Vector{UInt8}:\n 0x48\n 0x65\n 0x6c\n 0x6c\n 0x6f\n 0x21\n\njulia> String(b)\n\"Hello!\"\n\n\n\n\n\n"},{"location":"stdlib/Base64.html#Base64.stringmime","page":"Base64","title":"Base64.stringmime","category":"function","text":"stringmime(mime, x; context=nothing)\n\nReturn an AbstractString containing the representation of x in the requested mime type. This is similar to repr(mime, x) except that binary data is base64-encoded as an ASCII string.\n\nThe optional keyword argument context can be set to :key=>value pair or an IO or IOContext object whose attributes are used for the I/O stream passed to show.\n\n\n\n\n\n"},{"location":"base/parallel.html#Tasks","page":"Tasks","title":"Tasks","category":"section","text":""},{"location":"base/parallel.html#Scheduling","page":"Tasks","title":"Scheduling","category":"section","text":""},{"location":"base/parallel.html#lib-task-sync","page":"Tasks","title":"Synchronization","category":"section","text":""},{"location":"base/parallel.html#Channels","page":"Tasks","title":"Channels","category":"section","text":""},{"location":"base/parallel.html#low-level-schedule-wait","page":"Tasks","title":"Low-level synchronization using schedule and wait","category":"section","text":"The easiest correct use of schedule is on a Task that is not started (scheduled) yet. However, it is possible to use schedule and wait as a very low-level building block for constructing synchronization interfaces. A crucial pre-condition of calling schedule(task) is that the caller must \"own\" the task; i.e., it must know that the call to wait in the given task is happening at the locations known to the code calling schedule(task). One strategy for ensuring such pre-condition is to use atomics, as demonstrated in the following example:\n\n@enum OWEState begin\n    OWE_EMPTY\n    OWE_WAITING\n    OWE_NOTIFYING\nend\n\nmutable struct OneWayEvent\n    @atomic state::OWEState\n    task::Task\n    OneWayEvent() = new(OWE_EMPTY)\nend\n\nfunction Base.notify(ev::OneWayEvent)\n    state = @atomic ev.state\n    while state !== OWE_NOTIFYING\n        # Spin until we successfully update the state to OWE_NOTIFYING:\n        state, ok = @atomicreplace(ev.state, state => OWE_NOTIFYING)\n        if ok\n            if state == OWE_WAITING\n                # OWE_WAITING -> OWE_NOTIFYING transition means that the waiter task is\n                # already waiting or about to call `wait`. The notifier task must wake up\n                # the waiter task.\n                schedule(ev.task)\n            else\n                @assert state == OWE_EMPTY\n                # Since we are assuming that there is only one notifier task (for\n                # simplicity), we know that the other possible case here is OWE_EMPTY.\n                # We do not need to do anything because we know that the waiter task has\n                # not called `wait(ev::OneWayEvent)` yet.\n            end\n            break\n        end\n    end\n    return\nend\n\nfunction Base.wait(ev::OneWayEvent)\n    ev.task = current_task()\n    state, ok = @atomicreplace(ev.state, OWE_EMPTY => OWE_WAITING)\n    if ok\n        # OWE_EMPTY -> OWE_WAITING transition means that the notifier task is guaranteed to\n        # invoke OWE_WAITING -> OWE_NOTIFYING transition. The waiter task must call\n        # `wait()` immediately. In particular, it MUST NOT invoke any function that may\n        # yield to the scheduler at this point in code.\n        wait()\n    else\n        @assert state == OWE_NOTIFYING\n        # Otherwise, the `state` must have already been moved to OWE_NOTIFYING by the\n        # notifier task.\n    end\n    return\nend\n\nev = OneWayEvent()\n@sync begin\n    Threads.@spawn begin\n        wait(ev)\n        println(\"done\")\n    end\n    println(\"notifying...\")\n    notify(ev)\nend\n\n# output\nnotifying...\ndone\n\nOneWayEvent lets one task to wait for another task's notify. It is a limited communication interface since wait can only be used once from a single task (note the non-atomic assignment of ev.task)\n\nIn this example, notify(ev::OneWayEvent) is allowed to call schedule(ev.task) if and only if it modifies the state from OWE_WAITING to OWE_NOTIFYING. This lets us know that the task executing wait(ev::OneWayEvent) is now in the ok branch and that there cannot be other tasks that try to schedule(ev.task) since their @atomicreplace(ev.state, state => OWE_NOTIFYING) will fail."},{"location":"base/parallel.html#Core.Task","page":"Tasks","title":"Core.Task","category":"type","text":"Task(func[, reserved_stack::Int])\n\nCreate a Task (i.e. coroutine) to execute the given function func (which must be callable with no arguments). The task exits when this function returns. The task will run in the \"world age\" from the parent at construction when scheduled.\n\nThe optional reserved_stack argument specifies the size of the stack available for this task, in bytes. The default, 0, uses the system-dependent stack size default.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a() = sum(i for i in 1:1000);\n\njulia> b = Task(a);\n\nIn this example, b is a runnable Task that hasn't started yet.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.@task","page":"Tasks","title":"Base.@task","category":"macro","text":"@task\n\nWrap an expression in a Task without executing it, and return the Task. This only creates a task, and does not run it.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a1() = sum(i for i in 1:1000);\n\njulia> b = @task a1();\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.@async","page":"Tasks","title":"Base.@async","category":"macro","text":"@async\n\nWrap an expression in a Task and add it to the local machine's scheduler queue.\n\nValues can be interpolated into @async via $, which copies the value directly into the constructed underlying closure. This allows you to insert the value of a variable, isolating the asynchronous code from changes to the variable's value in the current task.\n\nwarning: Warning\nIt is strongly encouraged to favor Threads.@spawn over @async always even when no parallelism is required especially in publicly distributed libraries.  This is because a use of @async disables the migration of the parent task across worker threads in the current implementation of Julia.  Thus, seemingly innocent use of @async in a library function can have a large impact on the performance of very different parts of user applications.\n\ncompat: Julia 1.4\nInterpolating values via $ is available as of Julia 1.4.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.asyncmap","page":"Tasks","title":"Base.asyncmap","category":"function","text":"asyncmap(f, c...; ntasks=0, batch_size=nothing)\n\nUses multiple concurrent tasks to map f over a collection (or multiple equal length collections). For multiple collection arguments, f is applied elementwise.\n\nThe output is guaranteed to be the same order as the elements of the collection(s) c.\n\nntasks specifies the number of tasks to run concurrently. Depending on the length of the collections, if ntasks is unspecified, up to 100 tasks will be used for concurrent mapping.\n\nntasks can also be specified as a zero-arg function. In this case, the number of tasks to run in parallel is checked before processing every element and a new task started if the value of ntasks_func is greater than the current number of tasks.\n\nIf batch_size is specified, the collection is processed in batch mode. f must then be a function that must accept a Vector of argument tuples and must return a vector of results. The input vector will have a length of batch_size or less.\n\nThe following examples highlight execution in different tasks by returning the objectid of the tasks in which the mapping function is executed.\n\nFirst, with ntasks undefined, each element is processed in a different task.\n\njulia> tskoid() = objectid(current_task());\n\njulia> asyncmap(x->tskoid(), 1:5)\n5-element Vector{UInt64}:\n 0x6e15e66c75c75853\n 0x440f8819a1baa682\n 0x9fb3eeadd0c83985\n 0xebd3e35fe90d4050\n 0x29efc93edce2b961\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5)))\n5\n\nWith ntasks=2 all elements are processed in 2 tasks.\n\njulia> asyncmap(x->tskoid(), 1:5; ntasks=2)\n5-element Vector{UInt64}:\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n 0xa23d2f80cd7cf157\n 0x027ab1680df7ae94\n\njulia> length(unique(asyncmap(x->tskoid(), 1:5; ntasks=2)))\n2\n\nWith batch_size defined, the mapping function needs to be changed to accept an array of argument tuples and return an array of results. map is used in the modified mapping function to achieve this.\n\njulia> batch_func(input) = map(x->string(\"args_tuple: \", x, \", element_val: \", x[1], \", task: \", tskoid()), input)\nbatch_func (generic function with 1 method)\n\njulia> asyncmap(batch_func, 1:5; ntasks=2, batch_size=2)\n5-element Vector{String}:\n \"args_tuple: (1,), element_val: 1, task: 9118321258196414413\"\n \"args_tuple: (2,), element_val: 2, task: 4904288162898683522\"\n \"args_tuple: (3,), element_val: 3, task: 9118321258196414413\"\n \"args_tuple: (4,), element_val: 4, task: 4904288162898683522\"\n \"args_tuple: (5,), element_val: 5, task: 9118321258196414413\"\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.asyncmap!","page":"Tasks","title":"Base.asyncmap!","category":"function","text":"asyncmap!(f, results, c...; ntasks=0, batch_size=nothing)\n\nLike asyncmap, but stores output in results rather than returning a collection.\n\nwarning: Warning\nBehavior can be unexpected when any mutated argument shares memory with any other argument.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.current_task","page":"Tasks","title":"Base.current_task","category":"function","text":"current_task()\n\nGet the currently running Task.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.istaskdone","page":"Tasks","title":"Base.istaskdone","category":"function","text":"istaskdone(t::Task)::Bool\n\nDetermine whether a task has exited.\n\nExamples\n\njulia> a2() = sum(i for i in 1:1000);\n\njulia> b = Task(a2);\n\njulia> istaskdone(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.istaskstarted","page":"Tasks","title":"Base.istaskstarted","category":"function","text":"istaskstarted(t::Task)::Bool\n\nDetermine whether a task has started executing.\n\nExamples\n\njulia> a3() = sum(i for i in 1:1000);\n\njulia> b = Task(a3);\n\njulia> istaskstarted(b)\nfalse\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.istaskfailed","page":"Tasks","title":"Base.istaskfailed","category":"function","text":"istaskfailed(t::Task)::Bool\n\nDetermine whether a task has exited because an exception was thrown.\n\nExamples\n\njulia> a4() = error(\"task failed\");\n\njulia> b = Task(a4);\n\njulia> istaskfailed(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskfailed(b)\ntrue\n\ncompat: Julia 1.3\nThis function requires at least Julia 1.3.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.task_local_storage-Tuple{Any}","page":"Tasks","title":"Base.task_local_storage","category":"method","text":"task_local_storage(key)\n\nLook up the value of a key in the current task's task-local storage.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.task_local_storage-Tuple{Any, Any}","page":"Tasks","title":"Base.task_local_storage","category":"method","text":"task_local_storage(key, value)\n\nAssign a value to a key in the current task's task-local storage.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.task_local_storage-Tuple{Function, Any, Any}","page":"Tasks","title":"Base.task_local_storage","category":"method","text":"task_local_storage(body, key, value)\n\nCall the function body with a modified task-local storage, in which value is assigned to key; the previous value of key, or lack thereof, is restored afterwards. Useful for emulating dynamic scoping.\n\n\n\n\n\n"},{"location":"base/parallel.html#Core.ConcurrencyViolationError","page":"Tasks","title":"Core.ConcurrencyViolationError","category":"type","text":"ConcurrencyViolationError(msg) <: Exception\n\nAn error thrown when a detectable violation of concurrent semantics has occurred.\n\nA non-exhaustive list of examples of when this is used include:\n\nThrowing when a deadlock has been detected (e.g. wait(current_task()))\nKnown-unsafe behavior is attempted (e.g. yield(current_task))\nA known non-threadsafe datastructure is attempted to be modified from multiple concurrent tasks\nA lock is being unlocked that wasn't locked by this task\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.yield","page":"Tasks","title":"Base.yield","category":"function","text":"yield(t::Task, arg = nothing)\n\nA fast, unfair-scheduling version of schedule(t, arg); yield() which immediately yields to t before calling the scheduler.\n\nThrows a ConcurrencyViolationError if t is the currently running task.\n\n\n\n\n\nyield()\n\nSwitch to the scheduler to allow another scheduled task to run. A task that calls this function is still runnable, and will be restarted immediately if there are no other runnable tasks.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.yieldto","page":"Tasks","title":"Base.yieldto","category":"function","text":"yieldto(t::Task, arg = nothing)\n\nSwitch to the given task. The first time a task is switched to, the task's function is called with no arguments. On subsequent switches, arg is returned from the task's last call to yieldto. This is a low-level call that only switches tasks, not considering states or scheduling in any way. Its use is discouraged.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.sleep","page":"Tasks","title":"Base.sleep","category":"function","text":"sleep(seconds)\n\nBlock the current task for a specified number of seconds. The minimum sleep time is 1 millisecond or input of 0.001.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.schedule","page":"Tasks","title":"Base.schedule","category":"function","text":"schedule(t::Task, [val]; error=false)\n\nAdd a Task to the scheduler's queue. This causes the task to run constantly when the system is otherwise idle, unless the task performs a blocking operation such as wait.\n\nIf a second argument val is provided, it will be passed to the task (via the return value of yieldto) when it runs again. If error is true, the value is raised as an exception in the woken task.\n\nwarning: Warning\nIt is incorrect to use schedule on an arbitrary Task that has already been started. See the API reference for more information.\n\nwarning: Warning\nBy default tasks will have the sticky bit set to true t.sticky. This models the historic default for @async. Sticky tasks can only be run on the worker thread they are first scheduled on, and when scheduled will make the task that they were scheduled from sticky. To obtain the behavior of Threads.@spawn set the sticky bit manually to false.\n\nExamples\n\njulia> a5() = sum(i for i in 1:1000);\n\njulia> b = Task(a5);\n\njulia> istaskstarted(b)\nfalse\n\njulia> schedule(b);\n\njulia> yield();\n\njulia> istaskstarted(b)\ntrue\n\njulia> istaskdone(b)\ntrue\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.errormonitor","page":"Tasks","title":"Base.errormonitor","category":"function","text":"errormonitor(t::Task)\n\nPrint an error log to stderr if task t fails.\n\nExamples\n\njulia> wait(errormonitor(Threads.@spawn error(\"task failed\")); throw = false)\nUnhandled Task ERROR: task failed\nStacktrace:\n[...]\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.@sync","page":"Tasks","title":"Base.@sync","category":"macro","text":"@sync\n\nWait until all lexically-enclosed uses of @async, @spawn, Distributed.@spawnat and Distributed.@distributed are complete. All exceptions thrown by enclosed async operations are collected and thrown as a CompositeException.\n\nExamples\n\njulia> Threads.nthreads()\n4\n\njulia> @sync begin\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 1\")\n           Threads.@spawn println(\"Thread-id $(Threads.threadid()), task 2\")\n       end;\nThread-id 3, task 1\nThread-id 1, task 2\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.wait","page":"Tasks","title":"Base.wait","category":"function","text":"wait([x])\n\nBlock the current task until some event occurs.\n\nChannel: Wait for a value to be appended to the channel.\nCondition: Wait for notify on a condition and return the val parameter passed to notify. See the Condition-specific docstring of wait for the exact behavior.\nProcess: Wait for a process or process chain to exit. The exitcode field of a process can be used to determine success or failure.\nTask: Wait for a Task to finish. See the Task-specific docstring of wait for the exact behavior.\nRawFD: Wait for changes on a file descriptor (see the FileWatching package).\n\nIf no argument is passed, the task blocks for an undefined period. A task can only be restarted by an explicit call to schedule or yieldto.\n\nOften wait is called within a while loop to ensure a waited-for condition is met before proceeding.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.waitany","page":"Tasks","title":"Base.waitany","category":"function","text":"waitany(tasks; throw=true) -> (done_tasks, remaining_tasks)\n\nWait until at least one of the given tasks have been completed.\n\nIf throw is true, throw CompositeException when one of the completed tasks completes with an exception.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\nwarning: Warning\nThis may scale poorly compared to writing code that uses multiple individual tasks that each runs serially, since this needs to scan the list of tasks each time and synchronize with each one every time this is called. Or consider using waitall(tasks; failfast=true) instead.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.waitall","page":"Tasks","title":"Base.waitall","category":"function","text":"waitall(tasks; failfast=true, throw=true) -> (done_tasks, remaining_tasks)\n\nWait until all the given tasks have been completed.\n\nIf failfast is true, the function will return when at least one of the given tasks is finished by exception. If throw is true, throw CompositeException when one of the completed tasks has failed.\n\nfailfast and throw keyword arguments work independently; when only throw=true is specified, this function waits for all the tasks to complete.\n\nThe return value consists of two task vectors. The first one consists of completed tasks, and the other consists of uncompleted tasks.\n\ncompat: Julia 1.12\nThis function requires at least Julia 1.12.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.fetch-Tuple{Task}","page":"Tasks","title":"Base.fetch","category":"method","text":"fetch(t::Task)\n\nWait for a Task to finish, then return its result value. If the task fails with an exception, a TaskFailedException (which wraps the failed task) is thrown.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.fetch-Tuple{Any}","page":"Tasks","title":"Base.fetch","category":"method","text":"fetch(x::Any)\n\nReturn x.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.timedwait","page":"Tasks","title":"Base.timedwait","category":"function","text":"timedwait(testcb, timeout::Real; pollint::Real=0.1)\n\nWait until testcb() returns true or timeout seconds have passed, whichever is earlier. The test function is polled every pollint seconds. The minimum value for pollint is 0.001 seconds, that is, 1 millisecond.\n\nReturn :ok or :timed_out.\n\nExamples\n\njulia> cb() = (sleep(5); return);\n\njulia> t = @async cb();\n\njulia> timedwait(()->istaskdone(t), 1)\n:timed_out\n\njulia> timedwait(()->istaskdone(t), 6.5)\n:ok\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Condition","page":"Tasks","title":"Base.Condition","category":"type","text":"Condition()\n\nCreate an edge-triggered event source that tasks can wait for. Tasks that call wait on a Condition are suspended and queued. Tasks are woken up when notify is later called on the Condition. Waiting on a condition can return a value or raise an error if the optional arguments of notify are used. Edge triggering means that only tasks waiting at the time notify is called can be woken up. For level-triggered notifications, you must keep extra state to keep track of whether a notification has happened. The Channel and Threads.Event types do this, and can be used for level-triggered events.\n\nThis object is NOT thread-safe. See Threads.Condition for a thread-safe version.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Threads.Condition","page":"Tasks","title":"Base.Threads.Condition","category":"type","text":"Threads.Condition([lock])\n\nA thread-safe version of Base.Condition.\n\nTo call wait or notify on a Threads.Condition, you must first call lock on it. When wait is called, the lock is atomically released during blocking, and will be reacquired before wait returns. Therefore idiomatic use of a Threads.Condition c looks like the following:\n\nlock(c)\ntry\n    while !thing_we_are_waiting_for\n        wait(c)\n    end\nfinally\n    unlock(c)\nend\n\ncompat: Julia 1.2\nThis functionality requires at least Julia 1.2.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Event","page":"Tasks","title":"Base.Event","category":"type","text":"Event([autoreset=false])\n\nCreate a level-triggered event source. Tasks that call wait on an Event are suspended and queued until notify is called on the Event. After notify is called, the Event remains in a signaled state and tasks will no longer block when waiting for it, until reset is called.\n\nIf autoreset is true, at most one task will be released from wait for each call to notify.\n\nThis provides an acquire & release memory ordering on notify/wait.\n\ncompat: Julia 1.1\nThis functionality requires at least Julia 1.1.\n\ncompat: Julia 1.8\nThe autoreset functionality and memory ordering guarantee requires at least Julia 1.8.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.notify","page":"Tasks","title":"Base.notify","category":"function","text":"notify(condition, val=nothing; all=true, error=false)\n\nWake up tasks waiting for a condition, passing them val. If all is true (the default), all waiting tasks are woken, otherwise only one is. If error is true, the passed value is raised as an exception in the woken tasks.\n\nReturn the count of tasks woken up. Return 0 if no tasks are waiting on condition.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.reset-Tuple{Base.Event}","page":"Tasks","title":"Base.reset","category":"method","text":"reset(::Event)\n\nReset an Event back into an un-set state. Then any future calls to wait will block until notify is called again.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Semaphore","page":"Tasks","title":"Base.Semaphore","category":"type","text":"Semaphore(sem_size)\n\nCreate a counting semaphore that allows at most sem_size acquires to be in use at any time. Each acquire must be matched with a release.\n\nThis provides a acquire & release memory ordering on acquire/release calls.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.acquire","page":"Tasks","title":"Base.acquire","category":"function","text":"acquire(f, s::Semaphore)\n\nExecute f after acquiring from Semaphore s, and release on completion or error.\n\nFor example, a do-block form that ensures only 2 calls of foo will be active at the same time:\n\ns = Base.Semaphore(2)\n@sync for _ in 1:100\n    Threads.@spawn begin\n        Base.acquire(s) do\n            foo()\n        end\n    end\nend\n\ncompat: Julia 1.8\nThis method requires at least Julia 1.8.\n\n\n\n\n\nacquire(s::Semaphore)\n\nWait for one of the sem_size permits to be available, blocking until one can be acquired.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.release","page":"Tasks","title":"Base.release","category":"function","text":"release(s::Semaphore)\n\nReturn one permit to the pool, possibly allowing another task to acquire it and resume execution.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.AbstractLock","page":"Tasks","title":"Base.AbstractLock","category":"type","text":"AbstractLock\n\nAbstract supertype describing types that implement the synchronization primitives: lock, trylock, unlock, and islocked.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.lock","page":"Tasks","title":"Base.lock","category":"function","text":"lock(f::Function, l::Lockable)\n\nAcquire the lock associated with l, execute f with the lock held, and release the lock when f returns. f will receive one positional argument: the value wrapped by l. If the lock is already locked by a different task/thread, wait for it to become available. When this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\n\n\n\n\nlock(f::Function, lock)\n\nAcquire the lock, execute f with the lock held, and release the lock when f returns. If the lock is already locked by a different task/thread, wait for it to become available.\n\nWhen this function returns, the lock has been released, so the caller should not attempt to unlock it.\n\nSee also: @lock.\n\ncompat: Julia 1.7\nUsing a Channel as the second argument requires Julia 1.7 or later.\n\n\n\n\n\nlock(lock)\n\nAcquire the lock when it becomes available. If the lock is already locked by a different task/thread, wait for it to become available.\n\nEach lock must be matched by an unlock.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.unlock","page":"Tasks","title":"Base.unlock","category":"function","text":"unlock(lock)\n\nReleases ownership of the lock.\n\nIf this is a recursive lock which has been acquired before, decrement an internal counter and return immediately.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.trylock","page":"Tasks","title":"Base.trylock","category":"function","text":"trylock(lock) -> Success (Boolean)\n\nAcquire the lock if it is available, and return true if successful. If the lock is already locked by a different task/thread, return false.\n\nEach successful trylock must be matched by an unlock.\n\nFunction trylock combined with islocked can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.islocked","page":"Tasks","title":"Base.islocked","category":"function","text":"islocked(lock) -> Status (Boolean)\n\nCheck whether the lock is held by any task/thread. This function alone should not be used for synchronization. However, islocked combined with trylock can be used for writing the test-and-test-and-set or exponential backoff algorithms if it is supported by the typeof(lock) (read its documentation).\n\nExtended help\n\nFor example, an exponential backoff can be implemented as follows if the lock implementation satisfied the properties documented below.\n\nnspins = 0\nwhile true\n    while islocked(lock)\n        GC.safepoint()\n        nspins += 1\n        nspins > LIMIT && error(\"timeout\")\n    end\n    trylock(lock) && break\n    backoff()\nend\n\nImplementation\n\nA lock implementation is advised to define islocked with the following properties and note it in its docstring.\n\nislocked(lock) is data-race-free.\nIf islocked(lock) returns false, an immediate invocation of trylock(lock) must succeed (returns true) if there is no interference from other tasks.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.ReentrantLock","page":"Tasks","title":"Base.ReentrantLock","category":"type","text":"ReentrantLock()\n\nCreates a re-entrant lock for synchronizing Tasks. The same task can acquire the lock as many times as required (this is what the \"Reentrant\" part of the name means). Each lock must be matched with an unlock.\n\nCalling lock will also inhibit running of finalizers on that thread until the corresponding unlock. Use of the standard lock pattern illustrated below should naturally be supported, but beware of inverting the try/lock order or missing the try block entirely (e.g. attempting to return with the lock still held):\n\nThis provides a acquire/release memory ordering on lock/unlock calls.\n\nlock(l)\ntry\n    <atomic work>\nfinally\n    unlock(l)\nend\n\nIf !islocked(lck::ReentrantLock) holds, trylock(lck) succeeds unless there are other tasks attempting to hold the lock \"at the same time.\"\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.@lock","page":"Tasks","title":"Base.@lock","category":"macro","text":"@lock l expr\n\nMacro version of lock(f, l::AbstractLock) but with expr instead of f function. Expands to:\n\nlock(l)\ntry\n    expr\nfinally\n    unlock(l)\nend\n\nThis is similar to using lock with a do block, but avoids creating a closure and thus can improve the performance.\n\ncompat: Compat\n@lock was added in Julia 1.3, and exported in Julia 1.7.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Lockable","page":"Tasks","title":"Base.Lockable","category":"type","text":"Lockable(value, lock = ReentrantLock())\n\nCreates a Lockable object that wraps value and associates it with the provided lock. This object supports @lock, lock, trylock, unlock. To access the value, index the lockable object while holding the lock.\n\ncompat: Julia 1.11\nRequires at least Julia 1.11.\n\nExample\n\njulia> locked_list = Base.Lockable(Int[]);\n\njulia> @lock(locked_list, push!(locked_list[], 1)) # must hold the lock to access the value\n1-element Vector{Int64}:\n 1\n\njulia> lock(summary, locked_list)\n\"1-element Vector{Int64}\"\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.AbstractChannel","page":"Tasks","title":"Base.AbstractChannel","category":"type","text":"AbstractChannel{T}\n\nRepresentation of a channel passing objects of type T.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Channel","page":"Tasks","title":"Base.Channel","category":"type","text":"Channel{T=Any}(size::Int=0)\n\nConstructs a Channel with an internal buffer that can hold a maximum of size objects of type T. put! calls on a full channel block until an object is removed with take!.\n\nChannel(0) constructs an unbuffered channel. put! blocks until a matching take! is called. And vice-versa.\n\nOther constructors:\n\nChannel(): default constructor, equivalent to Channel{Any}(0)\nChannel(Inf): equivalent to Channel{Any}(typemax(Int))\nChannel(sz): equivalent to Channel{Any}(sz)\n\ncompat: Julia 1.3\nThe default constructor Channel() and default size=0 were added in Julia 1.3.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.Channel-Tuple{Function}","page":"Tasks","title":"Base.Channel","category":"method","text":"Channel{T=Any}(func::Function, size=0; taskref=nothing, spawn=false, threadpool=nothing)\n\nCreate a new task from func, bind it to a new channel of type T and size size, and schedule the task, all in a single call. The channel is automatically closed when the task terminates.\n\nfunc must accept the bound channel as its only argument.\n\nIf you need a reference to the created task, pass a Ref{Task} object via the keyword argument taskref.\n\nIf spawn=true, the Task created for func may be scheduled on another thread in parallel, equivalent to creating a task via Threads.@spawn.\n\nIf spawn=true and the threadpool argument is not set, it defaults to :default.\n\nIf the threadpool argument is set (to :default or :interactive), this implies that spawn=true and the new Task is spawned to the specified threadpool.\n\nReturn a Channel.\n\nExamples\n\njulia> chnl = Channel() do ch\n           foreach(i -> put!(ch, i), 1:4)\n       end;\n\njulia> typeof(chnl)\nChannel{Any}\n\njulia> for i in chnl\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\nReferencing the created task:\n\njulia> taskref = Ref{Task}();\n\njulia> chnl = Channel(taskref=taskref) do ch\n           println(take!(ch))\n       end;\n\njulia> istaskdone(taskref[])\nfalse\n\njulia> put!(chnl, \"Hello\");\nHello\n\njulia> istaskdone(taskref[])\ntrue\n\ncompat: Julia 1.3\nThe spawn= parameter was added in Julia 1.3. This constructor was added in Julia 1.3. In earlier versions of Julia, Channel used keyword arguments to set size and T, but those constructors are deprecated.\n\ncompat: Julia 1.9\nThe threadpool= argument was added in Julia 1.9.\n\njulia> chnl = Channel{Char}(1, spawn=true) do ch\n           for c in \"hello world\"\n               put!(ch, c)\n           end\n       end;\n\njulia> String(collect(chnl))\n\"hello world\"\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.put!-Tuple{Channel, Any}","page":"Tasks","title":"Base.put!","category":"method","text":"put!(c::Channel, v)\n\nAppend an item v to the channel c. Blocks if the channel is full.\n\nFor unbuffered channels, blocks until a take! is performed by a different task.\n\ncompat: Julia 1.1\nv now gets converted to the channel's type with convert as put! is called.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.take!-Tuple{Channel}","page":"Tasks","title":"Base.take!","category":"method","text":"take!(c::Channel)\n\nRemoves and returns a value from a Channel in order. Blocks until data is available. For unbuffered channels, blocks until a put! is performed by a different task.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> put!(c, 1);\n\njulia> take!(c)\n1\n\nUnbuffered channel\n\njulia> c = Channel(0);\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);\n\njulia> take!(c)\n1\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.isfull-Tuple{Channel}","page":"Tasks","title":"Base.isfull","category":"method","text":"isfull(c::Channel)\n\nDetermines if a Channel is full, in the sense that calling put!(c, some_value) would have blocked. Returns immediately, does not block.\n\nNote that it may frequently be the case that put! will not block after this returns true. Users must take precautions not to accidentally create live-lock bugs in their code by calling this method, as these are generally harder to debug than deadlocks. It is also possible that put! will block after this call returns false, if there are multiple producer tasks calling put! in parallel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1); # capacity = 1\n\njulia> isfull(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isfull(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel(); # capacity = 0\n\njulia> isfull(c) # unbuffered channel is always full\ntrue\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.isready-Tuple{Channel}","page":"Tasks","title":"Base.isready","category":"method","text":"isready(c::Channel)\n\nDetermines whether a Channel has a value stored in it. Returns immediately, does not block.\n\nFor unbuffered channels, return true if there are tasks waiting on a put!.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(1);\n\njulia> isready(c)\nfalse\n\njulia> put!(c, 1);\n\njulia> isready(c)\ntrue\n\nUnbuffered channel\n\njulia> c = Channel();\n\njulia> isready(c)  # no tasks waiting to put!\nfalse\n\njulia> task = Task(() -> put!(c, 1));\n\njulia> schedule(task);  # schedule a put! task\n\njulia> isready(c)\ntrue\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.isopen-Tuple{Channel}","page":"Tasks","title":"Base.isopen","category":"method","text":"isopen(c::Channel)\n\nDetermines whether a Channel is open for new put! operations. Notice that a Channel can be closed and still have buffered elements which can be consumed with take!.\n\nExamples\n\nBuffered channel with task\n\njulia> c = Channel(ch -> put!(ch, 1), 1);\n\njulia> isopen(c) # The channel is closed to new `put!`s\nfalse\n\njulia> isready(c) # The channel is closed but still contains elements\ntrue\n\njulia> take!(c)\n1\n\njulia> isready(c)\nfalse\n\nUnbuffered channel\n\njulia> c = Channel{Int}();\n\njulia> isopen(c)\ntrue\n\njulia> close(c)\n\njulia> isopen(c)\nfalse\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.fetch-Tuple{Channel}","page":"Tasks","title":"Base.fetch","category":"method","text":"fetch(c::Channel)\n\nWaits for and returns (without removing) the first available item from the Channel. Note: fetch is unsupported on an unbuffered (0-size) Channel.\n\nExamples\n\nBuffered channel\n\njulia> c = Channel(3) do ch\n           foreach(i -> put!(ch, i), 1:3)\n       end;\n\njulia> fetch(c)\n1\n\njulia> collect(c)  # item is not removed\n3-element Vector{Any}:\n 1\n 2\n 3\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.close-Tuple{Channel}","page":"Tasks","title":"Base.close","category":"method","text":"close(c::Channel[, excp::Exception])\n\nClose a channel. An exception (optionally given by excp), is thrown by:\n\nput! on a closed channel.\ntake! and fetch on an empty, closed channel.\n\n\n\n\n\n"},{"location":"base/parallel.html#Base.bind-Tuple{Channel, Task}","page":"Tasks","title":"Base.bind","category":"method","text":"bind(chnl::Channel, task::Task)\n\nAssociate the lifetime of chnl with a task. Channel chnl is automatically closed when the task terminates. Any uncaught exception in the task is propagated to all waiters on chnl.\n\nThe chnl object can be explicitly closed independent of task termination. Terminating tasks have no effect on already closed Channel objects.\n\nWhen a channel is bound to multiple tasks, the first task to terminate will close the channel. When multiple channels are bound to the same task, termination of the task will close all of the bound channels.\n\nExamples\n\njulia> c = Channel(0);\n\njulia> task = @async foreach(i->put!(c, i), 1:4);\n\njulia> bind(c,task);\n\njulia> for i in c\n           @show i\n       end;\ni = 1\ni = 2\ni = 3\ni = 4\n\njulia> isopen(c)\nfalse\n\njulia> c = Channel(0);\n\njulia> task = @async (put!(c, 1); error(\"foo\"));\n\njulia> bind(c, task);\n\njulia> take!(c)\n1\n\njulia> put!(c, 1);\nERROR: TaskFailedException\nStacktrace:\n[...]\n    nested task error: foo\n[...]\n\n\n\n\n\n"},{"location":"devdocs/aot.html#Ahead-of-Time-Compilation","page":"Ahead of Time Compilation","title":"Ahead of Time Compilation","category":"section","text":"This document describes the design and structure of the ahead-of-time (AOT) compilation system in Julia. This system is used when generating system images and package images. Much of the implementation described here is located in aotcompile.cpp, staticdata.c, and processor.cpp"},{"location":"devdocs/aot.html#Introduction","page":"Ahead of Time Compilation","title":"Introduction","category":"section","text":"Though Julia normally compiles code just-in-time (JIT), it is possible to compile code ahead of time and save the resulting code to a file. This can be useful for a number of reasons:\n\nTo reduce the time it takes to start a Julia process.\nTo reduce the time spent in the JIT compiler instead of executing code (time to first execution, TTFX).\nTo reduce the amount of memory used by the JIT compiler."},{"location":"devdocs/aot.html#High-Level-Overview","page":"Ahead of Time Compilation","title":"High-Level Overview","category":"section","text":"The following descriptions are a snapshot of the current implementation details of the end-to-end pipeline that happens internally when the user compiles a new AOT module, such as occurs when they type using Foo. These details are likely to change over time as we implement better ways to handle them, so current implementations may not exactly match the dataflow and functions described below."},{"location":"devdocs/aot.html#Compiling-Code-Images","page":"Ahead of Time Compilation","title":"Compiling Code Images","category":"section","text":"Firstly, the methods that need to be compiled to native code must be identified. This can only be done by actually executing the code to be compiled, as the set of methods that need to be compiled depends on the types of the arguments passed to the methods, and method invocations with certain combinations of types may not be known until runtime. During this process, the exact methods that the compiler sees are tracked for later compilation, producing a compilation trace.\n\nnote: Note\nCurrently when compiling images, Julia runs the trace generation in a different process than the process performing the AOT compilation. This can have impacts when attempting to use a debugger during precompilation. The best way to debug precompilation with a debugger is to use the rr debugger, record the entire process tree, use rr ps to identify the relevant failing process, and then use rr replay -p PID to replay just the failing process.\n\nOnce the methods to be compiled have been identified, they are passed to the jl_create_system_image function. This function sets up a number of data structures that will be used when serializing native code to a file, and then calls jl_create_native with the array of methods. jl_create_native runs codegen on the methods produces one or more LLVM modules. jl_create_system_image then records some useful information about what codegen produced from the module(s).\n\nThe module(s) are then passed to jl_dump_native, along with the information recorded by jl_create_system_image. jl_dump_native contains the code necessary to serialize the module(s) to bitcode, object, or assembly files depending on the command-line options passed to Julia. The serialized code and information are then written to a file as an archive.\n\nThe final step is to run a system linker on the object files in the archive produced by jl_dump_native. Once this step is complete, a shared library containing the compiled code is produced."},{"location":"devdocs/aot.html#Loading-Code-Images","page":"Ahead of Time Compilation","title":"Loading Code Images","category":"section","text":"When loading a code image, the shared library produced by the linker is loaded into memory. The system image data is then loaded from the shared library. This data contains information about the types, methods, and code instances that were compiled into the shared library. This data is used to restore the state of the runtime to what it was when the code image was compiled.\n\nIf the code image was compiled with multiversioning, the loader will pick the appropriate version of each function to use based on the CPU features available on the current machine.\n\nFor system images, since no other code has been loaded, the state of the runtime is now the same as it was when the code image was compiled. For package images, the environment may have changed compared to when the code was compiled, so each method must be checked against the global method table to determine if it is still valid code."},{"location":"devdocs/aot.html#Compiling-Methods","page":"Ahead of Time Compilation","title":"Compiling Methods","category":"section","text":""},{"location":"devdocs/aot.html#Tracing-Compiled-Methods","page":"Ahead of Time Compilation","title":"Tracing Compiled Methods","category":"section","text":"Julia has a command-line flag to record all of the methods that are compiled by the JIT compiler, --trace-compile=filename. When a function is compiled and this flag has a filename, Julia will print out a precompile statement to that file with the method and argument types it was called with. This therefore generates a precompile script that can be used later in the AOT compilation process. The PrecompileTools package has tooling that can make taking advantage of this functionality easier for package developers."},{"location":"devdocs/aot.html#jl_create_system_image","page":"Ahead of Time Compilation","title":"jl_create_system_image","category":"section","text":"jl_create_system_image saves all of the Julia-specific metadata necessary to later restore the state of the runtime. This includes data such as code instances, method instances, method tables, and type information. This function also sets up the data structures necessary to serialize the native code to a file. Finally, it calls jl_create_native to create one or more LLVM modules containing the native code for the methods passed to it. jl_create_native is responsible for running codegen on the methods passed to it."},{"location":"devdocs/aot.html#jl_dump_native","page":"Ahead of Time Compilation","title":"jl_dump_native","category":"section","text":"jl_dump_native is responsible for serializing the LLVM module containing the native code to a file. In addition to the module, the system image data produced by jl_create_system_image is compiled as a global variable. The output of this method is bitcode, object, and/or assembly archives containing the code and system image data.\n\njl_dump_native is typically one of the larger time sinks when emitting native code, with much of the time spent in optimizing LLVM IR and emitting machine code. Therefore, this function is capable of multithreading the optimization and machine code emission steps. This multithreading is parameterized on the size of the module, but can be explicitly overridden by setting the JULIA_IMAGE_THREADS environment variable. The default maximum number of threads is half the number of available threads, but setting it to be lower can reduce peak memory usage during compilation.\n\njl_dump_native can also produce native code optimized for multiple architectures, when integrated with the Julia loader. This is triggered by setting the JULIA_CPU_TARGET environment variable and mediated by the multiversioning pass in the optimization pipeline. To make this work with multithreading, an annotation step is added before the module is split into submodules that are emitted on their own threads, and this annotation step uses information available throughout the entire module to decide what functions are cloned for different architectures. Once the annotation has happened, individual threads can emit code for different architectures in parallel, knowing that a different submodule is guaranteed to produce the necessary functions that will be called by a cloned function.\n\nSome other metadata about how the module was serialized is also stored in the archive, such as the number of threads used to serialize the module and the number of functions that were compiled."},{"location":"devdocs/aot.html#Static-Linking","page":"Ahead of Time Compilation","title":"Static Linking","category":"section","text":"The final step in the AOT compilation process is to run a linker on the object files in the archive produced by jl_dump_native. This produces a shared library containing the compiled code. This shared library can then be loaded by Julia to restore the state of the runtime. When compiling a system image, the native linker used by a C compiler is used to produce the final shared library. For package images, the LLVM linker LLD is used to provide a more consistent linking interface."},{"location":"devdocs/aot.html#Loading-Code-Images-2","page":"Ahead of Time Compilation","title":"Loading Code Images","category":"section","text":""},{"location":"devdocs/aot.html#Loading-the-Shared-Library","page":"Ahead of Time Compilation","title":"Loading the Shared Library","category":"section","text":"The first step in loading a code image is to load the shared library produced by the linker. This is done by calling jl_dlopen on the path to the shared library. This function is responsible for loading the shared library and resolving all of the symbols in the library."},{"location":"devdocs/aot.html#Loading-Native-Code","page":"Ahead of Time Compilation","title":"Loading Native Code","category":"section","text":"The loader first needs to identify whether the native code that was compiled is valid for the architecture that the loader is running on. This is necessary to avoid executing instructions that older CPUs do not recognize. This is done by checking the CPU features available on the current machine against the CPU features that the code was compiled for. When multiversioning is enabled, the loader will pick the appropriate version of each function to use based on the CPU features available on the current machine. If none of the feature sets that were multiversioned, the loader will throw an error.\n\nPart of the multiversioning pass creates a number of global arrays of all of the functions in the module. When this process is multithreaded, an array of arrays is created, which the loader reorganizes into one large array with all of the functions that were compiled for this architecture. A similar process occurs for the global variables in the module."},{"location":"devdocs/aot.html#Setting-Up-Julia-State","page":"Ahead of Time Compilation","title":"Setting Up Julia State","category":"section","text":"The loader then uses the global variables and functions produced from loading native code to set up Julia runtime core data structures in the current process. This setup involves adding types and methods to the Julia runtime, and making the cached native code available for use by other Julia functions and the interpreter. For package images, each method must be validated, in that the global method table's state must match the state that the package image was compiled for. In particular, if a different set of methods exists at the load time compared to compile time of the package image, the method must be invalidated and recompiled on first use. This is necessary to ensure that execution semantics remain the same regardless of if a package was precompiled or if the code was directly executed. System images do not need to perform this validation, since the global method table is empty at load time. Thus, system images have faster load times than package images."},{"location":"devdocs/boundscheck.html#Bounds-checking","page":"Bounds checking","title":"Bounds checking","category":"section","text":"Like many modern programming languages, Julia uses bounds checking to ensure program safety when accessing arrays. In tight inner loops or other performance critical situations, you may wish to skip these bounds checks to improve runtime performance. For instance, in order to emit vectorized (SIMD) instructions, your loop body cannot contain branches, and thus cannot contain bounds checks. Consequently, Julia includes an @inbounds(...) macro to tell the compiler to skip such bounds checks within the given block. User-defined array types can use the @boundscheck(...) macro to achieve context-sensitive code selection."},{"location":"devdocs/boundscheck.html#Eliding-bounds-checks","page":"Bounds checking","title":"Eliding bounds checks","category":"section","text":"The @boundscheck(...) macro marks blocks of code that perform bounds checking. When such blocks are inlined into an @inbounds(...) block, the compiler may remove these blocks. The compiler removes the @boundscheck block only if it is inlined into the calling function. For example, you might write the method sum as:\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in eachindex(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWith a custom array-like type MyArray having:\n\n@inline getindex(A::MyArray, i::Real) = (@boundscheck checkbounds(A, i); A.data[to_index(i)])\n\nThen when getindex is inlined into sum, the call to checkbounds(A, i) will be elided. If your function contains multiple layers of inlining, only @boundscheck blocks at most one level of inlining deeper are eliminated. The rule prevents unintended changes in program behavior from code further up the stack."},{"location":"devdocs/boundscheck.html#Caution!","page":"Bounds checking","title":"Caution!","category":"section","text":"It is easy to accidentally expose unsafe operations with @inbounds. You might be tempted to write the above example as\n\nfunction sum(A::AbstractArray)\n    r = zero(eltype(A))\n    for i in 1:length(A)\n        @inbounds r += A[i]\n    end\n    return r\nend\n\nWhich quietly assumes 1-based indexing and therefore exposes unsafe memory access when used with OffsetArrays:\n\njulia> using OffsetArrays\n\njulia> sum(OffsetArray([1, 2, 3], -10))\n9164911648 # inconsistent results or segfault\n\nWhile the original source of the error here is 1:length(A), the use of @inbounds increases the consequences from a bounds error to a less easily caught and debugged unsafe memory access. It is often difficult or impossible to prove that a method which uses @inbounds is safe, so one must weigh the benefits of performance improvements against the risk of segfaults and silent misbehavior, especially in public facing APIs."},{"location":"devdocs/boundscheck.html#Propagating-inbounds","page":"Bounds checking","title":"Propagating inbounds","category":"section","text":"There may be certain scenarios where for code-organization reasons you want more than one layer between the @inbounds and @boundscheck declarations. For instance, the default getindex methods have the chain getindex(A::AbstractArray, i::Real) calls getindex(IndexStyle(A), A, i) calls _getindex(::IndexLinear, A, i).\n\nTo override the \"one layer of inlining\" rule, a function may be marked with Base.@propagate_inbounds to propagate an inbounds context (or out of bounds context) through one additional layer of inlining."},{"location":"devdocs/boundscheck.html#The-bounds-checking-call-hierarchy","page":"Bounds checking","title":"The bounds checking call hierarchy","category":"section","text":"The overall hierarchy is:\n\ncheckbounds(A, I...) which calls\ncheckbounds(Bool, A, I...) which calls\ncheckbounds_indices(Bool, axes(A), I) which recursively calls\ncheckindex for each dimension\n\nHere A is the array, and I contains the \"requested\" indices. axes(A) returns a tuple of \"permitted\" indices of A.\n\ncheckbounds(A, I...) throws an error if the indices are invalid, whereas checkbounds(Bool, A, I...) returns false in that circumstance.  checkbounds_indices discards any information about the array other than its axes tuple, and performs a pure indices-vs-indices comparison: this allows relatively few compiled methods to serve a huge variety of array types. Indices are specified as tuples, and are usually compared in a 1-1 fashion with individual dimensions handled by calling another important function, checkindex: typically,\n\ncheckbounds_indices(Bool, (IA1, IA...), (I1, I...)) = checkindex(Bool, IA1, I1) &\n                                                      checkbounds_indices(Bool, IA, I)\n\nso checkindex checks a single dimension. All of these functions, including the unexported checkbounds_indices have docstrings accessible with ? .\n\nIf you have to customize bounds checking for a specific array type, you should specialize checkbounds(Bool, A, I...). However, in most cases you should be able to rely on checkbounds_indices as long as you supply useful axes for your array type.\n\nIf you have novel index types, first consider specializing checkindex, which handles a single index for a particular dimension of an array. If you have a custom multidimensional index type (similar to CartesianIndex), then you may have to consider specializing checkbounds_indices.\n\nNote this hierarchy has been designed to reduce the likelihood of method ambiguities. We try to make checkbounds the place to specialize on array type, and try to avoid specializations on index types; conversely, checkindex is intended to be specialized only on index type (especially, the last argument)."},{"location":"devdocs/boundscheck.html#Emit-bounds-checks","page":"Bounds checking","title":"Emit bounds checks","category":"section","text":"Julia can be launched with --check-bounds={yes|no|auto} to emit bounds checks always, never, or respect @inbounds declarations."},{"location":"devdocs/build/windows.html#Windows","page":"Windows","title":"Windows","category":"section","text":"This file describes how to install, or build, and use Julia on Windows.\n\nFor more general information about Julia, please see the main README or the documentation."},{"location":"devdocs/build/windows.html#General-Information-for-Windows","page":"Windows","title":"General Information for Windows","category":"section","text":"We highly recommend running Julia using a modern terminal application, in particular Windows Terminal, which can be installed from the Microsoft Store."},{"location":"devdocs/build/windows.html#Line-endings","page":"Windows","title":"Line endings","category":"section","text":"Julia uses binary-mode files exclusively. Unlike many other Windows programs, if you write \\n to a file, you get a \\n in the file, not some other bit pattern. This matches the behavior exhibited by other operating systems. If you have installed Git for Windows, it is suggested, but not required, that you configure your system Git to use the same convention:\n\ngit config --global core.eol lf\ngit config --global core.autocrlf input\n\nor edit %USERPROFILE%\\.gitconfig and add/edit the lines:\n\n[core]\n    eol = lf\n    autocrlf = input"},{"location":"devdocs/build/windows.html#Binary-distribution","page":"Windows","title":"Binary distribution","category":"section","text":"For the binary distribution installation notes on Windows please see the instructions at https://julialang.org/downloads/platform/#windows. Note, however, that on all platforms using juliaup is recommended over manually installing binaries."},{"location":"devdocs/build/windows.html#Source-distribution","page":"Windows","title":"Source distribution","category":"section","text":""},{"location":"devdocs/build/windows.html#Cygwin-to-MinGW-cross-compiling","page":"Windows","title":"Cygwin-to-MinGW cross-compiling","category":"section","text":"The recommended way of compiling Julia from source on Windows is by cross compiling from Cygwin, using versions of the MinGW-w64 compilers available through Cygwin's package manager.\n\nDownload and run Cygwin setup for 32 bit or 64 bit. Note, that you can compile either 32 or 64 bit Julia from either 32 or 64 bit Cygwin. 64 bit Cygwin has a slightly smaller but often more up-to-date selection of packages.\nAdvanced: you may skip steps 2-4 by running:\nsetup-x86_64.exe -s <url> -q -P cmake,gcc-g++=12.5.0-1,git,make,patch,curl,m4,python3,p7zip,mingw64-i686-gcc-g++=12.5.0-1,mingw64-i686-gcc-fortran=12.5.0-1,mingw64-i686-gcc-core=12.5.0-1,mingw64-i686-headers=12.0.0-1,mingw64-i686-runtime=12.0.0-1,mingw64-i686-winpthreads=12.0.0-1,mingw64-x86_64-gcc-g++=12.5.0-1,mingw64-x86_64-gcc-fortran=12.5.0-1,mingw64-x86_64-gcc-core=12.5.0-1,mingw64-x86_64-headers=12.0.0-1,mingw64-x86_64-runtime=12.0.0-1,mingw64-x86_64-winpthreads=12.0.0-1\nreplacing <url> with a site from https://cygwin.com/mirrors.html or run setup manually first and select a mirror.\nSelect installation location and a mirror to download from.\nAt the Select Packages step, select the following:\nFrom the Devel category: cmake, gcc-g++, git, make, patch\nFrom the Net category: curl\nFrom Interpreters (or Python) category: m4, python3\nFrom the Archive category: p7zip\nFor 32 bit Julia, and also from the Devel category: mingw64-i686-gcc-g++ and mingw64-i686-gcc-fortran and mingw64-i686-gcc-core (version \"12.5.0-1\") mingw64-i686-headers and mingw64-i686-runtime and mingw64-i686-winpthreads (version \"12.0.0-1\")\nFor 64 bit Julia, and also from the Devel category: mingw64-x86_64-gcc-g++ and mingw64-x86_64-gcc-fortran and mingw64-x86_64-gcc-core (version \"12.5.0-1\") mingw64-x86_64-headers and mingw64-x86_64-runtime and mingw64-x86_64-winpthreads (version \"12.0.0-1\")\nAllow Cygwin installation to finish, then start from the installed shortcut 'Cygwin Terminal', or 'Cygwin64 Terminal', respectively.\nBuild Julia and its dependencies from source:\nGet the Julia sources\ngit clone https://github.com/JuliaLang/julia.git\ncd julia\nTip: If you get an error: cannot fork() for fetch-pack: Resource temporarily unavailable from git, add alias git=\"env PATH=/usr/bin git\" to ~/.bashrc and restart Cygwin.\nSet the XC_HOST variable in Make.user to indicate MinGW-w64 cross compilation\necho 'XC_HOST = i686-w64-mingw32' > Make.user     # for 32 bit Julia\n# or\necho 'XC_HOST = x86_64-w64-mingw32' > Make.user   # for 64 bit Julia\nStart the build\nmake -j 4       # Adjust the number of threads (4) to match your build environment.\nmake -j 4 debug # This builds julia-debug.exe\nRun Julia using the Julia executables directly\nusr/bin/julia.exe\nusr/bin/julia-debug.exe\n\nnote: Pro tip: build both!\nmake O=julia-win32 configure\nmake O=julia-win64 configure\necho 'XC_HOST = i686-w64-mingw32' > julia-win32/Make.user\necho 'XC_HOST = x86_64-w64-mingw32' > julia-win64/Make.user\necho 'ifeq ($(BUILDROOT),$(JULIAHOME))\n        $(error \"in-tree build disabled\")\n      endif' >> Make.user\nmake -C julia-win32  # build for Windows x86 in julia-win32 folder\nmake -C julia-win64  # build for Windows x86-64 in julia-win64 folder"},{"location":"devdocs/build/windows.html#Compiling-with-MinGW/MSYS2","page":"Windows","title":"Compiling with MinGW/MSYS2","category":"section","text":"MSYS2 is a software distribution and build environment for Windows.\n\nNote: MSYS2 requires 64 bit Windows 7 or newer.\n\nInstall and configure MSYS2.\nDownload and run the latest installer for the  64-bit distribution.  The installer will have a name like msys2-x86_64-yyyymmdd.exe.\nOpen the MSYS2 shell. Update the package database and base packages:\npacman -Syu\nExit and restart MSYS2. Update the rest of the base packages:\npacman -Syu\nThen install tools required to build julia:\npacman -S diffutils git m4 make patch tar p7zip curl python\nFor 64 bit Julia, install the x86_64 version:\npacman -S mingw-w64-x86_64-gcc mingw-w64-x86_64-cmake mingw-w64-x86_64-clang\nFor 32 bit Julia, install the i686 version:\npacman -S mingw-w64-i686-gcc mingw-w64-i686-cmake mingw-w64-i686-clang\nConfiguration of MSYS2 is complete. Now exit the MSYS2 shell.\nBuild Julia and its dependencies with pre-build dependencies.\nOpen a new MINGW64/MINGW32 shell.  Currently we can't use both mingw32 and mingw64,  so if you want to build the x86_64 and i686 versions,  you'll need to build them in each environment separately.\nClone the Julia sources:\ngit clone https://github.com/JuliaLang/julia.git\ncd julia\nIf you want to use clang (currently required if building LLVM from source), put the following in your Make.user\nCC=/mingw64/bin/clang   CXX=/mingw64/bin/clang++\n\nwarning: UCRT Unsupported\n\n\nDo not try to use any other clang that MSYS2 may install (which may not have the correct default target) or the \"Clang\" environment(which defaults to the currently unsupported ucrt).\n\n4. Start the build\n\n   ```\n   make -j$(nproc)\n   ```\n\nnote: Pro tip: build in dir\nmake O=julia-mingw-w64 configure\necho 'ifeq ($(BUILDROOT),$(JULIAHOME))\n        $(error \"in-tree build disabled\")\n      endif' >> Make.user\nmake -C julia-mingw-w64"},{"location":"devdocs/build/windows.html#Cross-compiling-from-Unix-(Linux/Mac/WSL)","page":"Windows","title":"Cross-compiling from Unix (Linux/Mac/WSL)","category":"section","text":"You can also use MinGW-w64 cross compilers to build a Windows version of Julia from Linux, Mac, or the Windows Subsystem for Linux (WSL).\n\nFirst, you will need to ensure your system has the required dependencies. We need wine (>=1.7.5), a system compiler, and some downloaders. Note: a Cygwin install might interfere with this method if using WSL.\n\nOn Ubuntu (on other Linux systems the dependency names are likely to be similar):\n\napt-get install wine-stable gcc wget p7zip-full winbind mingw-w64 gfortran-mingw-w64\ndpkg --add-architecture i386 && apt-get update && apt-get install wine32 # add sudo to each if needed\n# switch all of the following to their \"-posix\" variants (interactively):\nfor pkg in i686-w64-mingw32-g++ i686-w64-mingw32-gcc i686-w64-mingw32-gfortran x86_64-w64-mingw32-g++ x86_64-w64-mingw32-gcc x86_64-w64-mingw32-gfortran; do\n    sudo update-alternatives --config $pkg\ndone\n\nOn Mac: Install XCode, XCode command line tools, X11 (now XQuartz), and MacPorts or Homebrew. Then run port install wine wget mingw-w64, or brew install wine wget mingw-w64, as appropriate.\n\nThen run the build:\n\ngit clone https://github.com/JuliaLang/julia.git julia-win32\ncd julia-win32\necho override XC_HOST = i686-w64-mingw32 >> Make.user\nmake\nmake win-extras (Necessary before running make binary-dist)\nmake binary-dist then make exe to create the Windows installer.\nmove the julia-*.exe installer to the target machine\n\nIf you are building for 64-bit Windows, the steps are essentially the same. Just replace i686 in XC_HOST with x86_64. (Note: on Mac, wine only runs in 32-bit mode)."},{"location":"devdocs/build/windows.html#Debugging-a-cross-compiled-build-under-wine","page":"Windows","title":"Debugging a cross-compiled build under wine","category":"section","text":"The most effective way to debug a cross-compiled version of Julia on the cross-compilation host is to install a Windows version of GDB and run it under wine as usual. The pre-built packages available as part of the MSYS2 project are known to work. Apart from the GDB package you may also need the python and termcap packages. Finally, GDB's prompt may not work when launched from the command line. This can be worked around by prepending wineconsole to the regular GDB invocation."},{"location":"devdocs/build/windows.html#After-compiling","page":"Windows","title":"After compiling","category":"section","text":"Compiling using one of the options above creates a basic Julia build, but not some extra components that are included if you run the full Julia binary installer. If you need these components, the easiest way to get them is to build the installer yourself using make win-extras followed by make binary-dist and make exe. Then run the resulting installer."},{"location":"devdocs/build/windows.html#Windows-Build-Debugging","page":"Windows","title":"Windows Build Debugging","category":"section","text":""},{"location":"devdocs/build/windows.html#GDB-hangs-with-Cygwin-mintty","page":"Windows","title":"GDB hangs with Cygwin mintty","category":"section","text":"Run GDB under the Windows console (cmd) instead. GDB may not function properly under mintty with non- Cygwin applications. You can use cmd /c start to start the Windows console from mintty if necessary."},{"location":"devdocs/build/windows.html#GDB-not-attaching-to-the-right-process","page":"Windows","title":"GDB not attaching to the right process","category":"section","text":"Use the PID from the Windows task manager or WINPID from the ps command instead of the PID from unix-style command line tools (e.g. pgrep). You may need to add the PID column if it is not shown by default in the Windows task manager."},{"location":"devdocs/build/windows.html#GDB-not-showing-the-right-backtrace","page":"Windows","title":"GDB not showing the right backtrace","category":"section","text":"When attaching to the julia process, GDB may not be attaching to the right thread. Use info threads command to show all the threads and thread <threadno> to switch threads.\nBe sure to use a 32 bit version of GDB to debug a 32 bit build of Julia, or a 64 bit version of GDB to debug a 64 bit build of Julia."},{"location":"devdocs/build/windows.html#Build-process-is-slow/eats-memory/hangs-my-computer","page":"Windows","title":"Build process is slow/eats memory/hangs my computer","category":"section","text":"Disable the Windows Superfetch and Program Compatibility Assistant services, as they are known to have spurious interactions with MinGW/Cygwin.\nAs mentioned in the link above: excessive memory use by svchost specifically may be investigated in the Task Manager by clicking on the high-memory svchost.exe process and selecting Go to Services. Disable child services one-by-one until a culprit is found.\nBeware of BLODA. The vmmap tool is indispensable for identifying such software conflicts. Use vmmap to inspect the list of loaded DLLs for bash, mintty, or another persistent process used to drive the build. Essentially any DLL outside of the Windows System directory is potential BLODA."},{"location":"manual/missing.html#missing","page":"Missing Values","title":"Missing Values","category":"section","text":"Julia provides support for representing missing values in the statistical sense. This is for situations where no value is available for a variable in an observation, but a valid value theoretically exists. Missing values are represented via the missing object, which is the singleton instance of the type Missing. missing is equivalent to NULL in SQL and NA in R, and behaves like them in most situations."},{"location":"manual/missing.html#Propagation-of-Missing-Values","page":"Missing Values","title":"Propagation of Missing Values","category":"section","text":"missing values propagate automatically when passed to standard mathematical operators and functions. For these functions, uncertainty about the value of one of the operands induces uncertainty about the result. In practice, this means a math operation involving a missing value generally returns missing:\n\njulia> missing + 1\nmissing\n\njulia> \"a\" * missing\nmissing\n\njulia> abs(missing)\nmissing\n\nSince missing is a normal Julia object, this propagation rule only works for functions which have opted in to implement this behavior. This can be achieved by:\n\nadding a specific method defined for arguments of type Missing,\naccepting arguments of this type, and passing them to functions which propagate them (like standard math operators).\n\nPackages should consider whether it makes sense to propagate missing values when defining new functions, and define methods appropriately if this is the case. Passing a missing value to a function which does not have a method accepting arguments of type Missing throws a MethodError, just like for any other type.\n\nFunctions that do not propagate missing values can be made to do so by wrapping them in the passmissing function provided by the Missings.jl package. For example, f(x) becomes passmissing(f)(x)."},{"location":"manual/missing.html#Equality-and-Comparison-Operators","page":"Missing Values","title":"Equality and Comparison Operators","category":"section","text":"Standard equality and comparison operators follow the propagation rule presented above: if any of the operands is missing, the result is missing. Here are a few examples:\n\njulia> missing == 1\nmissing\n\njulia> missing == missing\nmissing\n\njulia> missing < 1\nmissing\n\njulia> 2 >= missing\nmissing\n\nIn particular, note that missing == missing returns missing, so == cannot be used to test whether a value is missing. To test whether x is missing, use ismissing(x).\n\nSpecial comparison operators isequal and === are exceptions to the propagation rule. They will always return a Bool value, even in the presence of missing values, considering missing as equal to missing and as different from any other value. They can therefore be used to test whether a value is missing:\n\njulia> missing === 1\nfalse\n\njulia> isequal(missing, 1)\nfalse\n\njulia> missing === missing\ntrue\n\njulia> isequal(missing, missing)\ntrue\n\nThe isless operator is another exception: missing is considered as greater than any other value. This operator is used by sort!, which therefore places missing values after all other values:\n\njulia> isless(1, missing)\ntrue\n\njulia> isless(missing, Inf)\nfalse\n\njulia> isless(missing, missing)\nfalse"},{"location":"manual/missing.html#Logical-operators","page":"Missing Values","title":"Logical operators","category":"section","text":"Logical (or boolean) operators |, & and xor are another special case since they only propagate missing values when it is logically required. For these operators, whether or not the result is uncertain, depends on the particular operation. This follows the well-established rules of three-valued logic which are implemented by e.g. NULL in SQL and NA in R. This abstract definition corresponds to a relatively natural behavior which is best explained via concrete examples.\n\nLet us illustrate this principle with the logical \"or\" operator |. Following the rules of boolean logic, if one of the operands is true, the value of the other operand does not have an influence on the result, which will always be true:\n\njulia> true | true\ntrue\n\njulia> true | false\ntrue\n\njulia> false | true\ntrue\n\nBased on this observation, we can conclude if one of the operands is true and the other missing, we know that the result is true in spite of the uncertainty about the actual value of one of the operands. If we had been able to observe the actual value of the second operand, it could only be true or false, and in both cases the result would be true. Therefore, in this particular case, missingness does not propagate:\n\njulia> true | missing\ntrue\n\njulia> missing | true\ntrue\n\nOn the contrary, if one of the operands is false, the result could be either true or false depending on the value of the other operand. Therefore, if that operand is missing, the result has to be missing too:\n\njulia> false | true\ntrue\n\njulia> true | false\ntrue\n\njulia> false | false\nfalse\n\njulia> false | missing\nmissing\n\njulia> missing | false\nmissing\n\nThe behavior of the logical \"and\" operator & is similar to that of the | operator, with the difference that missingness does not propagate when one of the operands is false. For example, when that is the case of the first operand:\n\njulia> false & false\nfalse\n\njulia> false & true\nfalse\n\njulia> false & missing\nfalse\n\nOn the other hand, missingness propagates when one of the operands is true, for example the first one:\n\njulia> true & true\ntrue\n\njulia> true & false\nfalse\n\njulia> true & missing\nmissing\n\nFinally, the \"exclusive or\" logical operator xor always propagates missing values, since both operands always have an effect on the result. Also note that the negation operator ! returns missing when the operand is missing, just like other unary operators."},{"location":"manual/missing.html#Control-Flow-and-Short-Circuiting-Operators","page":"Missing Values","title":"Control Flow and Short-Circuiting Operators","category":"section","text":"Control flow operators including if, while and the ternary operator x ? y : z do not allow for missing values. This is because of the uncertainty about whether the actual value would be true or false if we could observe it. This implies we do not know how the program should behave. In this case, a TypeError is thrown as soon as a missing value is encountered in this context:\n\njulia> if missing\n           println(\"here\")\n       end\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\nFor the same reason, contrary to logical operators presented above, the short-circuiting boolean operators && and || do not allow for missing values in situations where the value of the operand determines whether the next operand is evaluated or not. For example:\n\njulia> missing || false\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\njulia> missing && false\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\njulia> true && missing && false\nERROR: TypeError: non-boolean (Missing) used in boolean context\n\nIn contrast, there is no error thrown when the result can be determined without the missing values. This is the case when the code short-circuits before evaluating the missing operand, and when the missing operand is the last one:\n\njulia> true && missing\nmissing\n\njulia> false && missing\nfalse"},{"location":"manual/missing.html#Arrays-With-Missing-Values","page":"Missing Values","title":"Arrays With Missing Values","category":"section","text":"Arrays containing missing values can be created like other arrays:\n\njulia> [1, missing]\n2-element Vector{Union{Missing, Int64}}:\n 1\n  missing\n\nAs this example shows, the element type of such arrays is Union{Missing, T}, with T the type of the non-missing values. This reflects the fact that array entries can be either of type T (here, Int64) or of type Missing. This kind of array uses an efficient memory storage equivalent to an Array{T} holding the actual values combined with an Array{UInt8} indicating the type of the entry (i.e. whether it is Missing or T).\n\nArrays allowing for missing values can be constructed with the standard syntax. Use Array{Union{Missing, T}}(missing, dims) to create arrays filled with missing values:\n\njulia> Array{Union{Missing, String}}(missing, 2, 3)\n2×3 Matrix{Union{Missing, String}}:\n missing  missing  missing\n missing  missing  missing\n\nnote: Note\nUsing undef or similar may currently give an array filled with missing, but this is not the correct way to obtain such an array. Use a missing constructor as shown above instead.\n\nAn array with element type allowing missing entries (e.g. Vector{Union{Missing, T}}) which does not contain any missing entries can be converted to an array type that does not allow for missing entries (e.g. Vector{T}) using convert. If the array contains missing values, a MethodError is thrown during conversion:\n\njulia> x = Union{Missing, String}[\"a\", \"b\"]\n2-element Vector{Union{Missing, String}}:\n \"a\"\n \"b\"\n\njulia> convert(Array{String}, x)\n2-element Vector{String}:\n \"a\"\n \"b\"\n\njulia> y = Union{Missing, String}[missing, \"b\"]\n2-element Vector{Union{Missing, String}}:\n missing\n \"b\"\n\njulia> convert(Array{String}, y)\nERROR: MethodError: Cannot `convert` an object of type Missing to an object of type String"},{"location":"manual/missing.html#Skipping-Missing-Values","page":"Missing Values","title":"Skipping Missing Values","category":"section","text":"Since missing values propagate with standard mathematical operators, reduction functions return missing when called on arrays which contain missing values:\n\njulia> sum([1, missing])\nmissing\n\nIn this situation, use the skipmissing function to skip missing values:\n\njulia> sum(skipmissing([1, missing]))\n1\n\nThis convenience function returns an iterator which filters out missing values efficiently. It can therefore be used with any function which supports iterators:\n\njulia> x = skipmissing([3, missing, 2, 1])\nskipmissing(Union{Missing, Int64}[3, missing, 2, 1])\n\njulia> maximum(x)\n3\n\njulia> sum(x)\n6\n\njulia> mapreduce(sqrt, +, x)\n4.146264369941973\n\nObjects created by calling skipmissing on an array can be indexed using indices from the parent array. Indices corresponding to missing values are not valid for these objects, and an error is thrown when trying to use them (they are also skipped by keys and eachindex):\n\njulia> x[1]\n3\n\njulia> x[2]\nERROR: MissingException: the value at index (2,) is missing\n[...]\n\nThis allows functions which operate on indices to work in combination with skipmissing. This is notably the case for search and find functions. These functions return indices valid for the object returned by skipmissing, and are also the indices of the matching entries in the parent array:\n\njulia> findall(==(1), x)\n1-element Vector{Int64}:\n 4\n\njulia> findfirst(!iszero, x)\n1\n\njulia> argmax(x)\n1\n\nUse collect to extract non-missing values and store them in an array:\n\njulia> collect(x)\n3-element Vector{Int64}:\n 3\n 2\n 1"},{"location":"manual/missing.html#Logical-Operations-on-Arrays","page":"Missing Values","title":"Logical Operations on Arrays","category":"section","text":"The three-valued logic described above for logical operators is also used by logical functions applied to arrays. Thus, array equality tests using the == operator return missing whenever the result cannot be determined without knowing the actual value of the missing entry. In practice, this means missing is returned if all non-missing values of the compared arrays are equal, but one or both arrays contain missing values (possibly at different positions):\n\njulia> [1, missing] == [2, missing]\nfalse\n\njulia> [1, missing] == [1, missing]\nmissing\n\njulia> [1, 2, missing] == [1, missing, 2]\nmissing\n\nAs for single values, use isequal to treat missing values as equal to other missing values, but different from non-missing values:\n\njulia> isequal([1, missing], [1, missing])\ntrue\n\njulia> isequal([1, 2, missing], [1, missing, 2])\nfalse\n\nFunctions any and all also follow the rules of three-valued logic. Thus, returning missing when the result cannot be determined:\n\njulia> all([true, missing])\nmissing\n\njulia> all([false, missing])\nfalse\n\njulia> any([true, missing])\ntrue\n\njulia> any([false, missing])\nmissing"},{"location":"manual/asynchronous-programming.html#man-asynchronous","page":"Asynchronous Programming","title":"Asynchronous Programming","category":"section","text":"When a program needs to interact with the outside world, for example communicating with another machine over the internet, operations in the program may need to happen in an unpredictable order. Say your program needs to download a file. We would like to initiate the download operation, perform other operations while we wait for it to complete, and then resume the code that needs the downloaded file when it is available. This sort of scenario falls in the domain of asynchronous programming, sometimes also referred to as concurrent programming (since, conceptually, multiple things are happening at once).\n\nTo address these scenarios, Julia provides Tasks (also known by several other names, such as symmetric coroutines, lightweight threads, cooperative multitasking, or one-shot continuations). When a piece of computing work (in practice, executing a particular function) is designated as a Task, it becomes possible to interrupt it by switching to another Task. The original Task can later be resumed, at which point it will pick up right where it left off. At first, this may seem similar to a function call. However there are two key differences. First, switching tasks does not use any space, so any number of task switches can occur without consuming the call stack. Second, switching among tasks can occur in any order, unlike function calls, where the called function must finish executing before control returns to the calling function."},{"location":"manual/asynchronous-programming.html#Basic-Task-operations","page":"Asynchronous Programming","title":"Basic Task operations","category":"section","text":"You can think of a Task as a handle to a unit of computational work to be performed. It has a create-start-run-finish lifecycle. Tasks are created by calling the Task constructor on a 0-argument function to run, or using the @task macro:\n\njulia> t = @task begin; sleep(5); println(\"done\"); end\nTask (runnable) @0x00007f13a40c0eb0\n\n@task x is equivalent to Task(()->x).\n\nThis task will wait for five seconds, and then print done. However, it has not started running yet. We can run it whenever we're ready by calling schedule:\n\njulia> schedule(t);\n\nIf you try this in the REPL, you will see that schedule returns immediately. That is because it simply adds t to an internal queue of tasks to run. Then, the REPL will print the next prompt and wait for more input. Waiting for keyboard input provides an opportunity for other tasks to run, so at that point t will start. t calls sleep, which sets a timer and stops execution. If other tasks have been scheduled, they could run then. After five seconds, the timer fires and restarts t, and you will see done printed. t is then finished.\n\nThe wait function blocks the calling task until some other task finishes. So for example if you type\n\njulia> schedule(t); wait(t)\n\ninstead of only calling schedule, you will see a five second pause before the next input prompt appears. That is because the REPL is waiting for t to finish before proceeding.\n\nIt is common to want to create a task and schedule it right away, so the macro Threads.@spawn is provided for that purpose — Threads.@spawn x is equivalent to task = @task x; task.sticky = false; schedule(task)."},{"location":"manual/asynchronous-programming.html#Communicating-with-Channels","page":"Asynchronous Programming","title":"Communicating with Channels","category":"section","text":"In some problems, the various pieces of required work are not naturally related by function calls; there is no obvious \"caller\" or \"callee\" among the jobs that need to be done. An example is the producer-consumer problem, where one complex procedure is generating values and another complex procedure is consuming them. The consumer cannot simply call a producer function to get a value, because the producer may have more values to generate and so might not yet be ready to return. With tasks, the producer and consumer can both run as long as they need to, passing values back and forth as necessary.\n\nJulia provides a Channel mechanism for solving this problem. A Channel is a waitable first-in first-out queue which can have multiple tasks reading from and writing to it.\n\nLet's define a producer task, which produces values via the put! call. To consume values, we need to schedule the producer to run in a new task. A special Channel constructor which accepts a 1-arg function as an argument can be used to run a task bound to a channel. We can then take! values repeatedly from the channel object:\n\njulia> function producer(c::Channel)\n           put!(c, \"start\")\n           for n=1:4\n               put!(c, 2n)\n           end\n           put!(c, \"stop\")\n       end;\n\njulia> chnl = Channel(producer);\n\njulia> take!(chnl)\n\"start\"\n\njulia> take!(chnl)\n2\n\njulia> take!(chnl)\n4\n\njulia> take!(chnl)\n6\n\njulia> take!(chnl)\n8\n\njulia> take!(chnl)\n\"stop\"\n\nOne way to think of this behavior is that producer was able to return multiple times. Between calls to put!, the producer's execution is suspended and the consumer has control.\n\nThe returned Channel can be used as an iterable object in a for loop, in which case the loop variable takes on all the produced values. The loop is terminated when the channel is closed.\n\njulia> for x in Channel(producer)\n           println(x)\n       end\nstart\n2\n4\n6\n8\nstop\n\nNote that we did not have to explicitly close the channel in the producer. This is because the act of binding a Channel to a Task associates the open lifetime of a channel with that of the bound task. The channel object is closed automatically when the task terminates. Multiple channels can be bound to a task, and vice-versa.\n\nWhile the Task constructor expects a 0-argument function, the Channel method that creates a task-bound channel expects a function that accepts a single argument of type Channel. A common pattern is for the producer to be parameterized, in which case a partial function application is needed to create a 0 or 1 argument anonymous function.\n\nFor Task objects this can be done either directly or by use of a convenience macro:\n\nfunction mytask(myarg)\n    ...\nend\n\ntaskHdl = Task(() -> mytask(7))\n# or, equivalently\ntaskHdl = @task mytask(7)\n\nTo orchestrate more advanced work distribution patterns, bind and schedule can be used in conjunction with Task and Channel constructors to explicitly link a set of channels with a set of producer/consumer tasks."},{"location":"manual/asynchronous-programming.html#More-on-Channels","page":"Asynchronous Programming","title":"More on Channels","category":"section","text":"A channel can be visualized as a pipe, i.e., it has a write end and a read end:\n\nMultiple writers in different tasks can write to the same channel concurrently via put! calls.\nMultiple readers in different tasks can read data concurrently via take! calls.\nAs an example:\n# Given Channels c1 and c2,\nc1 = Channel(32)\nc2 = Channel(32)\n\n# and a function `foo` which reads items from c1, processes the item read\n# and writes a result to c2,\nfunction foo()\n    while true\n        data = take!(c1)\n        [...]               # process data\n        put!(c2, result)    # write out result\n    end\nend\n\n# we can schedule `n` instances of `foo` to be active concurrently.\nfor _ in 1:n\n    errormonitor(Threads.@spawn foo())\nend\nChannels are created via the Channel{T}(sz) constructor. The channel will only hold objects of type T. If the type is not specified, the channel can hold objects of any type. sz refers to the maximum number of elements that can be held in the channel at any time. For example, Channel(32) creates a channel that can hold a maximum of 32 objects of any type. A Channel{MyType}(64) can hold up to 64 objects of MyType at any time.\nIf a Channel is empty, readers (on a take! call) will block until data is available (see isempty).\nIf a Channel is full, writers (on a put! call) will block until space becomes available (see isfull).\nisready tests for the presence of any object in the channel, while wait waits for an object to become available.\nNote that if another task is currently waiting to put! an object into a channel, a channel can have more items available than its capacity.\nA Channel is in an open state initially. This means that it can be read from and written to freely via take! and put! calls. close closes a Channel. On a closed Channel, put! will fail. For example:\njulia> c = Channel(2);\n\njulia> put!(c, 1) # `put!` on an open channel succeeds\n1\n\njulia> close(c);\n\njulia> put!(c, 2) # `put!` on a closed channel throws an exception.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]\ntake! and fetch (which retrieves but does not remove the value) on a closed channel successfully return any existing values until it is emptied. Continuing the above example:\njulia> fetch(c) # Any number of `fetch` calls succeed.\n1\n\njulia> fetch(c)\n1\n\njulia> take!(c) # The first `take!` removes the value.\n1\n\njulia> take!(c) # No more data available on a closed channel.\nERROR: InvalidStateException: Channel is closed.\nStacktrace:\n[...]\n\nConsider a simple example using channels for inter-task communication. We start 4 tasks to process data from a single jobs channel. Jobs, identified by an id (job_id), are written to the channel. Each task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id and the simulated time to the results channel. Finally all the results are printed out.\n\njulia> const jobs = Channel{Int}(32);\n\njulia> const results = Channel{Tuple}(32);\n\njulia> function do_work()\n           for job_id in jobs\n               exec_time = rand()\n               sleep(exec_time)                # simulates elapsed time doing actual work\n                                               # typically performed externally.\n               put!(results, (job_id, exec_time))\n           end\n       end;\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(Threads.@spawn make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for i in 1:4 # start 4 tasks to process requests in parallel\n           errormonitor(Threads.@spawn do_work())\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds\")\n           global n = n - 1\n       end\n4 finished in 0.22 seconds\n3 finished in 0.45 seconds\n1 finished in 0.5 seconds\n7 finished in 0.14 seconds\n2 finished in 0.78 seconds\n5 finished in 0.9 seconds\n9 finished in 0.36 seconds\n6 finished in 0.87 seconds\n8 finished in 0.79 seconds\n10 finished in 0.64 seconds\n12 finished in 0.5 seconds\n11 finished in 0.97 seconds\n0.029772311\n\nInstead of errormonitor(t), a more robust solution may be to use bind(results, t), as that will not only log any unexpected failures, but also force the associated resources to close and propagate the exception everywhere."},{"location":"manual/asynchronous-programming.html#More-task-operations","page":"Asynchronous Programming","title":"More task operations","category":"section","text":"Task operations are built on a low-level primitive called yieldto. yieldto(task, value) suspends the current task, switches to the specified task, and causes that task's last yieldto call to return the specified value. Notice that yieldto is the only operation required to use task-style control flow; instead of calling and returning we are always just switching to a different task. This is why this feature is also called \"symmetric coroutines\"; each task is switched to and from using the same mechanism.\n\nyieldto is powerful, but most uses of tasks do not invoke it directly. Consider why this might be. If you switch away from the current task, you will probably want to switch back to it at some point, but knowing when to switch back, and knowing which task has the responsibility of switching back, can require considerable coordination. For example, put! and take! are blocking operations, which, when used in the context of channels maintain state to remember who the consumers are. Not needing to manually keep track of the consuming task is what makes put! easier to use than the low-level yieldto.\n\nIn addition to yieldto, a few other basic functions are needed to use tasks effectively.\n\ncurrent_task gets a reference to the currently-running task.\nistaskdone queries whether a task has exited.\nistaskstarted queries whether a task has run yet.\ntask_local_storage manipulates a key-value store specific to the current task."},{"location":"manual/asynchronous-programming.html#Tasks-and-events","page":"Asynchronous Programming","title":"Tasks and events","category":"section","text":"Most task switches occur as a result of waiting for events such as I/O requests, and are performed by a scheduler included in Julia Base. The scheduler maintains a queue of runnable tasks, and executes an event loop that restarts tasks based on external events such as message arrival.\n\nThe basic function for waiting for an event is wait. Several objects implement wait; for example, given a Process object, wait will wait for it to exit. wait is often implicit; for example, a wait can happen inside a call to read to wait for data to be available.\n\nIn all of these cases, wait ultimately operates on a Condition object, which is in charge of queueing and restarting tasks. When a task calls wait on a Condition, the task is marked as non-runnable, added to the condition's queue, and switches to the scheduler. The scheduler will then pick another task to run, or block waiting for external events. If all goes well, eventually an event handler will call notify on the condition, which causes tasks waiting for that condition to become runnable again.\n\nA task created explicitly by calling Task is initially not known to the scheduler. This allows you to manage tasks manually using yieldto if you wish. However, when such a task waits for an event, it still gets restarted automatically when the event happens, as you would expect."},{"location":"manual/variables.html#man-variables","page":"Variables","title":"Variables","category":"section","text":"A variable, in Julia, is a name associated (or bound) to a value. It's useful when you want to store a value (that you obtained after some math, for example) for later use. For example:\n\njulia> x = 10   # Assign the value 10 to the variable x\n10\n\njulia> x + 1    # Doing math with x's value\n11\n\njulia> x = 1 + 1   # Reassign x's value\n2\n\njulia> x = \"Hello World!\"   # You can assign values of other types, like strings of text\n\"Hello World!\"\n\nJulia provides an extremely flexible system for naming variables. Variable names are case-sensitive, and have no semantic meaning (that is, the language will not treat variables differently based on their names).\n\njulia> x = 1.0\n1.0\n\njulia> y = -3\n-3\n\njulia> Z = \"My string\"\n\"My string\"\n\njulia> customary_phrase = \"Hello world!\"\n\"Hello world!\"\n\njulia> UniversalDeclarationOfHumanRightsStart = \"人人生而自由，在尊严和权利上一律平等。\"\n\"人人生而自由，在尊严和权利上一律平等。\"\n\nUnicode names (in UTF-8 encoding) are allowed:\n\njulia> δ = 0.00001\n1.0e-5\n\njulia> 안녕하세요 = \"Hello\"\n\"Hello\"\n\nIn the Julia REPL and several other Julia editing environments, you can type many Unicode math symbols by typing the backslashed LaTeX symbol name followed by tab. For example, the variable name δ can be entered by typing \\delta-tab, or even α̂⁽²⁾ by \\alpha-tab-\\hat- tab-\\^(2)-tab. (If you find a symbol somewhere, e.g. in someone else's code, that you don't know how to type, the REPL help will tell you: just type ? and then paste the symbol.)\n\nJulia will even let you shadow existing exported constants and functions with local ones (although this is not recommended to avoid potential confusions):\n\njulia> pi = 3\n3\n\njulia> pi\n3\n\njulia> sqrt = 4\n4\n\njulia> length() = 5\nlength (generic function with 1 method)\n\njulia> Base.length\nlength (generic function with 79 methods)\n\nHowever, if you try to redefine a built-in constant or function that you have explicitly imported, Julia will give you an error:\n\njulia> using Base: pi, sqrt\n\njulia> pi\nπ = 3.1415926535897...\n\njulia> pi = 3\nERROR: cannot assign a value to imported variable Base.pi from module Main\n\njulia> sqrt(100)\n10.0\n\njulia> sqrt = 4\nERROR: cannot assign a value to imported variable Base.sqrt from module Main\n\ncompat: Julia 1.12\n\n\nNote that in versions prior to Julia 1.12, these errors depended on use rather than definition of   the conflicting binding."},{"location":"manual/variables.html#man-allowed-variable-names","page":"Variables","title":"Allowed Variable Names","category":"section","text":"Variable names must begin with a letter (A-Z or a-z), underscore, or a subset of Unicode code points greater than 00A0; in particular, Unicode character categories Lu/Ll/Lt/Lm/Lo/Nl (letters), Sc/So (currency and other symbols), and a few other letter-like characters (e.g. a subset of the Sm math symbols) are allowed. Subsequent characters may also include ! and digits (0-9 and other characters in categories Nd/No), as well as other Unicode code points: diacritics and other modifying marks (categories Mn/Mc/Me/Sk), some punctuation connectors (category Pc), primes, and a few other characters.\n\nOperators like + are also valid identifiers, but are parsed specially. In some contexts, operators can be used just like variables; for example (+) refers to the addition function, and (+) = f will reassign it. Most of the Unicode infix operators (in category Sm), such as ⊕, are parsed as infix operators and are available for user-defined methods (e.g. you can use const ⊗ = kron to define ⊗ as an infix Kronecker product). Operators can also be suffixed with modifying marks, primes, and sub/superscripts, e.g. +̂ₐ″ is parsed as an infix operator with the same precedence as +. A space is required between an operator that ends with a subscript/superscript letter and a subsequent variable name. For example, if +ᵃ is an operator, then +ᵃx must be written as +ᵃ x to distinguish it from + ᵃx where ᵃx is the variable name.\n\nA particular class of variable names is one that contains only underscores. These identifiers are write-only. I.e. they can only be assigned values, which are immediately discarded, and their values cannot be used in any way.\n\njulia> x, ___ = size([2 2; 1 1])\n(2, 2)\n\njulia> y = ___\nERROR: syntax: all-underscore identifiers are write-only and their values cannot be used in expressions\n\njulia> println(___)\nERROR: syntax: all-underscore identifiers are write-only and their values cannot be used in expressions\n\nThe only explicitly disallowed names for variables are the names of the built-in Keywords:\n\njulia> else = false\nERROR: ParseError:\n# Error @ none:1:1\nelse = false\n└──┘ ── invalid identifier\n\njulia> try = \"No\"\nERROR: ParseError:\n# Error @ none:1:1\ntry = \"No\"\n└────────┘ ── try without catch or finally\n\nSome Unicode characters are considered to be equivalent in identifiers. Different ways of entering Unicode combining characters (e.g., accents) are treated as equivalent (specifically, Julia identifiers are NFC. Julia also includes a few non-standard equivalences for characters that are visually similar and are easily entered by some input methods. The Unicode characters ɛ (U+025B: Latin small letter open e) and µ (U+00B5: micro sign) are treated as equivalent to the corresponding Greek letters. The middle dot · (U+00B7) and the Greek interpunct · (U+0387) are both treated as the mathematical dot operator ⋅ (U+22C5). The minus sign − (U+2212) is treated as equivalent to the hyphen-minus sign - (U+002D)."},{"location":"manual/variables.html#man-assignment-expressions","page":"Variables","title":"Assignment expressions and assignment versus mutation","category":"section","text":"An assignment variable = value \"binds\" the name variable to the value computed on the right-hand side, and the whole assignment is treated by Julia as an expression equal to the right-hand-side value. This means that assignments can be chained (the same value assigned to multiple variables with variable1 = variable2 = value) or used in other expressions, and is also why their result is shown in the REPL as the value of the right-hand side.  (In general, the REPL displays the value of whatever expression you evaluate.)  For example, here the value 4 of b = 2+2 is used in another arithmetic operation and assignment:\n\njulia> a = (b = 2+2) + 3\n7\n\njulia> a\n7\n\njulia> b\n4\n\nA common confusion is the distinction between assignment (giving a new \"name\" to a value) and mutation (changing a value). If you run a = 2 followed by a = 3, you have changed the \"name\" a to refer to a new value 3 … you haven't changed the number 2, so 2+2 will still give 4 and not 6!   This distinction becomes more clear when dealing with mutable types like arrays, whose contents can be changed:\n\njulia> a = [1,2,3] # an array of 3 integers\n3-element Vector{Int64}:\n 1\n 2\n 3\n\njulia> b = a   # both b and a are names for the same array!\n3-element Vector{Int64}:\n 1\n 2\n 3\n\nHere, the line b = a does not make a copy of the array a, it simply binds the name b to the same array a: both b and a \"point\" to one array [1,2,3] in memory. In contrast, an assignment a[i] = value changes the contents of the array, and the modified array will be visible through both the names a and b:\n\njulia> a[1] = 42     # change the first element\n42\n\njulia> a = 3.14159   # a is now the name of a different object\n3.14159\n\njulia> b   # b refers to the original array object, which has been mutated\n3-element Vector{Int64}:\n 42\n  2\n  3\n\nThat is, a[i] = value (an alias for setindex!) mutates an existing array object in memory, accessible via either a or b. Subsequently setting a = 3.14159 does not change this array, it simply binds a to a different object; the array is still accessible via b. Another common syntax to mutate an existing object is a.field = value (an alias for setproperty!), which can be used to change a mutable struct. There is also mutation via dot assignment, for example b .= 5:7 (which mutates our array b in-place to contain [5,6,7]), as part of Julia's vectorized \"dot\" syntax.\n\nWhen you call a function in Julia, it behaves as if you assigned the argument values to new variable names corresponding to the function arguments, as discussed in Argument-Passing Behavior.  (By convention, functions that mutate one or more of their arguments have names ending with !.)"},{"location":"manual/variables.html#Stylistic-Conventions","page":"Variables","title":"Stylistic Conventions","category":"section","text":"While Julia imposes few restrictions on valid names, it has become useful to adopt the following conventions:\n\nNames of variables are in lower case.\nWord separation can be indicated by underscores ('_'), but use of underscores is discouraged unless the name would be hard to read otherwise.\nNames of Types and Modules begin with a capital letter and word separation is shown with upper camel case instead of underscores.\nNames of functions and macros are in lower case, without underscores.\nFunctions that write to their arguments have names that end in !. These are sometimes called \"mutating\" or \"in-place\" functions because they are intended to produce changes in their arguments after the function is called, not just return a value.\n\nFor more information about stylistic conventions, see the Style Guide."},{"location":"devdocs/contributing/code-changes.html#Code-changes","page":"Code changes","title":"Code changes","category":"section","text":""},{"location":"devdocs/contributing/code-changes.html#Contributing-to-core-functionality-or-base-libraries","page":"Code changes","title":"Contributing to core functionality or base libraries","category":"section","text":"By contributing code to Julia, you are agreeing to release it under the MIT License.\n\nThe Julia community uses GitHub issues to track and discuss problems, feature requests, and pull requests (PR).\n\nIssues and pull requests should have self explanatory titles such that they can be understood from the list of PRs and Issues. i.e. Add {feature} and Fix {bug} are good, Fix #12345. Corrects the bug. is bad.\n\nYou can make pull requests for incomplete features to get code review. The convention is to open these as draft PRs and prefix the pull request title with \"WIP:\" for Work In Progress, or \"RFC:\" for Request for Comments when work is completed and ready for merging. This will prevent accidental merging of work that is in progress.\n\nNote: These instructions are for adding to or improving functionality in the base library. Before getting started, it can be helpful to discuss the proposed changes or additions on the Julia Discourse forum or in a GitHub issue—it's possible your proposed change belongs in a package rather than the core language. Also, keep in mind that changing stuff in the base can potentially break a lot of things. Finally, because of the time required to build Julia, note that it's usually faster to develop your code in stand-alone files, get it working, and then migrate it into the base libraries.\n\nAdd new code to Julia's base libraries as follows (this is the \"basic\" approach; see a more efficient approach in the next section):\n\nEdit the appropriate file in the base/ directory, or add new files if necessary. Create tests for your functionality and add them to files in the test/ directory. If you're editing C or Scheme code, most likely it lives in src/ or one of its subdirectories, although some aspects of Julia's REPL initialization live in cli/.\nAdd any new files to sysimg.jl in order to build them into the Julia system image.\nAdd any necessary export symbols in exports.jl.\nInclude your tests in test/Makefile and test/choosetests.jl.\n\nBuild as usual, and do make clean testall to test your contribution. If your contribution includes changes to Makefiles or external dependencies, make sure you can build Julia from a clean tree using git clean -fdx or equivalent (be careful – this command will delete any files lying around that aren't checked into git)."},{"location":"devdocs/contributing/code-changes.html#Running-specific-tests","page":"Code changes","title":"Running specific tests","category":"section","text":"There are make targets for running specific tests:\n\nmake test-bitarray\n\nYou can also use the runtests.jl script, e.g. to run test/bitarray.jl and test/math.jl:\n\n./usr/bin/julia test/runtests.jl bitarray math"},{"location":"devdocs/contributing/code-changes.html#Modifying-base-more-efficiently-with-Revise.jl","page":"Code changes","title":"Modifying base more efficiently with Revise.jl","category":"section","text":"Revise is a package that tracks changes in source files and automatically updates function definitions in your running Julia session. Using it, you can make extensive changes to Base without needing to rebuild in order to test your changes.\n\nHere is the standard procedure:\n\nIf you are planning changes to any types or macros, make those changes and build julia using make. (This is necessary because Revise cannot handle changes to type definitions or macros.) Unless it's required to get Julia to build, you do not have to add any functionality based on the new types, just the type definitions themselves.\nStart a Julia REPL session. Then issue the following commands:\n\nusing Revise    # if you aren't launching it in your `.julia/config/startup.jl`\nRevise.track(Base)\n\nEdit files in base/, save your edits, and test the functionality.\n\nIf you need to restart your Julia session, just start at step 2 above. Revise.track(Base) will note any changes from when Julia was last built and incorporate them automatically. You only need to rebuild Julia if you made code-changes that Revise cannot handle.\n\nFor convenience, there are also test-revise-* targets for every test-* target that use Revise to load any modifications to Base into the current system image before running the corresponding test. This can be useful as a shortcut on the command line (since tests aren't always designed to be run outside the runtest harness)."},{"location":"devdocs/contributing/code-changes.html#Contributing-to-the-standard-library","page":"Code changes","title":"Contributing to the standard library","category":"section","text":"The standard library (stdlib) packages are baked into the Julia system image. When running the ordinary test workflow on the stdlib packages, the system image version overrides the version you are developing. To test stdlib packages, you can do the following steps:\n\nEdit the UUID field of the Project.toml in the stdlib package\nChange the current directory to the directory of the stdlib you are developing\nStart julia with julia --project=.\nYou can now test the package by running pkg> test in Pkg mode.\n\nBecause you changed the UUID, the package manager treats the stdlib package as different from the one in the system image, and the system image version will not override the package.\n\nBe sure to change the UUID value back before making the pull request."},{"location":"devdocs/contributing/code-changes.html#News-worthy-changes","page":"Code changes","title":"News-worthy changes","category":"section","text":"For new functionality and other substantial changes, add a brief summary to NEWS.md. The news item should cross reference the pull request (PR) parenthetically, in the form ([#pr]). To add the PR reference number, first create the PR, then push an additional commit updating NEWS.md with the PR reference number. We periodically run ./julia doc/NEWS-update.jl from the julia directory to update the cross-reference links, but this should not be done in a typical PR in order to avoid conflicting commits."},{"location":"devdocs/contributing/code-changes.html#Annotations-for-new-features,-deprecations-and-behavior-changes","page":"Code changes","title":"Annotations for new features, deprecations and behavior changes","category":"section","text":"API additions and deprecations, and minor behavior changes are allowed in minor version releases. For documented features that are part of the public API, a compatibility note should be added into the manual or the docstring. It should state the Julia minor version that changed the behavior and have a brief message describing the change.\n\nAt the moment, this should always be done with the following compat admonition (so that it would be possible to programmatically find the annotations in the future):\n\n!!! compat \"Julia 1.X\"       This method was added in Julia 1.X."},{"location":"manual/parallel-computing.html#Parallel-Computing","page":"Parallel Computing","title":"Parallel Computing","category":"section","text":"Julia supports these four categories of concurrent and parallel programming:\n\nAsynchronous \"tasks\", or coroutines:\nJulia Tasks allow suspending and resuming computations  for I/O, event handling, producer-consumer processes, and similar patterns.  Tasks can synchronize through operations like wait and fetch, and  communicate via Channels. While strictly not parallel computing by themselves,  Julia lets you schedule Tasks on several threads.\nMulti-threading:\nJulia's multi-threading provides the ability to schedule Tasks  simultaneously on more than one thread or CPU core, sharing memory. This is usually the easiest way  to get parallelism on one's PC or on a single large multi-core server. Julia's multi-threading  is composable. When one multi-threaded function calls another multi-threaded function, Julia  will schedule all the threads globally on available resources, without oversubscribing.\nDistributed computing:\nDistributed computing runs multiple Julia processes with separate memory spaces. These can be on the same  computer or multiple computers. The Distributed standard library provides the capability for remote execution  of a Julia function. With this basic building block, it is possible to build many different kinds of  distributed computing abstractions. Packages like DistributedArrays.jl  are an example of such an abstraction. On the other hand, packages like MPI.jl and  Elemental.jl provide access to the existing MPI ecosystem of libraries.\nGPU computing:\nThe Julia GPU compiler provides the ability to run Julia code natively on GPUs. There  is a rich ecosystem of Julia packages that target GPUs. The JuliaGPU.org  website provides a list of capabilities, supported GPUs, related packages and documentation."},{"location":"devdocs/types.html#More-about-types","page":"More about types","title":"More about types","category":"section","text":"If you've used Julia for a while, you understand the fundamental role that types play. Here we try to get under the hood, focusing particularly on Parametric Types."},{"location":"devdocs/types.html#Types-and-sets-(and-Any-and-Union{}/Bottom)","page":"More about types","title":"Types and sets (and Any and Union{}/Bottom)","category":"section","text":"It's perhaps easiest to conceive of Julia's type system in terms of sets. While programs manipulate individual values, a type refers to a set of values. This is not the same thing as a collection; for example a Set of values is itself a single Set value. Rather, a type describes a set of possible values, expressing uncertainty about which value we have.\n\nA concrete type T describes the set of values whose direct tag, as returned by the typeof function, is T. An abstract type describes some possibly-larger set of values.\n\nAny describes the entire universe of possible values. Integer is a subset of Any that includes Int, Int8, and other concrete types. Internally, Julia also makes heavy use of another type known as Bottom, which can also be written as Union{}. This corresponds to the empty set.\n\nJulia's types support the standard operations of set theory: you can ask whether T1 is a \"subset\" (subtype) of T2 with T1 <: T2. Likewise, you intersect two types using typeintersect, take their union with Union, and compute a type that contains their union with typejoin:\n\njulia> typeintersect(Int, Float64)\nUnion{}\n\njulia> Union{Int, Float64}\nUnion{Float64, Int64}\n\njulia> typejoin(Int, Float64)\nReal\n\njulia> typeintersect(Signed, Union{UInt8, Int8})\nInt8\n\njulia> Union{Signed, Union{UInt8, Int8}}\nUnion{UInt8, Signed}\n\njulia> typejoin(Signed, Union{UInt8, Int8})\nInteger\n\njulia> typeintersect(Tuple{Integer, Float64}, Tuple{Int, Real})\nTuple{Int64, Float64}\n\njulia> Union{Tuple{Integer, Float64}, Tuple{Int, Real}}\nUnion{Tuple{Int64, Real}, Tuple{Integer, Float64}}\n\njulia> typejoin(Tuple{Integer, Float64}, Tuple{Int, Real})\nTuple{Integer, Real}\n\nWhile these operations may seem abstract, they lie at the heart of Julia. For example, method dispatch is implemented by stepping through the items in a method list until reaching one for which the type of the argument tuple is a subtype of the method signature. For this algorithm to work, it's important that methods be sorted by their specificity, and that the search begins with the most specific methods. Consequently, Julia also implements a partial order on types; this is achieved by functionality that is similar to <:, but with differences that will be discussed below."},{"location":"devdocs/types.html#UnionAll-types","page":"More about types","title":"UnionAll types","category":"section","text":"Julia's type system can also express an iterated union of types: a union of types over all values of some variable. This is needed to describe parametric types where the values of some parameters are not known.\n\nFor example, Array has two parameters as in Array{Int,2}. If we did not know the element type, we could write Array{T,2} where T, which is the union of Array{T,2} for all values of T: Union{Array{Int8,2}, Array{Int16,2}, ...}.\n\nSuch a type is represented by a UnionAll object, which contains a variable (T in this example, of type TypeVar), and a wrapped type (Array{T,2} in this example).\n\nConsider the following methods:\n\nf1(A::Array) = 1\nf2(A::Array{Int}) = 2\nf3(A::Array{T}) where {T<:Any} = 3\nf4(A::Array{Any}) = 4\n\nThe signature - as described in Function calls - of f3 is a UnionAll type wrapping a tuple type: Tuple{typeof(f3), Array{T}} where T. All but f4 can be called with a = [1,2]; all but f2 can be called with b = Any[1,2].\n\nLet's look at these types a little more closely:\n\njulia> dump(Array)\nUnionAll\n  var: TypeVar\n    name: Symbol T\n    lb: Union{}\n    ub: abstract type Any\n  body: UnionAll\n    var: TypeVar\n      name: Symbol N\n      lb: Union{}\n      ub: abstract type Any\n    body: mutable struct Array{T, N} <: DenseArray{T, N}\n      ref::MemoryRef{T}\n      size::NTuple{N, Int64}\n\nThis indicates that Array actually names a UnionAll type. There is one UnionAll type for each parameter, nested. The syntax Array{Int,2} is equivalent to Array{Int}{2}; internally each UnionAll is instantiated with a particular variable value, one at a time, outermost-first. This gives a natural meaning to the omission of trailing type parameters; Array{Int} gives a type equivalent to Array{Int,N} where N.\n\nA TypeVar is not itself a type, but rather should be considered part of the structure of a UnionAll type. Type variables have lower and upper bounds on their values (in the fields lb and ub). The symbol name is purely cosmetic. Internally, TypeVars are compared by address, so they are defined as mutable types to ensure that \"different\" type variables can be distinguished. However, by convention they should not be mutated.\n\nOne can construct TypeVars manually:\n\njulia> TypeVar(:V, Signed, Real)\nSigned<:V<:Real\n\nThere are convenience versions that allow you to omit any of these arguments except the name symbol.\n\nThe syntax Array{T} where T<:Integer is lowered to\n\nlet T = TypeVar(:T,Integer)\n    UnionAll(T, Array{T})\nend\n\nso it is seldom necessary to construct a TypeVar manually (indeed, this is to be avoided)."},{"location":"devdocs/types.html#Free-variables","page":"More about types","title":"Free variables","category":"section","text":"The concept of a free type variable is extremely important in the type system. We say that a variable V is free in type T if T does not contain the UnionAll that introduces variable V. For example, the type Array{Array{V} where V<:Integer} has no free variables, but the Array{V} part inside of it does have a free variable, V.\n\nA type with free variables is, in some sense, not really a type at all. Consider the type Array{Array{T}} where T, which refers to all homogeneous arrays of arrays. The inner type Array{T}, seen by itself, might seem to refer to any kind of array. However, every element of the outer array must have the same array type, so Array{T} cannot refer to just any old array. One could say that Array{T} effectively \"occurs\" multiple times, and T must have the same value each \"time\".\n\nFor this reason, the function jl_has_free_typevars in the C API is very important. Types for which it returns true will not give meaningful answers in subtyping and other type functions."},{"location":"devdocs/types.html#TypeNames","page":"More about types","title":"TypeNames","category":"section","text":"The following two Array types are functionally equivalent, yet print differently:\n\njulia> TV, NV = TypeVar(:T), TypeVar(:N)\n(T, N)\n\njulia> Array\nArray\n\njulia> Array{TV, NV}\nArray{T, N}\n\nThese can be distinguished by examining the name field of the type, which is an object of type TypeName:\n\njulia> dump(Array{Int,1}.name)\nTypeName\n  name: Symbol Array\n  module: Module Core\n  singletonname: Symbol Array\n  names: SimpleVector\n    1: Symbol ref\n    2: Symbol size\n  atomicfields: Ptr{Nothing}(0x0000000000000000)\n  constfields: Ptr{Nothing}(0x0000000000000000)\n  wrapper: UnionAll\n    var: TypeVar\n      name: Symbol T\n      lb: Union{}\n      ub: abstract type Any\n    body: UnionAll\n      var: TypeVar\n        name: Symbol N\n        lb: Union{}\n        ub: abstract type Any\n      body: mutable struct Array{T, N} <: DenseArray{T, N}\n  Typeofwrapper: abstract type Type{Array} <: Any\n  cache: SimpleVector\n    ...\n  linearcache: SimpleVector\n    ...\n  hash: Int64 2594190783455944385\n  backedges: #undef\n  partial: #undef\n  max_args: Int32 0\n  n_uninitialized: Int32 0\n  flags: UInt8 0x02\n  cache_entry_count: UInt8 0x00\n  max_methods: UInt8 0x00\n  constprop_heuristic: UInt8 0x00\n\nIn this case, the relevant field is wrapper, which holds a reference to the top-level type used to make new Array types.\n\njulia> pointer_from_objref(Array)\nPtr{Cvoid} @0x00007fcc7de64850\n\njulia> pointer_from_objref(Array.body.body.name.wrapper)\nPtr{Cvoid} @0x00007fcc7de64850\n\njulia> pointer_from_objref(Array{TV,NV})\nPtr{Cvoid} @0x00007fcc80c4d930\n\njulia> pointer_from_objref(Array{TV,NV}.name.wrapper)\nPtr{Cvoid} @0x00007fcc7de64850\n\nThe wrapper field of Array points to itself, but for Array{TV,NV} it points back to the original definition of the type.\n\nWhat about the other fields? hash assigns an integer to each type. To examine the cache field, it's helpful to pick a type that is less heavily used than Array. Let's first create our own type:\n\njulia> struct MyType{T,N} end\n\njulia> MyType{Int,2}\nMyType{Int64, 2}\n\njulia> MyType{Float32, 5}\nMyType{Float32, 5}\n\nWhen you instantiate a parametric type, each concrete type gets saved in a type cache (MyType.body.body.name.cache). However, instances containing free type variables are not cached."},{"location":"devdocs/types.html#Tuple-types","page":"More about types","title":"Tuple types","category":"section","text":"Tuple types constitute an interesting special case. For dispatch to work on declarations like x::Tuple, the type has to be able to accommodate any tuple. Let's check the parameters:\n\njulia> Tuple\nTuple\n\njulia> Tuple.parameters\nsvec(Vararg{Any})\n\nUnlike other types, tuple types are covariant in their parameters, so this definition permits Tuple to match any type of tuple:\n\njulia> typeintersect(Tuple, Tuple{Int,Float64})\nTuple{Int64, Float64}\n\njulia> typeintersect(Tuple{Vararg{Any}}, Tuple{Int,Float64})\nTuple{Int64, Float64}\n\nHowever, if a variadic (Vararg) tuple type has free variables it can describe different kinds of tuples:\n\njulia> typeintersect(Tuple{Vararg{T} where T}, Tuple{Int,Float64})\nTuple{Int64, Float64}\n\njulia> typeintersect(Tuple{Vararg{T}} where T, Tuple{Int,Float64})\nUnion{}\n\nNotice that when T is free with respect to the Tuple type (i.e. its binding UnionAll type is outside the Tuple type), only one T value must work over the whole type. Therefore a heterogeneous tuple does not match.\n\nFinally, it's worth noting that Tuple{} is distinct:\n\njulia> Tuple{}\nTuple{}\n\njulia> Tuple{}.parameters\nsvec()\n\njulia> typeintersect(Tuple{}, Tuple{Int})\nUnion{}\n\nWhat is the \"primary\" tuple-type?\n\njulia> pointer_from_objref(Tuple)\nPtr{Cvoid} @0x00007f5998a04370\n\njulia> pointer_from_objref(Tuple{})\nPtr{Cvoid} @0x00007f5998a570d0\n\njulia> pointer_from_objref(Tuple.name.wrapper)\nPtr{Cvoid} @0x00007f5998a04370\n\njulia> pointer_from_objref(Tuple{}.name.wrapper)\nPtr{Cvoid} @0x00007f5998a04370\n\nso Tuple == Tuple{Vararg{Any}} is indeed the primary type."},{"location":"devdocs/types.html#Diagonal-types","page":"More about types","title":"Diagonal types","category":"section","text":"Consider the type Tuple{T,T} where T. A method with this signature would look like:\n\nf(x::T, y::T) where {T} = ...\n\nAccording to the usual interpretation of a UnionAll type, this T ranges over all types, including Any, so this type should be equivalent to Tuple{Any,Any}. However, this interpretation causes some practical problems.\n\nFirst, a value of T needs to be available inside the method definition. For a call like f(1, 1.0), it's not clear what T should be. It could be Union{Int,Float64}, or perhaps Real. Intuitively, we expect the declaration x::T to mean T === typeof(x). To make sure that invariant holds, we need typeof(x) === typeof(y) === T in this method. That implies the method should only be called for arguments of the exact same type.\n\nIt turns out that being able to dispatch on whether two values have the same type is very useful (this is used by the promotion system for example), so we have multiple reasons to want a different interpretation of Tuple{T,T} where T. To make this work we add the following rule to subtyping: if a variable occurs more than once in covariant position, it is restricted to ranging over only concrete types. (\"Covariant position\" means that only Tuple and Union types occur between an occurrence of a variable and the UnionAll type that introduces it.) Such variables are called \"diagonal variables\" or \"concrete variables\".\n\nSo for example, Tuple{T,T} where T can be seen as Union{Tuple{Int8,Int8}, Tuple{Int16,Int16}, ...}, where T ranges over all concrete types. This gives rise to some interesting subtyping results. For example Tuple{Real,Real} is not a subtype of Tuple{T,T} where T, because it includes some types like Tuple{Int8,Int16} where the two elements have different types. Tuple{Real,Real} and Tuple{T,T} where T have the non-trivial intersection Tuple{T,T} where T<:Real. However, Tuple{Real} is a subtype of Tuple{T} where T, because in that case T occurs only once and so is not diagonal.\n\nNext consider a signature like the following:\n\nf(a::Array{T}, x::T, y::T) where {T} = ...\n\nIn this case, T occurs in invariant position inside Array{T}. That means whatever type of array is passed unambiguously determines the value of T – we say T has an equality constraint on it. Therefore in this case the diagonal rule is not really necessary, since the array determines T and we can then allow x and y to be of any subtypes of T. So variables that occur in invariant position are never considered diagonal. This choice of behavior is slightly controversial – some feel this definition should be written as\n\nf(a::Array{T}, x::S, y::S) where {T, S<:T} = ...\n\nto clarify whether x and y need to have the same type. In this version of the signature they would, or we could introduce a third variable for the type of y if x and y can have different types.\n\nThe next complication is the interaction of unions and diagonal variables, e.g.\n\nf(x::Union{Nothing,T}, y::T) where {T} = ...\n\nConsider what this declaration means. y has type T. x then can have either the same type T, or else be of type Nothing. So all of the following calls should match:\n\nf(1, 1)\nf(\"\", \"\")\nf(2.0, 2.0)\nf(nothing, 1)\nf(nothing, \"\")\nf(nothing, 2.0)\n\nThese examples are telling us something: when x is nothing::Nothing, there are no extra constraints on y. It is as if the method signature had y::Any. Indeed, we have the following type equivalence:\n\n(Tuple{Union{Nothing,T},T} where T) == Union{Tuple{Nothing,Any}, Tuple{T,T} where T}\n\nThe general rule is: a concrete variable in covariant position acts like it's not concrete if the subtyping algorithm only uses it once. When x has type Nothing, we don't need to use the T in Union{Nothing,T}; we only use it in the second slot. This arises naturally from the observation that in Tuple{T} where T restricting T to concrete types makes no difference; the type is equal to Tuple{Any} either way.\n\nHowever, appearing in invariant position disqualifies a variable from being concrete whether that appearance of the variable is used or not. Otherwise types can behave differently depending on which other types they are compared to, making subtyping not transitive. For example, consider\n\nTuple{Int,Int8,Vector{Integer}} <: Tuple{T,T,Vector{Union{Integer,T}}} where T\n\nIf the T inside the Union is ignored, then T is concrete and the answer is \"false\" since the first two types aren't the same. But consider instead\n\nTuple{Int,Int8,Vector{Any}} <: Tuple{T,T,Vector{Union{Integer,T}}} where T\n\nNow we cannot ignore the T in the Union (we must have T == Any), so T is not concrete and the answer is \"true\". That would make the concreteness of T depend on the other type, which is not acceptable since a type must have a clear meaning on its own. Therefore the appearance of T inside Vector is considered in both cases."},{"location":"devdocs/types.html#Subtyping-diagonal-variables","page":"More about types","title":"Subtyping diagonal variables","category":"section","text":"The subtyping algorithm for diagonal variables has two components: (1) identifying variable occurrences, and (2) ensuring that diagonal variables range over concrete types only.\n\nThe first task is accomplished by keeping counters occurs_inv and occurs_cov (in src/subtype.c) for each variable in the environment, tracking the number of invariant and covariant occurrences, respectively. A variable is diagonal when occurs_inv == 0 && occurs_cov > 1.\n\nThe second task is accomplished by imposing a condition on a variable's lower bound. As the subtyping algorithm runs, it narrows the bounds of each variable (raising lower bounds and lowering upper bounds) to keep track of the range of variable values for which the subtype relation would hold. When we are done evaluating the body of a UnionAll type whose variable is diagonal, we look at the final values of the bounds. Since the variable must be concrete, a contradiction occurs if its lower bound could not be a subtype of a concrete type. For example, an abstract type like AbstractArray cannot be a subtype of a concrete type, but a concrete type like Int can be, and the empty type Bottom can be as well. If a lower bound fails this test the algorithm stops with the answer false.\n\nFor example, in the problem Tuple{Int,String} <: Tuple{T,T} where T, we derive that this would be true if T were a supertype of Union{Int,String}. However, Union{Int,String} is an abstract type, so the relation does not hold.\n\nThis concreteness test is done by the function is_leaf_bound. Note that this test is slightly different from jl_is_leaf_type, since it also returns true for Bottom. Currently this function is heuristic, and does not catch all possible concrete types. The difficulty is that whether a lower bound is concrete might depend on the values of other type variable bounds. For example, Vector{T} is equivalent to the concrete type Vector{Int} only if both the upper and lower bounds of T equal Int. We have not yet worked out a complete algorithm for this."},{"location":"devdocs/types.html#Introduction-to-the-internal-machinery","page":"More about types","title":"Introduction to the internal machinery","category":"section","text":"Most operations for dealing with types are found in the files jltypes.c and subtype.c. A good way to start is to watch subtyping in action. Build Julia with make debug and fire up Julia within a debugger. gdb debugging tips has some tips which may be useful.\n\nBecause the subtyping code is used heavily in the REPL itself – and hence breakpoints in this code get triggered often – it will be easiest if you make the following definition:\n\njulia> function mysubtype(a,b)\n           ccall(:jl_breakpoint, Cvoid, (Any,), nothing)\n           a <: b\n       end\n\nand then set a breakpoint in jl_breakpoint. Once this breakpoint gets triggered, you can set breakpoints in other functions.\n\nAs a warm-up, try the following:\n\nmysubtype(Tuple{Int, Float64}, Tuple{Integer, Real})\n\nWe can make it more interesting by trying a more complex case:\n\nmysubtype(Tuple{Array{Int,2}, Int8}, Tuple{Array{T}, T} where T)"},{"location":"devdocs/types.html#Subtyping-and-method-sorting","page":"More about types","title":"Subtyping and method sorting","category":"section","text":"The type_morespecific functions are used for imposing a partial order on functions in method tables (from most-to-least specific). Specificity is strict; if a is more specific than b, then a does not equal b and b is not more specific than a.\n\nIf a is a strict subtype of b, then it is automatically considered more specific. From there, type_morespecific employs some less formal rules. For example, subtype is sensitive to the number of arguments, but type_morespecific may not be. In particular, Tuple{Int,AbstractFloat} is more specific than Tuple{Integer}, even though it is not a subtype.  (Of Tuple{Int,AbstractFloat} and Tuple{Integer,Float64}, neither is more specific than the other.)  Likewise, Tuple{Int,Vararg{Int}} is not a subtype of Tuple{Integer}, but it is considered more specific. However, morespecific does get a bonus for length: in particular, Tuple{Int,Int} is more specific than Tuple{Int,Vararg{Int}}.\n\nAdditionally, if 2 methods are defined with identical signatures, per type-equal, then they will instead be compared by order of addition, such that the later method is more specific than the earlier one."},{"location":"stdlib/ArgTools.html#ArgTools","page":"ArgTools","title":"ArgTools","category":"section","text":""},{"location":"stdlib/ArgTools.html#Argument-Handling","page":"ArgTools","title":"Argument Handling","category":"section","text":""},{"location":"stdlib/ArgTools.html#Function-Testing","page":"ArgTools","title":"Function Testing","category":"section","text":""},{"location":"stdlib/ArgTools.html#ArgTools.ArgRead","page":"ArgTools","title":"ArgTools.ArgRead","category":"type","text":"ArgRead = Union{AbstractString, AbstractCmd, IO}\n\nThe ArgRead types is a union of the types that the arg_read function knows how to convert into readable IO handles. See arg_read for details.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.ArgWrite","page":"ArgTools","title":"ArgTools.ArgWrite","category":"type","text":"ArgWrite = Union{AbstractString, AbstractCmd, IO}\n\nThe ArgWrite types is a union of the types that the arg_write function knows how to convert into writeable IO handles, except for Nothing which arg_write handles by generating a temporary file. See arg_write for details.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_read","page":"ArgTools","title":"ArgTools.arg_read","category":"function","text":"arg_read(f::Function, arg::ArgRead) -> f(arg_io)\n\nThe arg_read function accepts an argument arg that can be any of these:\n\nAbstractString: a file path to be opened for reading\nAbstractCmd: a command to be run, reading from its standard output\nIO: an open IO handle to be read from\n\nWhether the body returns normally or throws an error, a path which is opened will be closed before returning from arg_read and an IO handle will be flushed but not closed before returning from arg_read.\n\nNote: when opening a file, ArgTools will pass lock = false to the file open(...) call. Therefore, the object returned by this function should not be used from multiple threads. This restriction may be relaxed in the future, which would not break any working code.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_write","page":"ArgTools","title":"ArgTools.arg_write","category":"function","text":"arg_write(f::Function, arg::ArgWrite) -> arg\narg_write(f::Function, arg::Nothing) -> tempname()\n\nThe arg_write function accepts an argument arg that can be any of these:\n\nAbstractString: a file path to be opened for writing\nAbstractCmd: a command to be run, writing to its standard input\nIO: an open IO handle to be written to\nNothing: a temporary path should be written to\n\nIf the body returns normally, a path that is opened will be closed upon completion; an IO handle argument is left open but flushed before return. If the argument is nothing then a temporary path is opened for writing and closed open completion and the path is returned from arg_write. In all other cases, arg itself is returned. This is a useful pattern since you can consistently return whatever was written, whether an argument was passed or not.\n\nIf there is an error during the evaluation of the body, a path that is opened by arg_write for writing will be deleted, whether it's passed in as a string or a temporary path generated when arg is nothing.\n\nNote: when opening a file, ArgTools will pass lock = false to the file open(...) call. Therefore, the object returned by this function should not be used from multiple threads. This restriction may be relaxed in the future, which would not break any working code.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_isdir","page":"ArgTools","title":"ArgTools.arg_isdir","category":"function","text":"arg_isdir(f::Function, arg::AbstractString) -> f(arg)\n\nThe arg_isdir function takes arg which must be the path to an existing directory (an error is raised otherwise) and passes that path to f finally returning the result of f(arg). This is definitely the least useful tool offered by ArgTools and mostly exists for symmetry with arg_mkdir and to give consistent error messages.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_mkdir","page":"ArgTools","title":"ArgTools.arg_mkdir","category":"function","text":"arg_mkdir(f::Function, arg::AbstractString) -> arg\narg_mkdir(f::Function, arg::Nothing) -> mktempdir()\n\nThe arg_mkdir function takes arg which must either be one of:\n\na path to an already existing empty directory,\na non-existent path which can be created as a directory, or\nnothing in which case a temporary directory is created.\n\nIn all cases the path to the directory is returned. If an error occurs during f(arg), the directory is returned to its original state: if it already existed but was empty, it will be emptied; if it did not exist it will be deleted.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_readers","page":"ArgTools","title":"ArgTools.arg_readers","category":"function","text":"arg_readers(arg :: AbstractString, [ type = ArgRead ]) do arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        arg :: ArgRead\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThe arg_readers function takes a path to be read and a single-argument do block, which is invoked once for each test reader type that arg_read can handle. If the optional type argument is given then the do block is only invoked for readers that produce arguments of that type.\n\nThe arg passed to the do block is not the argument value itself, because some of test argument types need to be initialized and finalized for each test case. Consider an open file handle argument: once you've used it for one test, you can't use it again; you need to close it and open the file again for the next test. This function arg can be converted into an ArgRead instance using @arg_test arg begin ... end.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.arg_writers","page":"ArgTools","title":"ArgTools.arg_writers","category":"function","text":"arg_writers([ type = ArgWrite ]) do path::String, arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        arg :: ArgWrite\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThe arg_writers function takes a do block, which is invoked once for each test writer type that arg_write can handle with a temporary (non-existent) path and arg which can be converted into various writable argument types which write to path. If the optional type argument is given then the do block is only invoked for writers that produce arguments of that type.\n\nThe arg passed to the do block is not the argument value itself, because some of test argument types need to be initialized and finalized for each test case. Consider an open file handle argument: once you've used it for one test, you can't use it again; you need to close it and open the file again for the next test. This function arg can be converted into an ArgWrite instance using @arg_test arg begin ... end.\n\nThere is also an arg_writers method that takes a path name like arg_readers:\n\narg_writers(path::AbstractString, [ type = ArgWrite ]) do arg::Function\n    ## pre-test setup ##\n    @arg_test arg begin\n        # here `arg :: ArgWrite`\n        ## test using `arg` ##\n    end\n    ## post-test cleanup ##\nend\n\nThis method is useful if you need to specify path instead of using path name generated by tempname(). Since path is passed from outside of arg_writers, the path is not an argument to the do block in this form.\n\n\n\n\n\n"},{"location":"stdlib/ArgTools.html#ArgTools.@arg_test","page":"ArgTools","title":"ArgTools.@arg_test","category":"macro","text":"@arg_test arg1 arg2 ... body\n\nThe @arg_test macro is used to convert arg functions provided by arg_readers and arg_writers into actual argument values. When you write @arg_test arg body it is equivalent to arg(arg -> body).\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA","page":"SHA","title":"SHA","category":"section","text":""},{"location":"stdlib/SHA.html#SHA-functions","page":"SHA","title":"SHA functions","category":"section","text":"Usage is very straightforward:\n\njulia> using SHA\n\njulia> bytes2hex(sha256(\"test\"))\n\"9f86d081884c7d659a2feaa0c55ad015a3bf4f1b2b0b822cd15d6c15b0f00a08\"\n\nEach exported function (at the time of this writing, SHA-1, SHA-2 224, 256, 384 and 512, and SHA-3 224, 256, 384 and 512 functions are implemented) takes in either an AbstractVector{UInt8}, an AbstractString or an IO object.  This makes it trivial to checksum a file:\n\nshell> cat /tmp/test.txt\ntest\njulia> using SHA\n\njulia> open(\"/tmp/test.txt\") do f\n           sha2_256(f)\n       end\n32-element Vector{UInt8}:\n 0x9f\n 0x86\n 0xd0\n 0x81\n 0x88\n 0x4c\n 0x7d\n 0x65\n    ⋮\n 0x5d\n 0x6c\n 0x15\n 0xb0\n 0xf0\n 0x0a\n 0x08"},{"location":"stdlib/SHA.html#All-SHA-functions","page":"SHA","title":"All SHA functions","category":"section","text":"Due to the colloquial usage of sha256 to refer to sha2_256, convenience functions are provided, mapping shaxxx() function calls to sha2_xxx(). For SHA-3, no such colloquialisms exist and the user must use the full sha3_xxx() names.\n\nshaxxx() takes AbstractString and array-like objects (NTuple and Vector) with elements of type UInt8.\n\nSHA-1\n\nSHA-2\n\nSHA-3"},{"location":"stdlib/SHA.html#Working-with-context","page":"SHA","title":"Working with context","category":"section","text":"To create a hash from multiple items the SHAX_XXX_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!\n\njulia> using SHA\n\njulia> ctx = SHA2_256_CTX()\nSHA2 256-bit hash state\n\njulia> update!(ctx, b\"some data\")\n0x0000000000000009\n\njulia> update!(ctx, b\"some more data\")\n0x0000000000000017\n\njulia> digest!(ctx)\n32-element Vector{UInt8}:\n 0xbe\n 0xcf\n 0x23\n 0xda\n 0xaf\n 0x02\n 0xf7\n 0xa3\n 0x57\n 0x92\n    ⋮\n 0x89\n 0x4f\n 0x59\n 0xd8\n 0xb3\n 0xb4\n 0x81\n 0x8b\n 0xc5\n\nNote that, at the time of this writing, the SHA3 code is not optimized, and as such is roughly an order of magnitude slower than SHA2."},{"location":"stdlib/SHA.html#All-SHA-context-types","page":"SHA","title":"All SHA context types","category":"section","text":"SHA-1\n\nSHA-2\n\nConvenience types are also provided, where SHAXXX_CTX is a type alias for SHA2_XXX_CTX.\n\nSHA-3"},{"location":"stdlib/SHA.html#HMAC-functions","page":"SHA","title":"HMAC functions","category":"section","text":"julia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> bytes2hex(hmac_sha3_256(key, \"test-message\"))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\"\n\nTo create a hash from multiple items, the HMAC_CTX() types can be used to create a stateful hash object that is updated with update! and finalized with digest!.\n\njulia> using SHA\n\njulia> key = collect(codeunits(\"key_string\"))\n10-element Vector{UInt8}:\n 0x6b\n 0x65\n 0x79\n 0x5f\n 0x73\n 0x74\n 0x72\n 0x69\n 0x6e\n 0x67\n\njulia> ctx = HMAC_CTX(SHA3_256_CTX(), key);\n\njulia> update!(ctx, b\"test-\")\n0x0000000000000000000000000000008d\n\njulia> update!(ctx, b\"message\")\n0x00000000000000000000000000000094\n\njulia> bytes2hex(digest!(ctx))\n\"bc49a6f2aa29b27ee5ed1e944edd7f3d153e8a01535d98b5e24dac9a589a6248\""},{"location":"stdlib/SHA.html#All-HMAC-functions","page":"SHA","title":"All HMAC functions","category":"section","text":"HMAC context type\n\nSHA-1\n\nSHA-2\n\nSHA-3"},{"location":"stdlib/SHA.html#SHA.sha1","page":"SHA","title":"SHA.sha1","category":"function","text":"sha1(data)\n\nHash data using the sha1 algorithm and return the resulting digest. See also SHA1_CTX.\n\n\n\n\n\nsha1(io::IO)\n\nHash data from io using sha1 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha224","page":"SHA","title":"SHA.sha224","category":"function","text":"sha224(data)\n\nHash data using the sha224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha224(io::IO)\n\nHash data from io using sha224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha256","page":"SHA","title":"SHA.sha256","category":"function","text":"sha256(data)\n\nHash data using the sha256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha256(io::IO)\n\nHash data from io using sha256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha384","page":"SHA","title":"SHA.sha384","category":"function","text":"sha384(data)\n\nHash data using the sha384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha384(io::IO)\n\nHash data from io using sha384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha512","page":"SHA","title":"SHA.sha512","category":"function","text":"sha512(data)\n\nHash data using the sha512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha512(io::IO)\n\nHash data from io using sha512 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_224","page":"SHA","title":"SHA.sha2_224","category":"function","text":"sha2_224(data)\n\nHash data using the sha2_224 algorithm and return the resulting digest. See also SHA2_224_CTX.\n\n\n\n\n\nsha2_224(io::IO)\n\nHash data from io using sha2_224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_256","page":"SHA","title":"SHA.sha2_256","category":"function","text":"sha2_256(data)\n\nHash data using the sha2_256 algorithm and return the resulting digest. See also SHA2_256_CTX.\n\n\n\n\n\nsha2_256(io::IO)\n\nHash data from io using sha2_256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_384","page":"SHA","title":"SHA.sha2_384","category":"function","text":"sha2_384(data)\n\nHash data using the sha2_384 algorithm and return the resulting digest. See also SHA2_384_CTX.\n\n\n\n\n\nsha2_384(io::IO)\n\nHash data from io using sha2_384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_512","page":"SHA","title":"SHA.sha2_512","category":"function","text":"sha2_512(data)\n\nHash data using the sha2_512 algorithm and return the resulting digest. See also SHA2_512_CTX.\n\n\n\n\n\nsha2_512(io::IO)\n\nHash data from io using sha2_512 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_512_224","page":"SHA","title":"SHA.sha2_512_224","category":"function","text":"sha2_512_224(data)\n\nHash data using the sha2_512_224 algorithm and return the resulting digest. See also SHA2_512_224_CTX.\n\n\n\n\n\nsha2_512_224(io::IO)\n\nHash data from io using sha2_512_224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha2_512_256","page":"SHA","title":"SHA.sha2_512_256","category":"function","text":"sha2_512_256(data)\n\nHash data using the sha2_512_256 algorithm and return the resulting digest. See also SHA2_512_256_CTX.\n\n\n\n\n\nsha2_512_256(io::IO)\n\nHash data from io using sha2_512_256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha3_224","page":"SHA","title":"SHA.sha3_224","category":"function","text":"sha3_224(data)\n\nHash data using the sha3_224 algorithm and return the resulting digest. See also SHA3_224_CTX.\n\n\n\n\n\nsha3_224(io::IO)\n\nHash data from io using sha3_224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha3_256","page":"SHA","title":"SHA.sha3_256","category":"function","text":"sha3_256(data)\n\nHash data using the sha3_256 algorithm and return the resulting digest. See also SHA3_256_CTX.\n\n\n\n\n\nsha3_256(io::IO)\n\nHash data from io using sha3_256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha3_384","page":"SHA","title":"SHA.sha3_384","category":"function","text":"sha3_384(data)\n\nHash data using the sha3_384 algorithm and return the resulting digest. See also SHA3_384_CTX.\n\n\n\n\n\nsha3_384(io::IO)\n\nHash data from io using sha3_384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.sha3_512","page":"SHA","title":"SHA.sha3_512","category":"function","text":"sha3_512(data)\n\nHash data using the sha3_512 algorithm and return the resulting digest. See also SHA3_512_CTX.\n\n\n\n\n\nsha3_512(io::IO)\n\nHash data from io using sha3_512 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.update!","page":"SHA","title":"SHA.update!","category":"function","text":"update!(context, data[, datalen])\n\nUpdate the SHA context with the bytes in data. See also digest! for finalizing the hash.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.digest!","page":"SHA","title":"SHA.digest!","category":"function","text":"digest!(context)\n\nFinalize the SHA context and return the hash as array of bytes (Vector{Uint8}). Updating the context after calling digest! on it will error.\n\nExamples\n\njulia> ctx = SHA1_CTX()\nSHA1 hash state\n\njulia> update!(ctx, b\"data to to be hashed\")\n\njulia> digest!(ctx)\n20-element Vector{UInt8}:\n 0x83\n 0xe4\n ⋮\n 0x89\n 0xf5\n\njulia> update!(ctx, b\"more data\")\nERROR: Cannot update CTX after `digest!` has been called on it\n[...]\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA1_CTX","page":"SHA","title":"SHA.SHA1_CTX","category":"type","text":"SHA1_CTX()\n\nConstruct an empty SHA1 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA224_CTX","page":"SHA","title":"SHA.SHA224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA256_CTX","page":"SHA","title":"SHA.SHA256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA384_CTX","page":"SHA","title":"SHA.SHA384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA512_CTX","page":"SHA","title":"SHA.SHA512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_224_CTX","page":"SHA","title":"SHA.SHA2_224_CTX","category":"type","text":"SHA2_224_CTX()\n\nConstruct an empty SHA2_224 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_256_CTX","page":"SHA","title":"SHA.SHA2_256_CTX","category":"type","text":"SHA2_256_CTX()\n\nConstruct an empty SHA2_256 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_384_CTX","page":"SHA","title":"SHA.SHA2_384_CTX","category":"type","text":"SHA2_384()\n\nConstruct an empty SHA2_384 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_512_CTX","page":"SHA","title":"SHA.SHA2_512_CTX","category":"type","text":"SHA2_512_CTX()\n\nConstruct an empty SHA2_512 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_512_224_CTX","page":"SHA","title":"SHA.SHA2_512_224_CTX","category":"type","text":"SHA2_512_224_CTX()\n\nConstruct an empty SHA2_512/224 context and set the initial hash value.\n\nFor the source of the initial value, refer to FIPS 180-4, 5.3.6.1 SHA-512/224\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA2_512_256_CTX","page":"SHA","title":"SHA.SHA2_512_256_CTX","category":"type","text":"SHA2_512_256_CTX()\n\nConstruct an empty SHA2_512/256 context and set the initial hash value.\n\nFor the source of the initial value, refer to FIPS 180-4, 5.3.6.2 SHA-512/256\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA3_224_CTX","page":"SHA","title":"SHA.SHA3_224_CTX","category":"type","text":"SHA3_224_CTX()\n\nConstruct an empty SHA3_224 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA3_256_CTX","page":"SHA","title":"SHA.SHA3_256_CTX","category":"type","text":"SHA3_256_CTX()\n\nConstruct an empty SHA3_256 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA3_384_CTX","page":"SHA","title":"SHA.SHA3_384_CTX","category":"type","text":"SHA3_384_CTX()\n\nConstruct an empty SHA3_384 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.SHA3_512_CTX","page":"SHA","title":"SHA.SHA3_512_CTX","category":"type","text":"SHA3_512_CTX()\n\nConstruct an empty SHA3_512 context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.HMAC_CTX","page":"SHA","title":"SHA.HMAC_CTX","category":"type","text":"HMAC_CTX(ctx::CTX, key::Vector{UInt8}) where {CTX<:SHA_CTX}\n\nConstruct an empty HMAC_CTX context.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha1","page":"SHA","title":"SHA.hmac_sha1","category":"function","text":"hmac_sha1(key, data)\n\nHash data using the sha1 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha1(key, io::IO)\n\nHash data from io with the passed key using sha1 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha224","page":"SHA","title":"SHA.hmac_sha224","category":"function","text":"hmac_sha224(key, data)\n\nHash data using the sha224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha224(key, io::IO)\n\nHash data from io with the passed key using sha224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha256","page":"SHA","title":"SHA.hmac_sha256","category":"function","text":"hmac_sha256(key, data)\n\nHash data using the sha256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha256(key, io::IO)\n\nHash data from io with the passed key using sha256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha384","page":"SHA","title":"SHA.hmac_sha384","category":"function","text":"hmac_sha384(key, data)\n\nHash data using the sha384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha384(key, io::IO)\n\nHash data from io with the passed key using sha384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha512","page":"SHA","title":"SHA.hmac_sha512","category":"function","text":"hmac_sha512(key, data)\n\nHash data using the sha512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha512(key, io::IO)\n\nHash data from io with the passed key using sha512 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha2_224","page":"SHA","title":"SHA.hmac_sha2_224","category":"function","text":"hmac_sha2_224(key, data)\n\nHash data using the sha2_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_224(key, io::IO)\n\nHash data from io with the passed key using sha2_224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha2_256","page":"SHA","title":"SHA.hmac_sha2_256","category":"function","text":"hmac_sha2_256(key, data)\n\nHash data using the sha2_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_256(key, io::IO)\n\nHash data from io with the passed key using sha2_256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha2_384","page":"SHA","title":"SHA.hmac_sha2_384","category":"function","text":"hmac_sha2_384(key, data)\n\nHash data using the sha2_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_384(key, io::IO)\n\nHash data from io with the passed key using sha2_384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha2_512","page":"SHA","title":"SHA.hmac_sha2_512","category":"function","text":"hmac_sha2_512(key, data)\n\nHash data using the sha2_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha2_512(key, io::IO)\n\nHash data from io with the passed key using sha2_512 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha3_224","page":"SHA","title":"SHA.hmac_sha3_224","category":"function","text":"hmac_sha3_224(key, data)\n\nHash data using the sha3_224 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_224(key, io::IO)\n\nHash data from io with the passed key using sha3_224 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha3_256","page":"SHA","title":"SHA.hmac_sha3_256","category":"function","text":"hmac_sha3_256(key, data)\n\nHash data using the sha3_256 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_256(key, io::IO)\n\nHash data from io with the passed key using sha3_256 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha3_384","page":"SHA","title":"SHA.hmac_sha3_384","category":"function","text":"hmac_sha3_384(key, data)\n\nHash data using the sha3_384 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_384(key, io::IO)\n\nHash data from io with the passed key using sha3_384 algorithm.\n\n\n\n\n\n"},{"location":"stdlib/SHA.html#SHA.hmac_sha3_512","page":"SHA","title":"SHA.hmac_sha3_512","category":"function","text":"hmac_sha3_512(key, data)\n\nHash data using the sha3_512 algorithm using the passed key. See also HMAC_CTX.\n\n\n\n\n\nhmac_sha3_512(key, io::IO)\n\nHash data from io with the passed key using sha3_512 algorithm.\n\n\n\n\n\n"},{"location":"manual/distributed-computing.html#Multi-processing-and-Distributed-Computing","page":"Multi-processing and Distributed Computing","title":"Multi-processing and Distributed Computing","category":"section","text":"An implementation of distributed memory parallel computing is provided by module Distributed as part of the standard library shipped with Julia.\n\nMost modern computers possess more than one CPU, and several computers can be combined together in a cluster. Harnessing the power of these multiple CPUs allows many computations to be completed more quickly. There are two major factors that influence performance: the speed of the CPUs themselves, and the speed of their access to memory. In a cluster, it's fairly obvious that a given CPU will have fastest access to the RAM within the same computer (node). Perhaps more surprisingly, similar issues are relevant on a typical multicore laptop, due to differences in the speed of main memory and the cache. Consequently, a good multiprocessing environment should allow control over the \"ownership\" of a chunk of memory by a particular CPU. Julia provides a multiprocessing environment based on message passing to allow programs to run on multiple processes in separate memory domains at once.\n\nJulia's implementation of message passing is different from other environments such as MPI[1]. Communication in Julia is generally \"one-sided\", meaning that the programmer needs to explicitly manage only one process in a two-process operation. Furthermore, these operations typically do not look like \"message send\" and \"message receive\" but rather resemble higher-level operations like calls to user functions.\n\nDistributed programming in Julia is built on two primitives: remote references and remote calls. A remote reference is an object that can be used from any process to refer to an object stored on a particular process. A remote call is a request by one process to call a certain function on certain arguments on another (possibly the same) process.\n\nRemote references come in two flavors: Future and RemoteChannel.\n\nA remote call returns a Future to its result. Remote calls return immediately; the process that made the call proceeds to its next operation while the remote call happens somewhere else. You can wait for a remote call to finish by calling wait on the returned Future, and you can obtain the full value of the result using fetch.\n\nOn the other hand, RemoteChannel s are rewritable. For example, multiple processes can coordinate their processing by referencing the same remote Channel.\n\nEach process has an associated identifier. The process providing the interactive Julia prompt always has an id equal to 1. The processes used by default for parallel operations are referred to as \"workers\". When there is only one process, process 1 is considered a worker. Otherwise, workers are considered to be all processes other than process 1. As a result, adding 2 or more processes is required to gain benefits from parallel processing methods like pmap. Adding a single process is beneficial if you just wish to do other things in the main process while a long computation is running on the worker.\n\nLet's try this out. Starting with julia -p n provides n worker processes on the local machine. Generally it makes sense for n to equal the number of CPU threads (logical cores) on the machine. Note that the -p argument implicitly loads module Distributed.\n\n$ julia -p 2\n\njulia> r = remotecall(rand, 2, 2, 2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat 2 1 .+ fetch(r)\nFuture(2, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.18526  1.50912\n 1.16296  1.60607\n\nThe first argument to remotecall is the function to call. Most parallel programming in Julia does not reference specific processes or the number of processes available, but remotecall is considered a low-level interface providing finer control. The second argument to remotecall is the id of the process that will do the work, and the remaining arguments will be passed to the function being called.\n\nAs you can see, in the first line we asked process 2 to construct a 2-by-2 random matrix, and in the second line we asked it to add 1 to it. The result of both calculations is available in the two futures, r and s. The @spawnat macro evaluates the expression in the second argument on the process specified by the first argument.\n\nOccasionally you might want a remotely-computed value immediately. This typically happens when you read from a remote object to obtain data needed by the next local operation. The function remotecall_fetch exists for this purpose. It is equivalent to fetch(remotecall(...)) but is more efficient.\n\njulia> remotecall_fetch(r-> fetch(r)[1, 1], 2, r)\n0.18526337335308085\n\nThis fetches the array on worker 2 and returns the first value. Note, that fetch doesn't move any data in this case, since it's executed on the worker that owns the array. One can also write:\n\njulia> remotecall_fetch(getindex, 2, r, 1, 1)\n0.10824216411304866\n\nRemember that getindex(r,1,1) is equivalent to r[1,1], so this call fetches the first element of the future r.\n\nTo make things easier, the symbol :any can be passed to @spawnat, which picks where to do the operation for you:\n\njulia> r = @spawnat :any rand(2,2)\nFuture(2, 1, 4, nothing)\n\njulia> s = @spawnat :any 1 .+ fetch(r)\nFuture(3, 1, 5, nothing)\n\njulia> fetch(s)\n2×2 Matrix{Float64}:\n 1.38854  1.9098\n 1.20939  1.57158\n\nNote that we used 1 .+ fetch(r) instead of 1 .+ r. This is because we do not know where the code will run, so in general a fetch might be required to move r to the process doing the addition. In this case, @spawnat is smart enough to perform the computation on the process that owns r, so the fetch will be a no-op (no work is done).\n\n(It is worth noting that @spawnat is not built-in but defined in Julia as a macro. It is possible to define your own such constructs.)\n\nAn important thing to remember is that, once fetched, a Future will cache its value locally. Further fetch calls do not entail a network hop. Once all referencing Futures have fetched, the remote stored value is deleted.\n\nThreads.@spawn is similar to @spawnat, but only runs tasks on the local process. We use it to create a \"feeder\" task for each process. Each task picks the next index that needs to be computed, then waits for its process to finish, then repeats until we run out of indices. Note that the feeder tasks do not begin to execute until the main task reaches the end of the @sync block, at which point it surrenders control and waits for all the local tasks to complete before returning from the function. As for v0.7 and beyond, the feeder tasks are able to share state via nextidx because they all run on the same process. Even if Tasks are scheduled cooperatively, locking may still be required in some contexts, as in asynchronous I/O. This means context switches only occur at well-defined points: in this case, when remotecall_fetch is called. This is the current state of implementation and it may change for future Julia versions, as it is intended to make it possible to run up to N Tasks on M Process, aka M:N Threading. Then a lock acquiring\\releasing model for nextidx will be needed, as it is not safe to let multiple processes read-write a resource at the same time."},{"location":"manual/distributed-computing.html#code-availability","page":"Multi-processing and Distributed Computing","title":"Code Availability and Loading Packages","category":"section","text":"Your code must be available on any process that runs it. For example, type the following into the Julia prompt:\n\njulia> function rand2(dims...)\n           return 2*rand(dims...)\n       end\n\njulia> rand2(2,2)\n2×2 Matrix{Float64}:\n 0.153756  0.368514\n 1.15119   0.918912\n\njulia> fetch(@spawnat :any rand2(2,2))\nERROR: RemoteException(2, CapturedException(UndefVarError(Symbol(\"#rand2\"))))\nStacktrace:\n[...]\n\nProcess 1 knew about the function rand2, but process 2 did not.\n\nMost commonly you'll be loading code from files or packages, and you have a considerable amount of flexibility in controlling which processes load code. Consider a file, DummyModule.jl, containing the following code:\n\nmodule DummyModule\n\nexport MyType, f\n\nmutable struct MyType\n    a::Int\nend\n\nf(x) = x^2+1\n\nprintln(\"loaded\")\n\nend\n\nIn order to refer to MyType across all processes, DummyModule.jl needs to be loaded on every process. Calling include(\"DummyModule.jl\") loads it only on a single process. To load it on every process, use the @everywhere macro (starting Julia with julia -p 2):\n\njulia> @everywhere include(\"DummyModule.jl\")\nloaded\n      From worker 3:    loaded\n      From worker 2:    loaded\n\nAs usual, this does not bring DummyModule into scope on any of the processes, which requires using or import. Moreover, when DummyModule is brought into scope on one process, it is not on any other:\n\njulia> using .DummyModule\n\njulia> MyType(7)\nMyType(7)\n\njulia> fetch(@spawnat 2 MyType(7))\nERROR: On worker 2:\nUndefVarError: `MyType` not defined in `Main`\n⋮\n\njulia> fetch(@spawnat 2 DummyModule.MyType(7))\nMyType(7)\n\nHowever, it's still possible, for instance, to send a MyType to a process which has loaded DummyModule even if it's not in scope:\n\njulia> put!(RemoteChannel(2), MyType(7))\nRemoteChannel{Channel{Any}}(2, 1, 13)\n\nA file can also be preloaded on multiple processes at startup with the -L flag, and a driver script can be used to drive the computation:\n\njulia -p <n> -L file1.jl -L file2.jl driver.jl\n\nThe Julia process running the driver script in the example above has an id equal to 1, just like a process providing an interactive prompt.\n\nFinally, if DummyModule.jl is not a standalone file but a package, then using DummyModule will load DummyModule.jl on all processes, but only bring it into scope on the process where using was called."},{"location":"manual/distributed-computing.html#Starting-and-managing-worker-processes","page":"Multi-processing and Distributed Computing","title":"Starting and managing worker processes","category":"section","text":"The base Julia installation has in-built support for two types of clusters:\n\nA local cluster specified with the -p option as shown above.\nA cluster spanning machines using the --machine-file option. This uses a passwordless ssh login to start Julia worker processes (from the same path as the current host) on the specified machines. Each machine definition takes the form [count*][user@]host[:port] [bind_addr[:port]]. user defaults to current user, port to the standard ssh port. count is the number of workers to spawn on the node, and defaults to 1. The optional bind-to bind_addr[:port] specifies the IP address and port that other workers should use to connect to this worker.\n\nnote: Note\nWhile Julia generally strives for backward compatibility, distribution of code to worker processes relies on Serialization.serialize. As pointed out in the corresponding documentation, this can not be guaranteed to work across different Julia versions, so it is advised that all workers on all machines use the same version.\n\nFunctions addprocs, rmprocs, workers, and others are available as a programmatic means of adding, removing and querying the processes in a cluster.\n\njulia> using Distributed\n\njulia> addprocs(2)\n2-element Vector{Int64}:\n 2\n 3\n\nModule Distributed must be explicitly loaded on the master process before invoking addprocs. It is automatically made available on the worker processes.\n\nnote: Note\nNote that workers do not run a ~/.julia/config/startup.jl startup script, nor do they synchronize their global state (such as command-line switches, global variables, new method definitions, and loaded modules) with any of the other running processes. You may use addprocs(exeflags=\"--project\") to initialize a worker with a particular environment, and then @everywhere using <modulename> or @everywhere include(\"file.jl\").\n\nOther types of clusters can be supported by writing your own custom ClusterManager, as described below in the ClusterManagers section."},{"location":"manual/distributed-computing.html#Data-Movement","page":"Multi-processing and Distributed Computing","title":"Data Movement","category":"section","text":"Sending messages and moving data constitute most of the overhead in a distributed program. Reducing the number of messages and the amount of data sent is critical to achieving performance and scalability. To this end, it is important to understand the data movement performed by Julia's various distributed programming constructs.\n\nfetch can be considered an explicit data movement operation, since it directly asks that an object be moved to the local machine. @spawnat (and a few related constructs) also moves data, but this is not as obvious, hence it can be called an implicit data movement operation. Consider these two approaches to constructing and squaring a random matrix:\n\nMethod 1:\n\njulia> A = rand(1000,1000);\n\njulia> Bref = @spawnat :any A^2;\n\n[...]\n\njulia> fetch(Bref);\n\nMethod 2:\n\njulia> Bref = @spawnat :any rand(1000,1000)^2;\n\n[...]\n\njulia> fetch(Bref);\n\nThe difference seems trivial, but in fact is quite significant due to the behavior of @spawnat. In the first method, a random matrix is constructed locally, then sent to another process where it is squared. In the second method, a random matrix is both constructed and squared on another process. Therefore the second method sends much less data than the first.\n\nIn this toy example, the two methods are easy to distinguish and choose from. However, in a real program designing data movement might require more thought and likely some measurement. For example, if the first process needs matrix A then the first method might be better. Or, if computing A is expensive and only the current process has it, then moving it to another process might be unavoidable. Or, if the current process has very little to do between the @spawnat and fetch(Bref), it might be better to eliminate the parallelism altogether. Or imagine rand(1000,1000) is replaced with a more expensive operation. Then it might make sense to add another @spawnat statement just for this step."},{"location":"manual/distributed-computing.html#Global-variables","page":"Multi-processing and Distributed Computing","title":"Global variables","category":"section","text":"Expressions executed remotely via @spawnat, or closures specified for remote execution using remotecall may refer to global variables. Global bindings under module Main are treated a little differently compared to global bindings in other modules. Consider the following code snippet:\n\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2)\n\nIn this case sum MUST be defined in the remote process. Note that A is a global variable defined in the local workspace. Worker 2 does not have a variable called A under Main. The act of shipping the closure ()->sum(A) to worker 2 results in Main.A being defined on 2. Main.A continues to exist on worker 2 even after the call remotecall_fetch returns. Remote calls with embedded global references (under Main module only) manage globals as follows:\n\nNew global bindings are created on destination workers if they are referenced as part of a remote call.\nGlobal constants are declared as constants on remote nodes too.\nGlobals are re-sent to a destination worker only in the context of a remote call, and then only if its value has changed. Also, the cluster does not synchronize global bindings across nodes. For example:\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 2) # worker 2\nA = rand(10,10)\nremotecall_fetch(()->sum(A), 3) # worker 3\nA = nothing\nExecuting the above snippet results in Main.A on worker 2 having a different value from Main.A on worker 3, while the value of Main.A on node 1 is set to nothing.\n\nAs you may have realized, while memory associated with globals may be collected when they are reassigned on the master, no such action is taken on the workers as the bindings continue to be valid. clear! can be used to manually reassign specific globals on remote nodes to nothing once they are no longer required. This will release any memory associated with them as part of a regular garbage collection cycle.\n\nThus programs should be careful referencing globals in remote calls. In fact, it is preferable to avoid them altogether if possible. If you must reference globals, consider using let blocks to localize global variables.\n\nFor example:\n\njulia> A = rand(10,10);\n\njulia> remotecall_fetch(()->A, 2);\n\njulia> B = rand(10,10);\n\njulia> let B = B\n           remotecall_fetch(()->B, 2)\n       end;\n\njulia> @fetchfrom 2 InteractiveUtils.varinfo()\nname           size summary\n––––––––– ––––––––– ––––––––––––––––––––––\nA         800 bytes 10×10 Array{Float64,2}\nBase                Module\nCore                Module\nMain                Module\n\nAs can be seen, global variable A is defined on worker 2, but B is captured as a local variable and hence a binding for B does not exist on worker 2."},{"location":"manual/distributed-computing.html#Parallel-Map-and-Loops","page":"Multi-processing and Distributed Computing","title":"Parallel Map and Loops","category":"section","text":"Fortunately, many useful parallel computations do not require data movement. A common example is a Monte Carlo simulation, where multiple processes can handle independent simulation trials simultaneously. We can use @spawnat to flip coins on two processes. First, write the following function in count_heads.jl:\n\nfunction count_heads(n)\n    c::Int = 0\n    for i = 1:n\n        c += rand(Bool)\n    end\n    c\nend\n\nThe function count_heads simply adds together n random bits. Here is how we can perform some trials on two machines, and add together the results:\n\njulia> @everywhere include_string(Main, $(read(\"count_heads.jl\", String)), \"count_heads.jl\")\n\njulia> a = @spawnat :any count_heads(100000000)\nFuture(2, 1, 6, nothing)\n\njulia> b = @spawnat :any count_heads(100000000)\nFuture(3, 1, 7, nothing)\n\njulia> fetch(a)+fetch(b)\n100001564\n\nThis example demonstrates a powerful and often-used parallel programming pattern. Many iterations run independently over several processes, and then their results are combined using some function. The combination process is called a reduction, since it is generally tensor-rank-reducing: a vector of numbers is reduced to a single number, or a matrix is reduced to a single row or column, etc. In code, this typically looks like the pattern x = f(x,v[i]), where x is the accumulator, f is the reduction function, and the v[i] are the elements being reduced. It is desirable for f to be associative, so that it does not matter what order the operations are performed in.\n\nNotice that our use of this pattern with count_heads can be generalized. We used two explicit @spawnat statements, which limits the parallelism to two processes. To run on any number of processes, we can use a parallel for loop, running in distributed memory, which can be written in Julia using @distributed like this:\n\nnheads = @distributed (+) for i = 1:200000000\n    Int(rand(Bool))\nend\n\nThis construct implements the pattern of assigning iterations to multiple processes, and combining them with a specified reduction (in this case (+)). The result of each iteration is taken as the value of the last expression inside the loop. The whole parallel loop expression itself evaluates to the final answer.\n\nNote that although parallel for loops look like serial for loops, their behavior is dramatically different. In particular, the iterations do not happen in a specified order, and writes to variables or arrays will not be globally visible since iterations run on different processes. Any variables used inside the parallel loop will be copied and broadcast to each process.\n\nFor example, the following code will not work as intended:\n\na = zeros(100000)\n@distributed for i = 1:100000\n    a[i] = i\nend\n\nThis code will not initialize all of a, since each process will have a separate copy of it. Parallel for loops like these must be avoided. Fortunately, Shared Arrays can be used to get around this limitation:\n\nusing SharedArrays\n\na = SharedArray{Float64}(10)\n@distributed for i = 1:10\n    a[i] = i\nend\n\nUsing \"outside\" variables in parallel loops is perfectly reasonable if the variables are read-only:\n\na = randn(1000)\n@distributed (+) for i = 1:100000\n    f(a[rand(1:end)])\nend\n\nHere each iteration applies f to a randomly-chosen sample from a vector a shared by all processes.\n\nAs you could see, the reduction operator can be omitted if it is not needed. In that case, the loop executes asynchronously, i.e. it spawns independent tasks on all available workers and returns an array of Future immediately without waiting for completion. The caller can wait for the Future completions at a later point by calling fetch on them, or wait for completion at the end of the loop by prefixing it with @sync, like @sync @distributed for.\n\nIn some cases no reduction operator is needed, and we merely wish to apply a function to all integers in some range (or, more generally, to all elements in some collection). This is another useful operation called parallel map, implemented in Julia as the pmap function. For example, we could compute the singular values of several large random matrices in parallel as follows:\n\njulia> M = Matrix{Float64}[rand(1000,1000) for i = 1:10];\n\njulia> pmap(svdvals, M);\n\nJulia's pmap is designed for the case where each function call does a large amount of work. In contrast, @distributed for can handle situations where each iteration is tiny, perhaps merely summing two numbers. Only worker processes are used by both pmap and @distributed for for the parallel computation. In case of @distributed for, the final reduction is done on the calling process."},{"location":"manual/distributed-computing.html#Remote-References-and-AbstractChannels","page":"Multi-processing and Distributed Computing","title":"Remote References and AbstractChannels","category":"section","text":"Remote references always refer to an implementation of an AbstractChannel.\n\nA concrete implementation of an AbstractChannel (like Channel), is required to implement put!, take!, fetch, isready and wait. The remote object referred to by a Future is stored in a Channel{Any}(1), i.e., a Channel of size 1 capable of holding objects of Any type.\n\nRemoteChannel, which is rewritable, can point to any type and size of channels, or any other implementation of an AbstractChannel.\n\nThe constructor RemoteChannel(f::Function, pid)() allows us to construct references to channels holding more than one value of a specific type. f is a function executed on pid and it must return an AbstractChannel.\n\nFor example, RemoteChannel(()->Channel{Int}(10), pid), will return a reference to a channel of type Int and size 10. The channel exists on worker pid.\n\nMethods put!, take!, fetch, isready and wait on a RemoteChannel are proxied onto the backing store on the remote process.\n\nRemoteChannel can thus be used to refer to user implemented AbstractChannel objects. A simple example of this is the following DictChannel which uses a dictionary as its remote store:\n\njulia> struct DictChannel{T} <: AbstractChannel{T}\n           d::Dict\n           cond_take::Threads.Condition    # waiting for data to become available\n           DictChannel{T}() where {T} = new(Dict(), Threads.Condition())\n           DictChannel() = DictChannel{Any}()\n       end\n\njulia> begin\n       function Base.put!(D::DictChannel, k, v)\n           @lock D.cond_take begin\n               D.d[k] = v\n               notify(D.cond_take)\n           end\n           return D\n       end\n       function Base.take!(D::DictChannel, k)\n           @lock D.cond_take begin\n               v = fetch(D, k)\n               delete!(D.d, k)\n               return v\n           end\n       end\n       Base.isready(D::DictChannel) = @lock D.cond_take !isempty(D.d)\n       Base.isready(D::DictChannel, k) = @lock D.cond_take haskey(D.d, k)\n       function Base.fetch(D::DictChannel, k)\n           @lock D.cond_take begin\n               wait(D, k)\n               return D.d[k]\n           end\n       end\n       function Base.wait(D::DictChannel, k)\n           @lock D.cond_take begin\n               while !isready(D, k)\n                   wait(D.cond_take)\n               end\n           end\n       end\n       end;\n\njulia> d = DictChannel();\n\njulia> isready(d)\nfalse\n\njulia> put!(d, :k, :v);\n\njulia> isready(d, :k)\ntrue\n\njulia> fetch(d, :k)\n:v\n\njulia> wait(d, :k)\n\njulia> take!(d, :k)\n:v\n\njulia> isready(d, :k)\nfalse"},{"location":"manual/distributed-computing.html#Channels-and-RemoteChannels","page":"Multi-processing and Distributed Computing","title":"Channels and RemoteChannels","category":"section","text":"A Channel is local to a process. Worker 2 cannot directly refer to a Channel on worker 3 and vice-versa. A RemoteChannel, however, can put and take values across workers.\nA RemoteChannel can be thought of as a handle to a Channel.\nThe process id, pid, associated with a RemoteChannel identifies the process where the backing store, i.e., the backing Channel exists.\nAny process with a reference to a RemoteChannel can put and take items from the channel. Data is automatically sent to (or retrieved from) the process a RemoteChannel is associated with.\nSerializing  a Channel also serializes any data present in the channel. Deserializing it therefore effectively makes a copy of the original object.\nOn the other hand, serializing a RemoteChannel only involves the serialization of an identifier that identifies the location and instance of Channel referred to by the handle. A deserialized RemoteChannel object (on any worker), therefore also points to the same backing store as the original.\n\nThe channels example from above can be modified for interprocess communication, as shown below.\n\nWe start 4 workers to process a single jobs remote channel. Jobs, identified by an id (job_id), are written to the channel. Each remotely executing task in this simulation reads a job_id, waits for a random amount of time and writes back a tuple of job_id, time taken and its own pid to the results channel. Finally all the results are printed out on the master process.\n\njulia> addprocs(4); # add worker processes\n\njulia> const jobs = RemoteChannel(()->Channel{Int}(32));\n\njulia> const results = RemoteChannel(()->Channel{Tuple}(32));\n\njulia> @everywhere function do_work(jobs, results) # define work function everywhere\n           while true\n               job_id = take!(jobs)\n               exec_time = rand()\n               sleep(exec_time) # simulates elapsed time doing actual work\n               put!(results, (job_id, exec_time, myid()))\n           end\n       end\n\njulia> function make_jobs(n)\n           for i in 1:n\n               put!(jobs, i)\n           end\n       end;\n\njulia> n = 12;\n\njulia> errormonitor(Threads.@spawn make_jobs(n)); # feed the jobs channel with \"n\" jobs\n\njulia> for p in workers() # start tasks on the workers to process requests in parallel\n           remote_do(do_work, p, jobs, results)\n       end\n\njulia> @elapsed while n > 0 # print out results\n           job_id, exec_time, worker_id = take!(results)\n           println(\"$job_id finished in $(round(exec_time; digits=2)) seconds on worker $worker_id\")\n           global n = n - 1\n       end\n1 finished in 0.18 seconds on worker 4\n2 finished in 0.26 seconds on worker 5\n6 finished in 0.12 seconds on worker 4\n7 finished in 0.18 seconds on worker 4\n5 finished in 0.35 seconds on worker 5\n4 finished in 0.68 seconds on worker 2\n3 finished in 0.73 seconds on worker 3\n11 finished in 0.01 seconds on worker 3\n12 finished in 0.02 seconds on worker 3\n9 finished in 0.26 seconds on worker 5\n8 finished in 0.57 seconds on worker 4\n10 finished in 0.58 seconds on worker 2\n0.055971741"},{"location":"manual/distributed-computing.html#Remote-References-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Remote References and Distributed Garbage Collection","category":"section","text":"Objects referred to by remote references can be freed only when all held references in the cluster are deleted.\n\nThe node where the value is stored keeps track of which of the workers have a reference to it. Every time a RemoteChannel or a (unfetched) Future is serialized to a worker, the node pointed to by the reference is notified. And every time a RemoteChannel or a (unfetched) Future is garbage collected locally, the node owning the value is again notified. This is implemented in an internal cluster aware serializer. Remote references are only valid in the context of a running cluster. Serializing and deserializing references to and from regular IO objects is not supported.\n\nThe notifications are done via sending of \"tracking\" messages–an \"add reference\" message when a reference is serialized to a different process and a \"delete reference\" message when a reference is locally garbage collected.\n\nSince Futures are write-once and cached locally, the act of fetching a Future also updates reference tracking information on the node owning the value.\n\nThe node which owns the value frees it once all references to it are cleared.\n\nWith Futures, serializing an already fetched Future to a different node also sends the value since the original remote store may have collected the value by this time.\n\nIt is important to note that when an object is locally garbage collected depends on the size of the object and the current memory pressure in the system.\n\nIn case of remote references, the size of the local reference object is quite small, while the value stored on the remote node may be quite large. Since the local object may not be collected immediately, it is a good practice to explicitly call finalize on local instances of a RemoteChannel, or on unfetched Futures. Since calling fetch on a Future also removes its reference from the remote store, this is not required on fetched Futures. Explicitly calling finalize results in an immediate message sent to the remote node to go ahead and remove its reference to the value.\n\nOnce finalized, a reference becomes invalid and cannot be used in any further calls."},{"location":"manual/distributed-computing.html#Local-invocations","page":"Multi-processing and Distributed Computing","title":"Local invocations","category":"section","text":"Data is necessarily copied over to the remote node for execution. This is the case for both remotecalls and when data is stored to a RemoteChannel / Future on a different node. As expected, this results in a copy of the serialized objects on the remote node. However, when the destination node is the local node, i.e. the calling process id is the same as the remote node id, it is executed as a local call. It is usually (not always) executed in a different task - but there is no serialization/deserialization of data. Consequently, the call refers to the same object instances as passed - no copies are created. This behavior is highlighted below:\n\njulia> using Distributed\n\njulia> rc = RemoteChannel(()->Channel(3));   # RemoteChannel created on local node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i                          # Reusing `v`\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[3], [3], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 1\n\njulia> addprocs(1);\n\njulia> rc = RemoteChannel(()->Channel(3), workers()[1]);   # RemoteChannel created on remote node\n\njulia> v = [0];\n\njulia> for i in 1:3\n           v[1] = i\n           put!(rc, v)\n       end;\n\njulia> result = [take!(rc) for _ in 1:3];\n\njulia> println(result);\n[[1], [2], [3]]\n\njulia> println(\"Num Unique objects : \", length(unique(map(objectid, result))));\nNum Unique objects : 3\n\nAs can be seen, put! on a locally owned RemoteChannel with the same object v modified between calls results in the same single object instance stored. As opposed to copies of v being created when the node owning rc is a different node.\n\nIt is to be noted that this is generally not an issue. It is something to be factored in only if the object is both being stored locally and modified post the call. In such cases it may be appropriate to store a deepcopy of the object.\n\nThis is also true for remotecalls on the local node as seen in the following example:\n\njulia> using Distributed; addprocs(1);\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), myid(), v);     # Executed on local node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[1], v2=[1], true\n\njulia> v = [0];\n\njulia> v2 = remotecall_fetch(x->(x[1] = 1; x), workers()[1], v); # Executed on remote node\n\njulia> println(\"v=$v, v2=$v2, \", v === v2);\nv=[0], v2=[1], false\n\nAs can be seen once again, a remote call onto the local node behaves just like a direct invocation. The call modifies local objects passed as arguments. In the remote invocation, it operates on a copy of the arguments.\n\nTo repeat, in general this is not an issue. If the local node is also being used as a compute node, and the arguments used post the call, this behavior needs to be factored in and if required deep copies of arguments must be passed to the call invoked on the local node. Calls on remote nodes will always operate on copies of arguments."},{"location":"manual/distributed-computing.html#man-shared-arrays","page":"Multi-processing and Distributed Computing","title":"Shared Arrays","category":"section","text":"Shared Arrays use system shared memory to map the same array across many processes. A SharedArray is a good choice when you want to have a large amount of data jointly accessible to two or more processes on the same machine. Shared Array support is available via the module SharedArrays, which must be explicitly loaded on all participating workers.\n\nA complementary data structure is provided by the external package DistributedArrays.jl in the form of a DArray. While there are some similarities to a SharedArray, the behavior of a DArray is quite different. In a SharedArray, each \"participating\" process has access to the entire array; in contrast, in a DArray, each process has local access to just a chunk of the data, and no two processes share the same chunk.\n\nSharedArray indexing (assignment and accessing values) works just as with regular arrays, and is efficient because the underlying memory is available to the local process. Therefore, most algorithms work naturally on SharedArrays, albeit in single-process mode. In cases where an algorithm insists on an Array input, the underlying array can be retrieved from a SharedArray by calling sdata. For other AbstractArray types, sdata just returns the object itself, so it's safe to use sdata on any Array-type object.\n\nThe constructor for a shared array is of the form:\n\nSharedArray{T,N}(dims::NTuple; init=false, pids=Int[])\n\nwhich creates an N-dimensional shared array of a bits type T and size dims across the processes specified by pids. Unlike distributed arrays, a shared array is accessible only from those participating workers specified by the pids named argument (and the creating process too, if it is on the same host). Note that only elements that are isbits are supported in a SharedArray.\n\nIf an init function, of signature initfn(S::SharedArray), is specified, it is called on all the participating workers. You can specify that each worker runs the init function on a distinct portion of the array, thereby parallelizing initialization.\n\nHere's a brief example:\n\njulia> using Distributed\n\njulia> addprocs(3)\n3-element Vector{Int64}:\n 2\n 3\n 4\n\njulia> @everywhere using SharedArrays\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[localindices(S)] = repeat([myid()], length(localindices(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  3  4  4\n\njulia> S[3,2] = 7\n7\n\njulia> S\n3×4 SharedMatrix{Int64}:\n 2  2  3  4\n 2  3  3  4\n 2  7  4  4\n\nSharedArrays.localindices provides disjoint one-dimensional ranges of indices, and is sometimes convenient for splitting up tasks among processes. You can, of course, divide the work any way you wish:\n\njulia> S = SharedArray{Int,2}((3,4), init = S -> S[indexpids(S):length(procs(S)):length(S)] = repeat([myid()], length( indexpids(S):length(procs(S)):length(S))))\n3×4 SharedMatrix{Int64}:\n 2  2  2  2\n 3  3  3  3\n 4  4  4  4\n\nSince all processes have access to the underlying data, you do have to be careful not to set up conflicts. For example:\n\n@sync begin\n    for p in procs(S)\n        Threads.@spawn begin\n            remotecall_wait(fill!, p, S, p)\n        end\n    end\nend\n\nwould result in undefined behavior. Because each process fills the entire array with its own pid, whichever process is the last to execute (for any particular element of S) will have its pid retained.\n\nAs a more extended and complex example, consider running the following \"kernel\" in parallel:\n\nq[i,j,t+1] = q[i,j,t] + u[i,j,t]\n\nIn this case, if we try to split up the work using a one-dimensional index, we are likely to run into trouble: if q[i,j,t] is near the end of the block assigned to one worker and q[i,j,t+1] is near the beginning of the block assigned to another, it's very likely that q[i,j,t] will not be ready at the time it's needed for computing q[i,j,t+1]. In such cases, one is better off chunking the array manually. Let's split along the second dimension. Define a function that returns the (irange, jrange) indices assigned to this worker:\n\njulia> @everywhere function myrange(q::SharedArray)\n           idx = indexpids(q)\n           if idx == 0 # This worker is not assigned a piece\n               return 1:0, 1:0\n           end\n           nchunks = length(procs(q))\n           splits = [round(Int, s) for s in range(0, stop=size(q,2), length=nchunks+1)]\n           1:size(q,1), splits[idx]+1:splits[idx+1]\n       end\n\nNext, define the kernel:\n\njulia> @everywhere function advection_chunk!(q, u, irange, jrange, trange)\n           @show (irange, jrange, trange)  # display so we can see what's happening\n           for t in trange, j in jrange, i in irange\n               q[i,j,t+1] = q[i,j,t] + u[i,j,t]\n           end\n           q\n       end\n\nWe also define a convenience wrapper for a SharedArray implementation\n\njulia> @everywhere advection_shared_chunk!(q, u) =\n           advection_chunk!(q, u, myrange(q)..., 1:size(q,3)-1)\n\nNow let's compare three different versions, one that runs in a single process:\n\njulia> advection_serial!(q, u) = advection_chunk!(q, u, 1:size(q,1), 1:size(q,2), 1:size(q,3)-1);\n\none that uses @distributed:\n\njulia> function advection_parallel!(q, u)\n           for t = 1:size(q,3)-1\n               @sync @distributed for j = 1:size(q,2)\n                   for i = 1:size(q,1)\n                       q[i,j,t+1]= q[i,j,t] + u[i,j,t]\n                   end\n               end\n           end\n           q\n       end;\n\nand one that delegates in chunks:\n\njulia> function advection_shared!(q, u)\n           @sync begin\n               for p in procs(q)\n                   Threads.@spawn remotecall_wait(advection_shared_chunk!, p, q, u)\n               end\n           end\n           q\n       end;\n\nIf we create SharedArrays and time these functions, we get the following results (with julia -p 4):\n\njulia> q = SharedArray{Float64,3}((500,500,500));\n\njulia> u = SharedArray{Float64,3}((500,500,500));\n\nRun the functions once to JIT-compile and @time them on the second run:\n\njulia> @time advection_serial!(q, u);\n(irange,jrange,trange) = (1:500,1:500,1:499)\n 830.220 milliseconds (216 allocations: 13820 bytes)\n\njulia> @time advection_parallel!(q, u);\n   2.495 seconds      (3999 k allocations: 289 MB, 2.09% gc time)\n\njulia> @time advection_shared!(q,u);\n        From worker 2:       (irange,jrange,trange) = (1:500,1:125,1:499)\n        From worker 4:       (irange,jrange,trange) = (1:500,251:375,1:499)\n        From worker 3:       (irange,jrange,trange) = (1:500,126:250,1:499)\n        From worker 5:       (irange,jrange,trange) = (1:500,376:500,1:499)\n 238.119 milliseconds (2264 allocations: 169 KB)\n\nThe biggest advantage of advection_shared! is that it minimizes traffic among the workers, allowing each to compute for an extended time on the assigned piece."},{"location":"manual/distributed-computing.html#Shared-Arrays-and-Distributed-Garbage-Collection","page":"Multi-processing and Distributed Computing","title":"Shared Arrays and Distributed Garbage Collection","category":"section","text":"Like remote references, shared arrays are also dependent on garbage collection on the creating node to release references from all participating workers. Code which creates many short lived shared array objects would benefit from explicitly finalizing these objects as soon as possible. This results in both memory and file handles mapping the shared segment being released sooner."},{"location":"manual/distributed-computing.html#ClusterManagers","page":"Multi-processing and Distributed Computing","title":"ClusterManagers","category":"section","text":"The launching, management and networking of Julia processes into a logical cluster is done via cluster managers. A ClusterManager is responsible for\n\nlaunching worker processes in a cluster environment\nmanaging events during the lifetime of each worker\noptionally, providing data transport\n\nA Julia cluster has the following characteristics:\n\nThe initial Julia process, also called the master, is special and has an id of 1.\nOnly the master process can add or remove worker processes.\nAll processes can directly communicate with each other.\n\nConnections between workers (using the in-built TCP/IP transport) is established in the following manner:\n\naddprocs is called on the master process with a ClusterManager object.\naddprocs calls the appropriate launch method which spawns required number of worker processes on appropriate machines.\nEach worker starts listening on a free port and writes out its host and port information to stdout.\nThe cluster manager captures the stdout of each worker and makes it available to the master process.\nThe master process parses this information and sets up TCP/IP connections to each worker.\nEvery worker is also notified of other workers in the cluster.\nEach worker connects to all workers whose id is less than the worker's own id.\nIn this way a mesh network is established, wherein every worker is directly connected with every other worker.\n\nWhile the default transport layer uses plain TCPSocket, it is possible for a Julia cluster to provide its own transport.\n\nJulia provides two in-built cluster managers:\n\nLocalManager, used when addprocs() or addprocs(np::Integer) are called\nSSHManager, used when addprocs(hostnames::Array) is called with a list of hostnames\n\nLocalManager is used to launch additional workers on the same host, thereby leveraging multi-core and multi-processor hardware.\n\nThus, a minimal cluster manager would need to:\n\nbe a subtype of the abstract ClusterManager\nimplement launch, a method responsible for launching new workers\nimplement manage, which is called at various events during a worker's lifetime (for example, sending an interrupt signal)\n\naddprocs(manager::FooManager) requires FooManager to implement:\n\nfunction launch(manager::FooManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nAs an example let us see how the LocalManager, the manager responsible for starting workers on the same host, is implemented:\n\nstruct LocalManager <: ClusterManager\n    np::Integer\nend\n\nfunction launch(manager::LocalManager, params::Dict, launched::Array, c::Condition)\n    [...]\nend\n\nfunction manage(manager::LocalManager, id::Integer, config::WorkerConfig, op::Symbol)\n    [...]\nend\n\nThe launch method takes the following arguments:\n\nmanager::ClusterManager: the cluster manager that addprocs is called with\nparams::Dict: all the keyword arguments passed to addprocs\nlaunched::Array: the array to append one or more WorkerConfig objects to\nc::Condition: the condition variable to be notified as and when workers are launched\n\nThe launch method is called asynchronously in a separate task. The termination of this task signals that all requested workers have been launched. Hence the launch function MUST exit as soon as all the requested workers have been launched.\n\nNewly launched workers are connected to each other and the master process in an all-to-all manner. Specifying the command line argument --worker[=<cookie>] results in the launched processes initializing themselves as workers and connections being set up via TCP/IP sockets.\n\nAll workers in a cluster share the same cookie as the master. When the cookie is unspecified, i.e, with the --worker option, the worker tries to read it from its standard input.  LocalManager and SSHManager both pass the cookie to newly launched workers via their  standard inputs.\n\nBy default a worker will listen on a free port at the address returned by a call to getipaddr(). A specific address to listen on may be specified by optional argument --bind-to bind_addr[:port]. This is useful for multi-homed hosts.\n\nAs an example of a non-TCP/IP transport, an implementation may choose to use MPI, in which case --worker must NOT be specified. Instead, newly launched workers should call init_worker(cookie) before using any of the parallel constructs.\n\nFor every worker launched, the launch method must add a WorkerConfig object (with appropriate fields initialized) to launched\n\nmutable struct WorkerConfig\n    # Common fields relevant to all cluster managers\n    io::Union{IO, Nothing}\n    host::Union{AbstractString, Nothing}\n    port::Union{Integer, Nothing}\n\n    # Used when launching additional workers at a host\n    count::Union{Int, Symbol, Nothing}\n    exename::Union{AbstractString, Cmd, Nothing}\n    exeflags::Union{Cmd, Nothing}\n\n    # External cluster managers can use this to store information at a per-worker level\n    # Can be a dict if multiple fields need to be stored.\n    userdata::Any\n\n    # SSHManager / SSH tunnel connections to workers\n    tunnel::Union{Bool, Nothing}\n    bind_addr::Union{AbstractString, Nothing}\n    sshflags::Union{Cmd, Nothing}\n    max_parallel::Union{Integer, Nothing}\n\n    # Used by Local/SSH managers\n    connect_at::Any\n\n    [...]\nend\n\nMost of the fields in WorkerConfig are used by the inbuilt managers. Custom cluster managers would typically specify only io or host / port:\n\nIf io is specified, it is used to read host/port information. A Julia worker prints out its bind address and port at startup. This allows Julia workers to listen on any free port available instead of requiring worker ports to be configured manually.\nIf io is not specified, host and port are used to connect.\ncount, exename and exeflags are relevant for launching additional workers from a worker. For example, a cluster manager may launch a single worker per node, and use that to launch additional workers.\ncount with an integer value n will launch a total of n workers.\ncount with a value of :auto will launch as many workers as the number of CPU threads (logical cores) on that machine.\nexename is the name of the julia executable including the full path.\nexeflags should be set to the required command line arguments for new workers.\ntunnel, bind_addr, sshflags and max_parallel are used when a ssh tunnel is required to connect to the workers from the master process.\nuserdata is provided for custom cluster managers to store their own worker-specific information.\n\nmanage(manager::FooManager, id::Integer, config::WorkerConfig, op::Symbol) is called at different times during the worker's lifetime with appropriate op values:\n\nwith :register/:deregister when a worker is added / removed from the Julia worker pool.\nwith :interrupt when interrupt(workers) is called. The ClusterManager should signal the appropriate worker with an interrupt signal.\nwith :finalize for cleanup purposes."},{"location":"manual/distributed-computing.html#Cluster-Managers-with-Custom-Transports","page":"Multi-processing and Distributed Computing","title":"Cluster Managers with Custom Transports","category":"section","text":"Replacing the default TCP/IP all-to-all socket connections with a custom transport layer is a little more involved. Each Julia process has as many communication tasks as the workers it is connected to. For example, consider a Julia cluster of 32 processes in an all-to-all mesh network:\n\nEach Julia process thus has 31 communication tasks.\nEach task handles all incoming messages from a single remote worker in a message-processing loop.\nThe message-processing loop waits on an IO object (for example, a TCPSocket in the default implementation), reads an entire message, processes it and waits for the next one.\nSending messages to a process is done directly from any Julia task–not just communication tasks–again, via the appropriate IO object.\n\nReplacing the default transport requires the new implementation to set up connections to remote workers and to provide appropriate IO objects that the message-processing loops can wait on. The manager-specific callbacks to be implemented are:\n\nconnect(manager::FooManager, pid::Integer, config::WorkerConfig)\nkill(manager::FooManager, pid::Int, config::WorkerConfig)\n\nThe default implementation (which uses TCP/IP sockets) is implemented as connect(manager::ClusterManager, pid::Integer, config::WorkerConfig).\n\nconnect should return a pair of IO objects, one for reading data sent from worker pid, and the other to write data that needs to be sent to worker pid. Custom cluster managers can use an in-memory BufferStream as the plumbing to proxy data between the custom, possibly non-IO transport and Julia's in-built parallel infrastructure.\n\nA BufferStream is an in-memory IOBuffer which behaves like an IO–it is a stream which can be handled asynchronously.\n\nThe folder clustermanager/0mq in the Examples repository contains an example of using ZeroMQ to connect Julia workers in a star topology with a 0MQ broker in the middle. Note: The Julia processes are still all logically connected to each other–any worker can message any other worker directly without any awareness of 0MQ being used as the transport layer.\n\nWhen using custom transports:\n\nJulia workers must NOT be started with --worker. Starting with --worker will result in the newly launched workers defaulting to the TCP/IP socket transport implementation.\nFor every incoming logical connection with a worker, Base.process_messages(rd::IO, wr::IO)() must be called. This launches a new task that handles reading and writing of messages from/to the worker represented by the IO objects.\ninit_worker(cookie, manager::FooManager) must be called as part of worker process initialization.\nField connect_at::Any in WorkerConfig can be set by the cluster manager when launch is called. The value of this field is passed in all connect callbacks. Typically, it carries information on how to connect to a worker. For example, the TCP/IP socket transport uses this field to specify the (host, port) tuple at which to connect to a worker.\n\nkill(manager, pid, config) is called to remove a worker from the cluster. On the master process, the corresponding IO objects must be closed by the implementation to ensure proper cleanup. The default implementation simply executes an exit() call on the specified remote worker.\n\nThe Examples folder clustermanager/simple is an example that shows a simple implementation using UNIX domain sockets for cluster setup."},{"location":"manual/distributed-computing.html#Network-Requirements-for-LocalManager-and-SSHManager","page":"Multi-processing and Distributed Computing","title":"Network Requirements for LocalManager and SSHManager","category":"section","text":"Julia clusters are designed to be executed on already secured environments on infrastructure such as local laptops, departmental clusters, or even the cloud. This section covers network security requirements for the inbuilt LocalManager and SSHManager:\n\nThe master process does not listen on any port. It only connects out to the workers.\nEach worker binds to only one of the local interfaces and listens on an ephemeral port number assigned by the OS.\nLocalManager, used by addprocs(N), by default binds only to the loopback interface. This means that workers started later on remote hosts (or by anyone with malicious intentions) are unable to connect to the cluster. An addprocs(4) followed by an addprocs([\"remote_host\"]) will fail. Some users may need to create a cluster comprising their local system and a few remote systems. This can be done by explicitly requesting LocalManager to bind to an external network interface via the restrict keyword argument: addprocs(4; restrict=false).\nSSHManager, used by addprocs(list_of_remote_hosts), launches workers on remote hosts via SSH. By default SSH is only used to launch Julia workers. Subsequent master-worker and worker-worker connections use plain, unencrypted TCP/IP sockets. The remote hosts must have passwordless login enabled. Additional SSH flags or credentials may be specified via keyword argument sshflags.\naddprocs(list_of_remote_hosts; tunnel=true, sshflags=<ssh keys and other flags>) is useful when we wish to use SSH connections for master-worker too. A typical scenario for this is a local laptop running the Julia REPL (i.e., the master) with the rest of the cluster on the cloud, say on Amazon EC2. In this case only port 22 needs to be opened at the remote cluster coupled with SSH client authenticated via public key infrastructure (PKI). Authentication credentials can be supplied via sshflags, for example sshflags=`-i <keyfile>`.\nIn an all-to-all topology (the default), all workers connect to each other via plain TCP sockets. The security policy on the cluster nodes must thus ensure free connectivity between workers for the ephemeral port range (varies by OS).\nSecuring and encrypting all worker-worker traffic (via SSH) or encrypting individual messages can be done via a custom ClusterManager.\nIf you specify multiplex=true as an option to addprocs, SSH multiplexing is used to create a tunnel between the master and workers. If you have configured SSH multiplexing on your own and the connection has already been established, SSH multiplexing is used regardless of multiplex option. If multiplexing is enabled, forwarding is set by using the existing connection (-O forward option in ssh). This is beneficial if your servers require password authentication; you can avoid authentication in Julia by logging in to the server ahead of addprocs. The control socket will be located at ~/.ssh/julia-%r@%h:%p during the session unless the existing multiplexing connection is used. Note that bandwidth may be limited if you create multiple processes on a node and enable multiplexing, because in that case processes share a single multiplexing TCP connection."},{"location":"manual/distributed-computing.html#man-cluster-cookie","page":"Multi-processing and Distributed Computing","title":"Cluster Cookie","category":"section","text":"All processes in a cluster share the same cookie which, by default, is a randomly generated string on the master process:\n\ncluster_cookie() returns the cookie, while cluster_cookie(cookie)() sets it and returns the new cookie.\nAll connections are authenticated on both sides to ensure that only workers started by the master are allowed to connect to each other.\nThe cookie may be passed to the workers at startup via argument --worker=<cookie>. If argument --worker is specified without the cookie, the worker tries to read the cookie from its standard input (stdin). The stdin is closed immediately after the cookie is retrieved.\nClusterManagers can retrieve the cookie on the master by calling cluster_cookie(). Cluster managers not using the default TCP/IP transport (and hence not specifying --worker) must call init_worker(cookie, manager) with the same cookie as on the master.\n\nNote that environments requiring higher levels of security can implement this via a custom ClusterManager. For example, cookies can be pre-shared and hence not specified as a startup argument."},{"location":"manual/distributed-computing.html#Specifying-Network-Topology-(Experimental)","page":"Multi-processing and Distributed Computing","title":"Specifying Network Topology (Experimental)","category":"section","text":"The keyword argument topology passed to addprocs is used to specify how the workers must be connected to each other:\n\n:all_to_all, the default: all workers are connected to each other.\n:master_worker: only the driver process, i.e. pid 1, has connections to the workers.\n:custom: the launch method of the cluster manager specifies the connection topology via the fields ident and connect_idents in WorkerConfig. A worker with a cluster-manager-provided identity ident will connect to all workers specified in connect_idents.\n\nKeyword argument lazy=true|false only affects topology option :all_to_all. If true, the cluster starts off with the master connected to all workers. Specific worker-worker connections are established at the first remote invocation between two workers. This helps in reducing initial resources allocated for intra-cluster communication. Connections are setup depending on the runtime requirements of a parallel program. Default value for lazy is true.\n\nCurrently, sending a message between unconnected workers results in an error. This behaviour, as with the functionality and interface, should be considered experimental in nature and may change in future releases."},{"location":"manual/distributed-computing.html#Noteworthy-external-packages","page":"Multi-processing and Distributed Computing","title":"Noteworthy external packages","category":"section","text":"Outside of Julia parallelism there are plenty of external packages that should be mentioned. For example, MPI.jl is a Julia wrapper for the MPI protocol, Dagger.jl provides functionality similar to Python's Dask, and DistributedArrays.jl provides array operations distributed across workers, as outlined above.\n\nA mention must be made of Julia's GPU programming ecosystem, which includes:\n\nCUDA.jl wraps the various CUDA libraries and supports compiling Julia kernels for Nvidia GPUs.\noneAPI.jl wraps the oneAPI unified programming model, and supports executing Julia kernels on supported accelerators. Currently only Linux is supported.\nAMDGPU.jl wraps the AMD ROCm libraries and supports compiling Julia kernels for AMD GPUs. Currently only Linux is supported.\nHigh-level libraries like KernelAbstractions.jl, Tullio.jl and ArrayFire.jl.\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes by first casting it through distribute() and CuArray().\n\nRemember when importing DistributedArrays.jl to import it across all processes using @everywhere\n\n$ ./julia -p 4\n\njulia> addprocs()\n\njulia> @everywhere using DistributedArrays\n\njulia> using CUDA\n\njulia> B = ones(10_000) ./ 2;\n\njulia> A = ones(10_000) .* π;\n\njulia> C = 2 .* A ./ B;\n\njulia> all(C .≈ 4*π)\ntrue\n\njulia> typeof(C)\nVector{Float64} (alias for Array{Float64, 1})\n\njulia> dB = distribute(B);\n\njulia> dA = distribute(A);\n\njulia> dC = 2 .* dA ./ dB;\n\njulia> all(dC .≈ 4*π)\ntrue\n\njulia> typeof(dC)\nDistributedArrays.DArray{Float64,1,Vector{Float64}}\n\njulia> cuB = CuArray(B);\n\njulia> cuA = CuArray(A);\n\njulia> cuC = 2 .* cuA ./ cuB;\n\njulia> all(cuC .≈ 4*π);\ntrue\n\njulia> typeof(cuC)\nCuArray{Float64,1}\n\nIn the following example we will use both DistributedArrays.jl and CUDA.jl to distribute an array across multiple processes and call a generic function on it.\n\nfunction power_method(M, v)\n    for i in 1:100\n        v = M*v\n        v /= norm(v)\n    end\n\n    return v, norm(M*v) / norm(v)  # or  (M*v) ./ v\nend\n\npower_method repeatedly creates a new vector and normalizes it. We have not specified any type signature in function declaration, let's see if it works with the aforementioned datatypes:\n\njulia> M = [2. 1; 1 1];\n\njulia> v = rand(2)\n2-element Vector{Float64}:\n0.40395\n0.445877\n\njulia> power_method(M,v)\n([0.850651, 0.525731], 2.618033988749895)\n\njulia> cuM = CuArray(M);\n\njulia> cuv = CuArray(v);\n\njulia> curesult = power_method(cuM, cuv);\n\njulia> typeof(curesult)\nCuArray{Float64,1}\n\njulia> dM = distribute(M);\n\njulia> dv = distribute(v);\n\njulia> dC = power_method(dM, dv);\n\njulia> typeof(dC)\nTuple{DistributedArrays.DArray{Float64,1,Vector{Float64}},Float64}\n\nTo end this short exposure to external packages, we can consider MPI.jl, a Julia wrapper of the MPI protocol. As it would take too long to consider every inner function, it would be better to simply appreciate the approach used to implement the protocol.\n\nConsider this toy script which simply calls each subprocess, instantiate its rank and when the master process is reached, performs the ranks' sum\n\nimport MPI\n\nMPI.Init()\n\ncomm = MPI.COMM_WORLD\nMPI.Barrier(comm)\n\nroot = 0\nr = MPI.Comm_rank(comm)\n\nsr = MPI.Reduce(r, MPI.SUM, root, comm)\n\nif(MPI.Comm_rank(comm) == root)\n   @printf(\"sum of ranks: %s\\n\", sr)\nend\n\nMPI.Finalize()\n\nmpirun -np 4 ./julia example.jl\n\n[1]: In this context, MPI refers to the MPI-1 standard. Beginning with MPI-2, the MPI standards committee introduced a new set of communication mechanisms, collectively referred to as Remote Memory Access (RMA). The motivation for adding rma to the MPI standard was to facilitate one-sided communication patterns. For additional information on the latest MPI standard, see https://mpi-forum.org/docs."},{"location":"manual/handling-operating-system-variation.html#Handling-Operating-System-Variation","page":"Handling Operating System Variation","title":"Handling Operating System Variation","category":"section","text":"When writing cross-platform applications or libraries, it is often necessary to allow for differences between operating systems. The variable Sys.KERNEL can be used to handle such cases. There are several functions in the Sys module intended to make this easier, such as isunix, islinux, isapple, isbsd, isfreebsd, and iswindows. These may be used as follows:\n\nif Sys.iswindows()\n    windows_specific_thing(a)\nend\n\nNote that islinux, isapple, and isfreebsd are mutually exclusive subsets of isunix. Additionally, there is a macro @static which makes it possible to use these functions to conditionally hide invalid code, as demonstrated in the following examples.\n\nSimple blocks:\n\nccall((@static Sys.iswindows() ? :_fopen : :fopen), ...)\n\nComplex blocks:\n\n@static if Sys.islinux()\n    linux_specific_thing(a)\nelseif Sys.isapple()\n    apple_specific_thing(a)\nelse\n    generic_thing(a)\nend\n\nWhen nesting conditionals, the @static must be repeated for each level (parentheses optional, but recommended for readability):\n\n@static Sys.iswindows() ? :a : (@static Sys.isapple() ? :b : :c)"},{"location":"manual/stacktraces.html#Stack-Traces","page":"Stack Traces","title":"Stack Traces","category":"section","text":"The StackTraces module provides simple stack traces that are both human readable and easy to use programmatically."},{"location":"manual/stacktraces.html#Viewing-a-stack-trace","page":"Stack Traces","title":"Viewing a stack trace","category":"section","text":"The primary function used to obtain a stack trace is stacktrace:\n\n6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n\nCalling stacktrace() returns a vector of StackTraces.StackFrame s. For ease of use, the alias StackTraces.StackTrace can be used in place of Vector{StackFrame}. (Examples with [...] indicate that output may vary depending on how the code is run.)\n\njulia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]\n\njulia> @noinline child() = stacktrace()\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> grandparent() = parent()\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\n9-element Array{Base.StackTraces.StackFrame,1}:\n child() at REPL[3]:1\n parent() at REPL[4]:1\n grandparent() at REPL[5]:1\n[...]\n\nNote that when calling stacktrace() you'll typically see a frame with eval at boot.jl. When calling stacktrace() from the REPL you'll also have a few extra frames in the stack from REPL.jl, usually looking something like this:\n\njulia> example() = stacktrace()\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[1]:1\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92"},{"location":"manual/stacktraces.html#Extracting-useful-information","page":"Stack Traces","title":"Extracting useful information","category":"section","text":"Each StackTraces.StackFrame contains the function name, file name, line number, lambda info, a flag indicating whether the frame has been inlined, a flag indicating whether it is a C function (by default C functions do not appear in the stack trace), and an integer representation of the pointer returned by backtrace:\n\njulia> frame = stacktrace()[3]\neval(::Module, ::Expr) at REPL.jl:5\n\njulia> frame.func\n:eval\n\njulia> frame.file\nSymbol(\"~/julia/usr/share/julia/stdlib/v0.7/REPL/src/REPL.jl\")\n\njulia> frame.line\n5\n\njulia> frame.linfo\nMethodInstance for eval(::Module, ::Expr)\n\njulia> frame.inlined\nfalse\n\njulia> frame.from_c\nfalse\n\njulia> frame.pointer\n0x00007f92d6293171\n\nThis makes stack trace information available programmatically for logging, error handling, and more."},{"location":"manual/stacktraces.html#Error-handling","page":"Stack Traces","title":"Error handling","category":"section","text":"While having easy access to information about the current state of the callstack can be helpful in many places, the most obvious application is in error handling and debugging.\n\njulia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace()\n       end\nexample (generic function with 1 method)\n\njulia> example()\n7-element Array{Base.StackTraces.StackFrame,1}:\n example() at REPL[2]:4\n top-level scope\n eval at boot.jl:317 [inlined]\n[...]\n\nYou may notice that in the example above the first stack frame points at line 4, where stacktrace is called, rather than line 2, where bad_function is called, and bad_function's frame is missing entirely. This is understandable, given that stacktrace is called from the context of the catch. While in this example it's fairly easy to find the actual source of the error, in complex cases tracking down the source of the error becomes nontrivial.\n\nThis can be remedied by passing the result of catch_backtrace to stacktrace. Instead of returning callstack information for the current context, catch_backtrace returns stack information for the context of the most recent exception:\n\njulia> @noinline bad_function() = undeclared_variable\nbad_function (generic function with 1 method)\n\njulia> @noinline example() = try\n           bad_function()\n       catch\n           stacktrace(catch_backtrace())\n       end\nexample (generic function with 1 method)\n\njulia> example()\n8-element Array{Base.StackTraces.StackFrame,1}:\n bad_function() at REPL[1]:1\n example() at REPL[2]:2\n[...]\n\nNotice that the stack trace now indicates the appropriate line number and the missing frame.\n\njulia> @noinline child() = error(\"Whoops!\")\nchild (generic function with 1 method)\n\njulia> @noinline parent() = child()\nparent (generic function with 1 method)\n\njulia> @noinline function grandparent()\n           try\n               parent()\n           catch err\n               println(\"ERROR: \", err.msg)\n               stacktrace(catch_backtrace())\n           end\n       end\ngrandparent (generic function with 1 method)\n\njulia> grandparent()\nERROR: Whoops!\n10-element Array{Base.StackTraces.StackFrame,1}:\n error at error.jl:33 [inlined]\n child() at REPL[1]:1\n parent() at REPL[2]:1\n grandparent() at REPL[3]:3\n[...]"},{"location":"manual/stacktraces.html#Exception-stacks-and-[current_exceptions](@ref)","page":"Stack Traces","title":"Exception stacks and current_exceptions","category":"section","text":"compat: Julia 1.1\nException stacks requires at least Julia 1.1.\n\nWhile handling an exception further exceptions may be thrown. It can be useful to inspect all these exceptions to identify the root cause of a problem. The julia runtime supports this by pushing each exception onto an internal exception stack as it occurs. When the code exits a catch normally, any exceptions which were pushed onto the stack in the associated try are considered to be successfully handled and are removed from the stack.\n\nThe stack of current exceptions can be accessed using the current_exceptions function. For example,\n\njulia> try\n           error(\"(A) The root cause\")\n       catch\n           try\n               error(\"(B) An exception while handling the exception\")\n           catch\n               for (exc, bt) in current_exceptions()\n                   showerror(stdout, exc, bt)\n                   println(stdout)\n               end\n           end\n       end\n(A) The root cause\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:2\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259\n(B) An exception while handling the exception\nStacktrace:\n [1] error(::String) at error.jl:33\n [2] top-level scope at REPL[7]:5\n [3] eval(::Module, ::Any) at boot.jl:319\n [4] eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n [5] macro expansion at REPL.jl:117 [inlined]\n [6] (::getfield(REPL, Symbol(\"##26#27\")){REPL.REPLBackend})() at task.jl:259\n\nIn this example the root cause exception (A) is first on the stack, with a further exception (B) following it. After exiting both catch blocks normally (i.e., without throwing a further exception) all exceptions are removed from the stack and are no longer accessible.\n\nThe exception stack is stored on the Task where the exceptions occurred. When a task fails with uncaught exceptions, current_exceptions(task) may be used to inspect the exception stack for that task."},{"location":"manual/stacktraces.html#Comparison-with-[backtrace](@ref)","page":"Stack Traces","title":"Comparison with backtrace","category":"section","text":"A call to backtrace returns a vector of Union{Ptr{Nothing}, Base.InterpreterIP}, which may then be passed into stacktrace for translation:\n\njulia> trace = backtrace()\n18-element Array{Union{Ptr{Nothing}, Base.InterpreterIP},1}:\n Ptr{Nothing} @0x00007fd8734c6209\n Ptr{Nothing} @0x00007fd87362b342\n Ptr{Nothing} @0x00007fd87362c136\n Ptr{Nothing} @0x00007fd87362c986\n Ptr{Nothing} @0x00007fd87362d089\n Base.InterpreterIP(CodeInfo(:(begin\n      Core.SSAValue(0) = backtrace()\n      trace = Core.SSAValue(0)\n      return Core.SSAValue(0)\n  end)), 0x0000000000000000)\n Ptr{Nothing} @0x00007fd87362e4cf\n[...]\n\njulia> stacktrace(trace)\n6-element Array{Base.StackTraces.StackFrame,1}:\n top-level scope\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n\nNotice that the vector returned by backtrace had 18 elements, while the vector returned by stacktrace only has 6. This is because, by default, stacktrace removes any lower-level C functions from the stack. If you want to include stack frames from C calls, you can do it like this:\n\njulia> stacktrace(trace, true)\n21-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n do_call at interpreter.c:324\n eval_value at interpreter.c:416\n eval_body at interpreter.c:559\n jl_interpret_toplevel_thunk_callback at interpreter.c:798\n top-level scope\n jl_interpret_toplevel_thunk at interpreter.c:807\n jl_toplevel_eval_flex at toplevel.c:856\n jl_toplevel_eval_in at builtins.c:624\n eval at boot.jl:317 [inlined]\n eval(::Module, ::Expr) at REPL.jl:5\n jl_apply_generic at gf.c:2167\n eval_user_input(::Any, ::REPL.REPLBackend) at REPL.jl:85\n jl_apply_generic at gf.c:2167\n macro expansion at REPL.jl:116 [inlined]\n (::getfield(REPL, Symbol(\"##28#29\")){REPL.REPLBackend})() at event.jl:92\n jl_fptr_trampoline at gf.c:1838\n jl_apply_generic at gf.c:2167\n jl_apply at julia.h:1540 [inlined]\n start_task at task.c:268\n ip:0xffffffffffffffff\n\nIndividual pointers returned by backtrace can be translated into StackTraces.StackFrame s by passing them into StackTraces.lookup:\n\njulia> pointer = backtrace()[1];\n\njulia> frame = StackTraces.lookup(pointer)\n1-element Array{Base.StackTraces.StackFrame,1}:\n jl_apply_generic at gf.c:2167\n\njulia> println(\"The top frame is from $(frame[1].func)!\")\nThe top frame is from jl_apply_generic!"},{"location":"stdlib/LazyArtifacts.html#Lazy-Artifacts","page":"Lazy Artifacts","title":"Lazy Artifacts","category":"section","text":"In order for a package to download artifacts lazily, LazyArtifacts must be explicitly listed as a dependency of that package.\n\nFor further information on artifacts, see Artifacts."},{"location":"devdocs/functions.html#Julia-Functions","page":"Julia Functions","title":"Julia Functions","category":"section","text":"This document will explain how functions, method definitions, and method tables work."},{"location":"devdocs/functions.html#Method-Tables","page":"Julia Functions","title":"Method Tables","category":"section","text":"Every function in Julia is a generic function. A generic function is conceptually a single function, but consists of many definitions, or methods. The methods of a generic function are stored in a method table. There is one global method table (type MethodTable) named Core.methodtable. Any default operation on methods (such as calls) uses that table."},{"location":"devdocs/functions.html#Function-calls","page":"Julia Functions","title":"Function calls","category":"section","text":"Given the call f(x, y), the following steps are performed: First, a tuple type is formed, Tuple{typeof(f), typeof(x), typeof(y)}. Note that the type of the function itself is the first element. This is because the function itself participates symmetrically in method lookup with the other arguments. This tuple type is looked up in the global method table. However, the system can then cache the results, so these steps can be skipped later for similar lookups.\n\nThis dispatch process is performed by jl_apply_generic, which takes two arguments: a pointer to an array of the values f, x, and y, and the number of values (in this case 3).\n\nThroughout the system, there are two kinds of APIs that handle functions and argument lists: those that accept the function and arguments separately, and those that accept a single argument structure. In the first kind of API, the \"arguments\" part does not contain information about the function, since that is passed separately. In the second kind of API, the function is the first element of the argument structure.\n\nFor example, the following function for performing a call accepts just an args pointer, so the first element of the args array will be the function to call:\n\njl_value_t *jl_apply(jl_value_t **args, uint32_t nargs)\n\nThis entry point for the same functionality accepts the function separately, so the args array does not contain the function:\n\njl_value_t *jl_call(jl_value_t *f, jl_value_t **args, int32_t nargs);"},{"location":"devdocs/functions.html#Adding-methods","page":"Julia Functions","title":"Adding methods","category":"section","text":"Given the above dispatch process, conceptually all that is needed to add a new method is (1) a tuple type, and (2) code for the body of the method. jl_method_def implements this operation."},{"location":"devdocs/functions.html#Creating-generic-functions","page":"Julia Functions","title":"Creating generic functions","category":"section","text":"Since every object is callable, nothing special is needed to create a generic function. Therefore jl_new_generic_function simply creates a new singleton (0 size) subtype of Function and returns its instance. A function can have a mnemonic \"display name\" which is used in debug info and when printing objects. For example the name of Base.sin is sin. By convention, the name of the created type is the same as the function name, with a # prepended. So typeof(sin) is Base.#sin."},{"location":"devdocs/functions.html#Closures","page":"Julia Functions","title":"Closures","category":"section","text":"A closure is simply a callable object with field names corresponding to captured variables. For example, the following code:\n\nfunction adder(x)\n    return y->x+y\nend\n\nis lowered to (roughly):\n\nstruct ##1{T}\n    x::T\nend\n\n(_::##1)(y) = _.x + y\n\nfunction adder(x)\n    return ##1(x)\nend"},{"location":"devdocs/functions.html#Constructors","page":"Julia Functions","title":"Constructors","category":"section","text":"A constructor call is just a call to a type, to a method defined on Type{T}."},{"location":"devdocs/functions.html#Builtins","page":"Julia Functions","title":"Builtins","category":"section","text":"The \"builtin\" functions, defined in the Core module, are:\n\nfunction lines(words)\n    io = IOBuffer()\n    n = 0\n    for w in words\n        if n+length(w) > 80\n            print(io, '\\n', w)\n            n = length(w)\n        elseif n == 0\n            print(io, w);\n            n += length(w)\n        else\n            print(io, ' ', w);\n            n += length(w)+1\n        end\n    end\n    takestring!(io)\nend\nimport Markdown\n[string(n) for n in names(Core;all=true)\n    if getfield(Core,n) isa Core.Builtin && nameof(getfield(Core,n)) === n] |>\n    lines |>\n    s ->  \"```\\n$s\\n```\" |>\n    Markdown.parse\n\nThese are mostly singleton objects all of whose types are subtypes of Builtin, which is a subtype of Function. Their purpose is to expose entry points in the run time that use the \"jlcall\" calling convention:\n\njl_value_t *(jl_value_t*, jl_value_t**, uint32_t)"},{"location":"devdocs/functions.html#Keyword-arguments","page":"Julia Functions","title":"Keyword arguments","category":"section","text":"Keyword arguments work by adding methods to the kwcall function. This function is usually the \"keyword argument sorter\" or \"keyword sorter\", which then calls the inner body of the function (defined anonymously). Every definition in the kwsorter function has the same arguments as some definition in the normal method table, except with a single NamedTuple argument prepended, which gives the names and values of passed keyword arguments. The kwsorter's job is to move keyword arguments into their canonical positions based on name, plus evaluate and substitute any needed default value expressions. The result is a normal positional argument list, which is then passed to yet another compiler-generated function.\n\nThe easiest way to understand the process is to look at how a keyword argument method definition is lowered. The code:\n\nfunction circle(center, radius; color = black, fill::Bool = true, options...)\n    # draw\nend\n\nactually produces three method definitions. The first is a function that accepts all arguments (including keyword arguments) as positional arguments, and includes the code for the method body. It has an auto-generated name:\n\nfunction #circle#1(color, fill::Bool, options, circle, center, radius)\n    # draw\nend\n\nThe second method is an ordinary definition for the original circle function, which handles the case where no keyword arguments are passed:\n\nfunction circle(center, radius)\n    #circle#1(black, true, pairs(NamedTuple()), circle, center, radius)\nend\n\nThis simply dispatches to the first method, passing along default values. pairs is applied to the named tuple of rest arguments to provide key-value pair iteration. Note that if the method doesn't accept rest keyword arguments then this argument is absent.\n\nFinally there is the kwsorter definition:\n\nfunction (::Core.kwcall)(kws, circle, center, radius)\n    if haskey(kws, :color)\n        color = kws.color\n    else\n        color = black\n    end\n    # etc.\n\n    # put remaining kwargs in `options`\n    options = structdiff(kws, NamedTuple{(:color, :fill)})\n\n    # if the method doesn't accept rest keywords, throw an error\n    # unless `options` is empty\n\n    #circle#1(color, fill, pairs(options), circle, center, radius)\nend"},{"location":"devdocs/functions.html#compiler-efficiency-issues","page":"Julia Functions","title":"Compiler efficiency issues","category":"section","text":"Generating a new type for every function has potentially serious consequences for compiler resource use when combined with Julia's \"specialize on all arguments by default\" design. Indeed, the initial implementation of this design suffered from much longer build and test times, higher memory use, and a system image nearly 2x larger than the baseline. In a naive implementation, the problem is bad enough to make the system nearly unusable. Several significant optimizations were needed to make the design practical.\n\nThe first issue is excessive specialization of functions for different values of function-valued arguments. Many functions simply \"pass through\" an argument to somewhere else, e.g. to another function or to a storage location. Such functions do not need to be specialized for every closure that might be passed in. Fortunately this case is easy to distinguish by simply considering whether a function calls one of its arguments (i.e. the argument appears in \"head position\" somewhere). Performance-critical higher-order functions like map certainly call their argument function and so will still be specialized as expected. This optimization is implemented by recording which arguments are called during the analyze-variables pass in the front end. When cache_method sees an argument in the Function type hierarchy passed to a slot declared as Any or Function, it behaves as if the @nospecialize annotation were applied. This heuristic seems to be extremely effective in practice.\n\nThe next issue concerns the structure of method tables. Empirical studies show that the vast majority of dynamically-dispatched calls involve one or two arguments. In turn, many of these cases can be resolved by considering only the first argument. (Aside: proponents of single dispatch would not be surprised by this at all. However, this argument means \"multiple dispatch is easy to optimize in practice\", and that we should therefore use it, not \"we should use single dispatch\"!). So the method table and cache splits up on the structure based on a left-to-right decision tree so allow efficient nearest-neighbor searches.\n\nThe front end generates type declarations for all closures. Initially, this was implemented by generating normal type declarations. However, this produced an extremely large number of constructors, all of which were trivial (simply passing all arguments through to new). Since methods are partially ordered, inserting all of these methods is O(n²), plus there are just too many of them to keep around. This was optimized by generating struct_type expressions directly (bypassing default constructor generation), and using new directly to create closure instances. Not the prettiest thing ever, but you do what you gotta do.\n\nThe next problem was the @test macro, which generated a 0-argument closure for each test case. This is not really necessary, since each test case is simply run once in place. Therefore, @test was modified to expand to a try-catch block that records the test result (true, false, or exception raised) and calls the test suite handler on it."},{"location":"manual/integers-and-floating-point-numbers.html#Integers-and-Floating-Point-Numbers","page":"Integers and Floating-Point Numbers","title":"Integers and Floating-Point Numbers","category":"section","text":"Integers and floating-point values are the basic building blocks of arithmetic and computation. Built-in representations of such values are called numeric primitives, while representations of integers and floating-point numbers as immediate values in code are known as numeric literals. For example, 1 is an integer literal, while 1.0 is a floating-point literal; their binary in-memory representations as objects are numeric primitives.\n\nJulia provides a broad range of primitive numeric types, and a full complement of arithmetic and bitwise operators as well as standard mathematical functions are defined over them. These map directly onto numeric types and operations that are natively supported on modern computers, thus allowing Julia to take full advantage of computational resources. Additionally, Julia provides software support for Arbitrary Precision Arithmetic, which can handle operations on numeric values that cannot be represented effectively in native hardware representations, but at the cost of relatively slower performance.\n\nThe following are Julia's primitive numeric types:\n\nInteger types:\n\nType Signed? Number of bits Smallest value Largest value\nInt8 ✓ 8 -2^7 2^7 - 1\nUInt8  8 0 2^8 - 1\nInt16 ✓ 16 -2^15 2^15 - 1\nUInt16  16 0 2^16 - 1\nInt32 ✓ 32 -2^31 2^31 - 1\nUInt32  32 0 2^32 - 1\nInt64 ✓ 64 -2^63 2^63 - 1\nUInt64  64 0 2^64 - 1\nInt128 ✓ 128 -2^127 2^127 - 1\nUInt128  128 0 2^128 - 1\nBool N/A 8 false (0) true (1)\n\nFloating-point types:\n\nType Precision Number of bits\nFloat16 half 16\nFloat32 single 32\nFloat64 double 64\n\nAdditionally, full support for Complex and Rational Numbers is built on top of these primitive numeric types. All numeric types interoperate naturally without explicit casting, thanks to a flexible, user-extensible type promotion system."},{"location":"manual/integers-and-floating-point-numbers.html#Integers","page":"Integers and Floating-Point Numbers","title":"Integers","category":"section","text":"Literal integers are represented in the standard manner:\n\njulia> 1\n1\n\njulia> 1234\n1234\n\nThe default type for an integer literal depends on whether the target system has a 32-bit architecture or a 64-bit architecture:\n\n# 32-bit system:\njulia> typeof(1)\nInt32\n\n# 64-bit system:\njulia> typeof(1)\nInt64\n\nThe Julia internal variable Sys.WORD_SIZE indicates whether the target system is 32-bit or 64-bit:\n\n# 32-bit system:\njulia> Sys.WORD_SIZE\n32\n\n# 64-bit system:\njulia> Sys.WORD_SIZE\n64\n\nJulia also defines the types Int and UInt, which are aliases for the system's signed and unsigned native integer types respectively:\n\n# 32-bit system:\njulia> Int\nInt32\njulia> UInt\nUInt32\n\n# 64-bit system:\njulia> Int\nInt64\njulia> UInt\nUInt64\n\nLarger integer literals that cannot be represented using only 32 bits but can be represented in 64 bits always create 64-bit integers, regardless of the system type:\n\n# 32-bit or 64-bit system:\njulia> typeof(3000000000)\nInt64\n\nUnsigned integers are input and output using the 0x prefix and hexadecimal (base 16) digits 0-9a-f (the capitalized digits A-F also work for input). The size of the unsigned value is determined by the number of hex digits used:\n\njulia> x = 0x1\n0x01\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x123\n0x0123\n\njulia> typeof(x)\nUInt16\n\njulia> x = 0x1234567\n0x01234567\n\njulia> typeof(x)\nUInt32\n\njulia> x = 0x123456789abcdef\n0x0123456789abcdef\n\njulia> typeof(x)\nUInt64\n\njulia> x = 0x11112222333344445555666677778888\n0x11112222333344445555666677778888\n\njulia> typeof(x)\nUInt128\n\nThis behavior is based on the observation that when one uses unsigned hex literals for integer values, one typically is using them to represent a fixed numeric byte sequence, rather than just an integer value.\n\nBinary and octal literals are also supported:\n\njulia> x = 0b10\n0x02\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0o010\n0x08\n\njulia> typeof(x)\nUInt8\n\njulia> x = 0x00000000000000001111222233334444\n0x00000000000000001111222233334444\n\njulia> typeof(x)\nUInt128\n\nAs for hexadecimal literals, binary and octal literals produce unsigned integer types. The size of the binary data item is the minimal needed size, if the leading digit of the literal is not 0. In the case of leading zeros, the size is determined by the minimal needed size for a literal, which has the same length but leading digit 1. It means that:\n\n0x1 and 0x12 are UInt8 literals,\n0x123 and 0x1234 are UInt16 literals,\n0x12345 and 0x12345678 are UInt32 literals,\n0x123456789 and 0x1234567890adcdef are UInt64 literals, etc.\n\nEven if there are leading zero digits which don’t contribute to the value, they count for determining storage size of a literal. So 0x01 is a UInt8 while 0x0001 is a UInt16.\n\nThat allows the user to control the size.\n\nUnsigned literals (starting with 0x) that encode integers too large to be represented as UInt128 values will construct BigInt values instead. This is not an unsigned type but it is the only built-in type big enough to represent such large integer values.\n\nBinary, octal, and hexadecimal literals may be signed by a - immediately preceding the unsigned literal. They produce an unsigned integer of the same size as the unsigned literal would do, with the two's complement of the value:\n\njulia> -0x2\n0xfe\n\njulia> -0x0002\n0xfffe\n\nThe minimum and maximum representable values of primitive numeric types such as integers are given by the typemin and typemax functions:\n\njulia> (typemin(Int32), typemax(Int32))\n(-2147483648, 2147483647)\n\njulia> for T in [Int8,Int16,Int32,Int64,Int128,UInt8,UInt16,UInt32,UInt64,UInt128]\n           println(\"$(lpad(T,7)): [$(typemin(T)),$(typemax(T))]\")\n       end\n   Int8: [-128,127]\n  Int16: [-32768,32767]\n  Int32: [-2147483648,2147483647]\n  Int64: [-9223372036854775808,9223372036854775807]\n Int128: [-170141183460469231731687303715884105728,170141183460469231731687303715884105727]\n  UInt8: [0,255]\n UInt16: [0,65535]\n UInt32: [0,4294967295]\n UInt64: [0,18446744073709551615]\nUInt128: [0,340282366920938463463374607431768211455]\n\nThe values returned by typemin and typemax are always of the given argument type. (The above expression uses several features that have yet to be introduced, including for loops, Strings, and Interpolation, but should be easy enough to understand for users with some existing programming experience.)"},{"location":"manual/integers-and-floating-point-numbers.html#Overflow-behavior","page":"Integers and Floating-Point Numbers","title":"Overflow behavior","category":"section","text":"In Julia, exceeding the maximum representable value of a given type results in a wraparound behavior:\n\njulia> x = typemax(Int64)\n9223372036854775807\n\njulia> x + 1\n-9223372036854775808\n\njulia> x + 1 == typemin(Int64)\ntrue\n\nArithmetic operations with Julia's integer types inherently perform modular arithmetic, mirroring the characteristics of integer arithmetic on modern computer hardware. In scenarios where overflow is a possibility, it is crucial to explicitly check for wraparound effects that can result from such overflows. The Base.Checked module provides a suite of arithmetic operations equipped with overflow checks, which trigger errors if an overflow occurs. For use cases where overflow cannot be tolerated under any circumstances, utilizing the BigInt type, as detailed in Arbitrary Precision Arithmetic, is advisable.\n\nAn example of overflow behavior and how to potentially resolve it is as follows:\n\njulia> 10^19\n-8446744073709551616\n\njulia> big(10)^19\n10000000000000000000"},{"location":"manual/integers-and-floating-point-numbers.html#Division-errors","page":"Integers and Floating-Point Numbers","title":"Division errors","category":"section","text":"Integer division (the div function) has two exceptional cases: dividing by zero, and dividing the lowest negative number (typemin) by -1. Both of these cases throw a DivideError. The remainder and modulus functions (rem and mod) throw a DivideError when their second argument is zero."},{"location":"manual/integers-and-floating-point-numbers.html#Floating-Point-Numbers","page":"Integers and Floating-Point Numbers","title":"Floating-Point Numbers","category":"section","text":"Literal floating-point numbers are represented in the standard formats, using E-notation when necessary:\n\njulia> 1.0\n1.0\n\njulia> 1.\n1.0\n\njulia> 0.5\n0.5\n\njulia> .5\n0.5\n\njulia> -1.23\n-1.23\n\njulia> 1e10\n1.0e10\n\njulia> 2.5e-4\n0.00025\n\nThe above results are all Float64 values. Literal Float32 values can be entered by writing an f in place of e:\n\njulia> x = 0.5f0\n0.5f0\n\njulia> typeof(x)\nFloat32\n\njulia> 2.5f-4\n0.00025f0\n\nValues can be converted to Float32 easily:\n\njulia> x = Float32(-1.5)\n-1.5f0\n\njulia> typeof(x)\nFloat32\n\nHexadecimal floating-point literals are also valid, but only as Float64 values, with p preceding the base-2 exponent:\n\njulia> 0x1p0\n1.0\n\njulia> 0x1.8p3\n12.0\n\njulia> x = 0x.4p-1\n0.125\n\njulia> typeof(x)\nFloat64\n\nHalf-precision floating-point numbers are also supported (Float16) on all platforms, with native instructions used on hardware which supports this number format. Otherwise, operations are implemented in software, and use Float32 for intermediate calculations. As an internal implementation detail, this is achieved under the hood by using LLVM's half type, which behaves similarly to what the GCC -fexcess-precision=16 flag does for C/C++ code.\n\njulia> sizeof(Float16(4.))\n2\n\njulia> 2*Float16(4.)\nFloat16(8.0)\n\nThe underscore _ can be used as digit separator:\n\njulia> 10_000, 0.000_000_005, 0xdead_beef, 0b1011_0010\n(10000, 5.0e-9, 0xdeadbeef, 0xb2)"},{"location":"manual/integers-and-floating-point-numbers.html#Floating-point-zero","page":"Integers and Floating-Point Numbers","title":"Floating-point zero","category":"section","text":"Floating-point numbers have two zeros, positive zero and negative zero. They are equal to each other but have different binary representations, as can be seen using the bitstring function:\n\njulia> 0.0 == -0.0\ntrue\n\njulia> bitstring(0.0)\n\"0000000000000000000000000000000000000000000000000000000000000000\"\n\njulia> bitstring(-0.0)\n\"1000000000000000000000000000000000000000000000000000000000000000\""},{"location":"manual/integers-and-floating-point-numbers.html#Special-floating-point-values","page":"Integers and Floating-Point Numbers","title":"Special floating-point values","category":"section","text":"There are three specified standard floating-point values that do not correspond to any point on the real number line:\n\nFloat16 Float32 Float64 Name Description\nInf16 Inf32 Inf positive infinity a value greater than all finite floating-point values\n-Inf16 -Inf32 -Inf negative infinity a value less than all finite floating-point values\nNaN16 NaN32 NaN not a number a value not == to any floating-point value (including itself)\n\nFor further discussion of how these non-finite floating-point values are ordered with respect to each other and other floats, see Numeric Comparisons. By the IEEE 754 standard, these floating-point values are the results of certain arithmetic operations:\n\njulia> 1/Inf\n0.0\n\njulia> 1/0\nInf\n\njulia> -5/0\n-Inf\n\njulia> 0.000001/0\nInf\n\njulia> 0/0\nNaN\n\njulia> 500 + Inf\nInf\n\njulia> 500 - Inf\n-Inf\n\njulia> Inf + Inf\nInf\n\njulia> Inf - Inf\nNaN\n\njulia> Inf * Inf\nInf\n\njulia> Inf / Inf\nNaN\n\njulia> 0 * Inf\nNaN\n\njulia> NaN == NaN\nfalse\n\njulia> NaN != NaN\ntrue\n\njulia> NaN < NaN\nfalse\n\njulia> NaN > NaN\nfalse\n\nThe typemin and typemax functions also apply to floating-point types:\n\njulia> (typemin(Float16),typemax(Float16))\n(-Inf16, Inf16)\n\njulia> (typemin(Float32),typemax(Float32))\n(-Inf32, Inf32)\n\njulia> (typemin(Float64),typemax(Float64))\n(-Inf, Inf)"},{"location":"manual/integers-and-floating-point-numbers.html#Machine-epsilon","page":"Integers and Floating-Point Numbers","title":"Machine epsilon","category":"section","text":"Most real numbers cannot be represented exactly with floating-point numbers, and so for many purposes it is important to know the distance between two adjacent representable floating-point numbers, which is often known as machine epsilon.\n\nJulia provides eps, which gives the distance between 1.0 and the next larger representable floating-point value:\n\njulia> eps(Float32)\n1.1920929f-7\n\njulia> eps(Float64)\n2.220446049250313e-16\n\njulia> eps() # same as eps(Float64)\n2.220446049250313e-16\n\nThese values are 2.0^-23 and 2.0^-52 as Float32 and Float64 values, respectively. The eps function can also take a floating-point value as an argument, and gives the absolute difference between that value and the next representable floating point value. That is, eps(x) yields a value of the same type as x such that x + eps(x) is the next representable floating-point value larger than x:\n\njulia> eps(1.0)\n2.220446049250313e-16\n\njulia> eps(1000.)\n1.1368683772161603e-13\n\njulia> eps(1e-27)\n1.793662034335766e-43\n\njulia> eps(0.0)\n5.0e-324\n\nThe distance between two adjacent representable floating-point numbers is not constant, but is smaller for smaller values and larger for larger values. In other words, the representable floating-point numbers are densest in the real number line near zero, and grow sparser exponentially as one moves farther away from zero. By definition, eps(1.0) is the same as eps(Float64) since 1.0 is a 64-bit floating-point value.\n\nJulia also provides the nextfloat and prevfloat functions which return the next largest or smallest representable floating-point number to the argument respectively:\n\njulia> x = 1.25f0\n1.25f0\n\njulia> nextfloat(x)\n1.2500001f0\n\njulia> prevfloat(x)\n1.2499999f0\n\njulia> bitstring(prevfloat(x))\n\"00111111100111111111111111111111\"\n\njulia> bitstring(x)\n\"00111111101000000000000000000000\"\n\njulia> bitstring(nextfloat(x))\n\"00111111101000000000000000000001\"\n\nThis example highlights the general principle that the adjacent representable floating-point numbers also have adjacent binary integer representations."},{"location":"manual/integers-and-floating-point-numbers.html#Rounding-modes","page":"Integers and Floating-Point Numbers","title":"Rounding modes","category":"section","text":"If a number doesn't have an exact floating-point representation, it must be rounded to an appropriate representable value. However, the manner in which this rounding is done can be changed if required according to the rounding modes presented in the IEEE 754 standard.\n\nThe default mode used is always RoundNearest, which rounds to the nearest representable value, with ties rounded towards the nearest value with an even least significant bit."},{"location":"manual/integers-and-floating-point-numbers.html#Background-and-References","page":"Integers and Floating-Point Numbers","title":"Background and References","category":"section","text":"Floating-point arithmetic entails many subtleties which can be surprising to users who are unfamiliar with the low-level implementation details. However, these subtleties are described in detail in most books on scientific computation, and also in the following references:\n\nThe definitive guide to floating point arithmetic is the IEEE 754-2008 Standard; however, it is not available for free online.\nFor a brief but lucid presentation of how floating-point numbers are represented, see John D. Cook's article on the subject as well as his introduction to some of the issues arising from how this representation differs in behavior from the idealized abstraction of real numbers.\nAlso recommended is Bruce Dawson's series of blog posts on floating-point numbers.\nFor an excellent, in-depth discussion of floating-point numbers and issues of numerical accuracy encountered when computing with them, see David Goldberg's paper What Every Computer Scientist Should Know About Floating-Point Arithmetic.\nFor even more extensive documentation of the history of, rationale for, and issues with floating-point numbers, as well as discussion of many other topics in numerical computing, see the collected writings of William Kahan, commonly known as the \"Father of Floating-Point\". Of particular interest may be An Interview with the Old Man of Floating-Point."},{"location":"manual/integers-and-floating-point-numbers.html#Arbitrary-Precision-Arithmetic","page":"Integers and Floating-Point Numbers","title":"Arbitrary Precision Arithmetic","category":"section","text":"To allow computations with arbitrary-precision integers and floating point numbers, Julia wraps the GNU Multiple Precision Arithmetic Library (GMP) and the GNU MPFR Library, respectively. The BigInt and BigFloat types are available in Julia for arbitrary precision integer and floating point numbers respectively.\n\nConstructors exist to create these types from primitive numerical types, and the string literal @big_str or parse can be used to construct them from AbstractStrings. BigInts can also be input as integer literals when they are too big for other built-in integer types. Note that as there is no unsigned arbitrary-precision integer type in Base (BigInt is sufficient in most cases), hexadecimal, octal and binary literals can be used (in addition to decimal literals).\n\nOnce created, they participate in arithmetic with all other numeric types thanks to Julia's type promotion and conversion mechanism:\n\njulia> BigInt(typemax(Int64)) + 1\n9223372036854775808\n\njulia> big\"123456789012345678901234567890\" + 1\n123456789012345678901234567891\n\njulia> parse(BigInt, \"123456789012345678901234567890\") + 1\n123456789012345678901234567891\n\njulia> string(big\"2\"^200, base=16)\n\"100000000000000000000000000000000000000000000000000\"\n\njulia> 0x100000000000000000000000000000000-1 == typemax(UInt128)\ntrue\n\njulia> 0x000000000000000000000000000000000\n0\n\njulia> typeof(ans)\nBigInt\n\njulia> big\"1.23456789012345678901\"\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> parse(BigFloat, \"1.23456789012345678901\")\n1.234567890123456789010000000000000000000000000000000000000000000000000000000004\n\njulia> BigFloat(2.0^66) / 3\n2.459565876494606882133333333333333333333333333333333333333333333333333333333344e19\n\njulia> factorial(BigInt(40))\n815915283247897734345611269596115894272000000000\n\nHowever, type promotion between the primitive types above and BigInt/BigFloat is not automatic and must be explicitly stated.\n\njulia> x = typemin(Int64)\n-9223372036854775808\n\njulia> x = x - 1\n9223372036854775807\n\njulia> typeof(x)\nInt64\n\njulia> y = BigInt(typemin(Int64))\n-9223372036854775808\n\njulia> y = y - 1\n-9223372036854775809\n\njulia> typeof(y)\nBigInt\n\nThe default precision (in number of bits of the significand) and rounding mode of BigFloat operations can be changed globally by calling setprecision and setrounding, and all further calculations will take these changes in account. Alternatively, the precision or the rounding can be changed only within the execution of a particular block of code by using the same functions with a do block:\n\njulia> setrounding(BigFloat, RoundUp) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.100000000000000000000000000000000000000000000000000000000000000000000000000003\n\njulia> setrounding(BigFloat, RoundDown) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.099999999999999999999999999999999999999999999999999999999999999999999999999986\n\njulia> setprecision(40) do\n           BigFloat(1) + parse(BigFloat, \"0.1\")\n       end\n1.1000000000004\n\nwarning: Warning\nThe relation between setprecision or setrounding and @big_str, the macro used for big string literals (such as big\"0.3\"), might not be intuitive, as a consequence of the fact that @big_str is a macro. See the @big_str documentation for details."},{"location":"manual/integers-and-floating-point-numbers.html#man-numeric-literal-coefficients","page":"Integers and Floating-Point Numbers","title":"Numeric Literal Coefficients","category":"section","text":"To make common numeric formulae and expressions clearer, Julia allows variables to be immediately preceded by a numeric literal, implying multiplication. This makes writing polynomial expressions much cleaner:\n\njulia> x = 3\n3\n\njulia> 2x^2 - 3x + 1\n10\n\njulia> 1.5x^2 - .5x + 1\n13.0\n\nIt also makes writing exponential functions more elegant:\n\njulia> 2^2x\n64\n\nThe precedence of numeric literal coefficients is slightly lower than that of unary operators such as negation. So -2x is parsed as (-2) * x and √2x is parsed as (√2) * x. However, numeric literal coefficients parse similarly to unary operators when combined with exponentiation. For example 2^3x is parsed as 2^(3x), and 2x^3 is parsed as 2*(x^3).\n\nNumeric literals also work as coefficients to parenthesized expressions:\n\njulia> 2(x-1)^2 - 3(x-1) + 1\n3\n\nnote: Note\nThe precedence of numeric literal coefficients used for implicit multiplication is higher than other binary operators such as multiplication (*), and division (/, \\, and //). This means, for example, that 1 / 2im equals -0.5im and 6 // 2(2 + 1) equals 1 // 1.\n\nAdditionally, parenthesized expressions can be used as coefficients to variables, implying multiplication of the expression by the variable:\n\njulia> (x-1)x\n6\n\nNeither juxtaposition of two parenthesized expressions, nor placing a variable before a parenthesized expression, however, can be used to imply multiplication:\n\njulia> (x-1)(x+1)\nERROR: MethodError: objects of type Int64 are not callable\n\njulia> x(x+1)\nERROR: MethodError: objects of type Int64 are not callable\n\nBoth expressions are interpreted as function application: any expression that is not a numeric literal, when immediately followed by a parenthetical, is interpreted as a function applied to the values in parentheses (see Functions for more about functions). Thus, in both of these cases, an error occurs since the left-hand value is not a function.\n\nThe above syntactic enhancements significantly reduce the visual noise incurred when writing common mathematical formulae. Note that no whitespace may come between a numeric literal coefficient and the identifier or parenthesized expression which it multiplies."},{"location":"manual/integers-and-floating-point-numbers.html#Syntax-Conflicts","page":"Integers and Floating-Point Numbers","title":"Syntax Conflicts","category":"section","text":"Juxtaposed literal coefficient syntax may conflict with some numeric literal syntaxes: hexadecimal, octal and binary integer literals and engineering notation for floating-point literals. Here are some situations where syntactic conflicts arise:\n\nThe hexadecimal integer literal expression 0xff could be interpreted as the numeric literal 0 multiplied by the variable xff. Similar ambiguities arise with octal and binary literals like 0o777 or 0b01001010.\nThe floating-point literal expression 1e10 could be interpreted as the numeric literal 1 multiplied by the variable e10, and similarly with the equivalent E form.\nThe 32-bit floating-point literal expression 1.5f22 could be interpreted as the numeric literal 1.5 multiplied by the variable f22.\n\nIn all cases the ambiguity is resolved in favor of interpretation as numeric literals:\n\nExpressions starting with 0x/0o/0b are always hexadecimal/octal/binary literals.\nExpressions starting with a numeric literal followed by e or E are always floating-point literals.\nExpressions starting with a numeric literal followed by f are always 32-bit floating-point literals.\n\nUnlike E, which is equivalent to e in numeric literals for historical reasons, F is just another letter and does not behave like f in numeric literals. Hence, expressions starting with a numeric literal followed by F are interpreted as the numerical literal multiplied by a variable, which means that, for example, 1.5F22 is equal to 1.5 * F22."},{"location":"manual/integers-and-floating-point-numbers.html#Literal-zero-and-one","page":"Integers and Floating-Point Numbers","title":"Literal zero and one","category":"section","text":"Julia provides functions which return literal 0 and 1 corresponding to a specified type or the type of a given variable.\n\nFunction Description\nzero(x) Literal zero of type x or type of variable x\none(x) Literal one of type x or type of variable x\n\nThese functions are useful in Numeric Comparisons to avoid overhead from unnecessary type conversion.\n\nExamples:\n\njulia> zero(Float32)\n0.0f0\n\njulia> zero(1.0)\n0.0\n\njulia> one(Int32)\n1\n\njulia> one(BigFloat)\n1.0"},{"location":"base/scopedvalues.html#scoped-values","page":"Scoped Values","title":"Scoped Values","category":"section","text":"Scoped values provide an implementation of dynamic scoping in Julia.\n\nnote: Lexical scoping vs dynamic scoping\nLexical scoping is the default behavior in Julia. Under lexical scoping the scope of a variable is determined by the lexical (textual) structure of a program. Under dynamic scoping a variable is bound to the most recent assigned value during the program's execution.\n\nThe state of a scoped value is dependent on the execution path of the program. This means that for a scoped value you may observe multiple different values concurrently.\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.\n\nIn its simplest form you can create a ScopedValue with a default value and then use with or @with to enter a new dynamic scope. The new scope will inherit all values from the parent scope (and recursively from all outer scopes) with the provided scoped value taking priority over previous definitions.\n\nLet's first look at an example of lexical scope. A let statement begins a new lexical scope within which the outer definition of x is shadowed by its inner definition.\n\njulia> x = 1\n1\n\njulia> let x = 5\n           @show x\n       end;\nx = 5\n\njulia> @show x;\nx = 1\n\nIn the following example, since Julia uses lexical scope, the variable x in the body of f refers to the x defined in the global scope, and entering a let scope does not change the value f observes.\n\njulia> x = 1\n1\n\njulia> f() = @show x\nf (generic function with 1 method)\n\njulia> let x = 5\n           f()\n       end;\nx = 1\n\njulia> f();\nx = 1\n\nNow using a ScopedValue we can use dynamic scoping.\n\njulia> using Base.ScopedValues\n\njulia> x = ScopedValue(1)\nScopedValue{Int64}(1)\n\njulia> f() = @show x[]\nf (generic function with 1 method)\n\njulia> with(x=>5) do\n           f()\n       end;\nx[] = 5\n\njulia> f();\nx[] = 1\n\nNote that the observed value of the ScopedValue is dependent on the execution path of the program.\n\nIt often makes sense to use a const variable to point to a scoped value, and you can set the value of multiple ScopedValues with one call to with.\n\nusing Base.ScopedValues\n\nf() = @show a[]\ng() = @show b[]\n\nconst a = ScopedValue(1)\nconst b = ScopedValue(2)\n\nf() # a[] = 1\ng() # b[] = 2\n\n# Enter a new dynamic scope and set value.\nwith(a => 3) do\n    f() # a[] = 3\n    g() # b[] = 2\n    with(a => 4, b => 5) do\n        f() # a[] = 4\n        g() # b[] = 5\n    end\n    f() # a[] = 3\n    g() # b[] = 2\nend\n\nf() # a[] = 1\ng() # b[] = 2\n\nScopedValues provides a macro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. @with var=>val expr is equivalent to with(var=>val) do expr end. However, with requires a zero-argument closure or function, which results in an extra call-frame. As an example, consider the following function f:\n\nusing Base.ScopedValues\nconst a = ScopedValue(1)\nf(x) = a[] + x\n\nIf you wish to run f in a dynamic scope with a set to 2, then you can use with:\n\nwith(() -> f(10), a=>2)\n\nHowever, this requires wrapping f in a zero-argument function. If you wish to avoid the extra call-frame, then you can use the @with macro:\n\n@with a=>2 f(10)\n\nnote: Note\nDynamic scopes are inherited by Tasks, at the moment of task creation. Dynamic scopes are not propagated through Distributed.jl operations.\n\nIn the example below we open a new dynamic scope before launching a task. The parent task and the two child tasks observe independent values of the same scoped value at the same time.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst scoped_val = ScopedValue(1)\n@sync begin\n    with(scoped_val => 2)\n        @spawn @show scoped_val[] # 2\n    end\n    with(scoped_val => 3)\n        @spawn @show scoped_val[] # 3\n    end\n    @show scoped_val[] # 1\nend\n\nScoped values are constant throughout a scope, but you can store mutable state in a scoped value. Just keep in mind that the usual caveats for global variables apply in the context of concurrent programming.\n\nCare is also required when storing references to mutable state in scoped values. You might want to explicitly unshare mutable state when entering a new dynamic scope.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# Example of using a mutable value wrongly\n@sync begin\n    # `Dict` is not thread-safe the usage below is invalid\n    @spawn (sval_dict[][:a] = 3)\n    @spawn (sval_dict[][:b] = 3)\nend\n\n@sync begin\n    # If we instead pass a unique dictionary to each\n    # task we can access the dictionaries race free.\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:a] = 3)\n    end\n    with(sval_dict => Dict()) do\n        @spawn (sval_dict[][:b] = 3)\n    end\nend"},{"location":"base/scopedvalues.html#Example","page":"Scoped Values","title":"Example","category":"section","text":"In the example below we use a scoped value to implement a permission check in a web-application. After determining the permissions of the request, a new dynamic scope is entered and the scoped value LEVEL is set. Other parts of the application can query the scoped value and will receive the appropriate value. Other alternatives like task-local storage and global variables are not well suited for this kind of propagation; our only alternative would have been to thread a value through the entire call-chain.\n\nusing Base.ScopedValues\n\nconst LEVEL = ScopedValue(:GUEST)\n\nfunction serve(request, response)\n    level = isAdmin(request) ? :ADMIN : :GUEST\n    with(LEVEL => level) do\n        Threads.@spawn handle(request, response)\n    end\nend\n\nfunction open(connection::Database)\n    level = LEVEL[]\n    if level !== :ADMIN\n        error(\"Access disallowed\")\n    end\n    # ... open connection\nend\n\nfunction handle(request, response)\n    # ...\n    open(Database(#=...=#))\n    # ...\nend"},{"location":"base/scopedvalues.html#Idioms","page":"Scoped Values","title":"Idioms","category":"section","text":""},{"location":"base/scopedvalues.html#unshare_mutable_state","page":"Scoped Values","title":"Unshare mutable state","category":"section","text":"using Base.ScopedValues\nimport Base.Threads: @spawn\n\nconst sval_dict = ScopedValue(Dict())\n\n# If you want to add new values to the dict, instead of replacing\n# it, unshare the values explicitly. In this example we use `merge`\n# to unshare the state of the dictionary in parent scope.\n@sync begin\n    with(sval_dict => merge(sval_dict[], Dict(:a => 10))) do\n        @spawn @show sval_dict[][:a]\n    end\n    @spawn sval_dict[][:a] = 3 # Not a race since they are unshared.\nend"},{"location":"base/scopedvalues.html#Scoped-values-as-globals","page":"Scoped Values","title":"Scoped values as globals","category":"section","text":"In order to access the value of a scoped value, the scoped value itself has to be in (lexical) scope. This means most often you likely want to use scoped values as constant globals.\n\nusing Base.ScopedValues\nconst sval = ScopedValue(1)\n\nIndeed one can think of scoped values as hidden function arguments.\n\nThis does not preclude their use as non-globals.\n\nusing Base.ScopedValues\nimport Base.Threads: @spawn\n\nfunction main()\n    role = ScopedValue(:client)\n\n    function launch()\n        #...\n        role[]\n    end\n\n    @with role => :server @spawn launch()\n    launch()\nend\n\nBut it might have been simpler to just directly pass the function argument in these cases."},{"location":"base/scopedvalues.html#Very-many-ScopedValues","page":"Scoped Values","title":"Very many ScopedValues","category":"section","text":"If you find yourself creating many ScopedValue's for one given module, it may be better to use a dedicated struct to hold them.\n\nusing Base.ScopedValues\n\nBase.@kwdef struct Configuration\n    color::Bool = false\n    verbose::Bool = false\nend\n\nconst CONFIG = ScopedValue(Configuration(color=true))\n\n@with CONFIG => Configuration(color=CONFIG[].color, verbose=true) begin\n    @show CONFIG[].color # true\n    @show CONFIG[].verbose # true\nend"},{"location":"base/scopedvalues.html#API-docs","page":"Scoped Values","title":"API docs","category":"section","text":""},{"location":"base/scopedvalues.html#Implementation-notes-and-performance","page":"Scoped Values","title":"Implementation notes and performance","category":"section","text":"Scopes use a persistent dictionary. Lookup and insertion is O(log(32, n)), upon dynamic scope entry a small amount of data is copied and the unchanged data is shared among other scopes.\n\nThe Scope object itself is not user-facing and may be changed in a future version of Julia."},{"location":"base/scopedvalues.html#Design-inspiration","page":"Scoped Values","title":"Design inspiration","category":"section","text":"This design was heavily inspired by JEPS-429, which in turn was inspired by dynamically scoped free variables in many Lisp dialects. In particular Interlisp-D and its deep binding strategy.\n\nA prior design discussed was context variables ala PEPS-567 and implemented in Julia as ContextVariablesX.jl."},{"location":"base/scopedvalues.html#Base.ScopedValues.ScopedValue","page":"Scoped Values","title":"Base.ScopedValues.ScopedValue","category":"type","text":"ScopedValue(x)\n\nCreate a container that propagates values across dynamic scopes. Use with to create and enter a new dynamic scope.\n\nValues can only be set when entering a new dynamic scope, and the value referred to will be constant during the execution of a dynamic scope.\n\nDynamic scopes are propagated across tasks.\n\nExamples\n\njulia> using Base.ScopedValues;\n\njulia> const sval = ScopedValue(1);\n\njulia> sval[]\n1\n\njulia> with(sval => 2) do\n           sval[]\n       end\n2\n\njulia> sval[]\n1\n\ncompat: Julia 1.11\nScoped values were introduced in Julia 1.11. In Julia 1.8+ a compatible implementation is available from the package ScopedValues.jl.\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.with","page":"Scoped Values","title":"Base.ScopedValues.with","category":"function","text":"with(f, (var::ScopedValue{T} => val)...)\n\nExecute f in a new dynamic scope with var set to val. val will be converted to type T.\n\nSee also: ScopedValues.@with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> f(10)\n11\n\njulia> with(a=>2) do\n           f(10)\n       end\n12\n\njulia> f(10)\n11\n\njulia> b = ScopedValue(2);\n\njulia> g(x) = a[] + b[] + x;\n\njulia> with(a=>10, b=>20) do\n           g(30)\n       end\n60\n\njulia> with(() -> a[] * b[], a=>3, b=>4)\n12\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.@with","page":"Scoped Values","title":"Base.ScopedValues.@with","category":"macro","text":"@with (var::ScopedValue{T} => val)... expr\n\nMacro version of with. The expression @with var=>val expr evaluates expr in a new dynamic scope with var set to val. val will be converted to type T. @with var=>val expr is equivalent to with(var=>val) do expr end, but @with avoids creating a closure.\n\nSee also: ScopedValues.with, ScopedValues.ScopedValue, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> const a = ScopedValue(1);\n\njulia> f(x) = a[] + x;\n\njulia> @with a=>2 f(10)\n12\n\njulia> @with a=>3 begin\n           x = 100\n           f(x)\n       end\n103\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.isassigned-Tuple{Base.ScopedValues.ScopedValue}","page":"Scoped Values","title":"Base.isassigned","category":"method","text":"isassigned(val::ScopedValue)\n\nTest whether a ScopedValue has an assigned value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.get.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(1); b = ScopedValue{Int}();\n\njulia> isassigned(a)\ntrue\n\njulia> isassigned(b)\nfalse\n\n\n\n\n\n"},{"location":"base/scopedvalues.html#Base.ScopedValues.get","page":"Scoped Values","title":"Base.ScopedValues.get","category":"function","text":"get(val::ScopedValue{T})::Union{Nothing, Some{T}}\nget(val::LazyScopedValue{T})::Union{Nothing, Some{T}}\n\nIf the scoped value isn't set and doesn't have a default value, return nothing. Otherwise returns Some{T} with the current value.\n\nSee also: ScopedValues.with, ScopedValues.@with, ScopedValues.ScopedValue.\n\nExamples\n\njulia> using Base.ScopedValues\n\njulia> a = ScopedValue(42); b = ScopedValue{Int}();\n\njulia> ScopedValues.get(a)\nSome(42)\n\njulia> isnothing(ScopedValues.get(b))\ntrue\n\n\n\n\n\n"},{"location":"manual/calling-c-and-fortran-code.html#Calling-C-and-Fortran-Code","page":"Calling C and Fortran Code","title":"Calling C and Fortran Code","category":"section","text":"Though most code can be written in Julia, there are many high-quality, mature libraries for numerical computing already written in C and Fortran. To allow easy use of this existing code, Julia makes it simple and efficient to call C and Fortran functions. Julia has a \"no boilerplate\" philosophy: functions can be called directly from Julia without any \"glue\" code, code generation, or compilation – even from the interactive prompt. This is accomplished just by making an appropriate call with the @ccall macro (or the less convenient ccall syntax, see the ccall syntax section).\n\nThe code to be called must be available as a shared library. Most C and Fortran libraries ship compiled as shared libraries already, but if you are compiling the code yourself using GCC (or Clang), you will need to use the -shared and -fPIC options. The machine instructions generated by Julia's JIT are the same as a native C call would be, so the resulting overhead is the same as calling a library function from C code. [1]\n\nBy default, Fortran compilers generate mangled names (for example, converting function names to lowercase or uppercase, often appending an underscore), and so to call a Fortran function you must pass the mangled identifier corresponding to the rule followed by your Fortran compiler. Also, when calling a Fortran function, all inputs must be passed as pointers to allocated values on the heap or stack. This applies not only to arrays and other mutable objects which are normally heap-allocated, but also to scalar values such as integers and floats which are normally stack-allocated and commonly passed in registers when using C or Julia calling conventions.\n\nThe syntax for @ccall to generate a call to the library function is:\n\n@ccall library.function_name(argvalue1::argtype1, ...)::returntype\n@ccall function_name(argvalue1::argtype1, ...)::returntype\n@ccall $function_pointer(argvalue1::argtype1, ...)::returntype\n\nwhere library is a string constant or global variable name (see Non-constant Function -Specifications below). The library can be just a name, or it can specify a full path to the library. The library may be omitted, in which case the function name is resolved in the current executable, the current libc, or libjulia(-internal). This form can be used to call C library functions, functions in the Julia runtime, or functions in an application linked to Julia. Omitting the library cannot be used to call a function in any library (like specifying RTLD_DEFAULT to dlsym) as such behavior is slow, complicated, and not implemented on all platforms. Alternatively, @ccall may also be used to call a function pointer $function_pointer, such as one returned by Libdl.dlsym. The argtypes corresponds to the C-function signature and the argvalues are the actual argument values to be passed to the function.\n\nnote: Note\nSee below for how to map C types to Julia types.\n\nAs a complete but simple example, the following calls the clock function from the standard C library on most Unix-derived systems:\n\njulia> t = @ccall clock()::Int32\n2292761\n\njulia> typeof(t)\nInt32\n\nclock takes no arguments and returns an Int32. To call the getenv function to get a pointer to the value of an environment variable, one makes a call like this:\n\njulia> path = @ccall getenv(\"SHELL\"::Cstring)::Cstring\nCstring(@0x00007fff5fbffc45)\n\njulia> unsafe_string(path)\n\"/bin/bash\"\n\nIn practice, especially when providing reusable functionality, one generally wraps @ccall uses in Julia functions that set up arguments and then check for errors in whatever manner the C or Fortran function specifies. If an error occurs it is thrown as a normal Julia exception. This is especially important since C and Fortran APIs are notoriously inconsistent about how they indicate error conditions. For example, the getenv C library function is wrapped in the following Julia function, which is a simplified version of the actual definition from env.jl:\n\nfunction getenv(var::AbstractString)\n    val = @ccall getenv(var::Cstring)::Cstring\n    if val == C_NULL\n        error(\"getenv: undefined variable: \", var)\n    end\n    return unsafe_string(val)\nend\n\nThe C getenv function indicates an error by returning C_NULL, but other standard C functions indicate errors in different ways, including by returning -1, 0, 1, and other special values. This wrapper throws an exception indicating the problem if the caller tries to get a non-existent environment variable:\n\njulia> getenv(\"SHELL\")\n\"/bin/bash\"\n\njulia> getenv(\"FOOBAR\")\nERROR: getenv: undefined variable: FOOBAR\n\nHere is a slightly more complex example that discovers the local machine's hostname.\n\nfunction gethostname()\n    hostname = Vector{UInt8}(undef, 256) # MAXHOSTNAMELEN\n    err = @ccall gethostname(hostname::Ptr{UInt8}, sizeof(hostname)::Csize_t)::Int32\n    Base.systemerror(\"gethostname\", err != 0)\n    hostname[end] = 0 # ensure null-termination\n    return GC.@preserve hostname unsafe_string(pointer(hostname))\nend\n\nThis example first allocates an array of bytes. It then calls the C library function gethostname to populate the array with the hostname. Finally, it takes a pointer to the hostname buffer, and converts the pointer to a Julia string, assuming that it is a null-terminated C string.\n\nIt is common for C libraries to use this pattern of requiring the caller to allocate memory to be passed to the callee and populated. Allocation of memory from Julia like this is generally accomplished by creating an uninitialized array and passing a pointer to its data to the C function. This is why we don't use the Cstring type here: as the array is uninitialized, it could contain null bytes. Converting to a Cstring as part of the @ccall checks for contained null bytes and could therefore throw a conversion error.\n\nDereferencing pointer(hostname) with unsafe_string is an unsafe operation as it requires access to the memory allocated for hostname that may have been in the meanwhile garbage collected. The macro GC.@preserve prevents this from happening and therefore accessing an invalid memory location.\n\nFinally, here is an example of specifying a library via a path. We create a shared library with the following content\n\n#include <stdio.h>\n\nvoid say_y(int y)\n{\n    printf(\"Hello from C: got y = %d.\\n\", y);\n}\n\nand compile it with gcc -fPIC -shared -o mylib.so mylib.c. It can then be called by specifying the (absolute) path as the library name:\n\njulia> @ccall \"./mylib.so\".say_y(5::Cint)::Cvoid\nHello from C: got y = 5."},{"location":"manual/calling-c-and-fortran-code.html#Creating-C-Compatible-Julia-Function-Pointers","page":"Calling C and Fortran Code","title":"Creating C-Compatible Julia Function Pointers","category":"section","text":"It is possible to pass Julia functions to native C functions that accept function pointer arguments. For example, to match C prototypes of the form:\n\ntypedef returntype (*functiontype)(argumenttype, ...)\n\nThe macro @cfunction generates the C-compatible function pointer for a call to a Julia function. The arguments to @cfunction are:\n\nA Julia function\nThe function's return type\nA tuple of input types, corresponding to the function signature\n\nnote: Note\nAs with @ccall, the return type and the input types must be literal constants.\n\nnote: Note\nCurrently, only the platform-default C calling convention is supported. This means that @cfunction-generated pointers cannot be used in calls where WINAPI expects a stdcall function on 32-bit Windows, but can be used on WIN64 (where stdcall is unified with the C calling convention).\n\nnote: Note\nCallback functions exposed via @cfunction should not throw errors, as that will return control to the Julia runtime unexpectedly and may leave the program in an undefined state.\n\nA classic example is the standard C library qsort function, declared as:\n\nvoid qsort(void *base, size_t nitems, size_t size,\n           int (*compare)(const void*, const void*));\n\nThe base argument is a pointer to an array of length nitems, with elements of size bytes each. compare is a callback function which takes pointers to two elements a and b and returns an integer less/greater than zero if a should appear before/after b (or zero if any order is permitted).\n\nNow, suppose that we have a 1-d array A of values in Julia that we want to sort using the qsort function (rather than Julia's built-in sort function). Before we consider calling qsort and passing arguments, we need to write a comparison function:\n\njulia> function mycompare(a, b)::Cint\n           return (a < b) ? -1 : ((a > b) ? +1 : 0)\n       end;\n\nqsort expects a comparison function that return a C int, so we annotate the return type to be Cint.\n\nIn order to pass this function to C, we obtain its address using the macro @cfunction:\n\njulia> mycompare_c = @cfunction(mycompare, Cint, (Ref{Cdouble}, Ref{Cdouble}));\n\n@cfunction requires three arguments: the Julia function (mycompare), the return type (Cint), and a literal tuple of the input argument types, in this case to sort an array of Cdouble (Float64) elements.\n\nThe final call to qsort looks like this:\n\njulia> A = [1.3, -2.7, 4.4, 3.1];\n\njulia> @ccall qsort(A::Ptr{Cdouble}, length(A)::Csize_t, sizeof(eltype(A))::Csize_t, mycompare_c::Ptr{Cvoid})::Cvoid\n\njulia> A\n4-element Vector{Float64}:\n -2.7\n  1.3\n  3.1\n  4.4\n\nAs the example shows, the original Julia array A has now been sorted: [-2.7, 1.3, 3.1, 4.4]. Note that Julia takes care of converting the array to a Ptr{Cdouble}, computing the size of the element type in bytes, and so on.\n\nFor fun, try inserting a println(\"mycompare($a, $b)\") line into mycompare, which will allow you to see the comparisons that qsort is performing (and to verify that it is really calling the Julia function that you passed to it)."},{"location":"manual/calling-c-and-fortran-code.html#mapping-c-types-to-julia","page":"Calling C and Fortran Code","title":"Mapping C Types to Julia","category":"section","text":"It is critical to exactly match the declared C type with its declaration in Julia. Inconsistencies can cause code that works correctly on one system to fail or produce indeterminate results on a different system.\n\nNote that no C header files are used anywhere in the process of calling C functions: you are responsible for making sure that your Julia types and call signatures accurately reflect those in the C header file.[2]"},{"location":"manual/calling-c-and-fortran-code.html#automatic-type-conversion","page":"Calling C and Fortran Code","title":"Automatic Type Conversion","category":"section","text":"Julia automatically inserts calls to the Base.cconvert function to convert each argument to the specified type. For example, the following call:\n\n@ccall \"libfoo\".foo(x::Int32, y::Float64)::Cvoid\n\nwill behave as if it were written like this:\n\nc_x = Base.cconvert(Int32, x)\nc_y = Base.cconvert(Float64, y)\nGC.@preserve c_x c_y begin\n    @ccall \"libfoo\".foo(\n        Base.unsafe_convert(Int32, c_x)::Int32,\n        Base.unsafe_convert(Float64, c_y)::Float64\n    )::Cvoid\nend\n\nBase.cconvert normally just calls convert, but can be defined to return an arbitrary new object more appropriate for passing to C. This should be used to perform all allocations of memory that will be accessed by the C code. For example, this is used to convert an Array of objects (e.g. strings) to an array of pointers.\n\nBase.unsafe_convert handles conversion to Ptr types. It is considered unsafe because converting an object to a native pointer can hide the object from the garbage collector, causing it to be freed prematurely."},{"location":"manual/calling-c-and-fortran-code.html#Type-Correspondences","page":"Calling C and Fortran Code","title":"Type Correspondences","category":"section","text":"First, let's review some relevant Julia type terminology:\n\nSyntax / Keyword Example Description\nmutable struct BitSet \"Concrete Type\" :: A group of related data that includes a type-tag, is managed by the Julia GC, and is defined by object-identity. The type parameters of a concrete type must be fully defined (no TypeVars are allowed) in order for the instance to be constructed. Also see isconcretetype.\nabstract type Any, AbstractArray{T, N}, Complex{T} \"Super Type\" :: A super-type (not a concrete type) that cannot be instantiated, but can be used to describe a group of types. Also see isabstracttype.\nT{A} Vector{Int} \"Type Parameter\" :: A specialization of a type (typically used for dispatch or storage optimization).\n  \"TypeVar\" :: The T in the type parameter declaration is referred to as a TypeVar (short for type variable).\nprimitive type Int, Float64 \"Primitive Type\" :: A type with no fields, but a size. It is stored and defined by-value.\nstruct Pair{Int, Int} \"Struct\" :: A type with all fields defined to be constant. It is defined by-value, and may be stored with a type-tag.\n ComplexF64 (isbits) \"Is-Bits\"   :: A primitive type, or a struct type where all fields are other isbits types. It is defined by-value, and is stored without a type-tag.\nstruct ...; end nothing \"Singleton\" :: a concrete Type or Struct with no fields.\n(...) or tuple(...) (1, 2, 3) \"Tuple\" :: an immutable data-structure similar to an anonymous struct type, or a constant array. Represented as either an array or a struct."},{"location":"manual/calling-c-and-fortran-code.html#man-bits-types","page":"Calling C and Fortran Code","title":"Bits Types","category":"section","text":"There are several special types to be aware of, as no other type can be defined to behave the same:\n\nFloat32\nExactly corresponds to the float type in C (or REAL*4 in Fortran).\nFloat64\nExactly corresponds to the double type in C (or REAL*8 in Fortran).\nComplexF32\nExactly corresponds to the complex float type in C (or COMPLEX*8 in Fortran).\nComplexF64\nExactly corresponds to the complex double type in C (or COMPLEX*16 in Fortran).\nSigned\nExactly corresponds to the signed type annotation in C (or any INTEGER type in Fortran). Any Julia type that is not a subtype of Signed is assumed to be unsigned.\n\nRef{T}\nBehaves like a Ptr{T} that can manage its memory via the Julia GC.\n\nArray{T,N}\nWhen an array is passed to C as a Ptr{T} argument, it is not reinterpret-cast: Julia requires that the element type of the array matches T, and the address of the first element is passed.\nTherefore, if an Array contains data in the wrong format, it will have to be explicitly converted using a call such as trunc.(Int32, A).\nTo pass an array A as a pointer of a different type without converting the data beforehand (for example, to pass a Float64 array to a function that operates on uninterpreted bytes), you can declare the argument as Ptr{Cvoid}.\nIf an array of eltype Ptr{T} is passed as a Ptr{Ptr{T}} argument, Base.cconvert will attempt to first make a null-terminated copy of the array with each element replaced by its Base.cconvert version. This allows, for example, passing an argv pointer array of type Vector{String} to an argument of type Ptr{Ptr{Cchar}}.\n\nOn all systems we currently support, basic C/C++ value types may be translated to Julia types as follows. Every C type also has a corresponding Julia type with the same name, prefixed by C. This can help when writing portable code (and remembering that an int in C is not the same as an Int in Julia).\n\nSystem Independent Types\n\nC name Fortran name Standard Julia Alias Julia Base Type\nunsigned char CHARACTER Cuchar UInt8\nbool (_Bool in C99+)  Cuchar UInt8\nshort INTEGER*2, LOGICAL*2 Cshort Int16\nunsigned short  Cushort UInt16\nint, BOOL (C, typical) INTEGER*4, LOGICAL*4 Cint Int32\nunsigned int  Cuint UInt32\nlong long INTEGER*8, LOGICAL*8 Clonglong Int64\nunsigned long long  Culonglong UInt64\nintmax_t  Cintmax_t Int64\nuintmax_t  Cuintmax_t UInt64\nfloat REAL*4 Cfloat Float32\ndouble REAL*8 Cdouble Float64\ncomplex float COMPLEX*8 ComplexF32 Complex{Float32}\ncomplex double COMPLEX*16 ComplexF64 Complex{Float64}\nptrdiff_t  Cptrdiff_t Int\nssize_t  Cssize_t Int\nsize_t  Csize_t UInt\nvoid   Cvoid\nvoid and [[noreturn]] or _Noreturn   Union{}\nvoid*   Ptr{Cvoid} (or similarly Ref{Cvoid})\nT* (where T represents an appropriately defined type)   Ref{T} (T may be safely mutated only if T is an isbits type)\nchar* (or char[], e.g. a string) CHARACTER*N  Cstring if null-terminated, or Ptr{UInt8} if not\nchar** (or *char[])   Ptr{Ptr{UInt8}}\njl_value_t* (any Julia Type)   Any\njl_value_t* const* (a reference to a Julia value)   Ref{Any} (const, since mutation would require a write barrier, which is not possible to insert correctly)\nva_arg   Not supported\n... (variadic function specification)   T... (where T is one of the above types, when using the ccall function)\n... (variadic function specification)   ; va_arg1::T, va_arg2::S, etc. (only supported with @ccall macro)\n\nThe Cstring type is essentially a synonym for Ptr{UInt8}, except the conversion to Cstring throws an error if the Julia string contains any embedded null characters (which would cause the string to be silently truncated if the C routine treats null as the terminator). If you are passing a char* to a C routine that does not assume null termination (e.g. because you pass an explicit string length), or if you know for certain that your Julia string does not contain null and want to skip the check, you can use Ptr{UInt8} as the argument type. Cstring can also be used as the ccall return type, but in that case it obviously does not introduce any extra checks and is only meant to improve the readability of the call.\n\nSystem Dependent Types\n\nC name Standard Julia Alias Julia Base Type\nchar Cchar Int8 (x86, x86_64), UInt8 (powerpc, arm)\nlong Clong Int (UNIX), Int32 (Windows)\nunsigned long Culong UInt (UNIX), UInt32 (Windows)\nwchar_t Cwchar_t Int32 (UNIX), UInt16 (Windows)\n\nnote: Note\nWhen calling Fortran, all inputs must be passed by pointers to heap- or stack-allocated values, so all type correspondences above should contain an additional Ptr{..} or Ref{..} wrapper around their type specification.\n\nwarning: Warning\nFor string arguments (char*) the Julia type should be Cstring (if null-terminated data is expected), or either Ptr{Cchar} or Ptr{UInt8} otherwise (these two pointer types have the same effect), as described above, not String. Similarly, for array arguments (T[] or T*), the Julia type should again be Ptr{T}, not Vector{T}.\n\nwarning: Warning\nJulia's Char type is 32 bits, which is not the same as the wide-character type (wchar_t or wint_t) on all platforms.\n\nwarning: Warning\nA return type of Union{} means the function will not return, i.e., C++11 [[noreturn]] or C11 _Noreturn (e.g. jl_throw or longjmp). Do not use this for functions that return no value (void) but do return, for those, use Cvoid instead.\n\nnote: Note\nFor wchar_t* arguments, the Julia type should be Cwstring (if the C routine expects a null-terminated string), or Ptr{Cwchar_t} otherwise. Note also that UTF-8 string data in Julia is internally null-terminated, so it can be passed to C functions expecting null-terminated data without making a copy (but using the Cwstring type will cause an error to be thrown if the string itself contains null characters).\n\nnote: Note\nC functions that take an argument of type char** can be called by using a Ptr{Ptr{UInt8}} type within Julia. For example, C functions of the form:int main(int argc, char **argv);can be called via the following Julia code:argv = [ \"a.out\", \"arg1\", \"arg2\" ]\n@ccall main(length(argv)::Int32, argv::Ptr{Ptr{UInt8}})::Int32\n\nnote: Note\nFor Fortran functions taking variable length strings of type character(len=*) the string lengths are provided as hidden arguments. Type and position of these arguments in the list are compiler specific, where compiler vendors usually default to using Csize_t as type and append the hidden arguments at the end of the argument list. While this behaviour is fixed for some compilers (GNU), others optionally permit placing hidden arguments directly after the character argument (Intel, PGI). For example, Fortran subroutines of the formsubroutine test(str1, str2)\ncharacter(len=*) :: str1,str2can be called via the following Julia code, where the lengths are appendedstr1 = \"foo\"\nstr2 = \"bar\"\nccall(:test, Cvoid, (Ptr{UInt8}, Ptr{UInt8}, Csize_t, Csize_t),\n                    str1, str2, sizeof(str1), sizeof(str2))\n\nwarning: Warning\nFortran compilers may also add other hidden arguments for pointers, assumed-shape (:) and assumed-size (*) arrays. Such behaviour can be avoided by using ISO_C_BINDING and including bind(c) in the definition of the subroutine, which is strongly recommended for interoperable code. In this case, there will be no hidden arguments, at the cost of some language features (e.g. only character(len=1) will be permitted to pass strings).\n\nnote: Note\nA C function declared to return Cvoid will return the value nothing in Julia."},{"location":"manual/calling-c-and-fortran-code.html#Struct-Type-Correspondences","page":"Calling C and Fortran Code","title":"Struct Type Correspondences","category":"section","text":"Composite types such as struct in C or TYPE in Fortran90 (or STRUCTURE / RECORD in some variants of F77), can be mirrored in Julia by creating a struct definition with the same field layout.\n\nWhen used recursively, isbits types are stored inline. All other types are stored as a pointer to the data. When mirroring a struct used by-value inside another struct in C, it is imperative that you do not attempt to manually copy the fields over, as this will not preserve the correct field alignment. Instead, declare an isbits struct type and use that instead. Unnamed structs are not possible in the translation to Julia.\n\nPacked structs and union declarations are not supported by Julia.\n\nYou can get an approximation of a union if you know, a priori, the field that will have the greatest size (potentially including padding). When translating your fields to Julia, declare the Julia field to be only of that type.\n\nArrays of parameters can be expressed with NTuple. For example, the struct in C notation is written as\n\nstruct B {\n    int A[3];\n};\n\nb_a_2 = B.A[2];\n\ncan be written in Julia as\n\nstruct B\n    A::NTuple{3, Cint}\nend\n\nb_a_2 = B.A[3]  # note the difference in indexing (1-based in Julia, 0-based in C)\n\nArrays of unknown size (C99-compliant variable length structs specified by [] or [0]) are not directly supported. Often the best way to deal with these is to deal with the byte offsets directly. For example, if a C library declared a proper string type and returned a pointer to it:\n\nstruct String {\n    int strlen;\n    char data[];\n};\n\nIn Julia, we can access the parts independently to make a copy of that string:\n\nstr = from_c::Ptr{Cvoid}\nlen = unsafe_load(Ptr{Cint}(str))\nunsafe_string(str + Core.sizeof(Cint), len)"},{"location":"manual/calling-c-and-fortran-code.html#Type-Parameters","page":"Calling C and Fortran Code","title":"Type Parameters","category":"section","text":"The type arguments to @ccall and @cfunction are evaluated statically, when the method containing the usage is defined. They therefore must take the form of a literal tuple, not a variable, and cannot reference local variables.\n\nThis may sound like a strange restriction, but remember that since C is not a dynamic language like Julia, its functions can only accept argument types with a statically-known, fixed signature.\n\nHowever, while the type layout must be known statically to compute the intended C ABI, the static parameters of the function are considered to be part of this static environment. The static parameters of the function may be used as type parameters in the call signature, as long as they don't affect the layout of the type. For example, f(x::T) where {T} = @ccall valid(x::Ptr{T})::Ptr{T} is valid, since Ptr is always a word-size primitive type. But, g(x::T) where {T} = @ccall notvalid(x::T)::T is not valid, since the type layout of T is not known statically."},{"location":"manual/calling-c-and-fortran-code.html#SIMD-Values","page":"Calling C and Fortran Code","title":"SIMD Values","category":"section","text":"If a C/C++ routine has an argument or return value that is a native SIMD type, the corresponding Julia type is a homogeneous tuple of VecElement that naturally maps to the SIMD type. Specifically:\n\nThe tuple must be the same size and elements as the SIMD type. For example, a tuple representing an __m128 on x86 must have a size of 16 bytes and Float32 elements.\nThe element type of the tuple must be an instance of VecElement{T} where T is a primitive type with a power-of-two number of bytes (e.g. 1, 2, 4, 8, 16, etc) such as Int8 or Float64.\n\nFor instance, consider this C routine that uses AVX intrinsics:\n\n#include <immintrin.h>\n\n__m256 dist( __m256 a, __m256 b ) {\n    return _mm256_sqrt_ps(_mm256_add_ps(_mm256_mul_ps(a, a),\n                                        _mm256_mul_ps(b, b)));\n}\n\nThe following Julia code calls dist using ccall:\n\nconst m256 = NTuple{8, VecElement{Float32}}\n\na = m256(ntuple(i -> VecElement(sin(Float32(i))), 8))\nb = m256(ntuple(i -> VecElement(cos(Float32(i))), 8))\n\nfunction call_dist(a::m256, b::m256)\n    @ccall \"libdist\".dist(a::m256, b::m256)::m256\nend\n\nprintln(call_dist(a,b))\n\nThe host machine must have the requisite SIMD registers. For example, the code above will not work on hosts without AVX support."},{"location":"manual/calling-c-and-fortran-code.html#Memory-Ownership","page":"Calling C and Fortran Code","title":"Memory Ownership","category":"section","text":"malloc/free\n\nMemory allocation and deallocation of such objects must be handled by calls to the appropriate cleanup routines in the libraries being used, just like in any C program. Do not try to free an object received from a C library with Libc.free in Julia, as this may result in the free function being called via the wrong library and cause the process to abort. The reverse (passing an object allocated in Julia to be freed by an external library) is equally invalid."},{"location":"manual/calling-c-and-fortran-code.html#When-to-use-T,-Ptr{T}-and-Ref{T}","page":"Calling C and Fortran Code","title":"When to use T, Ptr{T} and Ref{T}","category":"section","text":"In Julia code wrapping calls to external C routines, ordinary (non-pointer) data should be declared to be of type T inside the @ccall, as they are passed by value. For C code accepting pointers, Ref{T} should generally be used for the types of input arguments, allowing the use of pointers to memory managed by either Julia or C through the implicit call to Base.cconvert. In contrast, pointers returned by the C function called should be declared to be of the output type Ptr{T}, reflecting that the memory pointed to is managed by C only. Pointers contained in C structs should be represented as fields of type Ptr{T} within the corresponding Julia struct types designed to mimic the internal structure of corresponding C structs.\n\nIn Julia code wrapping calls to external Fortran routines, all input arguments should be declared as of type Ref{T}, as Fortran passes all variables by pointers to memory locations. The return type should either be Cvoid for Fortran subroutines, or a T for Fortran functions returning the type T."},{"location":"manual/calling-c-and-fortran-code.html#Mapping-C-Functions-to-Julia","page":"Calling C and Fortran Code","title":"Mapping C Functions to Julia","category":"section","text":""},{"location":"manual/calling-c-and-fortran-code.html#@ccall-/-@cfunction-argument-translation-guide","page":"Calling C and Fortran Code","title":"@ccall / @cfunction argument translation guide","category":"section","text":"For translating a C argument list to Julia:\n\nT, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nT, where T is an equivalent Julia Bits Type (per the table above)\nif T is an enum, the argument type should be equivalent to Cint or Cuint\nargument value will be copied (passed by value)\nstruct T (including typedef to a struct)\nT, where T is a concrete Julia type\nargument value will be copied (passed by value)\nvector T (or __attribute__ vector_size, or a typedef such as __m128)\nNTuple{N, VecElement{T}}, where T is a primitive Julia type of the correct size and N is the number of elements in the vector (equal to vector_size / sizeof T).\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Cvoid} if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\njl_value_t* const*\nRef{Any}\nargument list must be a valid Julia object (or C_NULL)\ncannot be used for an output parameter, unless the user is able to separately arrange for the object to be GC-preserved\nT*\nRef{T}, where T is the Julia type corresponding to T\nargument value will be copied if it is an inlinealloc type (which includes isbits otherwise, the value must be a valid Julia object\nT (*)(...) (e.g. a pointer to a function)\nPtr{Cvoid} (you may need to use @cfunction explicitly to create this pointer)\n... (e.g. a vararg)\n[for ccall]: T..., where T is the single Julia type of all remaining arguments\n[for @ccall]: ; va_arg1::T, va_arg2::S, etc, where T and S are the Julia type (i.e. separate the regular arguments from varargs with a ;)\ncurrently unsupported by @cfunction\nva_arg\nnot supported by ccall or @cfunction"},{"location":"manual/calling-c-and-fortran-code.html#@ccall-/-@cfunction-return-type-translation-guide","page":"Calling C and Fortran Code","title":"@ccall / @cfunction return type translation guide","category":"section","text":"For translating a C return type to Julia:\n\nvoid\nCvoid (this will return the singleton instance nothing::Cvoid)\nT, where T is one of the primitive types: char, int, long, short, float, double, complex, enum or any of their typedef equivalents\nsame as C argument list\nargument value will be copied (returned by-value)\nstruct T (including typedef to a struct)\nsame as C argument list\nargument value will be copied (returned by-value)\nvector T\nsame as C argument list\nvoid*\ndepends on how this parameter is used, first translate this to the intended pointer type, then determine the Julia equivalent using the remaining rules in this list\nthis argument may be declared as Ptr{Cvoid} if it really is just an unknown pointer\njl_value_t*\nAny\nargument value must be a valid Julia object\njl_value_t**\nPtr{Any} (Ref{Any} is invalid as a return type)\nT*\nIf the memory is already owned by Julia, or is an isbits type, and is known to be non-null:\nRef{T}, where T is the Julia type corresponding to T\na return type of Ref{Any} is invalid, it should either be Any (corresponding to jl_value_t*) or Ptr{Any} (corresponding to jl_value_t**)\nC MUST NOT modify the memory returned via Ref{T} if T is an isbits type\nIf the memory is owned by C:\nPtr{T}, where T is the Julia type corresponding to T\nT (*)(...) (e.g. a pointer to a function)\nPtr{Cvoid} to call this directly from Julia you will need to pass this as the first argument to @ccall. See Indirect Calls."},{"location":"manual/calling-c-and-fortran-code.html#Passing-Pointers-for-Modifying-Inputs","page":"Calling C and Fortran Code","title":"Passing Pointers for Modifying Inputs","category":"section","text":"Because C doesn't support multiple return values, often C functions will take pointers to data that the function will modify. To accomplish this within a @ccall, you need to first encapsulate the value inside a Ref{T} of the appropriate type. When you pass this Ref object as an argument, Julia will automatically pass a C pointer to the encapsulated data:\n\nwidth = Ref{Cint}(0)\nrange = Ref{Cfloat}(0)\n@ccall foo(width::Ref{Cint}, range::Ref{Cfloat})::Cvoid\n\nUpon return, the contents of width and range can be retrieved (if they were changed by foo) by width[] and range[]; that is, they act like zero-dimensional arrays."},{"location":"manual/calling-c-and-fortran-code.html#C-Wrapper-Examples","page":"Calling C and Fortran Code","title":"C Wrapper Examples","category":"section","text":"Let's start with a simple example of a C wrapper that returns a Ptr type:\n\nmutable struct gsl_permutation\nend\n\n# The corresponding C signature is\n#     gsl_permutation * gsl_permutation_alloc (size_t n);\nfunction permutation_alloc(n::Integer)\n    output_ptr = @ccall \"libgsl\".gsl_permutation_alloc(n::Csize_t)::Ptr{gsl_permutation}\n    if output_ptr == C_NULL # Could not allocate memory\n        throw(OutOfMemoryError())\n    end\n    return output_ptr\nend\n\nThe GNU Scientific Library (here assumed to be accessible through :libgsl) defines an opaque pointer, gsl_permutation *, as the return type of the C function gsl_permutation_alloc. As user code never has to look inside the gsl_permutation struct, the corresponding Julia wrapper simply needs a new type declaration, gsl_permutation, that has no internal fields and whose sole purpose is to be placed in the type parameter of a Ptr type. The return type of the ccall is declared as Ptr{gsl_permutation}, since the memory allocated and pointed to by output_ptr is controlled by C.\n\nThe input n is passed by value, and so the function's input signature is simply declared as ::Csize_t without any Ref or Ptr necessary. (If the wrapper was calling a Fortran function instead, the corresponding function input signature would instead be ::Ref{Csize_t}, since Fortran variables are passed by pointers.) Furthermore, n can be any type that is convertible to a Csize_t integer; the ccall implicitly calls Base.cconvert(Csize_t, n).\n\nHere is a second example wrapping the corresponding destructor:\n\n# The corresponding C signature is\n#     void gsl_permutation_free (gsl_permutation * p);\nfunction permutation_free(p::Ptr{gsl_permutation})\n    @ccall \"libgsl\".gsl_permutation_free(p::Ptr{gsl_permutation})::Cvoid\nend\n\nHere is a third example passing Julia arrays:\n\n# The corresponding C signature is\n#    int gsl_sf_bessel_Jn_array (int nmin, int nmax, double x,\n#                                double result_array[])\nfunction sf_bessel_Jn_array(nmin::Integer, nmax::Integer, x::Real)\n    if nmax < nmin\n        throw(DomainError())\n    end\n    result_array = Vector{Cdouble}(undef, nmax - nmin + 1)\n    errorcode = @ccall \"libgsl\".gsl_sf_bessel_Jn_array(\n                    nmin::Cint, nmax::Cint, x::Cdouble, result_array::Ref{Cdouble})::Cint\n    if errorcode != 0\n        error(\"GSL error code $errorcode\")\n    end\n    return result_array\nend\n\nThe C function wrapped returns an integer error code; the results of the actual evaluation of the Bessel J function populate the Julia array result_array. This variable is declared as a Ref{Cdouble}, since its memory is allocated and managed by Julia. The implicit call to Base.cconvert(Ref{Cdouble}, result_array) unpacks the Julia pointer to a Julia array data structure into a form understandable by C."},{"location":"manual/calling-c-and-fortran-code.html#Fortran-Wrapper-Example","page":"Calling C and Fortran Code","title":"Fortran Wrapper Example","category":"section","text":"The following example utilizes ccall to call a function in a common Fortran library (libBLAS) to compute a dot product. Notice that the argument mapping is a bit different here than above, as we need to map from Julia to Fortran. On every argument type, we specify Ref or Ptr. This mangling convention may be specific to your Fortran compiler and operating system and is likely undocumented. However, wrapping each in a Ref (or Ptr, where equivalent) is a frequent requirement of Fortran compiler implementations:\n\nfunction compute_dot(DX::Vector{Float64}, DY::Vector{Float64})\n    @assert length(DX) == length(DY)\n    n = length(DX)\n    incx = incy = 1\n    product = @ccall \"libLAPACK\".ddot(\n        n::Ref{Int32}, DX::Ptr{Float64}, incx::Ref{Int32}, DY::Ptr{Float64}, incy::Ref{Int32})::Float64\n    return product\nend"},{"location":"manual/calling-c-and-fortran-code.html#Garbage-Collection-Safety","page":"Calling C and Fortran Code","title":"Garbage Collection Safety","category":"section","text":"When passing data to a @ccall, it is best to avoid using the pointer function. Instead define a Base.cconvert method and pass the variables directly to the @ccall. @ccall automatically arranges that all of its arguments will be preserved from garbage collection until the call returns. If a C API will store a reference to memory allocated by Julia, after the @ccall returns, you must ensure that the object remains visible to the garbage collector. The suggested way to do this is to make a global variable of type Vector{Ref} to hold these values until the C library notifies you that it is finished with them.\n\nWhenever you have created a pointer to Julia data, you must ensure the original data exists until you have finished using the pointer. Many methods in Julia such as unsafe_load and String make copies of data instead of taking ownership of the buffer, so that it is safe to free (or alter) the original data without affecting Julia. A notable exception is unsafe_wrap which, for performance reasons, shares (or can be told to take ownership of) the underlying buffer.\n\nThe garbage collector does not guarantee any order of finalization. That is, if a contained a reference to b and both a and b are due for garbage collection, there is no guarantee that b would be finalized after a. If proper finalization of a depends on b being valid, it must be handled in other ways."},{"location":"manual/calling-c-and-fortran-code.html#Non-constant-Function-Specifications","page":"Calling C and Fortran Code","title":"Non-constant Function Specifications","category":"section","text":"In some cases, the exact name or path of the needed library is not known in advance and must be computed at run time. To handle such cases, the library component specification can be a value such as Libdl.LazyLibrary. The runtime will call Libdl.dlopen on that object when first used by a ccall."},{"location":"manual/calling-c-and-fortran-code.html#man-lazylibrary","page":"Calling C and Fortran Code","title":"Using LazyLibrary for Lazy Loading","category":"section","text":"Libdl.LazyLibrary provides a thread-safe mechanism for deferring library loading until first use. This is the recommended approach for library initialization in modern Julia code.\n\nA LazyLibrary represents a library that opens itself (and its dependencies) automatically on first use in a ccall(), @ccall, dlopen(), dlsym(), dlpath(), or cglobal(). The library is loaded exactly once in a thread-safe manner, and subsequent calls reuse the loaded library handle."},{"location":"manual/calling-c-and-fortran-code.html#Basic-Usage","page":"Calling C and Fortran Code","title":"Basic Usage","category":"section","text":"using Libdl\n\n# Define a LazyLibrary as a const for optimal performance\nconst libz = LazyLibrary(\"libz\")\n\n# Use directly in @ccall - library loads automatically on first call\n@ccall libz.deflate(strm::Ptr{Cvoid}, flush::Cint)::Cint\n\n# Also works with ccall\nccall((:inflate, libz), Cint, (Ptr{Cvoid}, Cint), strm, flush)"},{"location":"manual/calling-c-and-fortran-code.html#Platform-Specific-Libraries","page":"Calling C and Fortran Code","title":"Platform-Specific Libraries","category":"section","text":"For code that needs to work across different platforms:\n\nconst mylib = LazyLibrary(\n    if Sys.iswindows()\n        \"mylib.dll\"\n    elseif Sys.isapple()\n        \"libmylib.dylib\"\n    else\n        \"libmylib.so\"\n    end\n)"},{"location":"manual/calling-c-and-fortran-code.html#Libraries-with-Dependencies","page":"Calling C and Fortran Code","title":"Libraries with Dependencies","category":"section","text":"When a library depends on other libraries, specify the dependencies to ensure they load in the correct order:\n\nconst libfoo = LazyLibrary(\"libfoo\")\nconst libbar = LazyLibrary(\"libbar\"; dependencies=[libfoo])\n\n# When libbar is first used, libfoo is loaded first automatically\n@ccall libbar.bar_function(x::Cint)::Cint"},{"location":"manual/calling-c-and-fortran-code.html#Lazy-Path-Construction","page":"Calling C and Fortran Code","title":"Lazy Path Construction","category":"section","text":"For libraries whose paths are determined at runtime, use LazyLibraryPath:\n\n# Path is constructed when library is first accessed\nconst mylib = LazyLibrary(LazyLibraryPath(artifact_dir, \"lib\", \"libmylib.so\"))"},{"location":"manual/calling-c-and-fortran-code.html#Initialization-Callbacks","page":"Calling C and Fortran Code","title":"Initialization Callbacks","category":"section","text":"If a library requires initialization after loading:\n\nconst mylib = LazyLibrary(\"libmylib\";\n    on_load_callback = () -> @ccall mylib.initialize()::Cvoid\n)\n\nwarning: Warning\nThe on_load_callback should be minimal and must not call wait() on any tasks. It is called exactly once by the thread that loads the library."},{"location":"manual/calling-c-and-fortran-code.html#Conversion-from-__init__()-Pattern","page":"Calling C and Fortran Code","title":"Conversion from __init__() Pattern","category":"section","text":"Before LazyLibrary, library paths were often computed in __init__() functions. This pattern can be replaced with LazyLibrary for better performance and thread safety.\n\nOld pattern using __init__():\n\n# Old: Library path computed in __init__()\nlibmylib_path = \"\"\n\nfunction __init__(\n    # Loads library on startup, whether it is used or not\n    global libmylib_path = find_library([\"libmylib\"])\nend\n\nfunction myfunc(x)\n    ccall((:cfunc, libmylib_path), Cint, (Cint,), x)\nend\n\nNew pattern using LazyLibrary:\n\n# New: Library as const, no __init__() needed\nconst libmylib = LazyLibrary(\"libmylib\")\n\nfunction myfunc(x)\n    # Library loads automatically just before calling `cfunc`\n    @ccall libmylib.cfunc(x::Cint)::Cint\nend\n\nFor more details, see the Libdl.LazyLibrary documentation."},{"location":"manual/calling-c-and-fortran-code.html#Overloading-dlopen-for-Custom-Types","page":"Calling C and Fortran Code","title":"Overloading dlopen for Custom Types","category":"section","text":"The runtime will call dlsym(:function, dlopen(library)::Ptr{Cvoid}) when a @ccall is executed. The Libdl.dlopen function can be overloaded for custom types to provide alternate behaviors. However, it is assumed that the library location and handle does not change once it is determined, so the result of the call may be cached and reused. Therefore, the number of times the dlopen expression executes is unspecified, and returning different values for multiple calls will results in unspecified (but valid) behavior."},{"location":"manual/calling-c-and-fortran-code.html#Computed-Function-Names","page":"Calling C and Fortran Code","title":"Computed Function Names","category":"section","text":"If even more flexibility is needed, it is possible to use computed values as function names by staging through eval as follows:\n\n@eval @ccall \"lib\".$(string(\"a\", \"b\"))()::Cint\n\nThis expression constructs a name using string, then substitutes this name into a new @ccall expression, which is then evaluated. Keep in mind that eval only operates at the top level, so within this expression local variables will not be available (unless their values are substituted with $). For this reason, eval is typically only used to form top-level definitions, for example when wrapping libraries that contain many similar functions."},{"location":"manual/calling-c-and-fortran-code.html#Indirect-Calls","page":"Calling C and Fortran Code","title":"Indirect Calls","category":"section","text":"The first argument to @ccall can also be an expression to be evaluated at run time, each time it is used. In this case, the expression must evaluate to a Ptr, which will be used as the address of the native function to call. This behavior occurs when the first @ccall argument is marked with $ and when the first ccall argument is not a simple constant literal or expression in (). The argument can be any expression and can use local variables and arguments and can return a different value every time.\n\nFor example, you might implement a macro similar to cglobal that looks up the function via dlsym, then caches the pointer in a shared reference (which is auto reset to C_NULL during precompile saving). For example:\n\nmacro dlsym(lib, func)\n    z = Ref(C_NULL)\n    quote\n        local zlocal = $z[]\n        if zlocal == C_NULL\n            zlocal = dlsym($(esc(lib))::Ptr{Cvoid}, $(esc(func)))::Ptr{Cvoid}\n            $z[] = zlocal\n        end\n        zlocal\n    end\nend\n\nconst mylibvar = LazyLibrary(\"mylib\")\n@ccall $(@dlsym(dlopen(mylibvar), \"myfunc\"))()::Cvoid"},{"location":"manual/calling-c-and-fortran-code.html#Closure-cfunctions","page":"Calling C and Fortran Code","title":"Closure cfunctions","category":"section","text":"The first argument to @cfunction can be marked with a $, in which case the return value will instead be a struct CFunction which closes over the argument. You must ensure that this return object is kept alive until all uses of it are done. The contents and code at the cfunction pointer will be erased via a finalizer when this reference is dropped and atexit. This is not usually needed, since this functionality is not present in C, but can be useful for dealing with ill-designed APIs which don't provide a separate closure environment parameter.\n\nfunction qsort(a::Vector{T}, cmp) where T\n    isbits(T) || throw(ArgumentError(\"this method can only qsort isbits arrays\"))\n    callback = @cfunction $cmp Cint (Ref{T}, Ref{T})\n    # Here, `callback` isa Base.CFunction, which will be converted to Ptr{Cvoid}\n    # (and protected against finalization) by the ccall\n    @ccall qsort(a::Ptr{T}, length(a)::Csize_t, Base.elsize(a)::Csize_t, callback::Ptr{Cvoid})\n    # We could instead use:\n    #    GC.@preserve callback begin\n    #        use(Base.unsafe_convert(Ptr{Cvoid}, callback))\n    #    end\n    # if we needed to use it outside of a `ccall`\n    return a\nend\n\nnote: Note\nClosure @cfunction relies on LLVM trampolines, which are not available on all platforms (for example ARM and PowerPC)."},{"location":"manual/calling-c-and-fortran-code.html#Closing-a-Library","page":"Calling C and Fortran Code","title":"Closing a Library","category":"section","text":"It is sometimes useful to close (unload) a library so that it can be reloaded. For instance, when developing C code for use with Julia, one may need to compile, call the C code from Julia, then close the library, make an edit, recompile, and load in the new changes. One can either restart Julia or use the Libdl functions to manage the library explicitly, such as:\n\nlib = Libdl.dlopen(\"./my_lib.so\") # Open the library explicitly.\nsym = Libdl.dlsym(lib, :my_fcn)   # Get a symbol for the function to call.\n@ccall $sym(...) # Use the pointer `sym` instead of the library.symbol tuple.\nLibdl.dlclose(lib) # Close the library explicitly.\n\nNote that when using @ccall with the input (e.g., @ccall \"./my_lib.so\".my_fcn(...)::Cvoid), the library is opened implicitly and it may not be explicitly closed."},{"location":"manual/calling-c-and-fortran-code.html#Variadic-function-calls","page":"Calling C and Fortran Code","title":"Variadic function calls","category":"section","text":"To call variadic C functions a semicolon can be used in the argument list to separate required arguments from variadic arguments. An example with the printf function is given below:\n\njulia> @ccall printf(\"%s = %d\\n\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint\nfoo = 3\n8"},{"location":"manual/calling-c-and-fortran-code.html#ccall-interface","page":"Calling C and Fortran Code","title":"ccall interface","category":"section","text":"There is another alternative interface to @ccall. This interface is slightly less convenient but it does allow one to specify a calling convention.\n\nThe arguments to ccall are:\n\nA (:function, \"library\") pair (most common),\nOR\na :function name symbol or \"function\" name string (for symbols in the current process or libc),\nOR\na function pointer (for example, from dlsym).\nThe function's return type\nA tuple of input types, corresponding to the function signature. One common mistake is forgetting that a 1-tuple of argument types must be written with a trailing comma.\nThe actual argument values to be passed to the function, if any; each is a separate parameter.\n\nnote: Note\nThe (:function, \"library\") pair and the input type list must be syntactic tuples (i.e., they can't be variables or values with a type of Tuple.The rettype and argument type values are evaluated at when the containing method is defined, not runtime.\n\nnote: Function Name vs Pointer Syntax\nThe syntax of the first argument to ccall determines whether you're calling by name or by pointer:Name-based calls (tuple literal syntax):\nBoth the function and library names can be a quoted Symbol, a String, a variable name (a GlobalRef), or a dotted expression ending with a variable name.\nSingle name: (:function_name,) or \"function_name\" - uses default library lookup.\nName with library: (:function_name, \"library\") - specifies both function and library.\nSymbol, string, and tuple literal constants (not expressions that evaluate to those constants, but actual literals) are automatically normalized to tuple form.\nPointer-based calls (non-tuple syntax):\nAnything that is not a literal tuple expression specified above is assumed to be an expression that evaluates to a function pointers at runtime.\nFunction pointer variables: fptr where fptr is a runtime pointer value.\nFunction pointer computations: dlsym(:something) where the result is computed at runtime every time (usually along with some caching logic).\nLibrary name expressions:\nWhen given as a variable, the library name can resolve to a Symbol, a String, or any other value. The runtime will call Libdl.dlopen(name) on the value an unspecified number of times, caching the result. The result is not invalidated if the value of the binding changes or if it becomes undefined, as long as there exists any value for that binding in any past or future worlds, that value may be used.\nDot expressions, such as A.B().c, will be executed at method definition time up to the final c. The first part must resolve to a Module, and the second part to a quoted symbol. The value of that global will be resolved at runtime when the ccall is first executed.\n\nA table of translations between the macro and function interfaces is given below.\n\n@ccall ccall\n@ccall clock()::Int32 ccall(:clock, Int32, ())\n@ccall f(a::Cint)::Cint ccall(:f, Cint, (Cint,), a)\n@ccall \"mylib\".f(a::Cint, b::Cdouble)::Cvoid ccall((:f, \"mylib\"), Cvoid, (Cint, Cdouble), a, b)\n@ccall $fptr.f()::Cvoid ccall(fptr, f, Cvoid, ())\n@ccall printf(\"%s = %d\\n\"::Cstring ; \"foo\"::Cstring, foo::Cint)::Cint <unavailable>\n@ccall printf(\"%s = %s\\n\"::Cstring ; \"2 + 2\"::Cstring, \"5\"::Cstring)::Cint ccall(:printf, Cint, (Cstring, Cstring...), \"%s = %s\\n\", \"2 + 2\", \"5\")\n<unavailable> ccall(:gethostname, stdcall, Int32, (Ptr{UInt8}, UInt32), hn, length(hn))"},{"location":"manual/calling-c-and-fortran-code.html#calling-convention","page":"Calling C and Fortran Code","title":"Calling Convention","category":"section","text":"The second argument to ccall (immediately preceding return type) can optionally be a calling convention specifier (the @ccall macro currently does not support giving a calling convention). Without any specifier, the platform-default C calling convention is used. Other supported conventions are: stdcall, cdecl, fastcall, and thiscall (no-op on 64-bit Windows). For example (from base/libc.jl) we see the same gethostname ccall as above, but with the correct signature for Windows:\n\nhn = Vector{UInt8}(undef, 256)\nerr = ccall(:gethostname, stdcall, Int32, (Ptr{UInt8}, UInt32), hn, length(hn))\n\nFor more information, please see the LLVM Language Reference.\n\nThere is one additional special calling convention llvmcall, which allows inserting calls to LLVM intrinsics directly. This can be especially useful when targeting unusual platforms such as GPGPUs. For example, for CUDA, we need to be able to read the thread index:\n\nccall(\"llvm.nvvm.read.ptx.sreg.tid.x\", llvmcall, Int32, ())\n\nAs with any ccall, it is essential to get the argument signature exactly correct. Also, note that there is no compatibility layer that ensures the intrinsic makes sense and works on the current target, unlike the equivalent Julia functions exposed by Core.Intrinsics."},{"location":"manual/calling-c-and-fortran-code.html#Accessing-Global-Variables","page":"Calling C and Fortran Code","title":"Accessing Global Variables","category":"section","text":"Global variables exported by native libraries can be accessed by name using the cglobal function. The arguments to cglobal are a symbol specification identical to that used by ccall, and a type describing the value stored in the variable:\n\njulia> cglobal((:errno, :libc), Int32)\nPtr{Int32} @0x00007f418d0816b8\n\nThe result is a pointer giving the address of the value. The value can be manipulated through this pointer using unsafe_load and unsafe_store!.\n\nnote: Note\nThis errno symbol may not be found in a library named \"libc\", as this is an implementation detail of your system compiler. Typically standard library symbols should be accessed just by name, allowing the compiler to fill in the correct one. Also, however, the errno symbol shown in this example is special in most compilers, and so the value seen here is probably not what you expect or want. Compiling the equivalent code in C on any multi-threaded-capable system would typically actually call a different function (via macro preprocessor overloading), and may give a different result than the legacy value printed here."},{"location":"manual/calling-c-and-fortran-code.html#Accessing-Data-through-a-Pointer","page":"Calling C and Fortran Code","title":"Accessing Data through a Pointer","category":"section","text":"The following methods are described as \"unsafe\" because a bad pointer or type declaration can cause Julia to terminate abruptly.\n\nGiven a Ptr{T}, the contents of type T can generally be copied from the referenced memory into a Julia object using unsafe_load(ptr, [index]). The index argument is optional (default is 1), and follows the Julia-convention of 1-based indexing. This function is intentionally similar to the behavior of getindex and setindex! (e.g. [] access syntax).\n\nThe return value will be a new object initialized to contain a copy of the contents of the referenced memory. The referenced memory can safely be freed or released.\n\nIf T is Any, then the memory is assumed to contain a reference to a Julia object (a jl_value_t*), the result will be a reference to this object, and the object will not be copied. You must be careful in this case to ensure that the object was always visible to the garbage collector (pointers do not count, but the new reference does) to ensure the memory is not prematurely freed. Note that if the object was not originally allocated by Julia, the new object will never be finalized by Julia's garbage collector. If the Ptr itself is actually a jl_value_t*, it can be converted back to a Julia object reference by unsafe_pointer_to_objref(ptr). (Julia values v can be converted to jl_value_t* pointers, as Ptr{Cvoid}, by calling pointer_from_objref(v).)\n\nThe reverse operation (writing data to a Ptr{T}), can be performed using unsafe_store!(ptr, value, [index]). Currently, this is only supported for primitive types or other pointer-free (isbits) immutable struct types.\n\nAny operation that throws an error is probably currently unimplemented and should be posted as a bug so that it can be resolved.\n\nIf the pointer of interest is a plain-data array (primitive type or immutable struct), the function unsafe_wrap(Array, ptr,dims, own = false) may be more useful. The final parameter should be true if Julia should \"take ownership\" of the underlying buffer and call free(ptr) when the returned Array object is finalized. If the own parameter is omitted or false, the caller must ensure the buffer remains in existence until all access is complete.\n\nArithmetic on the Ptr type in Julia (e.g. using +) does not behave the same as C's pointer arithmetic. Adding an integer to a Ptr in Julia always moves the pointer by some number of bytes, not elements. This way, the address values obtained from pointer arithmetic do not depend on the element types of pointers."},{"location":"manual/calling-c-and-fortran-code.html#Thread-safety","page":"Calling C and Fortran Code","title":"Thread-safety","category":"section","text":"Some C libraries execute their callbacks from a different thread, and since Julia isn't thread-safe you'll need to take some extra precautions. In particular, you'll need to set up a two-layered system: the C callback should only schedule (via Julia's event loop) the execution of your \"real\" callback. To do this, create an AsyncCondition object and wait on it:\n\ncond = Base.AsyncCondition()\nwait(cond)\n\nThe callback you pass to C should only execute a ccall to :uv_async_send, passing cond.handle as the argument, taking care to avoid any allocations or other interactions with the Julia runtime.\n\nNote that events may be coalesced, so multiple calls to uv_async_send may result in a single wakeup notification to the condition."},{"location":"manual/calling-c-and-fortran-code.html#More-About-Callbacks","page":"Calling C and Fortran Code","title":"More About Callbacks","category":"section","text":"For more details on how to pass callbacks to C libraries, see this blog post."},{"location":"manual/calling-c-and-fortran-code.html#C","page":"Calling C and Fortran Code","title":"C++","category":"section","text":"For tools to create C++ bindings, see the CxxWrap package.\n\n[1]: Non-library function calls in both C and Julia can be inlined and thus may have even less overhead than calls to shared library functions. The point above is that the cost of actually doing foreign function call is about the same as doing a call in either native language.\n\n[2]: The Clang package can be used to auto-generate Julia code from a C header file."},{"location":"manual/installation.html#man-installation","page":"Installation","title":"Installation","category":"section","text":"There are many ways to install Julia. The following sections highlight the recommended method for each of the main supported platforms, and then present alternative ways that might be useful in specialized situations.\n\nThe current installation recommendation is a solution based on Juliaup. If you installed Julia previously with a method that is not based on Juliaup and want to switch your system to an installation that is based on Juliaup, we recommend that you uninstall all previous Julia versions, ensure that you remove anything Julia related from your PATH variable and then install Julia with one of the methods described below."},{"location":"manual/installation.html#Windows","page":"Installation","title":"Windows","category":"section","text":"On Windows Julia can be installed directly from the Windows store here. One can also install exactly the same version by executing\n\nwinget install --name Julia --id 9NJNWW8PVKMN -e -s msstore\n\nin any shell."},{"location":"manual/installation.html#Mac-and-Linux","page":"Installation","title":"Mac and Linux","category":"section","text":"Julia can be installed on Linux or Mac by executing\n\ncurl -fsSL https://install.julialang.org | sh\n\nin a shell."},{"location":"manual/installation.html#Command-line-arguments","page":"Installation","title":"Command line arguments","category":"section","text":"One can pass various command line arguments to the Julia installer. The syntax for installer arguments is\n\ncurl -fsSL https://install.julialang.org | sh -s -- <ARGS>\n\nHere <ARGS> should be replaced with one or more of the following arguments:\n\n--yes (or -y): Run the installer in a non-interactive mode. All configuration values use their default or a value supplied as a command line argument.\n--default-channel=<NAME>: Configure the default Juliaup channel. For example --default-channel lts would install the lts channel and configure it as the default.\n--add-to-path=<yes|no>: Configure whether Julia should be added to the PATH environment variable. Valid values are yes (default) and no.\n--background-selfupdate=<SECONDS>: Configure an optional CRON job that auto-updates Juliaup if <SECONDS> has a value larger than 0. The actual value controls how often the CRON job will run to check for a new Juliaup version in seconds. The default value is 0, i.e. no CRON job will be created.\n--startup-selfupdate=<MINUTES>: Configure how often Julia will check for new versions of Juliaup when Julia is started. The default is every 1440 minutes.\n-p=<PATH> (or --path): Configure where the Julia and Juliaup binaries are installed. The default is ~/.juliaup."},{"location":"manual/installation.html#Alternative-installation-methods","page":"Installation","title":"Alternative installation methods","category":"section","text":"Note that we recommend the following methods only if none of the installation methods described above work for your system.\n\nSome of the installation methods described below recommend installing a package called juliaup. Note that this nevertheless installs a fully functional Julia system, not just Juliaup."},{"location":"manual/installation.html#App-Installer-(Windows)","page":"Installation","title":"App Installer (Windows)","category":"section","text":"If the Windows Store is blocked on a system, we have an alternative MSIX App Installer based setup. To use the App Installer version, download this file and open it by double clicking on it. One can also install exactly the same version by executing the PowerShell command\n\nAdd-AppxPackage -AppInstallerFile https://install.julialang.org/Julia.appinstaller"},{"location":"manual/installation.html#MSI-Installer-(Windows)","page":"Installation","title":"MSI Installer (Windows)","category":"section","text":"If neither the Windows Store nor the App Installer version work on your Windows system, you can also use a MSI based installer. Note that this installation methods comes with serious limitations and is generally not recommended unless no other method works. For example, there is no automatic update mechanism for Juliaup with this installation method. The 64 bit version of the MSI installer can be downloaded from here and the 32 bit version from here.\n\nBy default the install will be a per-user install that does not require  elevation. You can also do a system install by running the following command  from a shell:\n\nmsiexec /i <PATH_TO_JULIA_MSI> ALLUSERS=1"},{"location":"manual/installation.html#[Homebrew](https://brew.sh)-(Mac-and-Linux)","page":"Installation","title":"Homebrew (Mac and Linux)","category":"section","text":"On systems with brew, you can install Julia by running\n\nbrew install juliaup\n\nin a shell. Note that you will have to update Juliaup with standard brew commands."},{"location":"manual/installation.html#[Arch-Linux-AUR](https://aur.archlinux.org/packages/juliaup/)-(Linux)","page":"Installation","title":"Arch Linux - AUR (Linux)","category":"section","text":"On Arch Linux, Juliaup is available in the Arch User Repository (AUR)."},{"location":"manual/installation.html#[openSUSE-Tumbleweed](https://get.ope